/**
 * @copyright    (C) 2012 - 2016 Robert Bosch GmbH.
 *               The reproduction, distribution and utilization of this file as well as the
 *               communication of its contents to others without express authorization is prohibited.
 *               Offenders will be held liable for the payment of damages.
 *               All rights reserved in the event of the grant of a patent, utility model or design.
 * @brief        get/set registry elements for datapool 
 *               only valid if registry for Datapool available
 * @addtogroup   Datapool master
 * @{
 */

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#define OSAL_S_IMPORT_INTERFACE_GENERIC		   
#include "osal_if.h"

#define SYSTEM_S_IMPORT_INTERFACE_LIST
#include "stl_pif.h"

#define REG_S_IMPORT_INTERFACE_GENERIC
#include "reg_if.h"

#define DP_S_IMPORT_INTERFACE_BASE
#define DP_S_IMPORT_INTERFACE_DPELEM
#include "dp_if.h"

#include "dpmaster.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS DP_MASTER_TRACE_CLASS
#include "trcGenProj/Header/dpmasterReg.cpp.trc.h"
#endif

#ifdef SYSTEM_S_USE_BPSTL
typedef bpstl::list<dp_tclBaseElement> TElemList;
#else
typedef std::list<dp_tclBaseElement> TElemList;
#endif

tVoid dp_tclDatapoolMaster::vStoreDpReg() {
   tS32 as32Data[255] = {0};
   tChar cRegKeyPath[] = "/dev/registry/LOCAL_MACHINE/SOFTWARE/BLAUPUNKT/SYSTEM/DATAPOOL";
   TElemList oRegStoreList;

   ETG_TRACE_USR1(("vStoreDpReg(): parse registry for elements to store."));

   OSAL_trIOCtrlDir rDirV;
   OSAL_trIOCtrlRegistry rReg;
   OSAL_tIODescriptor fd;

   fd = OSAL_IOOpen(cRegKeyPath, OSAL_EN_READONLY);
   if (fd != OSAL_ERROR)
   {
      /* Now show all the values of this key */
      rDirV.fd = fd;
      rDirV.s32Cookie = 0;

      while (OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_REGENUMVALUE, (intptr_t) & rDirV) != OSAL_ERROR)
      {

         rReg.pcos8Name = rDirV.dirent.s8Name;
         rReg.ps8Value  = (tU8*)as32Data;
         rReg.u32Size   = sizeof(as32Data);

         if (OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_REGGETVALUE, (intptr_t) & rReg) != OSAL_ERROR)
         {
            // This is correct because not all traces come out if this is missing
            if (rReg.s32Type == OSAL_C_S32_VALUE_S32)
            {
               ETG_TRACE_USR1(("VAL number: '%32s' 0x%08x (%d)", (const tChar*)rDirV.dirent.s8Name, as32Data[0], as32Data[0]));

               dp_tclBaseElement myDpElem((tChar*)rDirV.dirent.s8Name);
               myDpElem.bFillData((tVoid*)&as32Data[0], sizeof(tU32));
               oRegStoreList.push_back(myDpElem);
            }
            else if (rReg.s32Type == OSAL_C_S32_VALUE_STRING)
            {
               tChar* strData = (tChar*)as32Data;
               ETG_TRACE_USR1(("VAL string: '%32s' '%s'", (const tChar*)rDirV.dirent.s8Name, strData));

               dp_tclBaseElement myDpElem((tChar*)rDirV.dirent.s8Name);
               myDpElem.bFillData(strData, ((tU32)strlen(strData))+5); //make sure that strlen is >4
               oRegStoreList.push_back(myDpElem);
            }
         }
      }
      OSAL_s32IOClose(fd);

      tU32 u32FileLen = 0;
      TElemList::iterator posElement;
      for (posElement = oRegStoreList.begin(); posElement != oRegStoreList.end(); ++posElement) {
         //found element -> add length for this element
         u32FileLen +=posElement->u32GetStreamLength();
      }

      // all elements checked -> get memory for file to store
      tU8* pBuffer = new tU8[u32FileLen];
      if (pBuffer) {
         tU8* pu8CurPos = pBuffer;
         for (posElement = oRegStoreList.begin(); posElement != oRegStoreList.end(); ++posElement) {
            // and now stream data to current context
            tU32 u32Len = posElement->u32GetStreamLength();

            ETG_TRACE_USR1(("vStoreDpReg(): data: %d for element: %s", *(tU32*)posElement->pvGetData(), posElement->pcGetElementName()));

            dp_tclOutStreamCtxt oContext(pu8CurPos, u32Len);
            oContext << *posElement;

            pu8CurPos = (tU8*)oContext.pu8GetEndPosition();
         }
      }

      _poPersMemAccessHdl->bWriteFfsData("generic", pBuffer, u32FileLen,eDpLocation_FILE_SYSTEM,0x00, TRUE);

      delete[] pBuffer;
      pBuffer=NULL;/*Bug 123144: Safe handling*/
   }

   return;
}

tVoid dp_tclDatapoolMaster::vLoadDpReg() 
{
   tChar cRegKeyPath[] = "/dev/registry/LOCAL_MACHINE/SOFTWARE/BLAUPUNKT/SYSTEM/DATAPOOL";
   reg_tclRegKey  oReg;
   tS32 s32ReadBytes=0;
   ETG_TRACE_USR1(("vLoadDpReg(): parse registry for elements to store."));
   if( oReg.bOpen(cRegKeyPath)) 
   {
      tU8* pu8Buffer = NULL;
      s32ReadBytes = _poPersMemAccessHdl->s32ImportPersData("generic",&pu8Buffer,eDpLocation_FILE_SYSTEM,0x00);
      if (0 < s32ReadBytes) 
      {//extract data to pool
         tU8* pu8CurPos = pu8Buffer;
         while ((pu8CurPos+sizeof(tU32)) < (pu8Buffer + s32ReadBytes)) 
         {//lint !e662 :Possible creation of out-of-bounds pointer (4 beyond end of data) by operator 'ptr+int' -> buffer ranges checked
            //create empty element
            dp_tclBaseElement myDataElem;
            //stream data to context
            dp_tclInStreamCtxt oInContext(pu8CurPos);
            oInContext >> myDataElem;
            pu8CurPos = (tU8*)oInContext.pu8GetEndPosition();
            ETG_TRACE_USR1(("vLoadDpReg(): len: %d for element: %s", myDataElem.u32GetDataLength(), myDataElem.pcGetElementName()));
            tChar strValue[255] = {0};
            if (!oReg.bQueryString(myDataElem.pcGetElementName(), strValue, 255)) 
            {
               oReg.bSetString(myDataElem.pcGetElementName(), (tChar*)myDataElem.pvGetData(), myDataElem.u32GetDataLength());
            }
         }
      }
      delete[] pu8Buffer;
      pu8Buffer=NULL; /*Bug 123144: Safe handling*/
   }
   return;
}
//EOF
