/**
 * @copyright    (C) 2012 - 2016 Robert Bosch GmbH.
 *               The reproduction, distribution and utilization of this file as well as the
 *               communication of its contents to others without express authorization is prohibited.
 *               Offenders will be held liable for the payment of damages.
 *               All rights reserved in the event of the grant of a patent, utility model or design.
 * @brief        includes main datapool class, DP-Thread is spawned that mainly is used for
 *               communication with the DP-Master and to realize the time slices for the different
 *               storage strategies for persistence data handling.
 *               The registration at the DP-Master is also located here.
 *               The class follows a Singleton design pattern.
 *               There will be exactly one instance of this class in each process where a DP-Component should be installed.
 * @addtogroup   Datapool main
 * @{
 */

#include <sys/fcntl.h>
#include <unistd.h>

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#include "OsalConf.h"
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
#include "osal_public.h"

#include "oserror.h"
#ifndef REG_S_IMPORT_INTERFACE_MOCKUP
#define REG_S_IMPORT_INTERFACE_GENERIC
#include "reg_if.h"
#else
#include "../reg_mock.h"
#endif

#define DP_S_IMPORT_INTERFACE_BASE
#define DP_S_IMPORT_INTERFACE_DPPERSACCESS
#define DP_S_IMPORT_INTERFACE_FI
#define DP_S_IMPORT_INTERFACE_DPENDUSER
#include "dp_if.h"

#include "ccaservice.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS DP_TRACE_CLASS
#include "trcGenProj/Header/dpbase.cpp.trc.h"
#endif

#define DP_MAILBOX_MAX_MESSAGE_COUNT 20

tBool DP_bEnterCritical(OSAL_tSemHandle hSemId, OSAL_tMSecond tTimeout = OSAL_C_U32_INFINITE)
{
   tS32 s32Ret = OSAL_s32SemaphoreWait(hSemId, tTimeout);
   tBool bRet = TRUE;
   if (s32Ret != OSAL_OK)
   {
      if (tTimeout == OSAL_C_U32_INFINITE)
      { //fatal situation -> ERROR TRACE
         ETG_TRACE_ERR(("DP_bEnterCritical(): error ->Failed to get Semaphore!!!!"));
      }
      else
      {  //timeout detected
         bRet = FALSE;
      }
   }
   return bRet;
}

tBool DP_bReleaseCritical(OSAL_tSemHandle hSemId)
{
   tBool bSemPost = FALSE;
   tS32 s32Ret = OSAL_s32SemaphorePost(hSemId);
   if (s32Ret == OSAL_OK)
   {
      bSemPost = TRUE;
   }
   else
   {
      ETG_TRACE_ERR(("DP_bReleaseCritical(): error -> post Semaphore!!!!"));
   }
   return bSemPost;
}

// ****************************************************************************
// ********   initialization part 
// ****************************************************************************

dp_tclDpBase::dp_tclDpBase()
{
   _hMasterQueueHandle = OSAL_C_INVALID_HANDLE;
   _bTerminate = FALSE;
   _hMessageQueueHandle = OSAL_C_INVALID_HANDLE;
   _hEvSync = OSAL_C_INVALID_HANDLE;
   _u32LockMode = DP_U32_LOCK_MODE_NONE;
   _hDpAccess = OSAL_C_INVALID_HANDLE;
   _hDpSemSrvIf = OSAL_C_INVALID_HANDLE;
   _poPersMem = NULL;
   _tListSubPools.clear();
   _bSubListLoad = FALSE;
   _bListTraceElementsLoad = FALSE;
   _oListTraceElements.clear();
#ifdef DP_FEATURE_UNDEF_ELEMENT
   oDpUndefPoolProperty.clear();
#endif
   oDatapool.clear();
   _u32EvWaitForMask = 0;
   memset(_szMyProcSpecName, 0, sizeof(_szMyProcSpecName));
   memset(_szMyProcResourceName, 0, sizeof(_szMyProcResourceName));

   //get _szMyProcSpecName from /proc/%d/cmdline
   {
      int  ViFileHandle;
      char VcPathProcess[OSAL_C_U32_MAX_PATHLENGTH];
      snprintf(VcPathProcess, OSAL_C_U32_MAX_PATHLENGTH, "/proc/%d/cmdline",  (tS32)OSAL_ProcessWhoAmI());// Bug 247896: Fixes compiler warning.
      if ((ViFileHandle = open(VcPathProcess, O_RDONLY, 0)) != -1)
      {
         tS32 Vu32Lenght = (tS32)read(ViFileHandle, VcPathProcess, OSAL_C_U32_MAX_PATHLENGTH);
         if (Vu32Lenght > 0)
         {
            char *VcpPos;
            VcpPos = strrchr(VcPathProcess, '/');
            if (VcpPos != NULL)
            { /*copy process name*/
               char VszPrcName[DP_MAX_LEN_NAME_PROCESS];
               OSAL_szStringNCopy(VszPrcName, VcpPos + 1, DP_MAX_LEN_NAME_PROCESS);
               VszPrcName[DP_MAX_LEN_NAME_PROCESS - 1] = '\0';
               VcpPos = strchr(VszPrcName, '.');
               if (VcpPos != NULL)
               {
                  VcpPos[0] = '\0';
               }
               snprintf(_szMyProcSpecName, DP_MAX_LEN_NAME_PROCESS, "Dp_%s", VszPrcName);
               ETG_TRACE_FATAL(("dp_tclDpBase(): process pid:%d, name '%s'", OSAL_ProcessWhoAmI(), _szMyProcSpecName));
            }
            else
            {
               ETG_TRACE_FATAL(("dp_tclDpBase(): find no / in string '%s'  --> use ProcId.", VcPathProcess));
            }
         }
         else
         {
            ETG_TRACE_FATAL(("dp_tclDpBase(): read process name failed --> use ProcId."));
         }
         close(ViFileHandle);
      }
      else
      {
         ETG_TRACE_FATAL(("dp_tclDpBase():cannot open '%s'.", VcPathProcess));
      }
   }
   //create resources for this process
   //get resource name from process name max 32 byte
   strncpy(_szMyProcResourceName, _szMyProcSpecName, DP_MAX_LEN_NAME_RESOURCE - 1); //last byte should be zero
   if (OSAL_OK != OSAL_s32SemaphoreCreate(_szMyProcResourceName, &_hDpAccess, (tU32)1))
   {
      ETG_TRACE_FATAL(("dp_tclDpBase():  %x error while creating semaphore '%s'", OSAL_u32ErrorCode(), _szMyProcResourceName));
      if (OSAL_u32ErrorCode() == OSAL_E_ALREADYEXISTS)
      {
         /*If the process with the same name is created again after exit*/
         if (OSAL_OK != OSAL_s32SemaphoreOpen(_szMyProcResourceName, &_hDpAccess))
         {
            ETG_TRACE_FATAL(("dp_tclDpBase():  %x Error while Opening semaphore for '%s'", OSAL_u32ErrorCode(), _szMyProcResourceName));
            NORMAL_M_ASSERT_ALWAYS();
         }
         else
            NORMAL_M_ASSERT_ALWAYS();
      }
   }
   //create this semaphore once in the system
   if (OSAL_OK != OSAL_s32SemaphoreOpen("DpSemSrvIf", &_hDpSemSrvIf))
   { //create semaphore now
      if (OSAL_OK != OSAL_s32SemaphoreCreate("DpSemSrvIf", &_hDpSemSrvIf, (tU32)1))
      {
         ETG_TRACE_FATAL(("dp_tclDpBase(): error while creating semaphore 'DpSemSrvIf'"));
         NORMAL_M_ASSERT_ALWAYS();
      }
      else
      {
         ETG_TRACE_FATAL(("dp_tclDpBase(): dp master process %s", _szMyProcResourceName));
      }
   }
   // create my message queue
   if (OSAL_ERROR == OSAL_s32MessageQueueCreate(_szMyProcResourceName, DP_MAILBOX_MAX_MESSAGE_COUNT, sizeof(dp_tclDatapoolMaster::TDpPoolMsg), OSAL_EN_READWRITE, &_hMessageQueueHandle))
   {
      ETG_TRACE_FATAL(("dp_tclDpBase(): create message queue fails  with  %x for process '%s' ", OSAL_u32ErrorCode(), _szMyProcResourceName));
      if (OSAL_u32ErrorCode() == OSAL_E_ALREADYEXISTS)
      {
         /*If the process with the same name is created again after exit*/
         if (OSAL_ERROR == OSAL_s32MessageQueueOpen(_szMyProcResourceName, OSAL_EN_READWRITE, &_hMessageQueueHandle))
         {
            _hMessageQueueHandle = OSAL_C_INVALID_HANDLE;
            ETG_TRACE_FATAL(("dp_tclDpBase(): Open message queue fails with %x for process '%s' ", OSAL_u32ErrorCode(), _szMyProcResourceName));
         }
      }
   }
   //create event
   if (OSAL_s32EventCreate(_szMyProcResourceName, &_hEvSync) == OSAL_ERROR)
   {
      ETG_TRACE_FATAL(("dp_tclDpBase(): create event fails with %x for process '%s'", OSAL_u32ErrorCode(), _szMyProcResourceName));
      if (OSAL_u32ErrorCode() == OSAL_E_ALREADYEXISTS)
      {
         /*If the process with the same name is created again after exit*/
         if (OSAL_ERROR == OSAL_s32EventOpen(_szMyProcResourceName, &_hEvSync))
         {
            ETG_TRACE_FATAL(("dp_tclDpBase(): Open event fails with %x for process '%s'", OSAL_u32ErrorCode(), _szMyProcResourceName));
            NORMAL_M_ASSERT_ALWAYS();
         }
      }
      else
         NORMAL_M_ASSERT_ALWAYS();
   }
   //create thread
   OSAL_trThreadAttribute rAttr;
   rAttr.szName = _szMyProcResourceName;
   rAttr.s32StackSize = 10000;
   rAttr.u32Priority = 0;//NCG3D-103823: Modifies RT prio to nice level.
   rAttr.pfEntry = (OSAL_tpfThreadEntry)vDpMainThread;
   rAttr.pvArg = (tPVoid) this;
   _hThreadId = OSAL_ThreadSpawn(&rAttr);
   if (_hThreadId == OSAL_ERROR)
   {
      ETG_TRACE_FATAL(("dp_tclDpBase(): create thread fails with %x for process '%s' ", OSAL_u32ErrorCode(), _szMyProcResourceName));
      NORMAL_M_ASSERT_ALWAYS();
   }
   else 
   {
       tS32 s32ReturnValue = OSAL_s32ThreadPriority(_hThreadId,0);//NCG3D-103823: Modifies RT prio to nice level.
       if(s32ReturnValue != OSAL_OK)
       {
           ETG_TRACE_FATAL(("dp_tclDpBase(): change threadpriority failed %d '%s' ", OSAL_u32ErrorCode(), _szMyProcResourceName));
       }
       else
       {
           ETG_TRACE_USR4(("dp_tclDpBase(): change threadpriority success '%s' ", _szMyProcResourceName));
       }
   }
#ifndef DP_FEATURE_UNDEF_ELEMENT
   ETG_TRACE_USR1(("dp_tclDpBase(): Clear undef Pool LIst "));
   pDpUndefClearPoolList.clear();
#endif
}

dp_tclDpBase::~dp_tclDpBase()
{
   ETG_TRACE_USR4(("dp_tclDpBase(): destructor process name '%s'", _szMyProcSpecName));

   _poPersMem = NULL;

   if (_hDpAccess != OSAL_C_INVALID_HANDLE)
   {
      OSAL_s32SemaphoreClose(_hDpAccess);
      OSAL_s32SemaphoreDelete(_szMyProcResourceName);
   }
   if (_hDpSemSrvIf != OSAL_C_INVALID_HANDLE)
   {
      OSAL_s32SemaphoreClose(_hDpSemSrvIf);
   }
   if (_hMessageQueueHandle != OSAL_C_INVALID_HANDLE)
   {
      OSAL_s32MessageQueueClose(_hMessageQueueHandle);
      OSAL_s32MessageQueueDelete(_szMyProcResourceName);
      _hMessageQueueHandle = OSAL_C_INVALID_HANDLE;
   }
   if (_hMasterQueueHandle != OSAL_C_INVALID_HANDLE)
   {
      OSAL_s32MessageQueueClose(_hMasterQueueHandle);
      _hMasterQueueHandle = OSAL_C_INVALID_HANDLE;
   }
   if (_hEvSync != OSAL_C_INVALID_HANDLE)
   {
      OSAL_s32EventClose(_hEvSync);
      (void)OSAL_s32EventDelete(_szMyProcResourceName);
   }
}

tVoid dp_tclDpBase::vInitInstance(dp_tclDpPersMemAccess* poPersMem)
{
   _poPersMem = poPersMem;

   DP_NULL_POINTER_CHECK(_poPersMem);
   /*trace */
   ETG_TRACE_USR4(("vInitInstance(): function call by process '%s' ", _szMyProcSpecName));
   /*register master*/
   bRegisterMaster();
   /*register exit function*/
   if (s32RegisterExitFunction(DP_vDestroyDatapool, 0) != OSAL_OK)
   {/*trace */
      ETG_TRACE_FATAL(("vInitInstance(): error s32RegisterExitFunction() fails !"));
   }
   /*set trace configuration*/
   //abr: 26.05.2015: move to the access functions; 
   //this should not be done in the attach function, because file access is possible and the partion not mounted.
   //vSetTraceConfiguration();
}

#ifdef DP_DATAPOOL_UNIT_TEST
tVoid dp_tclDpBase::vResetUnitTest(char * PcSimProcessName)
{
   oDatapool.clear();
#ifdef DP_FEATURE_UNDEF_ELEMENT
   oDpUndefPoolProperty.clear();
#endif
   oHistoryList.clear();
   _tListSubPools.clear();
   _bSubListLoad = FALSE;
   _bListTraceElementsLoad = FALSE;
   _oListTraceElements.clear();
   OSAL_pvMemorySet(_szMyProcSpecName, 0, sizeof(_szMyProcSpecName));
   snprintf(_szMyProcSpecName, sizeof(_szMyProcSpecName), "%s", PcSimProcessName);
   OSAL_pvMemorySet(_szMyProcResourceName, 0, sizeof(_szMyProcResourceName));
   snprintf(_szMyProcResourceName, sizeof(_szMyProcResourceName) - 1, "%s", PcSimProcessName);
}
#endif

tS32 dp_tclDpBase::s32CheckPoolInit(tU32 u32PoolId)
{
   tS32 s32Ret = DP_S32_ERR_NO_ELEMENT;
   (tVoid)DP_bEnterCritical(_hDpAccess);
   TDatapool::iterator tDpPos = oDatapool.find(u32PoolId);
   //try to find pool
   if (tDpPos == oDatapool.end())
   {
      tBool VbResult = bInitPool(u32PoolId);
      /*check if pool exist!!!*/
      tDpPos = oDatapool.find(u32PoolId);
      if ((tDpPos == oDatapool.end()) || (VbResult == FALSE))
      { //error
         ETG_TRACE_ERR(("s32CheckPoolInit(): error pool '%08x' not created, VbResult: %d", u32PoolId, VbResult));
         s32Ret = DP_S32_ERR_NO_POOL;
      }
      else
      {
         ETG_TRACE_USR4(("s32CheckPoolInit(): pool '%08x' state bElemChanged '%d' ", u32PoolId, oDatapool[u32PoolId].bElemChanged));
      }
   }
#ifdef DP_U32_POOL_ID_DPENDUSERMODE
   //check for access only for eDpModeEndUserNo pools
   TDatapool::iterator VtDpPos = oDatapool.find(u32PoolId);
   if (VtDpPos != oDatapool.end())
   { /*if end user pool */
      if (oDatapool[u32PoolId].eModeEndUser != eDpModeEndUserNo)
      { /*if pool not locked for end user */
         if (_u32LockMode != DP_U32_LOCK_MODE_END_USER)
         { /*check if the pool is valid for the current end user => store old end user pool if changed*/
            s32Ret = s32CheckPoolInitEndUser(u32PoolId);
         }
         else
         {
            ETG_TRACE_USR4(("s32CheckPoolInit(): Pool %s is currently locked for end user pools", oDatapool[u32PoolId].strFileName));
            s32Ret = DP_S32_ERR_ACCESS_POOL_LOCKED;
         }
      }
   }
#endif
   DP_bReleaseCritical(_hDpAccess);
   return(s32Ret);
}

tVoid dp_tclDpBase::vCheckSubPoolInit(tBool VbDefSetAction)
{
   TDpPoolSubPoolList::iterator pDpSub;
#ifdef DP_FEATURE_UNDEF_ELEMENT
   tBool                        VbInitPool = FALSE;
#endif
   TDpPoolSubPoolClearList _tSubPoolClearList;
   _tSubPoolClearList.clear();

   /*check subpool list, if all changed pool loaded to process*/
   (tVoid)DP_bEnterCritical(_hDpAccess);
   /*load sub pool list, if not already loaded */
   vImportSubPoolList();
   /*load subpools to process if not load right now*/
   for (pDpSub = _tListSubPools.begin(); pDpSub != _tListSubPools.end(); ++pDpSub)
   {
      if (!pDpSub->bInitilazied)
      {
#ifdef DP_FEATURE_UNDEF_ELEMENT
         VbInitPool = TRUE;
         //for pool "undef element" check if pool property=> load to process
         if (((pDpSub->u32SubPool >> 16) >= CCA_C_U16_APP_UNDEFINED_BEGIN) && ((pDpSub->u32SubPool >> 16) <= CCA_C_U16_APP_UNDEFINED_END))
         { //pool undef element
            if (VbDefSetAction == TRUE)
            {/*defset for undef element;check if pool property=> load to process*/
               VbInitPool = bPerformDefSetUndefPool(pDpSub->u32SubPool);
            }
            else
            {/* bank action not possible for undef element */
               VbInitPool = FALSE;
            }
         }
         if (VbInitPool == TRUE)
#else
         OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(VbDefSetAction);
#endif
         {  //pool not load right now -> init now
            ETG_TRACE_USR1(("vCheckSubPoolInit():Init pool %08x now!!", pDpSub->u32SubPool));
            if (bInitPool(pDpSub->u32SubPool) == FALSE)
            {
               _tSubPoolClearList.push_back(pDpSub->u32SubPool); // Undefined Pools Exist
               ETG_TRACE_USR4(("vCheckSubPoolInit(): Undefined pool '%08x'!!!!!", pDpSub->u32SubPool));
            }
            //check if pool init/ available
            TDatapool::iterator VposDp = oDatapool.find(pDpSub->u32SubPool);
            if (VposDp != oDatapool.end())
            {
               ETG_TRACE_USR4(("vCheckSubPoolInit(): pool '%08x' state bElemChanged '%d' ", pDpSub->u32SubPool, oDatapool[pDpSub->u32SubPool].bElemChanged));
            }
         }
      }
#ifdef DP_U32_POOL_ID_DPENDUSERMODE
      else
      { //check for access
         TDatapool::iterator VtDpPos = oDatapool.find(pDpSub->u32SubPool);
         if (VtDpPos != oDatapool.end())
         { /*if end user pool */
            if (oDatapool[pDpSub->u32SubPool].eModeEndUser != eDpModeEndUserNo)
            { /*check if the load pool is valid for the current end user => store old end user pool if changed*/
               s32CheckPoolInitEndUser(pDpSub->u32SubPool);
            }
         }
      }
#endif
   }
   if (_tSubPoolClearList.size() != 0)
   {
      TDpPoolSubPoolClearList::iterator pClearSubpool;
      for (pClearSubpool = _tSubPoolClearList.begin(); pClearSubpool != _tSubPoolClearList.end(); ++pClearSubpool)
      {
         ETG_TRACE_USR1(("vCheckSubPoolInit():Clear pool %08x now/(%d)!!", *pClearSubpool, _tSubPoolClearList.size()));
         /*remove the files corresponding to this pool if exists
          & remove the entry from undefPool */
         vRemovePoolData(*pClearSubpool);
         //remove the entry from dp_processname file
         vRemoveAndExportSubPoolList(*pClearSubpool);
      }
   }
   DP_bReleaseCritical(_hDpAccess);
}

#ifdef DP_U32_POOL_ID_DPENDUSERMODE
tS32 dp_tclDpBase::s32CheckPoolInitEndUser(tU32 u32PoolId)
{
   tS32 s32Ret = DP_S32_ERR_NO_ELEMENT;
   /* check end user changed */
   dp_tclDpEndUser*    VpoMyDpEndUser = dp_tclDpEndUser::pGetInstance();
   if (VpoMyDpEndUser)
   {
      tU8 Vu8CurrentUser;
      if (VpoMyDpEndUser->s32GetEndUser(Vu8CurrentUser) == DP_S32_NO_ERR)
      {
         ETG_TRACE_USR4(("s32CheckPoolInitEndUser(): current pool '%08x' for end user '%d' ", u32PoolId, oDatapool[u32PoolId].u8EndUser));
         if ((oDatapool[u32PoolId].u8EndUser) != Vu8CurrentUser)
         { /* first store pool */
            ETG_TRACE_USR4(("s32CheckPoolInitEndUser(): pool '%08x' state bElemChanged '%d' ", u32PoolId, oDatapool[u32PoolId].bElemChanged));
            if (oDatapool[u32PoolId].bElemChanged == TRUE)
            {
               ETG_TRACE_USR4(("s32CheckPoolInitEndUser(): new end user first save old pool '%08x' '%s' ", u32PoolId, oDatapool[u32PoolId].strFileName));
               bExportPersData(u32PoolId, oDatapool[u32PoolId].strFileName, oDatapool[u32PoolId].eLocation, oDatapool[u32PoolId].u32PoolVersion, TRUE);
               oDatapool[u32PoolId].u32LastStoreTime = OSAL_ClockGetElapsedTime();;
               oDatapool[u32PoolId].bWaiting4Store = FALSE;
               oDatapool[u32PoolId].u32NextStoreTime = DP_U8_NOTHING_TO_STORE_TIME;
               oDatapool[u32PoolId].bDelayedElementActive = FALSE;
            }
            /*init new pool*/
            oDatapool.erase(u32PoolId);
            bInitPool(u32PoolId);
            //check if pool init/ available
            TDatapool::iterator VposDp = oDatapool.find(u32PoolId);
            if (VposDp != oDatapool.end())
            {
               ETG_TRACE_USR4(("s32CheckPoolInitEndUser(): new end user reload pool '%08x' '%s' ", u32PoolId, oDatapool[u32PoolId].strFileName));
               ETG_TRACE_USR4(("s32CheckPoolInitEndUser(): pool '%08x' state bElemChanged '%d' ", u32PoolId, oDatapool[u32PoolId].bElemChanged));
            }
         }
      }
      else
      {
         ETG_TRACE_USR4(("s32CheckPoolInitEndUser(): s32GetEndUser() fails (pool '%08x') ", u32PoolId));
         s32Ret = DP_S32_ERR_UNKNOWN;
      }
   }
   else
   {
      ETG_TRACE_USR4(("s32CheckPoolInitEndUser(): pGetInstance() fails (pool '%08x') ", u32PoolId));
      s32Ret = DP_S32_ERR_UNKNOWN;
   }
   return(s32Ret);
}
#endif

tVoid dp_tclDpBase::vInitDp(tU32 u32PoolId, const tChar* cPoolName, tU32 u32PoolVersion, eDpLocation PeLocation, tU8 u8Bank, eDpModeEndUser PeModeEndUser, tBool bDefset)
{
   //try to find in pool
   //(tVoid) DP_bEnterCritical(_hDpAccess);
   TDatapool::iterator tDpPos = oDatapool.find(u32PoolId);
   if (tDpPos == oDatapool.end()) {
      //pool not found -> create now
      TDpPoolProperty oNewList;

      oNewList.u32NextStoreTime = DP_U8_NOTHING_TO_STORE_TIME;
      oNewList.u32LastStoreTime = OSAL_ClockGetElapsedTime();
      oNewList.bWaiting4Store = FALSE;
      oNewList.bDelayedElementActive = FALSE;
      oNewList.bDefSet = bDefset;
      oNewList.bElemChanged = FALSE;
      oNewList.eLocation = PeLocation;
      oNewList.u32PoolVersion = u32PoolVersion;
      oNewList.eModeEndUser = PeModeEndUser;
      oNewList.u8EndUser = 0;
      oNewList.u32Handle = 0;
      //strFileName
      memset(oNewList.strFileName, 0, DP_MAX_LEN_FILE);
      DP_NULL_POINTER_CHECK(_poPersMem);
      _poPersMem->s32GetDataStreamName((tU16)(u32PoolId >> 16), cPoolName, oNewList.strFileName);
      //if end user pool get current end user and add end user file extansion
#ifdef DP_U32_POOL_ID_DPENDUSERMODE
      if (PeModeEndUser != eDpModeEndUserNo)
      {
         dp_tclDpEndUser*    VpoMyDpEndUser = dp_tclDpEndUser::pGetInstance();
         if (VpoMyDpEndUser)
         {
            VpoMyDpEndUser->s32GetEndUser(oNewList.u8EndUser);
            ETG_TRACE_USR4(("vInitDp(): oNewList.u8EndUser '%d' ", oNewList.u8EndUser));
            VpoMyDpEndUser->vAddCurrentEndUserExt(oNewList.strFileName);
            ETG_TRACE_USR4(("vInitDp(): strFileName '%s' ", oNewList.strFileName));
         }
      }
#else
      OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(u8Bank);
#endif

      oDatapool[u32PoolId] = oNewList;
      tDpPos = oDatapool.find(u32PoolId);

      TDpPoolSubPoolList::iterator pDpSub;
      tBool bFound = FALSE;
      for (pDpSub = _tListSubPools.begin(); (pDpSub != _tListSubPools.end()) && !bFound; ++pDpSub)
      {
         if (pDpSub->u32SubPool == u32PoolId)
         { /*subpool found set init state*/
            pDpSub->bInitilazied = TRUE;
            bFound = TRUE;
         }
      }
      if (!bFound)
      {  /*subpool not found, add subpool to list*/
         TDpListSubPool tListElement = { TRUE, FALSE, u32PoolId };
         _tListSubPools.push_back(tListElement);
      }
   }
   //DP_bReleaseCritical(_hDpAccess);
   //if bank active read from bank
#ifdef DP_U32_POOL_ID_DPENDUSERMODE
   if ((u8Bank != DP_U8_NO_BANK_ACTION) && (PeModeEndUser == eDpModeEndUserMultiBank))
   {
      dp_tclDpEndUser*    VpoMyDpEndUser = dp_tclDpEndUser::pGetInstance();
      if (VpoMyDpEndUser)
      {
         tChar  VstrFileNameBank[DP_MAX_LEN_FILE] = { 0 };
         strncpy(VstrFileNameBank, oDatapool[u32PoolId].strFileName, DP_MAX_LEN_FILE);
         VstrFileNameBank[DP_MAX_LEN_FILE - 1] = '\0';
         VpoMyDpEndUser->vAddEndUserBankExt(u8Bank, VstrFileNameBank);
         ETG_TRACE_USR4(("vInitDp(): VstrFileNameBank '%s' ", VstrFileNameBank));
         vImportPersData(u32PoolId, VstrFileNameBank, oDatapool[u32PoolId].eLocation, oDatapool[u32PoolId].u32PoolVersion);
      }
   }
   else
#endif
   { /*get persistent data from ,location*/
      vImportPersData(u32PoolId, oDatapool[u32PoolId].strFileName, oDatapool[u32PoolId].eLocation, oDatapool[u32PoolId].u32PoolVersion);
   }
}

tBool dp_tclDpBase::bRegisterMaster()
{
   ETG_TRACE_USR4(("bRegisterMaster(): Register to master."));
   //first try to open queue of DP master
   tS32 s32Ret = OSAL_s32MessageQueueOpen(DP_STR_MASTER_NAME, OSAL_EN_READWRITE, &_hMasterQueueHandle);
   if (s32Ret != OSAL_OK)
   {  //failed to open queue
      ETG_TRACE_FATAL(("bRegisterMaster(): failed to open master queue."));
      return FALSE;
   }
   // now register client at master
   dp_tclDatapoolMaster::TDpPoolMsg oMsg = { dp_tclDatapoolMaster::eRegisterClient, { 0 } };
   OSAL_szStringCopy(oMsg.u.tRegMaster.strProzess, _szMyProcSpecName);
   oMsg.u.tRegMaster.tPid = OSAL_ProcessWhoAmI();
   if (!bPostMasterMessage(&oMsg))
   {
      ETG_TRACE_FATAL(("bRegisterMaster(): failed to post message to master"));
   }
   return TRUE;
}

tBool dp_tclDpBase::bUnregisterMaster()
{
   tBool VbReturn = TRUE;
   ETG_TRACE_USR4(("bUnregisterMaster()"));
   //unregister client 
   dp_tclDatapoolMaster::TDpPoolMsg oMsg = { dp_tclDatapoolMaster::eUnregisterClient, { 0 } };
   OSAL_szStringCopy(oMsg.u.strData, _szMyProcSpecName);
   if (!bPostMasterMessage(&oMsg))
   {
      VbReturn = FALSE;
      ETG_TRACE_FATAL(("bUnregisterMaster(): failed to post message to master"));
   }
   return(VbReturn);
}

tBool dp_tclDpBase::bStopThreadWait()
{
   tBool VbReturn = TRUE;
   ETG_TRACE_USR4(("bStopThreadWait()"));
   //delete old existing events
   vWaitDeleteExistingEvents(DP_U32_EVENT_STOP_WAIT_THREAD_DONE);
   //send message to own thread to terminate wait
   dp_tclDatapoolMaster::TDpPoolMsg oMsg = { dp_tclDatapoolMaster::eStopWaitThread, { 0 } };
   if (!bPostOwnMessage(&oMsg))
   {
      VbReturn = FALSE;
      ETG_TRACE_FATAL(("bStopThreadWait(): failed to post message to master"));
   }
   else
   {//wait until done     
      tS32 Vs32Return = OSAL_s32ThreadJoin(_hThreadId, OSAL_C_TIMEOUT_FOREVER);
      //tS32 Vs32Return=s32WaitSync(DP_U32_EVENT_STOP_WAIT_THREAD_DONE);
      if (Vs32Return != OSAL_OK)
      {
         VbReturn = FALSE;
         ETG_TRACE_FATAL(("bStopThreadWait(): failed %d", Vs32Return));
      }
      else
      {
         ETG_TRACE_USR4(("bStopThreadWait(): success"));
      }
   }
   return(VbReturn);
}

// ****************************************************************************
// ********   element handling 
// ****************************************************************************

tVoid dp_tclDpBase::vInitElement(tU32 u32PoolId, dp_tclBaseElement* pDpElement, dp_tclBaseElement::EDpElemState eElemState)
{
   //try to find in pool
   TDatapool::iterator tDpPos = oDatapool.find(u32PoolId);
   if (tDpPos == oDatapool.end())
   {
      ETG_TRACE_ERR(("vInitElement(): error -> pool must be initialized first"));
      return;
   }
   //check if element in pool 
   TListDatapoolElements::iterator posElement = oDatapool[u32PoolId].tDpElementList.find(pDpElement->s32GetHash());
   if (posElement == oDatapool[u32PoolId].tDpElementList.end())
   { //element not found in pool -> create now
      if (bIsTracingEnabled(pDpElement)) ETG_TRACE_USR1(("vInitElement(): element '%s' not found in pool -> create now.", pDpElement->pcGetElementName()));
      //add element
      oDatapool[u32PoolId].tDpElementList[pDpElement->s32GetHash()] = *pDpElement;
      /* for v850,kernel pools and multi_bank pools, default data shall be saved. Therfore validate this element.*/
      /* reason: */
      /*  1) V850 and kernel pools: read from other location */
      /*  2) multi_bank pools:      the default value of an bank has to be stored to the current user at shutdown */
      if ((oDatapool[u32PoolId].eLocation == eDpLocation_V850) || (oDatapool[u32PoolId].eLocation == eDpLocation_PDD_KERNEL)
         || (oDatapool[u32PoolId].eModeEndUser == eDpModeEndUserMultiBank))
      {
         oDatapool[u32PoolId].tDpElementList[pDpElement->s32GetHash()].vValidateElement();
      }
      vTraceElement(pDpElement, eDpElemInit);
   }
   else
   {//element in pool => check version and datalength -> if compatible
      if (posElement->second.u8GetVersion() != pDpElement->u8GetVersion())
      { //version different
         if (DP_S32_ERR_NO_CONVERSION == pDpElement->s32ConvertElement(&posElement->second))
         {
            if (bIsTracingEnabled(pDpElement)) ETG_TRACE_USR1(("vInitElement(): version (%d -> %d) of element '%s' has changed -> set to init.", posElement->second.u8GetVersion(), pDpElement->u8GetVersion(), pDpElement->pcGetElementName()));
            oDatapool[u32PoolId].tDpElementList[pDpElement->s32GetHash()] = *pDpElement;
            vTraceElement(pDpElement, eDpElemVersionChange);
         }
         else
         { //here we have to check if it is possible to restore old stored data
            if (bIsTracingEnabled(pDpElement)) ETG_TRACE_USR1(("vInitElement(): Special convert function for version change (%d -> %d) is set for element '%s'.", posElement->second.u8GetVersion(), pDpElement->u8GetVersion(), pDpElement->pcGetElementName()));
            posElement->second.vFillElementProperty(pDpElement);
            vTraceElement(pDpElement, eDpElemVersionConversion);
         }
      }
      else
      {  //element was loaded from FFS -> merge now element properties
         if (bIsTracingEnabled(pDpElement)) ETG_TRACE_USR1(("vInitElement(): element '%s' was loaded from FFS -> merge now element properties.", pDpElement->pcGetElementName()));
         posElement->second.vFillElementProperty(pDpElement);
         vTraceElement(&posElement->second, eDpElemLoad);
         eElemState = dp_tclBaseElement::eDpElemSet; // value read from file => is set and can read
      }
   }
   //and now set default data
   oDatapool[u32PoolId].tDpElementList[pDpElement->s32GetHash()].vSetElementState(eElemState);
   oDatapool[u32PoolId].tDpElementList[pDpElement->s32GetHash()].bSetDefaultData(pDpElement->pvGetDataDefault(), pDpElement->u32GetDataLengthDefault());
   return;
}

tVoid dp_tclDpBase::vInitUnknownCodingElements(tU32 u32PoolId)
{
   tU16 u16SubpoolId = (u32PoolId & 0x0000ffff);
   if (u16SubpoolId >= DP_U16_SUBPOOL_CODING_RANGE)
   {  //try to find in pool  (function calls from vInitXXX XXX=Poolname function is protected)
      TDatapool::iterator posDp = oDatapool.find(u32PoolId);
      if (posDp != oDatapool.end())
      {
         TListDatapoolElements::iterator posElement;
         for (posElement = posDp->second.tDpElementList.begin(); posElement != posDp->second.tDpElementList.end(); ++posElement)
         {
            if (posElement->second.eGetElementState() == dp_tclBaseElement::eDpElemLoaded)
            {
               posElement->second.vChangeProperties(TRUE, 0x0446, dp_tclBaseElement::eDpElementTypePersistent, dp_tclBaseElement::eDpStoringTypeShutdown, DP_TYPE_STRUCT, 0);
            }
         }
      }
   }
   return;
}

#ifdef DP_FEATURE_UNDEF_ELEMENT
tVoid dp_tclDpBase::vInitUndefElements(tU32 u32PoolId)
{//try to find in pool
   TDatapool::iterator posDp = oDatapool.find(u32PoolId);
   ETG_TRACE_USR4(("vInitUndefElements(): pool id: '%x' ", u32PoolId));
   if (posDp != oDatapool.end())
   {
      TListDatapoolElements::iterator posElement;
      /*for all elements in pool*/
      for (posElement = posDp->second.tDpElementList.begin(); posElement != posDp->second.tDpElementList.end(); ++posElement)
      { /*if element loaded from file, but not set*/
         if (posElement->second.eGetElementState() == dp_tclBaseElement::eDpElemLoaded)
         { /*set to default properties:         bVarSize, Owner read/write, eDpElementTypePersistent,      TypeImmediately,                              bytes ,     DP_DEFSET_USER*/
            posElement->second.vChangeProperties(DP_UNDEF_ELEMENT_DEFAULT_PROPERTY_VAR_SIZE, DP_UNDEF_ELEMENT_DEFAULT_PROPERTY_ACCESSS_TYPE,
               DP_UNDEF_ELEMENT_DEFAULT_PROPERTY_ELEMENT_TYPE, DP_UNDEF_ELEMENT_DEFAULT_PROPERTY_STORING_TYPE,
               DP_TYPE_U8, DP_UNDEF_ELEMENT_DEFAULT_PROPERTY_DEF_SET_MODE);
            posElement->second.vSetVersion(DP_UNDEF_ELEMENT_DEFAULT_PROPERTY_VERSION);
            /*set default data with correct lenght*/
            tU32 Vu32Lenght = posElement->second.u32GetDataLength();
            tU8* Vu8DefaultDataBuffer = NULL;
            try
            {
               Vu8DefaultDataBuffer = new tU8[Vu32Lenght];
            }
            catch (const std::bad_alloc&)
            {
               Vu8DefaultDataBuffer = NULL;
               ETG_TRACE_ERR(("vInitUndefElements(): unable to allocate '%d' for PoolId ", u32PoolId));
            }
            if (Vu8DefaultDataBuffer)
            {
               memset(Vu8DefaultDataBuffer, 0, Vu32Lenght);
               posElement->second.bSetDefaultData(Vu8DefaultDataBuffer, Vu32Lenght);
               delete[] Vu8DefaultDataBuffer;
               Vu8DefaultDataBuffer = NULL;/*Bug 123144: Safe handling*/

               ETG_TRACE_USR4(("vInitUndefElements(): init element '%s' ", posElement->second.pcGetElementName()));
               ETG_TRACE_USR1(("vInitUndefElements(): element lenght '%d' ", Vu32Lenght));
            }
         }
         else
         {
            ETG_TRACE_ERR(("vInitUndefElements(): element state not loaded"));
         }
      }
   }
   else
   {
      ETG_TRACE_ERR(("vInitUndefElements(): pool '%d' not found", u32PoolId));
   }
}

tS32 dp_tclDpBase::s32InitUndefElement(const tChar* PpcUserName, const tChar* PpcPoolName, eDpLocation PpeLocation, dp_tclBaseElement* PpDpElement)
{
   tS32                    Vs32ReturnCode = DP_S32_NO_ERR;
   /*check parameter*/
   if ((*PpDpElement->pcGetElementName() == 0)
      || ((PpDpElement->u32GetDefSetMode() != DP_DEFSET_USER) && ((PpDpElement->u32GetDefSetMode() != 0))) //only default user possible
      )
   {
      Vs32ReturnCode = DP_S32_ERR_WRONG_PARAM;
   }
   else
   {
      tDpUndefPoolProperty    VtPoolProperty;
      dp_tclDpUndef*          VpoMyDpUndef = dp_tclDpUndef::pGetInstance();

      ETG_TRACE_USR4(("s32InitUndefElement(): init element '%s' ", PpDpElement->pcGetElementName()));
      memset(&VtPoolProperty, 0, sizeof(tDpUndefPoolProperty));
      strncpy(VtPoolProperty.strPoolName, PpcPoolName, DP_MAX_LEN_NAME_UNDEFINED_POOLS);
      VtPoolProperty.strPoolName[DP_MAX_LEN_NAME_UNDEFINED_POOLS - 1] = '\0';
      strncpy(VtPoolProperty.strUserName, PpcUserName, DP_MAX_LEN_NAME_USER);
      VtPoolProperty.strUserName[DP_MAX_LEN_NAME_USER - 1] = '\0';
      strncpy(VtPoolProperty.strProcessNameLocated, _szMyProcSpecName, DP_MAX_LEN_NAME_PROCESS);
      VtPoolProperty.strProcessNameLocated[DP_MAX_LEN_NAME_PROCESS - 1] = '\0';
      VtPoolProperty.eLocation = PpeLocation;
      /* get pool properties */
      Vs32ReturnCode = VpoMyDpUndef->s32GetPoolProperty(&VtPoolProperty, dp_tclDpUndef::eDpUndefInit, PpDpElement);
      /*check if location the same*/
      if (VtPoolProperty.eLocation != PpeLocation)
      {
         Vs32ReturnCode = DP_S32_ERR_DIFFERENT_LOCATION;
         ETG_TRACE_ERR(("s32InitUndefElement(): error -> different location for the same pool: %s", VtPoolProperty.strPoolName));
      }
      if (Vs32ReturnCode == DP_S32_NO_ERR)
      { /* pool should located in actual process*/
         if (strncmp(VtPoolProperty.strProcessNameLocated, _szMyProcSpecName, DP_MAX_LEN_NAME_PROCESS) == 0)
         {
            (tVoid)DP_bEnterCritical(_hDpAccess);
            /* check if pool exist in the process*/
            TDpUndefPoolProperty::iterator VtDpUndefPos = oDpUndefPoolProperty.find(VtPoolProperty.u32PoolId);
            if (VtDpUndefPos == oDpUndefPoolProperty.end())
            { /*add new property*/
               oDpUndefPoolProperty[VtPoolProperty.u32PoolId] = VtPoolProperty;
            }
            //check if pool init
            TDatapool::iterator VtDpPos = oDatapool.find(VtPoolProperty.u32PoolId);
            if (VtDpPos == oDatapool.end())
            {
               bInitPool(VtPoolProperty.u32PoolId);
            }
            //check for access
            VtDpPos = oDatapool.find(VtPoolProperty.u32PoolId);
            if (VtDpPos != oDatapool.end())
            {//check if element exist in the pool 
               TListDatapoolElements::iterator VposElement = oDatapool[VtPoolProperty.u32PoolId].tDpElementList.find(PpDpElement->s32GetHash());
               if (VposElement == oDatapool[VtPoolProperty.u32PoolId].tDpElementList.end())
               { /*element doesn't exist*/
                  ETG_TRACE_USR4(("s32InitUndefElement(): element %s not in pool", PpDpElement->pcGetElementName()));
                  PpDpElement->vValidateElement();
                  vInitElement(VtPoolProperty.u32PoolId, PpDpElement, dp_tclBaseElement::eDpElemInit);
                  /*try to find again*/
                  VposElement = oDatapool[VtPoolProperty.u32PoolId].tDpElementList.find(PpDpElement->s32GetHash());
               }
               else
               { /*check if the element name identical*/
                  if (strcmp(VposElement->second.pcGetElementName(), PpDpElement->pcGetElementName()) != 0)
                  {/*name are not identical, but have the same hash value*/
                     ETG_TRACE_FATAL(("s32InitUndefElement(): for element %s twice hash value exist in the same pool => please chaneg element name", PpDpElement->pcGetElementName()));
                     Vs32ReturnCode = DP_S32_ERR_EQUAL_HASH;
                  }
                  else
                  { /*change properties of element*/
                     ETG_TRACE_USR4(("s32InitUndefElement(): element %s in pool, change properties ", PpDpElement->pcGetElementName()));
                     VposElement->second.vChangeProperties(PpDpElement->bGetVarSize(), PpDpElement->u16GetAccessType(), PpDpElement->eGetElementType(),
                        PpDpElement->eGetStoringType(), DP_TYPE_U8, PpDpElement->u32GetDefSetMode());
                     VposElement->second.vSetVersion(PpDpElement->u8GetVersion());
                  }
               }
               if ((VposElement != oDatapool[VtPoolProperty.u32PoolId].tDpElementList.end()) && (Vs32ReturnCode == DP_S32_NO_ERR))
               {
                  tDpUndefElementProperty VtUndefElementProperty;
                  memset(&VtUndefElementProperty, 0, sizeof(tDpUndefElementProperty));
                  strncpy(VtUndefElementProperty.strElementName, PpDpElement->pcGetElementName(), DP_MAX_LEN_NAME_ELEMENT);
                  VtUndefElementProperty.strElementName[DP_MAX_LEN_NAME_ELEMENT - 1] = '\0';
                  VtUndefElementProperty.u8Version = VposElement->second.u8GetVersion();
                  VtUndefElementProperty.u32DataLength = VposElement->second.u32GetDataLength();
                  VtUndefElementProperty.bVarSize = VposElement->second.bGetVarSize();
                  VtUndefElementProperty.u16AccessType = VposElement->second.u16GetAccessType();
                  VtUndefElementProperty.tElemType = VposElement->second.eGetElementType();
                  VtUndefElementProperty.tStoringType = VposElement->second.eGetStoringType();
                  VtUndefElementProperty.u32StoringIntervall = VposElement->second.u32GetStoringIntervall();
                  VtUndefElementProperty.u32DefSetMode = VposElement->second.u32GetDefSetMode();
                  Vs32ReturnCode = VpoMyDpUndef->s32SaveElementProperty(VtPoolProperty.u32PoolId, &VtUndefElementProperty, PpeLocation);
               }
            }
            DP_bReleaseCritical(_hDpAccess);
         }
         else
         {
            Vs32ReturnCode = DP_S32_ERR_ACCESS_RIGHT;
         }
      }
   }
   return(Vs32ReturnCode);
}
#endif

tBool dp_tclDpBase::bSetElementToDefault(dp_tclBaseElement* pDpElement)
{
   (tVoid)DP_bEnterCritical(_hDpAccess);
   tBool bRet = pDpElement->bRestoreDefaultData();
   DP_bReleaseCritical(_hDpAccess);
   return bRet;
}

#ifdef DP_FEATURE_UNDEF_ELEMENT
tS32  dp_tclDpBase::s32SetUndefElement(const tChar* PpcUserName, const tChar* PpcPoolName, const tChar* PpcElementName, tU8* Ppu8WriteBuffer, tS32 Ps32Size)
{
   tS32                    Vs32ReturnCode = DP_S32_NO_ERR;
   tDpUndefPoolProperty    VtPoolProperty;
   dp_tclDpUndef*          VpoMyDpUndef = dp_tclDpUndef::pGetInstance();
   dp_tclBaseElement       VDpElem(PpcElementName, 0); //hash calc in constructor  

   ETG_TRACE_USR4(("s32SetUndefElement(): set element '%s' ", PpcElementName));
   memset(&VtPoolProperty, 0, sizeof(tDpUndefPoolProperty));
   strncpy(VtPoolProperty.strPoolName, PpcPoolName, DP_MAX_LEN_NAME_UNDEFINED_POOLS);
   VtPoolProperty.strPoolName[DP_MAX_LEN_NAME_UNDEFINED_POOLS - 1] = '\0';
   strncpy(VtPoolProperty.strUserName, PpcUserName, DP_MAX_LEN_NAME_USER);
   VtPoolProperty.strUserName[DP_MAX_LEN_NAME_USER - 1] = '\0';
   strncpy(VtPoolProperty.strProcessNameLocated, _szMyProcSpecName, DP_MAX_LEN_NAME_PROCESS);
   VtPoolProperty.strProcessNameLocated[DP_MAX_LEN_NAME_PROCESS - 1] = '\0';
   VtPoolProperty.eLocation = eDpLocation_FILE_SYSTEM; //default location
   /* get pool properties */
   Vs32ReturnCode = VpoMyDpUndef->s32GetPoolProperty(&VtPoolProperty, dp_tclDpUndef::eDpUndefSet, &VDpElem);
   if (Vs32ReturnCode == DP_S32_NO_ERR)
   { /* pool should located in actual process*/
      if (strncmp(VtPoolProperty.strProcessNameLocated, _szMyProcSpecName, DP_MAX_LEN_NAME_PROCESS) == 0)
      { /* check if pool exist in the process*/
         (tVoid)DP_bEnterCritical(_hDpAccess);
         TDpUndefPoolProperty::iterator VtDpPos = oDpUndefPoolProperty.find(VtPoolProperty.u32PoolId);
         if (VtDpPos == oDpUndefPoolProperty.end())
         { /*add new property*/
            oDpUndefPoolProperty[VtPoolProperty.u32PoolId] = VtPoolProperty;
         }
         DP_bReleaseCritical(_hDpAccess);
         (tVoid)VDpElem.bFillData((tVoid*)Ppu8WriteBuffer, Ps32Size);
         Vs32ReturnCode = s32SetElement(VtPoolProperty.u32PoolId, &VDpElem, (tU16)(VtPoolProperty.u32PoolId >> 16));
         if (Vs32ReturnCode == DP_S32_NO_ERR)
         {
            Vs32ReturnCode = Ps32Size;
         }
      }
      else
      {  /* pool not located in the actual process => set element should be set by the given located process*/
         Vs32ReturnCode = s32SharedUndefElementAccess(&VtPoolProperty, PpcElementName, FALSE, Ppu8WriteBuffer, Ps32Size);
      }
   }
   else
   {
      ETG_TRACE_ERR(("s32SetUndefElement(): error -> no pool property found for element '%s'.", PpcElementName));
   }
   return(Vs32ReturnCode);
}
#else
tVoid dp_tclDpBase::vLoadUndefClearPoolList()
{
   tU8*          Vpu8Buffer = NULL;
   tDpUndefPoolProperty *pDpUndefPoolProperty = NULL;
   tChar         VstrName[DP_MAX_LEN_FILE] = { 0 };
   tS32          Vs32ReadBytes;
   tU32          Vu32Count = 0, Vu32Index = 0;

   // Import data from File.
   if (_poPersMem != NULL)
   {
      OSAL_szStringNCopy((tString)VstrName, DP_UNDEF_FILE_NAME_POOL_PROPERTIES, DP_MAX_LEN_FILE);
      VstrName[DP_MAX_LEN_FILE - 1] = '\0';

      Vs32ReadBytes = _poPersMem->s32ImportPersData(VstrName, &Vpu8Buffer, eDpLocation_FILE_SYSTEM_SECURE, DP_UNDEF_VERSION_FILE_PROPERTIES);
      if (Vs32ReadBytes <= (tS32)DP_SIZE_HASH_TABLE)
      {
         ETG_TRACE_USR4(("vLoadUndefClearPoolList:: no pool property stored"));
      }
      else
      {
         pDpUndefPoolProperty = (tDpUndefPoolProperty *)(Vpu8Buffer + DP_SIZE_HASH_TABLE);

         ETG_TRACE_USR4(("vLoadUndefClearPoolList:: Vpu8Buffer = %x new pDpUndefPoolProperty = %x", Vpu8Buffer, pDpUndefPoolProperty));

         Vu32Count = (Vs32ReadBytes - (tU32)DP_SIZE_HASH_TABLE) / (tU32)(sizeof(tDpUndefPoolProperty));

         for (Vu32Index = 0; Vu32Index < Vu32Count; Vu32Index++)
         {
            ETG_TRACE_USR4(("vLoadUndefClearPoolList:: Vpu8Buffer = %x pDpUndefPoolProperty = %x(%x)", Vpu8Buffer + Vs32ReadBytes, pDpUndefPoolProperty,
               Vpu8Buffer + (Vu32Index*(sizeof(pDpUndefPoolProperty)))));

            if ((tU8*)pDpUndefPoolProperty < (tU8*)(Vpu8Buffer + Vs32ReadBytes))
            {
               tDpUndefClearPool pDpUndefPool;
               pDpUndefPool.u32PoolID = pDpUndefPoolProperty->u32PoolId;
               memset(pDpUndefPool.strPoolName, 0, DP_MAX_LEN_NAME_UNDEFINED_POOLS);
               strncpy(pDpUndefPool.strPoolName, pDpUndefPoolProperty->strPoolName, DP_MAX_LEN_NAME_UNDEFINED_POOLS - 1);
               ETG_TRACE_USR4(("vLoadUndefClearPoolList:: u32PoolID = %x strPoolName = %s", pDpUndefPool.u32PoolID, pDpUndefPool.strPoolName));
               pDpUndefClearPoolList.push_back(pDpUndefPool);
               pDpUndefPoolProperty++;
            }
         }
      }
   }
   //Check what will happen when the file does not exist.
}
#endif

tS32 dp_tclDpBase::s32SetElement(dp_tclBaseElement* pDpElement, tU16 u16AccessId)
{
   tS32 s32Ret = DP_S32_ERR_NO_ELEMENT;
   TDatapool::iterator posDp;
   for (posDp = oDatapool.begin(); (posDp != oDatapool.end()) && (s32Ret == DP_S32_ERR_NO_ELEMENT); ++posDp)
   {
      tU32 u32PoolId = posDp->first;
      s32Ret = s32SetElement(u32PoolId, pDpElement, u16AccessId);
   }
   return s32Ret;
}

tS32 dp_tclDpBase::s32SetElement(tU32 u32PoolId, dp_tclBaseElement* pDpElement, tU16 u16AccessId)
{
   tS32  s32Ret = DP_S32_ERR_NO_ELEMENT;
   tBool bOwner = ((tU32)(u16AccessId << (tU16)16) == (u32PoolId & (tU32)0xffff0000));
   tBool bCodingElem = FALSE;
   tBool bForceStore = FALSE;
#ifdef DP_FEATURE_UNDEF_ELEMENT
   tBool bUndefElement = FALSE;
#endif

   if (bIsTracingEnabled(pDpElement))
      ETG_TRACE_USR1(("bSetElement(): called for element (hash: 0x%08x) '%s' ", pDpElement->s32GetHash(), pDpElement->pcGetElementName()));

   //check for global/coding pool
   tU16 u16SubpoolId = (u32PoolId & 0x0000ffff);
   if (u16SubpoolId >= DP_U16_SUBPOOL_CODING_RANGE)
   {
      bCodingElem = TRUE;
   }
#ifdef DP_FEATURE_UNDEF_ELEMENT
   //check for undef element
   if ((u16AccessId >= CCA_C_U16_APP_UNDEFINED_BEGIN) && (u16AccessId <= CCA_C_U16_APP_UNDEFINED_END))
   {
      bUndefElement = TRUE;
   }
#endif
   if (bOwner)
   {//try to find pool
      s32Ret = s32CheckPoolInit(u32PoolId);
   }
   //trace element to write
   vTraceElement(pDpElement, eDpElemWrite);
   //check for access
   TDatapool::iterator tDpPos = oDatapool.find(u32PoolId);
   if ((tDpPos != oDatapool.end()) && (s32Ret == DP_S32_ERR_NO_ELEMENT))
   {
      tBool bDataChanged = TRUE;
      tBool bAccess = FALSE;
      (tVoid)DP_bEnterCritical(_hDpAccess);
      TListDatapoolElements::iterator posElement = oDatapool[u32PoolId].tDpElementList.find(pDpElement->s32GetHash());
      //for coding element, which are not defined in XML
      if ((posElement == oDatapool[u32PoolId].tDpElementList.end())
         && (bCodingElem == TRUE))
      {
         pDpElement->vChangeProperties(TRUE, 0x0446, dp_tclBaseElement::eDpElementTypePersistent, dp_tclBaseElement::eDpStoringTypeShutdown, DP_TYPE_STRUCT, 0);
         pDpElement->vValidateElement();
         vInitElement(u32PoolId, pDpElement, dp_tclBaseElement::eDpElemSet);
         bForceStore = TRUE;
         //try to find again
         posElement = oDatapool[u32PoolId].tDpElementList.find(pDpElement->s32GetHash());
      }
#ifdef DP_FEATURE_UNDEF_ELEMENT
      //for undefined element
      if ((posElement == oDatapool[u32PoolId].tDpElementList.end())
         && (bUndefElement == TRUE))
      {
         /*set to default properties: bVarSize, Owner read/write, eDpElementTypePersistent,      TypeImmediately,                       bytes ,     DP_DEFSET_USER*/
         pDpElement->vChangeProperties(DP_UNDEF_ELEMENT_DEFAULT_PROPERTY_VAR_SIZE, DP_UNDEF_ELEMENT_DEFAULT_PROPERTY_ACCESSS_TYPE,
            DP_UNDEF_ELEMENT_DEFAULT_PROPERTY_ELEMENT_TYPE, DP_UNDEF_ELEMENT_DEFAULT_PROPERTY_STORING_TYPE,
            DP_TYPE_U8, DP_UNDEF_ELEMENT_DEFAULT_PROPERTY_DEF_SET_MODE);
         pDpElement->vSetVersion(DP_UNDEF_ELEMENT_DEFAULT_PROPERTY_VERSION);
         pDpElement->vValidateElement();
         vInitElement(u32PoolId, pDpElement, dp_tclBaseElement::eDpElemSet);
         bForceStore = TRUE;
         //try to find again
         posElement = oDatapool[u32PoolId].tDpElementList.find(pDpElement->s32GetHash());
      }
      // check for undef element name identical
      if ((bUndefElement == TRUE) && (posElement != oDatapool[u32PoolId].tDpElementList.end()))
      {
         if (strcmp(posElement->second.pcGetElementName(), pDpElement->pcGetElementName()) != 0)
         {/*name are not identical, but have the same hash value*/
            ETG_TRACE_FATAL(("bSetElement(): for element %s twice hash value exist in the same pool => please chaneg element name", pDpElement->pcGetElementName()));
            s32Ret = DP_S32_ERR_EQUAL_HASH;
         }
      }
      //check access type
      if ((s32Ret != DP_S32_ERR_EQUAL_HASH) && (posElement != oDatapool[u32PoolId].tDpElementList.end()))
#else
      //check access type
      if (posElement != oDatapool[u32PoolId].tDpElementList.end())
#endif
      {
         if ((bOwner && (posElement->second.u16GetAccessType() & DP_U16_ACCESS_WR_OWNER_0002)) || (!bOwner && (posElement->second.u16GetAccessType() & DP_U16_ACCESS_WR_OTHER_0220)))
         {
            bAccess = TRUE;
         }
         else
         {
            ETG_TRACE_ERR(("bSetElement(): error -> no access right to change element '%s'.", pDpElement->pcGetElementName()));
            s32Ret = DP_S32_ERR_ACCESS_RIGHT;
         }
      }
      if (bAccess && (posElement != oDatapool[u32PoolId].tDpElementList.end()))
      {
         s32Ret = DP_S32_NO_ERR;
         //check if pool is locked
         switch (_u32LockMode)
         {
         case DP_U32_LOCK_MODE_SETDEF:
            if (posElement->second.u32GetDefSetMode() & DP_DEFSET_CHANGEABLE)
            {
               ETG_TRACE_USR1(("bSetElement(): Pool is currently locked because of DefSet, but element '%s' is configured as writeable while DefSet!", pDpElement->pcGetElementName()));
            }
            else
            {
               bDataChanged = FALSE;
            }
            break;
         case DP_U32_LOCK_MODE_MASTER:
            bDataChanged = FALSE;
            break;
         case DP_U32_LOCK_MODE_END_USER:
            if (oDatapool[u32PoolId].eModeEndUser != eDpModeEndUserNo)
            { /*pool "multi user" => set not possible*/
               ETG_TRACE_USR1(("bSetElement(): Pool %s is currently locked for end user pools", oDatapool[u32PoolId].strFileName));
               bDataChanged = FALSE;
            }
            break;
         default:
            bDataChanged = TRUE;
            break;
         }

         if (bDataChanged)
         {  //check if data is identical
            tVoid* pvNewData = pDpElement->pvGetData();
            tU32 u32NewDataLen = pDpElement->u32GetDataLength();
            tVoid* pvPoolData = posElement->second.pvGetData();
            tU32 u32PoolDataLen = posElement->second.u32GetDataLength();
            if (u32NewDataLen == u32PoolDataLen)
            {
               if (0 == memcmp(pvNewData, pvPoolData, u32PoolDataLen))
               {
                  bDataChanged = FALSE;
               }
            }
#ifdef DP_FEATURE_UNDEF_ELEMENT
            //for undef element: check if bVarSize==FALSE => lenght should be identical
            if (bDataChanged&&bUndefElement == TRUE&&posElement->second.bGetVarSize() == FALSE)
            {
               if (u32NewDataLen != u32PoolDataLen)
               {
                  bDataChanged = FALSE;
                  ETG_TRACE_ERR(("bSetElement(): The size to write is wrong: defined size %d", u32PoolDataLen));
                  s32Ret = DP_S32_ERR_WRONG_SIZE;
               }
            }
#endif
            //abr: 11.06.2014: pool location RAW_NOR and V850: default value should be stored(these data are read from different location) 
            if ((bDataChanged == FALSE) && (posElement->second.eGetElementState() == dp_tclBaseElement::eDpElemInit)
               && ((oDatapool[u32PoolId].eLocation >= eDpLocation_RAW_NOR) && (oDatapool[u32PoolId].eLocation <= eDpLocation_RAW_EMMC)))
            {
               ETG_TRACE_USR4(("bSetElement(): store default pool (state: %d, name: '%s')", posElement->second.eGetElementState(), oDatapool[u32PoolId].strFileName));
               bDataChanged = TRUE;
            }
         }
         else
         {
            ETG_TRACE_USR1(("bSetElement(): Pool is currently locked (mode: %d), cannot write element: '%s'!", _u32LockMode, pDpElement->pcGetElementName()));
            s32Ret = DP_S32_ERR_ACCESS_POOL_LOCKED;
         }
      }
      if (bForceStore)
      {
         bDataChanged = TRUE;
      }
      if ((tDpPos != oDatapool.end()) && bDataChanged && bAccess)
      {
         oDatapool[u32PoolId].tDpElementList[pDpElement->s32GetHash()].bFillData(pDpElement->pvGetData(), pDpElement->u32GetDataLength());
         //set state
         oDatapool[u32PoolId].tDpElementList[pDpElement->s32GetHash()].vSetElementState(dp_tclBaseElement::eDpElemSet);
         oDatapool[u32PoolId].bElemChanged = TRUE;

         if (oDatapool[u32PoolId].tDpElementList[pDpElement->s32GetHash()].eGetElementType() == dp_tclBaseElement::eDpElementTypePersistent)
         {
            tU32 u32CurrentTime = OSAL_ClockGetElapsedTime();
            tU32 u32CalcStoreTime = u32CurrentTime;

            //add element to history
            TDpHistoryProperty tChangedElem = { pDpElement->pcGetElementName(), u32CurrentTime };
            oDatapool[u32PoolId].tElemChangeMap[pDpElement->s32GetHash()] = tChangedElem;

            //validate element now
            oDatapool[u32PoolId].tDpElementList[pDpElement->s32GetHash()].vValidateElement();

            if (oDatapool[u32PoolId].tDpElementList[pDpElement->s32GetHash()].eGetStoringType() == dp_tclBaseElement::eDpStoringTypeCyclic)
            {
               u32CalcStoreTime = DP_U8_NOTHING_TO_STORE_TIME;

               tU32 u32CyclicTime = oDatapool[u32PoolId].tDpElementList[pDpElement->s32GetHash()].u32GetStoringIntervall() * 1000;
               if ((u32CyclicTime + oDatapool[u32PoolId].u32LastStoreTime) < oDatapool[u32PoolId].u32NextStoreTime)
               {
                  u32CalcStoreTime = u32CyclicTime + oDatapool[u32PoolId].u32LastStoreTime;
               }
            }
            else if (oDatapool[u32PoolId].tDpElementList[pDpElement->s32GetHash()].eGetStoringType() == dp_tclBaseElement::eDpStoringTypeImmediately)
            {
               u32CalcStoreTime = u32CurrentTime;
            }
            else if (oDatapool[u32PoolId].tDpElementList[pDpElement->s32GetHash()].eGetStoringType() == dp_tclBaseElement::eDpStoringTypeDelayed)
            {
               oDatapool[u32PoolId].tDpElementList[pDpElement->s32GetHash()].vSetNextStoreTime((u32CurrentTime + DP_U8_STORE_ELEMENT_DELAY_TIME));
               oDatapool[u32PoolId].bDelayedElementActive = TRUE;
               u32CalcStoreTime = DP_U8_NOTHING_TO_STORE_TIME;
            }
            else
            {
               u32CalcStoreTime = DP_U8_NOTHING_TO_STORE_TIME;
            }
            oDatapool[u32PoolId].bWaiting4Store = TRUE;

            if (bIsTracingEnabled(pDpElement)) ETG_TRACE_USR1(("bSetElement(): Element changed: store time: %d for Element '%s' ", u32CalcStoreTime, pDpElement->pcGetElementName()));
            if (oDatapool[u32PoolId].tDpElementList[pDpElement->s32GetHash()].eGetStoringType() == dp_tclBaseElement::eDpStoringTypeForce)
            {
               if (bExportPersData(u32PoolId, oDatapool[u32PoolId].strFileName, oDatapool[u32PoolId].eLocation, oDatapool[u32PoolId].u32PoolVersion, TRUE) == TRUE)
               {
                  oDatapool[u32PoolId].u32LastStoreTime = u32CurrentTime;
                  oDatapool[u32PoolId].bWaiting4Store = FALSE;
                  oDatapool[u32PoolId].u32NextStoreTime = DP_U8_NOTHING_TO_STORE_TIME;
               }/*else return error*/
               else
                  s32Ret = DP_S32_ERR_UNKNOWN;
            }
            else if (u32CalcStoreTime != DP_U8_NOTHING_TO_STORE_TIME)
            {
               if (u32CalcStoreTime < oDatapool[u32PoolId].u32NextStoreTime)
               {
                  if (u32CalcStoreTime < (oDatapool[u32PoolId].u32LastStoreTime + DP_U8_STORE_POOL_INTERVAL_TIME))
                  {
                     oDatapool[u32PoolId].u32NextStoreTime = oDatapool[u32PoolId].u32LastStoreTime + DP_U8_STORE_POOL_INTERVAL_TIME;
                  }
                  else
                  {
                     oDatapool[u32PoolId].u32NextStoreTime = u32CalcStoreTime;
                  }
               }
               if (bIsTracingEnabled(pDpElement))  ETG_TRACE_USR1(("bSetElement(): Element changed: Element will be stored at %d in poolid %d (element '%s')", oDatapool[u32PoolId].u32NextStoreTime, u32PoolId, pDpElement->pcGetElementName()));
            }
         }
      }
      DP_bReleaseCritical(_hDpAccess);

#ifdef DP_FEATURE_UNDEF_ELEMENT
      if (bAccess && bDataChanged && (bUndefElement == FALSE))
#else
      if (bAccess && bDataChanged)
#endif
      {  //inform registered via worker thread if element changed
         dp_tclDatapoolMaster::TDpPoolMsg oMsg = { dp_tclDatapoolMaster::eElementChanged, { 0 } };
         OSAL_szStringNCopy(oMsg.u.strData, posElement->second.pcGetElementName(), DP_MAX_LEN_NAME_ELEMENT);
         oMsg.u.strData[DP_MAX_LEN_NAME_ELEMENT - 1] = '\0';
         bPostOwnMessage(&oMsg);
      }
   }
   return s32Ret;
}

#ifdef DP_FEATURE_UNDEF_ELEMENT
tS32 dp_tclDpBase::s32GetUndefElement(const tChar* PpcUserName, const tChar* PpcPoolName, const tChar* PpcElementName, tU8* Ppu8ReadBuffer, tS32 Ps32Size)
{
   tS32                    Vs32ReturnCode = DP_S32_ERR_NO_ELEMENT;
   tDpUndefPoolProperty    VtPoolProperty;
   dp_tclDpUndef*          VpoMyDpUndef = dp_tclDpUndef::pGetInstance();
   dp_tclBaseElement       VDpElem(PpcElementName, 0); //hash calc in constructor    

   ETG_TRACE_USR4(("s32GetUndefElement(): get element '%s' ", PpcElementName));
   memset(&VtPoolProperty, 0, sizeof(tDpUndefPoolProperty));
   strncpy(VtPoolProperty.strPoolName, PpcPoolName, DP_MAX_LEN_NAME_UNDEFINED_POOLS);
   VtPoolProperty.strPoolName[DP_MAX_LEN_NAME_UNDEFINED_POOLS - 1] = '\0';
   strncpy(VtPoolProperty.strUserName, PpcUserName, DP_MAX_LEN_NAME_USER);
   VtPoolProperty.strUserName[DP_MAX_LEN_NAME_USER - 1] = '\0';
   strncpy(VtPoolProperty.strProcessNameLocated, _szMyProcSpecName, DP_MAX_LEN_NAME_PROCESS);
   VtPoolProperty.strProcessNameLocated[DP_MAX_LEN_NAME_PROCESS - 1] = '\0';
   /* get pool properties */
   Vs32ReturnCode = VpoMyDpUndef->s32GetPoolProperty(&VtPoolProperty, dp_tclDpUndef::eDpUndefGet, &VDpElem);
   if (Vs32ReturnCode == DP_S32_NO_ERR)
   { /* pool should located in actual process*/
      if (strncmp(VtPoolProperty.strProcessNameLocated, _szMyProcSpecName, DP_MAX_LEN_NAME_PROCESS) == 0)
      { /* check if pool exist in the process*/
         (tVoid)DP_bEnterCritical(_hDpAccess);
         TDpUndefPoolProperty::iterator VtDpPos = oDpUndefPoolProperty.find(VtPoolProperty.u32PoolId);
         if (VtDpPos == oDpUndefPoolProperty.end())
         { /*add new property*/
            oDpUndefPoolProperty[VtPoolProperty.u32PoolId] = VtPoolProperty;
         }
         DP_bReleaseCritical(_hDpAccess);
         Vs32ReturnCode = s32GetElement(VtPoolProperty.u32PoolId, &VDpElem, (tU16)(VtPoolProperty.u32PoolId >> 16));
         if (Vs32ReturnCode == DP_S32_NO_ERR)
         {
            if (VDpElem.u32GetDataLength() <= ((tU32)Ps32Size))
            {/*copy data into buffer*/
               vTraceElement(&VDpElem, eDpElemRead);
               memmove((void*)Ppu8ReadBuffer, VDpElem.pvGetData(), VDpElem.u32GetDataLength());
               ETG_TRACE_USR1(("s32GetUndefElement():%02x", ETG_LIST_LEN((VDpElem.u32GetDataLength())), ETG_LIST_PTR_T8(Ppu8ReadBuffer)));
               Vs32ReturnCode = (tS32)VDpElem.u32GetDataLength();
            }
            else
            {
               Vs32ReturnCode = DP_S32_ERR_WRONG_SIZE;
            }
         }
         else
         {
            ETG_TRACE_ERR(("s32GetUndefElement(): s32GetElement() returns error %d", Vs32ReturnCode));
         }
      }
      else
      {  /* pool not located in the actual process => get element should be get by the given located process*/
         Vs32ReturnCode = s32SharedUndefElementAccess(&VtPoolProperty, PpcElementName, TRUE, Ppu8ReadBuffer, Ps32Size);
      }
   }
   else
   {
      ETG_TRACE_USR4(("s32GetUndefElement(): element '%s' not located in this process", PpcElementName));
   }
   return(Vs32ReturnCode);
}
#endif

tS32 dp_tclDpBase::s32GetElement(dp_tclBaseElement* pDpElement, tU16 u16AccessId)
{
   tS32 s32Ret = DP_S32_ERR_NO_ELEMENT;
   TDatapool::iterator posDp;
   for (posDp = oDatapool.begin(); (posDp != oDatapool.end()) && (s32Ret == DP_S32_ERR_NO_ELEMENT); ++posDp)
   {
      tU32 u32PoolId = posDp->first;
      s32Ret = s32GetElement(u32PoolId, pDpElement, u16AccessId);
   }
   return s32Ret;
}

tS32 dp_tclDpBase::s32GetElement(tU32 u32PoolId, dp_tclBaseElement* pDpElement, tU16 u16AccessId)
{
   tS32 s32Ret = DP_S32_ERR_NO_ELEMENT;
   tBool bElementFound = FALSE;
   tBool bCodingElem = FALSE;
#ifdef DP_FEATURE_UNDEF_ELEMENT
   tBool bUndefElement = FALSE;
#endif
   tBool bOwner = ((tU32)(u16AccessId << (tU16)16) == (u32PoolId & (tU32)0xffff0000));

   if (bIsTracingEnabled(pDpElement))
      ETG_TRACE_USR1(("s32GetElement(): called for element (hash: 0x%08x) '%s' ", pDpElement->s32GetHash(), pDpElement->pcGetElementName()));

   ETG_TRACE_USR1(("s32GetElement():bOwner=%d,u16AccessId=%d,u32PoolId=%d", bOwner, u16AccessId, u32PoolId));
   ETG_TRACE_USR1(("s32GetElement():ProcSpecName='%s'", _szMyProcSpecName));
   //check for global/coding pool
   tU16 u16SubpoolId = (u32PoolId & 0x0000ffff);
   if (u16SubpoolId >= DP_U16_SUBPOOL_CODING_RANGE)
   {
      bCodingElem = TRUE;
   }
#ifdef DP_FEATURE_UNDEF_ELEMENT
   //check for undef element
   if ((u16AccessId >= CCA_C_U16_APP_UNDEFINED_BEGIN) && (u16AccessId <= CCA_C_U16_APP_UNDEFINED_END))
   {
      bUndefElement = TRUE;
   }
#endif
   /*if owner or coding element*/
   if (bOwner || bCodingElem)
   {  //check pool initu32PoolId
      s32Ret = s32CheckPoolInit(u32PoolId);
   }
   //check for access
   TDatapool::iterator tDpPos = oDatapool.find(u32PoolId);
   if ((tDpPos != oDatapool.end()) && (s32Ret == DP_S32_ERR_NO_ELEMENT))
   {//check list if element already exists
      bElementFound = TRUE;
      (tVoid)DP_bEnterCritical(_hDpAccess);
      TListDatapoolElements::iterator pos = oDatapool[u32PoolId].tDpElementList.find(pDpElement->s32GetHash());
      if (pos != oDatapool[u32PoolId].tDpElementList.end())
      {
         if ((bOwner && (pos->second.u16GetAccessType() & DP_U16_ACCESS_RD_OWNER_0004))
            || (!bOwner && (pos->second.u16GetAccessType() & DP_U16_ACCESS_RD_OTHER_0440)))
         {
#ifdef DP_FEATURE_UNDEF_ELEMENT
            tBool bElementCopy = TRUE;
            //if undef element and not state set
            if ((bUndefElement) && ((pos->second.eGetElementState() == dp_tclBaseElement::eDpElemInit)))
            { //not copied
               bElementCopy = FALSE;
               ETG_TRACE_ERR(("s32GetElement(): undef element '%s' only init", pDpElement->pcGetElementName()));
            }
            if (bElementCopy == TRUE)
#endif
            {
               pDpElement->vCopyElement(oDatapool[u32PoolId].tDpElementList[pDpElement->s32GetHash()]);
               //element found -> increment rd counter for this element
               oDatapool[u32PoolId].tDpElementList[pDpElement->s32GetHash()].vIncrementRdAccessCount();
               vTraceElement(pDpElement, eDpElemRead);
               s32Ret = DP_S32_NO_ERR;
            }
         }
         else
         {
            ETG_TRACE_ERR(("bGetElement(): error -> no access right to read element '%s'.", pDpElement->pcGetElementName()));
            s32Ret = DP_S32_ERR_ACCESS_RIGHT;
         }
      }
      else
      {
         bElementFound = FALSE;
      }
      (void)bElementFound;
      DP_bReleaseCritical(_hDpAccess);
   }
   if (s32Ret != DP_S32_NO_ERR)
      ETG_TRACE_ERR(("bGetElement(): returns %d.", s32Ret));
   return s32Ret;
}

#ifdef DP_FEATURE_UNDEF_ELEMENT
tS32     dp_tclDpBase::s32DeleteUndefElement(const tChar* PpcUserName, const tChar* PpcPoolName, const tChar* PpcElementName)
{
   tS32                    Vs32ReturnCode = DP_S32_NO_ERR;
   tDpUndefPoolProperty    VtPoolProperty;
   dp_tclDpUndef*          VpoMyDpUndef = dp_tclDpUndef::pGetInstance();
   dp_tclBaseElement       VDpElem(PpcElementName, 0); //hash calc in constructor    

   ETG_TRACE_USR4(("s32DeleteUndefElement(): delete element '%s' ", PpcElementName));
   memset(&VtPoolProperty, 0, sizeof(tDpUndefPoolProperty));
   strncpy(VtPoolProperty.strPoolName, PpcPoolName, DP_MAX_LEN_NAME_UNDEFINED_POOLS);
   VtPoolProperty.strPoolName[DP_MAX_LEN_NAME_UNDEFINED_POOLS - 1] = '\0';
   strncpy(VtPoolProperty.strUserName, PpcUserName, DP_MAX_LEN_NAME_USER);
   VtPoolProperty.strUserName[DP_MAX_LEN_NAME_USER - 1] = '\0';
   strncpy(VtPoolProperty.strProcessNameLocated, _szMyProcSpecName, DP_MAX_LEN_NAME_PROCESS);
   VtPoolProperty.strProcessNameLocated[DP_MAX_LEN_NAME_PROCESS - 1] = '\0';
   /* get pool properties */
   Vs32ReturnCode = VpoMyDpUndef->s32GetPoolProperty(&VtPoolProperty, dp_tclDpUndef::eDpUndefDelete, &VDpElem);
   if (Vs32ReturnCode == DP_S32_NO_ERR)
   { /* pool should located in actual process*/
      if (strncmp(VtPoolProperty.strProcessNameLocated, _szMyProcSpecName, DP_MAX_LEN_NAME_PROCESS) == 0)
      {  /*+++ check if pool properties exist in the process +++*/
         (tVoid)DP_bEnterCritical(_hDpAccess);
         TDpUndefPoolProperty::iterator VtDpPos = oDpUndefPoolProperty.find(VtPoolProperty.u32PoolId);
         if (VtDpPos == oDpUndefPoolProperty.end())
         { /*add new property*/
            oDpUndefPoolProperty[VtPoolProperty.u32PoolId] = VtPoolProperty;
         }
         DP_bReleaseCritical(_hDpAccess);

         /*++++   check pool init ++++ */
         (tVoid)DP_bEnterCritical(_hDpAccess);
         TDatapool::iterator tDpPos = oDatapool.find(VtPoolProperty.u32PoolId);
         //try to find pool
         if (tDpPos == oDatapool.end())
         { /*init pool*/
            bInitPool(VtPoolProperty.u32PoolId);
         }
         DP_bReleaseCritical(_hDpAccess);

         /*++++   check element exist in pool ++++ */
         tDpPos = oDatapool.find(VtPoolProperty.u32PoolId);
         if (tDpPos != oDatapool.end())
         {
            (tVoid)DP_bEnterCritical(_hDpAccess);
            Vs32ReturnCode = s32DeleteElement(&VtPoolProperty, &VDpElem);
            DP_bReleaseCritical(_hDpAccess);
         }
         else
         {
            Vs32ReturnCode = DP_S32_ERR_NO_POOL;
            ETG_TRACE_ERR(("s32DeleteUndefElement(): error -> pool not found '%s'.", PpcPoolName));
         }
      }
      else
      {
         Vs32ReturnCode = s32SharedUndefElementDelete(&VtPoolProperty, PpcElementName);
      }
   }
   return(Vs32ReturnCode);
}
#endif

// ****************************************************************************
// ********   storage handling 
// ****************************************************************************

tBool dp_tclDpBase::bExportPersData(tU32 u32PoolId, tChar* cPoolName, eDpLocation PeLocation, tU32 u32Version, tBool bfSync, tU8 Pu8Bank)
{  //get needed memory size first
   tBool   VbReturn = TRUE;
   tBool   VbSaveDataStream = TRUE;
   tU32    u32FileLen = 0;
   TDatapool::iterator tDpPos = oDatapool.find(u32PoolId);

   ETG_TRACE_USR4(("dp_tclDpBase::bExportPersData(): store pool '%s'.", cPoolName));

#ifdef DP_U32_POOL_ID_DPENDUSERMODE 
   if (tDpPos != oDatapool.end())
   { /*check if end user pool */
      if (tDpPos->second.eModeEndUser != eDpModeEndUserNo)
      { /*if end user mode activ; check if current end user should be stored */
         dp_tclDpEndUser*        VpoMyDpEndUser = dp_tclDpEndUser::pGetInstance();
         if (VpoMyDpEndUser)
         {
            VbSaveDataStream = VpoMyDpEndUser->bCheckEndUserShouldStore(cPoolName, Pu8Bank);
            if (VbSaveDataStream == FALSE)
            {
               ETG_TRACE_USR4(("dp_tclDpBase::bExportPersData(): end user pool '%s' should not be stored", cPoolName));
            }
         }
      }
   }
#endif
   if (VbSaveDataStream == TRUE)
   { /*load sub pool list, if not already loaded */
      vImportSubPoolList();
      /*export subpools list, if u32PoolId not already stored*/
      vExportSubPoolList(u32PoolId);

      if (tDpPos != oDatapool.end())
      {  //first get needed length for elements to store
         TListDatapoolElements::iterator posElement;
         for (posElement = oDatapool[u32PoolId].tDpElementList.begin(); posElement != oDatapool[u32PoolId].tDpElementList.end(); ++posElement)
         {
            if ((posElement->second.eGetElementType() == dp_tclBaseElement::eDpElementTypePersistent) && (posElement->second.bIsElementValidated()))
            {  //found element -> add length for this element
               u32FileLen += posElement->second.u32GetStreamLength();
            }
         }
         ETG_TRACE_USR4(("dp_tclDpBase::bExportPersData(): calculated file length: %d for Pool %s", u32FileLen, cPoolName));

         // all elements checked -> get memory for file to store
         tU8* pBuffer = NULL;
         try
         {
            pBuffer = new tU8[u32FileLen];
         }
         catch (const std::bad_alloc&)
         {
            pBuffer = NULL;
            ETG_TRACE_USR4(("dp_tclDpBase::bExportPersData(): unable to allocate length: %d for pool %s.", u32FileLen, cPoolName));
         }
         if (pBuffer)
         {
            tU8* pu8CurPos = pBuffer;
            tU32 u32TotalLen = 0;
            //ETG_TRACE_USR4(("dp_tclDpBase::bExportPersData(): check next element at pos %08x.", pu8CurPos));
            for (posElement = oDatapool[u32PoolId].tDpElementList.begin(); posElement != oDatapool[u32PoolId].tDpElementList.end(); ++posElement)
            {
               ETG_TRACE_USR4(("dp_tclDpBase::bExportPersData(): check for element '%s'.", posElement->second.pcGetElementName()));
               if ((posElement->second.eGetElementType() == dp_tclBaseElement::eDpElementTypePersistent) && (posElement->second.bIsElementValidated()))
               { //fill buffer -> element by element
                 //ETG_TRACE_USR4(("dp_tclDpBase::bExportPersData(): add element '%s'.", posElement->second.pcGetElementName()));
                 // and now stream data to current context
                  tU32 u32Len = posElement->second.u32GetStreamLength();
                  u32TotalLen += u32Len;
                  if ((pu8CurPos + u32Len) <= (pBuffer + u32FileLen))//lint !e662 :Possible creation of out-of-bounds pointer (4 beyond end of data) by operator 'ptr+int' -> buffer ranges checked
                  {
                     dp_tclOutStreamCtxt oContext(pu8CurPos, u32Len);
                     oContext << posElement->second;
                     pu8CurPos = oContext.pu8GetEndPosition();
                  }
                  else
                  {
                     ETG_TRACE_FATAL(("dp_tclDpBase::bExportPersData():Failed for element '%s'.", posElement->second.pcGetElementName()));
                     ETG_TRACE_FATAL(("dp_tclDpBase::bExportPersData():Overflow Detected!Needed = %d & available= %d", u32TotalLen, u32FileLen));
                  }
                  //ETG_TRACE_USR4(("dp_tclDpBase::bExportPersData(): check next element at pos %08x.", pu8CurPos));
               }
            }
            //add some history information
            TDpPoolHistoryEntry tHisEntry;
            tHisEntry.u32TimeStamp = OSAL_ClockGetElapsedTime();
            tHisEntry.strFileName = cPoolName;
            tHisEntry.u32PoolId = u32PoolId;
            tHisEntry.u32BytesWritten = u32FileLen;
            TElementChangeMap::iterator posElem;
            for (posElem = oDatapool[u32PoolId].tElemChangeMap.begin(); posElem != oDatapool[u32PoolId].tElemChangeMap.end(); ++posElem)
            {
               if (posElem->second.u32Timestamp != 0)
               {
                  tHisEntry.tElemChangeMap[posElem->first] = posElem->second;
                  oDatapool[u32PoolId].tElemChangeMap[posElem->first].u32Timestamp = 0;
               }
            }
            oHistoryList.push_back(tHisEntry);

            //and now store file
            VbReturn = _poPersMem->bWriteFfsData(cPoolName, pBuffer, u32FileLen, PeLocation, u32Version, bfSync);
            delete[] pBuffer;
            pBuffer = NULL;/*Bug 123144: Safe handling*/
         }
      }
   }
   return(VbReturn);
}

tVoid dp_tclDpBase::vImportPersData(tU32 u32PoolId, tChar* cPoolName, eDpLocation PeLocation, tU32 u32Version)
{
   tU8* pu8Buffer = NULL;
   tS32 s32ReadBytes = 0;

   DP_NULL_POINTER_CHECK(_poPersMem);
   s32ReadBytes = _poPersMem->s32ImportPersData(cPoolName, &pu8Buffer, PeLocation, u32Version);
   if (0 < s32ReadBytes)
   {  //extract data to pool
      tU8* pu8CurPos = pu8Buffer;
      tU16 u16Count = 0;
      while ((pu8CurPos + sizeof(tU32)) < (pu8Buffer + s32ReadBytes))
      { //lint !e662 :Possible creation of out-of-bounds pointer (4 beyond end of data) by operator 'ptr+int' -> buffer ranges checked
       //create empty element
         dp_tclBaseElement myDataElem;
         //stream data to context
         dp_tclInStreamCtxt oInContext(pu8CurPos);
         oInContext >> myDataElem;
         //now add to pool if data valid
         if (oInContext.bIsValid())
         {
            vInitElement(u32PoolId, &myDataElem, dp_tclBaseElement::eDpElemLoaded);
            pu8CurPos = oInContext.pu8GetEndPosition();
            u16Count++;
         }
         else
         {/*could not import pool element not valid => break */
            pu8CurPos = pu8Buffer + s32ReadBytes; //break loop
            ETG_TRACE_ERRMEM(("DATAPOOL: dp_tclDpBase::vImportPersData(): read invalid datastream poolname:'%s' ! ", cPoolName));
         }
      }
      ETG_TRACE_USR1(("vImportPersData(): %d elements added to pool.", u16Count));
   }
   else
      ETG_TRACE_USR1(("vImportPersData(): returns %d for pool %s.", s32ReadBytes, cPoolName));
   delete[] pu8Buffer;
   pu8Buffer = NULL;/*Bug 123144: Safe handling*/
}

tVoid dp_tclDpBase::vImportSubPoolList()
{
   tS32 s32ReadBytes = 0;
   tU8* pu8Buffer = NULL;
   tChar strName[DP_MAX_LEN_FILE] = { 0 };

   DP_NULL_POINTER_CHECK(_poPersMem);
   /*check _bSubList load */
   if (_bSubListLoad == FALSE)
   { /*import process subpool list*/
      OSAL_szStringNCopy((tString)strName, _szMyProcSpecName, DP_MAX_LEN_FILE - 1);
      s32ReadBytes = _poPersMem->s32ImportPersData(strName, &pu8Buffer, eDpLocation_FILE_SYSTEM_SECURE, DP_U32_FILE_VERSION);
      if ((s32ReadBytes >= 4) && ((s32ReadBytes % 4) == 0))
      { //must be 4 byte aligned
         tU32* pPos = (tU32*)pu8Buffer; //lint !e826: Suspicious pointer-to-pointer conversion (area too small) -> s32ReadBytes>=4 -> min. 4bytes allocated
         for (tU16 i = 0; i < (s32ReadBytes / 4); i++)
         {
            TDpPoolSubPoolList::iterator pDpSub;
            tBool bFound = FALSE;
            /*check if ID exist in list*/
            for (pDpSub = _tListSubPools.begin(); (pDpSub != _tListSubPools.end()) && !bFound; ++pDpSub)
            {
               if (pDpSub->u32SubPool == pPos[i])
               {
                  bFound = TRUE;
                  ETG_TRACE_USR4(("dp_tclDpBase::vImportSubPoolList(): SubPool already in list: %08x.(%i, %i)", pPos[i], (s32ReadBytes / 4), i));
               }
            }/*end for search in subpool in list*/
            if (!bFound)
            { /* add subpool to list */
               TDpListSubPool tListElement = { FALSE, TRUE, pPos[i] };
               _tListSubPools.push_back(tListElement);
            }
         }/*end for*/
      }
      delete[] pu8Buffer;
      pu8Buffer = NULL;/*Bug 123144: Safe handling*/
      _bSubListLoad = TRUE;
   }
}

tVoid dp_tclDpBase::vRemovePersistentPool(tU32 u32PoolId, tChar* PcPoolName, eDpLocation eDpLocation)
{
   tChar  VstrFileName[DP_MAX_LEN_FILE] = { 0 };
   if (_poPersMem != NULL)
   {
      if (PcPoolName != NULL)
      {
         snprintf(VstrFileName, DP_MAX_LEN_FILE - 1, "%04x/%s", (tU16)(u32PoolId >> 16), PcPoolName);
         ETG_TRACE_USR4(("dp_tclDpBase::vRemovePersistentPool(): %s ", VstrFileName));
         _poPersMem->s32DeletePersData(VstrFileName, eDpLocation);
      }
      else
      {
         snprintf(VstrFileName, DP_MAX_LEN_FILE - 1, "%04x", (tU16)(u32PoolId >> 16));
         ETG_TRACE_USR4(("dp_tclDpBase::vRemovePersistentPool(): %s ", VstrFileName));
         _poPersMem->s32DeleteOldPool(VstrFileName, eDpLocation);
      }
   }
}

tVoid dp_tclDpBase::vRemovePoolData(tU32 u32PoolId)
{
   //remove the files corresponding to this pool if exists
   //remove the entry from undefPool

#ifndef DP_FEATURE_UNDEF_ELEMENT
   tU16 Vu16AccessId = (tU16)(u32PoolId >> 16);
   tDpUndefClearPoolList::iterator pDpClearPool;

   if ((Vu16AccessId >= CCA_C_U16_APP_UNDEFINED_BEGIN) && (Vu16AccessId <= CCA_C_U16_APP_UNDEFINED_END))
   {
      ETG_TRACE_USR1(("dp_tclDpBase::vRemovePoolData():Remove Old Undef Pool 0x%8x size = %d", u32PoolId, pDpUndefClearPoolList.size()));

      if (pDpUndefClearPoolList.size() == 0)
      {
         // Load undef Pool Property with Lock
         ETG_TRACE_USR1(("dp_tclDpBase::vRemovePoolData():Call vLoadUndefClearPoolList"));
         vLoadUndefClearPoolList();
      }
      if (pDpUndefClearPoolList.size() != 0)
      {
         for (pDpClearPool = pDpUndefClearPoolList.begin(); pDpClearPool != pDpUndefClearPoolList.end(); ++pDpClearPool)
         {
            ETG_TRACE_USR1(("dp_tclDpBase::vRemovePoolData():Clear PoolId %x size %d", pDpClearPool->u32PoolID, pDpUndefClearPoolList.size()));
            if (pDpClearPool->u32PoolID == u32PoolId)
            {
               //remove the undef element property 
               vRemovePersistentPool(u32PoolId, (tChar *)DP_UNDEF_FILE_NAME_ELEMENT_PROPERTIES, eDpLocation_FILE_SYSTEM);
               //remove the undef pool data
               vRemovePersistentPool(u32PoolId, pDpClearPool->strPoolName, eDpLocation_FILE_SYSTEM);
               //remove the undef pool Folder

               vRemovePersistentPool(u32PoolId, NULL, eDpLocation_FILE_SYSTEM);

               //remove the tDpUndefClearPoolList entry
               pDpUndefClearPoolList.erase(pDpClearPool);
               if (pDpUndefClearPoolList.size() == 0)
               {
                  //remove undefpool property file
                  _poPersMem->s32DeletePersData(DP_UNDEF_FILE_NAME_POOL_PROPERTIES, eDpLocation_FILE_SYSTEM_SECURE);
               }
               break;
            }
         }
      }
   }
   else
#endif
   {
      ETG_TRACE_USR1(("dp_tclDpBase::vRemovePoolData():Cannot Remove Datapool Old Pool 0x%8x from disk", u32PoolId));
   }
}

tVoid dp_tclDpBase::vRemoveAndExportSubPoolList(tU32 u32PoolId)
{
   TDpPoolSubPoolList::iterator pDpSub;
   tU32 u32BufSize = ((tU32)_tListSubPools.size()) * 4;
   tU8* pBuffer = NULL;
   tBool bSubPoolAlreadyStored = TRUE;

   ETG_TRACE_USR1(("dp_tclDpBase::vRemoveAndExportSubPoolList(): Enter for poolid %x  subpool size %d", u32PoolId, (tU32)_tListSubPools.size()));

   DP_NULL_POINTER_CHECK(_poPersMem);
   try
   {
      pBuffer = new tU8[u32BufSize];
   }
   catch (const std::bad_alloc&)
   {
      pBuffer = NULL;
      ETG_TRACE_ERR(("dp_tclDpBase::vRemoveAndExportSubPoolList(): allocate failed for %x elements for poolid %x", u32BufSize, u32PoolId));
   }
   if (pBuffer != NULL)
   {
      tU32* pPos = (tU32*)pBuffer; //lint !e826: Suspicious pointer-to-pointer conversion (area too small) -> if "pBuffer!=NULL" min. 4bytes allocated
      for (pDpSub = _tListSubPools.begin(); pDpSub != _tListSubPools.end(); ++pDpSub)
      {
         ETG_TRACE_USR1(("dp_tclDpBase::vRemoveAndExportSubPoolList(): %x %x %x", pDpSub->u32SubPool, u32PoolId, _tListSubPools.size()));
         /*check actual u32SubPool*/
         if (pDpSub->u32SubPool == u32PoolId)
         {
            bSubPoolAlreadyStored = FALSE;
            /*set to T;RUE*/
            ETG_TRACE_USR1(("dp_tclDpBase::vRemoveAndExportSubPoolList(): %x %x %x", pDpSub->u32SubPool, u32PoolId, _tListSubPools.size()));
            pDpSub = _tListSubPools.erase(pDpSub);
            break;
         }
      }
      /*add u32SubPool to buffer*/
      if (bSubPoolAlreadyStored == FALSE)
      {
         // _tListSubPools.clear();
         for (pDpSub = _tListSubPools.begin(); pDpSub != _tListSubPools.end(); ++pDpSub)
         {
            // _tListSubPools.push_back(*pDpSub);
            ETG_TRACE_ERR(("dp_tclDpBase::vRemoveAndExportSubPoolList(): %x %x %x", pDpSub->u32SubPool, u32PoolId, _tListSubPools.size()));
            if (pDpSub->u32SubPool != u32PoolId)
            {
               if (pPos)
                  *pPos++ = pDpSub->u32SubPool;
            }
         }
         tChar strName[DP_MAX_LEN_FILE] = { 0 };
         OSAL_szStringNCopy((tString)strName, _szMyProcSpecName, (DP_MAX_LEN_FILE - 1));
         ETG_TRACE_USR1(("dp_tclDpBase::vRemoveAndExportSubPoolList():before write %x %x", u32PoolId, (tU32)_tListSubPools.size()))
            _poPersMem->bWriteFfsData(strName, pBuffer, u32BufSize, eDpLocation_FILE_SYSTEM_SECURE, DP_U32_FILE_VERSION, TRUE);
      }
      delete[] pBuffer;
      pBuffer = NULL;/*Bug 123144: Safe handling*/
   }
}

tVoid dp_tclDpBase::vExportSubPoolList(tU32 u32PoolId)
{
   TDpPoolSubPoolList::iterator pDpSub;
   tU32 u32BufSize = (tU32)_tListSubPools.size() * 4;
   tU8* pBuffer = NULL;
   tBool bSubPoolAlreadyStored = FALSE;

   DP_NULL_POINTER_CHECK(_poPersMem);
   try
   {
      pBuffer = new tU8[u32BufSize];
   }
   catch (const std::bad_alloc&)
   {
      pBuffer = NULL;
      ETG_TRACE_ERR(("dp_tclDpBase::vExportSubPoolList(): allocate failed for %d elements for poolid %d", u32BufSize, u32PoolId));
   }
   if (pBuffer != NULL)
   {
      tU32* pPos = (tU32*)pBuffer; //lint !e826: Suspicious pointer-to-pointer conversion (area too small) -> if "pBuffer!=NULL" min. 4bytes allocated
      for (pDpSub = _tListSubPools.begin(); pDpSub != _tListSubPools.end(); ++pDpSub)
      { /*add u32SubPool to buffer*/
         if (pPos)
            *pPos++ = pDpSub->u32SubPool;
         /*check actual u32SubPool*/
         if (pDpSub->u32SubPool == u32PoolId)
         { /*check state stored */
            if (pDpSub->bStored == TRUE)
            {
               bSubPoolAlreadyStored = TRUE;
            }
            /*set to TRUE*/
            pDpSub->bStored = TRUE;
         }
      }
      if (bSubPoolAlreadyStored == FALSE)
      {
         tChar strName[DP_MAX_LEN_FILE] = { 0 };
         OSAL_szStringNCopy((tString)strName, _szMyProcSpecName, (DP_MAX_LEN_FILE - 1));

         _poPersMem->bWriteFfsData(strName, pBuffer, u32BufSize, eDpLocation_FILE_SYSTEM_SECURE, DP_U32_FILE_VERSION, TRUE);
      }
      delete[] pBuffer;
      pBuffer = NULL;/*Bug 123144: Safe handling*/
   }
}

// ****************************************************************************
// ********   Thread handling 
// ****************************************************************************
void dp_tclDpBase::vWaitDeleteExistingEvents(tU32 u32Mask)
{
   _u32EvWaitForMask = u32Mask;
   _u32EvWaitForMask |= DP_U32_EVENT_BUSY;
   _u32EvWaitForMask |= DP_U32_EVENT_TIMEOUT;
   _u32EvWaitForMask |= DP_U32_EVENT_ERR_TYPE;
   _u32EvWaitForMask |= DP_U32_EVENT_ERR_ALREADY_LOCKED;
   OSAL_s32EventPost(_hEvSync, ~_u32EvWaitForMask, OSAL_EN_EVENTMASK_AND);
}

tS32 dp_tclDpBase::s32WaitSync(tU32 u32Mask, tU32 u32Timeout)
{
   tS32 s32Ret = DP_S32_ERR_UNKNOWN;
   _u32EvWaitForMask = u32Mask;

   _u32EvWaitForMask |= DP_U32_EVENT_BUSY;
   _u32EvWaitForMask |= DP_U32_EVENT_TIMEOUT;
   _u32EvWaitForMask |= DP_U32_EVENT_ERR_TYPE;
   _u32EvWaitForMask |= DP_U32_EVENT_ERR_ALREADY_LOCKED;

   ETG_TRACE_USR1(("s32WaitSync(): wait for Event 0x%08x.", _u32EvWaitForMask));

   OSAL_tEventMask hEvRequest = 0;
   //OSAL_s32EventPost(_hEvSync, ~_u32EvWaitForMask, OSAL_EN_EVENTMASK_AND);
   if (OSAL_ERROR == OSAL_s32EventWait(_hEvSync, _u32EvWaitForMask, OSAL_EN_EVENTMASK_OR, u32Timeout, &hEvRequest))
   {
      ETG_TRACE_USR1(("s32WaitSync(): Returned after %d without ACK.", u32Timeout));
      s32Ret = DP_S32_ERR_TIMEOUT;
   }
   else {
      ETG_TRACE_USR1(("s32WaitSync(): RECEIVED Event 0x%08x.", hEvRequest));
      if (DP_U32_EVENT_BUSY & hEvRequest)
      {
         s32Ret = DP_S32_ERR_BUSY;
      }
      else if (DP_U32_EVENT_TIMEOUT & hEvRequest)
      {
         s32Ret = DP_S32_ERR_TIMEOUT;
      }
      else if (DP_U32_EVENT_ERR_TYPE & hEvRequest)
      {
         s32Ret = DP_S32_ERR_TYPE;
      }
      else if (DP_U32_EVENT_ERR_ALREADY_LOCKED & hEvRequest)
      {
         s32Ret = DP_S32_ERR_ALREADY_LOCKED;
      }
      else if (u32Mask & hEvRequest)
      {
         s32Ret = DP_S32_NO_ERR;
      }
      OSAL_s32EventPost(_hEvSync, ~hEvRequest, OSAL_EN_EVENTMASK_AND);
   }
   _u32EvWaitForMask = 0;
   return s32Ret;
}

tBool dp_tclDpBase::bPostMasterMessage(dp_tclDatapoolMaster::TDpPoolMsg* pMsg)
{
   tBool bSuccess = FALSE;
   if (_hMasterQueueHandle != OSAL_C_INVALID_HANDLE)
   {
      tS32 s32Ret = OSAL_s32MessageQueuePost(_hMasterQueueHandle, (tCU8*)pMsg, sizeof(dp_tclDatapoolMaster::TDpPoolMsg), OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST);
      if (s32Ret == OSAL_OK)
      {
         bSuccess = TRUE;
      }
   }
   return bSuccess;
}

tBool dp_tclDpBase::bPostOwnMessage(dp_tclDatapoolMaster::TDpPoolMsg* pMsg)
{
   tBool bSuccess = FALSE;
   tS32 s32Ret = OSAL_s32MessageQueuePost(_hMessageQueueHandle, (tCU8*)pMsg, sizeof(dp_tclDatapoolMaster::TDpPoolMsg), OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST);

   if (s32Ret == OSAL_OK)
   {
      bSuccess = TRUE;
   }
   return bSuccess;
}

tBool dp_tclDpBase::bPostMessageToProcessLocatedData(tChar* PpstrProcessNameElementLocated, dp_tclDatapoolMaster::TDpPoolMsg* pMsg)
{
   tBool              VbSuccess = FALSE;
   OSAL_tMQueueHandle VHdlQueue;

   /*first get handle*/
   tS32 Vs32Ret = OSAL_s32MessageQueueOpen(PpstrProcessNameElementLocated, OSAL_EN_READWRITE, &VHdlQueue);
   if (Vs32Ret != OSAL_OK)
   {  //failed to open queue
      ETG_TRACE_FATAL(("dp_tclDpBase::bPostMessageToProcessLocatedData(): failed to open master queue."));
   }
   else
   { /*post message */
      Vs32Ret = OSAL_s32MessageQueuePost(VHdlQueue, (tCU8*)pMsg, sizeof(dp_tclDatapoolMaster::TDpPoolMsg), OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST);
      if (Vs32Ret == OSAL_OK)
      {
         VbSuccess = TRUE;
         ETG_TRACE_USR4(("dp_tclDpBase::bPostMessageToProcessLocatedData(): post message to process %s ", PpstrProcessNameElementLocated));
      }
   }
   return(VbSuccess);
}

tVoid dp_tclDpBase::vDpMainThread(tVoid *pvArg)
{
   dp_tclDpBase * myRef = (dp_tclDpBase*)pvArg;
   while (!(myRef->_bTerminate))
   {
      tU32 u32Prio;
      dp_tclDatapoolMaster::TDpPoolMsg oMsg;
      // wait for a message
      // writes the received OSALMsgHandle in the message object
      tS32 s32ReadBytes = OSAL_s32MessageQueueWait(myRef->_hMessageQueueHandle, (tU8*)&oMsg, /*pointer of the MsgHandle field*/
         sizeof(dp_tclDatapoolMaster::TDpPoolMsg), &u32Prio, DP_U32_CHECK_POOL_DELAY);
      if (s32ReadBytes > 0)
      {  //check message type
         ETG_TRACE_USR4(("vDpMainThread():: Message received: %d, process: %s.", oMsg.eCmd, myRef->_szMyProcSpecName));
         if (myRef->bMessageHandler(&oMsg) == TRUE)
         { //process terminates
            myRef->_bTerminate = TRUE;
         }
      }
      else if (s32ReadBytes == 0)
      {//no message received -> must be timeout: check if something has to be stored
         myRef->vTimerHandling();
      }
      else
      {
         ETG_TRACE_USR4(("vDpMainThread():: OSAL_s32MessageQueueWait returned with %08x.", s32ReadBytes));
      }
   }/*end while*/
   ETG_TRACE_USR4(("vDpMainThread():: call OSAL_vThreadExit"));
   OSAL_vThreadExit();
}

tBool dp_tclDpBase::bMessageHandler(dp_tclDatapoolMaster::TDpPoolMsg* pMsg)
{
   tBool VbTerminate = FALSE;
   switch (pMsg->eCmd)
   {
   case dp_tclDatapoolMaster::eClientStorePool:
   {//synchronous call -> send ack
      vStoreNow();
      dp_tclDatapoolMaster::TDpPoolMsg oNewMsg = { dp_tclDatapoolMaster::eAckPoolStored, { 0 } };
      if (!bPostMasterMessage(&oNewMsg))
      {
         ETG_TRACE_FATAL(("vDpMainThread(): failed to post message to master"));
      }
   }break;
   case dp_tclDatapoolMaster::eClientLockPool:
   {  //synchronous call -> send ack
      (tVoid)DP_bEnterCritical(_hDpAccess);
      if (DP_U32_LOCK_MODE_SETDEF == pMsg->u.u32Data)
      {
         _u32LockMode = DP_U32_LOCK_MODE_SETDEF;
      }
      else if (DP_U32_LOCK_MODE_MASTER == pMsg->u.u32Data)
      {
         _u32LockMode = DP_U32_LOCK_MODE_MASTER;
      }
      else if (DP_U32_LOCK_MODE_END_USER == pMsg->u.u32Data)
      {
         _u32LockMode = DP_U32_LOCK_MODE_END_USER;
      }
      DP_bReleaseCritical(_hDpAccess);
      dp_tclDatapoolMaster::TDpPoolMsg oNewMsg = { dp_tclDatapoolMaster::eAckPoolLocked, { 0 } };
      if (!bPostMasterMessage(&oNewMsg))
      {
         ETG_TRACE_FATAL(("vDpMainThread(): failed to post message to master"));
      }
   }break;
   case dp_tclDatapoolMaster::eClientUnlockPool:
   {//synchronous call -> send ack
      (tVoid)DP_bEnterCritical(_hDpAccess);
      _u32LockMode = DP_U32_LOCK_MODE_NONE;
      DP_bReleaseCritical(_hDpAccess);
      dp_tclDatapoolMaster::TDpPoolMsg oNewMsg = { dp_tclDatapoolMaster::eAckPoolUnlocked, { 0 } };
      if (!bPostMasterMessage(&oNewMsg))
      {
         ETG_TRACE_FATAL(("vDpMainThread(): failed to post message to master"));
      }
   }break;
   case dp_tclDatapoolMaster::eClientDefSet:
   {  //synchronous call -> send ack
      vPerformDefSet(pMsg->u.au32Data[0], (tU8)pMsg->u.au32Data[1]);
      dp_tclDatapoolMaster::TDpPoolMsg oNewMsg = { dp_tclDatapoolMaster::eAckDefSet, { 0 } };
      if (!bPostMasterMessage(&oNewMsg))
      {
         ETG_TRACE_FATAL(("vDpMainThread(): failed to post message to master"));
      }
   }break;
   case dp_tclDatapoolMaster::eClientCalcHash:
   {  //here calculate hashes for all subpools
      vCalcPoolHashes();
      //synchronous call -> send ack
      dp_tclDatapoolMaster::TDpPoolMsg oNewMsg = { dp_tclDatapoolMaster::eAckHashCalc, { 0 } };
      if (!bPostMasterMessage(&oNewMsg))
      {
         ETG_TRACE_FATAL(("vDpMainThread(): failed to post message to master"));
      }
   }break;
   case dp_tclDatapoolMaster::eClientTracePool:
   {
      if (pMsg->u.u32Data == 0xffffffff)
      {
         vTracePoolElement();
      }
      else
      {
         vTracePoolElement(pMsg->u.u32Data);
      }
   }break;
   case dp_tclDatapoolMaster::eClientTraceElement:
   {
      vTracePoolElement(pMsg->u.strData);
   }break;
   case dp_tclDatapoolMaster::eClientTraceHistory:
   {
      vTracePoolHistory();
   }break;
   case dp_tclDatapoolMaster::eClientAllPoolsStored:
   {
      OSAL_s32EventPost(_hEvSync, DP_U32_EVENT_STORE_FINISHED, OSAL_EN_EVENTMASK_OR);
   }break;
   case dp_tclDatapoolMaster::eClientAllPoolsLocked:
   {
      OSAL_s32EventPost(_hEvSync, DP_U32_EVENT_LOCKED_FINISHED, OSAL_EN_EVENTMASK_OR);
   }break;
   case dp_tclDatapoolMaster::eClientAllPoolsUnlocked:
   {
      OSAL_s32EventPost(_hEvSync, DP_U32_EVENT_UNLOCKED_FINISHED, OSAL_EN_EVENTMASK_OR);
   }break;
   case dp_tclDatapoolMaster::eClientAllDefSetsReady:
   {
      OSAL_s32EventPost(_hEvSync, DP_U32_EVENT_DEFSET_FINISHED, OSAL_EN_EVENTMASK_OR);
   }break;
   case dp_tclDatapoolMaster::eClientHashReady:
   {
      memmove((tU8*)_tCurentHash, (tU8*)pMsg->u.tHash.tValue.u32Hash, DP_MASTER_HASH_COUNT * sizeof(tU32));
      OSAL_s32EventPost(_hEvSync, DP_U32_EVENT_HASH_FINISHED, OSAL_EN_EVENTMASK_OR);
   }break;
   case dp_tclDatapoolMaster::eErrBusy:
   {
      OSAL_s32EventPost(_hEvSync, DP_U32_EVENT_BUSY, OSAL_EN_EVENTMASK_OR);
   }break;
   case dp_tclDatapoolMaster::eErrParamType:
   {
      OSAL_s32EventPost(_hEvSync, DP_U32_EVENT_ERR_TYPE, OSAL_EN_EVENTMASK_OR);
   }break;
   case dp_tclDatapoolMaster::eErrAlreadyLocked:
   {
      OSAL_s32EventPost(_hEvSync, DP_U32_EVENT_ERR_ALREADY_LOCKED, OSAL_EN_EVENTMASK_OR);
   }
   break;
   case dp_tclDatapoolMaster::eElementChanged:
   {
      vElementChanged(pMsg->u.strData);
   }break;
   case dp_tclDatapoolMaster::eStopWaitThread:
   {//send event process terminates
      ETG_TRACE_USR4(("vDpMainThread():: post message DP_U32_EVENT_STOP_WAIT_THREAD_DONE"));
      OSAL_s32EventPost(_hEvSync, DP_U32_EVENT_STOP_WAIT_THREAD_DONE, OSAL_EN_EVENTMASK_OR);
      VbTerminate = TRUE;
   }break;
   case dp_tclDatapoolMaster::eClientUpdatePool:
   {
      (tVoid)DP_bEnterCritical(_hDpAccess);
      TDatapool::iterator VposDp = oDatapool.find(pMsg->u.u32Data);
      if (VposDp != oDatapool.end())
      {
         oDatapool.erase(pMsg->u.u32Data);
         bInitPool(pMsg->u.u32Data);
         //check if pool init/ available
         VposDp = oDatapool.find(pMsg->u.u32Data);
         if (VposDp != oDatapool.end())
         {
            ETG_TRACE_USR4(("eClientUpdatePool(): pool '%08x' state bElemChanged '%d' ", pMsg->u.u32Data, oDatapool[pMsg->u.u32Data].bElemChanged));
         }
      }
      DP_bReleaseCritical(_hDpAccess);
      ETG_TRACE_USR1(("vDpMainThread():: Update of pool detected : %08x.", pMsg->u.u32Data));
      if (pMsg->u.u32Data == DP_U16_DP_INTERN_CONFIGPOOL)
      {  //reload trace config
         _bListTraceElementsLoad = FALSE;
         vSetTraceConfiguration();
      }
      //synchronous call -> send ack
      dp_tclDatapoolMaster::TDpPoolMsg oNewMsg = { dp_tclDatapoolMaster::eAckUpdatePool, { 0 } };
      if (!bPostMasterMessage(&oNewMsg))
      {
         ETG_TRACE_FATAL(("vDpMainThread(): failed to post message to master"));
      }
   }break;
   case dp_tclDatapoolMaster::eClientAllUpdatePoolReady:
   {
      ETG_TRACE_USR4(("vDpMainThread():: send DP_U32_EVENT_UPDATE_POOL_FINISHED"));
      OSAL_s32EventPost(_hEvSync, DP_U32_EVENT_UPDATE_POOL_FINISHED, OSAL_EN_EVENTMASK_OR);
   }break;
#ifdef DP_U32_POOL_ID_DPENDUSERMODE
   case dp_tclDatapoolMaster::eClientBankAction:
   { //synchronous call -> send ack
      vPerformBankAction((tU8)pMsg->u.au32Data[0], (eDpEndUserBankAction)pMsg->u.au32Data[1]);
      dp_tclDatapoolMaster::TDpPoolMsg oNewMsg = { dp_tclDatapoolMaster::eAckBankAction, { 0 } };
      if (!bPostMasterMessage(&oNewMsg))
      {
         ETG_TRACE_FATAL(("vDpMainThread(): failed to post message to master"));
      }
   }break;
   case dp_tclDatapoolMaster::eClientAllBankActionsReady:
   {
      ETG_TRACE_USR4(("vDpMainThread():: send DP_U32_EVENT_BANK_ACTION_FINISHED"));
      OSAL_s32EventPost(_hEvSync, DP_U32_EVENT_BANK_ACTION_FINISHED, OSAL_EN_EVENTMASK_OR);
   }break;
   case dp_tclDatapoolMaster::eClientEndUserCopy:
   { //synchronous call -> send ack
      vPerformCopyEndUser((tU8)pMsg->u.au32Data[0], (tU8)pMsg->u.au32Data[1]);
      dp_tclDatapoolMaster::TDpPoolMsg oNewMsg = { dp_tclDatapoolMaster::eAckEndUserCopy, { 0 } };
      if (!bPostMasterMessage(&oNewMsg))
      {
         ETG_TRACE_FATAL(("vDpMainThread(): failed to post message to master"));
      }
   }break;
   case dp_tclDatapoolMaster::eClientAllEndUserCopyReady:
   {
      ETG_TRACE_USR4(("vDpMainThread():: send DP_U32_EVENT_END_USER_COPY_FINISHED"));
      OSAL_s32EventPost(_hEvSync, DP_U32_EVENT_END_USER_COPY_FINISHED, OSAL_EN_EVENTMASK_OR);
   }break;
#endif
#ifdef DP_FEATURE_UNDEF_ELEMENT
   case dp_tclDatapoolMaster::eUndefElementShareCreate:
   { /*function pepare shm and copy element to shared memory */
      if (bSharedOwnUndefElementCreate(pMsg->u.tUndef.strPool, pMsg->u.tUndef.strElement))
      {  /*send message*/
         dp_tclDatapoolMaster::TDpPoolMsg VtMsg = { dp_tclDatapoolMaster::eUndefElementSharedCreated, { 0 } };
         if (bPostMessageToProcessLocatedData(pMsg->u.tUndef.strProcess, &VtMsg) == FALSE)
         {
            ETG_TRACE_FATAL(("vDpMainThread():: post message element eUndefElementShareCreateCreated fails"));
         }
      }
      else
      {
         ETG_TRACE_FATAL(("vDpMainThread():: element not shared: %s.", pMsg->u.tUndef.strElement));
      }
   }break;
   case dp_tclDatapoolMaster::eUndefElementShareDelete:
   {
      if (s32SharedOwnUndefElementDelete(pMsg->u.tUndef.strPool, pMsg->u.tUndef.strElement) == DP_S32_NO_ERR)
      {  /*send message*/
         dp_tclDatapoolMaster::TDpPoolMsg VtMsg = { dp_tclDatapoolMaster::eUndefElementSharedDeleteDone, { 0 } };
         if (bPostMessageToProcessLocatedData(pMsg->u.tUndef.strProcess, &VtMsg) == FALSE)
         {
            ETG_TRACE_FATAL(("vDpMainThread():: post message element eUndefElementSharedDeleteDone fails"));
         }
      }
      else
      {
         ETG_TRACE_FATAL(("vDpMainThread():: element not deleted: %s.", pMsg->u.tUndef.strElement));
      }
   }break;
   case dp_tclDatapoolMaster::eUndefElementSharedCreated:
   {/*send event */
      ETG_TRACE_USR4(("vDpMainThread():: post event DP_U32_EVENT_SHM_ELEMENT_CREATED"));
      OSAL_s32EventPost(_hEvSync, DP_U32_EVENT_SHM_ELEMENT_CREATED, OSAL_EN_EVENTMASK_OR);
   }break;
   case dp_tclDatapoolMaster::eUndefElementSharedAccessDone:
   {/*delete shared element*/
      vSharedOwnUndefElementDestroy(pMsg->u.tUndef.strPool, pMsg->u.tUndef.strElement);
   }break;
   case dp_tclDatapoolMaster::eUndefElementSharedDeleteDone:
   {/*send event */
      ETG_TRACE_USR4(("vDpMainThread():: post event DP_U32_EVENT_SHM_ELEMENT_DELETED"));
      OSAL_s32EventPost(_hEvSync, DP_U32_EVENT_SHM_ELEMENT_DELETED, OSAL_EN_EVENTMASK_OR);
   }break;
#endif
   default:
      ETG_TRACE_USR4(("vDpMainThread():: Unknown command in message: %d.", pMsg->eCmd));
      break;
   }
   return(VbTerminate);
}

tVoid dp_tclDpBase::vTimerHandling(tVoid)
{
   TDatapool::iterator posDp;

   (tVoid)DP_bEnterCritical(_hDpAccess);
   for (posDp = oDatapool.begin(); posDp != oDatapool.end(); ++posDp)
   {
      if (posDp->second.bWaiting4Store)
      {
         tU32 u32CurrentTime = OSAL_ClockGetElapsedTime();
         ETG_TRACE_USR4(("vDpMainThread(): someone is waiting to store some data in pool (cur: %d, next: %d, last: %d) (pool: '%s').", u32CurrentTime, posDp->second.u32NextStoreTime, posDp->second.u32LastStoreTime, posDp->second.strFileName));
         if ((u32CurrentTime > posDp->second.u32NextStoreTime) && (u32CurrentTime > DP_U32_STARTUP_STORING_DELAY))
         {
            ETG_TRACE_USR4(("vDpMainThread():: Store '%s' now.", posDp->second.strFileName));

            bExportPersData(posDp->first, posDp->second.strFileName, posDp->second.eLocation, posDp->second.u32PoolVersion, FALSE);
            posDp->second.u32LastStoreTime = u32CurrentTime;
            posDp->second.bWaiting4Store = FALSE;
            posDp->second.u32NextStoreTime = DP_U8_NOTHING_TO_STORE_TIME;
            posDp->second.bDelayedElementActive = FALSE;
         }
      }
      if (posDp->second.bDelayedElementActive)
      {
         //a delayed element is waiting to be stored -> search for this element
         tU32 u32CurrentTime = OSAL_ClockGetElapsedTime();
         TListDatapoolElements::iterator posElement;
         for (posElement = posDp->second.tDpElementList.begin(); posElement != posDp->second.tDpElementList.end(); ++posElement)
         {
            if (posElement->second.eGetStoringType() == dp_tclBaseElement::eDpStoringTypeDelayed)
            {
               if (u32CurrentTime > posElement->second.u32GetNextStoreTime())
               {
                  posDp->second.u32NextStoreTime = u32CurrentTime;
                  posDp->second.bWaiting4Store = TRUE;
                  ETG_TRACE_USR4(("vDpMainThread():: Store delayed element at %dms --> '%s'.", posDp->second.u32NextStoreTime, posElement->second.pcGetElementName()));
               }
            }
         }
      }
   }/*end for*/
   DP_bReleaseCritical(_hDpAccess);
}

// ****************************************************************************
// ********    methods triggered by master 
// ****************************************************************************
tVoid dp_tclDpBase::vPerformDefSet(tU32 u32DefSetType, tU8 Pu8EndUserOrBank)
{
   TDatapool::iterator               VposDp;
   TListDatapoolElements::iterator   VposElement;
   tBool                             VbPoolChanged;     //TRUE:  pool changed  => export 
   tBool                             VbCurrentPool;     //FALSE:  the current pool should not be set 

   /*check subpool list, if all changed pool loaded to process*/
   vCheckSubPoolInit(TRUE);
   /*set all pools to default*/
   for (VposDp = oDatapool.begin(); VposDp != oDatapool.end(); ++VposDp)
   {
      VbPoolChanged = FALSE;
      VbCurrentPool = TRUE;
#ifdef DP_U32_POOL_ID_DPENDUSERMODE
      //check if end user pool with bank
      if (VposDp->second.eModeEndUser == eDpModeEndUserMultiBank)
      { /*delete/change only the files*/
         dp_tclDpEndUser*    VpoMyDpEndUser = dp_tclDpEndUser::pGetInstance();
         VpoMyDpEndUser->bPerformDefSetEndUserBank(&VposDp->second.tDpElementList, Pu8EndUserOrBank, u32DefSetType, VposDp->second.strFileName, VposDp->second.eLocation, VposDp->second.u32PoolVersion);
      }
      /*check if end user pool*/
      if (VposDp->second.eModeEndUser != eDpModeEndUserNo)
      { //check if current pool should set to default
         dp_tclDpEndUser*    VpoMyDpEndUser = dp_tclDpEndUser::pGetInstance();
         VbCurrentPool = VpoMyDpEndUser->bPerformDefSetEndUserPool(&VposDp->second.tDpElementList, Pu8EndUserOrBank, u32DefSetType, VposDp->second.strFileName, VposDp->second.eLocation, VposDp->second.u32PoolVersion);
      }
#else
      OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(Pu8EndUserOrBank);
#endif
      /*for each element in the pool*/
      for (VposElement = VposDp->second.tDpElementList.begin(); VposElement != VposDp->second.tDpElementList.end(); ++VposElement)
      {  /*check type of default setting*/
         if (VposElement->second.u32GetDefSetMode() & u32DefSetType)
         {
            if (VbCurrentPool == TRUE)
            {
               if (bSetElementToDefault(&VposElement->second))
               {
                  ETG_TRACE_USR1(("vPerformDefSet(): Element has changed: '%s'!!", VposElement->second.pcGetElementName()));
                  vTraceElement(&VposElement->second, eDpElemWrite);
                  VbPoolChanged = TRUE;
               }
               else
               {
                  ETG_TRACE_USR1(("vPerformDefSet(): Element hasn't changed: '%s'!!", VposElement->second.pcGetElementName()));
                  vTraceElement(&VposElement->second, eDpElemWrite);
               }
            }
         }
      }/*end for*/
      if (VbPoolChanged == TRUE)
      {
         tU32 u32CurrentTime = OSAL_ClockGetElapsedTime();
         bExportPersData(VposDp->first, VposDp->second.strFileName, VposDp->second.eLocation, VposDp->second.u32PoolVersion, TRUE);
         VposDp->second.u32LastStoreTime = u32CurrentTime;
         VposDp->second.bWaiting4Store = FALSE;
         VposDp->second.u32NextStoreTime = DP_U8_NOTHING_TO_STORE_TIME;
         VposDp->second.bDelayedElementActive = FALSE;
      }
   }/*end for*/
}

#ifdef DP_FEATURE_UNDEF_ELEMENT
tBool dp_tclDpBase::bPerformDefSetUndefPool(tU32 u32PoolId)
{
   tBool VbInit = FALSE;
   TDpUndefPoolProperty::iterator VtDpPos = oDpUndefPoolProperty.find(u32PoolId);
   if (VtDpPos == oDpUndefPoolProperty.end())
   { //pool properties not exist; get name from ID 
      tDpUndefPoolProperty      VtPoolProperty;
      dp_tclDpUndef*            VpoMyDpUndef = dp_tclDpUndef::pGetInstance();
      ETG_TRACE_USR1(("bPerformDefSetUndefPool(): load properties of pool %08x now!!", u32PoolId));
      memset(&VtPoolProperty, 0, sizeof(tDpUndefPoolProperty));
      strncpy(VtPoolProperty.strProcessNameLocated, _szMyProcSpecName, DP_MAX_LEN_NAME_PROCESS);
      VtPoolProperty.strProcessNameLocated[DP_MAX_LEN_NAME_PROCESS - 1] = '\0';
      VtPoolProperty.u32PoolId = u32PoolId;
      if (VpoMyDpUndef->bGetPoolNameFromId(&VtPoolProperty) == TRUE)
      {
         tS32 Vs32ReturnCode = VpoMyDpUndef->s32GetPoolProperty(&VtPoolProperty, dp_tclDpUndef::eDpUndefPerformDefSet, NULL);
         if (Vs32ReturnCode == DP_S32_NO_ERR)
         {
            if (strncmp(VtPoolProperty.strProcessNameLocated, _szMyProcSpecName, DP_MAX_LEN_NAME_PROCESS) == 0)
            {
               oDpUndefPoolProperty[u32PoolId] = VtPoolProperty;
               VbInit = TRUE;
            }
         }
      }
   }
   return(VbInit);
}
#endif

tVoid dp_tclDpBase::vCalcPoolHashes()
{
   DP_NULL_POINTER_CHECK(_poPersMem);

   TDatapool::iterator posDp;
   for (posDp = oDatapool.begin(); posDp != oDatapool.end(); ++posDp)
   {
      if (posDp->second.bDefSet)
      {
         dp_tclDatapoolMaster::TDpPoolMsg oNewMsg = { dp_tclDatapoolMaster::eHashValue, { 0 } };
         if (TRUE == _poPersMem->bCalcHashForFile(posDp->second.strFileName, oNewMsg.u.tHash.tValue.u32Hash, posDp->second.eLocation, posDp->second.u32PoolVersion))
         {
            OSAL_szStringCopy(oNewMsg.u.tHash.strFile, posDp->second.strFileName);
            if (!bPostMasterMessage(&oNewMsg))
            {
               ETG_TRACE_FATAL(("vCalcPoolHashes(): failed to post message to master"));
            }
         }
      }
   }
}

tVoid dp_tclDpBase::vElementChanged(const tChar* strName)
{
   TDatapool::iterator posDp;
   (tVoid)DP_bEnterCritical(_hDpAccess);
   for (posDp = oDatapool.begin(); posDp != oDatapool.end(); ++posDp)
   {
      tU32 u32PoolId = posDp->first;
      TDatapool::iterator tDpPos = oDatapool.find(u32PoolId);
      if (tDpPos != oDatapool.end())
      {
         TListDatapoolElements::iterator posElement;
         for (posElement = oDatapool[u32PoolId].tDpElementList.begin(); posElement != oDatapool[u32PoolId].tDpElementList.end(); ++posElement)
         {
            if (0 == OSAL_s32StringNCompare(posElement->second.pcGetElementName(), strName, DP_MAX_LEN_NAME_ELEMENT))
            {
               posElement->second.vSendNotification();
            }
         }
      }
   }
   DP_bReleaseCritical(_hDpAccess);
}

tVoid dp_tclDpBase::vStoreNow()
{
   TDatapool::iterator posDp;
   ETG_TRACE_USR1(("vStoreNow():Extern trigger to store all pools now."));
   (tVoid)DP_bEnterCritical(_hDpAccess);
   for (posDp = oDatapool.begin(); posDp != oDatapool.end(); ++posDp)
   {
      tBool bStore = TRUE;
      ETG_TRACE_USR1(("vDpMainThread(): store pool '%s' now.", posDp->second.strFileName));


      if (!posDp->second.bElemChanged && ((posDp->first & 0x0000ffff) >= DP_U16_SUBPOOL_CODING_RANGE))
      {
         ETG_TRACE_USR1(("vDpMainThread(): Do not store pool %08x (Coding pool and no element has changed).", posDp->first));
         bStore = FALSE;
      }
      if (bStore)
      {
         bExportPersData(posDp->first, posDp->second.strFileName, posDp->second.eLocation, posDp->second.u32PoolVersion, TRUE);
         posDp->second.u32LastStoreTime = OSAL_ClockGetElapsedTime();
         posDp->second.bWaiting4Store = FALSE;
         posDp->second.u32NextStoreTime = DP_U8_NOTHING_TO_STORE_TIME;
      }
   }
   DP_bReleaseCritical(_hDpAccess);
}

#ifdef DP_U32_POOL_ID_DPENDUSERMODE
tVoid dp_tclDpBase::vPerformBankAction(tU8 Pu8Bank, eDpEndUserBankAction PeTypeAction)
{
   TDatapool::iterator           VposDp;
   TDpPoolSubPoolList::iterator  VpDpSub;
   /*check subpool list, if all changed pool loaded to process*/
   vCheckSubPoolInit(FALSE);
   //for all pools 
   (tVoid)DP_bEnterCritical(_hDpAccess);
   for (VpDpSub = _tListSubPools.begin(); VpDpSub != _tListSubPools.end(); ++VpDpSub)
   {
      VposDp = oDatapool.find(VpDpSub->u32SubPool);
      if (VposDp != oDatapool.end())
      {/*check if end user pool with banking*/
         if (VposDp->second.eModeEndUser == eDpModeEndUserMultiBank)
         {
            dp_tclDpEndUser*    VpoMyDpEndUser = dp_tclDpEndUser::pGetInstance();
            if (VpoMyDpEndUser)
            {
               if (PeTypeAction == eDPEndUserBankSave)
               {
                  ETG_TRACE_USR4(("vPerformBankAction(): type: eDPEndUserBankSave, bank:%d, filename:'%s'", Pu8Bank, VposDp->second.strFileName));
                  //add bank extension		    	    		 		 
                  tChar  VstrFileName[DP_MAX_LEN_FILE] = { 0 };
                  strncpy(VstrFileName, VposDp->second.strFileName, DP_MAX_LEN_FILE);
                  VstrFileName[DP_MAX_LEN_FILE - 1] = '\0';
                  VpoMyDpEndUser->vAddEndUserBankExt(Pu8Bank, VstrFileName);
                  //ETG_TRACE_USR4(("vPerformBankAction(): type eDPEndUserBankSave: file name '%s' ",VstrFileName));
                    //store pool
                  bExportPersData(VposDp->first, VstrFileName, VposDp->second.eLocation, VposDp->second.u32PoolVersion, TRUE, Pu8Bank);
                  //save used bank
                  VpoMyDpEndUser->s32SetBankUsed(Pu8Bank);
               }
               if (PeTypeAction == eDPEndUserBankLoad)
               {
                  ETG_TRACE_USR4(("vPerformBankAction(): type: eDPEndUserBankLoad, bank:%d, filename:'%s'", Pu8Bank, VposDp->second.strFileName));
                  //erase pool
                  oDatapool.erase(VpDpSub->u32SubPool);
                  //init pool  
                  bInitPool(VpDpSub->u32SubPool, Pu8Bank);
                  //set/save bank
                  VpoMyDpEndUser->s32SetBank(Pu8Bank);
                  //check if pool init/ available
                  VposDp = oDatapool.find(VpDpSub->u32SubPool);
                  if (VposDp != oDatapool.end())
                  {//set pool to change => so that the pool should bestored by shutdown or switch user
                     oDatapool[VpDpSub->u32SubPool].bElemChanged = TRUE;
                     ETG_TRACE_USR4(("eDPEndUserBankLoad(): pool '%08x' state bElemChanged '%d' ", VpDpSub->u32SubPool, oDatapool[VpDpSub->u32SubPool].bElemChanged));
                  }
               }
            }
         }
      }
   }
   DP_bReleaseCritical(_hDpAccess);
}

tVoid dp_tclDpBase::vPerformCopyEndUser(tU8 Pu8EndUserFrom, tU8 Pu8EndUserTo)
{
   ETG_TRACE_USR4(("vPerformCopyEndUser(): from EndUser '%d' to '%d'", Pu8EndUserFrom, Pu8EndUserTo));
   dp_tclDpEndUser*        VpoMyDpEndUser = dp_tclDpEndUser::pGetInstance();
   TDatapool::iterator     VposDp;
   if (VpoMyDpEndUser)
   {
      tU8 Vu8CurrentUser;
      if (VpoMyDpEndUser->s32GetEndUser(Vu8CurrentUser) == DP_S32_NO_ERR)
      {
         if (Vu8CurrentUser == Pu8EndUserFrom)
         { /*load all subpools to RAM */
            vCheckSubPoolInit(FALSE);
            /*for all end user pools; store to user */
            (tVoid)DP_bEnterCritical(_hDpAccess);
            for (VposDp = oDatapool.begin(); VposDp != oDatapool.end(); ++VposDp)
            { /*if end user pool*/
               if (VposDp->second.eModeEndUser != eDpModeEndUserNo)
               { /*store the current value of the pools to Pu8EndUserTo*/
                  tChar  VstrFileName[DP_MAX_LEN_FILE] = { 0 };
                  /*get the file name without extension */
                  VpoMyDpEndUser->vGetFileNameWithoutExt(VposDp->second.strFileName, VstrFileName);
                  /*get new name of the datastream*/
                  VpoMyDpEndUser->vAddExt(Pu8EndUserTo, VstrFileName);
                  ETG_TRACE_USR4(("vPerformCopyEndUser(): store to file '%s' ", VstrFileName));
                  bExportPersData(VposDp->first, VstrFileName, VposDp->second.eLocation, VposDp->second.u32PoolVersion, TRUE);
               }
            }
            DP_bReleaseCritical(_hDpAccess);
         }
         else
         { /*for all end user pools*/
            (tVoid)DP_bEnterCritical(_hDpAccess);
            for (VposDp = oDatapool.begin(); VposDp != oDatapool.end(); ++VposDp)
            { /*if end user pool*/
               if (VposDp->second.eModeEndUser != eDpModeEndUserNo)
               { /* copy file to Pu8EndUserTo*/
                  tChar  VstrFileName[DP_MAX_LEN_FILE] = { 0 };
                  /*check if new end user and the end user pool 'Pu8EndUserFrom' is load => then store to file, before copy */
                  if ((VposDp->second.u8EndUser != Vu8CurrentUser) && (VposDp->second.u8EndUser == Pu8EndUserFrom))
                  { /* store pool if changed */
                     ETG_TRACE_USR4(("vPerformCopyEndUser(): pool '%08x' state bElemChanged '%d' ", VposDp->first, VposDp->second.bElemChanged));
                     if (VposDp->second.bElemChanged == TRUE)
                     {
                        ETG_TRACE_USR4(("vPerformCopyEndUser(): new end user first save old pool '%08x' '%s' ", VposDp->first, VposDp->second.strFileName));
                        bExportPersData(VposDp->first, VposDp->second.strFileName, VposDp->second.eLocation, VposDp->second.u32PoolVersion, TRUE);
                        VposDp->second.u32LastStoreTime = OSAL_ClockGetElapsedTime();;
                        VposDp->second.bWaiting4Store = FALSE;
                        VposDp->second.u32NextStoreTime = DP_U8_NOTHING_TO_STORE_TIME;
                        VposDp->second.bDelayedElementActive = FALSE;
                     }
                  }
                  /*get the file name without extension */
                  VpoMyDpEndUser->vGetFileNameWithoutExt(VposDp->second.strFileName, VstrFileName);
                  /* call function of end user class, which reads the file and stores the file */
                  /* since default value for all end user equal => it is sufficient to copy the files */
                  VpoMyDpEndUser->s32CopyEndUserDataStream(Pu8EndUserFrom, Pu8EndUserTo, VstrFileName, VposDp->second.eLocation, VposDp->second.u32PoolVersion);
               }
            }
            DP_bReleaseCritical(_hDpAccess);
         }
      }
   }
}
#endif

// ****************************************************************************
// ********    methods triggered by user (synchronous via master)
// ****************************************************************************
tVoid dp_tclDpBase::vEnableElementForTrace(const tChar* strName)
{
   dp_tclBaseElement myDpElem("TrElements");
   if (strName != NULL)
   {
      dp_tclBaseElement myDpTraceElem(strName);
      ETG_TRACE_ERR(("vSetTraceConfiguration(): enable element '%40s' with hash 0x%08x.", strName, myDpTraceElem.s32GetHash()));
      if (DP_S32_NO_ERR == s32GetElement(DP_U32_POOL_ID_DPINTERNDATA, &myDpElem, DP_U32_POOL_ID_DPINTERNDATA >> 16))
      {
         if ((myDpElem.u32GetDataLength() % ((tU32)sizeof(tU32))) == 0)
         {
            tU32 u32Count = myDpElem.u32GetDataLength() / ((tU32)sizeof(tU32));
            ETG_TRACE_ERR(("vSetTraceConfiguration(): element 'TrElements' with %d entries found.", u32Count));
            tU32* pu32NewArray = NULL;
            try
            {
               pu32NewArray = new tU32[u32Count + 1];
            }
            catch (const std::bad_alloc&)
            {
               pu32NewArray = NULL;
               ETG_TRACE_ERR(("vEnableElementForTrace(): allocation failed %d elements for %s ", u32Count + 1, strName));
            }
            if (pu32NewArray)
            {
               (tVoid)memmove((tVoid*)pu32NewArray, myDpElem.pvGetData(), u32Count*sizeof(tU32));
               pu32NewArray[u32Count] = myDpTraceElem.s32GetHash();
               (tVoid)myDpElem.bFillData((tVoid*)pu32NewArray, ((tU32)sizeof(tU32))*(u32Count + 1));
               delete[] pu32NewArray;
               pu32NewArray = NULL;/*Bug 123144: Safe handling*/
               s32SetElement(DP_U32_POOL_ID_DPINTERNDATA, &myDpElem, DP_U32_POOL_ID_DPINTERNDATA >> 16);
            }
            _bListTraceElementsLoad = FALSE;
            vSetTraceConfiguration();
         }
      }
   }
   else
   {
      (tVoid)myDpElem.bFillData((tVoid*)NULL, 0);
      s32SetElement(DP_U32_POOL_ID_DPINTERNDATA, &myDpElem, DP_U32_POOL_ID_DPINTERNDATA >> 16);
   }
}

tS32 dp_tclDpBase::s32StoreNowSync(tU32 u32AccessId, tU32 u32Timeout)
{
   if ((u32AccessId == 0) || (u32AccessId == 0x0114)) //spm, fc_diagnosis
   {
      if (!DP_bEnterCritical(_hDpSemSrvIf, u32Timeout))
      { //waiting mask is set --> someone in this process is currently in a sync call
         ETG_TRACE_FATAL(("s32StoreNowSync(): Sync service call is currently in progress."));
         return DP_S32_ERR_BUSY;
      }
      //delete old existing events
      vWaitDeleteExistingEvents(DP_U32_EVENT_STORE_FINISHED);
      // send message
      dp_tclDatapoolMaster::TDpPoolMsg oMsg = { dp_tclDatapoolMaster::eStoreAllPools, { u32AccessId } };
      bPostMasterMessage(&oMsg);
      //wait event
      tS32 s32Ret = s32WaitSync(DP_U32_EVENT_STORE_FINISHED, u32Timeout);
      DP_bReleaseCritical(_hDpSemSrvIf);
      return s32Ret;
   }
   else
   {
      return DP_S32_ERR_ACCESS_RIGHT;
   }
}

tS32 dp_tclDpBase::s32SetDefault(tU32 u32AccessId, tU32 u32DefType, tU8 Pu8EndUserOrBank, tU32 u32Timeout)
{
   tBool VbAccess = FALSE;

   ETG_TRACE_USR4(("dp_tclDpBase::s32SetDefault() u32DefType=%d ", u32DefType));
   //check access
   if ((u32AccessId == 0) || (u32AccessId == 0x0114)) //spm, fc_diagnosis
   {
      if ((u32DefType != DP_DEFSET_END_USER) && (u32DefType != DP_DEFSET_END_USER_BANK))
      {
         VbAccess = TRUE;
      }
   }
#ifdef DP_U32_POOL_ID_DPENDUSERMODE
   if (u32AccessId == (tU16)(DP_ACCESS_ID_END_USER_APP)) // application with end user mode
   {
      if ((u32DefType == DP_DEFSET_END_USER) || (u32DefType == DP_DEFSET_END_USER_BANK))
      {
         VbAccess = TRUE;
      }
   }
#endif
   if (VbAccess == TRUE)
   {
      if (!DP_bEnterCritical(_hDpSemSrvIf, u32Timeout))
      { //waiting mask is set --> someone in this process is currently in a sync call
         ETG_TRACE_FATAL(("s32SetDefault(): Sync service call is currently in progress."));
         return DP_S32_ERR_BUSY;
      }
      //delete old existing events
      vWaitDeleteExistingEvents(DP_U32_EVENT_DEFSET_FINISHED);
      //post message
      dp_tclDatapoolMaster::TDpPoolMsg oMsg = { dp_tclDatapoolMaster::eDefSet, { 0 } };
      oMsg.u.au32Data[0] = u32DefType;
      oMsg.u.au32Data[1] = (tU32)Pu8EndUserOrBank;
      bPostMasterMessage(&oMsg);
      //wait  
      tS32 s32Ret = s32WaitSync(DP_U32_EVENT_DEFSET_FINISHED, u32Timeout);
#ifdef DP_U32_POOL_ID_DPENDUSERMODE
      //if factory default setting= set end user to default
      if (((u32DefType == DP_DEFSET_TEF) || (u32DefType == DP_DEFSET_USER) || (u32DefType == DP_DEFSET_END_USER)))
      {
         dp_tclDpEndUser*    VpoMyDpEndUser = dp_tclDpEndUser::pGetInstance();
         if (VpoMyDpEndUser)
         {
            if (u32DefType == DP_DEFSET_END_USER)
            {
               VpoMyDpEndUser->vSetBankForEndUserToDefault(Pu8EndUserOrBank);
            }
            else
            {
               VpoMyDpEndUser->vSetEndUserModeToDefault();
            }
         }
      }
#endif
      DP_bReleaseCritical(_hDpSemSrvIf);
      return s32Ret;
   }
   else
   {
      return DP_S32_ERR_ACCESS_RIGHT;
   }
}

tS32 dp_tclDpBase::s32SetConfig(tU32 u32PoolId, tU32 u32AccessId, tU32 u32Timeout)
{
   tS32 Vs32Return = DP_S32_NO_ERR;
   if ((u32AccessId == 0) || (u32AccessId == 0x0114)) //spm, fc_diagnosis
   { //check u32PoolId is coding range 
      tU16 u16SubpoolId = (u32PoolId & 0x0000ffff);
      if (u16SubpoolId >= DP_U16_SUBPOOL_CODING_RANGE)
      {
         if (!DP_bEnterCritical(_hDpSemSrvIf, u32Timeout))
         { //waiting mask is set --> someone in this process is currently in a sync call
            ETG_TRACE_FATAL(("s32SetConfig(): Sync service call is currently in progress."));
            Vs32Return = DP_S32_ERR_BUSY;
         }
         else
         { // check coding data changed
            if (oDatapool[u32PoolId].bElemChanged == TRUE)
            {// first save the pool
               bExportPersData(u32PoolId, oDatapool[u32PoolId].strFileName, oDatapool[u32PoolId].eLocation, oDatapool[u32PoolId].u32PoolVersion, TRUE);
               oDatapool[u32PoolId].u32LastStoreTime = OSAL_ClockGetElapsedTime();;
               oDatapool[u32PoolId].bWaiting4Store = FALSE;
               oDatapool[u32PoolId].u32NextStoreTime = DP_U8_NOTHING_TO_STORE_TIME;
            }
            //delete old existing events
            vWaitDeleteExistingEvents(DP_U32_EVENT_UPDATE_POOL_FINISHED);
            //post message
            dp_tclDatapoolMaster::TDpPoolMsg oMsg = { dp_tclDatapoolMaster::ePoolChanged, { u32PoolId } };
            bPostMasterMessage(&oMsg);
            // wait 
            Vs32Return = s32WaitSync(DP_U32_EVENT_UPDATE_POOL_FINISHED, u32Timeout);
            DP_bReleaseCritical(_hDpSemSrvIf);
         }
      }/*end if*/
      else
      {
         ETG_TRACE_FATAL(("s32SetConfig(): pool not defined for coding"));
         Vs32Return = DP_S32_ERR_ACCESS_RIGHT;
      }
   }/*end if no spm or diagnosis*/
   else
   {
      ETG_TRACE_FATAL(("s32SetConfig(): no access rights"));
      Vs32Return = DP_S32_ERR_ACCESS_RIGHT;
   }
   return(Vs32Return);
}

tS32 dp_tclDpBase::s32Lock(tU32 u32AccessId, tU32 u32LockType, tU32 u32Timeout)
{
   tBool VbAccess = FALSE;
   ETG_TRACE_USR4(("s32Lock(): %08x", u32AccessId));
   //check access
   if ((u32AccessId == 0) || (u32AccessId == 0x0114)) //spm, fc_diagnosis
   {
      if (u32LockType != DP_U32_LOCK_MODE_END_USER)
      {
         VbAccess = TRUE;
      }
   }

#ifdef DP_U32_POOL_ID_DPENDUSERMODE
   if (u32AccessId == (tU16)(DP_ACCESS_ID_END_USER_APP)) // application with end user mode
   {
      if (u32LockType == DP_U32_LOCK_MODE_END_USER)
      {
         VbAccess = TRUE;
      }
   }
#endif
   if (VbAccess == TRUE)
   {
      if (!DP_bEnterCritical(_hDpSemSrvIf, u32Timeout))
      { //waiting mask is set --> someone in this process is currently in a sync call
         ETG_TRACE_FATAL(("s32Lock(): Sync service call is currently in progress."));
         return DP_S32_ERR_BUSY;
      }
      //delete old existing events
      vWaitDeleteExistingEvents(DP_U32_EVENT_LOCKED_FINISHED);
      //prepare message
      dp_tclDatapoolMaster::TDpPoolMsg oMsg = { dp_tclDatapoolMaster::eLockPool, { 0 } };
      oMsg.u.au32Data[0] = u32LockType;
      oMsg.u.au32Data[1] = u32AccessId;
      //post message
      bPostMasterMessage(&oMsg);
      //wait lock finished
      tS32 s32Ret = s32WaitSync(DP_U32_EVENT_LOCKED_FINISHED, u32Timeout);
      DP_bReleaseCritical(_hDpSemSrvIf);
      return s32Ret;
   }
   else
   {
      return DP_S32_ERR_ACCESS_RIGHT;
   }
}

tS32 dp_tclDpBase::s32Unlock(tU32 u32AccessId, tU32 u32Timeout)
{
   ETG_TRACE_USR4(("s32Unlock(): %08x", u32AccessId));
   if ((u32AccessId == 0) || (u32AccessId == 0x0114)                //spm, fc_diagnosis
#ifdef DP_U32_POOL_ID_DPENDUSERMODE
      || (u32AccessId == (tU16)(DP_ACCESS_ID_END_USER_APP))
#endif
      ) //application with end user mode
   {
      if (!DP_bEnterCritical(_hDpSemSrvIf, u32Timeout))
      { //waiting mask is set --> someone in this process is currently in a sync call
         ETG_TRACE_FATAL(("s32Unlock(): Sync service call is currently in progress."));
         return DP_S32_ERR_BUSY;
      }
      //delete old existing events
      vWaitDeleteExistingEvents(DP_U32_EVENT_UNLOCKED_FINISHED);
      //prepare message
      dp_tclDatapoolMaster::TDpPoolMsg oMsg = { dp_tclDatapoolMaster::eUnlockPool, { u32AccessId } };
      bPostMasterMessage(&oMsg);
      //wait
      tS32 s32Ret = s32WaitSync(DP_U32_EVENT_UNLOCKED_FINISHED, u32Timeout);
      DP_bReleaseCritical(_hDpSemSrvIf);
      return s32Ret;
   }
   else
   {
      return DP_S32_ERR_ACCESS_RIGHT;
   }
}

tS32 dp_tclDpBase::s32CalcHash(tU32* pu32Data, tU32 u32DataLen, tU32 u32AccessId, tU32 u32Timeout)
{
   if ((u32AccessId == 0) || (u32AccessId == 0x0114)) //spm, fc_diagnosis
   {
      if (u32DataLen < DP_MASTER_HASH_COUNT)
      {
         return DP_S32_ERR_WR_BUF_LEN;
      }
      if (!DP_bEnterCritical(_hDpSemSrvIf, u32Timeout))
      { //waiting mask is set --> someone in this process is currently in a sync call
         ETG_TRACE_FATAL(("s32CalcHash(): Sync service call is currently in progress."));
         return DP_S32_ERR_BUSY;
      }

      //delete old existing events
      vWaitDeleteExistingEvents(DP_U32_EVENT_HASH_FINISHED);
      //post message
      dp_tclDatapoolMaster::TDpPoolMsg oMsg = { dp_tclDatapoolMaster::eCalcHash, { 0 } };
      bPostMasterMessage(&oMsg);
      //wait event
      tS32 s32Ret = s32WaitSync(DP_U32_EVENT_HASH_FINISHED, u32Timeout);
      if (s32Ret >= 0)
      {
         memmove((tU8*)pu32Data, (tU8*)_tCurentHash, DP_MASTER_HASH_COUNT * sizeof(tU32));
         s32Ret = DP_MASTER_HASH_COUNT;
      }
      DP_bReleaseCritical(_hDpSemSrvIf);
      return s32Ret;
   }
   else
   {
      return DP_S32_ERR_ACCESS_RIGHT;
   }
}

#ifdef DP_U32_POOL_ID_DPENDUSERMODE
tS32  dp_tclDpBase::s32EndUserBank(tU8 Pu8Bank, tU16 Pu16AccessId, eDpEndUserBankAction PeTypeAction, tU32 Pu32Timeout)
{
   tS32 Vs32Return = DP_S32_NO_ERR;
   if (Pu16AccessId == (tU16)(DP_ACCESS_ID_END_USER_APP))
   { //check u32PoolId is coding range 
      if (!DP_bEnterCritical(_hDpSemSrvIf, Pu32Timeout))
      { //waiting mask is set --> someone in this process is currently in a sync call
         ETG_TRACE_FATAL(("s32EndUserBank(): Sync service call is currently in progress."));
         Vs32Return = DP_S32_ERR_BUSY;
      }
      else
      { //delete old existing events
         vWaitDeleteExistingEvents(DP_U32_EVENT_BANK_ACTION_FINISHED);
         //post message
         dp_tclDatapoolMaster::TDpPoolMsg oMsg = { dp_tclDatapoolMaster::eBankAction, { 0 } };
         oMsg.u.au32Data[0] = (tU32)Pu8Bank;
         oMsg.u.au32Data[1] = (tU32)PeTypeAction;
         bPostMasterMessage(&oMsg);
         // wait 
         Vs32Return = s32WaitSync(DP_U32_EVENT_BANK_ACTION_FINISHED, Pu32Timeout);
         DP_bReleaseCritical(_hDpSemSrvIf);
      }
   }/*end if no spm or diagnosis*/
   else
   {
      ETG_TRACE_FATAL(("s32SetConfig(): no access rights"));
      Vs32Return = DP_S32_ERR_ACCESS_RIGHT;
   }
   return(Vs32Return);
}

tS32  dp_tclDpBase::s32CopyEndUser(tU8 u8EndUserFrom, tU8 u8EndUserTo, tU16 Pu16AccessId, tU32 Pu32Timeout)
{
   tS32 Vs32Return = DP_S32_NO_ERR;
   if (Pu16AccessId == (tU16)(DP_ACCESS_ID_END_USER_APP))
   { /*get current user*/
      dp_tclDpEndUser*    VpoMyDpEndUser = dp_tclDpEndUser::pGetInstance();
      if (VpoMyDpEndUser)
      {
         tU8 Vu8CurrentUser;
         if (VpoMyDpEndUser->s32GetEndUser(Vu8CurrentUser) == DP_S32_NO_ERR)
         { /*copy to the current end user not allowed */
            if (Vu8CurrentUser == u8EndUserTo)
            {
               ETG_TRACE_USR4(("s32CopyEndUser(): copy to the current end user not allowed: u8EndUserTo:%d, current end user:%d", u8EndUserTo, Vu8CurrentUser));
               Vs32Return = DP_S32_ERR_NOT_ALLOWED;
            }
            else
            {
               if (!DP_bEnterCritical(_hDpSemSrvIf, Pu32Timeout))
               { //waiting mask is set --> someone in this process is currently in a sync call
                  ETG_TRACE_FATAL(("s32CopyEndUser(): Sync service call is currently in progress."));
                  Vs32Return = DP_S32_ERR_BUSY;
               }
               else
               { //delete old existing events
                  vWaitDeleteExistingEvents(DP_U32_EVENT_END_USER_COPY_FINISHED);
                  //post message
                  dp_tclDatapoolMaster::TDpPoolMsg oMsg = { dp_tclDatapoolMaster::eEndUserCopy, { 0 } };
                  oMsg.u.au32Data[0] = (tU32)u8EndUserFrom;
                  oMsg.u.au32Data[1] = (tU32)u8EndUserTo;
                  bPostMasterMessage(&oMsg);
                  // wait 
                  Vs32Return = s32WaitSync(DP_U32_EVENT_END_USER_COPY_FINISHED, Pu32Timeout);
                  DP_bReleaseCritical(_hDpSemSrvIf);
               }
            }
         }
      }
   }
   else
   {
      ETG_TRACE_FATAL(("s32CopyEndUser(): no access rights"));
      Vs32Return = DP_S32_ERR_ACCESS_RIGHT;
   }
   return(Vs32Return);
}
#endif

// ****************************************************************************
// ********    methods triggered by user 
// ********    (asynchronous via master -> fire and return)
// ****************************************************************************

tVoid dp_tclDpBase::vHistoryTrace() {
   dp_tclDatapoolMaster::TDpPoolMsg oMsg = { dp_tclDatapoolMaster::eTraceHistory, { 0 } };
   bPostMasterMessage(&oMsg);
}

tVoid dp_tclDpBase::vTracePool(tU32 u32PoolId) {
   dp_tclDatapoolMaster::TDpPoolMsg oMsg = { dp_tclDatapoolMaster::eTracePools, { u32PoolId } };
   bPostMasterMessage(&oMsg);
}

tVoid dp_tclDpBase::vTraceElementByName(const tChar* strName) {
   dp_tclDatapoolMaster::TDpPoolMsg oMsg = { dp_tclDatapoolMaster::eTraceElement, { 0 } };
   OSAL_szStringNCopy(oMsg.u.strData, strName, DP_MAX_LEN_NAME_ELEMENT);
   oMsg.u.strData[DP_MAX_LEN_NAME_ELEMENT - 1] = '\0';
   bPostMasterMessage(&oMsg);
}

// ****************************************************************************
// ********    methods triggered by user (intern)
// ****************************************************************************

tBool dp_tclDpBase::bAddNotification(tU32 u32PoolId, const tChar* strElementName, tVoid* pvCallback, tU32 u32Event, tU16 u16AccessId)
{
   tBool   VbReturn = FALSE;
   (tVoid)u16AccessId;
   (tVoid)DP_bEnterCritical(_hDpAccess);
   TDatapool::iterator tDpPos = oDatapool.find(u32PoolId);
   if (tDpPos != oDatapool.end())
   {
      TListDatapoolElements::iterator posElement;
      for (posElement = oDatapool[u32PoolId].tDpElementList.begin(); posElement != oDatapool[u32PoolId].tDpElementList.end(); ++posElement)
      {
         if (strcmp(strElementName, posElement->second.pcGetElementName()) == 0)
         {
            dp_tclBaseElement::TFuncNotEntry tElement = { (DP_tpvDataUpdateCallback)pvCallback, u32Event }; //lint !e611: Suspicious cast --> convert void pointer to function pointer
            VbReturn = posElement->second.bAddNotification(tElement);
         }
      }
   }
   DP_bReleaseCritical(_hDpAccess);
   return(VbReturn);
}

tVoid dp_tclDpBase::vRemoveNotification(tU32 u32PoolId, const tChar* strElementName, tVoid* pvCallback, tU32 u32Event, tU16 u16AccessId)
{
   (tVoid)u16AccessId;
   (tVoid)DP_bEnterCritical(_hDpAccess);
   TDatapool::iterator tDpPos = oDatapool.find(u32PoolId);
   if (tDpPos != oDatapool.end())
   {
      TListDatapoolElements::iterator posElement;
      for (posElement = oDatapool[u32PoolId].tDpElementList.begin(); posElement != oDatapool[u32PoolId].tDpElementList.end(); ++posElement)
      {
         if (strcmp(strElementName, posElement->second.pcGetElementName()) == 0)
         {
            dp_tclBaseElement::TFuncNotEntry tElement = { (DP_tpvDataUpdateCallback)pvCallback, u32Event }; //lint !e611: Suspicious cast --> convert void pointer to function pointer
            posElement->second.bRemoveNotification(tElement);
         }
      }
   }
   DP_bReleaseCritical(_hDpAccess);
}

tS32  dp_tclDpBase::s32StorePoolNow(tU32 Pu32PoolId, tU32 Pu32AccessId)
{
   tS32   Vs32Return = DP_S32_NO_ERR;
   tBool  VbOwner = (Pu32AccessId << 16) == (Pu32PoolId & 0xffff0000);

   ETG_TRACE_USR4(("s32StorePoolNow(): call function s32StorePoolNow() for PoolId '%x'", Pu32PoolId));
   //check access
   if (VbOwner)
   {
      (tVoid)DP_bEnterCritical(_hDpAccess);
      //check if pool exist in RAM
      TDatapool::iterator tDpPos = oDatapool.find(Pu32PoolId);
      if ((tDpPos != oDatapool.end()))
      {// check if pool changed
         if (oDatapool[Pu32PoolId].bElemChanged == TRUE)
         {// save the pool
            if (bExportPersData(Pu32PoolId, oDatapool[Pu32PoolId].strFileName, oDatapool[Pu32PoolId].eLocation, oDatapool[Pu32PoolId].u32PoolVersion, TRUE) == TRUE)
            {
               oDatapool[Pu32PoolId].u32LastStoreTime = OSAL_ClockGetElapsedTime();;
               oDatapool[Pu32PoolId].bWaiting4Store = FALSE;
               oDatapool[Pu32PoolId].u32NextStoreTime = DP_U8_NOTHING_TO_STORE_TIME;
            }
            else
               Vs32Return = DP_S32_ERR_UNKNOWN;
         }
         else
         {
            ETG_TRACE_USR4(("s32StorePoolNow(): pool '%x' not changed", Pu32PoolId));
         }
      }
      else
      {
         ETG_TRACE_USR4(("s32StorePoolNow(): pool '%x' not load", Pu32PoolId));
      }
      DP_bReleaseCritical(_hDpAccess);
   }
   else
   {
      Vs32Return = DP_S32_ERR_ACCESS_RIGHT;
   }
   ETG_TRACE_USR4(("s32StorePoolNow(): function return %d ", Vs32Return));
   return(Vs32Return);
}

#ifdef DP_FEATURE_UNDEF_ELEMENT
// ****************************************************************************
// ********    methods for shared data
// ****************************************************************************
tS32    dp_tclDpBase::s32SharedUndefElementAccess(tDpUndefPoolProperty*  PptPoolProperty, const tChar* PpstrElementName, tBool PbAccessRead, tU8* Ppu8Buffer, tS32 Ps32Size)
{
   tS32                    Vs32Ret = DP_S32_NO_ERR;
   dp_tclDpUndef*          VpoMyDpUndef = dp_tclDpUndef::pGetInstance();

   if (!DP_bEnterCritical(_hDpAccess, DP_U32_WAIT_TIMEOUT_SHARED_DATA))
   { //waiting mask is set --> someone in this process is currently in a sync call
      ETG_TRACE_FATAL(("dp_tclDpBase::s32SharedUndefElementAccess(): Sync service call is currently in progress."));
      Vs32Ret = DP_S32_ERR_BUSY;
   }
   else
   {  //delete old existing events
      vWaitDeleteExistingEvents(DP_U32_EVENT_SHM_ELEMENT_CREATED);
      //send message
      dp_tclDatapoolMaster::TDpPoolMsg  VtMsg = { dp_tclDatapoolMaster::eUndefElementShareCreate, { 0 } };
      OSAL_szStringNCopy(VtMsg.u.tUndef.strElement, PpstrElementName, DP_MAX_LEN_NAME_ELEMENT);
      VtMsg.u.tUndef.strElement[DP_MAX_LEN_NAME_ELEMENT - 1] = '\0';
      OSAL_szStringNCopy(VtMsg.u.tUndef.strPool, PptPoolProperty->strPoolName, DP_MAX_LEN_NAME_UNDEFINED_POOLS);
      VtMsg.u.tUndef.strPool[DP_MAX_LEN_NAME_UNDEFINED_POOLS - 1] = '\0';
      OSAL_szStringNCopy(VtMsg.u.tUndef.strProcess, _szMyProcSpecName, DP_MAX_LEN_NAME_PROCESS);
      VtMsg.u.tUndef.strProcess[DP_MAX_LEN_NAME_PROCESS - 1] = '\0';
      if (bPostMessageToProcessLocatedData(PptPoolProperty->strProcessNameLocated, &VtMsg) == FALSE)
      {
         Vs32Ret = DP_S32_ERR_UNKNOWN;
      }
      else
      { /*wait for answer*/
         Vs32Ret = s32WaitSync(DP_U32_EVENT_SHM_ELEMENT_CREATED, DP_U32_WAIT_TIMEOUT_SHARED_DATA);
         if (Vs32Ret < 0)
         {
            ETG_TRACE_FATAL(("dp_tclDpBase::s32SharedUndefElementAccess(): s32WaitSync fails"));
         }
         else
         { //lock
            VpoMyDpUndef->vLockSemShmElement();
            if (PbAccessRead == TRUE)
            {
               Vs32Ret = VpoMyDpUndef->s32GetShmElement(PptPoolProperty->strPoolName, PpstrElementName, (tU32)Ps32Size, (void*)Ppu8Buffer);
            }
            else
            {
               Vs32Ret = VpoMyDpUndef->s32SetShmElement(PptPoolProperty->strPoolName, PpstrElementName, (tU32)Ps32Size, (void*)Ppu8Buffer);
            }
            //unlock
            VpoMyDpUndef->vUnLockSemShmElement();
         }
         /*send message done */
         dp_tclDatapoolMaster::TDpPoolMsg  VtMsgDone = { dp_tclDatapoolMaster::eUndefElementSharedAccessDone, { 0 } };
         OSAL_szStringNCopy(VtMsgDone.u.tUndef.strElement, PpstrElementName, DP_MAX_LEN_NAME_ELEMENT);
         VtMsgDone.u.tUndef.strElement[DP_MAX_LEN_NAME_ELEMENT - 1] = '\0';
         OSAL_szStringNCopy(VtMsgDone.u.tUndef.strPool, PptPoolProperty->strPoolName, DP_MAX_LEN_NAME_UNDEFINED_POOLS);
         VtMsgDone.u.tUndef.strPool[DP_MAX_LEN_NAME_UNDEFINED_POOLS - 1] = '\0';
         if (bPostMessageToProcessLocatedData(PptPoolProperty->strProcessNameLocated, &VtMsgDone) == FALSE)
         {
            Vs32Ret = DP_S32_ERR_UNKNOWN;
         }
      }
      DP_bReleaseCritical(_hDpAccess);
   }
   return(Vs32Ret);
}

tS32    dp_tclDpBase::s32SharedUndefElementDelete(tDpUndefPoolProperty*  PptPoolProperty, const tChar* PpstrElementName)
{
   tS32                    Vs32Ret = DP_S32_NO_ERR;

   if (!DP_bEnterCritical(_hDpAccess, DP_U32_WAIT_TIMEOUT_SHARED_DATA))
   { //waiting mask is set --> someone in this process is currently in a sync call
      ETG_TRACE_FATAL(("dp_tclDpBase::s32SharedUndefElementDelete(): Sync service call is currently in progress."));
      Vs32Ret = DP_S32_ERR_BUSY;
   }
   else
   {  //delete old existing events
      vWaitDeleteExistingEvents(DP_U32_EVENT_SHM_ELEMENT_DELETED);
      //send message
      dp_tclDatapoolMaster::TDpPoolMsg  VtMsg = { dp_tclDatapoolMaster::eUndefElementShareDelete, { 0 } };
      OSAL_szStringNCopy(VtMsg.u.tUndef.strElement, PpstrElementName, DP_MAX_LEN_NAME_ELEMENT);
      VtMsg.u.tUndef.strElement[DP_MAX_LEN_NAME_ELEMENT - 1] = '\0';
      OSAL_szStringNCopy(VtMsg.u.tUndef.strPool, PptPoolProperty->strPoolName, DP_MAX_LEN_NAME_UNDEFINED_POOLS);
      VtMsg.u.tUndef.strPool[DP_MAX_LEN_NAME_UNDEFINED_POOLS - 1] = '\0';
      OSAL_szStringNCopy(VtMsg.u.tUndef.strProcess, _szMyProcSpecName, DP_MAX_LEN_NAME_PROCESS);
      VtMsg.u.tUndef.strProcess[DP_MAX_LEN_NAME_PROCESS - 1] = '\0';
      if (bPostMessageToProcessLocatedData(PptPoolProperty->strProcessNameLocated, &VtMsg) == FALSE)
      {
         Vs32Ret = DP_S32_ERR_UNKNOWN;
      }
      else
      { /*wait for answer*/
         Vs32Ret = s32WaitSync(DP_U32_EVENT_SHM_ELEMENT_DELETED, DP_U32_WAIT_TIMEOUT_SHARED_DATA);
         if (Vs32Ret < 0)
         {
            ETG_TRACE_FATAL(("dp_tclDpBase::s32SharedUndefElementAccess(): s32WaitSync fails"));
         }
      }
      DP_bReleaseCritical(_hDpAccess);
   }
   return(Vs32Ret);
}

tBool   dp_tclDpBase::bSharedOwnUndefElementCreate(const tChar* PpstrPoolName, const tChar* PpstrElementName)
{
   tBool                   VbReturn = FALSE;
   tS32                    Vs32ReturnCode;
   tDpUndefPoolProperty    VtPoolProperty;
   dp_tclDpUndef*          VpoMyDpUndef = dp_tclDpUndef::pGetInstance();
   dp_tclBaseElement       VDpElem(PpstrElementName, 0); //hash calc in constructor  

   ETG_TRACE_USR4(("bSharedOwnUndefElementCreate(): prepare element '%s' for shared access", PpstrElementName));
   memset(&VtPoolProperty, 0, sizeof(tDpUndefPoolProperty));
   strncpy(VtPoolProperty.strPoolName, PpstrPoolName, DP_MAX_LEN_NAME_UNDEFINED_POOLS);
   VtPoolProperty.strPoolName[DP_MAX_LEN_NAME_UNDEFINED_POOLS - 1] = '\0';
   strncpy(VtPoolProperty.strProcessNameLocated, _szMyProcSpecName, DP_MAX_LEN_NAME_PROCESS);
   VtPoolProperty.strProcessNameLocated[DP_MAX_LEN_NAME_PROCESS - 1] = '\0';
   Vs32ReturnCode = VpoMyDpUndef->s32GetPoolProperty(&VtPoolProperty, dp_tclDpUndef::eDpUndefShared, &VDpElem);
   if (Vs32ReturnCode == DP_S32_NO_ERR)
   { /* pool should located in actual process*/
      if (strncmp(VtPoolProperty.strProcessNameLocated, _szMyProcSpecName, DP_MAX_LEN_NAME_PROCESS) == 0)
      { /* check if pool exist in the process*/
         (tVoid)DP_bEnterCritical(_hDpAccess);
         TDpUndefPoolProperty::iterator VtDpPos = oDpUndefPoolProperty.find(VtPoolProperty.u32PoolId);
         if (VtDpPos == oDpUndefPoolProperty.end())
         { /*add new property*/
            oDpUndefPoolProperty[VtPoolProperty.u32PoolId] = VtPoolProperty;
         }
         DP_bReleaseCritical(_hDpAccess);
         Vs32ReturnCode = s32GetElement(VtPoolProperty.u32PoolId, &VDpElem, (tU16)(VtPoolProperty.u32PoolId >> 16));
         if (Vs32ReturnCode == DP_S32_NO_ERR)
         { //create shared  element
            VpoMyDpUndef->vLockSemShmElement();
            Vs32ReturnCode = VpoMyDpUndef->s32CreateShmElement(PpstrPoolName, PpstrElementName, VDpElem.u32GetDataLength(), VDpElem.pvGetData());
            VpoMyDpUndef->vUnLockSemShmElement();
            if (Vs32ReturnCode == VDpElem.u32GetDataLength())
            {
               VbReturn = TRUE;
            }
         }
      }
   }
   if (VbReturn == FALSE)
   {
      ETG_TRACE_ERR(("bSharedOwnUndefElementCreate(): could not shared element '%s'", PpstrElementName));
   }
   return(VbReturn);
}

void    dp_tclDpBase::vSharedOwnUndefElementDestroy(const tChar* PpstrPoolName, const tChar* PpstrElementName)
{
   tS32                    Vs32ReturnCode;
   tDpUndefPoolProperty    VtPoolProperty;
   dp_tclDpUndef*          VpoMyDpUndef = dp_tclDpUndef::pGetInstance();
   dp_tclBaseElement       VDpElem(PpstrElementName, 0); //hash calc in constructor  

   ETG_TRACE_USR4(("vSharedOwnUndefElementDestroy(): destroy element '%s' for shared access", PpstrElementName));
   memset(&VtPoolProperty, 0, sizeof(tDpUndefPoolProperty));
   strncpy(VtPoolProperty.strPoolName, PpstrPoolName, DP_MAX_LEN_NAME_UNDEFINED_POOLS);
   VtPoolProperty.strPoolName[DP_MAX_LEN_NAME_UNDEFINED_POOLS - 1] = '\0';
   strncpy(VtPoolProperty.strProcessNameLocated, _szMyProcSpecName, DP_MAX_LEN_NAME_PROCESS);
   VtPoolProperty.strProcessNameLocated[DP_MAX_LEN_NAME_PROCESS - 1] = '\0';
   Vs32ReturnCode = VpoMyDpUndef->s32GetPoolProperty(&VtPoolProperty, dp_tclDpUndef::eDpUndefShared, &VDpElem);
   if (Vs32ReturnCode == DP_S32_NO_ERR)
   { /* pool should located in actual process*/
      if (strncmp(VtPoolProperty.strProcessNameLocated, _szMyProcSpecName, DP_MAX_LEN_NAME_PROCESS) == 0)
      { /* check if pool exist in the process*/
         (tVoid)DP_bEnterCritical(_hDpAccess);
         TDpUndefPoolProperty::iterator VtDpPos = oDpUndefPoolProperty.find(VtPoolProperty.u32PoolId);
         if (VtDpPos == oDpUndefPoolProperty.end())
         { /*add new property*/
            oDpUndefPoolProperty[VtPoolProperty.u32PoolId] = VtPoolProperty;
         }
         DP_bReleaseCritical(_hDpAccess);
         //lock
         VpoMyDpUndef->vLockSemShmElement();
         //get data
         tU8* VpBuffer = NULL;
         Vs32ReturnCode = VpoMyDpUndef->s32DecAccessCountAndGetDataShmElement(PpstrPoolName, PpstrElementName, &VpBuffer);
         if (Vs32ReturnCode > 0)
         { //fill data to element
            (tVoid)VDpElem.bFillData((tVoid*)VpBuffer, Vs32ReturnCode);
            delete[] VpBuffer;
            VpBuffer = NULL;/*Bug 123144: Safe handling*/
          //set data
            Vs32ReturnCode = s32SetElement(VtPoolProperty.u32PoolId, &VDpElem, (tU16)(VtPoolProperty.u32PoolId >> 16));
            if (Vs32ReturnCode < DP_S32_NO_ERR)
            {
               ETG_TRACE_FATAL(("vSharedOwnUndefElementDestroy(): destroy element '%s' could not copy to static pool", PpstrElementName));
            }
            //delete shared element
            if ((VpoMyDpUndef->bDeleteShmElement(PpstrPoolName, PpstrElementName)) == TRUE)
            {
               ETG_TRACE_USR4(("vSharedOwnUndefElementDestroy(): destroy element '%s' success", PpstrElementName));
            }
         }
         //unlock
         VpoMyDpUndef->vUnLockSemShmElement();
      }
      else
      {
         ETG_TRACE_ERR(("vSharedOwnUndefElementDestroy(): destroy element '%s' not in actual process ", PpstrElementName));
      }
   }
   else
   {
      ETG_TRACE_ERR(("vSharedOwnUndefElementDestroy(): destroy element '%s' not in actual process ", PpstrElementName));
   }
}

tS32 dp_tclDpBase::s32SharedOwnUndefElementDelete(const tChar* PpstrPoolName, const tChar* PpstrElementName)
{
   ETG_TRACE_USR4(("s32SharedOwnUndefElementDelete(): delete undef element '%s'", PpstrElementName));
   tS32                    Vs32ReturnCode = DP_S32_ERR_NO_POOL;
   tDpUndefPoolProperty    VtPoolProperty;
   dp_tclDpUndef*          VpoMyDpUndef = dp_tclDpUndef::pGetInstance();
   dp_tclBaseElement       VDpElem(PpstrElementName, 0); //hash calc in constructor    

   memset(&VtPoolProperty, 0, sizeof(tDpUndefPoolProperty));
   strncpy(VtPoolProperty.strPoolName, PpstrPoolName, DP_MAX_LEN_NAME_UNDEFINED_POOLS);
   VtPoolProperty.strPoolName[DP_MAX_LEN_NAME_UNDEFINED_POOLS - 1] = '\0';
   strncpy(VtPoolProperty.strProcessNameLocated, _szMyProcSpecName, DP_MAX_LEN_NAME_PROCESS);
   VtPoolProperty.strProcessNameLocated[DP_MAX_LEN_NAME_PROCESS - 1] = '\0';
   /* get pool properties */
   if (VpoMyDpUndef->s32GetPoolProperty(&VtPoolProperty, dp_tclDpUndef::eDpUndefDelete, &VDpElem) == DP_S32_NO_ERR)
   {/* pool should located in actual process*/
      if (strncmp(VtPoolProperty.strProcessNameLocated, _szMyProcSpecName, DP_MAX_LEN_NAME_PROCESS) == 0)
      {  /* check if pool properties exist in the process */
         (tVoid)DP_bEnterCritical(_hDpAccess);
         TDpUndefPoolProperty::iterator VtDpPosProp = oDpUndefPoolProperty.find(VtPoolProperty.u32PoolId);
         if (VtDpPosProp != oDpUndefPoolProperty.end())
         { //try to find pool
            TDatapool::iterator VtDpPos = oDatapool.find(VtPoolProperty.u32PoolId);
            if (VtDpPos != oDatapool.end())
            {
               Vs32ReturnCode = s32DeleteElement(&VtPoolProperty, &VDpElem);
            }
         }
         DP_bReleaseCritical(_hDpAccess);
      }
   }
   return(Vs32ReturnCode);
}

tS32 dp_tclDpBase::s32DeleteElement(tDpUndefPoolProperty*  PptPoolProperty, dp_tclBaseElement* PpDpElem)
{
   tS32             Vs32ReturnCode = DP_S32_NO_ERR;
   dp_tclDpUndef*   VpoMyDpUndef = dp_tclDpUndef::pGetInstance();

   TListDatapoolElements::iterator VtPosElement = oDatapool[PptPoolProperty->u32PoolId].tDpElementList.find(PpDpElem->s32GetHash());
   if (VtPosElement == oDatapool[PptPoolProperty->u32PoolId].tDpElementList.end())
   { /*element not found */
      Vs32ReturnCode = DP_S32_ERR_NO_ELEMENT;
   }
   else
   {/*element found */
      /*1) delete properties if not default*/
      tDpUndefElementProperty VtUndefElementProperty;
      memset(&VtUndefElementProperty, 0, sizeof(tDpUndefElementProperty));
      strncpy(VtUndefElementProperty.strElementName, PpDpElem->pcGetElementName(), DP_MAX_LEN_NAME_ELEMENT);
      VtUndefElementProperty.strElementName[DP_MAX_LEN_NAME_ELEMENT - 1] = '\0';
      VtUndefElementProperty.u8Version = VtPosElement->second.u8GetVersion();
      VtUndefElementProperty.u32DataLength = VtPosElement->second.u32GetDataLength();
      VtUndefElementProperty.bVarSize = VtPosElement->second.bGetVarSize();
      VtUndefElementProperty.u16AccessType = VtPosElement->second.u16GetAccessType();
      VtUndefElementProperty.tElemType = VtPosElement->second.eGetElementType();
      VtUndefElementProperty.tStoringType = VtPosElement->second.eGetStoringType();
      VtUndefElementProperty.u32StoringIntervall = VtPosElement->second.u32GetStoringIntervall();
      VtUndefElementProperty.u32DefSetMode = VtPosElement->second.u32GetDefSetMode();
      Vs32ReturnCode = VpoMyDpUndef->s32DeleteElementProperty(PptPoolProperty->u32PoolId, &VtUndefElementProperty, PptPoolProperty->eLocation);
      /*2) erase element from list */
      oDatapool[PptPoolProperty->u32PoolId].tDpElementList.erase(VtPosElement);
      /*check again if erase done */
      VtPosElement = oDatapool[PptPoolProperty->u32PoolId].tDpElementList.find(PpDpElem->s32GetHash());
      if ((VtPosElement != oDatapool[PptPoolProperty->u32PoolId].tDpElementList.end()) || (Vs32ReturnCode != DP_S32_NO_ERR))
      {
         Vs32ReturnCode = DP_S32_ERR_UNKNOWN;
         ETG_TRACE_ERR(("s32DeleteElement(): error delete element fails '%s'.", PpDpElem->pcGetElementName()));
      }
      else
      {
         ETG_TRACE_USR4(("s32DeleteElement(): element '%s' delete from pool.", PpDpElem->pcGetElementName()));
         /*if no element exist in pool => remove pool*/
         if (oDatapool[PptPoolProperty->u32PoolId].tDpElementList.size() == 0)
         {/*delete pool */
            ETG_TRACE_USR4(("s32DeleteElement(): no element in pool => delete pool"));
            oDatapool.erase(PptPoolProperty->u32PoolId);
            /*check pool */
            TDatapool::iterator VtDpPos = oDatapool.find(PptPoolProperty->u32PoolId);
            if (VtDpPos != oDatapool.end())
            {/*error delete pool */
               Vs32ReturnCode = DP_S32_ERR_UNKNOWN;
               ETG_TRACE_ERR(("s32DeleteElement(): error delete pool fails '%s'.", PptPoolProperty->strPoolName));
            }
            else
            { /*delete property*/
               oDpUndefPoolProperty.erase(PptPoolProperty->u32PoolId);
               TDpUndefPoolProperty::iterator VtDpPosProp = oDpUndefPoolProperty.find(PptPoolProperty->u32PoolId);
               if (VtDpPosProp != oDpUndefPoolProperty.end())
               { /*error delete pool properties*/
                  Vs32ReturnCode = DP_S32_ERR_UNKNOWN;
                  ETG_TRACE_ERR(("s32DeleteElement(): error delete pool properties fails '%s'.", PptPoolProperty->strPoolName));
               }
               else
               { /*delete property in shared memory*/
                  Vs32ReturnCode = VpoMyDpUndef->s32DeletePoolProperty(PptPoolProperty);
                  if (Vs32ReturnCode == DP_S32_NO_ERR)
                  {
                     if (_poPersMem == NULL)
                     {
                        FATAL_M_ASSERT_ALWAYS();
                     }
                     else
                     {
                        _poPersMem->s32DeletePersData(PptPoolProperty->strPoolName, PptPoolProperty->eLocation);
                        ETG_TRACE_USR4(("s32DeleteElement(): delete pool success '%s'.", PptPoolProperty->strPoolName));
                     }
                  }
                  else
                  {
                     ETG_TRACE_ERR(("s32DeleteElement(): error -> delete pool property '%s'.", PptPoolProperty->strPoolName));
                  }
               }
            }
         }
      }
   }
   return(Vs32ReturnCode);
}
#endif
tBool dp_tclDpBase::bInitPool(tU32 u32PoolId, tU8 u8Bank)
{
   tBool bRet = FALSE;
   if (oDatapool.find(u32PoolId) == oDatapool.end())
   {
#ifdef DP_FEATURE_UNDEF_ELEMENT
      tU16 Vu16AccessId = (tU16)(u32PoolId >> 16);
      if ((Vu16AccessId >= CCA_C_U16_APP_UNDEFINED_BEGIN) && (Vu16AccessId <= CCA_C_U16_APP_UNDEFINED_END))
      {
         vInitUndef(u32PoolId);
         bRet = TRUE;
      }
      else
      {
#endif
         TListInitPoolCallback::iterator iter = _olistInitPoolCallbacks.find(u32PoolId);
         if (iter == _olistInitPoolCallbacks.end())
         {
            ETG_TRACE_USR4(("dp_tclDpBase::bInitPool() set callback pointer because not in list _olistInitPoolCallbacks u32PoolId:0x%08x.", u32PoolId));
            bSetInitCallbackCore(u32PoolId);
         }
         //check again
         iter = _olistInitPoolCallbacks.find(u32PoolId);
         if (iter != _olistInitPoolCallbacks.end())
         { /*call init routine*/
           //ETG_TRACE_USR4(("dp_tclDpBase::bInitPool() u32PoolId:0x%08x", u32PoolId));
            DP_tpvInitPoolCallback pCallback = iter->second;
            if (pCallback != NULL)
            {
               ETG_TRACE_USR4(("dp_tclDpBase::bInitPool() call pCallback() u32PoolId:0x%08x for bank:%d", u32PoolId, u8Bank));
               pCallback(u8Bank);
               bRet = TRUE;
            }
         }
         else
         {
            ETG_TRACE_ERR(("dp_tclDpBase::bInitPool() error callback pointer not in list _olistInitPoolCallbacks u32PoolId:0x%08x.", u32PoolId));
         }
#ifdef DP_FEATURE_UNDEF_ELEMENT
      }
#endif
   }
   return bRet;
}

tVoid dp_tclDpBase::vSetPoolInitCallback(DP_tpvInitPoolCallback pInitCallback, tU32 u32PoolId)
{
   (tVoid)DP_bEnterCritical(_hDpAccess);
   ETG_TRACE_USR4(("dp_tclDpBase::vSetPoolInitCallback(): 0x%08x", u32PoolId));
   vAddInitCallbacktoList(pInitCallback, u32PoolId);
   DP_bReleaseCritical(_hDpAccess);
}

tBool dp_tclDpBase::bReloadPool(tU32 u32PoolId)
{
   tBool bRet = FALSE;
   (tVoid)DP_bEnterCritical(_hDpAccess);
   /*if the data is not found in the list then there is no point of reloading
   Also if the data is not found in the list then the segv will happen*/
   TDatapool::iterator VtDpPos = oDatapool.find(u32PoolId);
   if (VtDpPos != oDatapool.end())
   {
      if (oDatapool[u32PoolId].bElemChanged == TRUE)
      {
         ETG_TRACE_FATAL(("bReloadPool(): completion of previous update, still in progress for the pool: '0x%4x'.", u32PoolId));
      }
      else
      {
         /*init new pool*/
         oDatapool.erase(u32PoolId);
         bRet = bInitPool(u32PoolId);
      }
   }
   else
      ETG_TRACE_FATAL(("bReloadPool(): not possible for the requested element since the pool '0x%4x' is never read before.", u32PoolId));

   (tVoid)DP_bReleaseCritical(_hDpAccess);
   return bRet;
}
//EOF 
