/******************************************************************************/
/**
* \file    dispvidctrl_tclControlGraphic.cpp
* \ingroup
*
* \brief
*
* \remark  Copyright : (c) 2012 Robert Bosch GmbH, Hildesheim
* \remark  Author    :
* \remark  Scope     :
*
* \
*/
/******************************************************************************/


/******************************************************************************/
/* INCLUDES                                                                   */
/******************************************************************************/
#include "dispvidctrl_AppMain_Trace.h"

// ETG defines
#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_CLIENT_VIDEOPLAYER
#include "trcGenProj/Header/dispvidctrl_tclControlGraphic.cpp.trc.h"
#endif

#include "VideoPlayer_PNG_Decoder.h"
#include "dispvidctrl_tclPersistentData.h"
#include "dispvidctrl_datapool.h"

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include "system_types.h"
#include "ostypes.h"
#include <signal.h>
#include <sys/time.h>
#include <pthread.h>

#include "dispvidctrl_tclControlGraphic.h"

#include "VideoPlayer_shm.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"

/******************************************************************************/
/* DEFINES                                                                    */
/******************************************************************************/

#define PORTRAIT_BEGIN IDX_PORTRAIT_RVC
#define ENGLISH_UK  3
#define BOTTOM 0
#define TOP 1

/******************************************************************************/
/* FUNCTION     dispvidctrl_tclControlGraphic                            */
/******************************************************************************/
/**
*  \brief       default constructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/


/******************************************************************************/
/* FUNCTION     dispvidctrl_tclControlGraphic(vd_rvc_tclApp* poMainAppl) */
/******************************************************************************/
/**
*  \brief       constructor
*
*  \param       pointer to main application
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclControlGraphic::dispvidctrl_tclControlGraphic(const TEarlyConfiguration& tEarlyConfig):
m_bInitialized(false)
, m_poGraphics_DynGuideline(NULL)
, m_pU8Buffer_DynGuideline(NULL)
, m_bGraphicAvailable(false)
, m_szBasePath_Bitmap(NULL)
, m_GuidelinesDisplaying(false)
, guideline_thread()
{
    m_po_dispvidctrl_tclControlGraphic = this;
    m_warningTextPosition = BOTTOM; // the default value
    _vDetermineWarningTextPostion();
    m_bFirstDraw = false;
    m_bHmiAvailable = false;

    _dSteeringWheelAngle         = 0;
    _bSteeringWheelAngleChanged  = FALSE;

    _dRearWheelAngle             = 0;
    _bRearWheelAngleChanged      = FALSE;

    _bThreadActive               = FALSE;
    _bBlackScreenActive          = FALSE;

    //extract and hold relevant configuration data
    m_ScreenWidth                           = tEarlyConfig.tEarlyNorForce.u16Display1Width;
    m_ScreenHeight                          = tEarlyConfig.tEarlyNorForce.u16Display1Height;
    m_ScreenXOffset                         = tEarlyConfig.tEarlyNorForce.u16Display1XOffset;
    m_ScreenYOffset                         = tEarlyConfig.tEarlyNorForce.u16Display1YOffset;
    m_RVCLayer                              = tEarlyConfig.u32RvcLayer;
    m_RVCGraphicsLayer                      = tEarlyConfig.u32RvcGraphicsLayer;
    m_GuidelineSurface                      = tEarlyConfig.u32GuidelineSurface;
    m_RVCSurface                            = tEarlyConfig.u32RvcSurface;

    m_GuidelineWarningLanguage              = tEarlyConfig.tEarlyNor.u32GuidelineWarningLanguage;
    if (tEarlyConfig.tEarlyNorForce.bGuidelineVisibility == TRUE) {
        m_GuidelineVisibility               = tEarlyConfig.tEarlyNor.u32GuidelineVisibilityMode;
    }
    else {
        m_GuidelineVisibility               = 0;
    }
    m_GuidelineCameraType                   = tEarlyConfig.tEarlyNorForce.u32GuidelineCameraType;
    m_GuidelineCameraSystemType             = tEarlyConfig.tEarlyNorForce.u32GuidelineCameraSystemType;
    m_GuidelineBlackWarningTextBackground   = tEarlyConfig.tEarlyNorForce.u32GuidelineBlackWarningTextBackground;
    m_GuidelineWarningText                  = tEarlyConfig.tEarlyNorForce.u32GuidelineWarningText;
    m_GuidelineDrawingWidth                 = m_ScreenWidth;
    m_GuidelineDrawingHeight                = m_ScreenHeight;
    m_LeftHandSteeringPosition              = tEarlyConfig.tEarlyNorForce.bLeftHandSteeringPosition;
    m_DisplayOrientation                    = tEarlyConfig.tEarlyNorForce.u8EarlyDisplayOrientation;

    (tVoid) OSAL_pvMemoryCopy(&m_au8GuidelineVehicleParameter[0], &tEarlyConfig.tEarlyNorForce.au8GuidelineVehicleParameter[0], sizeof m_au8GuidelineVehicleParameter);

    // used as a divider, so cannot be 0 (for example, during first boot up)
    if ( m_ScreenWidth == 0 )
    {
       m_ScreenWidth = 800;
    }

    if ( m_ScreenHeight == 0 )
    {
       m_ScreenHeight = 480;
    }
    
    // buffer will be deleted in destructor
    m_pU8Buffer_DynGuideline = new tU8[(m_ScreenWidth * m_ScreenHeight * 4)];
    if(m_pU8Buffer_DynGuideline == 0)
    {
       ETG_TRACE_FATAL( ( "dispvidctrl_tclControlGraphic::dispvidctrl_tclControlGraphic() Failure to allocate guideline buffer"));
    }

    m_GuidelinesCurrentlyVisible = FALSE;
}

/******************************************************************************/
/* FUNCTION     _vDetermineWarningTextPostion                                      */
/******************************************************************************/
void dispvidctrl_tclControlGraphic::_vDetermineWarningTextPostion()
{
  if (DP_S32_NO_ERR == DP_s32GetConfigItem("CIS_Configuration","RVCTextPosition",(uint8_t*) &m_warningTextPosition,1))
  {
        dispvidctrl_tclAppMain::vWriteStartupLog(("Warning text Position read from kds"));
	if((m_warningTextPosition != BOTTOM) && (m_warningTextPosition != TOP))
	{
		m_warningTextPosition = BOTTOM;
	}
  }
  else
  {
       dispvidctrl_tclAppMain::vWriteStartupLog(("Error in Reading Warning text Position from Kds"));
  }

}


/******************************************************************************/
/* FUNCTION     ~dispvidctrl_tclControlGraphic                           */
/******************************************************************************/
/**
*  \brief       destructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclControlGraphic::~dispvidctrl_tclControlGraphic(void)
{
    // graphics interface
    m_po_dispvidctrl_tclControlGraphic = NULL;
    m_poGraphics_DynGuideline = NULL;

    if (m_pU8Buffer_DynGuideline != NULL)
    {
        delete[] m_pU8Buffer_DynGuideline;
        m_pU8Buffer_DynGuideline = NULL;
    }

    if (m_bInitialized == true)
    {
        // undo initialization
    }

    m_szBasePath_Bitmap = NULL;
}

/******************************************************************************/
/* FUNCTION     vOnPreInit                                                    */
/******************************************************************************/
/**
*  \brief       initializes (prepares) graphic layers for the RVC functionality
*               even before the first visualization ON request
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
void dispvidctrl_tclControlGraphic::vOnPreInit()
{
    if (!m_bInitialized) {
       vOnInit();
    }
}

/******************************************************************************/
/* FUNCTION     vOnInit                                                       */
/******************************************************************************/
/**
*  \brief       initializes (prepares) graphic layers for the RVC functionality
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
void dispvidctrl_tclControlGraphic::vOnInit()
{
    unsigned char u8VideoResizeParamIndex = IDX_LANDSCAPE_RVC;

    m_bInitialized = true;

    // will be called from worker thread vOnInit()
    m_GuidelinesDisplaying = false;

    _vConfiguration(); //Set up data based on screen size
    
    if(BOTTOM == m_warningTextPosition)
    {
     m_poGraphics_DynGuideline = new vd_rvc_tclGraphics_DynGuideline();

    DISPVIDCTRL_NULL_POINTER_CHECK(m_pU8Buffer_DynGuideline);
    memset(m_pU8Buffer_DynGuideline, 0, m_ScreenWidth * m_ScreenHeight * 4);
    }
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vOnInit() CA-GRF   ::  = m_GuidelineCameraSystemType ........................ %d", \
                        OSAL_ClockGetElapsedTime(), m_GuidelineCameraSystemType));

    switch (m_GuidelineCameraSystemType)
    {
       case CAMERA_TYPE_AVM_IPA:
          if (!m_bHmiAvailable)
          {
             // todo: Fetch the Pattern Information from INC ? Pattern 1 is shown by default
             _vUpdateDrawings_IpaMask(1);
             _vUpdateDrawings_IpaWarningText(); // will decode the IPA Mask and Warning Message and display it
          }
          break;
          
       case CAMERA_TYPE_AVM_ENTRY_RENAULT:
          if (!m_bHmiAvailable) {
             _vUpdateDrawings_IpaWarningText(); // will decode the IPA Mask and Warning Message and display it
          }
          break;
       case CAMERA_TYPE_MVC:
          u8VideoResizeParamIndex = IDX_LANDSCAPE_MVC;
          if (!m_bHmiAvailable) {
             _vUpdateDrawings_IpaWarningText(); // will decode the IPA Mask and Warning Message and display it
          }
          break;
          
       case CAMERA_TYPE_RVC:
          u8VideoResizeParamIndex = IDX_LANDSCAPE_RVC;
          // in case dynamic guideline is activated we need to decode only the warning text into the surfaces
          // static guideline will be drawn together with the dynamic guideline
          if (m_GuidelineWarningText == true)
          {
             _vUpdateDrawings_Rvc(); // will decode only warning text files into surfaces
          }
          break;
       case CAMERA_TYPE_RVC_RENAULT:
          u8VideoResizeParamIndex = IDX_LANDSCAPE_RVC_RENAULT;
          // in case dynamic guideline is activated we need to decode only the warning text into the surfaces
          // static guideline will be drawn together with the dynamic guideline
          if (!m_bHmiAvailable && m_GuidelineWarningText == true)
          {
             _vUpdateDrawings_Rvc(); // will decode only warning text files into surfaces
          }
          break;
       default:
          break;
    }
    
    if ( m_DisplayOrientation == DISPVIDCTRL_AllianceDisplay_Format_Portrait ) // Portrait
    {
       u8VideoResizeParamIndex += PORTRAIT_BEGIN;
    }

    // Set up buffer, image and capture parameters
    vOnSetupGuidelineParams(RVC_AVM_VIDEO_FULL_SCREEN, u8VideoResizeParamIndex);
    
    // Set up vehicle parameters
    vSetVehicleConfig_DynGuideline();
     if(NULL != m_poGraphics_DynGuideline)
     {	     
       m_poGraphics_DynGuideline->vAllowWarningIndIfUsingDummyConfig(true);
     }
}


/******************************************************************************/
/* FUNCTION     vOnSetupGuidelineParams                                       */
/******************************************************************************/
/**
*  \brief       initializes (prepares) graphic layers for the RVC functionality
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
void dispvidctrl_tclControlGraphic::vOnSetupGuidelineParams(unsigned char u8VideoType, unsigned char u8VideoResizeParamIndex)
{
   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vOnSetupGuidelineParams() CA-GRF   u8VideoType = %u u8VideoResizeParamIndex = %u", \
                       OSAL_ClockGetElapsedTime(), ETG_CENUM(tenControl_VideoResize, u8VideoType), ETG_CENUM(enVideoParamMapIndex, u8VideoResizeParamIndex)));

   unsigned int u32GuidelineHeight = (unsigned int) (RVC_DISPLAY_RESOLUTION_7_INCH_HEIGHT - RVC_WARNINGTEXT_7_INCH_HIGH);
   
   // Guidelines are only applicable to RVC or MVC 
   if ( m_GuidelineCameraSystemType == CAMERA_TYPE_RVC || m_GuidelineCameraSystemType == CAMERA_TYPE_RVC_RENAULT || m_GuidelineCameraSystemType == CAMERA_TYPE_MVC )
   {
      if ( m_DisplayOrientation == DISPVIDCTRL_AllianceDisplay_Format_Portrait ) // Portrait
      {
         u32GuidelineHeight = (unsigned int) (m_ScreenHeight - 170); // Height of Zone3 + Zone4 is subtracted
      }
   
      if (m_poGraphics_DynGuideline && m_pU8Buffer_DynGuideline)
      {
          vd_rvc_tstOutputBufferParams stBufParams;
          vd_rvc_tstImageAreaParams    stImgParams;
          vd_rvc_tstCaptureAreaParams  stCaptureParams;
      
          //  NCG3D-68269  NCG3D-67807 clear the already existing buffer before re-drawing it
          (void) m_poGraphics_DynGuideline->bDraw(0, 0);

          stBufParams.pvBuf = (void*) m_pU8Buffer_DynGuideline; // start of buffer area

          if ( u8VideoType == RVC_AVM_VIDEO_SETTINGS_SCREEN )
          {
             m_GuidelineDrawingWidth = mapCameraTypeGraphicParam[u8VideoResizeParamIndex].tVideoSettingsDestination.u16Width;
             m_GuidelineDrawingHeight = mapCameraTypeGraphicParam[u8VideoResizeParamIndex].tVideoSettingsDestination.u16Height;
             m_ScreenXOffset = mapCameraTypeGraphicParam[u8VideoResizeParamIndex].tVideoSettingsDestination.u16xOffset;
             m_ScreenYOffset = mapCameraTypeGraphicParam[u8VideoResizeParamIndex].tVideoSettingsDestination.u16yOffset;
          }
          else 
          {
             m_GuidelineDrawingWidth = mapCameraTypeGraphicParam[u8VideoResizeParamIndex].tVideoDestination.u16Width;
             m_GuidelineDrawingHeight = mapCameraTypeGraphicParam[u8VideoResizeParamIndex].tVideoDestination.u16Height;
             m_ScreenXOffset = mapCameraTypeGraphicParam[u8VideoResizeParamIndex].tVideoDestination.u16xOffset;
             m_ScreenYOffset = mapCameraTypeGraphicParam[u8VideoResizeParamIndex].tVideoDestination.u16yOffset;
             
             // if LHD use the video resize parameters as it is 
             if ( FALSE == m_LeftHandSteeringPosition ) // RHD
             {
                // horizontal mirroring of the video window
                // only the x-offset needs to be adapted according to the LHD layout
                m_ScreenXOffset = m_ScreenWidth - ( m_ScreenXOffset + m_GuidelineDrawingWidth );
             }
          }

          stBufParams.u32NumBufXpix = (unsigned int) m_ScreenWidth; // number of logical pixels per row (usually ==screen width in pixels)          
          
          if ( m_DisplayOrientation == 0 ) // Landscape
          {
             stBufParams.u32NumBufYpix = (unsigned int) (m_ScreenHeight - RVC_WARNINGTEXT_7_INCH_HIGH);
          }
          else // Portrait
          {
             stBufParams.u32NumBufYpix = (unsigned int) u32GuidelineHeight;
          }
      
          // width and height of output image area (may be smaller than screen/buffer width/height if output image area doesn't cover the whole screen)
          // -> we don't need to draw into the area of the warning text
          stImgParams.u32NumImgXpix = (unsigned int) m_GuidelineDrawingWidth;
          stImgParams.u32NumImgYpix = (unsigned int) (m_GuidelineDrawingHeight - RVC_WARNINGTEXT_7_INCH_HIGH);
      
          // offset of the output image area within the screen/buffer area (in pixels; set to (0,0) if upper left corner of image area == upper left corner of screen)
          stImgParams.u32ImgXoffs = m_ScreenXOffset;  // offset from left
          stImgParams.u32ImgYoffs = m_ScreenYOffset;  // offset from top
      
          // size of the full video image as used as input for capturing (in video image - e.g. NTSC - pixels)
          stCaptureParams.u32NumVideoImgXpix = VIDEO_IMAGE_X_PIXELS;
          stCaptureParams.u32NumVideoImgYpix = VIDEO_IMAGE_Y_PIXELS;
      
          // size of the captured part of the video image (i.e. the part which will be mapped to the output image) (in video image - e.g. NTSC - pixels)
          stCaptureParams.u32NumCaptureAreaXpix = VIDEO_IMAGE_X_PIXELS;
          if ( m_ScreenHeight != 0 )
          {
             stCaptureParams.u32NumCaptureAreaYpix = VIDEO_IMAGE_Y_PIXELS * u32GuidelineHeight / m_ScreenHeight;
          }
      
          // offset of the captured part of the video image within the full video image (set to (0,0) if upper left corner of captured area == upper left corner of full video image area)
          stCaptureParams.u32CaptureAreaXoffs = 0;  // offset from left (in video image - e.g. NTSC - pixels)
          stCaptureParams.u32CaptureAreaYoffs = 0;  // offset from top (in video image - e.g. NTSC - pixels)
      
          m_poGraphics_DynGuideline->vSetOutBuffer(&stBufParams);
          m_poGraphics_DynGuideline->vSetImageConfig(&stImgParams);
          m_poGraphics_DynGuideline->vSetCaptureConfig(&stCaptureParams);
      
          ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vOnSetupGuidelineParams() CA-GRF   :: u32NumBufXpix = ........................ %d", OSAL_ClockGetElapsedTime(), stBufParams.u32NumBufXpix));
          ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vOnSetupGuidelineParams() CA-GRF   :: u32NumBufYpix = ........................ %d", OSAL_ClockGetElapsedTime(), stBufParams.u32NumBufYpix));
          ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vOnSetupGuidelineParams() CA-GRF   :: u32NumImgXpix = ........................ %d", OSAL_ClockGetElapsedTime(), stImgParams.u32NumImgXpix));
          ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vOnSetupGuidelineParams() CA-GRF   :: u32NumImgYpix = ........................ %d", OSAL_ClockGetElapsedTime(), stImgParams.u32NumImgYpix));
          ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vOnSetupGuidelineParams() CA-GRF   :: u32ImgXoffs   = ........................ %d", OSAL_ClockGetElapsedTime(), stImgParams.u32ImgXoffs));
          ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vOnSetupGuidelineParams() CA-GRF   :: u32ImgYoffs   = ........................ %d", OSAL_ClockGetElapsedTime(), stImgParams.u32ImgYoffs));
          ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vOnSetupGuidelineParams() CA-GRF   :: u32NumVideoImgXpix = ................... %d", OSAL_ClockGetElapsedTime(), stCaptureParams.u32NumVideoImgXpix));
          ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vOnSetupGuidelineParams() CA-GRF   :: u32NumVideoImgYpix = ................... %d", OSAL_ClockGetElapsedTime(), stCaptureParams.u32NumVideoImgYpix));
          ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vOnSetupGuidelineParams() CA-GRF   :: u32NumCaptureAreaXpix = ................ %d", OSAL_ClockGetElapsedTime(), stCaptureParams.u32NumCaptureAreaXpix));
          ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vOnSetupGuidelineParams() CA-GRF   :: u32NumCaptureAreaYpix = ................ %d", OSAL_ClockGetElapsedTime(), stCaptureParams.u32NumCaptureAreaYpix));
          ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vOnSetupGuidelineParams() CA-GRF   :: u32CaptureAreaXoffs = .................. %d", OSAL_ClockGetElapsedTime(), stCaptureParams.u32CaptureAreaXoffs));
          ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vOnSetupGuidelineParams() CA-GRF   :: u32CaptureAreaXoffs = .................. %d", OSAL_ClockGetElapsedTime(), stCaptureParams.u32CaptureAreaYoffs));
          
          // but *don't* call m_poGraphics_DynGuideline->vSetVehicleConfig(&stKdsVehConfig) as well here -
          // this has to be done only once in vOnInit() above.
          // (Otherwise, each time we get here (basically each time the RVC gets (de)activated), we would overwrite all
          // guideline parameter changes which may have been done in the service test mode with the KDS config again,
          // to the effect that we would never be able to see how the changed parameters affect the guidelines.)
      }
   }
}


/******************************************************************************/
/* FUNCTION     vDrawGuidelines                                               */
/******************************************************************************/
/**
*  \brief       Start drawing Guidelines
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
void dispvidctrl_tclControlGraphic::vDrawGuidelines(void)
{
    tDouble dSteerAngleDeg = 0;

    //Check whether dynamic guidelines are required
    if(m_GuidelineVisibility & DYN_GUIDELINE_DRAW_DYNAMIC)
    {
        dSteerAngleDeg = _dSteeringWheelAngle;
        if(_bSteeringWheelAngleChanged == TRUE)
        {
            _bSteeringWheelAngleChanged = FALSE;

            ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vDrawGuidelines() Steering Wheel Angle [%f]", OSAL_ClockGetElapsedTime(),dSteerAngleDeg));
        }
    }

    if (m_poGraphics_DynGuideline)
    {
        // returns true when buffer content has changed

        //call dynamic guideline calculation function

        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vDrawGuidelines() Dynamic Guideline start draw - Steering Wheel Angle [%f] Guideline Visibility [%d]", OSAL_ClockGetElapsedTime(),dSteerAngleDeg,m_GuidelineVisibility));

        // returns true when buffer content has changed
        tU32 u32Mode = m_GuidelineVisibility | DYN_GUIDELINE_FORCE_UPDATE;
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vDrawGuidelines() Dynamic Guideline start draw - Draw Guideline Mode [%d]", OSAL_ClockGetElapsedTime(),u32Mode));
        if (true == m_poGraphics_DynGuideline->bDraw(dSteerAngleDeg, u32Mode))
        {
            ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vDrawGuidelines() m_poGraphics_DynGuideline->bDraw updated buffer", OSAL_ClockGetElapsedTime()));
        }
        else
        {
            ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vDrawGuidelines() - m_poGraphics_DynGuideline->bDraw no update", OSAL_ClockGetElapsedTime()));
        }
    }
}


/******************************************************************************/
/* FUNCTION     vStartGuidelines                                                       */
/******************************************************************************/
/**
*  \brief       Start drawing Guidelines
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
void dispvidctrl_tclControlGraphic::vStartGuidelines(void)
{
    if (!_bThreadActive) {

        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vStartGuidelines() Language [%d] WarningText [%d] Visibility [%d] CameraType [%d] CameraSystemType [%d] WarningTextBackground [%d]", OSAL_ClockGetElapsedTime(), (int)m_GuidelineWarningLanguage, (int)m_GuidelineWarningText, (int)m_GuidelineVisibility, (int)m_GuidelineCameraType, (int)m_GuidelineCameraSystemType, (int)m_GuidelineBlackWarningTextBackground));

        if (!m_bInitialized) {
            vOnInit();
        }

        m_GuidelinesDisplaying = true;

        vDrawGuidelines();

        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vStartGuidelines() Starting Guideline draw thread", OSAL_ClockGetElapsedTime()));

        int thread_ret = pthread_create(&guideline_thread, NULL, dispvidctrl_tclControlGraphic::vGuidelineThread, this);
        if (thread_ret  == 0)
        {
            pthread_detach(guideline_thread);
        }
        thread_ret = pthread_join(guideline_thread, NULL);
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vStartGuidelines() pthread_join result [%d]", OSAL_ClockGetElapsedTime(), thread_ret));

        // Wait for thread to start
        int thread_cnt = 0;
        while( (_bThreadActive == false) && (thread_cnt < 50) )
        {
            thread_cnt++;
            usleep(1000);
        }
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vStartGuidelines() thread started after %d ms - wait for graphics to initialise", OSAL_ClockGetElapsedTime(), thread_cnt));

        // Wait for Graphics to initialise
        int graphics_cnt = 0;
        while( (bGraphicsSurfaceReady() == false) && (graphics_cnt < 200) )
        {
            graphics_cnt++;
            usleep(1000);
        }
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vStartGuidelines() graphics ready after %d ms", OSAL_ClockGetElapsedTime(), graphics_cnt));
    }

}


/******************************************************************************/
/* FUNCTION     vStopGuidelines                                                       */
/******************************************************************************/
/**
*  \brief       Stop drawing Guidelines
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
void dispvidctrl_tclControlGraphic::vStopGuidelines(void)
{
    m_GuidelinesCurrentlyVisible = FALSE;

    // Make sure the graphics layer has been made invisible
    end_display_guidelines();
}


/******************************************************************************/
/* FUNCTION     vGuidelineThreadMainLoop                                                       */
/******************************************************************************/
/**
*  \brief       Guideline drawing main loop
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
void* dispvidctrl_tclControlGraphic::vGuidelineThreadMainLoop(void)
{
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vGuidelineThreadMainLoop() calling display_guidelines()", OSAL_ClockGetElapsedTime()));
    _bThreadActive = TRUE;

    display_guidelines(m_ScreenWidth, m_ScreenHeight, m_RVCLayer, m_RVCGraphicsLayer, m_GuidelineSurface, m_RVCSurface, (char*)m_pU8Buffer_DynGuideline);

    _bThreadActive = FALSE;
    return NULL;
}


/******************************************************************************/
/******************************************************************************/
/******************************************************************************/
/******************************************************************************/
/* FUNCTION     _vConfiguration                                               */
/******************************************************************************/
/**
*  \brief       determines relevant configuration details
*
*  \param       pointer to configuration data structure
*  \return      none
*/
/******************************************************************************/
void dispvidctrl_tclControlGraphic::_vConfiguration()
{
    switch (m_ScreenHeight)
    {
    case RVC_DISPLAY_RESOLUTION_7_INCH_HEIGHT:
        {
            ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::_vConfiguration() u8DisplayType = ..... 7 Inch Configuration", OSAL_ClockGetElapsedTime()));

            // set base path for bitmaps
            m_szBasePath_Bitmap = (tCString) "/opt/bosch/early/warningtexts/800x480/";
            break;
        }

    case RVC_DISPLAY_RESOLUTION_5_8_INCH_HEIGHT:
        {
            ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::_vConfiguration() u8DisplayType = ..... 5.8 Inch Configuration", OSAL_ClockGetElapsedTime()));

            // set base path for bitmaps
            m_szBasePath_Bitmap = (tCString) "/opt/bosch/early/warningtexts/480x272/";
            break;
        }

    case RVC_DISPLAY_RESOLUTION_3_5_INCH_HEIGHT:
        {
            ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::_vConfiguration() u8DisplayType = ..... 3.5 Inch Configuration", OSAL_ClockGetElapsedTime()));

            // set base path for bitmaps
            m_szBasePath_Bitmap = (tCString) "/opt/bosch/early/warningtexts/400x240/";
            break;
        }

    default:
        {
            ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::_vConfiguration() Invalid value for m_ScreenHeight [%d]", OSAL_ClockGetElapsedTime(), (int)m_ScreenHeight));
            break;
        }
    }
}

/******************************************************************************/
/* FUNCTION     _vUpdateDrawings_Rvc                                          */
/******************************************************************************/
/**
*  \brief       - determines path and filenames for the bitmaps to decode
*               - calls bitmap decoding/drawing functions of graphic interface
*
*  \param       enum - control action
*  \return      tBool - success state
*/
/******************************************************************************/
void dispvidctrl_tclControlGraphic::_vUpdateDrawings_Rvc(void)
{
    char  as8Filename_WarningText[9];
    char  as8path_WarningText[50];
    as8Filename_WarningText[0] = '\0';
    as8path_WarningText[0] = '\0';
	ETG_TRACE_FATAL( ( "dispvidctrl_tclControlGraphic::_vUpdateDrawings_Rvc()"));
    //Clear the whole buffer - including warning text
    DISPVIDCTRL_NULL_POINTER_CHECK(m_pU8Buffer_DynGuideline);
    memset(m_pU8Buffer_DynGuideline, 0, m_ScreenWidth * m_ScreenHeight * 4);

    //Calculate position in buffer for start of warning text
    unsigned int buffer_offset = 0;
    unsigned int warning_text_height = 0;
    unsigned int left_buffer_offset = (((m_ScreenWidth - RVC_DISPLAY_RESOLUTION_7_INCH_WIDTH)/2) * 4);

    if (m_ScreenHeight == RVC_DISPLAY_RESOLUTION_7_INCH_HEIGHT)
    {
		//ETG_TRACE_FATAL( ( "dispvidctrl_tclControlGraphic::_vUpdateDrawings_Rvc() 1"));
        //if((m_GuidelineCameraSystemType == CAMERA_TYPE_RVC_RENAULT) && (m_DisplayOrientation != DISPVIDCTRL_AllianceDisplay_Format_Portrait))
		if ( (m_GuidelineCameraSystemType == CAMERA_TYPE_RVC) || (m_GuidelineCameraSystemType == CAMERA_TYPE_RVC_RENAULT) )
        {
          // ETG_TRACE_FATAL( ( "dispvidctrl_tclControlGraphic::_vUpdateDrawings_Rvc() to draw blackscreen above warning text."));
           buffer_offset = (unsigned int) (m_ScreenWidth * (m_ScreenHeight - RVC_ZONE3_LANDSCAPE_HIGH) * 4);
        //   vDrawBlackscreen(RVC_WARNINGTEXT_7_INCH_WIDE, (RVC_ZONE3_LANDSCAPE_HIGH - RVC_WARNINGTEXT_7_INCH_HIGH), buffer_offset, left_buffer_offset); //Drawing black screen for 22 lines to avoid map only for renault landscape.
		 //  ETG_TRACE_FATAL( ( "dispvidctrl_tclControlGraphic::_vUpdateDrawings_Rvc() 2"));
        }

        buffer_offset = (unsigned int) (m_ScreenWidth * (m_ScreenHeight - RVC_WARNINGTEXT_7_INCH_HIGH) * 4);
        warning_text_height = RVC_WARNINGTEXT_7_INCH_HIGH;
    }
    else
    {
		ETG_TRACE_FATAL( ( "dispvidctrl_tclControlGraphic::_vUpdateDrawings_Rvc() 3"));
        if (m_ScreenHeight == RVC_DISPLAY_RESOLUTION_5_8_INCH_HEIGHT)
        {
            buffer_offset = (unsigned int) (m_ScreenWidth * (m_ScreenHeight - RVC_WARNINGTEXT_5_8_INCH_HIGH) * 4);
            warning_text_height = RVC_WARNINGTEXT_5_8_INCH_HIGH;
        }
        else
        {
            buffer_offset = (unsigned int) (m_ScreenWidth * (m_ScreenHeight - RVC_WARNINGTEXT_3_5_INCH_HIGH) * 4);
            warning_text_height = RVC_WARNINGTEXT_3_5_INCH_HIGH;
        }
    }
    if(TOP == m_warningTextPosition)
    {
         buffer_offset = 0;
    }

    //Decode the png
    sDecodedImageData l_pngDecodedImageData;

    // Clear the mask PNG data
    memset(&l_pngDecodedImageData, 0, sizeof(sDecodedImageData));
    
   // if ( m_GuidelineWarningText )
    {
       //get file name extensions from configuration value
       /* put each nibble of configuration value as char in the string
       *  (0x30 to 0x39 if nibble value is going from  0 to 9
       *   and 0x61 to 0x66 if nibble value is going from 0xA to 0xF (10 to 15))    add 39 for hex a..f (use small letters in file name)
       */
	 //  ETG_TRACE_FATAL( ( "dispvidctrl_tclControlGraphic::_vUpdateDrawings_Rvc() 4"));
       as8Filename_WarningText[0] = (char) 'w';
       as8Filename_WarningText[1] = (char) 't';
       if (m_GuidelineBlackWarningTextBackground)
       {
           as8Filename_WarningText[1] = (char) 'b';
	  // ETG_TRACE_FATAL( ( "dispvidctrl_tclControlGraphic::_vUpdateDrawings_Rvc() 5"));

       }        
       as8Filename_WarningText[2] = (char)(0x30 + (0x0f & (m_GuidelineWarningLanguage >> 4)) + (((0x0f & (m_GuidelineWarningLanguage >> 4)) < 10) ? 0:39));
       as8Filename_WarningText[3] = (char)(0x30 + (0x0f & m_GuidelineWarningLanguage) + (((0x0f & m_GuidelineWarningLanguage) <10) ? 0:39));
       as8Filename_WarningText[4] = (char) '.';
       as8Filename_WarningText[5] = (char) 'p';
       as8Filename_WarningText[6] = (char) 'n';
       as8Filename_WarningText[7] = (char) 'g';
       as8Filename_WarningText[8] = 0; // zero termination
       
       // combine base path and file name
       
       if(m_szBasePath_Bitmap && *m_szBasePath_Bitmap && // at least one char in base path name
         (strlen(m_szBasePath_Bitmap)+strlen(as8Filename_WarningText)) < sizeof(as8path_WarningText) )
       {
           strcpy(as8path_WarningText, m_szBasePath_Bitmap);
           strcpy(as8path_WarningText + strlen(as8path_WarningText), as8Filename_WarningText);
       }
       
       ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::_vUpdateDrawings_Rvc() PNG for warning text is %s", OSAL_ClockGetElapsedTime(), as8path_WarningText));
       
       if( as8path_WarningText[0] != '\0' )
       {            
           if(_intDecodePNGIntoBuffer(as8path_WarningText,l_pngDecodedImageData,false,true,true) == 0)
           {
              ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::_vUpdateDrawings_Rvc() PNG for warning text decoded", OSAL_ClockGetElapsedTime()));
              
             _vUpdateDrawings_WarningText(&l_pngDecodedImageData, RVC_WARNINGTEXT_7_INCH_WIDE, RVC_WARNINGTEXT_7_INCH_HIGH, buffer_offset, left_buffer_offset);
           }
           else
           {
              vSetDefalutLanguage();
           }
       }
    }

   // _vUpdateDrawings_WarningText(&l_pngDecodedImageData, RVC_WARNINGTEXT_7_INCH_WIDE, RVC_WARNINGTEXT_7_INCH_HIGH, buffer_offset, left_buffer_offset);
}
/******************************************************************************/
/* FUNCTION     vSetDefalutLanguage                                          */
/******************************************************************************/
/**
*  \brief       - Set the default language whenever there is a error condition
*
*  \param       - none
*  \return      - none
*******************************************************************************/
tVoid dispvidctrl_tclControlGraphic::vSetDefalutLanguage()
{
   m_GuidelineWarningLanguage = ENGLISH_UK;
   _vUpdateDrawings_Rvc();
}


/******************************************************************************/
/* FUNCTION     vDrawBlackscreen                                          */
/******************************************************************************/
/**
*  \brief       - Draws blackscreen between video and warning text
*
*  \param       - none
*  \return      - none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControlGraphic::vDrawBlackscreen(int width, int height, int top_buffer_offset, int left_buffer_offset)
{
   ETG_TRACE_FATAL( ( "dispvidctrl_tclControlGraphic::vDrawBlackscreen() entered."));
   tU8 *bp = (tU8*)m_pU8Buffer_DynGuideline + top_buffer_offset;
   for(int i=0; i<(m_ScreenWidth*height); i++)
   {
      *(bp++) = 0x00;
      *(bp++) = 0x00;
      *(bp++) = 0x00;
      *(bp++) = 0xFF;
   }
   (tVoid)width;
   (tVoid)left_buffer_offset;
}


/******************************************************************************/
/* FUNCTION     _vUpdateDrawings_IpaMask()                                    */
/******************************************************************************/
/**
*  \brief       Draw the IPA Mask
*
*  \param       None
*  \return      None
*/
/******************************************************************************/
void dispvidctrl_tclControlGraphic::_vUpdateDrawings_IpaMask(const tU8 u8Pattern)
{
    char  as8Filename_IPAVideoMask[8];
    char  as8path_IPAVideoMask[50];
    as8Filename_IPAVideoMask[0] = '\0';
    as8path_IPAVideoMask[0] = '\0';

    //Clear the whole buffer - including warning text
    DISPVIDCTRL_NULL_POINTER_CHECK(m_pU8Buffer_DynGuideline);
    memset(m_pU8Buffer_DynGuideline, 0, m_ScreenWidth * m_ScreenHeight * 4);

    //Calculated position in buffer for start of IPA mask is always 0
    unsigned int buffer_offset = 0;

    // All patterns other than Pattern 15 has half Screen Mask
    if(u8Pattern != 0x0F)
    {
        //get file name extensions from configuration value
        as8Filename_IPAVideoMask[0] = (char) 'm';
        as8Filename_IPAVideoMask[1] = (char) 'r';
        if (m_LeftHandSteeringPosition)
        {
            as8Filename_IPAVideoMask[1] = (char) 'l';
        }
        as8Filename_IPAVideoMask[2] = (char) 'r';
        as8Filename_IPAVideoMask[3] = (char) '.';
        as8Filename_IPAVideoMask[4] = (char) 'p';
        as8Filename_IPAVideoMask[5] = (char) 'n';
        as8Filename_IPAVideoMask[6] = (char) 'g';
        as8Filename_IPAVideoMask[7] = 0; // zero termination
    }
    else // Only Pattern 15 has Full Screen Mask
    {
        as8Filename_IPAVideoMask[0] = (char) 'm';
        as8Filename_IPAVideoMask[1] = (char) 'f'; // f = Full Screen Image
        as8Filename_IPAVideoMask[2] = (char) 'r';
        as8Filename_IPAVideoMask[3] = (char) '.';
        as8Filename_IPAVideoMask[4] = (char) 'p';
        as8Filename_IPAVideoMask[5] = (char) 'n';
        as8Filename_IPAVideoMask[6] = (char) 'g';
        as8Filename_IPAVideoMask[7] = 0; // zero termination
    }

    // combine base path and file name

    if(m_szBasePath_Bitmap && *m_szBasePath_Bitmap &&  // at least one char in base path name
       (strlen(m_szBasePath_Bitmap)+strlen(as8Filename_IPAVideoMask)) < sizeof(as8path_IPAVideoMask) )
    {
        strcpy(as8path_IPAVideoMask, m_szBasePath_Bitmap);
        strcpy(as8path_IPAVideoMask + strlen(as8path_IPAVideoMask), as8Filename_IPAVideoMask);
    }

    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::_vUpdateDrawings_IpaMask() PNG for Mask is %s", OSAL_ClockGetElapsedTime(), as8path_IPAVideoMask));

    //Decode the png
    sDecodedImageData l_pngDecodedImageData;

    if( as8path_IPAVideoMask[0] != '\0' )
    {
        if(_intDecodePNGIntoBuffer(as8path_IPAVideoMask,l_pngDecodedImageData,false,true,true) == 0)
        {
            ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::_vUpdateDrawings_IpaMask() Rendering png Mask h=%d, w=%d, %s", OSAL_ClockGetElapsedTime(), (int)l_pngDecodedImageData.m_u32Height, (int)l_pngDecodedImageData.m_u32Width, as8path_IPAVideoMask));

            ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::_vUpdateDrawings_IpaMask() buffer_offset [%d], m_ScreenHeight [%d], m_ScreenWidth [%d], m_pU8Buffer_DynGuideline [%d]", OSAL_ClockGetElapsedTime(), (int)buffer_offset, (int)m_ScreenHeight, (int)m_ScreenWidth, (intptr_t)m_pU8Buffer_DynGuideline));

            // Copy from png buffer to Overlay Buffer
            void* vp = m_pU8Buffer_DynGuideline;
            char* cp = (char*)vp;
            char* cs = (char*)l_pngDecodedImageData.m_pu8Data;

            ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::_vUpdateDrawings_IpaMask() cp [%d], cs [%d], m_ScreenHeight*m_ScreenWidth*4 [%d]", OSAL_ClockGetElapsedTime(), (intptr_t)cp, (intptr_t)cs, (int)(m_ScreenHeight*m_ScreenWidth*4)));
            memcpy(cp, cs, (m_ScreenHeight*m_ScreenWidth*4));
        }
    }
}

/******************************************************************************/
/* FUNCTION     _vUpdateDrawings_IpaWarningText()                             */
/******************************************************************************/
/**
*  \brief       Draw the IPA Mask
*
*  \param       None
*  \return      None
*/
/******************************************************************************/
void dispvidctrl_tclControlGraphic::_vUpdateDrawings_IpaWarningText(void)
{
    unsigned int warning_text_height = IPA_WARNINGTEXT_7_INCH_HIGH;
    unsigned int buffer_offset = 0;
    unsigned int left_buffer_offset = 0;
    
    if (m_GuidelineCameraSystemType == CAMERA_TYPE_AVM_ENTRY_RENAULT || m_GuidelineCameraSystemType == CAMERA_TYPE_MVC)
    {
       // Hardcoded 11 --> Warning text is displayed in center aligned (height) with zone3
       buffer_offset = ((m_ScreenWidth) * (m_ScreenHeight - IPA_WARNINGTEXT_7_INCH_HIGH - 11) * 4);
       left_buffer_offset = (((m_ScreenWidth - IPA_WARNINGTEXT_7_INCH_WIDE)/2) * 4);
    }
    else if (m_GuidelineCameraSystemType == CAMERA_TYPE_AVM_IPA) // AVM Entry with IPA for Nissan
    {
       if (FALSE == m_LeftHandSteeringPosition)
       {
          buffer_offset = ((m_ScreenWidth - IPA_WARNINGTEXT_7_INCH_WIDE) * 4);
       }
    }
    else
    {}
    
    //Decode the png
    sDecodedImageData l_pngDecodedImageData;

    // Clear the mask PNG data
    memset(&l_pngDecodedImageData, 0, sizeof(sDecodedImageData));

    if ( m_GuidelineWarningText )
    {
       char  as8Filename_IPAWarningText[9];
       char  as8path_IPAWarningText[50];
       as8Filename_IPAWarningText[0] = '\0';
       as8path_IPAWarningText[0] = '\0';
       
       //get file name extensions from configuration value
       /* put each nibble of configuration value as char in the string
       *  (0x30 to 0x39 if nibble value is going from  0 to 9
       *   and 0x61 to 0x66 if nibble value is going from 0xA to 0xF (10 to 15))    add 39 for hex a..f (use small letters in file name)
       */
       as8Filename_IPAWarningText[0] = (char) 'w';
       as8Filename_IPAWarningText[1] = (char) 'i';
       as8Filename_IPAWarningText[2] = (char)(0x30 + (0x0f & (m_GuidelineWarningLanguage >> 4)) + (((0x0f & (m_GuidelineWarningLanguage >> 4)) < 10) ? 0:39));
       as8Filename_IPAWarningText[3] = (char)(0x30 + (0x0f & m_GuidelineWarningLanguage) + (((0x0f & m_GuidelineWarningLanguage) <10) ? 0:39));
       as8Filename_IPAWarningText[4] = (char) '.';
       as8Filename_IPAWarningText[5] = (char) 'p';
       as8Filename_IPAWarningText[6] = (char) 'n';
       as8Filename_IPAWarningText[7] = (char) 'g';
       as8Filename_IPAWarningText[8] = 0; // zero termination
       
       // combine base path and file name
       
       if(m_szBasePath_Bitmap && *m_szBasePath_Bitmap && // at least one char in base path name
         (strlen(m_szBasePath_Bitmap)+strlen(as8Filename_IPAWarningText)) < sizeof(as8path_IPAWarningText) )
       {
           strcpy(as8path_IPAWarningText, m_szBasePath_Bitmap);
           strcpy(as8path_IPAWarningText + strlen(as8path_IPAWarningText), as8Filename_IPAWarningText);
       }
       
       ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::_vUpdateDrawings_IpaWarningText() PNG for warning text is %s", OSAL_ClockGetElapsedTime(), as8path_IPAWarningText));
       
       //Decode the png
       if( as8path_IPAWarningText[0] != '\0' )
       {
           if(_intDecodePNGIntoBuffer(as8path_IPAWarningText,l_pngDecodedImageData,false,true,true) == 0)
           {
              ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::_vUpdateDrawings_IpaWarningText() PNG for warning text decoded", OSAL_ClockGetElapsedTime()));
           }
       }
    }

   _vUpdateDrawings_WarningText(&l_pngDecodedImageData, IPA_WARNINGTEXT_7_INCH_WIDE, IPA_WARNINGTEXT_7_INCH_HIGH, buffer_offset, left_buffer_offset);
}

/******************************************************************************/
/* FUNCTION     _vUpdateDrawings_WarningText()                                */
/******************************************************************************/
/**
*  \brief       Draw/Clear the Warning Text
*
*  \param       None
*  \return      None
*/
/******************************************************************************/
void dispvidctrl_tclControlGraphic::_vUpdateDrawings_WarningText(sDecodedImageData* l_pngDecodedImageData, int width, int height, int buffer_offset, int left_buffer_offset)
{
       ETG_TRACE_FATAL(("dispvidctrl_tclControlGraphic::_vUpdateDrawings_WarningText() PNG for warning text"));

   // Warning Text width is less than the screen size
   // copy the decoded text png frame by frame into the buffer "text width pixels" = frame_buffer_width
   int frame_buffer_width = width * 4;
   int max_fill = (intptr_t)m_pU8Buffer_DynGuideline + (m_ScreenWidth * m_ScreenHeight * 4);

   // Copy from png buffer to Overlay Buffer
   void* vp = m_pU8Buffer_DynGuideline;
   char* cp = (char*)vp + buffer_offset + left_buffer_offset;
   
   if ( NULL != l_pngDecodedImageData && NULL != l_pngDecodedImageData->m_pu8Data ) // Something is decoded so draw the warning Text
   {
      ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::_vUpdateDrawings_WarningText() Rendering png warning text h=%d, w=%d", OSAL_ClockGetElapsedTime(), (int)height, (int)width));
     
      char* cs = (char*)l_pngDecodedImageData->m_pu8Data;
      int max_data = (intptr_t)l_pngDecodedImageData->m_pu8Data + (height * width * 4);
      
      ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::_vUpdateDrawings_WarningText() frame_buffer_width [%d], max_fill [%d]", OSAL_ClockGetElapsedTime(), (int)frame_buffer_width, (int)max_fill));
      
      // each loop should copy "text-width" number of pixels
      // in the next loop we move to the next row of pixels
      while ((((intptr_t)cp + frame_buffer_width) <= max_fill) && (((intptr_t)cs + frame_buffer_width) <= max_data))
      {
         memcpy(cp,cs,frame_buffer_width); // copy one row
         // move to the next row
         cp += (m_ScreenWidth*4);
         cs += frame_buffer_width;
         ETG_TRACE_USR4( ( "[%d ms] dispvidctrl_tclControlGraphic::_vUpdateDrawings_WarningText() cp [%d], cs [%d]", OSAL_ClockGetElapsedTime(), (intptr_t)cp, (intptr_t)cs));
      }
   }
   else // Clear the already Drawn Warning Text
   {
      int fill_length = max_fill - (intptr_t)cp;
      memset(cp,0,fill_length);
   }
}

/******************************************************************************/
/* FUNCTION     void vUpdateGuidelinesDisplay(tBool bBlackScreen)             */
/******************************************************************************/
/**
*  \brief       Force update to guidelines graphic display
*
*  \param       None
*  \return      None
*/
/******************************************************************************/
void dispvidctrl_tclControlGraphic::vUpdateGuidelinesDisplay(tU8 u8BlackScreenType)
{
    if (_bThreadActive) {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vUpdateGuidelinesDisplay() - Display new buffer contents - bBlackScreenType [%d], m_GuidelinesCurrentlyVisible [%d]", OSAL_ClockGetElapsedTime(), u8BlackScreenType, m_GuidelinesCurrentlyVisible));
        if( (m_GuidelinesCurrentlyVisible == TRUE) || (u8BlackScreenType != 0) ) {
            update_guidelines((enum ten_VideoPlayer_BlackScreenType) u8BlackScreenType);
        }
    } else {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vUpdateGuidelinesDisplay(): Guidelines thread currently not running --> ignore update.", OSAL_ClockGetElapsedTime()));
    }
}

void dispvidctrl_tclControlGraphic::vSwitchToBlackScreen(tU8 u8BlackScreenType)
{
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vSwitchToBlackScreen() - Display buffer start switch to black", OSAL_ClockGetElapsedTime()));
    m_GuidelinesCurrentlyVisible = FALSE; //Indicates Guidelines are currently not being displayed - either RVC is off or Black Screen is active
    vUpdateGuidelinesDisplay(TRUE);
}

void dispvidctrl_tclControlGraphic::vSwitchToGuidelinesDisplay(void)
{
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vSwitchToGuidelinesDisplay() - Display buffer switch to guidelines", OSAL_ClockGetElapsedTime()));
    m_GuidelinesCurrentlyVisible = TRUE;
	vDrawGuidelines(); //Force redraw of guidelines 
    vUpdateGuidelinesDisplay(0);

}

tVoid dispvidctrl_tclControlGraphic::vSetWarningTextLanguage(tU8 u8Language) {
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vSetWarningTextLanguage(): update -> %d [prev %d]", OSAL_ClockGetElapsedTime(), u8Language, m_GuidelineWarningLanguage));

    if(m_GuidelineWarningLanguage != u8Language) {
        m_GuidelineWarningLanguage = u8Language;
		///ETG_TRACE_FATAL(("dispvidctrl_tclControlGraphic::vSetWarningTextLanguage(): inside 1"));
		//if(m_GuidelinesCurrentlyVisible == TRUE) 
		{
			//ETG_TRACE_FATAL(("dispvidctrl_tclControlGraphic::vSetWarningTextLanguage(): inside 2"));

            if (m_bInitialized)
            {
              // ETG_TRACE_FATAL(("dispvidctrl_tclControlGraphic::vSetWarningTextLanguage(): inside 3"));  
               if ( (m_GuidelineCameraSystemType == CAMERA_TYPE_RVC) || (m_GuidelineCameraSystemType == CAMERA_TYPE_RVC_RENAULT) )
               {
				  // ETG_TRACE_FATAL(("dispvidctrl_tclControlGraphic::vSetWarningTextLanguage(): inside 4"));
                  _vUpdateDrawings_Rvc();
               }
               else if ((m_GuidelineCameraSystemType == CAMERA_TYPE_AVM_ENTRY_RENAULT) || (m_GuidelineCameraSystemType == CAMERA_TYPE_MVC) || ((m_GuidelineCameraSystemType == CAMERA_TYPE_AVM_IPA) && (!m_bHmiAvailable)))
               {
                  _vUpdateDrawings_IpaWarningText();
               }
               else
               {}
            }

            if (_bThreadActive) {
                //_vUpdateDrawings_Rvc();
                vDrawGuidelines();
                vUpdateGuidelinesDisplay(0);
            } else {
                ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vSetWarningTextLanguage(): Guidelines thread currently not running --> ignore update.", OSAL_ClockGetElapsedTime()));
			}
        }
    }
}

tVoid dispvidctrl_tclControlGraphic::vSetGuideLineVisibility(tU32 u32Visibility) {
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vSetGuideLineVisibility(): update -> %d [prev %d]", OSAL_ClockGetElapsedTime(), u32Visibility, m_GuidelineVisibility));

    if(m_GuidelineVisibility != u32Visibility) {
        m_GuidelineVisibility = (u32Visibility & (~DYN_GUIDELINE_FORCE_UPDATE));
		
		if(m_GuidelinesCurrentlyVisible == TRUE) {
            if (_bThreadActive) {
                vDrawGuidelines();
                vUpdateGuidelinesDisplay(0);
            } else {
                ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vSetGuideLineVisibility(): Guidelines thread currently not running --> ignore update.", OSAL_ClockGetElapsedTime()));
			}
        }
    }
}

tVoid dispvidctrl_tclControlGraphic::vSetWarningTextVisibility(tU32 u32Visibility) {
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vSetWarningTextVisibility(): update -> %d [prev %d]", OSAL_ClockGetElapsedTime(), u32Visibility, m_GuidelineWarningText));

    if(m_GuidelineWarningText != u32Visibility) {
        m_GuidelineWarningText = u32Visibility;
		
		if(m_GuidelinesCurrentlyVisible == TRUE) {

        if (m_bInitialized)
        {
           if ( (m_GuidelineCameraSystemType == CAMERA_TYPE_RVC) || (m_GuidelineCameraSystemType == CAMERA_TYPE_RVC_RENAULT) )
           {
              _vUpdateDrawings_Rvc();
           }
           else if ((m_GuidelineCameraSystemType == CAMERA_TYPE_AVM_ENTRY_RENAULT) || (m_GuidelineCameraSystemType == CAMERA_TYPE_MVC) || ((m_GuidelineCameraSystemType == CAMERA_TYPE_AVM_IPA) && (!m_bHmiAvailable)))
           {
              _vUpdateDrawings_IpaWarningText();
           }
           else
           {}
        }

            if (_bThreadActive) {
                vDrawGuidelines();
                vUpdateGuidelinesDisplay(0);
            } else {
                ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vSetWarningTextVisibility(): Guidelines thread currently not running --> ignore update.", OSAL_ClockGetElapsedTime()));
			}
        }
    }
}

tVoid dispvidctrl_tclControlGraphic::vSetSteeringWheelAngle(tS32 s32SteeringWheelAngle) {
    if ((tDouble) s32SteeringWheelAngle != _dSteeringWheelAngle) {
        _dSteeringWheelAngle        = (tDouble)s32SteeringWheelAngle;
        _bSteeringWheelAngleChanged = TRUE;

		if(m_GuidelinesCurrentlyVisible == TRUE) {
            ETG_TRACE_FATAL( ( "[%d ms]dispvidctrl_tclControlGraphic:: vSetSteeringWheelAngle() s32SteeringWheelAngle [%d], m_GuidelinesDisplaying [%d], m_GuidelineVisibility [%d]", OSAL_ClockGetElapsedTime(), s32SteeringWheelAngle, m_GuidelinesDisplaying, m_GuidelineVisibility));

            if((m_GuidelinesDisplaying == true) && (m_GuidelineVisibility & DYN_GUIDELINE_DRAW_DYNAMIC))
            {
               //Draw dynamic guidelines

               ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vSetSteeringWheelAngle() Dynamic guidelines update required", OSAL_ClockGetElapsedTime()));
               vDrawGuidelines();
            }
            vUpdateGuidelinesDisplay(0);
		}
    }
}

tVoid dispvidctrl_tclControlGraphic::vSetRearWheelAngle(tS32 s32RearWheelAngle) {
    if ((tDouble) s32RearWheelAngle != _dRearWheelAngle) {
        _dRearWheelAngle        = (tDouble)s32RearWheelAngle;
        _bRearWheelAngleChanged = TRUE;

		if(m_GuidelinesCurrentlyVisible == TRUE) {
            if((m_GuidelinesDisplaying == true) && (m_GuidelineVisibility & DYN_GUIDELINE_DRAW_DYNAMIC))
            {
               //Draw dynamic guidelines
               ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vSetRearWheelAngle() Dynamic guidelines update required", OSAL_ClockGetElapsedTime()));
               vDrawGuidelines();
            }
            vUpdateGuidelinesDisplay(0);
		}
    }
}

tVoid dispvidctrl_tclControlGraphic::vTraceDebugInformation(tVoid)
{
    ETG_TRACE_FATAL((" CA-RVCG  :: RVC Graphic Configuration"));
    ETG_TRACE_FATAL((" CA-RVCG  :: m_ScreenWidth = .................. %d", m_ScreenWidth));
    ETG_TRACE_FATAL((" CA-RVCG  :: m_ScreenHeight = ................. %d", m_ScreenHeight));
    ETG_TRACE_FATAL((" CA-RVCG  :: m_RVCLayer = ..................... %d", m_RVCLayer));
    ETG_TRACE_FATAL((" CA-RVCG  :: m_RVCGraphicsLayer = ............. %d", m_RVCGraphicsLayer));
    ETG_TRACE_FATAL((" CA-RVCG  :: m_GuidelineSurface = ............. %d", m_GuidelineSurface));
    ETG_TRACE_FATAL((" CA-RVCG  :: m_RVCSurface = ................... %d", m_RVCSurface));
    ETG_TRACE_FATAL((" CA-RVCG  :: m_GuidelineWarningLanguage = ..... %d", m_GuidelineWarningLanguage));
    ETG_TRACE_FATAL((" CA-RVCG  :: m_GuidelineVisibility = .......... %d", m_GuidelineVisibility));
    ETG_TRACE_FATAL((" CA-RVCG  :: m_GuidelineBlackWarningTextBackground = %d", m_GuidelineBlackWarningTextBackground));
    ETG_TRACE_FATAL((" CA-RVCG  :: m_GuidelineWarningText = ......... %d", m_GuidelineWarningText));
    ETG_TRACE_FATAL((" CA-RVCG  :: m_GuidelineDrawingHeight = ....... %d", m_GuidelineDrawingHeight));
    ETG_TRACE_FATAL((" CA-RVCG  :: m_GuidelinesDisplaying = ......... %d", m_GuidelinesDisplaying));
    ETG_TRACE_FATAL((" CA-RVCG  :: m_GuidelinesCurrentlyVisible = ... %d", m_GuidelinesCurrentlyVisible));
    ETG_TRACE_FATAL((" CA-RVCG  :: m_bInitialized = ................. %d", m_bInitialized));
    ETG_TRACE_FATAL((" CA-RVCG  :: m_bGraphicAvailable = ............ %d", m_bGraphicAvailable));
    ETG_TRACE_FATAL((" CA-RVCG  :: m_bFirstDraw = ................... %d", m_bFirstDraw));
    ETG_TRACE_FATAL((" CA-RVCG  :: m_bHmiAvailable = ................ %d", m_bHmiAvailable));
    ETG_TRACE_FATAL((" CA-RVCG  :: m_ScreenXOffset = ................ %d", m_ScreenXOffset));
    ETG_TRACE_FATAL((" CA-RVCG  :: m_ScreenYOffset = ................ %d", m_ScreenYOffset));
    ETG_TRACE_FATAL((" CA-RVCG  :: _dSteeringWheelAngle = ........... %f", _dSteeringWheelAngle));
    ETG_TRACE_FATAL((" CA-RVCG  :: _dRearWheelAngle = ............... %f", _dRearWheelAngle));
    ETG_TRACE_FATAL((" CA-RVCG  :: _bSteeringWheelAngleChanged = .... %d", _bSteeringWheelAngleChanged));
    ETG_TRACE_FATAL((" CA-RVCG  :: _bRearWheelAngleChanged = ........ %d", _bRearWheelAngleChanged));
    ETG_TRACE_FATAL((" CA-RVCG  :: _bThreadActive = ................. %d", _bThreadActive));
    ETG_TRACE_FATAL((" CA-RVCG  :: m_GuidelineCameraType = .......... %d", m_GuidelineCameraType));
    ETG_TRACE_FATAL((" CA-RVCG  :: m_GuidelineCameraSystemType = .... %d", m_GuidelineCameraSystemType));

   if (m_poGraphics_DynGuideline)
   {
      m_poGraphics_DynGuideline->vTraceDebugInformation();
   }
}



tVoid dispvidctrl_tclControlGraphic::vIndVehicleConfigChanged_DynGuideline(tVoid)
{
   if (m_poGraphics_DynGuideline)
   {
      m_poGraphics_DynGuideline->vIndVehicleConfigChanged();
   }
}

vd_rvc_tstVehicleParams* dispvidctrl_tclControlGraphic::vGetVehicleConfigPtr_DynGuideline(tVoid)
{
   if (m_poGraphics_DynGuideline)
   {
      return m_poGraphics_DynGuideline->vGetVehicleConfigPtr();
   }
   else
   {
      return NULL;
   }
}

tVoid dispvidctrl_tclControlGraphic::vSetTemporaryCalibGuidelnStyle(tBool bActive)
{
   if (m_poGraphics_DynGuideline)
   {
      m_poGraphics_DynGuideline->vSetTemporaryCalibGuidelnStyle(bActive);
   }
}
tVoid dispvidctrl_tclControlGraphic::vSetInitialblackscreenMask(tU8 u8camConfigType)
{
    tU8 u8BlackScreenType  = EN_VIDEOPLAYER__BLACKSCREENTYPE_NONE;
    ETG_TRACE_FATAL((" CA-RVCG  :: vSetInitialblackscreenMask = .................. %d", u8camConfigType));
    switch (u8camConfigType)
    {
        case IDX_PORTRAIT_RVC_RENAULT:
        case IDX_PORTRAIT_RVC_RENAULT_UPA:
        case IDX_PORTRAIT_RVC_RENAULT_UPA_HFP:
            //todo: update the blackscreentype for rvc
            //u8BlackScreenType  = EN_VIDEOPLAYER__BLACKSCREENTYPE_RVC_P;
            break;
        case IDX_PORTRAIT_AVM_ENTRY_RENAULT:
        case IDX_PORTRAIT_AVM_ENTRY_RENAULT_UPA:
        case IDX_PORTRAIT_AVM_ENTRY_RENAULT_UPA_HFP:
            u8BlackScreenType  = EN_VIDEOPLAYER__BLACKSCREENTYPE_AVM_P;
            break;
        default:
            break;
    }
    vUpdateGuidelinesDisplay(u8BlackScreenType);
}

tVoid dispvidctrl_tclControlGraphic::vSetVehicleConfig_DynGuideline(tVoid)
{
   vd_rvc_tstKdsVehicleConfig   stKdsVehConfig;

   if (m_poGraphics_DynGuideline)
   {
      stKdsVehConfig.pu8ConfigString = &m_au8GuidelineVehicleParameter[0];  // pointer to 80-byte guideline configuration string as read from early configuration data
      stKdsVehConfig.u8CameraId      = m_GuidelineCameraType;  // camera ID as read from early configuration data
      ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vSetVehicleConfig_DynGuideline() calling m_poGraphics_DynGuideline->vSetVehicleConfig Values from persistent data", OSAL_ClockGetElapsedTime()));
      m_poGraphics_DynGuideline->vSetVehicleConfig(&stKdsVehConfig);
   }
}

int dispvidctrl_tclControlGraphic::_intDecodePNGIntoBuffer(const char* f_fname,  //!< file name
                                                           struct sDecodedImageData& f_decodedData, //!< decoded Data
                                                           const bool& f_corfbFlipImage,  //! flip image
                                                           const bool& f_corfbPadImage,
                                                           bool bSwapBGR)
{
   int readPNGReturnCode = readPNGAndDecodeFromFile(f_fname,f_decodedData,f_corfbFlipImage,f_corfbPadImage,bSwapBGR);
   switch(readPNGReturnCode)
   {
   case 0:
      {
         break;
      }
   case 1:
      {
         ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::_intDecodePNGIntoBuffer() Error opening file handle for %s", OSAL_ClockGetElapsedTime(), f_fname));
         break;
      }
   case 2:
      {
         ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::_intDecodePNGIntoBuffer() Decoding PNG failed - memory allocation for %s", OSAL_ClockGetElapsedTime(), f_fname));
         break;
      }
   case 3:
      {
         ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::_intDecodePNGIntoBuffer() Decoding PNG failed - Create the PNG Info structure for %s", OSAL_ClockGetElapsedTime(), f_fname));
         break;
      }
   case 4:
      {
         ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::_intDecodePNGIntoBuffer() Decoding PNG failed - Create the PNG End Info structure for %s", OSAL_ClockGetElapsedTime(), f_fname));
         break;
      }
   default:
      {
         ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::_intDecodePNGIntoBuffer() Unknown error from readPNGAndDecodeFromFile [%d]", OSAL_ClockGetElapsedTime(), readPNGReturnCode));
         break;
      }
   }
   
   return readPNGReturnCode;
}

tVoid dispvidctrl_tclControlGraphic::vRemoveEarlyOnlyGraphics(void)
{
   // clear the IPA related early masks because HMI will take care of this once available
   if (m_GuidelineCameraSystemType == CAMERA_TYPE_AVM_IPA)
   {
      m_bHmiAvailable = TRUE;

      //Clear the whole buffer - including warning text
      DISPVIDCTRL_NULL_POINTER_CHECK(m_pU8Buffer_DynGuideline);
      memset(m_pU8Buffer_DynGuideline, 0, m_ScreenWidth * m_ScreenHeight * 4);
      
      if (_bThreadActive) {
          vDrawGuidelines();
          vUpdateGuidelinesDisplay(0);
      } else {
          ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclControlGraphic::vRemoveEarlyOnlyGraphics(): Guidelines thread currently not running --> ignore update.", OSAL_ClockGetElapsedTime()));
      }
   }
}

