/*
 * Copyright (C) 2011 Benjamin Franzke
 * Copyright (C) 2010 Intel Corporation
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting documentation, and
 * that the name of the copyright holders not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  The copyright holders make no representations
 * about the suitability of this software for any purpose.  It is provided "as
 * is" without express or implied warranty.
 *
 * THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
 * OF THIS SOFTWARE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include <assert.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/epoll.h>

#include <ilm/ilm_client.h>
#include <ilm/ilm_control.h>

// possibly needed for move to Wayland 1.11
//#include <wayland-server.h>

#include <wayland-client.h>
#include "VideoPlayer_shm.h"
#include "dispvidctrl_AppMain_Trace.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_CLIENT_VIDEOPLAYER
#include "trcGenProj/Header/VideoPlayer_shm.c.trc.h"
#endif

#ifndef UNUSED
#define UNUSED(a) ((a) = (a))
#endif

#ifndef ARRAY_LENGTH
#define ARRAY_LENGTH(a) (sizeof (a) / sizeof (a)[0])
#endif

#define SERVERINFO_GET_CONNECTION_ID    0

#define NUM_BUFFERS 10

static const struct wl_interface *types[] = {
   NULL,
};

static const struct wl_message serverinfo_requests[] = {
   { "get_connection_id", "", types + 0 },
};

static const struct wl_message serverinfo_events[] = {
   { "connection_id", "u", types + 0 },
};

const struct wl_interface serverinfo_interface = {
   "serverinfo", 1,
   ARRAY_LENGTH(serverinfo_requests), serverinfo_requests,
   ARRAY_LENGTH(serverinfo_events), serverinfo_events,
};


struct display {
   struct wl_display *display;
   struct wl_registry *registry;
   struct wl_compositor *compositor;
   struct wl_shm *shm;
   struct serverinfo* wl_ext_serverinfo;
   unsigned int wl_connect_id;
   uint32_t formats;
};

struct buffer {
   struct wl_buffer *buffer;
   void *shm_data;
   int busy;
};

struct window {
   struct display *display;
   int width, height;
   struct wl_surface *surface;
   struct buffer buffers[NUM_BUFFERS];
   struct wl_callback *callback;

   unsigned int surface_id;
   unsigned int layer_id;
};

struct display *display;
struct window *window;

int ret = 0;
unsigned int graphics_layerid = 5001;
unsigned int rvc_layerid;
unsigned int graphics_surfaceid;
unsigned int graphics_layervisibility = 1;
unsigned int rvc_surfaceid;
unsigned int window_height;
unsigned int window_width;
char* guideline_buffer;
static int redrawGuidelines = 0;
enum ten_VideoPlayer_BlackScreenType enBlackScreenTypeRequest = EN_VIDEOPLAYER__BLACKSCREENTYPE_NONE;
bool green_screen;
static bool stop_guideline_visibility = false;
#ifdef GEN3X86
//For LSIM this is always true
bool shm_graphics_ready = true;
#else
bool shm_graphics_ready = false;
#endif //GEN3X86

static int
set_cloexec_or_close(int fd)
{
	long flags;

	if (fd == -1)
		return -1;

	flags = fcntl(fd, F_GETFD);
	if (flags == -1)
    {
	   close(fd);
	   return -1;
    }

	if (fcntl(fd, F_SETFD, flags | FD_CLOEXEC) == -1)
    {
	   close(fd);
	   return -1;
    }

	return fd;

// err:
// 	close(fd);
// 	return -1;
}


static int
create_tmpfile_cloexec(char *tmpname)
{
	int fd;

#ifdef HAVE_MKOSTEMP
	fd = mkostemp(tmpname, O_CLOEXEC);
	if (fd >= 0)
		unlink(tmpname);
#else
	fd = mkstemp(tmpname);
	if (fd >= 0) {
		fd = set_cloexec_or_close(fd);
		unlink(tmpname);
	}
#endif

	return fd;
}

/*
 * Create a new, unique, anonymous file of the given size, and
 * return the file descriptor for it. The file descriptor is set
 * CLOEXEC. The file is immediately suitable for mmap()'ing
 * the given size at offset zero.
 *
 * The file should not have a permanent backing store like a disk,
 * but may have if XDG_RUNTIME_DIR is not properly implemented in OS.
 *
 * The file name is deleted from the file system.
 *
 * The file is suitable for buffer sharing between processes by
 * transmitting the file descriptor over Unix sockets using the
 * SCM_RIGHTS methods.
 */
int
os_create_anonymous_file(off_t size)
{
	static const char template[] = "/weston-shared-XXXXXX";
	const char *path;
	char *name;
	int fd;

	path = getenv("XDG_RUNTIME_DIR");
	if (!path) {
		errno = ENOENT;
		return -1;
	}

	name = malloc(strlen(path) + sizeof(template));
	if (!name)
		return -1;

	strcpy(name, path);
	strcat(name, template);

	fd = create_tmpfile_cloexec(name);

	free(name);

	if (fd < 0)
		return -1;

	if (ftruncate(fd, size) < 0) {
		close(fd);
		return -1;
	}

	return fd;
}

bool bGraphicsSurfaceReady(void)
{
   return shm_graphics_ready;
}

void draw_guidelines(void*);

static void serverinfo_cb_impl(void *data, struct serverinfo *pServerinfo, unsigned int client_handle)
{
   UNUSED(pServerinfo);

   struct display * disp = (struct display *)data;
   disp->wl_connect_id = client_handle;
}

struct serverinfo_listener {
   void (*connection_id)(void *data,
                         struct serverinfo *serverinfo,
                         uint32_t);
}; //lint !e754

struct serverinfo_listener serverinfo_cb = {
   serverinfo_cb_impl
};

static void
buffer_release(void *data, struct wl_buffer *buffer)
{
   struct buffer *mybuf = data;
   UNUSED(buffer);
   mybuf->busy = 0;
}

static const struct wl_buffer_listener buffer_listener = {
   buffer_release
};

void graphiclayervisibility(unsigned int visibility)
{
	ETG_TRACE_FATAL( ( "Recieved visibility is %d",visibility));
	graphics_layervisibility = visibility;
        bool lrc = (ILM_SUCCESS == ilm_layerSetVisibility((unsigned int)graphics_layerid,graphics_layervisibility));
        lrc = (ILM_SUCCESS == ilm_commitChanges());
        ETG_TRACE_USR4(("Comitted done %d",lrc));
}

static int
create_shm_buffer(struct display *display, struct buffer *buffer,
                  int width, int height, uint32_t format)
{ //lint !e578
   struct wl_shm_pool *pool;
   int fd, size, stride;
   void *data;

   stride = width * 4;
   size = stride * height;

   fd = os_create_anonymous_file(size);
   if (fd < 0) {
      ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm create_shm_buffer() creating a buffer file for %d B failed", OSAL_ClockGetElapsedTime(), size));
      return -1;
   }

   /*pointer to the buffer memory which will be displayed*/
   data = mmap(NULL, size, PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0);
   if (data == MAP_FAILED) {
      ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm create_shm_buffer() mmap failed", OSAL_ClockGetElapsedTime()));

      close(fd);
      return -1;
   }

   pool = wl_shm_create_pool(display->shm, fd, size);
   buffer->buffer = wl_shm_pool_create_buffer(pool, 0,
                     width, height,
                     stride, format);
   wl_buffer_add_listener(buffer->buffer, &buffer_listener, buffer);
   wl_shm_pool_destroy(pool);
   close(fd);

   buffer->shm_data = data;

   return 0;
}

static struct window *
create_window(struct display *display, int width, int height)
{//lint !e578
   struct window *window; //lint !e578

   window = calloc(1, sizeof *window);
   if (!window)
      return NULL;

   window->callback = NULL;
   window->display = display;
   window->width = width;
   window->height = height;
   window->surface = wl_compositor_create_surface(display->compositor);

   window->layer_id = 0xFFFFFFFF;
   window->surface_id = 0xFFFFFFFF;

   return window;
}

static void
destroy_window(struct window *window)
{//lint !e578
    int i = 0;
   if (window->callback)
      wl_callback_destroy(window->callback);

   for (i = 0; i < NUM_BUFFERS; i++)
   {
        if (window->buffers[i].buffer)
            wl_buffer_destroy(window->buffers[i].buffer);
   }
   wl_surface_destroy(window->surface);
   free(window);
}

static struct buffer *
window_next_buffer(struct window *window)
{//lint !e578
   struct buffer *buffer = NULL;
   int ret = 0;//lint !e578
   int cur_buf = 0;

   for (cur_buf = 0; cur_buf < NUM_BUFFERS; cur_buf++)
   {
        if (!window->buffers[cur_buf].busy)
        {
            buffer = &window->buffers[cur_buf];
            break;
        }
   }
   if (buffer == NULL)
   {
       /*all buffers are busy, no free buffer available*/
       return NULL;
   }

   if (!buffer->buffer) {
       /*here the memory is actually allocated
        * only ones, in the first run*/
      ret = create_shm_buffer(window->display, buffer,
               window->width, window->height,
               WL_SHM_FORMAT_ARGB8888);

      if (ret < 0)
         return NULL;

      /* paint the padding */
      memset(buffer->shm_data, 0xff,
             window->width * window->height * 4);
   }

   return buffer;
}

static void my_frame_callback(void *data, struct wl_callback *callback, uint32_t time)
{//lint !e578
    UNUSED(data);
    UNUSED(callback);
    UNUSED(time);
    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm my_frame_callback() - frame processed", OSAL_ClockGetElapsedTime()));
}

static const struct wl_callback_listener frame_listener = {
   my_frame_callback
};

static void redraw(void *data)
{
   struct window *window = data; //lint !e578
   struct buffer *buffer;

   if(redrawGuidelines == 1)
   {
      ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm redraw() - redrawGuidelines [%d] - update surface", OSAL_ClockGetElapsedTime(), redrawGuidelines));
      redrawGuidelines = 0;

      buffer = window_next_buffer(window);
      if (!buffer) 
      {
         ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm redraw() Failed to create the first buffer or Both buffers busy at redraw(). Server bug?", OSAL_ClockGetElapsedTime()));
         abort();
      }

      draw_guidelines((void*)buffer->shm_data);

      if (window->callback)
      {
         wl_callback_destroy(window->callback);
      }

      window->callback = wl_surface_frame(window->surface);
      wl_callback_add_listener(window->callback, &frame_listener, window);

      wl_surface_attach(window->surface, buffer->buffer, 0, 0);
      wl_surface_damage(window->surface, 0, 0, window->width, window->height);
      wl_surface_commit(window->surface);
      buffer->busy = 1;
   }
   else
   {
      ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm redraw() - redrawGuidelines [0] - nothing to do", OSAL_ClockGetElapsedTime()));
   }
}

static void
shm_format(void *data, struct wl_shm *wl_shm, uint32_t format)
{
   struct display *d = data;
   UNUSED(wl_shm);
   d->formats |= (1 << format);
}

struct wl_shm_listener shm_listenter = {
   shm_format
};

static void
registry_handle_global(void *data, struct wl_registry *registry,
             uint32_t id, const char *interface, uint32_t version)
{
   struct display *d = data;
   UNUSED(version);

   if (strcmp(interface, "wl_compositor") == 0)
   {
      d->compositor =
         wl_registry_bind(registry,
                id, &wl_compositor_interface, 1);
   }
   else if (strcmp(interface, "wl_shm") == 0)
   {
      d->shm = wl_registry_bind(registry,
                 id, &wl_shm_interface, 1);
      wl_shm_add_listener(d->shm, &shm_listenter, d);
   }
   else if (!strcmp(interface, "serverinfo"))
   {
        d->wl_ext_serverinfo = (struct serverinfo*)wl_registry_bind( registry, id, &serverinfo_interface, 1);

        wl_proxy_add_listener((struct wl_proxy *) d->wl_ext_serverinfo,
                             (void (**)(void)) &serverinfo_cb, data);

        wl_proxy_marshal((struct wl_proxy *) d->wl_ext_serverinfo,
                         SERVERINFO_GET_CONNECTION_ID);
    }
}

static void
registry_handle_global_remove(void *data, struct wl_registry *registry,
               uint32_t name)
{
    UNUSED(data);
    UNUSED(registry);
    UNUSED(name);
}

static const struct wl_registry_listener registry_listener = {
   registry_handle_global,
   registry_handle_global_remove
};

static struct display *
create_display(void)
{
   struct display *display; //lint !e578

   display = malloc(sizeof *display);
   if(display != 0)
   {
      display->display = wl_display_connect(NULL);
      assert(display->display);

      display->formats = 0;
      display->registry = wl_display_get_registry(display->display);
      wl_registry_add_listener(display->registry, &registry_listener, display);
      wl_display_roundtrip(display->display);
      if (display->shm == NULL) {
         ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm create_display() No wl_shm global", OSAL_ClockGetElapsedTime()));
         exit(1);
      }

      wl_display_roundtrip(display->display);

      if (!(display->formats & (1 << WL_SHM_FORMAT_XRGB8888))) {
         ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm create_display() WL_SHM_FORMAT_XRGB32 not available", OSAL_ClockGetElapsedTime()));
         exit(1);
      }
   
      return display;
   }
   return 0;
}

static void
destroy_display(struct display *display)
{//lint !e578
   if (display->shm)
      wl_shm_destroy(display->shm);

   if (display->compositor)
      wl_compositor_destroy(display->compositor);

   wl_registry_destroy(display->registry);
   wl_display_flush(display->display);
   wl_display_disconnect(display->display);
   free(display);
}

static void init_lm_client(struct window* w, uint layer_id, uint surface_id)
{
    ilmErrorTypes ilm_error = (ilmErrorTypes)ILM_FALSE;

    if(NULL == getenv("XDG_RUNTIME_DIR"))
    {
      setenv("XDG_RUNTIME_DIR","/tmp/", 1);
    }

    ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm init_lm_client() CREATE ON LAYERID: %d ,SURFACEID :%d", OSAL_ClockGetElapsedTime(), layer_id, surface_id));

    w->layer_id = layer_id;
    w->surface_id = surface_id;

    // New code added for move to Wayland 1.11
    struct wl_proxy* pxy = (struct wl_proxy*)w->surface;
    t_ilm_nativehandle native_ilm_handle = (w->display->wl_connect_id << 16) | (uint32_t) wl_proxy_get_id(pxy);

    /* Old Code
    struct wl_object* p_obj = (struct wl_object*)w->surface;
    uint native_ilm_handle = (w->display->wl_connect_id << 16) | (uint)p_obj->id;
    */

    ETG_TRACE_FATAL (("[%d ms] dispvidctrl shm init_lm_client() - LayerManagerControl Create Surface %d Layer %d", OSAL_ClockGetElapsedTime(), surface_id, layer_id));
    ilm_error = ilm_surfaceCreate( (t_ilm_nativehandle) native_ilm_handle,
                                                w->width,
                                                w->height,
                                                ILM_PIXELFORMAT_RGBA_8888,
                                                (t_ilm_surface*)&(w->surface_id));
    if (ilm_error == ILM_FAILED)
    {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm init_lm_client() surface could not be created, native_handle: %d", OSAL_ClockGetElapsedTime(), native_ilm_handle));
        return ;
    }

#ifndef VARIANT_S_FTR_DISABLE_FOR_GEN4_TEMP
    ilm_error = ilm_UpdateInputEventAcceptanceOn(surface_id, (ilmInputDevice) (ILM_INPUT_DEVICE_ALL), false);

    if (ilm_error == ILM_FAILED)
    {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm init_lm_client() surface inputs could not be disabled, native_handle: %d", OSAL_ClockGetElapsedTime(), native_ilm_handle));
        return ;
    }
#endif

    ilm_error = ilm_surfaceSetDestinationRectangle((t_ilm_surface)w->surface_id,
                                                    0, 0,
                                                    w->width, w->height);
    if (ilm_error == ILM_FAILED)
    {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm init_lm_client() destination rectangle could not be set, surface id: %d", OSAL_ClockGetElapsedTime(), w->surface_id));
        return;
    }

    ilm_error = ilm_surfaceSetSourceRectangle((t_ilm_surface)w->surface_id,
                                              0, 0,
                                              w->width, w->height);
    if (ilm_error == ILM_FAILED)
    {
       ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm init_lm_client() source rectangle could not be set, surface id: %d", OSAL_ClockGetElapsedTime(), w->surface_id));
       return;
    }

    // Don't set visible until draw complete
    ETG_TRACE_FATAL (("[%d ms] dispvidctrl shm init_lm_client() - LayerManagerControl Surface %d Visibility False", OSAL_ClockGetElapsedTime(), surface_id));
    ilm_error = ilm_surfaceSetVisibility((t_ilm_surface)w->surface_id, ILM_FALSE);
    if (ilm_error == ILM_FAILED)
    {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm init_lm_client() set visibility of surface failed, surface id: %d", OSAL_ClockGetElapsedTime(), w->surface_id));
        return ;
    }

    ETG_TRACE_FATAL (("[%d ms] dispvidctrl shm init_lm_client() - LayerManagerControl Add Surface %d to Layer %d", OSAL_ClockGetElapsedTime(), surface_id, layer_id));
    ilm_error = ilm_layerAddSurface((t_ilm_layer)w->layer_id,
                                    (t_ilm_surface)w->surface_id);
    if (ilm_error == ILM_FAILED)
    {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm init_lm_client() adding surface(id:%d) to layer(id:%d) failed", OSAL_ClockGetElapsedTime(), w->surface_id, w->layer_id));
        return;
    }

    ilm_error = ilm_commitChanges(); //lint !e746
    if (ilm_error == ILM_FAILED)
    {
        ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm init_lm_client() commit changes to Layer Manager failed", OSAL_ClockGetElapsedTime()));
        return ;
    }
}

static void deinit_lm_client(struct window* w)
{
    wl_display_flush( w->display->display );
    wl_display_roundtrip( w->display->display);
    ilm_surfaceRemove(w->surface_id);
    ilm_commitChanges();
}

static int running = 0;
static int stopped = 0;
static int display_control = 0;

void set_green_screen(void)
{
   // For LVDS signal failure only
   // We need to show a green screen
   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm set_green_screen()", OSAL_ClockGetElapsedTime()));
   green_screen = true;
}

void draw_guidelines(void *image)
{
   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm draw_guidelines() - updating window - black screen [%d]", OSAL_ClockGetElapsedTime(), ETG_CENUM(enum ten_VideoPlayer_BlackScreenType, enBlackScreenTypeRequest)));
   void* vp = image;
   char* cp = (char*)vp;
   unsigned int i = 0;
   unsigned int j = 0; // line count
   unsigned int k = 0; // line pixel count

   if(enBlackScreenTypeRequest == EN_VIDEOPLAYER__BLACKSCREENTYPE_NONE) {
      ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm draw_guidelines() - display guidelines", OSAL_ClockGetElapsedTime()));
      memcpy(cp,guideline_buffer,(window_height * window_width * 4));
   }
   else {
      ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm draw_guidelines() - display black screen type", OSAL_ClockGetElapsedTime()));
      
      switch (enBlackScreenTypeRequest)
      {
         case EN_VIDEOPLAYER__BLACKSCREENTYPE_FULL:
            {
               for(i = 0; (i < (window_width * window_height * 4) );)
               {
                  cp[i++] = 0x00; //Blue

                  // For LVDS it is required to show a green screen when there is no LVDS signal
                  if (green_screen == true) {
                     cp[i++] = 0xFF; //Green
                  }
                  else {
                     cp[i++] = 0x00; //No Green
                  }
                  cp[i++] = 0x00; //Red
                  cp[i++] = 0xFF; //Alpha - set value for the pixel to "not transparent"
               }
            }
            break;
         case EN_VIDEOPLAYER__BLACKSCREENTYPE_LHD:
            {
               // for 128 lines all pixel = transparent
               // for 533 lines the first 781 pixel of each line = black,
               //     the remaining 499 pixel of these line are transparent
               // for 107 lines all pixel = transparent
               
               if ((window_width == 1280) && (window_height == 768))
               {
                  i = 0;                  
                  for(j = 0; (j < (window_width * 128)); j++) // for 128 lines * 1280 pixel
                  {
                     cp[i++] = 0x00; //Blue
                     cp[i++] = 0x00; //Green
                     cp[i++] = 0x00; //Red
                     cp[i++] = 0x00; //Alpha - set value for the pixel to "transparent"
                  }
                  for(j = 0; (j < 533); j++)  // for 533 lines
                  {
                     // for 1280 pixel

                     for(k = 0; (k < 781); k++) // for 781 pixel
                     {
                        cp[i++] = 0x00; //Blue
                        cp[i++] = 0x00; //Green
                        cp[i++] = 0x00; //Red
                        cp[i++] = 0xFF; //Alpha - set value for the pixel to "not transparent"
                     }
                     for(k = 0; (k < 499); k++) // for 499 pixel
                     {
                        cp[i++] = 0x00; //Blue
                        cp[i++] = 0x00; //Green
                        cp[i++] = 0x00; //Red
                        cp[i++] = 0x00; //Alpha - set value for the pixel to "transparent"
                     }
                  }
                  for(j = 0; (j < (window_width * 107)); j++) // for 107 lines * 1280 pixel
                  {
                     cp[i++] = 0x00; //Blue
                     cp[i++] = 0x00; //Green
                     cp[i++] = 0x00; //Red
                     cp[i++] = 0x00; //Alpha - set value for the pixel to "transparent"
                  }
               }
            }
            break;
         case EN_VIDEOPLAYER__BLACKSCREENTYPE_RHD:
            {
               // for 128 lines all pixel = transparent
               // for 533 lines the first 499 pixel of each line = transparent,
               //     the remaining 781 pixel of these line are black
               // for 107 lines all pixel = transparent
               
               if ((window_width == 1280) && (window_height == 768))
               {
                  i = 0;                  
                  for(j = 0; (j < (window_width * 128)); j++) // for 128 lines * 1280 pixel
                  {
                     cp[i++] = 0x00; //Blue
                     cp[i++] = 0x00; //Green
                     cp[i++] = 0x00; //Red
                     cp[i++] = 0x00; //Alpha - set value for the pixel to "transparent"
                  }
                  for(j = 0; (j < 533); j++)  // for 533 lines
                  {
                     // for 1280 pixel

                     for(k = 0; (k < 499); k++) // for 499 pixel
                     {
                        cp[i++] = 0x00; //Blue
                        cp[i++] = 0x00; //Green
                        cp[i++] = 0x00; //Red
                        cp[i++] = 0x00; //Alpha - set value for the pixel to "transparent"
                     }
                     for(k = 0; (k < 781); k++) // for 781 pixel
                     {
                        cp[i++] = 0x00; //Blue
                        cp[i++] = 0x00; //Green
                        cp[i++] = 0x00; //Red
                        cp[i++] = 0xFF; //Alpha - set value for the pixel to "not transparent"
                     }
                  }
                  for(j = 0; (j < (window_width * 107)); j++) // for 107 lines * 1280 pixel
                  {
                     cp[i++] = 0x00; //Blue
                     cp[i++] = 0x00; //Green
                     cp[i++] = 0x00; //Red
                     cp[i++] = 0x00; //Alpha - set value for the pixel to "transparent"
                  }
               }
            }
            break;
         case EN_VIDEOPLAYER__BLACKSCREENTYPE_WIDE:
            {
               // for 71  lines all pixel = transparent
               // for 590 lines all pixel are black
               // for 107 lines all pixel = transparent
               
               if ((window_width == 1280) && (window_height == 768))
               {
                  i = 0;                  
                  for(j = 0; (j < (window_width * 71)); j++) // for 128 lines * 1280 pixel
                  {
                     cp[i++] = 0x00; //Blue
                     cp[i++] = 0x00; //Green
                     cp[i++] = 0x00; //Red
                     cp[i++] = 0x00; //Alpha - set value for the pixel to "transparent"
                  }
                  for(j = 0; (j < (window_width *590)); j++)  // for 533 lines * 1280 pixel
                  {
                     cp[i++] = 0x00; //Blue
                     cp[i++] = 0x00; //Green
                     cp[i++] = 0x00; //Red
                     cp[i++] = 0xFF; //Alpha - set value for the pixel to "transparent"
                  }
                  for(j = 0; (j < (window_width * 107)); j++) // for 107 lines * 1280 pixel
                  {
                     cp[i++] = 0x00; //Blue
                     cp[i++] = 0x00; //Green
                     cp[i++] = 0x00; //Red
                     cp[i++] = 0x00; //Alpha - set value for the pixel to "transparent"
                  }
               }
            }
            break;
         case EN_VIDEOPLAYER__BLACKSCREENTYPE_RVC_P:
            break;
         case EN_VIDEOPLAYER__BLACKSCREENTYPE_RVC_L:
            break;
         case EN_VIDEOPLAYER__BLACKSCREENTYPE_AVM_P:
		    {
               // for 205  lines all pixel = black
               // for 489 lines all pixel are transparent
		        //Avm video width is 800*489
		        //in 489 picel first 10 lines are black, next 800 lines are transparent and last 10 lines are black
               // for 330 lines all pixel = black
               
               if ((window_width == 820) && (window_height == 1024))
               {
                  i = 0;                  
                  for(j = 0; (j < (window_width * 205)); j++) // for 128 lines * 1280 pixel
                  {
                     cp[i++] = 0x00; //Blue
                     cp[i++] = 0x00; //Green
                     cp[i++] = 0x00; //Red
                     cp[i++] = 0xFF; //Alpha - set value for the pixel to "transparent"
                  }
                  for(j = 0; (j < (window_width *489)); j++)  // for 533 lines * 1280 pixel
                  {
                     /*for(k = 0; (k < 10); k++) // for 499 pixel
                     {
                        cp[i++] = 0x00; //Blue
                        cp[i++] = 0x00; //Green
                        cp[i++] = 0x00; //Red
                        cp[i++] = 0xFF; //Alpha - set value for the pixel to "black"
                     }
                     for(k = 10; (k < 800); k++) // for 800 pixel
                     {
                        cp[i++] = 0x00; //Blue
                        cp[i++] = 0x00; //Green
                        cp[i++] = 0x00; //Red
                        cp[i++] = 0x00; //Alpha - set value for the pixel to " transparent"
                     }
                     for(k = 800; (k < 820); k++) // for 800 pixel
                     {*/
                        cp[i++] = 0x00; //Blue
                        cp[i++] = 0x00; //Green
                        cp[i++] = 0x00; //Red
                        cp[i++] = 0x00; //Alpha - set value for the pixel to "black"

                  }
                  for(j = 0; (j < (window_width * 330)); j++) // for 107 lines * 1280 pixel
                  {
                     cp[i++] = 0x00; //Blue
                     cp[i++] = 0x00; //Green
                     cp[i++] = 0x00; //Red
                     cp[i++] = 0xFF; //Alpha - set value for the pixel to "transparent"
                  }
               }
            }
            break;
         case EN_VIDEOPLAYER__BLACKSCREENTYPE_AVM_L:
            break;
         case EN_VIDEOPLAYER__BLACKSCREENTYPE_MVC_L_RHD:
            {
               // for 402 lines (first 710 pixels) are transparent
               // for 402 lines (remaining 90 pixels) are black
               // for remaining 78 lines all pixel are transparent
               
               if ((window_width == 800) && (window_height == 480))
               {
                  i = 0;
                  for(j = 0; j < 402; j++) // for 402 lines
                  {
                     // next 710 pixels
                     memset((cp+i), 0x00, (710 << 2));
                     i += 710 << 2; // (710*4)
                     
                     for (k = 0; k < 90; k++) // first 90 pixels
                     {
                        cp[i++] = 0x00; //Blue
                        cp[i++] = 0x00; //Green
                        cp[i++] = 0x00; //Red
                        cp[i++] = 0xFF; //Alpha - set value for the pixel to "not transparent"
                     }

                     //for (k = 0; k < 710; k++)
                     //{
                     //   cp[i++] = 0x00; //Blue
                     //   cp[i++] = 0x00; //Green
                     //   cp[i++] = 0x00; //Red
                     //   cp[i++] = 0x00; //Alpha - set value for the pixel to "transparent"
                     //}
                  }

                  //for(j = 0; j < 78; j++)  // for 78 lines (zone3)
                  {
                     // set value for the pixel to "transparent"
                     memset((cp+i), 0x00, (4*78*window_width));
                     i += (4*78*window_width);
                     
                     //cp[i++] = 0x00; //Blue
                     //cp[i++] = 0x00; //Green
                     //cp[i++] = 0x00; //Red
                     //cp[i++] = 0x00; //Alpha - set value for the pixel to "transparent"
                  }
               }
            }
            break;
         case EN_VIDEOPLAYER__BLACKSCREENTYPE_MVC_L_LHD:
            {
               // for 402 lines (first 90 pixels) are black
               // for 402 lines (remaining 710 pixels) are transparent
               // for remaining 78 lines all pixel are transparent
               
               if ((window_width == 800) && (window_height == 480))
               {
                  i = 0;
                  for(j = 0; j < 402; j++) // for 402 lines
                  {
                     for (k = 0; k < 90; k++) // first 90 pixels
                     {
                        cp[i++] = 0x00; //Blue
                        cp[i++] = 0x00; //Green
                        cp[i++] = 0x00; //Red
                        cp[i++] = 0xFF; //Alpha - set value for the pixel to "not transparent"
                     }
                     
                     // next 710 pixels
                     memset((cp+i), 0x00, (4*710));
                     i += (4*710);
                     
                     //for (k = 0; k < 710; k++)
                     //{
                     //   cp[i++] = 0x00; //Blue
                     //   cp[i++] = 0x00; //Green
                     //   cp[i++] = 0x00; //Red
                     //   cp[i++] = 0x00; //Alpha - set value for the pixel to "transparent"
                     //}
                  }

                  for(j = 0; j < 78; j++)  // for 78 lines (zone3)
                  {
                     // set value for the pixel to "not transparent"

                     cp[i++] = 0x00; //Blue
                     cp[i++] = 0x00; //Green
                     cp[i++] = 0x00; //Red
                     cp[i++] = 0xFF; //Alpha - set value for the pixel to "not transparent"
                  }
               }
            }
            break;
         default:
            break;
      }
      
      
      //enBlackScreenTypeRequest = EN_VIDEOPLAYER__BLACKSCREENTYPE_NONE;
   }
}

// For LSIM - there is no graphics
void display_guidelines(unsigned int width, unsigned int height, unsigned int layer, unsigned int graphics_layer, unsigned int surface, unsigned int rvc_surface, char* buff)
{
   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_guidelines() - width [%d], height [%d], layer [%d], graphics layer [%d], surface [%d] rvc surface [%d]", OSAL_ClockGetElapsedTime(), width, height, layer, graphics_layer, surface, rvc_surface));
#ifndef GEN3X86
   rvc_layerid = layer;
   graphics_layerid = graphics_layer;
   graphics_surfaceid = surface;
   rvc_surfaceid = rvc_surface;
   window_height = height;
   window_width = width;
   guideline_buffer = buff;

   int ret = 0;//lint !e578

   display = create_display();
   window = create_window(display, window_width, window_height);
   if (!window)
   {
      ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_guidelines() - window could not be created", OSAL_ClockGetElapsedTime()));
      return;
   }
   init_lm_client(window,graphics_layerid,graphics_surfaceid);

   /* Initialise damage to full surface, so the padding gets painted */
   wl_surface_damage(window->surface, 0, 0, window->width, window->height);

   redrawGuidelines = 1;
   display_control = 0;
   stopped = 0;
   running = 1;

   //frame callback has to be deployed before surface_commit(in redraw function)
   if (window->callback)
   {
      wl_callback_destroy(window->callback);
   }

   window->callback = wl_surface_frame(window->surface);
   wl_callback_add_listener(window->callback, &frame_listener, window);

   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_guidelines() - graphics ready", OSAL_ClockGetElapsedTime()));
   shm_graphics_ready = true;

   redraw(window);

   while (running && ret != -1)
   {
      ret = wl_display_dispatch(display->display);

      // wl_display_dispatch exits after surface drawn and callback has returned
      ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_guidelines() - wl_display_dispatch() exited - redrawGuidelines [%d]", OSAL_ClockGetElapsedTime(), redrawGuidelines));

      //Perform LayerManagerControl commands after surface has been updated
      if(enBlackScreenTypeRequest != EN_VIDEOPLAYER__BLACKSCREENTYPE_NONE)
      {
          enBlackScreenTypeRequest = EN_VIDEOPLAYER__BLACKSCREENTYPE_NONE;
          redrawGuidelines = 0;

          ETG_TRACE_FATAL (("[%d ms] dispvidctrl shm display_guidelines() - LayerManagerControl Set Layer %d Visibility -> 1, Surface %d Visibility -> 1, Layer %d Visibility -> 1", OSAL_ClockGetElapsedTime(), graphics_layerid, graphics_surfaceid, rvc_layerid));
          bool lRc = (ILM_SUCCESS == ilm_surfaceSetVisibility((unsigned int)graphics_surfaceid,1));
          lRc = lRc && (ILM_SUCCESS == ilm_layerSetVisibility((unsigned int)graphics_layerid,graphics_layervisibility));
          lRc = lRc && (ILM_SUCCESS == ilm_layerSetVisibility((unsigned int)rvc_layerid,1));

          // Commit changes to ILM
          lRc = lRc && (ILM_SUCCESS == ilm_commitChanges());
          if(lRc != TRUE) {
              ETG_TRACE_FATAL (("[%d ms] dispvidctrl shm display_guidelines() - LayerManagerControl Commit Failed [%d]", OSAL_ClockGetElapsedTime(), lRc));
          }
      }
      else
      {
          ETG_TRACE_FATAL (("[%d ms] dispvidctrl shm display_guidelines() - LayerManagerControl Set Surface [%d] Visibility -> 1, both Layers [%d, %d] Visibility -> 1", OSAL_ClockGetElapsedTime(), graphics_surfaceid, rvc_layerid, graphics_layerid));
          bool lRc = (ILM_SUCCESS == ilm_layerAddSurface((unsigned int)graphics_layerid, (unsigned int)graphics_surfaceid));
          lRc = lRc && (ILM_SUCCESS == ilm_surfaceSetVisibility((unsigned int)graphics_surfaceid,1));
          lRc = lRc && (ILM_SUCCESS == ilm_layerSetVisibility((unsigned int)rvc_layerid,1));
          lRc = lRc && (ILM_SUCCESS == ilm_layerSetVisibility((unsigned int)graphics_layerid,graphics_layervisibility));

          // Commit changes to ILM
          lRc = lRc && (ILM_SUCCESS == ilm_commitChanges());
          if(lRc != TRUE) {
              ETG_TRACE_FATAL (("[%d ms] dispvidctrl shm display_guidelines() - LayerManagerControl Commit Failed [%d]", OSAL_ClockGetElapsedTime(), lRc));
          }
      }

      // Wait for any update to the Guidelines buffer
      while(redrawGuidelines == 0)
      {
          usleep(1000);
      }

      ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_guidelines() - redrawGuidelines [%d] - perform update", OSAL_ClockGetElapsedTime(), redrawGuidelines));
      //Perform the update
      //frame callback has to be deployed before surface_commit(in redraw function)
      if (window->callback)
      {
         wl_callback_destroy(window->callback);
      }

      window->callback = wl_surface_frame(window->surface);
      wl_callback_add_listener(window->callback, &frame_listener, window);

      redraw(window);
   }

   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm display_guidelines() - exiting", OSAL_ClockGetElapsedTime()));

   deinit_lm_client(window);
   destroy_window(window);
   destroy_display(display);
   stopped = 1;
#endif //GEN3X86
}

// For LSIM - there is no graphics
void update_guidelines(enum ten_VideoPlayer_BlackScreenType enBlackScreenType)
{
#ifndef GEN3X86
   ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm update_guidelines() - entered - enBlackScreenType [%d], running [%d]", OSAL_ClockGetElapsedTime(), ETG_CENUM(enum ten_VideoPlayer_BlackScreenType, enBlackScreenType), running));
   if(running == 1)
   {
      ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm update_guidelines() - currently displaying graphics - setting redrawGuidelines = 1 to force update", OSAL_ClockGetElapsedTime()));
      redrawGuidelines = 1;
      enBlackScreenTypeRequest = enBlackScreenType;
   }
   else
   {
      ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl shm update_guidelines() - not currently displaying graphics - ignoring", OSAL_ClockGetElapsedTime()));
   }
#endif //GEN3X86
}

// For LSIM - there is no graphics
void end_display_guidelines()
{
#ifndef GEN3X86
   stop_guideline_visibility = true;
   green_screen = false;
#endif //GEN3X86
}
