/*****************************************************************************************
*FILE         : VideoPlayer_PNG_Decoder.cpp
*
*SW-COMPONENT :
*DESCRIPTION  : Implementation of the PNG Decoder using libpng, with modifications from CM-AI/EAR-Abraham
*
*AUTHOR       : RBIN/ECM1-Thomas
*COPYRIGHT    : (c) 2007 Robert Bosch India
*
*HISTORY      : 18/06/2007 Initial Revision
****************************************************************************************/
/**
* \file VideoPlayer_PNG_Decoder.cpp
*
* Implementation of the PNG Decoder using libpng
*
* Initial Version: 18/06/2007  RBIN/ECM1-Thomas
*/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "VideoPlayer_PNG_Decoder.h"
#include "dispvidctrl_AppMain_Trace.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_CLIENT_VIDEOPLAYER
#include "trcGenProj/Header/VideoPlayer_PNG_Decoder.cpp.trc.h"
#endif

//! function to read and decode png from file
int readPNGAndDecodeFromFile(const char* f_fname,  //!< file name
                             struct sDecodedImageData& f_decodedData, //!< decoded Data
                             const bool& f_corfbFlipImage,  //! flip image
                             const bool& f_corfbPadImage,
                             bool bSwapBGR)
{
   int rc;
   sPNGData l_pngData;
   if(readPngDataFromFile(f_fname,l_pngData)!=0)
   {
      // error reading file
      return 1;
   }
   // decoding
   int decodedReturn = rl_tclPNG_Decoder::bDecodePNG(f_decodedData,l_pngData,f_corfbFlipImage,f_corfbPadImage,bSwapBGR);
   if (decodedReturn != 0)
   {
      return decodedReturn;
   }
   else
   {
      rc=0;
   }
   free(l_pngData.m_pu8Data);
   return rc;
}


//! reading data from file into an buffer
int readPngDataFromFile(const char* f_fname,struct sPNGData& f_pngData)
{
   FILE* l_fid=fopen(f_fname,"rb");
   if(l_fid==0L)
   {
      return 1;
   }
   // feststellen Dateigroesse
   fseek(l_fid, 0L, SEEK_END);
   f_pngData.m_u32Size=(unsigned int)ftell(l_fid);
   // Eingabebuffer allokieren
   f_pngData.m_pu8Data = (unsigned char*)malloc(f_pngData.m_u32Size);

   // Dateiinhalte lesen
   fseek(l_fid,0L,SEEK_SET);
   int retVal = ( fread(f_pngData.m_pu8Data,1,f_pngData.m_u32Size,l_fid) == f_pngData.m_u32Size ) ? 0 : 1;

   // Datei schliessen
   fclose(l_fid);

   // flag setzen
   f_pngData.m_u32DataAlreadyRead = 0;

   return retVal;

}

//! get nearest power of 2 value
unsigned int u32GetNearestPowerOfTwoValue(unsigned int u32Value)
{
   unsigned int u32PowerOfTwoValue = 2;

   if (u32Value > 2)
   {
      if(0 == (u32Value & (u32Value - 1))) //check if u32Value is already a power of two
      {
         u32PowerOfTwoValue = u32Value;
      }
      else
      {
         u32Value <<= 1; //We want the next highest power of two
         u32PowerOfTwoValue = 0x80000000; //highest power of two if tU32 is used for storage
         while (!(u32PowerOfTwoValue & u32Value))
         {
            u32PowerOfTwoValue >>= 1;
         }
      }
   }
   return u32PowerOfTwoValue;
}



/*****************************************************************************************
* FUNCTION    : pvMemoryAllocator
*
* DESCRIPTION : The memory allocator function to be handed over to libpng
*
* PARAMETER   : png_size_t size
* RETURNVALUE : tVoid*
*
* HISTORY     :
*  18/06/2007 ECM1-Thomas  Initial Revision
****************************************************************************************/
void* rl_tclPNG_Decoder::pvMemoryAllocator(png_structp, png_size_t size)
{
   return (void*) malloc(size);
}


/*****************************************************************************************
* FUNCTION    : vMemoryDeallocator
*
* DESCRIPTION : The memory deallocator function to be handed over to libpng
*
* PARAMETER   : png_structp png_ptr, png_voidp ptr
* RETURNVALUE : tVoid
*
* HISTORY     :
*  18/06/2007 ECM1-Thomas  Initial Revision
****************************************************************************************/
void rl_tclPNG_Decoder::vMemoryDeallocator(png_structp, png_voidp ptr)
{
   if (0L != ptr)
   {
      free(ptr);
   }
}


/*****************************************************************************************
* FUNCTION    : vRead_Function
*
* DESCRIPTION : The read function that reads data from the sPNGData structure. Handed over to libpng
*
* PARAMETER   : png_structp png_ptr, png_bytep data, png_size_t length
* RETURNVALUE : tVoid
*
* HISTORY     :
*  18/06/2007 ECM1-Thomas  Initial Revision
****************************************************************************************/
void rl_tclPNG_Decoder::vRead_Function(png_structp png_ptr, png_bytep data, png_size_t length)
{
   png_size_t u32NoOfBytesThatCanBeRead = length;

   if(0L == png_ptr)
   {
      return;
   }

   sPNGData* poPNGFileData = (sPNGData*)png_get_io_ptr(png_ptr);
   if (0L != poPNGFileData && 0L != poPNGFileData->m_pu8Data)
   {
      if ((poPNGFileData->m_u32DataAlreadyRead + length) > poPNGFileData->m_u32Size)
      {
         u32NoOfBytesThatCanBeRead = poPNGFileData->m_u32Size - poPNGFileData->m_u32DataAlreadyRead;
      }
      (void)memcpy(data,
                   &(poPNGFileData->m_pu8Data[poPNGFileData->m_u32DataAlreadyRead]),
                   u32NoOfBytesThatCanBeRead);
      poPNGFileData->m_u32DataAlreadyRead += u32NoOfBytesThatCanBeRead;
      if (u32NoOfBytesThatCanBeRead != length)
      {
         rl_tclPNG_Decoder::vError_Function(png_ptr, "Read Error");
      }
   }
}


/*****************************************************************************************
* FUNCTION    : vError_Function
*
* DESCRIPTION : The error function that reports errors. Handed over to libpng
*
* PARAMETER   : png_structp png_ptr, png_const_charp error_msg
* RETURNVALUE : tVoid
*
* HISTORY     :
*  18/06/2007 ECM1-Thomas  Initial Revision
****************************************************************************************/
void rl_tclPNG_Decoder::vError_Function(png_structp, png_const_charp error_msg)
{
   ETG_TRACE_FATAL( ( "[%d ms] vError_Function() decoder error: %s", OSAL_ClockGetElapsedTime(), error_msg));
}


/*****************************************************************************************
* FUNCTION    : vWarning_Function
*
* DESCRIPTION : The warning function that reports warnings. Handed over to libpng
*
* PARAMETER   : png_structp png_ptr, png_const_charp error_msg
* RETURNVALUE : tVoid
*
* HISTORY     :
*  18/06/2007 ECM1-Thomas  Initial Revision
****************************************************************************************/
void rl_tclPNG_Decoder::vWarning_Function(png_structp, png_const_charp warning_msg)
{
   ETG_TRACE_FATAL( ( "[%d ms] vWarning_Function() decoder warning: %s", OSAL_ClockGetElapsedTime(), warning_msg));
}


/*****************************************************************************************
* FUNCTION    : bDecodePNG
*
* DESCRIPTION : Decodes the PNG data in roPNGFileData to fill the roDecodedImage object
*
* PARAMETER   : struct sDecodedImageData& roDecodedImage, const struct sPNGData& roPNGFileData
* RETURNVALUE : tBool
*
* HISTORY     :
*  18/06/2007 ECM1-Thomas  Initial Revision
****************************************************************************************/
int rl_tclPNG_Decoder::bDecodePNG(struct sDecodedImageData& roDecodedImage,
                                  struct sPNGData& roPNGFileData,
                                  const bool& corfbFlipImage,
                                  const bool& corfbPadImage,
                                  bool bSwapBGR)
{
   bool bReturn = false;

   //Create the PNG structure for decoding
   png_structp png_ptr = png_create_read_struct_2(PNG_LIBPNG_VER_STRING,
                         /*(png_voidp)user_error_ptr*/0L,
                         vError_Function, vWarning_Function,
                         /*(png_voidp)user_mem_ptr*/ 0L,
                         pvMemoryAllocator, vMemoryDeallocator);

   if (0L == png_ptr)
   {
      //memory allocation failed
      return 2; // bReturn;
   }

   //Create the PNG Info structure
   png_infop info_ptr = png_create_info_struct(png_ptr);
   if (0L == info_ptr)
   {
      //Destroy the PNG structure
      png_destroy_read_struct(&png_ptr, (png_infopp)NULL, (png_infopp)NULL);
      return 3; // bReturn;
   }

   //Create the PNG End Info structure
   png_infop end_info = png_create_info_struct(png_ptr);
   if (0L == end_info)
   {
      //Destroy the PNG structure and the PNG Info Structure
      png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);
      return 4; // bReturn;
   }

   //Set the read function to read PNG encoded data
   png_set_read_fn(png_ptr, (png_voidp) (&roPNGFileData), vRead_Function);

   //Handle all chunks
   png_set_keep_unknown_chunks(png_ptr, PNG_HANDLE_CHUNK_ALWAYS, NULL, 0);

   //Read Image info
   png_read_info(png_ptr, info_ptr);

   //Get image height and width
   unsigned int u32PNGWidth                = png_get_image_width(png_ptr, info_ptr);
   roDecodedImage.m_u32Width       =   corfbPadImage ? ((u32PNGWidth + 3) & (~3)) : u32PNGWidth; //padding
   roDecodedImage.m_u32Height      = png_get_image_height(png_ptr, info_ptr);

   unsigned char u8BitDepth  = png_get_bit_depth(png_ptr, info_ptr);
   //We need bit depth to be 8
   if (16 == u8BitDepth)  //Strip the upper byte
   {
      png_set_strip_16(png_ptr);
   }
   else if (u8BitDepth < 8) //Convert to bit depth of 8
   {
      png_set_packing(png_ptr);
   }

   unsigned char u8ColorType = png_get_color_type(png_ptr, info_ptr);
   //We need color to be in RGB
   if (PNG_COLOR_TYPE_GRAY == u8ColorType
         ||
         PNG_COLOR_TYPE_GRAY_ALPHA == u8ColorType)
   {
      png_set_gray_to_rgb(png_ptr);
   }
   else if (u8ColorType == PNG_COLOR_TYPE_RGB)
   {
      png_set_filler(png_ptr, 0xFF, PNG_FILLER_AFTER);
   }
   else if (u8ColorType == PNG_COLOR_TYPE_PALETTE)
   {
      png_set_palette_to_rgb(png_ptr);
   }
   if (bSwapBGR && (u8ColorType == PNG_COLOR_TYPE_RGB || u8ColorType == PNG_COLOR_TYPE_RGB_ALPHA))
      png_set_bgr(png_ptr);
   
   //Get No of channels
   roDecodedImage.m_u8NoOfChannels = png_get_channels(png_ptr, info_ptr);

   //No Of bytes per row (width * no of channels)
   unsigned int u32RowBytes  = png_get_rowbytes(png_ptr, info_ptr);
   //To take care of padding
   u32RowBytes *= roDecodedImage.m_u32Width;
   if(u32PNGWidth!=0){ 	 //FOR  COVERITY   
   u32RowBytes /= u32PNGWidth;
   }

   if (PNG_COLOR_TYPE_RGB == u8ColorType) //Add the filler bytes to channels and rowbytes
   {
      ++(roDecodedImage.m_u8NoOfChannels);
      u32RowBytes += roDecodedImage.m_u32Width;
   }
   else if(PNG_COLOR_TYPE_PALETTE == u8ColorType)
   {
      roDecodedImage.m_u8NoOfChannels = 4;
      u32RowBytes = roDecodedImage.m_u32Width * roDecodedImage.m_u8NoOfChannels;
   }
   roDecodedImage.m_u8BitsPerPixel = (unsigned char)((roDecodedImage.m_u8NoOfChannels * 8) & 0xFF);
   // check palette transparency
   bool bUseColorKey = false;
   png_bytep trans;
   int num_trans;
   png_color_16p trans_values;

   if (png_get_tRNS(png_ptr, info_ptr, &trans, &num_trans,
                    &trans_values))
   {
      png_uint_32 bit_depth = png_get_bit_depth(png_ptr, info_ptr);
      png_uint_32 color_type = png_get_color_type(png_ptr, info_ptr);

      int sample_max = (1 << bit_depth); //lint !e701 //lint !e451 repeatedly included header file without standard include guard
      /* libpng doesn't reject a tRNS chunk with out-of-range samples */
      if (!((bit_depth != 8) ||
            (color_type == PNG_COLOR_TYPE_GRAY && (int)trans_values->gray > sample_max) ||
            (color_type == PNG_COLOR_TYPE_RGB &&
             ((int)trans_values->red > sample_max ||(int)trans_values->green > sample_max || (int)trans_values->blue > sample_max))))
      {
         bUseColorKey = true;
         roDecodedImage.m_bAlpha = true;
      }
   }
   if (PNG_COLOR_MASK_ALPHA == u8ColorType)
   {
      roDecodedImage.m_bAlpha = true;
   }

   //Allocate memory for the decoded image
   roDecodedImage.m_pu8Data = (unsigned char*)malloc(roDecodedImage.m_u32Height * u32RowBytes);
   if (0L == roDecodedImage.m_pu8Data)
   {
      png_destroy_read_struct(&png_ptr, &info_ptr, &end_info);
      return bReturn;
   }
   if (u32PNGWidth != roDecodedImage.m_u32Width)
   {
      //for the padding bytes to be set to zero
      memset(roDecodedImage.m_pu8Data, 0, roDecodedImage.m_u32Height * u32RowBytes);
   }

   //pointers to each row (no of rows = height) of the decoded image
   unsigned char** ppu8RowPointers = (unsigned char**)malloc(sizeof(unsigned char*) * roDecodedImage.m_u32Height);
   if (0L == ppu8RowPointers)
   {
      png_destroy_read_struct(&png_ptr, &info_ptr, &end_info);
      free(roDecodedImage.m_pu8Data);
      roDecodedImage.m_pu8Data = 0L;
      return bReturn;
   }

   //Set the row pointers
   if (true == corfbFlipImage)//PNG's origin is the top-left corner, we'll flip it vertically to make the bottom-left the origin
   {
      unsigned int u32Ind = roDecodedImage.m_u32Height;
      unsigned int u32RowInd = 0;
      while (u32Ind--)
      {
         ppu8RowPointers[u32RowInd++] = &(roDecodedImage.m_pu8Data[u32Ind * u32RowBytes]);
      }
   }
   else
   {
      for (unsigned int u32Ind = 0; u32Ind < roDecodedImage.m_u32Height; ++u32Ind)
      {
         ppu8RowPointers[u32Ind] = &(roDecodedImage.m_pu8Data[u32Ind * u32RowBytes]);
      }
   }

   //Decode the image
   png_read_image(png_ptr, ppu8RowPointers);

   //Destroy PNG structure
   png_read_end(png_ptr, end_info);
   png_destroy_read_struct(&png_ptr, &info_ptr, &end_info);

   //delete row pointers
   free(ppu8RowPointers);

   if (bUseColorKey)
   {
      // png with transparency -> set color key to black without alpha
      for (unsigned int i = 0; i < (roDecodedImage.m_u32Height * u32RowBytes); i += 4)
      {
         if (roDecodedImage.m_pu8Data[i ] == trans_values->red &&
               roDecodedImage.m_pu8Data[i + 1] == trans_values->green &&
               roDecodedImage.m_pu8Data[i + 2] == trans_values->blue)
         {
            roDecodedImage.m_pu8Data[i]      = 0x00;
            roDecodedImage.m_pu8Data[i + 1]  = 0x00;
            roDecodedImage.m_pu8Data[i + 2]  = 0x00;
            roDecodedImage.m_pu8Data[i + 3]  = 0x00;
         }
      }
   }
   else
   {
      if (PNG_COLOR_TYPE_RGB == u8ColorType
            ||
            PNG_COLOR_TYPE_PALETTE == u8ColorType
            ||
            PNG_COLOR_TYPE_RGB_ALPHA == u8ColorType)
      {
         for (unsigned int i = 0; i < (roDecodedImage.m_u32Height * u32RowBytes); i += 4)
         {
            if (roDecodedImage.m_pu8Data[i ] == 0xff
                  &&
                  roDecodedImage.m_pu8Data[i + 1] == 0x00
                  &&
                  roDecodedImage.m_pu8Data[i + 2] == 0xff)
            {
               roDecodedImage.m_pu8Data[i]      = 0x00;
               roDecodedImage.m_pu8Data[i + 1]  = 0x00;
               roDecodedImage.m_pu8Data[i + 2]  = 0x00;
               roDecodedImage.m_pu8Data[i + 3]  = 0x00;
            }
         }
      }
   }

   return 0; //bReturn;
}
