/******************************************************************************/
/**
* \file    vd_rvc_tclGraphics_DynGuideline_Util.cpp
* \ingroup
*
* \brief
*
* \remark  Copyright : (c) 2012 Robert Bosch GmbH, Hildesheim
* \remark  Author    :
* \remark  Scope     :
*
* \todo
*/
/******************************************************************************/


/*******************************************************************************
                        Includes
*******************************************************************************/

#include "vd_rvc_tclGraphics_DynGuideline_Util.h"

#include <cmath>

/**************************************************************************************************/
// methods of the vd_rvc_t3Vector class
/**************************************************************************************************/ 

vd_rvc_t3Vector::vd_rvc_t3Vector(tVoid)
   : _x(0), _y(0), _z(0)
{
}

vd_rvc_t3Vector::vd_rvc_t3Vector(tDouble dX, tDouble dY, tDouble dZ)
   : _x(dX), _y(dY), _z(dZ)
{
}

vd_rvc_t3Vector::vd_rvc_t3Vector(const vd_rvc_t2Vector & oVec2D)
   : _x(oVec2D.x()), _y(oVec2D.y()), _z(0.0)
{
}

const vd_rvc_t3Vector & vd_rvc_t3Vector::operator*= (tDouble dFact)
{
   _x *= dFact;
   _y *= dFact;
   _z *= dFact;
   return *this;
}

const vd_rvc_t3Vector & vd_rvc_t3Vector::operator+= (const vd_rvc_t3Vector & oVec)
{
   _x += oVec._x;
   _y += oVec._y;
   _z += oVec._z;
   return *this;
}

const vd_rvc_t3Vector & vd_rvc_t3Vector::operator-= (const vd_rvc_t3Vector & oVec)
{
   _x -= oVec._x;
   _y -= oVec._y;
   _z -= oVec._z;
   return *this;
}

vd_rvc_t3Vector vd_rvc_t3Vector::operator* (tDouble dFact) const
{
   vd_rvc_t3Vector oProd(*this);
   return oProd *= dFact;
}

vd_rvc_t3Vector vd_rvc_t3Vector::operator+ (const vd_rvc_t3Vector & oVec) const
{
   vd_rvc_t3Vector oDiff(*this);
   return oDiff += oVec;
}

vd_rvc_t3Vector vd_rvc_t3Vector::operator- (const vd_rvc_t3Vector & oVec) const
{
   vd_rvc_t3Vector oDiff(*this);
   return oDiff -= oVec;
}


/**************************************************************************************************/
// methods of the vd_rvc_t2Vector class
/**************************************************************************************************/ 

vd_rvc_t2Vector::vd_rvc_t2Vector(tVoid)
   : _x(0.0), _y(0.0)
{
}

vd_rvc_t2Vector::vd_rvc_t2Vector(tDouble dX, tDouble dY)
   : _x(dX), _y(dY)
{
}

vd_rvc_t2Vector::vd_rvc_t2Vector(const vd_rvc_t3Vector & oVec3D)
   : _x(oVec3D.x()), _y(oVec3D.y())  // and ignore z component of input vector
{
}

const vd_rvc_t2Vector & vd_rvc_t2Vector::operator*= (tDouble dFact)
{
   _x *= dFact;
   _y *= dFact;
   return *this;
}

const vd_rvc_t2Vector & vd_rvc_t2Vector::operator+= (const vd_rvc_t2Vector & oVec)
{
   _x += oVec._x;
   _y += oVec._y;
   return *this;
}

const vd_rvc_t2Vector & vd_rvc_t2Vector::operator-= (const vd_rvc_t2Vector & oVec)
{
   _x -= oVec._x;
   _y -= oVec._y;
   return *this;
}

vd_rvc_t2Vector vd_rvc_t2Vector::operator* (tDouble dFact) const
{
   vd_rvc_t2Vector oProd(*this);
   return oProd *= dFact;
}

vd_rvc_t2Vector vd_rvc_t2Vector::operator+ (const vd_rvc_t2Vector & oVec) const
{
   vd_rvc_t2Vector oSum(*this);
   return oSum += oVec;
}

vd_rvc_t2Vector vd_rvc_t2Vector::operator- (const vd_rvc_t2Vector & oVec) const
{
   vd_rvc_t2Vector oDiff(*this);
   return oDiff -= oVec;
}

tDouble vd_rvc_t2Vector::operator* (const vd_rvc_t2Vector & oVec) const
{
   return _x*oVec._x + _y*oVec._y;
}


/**************************************************************************************************/
// methods of the vd_rvc_t2IntVect class
/**************************************************************************************************/ 

vd_rvc_t2IntVect::vd_rvc_t2IntVect(tVoid)
   : _x(0), _y(0)
{
}

vd_rvc_t2IntVect::vd_rvc_t2IntVect(tS32 s32X, tS32 s32Y)
   : _x(s32X), _y(s32Y)
{
}

vd_rvc_t2IntVect::vd_rvc_t2IntVect(const vd_rvc_t2Vector & oP2D)
   : _x((tS32)floor(oP2D.x()+0.5)), _y((tS32)floor(oP2D.y()+0.5))
{
}

tVoid vd_rvc_t2IntVect::vSet(tFloat x_, tS32 y_ )
{
   _x = (tS32)floor(x_+0.5);
   _y = y_;
}

tVoid vd_rvc_t2IntVect::vSet(tS32 x_ , tFloat y_)
{
   _x = x_;
   _y = (tS32)floor(y_+0.5);
}


/**************************************************************************************************/
// methods of the vd_rvc_tcl3x3Matrix class
/**************************************************************************************************/ 

vd_rvc_tcl3x3Matrix::vd_rvc_tcl3x3Matrix(tVoid)
{
   (tVoid)memset( & _A[0][0], 0x00, sizeof(_A) );
   _A[0][0] = _A[1][1] = _A[2][2] = 1;
}

vd_rvc_tcl3x3Matrix::vd_rvc_tcl3x3Matrix(tDouble A00, tDouble A01, tDouble A02,   tDouble A10, tDouble A11, tDouble A12,   tDouble A20, tDouble A21, tDouble A22)
{
   _A[0][0] = A00;  _A[0][1] = A01;  _A[0][2] = A02;
   _A[1][0] = A10;  _A[1][1] = A11;  _A[1][2] = A12;
   _A[2][0] = A20;  _A[2][1] = A21;  _A[2][2] = A22;
}

const vd_rvc_tcl3x3Matrix & vd_rvc_tcl3x3Matrix::operator*= ( const vd_rvc_tcl3x3Matrix & oRight )
{
   vd_rvc_tcl3x3Matrix mProd;
   tU32 i, j;
   for(i=0; i<3; i++)
   {
      for(j=0; j<3; j++)
      {
         mProd._A[i][j] = _A[i][0]*oRight._A[0][j] + _A[i][1]*oRight._A[1][j] + _A[i][2]*oRight._A[2][j];
      }
   }
   *this = mProd;
   return *this;
}

vd_rvc_t3Vector vd_rvc_tcl3x3Matrix::operator* (const vd_rvc_t3Vector & oVec) const
{
   vd_rvc_t3Vector oOut( _A[0][0]*oVec.x() + _A[0][1]*oVec.y() + _A[0][2]*oVec.z() ,
                         _A[1][0]*oVec.x() + _A[1][1]*oVec.y() + _A[1][2]*oVec.z() ,
                         _A[2][0]*oVec.x() + _A[2][1]*oVec.y() + _A[2][2]*oVec.z() );
   return oOut;
}

vd_rvc_tcl3x3Matrix vd_rvc_tcl3x3Matrix::operator* (const vd_rvc_tcl3x3Matrix & oRight) const
{
   vd_rvc_tcl3x3Matrix mProd(*this);
   return mProd *= oRight;
}

tDouble vd_rvc_tcl3x3Matrix::dDet(void) const  // calculates the determinant of the matrix using the rule of Sarrus
{
   return + (_A[0][0]*_A[1][1]*_A[2][2] + _A[0][1]*_A[1][2]*_A[2][0] + _A[0][2]*_A[1][0]*_A[2][1])
          - (_A[2][0]*_A[1][1]*_A[0][2] + _A[2][1]*_A[1][2]*_A[0][0] + _A[2][2]*_A[1][0]*_A[0][1]);
}

tBool vd_rvc_tcl3x3Matrix::bInvert(vd_rvc_tcl3x3Matrix * poInv) // calculates the inverse of the matrix using Cramer's rule
{
   tDouble dMyDet = dDet();
   if (0 != dMyDet)
   {
      poInv->_A[0][0] = (_A[1][1]*_A[2][2] - _A[1][2]*_A[2][1]) / dMyDet;
      poInv->_A[0][1] = (_A[0][2]*_A[2][1] - _A[0][1]*_A[2][2]) / dMyDet;
      poInv->_A[0][2] = (_A[0][1]*_A[1][2] - _A[0][2]*_A[1][1]) / dMyDet;
      poInv->_A[1][0] = (_A[1][2]*_A[2][0] - _A[1][0]*_A[2][2]) / dMyDet;
      poInv->_A[1][1] = (_A[0][0]*_A[2][2] - _A[0][2]*_A[2][0]) / dMyDet;
      poInv->_A[1][2] = (_A[0][2]*_A[1][0] - _A[0][0]*_A[1][2]) / dMyDet; 
      poInv->_A[2][0] = (_A[1][0]*_A[2][1] - _A[1][1]*_A[2][0]) / dMyDet;
      poInv->_A[2][1] = (_A[0][1]*_A[2][0] - _A[0][0]*_A[2][1]) / dMyDet;
      poInv->_A[2][2] = (_A[0][0]*_A[1][1] - _A[0][1]*_A[1][0]) / dMyDet;      
      return true;  // matrix has an inverse (determinant is non-zero)
   }
   return false;  // matrix does not have an inverse (determinant is zero)
}

/**************************************************************************************************/
// methods of the vd_rvc_tcl2x2Matrix class
/**************************************************************************************************/ 

vd_rvc_tcl2x2Matrix::vd_rvc_tcl2x2Matrix(tVoid)
{
   (tVoid)memset( & _A[0][0], 0x00, sizeof(_A) );
   _A[0][0] = _A[1][1] = 1.0;
}

vd_rvc_tcl2x2Matrix::vd_rvc_tcl2x2Matrix(tDouble A00, tDouble A01,   tDouble A10, tDouble A11)
{
   _A[0][0] = A00;  _A[0][1] = A01;
   _A[1][0] = A10;  _A[1][1] = A11;
}

const vd_rvc_tcl2x2Matrix & vd_rvc_tcl2x2Matrix::operator*= ( const vd_rvc_tcl2x2Matrix & oRight )
{
   vd_rvc_tcl2x2Matrix mProd;
   mProd._A[0][0] = _A[0][0]*oRight._A[0][0] + _A[0][1]*oRight._A[1][0];
   mProd._A[0][1] = _A[0][0]*oRight._A[0][1] + _A[0][1]*oRight._A[1][1];
   mProd._A[1][0] = _A[1][0]*oRight._A[0][0] + _A[1][1]*oRight._A[1][0];
   mProd._A[1][1] = _A[1][0]*oRight._A[0][1] + _A[1][1]*oRight._A[1][1];
   *this = mProd;
   return *this;
}

vd_rvc_tcl2x2Matrix vd_rvc_tcl2x2Matrix::operator* (const vd_rvc_tcl2x2Matrix & oRight) const
{
   vd_rvc_tcl2x2Matrix mProd = *this;
   return mProd *= oRight;
}

vd_rvc_t2Vector vd_rvc_tcl2x2Matrix::operator* (const vd_rvc_t2Vector & oVec) const
{
   vd_rvc_t2Vector oOut( _A[0][0]*oVec.x() + _A[0][1]*oVec.y() ,
                         _A[1][0]*oVec.x() + _A[1][1]*oVec.y() );
   return oOut;
}

tDouble vd_rvc_tcl2x2Matrix::dDet(void) const  // calculates the determinant of the matrix using the rule of Sarrus
{
   return _A[0][0]*_A[1][1] - _A[1][0]*_A[0][1];
}

tBool vd_rvc_tcl2x2Matrix::bInvert(vd_rvc_tcl2x2Matrix * poInv) // calculates the inverse of the matrix using Cramer's rule
{
   tDouble dMyDet = dDet();
   if (0 != dMyDet)
   {
      poInv->_A[0][0] = (  _A[1][1]) / dMyDet;
      poInv->_A[0][1] = (- _A[0][1]) / dMyDet;
      poInv->_A[1][0] = (- _A[1][0]) / dMyDet;
      poInv->_A[1][1] = (  _A[0][0]) / dMyDet;
      return true;  // matrix has an inverse (determinant is non-zero)
   }
   return false;  // matrix does not have an inverse (determinant is zero)
}


/**************************************************************************************************/
// methods of the FlexFloat class
/**************************************************************************************************/ 

const tDouble vd_rvc_tclFlexFloat::s_dInfValue(-log(0.0));                              // dirty trick to deliberately produce an "Infinity" value without upsetting lint
const tDouble vd_rvc_tclFlexFloat::s_dNaNValue(0.0*vd_rvc_tclFlexFloat::s_dInfValue);   // dirty trick to deliberately produce a (quiet) "NaN" value

vd_rvc_tclFlexFloat::vd_rvc_tclFlexFloat(tVoid)
{
   vSetup(0,0);  // invalid setup => instances constructed this way must be initialized via vSetup() before they can be used
//   vSetup(32,8,0,true);  // setup corresponding to the standard IEEE float
}

vd_rvc_tclFlexFloat::vd_rvc_tclFlexFloat(tU8 u8NumBits, tU8 u8NumExpBits, tS16 s16ExponentBias, tBool bWithSignBit)
{
   vSetup(u8NumBits,u8NumExpBits,s16ExponentBias,bWithSignBit);
}

vd_rvc_tclFlexFloat::vd_rvc_tclFlexFloat(tU8 u8NumBits, tDouble dMinAbsVal, tDouble dMaxAbsVal, tBool bWithSignBit)
{
   if (dMinAbsVal<=0 || dMaxAbsVal<dMinAbsVal)
   {
      vSetup(0,0);  // invalid setup => instance must be re-initialized via vSetup() with valid parameters before it can be used
   }
   else
   {
      tDouble dExp2Min  = floor(log2(dMinAbsVal));
      tDouble dExp2Max  = floor(log2(dMaxAbsVal) + 1e-15);
      tDouble dNumExp2Vals = (1+dExp2Max-dExp2Min) + 2;  // "+2" because exponent values 00..00b and 11..11b are reserved to code zero, denormalized values, +/-Infinity and NaNs    
      tU8  u8NumExpBits = (tU8)ceil(log2(dNumExp2Vals) - 1e-15);
      tS16 s16ExpBias   = (tS16)floor((dExp2Min+dExp2Max)/2);
      vSetup(u8NumBits,u8NumExpBits,s16ExpBias,bWithSignBit);
   }
}

tVoid vd_rvc_tclFlexFloat::vSetup(tU8 u8NumBits, tU8 u8NumExpBits, tS16 s16ExponentBias, tBool bWithSignBit)
{
   m_bSetupOk = (u8NumBits>=3 && u8NumBits<=32 && u8NumExpBits>=2 && (u8NumExpBits+(bWithSignBit?1:0)+1)<=u8NumBits);
   if (m_bSetupOk)
   {
      // store user-supplied configuration values
      m_u8NumBits     = u8NumBits;
      m_u8NumExpBits  = u8NumExpBits;
      m_s16ExpBias    = s16ExponentBias;
      m_bSigned       = bWithSignBit;

      // compute derived helper values
      m_u8NumBytes    = (m_u8NumBits+7) >> 3;
      m_u8NumMantBits = (u8NumBits - u8NumExpBits) - (bWithSignBit ? 1 : 0);
      
      m_u32SignBit    = bWithSignBit ? (0x01u << (u8NumBits - 1)) : 0x00;
      m_u32HiddenBit  = 0x01u << m_u8NumMantBits;
      m_u32ExpMask    = ~(0xFFFFFFFF << u8NumExpBits);
      m_u32MantMask   = ~(0xFFFFFFFF << m_u8NumMantBits);
      m_u32AbsValMask = m_u32MantMask | (m_u32ExpMask << m_u8NumMantBits);

      m_s16ExpBase    = (tS16)((m_u32ExpMask-1) >> 1);  // e.g. for 8 exponent bits => exp mask = 0xFF = 255 => exp base = 0x7F = 127
      m_s16ExpMax     = m_s16ExpBase;                   // => exp max = +127 = maximum logical exponent value, will be coded as (max + base) = 254 = 0xFE
      m_s16ExpMin     = -(m_s16ExpBase-1);              // => exp min = -126 = minimum logical exponent value, will be coded as (min + base) =   1 = 0x01
      // note: coded exponent values 0x00 and 0xFF are used to indicate demormalized numbers resp. +/-Inf and NaN

      m_u32InfCode    = m_u32ExpMask << m_u8NumMantBits;                // bit pattern which codes a positive infinity: all exponent bits 1, all other bits 0
      m_u32NaNCode    = m_u32InfCode | (0x01u << (m_u8NumMantBits-1));  // bit pattern which codes a NaN: all exponent bits 1 (as for +Inf), + MSbit of mantissa 1, all other bits 0  
  
      // determine limits and related values
      m_dEpsVal          =                pow(2.0,(tDouble)(-(m_u8NumMantBits+1)));             // epsilon value (relative distance of two consecutive numbers is always >epsilon and <=2*epsilon) 
      m_dCenAbsVal       =                pow(2.0,(tDouble)(m_s16ExpBias));                     // "central" codable value (e.g. of 1.0/4.0/0.25 if expBias==0/+2/-2)
      m_dMaxAbsVal       = m_dCenAbsVal * pow(2.0,(tDouble)(m_s16ExpMax+1)) * (1 - m_dEpsVal);  // largest codable value
      m_dMinAbsVal       = m_dCenAbsVal * pow(2.0,(tDouble)(m_s16ExpMin));                      // smallest (absolute) value codable in normalized form
      m_dMinAbsValDenorm = m_dMinAbsVal * pow(2.0,(tDouble)(-m_u8NumMantBits));                 // smallest (absolute) value codable in denormalized form
   }
   else  // some code clutter to make Coverity happy
   {
      m_u8NumBits  = m_u8NumExpBits = m_u8NumBytes = m_u8NumMantBits = 0;
      m_s16ExpBias = m_s16ExpBase   = m_s16ExpMax  = m_s16ExpMin     = 0;
      m_bSigned    = false;
      m_u32SignBit = m_u32HiddenBit = m_u32ExpMask = m_u32MantMask = m_u32AbsValMask    = m_u32InfCode = 0;
      m_dEpsVal    = m_dCenAbsVal   = m_dMaxAbsVal = m_dMinAbsVal  = m_dMinAbsValDenorm = 0.0;
      m_u32NaNCode = 0xFFFFFFFF;
   }
}

vd_rvc_tclFlexFloat::~vd_rvc_tclFlexFloat()
{
}

tBool vd_rvc_tclFlexFloat::bIsInf(tU32 u32CodeValue) const
{
   if (!m_bSetupOk)
      return false;
   u32CodeValue &= m_u32AbsValMask;  // ignore sign bit and all bits beyond the bit size of the instance
   return (u32CodeValue == m_u32InfCode) ? true : false;   // (positive) Infinity is one specific bit pattern with exponent 11..11b and mantissa ==0x00
}

tBool vd_rvc_tclFlexFloat::bIsNaN(tU32 u32CodeValue) const
{
   if (!m_bSetupOk)
      return false;
   u32CodeValue   &= m_u32AbsValMask;  // ignore sign bit and all bits beyond the bit size of the instance
   tU32 u32ExpBits = (u32CodeValue >> m_u8NumMantBits) & m_u32ExpMask;  // NaNs are *all* bit patterns with exponent 11..11b (as Infinity), but with mantissa !=0x00
     // (note that while we must be able to *recognise* *all* possible NaN bit patterns, the *encoder* will *produce* only *one* of these, namely m_u32NaNCode)
   return ((u32ExpBits == m_u32ExpMask) && (u32CodeValue != m_u32InfCode)) ? true : false;  // because of the above, we can't simply check "u32CodeValue == m_u32NaNCode" here
}

tBool vd_rvc_tclFlexFloat::bIsDenorm(tU32 u32CodeValue) const
{
   if (!m_bSetupOk)
      return false;
   u32CodeValue   &= m_u32AbsValMask;  // ignore sign bit and all bits beyond the bit size of the instance
   tU32 u32ExpBits = (u32CodeValue >> m_u8NumMantBits) & m_u32ExpMask;
   return ((u32ExpBits == 0x00) && (u32CodeValue != 0x00)) ? true : false;  // denormalized values are values !=0 with exponent 0x00
}

tDouble vd_rvc_tclFlexFloat::dDecode(tU32 u32CodeValue) const
{
   if (!m_bSetupOk)
      return s_dNaNValue;
 
   tBool bIsNeg = (u32CodeValue & m_u32SignBit) ? true : false;  // if the type is unsigned, we have m_u32SignBit==0x000000, so this doesn't hurt and leads to bIsNeg==false
   tS16 s16Exp  = (tS16)((u32CodeValue >> m_u8NumMantBits) & m_u32ExpMask);
   tU32 u32Mant = u32CodeValue & m_u32MantMask;
   tDouble dValue;
   
   if ((0x00==s16Exp) && (0x00==u32Mant))  // +/- zero?
   {
     dValue = bIsNeg ? -0.0 : 0.0;
   }
   else if ((tU32)(tU16)s16Exp == m_u32ExpMask)  // exp==111...b signals codes special cases: +/-Inf or a NaN
   {
     if (u32Mant)
        dValue = s_dNaNValue;
     else
        dValue = bIsNeg ? -s_dInfValue : s_dInfValue;  // +/- Infinity
   }
   else // real number !=0
   {
      if (s16Exp)  // normalized case:
         u32Mant |= m_u32HiddenBit;  // paste-in back into the mantissa the hidden bit which was removed during coding
      else         // denormalized case:
         u32Mant <<= 1;              // just shift the mantissa up one bit
      s16Exp -= m_s16ExpBase;        // un-rebase the exponent from the range [00..01b,11..10b] to the range [min.exp.,max.exp.] ...
      s16Exp += m_s16ExpBias;        // ... un-bias it, ...
      dValue = u32Mant * pow(2.0,(tDouble)(s16Exp - m_u8NumMantBits));  // ... and finally apply it to the mantissa (without the portion which is already contained in the mantissa)
      if (bIsNeg)
         dValue = -dValue;
   }
   
   return dValue;
}

tU32 vd_rvc_tclFlexFloat::u32Encode(tDouble dValue) const
{
   if (!m_bSetupOk)
      return m_u32NaNCode;         // if the instance is not set-up properly, it can't code anything at all  =>  return code for NaN

   if (std::isnan(dValue))         // check if the passed value is a NaN
//   if (!(dValue>=0.0 || dValue<=0.0))   // use this if "isnan(x)" is unavailable (and maybe, if the compiler optimizes the condition away, use a volatile dValue)
      return m_u32NaNCode;
   
   if (0.0>dValue && !m_bSigned)   // check if the passed value is <0 but instance is set-up for positive numbers only
      return m_u32NaNCode;         //  negative values (including -Infinity) can't be coded by this instance  =>  return code for NaN
      
   // check if the passed value is zero in one of the possible two "flavours": +0 or -0 (yes, indeed - in the world of floats/doubles, both a signed and an unsigned zero exists!)
   tDouble dProbe = 1.0 / dValue;  // computing this "probe value" *inside* the "if(){}" body below would make lint unhappy, so we are doing it already now, *outside* the "if(){}" body 
   if (0.0 == dValue)              // this still catches both cases: unsigned zero ("+0") and signed zero ("-0")
   {
      // if we get here, we know (a) that the passed value is either +0 or -0, and (b) that therefore, the probe value is now either +Infinity or -Infinity (and no, with floats/doubles, x/0.0 *doesn't* crash!)
      if (0.0 < dProbe)  // probe value is >0, i.e. +Infinity  =>  passed value is +0.0  =>  return the code for +0.0:
         return 0x00;
      else               // probe value is <0, i.e. -Infinity  =>  passed value is -0.0  =>  return the code for -0.0, unless the instance is set-up for positive values only;
//         return (m_bSigned) ? m_u32SignBit : 0x00;          // in the latter case, silently drop the sign of -0.0 and return the code for +0.0
         return (m_bSigned) ? m_u32SignBit : m_u32NaNCode;    // in the latter case, instead of silently dropping the sign of -0.0, return the code for NaN, since -0.0 can't be coded by this instance
   }
   
   if (std::isinf(dValue))         // check if the passed value is +/-Infinity
//   if (0.0 == dProbe)            // use this if "isinf(x)" is unavailable; if x!=0 (which got ensured further up), this is only true for +/-Infinity
      return (0 < dValue) ? m_u32InfCode : (m_u32InfCode | m_u32SignBit);
   
   // we get here if and only if dValue is neither a NaN nor an Infinity nor a zero, and if its eventual negative sign can be coded by this instance
   
   tU32 u32Code = 0x00;  // start with code for positive zero
   if (0.0 > dValue)     // ensure positive value and set sign bit if necessary
   {
      dValue  = -dValue;
      u32Code = m_u32SignBit;
   }
   
   // compute binary exponent and use it to bring the input value into the range [1,2[
   tS16 s16Exp = (tS16)floor(log2(dValue));
   dValue *= pow(2.0,(tDouble)(-s16Exp)); // the result of pow(2,integer_value) is "exact" in the sense that it is a floating point value without any mantissa bits set, i.e. really a pure power of 2
                                          // => if the value delivered by log2() above had no numerical/rounding errors, the value would already now be *guaranteed* to be in the range [1,2[
   // However, as we know, log2() usually *will* produce rounding erros => if its input value is e.g. some value 2^n with integer n (i.e. a pure power of 2), it *can* still happen
   // that the value returned by log2() is *not* exactly n (as it should), but rather (n-delta) with some very small delta => floor-ing (n-delta) leads to s16Exp being (n-1) instead of n,
   // which in turn leads to dValue now being in the range [2,4[ instead of [1,2[. Therefore, we have to catch and correct these cases now:
   if (2.0 <= dValue)  // above the range [1,2[ ?
   {
      dValue /= 2;  // now inside [1,2[ (dividing floats by pure powers of 2 doesn't change any mantissa bytes, but only the binary exponent) 
      s16Exp++;     // corrected exponent
   }
   else if (1.0 > dValue)  // just for the sake of completeness, deal with the opposite case as well (though because of 'floor', I can't imagine any scenario where s16Exp would be too *large*)
   {
      dValue *= 2;
      s16Exp--;
   }
   s16Exp -= m_s16ExpBias;   // subtract the bias from the exponent prior to coding it

   if (s16Exp > m_s16ExpMax)   // exponent overflow?
   {
      u32Code |= m_u32InfCode;     // => code a +/-Infinity
   }
   else
   {
      tU32 u32Mant;
      tS16 s16ExpUflow = m_s16ExpMin - s16Exp;  // amount of exponent underflow (will be positive in case of underflow)
      if (0 >= s16ExpUflow)  // no underflow => can do normalized coding
      {
         u32Mant  = (tU32)floor(dValue * (tU32)(0x01u << m_u8NumMantBits));  // with the value being in the range [1,2[ now, this produces an integer with a leading 1-bit, followed by m_u8NumMantBits arbitrary bits
         u32Mant &= m_u32MantMask;  // mask out the said leading 1-bit (since, per construction, it's *always* 1 now, we don't need to store it in the code - "hidden bit"; it will be or'ed in again during decoding)
         s16Exp  += m_s16ExpBase;   // rebase the exponent from the range [min.exp.,max.exp.] to the range [00..01b,11..10b]
      }
      else  // underflow => try if value can be coded as denormalized
      {
         if (s16ExpUflow <= m_u8NumMantBits)  // denormalization (with loss of precision) possible?
            u32Mant = (tU32)floor(dValue * (tU32)(0x01u << (m_u8NumMantBits - s16ExpUflow)));  // with the value being in the range [1,2[ and exp.uflow>0, this produces an integer with set bits only within the lowest m_u8NumMantBits ones
         else  // can't even save the day by denormalizing => set mantissa to code a zero (which may be +0 or -0)
            u32Mant = 0x00;
         s16Exp = 0x00;  // exponent==0 in both the denormalized and the zero case
      }
      u32Code |= (((tU32)(tU16)s16Exp << m_u8NumMantBits) | u32Mant);  // combine mantissa and exponent with the sign bit (if any) which is already coded
   } // no exponent overflow
   
   return u32Code;
}

tDouble vd_rvc_tclFlexFloat::dDecode(const tU8 *pu8CodeBytes, tBool bLittleEndian) const
{
   if (!m_bSetupOk || NULL==pu8CodeBytes)
      return 0.0;

   tU32 u32Code;
   if (bLittleEndian) {  // little endian byte order in input buffer
      u32Code  = ((tU32)(*(  pu8CodeBytes))      );
      if (2 <= m_u8NumBytes) {
         u32Code += ((tU32)(*(++pu8CodeBytes)) <<  8);
         if (3 <= m_u8NumBytes) {
            u32Code += ((tU32)(*(++pu8CodeBytes)) << 16);
            if (4 <= m_u8NumBytes)
               u32Code += ((tU32)(*(++pu8CodeBytes)) << 24);
         }
      }
   } else {  // big endian byte order in input buffer
      pu8CodeBytes += m_u8NumBytes;  // this is one byte *beyond* the last byte of the code, hence ...
      u32Code  = ((tU32)(*(--pu8CodeBytes))      );   // ... '--' already *before* reading the first byte
      if (2 <= m_u8NumBytes) {
         u32Code += ((tU32)(*(--pu8CodeBytes)) <<  8);
         if (3 <= m_u8NumBytes) {
            u32Code += ((tU32)(*(--pu8CodeBytes)) << 16);
            if (4 <= m_u8NumBytes)
               u32Code += ((tU32)(*(--pu8CodeBytes)) << 24);
         }
      }
   } // endianness
   
   return dDecode(u32Code);
}

tVoid vd_rvc_tclFlexFloat::vEncode(tDouble dValue, tU8 *pu8CodeBytes, tBool bLittleEndian) const
{
   if (m_bSetupOk && NULL!=pu8CodeBytes)
   {
      tU32 u32Code = u32Encode(dValue);
      if (bLittleEndian) {  // little endian byte order requested for output buffer
         *(  pu8CodeBytes) = (tU8)(u32Code      );
         if (2 <= m_u8NumBytes) {
            *(++pu8CodeBytes) = (tU8)(u32Code >>  8);
            if (3 <= m_u8NumBytes) {
               *(++pu8CodeBytes) = (tU8)(u32Code >> 16);
               if (4 <= m_u8NumBytes) {
                  *(++pu8CodeBytes) = (tU8)(u32Code >> 24);
               }
            }
         }
      } else {  // big endian byte order requested for output buffer
         pu8CodeBytes += m_u8NumBytes;              // this is one byte *beyond* the last byte of the code, hence ...
         *(--pu8CodeBytes) = (tU8)(u32Code      );  // ... '--' already *before* writing the first byte
         if (2 <= m_u8NumBytes) {
            *(--pu8CodeBytes) = (tU8)(u32Code >>  8);
            if (3 <= m_u8NumBytes) {
               *(--pu8CodeBytes) = (tU8)(u32Code >> 16);
               if (4 <= m_u8NumBytes) {
                  *(--pu8CodeBytes) = (tU8)(u32Code >> 24);
               }
            }
         }
      } // endianness
   } // setup ok
}


/**************************************************************************************************/
// methods of the CRC32 class
/**************************************************************************************************/ 

vd_rvc_tclCrc32::vd_rvc_tclCrc32(vd_rvc_tclCrc32::vd_rvc_tclCrc32Type enCrcType)
{
   switch(enCrcType)
   {
      case CRC32_IEEE_802_3:
         m_u32CrcMask    = 0x04C11DB7;  // IEEE 802.3
         m_u32MagicValue = 0x38FB2284;
         break;
      case CRC32_CASTAGNOLI:
         m_u32CrcMask    = 0x1EDC6F41;  // Castagnoli
         m_u32MagicValue = 0xE3D2E612;
         break;
      case CRC32_KOOPMAN:
         m_u32CrcMask    = 0x741B8CD7;  // Koopman
         m_u32MagicValue = 0x23B33DEF;
         break;
      case CRC32_AIXM:
         m_u32CrcMask    = 0x814141AB;  // AIXM (aviation)
         m_u32MagicValue = 0x6F5ABCAD;
         break;
      default:
      case CRC32_CM_CR:
         m_u32CrcMask    = 0x10211021;  // CM-CR
         m_u32MagicValue = 0x7710643F;
         break;
   }
}

vd_rvc_tclCrc32::~vd_rvc_tclCrc32()
{
}

tU32 vd_rvc_tclCrc32::u32Calc(const tU8* pu8Bytes, tU32 u32NumBytes, tBool bInvertCrc, tU32 u32StartVal) const
{
  tU32 u32Crc = u32StartVal;
  for (tU32 u32ByteIdx=0; u32ByteIdx<u32NumBytes; u32ByteIdx++)
  {
     tU32 u32Bits = (tU32)pu8Bytes[u32ByteIdx] << 24;    // left-justify next byte in a dword
     for (tU32 u32BitCnt=0; u32BitCnt<8; u32BitCnt++)
     {
        if ( (u32Bits ^ u32Crc) & 0x80000000 )
           u32Crc = (u32Crc << 1) ^ m_u32CrcMask;
        else
           u32Crc <<= 1;
        u32Bits <<= 1;
     }
  }
  return bInvertCrc ? ~u32Crc : u32Crc; /* return unchanged or inverted calculated CRC-value. */
}

tBool vd_rvc_tclCrc32::bCheck(const tU8* pu8Bytes, tU32 u32NumBytes, tBool bCrcIsInverted, tU32 u32StartVal) const
{
   tU32 u32ActualCrc     = u32Calc(pu8Bytes,u32NumBytes,bCrcIsInverted,u32StartVal);
   tU32 u32ExpectedCrc32 = bCrcIsInverted ? m_u32MagicValue : 0x00000000;
   return (u32ActualCrc == u32ExpectedCrc32);
}
