/******************************************************************************/
/**
* \file    vd_rvc_tclGraphics_DynGuideline_Config.cpp
* \ingroup
*
* \brief
*
* \remark  Copyright : (c) 2015 Robert Bosch GmbH, Hildesheim
* \remark  Author    :
* \remark  Scope     :
*
* \todo
*/
/******************************************************************************/


/*******************************************************************************
                        Includes
*******************************************************************************/

#include "vd_rvc_tclGraphics_DynGuideline_Config.h"

#include "vd_rvc_tclGraphics_DynGuideline_Cameras.h"  // to get the VD_RVC_CAMERA_KDS_ID_* #define's
#include "vd_rvc_tclGraphics_DynGuideline_Calib.h"    // to get the DYN_GUIDELINE_TUNEMETH_* #define's
#include "vd_rvc_tclGraphics_DynGuideline_Util.h"     // to get the GET_U16_FROM_U8_ARRAY and GET_U16_FROM_U8_ARRAY macros

#include <cstring>


#include "dispvidctrl_AppMain_Trace.h"
// ETG defines
#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

// defines and include of generated code
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_CLIENT_VIDEOPLAYER
#include "trcGenProj/Header/vd_rvc_tclGraphics_DynGuideline_Config.cpp.trc.h"
#endif


/*******************************************************************************
                  Specific defines for this component
*******************************************************************************/


/*** graphics layout and content related defines ***/

// line lengths and widths
#define DYN_GUIDELINE_DEND_DYN_NISSAN      300 // in cm (changed from 370 to 300 to take over fix for NCG3D-56466 from 16.3 branch)
#define DYN_GUIDELINE_DEND_DYN_NISSAN_US   300 // in cm (for US/CAN, so that the dyn. GLs don't overlap with FMVSS-111 test pylons)


/*** utility defines ***/

#ifndef M_PI
  #define M_PI       3.141592653589793238462643383279
#endif
#define DEG2RAD(x)   ((x)*M_PI/180.0)
#define RAD2DEG(x)   ((x)*180.0/M_PI)


#define DYN_GUIDELINE_LIMIT_TO_RANGE(val,min,max)   { if ((val)<(min)) (val)=(min);  else if ((val)>(max))  (val)=(max); }


/*******************************************************************************
                  Local types
*******************************************************************************/

struct vd_rvc_tstPredefVehicleConfig
{
   const char*                      sCfgName;
   const vd_rvc_tstKdsVehicleConfig stCfgData;
};

/*******************************************************************************
                  Local constants
*******************************************************************************/

// >>> the following pre-defined configurations are intended only to ease development and allow for some manual testing - they are not needed anymore once config reading from KDS has been implemented
// offs:                                  0           2           4           6           8           10          12          14          16          18          20          22          24          26          28          30          32          34          36          38          40          42          44          46          48          50          52          54          56          58          60          62          64          66          68          70          72          74          76          78          80          82
static const tU8 KdsCfgStr_P32S[]     = { 0x00, 0x07, 0x0C, 0x0A, 0x55, 0x03, 0x45, 0x16, 0x27, 0x00, 0x00, 0x02, 0x3A, 0x00, 0x00, 0x00, 0x73, 0x14, 0x00, 0x37, 0x00, 0x50, 0x03, 0xCF, 0x00, 0x00, 0x01, 0xF4, 0x00, 0x00, 0x32, 0x14, 0x00, 0x00, 0x1E, 0x19, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xC0, 0x00, 0x00, 0x00, 0xFF, 0xF0, 0x80, 0x00, 0x44, 0x44, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
static const tU8 KdsCfgStr_GD1A[]     = { 0x00, 0x07, 0x0D, 0x0A, 0x8B, 0x03, 0x2E, 0x1A, 0xE6, 0x00, 0x00, 0x01, 0xD1, 0x00, 0x00, 0x00, 0x73, 0x14, 0x00, 0x2E, 0x00, 0x5B, 0x03, 0xBC, 0x00, 0x00, 0x01, 0x76, 0x00, 0x00, 0x32, 0x14, 0x00, 0x00, 0x1E, 0x19, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xC0, 0x00, 0x00, 0x00, 0xFF, 0xF0, 0x80, 0x00, 0x44, 0x44, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
static const tU8 KdsCfgStr_B02E[]     = { 0x00, 0x06, 0xCF, 0x09, 0xDD, 0x02, 0x88, 0x18, 0xBD, 0x00, 0x00, 0x02, 0x15, 0x00, 0x00, 0x00, 0x73, 0x14, 0x00, 0xD6, 0x00, 0x3C, 0x03, 0x97, 0x00, 0x00, 0x01, 0xC3, 0x00, 0x00, 0x32, 0x14, 0x00, 0x00, 0x1E, 0x19, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0xC0, 0x00, 0x00, 0x00, 0xFF, 0xF0, 0x80, 0x00, 0x44, 0x44, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
static const tU8 KdsCfgStr_JFC[]      = { 0x00, 0x07, 0x64, 0x0B, 0x44, 0x03, 0xAB, 0x17, 0x1B, 0x03, 0xDC, 0x02, 0x05, 0x01, 0x68, 0x00, 0x64, 0x22, 0x00, 0x26, 0x00, 0x3E, 0x03, 0x24, 0x00, 0x00, 0x01, 0xE9, 0x00, 0x00, 0x32, 0x14, 0x00, 0x00, 0x00, 0x19, 0xCA, 0x12, 0x12, 0xF9, 0xC4, 0x00, 0x80, 0xC2, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE1, 0x16, 0x16, 0x16, 0x1E, 0x28, 0x50, 0x96, 0x1E, 0xC8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
static const tU8 KdsCfgStr_HFE[]      = { 0x00, 0x06, 0xFE, 0x0A, 0x57, 0x03, 0x87, 0x16, 0x08, 0x00, 0x9A, 0x02, 0x3B, 0x01, 0x68, 0x00, 0x64, 0x22, 0xFF, 0xC9, 0x00, 0x35, 0x02, 0xCC, 0x00, 0x00, 0x01, 0xC2, 0x00, 0x00, 0x32, 0x14, 0x00, 0x00, 0x16, 0x19, 0xCA, 0x12, 0x12, 0xF9, 0xC4, 0x00, 0x80, 0xC2, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE1, 0x16, 0x16, 0x16, 0x1E, 0x28, 0x50, 0x96, 0x1E, 0xC8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
static const tU8 KdsCfgStr_LFD[]      = { 0x00, 0x07, 0x1C, 0x0A, 0xFA, 0x04, 0x2D, 0x17, 0xED, 0x00, 0x90, 0x02, 0x03, 0x01, 0x68, 0x00, 0x64, 0x22, 0x00, 0x00, 0x00, 0x4F, 0x03, 0x92, 0x00, 0x00, 0x01, 0x3F, 0x00, 0x00, 0x32, 0x14, 0x00, 0x00, 0x0C, 0x19, 0xCA, 0x12, 0x12, 0xF9, 0xC4, 0x00, 0x80, 0xC2, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE1, 0x16, 0x16, 0x16, 0x1E, 0x28, 0x50, 0x96, 0x1E, 0xC8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
static const tU8 KdsCfgStr_KFD[]      = { 0x00, 0x07, 0x1C, 0x0A, 0xFA, 0x04, 0x2D, 0x17, 0xED, 0x00, 0x90, 0x02, 0x03, 0x01, 0x68, 0x00, 0x64, 0x22, 0x00, 0x1E, 0x00, 0x40, 0x02, 0xCC, 0x00, 0x00, 0x01, 0xB4, 0x00, 0x00, 0x32, 0x14, 0x00, 0x00, 0x0F, 0x19, 0xCA, 0x12, 0x12, 0xF9, 0xC4, 0x00, 0x80, 0xC2, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE1, 0x16, 0x16, 0x16, 0x1E, 0x28, 0x50, 0x96, 0x1E, 0xC8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
static const tU8 KdsCfgStr_BFB[]      = { 0x00, 0x06, 0xEE, 0x0A, 0x6D, 0x03, 0x02, 0x17, 0xD6, 0x00, 0xC1, 0x02, 0x13, 0x01, 0x18, 0x00, 0x64, 0x22, 0x00, 0x00, 0x00, 0x32, 0x03, 0x80, 0x00, 0x00, 0x00, 0xF8, 0x00, 0x00, 0x32, 0x14, 0x00, 0x00, 0x14, 0x19, 0xCA, 0x12, 0x12, 0xF9, 0xC4, 0x00, 0x80, 0xC2, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE1, 0x16, 0x16, 0x16, 0x1E, 0x28, 0x50, 0x96, 0x1E, 0xC8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
static const tU8 KdsCfgStr_HZH[]      = { 0x00, 0x07, 0x1F, 0x0A, 0x55, 0x03, 0x9E, 0x16, 0x08, 0x00, 0x9A, 0x02, 0x3B, 0x01, 0x68, 0x00, 0x64, 0x22, 0xFE, 0xFC, 0x00, 0x4A, 0x02, 0xC4, 0x00, 0x00, 0x01, 0xB8, 0x00, 0x00, 0x32, 0x14, 0x00, 0x00, 0x16, 0x19, 0xCA, 0x12, 0x12, 0xF9, 0xC4, 0x00, 0x80, 0xC2, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE1, 0x16, 0x16, 0x16, 0x1E, 0x28, 0x50, 0x96, 0x1E, 0xC8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
static const tU8 KdsCfgStr_X62Tr[]    = { 0x00, 0x08, 0x16, 0x0E, 0x62, 0x04, 0x00, 0x13, 0x02, 0x00, 0x5B, 0x02, 0xA5, 0x01, 0x68, 0x00, 0x64, 0x22, 0xFF, 0x07, 0x00, 0x3E, 0x03, 0xAA, 0x00, 0x00, 0x01, 0xC5, 0x00, 0x00, 0x32, 0x14, 0x00, 0x00, 0x14, 0x19, 0xCA, 0x12, 0x12, 0xF9, 0xC4, 0x00, 0x80, 0xC2, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE1, 0x16, 0x16, 0x16, 0x1E, 0x28, 0x50, 0x96, 0x1E, 0xC8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
static const tU8 KdsCfgStr_X62Pr[]    = { 0x00, 0x08, 0x16, 0x0F, 0xA7, 0x06, 0x8A, 0x13, 0x02, 0x00, 0x5B, 0x02, 0xA5, 0x01, 0x68, 0x00, 0x64, 0x22, 0xFF, 0x07, 0x00, 0x3E, 0x03, 0xAA, 0x00, 0x00, 0x01, 0xC5, 0x00, 0x00, 0x32, 0x14, 0x00, 0x00, 0x14, 0x19, 0xCA, 0x12, 0x12, 0xF9, 0xC4, 0x00, 0x80, 0xC2, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE1, 0x16, 0x16, 0x16, 0x1E, 0x28, 0x50, 0x96, 0x1E, 0xC8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
static const tU8 KdsCfgStr_R4S[]      = { 0x00, 0x06, 0x6E, 0x09, 0xBE, 0x01, 0xDC, 0x17, 0x74, 0x00, 0x00, 0x02, 0xC6, 0x01, 0x68, 0x00, 0x64, 0x22, 0xFF, 0xD6, 0x00, 0x1D, 0x02, 0x97, 0x00, 0x00, 0x01, 0xC9, 0x00, 0x00, 0x32, 0x14, 0x00, 0x00, 0x14, 0x19, 0xCA, 0x12, 0x12, 0xF9, 0xC4, 0x00, 0x80, 0xC2, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE1, 0x16, 0x16, 0x16, 0x1E, 0x28, 0x50, 0x96, 0x1E, 0xC8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
static const tU8 KdsCfgStr_X82SD[]    = { 0x00, 0x07, 0xA5, 0x0C, 0xE2, 0x03, 0xC5, 0x16, 0x69, 0x00, 0xAA, 0x02, 0x3C, 0x01, 0x68, 0x00, 0x64, 0x22, 0xFF, 0x1D, 0x00, 0x9F, 0x07, 0x63, 0x00, 0x00, 0x01, 0xE8, 0x00, 0x00, 0x32, 0x14, 0x00, 0x00, 0x14, 0x19, 0xCA, 0x12, 0x12, 0xF9, 0xC4, 0x00, 0x80, 0xC2, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE1, 0x16, 0x16, 0x16, 0x1E, 0x28, 0x50, 0x96, 0x1E, 0xC8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
static const tU8 KdsCfgStr_X82Ha[]    = { 0x00, 0x07, 0xA5, 0x0C, 0xE2, 0x03, 0xC5, 0x16, 0x69, 0x00, 0xAA, 0x02, 0x3C, 0x01, 0x68, 0x00, 0x64, 0x22, 0xFF, 0xB8, 0x00, 0x99, 0x07, 0x61, 0x00, 0x00, 0x01, 0xE8, 0x00, 0x00, 0x32, 0x14, 0x00, 0x00, 0x14, 0x19, 0xCA, 0x12, 0x12, 0xF9, 0xC4, 0x00, 0x80, 0xC2, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE1, 0x16, 0x16, 0x16, 0x1E, 0x28, 0x50, 0x96, 0x1E, 0xC8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
static const tU8 KdsCfgStr_JFA[]      = { 0x00, 0x07, 0x0B, 0x0A, 0xAE, 0x02, 0xE5, 0x17, 0xB3, 0x00, 0xB1, 0x02, 0x01, 0x01, 0x68, 0x00, 0x64, 0x22, 0xFF, 0xDB, 0x00, 0x47, 0x03, 0x3F, 0x00, 0x07, 0x01, 0xB8, 0x00, 0x0B, 0x32, 0x14, 0x00, 0x00, 0x14, 0x19, 0xCA, 0x12, 0x12, 0xF9, 0xC4, 0x00, 0x80, 0xC2, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE1, 0x16, 0x16, 0x16, 0x1E, 0x28, 0x50, 0x96, 0x1E, 0xC8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
static const tU8 KdsCfgStr_RFA[]      = { 0x00, 0x07, 0x0B, 0x0A, 0xF4, 0x03, 0x82, 0x17, 0xB3, 0x00, 0xB1, 0x02, 0x01, 0x01, 0x68, 0x00, 0x64, 0x22, 0xFF, 0xDB, 0x00, 0x41, 0x03, 0x0F, 0x00, 0x00, 0x01, 0xCF, 0x00, 0x00, 0x32, 0x14, 0x00, 0x00, 0x14, 0x19, 0xCA, 0x12, 0x12, 0xF9, 0xC4, 0x00, 0x80, 0xC2, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE1, 0x16, 0x16, 0x16, 0x1E, 0x28, 0x50, 0x96, 0x1E, 0xC8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
static const tU8 KdsCfgStr_LFF[]      = { 0x00, 0x07, 0x10, 0x0A, 0x98, 0x03, 0xE8, 0x17, 0xD6, 0x00, 0xC1, 0x02, 0x13, 0x01, 0x18, 0x00, 0x64, 0x22, 0x00, 0x00, 0x00, 0x35, 0x03, 0x6E, 0x00, 0x00, 0x01, 0x25, 0x00, 0x00, 0x32, 0x14, 0x00, 0x00, 0x14, 0x19, 0xCA, 0x12, 0x12, 0xF9, 0xC4, 0x00, 0x80, 0xC2, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE1, 0x16, 0x16, 0x16, 0x1E, 0x28, 0x50, 0x96, 0x1E, 0xC8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
static const tU8 KdsCfgStr_KFB[]      = { 0x00, 0x07, 0x11, 0x0A, 0x98, 0x03, 0xE2, 0x17, 0xD6, 0x00, 0xC1, 0x02, 0x13, 0x01, 0x18, 0x00, 0x64, 0x22, 0xFF, 0xDB, 0x00, 0x30, 0x02, 0xE8, 0x00, 0x00, 0x01, 0xCC, 0x00, 0x00, 0x32, 0x14, 0x00, 0x00, 0x14, 0x19, 0xCA, 0x12, 0x12, 0xF9, 0xC4, 0x00, 0x80, 0xC2, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE1, 0x16, 0x16, 0x16, 0x1E, 0x28, 0x50, 0x96, 0x1E, 0xC8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
static const tU8 KdsCfgStr_J87ph2[]   = { 0x00, 0x06, 0xD5, 0x0A, 0x2E, 0x02, 0x8B, 0x1B, 0x8F, 0x00, 0x00, 0x01, 0xDB, 0x01, 0x68, 0x00, 0x64, 0x22, 0xFF, 0xDB, 0x00, 0x3F, 0x03, 0x6A, 0x00, 0x00, 0x01, 0xCD, 0x00, 0x00, 0x32, 0x14, 0x00, 0x00, 0x14, 0x19, 0xCA, 0x12, 0x12, 0xF9, 0xC4, 0x00, 0x80, 0xC2, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE1, 0x16, 0x16, 0x16, 0x1E, 0x28, 0x50, 0x96, 0x1E, 0xC8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
static const tU8 KdsCfgStr_K98ph2[]   = { 0x00, 0x06, 0xC8, 0x0A, 0x1D, 0x03, 0x37, 0x1B, 0xA1, 0x00, 0x00, 0x01, 0xEC, 0x01, 0x68, 0x00, 0x64, 0x22, 0x00, 0x3E, 0x00, 0x38, 0x02, 0x98, 0x00, 0x00, 0x01, 0xCC, 0x00, 0x00, 0x32, 0x14, 0x00, 0x00, 0x14, 0x19, 0xCA, 0x12, 0x12, 0xF9, 0xC4, 0x00, 0x80, 0xC2, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE1, 0x16, 0x16, 0x16, 0x1E, 0x28, 0x50, 0x96, 0x1E, 0xC8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
static const tU8 KdsCfgStr_B98ph2[]   = { 0x00, 0x06, 0xBC, 0x0A, 0x1D, 0x02, 0x6E, 0x1B, 0xA1, 0x00, 0x00, 0x01, 0xEC, 0x01, 0x68, 0x00, 0x64, 0x22, 0x00, 0x00, 0x00, 0x25, 0x03, 0x96, 0x00, 0x00, 0x01, 0x70, 0x00, 0x00, 0x32, 0x14, 0x00, 0x00, 0x14, 0x19, 0xCA, 0x12, 0x12, 0xF9, 0xC4, 0x00, 0x80, 0xC2, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE1, 0x16, 0x16, 0x16, 0x1E, 0x28, 0x50, 0x96, 0x1E, 0xC8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
static const tU8 KdsCfgStr_X61ph2[]   = { 0x00, 0x07, 0x25, 0x0B, 0x49, 0x02, 0xC5, 0x16, 0xF6, 0x00, 0x9C, 0x02, 0x3D, 0x01, 0x18, 0x00, 0x64, 0x22, 0xFF, 0xC8, 0x00, 0x34, 0x03, 0x28, 0x00, 0x00, 0x01, 0xA2, 0x00, 0x00, 0x32, 0x14, 0x00, 0x00, 0x14, 0x19, 0xCA, 0x12, 0x12, 0xF9, 0xC4, 0x00, 0x80, 0xC2, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE1, 0x16, 0x16, 0x16, 0x1E, 0x28, 0x50, 0x96, 0x1E, 0xC8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
static const tU8 KdsCfgStr_X61ph2L1[] = { 0x00, 0x07, 0x25, 0x0A, 0x89, 0x02, 0xC5, 0x16, 0xF6, 0x00, 0x9C, 0x02, 0x3D, 0x01, 0x18, 0x00, 0x64, 0x22, 0xFF, 0xC8, 0x00, 0x34, 0x03, 0x28, 0x00, 0x00, 0x01, 0xA2, 0x00, 0x00, 0x32, 0x14, 0x00, 0x00, 0x14, 0x19, 0xCA, 0x12, 0x12, 0xF9, 0xC4, 0x00, 0x80, 0xC2, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE1, 0x16, 0x16, 0x16, 0x1E, 0x28, 0x50, 0x96, 0x1E, 0xC8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
static const tU8 KdsCfgStr_X61ph2L2[] = { 0x00, 0x07, 0x25, 0x0C, 0x09, 0x02, 0xC5, 0x16, 0xF6, 0x00, 0x9C, 0x02, 0x3D, 0x01, 0x18, 0x00, 0x64, 0x22, 0xFF, 0xC8, 0x00, 0x34, 0x03, 0x28, 0x00, 0x00, 0x01, 0xA2, 0x00, 0x00, 0x32, 0x14, 0x00, 0x00, 0x14, 0x19, 0xCA, 0x12, 0x12, 0xF9, 0xC4, 0x00, 0x80, 0xC2, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE1, 0x16, 0x16, 0x16, 0x1E, 0x28, 0x50, 0x96, 0x1E, 0xC8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

// a HFE based dummy configuration:
static const tU8 KdsCfgStr_dummy[] = { 0x15, 0x06, 0xFE, 0x0A, 0x57, 0x03, 0x87, 0x16, 0x08, 0x00, 0x9A, 0x02, 0x3B, 0x01, 0x68, 0x00, 0x64, 0xA2, 0xFF, 0xC9, 0x00, 0x35, 0x02, 0xCC, 0x00, 0x00, 0x01, 0xC2, 0x00, 0x00, 0x32, 0x14, 0x00, 0x14, 0x16, 0x19, 0xCA, 0x12, 0x12, 0xF9, 0xC4, 0x00, 0x80, 0xC2, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE1, 0x16, 0x16, 0x16, 0x1E, 0x28, 0x50, 0x96, 0x1E, 0xC8, 0x71, 0x01, 0x02, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

#define VD_RVC_PREDEF_VEH_CFG(vehname,camname)   { #vehname, { & (KdsCfgStr_##vehname[0]), (tU8)VD_RVC_CAMERA_KDS_ID_##camname } }
static const vd_rvc_tstPredefVehicleConfig vd_rvc_astPredefVehicleConfigs[] =
{
   VD_RVC_PREDEF_VEH_CFG(P32S    ,SONY_2400S_NISSAN    ),
   VD_RVC_PREDEF_VEH_CFG(GD1A    ,SONY_2400S_NISSAN    ),
   VD_RVC_PREDEF_VEH_CFG(B02E    ,SONY_2400S_NISSAN    ),
   VD_RVC_PREDEF_VEH_CFG(JFC     ,SONY_2400S_RENAULT   ),
   VD_RVC_PREDEF_VEH_CFG(HFE     ,SONY_2400S_RENAULT   ),
   VD_RVC_PREDEF_VEH_CFG(LFD     ,SEMCO_LOGO           ),
   VD_RVC_PREDEF_VEH_CFG(KFD     ,SONY_2400S_RENAULT   ),
   VD_RVC_PREDEF_VEH_CFG(BFB     ,SEMCO_LOGO           ),
   VD_RVC_PREDEF_VEH_CFG(HZH     ,SONY_2400S_RENAULT   ),
   VD_RVC_PREDEF_VEH_CFG(X62Tr   ,SEMCO_GENERIC_VFOV109),
   VD_RVC_PREDEF_VEH_CFG(X62Pr   ,SEMCO_GENERIC_VFOV109),
   VD_RVC_PREDEF_VEH_CFG(R4S     ,SEMCO_GENERIC_VFOV109),
   VD_RVC_PREDEF_VEH_CFG(X82SD   ,SEMCO_GENERIC_VFOV109),
   VD_RVC_PREDEF_VEH_CFG(X82Ha   ,SEMCO_GENERIC_VFOV109),
   VD_RVC_PREDEF_VEH_CFG(JFA     ,SONY_2400S_RENAULT   ),
   VD_RVC_PREDEF_VEH_CFG(RFA     ,SONY_2400S_RENAULT   ),
   VD_RVC_PREDEF_VEH_CFG(LFF     ,SEMCO_LOGO           ),
   VD_RVC_PREDEF_VEH_CFG(KFB     ,SONY_2400S_RENAULT   ),
   VD_RVC_PREDEF_VEH_CFG(J87ph2  ,SONY_2400S_RENAULT   ),
   VD_RVC_PREDEF_VEH_CFG(K98ph2  ,SONY_2400S_RENAULT   ),
   VD_RVC_PREDEF_VEH_CFG(B98ph2  ,SEMCO_LOGO           ),
   VD_RVC_PREDEF_VEH_CFG(X61ph2  ,SEMCO_L47            ),
   VD_RVC_PREDEF_VEH_CFG(X61ph2L1,SEMCO_L47            ),
   VD_RVC_PREDEF_VEH_CFG(X61ph2L2,SEMCO_L47            ),

   VD_RVC_PREDEF_VEH_CFG(dummy,SONY_2400S_RENAULT   ),  // HFE configuration marked as dummy configuration 
};

// <<< pre-defined development-time-only configurations end

tBool vd_rvc_tclGraphics_DynGuideline_Config::bSetPredefConfig(const char * sPredefCfgName, vd_rvc_tstVehicleParams * pstVehParams)
{
   if (NULL==sPredefCfgName || NULL==pstVehParams)
      return false;
    
   for (tU32 u32Idx=0; u32Idx<sizeof(vd_rvc_astPredefVehicleConfigs)/sizeof(vd_rvc_astPredefVehicleConfigs[0]); u32Idx++)
   {
      const vd_rvc_tstPredefVehicleConfig* pPredefCfg = & vd_rvc_astPredefVehicleConfigs[u32Idx];
      if (0 == strcmp(pPredefCfg->sCfgName, sPredefCfgName))
      {
         return bEvalConfigString( & pPredefCfg->stCfgData, pstVehParams );
      }
   }
   return false;
}

/*******************************************************************************
    Class implementation
*******************************************************************************/

/*** static member variables ***/


/*** methods ***/

vd_rvc_tclGraphics_DynGuideline_Config::vd_rvc_tclGraphics_DynGuideline_Config(tVoid)
{
}


vd_rvc_tclGraphics_DynGuideline_Config::~vd_rvc_tclGraphics_DynGuideline_Config()
{
}


tBool vd_rvc_tclGraphics_DynGuideline_Config::bEvalConfigString(const vd_rvc_tstKdsVehicleConfig * pstKdsVehConfig, vd_rvc_tstVehicleParams * pstVehParams)
{
   if (NULL==pstKdsVehConfig || NULL==pstKdsVehConfig->pu8ConfigString || NULL==pstVehParams)
      return false;

   const tU8 * pu8ConfigBytes = & pstKdsVehConfig->pu8ConfigString[0];

   // config string version   
   pstVehParams->u8CfgStrVersNoHi    = (pu8ConfigBytes[0] >> 0) & 0x0F;
   pstVehParams->u8CfgStrVersNoLo    = (pu8ConfigBytes[0] >> 4) & 0x0F;
   pu8ConfigBytes += 1;
   
   // vehicle dimensions
   pstVehParams->dVehWidth           = GET_U16_FROM_U8_ARRAY(pu8ConfigBytes, 0) * 0.1;
   pstVehParams->dWheelBase          = GET_U16_FROM_U8_ARRAY(pu8ConfigBytes, 2) * 0.1;
   pstVehParams->dRearOverhang       = GET_U16_FROM_U8_ARRAY(pu8ConfigBytes, 4) * 0.1;
   pu8ConfigBytes += 6;
   
   // general steering system related parameters, plus general guideline style
   pstVehParams->dSWAtoFWA_Coeff1    = GET_S16_FROM_U8_ARRAY(pu8ConfigBytes, 0) * 1e-5;
   pstVehParams->dSWAtoFWA_Coeff3    = GET_S16_FROM_U8_ARRAY(pu8ConfigBytes, 2) * 1e-10;
   pstVehParams->dMaxSteerWhlAngle   = GET_U16_FROM_U8_ARRAY(pu8ConfigBytes, 4);
   pstVehParams->dMaxRearTireAngle   = GET_U16_FROM_U8_ARRAY(pu8ConfigBytes, 6) * 0.01;
   pstVehParams->dRadiusTuningFact   = GET_U16_FROM_U8_ARRAY(pu8ConfigBytes, 8) * 0.01;
   pstVehParams->bFourWheelSteering  = (pu8ConfigBytes[10] & 0x01) ? true : false;
   pstVehParams->bPosTireAngleIsCcw  = (pu8ConfigBytes[10] & 0x02) ? true : false;                                           
   pstVehParams->u8TireAngleRefPoint = (pu8ConfigBytes[10] >> 2) & 0x03;
   pstVehParams->u8GuidelnStyle      = (pu8ConfigBytes[10] >> 4) & 0x07;
   pstVehParams->bIsATestConfig      = (pu8ConfigBytes[10] & 0x80) ? true : false;
   pu8ConfigBytes += 11;
   
   // camera mounting position and angles
   pstVehParams->dCamXOffs           = GET_S16_FROM_U8_ARRAY(pu8ConfigBytes, 0) * 0.1;
   pstVehParams->dCamYOffs           = GET_S16_FROM_U8_ARRAY(pu8ConfigBytes, 2) * 0.1;
   pstVehParams->dCamZOffs           = GET_U16_FROM_U8_ARRAY(pu8ConfigBytes, 4) * 0.1;
   pstVehParams->dCamYaw             = GET_S16_FROM_U8_ARRAY(pu8ConfigBytes, 6) * 0.1;
   pstVehParams->dCamPitch           = GET_S16_FROM_U8_ARRAY(pu8ConfigBytes, 8) * 0.1;
   pstVehParams->dCamRoll            = GET_S16_FROM_U8_ARRAY(pu8ConfigBytes,10) * 0.1;
   pu8ConfigBytes += 12;

   // guideline drawing control parameters
   pstVehParams->dFrontTireThresh    = pu8ConfigBytes[0] * 0.01;
   pstVehParams->dMinFrontTireChg    = pu8ConfigBytes[1] * 0.01;
   pstVehParams->dRearTireThresh     = pu8ConfigBytes[2] * 0.01; 
   pstVehParams->dMinRearTireChg     = pu8ConfigBytes[3] * 0.01; 
   pu8ConfigBytes += 4;

   // general guideline layout parameters
   pstVehParams->dGuidelnStartDist   = pu8ConfigBytes[0];
   pstVehParams->dGuidelnLatMargin   = pu8ConfigBytes[1];
   (tVoid) OSAL_pvMemoryCopy(& pstVehParams->au8GuidelnRgb[0][0], & pu8ConfigBytes[2], sizeof(pstVehParams->au8GuidelnRgb));  
   pu8ConfigBytes += (2 + sizeof(pstVehParams->au8GuidelnRgb));
   
   pstVehParams->astLineWidth[DYN_GUIDELINE_LWIDTH_IDX_STAT].u8Near = (pu8ConfigBytes[0] >> 0) & 0x0F;
   pstVehParams->astLineWidth[DYN_GUIDELINE_LWIDTH_IDX_STAT].u8Far  = (pu8ConfigBytes[0] >> 4) & 0x0F;
   pstVehParams->astLineWidth[DYN_GUIDELINE_LWIDTH_IDX_DYN ].u8Near = (pu8ConfigBytes[1] >> 0) & 0x0F;
   pstVehParams->astLineWidth[DYN_GUIDELINE_LWIDTH_IDX_DYN ].u8Far  = (pu8ConfigBytes[1] >> 4) & 0x0F;
   pstVehParams->astLineWidth[DYN_GUIDELINE_LWIDTH_IDX_CENT].u8Near = (pu8ConfigBytes[2] >> 0) & 0x0F;
   pstVehParams->astLineWidth[DYN_GUIDELINE_LWIDTH_IDX_CENT].u8Far  = (pu8ConfigBytes[2] >> 4) & 0x0F;
   pu8ConfigBytes += 3;

   // customer specific layout parameters
   switch (pstVehParams->u8GuidelnStyle)
   {
      case DYN_GUIDELINE_STYLE_NISSAN:
      case DYN_GUIDELINE_STYLE_NISSAN_US:
      case DYN_GUIDELINE_STYLE_MITSUBISHI:
         // for the sake of completeness, set the following values (which are not used by Nissan/Mitsubishi systems) to default values:
         pstVehParams->au8GuidelnStatLen[0] = 0;
         pstVehParams->au8GuidelnStatLen[1] = 0;
         pstVehParams->au8GuidelnStatLen[2] = 0;
         pstVehParams->au8GuidelnStatLen[3] = 0;
         
         pstVehParams->dGuidelnLenHorBar     = 0.0;
         pstVehParams->fLineWdthWgtNearHbar  = 1.0F;
         pstVehParams->fLineWdthWgtVertLines = (tFloat) (pu8ConfigBytes[5] * 0.01);  // same byte position and coding as fLineWdthWgtNearHbar in Renault systems
         break;
         
      case DYN_GUIDELINE_STYLE_RENAULT:
         // read the raw static line section lengths
         pstVehParams->au8GuidelnStatLen[0] = pu8ConfigBytes[0];
         pstVehParams->au8GuidelnStatLen[1] = pu8ConfigBytes[1];
         pstVehParams->au8GuidelnStatLen[2] = pu8ConfigBytes[2];
         pstVehParams->au8GuidelnStatLen[3] = pu8ConfigBytes[3];
         
         // read further layout parameters resp. set values which are not used by Renault systems to default values:
         pstVehParams->dGuidelnLenHorBar     = (tDouble)(pu8ConfigBytes[4]);
         pstVehParams->fLineWdthWgtNearHbar  = (tFloat) (pu8ConfigBytes[5] * 0.01);  // same byte position and coding as fLineWdthWgtVertLines in Nissan/Mitsubishi systems
         pstVehParams->fLineWdthWgtVertLines = 1.0F;
         break;
         
      default:
         pstVehParams->u8GuidelnStyle = DYN_GUIDELINE_STYLE_UNDEFINED;
         break;
   }
   pu8ConfigBytes += 6;

   // camera ID
   pstVehParams->u8CameraType = pstKdsVehConfig->u8CameraId;

   tBool bCfgValid = bFinalizeVehicleConfig(pstVehParams);
   ETG_TRACE_USR1(("DYN-GL: pstVehParams (valid=%d):", bCfgValid));
   //vDebugVehicleParams(pstVehParams, bCfgValid);

   // tuning parameters
   tBool bCalibValid = bEvalCalibString(pu8ConfigBytes, pstVehParams);
   pu8ConfigBytes += 14;

   (tVoid)pu8ConfigBytes;
   return bCfgValid && bCalibValid;
}


tBool vd_rvc_tclGraphics_DynGuideline_Config::bEvalCalibString(const tU8 * pu8CalibString, vd_rvc_tstVehicleParams * pstVehParams)
{
   if (NULL==pu8CalibString || NULL==pstVehParams)
      return false;
      
   tBool bValid = true;

   // tuning string version and tuning/calibration method used
   pstVehParams->stTuneParams.u8TuneStrVersNo = (pu8CalibString[0] >> 0) & 0x0F;
   pstVehParams->stTuneParams.u8Method        = (pu8CalibString[0] >> 4) & 0x0F;
   pu8CalibString += 1;
   // camera mount tuning angles
   pstVehParams->stTuneParams.dCamYaw         = DEG2RAD(((tS8)(pu8CalibString[0])) * 0.1);
   pstVehParams->stTuneParams.dCamPitch       = DEG2RAD(((tS8)(pu8CalibString[1])) * 0.1);
   pstVehParams->stTuneParams.dCamRoll        = DEG2RAD(((tS8)(pu8CalibString[2])) * 0.1);
   pu8CalibString += 3;
   // reference point offsets
   // note: convention for x offsets is "negative value <=> offset towards vehicle center ('inward')" => for right line, we need to invert the sign of the input offset to achieve uniformity
   vEvalTuningOffsetParams(& (pstVehParams->stTuneParams.stOffsLft), pu8CalibString, (tBool)false ); 
   pu8CalibString += 3;
   vEvalTuningOffsetParams(& (pstVehParams->stTuneParams.stOffsRgt), pu8CalibString, (tBool)true  );
   pu8CalibString += 3;
   // line crop parameters
   pstVehParams->stTuneParams.stOffsLft.dNearLineLen = pu8CalibString[0];
   pstVehParams->stTuneParams.stOffsRgt.dNearLineLen = pu8CalibString[1];
   DYN_GUIDELINE_LIMIT_TO_RANGE(pstVehParams->stTuneParams.stOffsLft.dNearLineLen, 0, 127.0)
   DYN_GUIDELINE_LIMIT_TO_RANGE(pstVehParams->stTuneParams.stOffsRgt.dNearLineLen, 0, 127.0)
   pu8CalibString += 2;
   
   pstVehParams->stTuneParams.dSwaZeroOffset = (tDouble)((tS8)(pu8CalibString[0])       );
   pstVehParams->stTuneParams.dRwaZeroOffset = (tDouble)((tS8)(pu8CalibString[1]) * 0.01);
   DYN_GUIDELINE_LIMIT_TO_RANGE(pstVehParams->stTuneParams.dSwaZeroOffset, -25.0, 25.0)
   DYN_GUIDELINE_LIMIT_TO_RANGE(pstVehParams->stTuneParams.dRwaZeroOffset, -1.25, 1.25)
   pu8CalibString += 2;
   
   if (8 < pstVehParams->stTuneParams.u8Method)
      pstVehParams->stTuneParams.u8Method  = DYN_GUIDELINE_TUNEMETH_NONE;

   ETG_TRACE_USR1(("DYN-GL: pstVehParams->stTuneParams (valid=%d):",    bValid));
   //vDebugTuningParams(pstVehParams, bValid);

   (tVoid)pu8CalibString;
   return bValid;
}


tVoid vd_rvc_tclGraphics_DynGuideline_Config::vEvalTuningOffsetParams(vd_rvc_tstTuneOffsParams * pstTuningOffs, const tU8 * pu8ConfigBytes, tBool bInvertSignOfDx)
{
   tDouble dNDx = (tS8)pu8ConfigBytes[0];
   tDouble dNDy = (tS8)pu8ConfigBytes[1];
   tDouble dFDx = (tS8)pu8ConfigBytes[2];
   DYN_GUIDELINE_LIMIT_TO_RANGE(dNDx, -63.0, +63.0)
   DYN_GUIDELINE_LIMIT_TO_RANGE(dNDy, -63.0, +63.0)
   DYN_GUIDELINE_LIMIT_TO_RANGE(dFDx, -63.0, +63.0)
   pstTuningOffs->dNearDx      = (bInvertSignOfDx) ? -dNDx : dNDx;
   pstTuningOffs->dNearDy      = dNDy;
   pstTuningOffs->dFarDx       = (bInvertSignOfDx) ? -dFDx : dFDx;
}


tBool vd_rvc_tclGraphics_DynGuideline_Config::bFinalizeVehicleConfig(vd_rvc_tstVehicleParams * pstVehParams)
{
   if (NULL==pstVehParams)
      return false;

   // camera mounting position and angles
   pstVehParams->dCamYawRad   = DEG2RAD(pstVehParams->dCamYaw  );
   pstVehParams->dCamPitchRad = DEG2RAD(pstVehParams->dCamPitch);
   pstVehParams->dCamRollRad  = DEG2RAD(pstVehParams->dCamRoll );
      
   // general guideline layout parameters
   vEvalLineWidthParams(& pstVehParams->astLineWidth[DYN_GUIDELINE_LWIDTH_IDX_STAT]);
   vEvalLineWidthParams(& pstVehParams->astLineWidth[DYN_GUIDELINE_LWIDTH_IDX_DYN ]);
   vEvalLineWidthParams(& pstVehParams->astLineWidth[DYN_GUIDELINE_LWIDTH_IDX_CENT]);
         
   // customer specific layout parameters
   switch (pstVehParams->u8GuidelnStyle)
   {
      case DYN_GUIDELINE_STYLE_NISSAN:
      case DYN_GUIDELINE_STYLE_NISSAN_US:
      case DYN_GUIDELINE_STYLE_MITSUBISHI:
         // set the hard-coded static line section reference point positions (currently not used, but you never know ...)
         pstVehParams->adGuidelnStatDist[0] = 50.0;
         pstVehParams->adGuidelnStatDist[1] = 100.0;
         pstVehParams->adGuidelnStatDist[2] = 200.0;
         pstVehParams->adGuidelnStatDist[3] = 300.0;
         
         // limit further layout parameters
         DYN_GUIDELINE_LIMIT_TO_RANGE(pstVehParams->fLineWdthWgtVertLines, 1.0, 2.0)
         break;
         
      case DYN_GUIDELINE_STYLE_RENAULT:
         // limit input static line section lengths to the ranges specified in the Renault input parameter table
         DYN_GUIDELINE_LIMIT_TO_RANGE(pstVehParams->au8GuidelnStatLen[0],   5,  68)
         DYN_GUIDELINE_LIMIT_TO_RANGE(pstVehParams->au8GuidelnStatLen[1],  20,  83)
         DYN_GUIDELINE_LIMIT_TO_RANGE(pstVehParams->au8GuidelnStatLen[2],  40, 103)
         DYN_GUIDELINE_LIMIT_TO_RANGE(pstVehParams->au8GuidelnStatLen[3], 100, 163)
         // accumulate them to get the line section end positions (which are identical to the reference point positions) for internal usage
         pstVehParams->adGuidelnStatDist[0] =                            (tDouble) pstVehParams->au8GuidelnStatLen[0];
         pstVehParams->adGuidelnStatDist[1] = pstVehParams->adGuidelnStatDist[0] + pstVehParams->au8GuidelnStatLen[1];
         pstVehParams->adGuidelnStatDist[2] = pstVehParams->adGuidelnStatDist[1] + pstVehParams->au8GuidelnStatLen[2];
         pstVehParams->adGuidelnStatDist[3] = pstVehParams->adGuidelnStatDist[2] + pstVehParams->au8GuidelnStatLen[3];
         
         // limit further layout parameters
         DYN_GUIDELINE_LIMIT_TO_RANGE(pstVehParams->dGuidelnLenHorBar,      5,  68)
         DYN_GUIDELINE_LIMIT_TO_RANGE(pstVehParams->fLineWdthWgtNearHbar, 1.0, 2.0)
         break;
         
      default:
         pstVehParams->u8GuidelnStyle = DYN_GUIDELINE_STYLE_UNDEFINED;
         break;
   }
      
   // computed derived values
   pstVehParams->dGuidelnOffs = pstVehParams->dVehWidth / 2 + pstVehParams->dGuidelnLatMargin;
   switch (pstVehParams->u8GuidelnStyle)
   {
      case DYN_GUIDELINE_STYLE_NISSAN:
      case DYN_GUIDELINE_STYLE_MITSUBISHI:
         pstVehParams->dGuidelnLenDyn = DYN_GUIDELINE_DEND_DYN_NISSAN;      // so far, Nissan does not have any requirement to freely configure this (as Renault does), so we hardcode it
         pstVehParams->bConnectDynGlEnds = true;
         break;
      case DYN_GUIDELINE_STYLE_NISSAN_US:
         pstVehParams->dGuidelnLenDyn = DYN_GUIDELINE_DEND_DYN_NISSAN_US;   // for US/CAN, Nissan wants a different dyn. guideline length :-/ ...
         pstVehParams->bConnectDynGlEnds = true;
         break;
      case DYN_GUIDELINE_STYLE_RENAULT:
         pstVehParams->dGuidelnLenDyn = pstVehParams->adGuidelnStatDist[3]; // last value in static line end distance array is implicitly also the dynamic guideline length:
         pstVehParams->bConnectDynGlEnds = false;
         break;         
      default:
         pstVehParams->u8GuidelnStyle = DYN_GUIDELINE_STYLE_UNDEFINED;
         break;
   }
   // helper for calculation of distance-dependent line widths:
   // the length along the y-axis over which the guideline line width increases from its near end value to the far end value (see vCalcLineWidth())
   pstVehParams->dDistScaleForLineWidthCalc = 0.9 * (pstVehParams->dGuidelnLenDyn - pstVehParams->dGuidelnStartDist);  // this is only one model to achieve attractive results; but of course others would be possible, too
         
   // check the minimum requirements which must be fulfilled:
   tBool bCfgValid = ( (0.0 < pstVehParams->dVehWidth) && (0.0 < pstVehParams->dWheelBase) && (0.0 <= pstVehParams->dRearOverhang) && 
                       (0.0 != pstVehParams->dSWAtoFWA_Coeff1) /* dSWAtoFWA_Coeff3 is allowed to be 0 */                           &&
                       (2 >= pstVehParams->u8TireAngleRefPoint) && (DYN_GUIDELINE_STYLES_COUNT > pstVehParams->u8GuidelnStyle)      );

   return bCfgValid;
}

tVoid vd_rvc_tclGraphics_DynGuideline_Config::vEvalLineWidthParams(vd_rvc_tstLineWidthDef * pstLineWdthDef)
{
   DYN_GUIDELINE_LIMIT_TO_RANGE(pstLineWdthDef->u8Near, 1, 6)
   DYN_GUIDELINE_LIMIT_TO_RANGE(pstLineWdthDef->u8Far,  1, 6)
   
   // We explicitly don't assume u8NearEndWidth >= u8FarEndWidth, so that in principle line widths could even *in*crease towards the far end (if someone weird should wish so ...).
   // Therefore, we transform the near and far end line width into a decrease towards the far end (which might be negative) plus lower and upper limits:
   tU8 u8Near = pstLineWdthDef->u8Near;
   tU8 u8Far  = pstLineWdthDef->u8Far;
   pstLineWdthDef->s8Delta = (tS8)u8Far - (tS8)u8Near;
   pstLineWdthDef->u8LoLim = (u8Far<u8Near) ? u8Far : u8Near;
   pstLineWdthDef->u8HiLim = (u8Far>u8Near) ? u8Far : u8Near;
}


tVoid vd_rvc_tclGraphics_DynGuideline_Config::vDebugVehicleParams(const vd_rvc_tstVehicleParams * pstVehParams, tBool bValid)
{
   if (NULL == pstVehParams)
      return;

   ETG_TRACE_FATAL(("DYN-GL: Config string version = ........... %d.%d", pstVehParams->u8CfgStrVersNoHi, pstVehParams->u8CfgStrVersNoLo));

   ETG_TRACE_FATAL(("DYN-GL: pstVehParams (valid=%d):", bValid));
   ETG_TRACE_FATAL(("DYN-GL: ->dVehWidth = ..................... %f cm", (float)(pstVehParams->dVehWidth)        )); /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL: ->dWheelBase = .................... %f cm", (float)(pstVehParams->dWheelBase)       )); /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL: ->dRearOverhang = ................. %f cm", (float)(pstVehParams->dRearOverhang)    )); /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL: ->dSWAtoFWA_Coeff1 = .............. %f",    (float)(pstVehParams->dSWAtoFWA_Coeff1) )); /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL: ->dSWAtoFWA_Coeff3 = .............. %f",    (float)(pstVehParams->dSWAtoFWA_Coeff3) )); /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL: ->dMaxSteerWhlAngle = ............. %f",    (float)(pstVehParams->dMaxSteerWhlAngle))); /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL: ->dMaxRearRireAngle = ............. %f",    (float)(pstVehParams->dMaxRearTireAngle))); /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL: ->dRadiusTuningFact = ............. %f",    (float)(pstVehParams->dRadiusTuningFact))); /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL: ->bFourWheelSteering = ............ %d",    pstVehParams->bFourWheelSteering));
   ETG_TRACE_FATAL(("DYN-GL: ->bPosTireAngleIsCcw = ............ %d",    pstVehParams->bPosTireAngleIsCcw));
   ETG_TRACE_FATAL(("DYN-GL: ->u8TireAngleRefPoint = ........... %d",    pstVehParams->u8TireAngleRefPoint));
   ETG_TRACE_FATAL(("DYN-GL: ->bIsATestConfig = ................ %d",    pstVehParams->bIsATestConfig));

   ETG_TRACE_FATAL(("DYN-GL: ->dCamXOffs = ..................... %f cm", (float)(pstVehParams->dCamXOffs)       )); /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL: ->dCamYOffs = ..................... %f cm", (float)(pstVehParams->dCamYOffs)       )); /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL: ->dCamZOffs = ..................... %f cm", (float)(pstVehParams->dCamZOffs)       )); /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL: ->dCamYaw = ....................... %f deg",(float)(pstVehParams->dCamYaw)         )); /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL: ->dCamPitch = ..................... %f deg",(float)(pstVehParams->dCamPitch)       )); /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL: ->dCamRoll = ...................... %f deg",(float)(pstVehParams->dCamRoll)        )); /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL: ->dFrontTireThresh = .............. %f deg",(float)(pstVehParams->dFrontTireThresh))); /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL: ->dMinFrontTireChg = .............. %f deg",(float)(pstVehParams->dMinFrontTireChg))); /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL: ->dRearTireThresh = ............... %f deg",(float)(pstVehParams->dRearTireThresh) )); /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL: ->dMinRearTireChg = ............... %f deg",(float)(pstVehParams->dMinRearTireChg) )); /*lint !e740*/

   ETG_TRACE_FATAL(("DYN-GL: ->u8GuidelnStyle = ................ %d",    pstVehParams->u8GuidelnStyle));
   ETG_TRACE_FATAL(("DYN-GL: ->dGuidelnStartDist = ............. %f cm", (float)(pstVehParams->dGuidelnStartDist))); /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL: ->dGuidelnLatMargin = ............. %f cm", (float)(pstVehParams->dGuidelnLatMargin))); /*lint !e740*/

   const tU8 (*rgb)[3] = pstVehParams->au8GuidelnRgb;
   ETG_TRACE_FATAL(("DYN-GL: ->au8GuidelnRgb[STAT_NEAR] = ...... 0x%02X 0x%02X 0x%02X", rgb[0][0],rgb[0][1],rgb[0][2])); 
   ETG_TRACE_FATAL(("DYN-GL: ->au8GuidelnRgb[STAT_MID] = ....... 0x%02X 0x%02X 0x%02X", rgb[1][0],rgb[1][1],rgb[1][2])); 
   ETG_TRACE_FATAL(("DYN-GL: ->au8GuidelnRgb[STAT_FAR] = ....... 0x%02X 0x%02X 0x%02X", rgb[2][0],rgb[2][1],rgb[2][2])); 
   ETG_TRACE_FATAL(("DYN-GL: ->au8GuidelnRgb[STAT_CEN] = ....... 0x%02X 0x%02X 0x%02X", rgb[3][0],rgb[3][1],rgb[3][2])); 
   ETG_TRACE_FATAL(("DYN-GL: ->au8GuidelnRgb[DYN] = ............ 0x%02X 0x%02X 0x%02X", rgb[4][0],rgb[4][1],rgb[4][2]));
   ETG_TRACE_FATAL(("DYN-GL: ->astLineWidth[STAT] near/far = ... %d/%d pix",pstVehParams->astLineWidth[DYN_GUIDELINE_LWIDTH_IDX_STAT].u8Near, pstVehParams->astLineWidth[DYN_GUIDELINE_LWIDTH_IDX_STAT].u8Far));
   ETG_TRACE_FATAL(("DYN-GL: ->astLineWidth[DYN]  near/far = ... %d/%d pix",pstVehParams->astLineWidth[DYN_GUIDELINE_LWIDTH_IDX_DYN ].u8Near, pstVehParams->astLineWidth[DYN_GUIDELINE_LWIDTH_IDX_DYN ].u8Far));
   ETG_TRACE_FATAL(("DYN-GL: ->astLineWidth[CENT] near/far = ... %d/%d pix",pstVehParams->astLineWidth[DYN_GUIDELINE_LWIDTH_IDX_CENT].u8Near, pstVehParams->astLineWidth[DYN_GUIDELINE_LWIDTH_IDX_CENT].u8Far));

   ETG_TRACE_FATAL(("DYN-GL: ->au8GuidelnStatLen[NEAR] = ....... %d cm", pstVehParams->au8GuidelnStatLen[0]));
   ETG_TRACE_FATAL(("DYN-GL: ->au8GuidelnStatLen[MID1] = ....... %d cm", pstVehParams->au8GuidelnStatLen[1]));
   ETG_TRACE_FATAL(("DYN-GL: ->au8GuidelnStatLen[MID2] = ....... %d cm", pstVehParams->au8GuidelnStatLen[2]));
   ETG_TRACE_FATAL(("DYN-GL: ->au8GuidelnStatLen[FAR] = ........ %d cm", pstVehParams->au8GuidelnStatLen[3]));
   ETG_TRACE_FATAL(("DYN-GL: ->dGuidelnLenHorBar = ............. %f cm", (float)(pstVehParams->dGuidelnLenHorBar)));    /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL: ->fLineWdthWgtNearHbar = .......... %f",    (float)(pstVehParams->fLineWdthWgtNearHbar))); /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL: ->fLineWdthWgtVertLines = ......... %f",    (float)(pstVehParams->fLineWdthWgtVertLines))); /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL: ->u8CameraType = .................. %d",    pstVehParams->u8CameraType));
}


tVoid vd_rvc_tclGraphics_DynGuideline_Config::vDebugTuningParams (const vd_rvc_tstVehicleParams * pstVehParams, tBool bValid)
{
   if (NULL == pstVehParams)
      return;

   ETG_TRACE_FATAL(("DYN-GL: pstVehParams->stTuneParams (valid=%d):",    bValid));
   ETG_TRACE_FATAL(("DYN-GL: .u8TuneStrVersNo = ................ %d",    pstVehParams->stTuneParams.u8TuneStrVersNo));
   ETG_TRACE_FATAL(("DYN-GL: .u8Method = ....................... %d",    pstVehParams->stTuneParams.u8Method));
   ETG_TRACE_FATAL(("DYN-GL: .dCamYaw = ........................ %f deg",(float)(RAD2DEG(pstVehParams->stTuneParams.dCamYaw)  )));     /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL: .dCamPitch = ...................... %f deg",(float)(RAD2DEG(pstVehParams->stTuneParams.dCamPitch))));     /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL: .dCamRoll = ....................... %f deg",(float)(RAD2DEG(pstVehParams->stTuneParams.dCamRoll) )));     /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL: .stOffsLft.dNearDx = .............. %f cm", (float)(pstVehParams->stTuneParams.stOffsLft.dNearDx)));      /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL:    -''-   .dNearDy = .............. %f cm", (float)(pstVehParams->stTuneParams.stOffsLft.dNearDy)));      /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL:    -''-   .dFarDx = ............... %f cm", (float)(pstVehParams->stTuneParams.stOffsLft.dFarDx)));       /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL:    -''-   .dNearLineLen = ......... %f cm", (float)(pstVehParams->stTuneParams.stOffsLft.dNearLineLen))); /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL: .stOffsRgt.dNearDx = .............. %f cm", (float)(pstVehParams->stTuneParams.stOffsRgt.dNearDx)));      /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL:    -''-   .dNearDy = .............. %f cm", (float)(pstVehParams->stTuneParams.stOffsRgt.dNearDy)));      /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL:    -''-   .dFarDx = ............... %f cm", (float)(pstVehParams->stTuneParams.stOffsRgt.dFarDx)));       /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL:    -''-   .dNearLineLen = ......... %f cm", (float)(pstVehParams->stTuneParams.stOffsRgt.dNearLineLen))); /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL: .dSwaZeroOffset = ................. %f deg",(float)(pstVehParams->stTuneParams.dSwaZeroOffset)));         /*lint !e740*/
   ETG_TRACE_FATAL(("DYN-GL: .dRwaZeroOffset = ................. %f deg",(float)(pstVehParams->stTuneParams.dRwaZeroOffset)));         /*lint !e740*/
}
