/******************************************************************************/
/**
* \file    vd_rvc_tclGraphics_DynGuideline_CamPoint.cpp
* \ingroup
*
* \brief
*
* \remark  Copyright : (c) 2012 Robert Bosch GmbH, Hildesheim
* \remark  Author    :
* \remark  Scope     :
*
* \todo
*/
/******************************************************************************/


/*******************************************************************************
                        Includes
*******************************************************************************/

#include "vd_rvc_tclGraphics_DynGuideline_CamPoint.h"
#include "vd_rvc_tclGraphics_DynGuideline_Calib.h"
#include "vd_rvc_tclGraphics_DynGuideline_Cameras.h"

#include <cmath>

#define DYN_GUIDELINE_CAMIMG_IS_HORIZONTALLY_MIRRORED

/*******************************************************************************
                        Static member variables
*******************************************************************************/

const vd_rvc_tclCamera * vd_rvc_tclCamPoint::m_poCamera;
vd_rvc_t3Vector          vd_rvc_tclCamPoint::m_tCamMountOffs;
tDouble                  vd_rvc_tclCamPoint::m_dCamYaw;
tDouble                  vd_rvc_tclCamPoint::m_dCamPitch;
tDouble                  vd_rvc_tclCamPoint::m_dCamRoll;
vd_rvc_tcl3x3Matrix      vd_rvc_tclCamPoint::m_oGrnd2CamTransform;
vd_rvc_t2Vector          vd_rvc_tclCamPoint::m_tCamOptAxisOffs;
vd_rvc_tcl2x2Matrix      vd_rvc_tclCamPoint::m_oCamCoo2VideoPixScale;
vd_rvc_t2Vector          vd_rvc_tclCamPoint::m_tVideoCaptureCenterOffs;
vd_rvc_tcl2x2Matrix      vd_rvc_tclCamPoint::m_oVideoPix2ImgPixScale;
tDouble                  vd_rvc_tclCamPoint::m_dMaxR2norm;
vd_rvc_t2Vector          vd_rvc_tclCamPoint::m_tOutImgCenter;


/*******************************************************************************
                        Methods
*******************************************************************************/

vd_rvc_tclCamPoint::vd_rvc_tclCamPoint(tVoid) :
   m_tGroundCoo(),
   m_tRawCamCoo(),
   m_tSensCoo(),
   m_tRawImgCoo(),
   m_bRawImgCooValid(false),
   m_tPlotCoo()
{
}


vd_rvc_tclCamPoint::~vd_rvc_tclCamPoint(tVoid)
{
}

tVoid vd_rvc_tclCamPoint::vSetConfig(const vd_rvc_tstVehicleParams * ptVehParams,   const vd_rvc_tstCaptureAreaParams * ptCaptureParams,
                                     const vd_rvc_tstImageAreaParams * ptImgParams, const vd_rvc_tclCamera * poCamera )
{
   /* set values of static member variables */
   
   // camera mounting position and (nominal) angles
   m_tCamMountOffs = vd_rvc_t3Vector(ptVehParams->dCamXOffs, - ptVehParams->dCamYOffs, ptVehParams->dCamZOffs);
   m_dCamYaw       = ptVehParams->dCamYawRad;
   m_dCamPitch     = ptVehParams->dCamPitchRad;
   m_dCamRoll      = ptVehParams->dCamRollRad;
   
   // evaluate camera properties
   
   m_poCamera = poCamera;
   const vd_rvc_tstCameraParams * poCamParams = poCamera->pstGetCameraParams();
   m_tCamOptAxisOffs = vd_rvc_t2Vector(poCamParams->dOptAxisXoffs/1000.0, poCamParams->dOptAxisYoffs/1000.0);

   // maximum (squared) normalized image radius up to which the camera can deliver correct radial distortion correction values   
   m_dMaxR2norm = poCamera->dGetMaxRNormSquare();
   
   tDouble dXTemp, dYTemp;
   
   // calculate normalized-camera-coordinate - to - video-pixel scale factors
   dXTemp = poCamParams->dFocLen * ptCaptureParams->u32NumVideoImgXpix / poCamParams->dSensorXsize;
   dYTemp = poCamParams->dFocLen * ptCaptureParams->u32NumVideoImgYpix / poCamParams->dSensorYsize;
   m_oCamCoo2VideoPixScale = vd_rvc_tcl2x2Matrix(dXTemp,0.0, 0.0,dYTemp);

   // calculate offset vector between input video image center and capture area center
   dXTemp = ptCaptureParams->u32CaptureAreaXoffs - (ptCaptureParams->u32NumVideoImgXpix - ptCaptureParams->u32NumCaptureAreaXpix)/2.0;
   dYTemp = ptCaptureParams->u32CaptureAreaYoffs - (ptCaptureParams->u32NumVideoImgYpix - ptCaptureParams->u32NumCaptureAreaYpix)/2.0;
   m_tVideoCaptureCenterOffs = vd_rvc_t2Vector(dXTemp, dYTemp);
   
   // calculate video-pixel - to - output image pixel scale factors
   dXTemp = (tDouble)(ptImgParams->u32NumImgXpix) / ptCaptureParams->u32NumCaptureAreaXpix;
   dYTemp = (tDouble)(ptImgParams->u32NumImgYpix) / ptCaptureParams->u32NumCaptureAreaYpix;
   m_oVideoPix2ImgPixScale = vd_rvc_tcl2x2Matrix(dXTemp,0.0, 0.0,dYTemp);
   
   // image area center coordinates within the output buffer area (relative to buffer area top left corner)
   dXTemp = ptImgParams->u32NumImgXpix/2.0 + ptImgParams->u32ImgXoffs;
   dYTemp = ptImgParams->u32NumImgYpix/2.0 + ptImgParams->u32ImgYoffs;
   m_tOutImgCenter = vd_rvc_t2Vector(dXTemp, dYTemp);
   
   vd_rvc_tclGraphics_GuidelineCalib::vSetConfig(ptVehParams);  // this in turn will (amongst others) call vSetCamTuningAngles() below with the applicable tuning angles
}


tVoid vd_rvc_tclCamPoint::vSetCamTuningAngles(tDouble dCamTuneYaw, tDouble dCamTunePitch, tDouble dCamTuneRoll)
{
   // construct the ground-coordinates - to - camera-coordinates transformation matrix  M_total = M_permute * M_roll * M_pitch * M_yaw
   tDouble _angle, _sin, _cos;
   #if (defined DYN_GUIDELINE_CAMIMG_IS_HORIZONTALLY_MIRRORED)
   m_oGrnd2CamTransform = vd_rvc_tcl3x3Matrix(-1.0,0.0,0.0, 0.0,0.0,-1.0, 0.0,1.0,0.0);   // ground-to-camera axis permutation matrix - left/right mirrored
   #else
   m_oGrnd2CamTransform = vd_rvc_tcl3x3Matrix(+1.0,0.0,0.0, 0.0,0.0,-1.0, 0.0,1.0,0.0);   // ground-to-camera axis permutation matrix - left/right unchanged
   #endif
   
   if ((_angle = - (m_dCamRoll  + dCamTuneRoll))  != 0.0)
   {
      _sin = sin(_angle);
      _cos = cos(_angle);
      m_oGrnd2CamTransform *= vd_rvc_tcl3x3Matrix(_cos,0.0,_sin, 0.0,1.0,0.0, -_sin,0.0,_cos);  // roll matrix
   }
   if ((_angle = - (m_dCamPitch + dCamTunePitch)) != 0.0)
   {
      _sin = sin(_angle);
      _cos = cos(_angle);
      m_oGrnd2CamTransform *= vd_rvc_tcl3x3Matrix(1.0,0.0,0.0, 0.0,_cos,_sin, 0.0,-_sin,_cos);  // pitch matrix
   }
   if ((_angle =   (m_dCamYaw   + dCamTuneYaw))   != 0.0)
   {
      _sin = sin(_angle);
      _cos = cos(_angle);
      m_oGrnd2CamTransform *= vd_rvc_tcl3x3Matrix(_cos,_sin,0.0, -_sin,_cos,0.0, 0.0,0.0,1.0);  // yaw matrix
   }  
}


tVoid vd_rvc_tclCamPoint::vSetGroundCoords(const vd_rvc_t3Vector & oGrndCoo, tDouble dTuneSide)
{
   m_tGroundCoo = oGrndCoo;
   if (DYN_GUIDELINE_TUNE_SIDE_NONE != dTuneSide)
      vd_rvc_tclGraphics_GuidelineCalib::vTunePoint(& m_tGroundCoo, dTuneSide);
   m_tRawCamCoo = m_oGrnd2CamTransform * (m_tGroundCoo - m_tCamMountOffs);
   vCalcRawImgCoords();
}


tVoid vd_rvc_tclCamPoint::vSetCamCoords(tDouble dX, tDouble dY, tDouble dZ)
{
   m_tRawCamCoo = vd_rvc_t3Vector(dX,dY,dZ);
   vCalcRawImgCoords();
}


tVoid vd_rvc_tclCamPoint::vCalcRawImgCoords(tVoid)
{
   if ( m_tRawCamCoo.z() <= 0 )  // z_cam<=0 indicates a point exactly in or behind the focal plane, means: point is not visible
   {
      m_bRawImgCooValid = false;
   }
   else
   {
      vd_rvc_t2Vector oNormCamCoo(m_tRawCamCoo * (1.0/m_tRawCamCoo.z()));
      tDouble         dR2norm = oNormCamCoo*oNormCamCoo;
      m_bRawImgCooValid = (dR2norm <= m_dMaxR2norm);
      if ( m_bRawImgCooValid )
      {
         tDouble dRDF = m_poCamera->dCalcRadialDistortionFactor(dR2norm);
         m_tSensCoo   = oNormCamCoo*dRDF + m_tCamOptAxisOffs;
         m_tRawImgCoo = m_oVideoPix2ImgPixScale * (m_oCamCoo2VideoPixScale * m_tSensCoo - m_tVideoCaptureCenterOffs) + m_tOutImgCenter;
      }
   }
}


/**************************************************************************************************/
// methods to clip a line to the part with y_ground greater than the given threshold
/**************************************************************************************************/

tBool vd_rvc_tclCamPoint::bEnsureMinimumYgroundOnWholePathTo(vd_rvc_tclCamPoint * poEnd, tDouble dYMin)
{
   // this method reduces ("chops") the line from the current point to the passed end point to the part where y_ground>dYmin;
   // returns whether or not any part of the input line had y_ground>=dYmin (i.e. whether or not anything remains after the reduction) 

   vd_rvc_t3Vector * pS = & m_tGroundCoo;
   vd_rvc_t3Vector * pE = & poEnd->m_tGroundCoo;
   
   if (pS->y() < dYMin && pE->y() < dYMin)
   {
      return false;   // both points at or closer than the minimum y_ground => discard whole line
   }

   // if one point is closer than the minimum y_ground, chop off that part of the line
   if      (pS->y() < dYMin)
   {
      this ->vShiftToYgroundAlongPathTo(poEnd,dYMin);
   }
   else if (pE->y() < dYMin)
   {
      poEnd->vShiftToYgroundAlongPathTo(this, dYMin);
   }

   return true;
}


tVoid vd_rvc_tclCamPoint::vShiftToYgroundAlongPathTo(vd_rvc_tclCamPoint * poEnd, tDouble dYdest)
{
   // this method shifts the current point along the direction of the line to the end point so that the current point's new ground Y coordinate becomes dYdest
   
   vd_rvc_t3Vector * pS = & m_tGroundCoo;
   vd_rvc_t3Vector * pE = & poEnd->m_tGroundCoo;

   tDouble dFact  = (dYdest - pS->y()) / (pE->y() - pS->y());
   tDouble dXdest = pS->x() + dFact * (pE->x() - pS->x());
   tDouble dZdest = pS->z() + dFact * (pE->z() - pS->z());
   vSetGroundCoords(dXdest,dYdest,dZdest, DYN_GUIDELINE_TUNE_SIDE_NONE);  // no tuning/calibrating here - both line start and end point are expected to be already tuned, if required
}


/**************************************************************************************************/
// methods to clip a line to the part with positive z_camera
/**************************************************************************************************/

tBool vd_rvc_tclCamPoint::bEnsurePositiveZcamOnWholePathTo(vd_rvc_tclCamPoint * poEnd)
{
   // this method reduces ("chops") the line from the current point to the passed end point to the part where z_camera>0 (i.e. to the part strictly in front of the focal plane);
   // - returns whether or not any part of the input line had z_camera>0 (i.e. whether or not anything remains after the reduction) 
   // - if the line was chopped on one end and anything remained, the ground, camera, and (if possible) image pixel coordinates of the modified point will have been updated, too
   
   vd_rvc_t3Vector * pS = & m_tRawCamCoo;
   vd_rvc_t3Vector * pE = & poEnd->m_tRawCamCoo;

   if (pS->z() <= 0 && pE->z() <= 0)
   {
      return false;  // both points in or behind focal plane => discard whole line
   }
   
   // if one point is in or behind the focal plane, chop off that part of the line
   if      (pS->z() <= 0)
   {
      this ->vShiftToPositiveZcamAlongPathTo(poEnd);
   }
   else if (pE->z() <= 0)
   {
      poEnd->vShiftToPositiveZcamAlongPathTo(this);
   }

   return true;
}

tVoid vd_rvc_tclCamPoint::vShiftToPositiveZcamAlongPathTo(vd_rvc_tclCamPoint * poEnd)
{
   // This method shifts the cam coords of the current point along the direction of the line to the end point so that the current point's new camera Z coordinate becomes >0.
   // I.e.: after the call, the current point lies still on the same line of sight (<=> gets projected onto the same camera pixel), but now lies strictly in front of the focal plane
   // Note: The camera Z coordinate of the *end* point must be >0, i.e. the *end* point must already lie strictly in front of the focal plane.
   //       This is (resp. has to be) ensured by the caller!
   
   vd_rvc_t3Vector * pS = & m_tRawCamCoo;
   vd_rvc_t3Vector * pE = & poEnd->m_tRawCamCoo;

   // first shift camera Z coordinate of current point to half the focal plane distance of the end point, or to some maximum distance, whatever is closer
   tDouble dZCut = pE->z() / 2.0;
   #define MAX_SHIFTED_FOCAL_PLANE_DISTANCE 1.0
   if (dZCut > MAX_SHIFTED_FOCAL_PLANE_DISTANCE)
   {
       dZCut = MAX_SHIFTED_FOCAL_PLANE_DISTANCE;
   }
   // now shift camera x and y coordinates of current point accordingly
   tDouble dFact = (dZCut - pS->z()) / (pE->z() - pS->z());
   tDouble dXCut = pS->x() + dFact * (pE->x() - pS->x());
   tDouble dYCut = pS->y() + dFact * (pE->y() - pS->y());
   
   // set the new cam coords to update the normalized coordinates and re-compute the raw image coordinates according to the new values
   vSetCamCoords(dXCut,dYCut,dZCut);
}

/**************************************************************************************************/
// method to calculate the final screen plot positions
/**************************************************************************************************/

tBool vd_rvc_tclCamPoint::bCalcPlotCooForPathTo(vd_rvc_tclCamPoint * poEnd)
{
   if ( ! (bEnsurePositiveZcamOnWholePathTo(poEnd) && m_bRawImgCooValid && poEnd->m_bRawImgCooValid) )
   {
      return false;
   }
   
   // setup the plot coordinates with a copy of the raw image coordinates
   this ->m_tPlotCoo = this ->m_tRawImgCoo;
   poEnd->m_tPlotCoo = poEnd->m_tRawImgCoo;
   return true;
 }

