/*
 * dispvidctrl_tclSyncHandler.cpp
 *
 *  Created on: Aug 24, 2015
 *      Author: kol2hi
 */
/************************************************************************
* FILE:        dispvidctrl_tclSyncHandler.cpp
* PROJECT:
* SW-COMPONENT:vd_dimming
*----------------------------------------------------------------------
*
* DESCRIPTION: implementation of configuration manager class
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 20.08.2015| CM-AI/EPB2 Bernard | init
*
*************************************************************************/

/*******************************************************************************
| includes: system- and project- includes
|-----------------------------------------------------------------------------*/
// Basic OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

/*******************************************************************************
| set own module id
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| includes: needed interfaces from external components
|-----------------------------------------------------------------------------*/
#include <ilm/ilm_common.h>
#include <ilm/ilm_client.h>
#include <ilm/ilm_control.h>

#include <sys/types.h>
#include <unistd.h>

#include <systemd/sd-daemon.h> // you also need to add -lsystemd-daemon to the linker settings

/*******************************************************************************
| includes: internal and external interfaces from this component
|-----------------------------------------------------------------------------*/

#include "dispvidctrl_tclSyncHandler.h"
#include "dispvidctrl_AppMain.h"
#include "I_dispvidctrl_ClientDimmingIncAdaptor.h"
#include "I_dispvidctrl_tclVideoPlayer.h"
#include "Idispvidctrl_ClientVideomanager.h"
#include "I_dispvidctrl_ServiceSyncdisplay.h"
#include "I_dispvidctrl_ServiceRvcIf.h"
#include "Idispvidctrl_ClientEarly.h"
#include "dispvidctrl_datapool.h"
#include "dispvidctrl_tclControl_Avm.h"
#include "dispvidctrl_tclControl_Rvc.h"
#include "dispvidctrl_tclControl_Mvc.h"
#include "dispvidctrl_tclControl_Video.h"
#include "I_dispvidctrl_tclDisplaySettings.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_SERVER_SYNCDISPLAY
#include "trcGenProj/Header/dispvidctrl_tclSyncHandler.cpp.trc.h"
#endif

/*******************************************************************************
| defines and macros (scope: module-local)
|-----------------------------------------------------------------------------*/
#define DISPVIDCTRL_EVENT_VIDEO_WAIT  "ERALY_VIDEO_READY"
#define DISPVIDCTRL_EVENT_VIDEO_REQ   0x00000001

/*******************************************************************************
| typedefs (scope: module-local)
|-----------------------------------------------------------------------------*/
#define ELEMENTE(array)       ( (tU16)( sizeof( array ) / sizeof( array[0] ) ) )


/*******************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------------*/
dispvidctrl_tclSyncHandler* dispvidctrl_tclSyncHandler::_poMyStaticRef = NULL;

dispvidctrl_tclSyncHandler::TFsmStateConf dispvidctrl_tclSyncHandler::_aFsmStateTable[eSyncFsmMax] =
{
   { eSyncFsmIdle,
     0,                        // u32Timeout
     vEntryIdle,               // tFsmStateEntryFunc
     NULL,                     // tFsmStateReachedFunc
     NULL                      // tFsmStateExitFunc
   },

   { eSyncFsmInit,
     2000,                     // u32Timeout
     vEntryInit,               // tFsmStateEntryFunc
     NULL,                     // tFsmStateReachedFunc
     NULL                      // tFsmStateExitFunc
   },

   { eSyncFsmAnimation,
     40000,                    // u32Timeout
     vEntryAnimation,          // tFsmStateEntryFunc
     NULL,                     // tFsmStateReachedFunc
     vExitAnimation            // tFsmStateExitFunc
   },

   { eSyncFsmRvc,
     0,                        // u32Timeout
     vEntryRvc,                // tFsmStateEntryFunc
     NULL,                     // tFsmStateReachedFunc
     vExitRvc                  // tFsmStateExitFunc
   }
};

dispvidctrl_tclSyncHandler::TStateTransitions dispvidctrl_tclSyncHandler::_aFsmStateTransitionTable[] =
{
   { 100,                         // transitionId
     eSyncFsmInit,                // current FSM State
     FSM_DONT_CARE,               // trigger update
     eSyncTriggerLayerMgrRdy
     |eSyncTriggerVideoPlayerRdy
     |eSyncTriggerIncUp
     |eSyncTriggerRvcDebounced,   // current active trigger(s)
     eSyncTriggerDrvAssCtrl,      // current not active trigger(s)
     eSyncFsmRvc                  // next FSM State
   },
   { 102,                         // transitionId
     eSyncFsmInit,                // current FSM State
     FSM_DONT_CARE,               // trigger update
     eSyncTriggerLayerMgrRdy
     |eSyncTriggerVideoPlayerRdy
     |eSyncTriggerIncUp
     |eSyncTriggerRvcDebounced,   // current active trigger(s)
     FSM_DONT_CARE,               // current not active trigger(s)
     eSyncFsmRvc                  // next FSM State
   },
   { 104,                         // transitionId
     eSyncFsmInit,                // current FSM State
     FSM_DONT_CARE,               // trigger update
     eSyncTriggerLayerMgrRdy
     |eSyncTriggerVideoPlayerRdy
     |eSyncTriggerIncUp,          // current active trigger(s)
     FSM_DONT_CARE,               // current not active trigger(s)
     eSyncFsmIdle                 // next FSM State
   },
   { 203,                         // transitionId
     eSyncFsmIdle,                // current FSM State
     FSM_DONT_CARE,               // trigger update
     eSyncTriggerRvcDebounced,    // current active trigger(s)
     eSyncTriggerDrvAssCtrl,      // current not active trigger(s)
     eSyncFsmRvc                  // next FSM State
   },
   { 204,                         // transitionId
     eSyncFsmIdle,                // current FSM State
     FSM_DONT_CARE,               // trigger update
     eSyncTriggerRvcDebounced,    // current active trigger(s)
     FSM_DONT_CARE,               // current not active trigger(s)
     eSyncFsmRvc                  // next FSM State
   },
   { 210,                         // transitionId
     eSyncFsmIdle,                // current FSM State
     eSyncTriggerAnimation,       // trigger update
     eSyncTriggerAnimation,       // current active trigger(s)
     FSM_DONT_CARE,               // current not active trigger(s)
     eSyncFsmAnimation            // next FSM State
   },
   { 300,                         // transitionId
     eSyncFsmAnimation,           // current FSM State
     eSyncTriggerAnimation,       // trigger update
     eSyncTriggerHmiReady,        // current active trigger(s)
     eSyncTriggerAnimation,       // current not active trigger(s)
     eSyncFsmIdle                 // next FSM State
   },
   { 301,                         // transitionId
     eSyncFsmAnimation,           // current FSM State
     FSM_DONT_CARE,               // trigger update
     eSyncTriggerRvcDebounced,    // current active trigger(s)
     eSyncTriggerDrvAssCtrl,      // current not active trigger(s)
     eSyncFsmRvc                  // next FSM State
   },
   { 302,                         // transitionId
     eSyncFsmAnimation,           // current FSM State
     FSM_DONT_CARE,               // trigger update
     eSyncTriggerRvcDebounced,    // current active trigger(s)
     FSM_DONT_CARE,               // current not active trigger(s)
     eSyncFsmRvc                  // next FSM State
   },
   { 303,                       // transitionId
     eSyncFsmAnimation,         // current FSM State
     eSyncTriggerHmiReady,      // trigger update
     FSM_DONT_CARE,             // current active trigger(s)
     eSyncTriggerAnimation,     // current not active trigger(s)
     eSyncFsmIdle               // next FSM State
   },
   { 304,                       // transitionId
     eSyncFsmAnimation,         // current FSM State
     eSyncTriggerStateTimeOut,  // trigger update
     FSM_DONT_CARE,             // current active trigger(s)
     FSM_DONT_CARE,             // current not active trigger(s)
     eSyncFsmIdle               // next FSM State
   },
   { 400,                       // transitionId
     eSyncFsmRvc,               // current FSM State
     FSM_DONT_CARE,             // trigger update
     FSM_DONT_CARE,             // current active trigger(s)
     eSyncTriggerRvcDebounced
     |eSyncTriggerDrvAssCtrl
     |eSyncTriggerHmiReady,     // current not active trigger(s)
     eSyncFsmAnimation          // next FSM State
   },
   { 401,                       // transitionId
     eSyncFsmRvc,               // current FSM State
     FSM_DONT_CARE,             // trigger update
     FSM_DONT_CARE,             // current active trigger(s)
     eSyncTriggerRvcDebounced
     |eSyncTriggerDrvAssCtrl,   // current not active trigger(s)
     eSyncFsmIdle               // next FSM State
   },
   { 402,                       // transitionId
     eSyncFsmRvc,               // current FSM State
     FSM_DONT_CARE,             // trigger update
     eSyncTriggerDrvAssCtrl,    // current active trigger(s)
     eSyncTriggerRvcDebounced
     | eSyncTriggerHmiReady,    // current not active trigger(s)
     eSyncFsmAnimation          // next FSM State
   },
   { 403,                       // transitionId
     eSyncFsmRvc,               // current FSM State
     FSM_DONT_CARE,             // trigger update
     eSyncTriggerDrvAssCtrl,    // current active trigger(s)
     eSyncTriggerRvcDebounced,  // current not active trigger(s)
     eSyncFsmIdle               // next FSM State
   }
};


/*******************************************************************************
| function prototype (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function implementation (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
*
* FUNCTION:    dispvidctrl_tclSyncHandler()
*
* DESCRIPTION: constructor
*
* PARAMETER:   const dispvidctrl_tclAppMain*
*
* RETURNVALUE: none
*
*******************************************************************************/
dispvidctrl_tclSyncHandler::dispvidctrl_tclSyncHandler(const dispvidctrl_tclAppMain* poMainAppl)
: dispvidctrl_tclSyncHandlerFsmBase(eSyncFsmInit, ELEMENTE(_aFsmStateTransitionTable), _aFsmStateTransitionTable, ELEMENTE(_aFsmStateTable), _aFsmStateTable, TR_CLASS_DISPVIDCTRL_SERVER_SYNCDISPLAY)
, I_dispvidctrl_tclSyncHandler(poMainAppl)
, m_poDisplaySettings(OSAL_NULL)
, m_bEarlyCameraScreenControlOff(FALSE)
, m_bEarlyIncIgnition(FALSE)
, m_bEarlyIncAvmInitializing(TRUE)
, m_bEarlyIncAvmViewChangeRequest(FALSE)
, m_bAvmConfigured(FALSE)
, m_bAvmInitFlagTx(TRUE)
, m_u8AvmLanguage(0)
, m_enBacklightState_RVC(eBackLight_RVC_Auto)
, m_bDimmingModeRvc(FALSE)
, _hStateTimer(OSAL_C_INVALID_HANDLE)
, m_animationstatus(FALSE)
{
   _poMyStaticRef = this;

   _poSrvSync = NULL;

   _bRvcWasActive = FALSE;

   ETG_TRACE_USR4(("dispvidctrl_tclSyncHandler() entered."));

   if( OSAL_s32TimerCreate( (OSAL_tpfCallback)vStateTimer, ( tPVoid ) this,
                            &_hStateTimer) != OSAL_OK){
      _hStateTimer = OSAL_C_INVALID_HANDLE;
   }

   _hEvVideoReady = OSAL_C_INVALID_HANDLE;
   if (OSAL_s32EventCreate(DISPVIDCTRL_EVENT_VIDEO_WAIT, &_hEvVideoReady) == OSAL_ERROR){
      //ETG_TRACE_USR1(("Creation of SPM shutdown event failed"));
       ETG_TRACE_FATAL( ( "dispvidctrl_tclSyncHandler::dispvidctrl_tclSyncHandler(): Failed to create event" ) );
   }
}

/*******************************************************************************
*
* FUNCTION:    ~dispvidctrl_tclSyncHandler()
*
* DESCRIPTION: destructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*******************************************************************************/
dispvidctrl_tclSyncHandler::~dispvidctrl_tclSyncHandler()
{
   ETG_TRACE_USR4(("~dispvidctrl_tclSyncHandler() entered."));

   m_poDisplaySettings = OSAL_NULL;
   if (OSAL_C_INVALID_HANDLE != _hStateTimer) {
      if (OSAL_s32TimerSetTime( _hStateTimer, 0, 0) != OSAL_OK){     // Stop Timer before deleting
         ETG_TRACE_ERR( ( "~dispvidctrl_tclSyncHandler() SPM: !!!!!! Error detected !!!!!!" ) );
      }
      if( OSAL_s32TimerDelete( _hStateTimer) != OSAL_OK){            // Delete Timer
         ETG_TRACE_ERR( ( "~dispvidctrl_tclSyncHandler() SPM: !!!!!! Error detected !!!!!!" ) )
      }
      _hStateTimer = OSAL_C_INVALID_HANDLE;
   }
   
   if (OSAL_C_INVALID_HANDLE != _hEvVideoReady) {
      (tVoid) OSAL_s32EventClose(_hEvVideoReady);
      (tVoid) OSAL_s32EventDelete(DISPVIDCTRL_EVENT_VIDEO_WAIT);
      _hEvVideoReady = OSAL_C_INVALID_HANDLE;
   }
   
   _poSrvSync = NULL;

}

/*******************************************************************************
*
* FUNCTION:    vHandleMessage()
*
* DESCRIPTION: handle internal message
*
* PARAMETER:   dispvidctrl_tclBaseIf::TMsg* pMsg
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid dispvidctrl_tclSyncHandler::vHandleMessage(dispvidctrl_tclBaseIf::TMsg* pMsg)
{
    ETG_TRACE_USR1(("dispvidctrl_tclSyncHandler::vHandleMessage() entered %u -> data: %d.", ETG_CENUM(dispvidctrl_tclBaseIf::ECmdTypes , (tU32)pMsg->eCmd), pMsg->u.u32Data));

    switch (pMsg->eCmd)
    {
    case dispvidctrl_tclBaseIf::eSyncFirstFrame:
        {
            // Handle First Frame rendered
            I_dispvidctrl_tclVideoPlayer* poVideoPlayer = dynamic_cast<I_dispvidctrl_tclVideoPlayer*>(_poMyStaticRef->_cpoMain->getHandler("I_dispvidctrl_tclVideoPlayer"));
            if (poVideoPlayer){
                poVideoPlayer->vHandleFirstFrameRendered();
            }

            // switch on the light
            I_dispvidctrl_tclClientDimmingIncAdaptor* _poDimInc = dynamic_cast<I_dispvidctrl_tclClientDimmingIncAdaptor*>(_poMyStaticRef->_cpoMain->getHandler("I_dispvidctrl_tclClientDimmingIncAdaptor"));
            if (_poDimInc != NULL) {
                dispvidctrl_tclAppMain::vWriteStartupLog("[RVC-EARLY]: Send BACKLIGHT ON to V850.");
                m_enBacklightState_RVC = eBackLight_RVC_On;
                _poDimInc->sendDISPLAY_READY(::dimming_inc_fi::eLCD_READY);
                _poDimInc->sendDISPLAY_REQ(::dimming_inc_fi::eACTIVATE_DISPLAY);
            }

            m_bDimmingModeRvc = TRUE;
            if (m_poDisplaySettings)
            {
               ETG_TRACE_FATAL( ( "dispvidctrl_tclSyncHandler::call Dimming Mode RVC ON"));
               m_poDisplaySettings->vSetDimmingModeRvc();
            }
            if (OSAL_s32EventPost(_poMyStaticRef->_hEvVideoReady, DISPVIDCTRL_EVENT_VIDEO_REQ, OSAL_EN_EVENTMASK_OR) != OSAL_OK){
                ETG_TRACE_FATAL( ( "dispvidctrl_tclSyncHandler::vHandleMessage(): !!!!!! Error detected while posting event to _hEvVideoReady !!!!!!" ) );
            }
        }
        break;
    case dispvidctrl_tclBaseIf::eVideoPlayerReady:
        vNewTrigger(eSyncTriggerVideoPlayerRdy, TRUE);
        break;
    case dispvidctrl_tclBaseIf::eStartAnimation:
        vNewTrigger(eSyncTriggerAnimation, TRUE);
        break;
    case dispvidctrl_tclBaseIf::eStoptAnimation:
        vNewTrigger(eSyncTriggerAnimation, FALSE);
        //animation done / file played --> inform HMI
        {
            syncdisplay_fi_tcl_e8_EarlyVideoLayer_Status eVideoLayer;
            eVideoLayer.enType = syncdisplay_fi_tcl_e8_EarlyVideoLayer_Status::FI_EN_VIDEOLAYER_IDLE;
            if (_poSrvSync) _poSrvSync->sendEarlyVideoLayerStatusStatus(eVideoLayer);
        }
        break;
    case dispvidctrl_tclBaseIf::eSyncStateTimeout:
        vNewTrigger(eSyncTriggerStateTimeOut, TRUE);
        break;
    case dispvidctrl_tclBaseIf::eControlSyncHandler:
        vHandleControl((tenControl_SyncHandler) pMsg->u.u32Data);
        break;
    default:
        break;
    }
}

/******************************************************************************/
/* FUNCTION     vHandleControl()                                              */
/******************************************************************************/
/**
*  \brief       direct call interface
*
*  \param       enumeration control value
*               tU32 data value
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclSyncHandler::vHandleControl(tenControl_SyncHandler enControl_SyncHandler, tU32 u32Value)
{
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32Value );
   ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclSyncHandler::vHandleControl() entered %u -> data: %d.",OSAL_ClockGetElapsedTime(), ETG_CENUM(tenControl_SyncHandler, enControl_SyncHandler), u32Value));
   switch (enControl_SyncHandler)
   {
      case EN_CONTROL_SH__EARLY_CAMERA_SCREEN_CONTROL_OFF:
         m_bEarlyCameraScreenControlOff = TRUE;
         // for this event we want to update only the trigger state
         if (eSyncFsmRvc == _poMyStaticRef->_u32FsmState)
         {
            vUpdateTriggerState((eSyncFsmTrigger) (I_dispvidctrl_tclSyncHandler::eSyncTriggerDrvAssCtrl | I_dispvidctrl_tclSyncHandler::eSyncTriggerRvcDebounced), TRUE);
         }
         else
         {
            vUpdateTriggerState(I_dispvidctrl_tclSyncHandler::eSyncTriggerDrvAssCtrl, TRUE);
         }
         // disallow early INC spontaneous updates on signal changes (reverse gear, vehicle speed, steering angle ...)
         // (from V850 early App)
         {
            I_dispvidctrl_tclClientEarly* poClientEarly = dynamic_cast<I_dispvidctrl_tclClientEarly*>(_poMyStaticRef->_cpoMain->getHandler("I_dispvidctrl_tclClientEarly"));
            if (poClientEarly)
            {
               if (poClientEarly->sendCOMMUNICATION_CTRL(0, (tU8) FALSE) == FALSE)
               {
                  ETG_TRACE_FATAL(("dispvidctrl_tclSyncHandler:: Failed to stop early INC automatic indications!"));
               }
            }
         }
         break;
      case EN_CONTROL_SH__CAMERA_SCREEN_ON:
		// m_animationstatus = TRUE;
         vNewTrigger(eSyncTriggerRvcDebounced, TRUE);
         break;
      case EN_CONTROL_SH__CAMERA_SCREEN_OFF:
         
         // In Case of quick Reverse On Off it might happen that 
         // the Camera Screen is not switched On (for eg. while waiting for HMI Confirmation for Black-in)
         // But the Black-out will be drawn on middleware when entering Black-out after Reverse Off
         // to remove this black screen we need to explicity ask Video Player to do it
         // this is normally handled in vExitRvc
		// m_animationstatus = FALSE;
         if (eSyncFsmRvc != _poMyStaticRef->_u32FsmState)
         {
            ETG_TRACE_FATAL(("dispvidctrl_tclSyncHandler:: EN_CONTROL_SH__CAMERA_SCREEN_OFF Force Deactivate Rvc, Current Fsm State %u", ETG_CENUM(tenSyncFsmState, _poMyStaticRef->_u32FsmState)));
            // To make sure the graphics surface is invisible (In case of quick RVC Toggle)
            I_dispvidctrl_tclVideoPlayer* poVideoPlayer = dynamic_cast<I_dispvidctrl_tclVideoPlayer*>(_poMyStaticRef->_cpoMain->getHandler("I_dispvidctrl_tclVideoPlayer"));
            if (poVideoPlayer){
                poVideoPlayer->vActivateRvc(FALSE);
            }
         }
         
         vNewTrigger(eSyncTriggerRvcDebounced, FALSE);

         //backlight for RVC not required anymore
         m_enBacklightState_RVC = eBackLight_RVC_Auto;
         //post message to dimming client
         {
            tChar strIfName[] = "I_dispvidctrl_tclClientDimming";
            dispvidctrl_tclAppMain::theServer()->vPostMsgDimStateChanged(strIfName, (tU32) m_enBacklightState_RVC);
         //dimming mode RVC off
         m_bDimmingModeRvc = FALSE;
         if (m_poDisplaySettings)
         {
            ETG_TRACE_FATAL( ( "dispvidctrl_tclSyncHandler::call Dimming Mode RVC OFF"));
            m_poDisplaySettings->vSetDimmingModeRvc();
         }
         }
         break;
      case EN_CONTROL_SH__EARLY_INC_UP:
         break;
      case EN_CONTROL_SH__EARLY_INC_ACCESSORY:
      case EN_CONTROL_SH__EARLY_INC_REVERSE:
      case EN_CONTROL_SH__EARLY_INC_IGNITION:
      case EN_CONTROL_SH__EARLY_INC_AVM_INIT_FLAG_RX:
      case EN_CONTROL_SH__EARLY_INC_AVM_VIEW_CHG_REQ:
         if (m_bAvmConfigured)
         {
            _vHandleEarlyAvm(enControl_SyncHandler, u32Value);
         }
         break;
      case EN_CONTROL_SH__EARLY_INC_STEERING_ANGLE:
         {
            if (m_bEarlyCameraScreenControlOff == FALSE) {
               // get steering angle and state and update videoplayer
               // in case videoplayer can not process the value yet, videoplayer is holding the value
               I_dispvidctrl_tclClientEarly* poClientEarly = dynamic_cast<I_dispvidctrl_tclClientEarly*>(_poMyStaticRef->_cpoMain->getHandler("I_dispvidctrl_tclClientEarly"));
               if (poClientEarly)
               {
                  tS16 s16FrontSteeringAngle = 0;
                  tBool bFrontSteeringAngleValid;
               
                  bFrontSteeringAngleValid = poClientEarly->bGetFrontSteeringAngle(s16FrontSteeringAngle);
                  ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclSyncHandler::vHandleControl() angle %d  state: %d.",OSAL_ClockGetElapsedTime(), s16FrontSteeringAngle, bFrontSteeringAngleValid));
               
                  I_dispvidctrl_tclVideoPlayer* poVideoPlayer = dynamic_cast<I_dispvidctrl_tclVideoPlayer*>(_poMyStaticRef->_cpoMain->getHandler("I_dispvidctrl_tclVideoPlayer"));
                  if (poVideoPlayer) {
                     if (bFrontSteeringAngleValid) {
                        poVideoPlayer->vSetSteeringWheelAngle(s16FrontSteeringAngle);
                     }
                     else {
                        //set steering angle "0" to hide dynamic guideline
                        poVideoPlayer->vSetSteeringWheelAngle(0);
                     }
                  }
               }
            }
         }
         break;
      case EN_CONTROL_SH__DIMMING_INC_CAR_MODE_DAY:
         if (m_bEarlyCameraScreenControlOff == FALSE) {
            ETG_TRACE_FATAL(("dispvidctrl_tclSyncHandler::vHandleControl() CAR Mode: Day"));
            I_dispvidctrl_tclVideoPlayer* poVideoPlayer = dynamic_cast<I_dispvidctrl_tclVideoPlayer*>(_poMyStaticRef->_cpoMain->getHandler("I_dispvidctrl_tclVideoPlayer"));
            if (poVideoPlayer) {
               poVideoPlayer->vSetNight(VIDEOPLAYER_DAY);
            }
         }
         break;
      case EN_CONTROL_SH__DIMMING_INC_CAR_MODE_NIGHT:
         if (m_bEarlyCameraScreenControlOff == FALSE) {
            ETG_TRACE_FATAL(("dispvidctrl_tclSyncHandler::vHandleControl() CAR Mode: Night"));
            I_dispvidctrl_tclVideoPlayer* poVideoPlayer = dynamic_cast<I_dispvidctrl_tclVideoPlayer*>(_poMyStaticRef->_cpoMain->getHandler("I_dispvidctrl_tclVideoPlayer"));
            if (poVideoPlayer) {
               poVideoPlayer->vSetNight(VIDEOPLAYER_NIGHT);
            }
         }
         break;
      default:
         break;
   }
}


// Shall be only a temporary solution.
// We have to rework the sync handler state machine to handle AVM specific requirements also.
tVoid dispvidctrl_tclSyncHandler::_vHandleEarlyAvm(tenControl_SyncHandler enControl_SyncHandler, tU32 u32Value)
{
   if (m_bEarlyCameraScreenControlOff == FALSE)
   {
      switch (enControl_SyncHandler)
      {
         case EN_CONTROL_SH__EARLY_INC_IGNITION:
            m_bEarlyIncIgnition = (tBool) u32Value;
            ETG_TRACE_FATAL(("dispvidctrl_tclSyncHandler::_vHandleEarlyAvm() IGNITION: %d", m_bEarlyIncIgnition));
            break;
         case EN_CONTROL_SH__EARLY_INC_AVM_INIT_FLAG_RX:
            {
               m_bEarlyIncAvmInitializing = (tBool) u32Value;
               ETG_TRACE_FATAL(("dispvidctrl_tclSyncHandler::_vHandleEarlyAvm() AVM Initializing: %d", m_bEarlyIncAvmInitializing));
               if (m_bEarlyIncIgnition)
               {
                  // Ignition is ON
                  if (m_bEarlyIncAvmInitializing)
                  {
                     // Initializing active
                     // send USPStatusRequest(5FE message) initializing signal "set"
                     I_dispvidctrl_tclClientEarly* poClientEarly = dynamic_cast<I_dispvidctrl_tclClientEarly*>(_poMyStaticRef->_cpoMain->getHandler("I_dispvidctrl_tclClientEarly"));
                     if (poClientEarly)
                     {
                        (tVoid) poClientEarly->sendACTION(::early_inc_fi::eAVM_SET_INIT_FLAG, (tU8) TRUE);
                        m_bAvmInitFlagTx = TRUE;
                     }
                  }
                  else
                  {
                     I_dispvidctrl_tclClientEarly* poClientEarly = dynamic_cast<I_dispvidctrl_tclClientEarly*>(_poMyStaticRef->_cpoMain->getHandler("I_dispvidctrl_tclClientEarly"));
                     if (poClientEarly)
                     {
                        // send Language information
// todo: enable sending of language value, when multiple signal write is available to update the signals from IMX proxy side
//       (see also AVM Control when Control will be enabled)
                        //(tVoid) poClientEarly->sendACTION(::early_inc_fi::eAVM_SET_LANGUAGE, m_u8AvmLanguage);
                        // send USPStatusRequest(5FE message) initializing signal "cleared"
                        (tVoid) poClientEarly->sendACTION(::early_inc_fi::eAVM_SET_INIT_FLAG, (tU8) FALSE);
                        m_bAvmInitFlagTx = FALSE;
                     }
                  }
               }
               else
               {
                  // Ignition is OFF
                  // send USPStatusRequest(5FE message) initializing signal "set"
                  I_dispvidctrl_tclClientEarly* poClientEarly = dynamic_cast<I_dispvidctrl_tclClientEarly*>(_poMyStaticRef->_cpoMain->getHandler("I_dispvidctrl_tclClientEarly"));
                  if (poClientEarly)
                  {
                     (tVoid) poClientEarly->sendACTION(::early_inc_fi::eAVM_SET_INIT_FLAG, (tU8) TRUE);
                     m_bAvmInitFlagTx = TRUE;
                  }
               }
            }
            break;
         case EN_CONTROL_SH__EARLY_INC_AVM_VIEW_CHG_REQ:
            {
               m_bEarlyIncAvmViewChangeRequest = (tBool) u32Value;
               ETG_TRACE_FATAL(("dispvidctrl_tclSyncHandler::_vHandleEarlyAvm() AVM ViewChangeRequest: %d", m_bEarlyIncAvmViewChangeRequest));
               if (m_bEarlyIncIgnition)
               {
                  // Ignition is ON
                  if (m_bEarlyIncAvmInitializing == FALSE)
                  {
                     // Initialisation is done
                     if (m_bEarlyIncAvmViewChangeRequest)
                     {
                        // start visualisation
                        vNewTrigger(eSyncTriggerRvcDebounced, TRUE);
                     }
                     else
                     {
                        // stop visualisation
                        vNewTrigger(eSyncTriggerRvcDebounced, FALSE);
                     }
                  }
                  else
                  {
                     // Initialisation is not done
                     // stop visualisation
                     vNewTrigger(eSyncTriggerRvcDebounced, FALSE);
                  }
               }
               else
               {
                  // Ignition is OFF
                  // stop visualisation
                  vNewTrigger(eSyncTriggerRvcDebounced, FALSE);
               }
            }
            break;
         default:
            break;
      }
   }
   else
   {
      ETG_TRACE_FATAL(("dispvidctrl_tclSyncHandler::_vHandleEarlyAvm() Early Control is OFF"));
   }
}


/*******************************************************************************
*
* FUNCTION:    vHandleTraceMessage()
*
* DESCRIPTION:    handler for trace command
*
* PARAMETER:      const tUChar* puchData: reference of received message
*
* RETURNVALUE:    None.
*
*******************************************************************************/
tVoid dispvidctrl_tclSyncHandler::vHandleTraceMessage(const tUChar* puchData)
{
   ETG_TRACE_USR4(("dispvidctrl_tclSyncHandler::vHandleTraceMessage() entered (data: 0x%08x).", puchData ));
}

/*******************************************************************************
*
* FUNCTION:    vGetReferences(tVoid)
*
* DESCRIPTION:    Function to get all reference needed by this class.
*           A reference should always be the Interface class of the object
*
* PARAMETER:   None.
*
* RETURNVALUE:    None.
*
*******************************************************************************/
tVoid dispvidctrl_tclSyncHandler::vGetReferencesEarly(tVoid)
{
   ETG_TRACE_FATAL(("dispvidctrl_tclSyncHandler::vGetReferencesEarly() entered."));

   TEarlyConfiguration rEarlyConfig;
   _cpoMain->vGetEarlyConfig(rEarlyConfig);
   
   if ((rEarlyConfig.tEarlyNorForce.u32GuidelineCameraSystemType == CAMERA_TYPE_AVM_ENTRY)
   ||  (rEarlyConfig.tEarlyNorForce.u32GuidelineCameraSystemType == CAMERA_TYPE_AVM_IPA)
   ||  (rEarlyConfig.tEarlyNorForce.u32GuidelineCameraSystemType == CAMERA_TYPE_AVM_LOW)
   ||  (rEarlyConfig.tEarlyNorForce.u32GuidelineCameraSystemType == CAMERA_TYPE_AVM_ENTRY_RENAULT)
   ||  (rEarlyConfig.tEarlyNorForce.u32GuidelineCameraSystemType == CAMERA_TYPE_AVM_HIGH)
   ||  (rEarlyConfig.tEarlyNorForce.u32GuidelineCameraSystemType == CAMERA_TYPE_AVM_PHASE3))
   {
      m_bAvmConfigured = TRUE;
   }
   m_u8AvmLanguage = rEarlyConfig.tEarlyNor.u8AvmLanguage;
}

tVoid dispvidctrl_tclSyncHandler::vGetReferences(tVoid)
{
   ETG_TRACE_FATAL(("dispvidctrl_tclSyncHandler::vGetReferences() entered."));

   m_poDisplaySettings = dynamic_cast<I_dispvidctrl_tclDisplaySettings*>(_cpoMain->getHandler("I_dispvidctrl_tclDisplaySettings"));
   if(m_poDisplaySettings)
   {
       DISPVIDCTRL_NULL_POINTER_CHECK(m_poDisplaySettings);
   }
}

/*******************************************************************************
*
* FUNCTION:    tVoid vStartCommunication()
*
* DESCRIPTION:    Function to start all dynamic objects e.g. threads, ...
*
* PARAMETER:   None.
*
* RETURNVALUE:    None.
*
*******************************************************************************/
tVoid dispvidctrl_tclSyncHandler::vStartCommunicationEarly(tVoid)
{
   ETG_TRACE_FATAL(("dispvidctrl_tclSyncHandler::vStartCommunicationEarly() entered."));
}

tVoid dispvidctrl_tclSyncHandler::vStartCommunication(tVoid)
{
   ETG_TRACE_FATAL(("dispvidctrl_tclSyncHandler::vStartCommunication() entered."));

   _poSrvSync = dynamic_cast<Idispvidctrl_tclServiceSyncdisplay*>(_poMyStaticRef->_cpoMain->getHandler("Idispvidctrl_tclServiceSyncdisplay"));
   //update with current state
   if (_poSrvSync) {
       syncdisplay_fi_tcl_e8_EarlyVideoLayer_Status eVideoLayer;
       if ((tU32)eSyncFsmRvc == u32GetFsmState()){
           eVideoLayer.enType = syncdisplay_fi_tcl_e8_EarlyVideoLayer_Status::FI_EN_VIDEOLAYER_RVC;
       } else if ((tU32)eSyncFsmAnimation == u32GetFsmState()){
           eVideoLayer.enType = syncdisplay_fi_tcl_e8_EarlyVideoLayer_Status::FI_EN_VIDEOLAYER_ANIMATION;
       } else if ((tU32)eSyncFsmIdle == u32GetFsmState()){
           eVideoLayer.enType = syncdisplay_fi_tcl_e8_EarlyVideoLayer_Status::FI_EN_VIDEOLAYER_ANIMATION_NOT_YET_STARTED;
       } else {
           eVideoLayer.enType = syncdisplay_fi_tcl_e8_EarlyVideoLayer_Status::FI_EN_VIDEOLAYER_ANIMATION_NOT_YET_STARTED;
       }
       _poSrvSync->sendEarlyVideoLayerStatusStatus(eVideoLayer);
   }
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfo()
*
* DESCRIPTION:    Trace information
*
* PARAMETER:   None.
*
* RETURNVALUE:    None.
*
*******************************************************************************/
tVoid dispvidctrl_tclSyncHandler::vTraceInfo()
{
   ETG_TRACE_FATAL(("dispvidctrl_tclSyncHandler::vTraceInfo() *********** SyncHandler states ***************************************"));
   ETG_TRACE_FATAL(("dispvidctrl_tclSyncHandler::vTraceInfo() m_enBacklightState_RVC .................... %-40u", ETG_CENUM(tenBackLightState , m_enBacklightState_RVC)));
   ETG_TRACE_FATAL(("dispvidctrl_tclSyncHandler::vTraceInfo() m_bDimmingModeRvc ........................ %-40u", m_bDimmingModeRvc));

   vFsmInfo();
}

/******************************************************************************/
/* FUNCTION     vNewTrigger                                                   */
/******************************************************************************/
/**
*  \brief       To be called to enter a new Trigger
*
*  \param       trigger and state
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclSyncHandler::vNewTrigger(eSyncFsmTrigger eTrigger, tBool bTriggerState) const
{
   ETG_TRACE_FATAL(("dispvidctrl_tclSyncHandler::vNewTrigger() Trigger: %d, State: %d", ETG_CENUM(eSyncFsmTrigger, eTrigger), bTriggerState));
   _poMyStaticRef->u32CalcNewFsmState( (tU32)eTrigger, bTriggerState);
   return;
}

/******************************************************************************/
/* FUNCTION     vUpdateTriggerState                                           */
/******************************************************************************/
/**
*  \brief       To be called to correct Trigger states
*
*  \param       Trigger and State
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclSyncHandler::vUpdateTriggerState(eSyncFsmTrigger eTrigger, tBool bTriggerState) const
{
   ETG_TRACE_FATAL(("dispvidctrl_tclSyncHandler::vUpdateTrigger() Trigger: %d, State: %d", ETG_CENUM(eSyncFsmTrigger, eTrigger), bTriggerState));
   _poMyStaticRef->vUpdateTrigger((tU32) eTrigger, bTriggerState);
}


OSAL_tpfCallback dispvidctrl_tclSyncHandler::vStateTimer(tVoid *pArg){
   (tVoid)pArg;
   ETG_TRACE_USR1( ( "dispvidctrl_tclSyncHandler::vStateTimer() --> timeout detected!" ) );
   tChar strIfName[] = "I_dispvidctrl_tclSyncHandler";
   _poMyStaticRef->_cpoMain->theServer()->vPostMsgSyncStateTimeout(strIfName);
   return( 0 );
}

tVoid dispvidctrl_tclSyncHandler::vStartFsmStateTimer(tU32 u32Time){
   if (OSAL_C_INVALID_HANDLE != _hStateTimer) {
      ETG_TRACE_USR1( ( "dispvidctrl_tclSyncHandler::vStartFsmStateTimer() --> %dms!", u32Time ) );
      if(u32Time > 0){
         // set Shutdown-Timer active
         OSAL_s32TimerSetTime(_hStateTimer, u32Time, 0);
      } else {
         // reset Shutdown-Timer
         OSAL_s32TimerSetTime(_hStateTimer,       0, 0);
      }
   }
}

tVoid dispvidctrl_tclSyncHandler::vFsmInfo(tVoid){
   // get remaining state timeout time

   tU32 u32RemainingTime = 0;
   tU32 u32IntervalTime  = 0;

   if (OSAL_C_INVALID_HANDLE != _hStateTimer) {
      if (OSAL_s32TimerGetTime(_hStateTimer, &u32RemainingTime, &u32IntervalTime) != OSAL_OK){
         // don't need to error trace -> on ADIT platform even if timer isn't running return value is != OK
      }
   }

   vTraceStatusInfo(u32RemainingTime);
   return;
}

tVoid dispvidctrl_tclSyncHandler::vStateChangeDetected(){
   ETG_TRACE_FATAL(("dispvidctrl_tclSyncHandler::vStateChangeDetected() NewState: %d, Old State: %d", ETG_CENUM(tenSyncFsmState, _poMyStaticRef->_u32FsmState), ETG_CENUM(tenSyncFsmState, _poMyStaticRef->_u32LastFsmState)));
}


tVoid dispvidctrl_tclSyncHandler::vEntryIdle(tVoid){
   dispvidctrl_tclAppMain::vWriteStartupLog("[RVC-EARLY]: SyncHandler: vEntryIdle");

    if (OSAL_s32EventPost(_poMyStaticRef->_hEvVideoReady, DISPVIDCTRL_EVENT_VIDEO_REQ, OSAL_EN_EVENTMASK_OR) != OSAL_OK){
        ETG_TRACE_FATAL( ( "dispvidctrl_tclSyncHandler::vEntryIdle(): !!!!!! Error detected !!!!!!" ) );
    }
    I_dispvidctrl_tclVideoPlayer* poVideoPlayer = dynamic_cast<I_dispvidctrl_tclVideoPlayer*>(_poMyStaticRef->_cpoMain->getHandler("I_dispvidctrl_tclVideoPlayer"));
    if (poVideoPlayer){
        poVideoPlayer->vPrepareAnimation();
    }

    Idispvidctrl_tclClientVideomanager* poVideoManager = dynamic_cast<Idispvidctrl_tclClientVideomanager*>(_poMyStaticRef->_cpoMain->getHandler("Idispvidctrl_tclClientVideomanager"));
    if (poVideoManager && poVideoManager->bVideoManagerIsAvailable()){
        ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclSyncHandler::vEntryIdle() Set videomanager RVC state to inactive", OSAL_ClockGetElapsedTime()));
        poVideoManager->vSetRVCStateInactive();
    }
	if (_poMyStaticRef->_poSrvSync) {
		syncdisplay_fi_tcl_e8_Animation_Status animationstatus;
			animationstatus.enType = syncdisplay_fi_tcl_e8_Animation_Status::FI_EN_ANIMATION_FINISHED;
			_poMyStaticRef->_poSrvSync->sendAnimationStatusStatus(animationstatus);
	}

}
tVoid dispvidctrl_tclSyncHandler::vEntryInit(tVoid){
   dispvidctrl_tclAppMain::vWriteStartupLog("[RVC-EARLY]: SyncHandler: vEntryInit");
}
tVoid dispvidctrl_tclSyncHandler::vEntryAnimation(tVoid){
   dispvidctrl_tclAppMain::vWriteStartupLog("[RVC-EARLY]: SyncHandler: vEntryAnimation");

   if (_poMyStaticRef->_poSrvSync) {
       syncdisplay_fi_tcl_e8_EarlyVideoLayer_Status eVideoLayer;
       eVideoLayer.enType = syncdisplay_fi_tcl_e8_EarlyVideoLayer_Status::FI_EN_VIDEOLAYER_ANIMATION;
       _poMyStaticRef->_poSrvSync->sendEarlyVideoLayerStatusStatus(eVideoLayer);
	   syncdisplay_fi_tcl_e8_Animation_Status animationstatus;
			animationstatus.enType = syncdisplay_fi_tcl_e8_Animation_Status::FI_EN_ANIMATION_RUNNING;
			_poMyStaticRef->_poSrvSync->sendAnimationStatusStatus(animationstatus);
   }

   I_dispvidctrl_tclVideoPlayer* poVideoPlayer = dynamic_cast<I_dispvidctrl_tclVideoPlayer*>(_poMyStaticRef->_cpoMain->getHandler("I_dispvidctrl_tclVideoPlayer"));
   if (poVideoPlayer){
       poVideoPlayer->vStartAnimation(TRUE, _poMyStaticRef->_bRvcWasActive);
   }



}
tVoid dispvidctrl_tclSyncHandler::vExitAnimation(tVoid){
   dispvidctrl_tclAppMain::vWriteStartupLog("[RVC-EARLY]: SyncHandler: vExitAnimation");

   if (_poMyStaticRef->_poSrvSync) {
       syncdisplay_fi_tcl_e8_EarlyVideoLayer_Status eVideoLayer;
       eVideoLayer.enType = syncdisplay_fi_tcl_e8_EarlyVideoLayer_Status::FI_EN_VIDEOLAYER_IDLE;
       _poMyStaticRef->_poSrvSync->sendEarlyVideoLayerStatusStatus(eVideoLayer);
	   //if(m_animationstatus == FALSE)
	  // {
			//syncdisplay_fi_tcl_e8_Animation_Status animationstatus;
			//animationstatus.enType = syncdisplay_fi_tcl_e8_Animation_Status::FI_EN_ANIMATION_FINISHED;
			//_poMyStaticRef->_poSrvSync->sendAnimationStatusStatus(animationstatus);
	 //  }
   }


   I_dispvidctrl_tclVideoPlayer* poVideoPlayer = dynamic_cast<I_dispvidctrl_tclVideoPlayer*>(_poMyStaticRef->_cpoMain->getHandler("I_dispvidctrl_tclVideoPlayer"));
   if (poVideoPlayer){
       poVideoPlayer->vStartAnimation(FALSE);
   }
}
tVoid dispvidctrl_tclSyncHandler::vEntryRvc(tVoid){
   dispvidctrl_tclAppMain::vWriteStartupLog("[RVC-EARLY]: SyncHandler: vEntryRvc");

   if (_poMyStaticRef->_poSrvSync) {
       syncdisplay_fi_tcl_e8_EarlyVideoLayer_Status eVideoLayer;
       eVideoLayer.enType = syncdisplay_fi_tcl_e8_EarlyVideoLayer_Status::FI_EN_VIDEOLAYER_RVC;
       _poMyStaticRef->_poSrvSync->sendEarlyVideoLayerStatusStatus(eVideoLayer);
	   syncdisplay_fi_tcl_e8_Animation_Status animationstatus;
			animationstatus.enType = syncdisplay_fi_tcl_e8_Animation_Status::FI_EN_ANIMATION_RUNNING;
			_poMyStaticRef->_poSrvSync->sendAnimationStatusStatus(animationstatus);
   } else {
       dispvidctrl_tclAppMain::vWriteStartupLog("[RVC-EARLY]: SyncHandler: SyncHandler CCA service currently not ready. Update EarlyVideoLayerStatus later!");
   }

   I_dispvidctrl_tclVideoPlayer* poVideoPlayer = dynamic_cast<I_dispvidctrl_tclVideoPlayer*>(_poMyStaticRef->_cpoMain->getHandler("I_dispvidctrl_tclVideoPlayer"));
   if (poVideoPlayer){
       poVideoPlayer->vActivateRvc(TRUE);
   }

   _poMyStaticRef->_bRvcWasActive = TRUE;
}
tVoid dispvidctrl_tclSyncHandler::vExitRvc(tVoid){
   dispvidctrl_tclAppMain::vWriteStartupLog("[RVC-EARLY]: SyncHandler: vExitRvc");

   if (_poMyStaticRef->_poSrvSync) {
       syncdisplay_fi_tcl_e8_EarlyVideoLayer_Status eVideoLayer;
       eVideoLayer.enType = syncdisplay_fi_tcl_e8_EarlyVideoLayer_Status::FI_EN_VIDEOLAYER_IDLE;
       _poMyStaticRef->_poSrvSync->sendEarlyVideoLayerStatusStatus(eVideoLayer);
   }

   I_dispvidctrl_tclVideoPlayer* poVideoPlayer = dynamic_cast<I_dispvidctrl_tclVideoPlayer*>(_poMyStaticRef->_cpoMain->getHandler("I_dispvidctrl_tclVideoPlayer"));
   if (poVideoPlayer){
       poVideoPlayer->vActivateRvc(FALSE);
   }
}

tVoid dispvidctrl_tclSyncHandler::vWaitVideoChain(tU32 u32Timeout) {
   OSAL_tEventMask hEvRequest  = DISPVIDCTRL_EVENT_VIDEO_REQ;
   OSAL_s32EventWait(_hEvVideoReady, hEvRequest, OSAL_EN_EVENTMASK_OR, u32Timeout, &hEvRequest);
   vTriggerNotify();
}


tVoid dispvidctrl_tclSyncHandler::vTriggerNotify() {
    sd_notifyf(0, "READY=1\n"                       // Tells the init system that daemon startup is finished.
        // This is only used by systemd if the service definition file has Type=notify set.
        "STATUS=Processing requests...\n" // Passes a single-line status string back to the init system that describes the daemon state.
        // This is free-form and can be used for various purposes
        "MAINPID=%lu\nPARENTID=%lu",
        (unsigned long)getpid(),
        (unsigned long)getppid() );
}

