/******************************************************************************/
/**
* \file    dispvidctrl_tclFsm_ReverseSignal.cpp
* \ingroup
*
* \brief   state machine for reverse signal debouncing
*
* \remark  Copyright : (c) 2015 Robert Bosch GmbH, Hildesheim
* \remark  Author    : Michael Niemann CM-AI/PJ-CB32
* \remark  Scope     : A-IVI
*
* \todo
*/
/******************************************************************************/
/*
 * Debouncing happens in two steps/levels
 * 1th level is reached after 50 ms to debounce the PIN
 * -> reverse on/off event will be sent
 *
 * 2nd level is required nissan debouncing time for reverse signal
 * -> reverse debounced event will be sent
 */


#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#include "dispvidctrl_AppMain.h"
#include "dispvidctrl_tclControl_Rvc.h"
#include "dispvidctrl_tclFsm_ReverseSignal.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_APPLICATION
#include "trcGenProj/Header/dispvidctrl_tclFsm_ReverseSignal.cpp.trc.h"
#endif

#define DISPVIDCTRL_RVC_TRACE_ID_FSM_REVERSESIGNAL 0x02

#define ELEMENTE(array)       ((tU16)(sizeof(array)/sizeof(array[0])))

// Default Debouncing Time Values [ms]
#define DISPVIDCTRL_RVC_FSM_REVERSESIGNAL_DEBOUNCE_LEVEL_1_TIME_IN_MS_DEFAULT 50
#define DISPVIDCTRL_RVC_FSM_REVERSESIGNAL_DEBOUNCE_LEVEL_2_TIME_IN_MS_DEFAULT 300

dispvidctrl_tclFsm_ReverseSignal*   dispvidctrl_tclFsm_ReverseSignal::m_poInstance  = OSAL_NULL;

/*******************************************************************************
*                                  STATE TABLE
*******************************************************************************/
dispvidctrl_tclFsm_ReverseSignal::TFsmStateConf dispvidctrl_tclFsm_ReverseSignal::aFsm_ReverseSignal_StateTable[eState_Max] =
{
   {
      eState_OFF,
      1000,                                //u32Timeout
      vEntry_OFF,                          //tFsmStateEntryFunc
      vDo_OFF,                             //tFsmStateDoFunc
      NULL,                                //tFsmStateExitFunc
   },
   {
      eState_Debouncing_Level_1,
      0,                                   //u32Timeout
      vEntry_Debouncing_Level_1,           //tFsmStateEntryFunc
      NULL,                                //tFsmStateDoFunc
      NULL,                                //tFsmStateExitFunc
   },
   {
      eState_ON_Debouncing_Level_2,
      0,                                   //u32Timeout
      vEntry_ON_Debouncing_Level_2,        //tFsmStateEntryFunc
      NULL,                                //tFsmStateDoFunc
      NULL,                                //tFsmStateExitFunc
   },
   {
      eState_ON,
      1000,                                //u32Timeout
      vEntry_ON,                           //tFsmStateEntryFunc
      vDo_ON,                              //tFsmStateDoFunc
      NULL,                                //tFsmStateExitFunc
   }
};

/*******************************************************************************
 *                             STATE TRANSITION TABLE
 *******************************************************************************/
dispvidctrl_tclFsm_ReverseSignal::TStateTransitions dispvidctrl_tclFsm_ReverseSignal::aFsm_ReverseSignal_StateTransitionTable[] =
{
   // eState_OFF
   {  100,                                // transitionId
      eState_OFF,                         // current FSM State
      eTrigger_SignalChange,              // trigger update
      FSM_DONT_CARE,                      // current active trigger(s)
      FSM_DONT_CARE,                      // current not active trigger(s)
      eState_Debouncing_Level_1           // next FSM State
   },
   {  101,                                  // transitionId
      eState_OFF,                         // current FSM State
      eTrigger_StateVerificationTimeout,  // trigger update
      eTrigger_ReverseGpioHigh,           // current active trigger(s)
      FSM_DONT_CARE,                      // current not active trigger(s)
      eState_Debouncing_Level_1           // next FSM State
   },
   {  102,                                  // transitionId
      eState_OFF,                         // current FSM State
      eTrigger_StateVerificationTimeout,  // trigger update
      FSM_DONT_CARE,                      // current active trigger(s)
      eTrigger_ReverseGpioHigh,           // current not active trigger(s)
      eState_OFF                          // next FSM State
   },


   // eState_Debouncing_Level_1
   {  200,                                // transitionId
      eState_Debouncing_Level_1,          // current FSM State
      eTrigger_Level1Timeout,             // trigger update
      eTrigger_ReverseGpioHigh,           // current active trigger(s)
      FSM_DONT_CARE,                      // current not active trigger(s)
      eState_ON_Debouncing_Level_2        // next FSM State
   },
   {  201,                                // transitionId
      eState_Debouncing_Level_1,          // current FSM State
      eTrigger_Level1Timeout,             // trigger update
      FSM_DONT_CARE,                      // current active trigger(s)
      eTrigger_ReverseGpioHigh,           // current not active trigger(s)
      eState_OFF                          // next FSM State
   },


   // eState_ON_Debouncing_Level_2
   {  300,                                // transitionId
      eState_ON_Debouncing_Level_2,       // current FSM State
      eTrigger_Level2Timeout,             // trigger update
      eTrigger_ReverseGpioHigh,           // current active trigger(s)
      FSM_DONT_CARE,                      // current not active trigger(s)
      eState_ON                           // next FSM State
   },
   {  301,                                // transitionId
      eState_ON_Debouncing_Level_2,       // current FSM State
      eTrigger_Level2Timeout,             // trigger update
      FSM_DONT_CARE,                      // current active trigger(s)
      eTrigger_ReverseGpioHigh,           // current not active trigger(s)
      eState_OFF                          // next FSM State
   },
   {  302,                                // transitionId
      eState_ON_Debouncing_Level_2,       // current FSM State
      eTrigger_SignalChange,              // trigger update
      FSM_DONT_CARE,                      // current active trigger(s)
      FSM_DONT_CARE,                      // current not active trigger(s)
      eState_Debouncing_Level_1           // next FSM State
   },


   // eState_ON
   {  400,                                // transitionId
      eState_ON,                          // current FSM State
      eTrigger_StateVerificationTimeout,  // trigger update
      eTrigger_ReverseGpioHigh,           // current active trigger(s)
      FSM_DONT_CARE,                      // current not active trigger(s)
      eState_ON                           // next FSM State
   },
   {  401,                                // transitionId
      eState_ON,                          // current FSM State
      eTrigger_StateVerificationTimeout,  // trigger update
      FSM_DONT_CARE,                      // current active trigger(s)
      eTrigger_ReverseGpioHigh,           // current not active trigger(s)
      eState_Debouncing_Level_1           // next FSM State
   },
   {  402,                                // transitionId
      eState_ON,                          // current FSM State
      eTrigger_SignalChange,              // trigger update
      FSM_DONT_CARE,                      // current active trigger(s)
      FSM_DONT_CARE,                      // current not active trigger(s)
      eState_Debouncing_Level_1           // next FSM State
   },
   {  403,                                // transitionId
      eState_ON,                          // current FSM State
      eTrigger_AskForReverseDebouncedEvent,  // trigger update
      eTrigger_ReverseGpioHigh,           // current active trigger(s)
      FSM_DONT_CARE,                      // current not active trigger(s)
      eState_ON                           // next FSM State
   }
};


/******************************************************************************/
/* FUNCTION     ~dispvidctrl_tclFsm_ReverseSignal                                  */
/******************************************************************************/
/**
*  \brief       Constructor
*
*
*
*  \param       pointer to main application
*               ahl_tclFsmBase( tU32 u32InitFsmSTate,
*                               tU32 u32MaxTransitionEntry,
*                               TStateTransitions* paFsmTransitionTable,
*                               tU32 u32MaxStateEntry,
*                               FsmStateConf* paFsmStateTable,
*                               tU32 u32TraceClass
*                               tU8  u8FsmTraceId);
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclFsm_ReverseSignal::dispvidctrl_tclFsm_ReverseSignal(dispvidctrl_tclControl_Rvc* poControl_Rvc)
: dispvidctrl_tclFsm_Base(
     eState_OFF,
     ELEMENTE(aFsm_ReverseSignal_StateTransitionTable), aFsm_ReverseSignal_StateTransitionTable,
     ELEMENTE(aFsm_ReverseSignal_StateTable)          , aFsm_ReverseSignal_StateTable,
     (tU32) TR_CLASS_DISPVIDCTRL_APPLICATION,
     DISPVIDCTRL_RVC_TRACE_ID_FSM_REVERSESIGNAL)
, m_hTimerHandle_FsmStateTimeout(OSAL_C_INVALID_HANDLE)
, m_hTimerHandle_DebounceLevel1(OSAL_C_INVALID_HANDLE)
, m_hTimerHandle_DebounceLevel2(OSAL_C_INVALID_HANDLE)
, m_poControl_Rvc(poControl_Rvc)
, m_bStateTimeout(FALSE)
, m_u32DebounceLevel1Time(DISPVIDCTRL_RVC_FSM_REVERSESIGNAL_DEBOUNCE_LEVEL_1_TIME_IN_MS_DEFAULT)
, m_u32DebounceLevel2Time(DISPVIDCTRL_RVC_FSM_REVERSESIGNAL_DEBOUNCE_LEVEL_2_TIME_IN_MS_DEFAULT)
{
   // start timer for FSM state timeouts used for state verification
   if( OSAL_s32TimerCreate( (OSAL_tpfCallback)_pfCallbackTimer_FsmStateTimeout, (tPVoid)this, &m_hTimerHandle_FsmStateTimeout) != OSAL_OK)
   {
      ETG_TRACE_USR4(("ERROR - Could not create FSM timer!"));
      m_hTimerHandle_FsmStateTimeout = OSAL_C_INVALID_HANDLE;
   }
   // start timer for debounce level 1 timeout
   if( OSAL_s32TimerCreate( (OSAL_tpfCallback)_pfCallbackTimer_DebounceLevel1, (tPVoid)this, &m_hTimerHandle_DebounceLevel1) != OSAL_OK)
   {
      m_hTimerHandle_DebounceLevel1 = OSAL_C_INVALID_HANDLE;tU32 u32OSALError = OSAL_u32ErrorCode();
      ETG_TRACE_ERR((" FSM-RVS  :: E R R O R in dispvidctrl_tclFsm_Rvs(): could not create DebounceLevel1 timer! 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
   }
   else {
      ETG_TRACE_USR4((" FSM-RVS  :: CallbackTimer_DebounceLevel1 created: m_hTimerHandle_DebounceLevel1 = %u.", m_hTimerHandle_DebounceLevel1));
   }
   // start timer for debounce level 2 timeout
   if( OSAL_s32TimerCreate( (OSAL_tpfCallback)_pfCallbackTimer_DebounceLevel2, (tPVoid)this, &m_hTimerHandle_DebounceLevel2) != OSAL_OK)
   {
      m_hTimerHandle_DebounceLevel2 = OSAL_C_INVALID_HANDLE;tU32 u32OSALError = OSAL_u32ErrorCode();
      ETG_TRACE_ERR((" FSM-RVS  :: E R R O R in dispvidctrl_tclFsm_Rvs(): could not create DebounceLevel2 timer! 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
   }
   else {
      ETG_TRACE_USR4((" FSM-RVS  :: CallbackTimer_DebounceLevel2 created: m_hTimerHandle_DebounceLevel2 = %u.", m_hTimerHandle_DebounceLevel2));
   }
}


/******************************************************************************/
/* FUNCTION     ~dispvidctrl_tclFsm_ReverseSignal                                  */
/******************************************************************************/
/**
*  \brief       destructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclFsm_ReverseSignal::~dispvidctrl_tclFsm_ReverseSignal(tVoid)
{
   // delete timers
   if (OSAL_C_INVALID_HANDLE != m_hTimerHandle_FsmStateTimeout)
   {
      OSAL_s32TimerSetTime(m_hTimerHandle_FsmStateTimeout, 0, 0);
      OSAL_s32TimerDelete(m_hTimerHandle_FsmStateTimeout);
   }

   if (OSAL_C_INVALID_HANDLE != m_hTimerHandle_DebounceLevel1)
   {
      OSAL_s32TimerSetTime(m_hTimerHandle_DebounceLevel1, 0, 0);
      OSAL_s32TimerDelete(m_hTimerHandle_DebounceLevel1);
   }

   if (OSAL_C_INVALID_HANDLE != m_hTimerHandle_DebounceLevel2)
   {
      OSAL_s32TimerSetTime(m_hTimerHandle_DebounceLevel2, 0, 0);
      OSAL_s32TimerDelete(m_hTimerHandle_DebounceLevel2);
   }

   m_poInstance = NULL;
   m_poControl_Rvc = NULL;
}


/******************************************************************************/
/* FUNCTION     tclCreateInstance                                             */
/******************************************************************************/
/**
*  \brief       Create instance (singleton pattern)
*
*  \param       pointer to main application
*  \return      pointer to instance
*/
/******************************************************************************/
dispvidctrl_tclFsm_ReverseSignal* dispvidctrl_tclFsm_ReverseSignal::tclCreateInstance(dispvidctrl_tclControl_Rvc* poControl_Rvc)
{
   if (m_poInstance == NULL)
   {
      m_poInstance = OSAL_NEW dispvidctrl_tclFsm_ReverseSignal(poControl_Rvc);
   }
   return m_poInstance;
}


/******************************************************************************/
/* FUNCTION     vTraceDebugInfo                                               */
/******************************************************************************/
/**
*  \brief       to trace information for debug/analysis purposes
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_ReverseSignal::vTraceDebugInfo(tVoid)
{
   ETG_TRACE_FATAL(("  FSM-RVS :: Debounce Level 1 Time [ms]................. %-40u", m_u32DebounceLevel1Time));
   ETG_TRACE_FATAL(("  FSM-RVS :: Debounce Level 2 Time [ms]................. %-40u", m_u32DebounceLevel2Time));
   // todo: trace current FSM state and active trigger values
   ETG_TRACE_FATAL(("  FSM-RVS :: *********************************************************************************"));
}


/******************************************************************************/
/* FUNCTION                                                                   */
/******************************************************************************/
/**
*  \brief       project specific configuration
*               if not set a default value is used
*
*  \param       configuration value
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_ReverseSignal::vSetDebouncingLevel1TimeInMs(tU32 u32DebouncingLevel1TimeInMs)
{
   m_u32DebounceLevel1Time = u32DebouncingLevel1TimeInMs;
}

tVoid dispvidctrl_tclFsm_ReverseSignal::vSetDebouncingLevel2TimeInMs(tU32 u32DebouncingLevel2TimeInMs)
{
   m_u32DebounceLevel2Time = u32DebouncingLevel2TimeInMs;
}


/******************************************************************************/
/* FUNCTION     vNewTrigger                                                   */
/******************************************************************************/
/**
*  \brief       To be called to enter a new Trigger
*
*  \param       Trigger and State
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_ReverseSignal::vNewTrigger(tU32 u32Trigger, tBool bTriggerState) const
{
   tBool bCurrentState_ReverseGpio = FALSE;

   if (NULL != m_poInstance)
   {
      switch (u32Trigger)
      {
         case eTrigger_SignalChange:
         {
            // because of possible bouncing reading the GPIO values does not make sense for this event
         }
         break;

         default:
         {
            // read GPIO Value
            // get current reverse gear GPIO value
            if (m_poInstance->m_poControl_Rvc)
            {
               bCurrentState_ReverseGpio = m_poInstance->m_poControl_Rvc->bGetReverseState();
            }
            //set/clear eTrigger_ReverseGpioHigh status within FSM
            m_poInstance->vUpdateTrigger(eTrigger_ReverseGpioHigh, bCurrentState_ReverseGpio);
         }
         break;
      }

      m_poInstance->u32CalcNewFsmState(u32Trigger, bTriggerState);
   }
}


/******************************************************************************/
/* FUNCTION     vStateChangeDetected                                          */
/******************************************************************************/
/**
*  \brief       called when a state change happens
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_ReverseSignal::vStateChangeDetected(tVoid)
{
   if (NULL != m_poInstance)
   {
      m_poInstance->vTraceStatusInfo(0);
   }

   return;
}


/******************************************************************************/
/* FUNCTION     _pfCallbackTimer_FsmStateTimeout                              */
/******************************************************************************/
/**
*  \brief
*
*  \param       pArg - pointer to this class
*  \return      none
*/
/******************************************************************************/
OSAL_tpfCallback dispvidctrl_tclFsm_ReverseSignal::_pfCallbackTimer_FsmStateTimeout(tVoid* pArg)
{
   (tVoid) pArg;

   //ETG_TRACE_USR4(("  ====>>  : _pfCallbackTimer_FsmStateTimeout()"));

   //eTrigger_StateTimeout
   if ((m_poInstance) && (m_poInstance->m_poControl_Rvc))
   {
      m_poInstance->m_poControl_Rvc->vHandleFsmOutEvent_ReverseSignal(FSM_REVERSESIGNAL_OUT__FSM_REVERSESIGNAL_INPUT_EVENT,
                                                                         (tU32) eTrigger_StateVerificationTimeout);
   }

   return 0;
};


/******************************************************************************/
/* FUNCTION     vStartFsmStateTimer                                           */
/******************************************************************************/
/**
*  \brief       start or stop the timer
*
*  \param       timeout value
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_ReverseSignal::vStartFsmStateTimer(tU32 u32Time)
{
   if(u32Time > 0)
   {
      OSAL_s32TimerSetTime(m_hTimerHandle_FsmStateTimeout, u32Time, 0);      // set Shutdown-Timer active
   }
   else
   {
      OSAL_s32TimerSetTime(m_hTimerHandle_FsmStateTimeout,       0, 0);     // reset Shutdown-Timer
   }
}


/******************************************************************************/
/* FUNCTION     _pfCallbackTimer_DebounceLevel1                               */
/******************************************************************************/
/**
*  \brief
*
*  \param       pArg - pointer to this class
*  \return      none
*/
/******************************************************************************/
OSAL_tpfCallback dispvidctrl_tclFsm_ReverseSignal::_pfCallbackTimer_DebounceLevel1(tVoid* pArg)
{
   (tVoid) pArg;

   if ((m_poInstance) && (m_poInstance->m_poControl_Rvc)  )
   {
      m_poInstance->m_poControl_Rvc->vHandleFsmOutEvent_ReverseSignal(FSM_REVERSESIGNAL_OUT__FSM_REVERSESIGNAL_INPUT_EVENT,
                                                                                    (tU32) eTrigger_Level1Timeout);
      ETG_TRACE_USR4((" FSM-RVS  :: _pfCallbackTimer_DebounceLevel1(): called! m_hTimerHandle_DebounceLevel1 = %u",  m_poInstance->m_hTimerHandle_DebounceLevel1));
   }

   return 0;
};


/******************************************************************************/
/* FUNCTION     _bSetTimer_DebounceLevel1                                     */
/******************************************************************************/
/**
*  \brief       start or stop the timer
*
*  \param       timeout value
*  \return      tBool - timer start success state
*/
/******************************************************************************/
tBool  dispvidctrl_tclFsm_ReverseSignal::_bSetTimer_DebounceLevel1( OSAL_tMSecond msTimeout ) const

{
   tBool bSuccess = TRUE;
   if ( OSAL_C_INVALID_HANDLE != m_hTimerHandle_DebounceLevel1 )
   {
      if ( OSAL_s32TimerSetTime( m_hTimerHandle_DebounceLevel1, msTimeout, 0 ) != OSAL_OK )
      {
         bSuccess = FALSE;
         tU32 u32OSALError = OSAL_u32ErrorCode();
         ETG_TRACE_ERR((" FSM-RVS  :: E R R O R in _bSetTimer_DebounceLevel1(): timer set to %u ms failed! m_hTimerHandle_DebounceLevel1 = %u, 'Osal Error = %u', %s", msTimeout, m_hTimerHandle_DebounceLevel1, u32OSALError, OSAL_coszErrorText(u32OSALError)));
      }
      else if ( msTimeout != 0 ) {
         ETG_TRACE_USR4((" FSM-RVS  :: _bSetTimer_DebounceLevel1(): timer started! m_hTimerHandle_DebounceLevel1 = %u", m_hTimerHandle_DebounceLevel1));
      }
      else {
         ETG_TRACE_USR4((" FSM-RVS  :: _bSetTimer_DebounceLevel1(): timer stopped! m_hTimerHandle_DebounceLevel1 = %u", m_hTimerHandle_DebounceLevel1));
      }
   }
   return(bSuccess);
}


/******************************************************************************/
/* FUNCTION     _pfCallbackTimer_DebounceLevel2                               */
/******************************************************************************/
/**
*  \brief
*
*  \param       pArg - pointer to this class
*  \return      none
*/
/******************************************************************************/
OSAL_tpfCallback dispvidctrl_tclFsm_ReverseSignal::_pfCallbackTimer_DebounceLevel2(tVoid* pArg)
{
   (tVoid) pArg;

   if ((m_poInstance) && (m_poInstance->m_poControl_Rvc)  )
   {
      m_poInstance->m_poControl_Rvc->vHandleFsmOutEvent_ReverseSignal(FSM_REVERSESIGNAL_OUT__FSM_REVERSESIGNAL_INPUT_EVENT,
                                                                                    (tU32) eTrigger_Level2Timeout);
      ETG_TRACE_USR4((" FSM-RVS  :: _pfCallbackTimer_DebounceLevel2(): called! m_hTimerHandle_DebounceLevel2 = %u",  m_poInstance->m_hTimerHandle_DebounceLevel2));
   }

   return 0;
};


/******************************************************************************/
/* FUNCTION     _bSetTimer_DebounceLevel2                                     */
/******************************************************************************/
/**
*  \brief       start or stop the timer
*
*  \param       timeout value
*  \return      tBool - timer start success state
*/
/******************************************************************************/
tBool  dispvidctrl_tclFsm_ReverseSignal::_bSetTimer_DebounceLevel2( OSAL_tMSecond msTimeout ) const
{
   tBool bSuccess = TRUE;
   if ( OSAL_C_INVALID_HANDLE != m_hTimerHandle_DebounceLevel2 )
   {
      if ( OSAL_s32TimerSetTime( m_hTimerHandle_DebounceLevel2, msTimeout, 0 ) != OSAL_OK )
      {
         bSuccess = FALSE;
         tU32 u32OSALError = OSAL_u32ErrorCode();
         ETG_TRACE_ERR((" FSM-RVS  :: E R R O R in _bSetTimer_DebounceLevel2(): timer set to %u ms failed! m_hTimerHandle_DebounceLevel2 = %u, 'Osal Error = %u', %s", msTimeout, m_hTimerHandle_DebounceLevel2, u32OSALError, OSAL_coszErrorText(u32OSALError)));
      }
      else if ( msTimeout != 0 ){
         ETG_TRACE_USR4((" FSM-RVS  :: _bSetTimer_DebounceLevel2(): timer started! m_hTimerHandle_DebounceLevel2 = %u", m_hTimerHandle_DebounceLevel2));
      }
      else {
         ETG_TRACE_USR4((" FSM-RVS  :: _bSetTimer_DebounceLevel2(): timer stopped! m_hTimerHandle_DebounceLevel2 = %u", m_hTimerHandle_DebounceLevel2));
      }
   }
   return(bSuccess);
}


/******************************************************************************/
/* FUNCTION     vEntry_OFF                                                    */
/******************************************************************************/
/**
*  \brief       post event "Reverse OFF"
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_ReverseSignal::vEntry_OFF(tVoid)
{
   // post event Reverse OFF

   if (NULL != m_poInstance)
   {
      if (m_poInstance->m_poControl_Rvc)
      {
         m_poInstance->m_poControl_Rvc->vHandleFsmOutEvent_ReverseSignal(FSM_REVERSESIGNAL_OUT__REVERSE_OFF);
      }
   }
}

/******************************************************************************/
/* FUNCTION     vDo_OFF                                                       */
/******************************************************************************/
/**
*  \brief       restart state verification timer
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_ReverseSignal::vDo_OFF(tVoid)
{
// trigger(s):
// eTrigger_StateVerificationTimeout[eTrigger_ReverseGpioHigh = FALSE]


   // no actions for self transition triggers from state OFF to state OFF
//todo  // check why we do not restart the timer any more, I think this was concept in the past
}

/******************************************************************************/
/* FUNCTION     vEntry_Debouncing_Level_1                                     */
/******************************************************************************/
/**
*  \brief       start level 1 timer
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_ReverseSignal::vEntry_Debouncing_Level_1(tVoid)
{
   //start level 1 timer
   if (NULL != m_poInstance)
   {
      (tVoid) m_poInstance->_bSetTimer_DebounceLevel1(m_poInstance->m_u32DebounceLevel1Time);
   }
}


/******************************************************************************/
/* FUNCTION     vEntry_ON_Debouncing_Level_2                                  */
/******************************************************************************/
/**
*  \brief       post event "Reverse ON" and
*               start level 2 timer
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_ReverseSignal::vEntry_ON_Debouncing_Level_2(tVoid)
{
   // post event Reverse ON and
   // start level 2 timer
   if (NULL != m_poInstance)
   {
      if (m_poInstance->m_poControl_Rvc)
      {
         m_poInstance->m_poControl_Rvc->vHandleFsmOutEvent_ReverseSignal(FSM_REVERSESIGNAL_OUT__REVERSE_ON);
      }

      (tVoid) m_poInstance->_bSetTimer_DebounceLevel2(m_poInstance->m_u32DebounceLevel2Time);
   }
}


/******************************************************************************/
/* FUNCTION     vExit_ON_Debouncing_Level_2                                   */
/******************************************************************************/
/**
*  \brief       stop level 2 timer
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_ReverseSignal::vExit_ON_Debouncing_Level_2(tVoid)
{
   // stop level 2 timer
   if (NULL != m_poInstance)
   {
      (tVoid) m_poInstance->_bSetTimer_DebounceLevel2(0);
   }
}


/******************************************************************************/
/* FUNCTION     vEntry_ON                                                     */
/******************************************************************************/
/**
*  \brief       post event "Reverse ON Debounced"
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_ReverseSignal::vEntry_ON(tVoid)
{
   tU32 u32CurrentTrigger;

   // post event Reverse ON Debounced

   if (NULL != m_poInstance)
   {
      u32CurrentTrigger = m_poInstance->u32GetCurrentTrigger();
      switch (u32CurrentTrigger)
      {
         // no actions for self transition triggers from state ON to state ON
         case dispvidctrl_tclFsm_ReverseSignal::eTrigger_StateVerificationTimeout:
         {
         }
         break;

         // entry transitions
         default:
         {
            if ((m_poInstance) && (m_poInstance->m_poControl_Rvc))
            {
               m_poInstance->m_poControl_Rvc->vHandleFsmOutEvent_ReverseSignal(FSM_REVERSESIGNAL_OUT__REVERSE_ON_DEBOUNCED);
            }
         }
         break;
      }
   }
}

/******************************************************************************/
/* FUNCTION     vDo_ON                                                        */
/******************************************************************************/
/**
*  \brief       post event "Reverse ON Debounced"
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_ReverseSignal::vDo_ON(tVoid)
{
// trigger(s):
// eTrigger_StateVerificationTimeout[eTrigger_ReverseGpioHigh = TRUE]
// eTrigger_AskForReverseDebouncedEvent[eTrigger_ReverseGpioHigh = TRUE]
   tU32 u32CurrentTrigger;

   // post event Reverse ON Debounced

   if (NULL != m_poInstance)
   {
      u32CurrentTrigger = m_poInstance->u32GetCurrentTrigger();
      switch (u32CurrentTrigger)
      {
         // no actions for self transition triggers from state ON to state ON
         case dispvidctrl_tclFsm_ReverseSignal::eTrigger_StateVerificationTimeout:
         {
//todo  // check why we do not restart the timer any more, I think this was concept in the past
         }
         break;
         case dispvidctrl_tclFsm_ReverseSignal::eTrigger_AskForReverseDebouncedEvent:
         default:
         {
            if ((m_poInstance) && (m_poInstance->m_poControl_Rvc))
            {
               m_poInstance->m_poControl_Rvc->vHandleFsmOutEvent_ReverseSignal(FSM_REVERSESIGNAL_OUT__REVERSE_ON_DEBOUNCED);
            }
         }
         break;
      }
   }
}

/*******************************************************************************
*                                       E O F
*******************************************************************************/
