/******************************************************************************/
/**
* \file    dispvidctrl_tclFsm_CameraSupply.cpp
* \ingroup
*
* \brief   state machine for camera supply
*
* \remark  Copyright : (c) 2015 Robert Bosch GmbH, Hildesheim
* \remark  Author    : Michael Niemann CM-AI/PJ-CB32
* \remark  Scope     : A-IVI
*
* \todo
*/
/******************************************************************************/
/*
 *
 */


#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#include "dispvidctrl_AppMain.h"
#include "dispvidctrl_tclControl_Rvc.h"
#include "dispvidctrl_tclFsm_CameraSupply.h"
#include "dispvidctrl_tclFsm_Hmi.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_APPLICATION
#include "trcGenProj/Header/dispvidctrl_tclFsm_CameraSupply.cpp.trc.h"
#endif

#define ELEMENTE(array)       ((tU16)(sizeof(array)/sizeof(array[0])))

#define DISPVIDCTRL_RVC_FSM_CAMERASUPPLY_CAMERABOOT_TIME_IN_MS               300
#define DISPVIDCTRL_RVC_FSM_CAMERASUPPLY_CAMERAFASTBOOT_TIME_IN_MS            20
#define DISPVIDCTRL_RVC_FSM_CAMERASUPPLY_CAMERAOFFDELAY_TIME_NISSAN_IN_MS   17000
#define DISPVIDCTRL_RVC_FSM_CAMERASUPPLY_CAMERAOFFDELAY_TIME_RENAULT_IN_MS 60000
#define DISPVIDCTRL_RVC_FSM_CAMERASUPPLY_CAMERADISABLE_TIME_IN_MS            200
#define DISPVIDCTRL_RVC_FSM_CAMERASUPPLY_VIDEOHOLDING_TIME_NISSAN_IN_MS       20
#define DISPVIDCTRL_RVC_FSM_CAMERASUPPLY_VIDEOHOLDING_TIME_RENAULT_IN_MS    5000

#define DISPVIDCTRL_TRACE_ID_FSM_CAMERASUPPLY 0x03

dispvidctrl_tclFsm_CameraSupply*   dispvidctrl_tclFsm_CameraSupply::m_poInstance  = OSAL_NULL;

/*******************************************************************************
*                                  STATE TABLE
*******************************************************************************/
dispvidctrl_tclFsm_CameraSupply::TFsmStateConf dispvidctrl_tclFsm_CameraSupply::aFsm_CameraSupply_StateTable[eState_Max] =
{
      {
            eState_CameraOff,
            0,                                            //u32Timeout
            vEntry_CameraOff,                             //tFsmStateEntryFunc
            vDo_CameraOff,                                //tFsmStateDoFunc
            vExit_CameraOff,                              //tFsmStateExitFunc
      },
      {
            eState_CameraBoot_ReverseRequest,
            0,                                            //u32Timeout
            vEntry_CameraBoot,                            //tFsmStateEntryFunc
            NULL,                                         //tFsmStateDoFunc
            vExit_CameraBoot,                             //tFsmStateExitFunc
      },
      {
            eState_CameraBoot_SettingsRequest,
            0,                                            //u32Timeout
            vEntry_CameraBoot,                            //tFsmStateEntryFunc
            NULL,                                         //tFsmStateDoFunc
            vExit_CameraBoot,                             //tFsmStateExitFunc
      },
      {
            eState_CameraOn_ReverseRequest,
            0,                                            //u32Timeout
            vEntry_CameraOn_ReverseRequest,               //tFsmStateEntryFunc
            vDo_CameraOn_ReverseRequest,                  //tFsmStateDoFunc
            vExit_CameraOn,                               //tFsmStateExitFunc
      },
      {
            eState_CameraOn_SettingsRequest,
            0,                                            //u32Timeout
            vEntry_CameraOn_SettingsRequest,              //tFsmStateEntryFunc
            NULL,                                         //tFsmStateDoFunc
            vExit_CameraOn,                               //tFsmStateExitFunc
      },
      {
            eState_CameraOn_WaitForOffRequest,
            0,                                            //u32Timeout
            vEntry_CameraOn_WaitForOffRequest,            //tFsmStateEntryFunc
            vDo_CameraOn_WaitForOffRequest,               //tFsmStateDoFunc
            vExit_CameraOn_WaitForOffRequest,             //tFsmStateExitFunc
      }
};

/*******************************************************************************
 *                             STATE TRANSITION TABLE
 *******************************************************************************/
dispvidctrl_tclFsm_CameraSupply::TStateTransitions dispvidctrl_tclFsm_CameraSupply::aFsm_CameraSupply_StateTransitionTable[] =
{
      // ==== eState_CameraOff ================
      //  ---- transition actions -------------
      {   1,                                    // transitionId
            eState_CameraOff,                   // current FSM State
            eTrigger_ReverseOn,                 // trigger update
            FSM_DONT_CARE,                      // current active trigger(s)
            eTrigger_TailgateOpen,              // current not active trigger(s)
            eState_CameraBoot_ReverseRequest    // next FSM State
      },
      {   2,                                    // transitionId
            eState_CameraOff,                   // current FSM State
            eTrigger_ReverseDebounced,          // trigger update
            FSM_DONT_CARE,                      // current active trigger(s)
            eTrigger_TailgateOpen,              // current not active trigger(s)
            eState_CameraBoot_ReverseRequest    // next FSM State
      },
      {   3,                                    // transitionId
            eState_CameraOff,                   // current FSM State
            eTrigger_SettingsOn,                // trigger update
            FSM_DONT_CARE,                      // current active trigger(s)
            eTrigger_TailgateOpen,              // current not active trigger(s)
            eState_CameraBoot_SettingsRequest   // next FSM State
      },
      {   4,                                    // transitionId
            eState_CameraOff,                   // current FSM State
            eTrigger_DirectOnReverseReq,        // trigger update
            FSM_DONT_CARE,                      // current active trigger(s)
            FSM_DONT_CARE,                      // current not active trigger(s)
            eState_CameraOn_ReverseRequest      // next FSM State
      },
      {   5,                                    // transitionId
            eState_CameraOff,                   // current FSM State
            eTrigger_IgnitionOn,                // trigger update
            eTrigger_ReverseOn,                 // current active trigger(s)
            eTrigger_TailgateOpen,              // current not active trigger(s)
            eState_CameraBoot_ReverseRequest    // next FSM State
      },
      {   6,                                    // transitionId
            eState_CameraOff,                   // current FSM State
            eTrigger_TailgateClose,             // trigger update
            eTrigger_ReverseOn
            | eTrigger_IgnitionOn,              // current active trigger(s)
            FSM_DONT_CARE,                      // current not active trigger(s)
            eState_CameraBoot_ReverseRequest    // next FSM State
      },
      //  ---- do actions----------------------
      {   30,                                   // transitionId
            eState_CameraOff,                   // current FSM State
            eTrigger_CameraSupplyOff,           // trigger update
            FSM_DONT_CARE,                      // current active trigger(s)
            FSM_DONT_CARE,                      // current not active trigger(s)
            eState_CameraOff                    // next FSM State
      },

      // ==== eState_CameraBoot_ReverseRequest
      //  ---- transition actions -------------
      {   100,                                  // transitionId
            eState_CameraBoot_ReverseRequest,   // current FSM State
            eTrigger_ReverseOff,                // trigger update
            FSM_DONT_CARE,                      // current active trigger(s)
            FSM_DONT_CARE,                      // current not active trigger(s)
            eState_CameraOff                    // next FSM State
      },
      {   101,                                  // transitionId
            eState_CameraBoot_ReverseRequest,   // current FSM State
            eTrigger_IgnitionOff,               // trigger update
            FSM_DONT_CARE,                      // current active trigger(s)
            FSM_DONT_CARE,                      // current not active trigger(s)
            eState_CameraOff                    // next FSM State
      },
      {   102,                                  // transitionId
            eState_CameraBoot_ReverseRequest,   // current FSM State
            eTrigger_CameraBootTimeout,         // trigger update
            eTrigger_ReverseDebounced,         // | eTrigger_IgnitionOn), // current active trigger(s)
            FSM_DONT_CARE,                      // current not active trigger(s)
            eState_CameraOn_ReverseRequest      // next FSM State
      },
      {   103,                                  // transitionId
            eState_CameraBoot_ReverseRequest,   // current FSM State
            eTrigger_ReverseDebounced,          // trigger update
            eTrigger_CameraBootTimeout,            //| eTrigger_IgnitionOn,              // current active trigger(s)
            FSM_DONT_CARE,                      // current not active trigger(s)
            eState_CameraOn_ReverseRequest      // next FSM State
      },
      {   104,                                  // transitionId
            eState_CameraBoot_ReverseRequest,   // current FSM State
            eTrigger_IgnitionOn,                // trigger update
            eTrigger_CameraBootTimeout 
            | eTrigger_ReverseDebounced,        // current active trigger(s)
            FSM_DONT_CARE,                      // current not active trigger(s)
            eState_CameraOn_ReverseRequest      // next FSM State
      },

      // ==== eState_CameraBoot_SettingsRequest
      //  ---- transition actions -------------
      {   200,                                  // transitionId
            eState_CameraBoot_SettingsRequest,  // current FSM State
            eTrigger_ServiceRequestOff,         // trigger update
            FSM_DONT_CARE,                      // current active trigger(s)
            FSM_DONT_CARE,                      // current not active trigger(s)
            eState_CameraOff                    // next FSM State
      },
      {   201,                                  // transitionId
            eState_CameraBoot_SettingsRequest,  // current FSM State
            eTrigger_CameraBootTimeout,         // trigger update
            FSM_DONT_CARE,                      // current active trigger(s)
            eTrigger_ReverseDebounced,          // current not active trigger(s)
            eState_CameraOn_SettingsRequest     // next FSM State
      },
      {   202,                                  // transitionId
            eState_CameraBoot_SettingsRequest,  // current FSM State
            eTrigger_CameraBootTimeout,         // trigger update
            eTrigger_ReverseDebounced 
            | eTrigger_IgnitionOn,              // current active trigger(s)
            FSM_DONT_CARE,                      // current not active trigger(s)
            eState_CameraOn_ReverseRequest      // next FSM State
      },
      {   203,                                  // transitionId
            eState_CameraBoot_SettingsRequest,  // current FSM State
            eTrigger_CameraBootTimeout,         // trigger update
            eTrigger_ReverseDebounced 
            | eTrigger_IgnitionOff,             // current active trigger(s)
            FSM_DONT_CARE,                      // current not active trigger(s)
            eState_CameraOn_SettingsRequest     // next FSM State
      },

      // ==== eState_CameraOn_ReverseRequest ==
      //  ---- transition actions -------------
      {   300,                                  // transitionId
            eState_CameraOn_ReverseRequest,     // current FSM State
            eTrigger_CameraOff,                 // trigger update
            FSM_DONT_CARE,                      // current active trigger(s)
            FSM_DONT_CARE,                      // current not active trigger(s)
            eState_CameraOff                    // next FSM State
      },
      {   301,                                  // transitionId
            eState_CameraOn_ReverseRequest,     // current FSM State
            eTrigger_VideoHoldingTimeout,       // trigger update
            FSM_DONT_CARE,                      // current active trigger(s)
            eTrigger_ReverseDebounced,          // current not active trigger(s)
            eState_CameraOn_WaitForOffRequest   // next FSM State
      },
      {   302,                                  // transitionId
            eState_CameraOn_ReverseRequest,     // current FSM State
            eTrigger_ServiceRequestOff,         // trigger update
            FSM_DONT_CARE,                      // current active trigger(s)
            FSM_DONT_CARE,                      // current not active trigger(s)
            eState_CameraOn_WaitForOffRequest   // next FSM State
      },
      {   303,                                  // transitionId
            eState_CameraOn_ReverseRequest,     // current FSM State
            eTrigger_IgnitionOff,               // trigger update
            FSM_DONT_CARE,                      // current active trigger(s)
            FSM_DONT_CARE,                      // current not active trigger(s)
            eState_CameraOn_WaitForOffRequest   // next FSM State
      },
      {   304,                                  // transitionId
            eState_CameraOn_ReverseRequest,     // current FSM State
            eTrigger_SettingsOn,                // trigger update
            FSM_DONT_CARE,                      // current active trigger(s)
            FSM_DONT_CARE,                      // current not active trigger(s)
            eState_CameraOn_SettingsRequest     // next FSM State
      },
      {   305,                                  // transitionId
            eState_CameraOn_ReverseRequest,     // current FSM State
            eTrigger_TailgateOpen,              // trigger update
            FSM_DONT_CARE,                      // current active trigger(s)
            FSM_DONT_CARE,                      // current not active trigger(s)
            eState_CameraOff                    // next FSM State
      },
      //  ---- do actions----------------------
      {   330,                                  // transitionId
            eState_CameraOn_ReverseRequest,     // current FSM State
            eTrigger_ReverseOff,                // trigger update
            FSM_DONT_CARE,                      // current active trigger(s)
            FSM_DONT_CARE,                      // current not active trigger(s)
            eState_CameraOff      // next FSM State
      },
      {   331,                                  // transitionId
            eState_CameraOn_ReverseRequest,     // current FSM State
            eTrigger_ReverseDebounced,          // trigger update
            FSM_DONT_CARE,                      // current active trigger(s)
            FSM_DONT_CARE,                      // current not active trigger(s)
            eState_CameraOn_ReverseRequest      // next FSM State
      },

      // ==== eState_CameraOn_SettingsRequest =
      //  ---- transition actions -------------
      {   400,                                  // transitionId
            eState_CameraOn_SettingsRequest,    // current FSM State
            eTrigger_CameraOff,                 // trigger update
            FSM_DONT_CARE,                      // current active trigger(s)
            FSM_DONT_CARE,                      // current not active trigger(s)
            eState_CameraOff                    // next FSM State
      },
      {   401,                                  // transitionId
            eState_CameraOn_SettingsRequest,    // current FSM State
            eTrigger_ServiceRequestOff,         // trigger update
            FSM_DONT_CARE,                      // current active trigger(s)
            FSM_DONT_CARE,                      // current not active trigger(s)
            eState_CameraOn_WaitForOffRequest   // next FSM State
      },
      {   402,                                  // transitionId
            eState_CameraOn_SettingsRequest,    // current FSM State
            eTrigger_ReverseDebounced,          // trigger update
            eTrigger_IgnitionOn,                // current active trigger(s)
            FSM_DONT_CARE,                      // current not active trigger(s)
            eState_CameraOn_ReverseRequest      // next FSM State
      },
      {   403,                                  // transitionId
            eState_CameraOn_SettingsRequest,    // current FSM State
            eTrigger_IgnitionOn,                // trigger update
            eTrigger_ReverseDebounced,          // current active trigger(s)
            FSM_DONT_CARE,                      // current not active trigger(s)
            eState_CameraOn_ReverseRequest      // next FSM State
      },
      {   404,                                  // transitionId
            eState_CameraOn_SettingsRequest,    // current FSM State
            eTrigger_SettingsOff,               // trigger update
            eTrigger_ReverseDebounced |
            eTrigger_IgnitionOn,                // current active trigger(s)
            FSM_DONT_CARE,                      // current not active trigger(s)
            eState_CameraOn_ReverseRequest      // next FSM State
      },
      {   405,                                  // transitionId
            eState_CameraOn_SettingsRequest,    // current FSM State
            eTrigger_SettingsOff,               // trigger update
            eTrigger_ReverseOff,                // current active trigger(s)
            FSM_DONT_CARE,                      // current not active trigger(s)
            eState_CameraOn_WaitForOffRequest   // next FSM State
      },

      // ==== eState_CameraOn_WaitForOffRequest
      //  ---- transition actions -------------
      {   500,                                  // transitionId
            eState_CameraOn_WaitForOffRequest,  // current FSM State
            eTrigger_ReverseOn,                 // trigger update
            FSM_DONT_CARE,                      // current active trigger(s)
            eTrigger_TailgateOpen,              // current not active trigger(s)
            eState_CameraBoot_ReverseRequest    // next FSM State
      },
      {   501,                                  // transitionId
            eState_CameraOn_WaitForOffRequest,  // current FSM State
            eTrigger_ReverseDebounced,          // trigger update
            FSM_DONT_CARE,                      // current active trigger(s)
            eTrigger_TailgateOpen,              // current not active trigger(s)
            eState_CameraBoot_ReverseRequest    // next FSM State
      },
      {   502,                                  // transitionId
            eState_CameraOn_WaitForOffRequest,  // current FSM State
            eTrigger_SettingsOn,                // trigger update
            FSM_DONT_CARE,                      // current active trigger(s)
            eTrigger_TailgateOpen,              // current not active trigger(s)
            eState_CameraBoot_SettingsRequest   // next FSM State
      },
      {   503,                                  // transitionId
            eState_CameraOn_WaitForOffRequest,  // current FSM State
            eTrigger_CameraOffDelayTimeout,     // trigger update
            FSM_DONT_CARE,                      // current active trigger(s)
            FSM_DONT_CARE,                      // current not active trigger(s)
            eState_CameraOff                    // next FSM State
      },
      {   504,                                  // transitionId
            eState_CameraOn_WaitForOffRequest,  // current FSM State
            eTrigger_IgnitionOn,                // trigger update
            eTrigger_ReverseOn,                 // current active trigger(s)
            eTrigger_TailgateOpen,              // current not active trigger(s)
            eState_CameraBoot_ReverseRequest    // next FSM State
      },
      //  ---- do actions----------------------
      {   530,                                  // transitionId
            eState_CameraOn_WaitForOffRequest,  // current FSM State
            eTrigger_CameraOff,                 // trigger update
            FSM_DONT_CARE,                      // current active trigger(s)
            FSM_DONT_CARE,                      // current not active trigger(s)
            eState_CameraOn_WaitForOffRequest   // next FSM State
      },

      // ==== from all states =================
      {     0,                                  // transitionId
            FSM_DONT_CARE,                      // current FSM State
            eTrigger_Disable,                   // trigger update
            FSM_DONT_CARE,                      // current active trigger(s)
            FSM_DONT_CARE,                      // current not active trigger(s)
            eState_CameraOn_WaitForOffRequest   // next FSM State
      }
};


/******************************************************************************/
/* FUNCTION      dispvidctrl_tclFsm_CameraSupply                                   */
/******************************************************************************/
/**
*  \brief       Constructor
*
*
*
*  \param       pointer to main application
*               ahl_tclFsmBase( tU32 u32InitFsmSTate,
*                               tU32 u32MaxTransitionEntry,
*                               TStateTransitions* paFsmTransitionTable,
*                               tU32 u32MaxStateEntry,
*                               FsmStateConf* paFsmStateTable,
*                               tU32 u32TraceClass);
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclFsm_CameraSupply::dispvidctrl_tclFsm_CameraSupply(dispvidctrl_tclControl_Rvc* poControl_Rvc)
: dispvidctrl_tclFsm_Base(
     eState_CameraOff,
     ELEMENTE(aFsm_CameraSupply_StateTransitionTable), aFsm_CameraSupply_StateTransitionTable,
     ELEMENTE(aFsm_CameraSupply_StateTable)          , aFsm_CameraSupply_StateTable,
     (tU32) TR_CLASS_DISPVIDCTRL_APPLICATION,
     DISPVIDCTRL_TRACE_ID_FSM_CAMERASUPPLY)
, m_hTimerHandle_FsmStateTimeout(OSAL_C_INVALID_HANDLE)
, m_hTimerHandle_CameraBoot(OSAL_C_INVALID_HANDLE)
, m_hTimerHandle_CameraOffDelay(OSAL_C_INVALID_HANDLE)
, m_hTimerHandle_VideoHolding(OSAL_C_INVALID_HANDLE)
, m_poControl_Rvc(poControl_Rvc)
, m_bStateTimeout(FALSE)
, m_bRvcPowerHoldingActive(FALSE)
, m_u32CameraPowerHoldingTimeBrandspezificInMs(17000)
, m_u32CameraVideoHoldingTimeBrandspezificInMs(6000)
, m_u8Brand(1)
{
   // create timer for FSM state timeouts used for state verification
   if( OSAL_s32TimerCreate( (OSAL_tpfCallback)_pfCallbackTimer_FsmStateTimeout, (tPVoid)this, &m_hTimerHandle_FsmStateTimeout) != OSAL_OK)
   {
      ETG_TRACE_USR4(("ERROR - Could not create FSM timer!"));
      m_hTimerHandle_FsmStateTimeout = OSAL_C_INVALID_HANDLE;
   }
   // create timer for camera boot timeout
   if( OSAL_s32TimerCreate( (OSAL_tpfCallback)_pfCallbackTimer_CameraBoot, (tPVoid)this, &m_hTimerHandle_CameraBoot) != OSAL_OK)
   {
      ETG_TRACE_USR4(("ERROR - Could not create camera boot timer!"));
      m_hTimerHandle_CameraBoot = OSAL_C_INVALID_HANDLE;
      tU32 u32OSALError = OSAL_u32ErrorCode();
      ETG_TRACE_ERR((" FSM-CAM  :: E R R O R in dispvidctrl_tclFsm_Cam(): couldn't create CameraBoot timer! 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
   }
   else {
      ETG_TRACE_USR4((" FSM-CAM  :: CallbackTimer_CameraBoot created: m_hTimerHandle_CameraBoot = %u.", m_hTimerHandle_CameraBoot));
   }
   // create timer for camera off delay
   if( OSAL_s32TimerCreate( (OSAL_tpfCallback)_pfCallbackTimer_CameraOffDelay, (tPVoid)this, &m_hTimerHandle_CameraOffDelay) != OSAL_OK)
   {
      ETG_TRACE_USR4(("ERROR - Could not create camera off delay timer!"));
      m_hTimerHandle_CameraOffDelay = OSAL_C_INVALID_HANDLE;
   }
   // create timer for video holding timeout
   if( OSAL_s32TimerCreate( (OSAL_tpfCallback)_pfCallbackTimer_VideoHolding, (tPVoid)this, &m_hTimerHandle_VideoHolding) != OSAL_OK)
   {
      ETG_TRACE_USR4(("ERROR - Could not create video holding timer!"));
      m_hTimerHandle_VideoHolding = OSAL_C_INVALID_HANDLE;
      tU32 u32OSALError = OSAL_u32ErrorCode();
      ETG_TRACE_ERR((" FSM-CAM  :: E R R O R in dispvidctrl_tclFsm_Cam(): couldn't create VideoHolding timer! 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
   }
}


/******************************************************************************/
/* FUNCTION     ~dispvidctrl_tclFsm_ReverseSignal                                  */
/******************************************************************************/
/**
*  \brief       destructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclFsm_CameraSupply::~dispvidctrl_tclFsm_CameraSupply(tVoid)
{
   // delete timers
   if (OSAL_C_INVALID_HANDLE != m_hTimerHandle_FsmStateTimeout)
   {
      OSAL_s32TimerSetTime(m_hTimerHandle_FsmStateTimeout, 0, 0);
      OSAL_s32TimerDelete(m_hTimerHandle_FsmStateTimeout);
   }
   if (OSAL_C_INVALID_HANDLE != m_hTimerHandle_CameraBoot)
   {
      OSAL_s32TimerSetTime(m_hTimerHandle_CameraBoot, 0, 0);
      OSAL_s32TimerDelete(m_hTimerHandle_CameraBoot);
   }
   if (OSAL_C_INVALID_HANDLE != m_hTimerHandle_CameraOffDelay)
   {
      OSAL_s32TimerSetTime(m_hTimerHandle_CameraOffDelay, 0, 0);
      OSAL_s32TimerDelete(m_hTimerHandle_CameraOffDelay);
   }
   if (OSAL_C_INVALID_HANDLE != m_hTimerHandle_VideoHolding)
   {
      OSAL_s32TimerSetTime(m_hTimerHandle_VideoHolding, 0, 0);
      OSAL_s32TimerDelete(m_hTimerHandle_VideoHolding);
   }

   m_poInstance = NULL;
   m_poControl_Rvc = NULL;
}


/******************************************************************************/
/* FUNCTION     tclCreateInstance                                             */
/******************************************************************************/
/**
*  \brief       Create instance (singleton pattern)
*
*  \param       pointer to main application
*  \return      pointer to instance
*/
/******************************************************************************/
dispvidctrl_tclFsm_CameraSupply* dispvidctrl_tclFsm_CameraSupply::tclCreateInstance(dispvidctrl_tclControl_Rvc* poControl_Rvc)
{
   if (m_poInstance == NULL)
   {
      m_poInstance = OSAL_NEW dispvidctrl_tclFsm_CameraSupply(poControl_Rvc);
   }
   return m_poInstance;
}

/******************************************************************************/
/* FUNCTION     vConfigure                                                    */
/******************************************************************************/
/**
*  \brief       Called from Control to set configuration values
*
*  \param       pointer to configuration data structure
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_CameraSupply::vConfigure(const TConfiguration_Fsm_CameraSupply* pStConfiguration_Fsm_CameraSupply)
{
   m_u8Brand  = pStConfiguration_Fsm_CameraSupply->u8Brand;
   switch (m_u8Brand)
   {
      // Brand Renault
      case 0x0:
         {  m_u32CameraPowerHoldingTimeBrandspezificInMs = DISPVIDCTRL_RVC_FSM_CAMERASUPPLY_CAMERAOFFDELAY_TIME_RENAULT_IN_MS;
            m_u32CameraVideoHoldingTimeBrandspezificInMs = DISPVIDCTRL_RVC_FSM_CAMERASUPPLY_VIDEOHOLDING_TIME_RENAULT_IN_MS;
         } break;
      //Brand Nissan
      case 0x1:
      case 0x2:
         {  m_u32CameraPowerHoldingTimeBrandspezificInMs = DISPVIDCTRL_RVC_FSM_CAMERASUPPLY_CAMERAOFFDELAY_TIME_NISSAN_IN_MS;
            m_u32CameraVideoHoldingTimeBrandspezificInMs = DISPVIDCTRL_RVC_FSM_CAMERASUPPLY_VIDEOHOLDING_TIME_NISSAN_IN_MS;
         } break;
      default:
         break;
   }
}

/******************************************************************************/
/* FUNCTION     vUpdateRelatedTrigger                                         */
/******************************************************************************/
/**
*  \brief       to correct "hold" trigger states which are related to actual
*               trigger,
*               overwritten base class function
*
*  \param       actual Trigger
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_CameraSupply::vUpdateRelatedTrigger(tU32 u32Trigger)
{
   switch (u32Trigger)
   {
      case dispvidctrl_tclFsm_CameraSupply::eTrigger_IgnitionOff:
         vUpdateTrigger(eTrigger_IgnitionOn, FALSE);
         break;
      case dispvidctrl_tclFsm_CameraSupply::eTrigger_IgnitionOn:
         vUpdateTrigger(eTrigger_IgnitionOff, FALSE);
         break;
      case dispvidctrl_tclFsm_CameraSupply::eTrigger_ReverseOff:
         vUpdateTrigger(eTrigger_ReverseOn, FALSE);
         vUpdateTrigger(eTrigger_ReverseDebounced, FALSE);
         break;
      case dispvidctrl_tclFsm_CameraSupply::eTrigger_ReverseOn:
         vUpdateTrigger(eTrigger_ReverseDebounced, FALSE);
         vUpdateTrigger(eTrigger_ReverseOff, FALSE);
         break;
      case dispvidctrl_tclFsm_CameraSupply::eTrigger_Disable:
         vUpdateTrigger(eTrigger_ReverseOn, FALSE);
         vUpdateTrigger(eTrigger_ReverseDebounced, FALSE);
         vUpdateTrigger(eTrigger_IgnitionOn, FALSE);
         vUpdateTrigger(eTrigger_IgnitionOff, FALSE);
         break;
      case dispvidctrl_tclFsm_CameraSupply::eTrigger_ReverseDebounced:
         vUpdateTrigger(eTrigger_ReverseOff, FALSE);
         break;
      case dispvidctrl_tclFsm_CameraSupply::eTrigger_SettingsOn:
         vUpdateTrigger(eTrigger_SettingsOn, FALSE);
         break;
      case dispvidctrl_tclFsm_CameraSupply::eTrigger_SettingsOff:
         vUpdateTrigger(eTrigger_SettingsOff, FALSE);
         break;
         
      case dispvidctrl_tclFsm_CameraSupply::eTrigger_TailgateOpen:
         vUpdateTrigger(eTrigger_TailgateClose, FALSE);
         break;
      case dispvidctrl_tclFsm_CameraSupply::eTrigger_TailgateClose:
         vUpdateTrigger(eTrigger_TailgateOpen, FALSE);
         break;
         
      default:
         break;
   }
}

/******************************************************************************/
/* FUNCTION     vNewTrigger                                                   */
/******************************************************************************/
/**
*  \brief       To be called to enter a new Trigger
*
*  \param       Trigger and State
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_CameraSupply::vNewTrigger(tU32 u32Trigger, tBool bTriggerState) const
{
   if (NULL != m_poInstance)
   {
      m_poInstance->u32CalcNewFsmState(u32Trigger, bTriggerState);
   }
}

/******************************************************************************/
/* FUNCTION     vUpdateTriggerState                                           */
/******************************************************************************/
/**
*  \brief       To be called to correct Trigger states
*
*  \param       Trigger and State
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_CameraSupply::vUpdateTriggerState(tU32 u32Trigger, tBool bTriggerState)
{
   vUpdateTrigger(u32Trigger, bTriggerState);
}


/******************************************************************************/
/* FUNCTION     vStateChangeDetected                                          */
/******************************************************************************/
/**
*  \brief       called when a state change happens
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_CameraSupply::vStateChangeDetected(tVoid)
{
   if (NULL != m_poInstance)
   {
      m_poInstance->vTraceStatusInfo(0);
   }

   return;
}


/******************************************************************************/
/* FUNCTION     _pfCallbackTimer_FsmStateTimeout                              */
/******************************************************************************/
/**
*  \brief
*
*  \param       pArg - pointer to this class
*  \return      none
*/
/******************************************************************************/
OSAL_tpfCallback dispvidctrl_tclFsm_CameraSupply::_pfCallbackTimer_FsmStateTimeout(tVoid* pArg)
{
   (tVoid) pArg;

   //ETG_TRACE_USR4(("  ====>>  : _pfCallbackTimer_FsmStateTimeout()"));

   //eTrigger_StateTimeout
   if ((m_poInstance) && (m_poInstance->m_poControl_Rvc))
   {
      m_poInstance->m_poControl_Rvc->vHandleFsmOutEvent_CameraSupply(FSM_CAMERASUPPLY_OUT__FSM_CAMERASUPPLY_INPUT_EVENT,
                                                                         (tU32) eTrigger_StateTimeout);
   }

   return 0;
};


/******************************************************************************/
/* FUNCTION     vStartFsmStateTimer                                           */
/******************************************************************************/
/**
*  \brief       start or stop the timer
*
*  \param       timeout value
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_CameraSupply::vStartFsmStateTimer(tU32 u32Time)
{
   if(u32Time > 0)
   {
      OSAL_s32TimerSetTime(m_hTimerHandle_FsmStateTimeout, u32Time, 0);      // set Shutdown-Timer active
   }
   else
   {
      OSAL_s32TimerSetTime(m_hTimerHandle_FsmStateTimeout,       0, 0);     // reset Shutdown-Timer
   }
}


/******************************************************************************/
/* FUNCTION     _pfCallbackTimer_CameraBoot                                   */
/******************************************************************************/
/**
*  \brief
*
*  \param       pArg - pointer to this class
*  \return      none
*/
/******************************************************************************/
OSAL_tpfCallback dispvidctrl_tclFsm_CameraSupply::_pfCallbackTimer_CameraBoot(tVoid* pArg)
{
   (tVoid) pArg;

   if ((m_poInstance) && (m_poInstance->m_poControl_Rvc))
   {
      m_poInstance->m_poControl_Rvc->vHandleFsmOutEvent_CameraSupply(FSM_CAMERASUPPLY_OUT__FSM_CAMERASUPPLY_INPUT_EVENT,
                                                                         (tU32) eTrigger_CameraBootTimeout);
      ETG_TRACE_USR4((" FSM-CAM  :: _pfCallbackTimer_CameraBoot(): called! m_hTimerHandle_CameraBoot = %u",  m_poInstance->m_hTimerHandle_CameraBoot));
   }

   return 0;
};


/******************************************************************************/
/* FUNCTION     _bSetTimer_CameraBoot                                         */
/******************************************************************************/
/**
*  \brief       start or stop the timer
*
*  \param       timeout value
*  \return      tBool - timer start success state
*/
/******************************************************************************/
tBool  dispvidctrl_tclFsm_CameraSupply::_bSetTimer_CameraBoot( OSAL_tMSecond msTimeout )
{
   tBool bSuccess = TRUE;
   if ( OSAL_C_INVALID_HANDLE != m_hTimerHandle_CameraBoot )
   {
      if ( OSAL_s32TimerSetTime( m_hTimerHandle_CameraBoot, msTimeout, 0 ) != OSAL_OK ) {
         bSuccess = FALSE;
         tU32 u32OSALError = OSAL_u32ErrorCode();
         ETG_TRACE_ERR((" FSM-CAM  :: E R R O R in _bSetTimer_CameraBoot(): timer set to %u ms failed! m_hTimerHandle_CameraBoot = %u, 'Osal Error = %u', %s", msTimeout, m_hTimerHandle_CameraBoot, u32OSALError, OSAL_coszErrorText(u32OSALError)));
      }
      else if ( msTimeout != 0 ){
         ETG_TRACE_USR4((" FSM-CAM  :: _bSetTimer_CameraBoot(): timer started! m_hTimerHandle_CameraBoot = %u", m_hTimerHandle_CameraBoot));
      }
      else {
         ETG_TRACE_USR4((" FSM-CAM  :: _bSetTimer_CameraBoot(): timer stopped! m_hTimerHandle_CameraBoot = %u", m_hTimerHandle_CameraBoot));
      }
   }
   return(bSuccess);
}


/******************************************************************************/
/* FUNCTION     _pfCallbackTimer_CameraOffDelay                               */
/******************************************************************************/
/**
*  \brief
*
*  \param       pArg - pointer to this class
*  \return      none
*/
/******************************************************************************/
OSAL_tpfCallback dispvidctrl_tclFsm_CameraSupply::_pfCallbackTimer_CameraOffDelay(tVoid* pArg)
{
   (tVoid) pArg;

   if ((m_poInstance) && (m_poInstance->m_poControl_Rvc))
   {
      m_poInstance->m_poControl_Rvc->vHandleFsmOutEvent_CameraSupply(FSM_CAMERASUPPLY_OUT__FSM_CAMERASUPPLY_INPUT_EVENT,
                                                                         (tU32) eTrigger_CameraOffDelayTimeout);
   }

   return 0;
};


/******************************************************************************/
/* FUNCTION     _bSetTimer_CameraOffDelay                                     */
/******************************************************************************/
/**
*  \brief       start or stop the timer
*
*  \param       timeout value
*  \return      tBool - timer start success state
*/
/******************************************************************************/
tBool  dispvidctrl_tclFsm_CameraSupply::_bSetTimer_CameraOffDelay( OSAL_tMSecond msTimeout )
{
    tBool bSuccess = TRUE;
    if ( OSAL_C_INVALID_HANDLE != m_hTimerHandle_CameraOffDelay )
    {
        if ( OSAL_s32TimerSetTime( m_hTimerHandle_CameraOffDelay, msTimeout, 0 ) != OSAL_OK )
        {
            bSuccess = FALSE;
        }
    }
    return(bSuccess);
}

/******************************************************************************/
/* FUNCTION     _pfCallbackTimer_VideoHolding                                 */
/******************************************************************************/
/**
*  \brief
*
*  \param       pArg - pointer to this class
*  \return      none
*/
/******************************************************************************/
OSAL_tpfCallback dispvidctrl_tclFsm_CameraSupply::_pfCallbackTimer_VideoHolding(tVoid* pArg)
{
   (tVoid) pArg;

   if ((m_poInstance) && (m_poInstance->m_poControl_Rvc))
   {
      m_poInstance->m_poControl_Rvc->vHandleFsmOutEvent_CameraSupply(FSM_CAMERASUPPLY_OUT__FSM_CAMERASUPPLY_INPUT_EVENT,
                                                                         (tU32) eTrigger_VideoHoldingTimeout);
      ETG_TRACE_USR4((" FSM-CAM  :: _pfCallbackTimer_VideoHolding(): called! m_hTimerHandle_VideoHolding = %u",  m_poInstance->m_hTimerHandle_VideoHolding));
   }

   return 0;
};


/******************************************************************************/
/* FUNCTION     _bSetTimer_VideoHolding                                       */
/******************************************************************************/
/**
*  \brief       start or stop the timer
*
*  \param       timeout value
*  \return      tBool - timer start success state
*/
/******************************************************************************/
tBool  dispvidctrl_tclFsm_CameraSupply::_bSetTimer_VideoHolding( OSAL_tMSecond msTimeout )
{
   tBool bSuccess = TRUE;
   if ( OSAL_C_INVALID_HANDLE != m_hTimerHandle_VideoHolding )
   {
      if ( OSAL_s32TimerSetTime( m_hTimerHandle_VideoHolding, msTimeout, 0 ) != OSAL_OK )
      {
         bSuccess = FALSE;
      }
      ETG_TRACE_USR4((" FSM-CAM  :: _bSetTimer_VideoHolding(): timer set m_hTimerHandle_VideoHolding = %u", m_hTimerHandle_VideoHolding));
   }
   return(bSuccess);
}


/******************************************************************************/
/* FUNCTION     vEntry_CameraOff                                              */
/******************************************************************************/
/**
*  \brief       switch camera supply OFF
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_CameraSupply::vEntry_CameraOff(tVoid)
{
   if ((m_poInstance) && (m_poInstance->m_poControl_Rvc))
   {
      m_poInstance->m_poControl_Rvc->vHandleFsmOutEvent_CameraSupply(FSM_CAMERASUPPLY_OUT__SWITCH_CAMERA_VCC_OFF);
      
      //trigger clean up
      m_poInstance->vUpdateTrigger(eTrigger_CameraOff, FALSE);
      m_poInstance->vUpdateTrigger(eTrigger_Disable, FALSE);
      m_poInstance->vUpdateTrigger(eTrigger_CameraBootTimeout, FALSE);
      m_poInstance->vUpdateTrigger(eTrigger_VideoHoldingTimeout, FALSE);
      m_poInstance->vUpdateTrigger(eTrigger_CameraOffDelayTimeout, FALSE);
   }
}

/******************************************************************************/
/* FUNCTION     vDo_CameraSupplyOff                                           */
/******************************************************************************/
/**
*  \brief       switch  camera supply off
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_CameraSupply::vDo_CameraOff(tVoid)
{
// trigger(s):
// eTrigger_CameraSupplyOff

   if (NULL != m_poInstance)
   {
      if (m_poInstance->m_poControl_Rvc)
      {
         m_poInstance->m_poControl_Rvc->vHandleFsmOutEvent_CameraSupply(FSM_CAMERASUPPLY_OUT__SWITCH_CAMERA_VCC_OFF);
      }
   }
}

/******************************************************************************/
/* FUNCTION     vExit_CameraOff                                               */
/******************************************************************************/
/**
*  \brief
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_CameraSupply::vExit_CameraOff(tVoid)
{
   if (NULL != m_poInstance)
   {
      // remember camera holding state
      m_poInstance->m_bRvcPowerHoldingActive = FALSE;
   }
}


/******************************************************************************/
/* FUNCTION     vEntry_CameraBoot                                             */
/******************************************************************************/
/**
*  \brief       switch camera supply ON
*               start camera boot timer
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_CameraSupply::vEntry_CameraBoot(tVoid)
{
   if (NULL != m_poInstance)
   {
      // set GPIO for Camera supply to "on"
      if (m_poInstance->m_poControl_Rvc)
      {
         m_poInstance->m_poControl_Rvc->vHandleFsmOutEvent_CameraSupply(FSM_CAMERASUPPLY_OUT__SWITCH_CAMERA_VCC_ON);
      }

      //// if entering state with reverse debounced trigger do not clean this state
      //if (eTrigger_ReverseDebounced != m_poInstance->u32GetCurrentTrigger())
      //{  //clear old eTrigger_ReverseDebounced status within FSM
      //   m_poInstance->vUpdateTrigger((eTrigger_ReverseDebounced), FALSE);
      //}

      //start camera boot timer
      if (m_poInstance->m_bRvcPowerHoldingActive == TRUE)
      {
         //camera is active
         (tVoid) m_poInstance->_bSetTimer_CameraBoot(DISPVIDCTRL_RVC_FSM_CAMERASUPPLY_CAMERAFASTBOOT_TIME_IN_MS);
         //ETG_TRACE_USR4((" FSM-CAM  :: _bSetTimer_CameraBoot() fastboot after RVC Holding active"));
      }
      else
      {
         //camera is not active
         (tVoid) m_poInstance->_bSetTimer_CameraBoot(DISPVIDCTRL_RVC_FSM_CAMERASUPPLY_CAMERABOOT_TIME_IN_MS);
      }
   }
}

/******************************************************************************/
/* FUNCTION     vExit_CameraBoot                                              */
/******************************************************************************/
/**
*  \brief       stop camera boot timer
*               clear some states
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_CameraSupply::vExit_CameraBoot(tVoid)
{
   if (NULL != m_poInstance)
   {
      //stop camera boot timer
      (tVoid) m_poInstance->_bSetTimer_CameraBoot(0);

      //clear eTrigger_CameraBootTimeout status within FSM
      //clear eTrigger_SettingsOn status within FSM
      m_poInstance->vUpdateTrigger((eTrigger_CameraBootTimeout | eTrigger_SettingsOn), FALSE);
   }
}

/******************************************************************************/
/* FUNCTION     vEntry_CameraOn_ReverseRequest                                */
/******************************************************************************/
/**
*  \brief       post event to HMI Visualization state machine
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_CameraSupply::vEntry_CameraOn_ReverseRequest(tVoid)
{
   if ((m_poInstance) && (m_poInstance->m_poControl_Rvc))
   {
      if (eTrigger_DirectOnReverseReq != m_poInstance->u32GetCurrentTrigger())
      {
         m_poInstance->m_poControl_Rvc->vHandleFsmOutEvent_CameraSupply(FSM_CAMERASUPPLY_OUT__HMI_EV_VISUALISATION_ON);
      }
   }
}

/******************************************************************************/
/* FUNCTION     vDo_CameraOn_ReverseRequest                                   */
/******************************************************************************/
/**
*  \brief       Handle Video Holding Time
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_CameraSupply::vDo_CameraOn_ReverseRequest(tVoid)
{
// triggers:
// eTrigger_ReverseOff
// eTrigger_ReverseDebounced

   if (m_poInstance)
   {
      if ( m_poInstance->u32GetCurrentTrigger() == dispvidctrl_tclFsm_CameraSupply::eTrigger_ReverseDebounced )
      {
         // stop Rvc Video Holding timer because reverse is switched ON
         (tVoid) m_poInstance->_bSetTimer_VideoHolding(0);
      }
      else if ( m_poInstance->u32GetCurrentTrigger() == dispvidctrl_tclFsm_CameraSupply::eTrigger_ReverseOff )
      {
         // start Rvc Video Holding timer
         (tVoid) m_poInstance->_bSetTimer_VideoHolding(m_poInstance->m_u32CameraVideoHoldingTimeBrandspezificInMs);
      }
   }
}

/******************************************************************************/
/* FUNCTION     vEntry_CameraOn_SettingsRequest                               */
/******************************************************************************/
/**
*  \brief       post event to HMI Visualization state machine
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_CameraSupply::vEntry_CameraOn_SettingsRequest(tVoid)
{
   if ((m_poInstance) && (m_poInstance->m_poControl_Rvc))
   {
      m_poInstance->m_poControl_Rvc->vHandleFsmOutEvent_CameraSupply(FSM_CAMERASUPPLY_OUT__HMI_EV_VISUALISATION_ON_CAMERA_SETTINGS);
   }
}

/******************************************************************************/
/* FUNCTION     vExit_CameraOn                                                */
/******************************************************************************/
/**
*  \brief
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_CameraSupply::vExit_CameraOn(tVoid)
{
   if (m_poInstance)
   {
      // stop Rvc Video Holding timer
      (tVoid) m_poInstance->_bSetTimer_VideoHolding(0);
   }
}


/******************************************************************************/
/* FUNCTION     vEntry_CameraOn_WaitForOffRequest                             */
/******************************************************************************/
/**
*  \brief       post event to HMI Visualization state machine
*               in case of camera OFF trigger start camera OFF delay timer
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_CameraSupply::vEntry_CameraOn_WaitForOffRequest(tVoid)
{
// trigger(s):
// eTrigger_Disable
// eTrigger_VideoHoldingTimeout
// eTrigger_ServiceRequestOff
   if (NULL != m_poInstance)
   {
      tU32 u32OffReason = 0;
      if ( (TRUE == m_poInstance->bGetTriggerState(dispvidctrl_tclFsm_CameraSupply::eTrigger_ServiceRequestOff)) 
      ||   (TRUE == m_poInstance->bGetTriggerState(dispvidctrl_tclFsm_CameraSupply::eTrigger_Disable)) 
      ||   (TRUE == m_poInstance->bGetTriggerState(dispvidctrl_tclFsm_CameraSupply::eTrigger_IgnitionOff)) )
      {
         u32OffReason = dispvidctrl_tclFsm_Hmi::EN_VD_RVC_HMI_VISUALISATION_OFF__SERVICE_REQUEST_OFF;
      }

      // post event VisualisationOff
      if (m_poInstance->m_poControl_Rvc)
      {
         m_poInstance->m_poControl_Rvc->vHandleFsmOutEvent_CameraSupply(FSM_CAMERASUPPLY_OUT__HMI_EV_VISUALISATION_OFF, u32OffReason);
      }

      if (eTrigger_Disable == m_poInstance->u32GetCurrentTrigger())
      {
         // start camera OFF delay timer
         (tVoid) m_poInstance->_bSetTimer_CameraOffDelay(DISPVIDCTRL_RVC_FSM_CAMERASUPPLY_CAMERADISABLE_TIME_IN_MS);
      }
      else
      {
         // remember camera holding state
         m_poInstance->m_bRvcPowerHoldingActive = TRUE;
      }
   }
}

/******************************************************************************/
/* FUNCTION     vDo_CameraOn_WaitForOffRequest                                */
/******************************************************************************/
/**
*  \brief       in case of camera OFF trigger start camera OFF delay timer
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_CameraSupply::vDo_CameraOn_WaitForOffRequest(tVoid)
{
// trigger(s):
// eTrigger_CameraOff
// eTrigger_Disable

   if (NULL != m_poInstance)
   {
      //start camera OFF delay timer with CAMERAOFFDELAY_TIME
      if (eTrigger_CameraOff == m_poInstance->u32GetCurrentTrigger())
      {
         (tVoid) m_poInstance->_bSetTimer_CameraOffDelay(m_poInstance->m_u32CameraPowerHoldingTimeBrandspezificInMs);
      }
      //start camera OFF delay timer with CAMERADISABLE_TIME
      if (eTrigger_Disable == m_poInstance->u32GetCurrentTrigger())
      {
         (tVoid) m_poInstance->_bSetTimer_CameraOffDelay(DISPVIDCTRL_RVC_FSM_CAMERASUPPLY_CAMERADISABLE_TIME_IN_MS);
      }
   }
}

/******************************************************************************/
/* FUNCTION     vExit_CameraOn_WaitForOffRequest                              */
/******************************************************************************/
/**
*  \brief       stop camera OFF delay timer
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclFsm_CameraSupply::vExit_CameraOn_WaitForOffRequest(tVoid)
{
   if (NULL != m_poInstance)
   {
      //stop camera off delay timer
      (tVoid) m_poInstance->_bSetTimer_CameraOffDelay(0);
   }
}


/*******************************************************************************
*                                       E O F
*******************************************************************************/
