/******************************************************************************/
/**
* \file    dispvidctrl_tclControl_Video_GdlAdj.cpp
* \ingroup
*
* \brief   functionality for Guideline Adjustment
*
* \remark  Copyright : (c) 2016 Robert Bosch GmbH, Hildesheim
* \remark  Author    : Michael Niemann
* \remark  Scope     : AIVI
*
*/
/******************************************************************************/

/*******************************************************************************
                        Includes
*******************************************************************************/
#include "dispvidctrl_tclControl_Video.h"
#include "Idispvidctrl_ClientEarly.h"
#include "dispvidctrl_tclControl_Hmi.h"
#include "I_dispvidctrl_ServiceRvcIf.h"
#include "Idispvidctrl_ClientVcan.h"

#include "vd_rvc_tclGraphics_DynGuideline.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_APPLICATION
#include "trcGenProj/Header/dispvidctrl_tclControl_Video_GdlAdj.cpp.trc.h"
#endif

/*******************************************************************************
                        Defines
*******************************************************************************/

// #define GDL_ADJ__ALLOW_ACCESS_TO_TUNING_PARAMS    // uncomment this define to enable modification of these parameters as well
// #define GDL_ADJ__ALLOW_CAMERA_TYPE_SELECTION      // uncomment this define to enable modification of the camera type as well // not supported by HMI (text) yet => keep disabled for now


#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE1
#else
#endif

#define GDL_ADJ__LIMIT_TO_RANGE__BOOL(val)            { if ((val)<(0.5)) (val)=(0);  else (val)=(1); }
#define GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(val)   { if ((val)<(0)) (val)=(0);  else if ((val)>(255))  (val)=(255); }
#define GDL_ADJ__LIMIT_TO_RANGE(val,min,max)   { if ((val)<(min)) (val)=(min);  else if ((val)>(max))  (val)=(max); }

/******************************************************************************/
/* FUNCTION     vHandleControl_GdlAdj()                                       */
/******************************************************************************/
/**
*  \brief
*
*  \param
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::_vGdlAdj_HandleControl(tenControl_Video_GdlAdj enGdlAdjControl, tU32 u32Value)
{
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32Value );

   ETG_TRACE_USR4(("   CA-VID :: ........... vHandleControlGdlAdj(): %d     ", ETG_CENUM(tenControl_Video_GdlAdj, enGdlAdjControl)));

   switch (enGdlAdjControl)
   {
      case EN_CTRL_GDLADJ__SERVICE_REQUEST_VISUALISATION_ON_GUIDELINE_ADJUSTMENT:
         // request to Control HMI for activation of this special screen
         if (m_poControl_Hmi)
         {
            m_poControl_Hmi->vHandleControl(EN_CONTROL_HMI__GDL_ADJ_SCREEN, (tU32) TRUE);
         }
         _vGdlAdj_UpdateProperty_CameraAngle();
         _vGdlAdj_UpdateProperty_SteeringAngle();
         break;
      case EN_CTRL_GDLADJ__SERVICE_REQUEST_VISUALISATION_OFF:
         if (m_poControl_Hmi)
         {
            m_poControl_Hmi->vHandleControl(EN_CONTROL_HMI__GDL_ADJ_SCREEN, (tU32) FALSE);
         }
         break;
      case EN_CTRL_GDLADJ__ADJ_SCREEN_ON_CONFIRMATION:
         // "CorrectDrawLine" Adjustment screen to modify Camera Roll, Yaw and Pitch Angle is visible
         m_GdlAdj_bCorrectDrawLineScreen_Active = TRUE;

         //activate data update of properties relevant for the Screen
         // -> Steering Angle and State
         // -> Camera Yaw, Pitch, Roll

         break;
      case EN_CTRL_GDLADJ__ADJ_SCREEN_OFF_CONFIRMATION:
         // "CorrectDrawLine" Adjustment screen to modify Camera Roll, Yaw and Pitch Angle is not visible
         m_GdlAdj_bCorrectDrawLineScreen_Active = FALSE;

         //deactivate data update of properties relevant for the Screen

         break;
      case EN_CTRL_GDLADJ__ADJUST_BUTTON_UP:
      case EN_CTRL_GDLADJ__ADJUST_BUTTON_DOWN:
      case EN_CTRL_GDLADJ__ADJUST_BUTTON_LEFT:
      case EN_CTRL_GDLADJ__ADJUST_BUTTON_RIGHT:
      case EN_CTRL_GDLADJ__ADJUST_BUTTON_ROTATION_LEFT:
      case EN_CTRL_GDLADJ__ADJUST_BUTTON_ROTATION_RIGHT:
         _vGdlAdj_HandleAdjustButton(enGdlAdjControl);
         break;
      case EN_CTRL_GDLADJ__ADJUST_COMPLETE_CONFIRM:
         _vGdlAdj_HandleComplete();
         break;
      case EN_CTRL_GDLADJ__ADJUST_COMPLETE_REFUSED:
         _vGdlAdj_HandleRefused();
         break;
      case EN_CTRL_GDLADJ__ADJUST_COMPLETE_RESET:
         _vGdlAdj_HandleReset();
         break;
      //------------------------------------------
      case EN_CTRL_GDLADJ__UPDATE_STEERING_ANGLE:
         _vGdlAdj_UpdateProperty_SteeringAngle();
         break;
      default:
         // should never be the case
         break;
   }
}

/******************************************************************************/
/* FUNCTION     bGdlAdj_ConfigurationDataGetAll                               */
/******************************************************************************/
/**
*  \brief       The "GetAll" is providing the original configuration values
*               (or the modified values if a final confirmation was done inbetween)
*
*  \param
*  \return
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Video::bGdlAdj_ConfigurationDataGetAll(tU8 au8Item[], tF64 af64Value[])
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Video::bGdlAdj_ConfigurationDataGetAll() entered."));

   tBool bReturnValue;

   bReturnValue = _bGdlAdj_GetAllDynGdlVehicleParamsElement(au8Item, af64Value);

   return (bReturnValue);
}


/******************************************************************************/
/* FUNCTION     bGdlAdj_ConfigurationDataGet                                  */
/******************************************************************************/
/**
*  \brief       The "Get" is providing the original configuration value
*               (or the modified one if a final confirmation was done inbetween)
*
*  \param
*  \return
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Video::bGdlAdj_ConfigurationDataGet(tU8 u8Item, tF64& f64Value)
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Video::bGdlAdj_ConfigurationDataGet() entered: item=%u",
                   ETG_CENUM(tenControl_Video_GdlAdjConfigurationDataItem,u8Item) ));

   tBool bReturnValue = FALSE;
   tU8 u8NumberOfListElements;
   tBool bItemWithinList;

   // check first if the requested value was overwritten temporarely
   // we hold the original values which were overwritten in vecConfigurationDataSet

   // iterate through the list and check if a entry for the item is present
   // if the item is present take this one
   bItemWithinList = FALSE;
   u8NumberOfListElements = vecConfigurationDataSet.size();

   for (tU8 i = 0; i < u8NumberOfListElements; i++)
   {
      if ( u8Item ==  vecConfigurationDataSet[i].u8Item)
      {
         bItemWithinList = TRUE;
         f64Value = vecConfigurationDataSet[i].f64Value;
         bReturnValue = TRUE;
         ETG_TRACE_USR4(("dispvidctrl_tclControl_Video::bGdlAdj_ConfigurationDataGet(): item found in list, returning list val=%f",f64Value))
         break;
      }
   }
   // if the item is not present in the list read the original from DynGuideline
   if ( !bItemWithinList )
   {
      // get the value from DynGuideline
      bReturnValue = _bGdlAdj_GetDynGdlVehicleParamsElement(u8Item, f64Value);
      ETG_TRACE_USR4(("dispvidctrl_tclControl_Video::bGdlAdj_ConfigurationDataGet(): not found in list, returning DynGl val=%f",f64Value))
   }

   return (bReturnValue);
}


/******************************************************************************/
/* FUNCTION     bGdlAdj_ConfigurationDataSet                                  */
/******************************************************************************/
/**
*  \brief       called from HMI to set a value to become effective temporarely
*               until completion confirmation
*
*  \param       Liste Item, reference to value
*  \return      sucess state, range checked value (in reference variable)
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Video::bGdlAdj_ConfigurationDataSet(tU8 u8Item, tF64& f64Value)
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Video::bGdlAdj_ConfigurationDataSet() entered: item=%u val=%f",
                   ETG_CENUM(tenControl_Video_GdlAdjConfigurationDataItem,u8Item), (tF32)f64Value));

   tBool bReturnValue = FALSE;
   tU8 u8NumberOfListElements;
   tBool bItemWithinList;

   // we have to hold the original value to set it back when Completion-Refused will be triggered
   // we hold the original values in vecConfigurationDataSet

   // iterate through the list and check if a entry for the item is already present
   bItemWithinList = FALSE;
   u8NumberOfListElements = vecConfigurationDataSet.size();

   for (tU8 i = 0; i < u8NumberOfListElements; i++)
   {
      if ( u8Item ==  vecConfigurationDataSet[i].u8Item)
      {
         bItemWithinList = TRUE;
         ETG_TRACE_USR4(("dispvidctrl_tclControl_Video::bGdlAdj_ConfigurationDataSet(): item found in list, list val=%f",vecConfigurationDataSet[i].f64Value))
         break;
      }
   }
   // if the Item is not present in the list already
   // add a new entry in the list
   if ( !bItemWithinList )
   {
      stGdlAdj_ConfigurationData stConfigurationdataSet;
      tF64 f64ValueOriginal;

      // get the value from DynGuideline
      bReturnValue = _bGdlAdj_GetDynGdlVehicleParamsElement(u8Item, f64ValueOriginal);

      stConfigurationdataSet.u8Item = u8Item;
      stConfigurationdataSet.f64Value = f64ValueOriginal;
      vecConfigurationDataSet.push_back(stConfigurationdataSet);
      
      ETG_TRACE_USR4(("dispvidctrl_tclControl_Video::bGdlAdj_ConfigurationDataSet(): item added to list, DynGl val=%f",stConfigurationdataSet.f64Value))
   }

   //---------------------------------------------------------------------------

   ETG_TRACE_USR4(("dispvidctrl_tclControl_Video::bGdlAdj_ConfigurationDataSet() not corrected value %f", f64Value));

   // correct the value according to the datatype limitations before doing the set
   (tVoid) _bGdlAdj_DoRangeCorrectionForElementDataValue(u8Item, f64Value);

   ETG_TRACE_USR4(("dispvidctrl_tclControl_Video::bGdlAdj_ConfigurationDataSet() DataType corrected value %f", f64Value));

   //set the value
   (tVoid) _bGdlAdj_SetDynGdlVehicleParamsElement(u8Item, f64Value);
   // make the changed setting effective (is alos doing some additional range checkes)
   if (m_poClient_VideoPlayer)
   {
      m_poClient_VideoPlayer->vIndVehicleConfigChanged_DynGuideline();
   }
   // read the value back and update reference variable
   (tVoid) _bGdlAdj_GetDynGdlVehicleParamsElement(u8Item, f64Value);

   ETG_TRACE_USR4(("dispvidctrl_tclControl_Video::bGdlAdj_ConfigurationDataSet() read back corrected value %f", f64Value));

   // start or stop SteeringAngleAdjustment which is related to AnticipatoryLine ON/OFF
   if (u8Item == EN_GDLADJ__ANTICIPATORYLINE)
   {
      _vGuidelineToggleSequence_Initialize();
      _vStartStopSteeringAngleAdjustment();
   }

   return (TRUE);
}

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
/******************************************************************************/
/* FUNCTION     _vGdlAdj_HandleComplete()                                     */
/******************************************************************************/
/**
*  \brief       we can erase original values from vecConfigurationDataSet
*               modified values are valid
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::_vGdlAdj_HandleComplete(tVoid)
{
   // changes done (with set) in data structure of DynGuideline are valid now
   // we can clean the list which is holding the original values

   // erase all list elements
   vecConfigurationDataSet.clear();

   m_f32CameraRollAngle  = 0;
   m_f32CameraYawAngle   = 0;
   m_f32CameraPitchAngle = 0;
}

/******************************************************************************/
/* FUNCTION     _vGdlAdj_HandleRefused()                                      */
/******************************************************************************/
/**
*  \brief       set back original values in vd_rvc_tstVehicleParams structure of
*               DynGuideline module with values which were modified
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::_vGdlAdj_HandleRefused(tVoid)
{
   tU8 u8NumberOfListElements;

   // set back the values from our list
   // iterate through the local list and check if a entry for the item is present
   u8NumberOfListElements = vecConfigurationDataSet.size();

   for (tU8 i = 0; i < u8NumberOfListElements; i++)
   {
      (tVoid) _bGdlAdj_SetDynGdlVehicleParamsElement(vecConfigurationDataSet[i].u8Item, vecConfigurationDataSet[i].f64Value);
   }

   // make the changed settings effective
   if (m_poClient_VideoPlayer)
   {
      m_poClient_VideoPlayer->vIndVehicleConfigChanged_DynGuideline();
   }

   // erase all list elements
   vecConfigurationDataSet.clear();

   m_f32CameraRollAngle  = 0;
   m_f32CameraYawAngle   = 0;
   m_f32CameraPitchAngle = 0;

   _vGuidelineToggleSequence_Initialize();
   _vStartStopSteeringAngleAdjustment();
}

/******************************************************************************/
/* FUNCTION     _vGdlAdj_HandleReset()                                        */
/******************************************************************************/
/**
*  \brief       change back to original configuration data
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::_vGdlAdj_HandleReset(tVoid)
{
   if (m_poClient_VideoPlayer)
   {
      m_bAnticipatoryLine = m_Config_bAnticipatoryLine;
      m_poClient_VideoPlayer->vSetVehicleConfig_DynGuideline();
   }

   _vGuidelineToggleSequence_Initialize();
   _vStartStopSteeringAngleAdjustment();
}


/******************************************************************************/
/* FUNCTION     _vGdlAdj_HandleAdjustButton()                                 */
/******************************************************************************/
/**
*  \brief       
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::_vGdlAdj_HandleAdjustButton(tenControl_Video_GdlAdj enGdlAdjControl)
{
   tF64 f64CameraRollAngle;
   tF64 f64CameraYawAngle;
   tF64 f64CameraPitchAngle;


   // we have to add or substract 0.2 degree up to +/- 10 degree
   // the get and set interface functions are requireing value with factor 10 as used in config string

   switch (enGdlAdjControl)
   {
      case EN_CTRL_GDLADJ__ADJUST_BUTTON_UP:
         // Pitch Angle +0.2
         m_f32CameraPitchAngle += (tF32) 2;
         if (m_f32CameraPitchAngle > (tF32) 100) {
            m_f32CameraPitchAngle = (tF32) 100;
         }
         break;
      case EN_CTRL_GDLADJ__ADJUST_BUTTON_DOWN:
         // Pitch Angle -0.2
         m_f32CameraPitchAngle -= (tF32) 2;
         if (m_f32CameraPitchAngle < (tF32) -100) {
            m_f32CameraPitchAngle = (tF32) -100;
         }
         break;
      case EN_CTRL_GDLADJ__ADJUST_BUTTON_RIGHT:
         // Yaw Angle -0.2
         m_f32CameraYawAngle -= (tF32) 2;
         if (m_f32CameraYawAngle < (tF32) -100) {
            m_f32CameraYawAngle = (tF32) -100;
         }
         break;
      case EN_CTRL_GDLADJ__ADJUST_BUTTON_LEFT:
         // Yaw Angle +0.2
         m_f32CameraYawAngle += (tF32) 2;
         if (m_f32CameraYawAngle > (tF32) 100) {
            m_f32CameraYawAngle = (tF32) 100;
         }
         break;
      case EN_CTRL_GDLADJ__ADJUST_BUTTON_ROTATION_LEFT:
         // Roll Angle -0.2
         m_f32CameraRollAngle -= (tF32) 2;
         if (m_f32CameraRollAngle < (tF32) -100) {
            m_f32CameraRollAngle = (tF32) -100;
         }
         break;
      case EN_CTRL_GDLADJ__ADJUST_BUTTON_ROTATION_RIGHT:
         // Roll Angle +0.2
         m_f32CameraRollAngle += (tF32) 2;
         if (m_f32CameraRollAngle > (tF32) 100) {
            m_f32CameraRollAngle = (tF32) 100;
         }
         break;
      default:
         // should never be the case
         break;
   }

   (tVoid) bGdlAdj_ConfigurationDataGet((tU8) EN_GDLADJ__CAMROLL,  f64CameraRollAngle);
   (tVoid) bGdlAdj_ConfigurationDataGet((tU8) EN_GDLADJ__CAMYAW,   f64CameraYawAngle);
   (tVoid) bGdlAdj_ConfigurationDataGet((tU8) EN_GDLADJ__CAMPITCH, f64CameraPitchAngle);

   f64CameraRollAngle  += m_f32CameraRollAngle;
   f64CameraYawAngle   += m_f32CameraYawAngle;
   f64CameraPitchAngle += m_f32CameraPitchAngle;

   (tVoid) bGdlAdj_ConfigurationDataSet((tU8) EN_GDLADJ__CAMROLL,  f64CameraRollAngle);
   (tVoid) bGdlAdj_ConfigurationDataSet((tU8) EN_GDLADJ__CAMYAW,   f64CameraYawAngle);
   (tVoid) bGdlAdj_ConfigurationDataSet((tU8) EN_GDLADJ__CAMPITCH, f64CameraPitchAngle);

   _vGdlAdj_UpdateProperty_CameraAngle();

   // set guideline visibility with force update to get updated guideline drawing
   if(m_poClient_VideoPlayer != 0)
   {
      m_poClient_VideoPlayer->vSetGuideLineVisibility(DYN_GUIDELINE_DRAW_STATIC_SIDE | DYN_GUIDELINE_FORCE_UPDATE);
   }
}

/******************************************************************************/
/* FUNCTION     _vGdlAdj_UpdateProperty_CameraAngle()                         */
/******************************************************************************/
/**
*  \brief       
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::_vGdlAdj_UpdateProperty_CameraAngle(tVoid)
{
   tF32 f32CameraRollAngle;
   tF32 f32CameraYawAngle;
   tF32 f32CameraPitchAngle;
   vd_rvc_tstVehicleParams* pstVehicleParameter = NULL;

   if (m_poClient_VideoPlayer)
   {
      pstVehicleParameter = m_poClient_VideoPlayer->vGetVehicleConfigPtr_DynGuideline();
   }

   if (pstVehicleParameter)
   {
      // get Pitch, Yaw and Roll Value
      f32CameraRollAngle = (tF32) pstVehicleParameter->dCamRoll;
      f32CameraYawAngle = (tF32) pstVehicleParameter->dCamYaw;
      f32CameraPitchAngle = (tF32) pstVehicleParameter->dCamPitch;

      if ( m_poServiceRvcIf ) {
         ETG_TRACE_USR4(("dispvidctrl_tclControl_Video::_vGdlAdj_UpdateProperty_CameraAngle() Roll %f, Yaw %f, Pitch %f", f32CameraRollAngle, f32CameraYawAngle, f32CameraPitchAngle));
         m_poServiceRvcIf->sendGdlAdj_CameraAngleStatus(f32CameraRollAngle, f32CameraYawAngle, f32CameraPitchAngle);
      }
   }
}

/******************************************************************************/
/* FUNCTION     _vGdlAdj_UpdateProperty_SteeringAngle()                       */
/******************************************************************************/
/**
*  \brief       
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::_vGdlAdj_UpdateProperty_SteeringAngle(tVoid)
{
   tBool bShowValues;
   tU8 u8CentreAdjustmentState;
   tS32 s32SteeringAngleCentreAdjusted;
   rvc_fi_tclString SteeringAngleSignalState;

   tS32  s32SteeringAngleDeg = 0;
   tBool bSteeringAngleReadSuccess = FALSE;

   if (m_poClientVcan)
   {
      //read steering angle CAN Signal from message 002
      bSteeringAngleReadSuccess = m_poClientVcan->bGet_SteeringAngleSignal(s32SteeringAngleDeg);



      //adjust steering angle with gained offset
      ETG_TRACE_USR4((" CA-GRF   :: _vOnUpdateSteeringAngle(): Steering Angle_raw =  %d Offset_raw = %d ", s32SteeringAngleDeg, m_SteeringAngleAdjust_s32SteeringAngleOffset));
      //calculate physical value from raw data (raw data * 0.1)
      s32SteeringAngleDeg =  (tS32) ((s32SteeringAngleDeg - m_SteeringAngleAdjust_s32SteeringAngleOffset) * 0.1);
      ETG_TRACE_USR4((" CA-GRF   :: _vOnUpdateSteeringAngle(): Adjusted Steering Angle =  %d ", s32SteeringAngleDeg));
   }


   if (m_bAnticipatoryLine)
   {
      bShowValues = TRUE;
   }
   else
   {
      bShowValues = FALSE;
   }

   if (bSteeringAngleReadSuccess)
   {
      s32SteeringAngleCentreAdjusted = s32SteeringAngleDeg;
      SteeringAngleSignalState = "OK";
   }
   else
   {
      s32SteeringAngleCentreAdjusted = 0;
      SteeringAngleSignalState = "E00";
   }

   u8CentreAdjustmentState = 1;

   if ( m_poServiceRvcIf ) {
      ETG_TRACE_USR4(("dispvidctrl_tclControl_Video::_vGdlAdj_UpdateProperty_SteeringAngle() bShowValues %d, u8CentreAdjustmentState %d, s32SteeringAngleCentreAdjusted %d", bShowValues, u8CentreAdjustmentState, s32SteeringAngleCentreAdjusted));
      m_poServiceRvcIf->sendGdlAdj_SteeringAngleStatus(bShowValues, u8CentreAdjustmentState, s32SteeringAngleCentreAdjusted, SteeringAngleSignalState);
   }
}


/******************************************************************************/
/* FUNCTION     _bGdlAdj_GetDynGdlVehicleParamsElement()                      */
/******************************************************************************/
/**
*  \brief       to get a value from stVehicleParams
*               corrected to the format for the HMI list
*
*  \param
*  \return
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Video::_bGdlAdj_GetDynGdlVehicleParamsElement(tU8 u8Item, tF64& f64Value)
{
   tBool bReturnValue = FALSE;
   vd_rvc_tstVehicleParams* pstVehicleParameter = NULL;

   if (m_poClient_VideoPlayer)
   {
      pstVehicleParameter = m_poClient_VideoPlayer->vGetVehicleConfigPtr_DynGuideline();
   }

   if (pstVehicleParameter)
   {
      bReturnValue = TRUE;
      switch ((tenControl_Video_GdlAdjConfigurationDataItem) u8Item)
      {
         case EN_GDLADJ__ANTICIPATORYLINE:               f64Value =  (tF64) m_bAnticipatoryLine;                               break;
         case EN_GDLADJ__VEHICLEWIDTH:                   f64Value =  (tF64) (pstVehicleParameter->dVehWidth         * 10);     break;
         case EN_GDLADJ__WHEELBASE:                      f64Value =  (tF64) (pstVehicleParameter->dWheelBase        * 10);     break;
         case EN_GDLADJ__REAROVERHANG:                   f64Value =  (tF64) (pstVehicleParameter->dRearOverhang     * 10);     break;
         case EN_GDLADJ__STEERWHEELCALIB_F1:             f64Value =  (tF64) (pstVehicleParameter->dSWAtoFWA_Coeff1  * 1e5);    break;
         case EN_GDLADJ__STEERWHEELCALIB_F3:             f64Value =  (tF64) (pstVehicleParameter->dSWAtoFWA_Coeff3  * 1e10);   break;
         case EN_GDLADJ__MAXSTEERWHEELANGLE:             f64Value =  (tF64) pstVehicleParameter->dMaxSteerWhlAngle;            break;
         case EN_GDLADJ__MAXREARWHEELANGLE:              f64Value =  (tF64) (pstVehicleParameter->dMaxRearTireAngle * 100);    break;
         case EN_GDLADJ__RADIUSTUNINGFACT:               f64Value =  (tF64) (pstVehicleParameter->dRadiusTuningFact * 100);    break;
         case EN_GDLADJ__FOURWHEELSTEERING:              f64Value =  (tF64) pstVehicleParameter->bFourWheelSteering;           break;
         case EN_GDLADJ__TIREANGLETURNSENSE:             f64Value =  (tF64) pstVehicleParameter->bPosTireAngleIsCcw;           break;
         case EN_GDLADJ__TIREANGLEREFERENCEPOINT:        f64Value =  (tF64) pstVehicleParameter->u8TireAngleRefPoint;          break;
         #if(0)  // we will *not* allow to modify the guideline style at run-time, since the activation status of some other items depend on it
         case EN_GDLADJ__GUIDELINESTYLE:                 f64Value =  (tF64) pstVehicleParameter->u8GuidelnStyle;               break;
         #endif
         case EN_GDLADJ__CAMXOFFSET:                     f64Value =  (tF64) (pstVehicleParameter->dCamXOffs         * 10);     break;
         case EN_GDLADJ__CAMYOFFSET:                     f64Value =  (tF64) (pstVehicleParameter->dCamYOffs         * 10);     break;
         case EN_GDLADJ__CAMZOFFSET:                     f64Value =  (tF64) (pstVehicleParameter->dCamZOffs         * 10);     break;
         case EN_GDLADJ__CAMYAW:                         f64Value =  (tF64) (pstVehicleParameter->dCamYaw           * 10);     break;
         case EN_GDLADJ__CAMPITCH:                       f64Value =  (tF64) (pstVehicleParameter->dCamPitch         * 10);     break;
         case EN_GDLADJ__CAMROLL:                        f64Value =  (tF64) (pstVehicleParameter->dCamRoll          * 10);     break;
         case EN_GDLADJ__FRONTTIRETHRESH:                f64Value =  (tF64) (pstVehicleParameter->dFrontTireThresh  * 100);    break;
         case EN_GDLADJ__MINFRONTTIRECHANGE:             f64Value =  (tF64) (pstVehicleParameter->dMinFrontTireChg  * 100);    break;
         case EN_GDLADJ__REARTIRETHRESH:                 f64Value =  (tF64) (pstVehicleParameter->dRearTireThresh   * 100);    break;
         case EN_GDLADJ__MINREARTIRECHANGE:              f64Value =  (tF64) (pstVehicleParameter->dMinRearTireChg   * 100);    break;
         case EN_GDLADJ__GUIDELINESTARTDISTANCE:         f64Value =  (tF64) pstVehicleParameter->dGuidelnStartDist;            break;
         case EN_GDLADJ__GUIDELINESIDEMARGIN:            f64Value =  (tF64) pstVehicleParameter->dGuidelnLatMargin;            break;
         case EN_GDLADJ__RGBCOLORSTATICSIDENEAR_RED:     f64Value =  (tF64) pstVehicleParameter->au8GuidelnRgb[0][0];          break;
         case EN_GDLADJ__RGBCOLORSTATICSIDENEAR_GREEN:   f64Value =  (tF64) pstVehicleParameter->au8GuidelnRgb[0][1];          break;
         case EN_GDLADJ__RGBCOLORSTATICSIDENEAR_BLUE:    f64Value =  (tF64) pstVehicleParameter->au8GuidelnRgb[0][2];          break;
         case EN_GDLADJ__RGBCOLORSTATICSIDEMID_RED:      f64Value =  (tF64) pstVehicleParameter->au8GuidelnRgb[1][0];          break;
         case EN_GDLADJ__RGBCOLORSTATICSIDEMID_GREEN:    f64Value =  (tF64) pstVehicleParameter->au8GuidelnRgb[1][1];          break;
         case EN_GDLADJ__RGBCOLORSTATICSIDEMID_BLUE:     f64Value =  (tF64) pstVehicleParameter->au8GuidelnRgb[1][2];          break;
         case EN_GDLADJ__RGBCOLORSTATICSIDEFAR_RED:      f64Value =  (tF64) pstVehicleParameter->au8GuidelnRgb[2][0];          break;
         case EN_GDLADJ__RGBCOLORSTATICSIDEFAR_GREEN:    f64Value =  (tF64) pstVehicleParameter->au8GuidelnRgb[2][1];          break;
         case EN_GDLADJ__RGBCOLORSTATICSIDEFAR_BLUE:     f64Value =  (tF64) pstVehicleParameter->au8GuidelnRgb[2][2];          break;
         case EN_GDLADJ__RGBCOLORSTATICCENTER_RED:       f64Value =  (tF64) pstVehicleParameter->au8GuidelnRgb[3][0];          break;
         case EN_GDLADJ__RGBCOLORSTATICCENTER_GREEN:     f64Value =  (tF64) pstVehicleParameter->au8GuidelnRgb[3][1];          break;
         case EN_GDLADJ__RGBCOLORSTATICCENTER_BLUE:      f64Value =  (tF64) pstVehicleParameter->au8GuidelnRgb[3][2];          break;
         case EN_GDLADJ__RGBCOLORDYNAMIC_RED:            f64Value =  (tF64) pstVehicleParameter->au8GuidelnRgb[4][0];          break;
         case EN_GDLADJ__RGBCOLORDYNAMIC_GREEN:          f64Value =  (tF64) pstVehicleParameter->au8GuidelnRgb[4][1];          break;
         case EN_GDLADJ__RGBCOLORDYNAMIC_BLUE:           f64Value =  (tF64) pstVehicleParameter->au8GuidelnRgb[4][2];          break;
         case EN_GDLADJ__LINEWIDTHSTATICSIDENEAR:        f64Value =  (tF64) pstVehicleParameter->astLineWidth[0].u8Near;       break;
         case EN_GDLADJ__LINEWIDTHSTATICSIDEFAR:         f64Value =  (tF64) pstVehicleParameter->astLineWidth[0].u8Far;        break;
         case EN_GDLADJ__LINEWIDTHDYNAMICNEAR:           f64Value =  (tF64) pstVehicleParameter->astLineWidth[1].u8Near;       break;
         case EN_GDLADJ__LINEWIDTHDYNAMICFAR:            f64Value =  (tF64) pstVehicleParameter->astLineWidth[1].u8Far;        break;
         case EN_GDLADJ__LINEWIDTHSTATICCENTERNEAR:      f64Value =  (tF64) pstVehicleParameter->astLineWidth[2].u8Near;       break;
         case EN_GDLADJ__LINEWIDTHSTATICCENTERFAR:       f64Value =  (tF64) pstVehicleParameter->astLineWidth[2].u8Far;        break;
         case EN_GDLADJ__STATICLINELENGTHNEAR:           f64Value =  (tF64) pstVehicleParameter->au8GuidelnStatLen[0];         break;
         case EN_GDLADJ__STATICLINELENGTHMID:            f64Value =  (tF64) pstVehicleParameter->au8GuidelnStatLen[1];         break;
         case EN_GDLADJ__STATICLINELENGHTFAR1:           f64Value =  (tF64) pstVehicleParameter->au8GuidelnStatLen[2];         break;
         case EN_GDLADJ__STATICLINELENGHTFAR2:           f64Value =  (tF64) pstVehicleParameter->au8GuidelnStatLen[3];         break;
         case EN_GDLADJ__STATICHORIZLINELENGHT:          f64Value =  (tF64) pstVehicleParameter->dGuidelnLenHorBar;            break;
         case EN_GDLADJ__LINEWIDTHFACTORNEARHLINE:       f64Value =  (tF64) (pstVehicleParameter->fLineWdthWgtNearHbar * 100); break;
         #ifdef GDL_ADJ__ALLOW_ACCESS_TO_TUNING_PARAMS
         case EN_GDLADJ__TUNINGMETHOD:                   f64Value =  (tF64) pstVehicleParameter->stTuneParams.u8Method;                  break;
         case EN_GDLADJ__TUNINGDELTACAMYAW:              f64Value =  (tF64) pstVehicleParameter->stTuneParams.dCamYaw;                   break;
         case EN_GDLADJ__TUNINGDELTACAMPITCH:            f64Value =  (tF64) pstVehicleParameter->stTuneParams.dCamPitch;                 break;
         case EN_GDLADJ__TUNINGDELTACAMROLL:             f64Value =  (tF64) pstVehicleParameter->stTuneParams.dCamRoll;                  break;
         case EN_GDLADJ__TUNINGDELTAXNEARLEFT:           f64Value =  (tF64) pstVehicleParameter->stTuneParams.stOffsLft.dNearDx;         break;
         case EN_GDLADJ__TUNINGDELTAYNEARLEFT:           f64Value =  (tF64) pstVehicleParameter->stTuneParams.stOffsLft.dNearDy;         break;
         case EN_GDLADJ__TUNINGDELTAXFARLEFT:            f64Value =  (tF64) pstVehicleParameter->stTuneParams.stOffsLft.dFarDx;          break;
         case EN_GDLADJ__TUNINGNEARLINELENGTHLEFT:       f64Value =  (tF64) pstVehicleParameter->stTuneParams.stOffsLft.dNearLineLen;    break;
         case EN_GDLADJ__TUNINGDELTAXNEARRIGHT:          f64Value =  (tF64) pstVehicleParameter->stTuneParams.stOffsRgt.dNearDx;         break;
         case EN_GDLADJ__TUNINGDELTAYNEARRIGHT:          f64Value =  (tF64) pstVehicleParameter->stTuneParams.stOffsRgt.dNearDy;         break;
         case EN_GDLADJ__TUNINGDELTAXFARRIGHT:           f64Value =  (tF64) pstVehicleParameter->stTuneParams.stOffsRgt.dFarDx;          break;
         case EN_GDLADJ__TUNINGNEARLINELENGHTRIGHT:      f64Value =  (tF64) pstVehicleParameter->stTuneParams.stOffsRgt.dNearLineLen;    break;
         case EN_GDLADJ__TUNINGSWAZEROPOINTOFFSET:       f64Value =  (tF64) pstVehicleParameter->stTuneParams.dSwaZeroOffset;            break;
         case EN_GDLADJ__TUNINGRWAZEROPOINTOFFSET:       f64Value =  (tF64) pstVehicleParameter->stTuneParams.dRwaZeroOffset;            break;
         #endif
         #ifdef GDL_ADJ__ALLOW_CAMERA_TYPE_SELECTION
         case EN_GDLADJ__RVCLENSPARAMETER:               f64Value =  (tF64) pstVehicleParameter->u8CameraType;                 break;
         #endif
         case EN_GDLADJ__LINEWIDTHFACTORVERTLINES:       f64Value =  (tF64) (pstVehicleParameter->fLineWdthWgtVertLines * 100);break;
         
         default:
         {
            bReturnValue = FALSE;
         }
         break;
      }
   }

   return (bReturnValue);
}

/******************************************************************************/
/* FUNCTION     _bGdlAdj_GetAllDynGdlVehicleParamsElement()                   */
/******************************************************************************/
/**
*  \brief       to get all value from stVehicleParams
*               corrected to the format for the HMI list
*
*  \param
*  \return
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Video::_bGdlAdj_GetAllDynGdlVehicleParamsElement(tU8 au8Item[], tF64 af64Value[])
{
   tBool bReturnValue = FALSE;
   vd_rvc_tstVehicleParams* pstVehicleParameter = NULL;

   ETG_TRACE_USR4(("dispvidctrl_tclControl_Video::_bGdlAdj_GetAllDynGdlVehicleParamsElement() entered."));

   if (m_poClient_VideoPlayer)
   {
      pstVehicleParameter = m_poClient_VideoPlayer->vGetVehicleConfigPtr_DynGuideline();
   }
   else
   {
      ETG_TRACE_USR4(("dispvidctrl_tclControl_Video::_bGdlAdj_GetAllDynGdlVehicleParamsElement() m_poClient_VideoPlayer = NULL."));
   }

   if (pstVehicleParameter)
   {
      bReturnValue = TRUE;
      
      // Reset all item identifiers to "none", making them inaccessible:
      (tVoid) OSAL_pvMemorySet(au8Item, EN_GDLADJ__NONE, sizeof(*au8Item)*DISPVIDCTRL_VIDEO__NUMBER_OFF_CONFIGURATION_DATA );

      // setup/activate the identifiers of those items which are used only in a subset of all guideline styles
      switch(pstVehicleParameter->u8GuidelnStyle){
        case DYN_GUIDELINE_STYLE_NISSAN:
        case DYN_GUIDELINE_STYLE_NISSAN_US:
        case DYN_GUIDELINE_STYLE_MITSUBISHI:
          au8Item[ 8] = (tU8) EN_GDLADJ__RADIUSTUNINGFACT;
          
          au8Item[34] = (tU8) EN_GDLADJ__RGBCOLORSTATICCENTER_RED;
          au8Item[35] = (tU8) EN_GDLADJ__RGBCOLORSTATICCENTER_GREEN;
          au8Item[36] = (tU8) EN_GDLADJ__RGBCOLORSTATICCENTER_BLUE;

          au8Item[44] = (tU8) EN_GDLADJ__LINEWIDTHSTATICCENTERNEAR;
          au8Item[45] = (tU8) EN_GDLADJ__LINEWIDTHSTATICCENTERFAR;
          
          // au8Item[67] = (tU8) EN_GDLADJ__LINEWIDTHFACTORVERTLINES; // new item, not supported by HMI (text) yet => keep disabled for now
          break;

        case DYN_GUIDELINE_STYLE_RENAULT:
          au8Item[23] = (tU8) EN_GDLADJ__GUIDELINESTARTDISTANCE;

          au8Item[46] = (tU8) EN_GDLADJ__STATICLINELENGTHNEAR;
          au8Item[47] = (tU8) EN_GDLADJ__STATICLINELENGTHMID;
          au8Item[48] = (tU8) EN_GDLADJ__STATICLINELENGHTFAR1;
          au8Item[49] = (tU8) EN_GDLADJ__STATICLINELENGHTFAR2;

          au8Item[50] = (tU8) EN_GDLADJ__STATICHORIZLINELENGHT;
          au8Item[51] = (tU8) EN_GDLADJ__LINEWIDTHFACTORNEARHLINE;
          break;

        default:
          bReturnValue = FALSE;
          break;
      }
      
      // setup/activate the identifiers of those items which are used by all guideline styles
      au8Item[ 0] = (tU8) EN_GDLADJ__ANTICIPATORYLINE;
      au8Item[ 1] = (tU8) EN_GDLADJ__VEHICLEWIDTH;
      au8Item[ 2] = (tU8) EN_GDLADJ__WHEELBASE;
      au8Item[ 3] = (tU8) EN_GDLADJ__REAROVERHANG;
      au8Item[ 4] = (tU8) EN_GDLADJ__STEERWHEELCALIB_F1;
      au8Item[ 5] = (tU8) EN_GDLADJ__STEERWHEELCALIB_F3;
      au8Item[ 6] = (tU8) EN_GDLADJ__MAXSTEERWHEELANGLE;
      au8Item[ 7] = (tU8) EN_GDLADJ__MAXREARWHEELANGLE;
      // au8Item[ 8] gets activated depending on guideline style, see above
      au8Item[ 9] = (tU8) EN_GDLADJ__FOURWHEELSTEERING;
      au8Item[10] = (tU8) EN_GDLADJ__TIREANGLETURNSENSE;
      au8Item[11] = (tU8) EN_GDLADJ__TIREANGLEREFERENCEPOINT;
      #if(0)  // we will *not* allow to modify the guideline style at run-time, since the activation status of some other items depend on it
      au8Item[12] = (tU8) EN_GDLADJ__GUIDELINESTYLE;
      #endif
      au8Item[13] = (tU8) EN_GDLADJ__CAMXOFFSET;
      au8Item[14] = (tU8) EN_GDLADJ__CAMYOFFSET;
      au8Item[15] = (tU8) EN_GDLADJ__CAMZOFFSET;
      au8Item[16] = (tU8) EN_GDLADJ__CAMYAW;
      au8Item[17] = (tU8) EN_GDLADJ__CAMPITCH;
      au8Item[18] = (tU8) EN_GDLADJ__CAMROLL;
      au8Item[19] = (tU8) EN_GDLADJ__FRONTTIRETHRESH;
      au8Item[20] = (tU8) EN_GDLADJ__MINFRONTTIRECHANGE;
      au8Item[21] = (tU8) EN_GDLADJ__REARTIRETHRESH;
      au8Item[22] = (tU8) EN_GDLADJ__MINREARTIRECHANGE;
      // au8Item[23] gets activated depending on guideline style, see above
      au8Item[24] = (tU8) EN_GDLADJ__GUIDELINESIDEMARGIN;
      au8Item[25] = (tU8) EN_GDLADJ__RGBCOLORSTATICSIDENEAR_RED;
      au8Item[26] = (tU8) EN_GDLADJ__RGBCOLORSTATICSIDENEAR_GREEN;
      au8Item[27] = (tU8) EN_GDLADJ__RGBCOLORSTATICSIDENEAR_BLUE;
      au8Item[28] = (tU8) EN_GDLADJ__RGBCOLORSTATICSIDEMID_RED;
      au8Item[29] = (tU8) EN_GDLADJ__RGBCOLORSTATICSIDEMID_GREEN;
      au8Item[30] = (tU8) EN_GDLADJ__RGBCOLORSTATICSIDEMID_BLUE;
      au8Item[31] = (tU8) EN_GDLADJ__RGBCOLORSTATICSIDEFAR_RED;
      au8Item[32] = (tU8) EN_GDLADJ__RGBCOLORSTATICSIDEFAR_GREEN;
      au8Item[33] = (tU8) EN_GDLADJ__RGBCOLORSTATICSIDEFAR_BLUE;
      // au8Item[34..36] get activated depending on guideline style, see above
      au8Item[37] = (tU8) EN_GDLADJ__RGBCOLORDYNAMIC_RED;
      au8Item[38] = (tU8) EN_GDLADJ__RGBCOLORDYNAMIC_GREEN;
      au8Item[39] = (tU8) EN_GDLADJ__RGBCOLORDYNAMIC_BLUE;
      au8Item[40] = (tU8) EN_GDLADJ__LINEWIDTHSTATICSIDENEAR;
      au8Item[41] = (tU8) EN_GDLADJ__LINEWIDTHSTATICSIDEFAR;
      au8Item[42] = (tU8) EN_GDLADJ__LINEWIDTHDYNAMICNEAR;
      au8Item[43] = (tU8) EN_GDLADJ__LINEWIDTHDYNAMICFAR;
      // au8Item[44..51] get activated depending on guideline style, see above

      #ifdef GDL_ADJ__ALLOW_ACCESS_TO_TUNING_PARAMS
      au8Item[52] = (tU8) EN_GDLADJ__TUNINGMETHOD;
      au8Item[53] = (tU8) EN_GDLADJ__TUNINGDELTACAMYAW;
      au8Item[54] = (tU8) EN_GDLADJ__TUNINGDELTACAMPITCH;
      au8Item[55] = (tU8) EN_GDLADJ__TUNINGDELTACAMROLL;
      au8Item[56] = (tU8) EN_GDLADJ__TUNINGDELTAXNEARLEFT;
      au8Item[57] = (tU8) EN_GDLADJ__TUNINGDELTAYNEARLEFT;
      au8Item[58] = (tU8) EN_GDLADJ__TUNINGDELTAXFARLEFT;
      au8Item[59] = (tU8) EN_GDLADJ__TUNINGNEARLINELENGTHLEFT;
      au8Item[60] = (tU8) EN_GDLADJ__TUNINGDELTAXNEARRIGHT;
      au8Item[61] = (tU8) EN_GDLADJ__TUNINGDELTAYNEARRIGHT;
      au8Item[62] = (tU8) EN_GDLADJ__TUNINGDELTAXFARRIGHT;
      au8Item[63] = (tU8) EN_GDLADJ__TUNINGNEARLINELENGHTRIGHT;
      au8Item[64] = (tU8) EN_GDLADJ__TUNINGSWAZEROPOINTOFFSET;
      au8Item[65] = (tU8) EN_GDLADJ__TUNINGRWAZEROPOINTOFFSET;
      #endif  // else keep au8Item[52..65] at their init values == EN_GDLADJ__NONE, which disables the access to these items

      #ifdef GDL_ADJ__ALLOW_CAMERA_TYPE_SELECTION
      au8Item[66] = (tU8) EN_GDLADJ__RVCLENSPARAMETER;
      #endif
      // au8Item[67] gets activated depending on guideline style, see above
               
      // setup *all* config *values* (even those of config items which were not activated above)
      af64Value[ 0] =  (tF64) m_bAnticipatoryLine;                               // ANTICIPATORYLINE;
      af64Value[ 1] =  (tF64) (pstVehicleParameter->dVehWidth         * 10);     // VEHICLEWIDTH;
      af64Value[ 2] =  (tF64) (pstVehicleParameter->dWheelBase        * 10);     // WHEELBASE;
      af64Value[ 3] =  (tF64) (pstVehicleParameter->dRearOverhang     * 10);     // REAROVERHANG;
      af64Value[ 4] =  (tF64) (pstVehicleParameter->dSWAtoFWA_Coeff1  * 1e5);    // STEERWHEELCALIB_F1;
      af64Value[ 5] =  (tF64) (pstVehicleParameter->dSWAtoFWA_Coeff3  * 1e10);   // STEERWHEELCALIB_F3;
      af64Value[ 6] =  (tF64) pstVehicleParameter->dMaxSteerWhlAngle;            // MAXSTEERWHEELANGLE;
      af64Value[ 7] =  (tF64) (pstVehicleParameter->dMaxRearTireAngle * 100);    // MAXREARWHEELANGLE;
      af64Value[ 8] =  (tF64) (pstVehicleParameter->dRadiusTuningFact * 100);    // RADIUSTUNINGFACT;
      af64Value[ 9] =  (tF64) pstVehicleParameter->bFourWheelSteering;           // FOURWHEELSTEERING;
      af64Value[10] =  (tF64) pstVehicleParameter->bPosTireAngleIsCcw;           // TIREANGLETURNSENSE;
      af64Value[11] =  (tF64) pstVehicleParameter->u8TireAngleRefPoint;          // TIREANGLEREFERENCEPOINT;
      af64Value[12] =  (tF64) pstVehicleParameter->u8GuidelnStyle;               // GUIDELINESTYLE;
      af64Value[13] =  (tF64) (pstVehicleParameter->dCamXOffs         * 10);     // CAMXOFFSET;
      af64Value[14] =  (tF64) (pstVehicleParameter->dCamYOffs         * 10);     // CAMYOFFSET;
      af64Value[15] =  (tF64) (pstVehicleParameter->dCamZOffs         * 10);     // CAMZOFFSET;
      af64Value[16] =  (tF64) (pstVehicleParameter->dCamYaw           * 10);     // CAMYAW;
      af64Value[17] =  (tF64) (pstVehicleParameter->dCamPitch         * 10);     // CAMPITCH;
      af64Value[18] =  (tF64) (pstVehicleParameter->dCamRoll          * 10);     // CAMROLL;
      af64Value[19] =  (tF64) (pstVehicleParameter->dFrontTireThresh  * 100);    // FRONTTIRETHRESH;
      af64Value[20] =  (tF64) (pstVehicleParameter->dMinFrontTireChg  * 100);    // MINFRONTTIRECHANGE;
      af64Value[21] =  (tF64) (pstVehicleParameter->dRearTireThresh   * 100);    // REARTIRETHRESH;
      af64Value[22] =  (tF64) (pstVehicleParameter->dMinRearTireChg   * 100);    // MINREARTIRECHANGE;
      af64Value[23] =  (tF64) pstVehicleParameter->dGuidelnStartDist;            // GUIDELINESTARTDISTANCE;
      af64Value[24] =  (tF64) pstVehicleParameter->dGuidelnLatMargin;            // GUIDELINESIDEMARGIN;
      af64Value[25] =  (tF64) pstVehicleParameter->au8GuidelnRgb[0][0];          // RGBCOLORSTATICSIDENEAR_RED;
      af64Value[26] =  (tF64) pstVehicleParameter->au8GuidelnRgb[0][1];          // RGBCOLORSTATICSIDENEAR_GREEN;
      af64Value[27] =  (tF64) pstVehicleParameter->au8GuidelnRgb[0][2];          // RGBCOLORSTATICSIDENEAR_BLUE;
      af64Value[28] =  (tF64) pstVehicleParameter->au8GuidelnRgb[1][0];          // RGBCOLORSTATICSIDEMID_RED;
      af64Value[29] =  (tF64) pstVehicleParameter->au8GuidelnRgb[1][1];          // RGBCOLORSTATICSIDEMID_GREEN;
      af64Value[30] =  (tF64) pstVehicleParameter->au8GuidelnRgb[1][2];          // RGBCOLORSTATICSIDEMID_BLUE;
      af64Value[31] =  (tF64) pstVehicleParameter->au8GuidelnRgb[2][0];          // RGBCOLORSTATICSIDEFAR_RED;
      af64Value[32] =  (tF64) pstVehicleParameter->au8GuidelnRgb[2][1];          // RGBCOLORSTATICSIDEFAR_GREEN;
      af64Value[33] =  (tF64) pstVehicleParameter->au8GuidelnRgb[2][2];          // RGBCOLORSTATICSIDEFAR_BLUE;
      af64Value[34] =  (tF64) pstVehicleParameter->au8GuidelnRgb[3][0];          // RGBCOLORSTATICCENTER_RED;
      af64Value[35] =  (tF64) pstVehicleParameter->au8GuidelnRgb[3][1];          // RGBCOLORSTATICCENTER_GREEN;
      af64Value[36] =  (tF64) pstVehicleParameter->au8GuidelnRgb[3][2];          // RGBCOLORSTATICCENTER_BLUE;
      af64Value[37] =  (tF64) pstVehicleParameter->au8GuidelnRgb[4][0];          // RGBCOLORDYNAMIC_RED;
      af64Value[38] =  (tF64) pstVehicleParameter->au8GuidelnRgb[4][1];          // RGBCOLORDYNAMIC_GREEN;
      af64Value[39] =  (tF64) pstVehicleParameter->au8GuidelnRgb[4][2];          // RGBCOLORDYNAMIC_BLUE;
      af64Value[40] =  (tF64) pstVehicleParameter->astLineWidth[0].u8Near;       // LINEWIDTHSTATICSIDENEAR;
      af64Value[41] =  (tF64) pstVehicleParameter->astLineWidth[0].u8Far;        // LINEWIDTHSTATICSIDEFAR;
      af64Value[42] =  (tF64) pstVehicleParameter->astLineWidth[1].u8Near;       // LINEWIDTHDYNAMICNEAR;
      af64Value[43] =  (tF64) pstVehicleParameter->astLineWidth[1].u8Far;        // LINEWIDTHDYNAMICFAR;
      af64Value[44] =  (tF64) pstVehicleParameter->astLineWidth[2].u8Near;       // LINEWIDTHSTATICCENTERNEAR;
      af64Value[45] =  (tF64) pstVehicleParameter->astLineWidth[2].u8Far;        // LINEWIDTHSTATICCENTERFAR;
      af64Value[46] =  (tF64) pstVehicleParameter->au8GuidelnStatLen[0];         // STATICLINELENGTHNEAR;
      af64Value[47] =  (tF64) pstVehicleParameter->au8GuidelnStatLen[1];         // STATICLINELENGTHMID;
      af64Value[48] =  (tF64) pstVehicleParameter->au8GuidelnStatLen[2];         // STATICLINELENGHTFAR1;
      af64Value[49] =  (tF64) pstVehicleParameter->au8GuidelnStatLen[3];         // STATICLINELENGHTFAR2;
      af64Value[50] =  (tF64) pstVehicleParameter->dGuidelnLenHorBar;            // STATICHORIZLINELENGHT;
      af64Value[51] =  (tF64) (pstVehicleParameter->fLineWdthWgtNearHbar * 100); // LINEWIDTHFACTORNEARHLINE;
      af64Value[52] =  (tF64) pstVehicleParameter->stTuneParams.u8Method;                 //TUNINGMETHOD;
      af64Value[53] =  (tF64) pstVehicleParameter->stTuneParams.dCamYaw;                  //TUNINGDELTACAMYAW;
      af64Value[54] =  (tF64) pstVehicleParameter->stTuneParams.dCamPitch;                //TUNINGDELTACAMPITCH;
      af64Value[55] =  (tF64) pstVehicleParameter->stTuneParams.dCamRoll;                 //TUNINGDELTACAMROLL;
      af64Value[56] =  (tF64) pstVehicleParameter->stTuneParams.stOffsLft.dNearDx;        //TUNINGDELTAXNEARLEFT;
      af64Value[57] =  (tF64) pstVehicleParameter->stTuneParams.stOffsLft.dNearDy;        //TUNINGDELTAYNEARLEFT;
      af64Value[58] =  (tF64) pstVehicleParameter->stTuneParams.stOffsLft.dFarDx;         //TUNINGDELTAXFARLEFT;
      af64Value[59] =  (tF64) pstVehicleParameter->stTuneParams.stOffsLft.dNearLineLen;   //TUNINGNEARLINELENGTHLEFT;
      af64Value[60] =  (tF64) pstVehicleParameter->stTuneParams.stOffsRgt.dNearDx;        //TUNINGDELTAXNEARRIGHT;
      af64Value[61] =  (tF64) pstVehicleParameter->stTuneParams.stOffsRgt.dNearDy;        //TUNINGDELTAYNEARRIGHT;
      af64Value[62] =  (tF64) pstVehicleParameter->stTuneParams.stOffsRgt.dFarDx;         //TUNINGDELTAXFARRIGHT;
      af64Value[63] =  (tF64) pstVehicleParameter->stTuneParams.stOffsRgt.dNearLineLen;   //TUNINGNEARLINELENGHTRIGHT;
      af64Value[64] =  (tF64) pstVehicleParameter->stTuneParams.dSwaZeroOffset;           //TUNINGSWAZEROPOINTOFFSET;
      af64Value[65] =  (tF64) pstVehicleParameter->stTuneParams.dRwaZeroOffset;           //TUNINGRWAZEROPOINTOFFSET;
      af64Value[66] =  (tF64) pstVehicleParameter->u8CameraType;                 // RVCLENSPARAMETER;
      af64Value[67] =  (tF64) (pstVehicleParameter->fLineWdthWgtVertLines * 100);// LINEWIDTHFACTORVERTLINES;
   }
   else
   {
      ETG_TRACE_USR4(("dispvidctrl_tclControl_Video::_bGdlAdj_GetAllDynGdlVehicleParamsElement() pstVehicleParameter = NULL."));
   }

   return (bReturnValue);
}


/******************************************************************************/
/* FUNCTION     _bGdlAdj_SetDynGdlVehicleParamsElement()                      */
/******************************************************************************/
/**
*  \brief       to set a value in stVehicleParams
*
*  \param
*  \return
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Video::_bGdlAdj_SetDynGdlVehicleParamsElement(tU8 u8Item, tF64 f64Value)
{
   tBool bReturnValue = FALSE;
   vd_rvc_tstVehicleParams* pstVehicleParameter = NULL;

   if (m_poClient_VideoPlayer)
   {
      pstVehicleParameter = m_poClient_VideoPlayer->vGetVehicleConfigPtr_DynGuideline();
   }

   if (pstVehicleParameter)
   {
      bReturnValue = TRUE;
      switch ((tenControl_Video_GdlAdjConfigurationDataItem) u8Item)
      {
         case EN_GDLADJ__ANTICIPATORYLINE:             m_bAnticipatoryLine                         = (tBool) f64Value;         break;
         case EN_GDLADJ__VEHICLEWIDTH:                 pstVehicleParameter->dVehWidth              =         f64Value * 0.1;   break;
         case EN_GDLADJ__WHEELBASE:                    pstVehicleParameter->dWheelBase             =         f64Value * 0.1;   break;
         case EN_GDLADJ__REAROVERHANG:                 pstVehicleParameter->dRearOverhang          =         f64Value * 0.1;   break;
         case EN_GDLADJ__STEERWHEELCALIB_F1:           pstVehicleParameter->dSWAtoFWA_Coeff1       =         f64Value * 1e-5;  break;
         case EN_GDLADJ__STEERWHEELCALIB_F3:           pstVehicleParameter->dSWAtoFWA_Coeff3       =         f64Value * 1e-10; break;
         case EN_GDLADJ__MAXSTEERWHEELANGLE:           pstVehicleParameter->dMaxSteerWhlAngle      =         f64Value;         break;
         case EN_GDLADJ__MAXREARWHEELANGLE:            pstVehicleParameter->dMaxRearTireAngle      =         f64Value * 0.01;  break;
         case EN_GDLADJ__RADIUSTUNINGFACT:             pstVehicleParameter->dRadiusTuningFact      =         f64Value * 0.01;  break;
         case EN_GDLADJ__FOURWHEELSTEERING:            pstVehicleParameter->bFourWheelSteering     = (tBool) f64Value;         break;
         case EN_GDLADJ__TIREANGLETURNSENSE:           pstVehicleParameter->bPosTireAngleIsCcw     = (tBool) f64Value;         break;
         case EN_GDLADJ__TIREANGLEREFERENCEPOINT:      pstVehicleParameter->u8TireAngleRefPoint    = (tU8)   f64Value;         break;
         #if(0)  // we will *not* allow to modify the guideline style at run-time, since the activation status of some other items depend on it
         case EN_GDLADJ__GUIDELINESTYLE:               pstVehicleParameter->u8GuidelnStyle         = (tU8)   f64Value;         break;
         #endif
         case EN_GDLADJ__CAMXOFFSET:                   pstVehicleParameter->dCamXOffs              =         f64Value * 0.1;   break;
         case EN_GDLADJ__CAMYOFFSET:                   pstVehicleParameter->dCamYOffs              =         f64Value * 0.1;   break;
         case EN_GDLADJ__CAMZOFFSET:                   pstVehicleParameter->dCamZOffs              =         f64Value * 0.1;   break;
         case EN_GDLADJ__CAMYAW:                       pstVehicleParameter->dCamYaw                =         f64Value * 0.1;   break;
         case EN_GDLADJ__CAMPITCH:                     pstVehicleParameter->dCamPitch              =         f64Value * 0.1;   break;
         case EN_GDLADJ__CAMROLL:                      pstVehicleParameter->dCamRoll               =         f64Value * 0.1;   break;
         case EN_GDLADJ__FRONTTIRETHRESH:              pstVehicleParameter->dFrontTireThresh       =         f64Value * 0.01;  break;
         case EN_GDLADJ__MINFRONTTIRECHANGE:           pstVehicleParameter->dMinFrontTireChg       =         f64Value * 0.01;  break;
         case EN_GDLADJ__REARTIRETHRESH:               pstVehicleParameter->dRearTireThresh        =         f64Value * 0.01;  break;
         case EN_GDLADJ__MINREARTIRECHANGE:            pstVehicleParameter->dMinRearTireChg        =         f64Value * 0.01;  break;
         case EN_GDLADJ__GUIDELINESTARTDISTANCE:       pstVehicleParameter->dGuidelnStartDist      =         f64Value;         break;
         case EN_GDLADJ__GUIDELINESIDEMARGIN:          pstVehicleParameter->dGuidelnLatMargin      =         f64Value;         break;
         case EN_GDLADJ__RGBCOLORSTATICSIDENEAR_RED:   pstVehicleParameter->au8GuidelnRgb[0][0]    = (tU8)   f64Value;         break;
         case EN_GDLADJ__RGBCOLORSTATICSIDENEAR_GREEN: pstVehicleParameter->au8GuidelnRgb[0][1]    = (tU8)   f64Value;         break;
         case EN_GDLADJ__RGBCOLORSTATICSIDENEAR_BLUE:  pstVehicleParameter->au8GuidelnRgb[0][2]    = (tU8)   f64Value;         break;
         case EN_GDLADJ__RGBCOLORSTATICSIDEMID_RED:    pstVehicleParameter->au8GuidelnRgb[1][0]    = (tU8)   f64Value;         break;
         case EN_GDLADJ__RGBCOLORSTATICSIDEMID_GREEN:  pstVehicleParameter->au8GuidelnRgb[1][1]    = (tU8)   f64Value;         break;
         case EN_GDLADJ__RGBCOLORSTATICSIDEMID_BLUE:   pstVehicleParameter->au8GuidelnRgb[1][2]    = (tU8)   f64Value;         break;
         case EN_GDLADJ__RGBCOLORSTATICSIDEFAR_RED:    pstVehicleParameter->au8GuidelnRgb[2][0]    = (tU8)   f64Value;         break;
         case EN_GDLADJ__RGBCOLORSTATICSIDEFAR_GREEN:  pstVehicleParameter->au8GuidelnRgb[2][1]    = (tU8)   f64Value;         break;
         case EN_GDLADJ__RGBCOLORSTATICSIDEFAR_BLUE:   pstVehicleParameter->au8GuidelnRgb[2][2]    = (tU8)   f64Value;         break;
         case EN_GDLADJ__RGBCOLORSTATICCENTER_RED:     pstVehicleParameter->au8GuidelnRgb[3][0]    = (tU8)   f64Value;         break;
         case EN_GDLADJ__RGBCOLORSTATICCENTER_GREEN:   pstVehicleParameter->au8GuidelnRgb[3][1]    = (tU8)   f64Value;         break;
         case EN_GDLADJ__RGBCOLORSTATICCENTER_BLUE:    pstVehicleParameter->au8GuidelnRgb[3][2]    = (tU8)   f64Value;         break;
         case EN_GDLADJ__RGBCOLORDYNAMIC_RED:          pstVehicleParameter->au8GuidelnRgb[4][0]    = (tU8)   f64Value;         break;
         case EN_GDLADJ__RGBCOLORDYNAMIC_GREEN:        pstVehicleParameter->au8GuidelnRgb[4][1]    = (tU8)   f64Value;         break;
         case EN_GDLADJ__RGBCOLORDYNAMIC_BLUE:         pstVehicleParameter->au8GuidelnRgb[4][2]    = (tU8)   f64Value;         break;
         case EN_GDLADJ__LINEWIDTHSTATICSIDENEAR:      pstVehicleParameter->astLineWidth[0].u8Near = (tU8)   f64Value;         break;
         case EN_GDLADJ__LINEWIDTHSTATICSIDEFAR:       pstVehicleParameter->astLineWidth[0].u8Far  = (tU8)   f64Value;         break;
         case EN_GDLADJ__LINEWIDTHDYNAMICNEAR:         pstVehicleParameter->astLineWidth[1].u8Near = (tU8)   f64Value;         break;
         case EN_GDLADJ__LINEWIDTHDYNAMICFAR:          pstVehicleParameter->astLineWidth[1].u8Far  = (tU8)   f64Value;         break;
         case EN_GDLADJ__LINEWIDTHSTATICCENTERNEAR:    pstVehicleParameter->astLineWidth[2].u8Near = (tU8)   f64Value;         break;
         case EN_GDLADJ__LINEWIDTHSTATICCENTERFAR:     pstVehicleParameter->astLineWidth[2].u8Far  = (tU8)   f64Value;         break;
         case EN_GDLADJ__STATICLINELENGTHNEAR:         pstVehicleParameter->au8GuidelnStatLen[0]   = (tU8)   f64Value;         break;
         case EN_GDLADJ__STATICLINELENGTHMID:          pstVehicleParameter->au8GuidelnStatLen[1]   = (tU8)   f64Value;         break;
         case EN_GDLADJ__STATICLINELENGHTFAR1:         pstVehicleParameter->au8GuidelnStatLen[2]   = (tU8)   f64Value;         break;
         case EN_GDLADJ__STATICLINELENGHTFAR2:         pstVehicleParameter->au8GuidelnStatLen[3]   = (tU8)   f64Value;         break;
         case EN_GDLADJ__STATICHORIZLINELENGHT:        pstVehicleParameter->dGuidelnLenHorBar      = (tU8)   f64Value;         break;
         case EN_GDLADJ__LINEWIDTHFACTORNEARHLINE:     pstVehicleParameter->fLineWdthWgtNearHbar   = (tF32) (f64Value * 0.01); break;
         #ifdef GDL_ADJ__ALLOW_ACCESS_TO_TUNING_PARAMS         
         case EN_GDLADJ__TUNINGMETHOD:                 pstVehicleParameter->stTuneParams.u8Method                 = (tU8)   f64Value;         break;
         case EN_GDLADJ__TUNINGDELTACAMYAW:            pstVehicleParameter->stTuneParams.dCamYaw                  =         f64Value;         break;
         case EN_GDLADJ__TUNINGDELTACAMPITCH:          pstVehicleParameter->stTuneParams.dCamPitch                =         f64Value;         break;
         case EN_GDLADJ__TUNINGDELTACAMROLL:           pstVehicleParameter->stTuneParams.dCamRoll                 =         f64Value;         break;
         case EN_GDLADJ__TUNINGDELTAXNEARLEFT:         pstVehicleParameter->stTuneParams.stOffsLft.dNearDx        =         f64Value;         break;
         case EN_GDLADJ__TUNINGDELTAYNEARLEFT:         pstVehicleParameter->stTuneParams.stOffsLft.dNearDy        =         f64Value;         break;
         case EN_GDLADJ__TUNINGDELTAXFARLEFT:          pstVehicleParameter->stTuneParams.stOffsLft.dFarDx         =         f64Value;         break;
         case EN_GDLADJ__TUNINGNEARLINELENGTHLEFT:     pstVehicleParameter->stTuneParams.stOffsLft.dNearLineLen   =         f64Value;         break;
         case EN_GDLADJ__TUNINGDELTAXNEARRIGHT:        pstVehicleParameter->stTuneParams.stOffsRgt.dNearDx        =         f64Value;         break;
         case EN_GDLADJ__TUNINGDELTAYNEARRIGHT:        pstVehicleParameter->stTuneParams.stOffsRgt.dNearDy        =         f64Value;         break;
         case EN_GDLADJ__TUNINGDELTAXFARRIGHT:         pstVehicleParameter->stTuneParams.stOffsRgt.dFarDx         =         f64Value;         break;
         case EN_GDLADJ__TUNINGNEARLINELENGHTRIGHT:    pstVehicleParameter->stTuneParams.stOffsRgt.dNearLineLen   =         f64Value;         break;
         case EN_GDLADJ__TUNINGSWAZEROPOINTOFFSET:     pstVehicleParameter->stTuneParams.dSwaZeroOffset           =         f64Value;         break;
         case EN_GDLADJ__TUNINGRWAZEROPOINTOFFSET:     pstVehicleParameter->stTuneParams.dRwaZeroOffset           =         f64Value;         break;
         #endif
         #ifdef GDL_ADJ__ALLOW_CAMERA_TYPE_SELECTION
         case EN_GDLADJ__RVCLENSPARAMETER:             pstVehicleParameter->u8CameraType           = (tU8)  f64Value;          break;
         #endif
         case EN_GDLADJ__LINEWIDTHFACTORVERTLINES:     pstVehicleParameter->fLineWdthWgtVertLines  = (tF32) (f64Value * 0.01); break;
         default:
            bReturnValue = FALSE;
            break;
      }
   }

   return (bReturnValue);
}

/******************************************************************************/
/* FUNCTION     _bGdlAdj_DoRangeCorrectionForElementDataValue()               */
/******************************************************************************/
/**
*  \brief       to correct the range of a value which was set from HMI
*
*  \param
*  \return
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Video::_bGdlAdj_DoRangeCorrectionForElementDataValue(tU8 u8Item, tF64& f64Value)
{
   tBool bReturnValue = TRUE;

   switch ((tenControl_Video_GdlAdjConfigurationDataItem) u8Item)
   {
      case EN_GDLADJ__ANTICIPATORYLINE:              GDL_ADJ__LIMIT_TO_RANGE__BOOL(f64Value)           break; //m_bAnticipatoryLine
      case EN_GDLADJ__VEHICLEWIDTH:                                                                    break; //pstVehicleParameter->dVehWidth
      case EN_GDLADJ__WHEELBASE:                                                                       break; //pstVehicleParameter->dWheelBase
      case EN_GDLADJ__REAROVERHANG:                                                                    break; //pstVehicleParameter->dRearOverhang
      case EN_GDLADJ__STEERWHEELCALIB_F1:                                                              break; //pstVehicleParameter->dSWAtoFWA_Coeff1
      case EN_GDLADJ__STEERWHEELCALIB_F3:                                                              break; //pstVehicleParameter->dSWAtoFWA_Coeff3
      case EN_GDLADJ__MAXSTEERWHEELANGLE:                                                              break; //pstVehicleParameter->dMaxSteerWhlAngle;
      case EN_GDLADJ__MAXREARWHEELANGLE:                                                               break; //pstVehicleParameter->dMaxRearTireAngle
      case EN_GDLADJ__RADIUSTUNINGFACT:                                                                break; //pstVehicleParameter->dRadiusTuningFact
      case EN_GDLADJ__FOURWHEELSTEERING:             GDL_ADJ__LIMIT_TO_RANGE__BOOL(f64Value)           break; //pstVehicleParameter->bFourWheelSteering
      case EN_GDLADJ__TIREANGLETURNSENSE:            GDL_ADJ__LIMIT_TO_RANGE__BOOL(f64Value)           break; //pstVehicleParameter->bPosTireAngleIsCcw
      case EN_GDLADJ__TIREANGLEREFERENCEPOINT:       GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->u8TireAngleRefPoint
      #if(0)  // we will *not* allow to modify the guideline style at run-time, since the activation status of some other items depend on it
      case EN_GDLADJ__GUIDELINESTYLE:                GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->u8GuidelnStyle
      #endif
      case EN_GDLADJ__CAMXOFFSET:                                                                      break; //pstVehicleParameter->dCamXOffs
      case EN_GDLADJ__CAMYOFFSET:                                                                      break; //pstVehicleParameter->dCamYOffs
      case EN_GDLADJ__CAMZOFFSET:                                                                      break; //pstVehicleParameter->dCamZOffs
      case EN_GDLADJ__CAMYAW:                                                                          break; //pstVehicleParameter->dCamYaw
      case EN_GDLADJ__CAMPITCH:                                                                        break; //pstVehicleParameter->dCamPitch
      case EN_GDLADJ__CAMROLL:                                                                         break; //pstVehicleParameter->dCamRoll
      case EN_GDLADJ__FRONTTIRETHRESH:                                                                 break; //pstVehicleParameter->dFrontTireThresh
      case EN_GDLADJ__MINFRONTTIRECHANGE:                                                              break; //pstVehicleParameter->dMinFrontTireChg
      case EN_GDLADJ__REARTIRETHRESH:                                                                  break; //pstVehicleParameter->dRearTireThresh
      case EN_GDLADJ__MINREARTIRECHANGE:                                                               break; //pstVehicleParameter->dMinRearTireChg
      case EN_GDLADJ__GUIDELINESTARTDISTANCE:                                                          break; //pstVehicleParameter->dGuidelnStartDist
      case EN_GDLADJ__GUIDELINESIDEMARGIN:                                                             break; //pstVehicleParameter->dGuidelnLatMargin
      case EN_GDLADJ__RGBCOLORSTATICSIDENEAR_RED:    GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->au8GuidelnRgb[0][0]
      case EN_GDLADJ__RGBCOLORSTATICSIDENEAR_GREEN:  GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->au8GuidelnRgb[0][1]
      case EN_GDLADJ__RGBCOLORSTATICSIDENEAR_BLUE:   GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->au8GuidelnRgb[0][2]
      case EN_GDLADJ__RGBCOLORSTATICSIDEMID_RED:     GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->au8GuidelnRgb[1][0]
      case EN_GDLADJ__RGBCOLORSTATICSIDEMID_GREEN:   GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->au8GuidelnRgb[1][1]
      case EN_GDLADJ__RGBCOLORSTATICSIDEMID_BLUE:    GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->au8GuidelnRgb[1][2]
      case EN_GDLADJ__RGBCOLORSTATICSIDEFAR_RED:     GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->au8GuidelnRgb[2][0]
      case EN_GDLADJ__RGBCOLORSTATICSIDEFAR_GREEN:   GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->au8GuidelnRgb[2][1]
      case EN_GDLADJ__RGBCOLORSTATICSIDEFAR_BLUE:    GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->au8GuidelnRgb[2][2]
      case EN_GDLADJ__RGBCOLORSTATICCENTER_RED:      GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->au8GuidelnRgb[3][0]
      case EN_GDLADJ__RGBCOLORSTATICCENTER_GREEN:    GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->au8GuidelnRgb[3][1]
      case EN_GDLADJ__RGBCOLORSTATICCENTER_BLUE:     GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->au8GuidelnRgb[3][2]
      case EN_GDLADJ__RGBCOLORDYNAMIC_RED:           GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->au8GuidelnRgb[4][0]
      case EN_GDLADJ__RGBCOLORDYNAMIC_GREEN:         GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->au8GuidelnRgb[4][1]
      case EN_GDLADJ__RGBCOLORDYNAMIC_BLUE:          GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->au8GuidelnRgb[4][2]
      case EN_GDLADJ__LINEWIDTHSTATICSIDENEAR:       GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->astLineWidth[0].u8Near
      case EN_GDLADJ__LINEWIDTHSTATICSIDEFAR:        GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->astLineWidth[0].u8Far
      case EN_GDLADJ__LINEWIDTHDYNAMICNEAR:          GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->astLineWidth[1].u8Near
      case EN_GDLADJ__LINEWIDTHDYNAMICFAR:           GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->astLineWidth[1].u8Far
      case EN_GDLADJ__LINEWIDTHSTATICCENTERNEAR:     GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->astLineWidth[2].u8Near
      case EN_GDLADJ__LINEWIDTHSTATICCENTERFAR:      GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->astLineWidth[2].u8Far
      case EN_GDLADJ__STATICLINELENGTHNEAR:          GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->au8GuidelnStatLen[0]
      case EN_GDLADJ__STATICLINELENGTHMID:           GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->au8GuidelnStatLen[1]
      case EN_GDLADJ__STATICLINELENGHTFAR1:          GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->au8GuidelnStatLen[2]
      case EN_GDLADJ__STATICLINELENGHTFAR2:          GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->au8GuidelnStatLen[3]
      case EN_GDLADJ__STATICHORIZLINELENGHT:                                                           break; //pstVehicleParameter->dGuidelnLenHorBar
      case EN_GDLADJ__LINEWIDTHFACTORNEARHLINE:                                                        break; //pstVehicleParameter->fLineWdthWgtNearHbar
      #ifdef GDL_ADJ__ALLOW_ACCESS_TO_TUNING_PARAMS
      case EN_GDLADJ__TUNINGMETHOD:                  GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->stTuneParams.u8Method
      case EN_GDLADJ__TUNINGDELTACAMYAW:                                                               break; //pstVehicleParameter->stTuneParams.dCamYaw
      case EN_GDLADJ__TUNINGDELTACAMPITCH:                                                             break; //pstVehicleParameter->stTuneParams.dCamPitch
      case EN_GDLADJ__TUNINGDELTACAMROLL:                                                              break; //pstVehicleParameter->stTuneParams.dCamRoll
      case EN_GDLADJ__TUNINGDELTAXNEARLEFT:                                                            break; //pstVehicleParameter->stTuneParams.stOffsLft.dNearDx
      case EN_GDLADJ__TUNINGDELTAYNEARLEFT:                                                            break; //pstVehicleParameter->stTuneParams.stOffsLft.dNearDy
      case EN_GDLADJ__TUNINGDELTAXFARLEFT:                                                             break; //pstVehicleParameter->stTuneParams.stOffsLft.dFarDx
      case EN_GDLADJ__TUNINGNEARLINELENGTHLEFT:                                                        break; //pstVehicleParameter->stTuneParams.stOffsLft.dNearLineLen
      case EN_GDLADJ__TUNINGDELTAXNEARRIGHT:                                                           break; //pstVehicleParameter->stTuneParams.stOffsRgt.dNearDx
      case EN_GDLADJ__TUNINGDELTAYNEARRIGHT:                                                           break; //pstVehicleParameter->stTuneParams.stOffsRgt.dNearDy
      case EN_GDLADJ__TUNINGDELTAXFARRIGHT:                                                            break; //pstVehicleParameter->stTuneParams.stOffsRgt.dFarDx
      case EN_GDLADJ__TUNINGNEARLINELENGHTRIGHT:                                                       break; //pstVehicleParameter->stTuneParams.stOffsRgt.dNearLineLen
      case EN_GDLADJ__TUNINGSWAZEROPOINTOFFSET:                                                        break; //pstVehicleParameter->stTuneParams.dSwaZeroOffset
      case EN_GDLADJ__TUNINGRWAZEROPOINTOFFSET:                                                        break; //pstVehicleParameter->stTuneParams.dRwaZeroOffset
      #endif
      #ifdef GDL_ADJ__ALLOW_CAMERA_TYPE_SELECTION
      case EN_GDLADJ__RVCLENSPARAMETER:              GDL_ADJ__LIMIT_TO_RANGE__UNSIGNED_BYTE(f64Value)  break; //pstVehicleParameter->u8CameraType
      #endif
      case EN_GDLADJ__LINEWIDTHFACTORVERTLINES:                                                        break; //pstVehicleParameter->fLineWdthWgtVertLines
      default:
         bReturnValue = FALSE;
         break;
   }

   return (bReturnValue);
}


/******************************************************************************/
/* FUNCTION     _vGdlAdj_TraceInfo()                                          */
/******************************************************************************/
/**
*  \brief       trace information
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::_vGdlAdj_TraceInfo(tVoid)
{
   ETG_TRACE_FATAL(("   CA-VID :: vTraceInfo() ************ CA-VID states ****************************************"));
   ETG_TRACE_FATAL(("   CA-VID :: m_GdlAdj_bCorrectDrawLineScreen_Active .... %-40u", m_GdlAdj_bCorrectDrawLineScreen_Active));
   ETG_TRACE_FATAL(("   CA-VID :: *********************************************************************************"));
}


