/******************************************************************************/
/**
* \file    dispvidctrl_tclControl_Video.cpp
* \ingroup
*
* \brief
*
* \remark  Copyright : (c) 2015 Robert Bosch GmbH, Hildesheim
* \remark  Author    : Ian Lisney, Michael Niemann
* \remark  Scope     : AIVI
*
*/
/******************************************************************************/

/*******************************************************************************
                        Includes
*******************************************************************************/
#include "dispvidctrl_tclControl.h"
#include "dispvidctrl_tclControl_Video.h"
#include "Idispvidctrl_ClientVcan.h"
#include "Idispvidctrl_ClientEarly.h"
#include "dispvidctrl_tclPersistentData.h"
#include "I_dispvidctrl_tclSyncHandler.h"
#include "dispvidctrl_datapool.h"
#include "dispvidctrl_tclControl_Hmi.h"
#include "I_dispvidctrl_ServiceRvcIf.h"
#include "dispvidctrl_tclControl_Mvc.h"

#include "vd_rvc_tclGraphics_DynGuideline.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_APPLICATION
#include "trcGenProj/Header/dispvidctrl_tclControl_Video.cpp.trc.h"
#endif

/*******************************************************************************
                        Defines
*******************************************************************************/
// defines for mapping the user setting value 0..100% to the range used from the
// register (start, middle and end point)
//------------------------------------------------------------------------------
#define DISPVIDCTRL_VIDEO__BRIGHTNESS_MIN_VALUE_DAY   (tU8)  0
#define DISPVIDCTRL_VIDEO__BRIGHTNESS_MID_VALUE_DAY   (tU8)  30
#define DISPVIDCTRL_VIDEO__BRIGHTNESS_MAX_VALUE_DAY   (tU8)  60
#define DISPVIDCTRL_VIDEO__BRIGHTNESS_MIN_VALUE_NIGHT (tU8)  0
#define DISPVIDCTRL_VIDEO__BRIGHTNESS_MID_VALUE_NIGHT (tU8)  30
#define DISPVIDCTRL_VIDEO__BRIGHTNESS_MAX_VALUE_NIGHT (tU8)  60
//------------------------------------------------------------------------------
#define DISPVIDCTRL_VIDEO__CONTRAST_MIN_VALUE_DAY     (tU8)  15
#define DISPVIDCTRL_VIDEO__CONTRAST_MID_VALUE_DAY     (tU8)  50
#define DISPVIDCTRL_VIDEO__CONTRAST_MAX_VALUE_DAY     (tU8)  85
#define DISPVIDCTRL_VIDEO__CONTRAST_MIN_VALUE_NIGHT   (tU8)  15
#define DISPVIDCTRL_VIDEO__CONTRAST_MID_VALUE_NIGHT   (tU8)  50
#define DISPVIDCTRL_VIDEO__CONTRAST_MAX_VALUE_NIGHT   (tU8)  85
//------------------------------------------------------------------------------
#define DISPVIDCTRL_VIDEO__COLOUR_MIN_VALUE_DAY       (tU8)   0
#define DISPVIDCTRL_VIDEO__COLOUR_MID_VALUE_DAY       (tU8)  50
#define DISPVIDCTRL_VIDEO__COLOUR_MAX_VALUE_DAY       (tU8) 100
#define DISPVIDCTRL_VIDEO__COLOUR_MIN_VALUE_NIGHT     (tU8)   0
#define DISPVIDCTRL_VIDEO__COLOUR_MID_VALUE_NIGHT     (tU8)  50
#define DISPVIDCTRL_VIDEO__COLOUR_MAX_VALUE_NIGHT     (tU8) 100
//------------------------------------------------------------------------------
#define DISPVIDCTRL_VIDEO__BLACKLEVEL_MIN_VALUE_DAY   (tU8)   0
#define DISPVIDCTRL_VIDEO__BLACKLEVEL_MID_VALUE_DAY   (tU8)   20
#define DISPVIDCTRL_VIDEO__BLACKLEVEL_MAX_VALUE_DAY   (tU8)   40
#define DISPVIDCTRL_VIDEO__BLACKLEVEL_MIN_VALUE_NIGHT (tU8)   0
#define DISPVIDCTRL_VIDEO__BLACKLEVEL_MID_VALUE_NIGHT (tU8)   20
#define DISPVIDCTRL_VIDEO__BLACKLEVEL_MAX_VALUE_NIGHT (tU8)   40
//------------------------------------------------------------------------------
#define DISPVIDCTRL_VIDEO__BACKLIGHT_DIMMING_MIN_VALUE_DAY   (tU8)   0
#define DISPVIDCTRL_VIDEO__BACKLIGHT_DIMMING_MID_VALUE_DAY   (tU8)  50
#define DISPVIDCTRL_VIDEO__BACKLIGHT_DIMMING_MAX_VALUE_DAY   (tU8) 100
#define DISPVIDCTRL_VIDEO__BACKLIGHT_DIMMING_MIN_VALUE_NIGHT (tU8)   0
#define DISPVIDCTRL_VIDEO__BACKLIGHT_DIMMING_MID_VALUE_NIGHT (tU8)  50
#define DISPVIDCTRL_VIDEO__BACKLIGHT_DIMMING_MAX_VALUE_NIGHT (tU8) 100
//------------------------------------------------------------------------------
#define DISPVIDCTRL_VIDEO__CAMERA_STEERING_MESSAGE_TYPE__002 (tU8) 0x00 // CAN message ID 002
#define DISPVIDCTRL_VIDEO__CAMERA_STEERING_MESSAGE_TYPE__0C1 (tU8) 0x01 // CAM message ID 0C1

#define DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED (tU8) 0xFF

#define DISPVIDCTRL_VIDEO__STEERING_ANGLE_ADJUST__RECHECK_CONST_WITH_BACKUP    360000
#define DISPVIDCTRL_VIDEO__STEERING_ANGLE_ADJUST__RECHECK_CONST_NO_BACKUP       45000
#define DISPVIDCTRL_VIDEO__STEERING_ANGLE_ADJUST__RECHECK_TIME_NO_ADJUST_IN_MS   1000


// Lookup table for video settings used for mapping the HMI user setting value to DevVideo input value (ADV register value))
const static StVideoSettingHmiToDevVideoValueMapping stVideoSettingHmiToDevVideoValueMapping[] =
{
        //    HMI,       Hue    Bright,   Black,     Color,   Contrast
        {(tU8)  0, (tU8) 176, (tU8) 15, (tU8)  5, (tU8)  0, (tU8)  15},
        {(tU8)  5, (tU8) 170, (tU8) 17, (tU8)  7, (tU8)  5, (tU8)  18},
        {(tU8) 10, (tU8) 164, (tU8) 18, (tU8)  8, (tU8) 10, (tU8)  22},
        {(tU8) 15, (tU8) 159, (tU8) 20, (tU8) 10, (tU8) 15, (tU8)  25},
        {(tU8) 20, (tU8) 154, (tU8) 21, (tU8) 11, (tU8) 20, (tU8)  29},
        {(tU8) 25, (tU8) 149, (tU8) 23, (tU8) 13, (tU8) 25, (tU8)  32},
        {(tU8) 30, (tU8) 144, (tU8) 24, (tU8) 14, (tU8) 30, (tU8)  36},
        {(tU8) 35, (tU8) 139, (tU8) 26, (tU8) 16, (tU8) 35, (tU8)  39},
        {(tU8) 40, (tU8) 134, (tU8) 27, (tU8) 17, (tU8) 40, (tU8)  43},
        {(tU8) 45, (tU8) 130, (tU8) 29, (tU8) 19, (tU8) 45, (tU8)  46},
        {(tU8) 50, (tU8) 127, (tU8) 30, (tU8) 20, (tU8) 50, (tU8)  50}, // 0
        {(tU8) 55, (tU8) 124, (tU8) 33, (tU8) 22, (tU8) 55, (tU8)  53},
        {(tU8) 60, (tU8) 120, (tU8) 35, (tU8) 23, (tU8) 60, (tU8)  57},
        {(tU8) 65, (tU8) 115, (tU8) 38, (tU8) 25, (tU8) 65, (tU8)  60},
        {(tU8) 70, (tU8) 110, (tU8) 40, (tU8) 26, (tU8) 70, (tU8)  64},
        {(tU8) 75, (tU8) 105, (tU8) 43, (tU8) 28, (tU8) 75, (tU8)  67},
        {(tU8) 80, (tU8) 100, (tU8) 45, (tU8) 29, (tU8) 80, (tU8)  71},
        {(tU8) 85, (tU8)  95, (tU8) 48, (tU8) 31, (tU8) 85, (tU8)  74},
        {(tU8) 90, (tU8)  90, (tU8) 50, (tU8) 32, (tU8) 90, (tU8)  78},
        {(tU8) 95, (tU8)  84, (tU8) 53, (tU8) 34, (tU8) 95, (tU8)  81},
        {(tU8)100, (tU8)  78, (tU8) 55, (tU8) 35, (tU8)100, (tU8)  85}
};
#define DISPVIDCTRL_DATA_VIDEO_SETTING_MAPPING_TABLE_SIZE  (sizeof(stVideoSettingHmiToDevVideoValueMapping)/sizeof(stVideoSettingHmiToDevVideoValueMapping[0]))


/******************************************************************************/
/* FUNCTION     dispvidctrl_tclControl_Video()                                */
/******************************************************************************/
/**
*  \brief       default constructor
*
*  \param       pointer to main application
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclControl_Video::dispvidctrl_tclControl_Video()
: dispvidctrl_tclBaseIf(OSAL_NULL)
, m_poClientVcan(OSAL_NULL)
, m_poClient_DevVideo(OSAL_NULL)
, m_poClient_VideoPlayer(OSAL_NULL)
, m_poPersistentData(OSAL_NULL)
, m_poControl_Hmi(OSAL_NULL)
, m_poControl_Mvc(OSAL_NULL)
, m_poServiceRvcIf(OSAL_NULL)
, m_hTimerHandle_DynGuidelineUpdate(OSAL_C_INVALID_HANDLE)
, m_hTimerHandle_SteeringAngleAdjustment(OSAL_C_INVALID_HANDLE)
, m_hTimerHandle_SteeringAngleGuidelineAdjustment(OSAL_C_INVALID_HANDLE)
//-------------------------------------------
, m_u8CurrentGuidelineIndex(0)
, m_u8MaxGuidelineTogglesAvailable(0)
, m_u8DisplayOrientation(DISPVIDCTRL_AllianceDisplay_Format_Landscape)
, m_u32DisplayDimmingMode(DISPVIDCTRL_VIDEO__DISPLAY_DIMMING_MODE_DAY)
, m_u32VideoBrightnessDay(0)
, m_u32VideoBrightnessNight(0)
, m_u32VideoContrastDay(0)
, m_u32VideoContrastNight(0)
, m_u32VideoColourDay(0)
, m_u32VideoColourNight(0)
, m_u32VideoHueDay(0)
, m_u32VideoHueNight(0)
, m_u32VideoBlackLevelDay(0)
, m_u32VideoBlackLevelNight(0)
, m_u32VideoDimmingDay(0)
, m_u32VideoDimmingNight(0)
, m_u32GuidelineWarningLanguage(0)
, m_u32GuidelineStaticSide(0)
, m_u32GuidelineDynamic(0)
//------------------------------ runtime states and values
, m_bDiagScreenRequest(FALSE)
, m_bGraphicsVisibility(TRUE)
, rvc_guard_time_remaining(0)
, rvc_video_active(FALSE)
, rvc_video_status(FALSE)
, u32CurrentAnimationStatus(0)
, m_SteeringAngleAdjust_bIgnition(FALSE)
, m_SteeringAngleAdjust_bRecheck(FALSE)
, m_SteeringAngleAdjust_bBackupAvailable(FALSE)
, m_SteeringAngleAdjust_u32RecheckTimeInMs(1000)
, m_SteeringAngleAdjust_s32SteeringAngleOffset(0)
, m_SteeringAngleAdjust_bActive(FALSE)
, m_bAnticipatoryLine(FALSE)
//------------------------------ system (KDS) configuration info
, m_Config_bAvmMode(FALSE)
, m_Config_bAvmRenaultMode(FALSE)
, m_Config_bRvcMode(FALSE)
, m_Config_bRvcRenaultMode(FALSE)
, m_Config_bMvcMode(FALSE)
, m_Config_bCentreGuideline(FALSE)
, m_Config_bAnticipatoryLine(FALSE)
, m_Config_u8CameraSteeringMessageType(0)
, m_Config_bLeftHandSteeringPosition(FALSE)
, m_Config_u8BrightnessValueDay(0)
, m_Config_u8BrightnessValueNight(0)
, m_Config_bBlackWarningTextBackground(0)
, m_Config_u8RvcLensParameterSet(0)
, m_Config_u8Brand(0xFF) // configured to brand none
, m_Config_u16VehicleType(DISPVIDCTRL_VEHICLE_TYPE_UNKNOWN)
//------------------------------ HMI user setting values
, m_HmiUserSetting_u8GuidelineStaticSide(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8GuidelineDynamic(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8Language(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8DisplayDimmingMode(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8VideoBlackLevelDay(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8VideoBlackLevelNight(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8VideoBrightnessDay(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8VideoBrightnessNight(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8VideoColourDay(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8VideoColourNight(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8VideoContrastDay(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8VideoContrastNight(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8VideoHueDay(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8VideoHueNight(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8VideoDimmingDay(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8VideoDimmingNight(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
//------------------------------ Guideline Adjustment
, m_GdlAdj_bCorrectDrawLineScreen_Active(FALSE)
, m_f32CameraRollAngle(0)
, m_f32CameraYawAngle(0)
, m_f32CameraPitchAngle(0)
, m_u8ConfiguredVideoType(IDX_MAX)
{
   // set default values to avoid lint issues
   // correct start values will be set with reading persistent data in vGetConfiguration()
   // resp. in _vGuidelineToggleSequence_Initialize() being called by vGetConfiguration()
   (tVoid) OSAL_pvMemorySet(m_Config_aVehicleParameter, 0x00, sizeof m_Config_aVehicleParameter);
   (tVoid) OSAL_pvMemorySet(m_au32GuidelineToggleSequence, 0x00, sizeof m_au32GuidelineToggleSequence);
}

/******************************************************************************/
/* FUNCTION     dispvidctrl_tclControl_Video()                                */
/******************************************************************************/
/**
*  \brief       constructor
*
*  \param       pointer to main application
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclControl_Video::dispvidctrl_tclControl_Video(dispvidctrl_tclAppMain* poMainAppl)
: dispvidctrl_tclBaseIf(poMainAppl)
, m_poClientVcan(OSAL_NULL)
, m_poClient_DevVideo(OSAL_NULL)
, m_poClient_VideoPlayer(OSAL_NULL)
, m_poPersistentData(OSAL_NULL)
, m_poControl_Hmi(OSAL_NULL)
, m_poControl_Mvc(OSAL_NULL)
, m_poServiceRvcIf(OSAL_NULL)
, m_hTimerHandle_DynGuidelineUpdate(OSAL_C_INVALID_HANDLE)
, m_hTimerHandle_SteeringAngleAdjustment(OSAL_C_INVALID_HANDLE)
, m_hTimerHandle_SteeringAngleGuidelineAdjustment(OSAL_C_INVALID_HANDLE)
//-------------------------------------------
, m_u8CurrentGuidelineIndex(0)
, m_u8MaxGuidelineTogglesAvailable(0)
, m_u8DisplayOrientation(DISPVIDCTRL_AllianceDisplay_Format_Landscape)
, m_u32DisplayDimmingMode(DISPVIDCTRL_VIDEO__DISPLAY_DIMMING_MODE_DAY)
, m_u32VideoBrightnessDay(0)
, m_u32VideoBrightnessNight(0)
, m_u32VideoContrastDay(0)
, m_u32VideoContrastNight(0)
, m_u32VideoColourDay(0)
, m_u32VideoColourNight(0)
, m_u32VideoHueDay(0)
, m_u32VideoHueNight(0)
, m_u32VideoBlackLevelDay(0)
, m_u32VideoBlackLevelNight(0)
, m_u32VideoDimmingDay(0)
, m_u32VideoDimmingNight(0)
, m_u32GuidelineWarningLanguage(0)
, m_u32GuidelineStaticSide(0)
, m_u32GuidelineDynamic(0)
//------------------------------ runtime states and values
, m_bDiagScreenRequest(FALSE)
, m_bGraphicsVisibility(TRUE)
, rvc_guard_time_remaining(0)
, rvc_video_active(FALSE)
, rvc_video_status(FALSE)
, u32CurrentAnimationStatus(0)
, m_SteeringAngleAdjust_bIgnition(FALSE)
, m_SteeringAngleAdjust_bRecheck(FALSE)
, m_SteeringAngleAdjust_bBackupAvailable(FALSE)
, m_SteeringAngleAdjust_u32RecheckTimeInMs(1000)
, m_SteeringAngleAdjust_s32SteeringAngleOffset(0)
, m_SteeringAngleAdjust_bActive(FALSE)
, m_bAnticipatoryLine(FALSE)
//------------------------------ system (KDS) configuration info
, m_Config_bAvmMode(FALSE)
, m_Config_bAvmRenaultMode(FALSE)
, m_Config_bRvcMode(FALSE)
, m_Config_bRvcRenaultMode(FALSE)
, m_Config_bMvcMode(FALSE)
, m_Config_bCentreGuideline(FALSE)
, m_Config_bAnticipatoryLine(FALSE)
, m_Config_u8CameraSteeringMessageType(0)
, m_Config_bLeftHandSteeringPosition(FALSE)
, m_Config_u8BrightnessValueDay(0)
, m_Config_u8BrightnessValueNight(0)
, m_Config_bBlackWarningTextBackground(0)
, m_Config_u8RvcLensParameterSet(0)
, m_Config_u8Brand(0xFF) // configured to brand none
, m_Config_u16VehicleType(DISPVIDCTRL_VEHICLE_TYPE_UNKNOWN)
//------------------------------ HMI user setting values
, m_HmiUserSetting_u8GuidelineStaticSide(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8GuidelineDynamic(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8Language(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8DisplayDimmingMode(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8VideoBlackLevelDay(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8VideoBlackLevelNight(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8VideoBrightnessDay(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8VideoBrightnessNight(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8VideoColourDay(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8VideoColourNight(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8VideoContrastDay(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8VideoContrastNight(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8VideoHueDay(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8VideoHueNight(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8VideoDimmingDay(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
, m_HmiUserSetting_u8VideoDimmingNight(DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
//------------------------------ Guideline Adjustment
, m_GdlAdj_bCorrectDrawLineScreen_Active(FALSE)
, m_f32CameraRollAngle(0)
, m_f32CameraYawAngle(0)
, m_f32CameraPitchAngle(0)
, m_u8ConfiguredVideoType(IDX_MAX)
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Video() entered."));

   // set default values to avoid lint issues
   // correct start values will be set with reading persistent data in vGetConfiguration()
   // resp. in _vGuidelineToggleSequence_Initialize() being called by vGetConfiguration()
   (tVoid) OSAL_pvMemorySet(m_Config_aVehicleParameter, 0x00, sizeof m_Config_aVehicleParameter);
   (tVoid) OSAL_pvMemorySet(m_au32GuidelineToggleSequence, 0x00, sizeof m_au32GuidelineToggleSequence);
}


/******************************************************************************/
/* FUNCTION     ~dispvidctrl_tclControl_Video                                 */
/******************************************************************************/
/**
*  \brief       destructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclControl_Video::~dispvidctrl_tclControl_Video(tVoid)
{
   ETG_TRACE_USR4(("~dispvidctrl_tclControl_Video() entered."));

   //stop and delete all timer
   if (OSAL_C_INVALID_HANDLE != m_hTimerHandle_DynGuidelineUpdate)
   {
      (tVoid) OSAL_s32TimerSetTime(m_hTimerHandle_DynGuidelineUpdate, 0, 0);
      (tVoid) OSAL_s32TimerDelete(m_hTimerHandle_DynGuidelineUpdate);
   }
   if (OSAL_C_INVALID_HANDLE != m_hTimerHandle_SteeringAngleAdjustment)
   {
      (tVoid) OSAL_s32TimerSetTime(m_hTimerHandle_SteeringAngleAdjustment, 0, 0);
      (tVoid) OSAL_s32TimerDelete(m_hTimerHandle_SteeringAngleAdjustment);
   }
   if (OSAL_C_INVALID_HANDLE != m_hTimerHandle_SteeringAngleGuidelineAdjustment)
   {
      (tVoid) OSAL_s32TimerSetTime(m_hTimerHandle_SteeringAngleGuidelineAdjustment, 0, 0);
      (tVoid) OSAL_s32TimerDelete(m_hTimerHandle_SteeringAngleGuidelineAdjustment);
   }

   m_poClientVcan = OSAL_NULL;
   m_poClient_DevVideo = OSAL_NULL;
   m_poClient_VideoPlayer = OSAL_NULL;
   m_poPersistentData = OSAL_NULL;
   m_poControl_Hmi = OSAL_NULL;
   m_poServiceRvcIf = (OSAL_NULL);
   m_poControl_Mvc = OSAL_NULL;
}

/******************************************************************************/
/* FUNCTION     vGetConfiguration                                             */
/******************************************************************************/
/**
*  \brief       Function to get all configuration values needed by this class.
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::vGetConfiguration(const TConfiguration* pStConfigurationValues)
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Video::vGetConfiguration() entered."));

   m_Config_bAvmMode           = pStConfigurationValues->bAvmMode;
   m_Config_bAvmRenaultMode    = pStConfigurationValues->bAvmType_Renault;
   m_Config_bRvcMode           = pStConfigurationValues->bRvcMode;
   m_Config_bRvcRenaultMode    = pStConfigurationValues->bRvcType_Renault;
   m_Config_bMvcMode           = pStConfigurationValues->bMvcMode;
   m_Config_bCentreGuideline   = pStConfigurationValues->bCentreGuideline;
   m_Config_u8Brand            = pStConfigurationValues->u8Brand;
   m_Config_u16VehicleType     = pStConfigurationValues->u16VehicleType;
   m_u8DisplayOrientation      = pStConfigurationValues->u8DisplayOrientation;
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE1
   //in Scope1 dynamic guideline feature not active
   m_Config_bAnticipatoryLine  = FALSE;
   m_bAnticipatoryLine = FALSE;
#else
   m_Config_bAnticipatoryLine  = pStConfigurationValues->bAnticipatoryLine;    // active state for dynamic guideline
   m_bAnticipatoryLine         = pStConfigurationValues->bAnticipatoryLine;    // active state for dynamic guideline
#endif
   m_Config_bLeftHandSteeringPosition    = pStConfigurationValues->bLeftHandSteeringPosition;
   m_Config_u8BrightnessValueDay         = pStConfigurationValues->u8BrightnessDay;
   m_Config_u8BrightnessValueNight       = pStConfigurationValues->u8BrightnessNight;
   m_Config_bBlackWarningTextBackground  = pStConfigurationValues->bBlackWarningTextBackground;
   m_Config_u8RvcLensParameterSet        = pStConfigurationValues->u8RvcLensParameterSet;
   (tVoid) OSAL_pvMemoryCopy(&m_Config_aVehicleParameter[0], &pStConfigurationValues->VehicleParameter[0], sizeof m_Config_aVehicleParameter);

   // Guidelines should be shown for both RVC and MVC
   if (m_Config_bRvcMode || m_Config_bMvcMode) {
      _vGuidelineToggleSequence_Initialize();
   }
}

/******************************************************************************/
/* FUNCTION     _vGuidelineToggleSequence_Initialize                          */
/******************************************************************************/
/**
*  \brief       Function to decide the guideline toggling sequence depending on
*               => Scope1, Scope2, Scope2.1 etc.
*               => Config item
*               Fill the array with the toggling sequence
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::_vGuidelineToggleSequence_Initialize()
{
   tU32 u32StaticCent_IfConfigured = (tU32)(m_Config_bCentreGuideline ? GUIDELINE_DRAW_STATIC_CENT : GUIDELINE_DRAW_NONE);
     // note: currently, m_Config_bCentreGuideline==TRUE only for the retrofitted vehicle Nissan H61L

#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE1

   // static (side or side+center) --> no guidelines    (in Scope1 dynamic guideline feature not active)
   m_au32GuidelineToggleSequence[0] = (tU32) GUIDELINE_DRAW_STATIC_SIDE | u32StaticCent_IfConfigured;
   m_au32GuidelineToggleSequence[1] = (tU32) GUIDELINE_DRAW_NONE;
   m_u8MaxGuidelineTogglesAvailable = 2;

#elif VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1 // Renault Specific

   if (m_Config_bRvcMode) {
      // Dynamic Guidelines config item is only meant for HMI Setting in Renault
      // So we enable it always when Rvc is configured
      // Dynamic Guidelines should be drawn only for RVC in Renault
      
      // static (side or side+center) + dynamic --> static (side or side+center) only --> dynamic only --> no guidelines
      m_au32GuidelineToggleSequence[0] = (tU32) GUIDELINE_DRAW_STATIC_SIDE | u32StaticCent_IfConfigured | (tU32) GUIDELINE_DRAW_DYNAMIC;
      m_au32GuidelineToggleSequence[1] = (tU32) GUIDELINE_DRAW_STATIC_SIDE | u32StaticCent_IfConfigured;
      m_au32GuidelineToggleSequence[2] = (tU32) GUIDELINE_DRAW_DYNAMIC;
      m_au32GuidelineToggleSequence[3] = (tU32) GUIDELINE_DRAW_NONE;
      m_u8MaxGuidelineTogglesAvailable = 4;
   } else {
      m_au32GuidelineToggleSequence[0] = (tU32) GUIDELINE_DRAW_STATIC_SIDE | u32StaticCent_IfConfigured;
      m_au32GuidelineToggleSequence[1] = (tU32) GUIDELINE_DRAW_NONE;
      m_u8MaxGuidelineTogglesAvailable = 2;
   }

#else // Nissan Specific VARIANT_S_FTR_ENABLE_AIVI_SCOPE2 || VARIANT_S_FTR_ENABLE_AIVI_NPIVI

   if (m_bAnticipatoryLine && m_Config_bRvcMode) {
      // Dynamic Guidelines should be drawn only for RVC
      if (m_Config_bCentreGuideline) {  // currently the case only for the retrofitted H61L
         m_au32GuidelineToggleSequence[0] = (tU32) GUIDELINE_DRAW_STATIC_SIDE | (tU32) GUIDELINE_DRAW_STATIC_CENT | (tU32) GUIDELINE_DRAW_DYNAMIC;
         m_au32GuidelineToggleSequence[1] = (tU32) GUIDELINE_DRAW_STATIC_SIDE                                     | (tU32) GUIDELINE_DRAW_DYNAMIC;
         m_au32GuidelineToggleSequence[2] = (tU32) GUIDELINE_DRAW_STATIC_SIDE | (tU32) GUIDELINE_DRAW_STATIC_CENT;
         m_au32GuidelineToggleSequence[3] = (tU32) GUIDELINE_DRAW_STATIC_SIDE;
         m_u8MaxGuidelineTogglesAvailable = 4;
      } else {
         m_au32GuidelineToggleSequence[0] = (tU32) GUIDELINE_DRAW_STATIC_SIDE                                     | (tU32) GUIDELINE_DRAW_DYNAMIC;
         m_au32GuidelineToggleSequence[1] = (tU32) GUIDELINE_DRAW_STATIC_SIDE;
         m_u8MaxGuidelineTogglesAvailable = 2;
      }
   } else {
      // no dynamic guidelines
      if (m_Config_bCentreGuideline) {  // currently the case only for the retrofitted H61L
         m_au32GuidelineToggleSequence[0] = (tU32) GUIDELINE_DRAW_STATIC_SIDE | (tU32) GUIDELINE_DRAW_STATIC_CENT;
         m_au32GuidelineToggleSequence[1] = (tU32) GUIDELINE_DRAW_STATIC_SIDE;
         m_u8MaxGuidelineTogglesAvailable = 2;
      } else {
         m_au32GuidelineToggleSequence[0] = (tU32) GUIDELINE_DRAW_STATIC_SIDE;
         m_u8MaxGuidelineTogglesAvailable = 1;
      }
   }
   
   // In Scope2 Static Side Guidelines should always be displayed, independent of config   
   vHandleControl(EN_CONTROL_VIDEO__USER_SETTING_VALUE__GUIDELINE_STATIC_SIDE, 1);
#endif
}

/******************************************************************************/
/* FUNCTION     vGetReferences                                                */
/******************************************************************************/
/**
*  \brief       Function to get all reference needed by this class.
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::vGetReferences(tVoid)
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Video::vGetReferences() entered."));

   m_poClientVcan = dynamic_cast<I_dispvidctrl_tclClientVcan*>(_cpoMain->getHandler("I_dispvidctrl_tclClientVcan"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poClientVcan);

   m_poClient_DevVideo = dynamic_cast<dispvidctrl_tclClient_DevVideo*>(_cpoMain->getHandler("dispvidctrl_tclClient_DevVideo"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poClient_DevVideo);

   m_poClient_VideoPlayer = dynamic_cast<I_dispvidctrl_tclVideoPlayer*>(_cpoMain->getHandler("I_dispvidctrl_tclVideoPlayer"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poClient_VideoPlayer);

   m_poPersistentData = dynamic_cast<dispvidctrl_tclPersistentData*>(_cpoMain->getHandler("dispvidctrl_tclPersistentData"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poPersistentData);

   m_poControl_Hmi = dynamic_cast<dispvidctrl_tclControl_Hmi*>(_cpoMain->getHandler("dispvidctrl_tclControl_Hmi"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poControl_Hmi);

   m_poServiceRvcIf = dynamic_cast<Idispvidctrl_tclServiceRvcIf*>(_cpoMain->getHandler("Idispvidctrl_tclServiceRvcIf"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poServiceRvcIf);

   m_poControl_Mvc = dynamic_cast<dispvidctrl_tclControl_Mvc*>(_cpoMain->getHandler("dispvidctrl_tclControl_Mvc"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poControl_Mvc);
}

/******************************************************************************/
/* FUNCTION     vStartCommunication                                           */
/******************************************************************************/
/**
*  \brief       Function to start all dynamic objects e.g. threads, ...
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::vStartCommunication(tVoid)
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Video::vStartCommunication() entered."));

#ifndef VARIANT_S_FTR_ENABLE_AIVI_SCOPE1 // Only needed for dynamic Guidelines, not needed for Scope1
   // create timer for Dynamic Guideline updates
   if( OSAL_s32TimerCreate((OSAL_tpfCallback) _pfCallbackTimer_DynGuidelineUpdate, (tVoid*) this, &m_hTimerHandle_DynGuidelineUpdate) != OSAL_OK)
   {
      //FATAL_M_ASSERT_ALWAYS();
      ETG_TRACE_USR4((" CA-GRF   :: E R R O R in vStartCommunication(): ..... create Timer DynGuidelineUpdate failed! "));
      m_hTimerHandle_DynGuidelineUpdate =  OSAL_C_INVALID_HANDLE;
   }
   // create timer for Steering angle adjustment
   if( OSAL_s32TimerCreate((OSAL_tpfCallback) _pfCallbackTimer_SteeringAngleAdjustment, (tVoid*) this, &m_hTimerHandle_SteeringAngleAdjustment) != OSAL_OK)
   {
      //FATAL_M_ASSERT_ALWAYS();
      ETG_TRACE_USR4((" CA-GRF   :: E R R O R in vStartCommunication(): ..... create Timer SteeringAngleAdjustment failed! "));
      m_hTimerHandle_SteeringAngleAdjustment =  OSAL_C_INVALID_HANDLE;
   }
   else
   {
      if ((m_Config_bRvcMode == TRUE) && (m_bAnticipatoryLine == TRUE))
      {
         _vStartStopSteeringAngleAdjustment();
      }
   }

   if( OSAL_s32TimerCreate((OSAL_tpfCallback) _pfCallbackTimer_SteeringAngleGuidelineAdjustment, (tVoid*) this, &m_hTimerHandle_SteeringAngleGuidelineAdjustment) != OSAL_OK)
   {
      //FATAL_M_ASSERT_ALWAYS();
      ETG_TRACE_USR4((" CA-GRF   :: E R R O R in vStartCommunication(): ..... create Timer SteeringAngleGuidelineAdjustment failed! "));
      m_hTimerHandle_SteeringAngleGuidelineAdjustment =  OSAL_C_INVALID_HANDLE;
   }
#endif

   // for early video activation we hold required user settings and configuration data in an own data-pool.
   // before HMI is sending us the last user settings we use the values from our data-pool.
   _vGetPersistentData();
   _vGuidelineToggleSequence_ResetIndex();
   // the user settings data will be synchronised with the last user settings from HMI when HMI is sending these data on HMI start-up.
   // For Scope1 (NIVIC) the HMI is not sending HUE and BlackLevel values on start-up for synchronisation with early data pool.
   // We have to set it here manually but it will be processed first when HMI has sent the other values (brightness, colour...) too.
   // In case of Scope2 the HMI will overwrite the settings we have done here.
   _vOnUserSettingValue(EN_CONTROL_VIDEO__USER_SETTING_VALUE__BLACKLEVEL_DAY, (tU32) 50);
   _vOnUserSettingValue(EN_CONTROL_VIDEO__USER_SETTING_VALUE__BLACKLEVEL_NIGHT, (tU32) 50);
   _vOnUserSettingValue(EN_CONTROL_VIDEO__USER_SETTING_VALUE__HUE_DAY, (tU32) 50);
   _vOnUserSettingValue(EN_CONTROL_VIDEO__USER_SETTING_VALUE__HUE_NIGHT, (tU32) 50);
}

void dispvidctrl_tclControl_Video::setGraphicslayerVisibility(unsigned int visibility)
{
	ETG_TRACE_USR4(("dispvidctrl_tclControl_Video::Recieved Visibility %d",visibility));
	m_poClient_VideoPlayer->GraphicslayerVisibility(visibility);
}

/******************************************************************************/
/* FUNCTION     vHandleMessage(TMsg* pMsg)                                    */
/******************************************************************************/
/**
*  \brief       Handle worker events.
*
*  \param       message pointer
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::vHandleMessage(dispvidctrl_tclBaseIf::TMsg* pMsg)
{
   ETG_TRACE_USR4(("dispvidctrl_tclControl_Video::vHandleMessage() entered %u -> data: %d.", ETG_CENUM(dispvidctrl_tclBaseIf::ECmdTypes , (tU32)pMsg->eCmd), pMsg->u.u32Data));

   switch (pMsg->eCmd)
   {
      case dispvidctrl_tclBaseIf::eControlVideo:
         vHandleControl((tenControl_Video) pMsg->u.u32Data);
         break;
      case dispvidctrl_tclBaseIf::eRvcCameraState:
         {
              tBool bRvcActive = FALSE;
              tBool bRvcErr = FALSE;
              if (m_poClient_VideoPlayer)
              {
                 m_poClient_VideoPlayer->vGetRvcState(bRvcActive, bRvcErr);
              }
              bSetRVCSTATUSINFO(bRvcActive, bRvcErr);
         }
         break;
      case dispvidctrl_tclBaseIf::eDimStateChanged:
         {
            if ( m_HmiUserSetting_u8DisplayDimmingMode != pMsg->u.u32Data )
            {
               vHandleControl((tenControl_Video)EN_CONTROL_VIDEO__USER_SETTING_VALUE__DISPLAY_DIMMING_MODE, pMsg->u.u32Data);
               vHandleControl((tenControl_Video)EN_CONTROL_VIDEO__USER_SETTING_UPDATE__VIDEO_CONTROL_DIMMING, 0);
               if (m_bDiagScreenRequest)
               {
                  vHandleControl((tenControl_Video)EN_CONTROL_VIDEO__SET_VIDEO_CONTROLS, 0);
               }
            }
         }
         break;
      default:
         break;
   }
}


/******************************************************************************/
/* FUNCTION     vHandleControl()                                              */
/******************************************************************************/
/**
*  \brief       trace information
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::vHandleControl(tenControl_Video enControl, tU32 u32Value)
{
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32Value );

   ETG_TRACE_USR4(("   CA-VID :: ........... vHandleControl(): %d     ", ETG_CENUM(tenControl_Video, enControl)));

   switch (enControl)
   {
      case EN_CONTROL_VIDEO__DISABLE:
      case EN_CONTROL_VIDEO__ENABLE:
         break;
      case EN_CONTROL_VIDEO__EARLY_VIDEO_CONTROL_OFF:
            // Tell VideoPlayer that DispVidCtrl is assuming control.
            if(m_poClient_VideoPlayer)
            {
               tBool bRvcActive = FALSE;
               tBool bRvcErr = FALSE;

               m_poClient_VideoPlayer->vGetRvcState(bRvcActive, bRvcErr);
               bSetRVCSTATUSINFO(bRvcActive, bRvcErr);
               m_poClient_VideoPlayer->vSetInitialblackscreen(m_u8ConfiguredVideoType);
            }
         break;
      case EN_CONTROL_VIDEO__HMI_AVAILABLE:
         if(m_poClient_VideoPlayer)
         {
            m_poClient_VideoPlayer->vSetHmiAvailable();
         }
         break;
      //----------------------------------------------------------------------- from Control_Hmi
      case EN_CONTROL_VIDEO__DIRECT_ON:  // transition to late while early Video is active
         {
            //start dynamic guideline update timer
            if (m_bAnticipatoryLine && m_Config_bRvcMode)
            {
               (tVoid) _bSetTimer_DynGuidelineUpdate( (OSAL_tMSecond) 100);
            }
            if (m_Config_bRvcMode || m_Config_bMvcMode)
            {
               //_vGuidelineToggleSequence_ResetIndex();

               // switch warning-text and GuidelineVisibility according to user setup
               m_bGraphicsVisibility = TRUE;
               //_vSetVisibility_WarningText();
               //_vSetVisibility_Guideline();
            }
            _vSetVideoControls();
         }
         break;
      case EN_CONTROL_VIDEO__PREPARE_GRAPHICS:
         if(m_poClient_VideoPlayer)
         {
            m_poClient_VideoPlayer->vPreInitGraphics();
         }
         break;
      case EN_CONTROL_VIDEO__CAPTURING_OFF:
         {
            m_bDiagScreenRequest = FALSE;
            tChar strIfName[] = "I_dispvidctrl_tclSyncHandler";
            dispvidctrl_tclAppMain::theServer()->vPostMsgControlSyncHandler(strIfName, EN_CONTROL_SH__CAMERA_SCREEN_OFF);
            //stop dynamic guideline update timer
            (tVoid) _bSetTimer_DynGuidelineUpdate( (OSAL_tMSecond) 0);
            (tVoid) _bSetTimer_SteeringAngleGuidelineAdjustment( (OSAL_tMSecond) 0);
            // set special guideline style for guidleine adjustment to OFF
            if(m_poClient_VideoPlayer != 0)
            {
               m_poClient_VideoPlayer->vSetTemporaryCalibGuidelnStyle((tBool) FALSE);
            }
            // close the video device after exiting the video screen 
            tChar strIfName1[] = "dispvidctrl_tclControl_Video";
            dispvidctrl_tclAppMain::theServer()->vPostMsgControlVideo(strIfName1, EN_CONTROL_VIDEO__DEVICE_CLOSE);
         }
         break;
      case EN_CONTROL_VIDEO__CAPTURING_ON:
         {
            // first open the Video Device in Gstreamer so that it can switch the channels to LVDS or CVBS etc..
            tChar strIfName[] = "I_dispvidctrl_tclSyncHandler";
            dispvidctrl_tclAppMain::theServer()->vPostMsgControlSyncHandler(strIfName, EN_CONTROL_SH__CAMERA_SCREEN_ON);
            
            // keep the video device open while in camera screen for changing (brightness, color etc..)
            tChar strIfName1[] = "dispvidctrl_tclControl_Video";
            dispvidctrl_tclAppMain::theServer()->vPostMsgControlVideo(strIfName1, EN_CONTROL_VIDEO__DEVICE_OPEN);
            
            //start dynamic guideline update timer
            if (m_bAnticipatoryLine && m_Config_bRvcMode)
            {
               (tVoid) _bSetTimer_DynGuidelineUpdate( (OSAL_tMSecond) 100);
            }
         }
         break;
      case EN_CONTROL_VIDEO__CAPTURING_ON_DIAG:
         {
            m_bDiagScreenRequest = TRUE;
            // switch warning text and guideline to OFF
            m_bGraphicsVisibility = FALSE;
            _vSetVisibility_WarningText();
            _vSetVisibility_Guideline();

            // TODO SET_VIDEO_GAMMA_OFF

            // capturing on
            tChar strIfName[] = "I_dispvidctrl_tclSyncHandler";
            dispvidctrl_tclAppMain::theServer()->vPostMsgControlSyncHandler(strIfName, EN_CONTROL_SH__CAMERA_SCREEN_ON);
         }
         break;
      case EN_CONTROL_VIDEO__CAPTURING_ON_GDL_ADJ:
         {
            // switch warning text and guideline to OFF
            m_bGraphicsVisibility = FALSE;
            _vSetVisibility_WarningText();
            //_vSetVisibility_Guideline();
            //set special guideline style (will become visible only when guideline style is NISSAN and not Renault)
            if(m_poClient_VideoPlayer != 0)
            {
               m_poClient_VideoPlayer->vSetTemporaryCalibGuidelnStyle((tBool) TRUE);
               m_poClient_VideoPlayer->vSetGuideLineVisibility(DYN_GUIDELINE_DRAW_STATIC_SIDE | DYN_GUIDELINE_FORCE_UPDATE);
            }
            // TODO SET_VIDEO_GAMMA_OFF

            // capturing on
            tChar strIfName[] = "I_dispvidctrl_tclSyncHandler";
            dispvidctrl_tclAppMain::theServer()->vPostMsgControlSyncHandler(strIfName, EN_CONTROL_SH__CAMERA_SCREEN_ON);

            if (m_bAnticipatoryLine && m_Config_bRvcMode)
            {
               (tVoid) _bSetTimer_SteeringAngleGuidelineAdjustment( (OSAL_tMSecond) 250);
            }
         }
         break;
      case EN_CONTROL_VIDEO__GRAPHICS_OFF:
            // in user settings screen for brightness, contrast, colour we don't want to show graphics
            if (m_Config_bRvcMode || m_Config_bMvcMode || m_Config_bAvmRenaultMode)
            {
               // switch warning text and guideline to OFF
               m_bGraphicsVisibility = FALSE;
               _vSetVisibility_WarningText();
               _vSetVisibility_Guideline();
            }
         break;
      case EN_CONTROL_VIDEO__GRAPHICS_ON:
            if (m_Config_bRvcMode || m_Config_bMvcMode || m_Config_bAvmRenaultMode)
            {
               // switch warning-text and GuidelineVisibility according to user setup
               m_bGraphicsVisibility = TRUE;
               _vSetVisibility_WarningText();
               _vSetVisibility_Guideline();
            }
         break;
      case EN_CONTROL_VIDEO__SET_B_HANDOVER_READY:
            if(m_poClient_VideoPlayer)
            {
               m_poClient_VideoPlayer->vSetHandoverReady();
            }
         break;
      case EN_CONTROL_VIDEO__BLACK_SCREEN_ON:
            if(m_poClient_VideoPlayer)
            {
               m_poClient_VideoPlayer->vShowBlackScreen((tU8) u32Value);
            }
         break;
      case EN_CONTROL_VIDEO__SHOW_RVC:
            if(m_poClient_VideoPlayer)
            {
               m_poClient_VideoPlayer->vShowRvc();
            }
         break;
      case EN_CONTROL_VIDEO__SET_VIDEO_CONTROLS:
            if (m_bDiagScreenRequest)
            {
               _vResetVideoControls();
            }
            else
            {
               _vSetVideoControls();
            }
         break;
      case EN_CONTROL_VIDEO__SET_VIDEO_GAMMA_OFF:
      case EN_CONTROL_VIDEO__SET_VIDEO_GAMMA_ON:
            ETG_TRACE_USR4(("   CA-VID :: Control not handled currently(): %d     ", ETG_CENUM(tenControl_Video, enControl)));
         break;
      case EN_CONTROL_VIDEO__TOGGLE_GUIDELINE: //temporarily switching while camera screen is active
            _vGuidelineToggleSequence_ToggleIndex();
            _vSetVisibility_Guideline();
         break;
      //----------------------------------------------------------------------- updates from HMI or DIMMING
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__GUIDELINE_DYNAMIC:
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__GUIDELINE_STATIC_CENTRE:
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__GUIDELINE_STATIC_SIDE:
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__LANGUAGE:
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__DISPLAY_DIMMING_MODE:
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__BLACKLEVEL_DAY:
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__BLACKLEVEL_NIGHT:
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__BRIGHTNESS_DAY:
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__BRIGHTNESS_NIGHT:
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__COLOUR_DAY:
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__COLOUR_NIGHT:
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__CONTRAST_DAY:
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__CONTRAST_NIGHT:
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__HUE_DAY:
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__HUE_NIGHT:
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__DIMMING_DAY:
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__DIMMING_NIGHT:
            _vOnUserSettingValue(enControl, u32Value);
         break;
      case EN_CONTROL_VIDEO__USER_SETTING_UPDATE__GUIDELINE:
            _vOnUserSettingUpdate_GuideLine();
            _vSetVisibility_Guideline();
         break;
      case EN_CONTROL_VIDEO__USER_SETTING_UPDATE__LANGUAGE:
            _vOnUserSettingUpdate_Language();
         break;
      case EN_CONTROL_VIDEO__USER_SETTING_UPDATE__VIDEO_CONTROL:
            _vOnUserSettingUpdate_VideoControl();
         break;
      case EN_CONTROL_VIDEO__USER_SETTING_UPDATE__VIDEO_CONTROL_DIMMING:
            _vOnUserSettingUpdate_DimmingMode();
         break;
      //-----------------------------------------------------------------------
      case EN_CONTROL_VIDEO__DYNAMIC_GUIDELINE_UPDATE:
            _vOnDynamicGuidelineUpdate();
         break;
      case EN_CONTROL_VIDEO__STEERING_ANGLE_ADJUSTMENT:
            _vOnSteeringAngleAdjustment();
         break;
      //---------------------------------------------------- from clientSpm
      case EN_CONTROL_VIDEO__IGNITION:
            if (TRUE == (tBool) u32Value)
            {
               m_SteeringAngleAdjust_bIgnition = TRUE;
            }
            else
            {
               m_SteeringAngleAdjust_bIgnition = FALSE;
            }
         break;
      //------------------------------------
      case EN_CONTROL_VIDEO__GUIDELINE_ADJUSTMENT:
         _vGdlAdj_HandleControl((tenControl_Video_GdlAdj) u32Value);
         break;
      case EN_CONTROL_VIDEO__GUIDELINE_ADJUSTMENT_UPDATE:
         _vGdlAdj_HandleControl(EN_CTRL_GDLADJ__UPDATE_STEERING_ANGLE);
         break;
         
      case EN_CONTROL_VIDEO__DEVICE_OPEN:
         // close the video device after exiting the video screen 
         if (m_poClient_DevVideo != 0)
         {
            m_poClient_DevVideo->vOpenDevVideo();
         }
         break;
      case EN_CONTROL_VIDEO__DEVICE_CLOSE:
         // close the video device after exiting the video screen 
         if (m_poClient_DevVideo != 0)
         {
            m_poClient_DevVideo->vCloseDevVideo();
         }
         break;
         
      case EN_CONTROL_VIDEO__RESIZE_VIDEO:
         _vSetLayerSetting(u32Value);
         break;

      case EN_CONTROL_VIDEO__STATIC_GUIDELINE_UPDATE:  //Handle Static Guideline update for MVC View Modes.
         _vOnStaticGuidelineUpdate(u32Value);
         _vSetVisibility_Guideline();
         break;

      default:
         break;
   }
}

/******************************************************************************/
/* FUNCTION     _vOnStaticGuidelineUpdate                         */
/******************************************************************************/
/**
*  \brief       Static Guideline Update for MVC Views
*
*  \param       none
*
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::_vOnStaticGuidelineUpdate(tU32 u32StaticGuidelineVisibility)
{
   ETG_TRACE_FATAL(("   CA-VID :: _vOnStaticGuidelineUpdate() Static Guideline Visibility : %d     ", u32StaticGuidelineVisibility));

   if(u32StaticGuidelineVisibility)  //If MVC mode is rear, update guidelines
   {
      if(m_HmiUserSetting_u8GuidelineStaticSide)  //If HMI User settings is ON, then set MVC static guidelines to ON.
      {
         //If current index is not static, toggle the index to STATIC.
         if (((m_au32GuidelineToggleSequence[m_u8CurrentGuidelineIndex]) & ((tU32) DYN_GUIDELINE_DRAW_STATIC_SIDE)) != (tU32) DYN_GUIDELINE_DRAW_STATIC_SIDE)
         {
            _vGuidelineToggleSequence_ToggleIndex();
            ETG_TRACE_FATAL(("   CA-VID :: _vOnStaticGuidelineUpdate() setting current index to DYN_GUIDELINE_DRAW_STATIC_SIDE"));
         }
         else
         {
            ETG_TRACE_FATAL(("   CA-VID :: _vOnStaticGuidelineUpdate() current index is already DYN_GUIDELINE_DRAW_STATIC_SIDE"));
         }
      }
      else    //If HMI User settings is OFF, then set MVC static guidelines to OFF.
      {
         //If current index is static, toggle the index to DRAW_NONE
         if (((m_au32GuidelineToggleSequence[m_u8CurrentGuidelineIndex]) & ((tU32) DYN_GUIDELINE_DRAW_STATIC_SIDE)) == (tU32) DYN_GUIDELINE_DRAW_STATIC_SIDE)
         {
            _vGuidelineToggleSequence_ToggleIndex();
            ETG_TRACE_FATAL(("   CA-VID :: _vOnStaticGuidelineUpdate() HMI User settings is OFF setting current index to DYN_GUIDELINE_DRAW_NONE"));
         }
         else
         {
            ETG_TRACE_FATAL(("   CA-VID :: _vOnStaticGuidelineUpdate() HMI User settings is OFF current index is already DYN_GUIDELINE_DRAW_NONE"));
         }
      }
   }
   else    //MVC View mode is other than rear view, do not display guidelines.
   {
      //If current index is static, toggle the index to DRAW_NONE
      if (((m_au32GuidelineToggleSequence[m_u8CurrentGuidelineIndex]) & ((tU32) DYN_GUIDELINE_DRAW_STATIC_SIDE)) == (tU32) DYN_GUIDELINE_DRAW_STATIC_SIDE)
      {
         _vGuidelineToggleSequence_ToggleIndex();
         ETG_TRACE_FATAL(("   CA-VID :: _vOnStaticGuidelineUpdate() MVC view is Not Rear view, setting current index to DYN_GUIDELINE_DRAW_NONE"));
      }
      else
      {
         ETG_TRACE_FATAL(("   CA-VID :: _vOnStaticGuidelineUpdate() MVC view is Not Rear view, current index is already DYN_GUIDELINE_DRAW_NONE"));
      }
   }
}

/******************************************************************************/
/* FUNCTION     _vGuidelineToggleSequence_ToggleIndex                         */
/******************************************************************************/
/**
*  \brief       circular iteration through the toggling sequence,
*
*  \param       none
*
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::_vGuidelineToggleSequence_ToggleIndex()
{
   // continue iterating until the last sequence and then fall back to the first sequence
   if ( (++m_u8CurrentGuidelineIndex) >= m_u8MaxGuidelineTogglesAvailable )
   {
       m_u8CurrentGuidelineIndex = 0;
   }
}

/******************************************************************************/
/* FUNCTION     _vGuidelineToggleSequence_ResetIndex                          */
/******************************************************************************/
/**
*  \brief       fall back to the sequence start depending on user setting
*               and exit/entry of RVC
*
*  \param       none
*
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::_vGuidelineToggleSequence_ResetIndex()
{
   m_u8CurrentGuidelineIndex = 0;
}

/******************************************************************************/
/*                                                                            */
/*   H M I   U S E R   S E T T I N G                                          */
/*                                                                            */
/******************************************************************************/
/******************************************************************************/
/* FUNCTION     _vOnUserSettingValue                                          */
/******************************************************************************/
/**
*  \brief       - hold all new HMI user setting value(s)
*
*
*  \param       data control name, data value
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::_vOnUserSettingValue(tenControl_Video enControl, tU32 u32Value)
{
   ETG_TRACE_USR4(("   CA-VID :: _vOnUserSettingValue(): %d     ", ETG_CENUM(tenControl_Video, enControl)));

   switch (enControl)
   {
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__GUIDELINE_DYNAMIC:
         if (m_Config_bAnticipatoryLine) {
             m_HmiUserSetting_u8GuidelineDynamic = (tU8) u32Value;
         }
         break;
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__GUIDELINE_STATIC_SIDE:
         m_HmiUserSetting_u8GuidelineStaticSide = (tU8) u32Value;
         break;
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__LANGUAGE:
         m_HmiUserSetting_u8Language = (tU8) u32Value;
         break;
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__DISPLAY_DIMMING_MODE:
         m_HmiUserSetting_u8DisplayDimmingMode = (tU8) u32Value;
         break;
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__BLACKLEVEL_DAY:
         m_HmiUserSetting_u8VideoBlackLevelDay = (tU8) u32Value;
         break;
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__BLACKLEVEL_NIGHT:
         m_HmiUserSetting_u8VideoBlackLevelNight = (tU8) u32Value;
         break;
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__BRIGHTNESS_DAY:
         m_HmiUserSetting_u8VideoBrightnessDay = (tU8) u32Value;
         break;
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__BRIGHTNESS_NIGHT:
         m_HmiUserSetting_u8VideoBrightnessNight = (tU8) u32Value;
         break;
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__COLOUR_DAY:
         m_HmiUserSetting_u8VideoColourDay = (tU8) u32Value;
         break;
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__COLOUR_NIGHT:
         m_HmiUserSetting_u8VideoColourNight = (tU8) u32Value;
         break;
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__CONTRAST_DAY:
         m_HmiUserSetting_u8VideoContrastDay = (tU8) u32Value;
         break;
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__CONTRAST_NIGHT:
         m_HmiUserSetting_u8VideoContrastNight = (tU8) u32Value;
         break;
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__HUE_DAY:
         m_HmiUserSetting_u8VideoHueDay = (tU8) u32Value;
         break;
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__HUE_NIGHT:
         m_HmiUserSetting_u8VideoHueNight = (tU8) u32Value;
         break;
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__DIMMING_DAY:
         m_HmiUserSetting_u8VideoDimmingDay = (tU8) u32Value;
         break;
      case EN_CONTROL_VIDEO__USER_SETTING_VALUE__DIMMING_NIGHT:
         m_HmiUserSetting_u8VideoDimmingNight = (tU8) u32Value;
         break;

      default:
         break;
   }
}


/******************************************************************************/
/* FUNCTION     _vOnUserSettingUpdate_VideoControl                            */
/******************************************************************************/
/**
*  \brief       - called when we have received all linked values relevant for
*                 one user setting update
*
*               - do required mapping/calculation to internal values
*               for Video Controls:
*               -------------------
*               The HMI video control values are going from 0 to 100%, 50% means
*                  neutral setting (HMI user setting)
*                  The video player interface requires also 0 to 100%
*                  Video player will do the mapping to gStreamer API which requires
*                  the register range 0..256 or a signed value -127..0..128
*               To realize an suitable setting, the middle position and end points
*                  of the values are set via defines and an mapping will be done.
*                  (to use the full range makes no sense for the customer,  e.g.
*                  contrast and brightness register values set to minimum or maximum
*                  will produce pictures near to black or white)
*
*               store values persistent in data pool for vd_early/VideoPlayer early
*               video use case
*               - update current settings (send settings to VideoPlayer or dev/video)
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::_vOnUserSettingUpdate_VideoControl(tVoid)
{
   tU32 u32OldValue;

   if (m_poClient_DevVideo && m_poPersistentData)
   {
      // B L A C K L E V E L  D A Y
      if (m_HmiUserSetting_u8VideoBlackLevelDay <= 100)
      {
         u32OldValue = m_u32VideoBlackLevelDay;

         m_u32VideoBlackLevelDay = stVideoSettingHmiToDevVideoValueMapping[(m_HmiUserSetting_u8VideoBlackLevelDay/5)].u8DevVideoValue_Blacklevel;

//         if (m_HmiUserSetting_u8VideoBlackLevelDay < 50) {
//            m_u32VideoBlackLevelDay = ((((DISPVIDCTRL_VIDEO__BLACKLEVEL_MID_VALUE_DAY - DISPVIDCTRL_VIDEO__BLACKLEVEL_MIN_VALUE_DAY) * 1000) / 50) * m_HmiUserSetting_u8VideoBlackLevelDay / 1000) + DISPVIDCTRL_VIDEO__BLACKLEVEL_MIN_VALUE_DAY;
//         } else {
//            m_u32VideoBlackLevelDay = ((((DISPVIDCTRL_VIDEO__BLACKLEVEL_MAX_VALUE_DAY - DISPVIDCTRL_VIDEO__BLACKLEVEL_MID_VALUE_DAY) * 1000) / 50) * (m_HmiUserSetting_u8VideoBlackLevelDay - 50) / 1000) + DISPVIDCTRL_VIDEO__BLACKLEVEL_MID_VALUE_DAY;
//         }

         if (u32OldValue != m_u32VideoBlackLevelDay)
         {
            // store persistent for early video usage and update current setting
            m_poPersistentData->vSet_BlackLevel_Day(m_u32VideoBlackLevelDay);
            m_poClient_DevVideo->vSet_BlackLevel((tU8) m_u32VideoBlackLevelDay);
         }
      }

      // B L A C K L E V E L  N I G H T
      if (m_HmiUserSetting_u8VideoBlackLevelNight <= 100)
      {
         u32OldValue = m_u32VideoBlackLevelNight;

         m_u32VideoBlackLevelNight = stVideoSettingHmiToDevVideoValueMapping[m_HmiUserSetting_u8VideoBlackLevelNight/5].u8DevVideoValue_Blacklevel;

//         if (m_HmiUserSetting_u8VideoBlackLevelNight < 50) {
//            m_u32VideoBlackLevelNight = ((((DISPVIDCTRL_VIDEO__BLACKLEVEL_MID_VALUE_NIGHT - DISPVIDCTRL_VIDEO__BLACKLEVEL_MIN_VALUE_NIGHT) * 1000) / 50) * m_HmiUserSetting_u8VideoBlackLevelNight / 1000) + DISPVIDCTRL_VIDEO__BLACKLEVEL_MIN_VALUE_NIGHT;
//         } else {
//            m_u32VideoBlackLevelNight = ((((DISPVIDCTRL_VIDEO__BLACKLEVEL_MAX_VALUE_NIGHT - DISPVIDCTRL_VIDEO__BLACKLEVEL_MID_VALUE_NIGHT) * 1000) / 50) * (m_HmiUserSetting_u8VideoBlackLevelNight - 50) / 1000) + DISPVIDCTRL_VIDEO__BLACKLEVEL_MID_VALUE_NIGHT;
//         }

         if (u32OldValue != m_u32VideoBlackLevelNight)
         {
            // store persistent for early video usage and update current setting
            m_poPersistentData->vSet_BlackLevel_Night(m_u32VideoBlackLevelNight);
            m_poClient_DevVideo->vSet_BlackLevel((tU8) m_u32VideoBlackLevelNight);
         }
      }

      // B R I G H T N E S S  D A Y
      if (m_HmiUserSetting_u8VideoBrightnessDay <= 100)
      {
         if(m_HmiUserSetting_u8VideoBrightnessDay <= 40)
         {
             m_HmiUserSetting_u8VideoBrightnessDay = 40;
         }
         if(m_HmiUserSetting_u8VideoBrightnessDay > 65)
         {
            m_HmiUserSetting_u8VideoBrightnessDay = 65;
         }
         u32OldValue = m_u32VideoBrightnessDay;

         m_u32VideoBrightnessDay = stVideoSettingHmiToDevVideoValueMapping[m_HmiUserSetting_u8VideoBrightnessDay/5].u8DevVideoValue_Brightness;

//         if (m_HmiUserSetting_u8VideoBrightnessDay < 50) {
//            m_u32VideoBrightnessDay = ((((DISPVIDCTRL_VIDEO__BRIGHTNESS_MID_VALUE_DAY - DISPVIDCTRL_VIDEO__BRIGHTNESS_MIN_VALUE_DAY) * 1000) / 50) * m_HmiUserSetting_u8VideoBrightnessDay / 1000) + DISPVIDCTRL_VIDEO__BRIGHTNESS_MIN_VALUE_DAY;
//         } else {
//            m_u32VideoBrightnessDay = ((((DISPVIDCTRL_VIDEO__BRIGHTNESS_MAX_VALUE_DAY - DISPVIDCTRL_VIDEO__BRIGHTNESS_MID_VALUE_DAY) * 1000) / 50) * (m_HmiUserSetting_u8VideoBrightnessDay - 50) / 1000) + DISPVIDCTRL_VIDEO__BRIGHTNESS_MID_VALUE_DAY;
//         }
        // only for Brightness we can shift up or down the settings range
        // value will be shifted according brightness configuration value (here used as offset)
        if (m_Config_u8BrightnessValueDay < 50) {   // negative shifting
           if (m_Config_u8BrightnessValueDay + m_u32VideoBrightnessDay >= 50) { // value should not below "0" (negative)
              m_u32VideoBrightnessDay = m_u32VideoBrightnessDay - (50 - m_Config_u8BrightnessValueDay);
           }
           else {
              m_u32VideoBrightnessDay = 0;
           }
        } else {                                    // positive shifting
           m_u32VideoBrightnessDay = m_u32VideoBrightnessDay + (m_Config_u8BrightnessValueDay - 50);
           if (m_u32VideoBrightnessDay > 100) {             // limit to 100
              m_u32VideoBrightnessDay = 100;
           }
        }

         if (u32OldValue != m_u32VideoBrightnessDay)
         {
            // store persistent for early video usage and update current setting
            m_poPersistentData->vSet_Brightness_Day(m_u32VideoBrightnessDay);
            m_poClient_DevVideo->vSet_Brightness((tU8) m_u32VideoBrightnessDay);
         }
      }

      // B R I G H T N E S S  N I G H T
      if (m_HmiUserSetting_u8VideoBrightnessNight <= 100)
      {
         if(m_HmiUserSetting_u8VideoBrightnessNight < 45)
         {
            m_HmiUserSetting_u8VideoBrightnessNight = 45;
         }
         if(m_HmiUserSetting_u8VideoBrightnessNight > 65)
         {
            m_HmiUserSetting_u8VideoBrightnessNight = 65;
         }
         u32OldValue = m_u32VideoBrightnessNight;

         m_u32VideoBrightnessNight = stVideoSettingHmiToDevVideoValueMapping[m_HmiUserSetting_u8VideoBrightnessNight/5].u8DevVideoValue_Brightness;

//         if (m_HmiUserSetting_u8VideoBrightnessNight < 50) {
//            m_u32VideoBrightnessNight = ((((DISPVIDCTRL_VIDEO__BRIGHTNESS_MID_VALUE_NIGHT - DISPVIDCTRL_VIDEO__BRIGHTNESS_MIN_VALUE_NIGHT) * 1000) / 50) * m_HmiUserSetting_u8VideoBrightnessNight / 1000) + DISPVIDCTRL_VIDEO__BRIGHTNESS_MIN_VALUE_NIGHT;
//         } else {
//            m_u32VideoBrightnessNight = ((((DISPVIDCTRL_VIDEO__BRIGHTNESS_MAX_VALUE_NIGHT - DISPVIDCTRL_VIDEO__BRIGHTNESS_MID_VALUE_NIGHT) * 1000) / 50) * (m_HmiUserSetting_u8VideoBrightnessNight - 50) / 1000) + DISPVIDCTRL_VIDEO__BRIGHTNESS_MID_VALUE_NIGHT;
//         }
        // only for Brightness we can shift up or down the settings range
        // value will be shifted according brightness configuration value (here used as offset)
        if (m_Config_u8BrightnessValueNight < 50) {   // negative shifting
           if (m_Config_u8BrightnessValueNight + m_u32VideoBrightnessNight >= 50) { // value should not below "0" (negative)
              m_u32VideoBrightnessNight = m_u32VideoBrightnessNight - (50 - m_Config_u8BrightnessValueNight);
           }
           else {
              m_u32VideoBrightnessNight = 0;
           }
        } else {                                    // positive shifting
           m_u32VideoBrightnessNight = m_u32VideoBrightnessNight + (m_Config_u8BrightnessValueNight - 50);
           if (m_u32VideoBrightnessNight > 100) {             // limit to 100
              m_u32VideoBrightnessNight = 100;
           }
        }
//         // only for Brightness we can shift up or down the settings range
//         // value will be shifted according brightness configuration value (here used as offset)
//         if (m_Config_u8BrightnessValueNight < 50) {   // negative shifting
//            m_u32VideoBrightnessNight = m_u32VideoBrightnessNight - (50 - m_Config_u8BrightnessValueNight);
//         } else {                                    // positive shifting
//            m_u32VideoBrightnessNight = m_u32VideoBrightnessNight + (m_Config_u8BrightnessValueNight - 50);
//            if (m_u32VideoBrightnessNight > 100) {             // limit to 100
//               m_u32VideoBrightnessNight = 100;
//            }
//         }

         if (u32OldValue != m_u32VideoBrightnessNight)
         {
            // store persistent for early video usage and update current setting
            m_poPersistentData->vSet_Brightness_Night(m_u32VideoBrightnessNight);
            m_poClient_DevVideo->vSet_Brightness((tU8) m_u32VideoBrightnessNight);
         }
      }

      // C O L O U R  D A Y
      if (m_HmiUserSetting_u8VideoColourDay <= 100)
      {
         u32OldValue = m_u32VideoColourDay;

         m_u32VideoColourDay = stVideoSettingHmiToDevVideoValueMapping[m_HmiUserSetting_u8VideoColourDay/5].u8DevVideoValue_Color;

//         if (m_HmiUserSetting_u8VideoColourDay < 50) {
//            m_u32VideoColourDay = ((((DISPVIDCTRL_VIDEO__COLOUR_MID_VALUE_DAY - DISPVIDCTRL_VIDEO__COLOUR_MIN_VALUE_DAY) * 1000) / 50) * m_HmiUserSetting_u8VideoColourDay / 1000) + DISPVIDCTRL_VIDEO__COLOUR_MIN_VALUE_DAY;
//         } else {
//            m_u32VideoColourDay = ((((DISPVIDCTRL_VIDEO__COLOUR_MAX_VALUE_DAY - DISPVIDCTRL_VIDEO__COLOUR_MID_VALUE_DAY) * 1000) / 50) * (m_HmiUserSetting_u8VideoColourDay - 50) / 1000) + DISPVIDCTRL_VIDEO__COLOUR_MID_VALUE_DAY;
//         }

         if (u32OldValue != m_u32VideoColourDay)
         {
            // store persistent for early video usage and update current setting
            m_poPersistentData->vSet_Colour_Day(m_u32VideoColourDay);
            m_poClient_DevVideo->vSet_Colour((tU8) m_u32VideoColourDay);
         }
      }

      // C O L O U R  N I G H T
      if (m_HmiUserSetting_u8VideoColourNight <= 100)
      {
         u32OldValue = m_u32VideoColourNight;

         m_u32VideoColourNight = stVideoSettingHmiToDevVideoValueMapping[m_HmiUserSetting_u8VideoColourNight/5].u8DevVideoValue_Color;

//         if (m_HmiUserSetting_u8VideoColourNight < 50) {
//            m_u32VideoColourNight = ((((DISPVIDCTRL_VIDEO__COLOUR_MID_VALUE_NIGHT - DISPVIDCTRL_VIDEO__COLOUR_MIN_VALUE_NIGHT) * 1000) / 50) * m_HmiUserSetting_u8VideoColourNight / 1000) + DISPVIDCTRL_VIDEO__COLOUR_MIN_VALUE_NIGHT;
//         } else {
//            m_u32VideoColourNight = ((((DISPVIDCTRL_VIDEO__COLOUR_MAX_VALUE_NIGHT - DISPVIDCTRL_VIDEO__COLOUR_MID_VALUE_NIGHT) * 1000) / 50) * (m_HmiUserSetting_u8VideoColourNight - 50) / 1000) + DISPVIDCTRL_VIDEO__COLOUR_MID_VALUE_NIGHT;
//         }

         if (u32OldValue != m_u32VideoColourNight)
         {
            // store persistent for early video usage and update current setting
            m_poPersistentData->vSet_Colour_Night(m_u32VideoColourNight);
            m_poClient_DevVideo->vSet_Colour((tU8) m_u32VideoColourNight);
         }
      }

      // C O N T R A S T  D A Y
      if (m_HmiUserSetting_u8VideoContrastDay <= 100)
      {
         if(m_HmiUserSetting_u8VideoContrastDay < 35)
         {
             m_HmiUserSetting_u8VideoContrastDay = 35;
         }
         if(m_HmiUserSetting_u8VideoContrastDay > 65)
         {
            m_HmiUserSetting_u8VideoContrastDay = 65;
         }

         u32OldValue = m_u32VideoContrastDay;

         m_u32VideoContrastDay = stVideoSettingHmiToDevVideoValueMapping[m_HmiUserSetting_u8VideoContrastDay/5].u8DevVideoValue_Contrast;

//         if (m_HmiUserSetting_u8VideoContrastDay < 50) {
//            m_u32VideoContrastDay = ((((DISPVIDCTRL_VIDEO__CONTRAST_MID_VALUE_DAY - DISPVIDCTRL_VIDEO__CONTRAST_MIN_VALUE_DAY) * 1000) / 50) * m_HmiUserSetting_u8VideoContrastDay / 1000) + DISPVIDCTRL_VIDEO__CONTRAST_MIN_VALUE_DAY;
//         } else {
//            m_u32VideoContrastDay = ((((DISPVIDCTRL_VIDEO__CONTRAST_MAX_VALUE_DAY - DISPVIDCTRL_VIDEO__CONTRAST_MID_VALUE_DAY) * 1000) / 50) * (m_HmiUserSetting_u8VideoContrastDay - 50) / 1000) + DISPVIDCTRL_VIDEO__CONTRAST_MID_VALUE_DAY;
//         }

         if (u32OldValue != m_u32VideoContrastDay)
         {
            // store persistent for early video usage and update current setting
            m_poPersistentData->vSet_Contrast_Day(m_u32VideoContrastDay);
            m_poClient_DevVideo->vSet_Contrast((tU8) m_u32VideoContrastDay);
         }
      }

      // C O N T R A S T  N I G H T
      if (m_HmiUserSetting_u8VideoContrastNight <= 100)
      {
         if(m_HmiUserSetting_u8VideoContrastNight < 45)
         {  
            m_HmiUserSetting_u8VideoContrastNight = 45;
         }
         if(m_HmiUserSetting_u8VideoContrastNight > 65)
         {  
            m_HmiUserSetting_u8VideoContrastNight = 65;
         }

         u32OldValue = m_u32VideoContrastNight;

         m_u32VideoContrastNight = stVideoSettingHmiToDevVideoValueMapping[m_HmiUserSetting_u8VideoContrastNight/5].u8DevVideoValue_Contrast;

//         if (m_HmiUserSetting_u8VideoContrastNight < 50) {
//            m_u32VideoContrastNight = ((((DISPVIDCTRL_VIDEO__CONTRAST_MID_VALUE_NIGHT - DISPVIDCTRL_VIDEO__CONTRAST_MIN_VALUE_NIGHT) * 1000) / 50) * m_HmiUserSetting_u8VideoContrastNight / 1000) + DISPVIDCTRL_VIDEO__CONTRAST_MIN_VALUE_NIGHT;
//         } else {
//            m_u32VideoContrastNight = ((((DISPVIDCTRL_VIDEO__CONTRAST_MAX_VALUE_NIGHT - DISPVIDCTRL_VIDEO__CONTRAST_MID_VALUE_NIGHT) * 1000) / 50) * (m_HmiUserSetting_u8VideoContrastNight - 50) / 1000) + DISPVIDCTRL_VIDEO__CONTRAST_MID_VALUE_NIGHT;
//         }

         if (u32OldValue != m_u32VideoContrastNight)
         {
            // store persistent for early video usage and update current setting
            m_poPersistentData->vSet_Contrast_Night(m_u32VideoContrastNight);
            m_poClient_DevVideo->vSet_Contrast((tU8) m_u32VideoContrastNight);
         }
      }

      // H U E  D A Y
      if (m_HmiUserSetting_u8VideoHueDay <= 100)
      {
         u32OldValue = m_u32VideoHueDay;

         m_u32VideoHueDay = stVideoSettingHmiToDevVideoValueMapping[m_HmiUserSetting_u8VideoHueDay/5].u8DevVideoValue_Hue;

//         for(tU8 u8Count = 0; u8Count < DISPVIDCTRL_DATA_VIDEO_SETTING_MAPPING_TABLE_SIZE; u8Count++)
//         {
//            if(stVideoSettingHmiToDevVideoValueMapping[u8Count].u8HmiValue == (tU8) m_HmiUserSetting_u8VideoHueDay) {
//               m_u32VideoHueDay = stVideoSettingHmiToDevVideoValueMapping[u8Count].u8DevVideoValue_Hue;
//               //ETG_TRACE_USR4((" CA-GRF   :: _vOnUserSettingUpdate_VideoControl(): HUE DAY  HmiValue [%d] DevVideoValue [%d] ", m_HmiUserSetting_u8VideoHueDay, m_u32VideoHueDay));
//            }
//         }

         if (u32OldValue != m_u32VideoHueDay)
         {
            // store persistent for early video usage and update current setting
            m_poPersistentData->vSet_Hue_Day(m_u32VideoHueDay);
            m_poClient_DevVideo->vSet_Hue((tU8) m_u32VideoHueDay);
         }
      }

      // H U E  N I G H T
      if (m_HmiUserSetting_u8VideoHueNight <= 100)
      {
         u32OldValue = m_u32VideoHueNight;

         m_u32VideoHueNight = stVideoSettingHmiToDevVideoValueMapping[m_HmiUserSetting_u8VideoHueNight/5].u8DevVideoValue_Hue;

//         for(tU8 u8Count = 0; u8Count < DISPVIDCTRL_DATA_VIDEO_SETTING_MAPPING_TABLE_SIZE; u8Count++)
//         {
//            if(stVideoSettingHmiToDevVideoValueMapping[u8Count].u8HmiValue == (tU8) m_HmiUserSetting_u8VideoHueNight) {
//               m_u32VideoHueNight = stVideoSettingHmiToDevVideoValueMapping[u8Count].u8DevVideoValue_Hue;
//               //ETG_TRACE_USR4((" CA-GRF   :: _vOnUserSettingUpdate_VideoControl(): HUE NIGHT HmiValue [%d] DevVideoValue [%d] ", m_HmiUserSetting_u8VideoHueNight, m_u32VideoHueNight));
//            }
//         }

         if (u32OldValue != m_u32VideoHueNight)
         {
            // store persistent for early video usage and update current setting
            m_poPersistentData->vSet_Hue_Night(m_u32VideoHueNight);
            m_poClient_DevVideo->vSet_Hue((tU8) m_u32VideoHueNight);
         }
      }

      // B A C K L I G H T  D I M M I N G  D A Y
      if (m_HmiUserSetting_u8VideoDimmingDay != DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
      {
         u32OldValue = m_u32VideoDimmingDay;

         if (m_HmiUserSetting_u8VideoDimmingDay < 50) {
            m_u32VideoDimmingDay = ((((DISPVIDCTRL_VIDEO__BACKLIGHT_DIMMING_MID_VALUE_DAY - DISPVIDCTRL_VIDEO__BACKLIGHT_DIMMING_MIN_VALUE_DAY) * 1000) / 50) * m_HmiUserSetting_u8VideoDimmingDay / 1000) + DISPVIDCTRL_VIDEO__BACKLIGHT_DIMMING_MIN_VALUE_DAY;
         } else {
            m_u32VideoDimmingDay = ((((DISPVIDCTRL_VIDEO__BACKLIGHT_DIMMING_MAX_VALUE_DAY - DISPVIDCTRL_VIDEO__BACKLIGHT_DIMMING_MID_VALUE_DAY) * 1000) / 50) * (m_HmiUserSetting_u8VideoDimmingDay - 50) / 1000) + DISPVIDCTRL_VIDEO__BACKLIGHT_DIMMING_MID_VALUE_DAY;
         }

         if (u32OldValue != m_u32VideoDimmingDay)
         {
            // store persistent for early video usage and update current setting
            m_poPersistentData->vSet_BacklightDimming_Day(m_u32VideoDimmingDay);
//todo: update of back-light dimming (but only when Camera screen active)
         }
      }

      // B A C K L I G H T  D I M M I N G  N I G H T
      if (m_HmiUserSetting_u8VideoDimmingNight != DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
      {
         u32OldValue = m_u32VideoDimmingNight;

         if (m_HmiUserSetting_u8VideoDimmingNight < 50) {
            m_u32VideoDimmingNight = ((((DISPVIDCTRL_VIDEO__BACKLIGHT_DIMMING_MID_VALUE_NIGHT - DISPVIDCTRL_VIDEO__BACKLIGHT_DIMMING_MIN_VALUE_NIGHT) * 1000) / 50) * m_HmiUserSetting_u8VideoDimmingNight / 1000) + DISPVIDCTRL_VIDEO__BACKLIGHT_DIMMING_MIN_VALUE_NIGHT;
         } else {
            m_u32VideoDimmingNight = ((((DISPVIDCTRL_VIDEO__BACKLIGHT_DIMMING_MAX_VALUE_NIGHT - DISPVIDCTRL_VIDEO__BACKLIGHT_DIMMING_MID_VALUE_NIGHT) * 1000) / 50) * (m_HmiUserSetting_u8VideoDimmingNight - 50) / 1000) + DISPVIDCTRL_VIDEO__BACKLIGHT_DIMMING_MID_VALUE_NIGHT;
         }

         if (u32OldValue != m_u32VideoDimmingNight)
         {
            // store persistent for early video usage and update current setting
            m_poPersistentData->vSet_BacklightDimming_Night(m_u32VideoDimmingNight);
//todo: update of back-light dimming (but only when Camera screen active)
         }
      }

      //ETG_TRACE_FATAL(("_vOnUserSettingUpdate_VideoControl write configuration"));
      (tVoid) m_poPersistentData->bSetEarlyNorConfigurationPersistent();
   }
}

/******************************************************************************/
/* FUNCTION     _vOnUserSettingUpdate_DimmingMode                             */
/******************************************************************************/
/**
*  \brief       - called when we have received all linked values relevant for
*                 Dimming Mode update
*
*               - force an update of current settings
*                 (send settings to VideoPlayer or dev/video)
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::_vOnUserSettingUpdate_DimmingMode(tVoid)
{
   if (m_HmiUserSetting_u8DisplayDimmingMode != DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
   {
      if (m_u32DisplayDimmingMode != m_HmiUserSetting_u8DisplayDimmingMode)
      {
         m_u32DisplayDimmingMode = m_HmiUserSetting_u8DisplayDimmingMode;
         // force to send an update of current setting values
         _vSetVideoControls();
      }
   }
}


/******************************************************************************/
/* FUNCTION     _vOnUserSettingUpdate_GuideLine                               */
/******************************************************************************/
/**
*  \brief       - called when we have received all linked values relevant for
*                 one user setting update
*
*                 store values persistent in data pool for vd_early/VideoPlayer
*               - send settings to VideoPlayer
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::_vOnUserSettingUpdate_GuideLine(tVoid)
{
   tBool bMvcRearView = FALSE;
   if((m_Config_bMvcMode) && (OSAL_NULL != m_poControl_Mvc))
   {
      bMvcRearView = m_poControl_Mvc->IsMvcViewModeRear();
      ETG_TRACE_FATAL((" dispvidctrl_tclControl_Video::_vOnUserSettingUpdate_GuideLine MVC Rear View Mode : %d", bMvcRearView));
   }

   if ((m_Config_bRvcMode) || ((m_Config_bMvcMode) && (bMvcRearView))) //For MVC, Update the static guideline toggle index only if MVC view mode is rear view.
   {
      ETG_TRACE_USR4(("dispvidctrl_tclControl_Video::_vOnUserSettingUpdate_GuideLine Dynamic %u Static %u", m_HmiUserSetting_u8GuidelineDynamic, m_HmiUserSetting_u8GuidelineStaticSide));
      if ((m_HmiUserSetting_u8GuidelineDynamic    != DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
      ||  (m_HmiUserSetting_u8GuidelineStaticSide != DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED))
      {
         tU32 u32NewPersistentValue = GUIDELINE_DRAW_NONE;
         
         if(m_Config_bRvcMode)  //update center and dynamic guidelines only for RVC
         {
            if ( m_Config_bCentreGuideline == TRUE &&
                 ((m_au32GuidelineToggleSequence[m_u8CurrentGuidelineIndex]) & ((tU32) GUIDELINE_DRAW_STATIC_CENT)) == (tU32) GUIDELINE_DRAW_STATIC_CENT )
            {
               u32NewPersistentValue |= GUIDELINE_DRAW_STATIC_CENT;
            }

            if (m_Config_bAnticipatoryLine == TRUE && m_HmiUserSetting_u8GuidelineDynamic == 1)
            {
               u32NewPersistentValue |= GUIDELINE_DRAW_DYNAMIC;
               m_u32GuidelineDynamic = TRUE;
            }
            else
            {
               m_u32GuidelineDynamic = FALSE;
            }
         }

         if (m_HmiUserSetting_u8GuidelineStaticSide == 1)
         {
            u32NewPersistentValue |= GUIDELINE_DRAW_STATIC_SIDE;
            m_u32GuidelineStaticSide = TRUE;
         }
         else
         {
            m_u32GuidelineStaticSide = FALSE;
         }

         tU8 u8Index = 0;
         while ( u8Index < m_u8MaxGuidelineTogglesAvailable )
         {
            if ( m_au32GuidelineToggleSequence[u8Index] == u32NewPersistentValue )
            {
               m_u8CurrentGuidelineIndex = u8Index;
               break;
            }
            ++u8Index;
         }

         ETG_TRACE_USR4(("dispvidctrl_tclControl_Video::_vOnUserSettingUpdate_GuideLine Index %u, GuidelinesDisplays 0x%08X", \
                          m_u8CurrentGuidelineIndex, m_au32GuidelineToggleSequence[m_u8CurrentGuidelineIndex]));

         // hold in data pool
         if (m_poPersistentData)
         {
            m_poPersistentData->vSet_GuideLineVisibilityMode(u32NewPersistentValue);
            (tVoid) m_poPersistentData->bSetEarlyNorConfigurationPersistent();
         }
      }
   }
   else if((m_Config_bMvcMode) && (!bMvcRearView))  //Update User setting persistent data for MVC views other than Rear View. 
   {
      tU32 u32NewPersistentValue = GUIDELINE_DRAW_NONE;
      if (m_HmiUserSetting_u8GuidelineStaticSide == 1)
      {
         u32NewPersistentValue |= GUIDELINE_DRAW_STATIC_SIDE;
      }

      // hold in data pool
      if (m_poPersistentData)
      {
         m_poPersistentData->vSet_GuideLineVisibilityMode(u32NewPersistentValue);
         (tVoid) m_poPersistentData->bSetEarlyNorConfigurationPersistent();
      }
   }
}


/******************************************************************************/
/* FUNCTION     _vOnUserSettingUpdate_Language                                */
/******************************************************************************/
/**
*  \brief       - called when we have received all linked values relevant for
*                 one user setting update from HMI
*
*               - store values persistent in data pool for vd_early/VideoPlayer
*               - send settings to VideoPlayer
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::_vOnUserSettingUpdate_Language(tVoid)
{
   if (m_HmiUserSetting_u8Language != DISPVIDCTRL_VIDEO__HMI_USER_SETTING_VALUE_NOT_RECEIVED)
   {
      if (m_u32GuidelineWarningLanguage != m_HmiUserSetting_u8Language)
      {
         // hold local and persistent and send value to player
         // persistent value will be used as AVM language in early use case also
         m_u32GuidelineWarningLanguage = m_HmiUserSetting_u8Language;
         if (m_poClient_VideoPlayer && m_poPersistentData)
         {
            m_poPersistentData->vSet_WarningTextLanguage(m_HmiUserSetting_u8Language);
            //ETG_TRACE_FATAL(("_vOnUserSettingUpdate_Language write configuration"));
            (tVoid) m_poPersistentData->bSetEarlyNorConfigurationPersistent();
            m_poClient_VideoPlayer->vSetWarningTextLanguage(m_HmiUserSetting_u8Language);
         }
      }
   }
}

/******************************************************************************/
/* FUNCTION     _vOnDynamicGuidelineUpdate                                    */
/******************************************************************************/
/**
*  \brief       verify steering angle signals and send update to video player
*
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::_vOnDynamicGuidelineUpdate(tVoid)
{
   tS32  s32SteeringAngleRaw = 0;
   tS32  s32SteeringAnglePhysicalDegree = 0;
   tBool bSteeringAngleReadSuccess = FALSE;

   if (((m_au32GuidelineToggleSequence[m_u8CurrentGuidelineIndex]) & ((tU32) DYN_GUIDELINE_DRAW_DYNAMIC)) == (tU32) DYN_GUIDELINE_DRAW_DYNAMIC)
   {
      if (m_poClientVcan)
      {
         //read steering angle CAN Signal from message 002 or 0C1
         switch (m_Config_u8CameraSteeringMessageType)
         {
            case 0x00: // Msg ID 002
               bSteeringAngleReadSuccess = m_poClientVcan->bGet_SteeringAngleSignal(s32SteeringAngleRaw);
               break;
            case 0x01: // Msg ID 01C
               bSteeringAngleReadSuccess = m_poClientVcan->bGet_FrontSteerPinionAngleSignal(s32SteeringAngleRaw);
               break;
            default:
               break;
         }
      }

      //steering angle signal OK -> draw dynamic guideline
      if(TRUE == bSteeringAngleReadSuccess)
      {
         //adjust steering angle with gained offset
         ETG_TRACE_USR4((" CA-GRF   :: _vOnDynamicGuidelineUpdate(): Steering Angle_raw =  %d Offset_raw = %d ", s32SteeringAngleRaw, m_SteeringAngleAdjust_s32SteeringAngleOffset));
         //calculate physical value from raw data (raw data * 0.1)
         s32SteeringAnglePhysicalDegree =  (tS32) ((s32SteeringAngleRaw - m_SteeringAngleAdjust_s32SteeringAngleOffset) * 0.1);
         ETG_TRACE_USR4((" CA-GRF   :: _vOnDynamicGuidelineUpdate(): Adjusted Steering Angle =  %d ", s32SteeringAnglePhysicalDegree));
      }
   }

   // update video player with steering angle if signal is valid
   // or with steering angle = 0 if signal is not valid or if dynamic guideline shall be not visible
   if(m_poClient_VideoPlayer)
   {
      m_poClient_VideoPlayer->vSetSteeringWheelAngle(s32SteeringAnglePhysicalDegree);
   }
   else
   {
      ETG_TRACE_USR4(("_vOnDynamicGuidelineUpdate() m_poClient_VideoPlayer is null\n"));
   }
}


/******************************************************************************/
/* FUNCTION     _vStartStopSteeringAngleAdjustment                            */
/******************************************************************************/
/**
*  \brief       to start or stop Steering Angle Adjustment depending on
*               AnticipatoryLine setting
*
*               will be called from vStartCommunication
*               or later when m_bAnticipatoryLine is getting changed from
*               Guideline Adjustment Menu (after/confirm configuration list)
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::_vStartStopSteeringAngleAdjustment(tVoid)
{
#ifndef VARIANT_S_FTR_ENABLE_AIVI_SCOPE1 // Only needed for dynamic Guidelines, not needed for Scope1
   if (m_hTimerHandle_SteeringAngleAdjustment !=  OSAL_C_INVALID_HANDLE)
   {
      if ((m_Config_bRvcMode == TRUE) && (m_bAnticipatoryLine == TRUE))
      {
         if (FALSE == m_SteeringAngleAdjust_bActive)
         {
            // start
            (tVoid) OSAL_s32TimerSetTime(m_hTimerHandle_SteeringAngleAdjustment, (OSAL_tMSecond) DISPVIDCTRL_VIDEO__STEERING_ANGLE_ADJUST__RECHECK_TIME_NO_ADJUST_IN_MS, 0);
            // set state SteeringAngleAdjustment to active
            m_SteeringAngleAdjust_bActive = TRUE;
         }
      }
      else
      {
         if (TRUE == m_SteeringAngleAdjust_bActive)
         {
            // stop and reset relevant variables to inital values
            (tVoid) OSAL_s32TimerSetTime(m_hTimerHandle_SteeringAngleAdjustment, 0, 0);

            m_SteeringAngleAdjust_bRecheck = FALSE;
            m_SteeringAngleAdjust_bBackupAvailable = FALSE;
            m_SteeringAngleAdjust_u32RecheckTimeInMs = 1000;
            m_SteeringAngleAdjust_s32SteeringAngleOffset = 0;

            // set state SteeringAngleAdjustment to not active
            m_SteeringAngleAdjust_bActive = FALSE;
         }
      }
   }
#endif
}

/******************************************************************************/
/* FUNCTION     _vOnSteeringAngleAdjustment                                   */
/******************************************************************************/
/**
*  \brief       check if all conditions are given to calculate
*               new steering angle center offset (calculation with raw data)
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::_vOnSteeringAngleAdjustment(tVoid)
{
   tS32  s32SteeringAngleDeg = 0;
   tU32  u32VehicleSpeedValue = 0;
   tBool bSteeringAngleReadSuccess = FALSE;
   tBool bNewRecheckRequest = FALSE;

   //------- check if all preconditions for calibration are given --------------
   //Ignition is ON
   if ((m_poClientVcan) && (TRUE  == m_SteeringAngleAdjust_bIgnition))
   {
      //SteeringAngle is between +/- 10 degree
      switch (m_Config_u8CameraSteeringMessageType)
      {
         //read steering angle CAN Signal from message 002 or 0C1
         case 0x00: // Msg ID 002
            bSteeringAngleReadSuccess = m_poClientVcan->bGet_SteeringAngleSignal(s32SteeringAngleDeg);
            break;
         case 0x01: // Msg ID 01C
            bSteeringAngleReadSuccess = m_poClientVcan->bGet_FrontSteerPinionAngleSignal(s32SteeringAngleDeg);
            break;
         default:
            break;
      }
      if((TRUE == bSteeringAngleReadSuccess) && ((s32SteeringAngleDeg >= -100) && (s32SteeringAngleDeg <= 100)) )
      {
         //VehicleSpeed > 20 km/h
         if ( (TRUE == m_poClientVcan->bGet_VehicleSpeedSignal(u32VehicleSpeedValue)) &&
                  (u32VehicleSpeedValue > (tU32) 20) )
         {
            //------- start calibration after successful validation, or prepare the recheck timing -------

            bNewRecheckRequest = TRUE;
            //after validation timeout the conditions are still fulfilled, calculate now the offset.
            if (TRUE == m_SteeringAngleAdjust_bRecheck)
            {
               if(TRUE == m_SteeringAngleAdjust_bBackupAvailable )
               {
                  //average steering angle offset with backupdata
                  if (0 != m_SteeringAngleAdjust_s32SteeringAngleOffset)
                  {
                     m_SteeringAngleAdjust_s32SteeringAngleOffset = (tS32) ((m_SteeringAngleAdjust_s32SteeringAngleOffset + s32SteeringAngleDeg) / 2);
                  }
                  else
                  {
                     m_SteeringAngleAdjust_s32SteeringAngleOffset = (tS32) (s32SteeringAngleDeg / 2);
                  }
               }
               else
               {
                  //calculate first SteeringAngleOffset value
                  if (0 != s32SteeringAngleDeg)
                  {
                     m_SteeringAngleAdjust_s32SteeringAngleOffset = s32SteeringAngleDeg / 2;
                  }
                  //Up now a backup is available
                  m_SteeringAngleAdjust_bBackupAvailable = TRUE;
               }
               //derive from vehicle speed the time in which the car will probably move 100 meter.
               m_SteeringAngleAdjust_u32RecheckTimeInMs = (DISPVIDCTRL_VIDEO__STEERING_ANGLE_ADJUST__RECHECK_CONST_WITH_BACKUP / u32VehicleSpeedValue);
               ETG_TRACE_USR4((" CA-VID   :: _vOnSteeringAngleAdjustment() m_SteeringAngleAdjust_s32SteeringAngleOffset = %d deg(raw)", m_SteeringAngleAdjust_s32SteeringAngleOffset));
            }
            else
            {
               //select and derive proper recheck timing for validation
               m_SteeringAngleAdjust_bRecheck = TRUE;
               if (TRUE == m_SteeringAngleAdjust_bBackupAvailable)
               {
                  //derive from vehicle speed the time in which the car will probably move 100 meter
                  m_SteeringAngleAdjust_u32RecheckTimeInMs = ((tU32) DISPVIDCTRL_VIDEO__STEERING_ANGLE_ADJUST__RECHECK_CONST_WITH_BACKUP / u32VehicleSpeedValue);
               }
               else
               {
                  //derive from vehicle speed the time in which the car will probably move 12.5 meter
                  m_SteeringAngleAdjust_u32RecheckTimeInMs = (DISPVIDCTRL_VIDEO__STEERING_ANGLE_ADJUST__RECHECK_CONST_NO_BACKUP / u32VehicleSpeedValue);
               }
            }
         }
      }
   }

   //start recheck timer with gained value
   if (FALSE == bNewRecheckRequest)
   {
      m_SteeringAngleAdjust_u32RecheckTimeInMs = DISPVIDCTRL_VIDEO__STEERING_ANGLE_ADJUST__RECHECK_TIME_NO_ADJUST_IN_MS;
      m_SteeringAngleAdjust_bRecheck = FALSE;
   }
   if (OSAL_OK != OSAL_s32TimerSetTime(m_hTimerHandle_SteeringAngleAdjustment, (OSAL_tMSecond) m_SteeringAngleAdjust_u32RecheckTimeInMs, 0))
   {
      ETG_TRACE_ERR((" CA-VID   :: _vOnSteeringAngleAdjustment() set OSAL_s32TimerSetTime to %d ms FAILED!!!        ", m_SteeringAngleAdjust_u32RecheckTimeInMs));
   }
}

/******************************************************************************/
/* FUNCTION     _vSetVideoControls                                            */
/******************************************************************************/
/**
*  \brief       set values via /dev/video
*
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::_vSetVideoControls(tVoid)
{
   // Dimming mode has changed from day to night or vice versa or we want to have a update.
   // send values to dev/video
   if(m_poClient_DevVideo != 0)
   {
      if (m_u32DisplayDimmingMode == DISPVIDCTRL_VIDEO__DISPLAY_DIMMING_MODE_DAY)
      {
         ETG_TRACE_USR4(("_vSetVideoControls - m_u32VideoBlackLevelDay: [%d], m_u32VideoBrightnessDay: [%d], m_u32VideoColourDay: [%d], m_u32VideoContrastDay: [%d], m_u32VideoHueDay: [%d], m_u32VideoDimmingDay: [%d]", m_u32VideoBlackLevelDay, m_u32VideoBrightnessDay, m_u32VideoColourDay, m_u32VideoContrastDay, m_u32VideoHueDay, m_u32VideoDimmingDay));

         m_poClient_DevVideo->vSet_BlackLevel((tU8) m_u32VideoBlackLevelDay);
         m_poClient_DevVideo->vSet_Brightness((tU8) m_u32VideoBrightnessDay);
         m_poClient_DevVideo->vSet_Colour((tU8) m_u32VideoColourDay);
         m_poClient_DevVideo->vSet_Contrast((tU8) m_u32VideoContrastDay);
         m_poClient_DevVideo->vSet_Hue((tU8) m_u32VideoHueDay);
//todo: backlight dimming value is going to client handler dimming maybe
//         if (FALSE == m_poClient_DevVideo->bSend_Dimming(m_u32VideoDimmingDay))
//         {
//            ETG_TRACE_ERR(("setting of m_u32VideoDimmingDay failed"));
//         }
      }
      else
      {
         ETG_TRACE_USR4(("_vSetVideoControls - m_u32VideoBlackLevelNight: [%d], m_u32VideoBrightnessNight: [%d], m_u32VideoColourNight: [%d], m_u32VideoContrastNight: [%d], m_u32VideoHueNight:  [%d], m_u32VideoDimmingNight:  [%d]", m_u32VideoBlackLevelNight, m_u32VideoBrightnessNight, m_u32VideoColourNight, m_u32VideoContrastNight, m_u32VideoHueNight, m_u32VideoDimmingNight));

         m_poClient_DevVideo->vSet_BlackLevel((tU8) m_u32VideoBlackLevelNight);
         m_poClient_DevVideo->vSet_Brightness((tU8) m_u32VideoBrightnessNight);
         m_poClient_DevVideo->vSet_Contrast((tU8) m_u32VideoContrastNight);
         m_poClient_DevVideo->vSet_Colour((tU8) m_u32VideoColourNight);
         m_poClient_DevVideo->vSet_Hue((tU8) m_u32VideoHueNight);
//todo: backlight dimming value is going to client handler dimming maybe
//         if (FALSE == m_poClient_DevVideo->bSend_Dimming(m_u32VideoDimmingNight))
//         {
//            ETG_TRACE_ERR(("setting of m_u32VideoDimmingNight failed"));
//         }
      }
   }
}

/******************************************************************************/
/* FUNCTION     _vResetVideoControls                                          */
/******************************************************************************/
/**
*  \brief       used when activating Diagnosis Camera Screen to get a "neutral"
*               setting for the video controls
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::_vResetVideoControls(tVoid)
{
   //set values via dev/video
   if(m_poClient_DevVideo)
   {
      ETG_TRACE_USR4(("_vSetVideoControls - DiagScreen: VideoBrightnessDay: [50], VideoContrast: [50], VideoColour: [50], BlackLevel: [50], VideoHue: [50]"));
      m_poClient_DevVideo->vSet_BlackLevel(50);
      m_poClient_DevVideo->vSet_Brightness(50);
      m_poClient_DevVideo->vSet_Colour(50);
      m_poClient_DevVideo->vSet_Contrast(50);
      m_poClient_DevVideo->vSet_Hue(50);

      // disable also Adaptive Contrast Enhancement
      m_poClient_DevVideo->vSet_AceEnable(FALSE);
   }
}

template<typename T>
tVoid dispvidctrl_tclControl_Video::_vSwapContents(T& a, T& b)
{
   T temp;
   temp = static_cast<T>(a);
   a = static_cast<T>(b);
   b = static_cast<T>(temp);

   return;
}

/******************************************************************************/
/* FUNCTION     _vSetVisibility_Guideline                                     */
/******************************************************************************/
/**
*  \brief       send guideline visibility to video player
*
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::_vSetVisibility_Guideline(tVoid)
{
   //send values to player
   if(m_poClient_VideoPlayer != NULL)
   {
      tU32 u32ValueToSend = (tU32) GUIDELINE_DRAW_NONE;

      if (m_bGraphicsVisibility)
      {
         if (m_Config_bRvcMode || m_Config_bMvcMode)
         {
            u32ValueToSend = m_au32GuidelineToggleSequence[m_u8CurrentGuidelineIndex];
         }
      }

      ETG_TRACE_USR4(("_vSetVisibility_Guideline() sending CMD_SETTING [%d] GuidelineVisibility = [%d] to VideoPlayer", SETTING_GUIDELINEVISIBILITY, u32ValueToSend));
      m_poClient_VideoPlayer->vSetGuideLineVisibility(u32ValueToSend | GUIDELINE_FORCE_UPDATE);
   }
}

/******************************************************************************/
/* FUNCTION     _vSetVisibility_WarningText                                   */
/******************************************************************************/
/**
*  \brief       send warning text visibility to video player
*
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::_vSetVisibility_WarningText(tVoid)
{
   //send values to player
   if(m_poClient_VideoPlayer != 0)
   {
      tU32 u32ValueToSend = (tU32) DISPVIDCTRL_VIDEO__WARNINGTEXT_NOT_VISIBLE;

      if (m_bGraphicsVisibility) {         
         u32ValueToSend = (tU32) DISPVIDCTRL_VIDEO__WARNINGTEXT_VISIBLE;
      }

      ETG_TRACE_USR4(("_vSetVisibility_WarningText() sending CMD_SETTING [%d] GuidelineWarningText = [%d] to VideoPlayer", SETTING_GUIDELINEWARNINGTEXT, u32ValueToSend));
      m_poClient_VideoPlayer->vSetWarningTextVisibility(u32ValueToSend);
   }
}

/******************************************************************************/
/* FUNCTION     _vSetLayerSetting                                             */
/******************************************************************************/
/**
*  \brief       send warning text visibility to video player
*
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::_vSetLayerSetting(const tU32 VideoState)
{
   ETG_TRACE_USR4(("_vSetLayerSetting() sending VideoState [%d] to VideoPlayer", ETG_CENUM(tenControl_VideoResize, VideoState)));

   tU8 u8TempVideoResizeIndex = m_u8ConfiguredVideoType;

   TDisplayWindowLayout VideoDestination = {0,0,0,0};
   TDisplayWindowLayout VideoSource      = {0,0,0,0};
   
   if ( RVC_AVM_VIDEO_SETTINGS_SCREEN == VideoState )
   {
      if ( u8TempVideoResizeIndex < IDX_MAX )
      {
         VideoSource = mapCameraTypeVideoParam[u8TempVideoResizeIndex].tVideoSettingsSource;
         VideoDestination = mapCameraTypeVideoParam[u8TempVideoResizeIndex].tVideoSettingsDestination;
      }
   }
   else if ( RVC_AVM_VIDEO_FORCE_FULL_SCREEN == VideoState ) // full video display size
   {
      VideoSource.u16xOffset = DISPVIDCTRL_VIDEO__CAMERAXOFFSET;
      VideoSource.u16yOffset = DISPVIDCTRL_VIDEO__CAMERAYOFFSET;
      VideoSource.u16Width   = DISPVIDCTRL_VIDEO__CAMERAWIDTH;
      VideoSource.u16Height  = DISPVIDCTRL_VIDEO__CAMERAHEIGHT;
      
      VideoDestination.u16xOffset = DISPVIDCTRL_VIDEO__CAMERAXOFFSET;
      VideoDestination.u16yOffset = DISPVIDCTRL_VIDEO__CAMERAYOFFSET;
      
      // Even for portrait the height & width of the destination rectangle must be same as Landscape - according to the current PSD
      VideoDestination.u16Width   = DISPVIDCTRL_VIDEO__LANDSCAPE_DISPLAY1WIDTH;
      VideoDestination.u16Height  = DISPVIDCTRL_VIDEO__LANDSCAPE_DISPLAY1HEIGHT;
   }
   else // according to the HMI PSD's
   {
      // If full video screen  = 0
      // If video with Upa     = 1
      // If video with Upa Hfp = 2
      u8TempVideoResizeIndex += VideoState;
      
      if ( u8TempVideoResizeIndex < IDX_MAX )
      {
         VideoSource = mapCameraTypeVideoParam[u8TempVideoResizeIndex].tVideoSource;
         VideoDestination = mapCameraTypeVideoParam[u8TempVideoResizeIndex].tVideoDestination;
      
         // if LHD use the video resize parameters as it is 
         if ( FALSE == m_Config_bLeftHandSteeringPosition ) // RHD
         {
            // horizontal mirroring of the video window
            // only the x-offset needs to be adapted according to the LHD layout
            if ( DISPVIDCTRL_AllianceDisplay_Format_Landscape == m_u8DisplayOrientation )
            {
               VideoDestination.u16xOffset = DISPVIDCTRL_VIDEO__LANDSCAPE_DISPLAY1WIDTH - ( VideoDestination.u16xOffset + VideoDestination.u16Width );
            }
            else
            {
               VideoDestination.u16xOffset = DISPVIDCTRL_VIDEO__PORTRAIT_DISPLAY1WIDTH - ( VideoDestination.u16xOffset + VideoDestination.u16Width );
            }
         }
      }
   }
   
   if ( m_poClient_VideoPlayer && (u8TempVideoResizeIndex < IDX_MAX) )  // if u8TempVideoResizeIndex were >=IDX_MAX, VideoSource and VideoDestination would be uninitialized, so better check
   {
      // Video Layer
      m_poClient_VideoPlayer->vSetLayerOrSurfaceSettings(SURFACE_SOURCE_RECTANGLE, 
                                                         DEFAULT_RVCSURFACE, 
                                                         VideoSource.u16xOffset, 
                                                         VideoSource.u16yOffset, 
                                                         VideoSource.u16Width,
                                                         VideoSource.u16Height,
                                                         FALSE);

      m_poClient_VideoPlayer->vSetLayerOrSurfaceSettings(LAYER_DESTINATION_RECTANGLE, 
                                                         DEFAULT_RVCLAYER, 
                                                         VideoDestination.u16xOffset,
                                                         VideoDestination.u16yOffset,
                                                         VideoDestination.u16Width,
                                                         VideoDestination.u16Height,
                                                         FALSE);

      // Graphics Layer   
      m_poClient_VideoPlayer->vReConfigGuidelineParams(VideoState, u8TempVideoResizeIndex);
      
      _vSetVisibility_Guideline();
      
      m_poClient_VideoPlayer->vSetLayerOrSurfaceSettings(COMMIT_CHANGES, 0, 0, 0, 0, 0, TRUE);
   }
}

//##############################################################################
// Data Management
//##############################################################################

/******************************************************************************/
/* FUNCTION     _vGetPersistentData                                           */
/******************************************************************************/
/**
*  \brief       Read the persistent data and set the member data values
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::_vGetPersistentData(tVoid)
{
   // Read all data values
   if (m_poPersistentData)
   {
      (tVoid) m_poPersistentData->bGetPersistent_BlackLevel_Day(m_u32VideoBlackLevelDay);
      (tVoid) m_poPersistentData->bGetPersistent_BlackLevel_Night(m_u32VideoBlackLevelNight);
      (tVoid) m_poPersistentData->bGetPersistent_Brightness_Day(m_u32VideoBrightnessDay);
      (tVoid) m_poPersistentData->bGetPersistent_Brightness_Night(m_u32VideoBrightnessNight);
      (tVoid) m_poPersistentData->bGetPersistent_Colour_Day(m_u32VideoColourDay);
      (tVoid) m_poPersistentData->bGetPersistent_Colour_Night(m_u32VideoColourNight);
      (tVoid) m_poPersistentData->bGetPersistent_Contrast_Day(m_u32VideoContrastDay);
      (tVoid) m_poPersistentData->bGetPersistent_Contrast_Night(m_u32VideoContrastNight);
      (tVoid) m_poPersistentData->bGetPersistent_Hue_Day(m_u32VideoHueDay);
      (tVoid) m_poPersistentData->bGetPersistent_Hue_Night(m_u32VideoHueNight);
      (tVoid) m_poPersistentData->bGetPersistent_BacklightDimming_Day(m_u32VideoDimmingDay);
      (tVoid) m_poPersistentData->bGetPersistent_BacklightDimming_Night(m_u32VideoDimmingNight);

      (tVoid) m_poPersistentData->bGetPersistent_WarningTextLanguage(m_u32GuidelineWarningLanguage);

      tU32 u32Temp = 0;
      (tVoid) m_poPersistentData->bGetPersistent_GuideLineVisibilityMode(u32Temp);

      if (((u32Temp) & ((tU32) DYN_GUIDELINE_DRAW_DYNAMIC)) == (tU32) DYN_GUIDELINE_DRAW_DYNAMIC)
      {
          m_u32GuidelineDynamic = TRUE;
      }
      else
      {
          m_u32GuidelineDynamic = FALSE;
      }
      if (((u32Temp) & ((tU32) GUIDELINE_DRAW_STATIC_SIDE)) == (tU32) GUIDELINE_DRAW_STATIC_SIDE)
      {
          m_u32GuidelineStaticSide = TRUE;
      }
      else
      {
          m_u32GuidelineStaticSide = FALSE;
      }
      
      tU8 u8VideoType = static_cast<tU8>(IDX_LANDSCAPE_RVC);
      (tVoid) m_poPersistentData->bGet_VideoResizeIndex(u8VideoType);

      if ( u8VideoType < IDX_MAX )
      {
         m_u8ConfiguredVideoType = u8VideoType;
         ETG_TRACE_FATAL(("_vGetPersistentData() m_u8ConfiguredVideoType = %u", ETG_CENUM(enVideoParamMapIndex, m_u8ConfiguredVideoType)));
      }

      if(m_Config_bMvcMode)
      {
         m_HmiUserSetting_u8GuidelineStaticSide = m_u32GuidelineStaticSide;  //At startup get the HMI User GuideLineVisibilityMode - NCG3D-77436
      }
   }
}

/******************************************************************************/
/* FUNCTION     vSetGuardTimeRemaining                                        */
/******************************************************************************/
/**
*  \brief       Set new guard time remaining for video camera
*
*  \param       value as tU32
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::vSetGuardTimeRemaining(tU32 new_guard_time_remaining, tBool& bValueChanged)
{
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(bValueChanged);

   ETG_TRACE_USR4(("bSetGuardTimeRemaining() new Guard Time remaining = %d [%d]",
                   new_guard_time_remaining,
                   rvc_guard_time_remaining));

   //Should be nothing to do as we have set the property
}





//#########################################################################################
// Functions for Incoming Status Handling
//#########################################################################################

/******************************************************************************/
/* FUNCTION     bSetRVCSTATUSINFO                                             */
/******************************************************************************/
/**
*  \brief       Set PARKINGSYSTEMSTATUSINFO property (VideoActive and enRvcError)
*
*  \param       tBool video_active - TRUE if video is being captured at this time
*               tBool rvc_error_state - indicating current video signal state
*  \return      none
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Video::bSetRVCSTATUSINFO(tBool bVideoActive, tBool bVideoSignalError)
{
   ETG_TRACE_USR1(("bSetRVCSTATUSINFO() entered - setting VideoCapturingActive = %d [%d], video signal state = %d [%d]", bVideoActive, rvc_video_active, (tU8) bVideoSignalError, rvc_video_status));
   tBool bRetVal = FALSE;
   tU32 tU32FlagVideoStatusActive = 0x00000000;

   rvc_video_active = bVideoActive;
   rvc_video_status = bVideoSignalError;

   if (bVideoActive)
   {
      tU32FlagVideoStatusActive = (tU32) EN_VIDEO_STATUS__ON;
      
      // only when video is active we can have "no video signal error"
      if (FALSE == bVideoSignalError)
      {
         // bitwise OR
         tU32FlagVideoStatusActive |= DISPVIDCTRL_VIDEO__MASK_VIDEO_SIGNAL_STATUS;
      }
   }
   else
   {
      tU32FlagVideoStatusActive = (tU32) EN_VIDEO_STATUS__OFF;

      // video signal error by default when video capturing is not active
   }

   // inform control about video active / signal status
   dispvidctrl_tclControl* m_poControl = dynamic_cast<dispvidctrl_tclControl*>(_cpoMain->getHandler("dispvidctrl_tclControl"));
   DISPVIDCTRL_NULL_POINTER_CHECK_VAL(m_poControl);

   m_poControl->vHandleControl(EN_CONTROL__VIDEO_STATUS_RECEIVED, tU32FlagVideoStatusActive);
   tChar strIfName[] = "";
   dispvidctrl_tclAppMain::theServer()->vPostMsgVideoStatus(strIfName, tU32FlagVideoStatusActive);

   return bRetVal;
}

/******************************************************************************/
/* FUNCTION     bSetGUARDTIMEREMAINING                                        */
/******************************************************************************/
/**
*  \brief       Set GUARD_TIME_REMAINING property (time remaining)
*
*  \param       tU32 time_remaining - time remaining to display RVC in milliseconds
*  \return      none
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Video::bSetGUARDTIMEREMAINING(tU32 time_remaining)
{
   ETG_TRACE_USR4(("bSetGUARDTIMEREMAINING() entered - setting time_remaining = %d [%d]", time_remaining, rvc_guard_time_remaining));
   tBool bRetVal = TRUE;

   rvc_guard_time_remaining = time_remaining;

   return bRetVal;
}

/******************************************************************************/
/* FUNCTION     bSetANIMATIONSTATUS                                           */
/******************************************************************************/
/**
*  \brief       Set ANIMATIONSTATUS property (Animation State)
*
*  \param       int animation_state - current value of Animation Status
*  \return      none
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Video::bSetANIMATIONSTATUS(tU32  u32AnimationState)
{
   ETG_TRACE_USR4(("bSetANIMATIONSTATUS() entered - setting Animation State = %d [%d]", u32AnimationState, u32CurrentAnimationStatus));
   tBool bRetVal = FALSE;

   u32CurrentAnimationStatus = u32AnimationState;

   return bRetVal;
}

/******************************************************************************/
/* FUNCTION     bIsVideoSourceFound                                           */
/******************************************************************************/
/**
*  \brief       Set ANIMATIONSTATUS property (Animation State)
*
*  \param       int animation_state - current value of Animation Status
*  \return      none
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Video::bIsVideoSourceFound() const
{
   DISPVIDCTRL_NULL_POINTER_CHECK_VAL(m_poClient_VideoPlayer);
   return (m_poClient_VideoPlayer->bIsVideoSourceFound());
}

/******************************************************************************/
/* FUNCTION     bFindVideoSource                                              */
/******************************************************************************/
/**
*  \brief       Set ANIMATIONSTATUS property (Animation State)
*
*  \param       int animation_state - current value of Animation Status
*  \return      none
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Video::bFindVideoSource() const
{
   DISPVIDCTRL_NULL_POINTER_CHECK_VAL(m_poClient_VideoPlayer);
   return (m_poClient_VideoPlayer->bFindVideoSource());
}

/******************************************************************************/
/* FUNCTION     _bSetTimer_DynGuidelineUpdate                                 */
/******************************************************************************/
/**
*  \brief       start or stop the timer
*
*  \param       timeout value
*  \return      tBool - timer start success state
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Video::_bSetTimer_DynGuidelineUpdate( OSAL_tMSecond msTimeout )
{
   tBool bSuccess = TRUE;
   if ( OSAL_C_INVALID_HANDLE != m_hTimerHandle_DynGuidelineUpdate )   {
      if (msTimeout)
      {
         if ( OSAL_s32TimerSetTime( m_hTimerHandle_DynGuidelineUpdate, msTimeout, msTimeout ) != OSAL_OK )   {
            bSuccess = FALSE;
         }
      }
      else
      {
         if ( OSAL_s32TimerSetTime( m_hTimerHandle_DynGuidelineUpdate, 0, 0 ) != OSAL_OK )   {
            bSuccess = FALSE;
         }
      }
   }
   return(bSuccess);
}

/******************************************************************************/
/* FUNCTION     _pfCallbackTimer_DynGuidelineUpdate                           */
/******************************************************************************/
/**
*  \brief
*
*  \param       pArg - pointer to this class
*  \return      none
*/
/******************************************************************************/
OSAL_tpfCallback dispvidctrl_tclControl_Video::_pfCallbackTimer_DynGuidelineUpdate(tVoid* pArg)
{
   // get pointer to myself from argument
   dispvidctrl_tclControl_Video* po_tclControl_Video = (dispvidctrl_tclControl_Video*)pArg;
   if (po_tclControl_Video)
   {
      if ((po_tclControl_Video->_cpoMain) && (po_tclControl_Video->_cpoMain->theServer()))
      {
         // post loop-back message to Control_Video
         tChar strIfName[] = "dispvidctrl_tclControl_Video";
         po_tclControl_Video->_cpoMain->theServer()->vPostMsgControlVideo(strIfName, EN_CONTROL_VIDEO__DYNAMIC_GUIDELINE_UPDATE);
      }
   }
   return 0;
}

/******************************************************************************/
/* FUNCTION     _pfCallbackTimer_SteeringAngleAdjustment                      */
/******************************************************************************/
/**
*  \brief
*
*  \param       pArg - pointer to this class
*  \return      none
*/
/******************************************************************************/
OSAL_tpfCallback dispvidctrl_tclControl_Video::_pfCallbackTimer_SteeringAngleAdjustment(tVoid* pArg)
{
   // get pointer to myself from argument
   dispvidctrl_tclControl_Video* po_tclControl_Video = (dispvidctrl_tclControl_Video*)pArg;
   if (po_tclControl_Video)
   {
      if ((po_tclControl_Video->_cpoMain) && (po_tclControl_Video->_cpoMain->theServer()))
      {
         // post loop-back message to Control_Video
         tChar strIfName[] = "dispvidctrl_tclControl_Video";
         po_tclControl_Video->_cpoMain->theServer()->vPostMsgControlVideo(strIfName, EN_CONTROL_VIDEO__STEERING_ANGLE_ADJUSTMENT);
      }
   }
   return 0;
}

/******************************************************************************/
/* FUNCTION     _pfCallbackTimer_SteeringAngleGuidlineAdjustment              */
/******************************************************************************/
/**
*  \brief
*
*  \param       pArg - pointer to this class
*  \return      none
*/
/******************************************************************************/
OSAL_tpfCallback dispvidctrl_tclControl_Video::_pfCallbackTimer_SteeringAngleGuidelineAdjustment(tVoid* pArg)
{
   // get pointer to myself from argument
   dispvidctrl_tclControl_Video* po_tclControl_Video = (dispvidctrl_tclControl_Video*)pArg;
   if (po_tclControl_Video)
   {
      if ((po_tclControl_Video->_cpoMain) && (po_tclControl_Video->_cpoMain->theServer()))
      {
         // post loop-back message to Control_Video
         tChar strIfName[] = "dispvidctrl_tclControl_Video";
         po_tclControl_Video->_cpoMain->theServer()->vPostMsgControlVideo(strIfName, EN_CONTROL_VIDEO__GUIDELINE_ADJUSTMENT_UPDATE);
      }
   }
   return 0;
}

/******************************************************************************/
/* FUNCTION     _bSetTimer_SteeringAngleGuidelineAdjustment                   */
/******************************************************************************/
/**
*  \brief       start or stop the timer
*
*  \param       timeout value
*  \return      tBool - timer start success state
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Video::_bSetTimer_SteeringAngleGuidelineAdjustment( OSAL_tMSecond msTimeout )
{
   tBool bSuccess = TRUE;
   if ( OSAL_C_INVALID_HANDLE != m_hTimerHandle_SteeringAngleGuidelineAdjustment )   {
      if (msTimeout)
      {
         if ( OSAL_s32TimerSetTime( m_hTimerHandle_SteeringAngleGuidelineAdjustment, msTimeout, msTimeout ) != OSAL_OK )   {
            bSuccess = FALSE;
         }
      }
      else
      {
         if ( OSAL_s32TimerSetTime( m_hTimerHandle_SteeringAngleGuidelineAdjustment, 0, 0 ) != OSAL_OK )   {
            bSuccess = FALSE;
         }
      }
   }
   return(bSuccess);
}

//##############################################################################
// TTFis
//##############################################################################
/******************************************************************************/
/* FUNCTION     vResetConfig                                                  */
/******************************************************************************/
/**
*  \brief       Reset configuration to initial settings
*
*  \param       value as tU32
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::vResetConfig(tVoid)
{
   ETG_TRACE_FATAL(("vResetConfig() Set configuration to initial settings"));

   _cpoMain->dispvidctrl_tclAppMain::theServer()->vHandleConfigUpdate();

   //Now restart
}


/******************************************************************************/
/* FUNCTION:     _bReadAdvRegister                                            */
/******************************************************************************/
/**
*  \brief       to read ADV register via I2C.
*
*  \param       u8RegisterAddress, tU8& u8RegisterData
*  \return      read success state
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Video::_bReadAdvRegister(const tU8 u8SlaveAddress, const tU8 u8RegisterAddress, tU8& u8RegisterData)
{
   // To read an register value the ADV requires an 4 Byte sequence.
   // The first Byte contains the address of the device(7bit) and the R/W bit. For
   // transfer master -> slave ( write register) this bit has to be 0.
   // The next Byte contains the address of the register which shall
   // be accessed. After the second byte one more start condition (repeated start condition)
   // has to be sent followed by a byte again containing the device address and the R/W Bit. After
   // that the device will send the contents of the register and waits for an acknowledge from the master.

   // This sequence can't be realised with an standard "write" followed by an "read" access because
   // in between there would be an stop condition and not an repeated start condition.
   // We need to use the ioctl "I2C_RDWR" which is doing combined read/write transaction without stop in between.

   // https://www.kernel.org/doc/Documentation/i2c/dev-interface

   ETG_TRACE_USR4(("bReadAdvRegister(register address: 0x%02x) entered.", u8RegisterAddress ));

   tBool bResult = FALSE;
   const char *dev = "/dev/i2c-1";
   //const tU8  u8SlaveAddress = 0x20;
   tS32 fd;
   tS32 s32ReturnValue;

   // open I2C bus to read the register
   fd = open(dev, O_RDWR);
   if(fd == -1)
   {
      ETG_TRACE_ERR(("bReadAdvRegister: I2C bus open failed!"));
   }
   else
   {
      // set device address of the slave
      s32ReturnValue = ioctl(fd, I2C_SLAVE, u8SlaveAddress);
      if(s32ReturnValue < 0)
      {
         ETG_TRACE_ERR(("bReadAdvRegister: setting slave address failed! - error: %d", s32ReturnValue));
      }
      else
      {
         // prepare message sequence
         tU8 aReadBuffer[1] = {0};
         tU8 aWriteBuffer[1] = {0};
         struct i2c_msg messages[2];
         memset(messages, OSAL_NULL, sizeof(messages));
         struct i2c_rdwr_ioctl_data data;


         messages[0].addr = u8SlaveAddress;
         messages[0].flags = 0x0; //write mode
         messages[0].len = 1;
         messages[0].buf = aWriteBuffer;
         messages[0].buf[0] = u8RegisterAddress;

         messages[1].addr = u8SlaveAddress;
         messages[1].flags = I2C_M_RD; // read mode
         messages[1].len = 1;
         messages[1].buf = aReadBuffer;

         data.msgs = messages;
         data.nmsgs = 2;

         // read data / register
         s32ReturnValue = ioctl(fd, I2C_RDWR, &data);
         if(s32ReturnValue < 0)
         {
           ETG_TRACE_ERR(("bReadAdvRegister: reading of data failed! - error: %d", s32ReturnValue));
         }
         else
         {
            // set return data
            u8RegisterData = aReadBuffer[0];
            bResult = TRUE;
         }
      }
      // close I2C bus
      close(fd);
   }
   ETG_TRACE_FATAL(("bReadAdvRegister success:%d, register: 0x%02x data: 0x%02x) ", bResult, u8RegisterAddress, u8RegisterData));
   return bResult;
}



/******************************************************************************/
/* FUNCTION:     _bWriteAdvRegister                                           */
/******************************************************************************/
/**
*  \brief       to write ADV register via I2C.
*
*  \param       const tU8 u8RegisterAddress, tU8 u8RegisterData
*  \return      tBool - write success
*/
/******************************************************************************/
tBool dispvidctrl_tclControl_Video::_bWriteAdvRegister(const tU8 u8RegisterAddress, tU8 u8RegisterData) const
{
   tBool bResult = FALSE;
   const char *dev = "/dev/i2c-1";
   const tU8  u8SlaveAddress = 0x20;
   tS32 fd;

   // open I2C bus to read the register
   fd = open(dev, O_RDWR);
   if(fd == -1)
   {
      ETG_TRACE_ERR(("bWriteAdvRegister: I2C bus open failed!"));
   }
   else
   {
      // set device address of the slave
      if( ioctl(fd, I2C_SLAVE, u8SlaveAddress) < 0)
      {
         ETG_TRACE_ERR(("bWriteAdvRegister: setting slave address failed!"));
      }
      else
      {
         tU8 aWriteBuffer[2] = {0};
         aWriteBuffer[0] = u8RegisterAddress; //Address of Register
         aWriteBuffer[1] = u8RegisterData; //Data
         if (write(fd, aWriteBuffer, 2) != 2)
         {
            ETG_TRACE_ERR(("bWriteAdvRegister: Error in writing Register"));
         }
         else
         {
            bResult = TRUE;
         }
      }
      close(fd);
   }
   return (bResult);
}


/******************************************************************************/
/* FUNCTION     vTraceInfo()                                                  */
/******************************************************************************/
/**
*  \brief       trace information
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclControl_Video::vTraceInfo()
{
   _vGdlAdj_TraceInfo();

   ETG_TRACE_FATAL(("   CA-VID :: vTraceInfo() ************ CA-VID states ****************************************"));
   ETG_TRACE_FATAL(("   CA-VID :: m_Config_bRvcMode ......................... %-40u", m_Config_bRvcMode));
   ETG_TRACE_FATAL(("   CA-VID :: m_Config_bMvcMode ......................... %-40u", m_Config_bMvcMode));
   ETG_TRACE_FATAL(("   CA-VID :: m_Config_bAvmMode ......................... %-40u", m_Config_bAvmMode));
   ETG_TRACE_FATAL(("   CA-VID :: m_Config_bAvmRenaultMode .................. %-40u", m_Config_bAvmRenaultMode));
   
   ETG_TRACE_FATAL(("   CA-VID :: m_Config_bCentreGuideline ................. %-40u", m_Config_bCentreGuideline));
   ETG_TRACE_FATAL(("   CA-VID :: m_Config_bAnticipatoryLine ................ %-40u", m_Config_bAnticipatoryLine));
   ETG_TRACE_FATAL(("   CA-VID :: m_Config_bLeftHandSteeringPosition ........ %-40u", m_Config_bLeftHandSteeringPosition));
   ETG_TRACE_FATAL(("   CA-VID :: m_Config_u8BrightnessValueDay ............. %-40u", m_Config_u8BrightnessValueDay));
   ETG_TRACE_FATAL(("   CA-VID :: m_Config_u8BrightnessValueNight ........... %-40u", m_Config_u8BrightnessValueNight));
   ETG_TRACE_FATAL(("   CA-VID :: m_Config_bBlackWarningTextBackground ...... %-40u", m_Config_bBlackWarningTextBackground));
   ETG_TRACE_FATAL(("   CA-VID :: m_Config_u8RvcLensParameterSet ............ %-40u", m_Config_u8RvcLensParameterSet));
   ETG_TRACE_FATAL(("   CA-VID :: m_Config_u8Brand .......................... %-40u", m_Config_u8Brand));
   ETG_TRACE_FATAL(("   CA-VID :: m_Config_u16VehicleType ................... 0x%04X",m_Config_u16VehicleType));

   ETG_TRACE_FATAL(("   CA-VID :: m_u32GuidelineWarningLanguage ............. %-40u", m_u32GuidelineWarningLanguage));

   ETG_TRACE_FATAL(("   CA-VID :: m_bDiagScreenRequest ...................... %-40u", m_bDiagScreenRequest));
   ETG_TRACE_FATAL(("   CA-VID :: rvc_guard_time_remaining .................. %-40u", rvc_guard_time_remaining));
   ETG_TRACE_FATAL(("   CA-VID :: rvc_video_active .......................... %-40u", rvc_video_active));
   ETG_TRACE_FATAL(("   CA-VID :: rvc_video_status .......................... %-40u", rvc_video_status));
   ETG_TRACE_FATAL(("   CA-VID :: u32CurrentAnimationStatus ................. %-40u", u32CurrentAnimationStatus));
   ETG_TRACE_FATAL(("   CA-VID :: m_bAnticipatoryLine ....................... %-40u", m_bAnticipatoryLine));

   ETG_TRACE_FATAL(("   CA-VID :: m_bGraphicsVisibility...................... %-40u", m_bGraphicsVisibility));
   ETG_TRACE_FATAL(("   CA-VID :: m_u32GuidelineStaticSide (HMI Usr Settg) .. %-40u", m_HmiUserSetting_u8GuidelineStaticSide));
   ETG_TRACE_FATAL(("   CA-VID :: m_u32GuidelineDynamic (HMI User Setting) .. %-40u", m_HmiUserSetting_u8GuidelineDynamic));

   ETG_TRACE_FATAL(("   CA-VID :: m_u32GuidelineStaticSide .................. %-40u", m_u32GuidelineStaticSide));
   ETG_TRACE_FATAL(("   CA-VID :: m_u32GuidelineDynamic ..................... %-40u", m_u32GuidelineDynamic));

   ETG_TRACE_FATAL(("   CA-VID :: m_u32DisplayDimmingMode ................... %-40u", m_u32DisplayDimmingMode));
   ETG_TRACE_FATAL(("   CA-VID :: m_u32VideoBlackLevelDay ................... %-40u", m_u32VideoBlackLevelDay));
   ETG_TRACE_FATAL(("   CA-VID :: m_u32VideoBlackLevelNight ................. %-40u", m_u32VideoBlackLevelNight));
   ETG_TRACE_FATAL(("   CA-VID :: m_u32VideoBrightnessDay ................... %-40u", m_u32VideoBrightnessDay));
   ETG_TRACE_FATAL(("   CA-VID :: m_u32VideoBrightnessNight ................. %-40u", m_u32VideoBrightnessNight));
   ETG_TRACE_FATAL(("   CA-VID :: m_u32VideoColourDay ....................... %-40u", m_u32VideoColourDay));
   ETG_TRACE_FATAL(("   CA-VID :: m_u32VideoColourNight ..................... %-40u", m_u32VideoColourNight));
   ETG_TRACE_FATAL(("   CA-VID :: m_u32VideoContrastDay ..................... %-40u", m_u32VideoContrastDay));
   ETG_TRACE_FATAL(("   CA-VID :: m_u32VideoContrastNight ................... %-40u", m_u32VideoContrastNight));
   ETG_TRACE_FATAL(("   CA-VID :: m_u32VideoHueDay .......................... %-40u", m_u32VideoHueDay));
   ETG_TRACE_FATAL(("   CA-VID :: m_u32VideoHueNight ........................ %-40u", m_u32VideoHueNight));

   ETG_TRACE_FATAL(("   CA-VID :: m_SteeringAngleAdjust_bIgnition ........... %-40u", m_SteeringAngleAdjust_bIgnition));
   ETG_TRACE_FATAL(("   CA-VID :: m_SteeringAngleAdjust_bRecheck ............ %-40u", m_SteeringAngleAdjust_bRecheck));
   ETG_TRACE_FATAL(("   CA-VID :: m_SteeringAngleAdjust_bBackupAvailable .... %-40u", m_SteeringAngleAdjust_bBackupAvailable));
   ETG_TRACE_FATAL(("   CA-VID :: m_SteeringAngleAdjust_u32RecheckTimeInMs .. %-40u", m_SteeringAngleAdjust_u32RecheckTimeInMs));
   ETG_TRACE_FATAL(("   CA-VID :: m_SteeringAngleAdjust_s32SteeringAngleOffset %39u", m_SteeringAngleAdjust_s32SteeringAngleOffset));
   ETG_TRACE_FATAL(("   CA-VID :: m_SteeringAngleAdjust_bActive ............. %-40u", m_SteeringAngleAdjust_bActive));
   
   ETG_TRACE_FATAL(("   CA-VID :: m_u8ConfiguredVideoType ................... %u",    ETG_CENUM(enVideoParamMapIndex, m_u8ConfiguredVideoType)));
   ETG_TRACE_FATAL(("   CA-VID :: *********************************************************************************"));
}

