/******************************************************************************/
/**
* \file    dispvidctrl_tclClient_DevVideo.cpp
* \ingroup
*
* \brief   Interface to dev/video (video input) to control setting
*          (e.g. brightness, colour, ...)
*
* \remark  Copyright : (c) 2015 Robert Bosch GmbH, Hildesheim
* \remark  Author    : Michael Niemann
* \remark  Scope     : AIVI
*
* \todo
*/
/******************************************************************************/

/*******************************************************************************
                        Includes
*******************************************************************************/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#include "dispvidctrl_tclClient_DevVideo.h"

#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <linux/videodev2.h>
#include <linux/v4l2-controls.h>
#include <sys/ioctl.h>
#include "dispvidctrl_datapool.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_CLIENT_DEV_VIDEO
#include "trcGenProj/Header/dispvidctrl_tclClient_DevVideo.cpp.trc.h"
#endif



/*******************************************************************************
                        Defines
*******************************************************************************/
#define SCALE_INPUT_RANGE_MAX 100  // 0 to 100%
#define SCALE_INPUT_RANGE_MIDDLE 50   // to get 0x00 for two's complement register value after scaling
#define SCALE_OUTPUT_RANGE_MAX 255 // 0 to 255 (8 Bit ADV-register value)
#define SCALE_OUTPUT_RANGE_MIDDLE 128   // to get 0x00 for two's complement register value after scaling

#define CONVERSION_TO_TWOS_COMPLEMENT 1
#define CONVERSION_TO_SIGNED          2

enum tenDevVideo_Controls
{
   EN_CONTROL_DEV_VIDEO__BRIGHTNESS = 0x00980900,
   EN_CONTROL_DEV_VIDEO__CONTRAST,
   EN_CONTROL_DEV_VIDEO__COLOUR,
   EN_CONTROL_DEV_VIDEO__HUE,
   EN_CONTROL_DEV_VIDEO__GAMMA      = 0x00980916,
   EN_CONTROL_DEV_VIDEO__AUTOGAIN   = 0x00980918,
   EN_CONTROL_DEV_VIDEO__GAIN,

   EN_CONTROL_DEV_VIDEO__ADV718x_OFF_CB = 0x00981980,
   EN_CONTROL_DEV_VIDEO__ADV718x_OFF_CR,
   EN_CONTROL_DEV_VIDEO__ADV718x_FREERUN_ENABLE,
   EN_CONTROL_DEV_VIDEO__ADV718x_FORCE_FREERUN,
   EN_CONTROL_DEV_VIDEO__ADV718x_FREERUN_Y,
   EN_CONTROL_DEV_VIDEO__ADV718x_FREERUN_CB,
   EN_CONTROL_DEV_VIDEO__ADV718x_FREERUN_CR,
   /* Chroma Transient Improvement Controls */
   EN_CONTROL_DEV_VIDEO__ADV718x_CTI_ENABLE,
   EN_CONTROL_DEV_VIDEO__ADV718x_CTI_AB_ENABLE,
   EN_CONTROL_DEV_VIDEO__ADV718x_CTI_AB,
   EN_CONTROL_DEV_VIDEO__ADV718x_CTI_THRESH,
   /* Digital Noise Reduction and Luminance Peaking Gain Controls */
   EN_CONTROL_DEV_VIDEO__ADV718x_DNR_ENABLE,
   EN_CONTROL_DEV_VIDEO__ADV718x_DNR_THRESH1,
   EN_CONTROL_DEV_VIDEO__ADV718x_LUMA_PEAK_GAIN,
   EN_CONTROL_DEV_VIDEO__ADV718x_DNR_THRESH2,
   /* ADV7182 specific controls */
   EN_CONTROL_DEV_VIDEO__ADV7182_FREERUN_PAT_SEL,
   EN_CONTROL_DEV_VIDEO__ADV7182_ACE_ENABLE,
   EN_CONTROL_DEV_VIDEO__ADV7182_ACE_LUMA_GAIN,
   EN_CONTROL_DEV_VIDEO__ADV7182_ACE_RESPONSE_SPEED,
   EN_CONTROL_DEV_VIDEO__ADV7182_ACE_CHROMA_GAIN,
   EN_CONTROL_DEV_VIDEO__ADV7182_ACE_CHROMA_MAX,
   EN_CONTROL_DEV_VIDEO__ADV7182_ACE_GAMMA_GAIN,
   EN_CONTROL_DEV_VIDEO__ADV7182_DITHER_ENABLE
};


/******************************************************************************/
/* FUNCTION     dispvidctrl_tclClient_DevVideo()                              */
/******************************************************************************/
/**
*  \brief       constructor
*
*  \param       pointer to main application
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclClient_DevVideo::dispvidctrl_tclClient_DevVideo(const dispvidctrl_tclAppMain* poMainAppl)
: dispvidctrl_tclBaseIf(poMainAppl)
, m_fd_DevVideo(-1)
, m_pocdevVideo_In(NULL)
, m_Config_u8CameraConnectionType(DISPVIDCTRL_AllianceCameraConnectionType_CVBS)
{
   ETG_TRACE_USR4(("dispvidctrl_tclClient_DevVideo() entered."));

   // To avoid lint info for not referenced enum values lint Info 749.
   // Try to find a better solution for this. "//lint !749" seems not working.
   (tVoid) EN_CONTROL_DEV_VIDEO__BRIGHTNESS;
   (tVoid) EN_CONTROL_DEV_VIDEO__CONTRAST;
   (tVoid) EN_CONTROL_DEV_VIDEO__COLOUR;
   (tVoid) EN_CONTROL_DEV_VIDEO__HUE;
   (tVoid) EN_CONTROL_DEV_VIDEO__GAMMA;
   (tVoid) EN_CONTROL_DEV_VIDEO__AUTOGAIN;
   (tVoid) EN_CONTROL_DEV_VIDEO__GAIN;
   (tVoid) EN_CONTROL_DEV_VIDEO__ADV718x_OFF_CB;
   (tVoid) EN_CONTROL_DEV_VIDEO__ADV718x_OFF_CR;
   (tVoid) EN_CONTROL_DEV_VIDEO__ADV718x_FREERUN_ENABLE;
   (tVoid) EN_CONTROL_DEV_VIDEO__ADV718x_FORCE_FREERUN;
   (tVoid) EN_CONTROL_DEV_VIDEO__ADV718x_FREERUN_Y;
   (tVoid) EN_CONTROL_DEV_VIDEO__ADV718x_FREERUN_CB;
   (tVoid) EN_CONTROL_DEV_VIDEO__ADV718x_FREERUN_CR;
   (tVoid) EN_CONTROL_DEV_VIDEO__ADV718x_CTI_ENABLE;
   (tVoid) EN_CONTROL_DEV_VIDEO__ADV718x_CTI_AB_ENABLE;
   (tVoid) EN_CONTROL_DEV_VIDEO__ADV718x_CTI_AB;
   (tVoid) EN_CONTROL_DEV_VIDEO__ADV718x_CTI_THRESH;
   (tVoid) EN_CONTROL_DEV_VIDEO__ADV718x_DNR_ENABLE;
   (tVoid) EN_CONTROL_DEV_VIDEO__ADV718x_DNR_THRESH1;
   (tVoid) EN_CONTROL_DEV_VIDEO__ADV718x_LUMA_PEAK_GAIN;
   (tVoid) EN_CONTROL_DEV_VIDEO__ADV718x_DNR_THRESH2;
   (tVoid) EN_CONTROL_DEV_VIDEO__ADV7182_FREERUN_PAT_SEL;
   (tVoid) EN_CONTROL_DEV_VIDEO__ADV7182_ACE_ENABLE;
   (tVoid) EN_CONTROL_DEV_VIDEO__ADV7182_ACE_LUMA_GAIN;
   (tVoid) EN_CONTROL_DEV_VIDEO__ADV7182_ACE_RESPONSE_SPEED;
   (tVoid) EN_CONTROL_DEV_VIDEO__ADV7182_ACE_CHROMA_GAIN;
   (tVoid) EN_CONTROL_DEV_VIDEO__ADV7182_ACE_CHROMA_MAX;
   (tVoid) EN_CONTROL_DEV_VIDEO__ADV7182_ACE_GAMMA_GAIN;
   (tVoid) EN_CONTROL_DEV_VIDEO__ADV7182_ACE_CHROMA_GAIN;
   (tVoid) EN_CONTROL_DEV_VIDEO__ADV7182_DITHER_ENABLE;

   _u16Brightness = 0;
   _u16BlackLevel = 0;

   m_poClient_VideoPlayer = NULL;
}

/******************************************************************************/
/* FUNCTION     ~dispvidctrl_tclClient_DevVideo                               */
/******************************************************************************/
/**
*  \brief       destructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclClient_DevVideo::~dispvidctrl_tclClient_DevVideo(tVoid)
{
   ETG_TRACE_USR4(("~dispvidctrl_tclClient_DevVideo() entered."));

   vCloseDevVideo();
   
   if (m_pocdevVideo_In)
   {
      delete m_pocdevVideo_In;
      m_pocdevVideo_In = NULL;
   }

   m_poClient_VideoPlayer = NULL;
}

/******************************************************************************/
/* FUNCTION     vGetReferences                                                */
/******************************************************************************/
/**
*  \brief       Function to get all reference needed by this class.
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclClient_DevVideo::vGetReferences(tVoid)
{
   ETG_TRACE_USR4(("dispvidctrl_tclClient_DevVideo::vGetReferences() entered."));

   m_poClient_VideoPlayer = dynamic_cast<I_dispvidctrl_tclVideoPlayer*>(_cpoMain->getHandler("I_dispvidctrl_tclVideoPlayer"));
   DISPVIDCTRL_NULL_POINTER_CHECK(m_poClient_VideoPlayer);
}

/******************************************************************************/
/* FUNCTION     vGetConfiguration                                             */
/******************************************************************************/
/**
*  \brief       Function to get all configuration values needed by this class.
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclClient_DevVideo::vGetConfiguration(const TConfiguration* pStConfigurationValues)
{
   ETG_TRACE_USR4(("dispvidctrl_tclClient_DevVideo::vGetConfiguration() entered."));

   m_Config_u8CameraConnectionType = pStConfigurationValues->u8CameraConnectionType;
   ETG_TRACE_USR4(("dispvidctrl_tclClient_DevVideo::vGetConfiguration() CameraConnectionType %d", m_Config_u8CameraConnectionType));
}

/******************************************************************************/
/* FUNCTION     vStartCommunication                                           */
/******************************************************************************/
/**
*  \brief       Function to start all dynamic objects e.g. threads, ...
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclClient_DevVideo::vStartCommunication(tVoid)
{
   ETG_TRACE_USR4(("dispvidctrl_tclClient_DevVideo::vStartCommunication() entered."));

   // find the /dev/video to use it for video control settings (Brightness, Colour, ...)
   (tVoid) _bFindDevVideo();
}

/******************************************************************************/
/* FUNCTION     vHandleMessage(TMsg* pMsg)                                    */
/******************************************************************************/
/**
*  \brief       Handle worker events.
*
*  \param       message pointer
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclClient_DevVideo::vHandleMessage(dispvidctrl_tclBaseIf::TMsg* pMsg)
{
   ETG_TRACE_USR4(("dispvidctrl_tclClient_DevVideo::vHandleMessage() entered %u -> data: %d.", ETG_CENUM(dispvidctrl_tclBaseIf::ECmdTypes , (tU32)pMsg->eCmd), pMsg->u.u32Data));
}


/******************************************************************************/
/* FUNCTION     _bFindDevVideo                                                */
/******************************************************************************/
/**
*  \brief       to open dev video
*
*  \param       none
*  \return      tBool (Result state)
*/
/******************************************************************************/
tBool dispvidctrl_tclClient_DevVideo::_bFindDevVideo(tVoid)
{
   // In order to find out which device to use, we have to read the "name" file
   // present in each videoX subdirectory of /sys/class/video4linux/ (X is 0,1,2 or 3).
   // For the Analogue Camera, if the name file content matches with "ipu1_csi0 capture" we have
   // found the right video device (videoX)
   // For the Digital Camera, if the name file content matches with "ipu1_csi1 capture" we have
   // found the right video device (videoX)

   tBool bResult = FALSE;
   char  aDeviceName[30];
   FILE* fp;
   char devVideo_0[] = "/dev/video0";
   char devVideo_1[] = "/dev/video1";
   char devVideo_2[] = "/dev/video2";
   char devVideo_3[] = "/dev/video3";
   char devVideo_4[] = "/dev/video4";
   char* devVideo_In = 0;

   // close device in case it was open for any reason
   if (m_fd_DevVideo != -1)
   {
      close(m_fd_DevVideo);
   }
   m_fd_DevVideo = -1;


   // find out the device name of video0 is correct
   fp = fopen("/sys/class/video4linux/video0/name", "r");
   if (fp == 0)
   {
      //error for name file open
   }
   else
   {
      if (fgets(aDeviceName, 30, fp) == 0)
      {
         //error for getting the string
      }
      else
      {
         ETG_TRACE_USR1(("_bFindDevVideo() /sys/class/video4linux/video0/name = %s", aDeviceName));
         if (m_Config_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_CVBS)
         {
            if (0 == strcmp("ipu1_csi0 capture\n", aDeviceName))
            {
               bResult = TRUE;
               ETG_TRACE_USR1(("_bFindDevVideo() Analogue Camera device is /dev/video0"));
               devVideo_In = devVideo_0;
            }
         }
         if (m_Config_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_LVDS)
         {
            if (0 == strcmp("ipu1_csi1 capture\n", aDeviceName))
            {
               bResult = TRUE;
               ETG_TRACE_USR1(("_bFindDevVideo() Digital Camera device is /dev/video0"));
               devVideo_In = devVideo_0;
            }
         }
      }
      fclose(fp);
   }

   if (devVideo_In == 0)
   {
      // check if device name of video1 is correct
      fp = fopen("/sys/class/video4linux/video1/name", "r");
      if (fp == 0)
      {
         //error for name file open
      }
      else
      {
         if (fgets(aDeviceName, 30, fp) == 0)
         {
            //error for getting the string
         }
         else
         {
            ETG_TRACE_USR1(("_bFindDevVideo() /sys/class/video4linux/video1/name = %s", aDeviceName));
            if (m_Config_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_CVBS)
            {
               if (0 == strcmp("ipu1_csi0 capture\n", aDeviceName))
               {
                  bResult = TRUE;
                  ETG_TRACE_USR1(("_bFindDevVideo() Analogue Camera device is /dev/video1"));
                  devVideo_In = devVideo_1;
               }
            }
            if (m_Config_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_LVDS)
            {
               if (0 == strcmp("ipu1_csi1 capture\n", aDeviceName))
               {
                  bResult = TRUE;
                  ETG_TRACE_USR1(("_bFindDevVideo() Digital Camera device is /dev/video1"));
                  devVideo_In = devVideo_1;
               }
            }
         }
         fclose(fp);
      }
   }

   if (devVideo_In == 0)
   {
      // check if device name of video2 is correct
      fp = fopen("/sys/class/video4linux/video2/name", "r");
      if (fp == 0)
      {
         //error for name file open
      }
      else
      {
         if (fgets(aDeviceName, 30, fp) == 0)
         {
            //error for getting the string
         }
         else
         {
            ETG_TRACE_USR1(("_bFindDevVideo() /sys/class/video4linux/video2/name = %s", aDeviceName));
            if (m_Config_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_CVBS)
            {
               if (0 == strcmp("ipu1_csi0 capture\n", aDeviceName))
               {
                  bResult = TRUE;
                  ETG_TRACE_USR1(("_bFindDevVideo() Analogue Camera device is /dev/video2"));
                  devVideo_In = devVideo_2;
               }
            }
            if (m_Config_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_LVDS)
            {
               if (0 == strcmp("ipu1_csi1 capture\n", aDeviceName))
               {
                  bResult = TRUE;
                  ETG_TRACE_USR1(("_bFindDevVideo() Digital Camera device is /dev/video2"));
                  devVideo_In = devVideo_2;
               }
            }
         }
         fclose(fp);
      }
   }
   
   if (devVideo_In == 0)
   {
      // check if device name of video3 is correct
      fp = fopen("/sys/class/video4linux/video3/name", "r");
      if (fp == 0)
      {
         //error for name file open
      }
      else
      {
         if (fgets(aDeviceName, 30, fp) == 0)
         {
            //error for getting the string
         }
         else
         {
            ETG_TRACE_USR1(("_bFindDevVideo() /sys/class/video4linux/video3/name = %s", aDeviceName));
            if (m_Config_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_CVBS)
            {
               if (0 == strcmp("ipu1_csi0 capture\n", aDeviceName))
               {
                  bResult = TRUE;
                  ETG_TRACE_USR1(("_bFindDevVideo() Analogue Camera device is /dev/video3"));
                  devVideo_In = devVideo_3;
               }
            }
            if (m_Config_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_LVDS)
            {
               if (0 == strcmp("ipu1_csi1 capture\n", aDeviceName))
               {
                  bResult = TRUE;
                  ETG_TRACE_USR1(("_bFindDevVideo() Digital Camera device is /dev/video3"));
                  devVideo_In = devVideo_3;
               }
            }
         }
         fclose(fp);
      }
   }

   if (devVideo_In == 0)
   {
      // check if device name of video3 is correct
      fp = fopen("/sys/class/video4linux/video4/name", "r");
      if (fp == 0)
      {
         //error for name file open
      }
      else
      {
         if (fgets(aDeviceName, 30, fp) == 0)
         {
            //error for getting the string
         }
         else
         {
            ETG_TRACE_USR1(("_bFindDevVideo() /sys/class/video4linux/video4/name = %s", aDeviceName));
            if (m_Config_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_CVBS)
            {
               if (0 == strcmp("ipu1_csi0 capture\n", aDeviceName))
               {
                  bResult = TRUE;
                  ETG_TRACE_USR1(("_bFindDevVideo() Analogue Camera device is /dev/video4"));
                  devVideo_In = devVideo_4;
               }
            }
            if (m_Config_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_LVDS)
            {
               if (0 == strcmp("ipu1_csi1 capture\n", aDeviceName))
               {
                  bResult = TRUE;
                  ETG_TRACE_USR1(("_bFindDevVideo() Digital Camera device is /dev/video4"));
                  devVideo_In = devVideo_4;
               }
            }
         }
         fclose(fp);
      }
   }

   if (bResult)
   {
      tU8 u8Size = strlen(devVideo_In);
      m_pocdevVideo_In = new char[u8Size + 1];
      if (m_pocdevVideo_In)
      {
         strncpy(m_pocdevVideo_In, devVideo_In, u8Size);
         m_pocdevVideo_In[u8Size] = 0;
         ETG_TRACE_FATAL(("_bFindDevVideo() Camera device is %s", m_pocdevVideo_In));
      }
      else 
      {
         ETG_TRACE_FATAL(("_bFindDevVideo() Camera device is %s Not Selected", m_pocdevVideo_In));
      }
   }
   
   return bResult;
}

/******************************************************************************/
/* FUNCTION     vOpenDevVideo                                                 */
/******************************************************************************/
/**
*  \brief       to open dev video
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclClient_DevVideo::vOpenDevVideo()
{
   if (m_pocdevVideo_In)
   {
      m_fd_DevVideo = open(m_pocdevVideo_In, O_RDWR);
      if (m_fd_DevVideo != -1)
      {
         ETG_TRACE_FATAL(("vOpenDevVideo() open dev video success"));
      }
      else
      {
         ETG_TRACE_FATAL(("vOpenDevVideo() open dev video failed"));
      }
   }
}

/******************************************************************************/
/* FUNCTION     vCloseDevVideo                                                */
/******************************************************************************/
/**
*  \brief       to close dev video
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclClient_DevVideo::vCloseDevVideo(tVoid)
{
   if (m_fd_DevVideo != -1)
   {
      close(m_fd_DevVideo);
   }

   m_fd_DevVideo = -1;
}


/******************************************************************************/
/* FUNCTION     vSet_BlackLevel                                               */
/******************************************************************************/
/**
*  \brief       set new value
*
*  \param       settings value
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclClient_DevVideo::vSet_BlackLevel(tU8 u8Value)
{
   // Save setting for Black Level and set the new Combined Brightness level
   _u16BlackLevel = u8Value;
   vSet_Brightness((tU8)_u16Brightness);
}

/******************************************************************************/
/* FUNCTION     vSet_Brightness                                               */
/******************************************************************************/
/**
*  \brief       set new value
*
*  \param       settings value
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclClient_DevVideo::vSet_Brightness(tU8 u8Value)
{
   // Save setting for Brightness and set the new Combined Brightness level
   _u16Brightness = u8Value;

   if (m_Config_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_LVDS) {
       if(m_poClient_VideoPlayer != 0) {
          m_poClient_VideoPlayer->vSetLVDSBrightness(_u16Brightness, _u16BlackLevel);
       }
       else {
          ETG_TRACE_FATAL(("dispvidctrl_tclClient_DevVideo::vSet_Brightness() m_poClient_VideoPlayer is NULL"));
       }
   }
   else {
      struct v4l2_control rCtrl;
      //tU8  u8ApiValue_Brightness;
      tU32 u32ScaledValue;


      tU16 u16CombinedBrightness = u8Value + _u16BlackLevel;

      if(u16CombinedBrightness > 100) {
         ETG_TRACE_FATAL( ( "vSet_Brightness() Combined Brightness > 100 - brightness [%d], black level [%d]", (int)u8Value, _u16BlackLevel));
         u16CombinedBrightness = 100;
      }

      // scaling from 0..100% to 0..255 ADV-register value
      if (u16CombinedBrightness ==  SCALE_INPUT_RANGE_MIDDLE) // to get 0x00 for two's complement register value after scaling
      {
         u32ScaledValue = SCALE_OUTPUT_RANGE_MIDDLE;
      }
      else
      {
         u32ScaledValue = (u16CombinedBrightness * SCALE_OUTPUT_RANGE_MAX) / SCALE_INPUT_RANGE_MAX;
      }

//   // ADV brightness register requires a two's complement format
//   if (u32ScaledValue >= 128)
//   {
//      // API value 0..127 for increase (0x00..0x7F)
//      u8ApiValue_Brightness = (tU8) (u32ScaledValue - 128);
//   }
//   else
//   {
//      // API value from -1 to -128 for decrease (0xFF..0x80)
//      u8ApiValue_Brightness = (tU8) (u32ScaledValue + 128);
//   }

      ETG_TRACE_USR1( ( "vSet_Brightness() brightness [%d], black level [%d], scaled [%d]", (int)u8Value, _u16BlackLevel, u32ScaledValue));

      rCtrl.id = V4L2_CID_BRIGHTNESS;
      //rCtrl.value = (tS32) u8ApiValue_Brightness;
      rCtrl.value = (tS32) u32ScaledValue;

      if (m_fd_DevVideo != -1)
      {
         if (ioctl(m_fd_DevVideo, VIDIOC_S_CTRL, &rCtrl) < 0)
         {
            ETG_TRACE_USR1(("ERROR vSet_Brightness(): ioctl failed to set"));
         }
      }
      else
      {
         ETG_TRACE_USR1(("ERROR vSet_Brightness():m_fd_DevVideo invalid"));
      }
   }
}

/******************************************************************************/
/* FUNCTION     vSet_Colour                                                   */
/******************************************************************************/
/**
*  \brief       set new value
*
*  \param       settings value
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclClient_DevVideo::vSet_Colour(tU8 u8Value) const
{
   if (m_Config_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_LVDS) {
       if(m_poClient_VideoPlayer != 0) {
          m_poClient_VideoPlayer->vSetLVDSColour(u8Value);
       }
       else {
          ETG_TRACE_FATAL(("dispvidctrl_tclClient_DevVideo::vSet_Colour() m_poClient_VideoPlayer is NULL"));
       }
   }
   else {
      struct v4l2_control rCtrl;
      tU32 u32ScaledValue;

      // scaling from 0..100% to 0..255 ADV-register value
      u32ScaledValue = (u8Value * SCALE_OUTPUT_RANGE_MAX) / SCALE_INPUT_RANGE_MAX;

      ETG_TRACE_USR1(("vSet_Colour() colour [%d], scaled [%d]",(int) u8Value, u32ScaledValue));

      rCtrl.id = V4L2_CID_SATURATION;
      rCtrl.value = (tS32) u32ScaledValue;

      if (m_fd_DevVideo != -1)
      {
         if (ioctl(m_fd_DevVideo, VIDIOC_S_CTRL, &rCtrl) < 0)
         {
            ETG_TRACE_USR1(("ERROR vSet_Colour(): ioctl failed to set"));
         }
      }
      else
      {
         ETG_TRACE_USR1(("ERROR vSet_Colour():m_fd_DevVideo invalid"));
      }
   }
}

/******************************************************************************/
/* FUNCTION     vSet_Contrast                                                 */
/******************************************************************************/
/**
*  \brief       set new value
*
*  \param       settings value
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclClient_DevVideo::vSet_Contrast(tU8 u8Value) const
{
   if (m_Config_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_LVDS) {
       if(m_poClient_VideoPlayer != 0) {
          m_poClient_VideoPlayer->vSetLVDSContrast(u8Value);
       }
       else {
          ETG_TRACE_FATAL(("dispvidctrl_tclClient_DevVideo::vSet_Contrast() m_poClient_VideoPlayer is NULL"));
       }
   }
   else {
      struct v4l2_control rCtrl;
      tU32 u32ScaledValue;

      // scaling from 0..100% to 0..255 ADV-register value
      u32ScaledValue = (u8Value * SCALE_OUTPUT_RANGE_MAX) / SCALE_INPUT_RANGE_MAX;

      ETG_TRACE_USR1(("vSet_Contrast() contrast [%d], scaled [%d] ", (int) u8Value, u32ScaledValue));

      rCtrl.id = V4L2_CID_CONTRAST;
      rCtrl.value = (tS32) u32ScaledValue;

      if (m_fd_DevVideo != -1)
      {
         if (ioctl(m_fd_DevVideo, VIDIOC_S_CTRL, &rCtrl) < 0)
         {
            ETG_TRACE_USR1(("ERROR vSet_Contrast(): ioctl failed to set"));
         }
      }
      else
      {
         ETG_TRACE_USR1(("ERROR vSet_Contrast():m_fd_DevVideo invalid"));
      }
   }
}

/******************************************************************************/
/* FUNCTION     vSet_Hue                                                      */
/******************************************************************************/
/**
*  \brief       set new value
*
*  \param       settings value
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclClient_DevVideo::vSet_Hue(tU8 u8Value) const
{
   if (m_Config_u8CameraConnectionType == DISPVIDCTRL_AllianceCameraConnectionType_LVDS) {
       if(m_poClient_VideoPlayer != 0) {
          m_poClient_VideoPlayer->vSetLVDSHue(u8Value);
       }
       else {
          ETG_TRACE_FATAL(("dispvidctrl_tclClient_DevVideo::vSet_Hue() m_poClient_VideoPlayer is NULL"));
       }
   }
   else {
      struct v4l2_control rCtrl;
      tS32 s32ScaledValue;

      s32ScaledValue = (tS32) u8Value;

      // The ADV Hue register requires a value -128..0..127.
      // A positive register value is doing a negative phase shift and a negative value a positive phase shift of the chroma signal
      // Maybe because of that phase inversion, the dev_video interface is doing an inversion also and therefore the value range we can set is -127..0..128 and not -128..0..127.
      // Here a code snipet which I got from Suresh Dhandapani (RBEI/ECF11) which is showing the inversion/negation of variable "tmp":
      // case V4L2_CID_HUE:
      //     dev_dbg(sensor->dev, "   V4L2_CID_HUE\n");
      //     tmp = ctrl->val
      //     /* Hue is inverted according to HSL chart */
      //     adv7182_write_reg(sensor, ADV_REG_ADDR_HUE, -tmp);
      //     break;

      s32ScaledValue = s32ScaledValue - 127;

      ETG_TRACE_USR1(("vSet_Hue() hue [%d], scaled [%d]", (int) u8Value, s32ScaledValue));

      rCtrl.id = V4L2_CID_HUE;
      rCtrl.value = s32ScaledValue;

      if (m_fd_DevVideo != -1)
      {
         if (ioctl(m_fd_DevVideo, VIDIOC_S_CTRL, &rCtrl) < 0)
         {
            ETG_TRACE_USR1(("ERROR vSet_Hue(): ioctl failed to set"));
         }
      }
      else
      {
         ETG_TRACE_USR1(("ERROR vSet_Hue():m_fd_DevVideo invalid"));
      }
   }
}


/******************************************************************************/
/* Set Functions related to ADAPTIVE CONTRAST ENHAMCEMENT                     */
/******************************************************************************/
/**
*  \brief       set functions for ADAPTIVE CONTRAST ENHAMCEMENT (ACE)
*               to enable/disable ACE
*               as well as to set
*                Luma Gain,
*                Response Speed,
*                Chroma Gain,
*                Chroma Max,
*                Gamma Gain
*/
/******************************************************************************/
tVoid dispvidctrl_tclClient_DevVideo::vSet_AceEnable(tBool bEnable) const
{
   struct v4l2_control rCtrl;

   //rCtrl.id    = V4L2_CID_ADV7182_ACE_ENABLE;
   rCtrl.id    = EN_CONTROL_DEV_VIDEO__ADV7182_ACE_ENABLE;
   rCtrl.value = (tS32) bEnable;

   if (m_fd_DevVideo != -1)
   {
      if (ioctl(m_fd_DevVideo, VIDIOC_S_CTRL, &rCtrl) < 0)
      {
         ETG_TRACE_USR1(("ERROR vSet_AceEnable(): ioctl failed to set"));
      }
   }
}

tVoid dispvidctrl_tclClient_DevVideo::vSet_AceLumeGain(tU8 u8Value) const
{
   struct v4l2_control rCtrl;

   //rCtrl.id    = V4L2_CID_ADV7182_ACE_LUMA_GAIN;
   rCtrl.id    = EN_CONTROL_DEV_VIDEO__ADV7182_ACE_LUMA_GAIN;
   rCtrl.value = (tS32) u8Value;

   if (m_fd_DevVideo != -1)
   {
      if (ioctl(m_fd_DevVideo, VIDIOC_S_CTRL, &rCtrl) < 0)
      {
         ETG_TRACE_USR1(("ERROR vSet_AceLumeGain(): ioctl failed to set"));
      }
   }
}

tVoid dispvidctrl_tclClient_DevVideo::vSet_AceResponseSpeed(tU8 u8Value) const
{
   struct v4l2_control rCtrl;

   //rCtrl.id    = V4L2_CID_ADV7182_ACE_RESPONSE_SPEED;
   rCtrl.id    = EN_CONTROL_DEV_VIDEO__ADV7182_ACE_RESPONSE_SPEED;
   rCtrl.value = (tS32) u8Value;

   if (m_fd_DevVideo != -1)
   {
      if (ioctl(m_fd_DevVideo, VIDIOC_S_CTRL, &rCtrl) < 0)
      {
         ETG_TRACE_USR1(("ERROR vSet_AceResponseSpeed(): ioctl failed to set"));
      }
   }
}

tVoid dispvidctrl_tclClient_DevVideo::vSet_AceChromaGain(tU8 u8Value) const
{
   struct v4l2_control rCtrl;

   //rCtrl.id    = V4L2_CID_ADV7182_ACE_CHROMA_GAIN;
   rCtrl.id    = EN_CONTROL_DEV_VIDEO__ADV7182_ACE_CHROMA_GAIN;
   rCtrl.value = (tS32) u8Value;

   if (m_fd_DevVideo != -1)
   {
      if (ioctl(m_fd_DevVideo, VIDIOC_S_CTRL, &rCtrl) < 0)
      {
         ETG_TRACE_USR1(("ERROR vSet_AceChromaGain(): ioctl failed to set"));
      }
   }
}

tVoid dispvidctrl_tclClient_DevVideo::vSet_AceChromeMax(tU8 u8Value) const
{
   struct v4l2_control rCtrl;

   //rCtrl.id    = V4L2_CID_ADV7182_ACE_CHROMA_MAX;
   rCtrl.id    = EN_CONTROL_DEV_VIDEO__ADV7182_ACE_CHROMA_MAX;
   rCtrl.value = (tS32) u8Value;

   if (m_fd_DevVideo != -1)
   {
      if (ioctl(m_fd_DevVideo, VIDIOC_S_CTRL, &rCtrl) < 0)
      {
         ETG_TRACE_USR1(("ERROR vSet_AceChromeMax(): ioctl failed to set"));
      }
   }
}

tVoid dispvidctrl_tclClient_DevVideo::vSet_AceGammaGain(tU8 u8Value) const
{
   struct v4l2_control rCtrl;

   //rCtrl.id    = V4L2_CID_ADV7182_ACE_GAMMA_GAIN;
   rCtrl.id    = EN_CONTROL_DEV_VIDEO__ADV7182_ACE_GAMMA_GAIN;
   rCtrl.value = (tS32) u8Value;

   if (m_fd_DevVideo != -1)
   {
      if (ioctl(m_fd_DevVideo, VIDIOC_S_CTRL, &rCtrl) < 0)
      {
         ETG_TRACE_USR1(("ERROR vSet_AceGammaGain(): ioctl failed to set"));
      }
   }
}


//##############################################################################
//##############################################################################
// TTFis Command handling
//##############################################################################
/******************************************************************************/
/* FUNCTION     vTraceInfo()                                                  */
/******************************************************************************/
/**
*  \brief       trace information
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclClient_DevVideo::vTraceInfo()
{

}

/******************************************************************************/
/* FUNCTION     vHandleTraceMessage                                           */
/******************************************************************************/
/**
*  \brief       handle TTFis commands
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
tVoid dispvidctrl_tclClient_DevVideo::vHandleTraceMessage(const tUChar* puchData)
{
   DISPVIDCTRL_NULL_POINTER_CHECK(puchData);

   tU32 u32MsgCode = ((puchData[1]<<8) | puchData[2]);
   ETG_TRACE_USR4(("dispvidctrl_tclClient_DevVideo::vHandleTraceMessage(): trace command %d", u32MsgCode ));

   switch (u32MsgCode)
   {
   case DISPVIDCTRL_DISPVIDCTRL_VIDEO_DEVVIDEO_READ:
      {
         // read a value via dev video interface (parameter is control).
         tU32 param1 = ((tU32)puchData[3] << 24) | ((tU32)puchData[4] << 16) | ((tU32)puchData[5] <<  8) | (tU32)puchData[6];

         tS32 s32ControlDataRead = 0;
         (tVoid) _bReadDevVideo(param1, s32ControlDataRead);
      }
      break;
   case DISPVIDCTRL_DISPVIDCTRL_VIDEO_DEVVIDEO_WRITE:
      {
         // write a value via dev video (parameter are control and value).
         tU32 param1 = ((tU32)puchData[3] << 24) | ((tU32)puchData[4] << 16) | ((tU32)puchData[5] <<  8) | (tU32)puchData[6];
         tU8 param2 = puchData[7];
         tU32 param3 = ((tU32)puchData[8] << 24) | ((tU32)puchData[9] << 16) | ((tU32)puchData[10] <<  8) | (tU32)puchData[11];

         if (CONVERSION_TO_TWOS_COMPLEMENT == param2)
         {
            if (param3 >= 128)
            {
               // API value 0..127 for increase (0x00..0x7F)
               param3 = (param3 - 128);
            }
            else
            {
               // API value from -1 to -128 for decrease (0xFF..0x80)
               param3 = (param3 + 128);
            }
         }
         else if (CONVERSION_TO_SIGNED == param2)
         {
            (tVoid) _bWriteDevVideo(param1, (tS32) (param3 - 128));
         }
         else
         {
            (tVoid) _bWriteDevVideo(param1, (tS32)param3);
         }
      }
      break;

      default:
      break;
   }
}


tBool dispvidctrl_tclClient_DevVideo::_bReadDevVideo(const tU32 u32Control, tS32& s32ControlData) const
{
   ETG_TRACE_USR1(("CL-DevVID :: ........... _bReadDevVideo(): %d     ", ETG_CENUM(tenDevVideo_Controls, (tenDevVideo_Controls) u32Control)));

   tBool bResult = FALSE;
   struct v4l2_control rCtrl;

   rCtrl.id = u32Control;
   rCtrl.value = 0;

   if (m_fd_DevVideo != -1)
   {
      if (ioctl(m_fd_DevVideo, VIDIOC_G_CTRL, &rCtrl) < 0)
      {
         ETG_TRACE_FATAL(("_bReadDevVideo(): ioctl failed to get the value"));
      }
      else
      {
         ETG_TRACE_USR1(("_bReadDevVideo(): ioctl data read: %d ", rCtrl.value));
         s32ControlData = rCtrl.value;
         bResult = TRUE;
      }
   }
   else
   {
      ETG_TRACE_FATAL(("_bReadDevVideo(): m_fd_DevVideo invalid, failed to get the value"));
   }

   return (bResult);
}

tBool dispvidctrl_tclClient_DevVideo::_bWriteDevVideo(const tU32 u32Control, tS32 s32ControlData) const
{
   ETG_TRACE_USR1(("CL-DevVID :: .......... _bWriteDevVideo(): %d with data %d    ", ETG_CENUM(tenDevVideo_Controls, (tenDevVideo_Controls) u32Control), s32ControlData));

   tBool bResult = FALSE;
   struct v4l2_control rCtrl;

   rCtrl.id = u32Control;
   rCtrl.value = s32ControlData;

   if (m_fd_DevVideo != -1)
   {
      if (ioctl(m_fd_DevVideo, VIDIOC_S_CTRL, &rCtrl) < 0)
      {
         ETG_TRACE_FATAL(("_bWriteDevVideo(): ioctl failed to set the value"));
      }
      else
      {
         bResult = TRUE;
      }
   }
   else
   {
      ETG_TRACE_FATAL(("_bWriteDevVideo(): m_fd_DevVideo invalid, failed to set the value"));
   }

   return (bResult);
}

