/************************************************************************
* FILE:        dispvidctrl_tclConfigMgr.cpp
* PROJECT:
* SW-COMPONENT:vd_dimming
*----------------------------------------------------------------------
*
* DESCRIPTION: implementation of configuration manager class
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 20.08.2015| CM-AI/EPB2 Bernard | init
*
*************************************************************************/

/*******************************************************************************
| includes: system- and project- includes
|-----------------------------------------------------------------------------*/
// Basic OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

/*******************************************************************************
| set own module id
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| includes: needed interfaces from external components
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| includes: internal and external interfaces from this component
|-----------------------------------------------------------------------------*/

#include "dispvidctrl_tclConfigMgr.h"
#include "Idispvidctrl_ClientDimming.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_APPLICATION
#include "trcGenProj/Header/dispvidctrl_tclConfigMgr.cpp.trc.h"
#endif

/*******************************************************************************
| defines and macros (scope: module-local)
|-----------------------------------------------------------------------------*/
#define TOTAL_DAY_NIGHT_BYTES 1764
/*******************************************************************************
| typedefs (scope: module-local)
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function prototype (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function implementation (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
*
* FUNCTION:    dispvidctrl_tclConfigMgr()
*
* DESCRIPTION: constructor
*
* PARAMETER:   const dispvidctrl_tclAppMain*
*
* RETURNVALUE: none
*
*******************************************************************************/
dispvidctrl_tclConfigMgr::dispvidctrl_tclConfigMgr(const dispvidctrl_tclAppMain* poMainAppl)
: I_dispvidctrl_tclConfigMgr(poMainAppl)
{
   ETG_TRACE_USR1(("dispvidctrl_tclConfigMgr() entered."));
}

/*******************************************************************************
*
* FUNCTION:    ~dispvidctrl_tclConfigMgr()
*
* DESCRIPTION: destructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*******************************************************************************/
dispvidctrl_tclConfigMgr::~dispvidctrl_tclConfigMgr()
{
   ETG_TRACE_USR4(("~dispvidctrl_tclConfigMgr() entered."));
}

/*******************************************************************************
*
* FUNCTION:    vHandleMessage()
*
* DESCRIPTION: handle internal message
*
* PARAMETER:   dispvidctrl_tclBaseIf::TMsg* pMsg
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid dispvidctrl_tclConfigMgr::vHandleMessage(dispvidctrl_tclBaseIf::TMsg* pMsg)
{
	   DISPVIDCTRL_NULL_POINTER_CHECK(pMsg);
	   ETG_TRACE_USR1(("vHandleMessage() entered %u -> data: %d.", ETG_ENUM(HC_INT_MSG , (tU32)pMsg->eCmd), pMsg->u.u32Data));

	   switch(pMsg->eCmd)
	   {
	   case dispvidctrl_tclBaseIf::eTriggerKDSChanged:
		   {
			   //Also here I shall update Dimming Matrix with KDS data Sandeep++
			   dp_tclDimmingDPCALIBRATION_BrightnessMatrix_Selection dpBrightnessSelectionMatrix;
			   dpBrightnessSelectionMatrix.s32SetData((tU8)dispvidctrl_ConfigMgr::EBrightnessCalib_None);
			   //Setting this None and then calling load function will Sandeep --

			   //ensure that Matrix datapool is updated with New Diagnostics Data by Resetting Above DataPool
			   vOnLoadSettings();
		   }
		   break;
	   default:
		   {
		   }
		   break;
	   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceMessage()
*
* DESCRIPTION: 	handler for trace command
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid dispvidctrl_tclConfigMgr::vHandleTraceMessage(const tUChar* puchData)
{
	ETG_TRACE_USR4(("vHandleTraceMessage() entered (data: 0x%08x).", puchData ));
	DISPVIDCTRL_NULL_POINTER_CHECK(puchData);

    tU32 u32MsgCode = (puchData[1]<<8) | puchData[2];
	switch (u32MsgCode)
	{
	case DISPVIDCTRL_EARLYCONFIG_CMDGETCONFIG:
		{
            // Prints the early config information.
			ETG_TRACE_USR1(("%26s: EARLYCONFIG_CmdGetConfig ",
				   "DIM.APPL.CFG_.CMD__.START" ));
			tChar _strIdentAttr[30]  = "DIM.APPL.CFG_.MGR__.ATTR_";
			m_oEarlyConfig.vTraceInfoEarlyConfig(_strIdentAttr);
			ETG_TRACE_USR1(("%26s: finished without errors ",
				   "DIM.APPL.CFG_.CMD__.RES__" ));
		}
		break;
	case DISPVIDCTRL_EARLYCONFIG_CMDWRITECONFIG:
		{
            // Writes the early config information.
			ETG_TRACE_USR1(("%26s: EARLYCONFIG_CmdWriteConfig ",
				   "DIM.APPL.CFG_.CMD__.START" ));
			if (TRUE == m_oEarlyConfig.bWriteEarlyConfig())
			{
				ETG_TRACE_USR1(("%26s: finished without errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
			else
			{
				ETG_TRACE_FATAL(("%26s: finished with errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
		}
		break;
    case DISPVIDCTRL_EARLYCONFIG_CMDLOADCONFIG:
        {
            // Loads the early config information.
			ETG_TRACE_USR1(("%26s: EARLYCONFIG_CmdLoadConfig ",
				   "DIM.APPL.CFG_.CMD__.START" ));
			if (TRUE == m_oEarlyConfig.bLoadKDSEarlyConfig())
			{
				ETG_TRACE_USR1(("%26s: finished without errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
			else
			{
				ETG_TRACE_FATAL(("%26s: finished with errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
        }
        break;
    case DISPVIDCTRL_EARLYCONFIG_CMDSETDISPLAYRESOLUTION:
        {
            // Writes the early config display resolution information (max. 31 char) directly without validation. Please use '\\0' to reset manually settings.
            const tChar* pString1 = (const tChar*)&puchData[3];

			ETG_TRACE_USR1(("%26s: EARLYCONFIG_CmdSetDisplayResolution %31s (length %d) ",
				   "DIM.APPL.CFG_.CMD__.START", pString1, strlen(pString1) ));
			if (TRUE == m_oEarlyConfig.bWriteEarlyConfig_DisplayResolution(pString1))
			{
				ETG_TRACE_USR1(("%26s: finished without errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
			else
			{
				ETG_TRACE_FATAL(("%26s: finished with errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
        }
        break;
    case DISPVIDCTRL_EARLYCONFIG_CMDSETBACKWARDCOMP:
        {
            // Writes the early config display backward comp configuration (possible values: -1, 0, 1). Please use an undefined value to reset mannually settings.
            tU32 param1 = ((tU32)puchData[3] << 24) | ((tU32)puchData[4] << 16) | ((tU32)puchData[5] <<  8) | (tU32)puchData[6];


            // DISPVIDCTRL_EARLYCONFIG_CMDSETBACKWARDCOMP: user specific code start
			ETG_TRACE_USR1(("%26s: EARLYCONFIG_CmdSetBackwardComp %d ", "DIM.APPL.CFG_.CMD__.START", (tS32)param1 ));
			if (TRUE == m_oEarlyConfig.bWriteEarlyConfig_BackwardComp((tS32)param1))
			{
				ETG_TRACE_USR1(("%26s: finished without errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
			else
			{
				ETG_TRACE_FATAL(("%26s: finished with errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
            // DISPVIDCTRL_EARLYCONFIG_CMDSETBACKWARDCOMP: user specific code end

        }
        break;
    case DISPVIDCTRL_EARLYCONFIG_CMDSETLOWFREQUENCY:
        {
            // Writes the early config display low frequence configuration (possible values: -1, 0, 1). Please use an undefined value to reset mannually settings.
            tU32 param1 = ((tU32)puchData[3] << 24) | ((tU32)puchData[4] << 16) | ((tU32)puchData[5] <<  8) | (tU32)puchData[6];


            // DISPVIDCTRL_EARLYCONFIG_CMDSETLOWFREQUENCY: user specific code start
			ETG_TRACE_USR1(("%26s: EARLYCONFIG_CmdSetLowFrequency %d ", "DIM.APPL.CFG_.CMD__.START", (tS32)param1 ));
			if (TRUE == m_oEarlyConfig.bWriteEarlyConfig_LowFrequency((tS32)param1))
			{
				ETG_TRACE_USR1(("%26s: finished without errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
			else
			{
				ETG_TRACE_FATAL(("%26s: finished with errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
            // DISPVIDCTRL_EARLYCONFIG_CMDSETLOWFREQUENCY: user specific code end

        }
        break;
    case DISPVIDCTRL_EARLYCONFIG_CMDSETRTCDRIVER:
        {
            // Writes the early config RTC driver name (max. 63 char) directly without validation. Please use '\\0' to reset manually settings.
            const tChar* pString1 = (const tChar*)&puchData[3];

			ETG_TRACE_USR1(("%26s: EARLYCONFIG_CmdSetRtcDriver %63s (length %d) ",
				   "DIM.APPL.CFG_.CMD__.START", pString1, strlen(pString1) ));
			if (TRUE == m_oEarlyConfig.bWriteEarlyConfig_RTCDriverName(pString1))
			{
				ETG_TRACE_USR1(("%26s: finished without errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
			else
			{
				ETG_TRACE_FATAL(("%26s: finished with errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
        }
        break;
    case DISPVIDCTRL_EARLYCONFIG_CMDSETTOUCHDRIVER:
        {
            // Writes the early config touch driver name (max. 63 char) directly without validation. Please use '\\0' to reset manually settings.
            const tChar* pString1 = (const tChar*)&puchData[3];

			ETG_TRACE_USR1(("%26s: EARLYCONFIG_CmdSetTouchDriver %63s (length %d) ",
				   "DIM.APPL.CFG_.CMD__.START", pString1, strlen(pString1) ));
			if (TRUE == m_oEarlyConfig.bWriteEarlyConfig_TouchDriverName(pString1))
			{
				ETG_TRACE_USR1(("%26s: finished without errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
			else
			{
				ETG_TRACE_FATAL(("%26s: finished with errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
        }
        break;
    case DISPVIDCTRL_EARLYCONFIG_CMDSETTOUCHCONFIGFILE:
        {
            // Writes the early config touch configuration file name (max. 63 char) directly without validation. Please use '\\0' to reset manually settings.
            const tChar* pString1 = (const tChar*)&puchData[3];
			ETG_TRACE_USR1(("%26s: EARLYCONFIG_CmdSetTouchConfigFile %63s (length %d) ",
				   "DIM.APPL.CFG_.CMD__.START", pString1, strlen(pString1) ));
			if (TRUE == m_oEarlyConfig.bWriteEarlyConfig_TouchDriverConfigFile(pString1))
			{
				ETG_TRACE_USR1(("%26s: finished without errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
			else
			{
				ETG_TRACE_FATAL(("%26s: finished with errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
        }
        break;
    case DISPVIDCTRL_EARLYCONFIG_CMDSETTOUCHI2CADDRESS:
        {
            // Writes the early config I2C address (max. 11 char, e.g. 'i2c-6/') of first touch device directly without validation. Please use '\\0' to reset manually settings.
            const tChar* pString1 = (const tChar*)&puchData[3];


            // DISPVIDCTRL_EARLYCONFIG_CMDSETTOUCHI2CADDRESS: user specific code start
			ETG_TRACE_USR1(("%26s: EARLYCONFIG_CmdSetTouchI2CAddress %11s (length %d) ",
				   "DIM.APPL.CFG_.CMD__.START", pString1, strlen(pString1) ));
			if (TRUE == m_oEarlyConfig.bWriteEarlyConfig_TouchDriverI2CAddress(pString1))
			{
				ETG_TRACE_USR1(("%26s: finished without errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
			else
			{
				ETG_TRACE_FATAL(("%26s: finished with errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
            // DISPVIDCTRL_EARLYCONFIG_CMDSETTOUCHI2CADDRESS: user specific code end

        }
        break;
    case DISPVIDCTRL_EARLYCONFIG_CMDSETTOUCHDRIVER2:
        {
            // Writes the early config driver name (max. 63 char) of second touch device directly without validation. Please use '\\0' to reset manually settings.
            const tChar* pString1 = (const tChar*)&puchData[3];


            // DISPVIDCTRL_EARLYCONFIG_CMDSETTOUCHDRIVER2: user specific code start
			ETG_TRACE_USR1(("%26s: EARLYCONFIG_CmdSetTouchDriver2 %63s (length %d) ",
				   "DIM.APPL.CFG_.CMD__.START", pString1, strlen(pString1) ));
			if (TRUE == m_oEarlyConfig.bWriteEarlyConfig_TouchDriverName2(pString1))
			{
				ETG_TRACE_USR1(("%26s: finished without errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
			else
			{
				ETG_TRACE_FATAL(("%26s: finished with errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
            // DISPVIDCTRL_EARLYCONFIG_CMDSETTOUCHDRIVER2: user specific code end

        }
        break;
    case DISPVIDCTRL_EARLYCONFIG_CMDSETTOUCHCONFIGFILE2:
        {
            // Writes the early config configuration file name (max. 63 char) of second touch device directly without validation. Please use '\\0' to reset manually settings.
            const tChar* pString1 = (const tChar*)&puchData[3];


            // DISPVIDCTRL_EARLYCONFIG_CMDSETTOUCHCONFIGFILE2: user specific code start
			ETG_TRACE_USR1(("%26s: EARLYCONFIG_CmdSetTouchConfigFile2 %63s (length %d) ",
				   "DIM.APPL.CFG_.CMD__.START", pString1, strlen(pString1) ));
			if (TRUE == m_oEarlyConfig.bWriteEarlyConfig_TouchDriverConfigFile2(pString1))
			{
				ETG_TRACE_USR1(("%26s: finished without errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
			else
			{
				ETG_TRACE_FATAL(("%26s: finished with errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
            // DISPVIDCTRL_EARLYCONFIG_CMDSETTOUCHCONFIGFILE2: user specific code end

        }
        break;
    case DISPVIDCTRL_EARLYCONFIG_CMDSETTOUCHI2CADDRESS2:
        {
            // Writes the early config I2C address (max. 11 char, e.g. 'i2c-6/') of second touch device directly without validation. Please use '\\0' to reset manually settings.
            const tChar* pString1 = (const tChar*)&puchData[3];


            // DISPVIDCTRL_EARLYCONFIG_CMDSETTOUCHI2CADDRESS2: user specific code start
			ETG_TRACE_USR1(("%26s: EARLYCONFIG_CmdSetTouchI2CAddress2 %11s (length %d) ",
				   "DIM.APPL.CFG_.CMD__.START", pString1, strlen(pString1) ));
			if (TRUE == m_oEarlyConfig.bWriteEarlyConfig_TouchDriverI2CAddress2(pString1))
			{
				ETG_TRACE_USR1(("%26s: finished without errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
			else
			{
				ETG_TRACE_FATAL(("%26s: finished with errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
            // DISPVIDCTRL_EARLYCONFIG_CMDSETTOUCHI2CADDRESS2: user specific code end

        }
        break;
    case DISPVIDCTRL_EARLYCONFIG_CMDSETTIMINGLVDS1:
        {
            // Writes the early config LVDS 1 timing configuration (max. 31 char) directly without validation. Please use '\\0' to reset manually settings.
            const tChar* pString1 = (const tChar*)&puchData[3];

            // DISPVIDCTRL_EARLYCONFIG_CMDSETTIMINGLVDS1: user specific code start
			ETG_TRACE_USR1(("%26s: EARLYCONFIG_CmdSetTimingLvds1 %31s (length %d) ",
				   "DIM.APPL.CFG_.CMD__.START", pString1, strlen(pString1) ));
			if (TRUE == m_oEarlyConfig.bWriteEarlyConfig_TimingLvds1(pString1))
			{
				ETG_TRACE_USR1(("%26s: finished without errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
			else
			{
				ETG_TRACE_FATAL(("%26s: finished with errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
            // DISPVIDCTRL_EARLYCONFIG_CMDSETTIMINGLVDS1: user specific code end

        }
        break;
    case DISPVIDCTRL_EARLYCONFIG_CMDSETCLOCKGENERATIONLVDS1:
        {
            // Writes the early config LVDS 1 spread spectrum clock generation configuration (possible values: -1, 0, 8-15). Please use an undefined value to reset mannually settings.
            tU32 param1 = ((tU32)puchData[3] << 24) | ((tU32)puchData[4] << 16) | ((tU32)puchData[5] <<  8) | (tU32)puchData[6];


            // DISPVIDCTRL_EARLYCONFIG_CMDSETCLOCKGENERATIONLVDS1: user specific code start
			ETG_TRACE_USR1(("%26s: EARLYCONFIG_CmdSetClockGenerationLvds1 %d ",
				   "DIM.APPL.CFG_.CMD__.START", (tS32)param1 ));
			if (TRUE == m_oEarlyConfig.bWriteEarlyConfig_ClockGenerationLvds1((tS32)param1))
			{
				ETG_TRACE_USR1(("%26s: finished without errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
			else
			{
				ETG_TRACE_FATAL(("%26s: finished with errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
            // DISPVIDCTRL_EARLYCONFIG_CMDSETCLOCKGENERATIONLVDS1: user specific code end

        }
        break;
    case DISPVIDCTRL_EARLYCONFIG_CMDSETTIMINGLVDS2:
        {
            // Writes the early config LVDS 2 timing configuration (max. 31 char) directly without validation. Please use '\\0' to reset manually settings.
            const tChar* pString1 = (const tChar*)&puchData[3];

            // DISPVIDCTRL_EARLYCONFIG_CMDSETTIMINGLVDS2: user specific code start
			ETG_TRACE_USR1(("%26s: EARLYCONFIG_CmdSetTimingLvds2 %31s (length %d) ",
				   "DIM.APPL.CFG_.CMD__.START", pString1, strlen(pString1) ));
			if (TRUE == m_oEarlyConfig.bWriteEarlyConfig_TimingLvds2(pString1))
			{
				ETG_TRACE_USR1(("%26s: finished without errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
			else
			{
				ETG_TRACE_FATAL(("%26s: finished with errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
            // DISPVIDCTRL_EARLYCONFIG_CMDSETTIMINGLVDS2: user specific code end

        }
        break;
    case DISPVIDCTRL_EARLYCONFIG_CMDSETCLOCKGENERATIONLVDS2:
        {
            // Writes the early config LVDS 2 spread spectrum clock generation configuration (possible values: -1, 0, 8-15). Please use an undefined value to reset mannually settings.
            tU32 param1 = ((tU32)puchData[3] << 24) | ((tU32)puchData[4] << 16) | ((tU32)puchData[5] <<  8) | (tU32)puchData[6];


            // DISPVIDCTRL_EARLYCONFIG_CMDSETCLOCKGENERATIONLVDS2: user specific code start
			ETG_TRACE_USR1(("%26s: EARLYCONFIG_CmdSetClockGenerationLvds2 %d ",
				   "DIM.APPL.CFG_.CMD__.START", (tS32)param1 ));
			if (TRUE == m_oEarlyConfig.bWriteEarlyConfig_ClockGenerationLvds2((tS32)param1))
			{
				ETG_TRACE_USR1(("%26s: finished without errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
			else
			{
				ETG_TRACE_FATAL(("%26s: finished with errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
            // DISPVIDCTRL_EARLYCONFIG_CMDSETCLOCKGENERATIONLVDS2: user specific code end

        }
        break;
    case DISPVIDCTRL_EARLYCONFIG_CMDSETDISPLAYORDER:
        {
            // Writes the early config display order configuration directly without validation. Please use 'UNDEF' to reset manually settings.
            tU8 param1 = puchData[3];
            tU8 param2 = puchData[4];
            tU8 param3 = puchData[5];
            tU8 param4 = puchData[6];


            // DISPVIDCTRL_EARLYCONFIG_CMDSETDISPLAYORDER: user specific code start
			ETG_TRACE_USR1(("%26s: EARLYCONFIG_CmdSetDisplayOrder %u %u %u %u ",
				   "DIM.APPL.CFG_.CMD__.START",
				   ETG_ENUM(EC_SCREENID, param1),
				   ETG_ENUM(EC_SCREENID, param2),
				   ETG_ENUM(EC_SCREENID, param3),
				   ETG_ENUM(EC_SCREENID, param4) ));
			tU8 _au8Parameter[4] = {param1, param2, param3, param4};
			if (TRUE == m_oEarlyConfig.bWriteEarlyConfig_DisplayOrder(_au8Parameter))
			{
				ETG_TRACE_USR1(("%26s: finished without errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
			else
			{
				ETG_TRACE_FATAL(("%26s: finished with errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
            // DISPVIDCTRL_EARLYCONFIG_CMDSETDISPLAYORDER: user specific code end

        }
        break;
    case DISPVIDCTRL_EARLYCONFIG_CMDSETFLUSHDESIGNCONFIGFILE:
        {
            // Writes the early config flush design configuration file name (max. 63 char) directly without validation. Please use '\\0' to reset manually settings.
            const tChar* pString1 = (const tChar*)&puchData[3];


            // DISPVIDCTRL_EARLYCONFIG_CMDSETFLUSHDESIGNCONFIGFILE: user specific code start
			ETG_TRACE_USR1(("%26s: EARLYCONFIG_CmdSetFlushDesignConfigFile %64s (length %d) ",
				   "DIM.APPL.CFG_.CMD__.START", pString1, strlen(pString1) ));
			if (TRUE == m_oEarlyConfig.bWriteEarlyConfig_FlushDesignConfigFile(pString1))
			{
				ETG_TRACE_USR1(("%26s: finished without errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
			else
			{
				ETG_TRACE_FATAL(("%26s: finished with errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
            // DISPVIDCTRL_EARLYCONFIG_CMDSETFLUSHDESIGNCONFIGFILE: user specific code end

        }
        break;
    case DISPVIDCTRL_EARLYCONFIG_CMDSETCLOCKEDGELVDS1:
        {
            // Writes the early config LVDS 1 clock edge configuration (possible values: -1, 0, 1). Please use an undefined value to reset mannually settings.
            tU32 param1 = ((tU32)puchData[3] << 24) | ((tU32)puchData[4] << 16) | ((tU32)puchData[5] <<  8) | (tU32)puchData[6];


            // DISPVIDCTRL_EARLYCONFIG_CMDSETCLOCKEDGELVDS1: user specific code start
			if (TRUE == m_oEarlyConfig.bWriteEarlyConfig_ClockEdgeLvds1((tS32)param1))
			{
				ETG_TRACE_USR1(("%26s: finished without errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
			else
			{
				ETG_TRACE_FATAL(("%26s: finished with errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
            // DISPVIDCTRL_EARLYCONFIG_CMDSETCLOCKEDGELVDS1: user specific code end

        }
        break;
    case DISPVIDCTRL_EARLYCONFIG_CMDSETCLOCKEDGELVDS2:
        {
            // Writes the early config LVDS 1 clock edge configuration (possible values: -1, 0, 1). Please use an undefined value to reset mannually settings.
            tU32 param1 = ((tU32)puchData[3] << 24) | ((tU32)puchData[4] << 16) | ((tU32)puchData[5] <<  8) | (tU32)puchData[6];


            // DISPVIDCTRL_EARLYCONFIG_CMDSETCLOCKEDGELVDS2: user specific code start
			if (TRUE == m_oEarlyConfig.bWriteEarlyConfig_ClockEdgeLvds2((tS32)param1))
			{
				ETG_TRACE_USR1(("%26s: finished without errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
			else
			{
				ETG_TRACE_FATAL(("%26s: finished with errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
            // DISPVIDCTRL_EARLYCONFIG_CMDSETCLOCKEDGELVDS2: user specific code end

        }
        break;
    case DISPVIDCTRL_DIMMING_CMDGETPDDCONFIG:
        {
            // Prints the PDD dimming config information.
			ETG_TRACE_USR1(("%26s: DIMMING_CmdGetPDDConfig ",
					   "DIM.APPL.CFG_.CMD__.START" ));
			tChar _strIdentAttr[30]  = "DIM.APPL.CFG_.MGR__.ATTR_";
			m_oPDDDimming.vTraceInfoConfigPDD(_strIdentAttr);
			ETG_TRACE_USR1(("%26s: finished without errors ",
				   "DIM.APPL.CFG_.CMD__.RES__" ));
        }
        break;
	case DISPVIDCTRL_DIMMING_CMDSETPDDCONFIG:
		{
			// Writes the PDD dimming config information.
			ETG_TRACE_USR1(("%26s: DIMMING_CmdSetPDDConfig ",
				   "DIM.APPL.CFG_.CMD__.START" ));
			if (TRUE == m_oPDDDimming.bWritePddData())
			{
				ETG_TRACE_USR1(("%26s: finished without errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
			else
			{
				ETG_TRACE_FATAL(("%26s: finished with errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
		}
		break;
    case DISPVIDCTRL_DIMMING_CMDLOADPDDCONFIG:
        {
            // Loads the PDD dimming config information.
			ETG_TRACE_USR1(("%26s: DIMMING_CmdLoadPDDConfig ",
				   "DIM.APPL.CFG_.CMD__.START" ));
			if (TRUE == m_oPDDDimming.bLoadKDSData())
			{
				ETG_TRACE_USR1(("%26s: finished without errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
			else
			{
				ETG_TRACE_FATAL(("%26s: finished with errors ",
					   "DIM.APPL.CFG_.CMD__.RES__" ));
			}
        }
        break;

	default:
		break;
	}
}

/*******************************************************************************
*
* FUNCTION: 	vGetReferences(tVoid)
*
* DESCRIPTION: 	Function to get all reference needed by this class.
* 				A reference should always be the Interface class of the object
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid dispvidctrl_tclConfigMgr::vGetReferences(tVoid)
{
   ETG_TRACE_USR4(("vGetReferences() entered."));
}

/*******************************************************************************
*
* FUNCTION: 	tVoid vStartCommunication()
*
* DESCRIPTION: 	Function to start all dynamic objects e.g. threads, ...
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid dispvidctrl_tclConfigMgr::vStartCommunication(tVoid)
{
   ETG_TRACE_USR4(("vStartCommunication() entered."));
   vStartConfig();
}

/*******************************************************************************
*
* FUNCTION: 	vTraceInfo()
*
* DESCRIPTION: 	Trace information
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid dispvidctrl_tclConfigMgr::vTraceInfo()
{
	ETG_TRACE_USR4(("vTraceInfo() entered."));
	tChar _strIdentStart[30] = "DIM.APPL.CFG_.MGR__.START";
	tChar _strIdentAttr[30]  = "DIM.APPL.CFG_.MGR__.ATTR_";
	tChar _strIdentEnd[30]   = "DIM.APPL.CFG_.MGR__.END__";
	ETG_TRACE_USR2(("%026s: ", _strIdentStart ));
	m_oEarlyConfig.vTraceInfoEarlyConfig(_strIdentAttr);
	m_oPDDDimming.vTraceInfoConfigPDD(_strIdentAttr);
	ETG_TRACE_USR2(("%026s: ", _strIdentEnd ));
}

/*******************************************************************************
*
* FUNCTION:    vOnLoadSettings()
*
* DESCRIPTION: loading of all settings
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid dispvidctrl_tclConfigMgr::vOnLoadSettings(tVoid)
{
   ETG_TRACE_FATAL(("vOnLoadSettings() entered."));

   // update EarlyConfig
   if (TRUE == m_oEarlyConfig.bLoadKDSEarlyConfig())
   {
#ifdef VARIANT_S_FTR_ENABLE_EARLYCONFIG_AUTO_UPDATE
	   (tVoid)m_oEarlyConfig.bWriteEarlyConfig();
#else
	   ETG_TRACE_USR1(("vOnLoadSettings: Early config will not update automatically!" ));
#endif
   }
   else
   {
	   ETG_TRACE_USR1(("vOnLoadSettings: Loading of Early config failed!" ));
   }
   // update PDD Dimming
   if (TRUE == m_oPDDDimming.bLoadKDSData())
   {
#ifdef VARIANT_S_FTR_ENABLE_PDDCONFIG_AUTO_UPDATE
	   (tVoid)m_oPDDDimming.bWritePddData();
#else
	   ETG_TRACE_USR1(("vOnLoadSettings: PDD Dimming config will not update automatically!" ));
#endif
   }
   else
   {
	   ETG_TRACE_USR1(("vOnLoadSettings: Loading of PDD Dimming config failed!" ));
   }

   //Update BrightnessData for Display
#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1
   if( TRUE == loadDataBrightnessFromDIAG() )
   {
	   ETG_TRACE_USR1(("vOnLoadSettings: SUCCESSFULLY copied data from Diagnostics to internal datapool !!" ));
   }
   //Populate Internal Datapool into Matrix send Update
   vUpdateDimmINCBrightnessMatrix();
#endif

}

tVoid dispvidctrl_tclConfigMgr::vStartConfig() {
   OSAL_trThreadAttribute rAttr;
   tC8                    szThreadName[] = "dispvidctrlCfg";

   rAttr.szName       = szThreadName;
   rAttr.s32StackSize = 2048;
   rAttr.u32Priority  = 100;
   rAttr.pfEntry      = (OSAL_tpfThreadEntry)vDoConfigThread;
   rAttr.pvArg        = ( tPVoid ) this;

   OSAL_tThreadID hThreadId = OSAL_ThreadSpawn(&rAttr);

   if ( hThreadId == OSAL_ERROR ){
       ETG_TRACE_FATAL( ( "dispvidctrl_tclConfigMgr::vStartConfig(): Failed to spawn thread!" ) );
   }
}

tVoid dispvidctrl_tclConfigMgr::vDoConfigThread(tVoid *pvArg) {
    dispvidctrl_tclConfigMgr* _MyRef = (dispvidctrl_tclConfigMgr*)pvArg;
    OSAL_s32ThreadWait(3000);
    _MyRef->vOnLoadSettings();
}

#ifdef VARIANT_S_FTR_ENABLE_AIVI_SCOPE2_1
tBool dispvidctrl_tclConfigMgr::loadDataBrightnessFromDIAG()
{
	tS32 s32Result1 = -1;
	tS32 s32Result2 = -1;
	tS32 s32Result3 = -1;

	dp_tclDimmingDPCALIBRATION_BrightnessMatrix_Selection dpBrightnessSelectionMatrix;
	tU8 u8BrightnessSelectionDP = dpBrightnessSelectionMatrix.tGetData();

	if(u8BrightnessSelectionDP != (tU8)dispvidctrl_ConfigMgr::EBrightnessCalib_USER) //If not user then update from Diagnostics
	{
	    dp_tclDPAID_DatasetProjectDataBrightnessParam _aDiagDataBrightness; //contains 1764 bytes of data combined
	    tU8 _u8DIAGDayNightBrightnessData[TOTAL_DAY_NIGHT_BYTES]={0};

		//Read Data from DataPool and copy all 1764 bytes in tU8 Array  _u8DIAGDayNightBrightnessData;
		s32Result1 = _aDiagDataBrightness.s32GetData(&_u8DIAGDayNightBrightnessData[0], sizeof(_u8DIAGDayNightBrightnessData));
		ETG_TRACE_FATAL((":Datapool \_aDiagDataBrightness:{%*x}",
						ETG_LIST_LEN(1764), ETG_LIST_PTR_T8(&_u8DIAGDayNightBrightnessData[0])));
						
		I_dispvidctrl_tclClientDimming* pIncDimmingClient = dynamic_cast<I_dispvidctrl_tclClientDimming*>(_cpoMain->getHandler("I_dispvidctrl_tclClientDimming"));
		if(pIncDimmingClient != NULL && s32Result1 != -1)
		{
			tU16 u16DimmValue = 0;
			tU32 u32ArrIndex = 0;

			for(int row = 0; row<21; row++){
				for(int col = 0; col<21; col++){
					if(u32ArrIndex < TOTAL_DAY_NIGHT_BYTES)
					{
						u16DimmValue = tU16(_u8DIAGDayNightBrightnessData[u32ArrIndex]<<8) | _u8DIAGDayNightBrightnessData[u32ArrIndex+1];
						pIncDimmingClient->vSetDayMatrixCellValue(row,col, u16DimmValue);
						u32ArrIndex = u32ArrIndex+2;
					}
				}//Endof Col Loop
			}//Endof For loop Row 21x21 = 441 iterations

			u32ArrIndex = 882;
			for(int row = 0; row<21; row++){
				for(int col = 0; col<21; col++){
					if(u32ArrIndex < TOTAL_DAY_NIGHT_BYTES)
					{
						u16DimmValue = tU16(_u8DIAGDayNightBrightnessData[u32ArrIndex]<<8) | _u8DIAGDayNightBrightnessData[u32ArrIndex+1];
						pIncDimmingClient->vSetNightMatrixCellValue(row,col, u16DimmValue);
						u32ArrIndex = u32ArrIndex+2;
					}
				}//Endof Col Loop
			}//Endof For loop Row 21x21 = 441 iterations

			pIncDimmingClient->vStoreDayMatrixToDatapool();
			pIncDimmingClient->vStoreNightMatrixToDatapool();

			ETG_TRACE_FATAL( ( ": loadDataBrightnessFromDIAG(): VALID dp_tclDPAID_DatasetProjectDataBrightnessParam DIAG Loaded Success!! s32Result1=%d, s32Result2=%d, s32Result3=%d" ,s32Result1, s32Result2, s32Result3) );
		}
		else
		{
			ETG_TRACE_FATAL( ( ": loadDataBrightnessFromDIAG(): Invalid dp_tclDPAID_DatasetProjectDataBrightnessParam DIAG Failure s32Result1=%d!!", s32Result1 ) );
			if( -1 != loadDefaultDispBrightnessConfig() )
			{
				s32Result2 = 0;
				s32Result3 = 0;
			}
			else
			{
				s32Result2 = -1;
				s32Result3 = -1;
			}
		}

	}
	else
	{
		//This part of code gets executed when User calibrated data is present in Datapool
		ETG_TRACE_FATAL( ( ": loadDataBrightnessFromDIAG(): SKIP updating Datapool Matrix from Diagnostics!\nUsing User Calibrated Data....." ) );

	}

	//Return Status ERROR Checking 0 means all went well
	if(s32Result1 != -1 && s32Result2 != -1 && s32Result3 != -1)
	{
		ETG_TRACE_USR1( ( ": loadDataBrightnessFromDIAG(): Successfully updated !" ) );
        //Now set the Datapool selection to DIagnostics
		dpBrightnessSelectionMatrix.s32SetData((tU8)dispvidctrl_ConfigMgr::EBrightnessCalib_DIAG);

		return TRUE;
	}
	else
	{
		ETG_TRACE_USR1( ( ": loadDataBrightnessFromDIAG(): Not updated !" ) );
		return FALSE;
	}

}

tVoid dispvidctrl_tclConfigMgr::vGetKeyIllumMatrixValues(tU16* u16Array, tU8 length)
{
	ETG_TRACE_FATAL((" :vGetKeyIllumMatrixValues() - > Enter to Get key Illumination Array Values"));
	if(u16Array != NULL)
	{	//Copy the Datapool Value into array
		(tVoid)OSAL_pvMemoryCopy(
				(tVoid*)u16Array,	//Dest
				(tVoid*)m_oPDDDimming.m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON,
				length );
	}
}

tVoid dispvidctrl_tclConfigMgr::vSetKeyIllumMatrixValues(tU16* u16Array, tU8 length)
{
	//Later
	ETG_TRACE_FATAL((" :vSetKeyIllumMatrixValues() - > Enter to Set key Illumination Array Values"));

	if( u16Array != NULL )
	{	//Copy the Array to Datapool
		(tVoid)OSAL_pvMemoryCopy(
				(tVoid*)m_oPDDDimming.m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON,
    			(tVoid*)u16Array, //Source
				length );
	}
}

tVoid dispvidctrl_tclConfigMgr::vSetT4VSKeyIllumDataPool(tU16 u16Day_Day1, tU16 u16Day_night1, tU16 u16Night_Night1)
{
	ETG_TRACE_FATAL((" :vSetT4VSKeyIllumDataPool() - > Enter to Set key Illumination 3 Parameters"));

	m_oPDDDimming.m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY = u16Day_Day1;
	m_oPDDDimming.m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT = u16Day_night1;
	m_oPDDDimming.m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT.BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT = u16Night_Night1;

	ETG_TRACE_FATAL(("DAY_BUTTONS_DAY = %d", m_oPDDDimming.m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY ));
	ETG_TRACE_FATAL(("DAY_BUTTONS_NIGHT = %d", m_oPDDDimming.m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT ));
	ETG_TRACE_FATAL(("NIGHT_BUTTONS_NIGHT = %d", m_oPDDDimming.m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT.BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT));
}

tVoid dispvidctrl_tclConfigMgr::vGetT4VSKeyIllumDataPool(tU16 (&u16Array)[3])
{
	ETG_TRACE_FATAL((":vGetT4VSKeyIllumDataPool() - > Enter to Set key Illumination 3 Parameters"));
	if( u16Array != NULL )
	{	//Copy the Array to Datapool
		u16Array[0] = m_oPDDDimming.m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY;
		u16Array[1] = m_oPDDDimming.m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT;
		u16Array[2] = m_oPDDDimming.m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT.BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT;

		ETG_TRACE_FATAL(("DAY_BUTTONS_DAY = %d", u16Array[0] ));
		ETG_TRACE_FATAL(("DAY_BUTTONS_NIGHT = %d", u16Array[1]));
		ETG_TRACE_FATAL(("NIGHT_BUTTONS_NIGHT = %d", u16Array[2]));
	}
}

tVoid dispvidctrl_tclConfigMgr::vUpdateDimmINCBrightnessMatrix()
{
	I_dispvidctrl_tclClientDimming* pIncDimmingClient = dynamic_cast<I_dispvidctrl_tclClientDimming*>(_cpoMain->getHandler("I_dispvidctrl_tclClientDimming"));
	if(pIncDimmingClient != NULL)
	{
	   pIncDimmingClient->vUpdateDayBrightnessMatrix();
	   pIncDimmingClient->vUpdateNightBrightnessMatrix();
	}
}


tVoid dispvidctrl_tclConfigMgr::vReset_DispRenaultDatapool()
{
	//Reset Brightness Selection Datapool
	dp_tclDimmingDPCALIBRATION_BrightnessMatrix_Selection dpBrightnessSelectionMatrix;
	dpBrightnessSelectionMatrix.s32SetData((tU8)dispvidctrl_ConfigMgr::EBrightnessCalib_None);

	//Load Default Values into Datapool
	loadDefaultDispBrightnessConfig();
}

tS32 dispvidctrl_tclConfigMgr::loadDefaultDispBrightnessConfig()
{
	tS32 s32RetStatus = -1;

	tU16 _u8DIAGDayNightBrightnessData[(TOTAL_DAY_NIGHT_BYTES/2)]={
		//-------------------------------------------------------------------------------------------------------------------------------------------------------------------------
		//	D A Y     R O W S   2 1    M A T R I X
		//-------------------------------------------------------------------------------------------------------------------------------------------------------------------------	
		0x000A, 0x0014, 0x0032, 0x0064, 0x0096, 0x00C8, 0x00FA, 0x012C, 0x015E, 0x0190, 0x01C2, 0x01F4, 0x0226, 0x0258, 0x028A, 0x02BC, 0x02EE, 0x0320, 0x0352, 0x0384, 0x03B6,     //|   |
		0x001E, 0x0014, 0x0046, 0x0078, 0x00AA, 0x00DC, 0x010E, 0x0140, 0x0172, 0x01A4, 0x01D6, 0x0208, 0x023A, 0x026C, 0x029E, 0x02D0, 0x0302, 0x0334, 0x0366, 0x0398, 0x03CA,     //| D |
		0x001E, 0x0028, 0x005A, 0x008C, 0x00BE, 0x00F0, 0x0122, 0x0154, 0x0186, 0x01B8, 0x01EA, 0x021C, 0x024E, 0x0280, 0x02B2, 0x02E4, 0x0316, 0x0348, 0x037A, 0x03AC, 0x03DE,     //| A |
		0x001E, 0x003C, 0x006E, 0x00A0, 0x00D2, 0x0104, 0x0136, 0x0168, 0x019A, 0x01CC, 0x01FE, 0x0230, 0x0262, 0x0294, 0x02C6, 0x02F8, 0x032A, 0x035C, 0x038E, 0x03C0, 0x03E8,     //| Y |
		0x0028, 0x0050, 0x0082, 0x00B4, 0x00E6, 0x0118, 0x014A, 0x017C, 0x01AE, 0x01E0, 0x0212, 0x0244, 0x0276, 0x02A8, 0x02DA, 0x030C, 0x033E, 0x0370, 0x03A2, 0x03D4, 0x03E8,     //|   |
		0x0032, 0x0064, 0x0096, 0x00C8, 0x00FA, 0x012C, 0x015E, 0x0190, 0x01C2, 0x01F4, 0x0226, 0x0258, 0x028A, 0x02BC, 0x02EE, 0x0320, 0x0352, 0x0384, 0x03B6, 0x03E8, 0x03E8,     //| C |
		0x0046, 0x0078, 0x00AA, 0x00DC, 0x010E, 0x0140, 0x0172, 0x01A4, 0x01D6, 0x0208, 0x023A, 0x026C, 0x029E, 0x02D0, 0x0302, 0x0334, 0x0366, 0x0398, 0x03CA, 0x03E8, 0x03E8,     //| O |
		0x005A, 0x008C, 0x00BE, 0x00F0, 0x0122, 0x0154, 0x0186, 0x01B8, 0x01EA, 0x021C, 0x024E, 0x0280, 0x02B2, 0x02E4, 0x0316, 0x0348, 0x037A, 0x03AC, 0x03DE, 0x03E8, 0x03E8,     //| L |
		0x006E, 0x00A0, 0x00D2, 0x0104, 0x0136, 0x0168, 0x019A, 0x01CC, 0x01FE, 0x0230, 0x0262, 0x0294, 0x02C6, 0x02F8, 0x032A, 0x035C, 0x038E, 0x03C0, 0x03E8, 0x03E8, 0x03E8,     //| U |
		0x0082, 0x00B4, 0x00E6, 0x0118, 0x014A, 0x017C, 0x01AE, 0x01E0, 0x0212, 0x0244, 0x0276, 0x02A8, 0x02DA, 0x030C, 0x033E, 0x0370, 0x03A2, 0x03D4, 0x03E8, 0x03E8, 0x03E8,     //| M |
		0x0096, 0x00C8, 0x00FA, 0x012C, 0x015E, 0x0190, 0x01C2, 0x01F4, 0x0226, 0x0258, 0x028A, 0x02BC, 0x02EE, 0x0320, 0x0352, 0x0384, 0x03B6, 0x03E8, 0x03E8, 0x03E8, 0x03E8,     //| N |
		0x00AA, 0x00DC, 0x010E, 0x0140, 0x0172, 0x01A4, 0x01D6, 0x0208, 0x023A, 0x026C, 0x029E, 0x02D0, 0x0302, 0x0334, 0x0366, 0x0398, 0x03CA, 0x03E8, 0x03E8, 0x03E8, 0x03E8,     //|   |
		0x00BE, 0x00F0, 0x0122, 0x0154, 0x0186, 0x01B8, 0x01EA, 0x021C, 0x024E, 0x0280, 0x02B2, 0x02E4, 0x0316, 0x0348, 0x037A, 0x03AC, 0x03DE, 0x03E8, 0x03E8, 0x03E8, 0x03E8,     //| 2 |
		0x00D2, 0x0104, 0x0136, 0x0168, 0x019A, 0x01CC, 0x01FE, 0x0230, 0x0262, 0x0294, 0x02C6, 0x02F8, 0x032A, 0x035C, 0x038E, 0x03C0, 0x03E8, 0x03E8, 0x03E8, 0x03E8, 0x03E8,     //| 1 |
		0x00E6, 0x0118, 0x014A, 0x017C, 0x01AE, 0x01E0, 0x0212, 0x0244, 0x0276, 0x02A8, 0x02DA, 0x030C, 0x033E, 0x0370, 0x03A2, 0x03D4, 0x03E8, 0x03E8, 0x03E8, 0x03E8, 0x03E8,     //|   |
		0x00FA, 0x012C, 0x015E, 0x0190, 0x01C2, 0x01F4, 0x0226, 0x0258, 0x028A, 0x02BC, 0x02EE, 0x0320, 0x0352, 0x0384, 0x03B6, 0x03E8, 0x03E8, 0x03E8, 0x03E8, 0x03E8, 0x03E8,     //| N |
		0x010E, 0x0140, 0x0172, 0x01A4, 0x01D6, 0x0208, 0x023A, 0x026C, 0x029E, 0x02D0, 0x0302, 0x0334, 0x0366, 0x0398, 0x03CA, 0x03E8, 0x03E8, 0x03E8, 0x03E8, 0x03E8, 0x03E8,     //| O |
		0x0122, 0x0154, 0x0186, 0x01B8, 0x01EA, 0x021C, 0x024E, 0x0280, 0x02B2, 0x02E4, 0x0316, 0x0348, 0x037A, 0x03AC, 0x03DE, 0x03E8, 0x03E8, 0x03E8, 0x03E8, 0x03E8, 0x03E8,     //|   |
		0x0136, 0x0168, 0x019A, 0x01CC, 0x01FE, 0x0230, 0x0262, 0x0294, 0x02C6, 0x02F8, 0x032A, 0x035C, 0x038E, 0x03C0, 0x03E8, 0x03E8, 0x03E8, 0x03E8, 0x03E8, 0x03E8, 0x03E8,     //|   |
		0x014A, 0x017C, 0x01AE, 0x01E0, 0x0212, 0x0244, 0x0276, 0x02A8, 0x02DA, 0x030C, 0x033E, 0x0370, 0x03A2, 0x03D4, 0x03E8, 0x03E8, 0x03E8, 0x03E8, 0x03E8, 0x03E8, 0x03E8,     //|   |
		0x015E, 0x0190, 0x01C2, 0x01F4, 0x0226, 0x0258, 0x028A, 0x02BC, 0x02EE, 0x0320, 0x0352, 0x0384, 0x03B6, 0x03E8, 0x03E8, 0x03E8, 0x03E8, 0x03E8, 0x03E8, 0x03E8, 0x03E8,     //|   |
																																												   
		//-------------------------------------------------------------------------------------------------------------------------------------------------------------------------
		//	N I G H T     R O W S   2 1    M A T R I X
		//-------------------------------------------------------------------------------------------------------------------------------------------------------------------------																																										   
		0x0005, 0x000A, 0x0019, 0x0032, 0x004B, 0x0064, 0x007D, 0x0096, 0x00AF, 0x00C8, 0x00E1, 0x00FA, 0x0113, 0x012C, 0x0145, 0x015E, 0x0177, 0x0190, 0x01A9, 0x01C2, 0x01DB,     //|   |
		0x000F, 0x000A, 0x0023, 0x003C, 0x0055, 0x006E, 0x0087, 0x00A0, 0x00B9, 0x00D2, 0x00EB, 0x0104, 0x011D, 0x0136, 0x014F, 0x0168, 0x0181, 0x019A, 0x01B3, 0x01CC, 0x01E5,     //| N |
		0x000F, 0x0014, 0x002D, 0x0046, 0x005F, 0x0078, 0x0091, 0x00AA, 0x00C3, 0x00DC, 0x00F5, 0x010E, 0x0127, 0x0140, 0x0159, 0x0172, 0x018B, 0x01A4, 0x01BD, 0x01D6, 0x01EF,     //| I |
		0x000F, 0x001E, 0x0037, 0x0050, 0x0069, 0x0082, 0x009B, 0x00B4, 0x00CD, 0x00E6, 0x00FF, 0x0118, 0x0131, 0x014A, 0x0163, 0x017C, 0x0195, 0x01AE, 0x01C7, 0x01E0, 0x01F4,     //| G |
		0x0014, 0x0028, 0x0041, 0x005A, 0x0073, 0x008C, 0x00A5, 0x00BE, 0x00D7, 0x00F0, 0x0109, 0x0122, 0x013B, 0x0154, 0x016D, 0x0186, 0x019F, 0x01B8, 0x01D1, 0x01EA, 0x01F4,     //| H |
		0x0019, 0x0032, 0x004B, 0x0064, 0x007D, 0x0096, 0x00AF, 0x00C8, 0x00E1, 0x00FA, 0x0113, 0x012C, 0x0145, 0x015E, 0x0177, 0x0190, 0x01A9, 0x01C2, 0x01DB, 0x01F4, 0x01F4,     //| T |
		0x0023, 0x003C, 0x0055, 0x006E, 0x0087, 0x00A0, 0x00B9, 0x00D2, 0x00EB, 0x0104, 0x011D, 0x0136, 0x014F, 0x0168, 0x0181, 0x019A, 0x01B3, 0x01CC, 0x01E5, 0x01F4, 0x01F4,     //|   |
		0x002D, 0x0046, 0x005F, 0x0078, 0x0091, 0x00AA, 0x00C3, 0x00DC, 0x00F5, 0x010E, 0x0127, 0x0140, 0x0159, 0x0172, 0x018B, 0x01A4, 0x01BD, 0x01D6, 0x01EF, 0x01F4, 0x01F4,     //| C |
		0x0037, 0x0050, 0x0069, 0x0082, 0x009B, 0x00B4, 0x00CD, 0x00E6, 0x00FF, 0x0118, 0x0131, 0x014A, 0x0163, 0x017C, 0x0195, 0x01AE, 0x01C7, 0x01E0, 0x01F4, 0x01F4, 0x01F4,     //| O |
		0x0041, 0x005A, 0x0073, 0x008C, 0x00A5, 0x00BE, 0x00D7, 0x00F0, 0x0109, 0x0122, 0x013B, 0x0154, 0x016D, 0x0186, 0x019F, 0x01B8, 0x01D1, 0x01EA, 0x01F4, 0x01F4, 0x01F4,     //| L |
		0x004B, 0x0064, 0x007D, 0x0096, 0x00AF, 0x00C8, 0x00E1, 0x00FA, 0x0113, 0x012C, 0x0145, 0x015E, 0x0177, 0x0190, 0x01A9, 0x01C2, 0x01DB, 0x01F4, 0x01F4, 0x01F4, 0x01F4,     //| U |
		0x0055, 0x006E, 0x0087, 0x00A0, 0x00B9, 0x00D2, 0x00EB, 0x0104, 0x011D, 0x0136, 0x014F, 0x0168, 0x0181, 0x019A, 0x01B3, 0x01CC, 0x01E5, 0x01F4, 0x01F4, 0x01F4, 0x01F4,     //| M |
		0x005F, 0x0078, 0x0091, 0x00AA, 0x00C3, 0x00DC, 0x00F5, 0x010E, 0x0127, 0x0140, 0x0159, 0x0172, 0x018B, 0x01A4, 0x01BD, 0x01D6, 0x01EF, 0x01F4, 0x01F4, 0x01F4, 0x01F4,     //| N |
		0x0069, 0x0082, 0x009B, 0x00B4, 0x00CD, 0x00E6, 0x00FF, 0x0118, 0x0131, 0x014A, 0x0163, 0x017C, 0x0195, 0x01AE, 0x01C7, 0x01E0, 0x01F4, 0x01F4, 0x01F4, 0x01F4, 0x01F4,     //|   |
		0x0073, 0x008C, 0x00A5, 0x00BE, 0x00D7, 0x00F0, 0x0109, 0x0122, 0x013B, 0x0154, 0x016D, 0x0186, 0x019F, 0x01B8, 0x01D1, 0x01EA, 0x01F4, 0x01F4, 0x01F4, 0x01F4, 0x01F4,     //| 2 |
		0x007D, 0x0096, 0x00AF, 0x00C8, 0x00E1, 0x00FA, 0x0113, 0x012C, 0x0145, 0x015E, 0x0177, 0x0190, 0x01A9, 0x01C2, 0x01DB, 0x01F4, 0x01F4, 0x01F4, 0x01F4, 0x01F4, 0x01F4,     //| 1 |
		0x0087, 0x00A0, 0x00B9, 0x00D2, 0x00EB, 0x0104, 0x011D, 0x0136, 0x014F, 0x0168, 0x0181, 0x019A, 0x01B3, 0x01CC, 0x01E5, 0x01F4, 0x01F4, 0x01F4, 0x01F4, 0x01F4, 0x01F4,     //|   |
		0x0091, 0x00AA, 0x00C3, 0x00DC, 0x00F5, 0x010E, 0x0127, 0x0140, 0x0159, 0x0172, 0x018B, 0x01A4, 0x01BD, 0x01D6, 0x01EF, 0x01F4, 0x01F4, 0x01F4, 0x01F4, 0x01F4, 0x01F4,     //| N |
		0x009B, 0x00B4, 0x00CD, 0x00E6, 0x00FF, 0x0118, 0x0131, 0x014A, 0x0163, 0x017C, 0x0195, 0x01AE, 0x01C7, 0x01E0, 0x01F4, 0x01F4, 0x01F4, 0x01F4, 0x01F4, 0x01F4, 0x01F4,     //| O |
		0x00A5, 0x00BE, 0x00D7, 0x00F0, 0x0109, 0x0122, 0x013B, 0x0154, 0x016D, 0x0186, 0x019F, 0x01B8, 0x01D1, 0x01EA, 0x01F4, 0x01F4, 0x01F4, 0x01F4, 0x01F4, 0x01F4, 0x01F4,     //|   |
		0x00AF, 0x00C8, 0x00E1, 0x00FA, 0x0113, 0x012C, 0x0145, 0x015E, 0x0177, 0x0190, 0x01A9, 0x01C2, 0x01DB, 0x01F4, 0x01F4, 0x01F4, 0x01F4, 0x01F4, 0x01F4, 0x01F4, 0x01F4      //|   |
	};
	
	tS32 s32Result2 = -1;
	tS32 s32Result3 = -1;

	ETG_TRACE_FATAL( ( ": loadDefaultDispBrightnessConfig(): Diagnostics data load failed..so loading some default brightness!!" ) );

	dp_tclDimmingDPDISP_Brightness_Calibration_Day_Set _aCalibDayBrightnessValues;
	s32Result2 = _aCalibDayBrightnessValues.s32SetData(&_u8DIAGDayNightBrightnessData[0], 441);

	dp_tclDimmingDPDISP_Brightness_Calibration_Night_Set _aCalibNightBrightnessValues;
	s32Result3 = _aCalibNightBrightnessValues.s32SetData(&_u8DIAGDayNightBrightnessData[442], 441);

	if(s32Result2 != -1  && s32Result3 != -1)
	{
		s32RetStatus = 0;
		vUpdateDimmINCBrightnessMatrix();
	}

	ETG_TRACE_FATAL( ( ": loadDefaultDispBrightnessConfig(): End -- s32RetStatus = %d, s32Result2=%d, s32Result3=%d !!",s32RetStatus, s32Result2, s32Result3 ) );

	return s32RetStatus;
}

#endif
