/************************************************************************
* FILE:        dispvidctrl_config_pdd.cpp
* PROJECT:
* SW-COMPONENT:vd_dimming
*----------------------------------------------------------------------
*
* DESCRIPTION: implementation of PDD dimming class
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 13.10.2015| CM-AI/EPB2 Bernard | rework
*
*************************************************************************/

/*******************************************************************************
| includes: system- and project- includes
|-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

/*******************************************************************************
| set own module id
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| includes: needed interfaces from external components
|-----------------------------------------------------------------------------*/
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#include "dp_drvass_video_if.h"

/*******************************************************************************
| includes: internal and external interfaces from this component
|-----------------------------------------------------------------------------*/
#include "dispvidctrl_config_pdd.h"
#include "PDDDefault_Data.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_APPLICATION
#include "trcGenProj/Header/dispvidctrl_config_pdd.cpp.trc.h"
#endif

/*******************************************************************************
| defines and macros (scope: module-local)
|-----------------------------------------------------------------------------*/


#define NISSAN_TYPE 	0x01
#define RENAULT_TYPE	0x00
#define P42M			0x424D

#define CHECK_VARIANT_NISSAN(TYPE)  (tU8)((TYPE >= 0x01 && TYPE<=0x03) ? (0x01): (0xFF))
#define CHECK_VARIANT_RENAULT(TYPE) (tU8)(((TYPE >= 0x04 && TYPE<=0x07) || TYPE==0x00 )?(0x00):(0xFF))

/*******************************************************************************
| typedefs (scope: module-local)
|-----------------------------------------------------------------------------*/
typedef enum brand
{
	Renault	= 	0x00,
	Nissan	= 	0x01,
	PIVI	=	0x02,
	EV		=	0x03,
	Dacia	=	0x04,
	Daimler	=	0x05,
	Avtovaz	=	0x06,
	RSM		=	0x07
}ebrandtype;

typedef enum vcanDiversity
{
	slot0	=	0x00,
	slot1	=	0x01,
	slot2	=	0x02,
	slot3	=	0x03,
	slot4	=	0x04,
	slot5	=	0x05,
	slot6	=	0x06
}eVanDiversityType;

typedef enum architecture
{
	default_type	= 	0x00,
	Nissan_Slot6	=	0x01,
	Nissan_Slot5	=	0x02,
	Nissan_Slot1	=	0x03,
	Renault_Slot1	=	0x04,
	Renault_Slot3	=	0x05,
	Nissan_P42M		=	0x06,
	PIVI_device		=	0x07
}eDimmingArchitectureType;

/*******************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function prototype (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function implementation (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
*
* FUNCTION:    dispvidctrl_tclConfigPdd()
*
* DESCRIPTION: constructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*******************************************************************************/
dispvidctrl_tclConfigPdd::dispvidctrl_tclConfigPdd()
{
    (tVoid)OSAL_pvMemorySet((tVoid*)&m_oKDSConfigData, OSAL_NULL, sizeof(m_oKDSConfigData));
    m_bKDSLoaded = FALSE;
    m_bUseDefaultCalibrationValues = FALSE;
    m_u8MajorVersion = OSAL_C_U8_MAX;
    m_u8MinorVersion = OSAL_C_U8_MAX;
    vReadDefaultValues();
}


/*******************************************************************************
*
* FUNCTION: 	vReadDefaultValues
*
* DESCRIPTION: 	read KDS configuration data
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid dispvidctrl_tclConfigPdd::vReadDefaultValues()
{
	ETG_TRACE_USR4(("vReadDefaultValues() entered."));
	m_bKDSLoaded = FALSE;
	m_bUseDefaultCalibrationValues = FALSE;
    m_u8MajorVersion = OSAL_C_U8_MAX;
    m_u8MinorVersion = OSAL_C_U8_MAX;


    (tVoid)OSAL_pvMemorySet((tVoid*)&m_oKDSConfigData, OSAL_NULL, sizeof(m_oKDSConfigData));
	// ... BRIGTHNESS_CURVE_HMI_NIGHT
    tU16 _au16BRIGTHNESS_CURVE_HMI_NIGHT[DATAPOOL_SCC_BRIGTHNESS_CURVE_HMI_NIGHT_NUMBER] = {DEFAULT_C_AU16_BRIGTHNESS_CURVE_HMI_NIGHT};
    (tVoid)OSAL_pvMemoryCopy((tVoid*)m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_NIGHT.BRIGTHNESS_CURVE_HMI_NIGHT,
    		(tVoid*)_au16BRIGTHNESS_CURVE_HMI_NIGHT,
    		sizeof(m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_NIGHT.BRIGTHNESS_CURVE_HMI_NIGHT));
	// BRIGTHNESS_CURVE_HMI_DAY
    tU16 _au16BRIGTHNESS_CURVE_HMI_DAY[DATAPOOL_SCC_BRIGTHNESS_CURVE_HMI_DAY_NUMBER] = {DEFAULT_C_AU16_BRIGTHNESS_CURVE_HMI_DAY};
    (tVoid)OSAL_pvMemoryCopy((tVoid*)m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_DAY.BRIGTHNESS_CURVE_HMI_DAY,
    		(tVoid*)_au16BRIGTHNESS_CURVE_HMI_DAY,
    		sizeof(m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_DAY.BRIGTHNESS_CURVE_HMI_DAY));
	// DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY
    m_oKDSConfigData.DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY.DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY = DEFAULT_C_U16_DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY;
	// KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY
    m_oKDSConfigData.KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY.KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY = DEFAULT_C_U16_KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY;
    // BRIGTHNESS_NO_OF_HMI_OFFSET
    m_oKDSConfigData.BRIGTHNESS_NO_OF_HMI_OFFSET.BRIGTHNESS_NO_OF_HMI_OFFSET = DEFAULT_C_U8_BRIGTHNESS_NO_OF_HMI_OFFSET;
    // NUMBER_OF_KEYS
    m_oKDSConfigData.NUMBER_OF_KEYS.NUMBER_OF_KEYS = DEFAUTL_C_U8_NUMBER_OF_KEYS;
    // HIGH_TEMPERATURE
    m_oKDSConfigData.HIGH_TEMPERATURE.HIGH_TEMPERATURE = DEFAULT_C_U8_HIGH_TEMPERATURE;
    // CRITICAL_TEMPERATURE
    m_oKDSConfigData.CRITICAL_TEMPERATURE.CRITICAL_TEMPERATURE = DEFAULT_C_U8_CRITICAL_TEMPERATURE;
    // LOW_TEMPERATURE
    m_oKDSConfigData.LOW_TEMPERATURE.LOW_TEMPERATURE = DEFAUTL_C_S8_LOW_TEMPERATURE;
    // TEMP_HYSTERESIS
    m_oKDSConfigData.TEMP_HYSTERESIS.TEMP_HYSTERESIS = DEFAULT_C_U8_TEMP_HYSTERESIS;
    // TEMP_COMPENSATION_PWM_MIN
    m_oKDSConfigData.TEMP_COMPENSATION_PWM_MIN.TEMP_COMPENSATION_PWM_MIN = DEFAULT_C_U8_TEMP_COMPENSATION_PWM_MIN;
    // DISPLAY_TYPE
    m_oKDSConfigData.DISPLAY_TYPE.DISPLAY_TYPE = DEFAULT_C_AU16_DISPLAY_TYPE_ATTACHED_VALUE;
    // DIMMING COCKPIT STRATEGY
    m_oKDSConfigData.DIMMING_COCKPIT_STRATEGY.DIMMING_COCKPIT_STRATEGY = DEFAULT_C_AU16_DIMMING_COCKPIT_STRATEGY_VALUE;
    // KEY_BRIGHTNESS_NUMOF_OFFSETS
    m_oKDSConfigData.KEY_BRIGHTNESS_NUMOF_OFFSETS.KEY_BRIGHTNESS_NUMOF_OFFSETS = DEFAULT_C_AU16_KEY_BRIGHTNESS_NUMOF_OFFSETS_VALUE;
    // DAMPING_STRATEGY
    m_oKDSConfigData.DAMPING_STRATEGY.DAMPING_STRATEGY = DEFAULT_C_AU16_DAMPING_STRATEGY_DISABLED_VALUE;
    // DAMPING_SAMPLING_TIMETE
    m_oKDSConfigData.DAMPING_SAMPLING_TIMETE.DAMPING_SAMPLING_TIMETE = DEFAULT_C_AU16_DAMPING_SAMPLING_TIMETE_VALUE;
    // DAMPING_CONSTANTK
    m_oKDSConfigData.DAMPING_CONSTANTK.DAMPING_CONSTANTK = DEFAULT_C_AU16_DAMPING_CONSTANTK_VALUE;
    // DAMPING_CONSTANTJ
    m_oKDSConfigData.DAMPING_CONSTANTJ.DAMPING_CONSTANTJ = DEFAULT_C_AU16_DAMPING_CONSTANTJ_VALUE;
    // BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY
    m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY = DEFAULT_C_AU16_BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY_VALUE;
    // BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT
    m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT = DEFAULT_C_AU16_BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT_VALUE;
    // BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT
    m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT.BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT = DEFAULT_C_AU16_BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT_VALUE;
    // BRIGTHNESSBACKLIGHT_DAYBUTTONS_ON
    m_oKDSConfigData.BRIGTHNESSBACKLIGHT_DAYBUTTONS_ON.BRIGTHNESSBACKLIGHT_DAYBUTTONS_ON = DEFAULT_C_AU16_BRIGTHNESSBACKLIGHT_DAYBUTTONS_ON_VALUE;
    // BRIGTHNESS_BACKLIGHT_DAYBUTTONS_OFF
    m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_DAYBUTTONS_OFF.BRIGTHNESS_BACKLIGHT_DAYBUTTONS_OFF = DEFAULT_C_AU16_BRIGTHNESS_BACKLIGHT_DAYBUTTONS_OFF_VALUE;
    // BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON ; This is Key Illumination Matrix Scope 2.1 (21 rows)
    tU16 _au16BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON[DATAPOOL_SCC_BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON_NUMBER] = {DEFAULT_C_AU16_BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON_VALUE};
    (tVoid)OSAL_pvMemoryCopy((tVoid*)m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON,
            		(tVoid*)_au16BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON,
            		sizeof(m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON));

    // KEYB_LED_CALIBRATION_VALUE
    tU16 _au16KEYB_LED_CALIBRATION_VALUE[DATAPOOL_SCC_KEYB_LED_CALIBRATION_VALUE_NUMBER] = {DEFAULT_C_AU16_KEYB_LED_CALIBRATION_VALUE};
    (tVoid)OSAL_pvMemoryCopy((tVoid*)m_oKDSConfigData.KEYB_LED_CALIBRATION_VALUE.KEYB_LED_CALIBRATION_VALUE,
    		(tVoid*)_au16KEYB_LED_CALIBRATION_VALUE,
    		sizeof(m_oKDSConfigData.KEYB_LED_CALIBRATION_VALUE.KEYB_LED_CALIBRATION_VALUE));

    // DAY_NIGHT_DELAY
    m_oKDSConfigData.DAY_NIGHT_DELAY.DAY_NIGHT_DELAY = DEFAULT_C_AU16_LIGHTENING_PARAMETER_VALUE;

    //BRANDNAME
    m_oKDSConfigData.BRANDNAME.BRANDNAME = NISSAN_TYPE;		//Set Nissan as default

    //DIMMING_ARCHITECTURE
    m_oKDSConfigData.DIMMING_ARCHITECTURE.DIMMING_ARCHITECTURE = DEFAULT_C_U8_DIMMING_ARCHITECTURE_VALUE;
}

/*******************************************************************************
*
* FUNCTION: 	bLoadKDSData
*
* DESCRIPTION: 	load KDS configuration data
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	tBool 	{ TRUE - successful | else - failed }
*
*******************************************************************************/
tBool dispvidctrl_tclConfigPdd::bLoadKDSData()
{
	ETG_TRACE_USR4(("bLoadKDSData() entered."));
	// reset all KDS information
	vReadDefaultValues();

    // load DimmingParameter
	tBool _bKDSLoaded_DimmingParameter = FALSE;
	tU8 _au8DimmingParameterMajorVersion = 0;
	tS32 _s32ReturnCode = DP_s32GetConfigItem( "DimmingParameter", "MajorVersion",
			&_au8DimmingParameterMajorVersion, sizeof(_au8DimmingParameterMajorVersion));
	if  (DP_S32_NO_ERR != _s32ReturnCode)
	{
		ETG_TRACE_USR1(("bLoadKDSData: MajorVersion number of KDS DimmingParameter could not read (error=0x%08x)", _s32ReturnCode));
	}
	else
	{
		ETG_TRACE_USR2(("bLoadKDSData: MajorVersion number %d of KDS DimmingParameter read successful (return=%d)",
				_au8DimmingParameterMajorVersion, _s32ReturnCode ));
		_bKDSLoaded_DimmingParameter = (0 < _au8DimmingParameterMajorVersion) ? bLoadKDSData_V1() : bLoadKDSData_V0();
	}

	tBool _bKDSLoaded_DimmingCalibrationParameterSet = bLoadCalibrationData();

	m_bKDSLoaded = ((TRUE == _bKDSLoaded_DimmingParameter)
			&& ((TRUE == _bKDSLoaded_DimmingCalibrationParameterSet) || (TRUE == m_bUseDefaultCalibrationValues)));
	return m_bKDSLoaded;
}

/*******************************************************************************
*
* FUNCTION: 	bLoadKDSData_V0
*
* DESCRIPTION: 	load KDS configuration data with major version 0
*
*   see https://hi-dms.de.bosch.com/docushare/dsweb/Get/Document-809480/VD_DIMMING_CFG_01.xlsx
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	tBool 	{ TRUE - successful | else - failed }
*
*******************************************************************************/
tBool dispvidctrl_tclConfigPdd::bLoadKDSData_V0()
{
	ETG_TRACE_USR4(("bLoadKDSData_V0() entered."));

	tBool _bReturn = FALSE;
	tU8 _au8KDSRawDataDimmingParameter[DP_U8_KDSLEN_DIMMINGPARAMETER_COMPLETE] = {0};
	tS32 _s32ReturnCode = DP_s32GetConfigItem( "DimmingParameter", "",
			_au8KDSRawDataDimmingParameter, sizeof(_au8KDSRawDataDimmingParameter));
	if  (DP_S32_NO_ERR != _s32ReturnCode || 0 != _au8KDSRawDataDimmingParameter[0])
	{
		ETG_TRACE_USR1(("bLoadKDSData_V0: KDS DimmingParameter could not read (error=0x%08x, major version=%u)",
				_s32ReturnCode, _au8KDSRawDataDimmingParameter[0] ));
	}
	else
	{
		tU8* _pu8Data = _au8KDSRawDataDimmingParameter;
		// set version information
		m_u8MajorVersion = 0;
		m_u8MinorVersion = 0;
		// BRIGTHNESS_CURVE_HMI_NIGHT
		for (tU32 _u32Idx = 0; (DATAPOOL_SCC_BRIGTHNESS_CURVE_HMI_NIGHT_NUMBER > _u32Idx); ++_u32Idx)
		{
			(tVoid)bConvertRawKDSData(m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_NIGHT.BRIGTHNESS_CURVE_HMI_NIGHT[_u32Idx], _pu8Data);
			_pu8Data += sizeof(m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_NIGHT.BRIGTHNESS_CURVE_HMI_NIGHT[0]);
		}
		// BRIGTHNESS_CURVE_HMI_DAY
		for (tU32 _u32Idx = 0; DATAPOOL_SCC_BRIGTHNESS_CURVE_HMI_DAY_NUMBER > _u32Idx; ++_u32Idx)
		{
			(tVoid)bConvertRawKDSData(m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_DAY.BRIGTHNESS_CURVE_HMI_DAY[_u32Idx], _pu8Data);
			_pu8Data += sizeof(m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_DAY.BRIGTHNESS_CURVE_HMI_DAY[0]);
		}
		// DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY
		(tVoid)bConvertRawKDSData(m_oKDSConfigData.DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY.DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY, _pu8Data);
		_pu8Data += sizeof(m_oKDSConfigData.DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY.DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY);
		// KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY
		(tVoid)bConvertRawKDSData(m_oKDSConfigData.KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY.KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY, _pu8Data);
		_pu8Data += sizeof(m_oKDSConfigData.KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY.KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY);
		// BRIGTHNESS_NO_OF_HMI_OFFSET
		(tVoid)bConvertRawKDSData(m_oKDSConfigData.BRIGTHNESS_NO_OF_HMI_OFFSET.BRIGTHNESS_NO_OF_HMI_OFFSET, _pu8Data);
		_pu8Data += sizeof(m_oKDSConfigData.BRIGTHNESS_NO_OF_HMI_OFFSET.BRIGTHNESS_NO_OF_HMI_OFFSET);
		// NUMBER_OF_KEYS
		(tVoid)bConvertRawKDSData(m_oKDSConfigData.NUMBER_OF_KEYS.NUMBER_OF_KEYS, _pu8Data);
		_pu8Data += sizeof(m_oKDSConfigData.NUMBER_OF_KEYS.NUMBER_OF_KEYS);
		// HIGH_TEMPERATURE
		(tVoid)bConvertRawKDSData(m_oKDSConfigData.HIGH_TEMPERATURE.HIGH_TEMPERATURE, _pu8Data);
		_pu8Data += sizeof(m_oKDSConfigData.HIGH_TEMPERATURE.HIGH_TEMPERATURE);
		// CRITICAL_TEMPERATURE
		(tVoid)bConvertRawKDSData(m_oKDSConfigData.CRITICAL_TEMPERATURE.CRITICAL_TEMPERATURE, _pu8Data);
		_pu8Data += sizeof(m_oKDSConfigData.CRITICAL_TEMPERATURE.CRITICAL_TEMPERATURE);
		// LOW_TEMPERATURE
		(tVoid)bConvertRawKDSData(m_oKDSConfigData.LOW_TEMPERATURE.LOW_TEMPERATURE, _pu8Data);
		_pu8Data += sizeof(m_oKDSConfigData.LOW_TEMPERATURE.LOW_TEMPERATURE);
		// TEMP_HYSTERESIS
		(tVoid)bConvertRawKDSData(m_oKDSConfigData.TEMP_HYSTERESIS.TEMP_HYSTERESIS, _pu8Data);
		_pu8Data += sizeof(m_oKDSConfigData.TEMP_HYSTERESIS.TEMP_HYSTERESIS);
		// TEMP_COMPENSATION_PWM_MIN
		(tVoid)bConvertRawKDSData(m_oKDSConfigData.TEMP_COMPENSATION_PWM_MIN.TEMP_COMPENSATION_PWM_MIN, _pu8Data);
		//_pu8Data += sizeof(m_oKDSConfigData.TEMP_COMPENSATION_PWM_MIN.TEMP_COMPENSATION_PWM_MIN);
		_bReturn = TRUE;
	}

	return _bReturn;
}

/*******************************************************************************
*
* FUNCTION: 	bLoadKDSData_V1
*
* DESCRIPTION: 	load KDS configuration data with major version 1 or higher
*
*   see https://hi-dms.de.bosch.com/docushare/dsweb/Get/Document-722374/A-IVI_CMDS_1v11.cdd
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	tBool 	{ TRUE - successful | else - failed }
*
*******************************************************************************/
tBool dispvidctrl_tclConfigPdd::bLoadKDSData_V1()
{
	ETG_TRACE_USR4(("bLoadKDSData_V1() entered."));

	tBool _bReturn = TRUE;
	// reading from KDS - PD
	tU8 _au8KDSRawDataDimmingParameter[DP_U8_KDSLEN_DIMMINGPARAMETER_COMPLETE] = {0};
	tS32 _s32ReturnCode = DP_s32GetConfigItem( "DimmingParameter", "", _au8KDSRawDataDimmingParameter, sizeof(_au8KDSRawDataDimmingParameter));
	if  (DP_S32_NO_ERR != _s32ReturnCode || 0 == _au8KDSRawDataDimmingParameter[0])
	{
		ETG_TRACE_USR1(("bLoadKDSData_V1: KDS DimmingParameter could not read (error=0x%08x, major version=%d)",
				_s32ReturnCode, _au8KDSRawDataDimmingParameter[0] ));
		_bReturn = FALSE;
	}
	else
	{
		tU8* _pu8Data = _au8KDSRawDataDimmingParameter;
		// u8GetMajorVersion ... Major Version
		(tVoid)bConvertRawKDSData(m_u8MajorVersion, _pu8Data);
		_pu8Data += sizeof(m_u8MajorVersion);
		// u8GetMinorVersion ... Minor Version
		(tVoid)bConvertRawKDSData(m_u8MinorVersion, _pu8Data);
		_pu8Data += sizeof(m_u8MinorVersion);
		// u8GetBRIGTHNESSCURVEHMINIGHTC* ... BRIGTHNESS_CURVE_HMI_NIGHT
		for (tU32 _u32Idx = 0; DATAPOOL_SCC_BRIGTHNESS_CURVE_HMI_NIGHT_NUMBER > _u32Idx; ++_u32Idx)
		{
			(tVoid)bConvertRawKDSData(m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_NIGHT.BRIGTHNESS_CURVE_HMI_NIGHT[_u32Idx], _pu8Data);
			_pu8Data += sizeof(m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_NIGHT.BRIGTHNESS_CURVE_HMI_NIGHT[0]);
		}
		// u8GetBRIGTHNESSCURVEHMIDAYC* ... BRIGTHNESS_CURVE_HMI_DAY
		for (tU32 _u32Idx = 0; DATAPOOL_SCC_BRIGTHNESS_CURVE_HMI_DAY_NUMBER > _u32Idx; ++_u32Idx)
		{
			(tVoid)bConvertRawKDSData(m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_DAY.BRIGTHNESS_CURVE_HMI_DAY[_u32Idx], _pu8Data);
			_pu8Data += sizeof(m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_DAY.BRIGTHNESS_CURVE_HMI_DAY[0]);
		}
		// u8GetDISPLAYILLUMINATIONOUTPUTFREQUENCY ... DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY
		(tVoid)bConvertRawKDSData(m_oKDSConfigData.DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY.DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY, _pu8Data);
		_pu8Data += sizeof(m_oKDSConfigData.DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY.DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY);
		// u8GetKEYSENCODERILLUMINATIONOUTPUTFREQUENCY ... KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY
		(tVoid)bConvertRawKDSData(m_oKDSConfigData.KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY.KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY, _pu8Data);
		_pu8Data += sizeof(m_oKDSConfigData.KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY.KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY);
		// u8GetTEMPCOMPENSATIONPWMMIN ... TEMP_COMPENSATION_PWM_MIN ... 2 bytes ... 1/10percent
		tU16 _u16TEMP_COMPENSATION_PWM_MIN = 0;
		(tVoid)bConvertRawKDSData(_u16TEMP_COMPENSATION_PWM_MIN, _pu8Data);
		_pu8Data += sizeof(_u16TEMP_COMPENSATION_PWM_MIN);
		m_oKDSConfigData.TEMP_COMPENSATION_PWM_MIN.TEMP_COMPENSATION_PWM_MIN = (tU8)(_u16TEMP_COMPENSATION_PWM_MIN / 10);
		// u8GetNUMBEROFKEYS ... NUMBER_OF_KEYS
		(tVoid)bConvertRawKDSData(m_oKDSConfigData.NUMBER_OF_KEYS.NUMBER_OF_KEYS, _pu8Data);
		_pu8Data += sizeof(m_oKDSConfigData.NUMBER_OF_KEYS.NUMBER_OF_KEYS);
		// u8GetHIGHTEMPERATURE ... HIGH_TEMPERATURE ... 2 bytes ... signed ... dC
		tS16 _s16HIGH_TEMPERATURE = 0;
		(tVoid)bConvertRawKDSData(_s16HIGH_TEMPERATURE, _pu8Data);
		_pu8Data += sizeof(_s16HIGH_TEMPERATURE);
		m_oKDSConfigData.HIGH_TEMPERATURE.HIGH_TEMPERATURE = (tU8)(_s16HIGH_TEMPERATURE / 10);
		// u8GetCRITICALTEMPERATURE ... CRITICAL_TEMPERATURE ... 2 bytes ... signed ... dC
		tS16 _s16CRITICAL_TEMPERATURE = 0;
		(tVoid)bConvertRawKDSData(_s16CRITICAL_TEMPERATURE, _pu8Data);
		_pu8Data += sizeof(_s16CRITICAL_TEMPERATURE);
		m_oKDSConfigData.CRITICAL_TEMPERATURE.CRITICAL_TEMPERATURE = (tU8)(_s16CRITICAL_TEMPERATURE / 10);
		// u8GetLOWTEMPERATURE ... LOW_TEMPERATURE ... 2 bytes ... signed ... dC
		tS16 _s16LOW_TEMPERATURE = 0;
		(tVoid)bConvertRawKDSData(_s16LOW_TEMPERATURE, _pu8Data);
		_pu8Data += sizeof(_s16LOW_TEMPERATURE);
		m_oKDSConfigData.LOW_TEMPERATURE.LOW_TEMPERATURE = (tS8)(_s16LOW_TEMPERATURE / 10);
		// u8GetBRIGTHNESSNOOFHMIOFFSET ... BRIGTHNESS_NO_OF_HMI_OFFSET ... new
		(tVoid)bConvertRawKDSData(m_oKDSConfigData.BRIGTHNESS_NO_OF_HMI_OFFSET.BRIGTHNESS_NO_OF_HMI_OFFSET, _pu8Data);
		_pu8Data += sizeof(m_oKDSConfigData.BRIGTHNESS_NO_OF_HMI_OFFSET.BRIGTHNESS_NO_OF_HMI_OFFSET);
		// u8GetTEMPHYSTERESIS ... TEMP_HYSTERESIS ... old ... obsolete
		(tVoid)bConvertRawKDSData(m_oKDSConfigData.TEMP_HYSTERESIS.TEMP_HYSTERESIS, _pu8Data);
		_pu8Data += sizeof(m_oKDSConfigData.TEMP_HYSTERESIS.TEMP_HYSTERESIS);
		// u8GetLIGHTNINGTIMINGDELAY ... DAY_NIGHT_DELAY ...
		(tVoid)bConvertRawKDSData(m_oKDSConfigData.DAY_NIGHT_DELAY.DAY_NIGHT_DELAY, _pu8Data);
		_pu8Data += sizeof(m_oKDSConfigData.DAY_NIGHT_DELAY.DAY_NIGHT_DELAY);
		// u8GetKEYBRIGHTNESSNUMOFOFFSETS ... KEY_BRIGHTNESS_NUMOF_OFFSETS ...
		(tVoid)bConvertRawKDSData(m_oKDSConfigData.KEY_BRIGHTNESS_NUMOF_OFFSETS.KEY_BRIGHTNESS_NUMOF_OFFSETS, _pu8Data);
		_pu8Data += sizeof(m_oKDSConfigData.KEY_BRIGHTNESS_NUMOF_OFFSETS.KEY_BRIGHTNESS_NUMOF_OFFSETS);
		// u8GetSOFTDIMMINGENABLED ... DAMPING_STRATEGY ...
		(tVoid)bConvertRawKDSData(m_oKDSConfigData.DAMPING_STRATEGY.DAMPING_STRATEGY, _pu8Data);
		_pu8Data += sizeof(m_oKDSConfigData.DAMPING_STRATEGY.DAMPING_STRATEGY);
		// u8GetSOFTDIMMINGSAMPLINGTIME ... DAMPING_SAMPLING_TIMETE ...
		(tVoid)bConvertRawKDSData(m_oKDSConfigData.DAMPING_SAMPLING_TIMETE.DAMPING_SAMPLING_TIMETE, _pu8Data);
		_pu8Data += sizeof(m_oKDSConfigData.DAMPING_SAMPLING_TIMETE.DAMPING_SAMPLING_TIMETE);
		// u8GetSOFTDIMMINGCONSTANTK ... DAMPING_CONSTANTK ...
		(tVoid)bConvertRawKDSData(m_oKDSConfigData.DAMPING_CONSTANTK.DAMPING_CONSTANTK, _pu8Data);
		_pu8Data += sizeof(m_oKDSConfigData.DAMPING_CONSTANTK.DAMPING_CONSTANTK);
		// u8GetSOFTDIMMINGCONSTANTJ ... DAMPING_CONSTANTJ ...
		(tVoid)bConvertRawKDSData(m_oKDSConfigData.DAMPING_CONSTANTJ.DAMPING_CONSTANTJ, _pu8Data);
		_pu8Data += sizeof(m_oKDSConfigData.DAMPING_CONSTANTJ.DAMPING_CONSTANTJ);
		// u8GetBRIGTHNESSBACKLIGHTDAYBUTTONSDAY1 ... BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY ... 2 bytes .. 1/10percent
		tU16 _u16BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY_1 = 0;
		(tVoid)bConvertRawKDSData(_u16BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY_1, _pu8Data);
		m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY = _u16BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY_1 / 10;
		_pu8Data += sizeof(m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY);
		// u8GetBRIGTHNESSBACKLIGHTDAYBUTTONSNIGHT1 ... BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT ... 2 bytes .. 1/10percent
		tU16 _u16BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT_1 = 0;
		(tVoid)bConvertRawKDSData(_u16BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT_1, _pu8Data);
		m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT = _u16BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT_1 / 10;
		_pu8Data += sizeof(m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT);
		// u8GetBRIGTHNESSBACKLIGHTNIGHTBUTTONSNIGHT1 ... BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT ... 2 bytes .. 1/10percent
		tU16 _u16BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT = 0;
		(tVoid)bConvertRawKDSData(_u16BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT, _pu8Data);
		m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT.BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT = _u16BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT / 10;
		_pu8Data += sizeof(_u16BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT);
		// u8GetBRIGTHNESSBACKLIGHTDAYBUTTONSON ... BRIGTHNESSBACKLIGHT_DAYBUTTONS_ON ... 2 bytes .. 1/10percent
		tU16 _u16BRIGTHNESSBACKLIGHT_DAYBUTTONS_ON = 0;
		(tVoid)bConvertRawKDSData(_u16BRIGTHNESSBACKLIGHT_DAYBUTTONS_ON, _pu8Data);
		m_oKDSConfigData.BRIGTHNESSBACKLIGHT_DAYBUTTONS_ON.BRIGTHNESSBACKLIGHT_DAYBUTTONS_ON = _u16BRIGTHNESSBACKLIGHT_DAYBUTTONS_ON / 10;
		_pu8Data += sizeof(_u16BRIGTHNESSBACKLIGHT_DAYBUTTONS_ON);
		// u8GetBRIGTHNESSBACKLIGHTDAYBUTTONSOFF1 ... BRIGTHNESS_BACKLIGHT_DAYBUTTONS_OFF ... 2 bytes .. 1/10percent
		tU16 _u16BRIGTHNESS_BACKLIGHT_DAYBUTTONS_OFF = 0;
		(tVoid)bConvertRawKDSData(_u16BRIGTHNESS_BACKLIGHT_DAYBUTTONS_OFF, _pu8Data);
		m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_DAYBUTTONS_OFF.BRIGTHNESS_BACKLIGHT_DAYBUTTONS_OFF = _u16BRIGTHNESS_BACKLIGHT_DAYBUTTONS_OFF / 10;
		_pu8Data += sizeof(_u16BRIGTHNESS_BACKLIGHT_DAYBUTTONS_OFF);
		// u8GetBRIGTHNESSBACKLIGHTNIGHTBUTTONSON21 ... BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS
		for (tU32 _u32Idx = 0; DATAPOOL_SCC_BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON_NUMBER > _u32Idx; ++_u32Idx)
		{
			tU16 _u16BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON = 0;
			(tVoid)bConvertRawKDSData(_u16BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON, _pu8Data);
			// NOTE: dimenssion in percent or 1/10percent?
			m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON[_u32Idx] = _u16BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON / 10;
			_pu8Data += sizeof(m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON[0]);
		}
	}
	// reading from KDS - CD
	// DISPLAY_TYPE ... from CD only ... SystemConfiguration1.TouchScreenConfiguration
	tU8 _u8DisplayType = 0;
	_s32ReturnCode = DP_s32GetConfigItem( "SystemConfiguration1", "TouchScreenConfiguration", &_u8DisplayType, sizeof(_u8DisplayType), FALSE );
	if  (DP_S32_NO_ERR > _s32ReturnCode)
	{
		ETG_TRACE_USR1(("bLoadKDSData_V1: KDS SystemConfiguration1.TouchScreenConfiguration could not read (error=0x%08x)",
				_s32ReturnCode ));
		_bReturn = FALSE;
	}
	else
	{
		m_oKDSConfigData.DISPLAY_TYPE.DISPLAY_TYPE = _u8DisplayType;
	}
	// DIMMING_COCKPIT_STRATEGY ... from CD only ... HMIConfig.CFGDimmingStrategy
	tU8 _u8CFGDimmingStrategy = 0;
	_s32ReturnCode = DP_s32GetConfigItem( "HMIConfig", "CFGDimmingStrategy", &_u8CFGDimmingStrategy, sizeof(_u8CFGDimmingStrategy), FALSE );
	if  (DP_S32_NO_ERR > _s32ReturnCode)
	{
		ETG_TRACE_USR1(("bLoadKDSData_V1: KDS HMIConfig.CFGDimmingStrategy could not read (error=0x%08x)",
				_s32ReturnCode ));
		_bReturn = FALSE;
	}
	else
	{
		m_oKDSConfigData.DIMMING_COCKPIT_STRATEGY.DIMMING_COCKPIT_STRATEGY = _u8CFGDimmingStrategy;
	}

	// For Dimming Architecture which depends on 3 config items
	{

	// VCAN DIVERSITY ... from CD only ... VehicleInformation.AllianceVCANDiversityType
		tU8 _u8AllianceVCANDiversityType = 0;
		_s32ReturnCode = DP_s32GetConfigItem( "VehicleInformation", "VCANDiversity", &_u8AllianceVCANDiversityType, sizeof(_u8AllianceVCANDiversityType), FALSE );
		if  (DP_S32_NO_ERR > _s32ReturnCode)
		{
			ETG_TRACE_USR1(("bLoadKDSData_V1: KDS VehicleInformation.AllianceVCANDiversityType could not read (error=0x%08x)",
					_s32ReturnCode ));
			_bReturn = FALSE;
		}
		else
		{
			ETG_TRACE_USR1(("bLoadKDSData_V1: KDS VehicleInformation.AllianceVCANDiversityType=%d ",_u8AllianceVCANDiversityType ));
		}

	// Vehicle Type ... from CD only ... VehicleInformation.AllianceVehicleType
		tU16 _u16AllianceVehicleType = 0;
		tU8 VehicleType[2]={0};
		_s32ReturnCode = DP_s32GetConfigItem( "VehicleInformation", "VehicleType", VehicleType, sizeof(VehicleType), FALSE );
		if  (DP_S32_NO_ERR > _s32ReturnCode)
		{
			ETG_TRACE_USR1(("bLoadKDSData_V1: KDS VehicleInformation.AllianceVehicleType could not read (error=0x%08x)",
					_s32ReturnCode ));
			_bReturn = FALSE;
		}
		else
		{
			_u16AllianceVehicleType = (  (((tU16)(VehicleType[0]))<<8) | (tU16)(VehicleType[1]) );
			ETG_TRACE_USR1(("bLoadKDSData_V1: KDS VehicleInformation.AllianceVehicleType=%d ",_u16AllianceVehicleType ));
		}

		tU8 u8BrandType = 0x01; // 0x01 means Nissan, default initialize to Nissan

	//Now Reading Brand Type from KDS
		/*AllianceBrandType 3 Bit
		* [0x00] Renault
		* [0x01] Nissan
		* [0x02] Infiniti*/
		if ( DP_S32_NO_ERR == DP_s32GetConfigItem("SystemConfiguration1", "Brand", &u8BrandType, 1) )
		{
			ETG_TRACE_USR1((" Brand Type is ..  0x%02x.", u8BrandType));
		}
		else
		{
			ETG_TRACE_ERR(("u8BrandType(): Read has failed !!! "));
		}
		switch(u8BrandType)
		{
			case Nissan:
			case EV:
			{
				if(_u16AllianceVehicleType == P42M)
				{
					m_oKDSConfigData.DIMMING_ARCHITECTURE.DIMMING_ARCHITECTURE = Nissan_P42M;
				}
				else if(_u8AllianceVCANDiversityType == slot6)
				{
					m_oKDSConfigData.DIMMING_ARCHITECTURE.DIMMING_ARCHITECTURE = Nissan_Slot6;
				}
				else if(_u8AllianceVCANDiversityType == slot5)
				{
					m_oKDSConfigData.DIMMING_ARCHITECTURE.DIMMING_ARCHITECTURE = Nissan_Slot5;
				}
				else if(_u8AllianceVCANDiversityType == slot1)
				{
					m_oKDSConfigData.DIMMING_ARCHITECTURE.DIMMING_ARCHITECTURE = Nissan_Slot1;
				}
			}
			break;
			case PIVI:
			{
				m_oKDSConfigData.DIMMING_ARCHITECTURE.DIMMING_ARCHITECTURE = PIVI_device;
			}
			break;
			case Renault:
			case Dacia:
			case Daimler:
			case Avtovaz:
			case RSM:
			{
				if(_u8AllianceVCANDiversityType == slot3)
				{
					m_oKDSConfigData.DIMMING_ARCHITECTURE.DIMMING_ARCHITECTURE = Renault_Slot3;
				}
				else if(_u8AllianceVCANDiversityType == slot1)
				{
					m_oKDSConfigData.DIMMING_ARCHITECTURE.DIMMING_ARCHITECTURE = Renault_Slot1;
				}
			}
			break;
			default:
			{
				m_oKDSConfigData.DIMMING_ARCHITECTURE.DIMMING_ARCHITECTURE = default_type;
			}
		}
	}


	return _bReturn;
}

/*******************************************************************************
*
* FUNCTION: 	bLoadCalibrationData
*
* DESCRIPTION: 	load KDS calibration data
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	tBool 	{ TRUE - successful | else - failed }
*
*******************************************************************************/
tBool dispvidctrl_tclConfigPdd::bLoadCalibrationData()
{
	ETG_TRACE_USR4(("bLoadCalibrationData() entered."));

	tBool _bReturn = FALSE;
	tU8 _au8KDSRawDataDimmingCalibrationParameterSet[DP_U8_KDSLEN_DIMMINGCALIBRATIONPARAMETERSET_COMPLETE] = {0};
	tS32 _s32ReturnCode = DP_s32GetConfigItem( "DimmingCalibrationParameterSet", "",
			_au8KDSRawDataDimmingCalibrationParameterSet, sizeof(_au8KDSRawDataDimmingCalibrationParameterSet));
	if  (DP_S32_NO_ERR != _s32ReturnCode)
	{
		m_bUseDefaultCalibrationValues = TRUE;
		ETG_TRACE_USR1(("bLoadKDSData: KDS _au8KDSRawDataDimmingCalibrationParameterSet could not read (error=0x%08x)", _s32ReturnCode));
	}
	else
	{
		ETG_TRACE_USR1(("bLoadKDSData: KDS _au8KDSRawDataDimmingCalibrationParameterSet read successful (return=%d)", _s32ReturnCode));
		tU8* _pu8Data = _au8KDSRawDataDimmingCalibrationParameterSet;
		// KEYB_LED_CALIBRATION_VALUE
		for (tU32 _u32Idx = 0; (DP_U8_KDSLEN_DIMMINGCALIBRATIONPARAMETERSET_COMPLETE / 2) > _u32Idx; ++_u32Idx)
		{
			(tVoid)bConvertRawKDSData(m_oKDSConfigData.KEYB_LED_CALIBRATION_VALUE.KEYB_LED_CALIBRATION_VALUE[_u32Idx], _pu8Data);
			_pu8Data += sizeof(m_oKDSConfigData.KEYB_LED_CALIBRATION_VALUE.KEYB_LED_CALIBRATION_VALUE[0]);
		}
		_bReturn = TRUE;
	}

	return _bReturn;
}

/*******************************************************************************
*
* FUNCTION: 	bIsValidKDSData
*
* DESCRIPTION: 	loaded KDS configuration data are valied
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	tBool 	{ TRUE - valid | else - invalid }
*
*******************************************************************************/
tBool dispvidctrl_tclConfigPdd::bIsValidKDSData() const
{
	ETG_TRACE_USR4(("bIsValidKDSData() entered."));
	tBool _bValid = TRUE;

	// KDS data should be loaded
	if (TRUE != m_bKDSLoaded)
		return FALSE;

	// BRIGTHNESS_NO_OF_HMI_OFFSET
	if (5 > m_oKDSConfigData.BRIGTHNESS_NO_OF_HMI_OFFSET.BRIGTHNESS_NO_OF_HMI_OFFSET
		|| DATAPOOL_SCC_BRIGTHNESS_CURVE_HMI_NIGHT_NUMBER < m_oKDSConfigData.BRIGTHNESS_NO_OF_HMI_OFFSET.BRIGTHNESS_NO_OF_HMI_OFFSET
		|| DATAPOOL_SCC_BRIGTHNESS_CURVE_HMI_DAY_NUMBER < m_oKDSConfigData.BRIGTHNESS_NO_OF_HMI_OFFSET.BRIGTHNESS_NO_OF_HMI_OFFSET)
	{
		_bValid = FALSE;
		ETG_TRACE_FATAL(("bIsValidKDSData: BRIGTHNESS_NO_OF_HMI_OFFSET=%d is invalid!",
				m_oKDSConfigData.BRIGTHNESS_NO_OF_HMI_OFFSET.BRIGTHNESS_NO_OF_HMI_OFFSET ));
	}
	// BRIGTHNESS_CURVE_HMI_NIGHT
	for (tU32 _u32Idx = 0; DATAPOOL_SCC_BRIGTHNESS_CURVE_HMI_NIGHT_NUMBER > _u32Idx && (tU32)m_oKDSConfigData.BRIGTHNESS_NO_OF_HMI_OFFSET.BRIGTHNESS_NO_OF_HMI_OFFSET > _u32Idx; ++_u32Idx)
	{
		if (0 == m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_NIGHT.BRIGTHNESS_CURVE_HMI_NIGHT[_u32Idx])
		{
			_bValid = FALSE;
			ETG_TRACE_FATAL(("bIsValidKDSData: BRIGTHNESS_CURVE_HMI_NIGHT[%d]=%d is invalid!",
					_u32Idx, m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_NIGHT.BRIGTHNESS_CURVE_HMI_NIGHT[_u32Idx] ));
			break;
		}
	}
	// BRIGTHNESS_CURVE_HMI_DAY
	for (tU32 _u32Idx = 0; DATAPOOL_SCC_BRIGTHNESS_CURVE_HMI_DAY_NUMBER > _u32Idx && (tU32)m_oKDSConfigData.BRIGTHNESS_NO_OF_HMI_OFFSET.BRIGTHNESS_NO_OF_HMI_OFFSET > _u32Idx; ++_u32Idx)
	{
		if (0 == m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_DAY.BRIGTHNESS_CURVE_HMI_DAY[_u32Idx])
		{
			_bValid = FALSE;
			ETG_TRACE_FATAL(("bIsValidKDSData: BRIGTHNESS_CURVE_HMI_DAY[%d]=%d is invalid!",
					_u32Idx, m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_DAY.BRIGTHNESS_CURVE_HMI_DAY[_u32Idx] ));
			break;
		}
	}
	// DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY
	if (0 == m_oKDSConfigData.DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY.DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY)
	{
		_bValid = FALSE;
		ETG_TRACE_FATAL(("bIsValidKDSData: DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY=%d is invalid!",
				m_oKDSConfigData.DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY.DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY ));
	}
	// KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY
	if (0 == m_oKDSConfigData.KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY.KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY)
	{
		_bValid = FALSE;
		ETG_TRACE_FATAL(("bIsValidKDSData: KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY=%d is invalid!",
				m_oKDSConfigData.KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY.KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY ));
	}
	// NUMBER_OF_KEYS
	if (DATAPOOL_SCC_KEYB_LED_CALIBRATION_VALUE_NUMBER < m_oKDSConfigData.NUMBER_OF_KEYS.NUMBER_OF_KEYS)
	{
		_bValid = FALSE;
		ETG_TRACE_FATAL(("bIsValidKDSData: NUMBER_OF_KEYS=%d is invalid!",
				m_oKDSConfigData.NUMBER_OF_KEYS.NUMBER_OF_KEYS ));
	}
	// HIGH_TEMPERATURE
	if (40 > m_oKDSConfigData.HIGH_TEMPERATURE.HIGH_TEMPERATURE)
	{
		_bValid = FALSE;
		ETG_TRACE_FATAL(("bIsValidKDSData: HIGH_TEMPERATURE=%d is invalid!",
				m_oKDSConfigData.HIGH_TEMPERATURE.HIGH_TEMPERATURE ));
	}
	// CRITICAL_TEMPERATURE
	if (m_oKDSConfigData.CRITICAL_TEMPERATURE.CRITICAL_TEMPERATURE < m_oKDSConfigData.HIGH_TEMPERATURE.HIGH_TEMPERATURE)
	{
		_bValid = FALSE;
		ETG_TRACE_FATAL(("bIsValidKDSData: CRITICAL_TEMPERATURE=%d is invalid!",
				m_oKDSConfigData.CRITICAL_TEMPERATURE.CRITICAL_TEMPERATURE ));
	}
	// LOW_TEMPERATURE
	if (0 < m_oKDSConfigData.LOW_TEMPERATURE.LOW_TEMPERATURE)
	{
		_bValid = FALSE;
		ETG_TRACE_FATAL(("bIsValidKDSData: LOW_TEMPERATURE=%d is invalid!",
				m_oKDSConfigData.LOW_TEMPERATURE.LOW_TEMPERATURE ));
	}
	// TEMP_HYSTERESIS
	// TEMP_COMPENSATION_PWM_MIN

	// KEYB_LED_CALIBRATION_VALUE
	for (tU32 _u32Idx = 0; DATAPOOL_SCC_KEYB_LED_CALIBRATION_VALUE_NUMBER > _u32Idx && (tU32)m_oKDSConfigData.NUMBER_OF_KEYS.NUMBER_OF_KEYS > _u32Idx; ++_u32Idx)
	{
		if (0 == m_oKDSConfigData.KEYB_LED_CALIBRATION_VALUE.KEYB_LED_CALIBRATION_VALUE[_u32Idx])
		{
			_bValid = FALSE;
			ETG_TRACE_FATAL(("bIsValidKDSData: KEYB_LED_CALIBRATION_VALUE[%d]=%d is invalid!",
					_u32Idx, m_oKDSConfigData.KEYB_LED_CALIBRATION_VALUE.KEYB_LED_CALIBRATION_VALUE[_u32Idx] ));
		}
	}

	return _bValid;
}
/*******************************************************************************
*
* FUNCTION: 	bIsChangedKDSData
*
* DESCRIPTION: 	compare the loaded KDS configuration data and PDD-DIM
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	tBool {TRUE - KDS data changed | else KDS data unchanged}
*
*******************************************************************************/
tBool dispvidctrl_tclConfigPdd::bIsChangedKDSData() const
{
	ETG_TRACE_USR4(("bIsChangedKDSData() entered."));

	if (TRUE != m_bKDSLoaded)
		return FALSE;

	tU32 _u32ChangedCount = 0;
	// BRIGTHNESS_CURVE_HMI_NIGHT
	dp_tclDimmingDpPDDBRIGTHNESS_CURVE_HMI_NIGHT _oDpBrightnessCurveHmiNight;
	tU16 _au16BRIGTHNESS_CURVE_HMI_NIGHT[DATAPOOL_SCC_BRIGTHNESS_CURVE_HMI_NIGHT_NUMBER] = {0};
	(tVoid)_oDpBrightnessCurveHmiNight.s32GetData(_au16BRIGTHNESS_CURVE_HMI_NIGHT,
			(sizeof(_au16BRIGTHNESS_CURVE_HMI_NIGHT)/sizeof(_au16BRIGTHNESS_CURVE_HMI_NIGHT[0])));
	if (OSAL_NULL != OSAL_s32MemoryCompare(m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_NIGHT.BRIGTHNESS_CURVE_HMI_NIGHT,
			_au16BRIGTHNESS_CURVE_HMI_NIGHT, sizeof(m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_NIGHT.BRIGTHNESS_CURVE_HMI_NIGHT)))
	{
		++_u32ChangedCount;
		ETG_TRACE_USR1(("bIsChangedKDSData: BRIGTHNESS_CURVE_HMI_NIGHT changed"));
	}
	// BRIGTHNESS_CURVE_HMI_DAY
	dp_tclDimmingDpPDDBRIGTHNESS_CURVE_HMI_DAY _oDpBrightnessCurveHmiDay;
	tU16 _au16BRIGTHNESS_CURVE_HMI_DAY[DATAPOOL_SCC_BRIGTHNESS_CURVE_HMI_DAY_NUMBER] = {0};
	(tVoid)_oDpBrightnessCurveHmiDay.s32GetData(_au16BRIGTHNESS_CURVE_HMI_DAY,
			(sizeof(_au16BRIGTHNESS_CURVE_HMI_DAY)/sizeof(_au16BRIGTHNESS_CURVE_HMI_DAY[0])));
	if (OSAL_NULL != OSAL_s32MemoryCompare(m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_DAY.BRIGTHNESS_CURVE_HMI_DAY,
			_au16BRIGTHNESS_CURVE_HMI_DAY, sizeof(m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_DAY.BRIGTHNESS_CURVE_HMI_DAY)))
	{
		++_u32ChangedCount;
		ETG_TRACE_USR1(("bIsChangedKDSData: BRIGTHNESS_CURVE_HMI_DAY changed"));
	}
	// DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY
	dp_tclDimmingDpPDDDISPLAY_ILLUMINATION_OUTPUT_FREQUENCY _oDpDisplayIlluminationOutputFrequency;
	if (_oDpDisplayIlluminationOutputFrequency.tGetData() != m_oKDSConfigData.DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY.DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY)
	{
		++_u32ChangedCount;
		ETG_TRACE_USR1(("bIsChangedKDSData: DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY changed"));
	}
	// KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY
	dp_tclDimmingDpPDDKEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY _oDbKeysEncoderIlluminationOutputFrequency;
	if (_oDbKeysEncoderIlluminationOutputFrequency.tGetData() != m_oKDSConfigData.KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY.KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY)
	{
		++_u32ChangedCount;
		ETG_TRACE_USR1(("bIsChangedKDSData: KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY changed"));
	}
	// BRIGTHNESS_NO_OF_HMI_OFFSET
	dp_tclDimmingDpPDDBRIGTHNESS_NO_OF_HMI_OFFSET _oDpBrightnessNoOfHmiOffset;
	if (_oDpBrightnessNoOfHmiOffset.tGetData() != m_oKDSConfigData.BRIGTHNESS_NO_OF_HMI_OFFSET.BRIGTHNESS_NO_OF_HMI_OFFSET)
	{
		++_u32ChangedCount;
		ETG_TRACE_USR1(("bIsChangedKDSData: BRIGTHNESS_NO_OF_HMI_OFFSET changed"));
	}
	// NUMBER_OF_KEYS
	dp_tclDimmingDpPDDNUMBER_OF_KEYS _oDpNumbersOfKeys;
	if (_oDpNumbersOfKeys.tGetData() != m_oKDSConfigData.NUMBER_OF_KEYS.NUMBER_OF_KEYS)
	{
		++_u32ChangedCount;
		ETG_TRACE_USR1(("bIsChangedKDSData: NUMBER_OF_KEYS changed"));
	}
	// HIGH_TEMPERATURE
	dp_tclDimmingDpPDDHIGH_TEMPERATURE _oDpHighTemperature;
	if (_oDpHighTemperature.tGetData() != m_oKDSConfigData.HIGH_TEMPERATURE.HIGH_TEMPERATURE)
	{
		++_u32ChangedCount;
		ETG_TRACE_USR1(("bIsChangedKDSData: HIGH_TEMPERATURE changed"));
	}
	// CRITICAL_TEMPERATURE
	dp_tclDimmingDpPDDCRITICAL_TEMPERATURE _oDpCriticalTemperature;
	if (_oDpCriticalTemperature.tGetData() != m_oKDSConfigData.CRITICAL_TEMPERATURE.CRITICAL_TEMPERATURE)
	{
		++_u32ChangedCount;
		ETG_TRACE_USR1(("bIsChangedKDSData: CRITICAL_TEMPERATURE changed"));
	}
	// LOW_TEMPERATURE
	dp_tclDimmingDpPDDLOW_TEMPERATURE _oDpLowTemperature;
	if (_oDpLowTemperature.tGetData() != m_oKDSConfigData.LOW_TEMPERATURE.LOW_TEMPERATURE)
	{
		++_u32ChangedCount;
		ETG_TRACE_USR1(("bIsChangedKDSData: LOW_TEMPERATURE changed"));
	}
	// TEMP_HYSTERESIS
	dp_tclDimmingDpPDDTEMP_HYSTERESIS _oDpTempHysteresis;
	if (_oDpTempHysteresis.tGetData() != m_oKDSConfigData.TEMP_HYSTERESIS.TEMP_HYSTERESIS)
	{
		++_u32ChangedCount;
		ETG_TRACE_USR1(("bIsChangedKDSData: TEMP_HYSTERESIS changed"));
	}
	// TEMP_COMPENSATION_PWM_MIN
	dp_tclDimmingDpPDDTEMP_COMPENSATION_PWM_MIN _oDpTempCompensationPwmMin;
	if (_oDpTempCompensationPwmMin.tGetData() != m_oKDSConfigData.TEMP_COMPENSATION_PWM_MIN.TEMP_COMPENSATION_PWM_MIN)
	{
		++_u32ChangedCount;
		ETG_TRACE_USR1(("bIsChangedKDSData: TEMP_COMPENSATION_PWM_MIN changed"));
	}
	// KEYB_LED_CALIBRATION_VALUE
	dp_tclDimmingDpPDDKEYB_LED_CALIBRATION_VALUE _oDpKeybLedCalibrationValue;
	tU16 _au16KEYB_LED_CALIBRATION_VALUE[DATAPOOL_SCC_KEYB_LED_CALIBRATION_VALUE_NUMBER] = {0};
	(tVoid)_oDpKeybLedCalibrationValue.s32GetData(_au16KEYB_LED_CALIBRATION_VALUE,
			(sizeof(_au16KEYB_LED_CALIBRATION_VALUE)/sizeof(_au16KEYB_LED_CALIBRATION_VALUE[0])));
	if (OSAL_NULL != OSAL_s32MemoryCompare(m_oKDSConfigData.KEYB_LED_CALIBRATION_VALUE.KEYB_LED_CALIBRATION_VALUE,
			_au16KEYB_LED_CALIBRATION_VALUE, sizeof(m_oKDSConfigData.KEYB_LED_CALIBRATION_VALUE.KEYB_LED_CALIBRATION_VALUE)))
	{
		++_u32ChangedCount;
		ETG_TRACE_USR1(("bIsChangedKDSData: KEYB_LED_CALIBRATION_VALUE changed"));
	}

	// DISPLAY_TYPE
	dp_tclDimmingDpPDDDISPLAY_TYPE _oDpDisplayType;
	if (_oDpDisplayType.tGetData() != m_oKDSConfigData.DISPLAY_TYPE.DISPLAY_TYPE)
	{
		++_u32ChangedCount;
		ETG_TRACE_USR1(("bIsChangedKDSData: DISPLAY_TYPE changed"));
	}

	// DIMMING COCKPIT STRATEGY
	dp_tclDimmingDpPDDDIMMING_COCKPIT_STRATEGY _oDpDimmingCockpitStrategy;
	if (_oDpDimmingCockpitStrategy.tGetData() != m_oKDSConfigData.DIMMING_COCKPIT_STRATEGY.DIMMING_COCKPIT_STRATEGY)
	{
		++_u32ChangedCount;
		ETG_TRACE_USR1(("bIsChangedKDSData: DIMMING_COCKPIT_STRATEGY changed"));
	}

	// KEY_BRIGHTNESS_NUMOF_OFFSETS
	dp_tclDimmingDpPDDKEY_BRIGHTNESS_NUMOF_OFFSETS _oDpKeyBrightnessNumofOffsets;
	if (_oDpKeyBrightnessNumofOffsets.tGetData() != m_oKDSConfigData.KEY_BRIGHTNESS_NUMOF_OFFSETS.KEY_BRIGHTNESS_NUMOF_OFFSETS)
	{
		++_u32ChangedCount;
		ETG_TRACE_USR1(("bIsChangedKDSData: KEY_BRIGHTNESS_NUMOF_OFFSETS changed"));
	}

	// DAMPING_STRATEGY
	dp_tclDimmingDpPDDDAMPING_STRATEGY _oDpKeyDampingStrategy;
	if (_oDpKeyDampingStrategy.tGetData() != m_oKDSConfigData.DAMPING_STRATEGY.DAMPING_STRATEGY)
	{
		++_u32ChangedCount;
		ETG_TRACE_USR1(("bIsChangedKDSData: DAMPING_STRATEGY changed"));
	}

	// DAMPING_STRATEGY
	dp_tclDimmingDpPDDDAMPING_SAMPLING_TIMETE _oDpKeyDampingSampling;
	if (_oDpKeyDampingSampling.tGetData() != m_oKDSConfigData.DAMPING_SAMPLING_TIMETE.DAMPING_SAMPLING_TIMETE)
	{
		++_u32ChangedCount;
		ETG_TRACE_USR1(("bIsChangedKDSData: DAMPING_SAMPLING_TIMETE changed"));
	}

	// DAMPING_CONSTANTK
	dp_tclDimmingDpPDDDAMPING_CONSTANTK _oDpKeyDampingConstantK;
	if (_oDpKeyDampingConstantK.tGetData() != m_oKDSConfigData.DAMPING_CONSTANTK.DAMPING_CONSTANTK)
	{
		++_u32ChangedCount;
		ETG_TRACE_USR1(("bIsChangedKDSData: DAMPING_CONSTANTK changed"));
	}

	// DAMPING_CONSTANTJ
	dp_tclDimmingDpPDDDAMPING_CONSTANTJ _oDpKeyDampingConstantJ;
	if (_oDpKeyDampingConstantJ.tGetData() != m_oKDSConfigData.DAMPING_CONSTANTJ.DAMPING_CONSTANTJ)
	{
		++_u32ChangedCount;
		ETG_TRACE_USR1(("bIsChangedKDSData: DAMPING_CONSTANTJ changed"));
	}

	// BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY
	dp_tclDimmingDpPDDBRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY _oDpBrightnessBacklightDayButtonsDay;
	if (_oDpBrightnessBacklightDayButtonsDay.tGetData() != m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY)
	{
		++_u32ChangedCount;
		ETG_TRACE_USR1(("bIsChangedKDSData: BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY changed"));
	}

	// BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT
	dp_tclDimmingDpPDDBRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT _oDpBrightnessBacklightDayButtonsNight;
	if (_oDpBrightnessBacklightDayButtonsNight.tGetData() != m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT)
	{
		++_u32ChangedCount;
		ETG_TRACE_USR1(("bIsChangedKDSData: BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT changed"));
	}

	// BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT
	dp_tclDimmingDpPDDBRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT _oDpBrightnessBacklightNightButtonsNight;
	if (_oDpBrightnessBacklightNightButtonsNight.tGetData() != m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT.BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT)
	{
		++_u32ChangedCount;
		ETG_TRACE_USR1(("bIsChangedKDSData: BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT changed"));
	}

	// BRIGTHNESSBACKLIGHT_DAYBUTTONS_ON
	dp_tclDimmingDpPDDBRIGTHNESSBACKLIGHT_DAYBUTTONS_ON _oDpBrightnessBacklightDaybuttonsOn;
	if (_oDpBrightnessBacklightDaybuttonsOn.tGetData() != m_oKDSConfigData.BRIGTHNESSBACKLIGHT_DAYBUTTONS_ON.BRIGTHNESSBACKLIGHT_DAYBUTTONS_ON)
	{
		++_u32ChangedCount;
		ETG_TRACE_USR1(("bIsChangedKDSData: BRIGTHNESSBACKLIGHT_DAYBUTTONS_ON changed"));
	}

	// BRIGTHNESSBACKLIGHT_DAYBUTTONS_OFF
	dp_tclDimmingDpPDDBRIGTHNESS_BACKLIGHT_DAYBUTTONS_OFF _oDpBrightnessBacklightDaybuttonsOFF;
	if (_oDpBrightnessBacklightDaybuttonsOFF.tGetData() != m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_DAYBUTTONS_OFF.BRIGTHNESS_BACKLIGHT_DAYBUTTONS_OFF)
	{
		++_u32ChangedCount;
		ETG_TRACE_USR1(("bIsChangedKDSData: BRIGTHNESS_BACKLIGHT_DAYBUTTONS_OFF changed"));
	}

	// BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON
	dp_tclDimmingDpPDDBRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON _oDpBrightnessBacklightNightbuttonsOff;
	tU16 _au16BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS[DATAPOOL_SCC_BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON_NUMBER] = {0};
	(tVoid)_oDpBrightnessBacklightNightbuttonsOff.s32GetData(_au16BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS,
			(sizeof(_au16BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS)/sizeof(_au16BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS[0])));
	if (OSAL_NULL != OSAL_s32MemoryCompare(m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON,
			_au16BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS, sizeof(m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON)))
	{
		++_u32ChangedCount;
		ETG_TRACE_USR1(("bIsChangedKDSData: KEYB_LED_CALIBRATION_VALUE changed"));
	}

	// DAY_NIGHT_DELAY

	dp_tclDimmingDpPDDDAY_NIGHT_DELAY _oDpDayNightDelayParameter;
	if (_oDpDayNightDelayParameter.tGetData() != m_oKDSConfigData.DAY_NIGHT_DELAY.DAY_NIGHT_DELAY)
	{
		++_u32ChangedCount;
		ETG_TRACE_USR1(("bIsChangedKDSData: LIGHTNETING_PARAMETER changed"));
	}

	//BRANDNAME
	dp_tclDimmingDpPDDBRANDNAME _oDpBrandValue;
	if(_oDpBrandValue.tGetData() != m_oKDSConfigData.BRANDNAME.BRANDNAME)
	{
		++_u32ChangedCount;
		ETG_TRACE_USR1(("bIsChangedKDSData: BRANDNAME changed"));
	}

	//DIMMING ARCHITECTURE
	dp_tclDimmingDpPDDDIMMING_ARCHITECTURE _oDpArchitectureValue;
	if(_oDpArchitectureValue.tGetData() != m_oKDSConfigData.DIMMING_ARCHITECTURE.DIMMING_ARCHITECTURE)
		{
			++_u32ChangedCount;
			ETG_TRACE_USR1(("bIsChangedKDSData: DIMMING_ARCHITECTURE changed"));
		}

	// update counter
	ETG_TRACE_USR1(("bIsChangedKDSData: %d changed KDS data elements", _u32ChangedCount));
	return (OSAL_NULL < _u32ChangedCount);
}

/*******************************************************************************
*
* FUNCTION: 	bWritePddData
*
* DESCRIPTION: 	write the loaded KDS configuration data to PDD-DIM
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	tBool {TRUE - successful | else failed}
*
*******************************************************************************/
tBool dispvidctrl_tclConfigPdd::bWritePddData()
{

	tU8 u8BrandType = 0x01; // 0x01 means Nissan, default initialize to Nissan

	//Now Reading Brand Type from KDS
    /*AllianceBrandType 3 Bit
    * [0x00] Renault
    * [0x01] Nissan
    * [0x02] Infiniti*/
	if ( DP_S32_NO_ERR == DP_s32GetConfigItem("SystemConfiguration1", "Brand", &u8BrandType, 1) )
	{
		m_oKDSConfigData.BRANDNAME.BRANDNAME = u8BrandType;
		ETG_TRACE_USR1((" Brand Type is ..  0x%02x.", u8BrandType));
	}
	else
	{
		ETG_TRACE_ERR(("u8BrandType(): Read has failed !!! "));
	}
    //Reading Brand Time Finished..

	ETG_TRACE_USR4(("vWritePddData() entered."));
	if (TRUE != m_bKDSLoaded || TRUE != bIsValidKDSData())
		return FALSE;

	if (TRUE == bIsChangedKDSData())
	{
		tS32 _s32ReturnValue = DP_S32_NO_ERR;
		ETG_TRACE_USR4(("vWritePddData: result %d ", _s32ReturnValue ));
		// BRIGTHNESS_CURVE_HMI_NIGHT
		{
			dp_tclDimmingDpPDDBRIGTHNESS_CURVE_HMI_NIGHT _oDpValue;
			_s32ReturnValue = _oDpValue.s32SetData(m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_NIGHT.BRIGTHNESS_CURVE_HMI_NIGHT,
					(sizeof(m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_NIGHT.BRIGTHNESS_CURVE_HMI_NIGHT)/sizeof(m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_NIGHT.BRIGTHNESS_CURVE_HMI_NIGHT[0]) ));
			ETG_TRACE_USR1(("vWritePddData: BRIGTHNESS_CURVE_HMI_NIGHT set with result %d ", _s32ReturnValue ));
		}
		// BRIGTHNESS_CURVE_HMI_DAY
		{
			dp_tclDimmingDpPDDBRIGTHNESS_CURVE_HMI_DAY _oDpValue;
			_s32ReturnValue = _oDpValue.s32SetData(m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_DAY.BRIGTHNESS_CURVE_HMI_DAY,
					(sizeof(m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_DAY.BRIGTHNESS_CURVE_HMI_DAY)/sizeof(m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_DAY.BRIGTHNESS_CURVE_HMI_DAY[0])));
			ETG_TRACE_USR1(("vWritePddData: BRIGTHNESS_CURVE_HMI_DAY set with result %d ", _s32ReturnValue ));
		}
		// DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY
		{
			dp_tclDimmingDpPDDDISPLAY_ILLUMINATION_OUTPUT_FREQUENCY _oDpValue;
			_s32ReturnValue = _oDpValue.s32SetData(m_oKDSConfigData.DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY.DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY);
			ETG_TRACE_USR1(("vWritePddData: DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY set with result %d ", _s32ReturnValue ));
		}
		// KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY
		{
			dp_tclDimmingDpPDDKEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY _oDpValue;
			_s32ReturnValue = _oDpValue.s32SetData(m_oKDSConfigData.KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY.KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY);
			ETG_TRACE_USR1(("vWritePddData: KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY set with result %d ", _s32ReturnValue ));
		}
		// BRIGTHNESS_NO_OF_HMI_OFFSET
		{
			dp_tclDimmingDpPDDBRIGTHNESS_NO_OF_HMI_OFFSET _oDpValue;
			_s32ReturnValue = _oDpValue.s32SetData(m_oKDSConfigData.BRIGTHNESS_NO_OF_HMI_OFFSET.BRIGTHNESS_NO_OF_HMI_OFFSET);
			ETG_TRACE_USR1(("vWritePddData: BRIGTHNESS_NO_OF_HMI_OFFSET set with result %d ", _s32ReturnValue ));
		}
		// NUMBER_OF_KEYS
		{
			dp_tclDimmingDpPDDNUMBER_OF_KEYS _oDpValue;
			_s32ReturnValue = _oDpValue.s32SetData(m_oKDSConfigData.NUMBER_OF_KEYS.NUMBER_OF_KEYS);
			ETG_TRACE_USR1(("vWritePddData: NUMBER_OF_KEYS set with result %d ", _s32ReturnValue ));
		}
		// HIGH_TEMPERATURE
		{
			dp_tclDimmingDpPDDHIGH_TEMPERATURE _oDpValue;
			_s32ReturnValue = _oDpValue.s32SetData(m_oKDSConfigData.HIGH_TEMPERATURE.HIGH_TEMPERATURE);
			ETG_TRACE_USR1(("vWritePddData: HIGH_TEMPERATURE set with result %d ", _s32ReturnValue ));
		}
		// CRITICAL_TEMPERATURE
		{
			dp_tclDimmingDpPDDCRITICAL_TEMPERATURE _oDpValue;
			_s32ReturnValue = _oDpValue.s32SetData(m_oKDSConfigData.CRITICAL_TEMPERATURE.CRITICAL_TEMPERATURE);
			ETG_TRACE_USR1(("vWritePddData: CRITICAL_TEMPERATURE set with result %d ", _s32ReturnValue ));
		}
		// LOW_TEMPERATURE
		{
			dp_tclDimmingDpPDDLOW_TEMPERATURE _oDpValue;
			_s32ReturnValue = _oDpValue.s32SetData(m_oKDSConfigData.LOW_TEMPERATURE.LOW_TEMPERATURE);
			ETG_TRACE_USR1(("vWritePddData: LOW_TEMPERATURE set with result %d ", _s32ReturnValue ));
		}
		// TEMP_HYSTERESIS
		{
			dp_tclDimmingDpPDDTEMP_HYSTERESIS _oDpValue;
			_s32ReturnValue = _oDpValue.s32SetData(m_oKDSConfigData.TEMP_HYSTERESIS.TEMP_HYSTERESIS);
			ETG_TRACE_USR1(("vWritePddData: TEMP_HYSTERESIS set with result %d ", _s32ReturnValue ));
		}
		// TEMP_COMPENSATION_PWM_MIN
		{
			dp_tclDimmingDpPDDTEMP_COMPENSATION_PWM_MIN _oDpValue;
			_s32ReturnValue = _oDpValue.s32SetData(m_oKDSConfigData.TEMP_COMPENSATION_PWM_MIN.TEMP_COMPENSATION_PWM_MIN);
			ETG_TRACE_USR1(("vWritePddData: TEMP_COMPENSATION_PWM_MIN set with result %d ", _s32ReturnValue ));
		}
		// KEYB_LED_CALIBRATION_VALUE
		{
			dp_tclDimmingDpPDDKEYB_LED_CALIBRATION_VALUE _oDpValue;
			_s32ReturnValue = _oDpValue.s32SetData(m_oKDSConfigData.KEYB_LED_CALIBRATION_VALUE.KEYB_LED_CALIBRATION_VALUE,
					(sizeof(m_oKDSConfigData.KEYB_LED_CALIBRATION_VALUE.KEYB_LED_CALIBRATION_VALUE)/sizeof(m_oKDSConfigData.KEYB_LED_CALIBRATION_VALUE.KEYB_LED_CALIBRATION_VALUE[0])));
			ETG_TRACE_USR1(("vWritePddData: KEYB_LED_CALIBRATION_VALUE set with result %d ", _s32ReturnValue ));
		}

		// DisplayType
		{
			
			// touch config file name

			tS32 _s32ReadStatus;
			dp_tclDimmingDPEarlyConfig_Display_Type _oDisplayTypeEarly;
			tU8 u8DisplayType = 0;
			_s32ReadStatus = _oDisplayTypeEarly.s32GetData(u8DisplayType);

			if(DP_S32_NO_ERR <= _s32ReadStatus)
			{	 if ((u8DisplayType == DEFAULT_C_AU16_DISPLAY_TYPE_ATTACHED_VALUE ))
			{
				m_oKDSConfigData.DISPLAY_TYPE.DISPLAY_TYPE = DEFAULT_C_AU16_DISPLAY_TYPE_ATTACHED_VALUE;
			}
			else
			{
				m_oKDSConfigData.DISPLAY_TYPE.DISPLAY_TYPE = 1;
			}

		dp_tclDimmingDpPDDDISPLAY_TYPE _oDpValue;
		
		_s32ReturnValue = _oDpValue.s32SetData(m_oKDSConfigData.DISPLAY_TYPE.DISPLAY_TYPE);
		ETG_TRACE_USR1(("vWritePddData: DISPLAY TYPE  set with result %d ", _s32ReturnValue ));
		}
		

		}

		// DimmingCockpitStrategy_u8
		{
			dp_tclDimmingDpPDDDIMMING_COCKPIT_STRATEGY _oDpValue;
		_s32ReturnValue = _oDpValue.s32SetData(m_oKDSConfigData.DIMMING_COCKPIT_STRATEGY.DIMMING_COCKPIT_STRATEGY);
		ETG_TRACE_USR1(("vWritePddData: DIMMING COCKPIT STRATEGY  set with result %d ", _s32ReturnValue ));
		}

		// KeyBrightness_NumOfOffsets_u8
		{
		dp_tclDimmingDpPDDKEY_BRIGHTNESS_NUMOF_OFFSETS _oDpValue;
		_s32ReturnValue = _oDpValue.s32SetData(m_oKDSConfigData.KEY_BRIGHTNESS_NUMOF_OFFSETS.KEY_BRIGHTNESS_NUMOF_OFFSETS);
		ETG_TRACE_USR1(("vWritePddData: KeyBrightness_NumOfOffsets  set with result %d ", _s32ReturnValue ));

		}

		//Damping_Strategy_u8
		{
					
			//Read Dimmming Cockpit Strategy from CD configuration !!
			
			//tU8 u8BrandType = 0xFF; Making global for BRANDNAME
			

			
			if(u8BrandType==0x00)
			{		
				m_oKDSConfigData.DAMPING_STRATEGY.DAMPING_STRATEGY = DEFAULT_C_AU16_DAMPING_STRATEGY_ENABLED_VALUE;
				
			}
			else
			{
				m_oKDSConfigData.DAMPING_STRATEGY.DAMPING_STRATEGY = DEFAULT_C_AU16_DAMPING_STRATEGY_DISABLED_VALUE;			
			}
			
			
			
		dp_tclDimmingDpPDDDAMPING_STRATEGY _oDpValue;
		_s32ReturnValue = _oDpValue.s32SetData(m_oKDSConfigData.DAMPING_STRATEGY.DAMPING_STRATEGY);
		ETG_TRACE_USR1(("vWritePddData: DAMPING STRATEGY  set with result %d ", _s32ReturnValue ));

		}

		// Damping_SamplingTimeTe_u8
		{
		dp_tclDimmingDpPDDDAMPING_SAMPLING_TIMETE _oDpValue;
		_s32ReturnValue = _oDpValue.s32SetData(m_oKDSConfigData.DAMPING_SAMPLING_TIMETE.DAMPING_SAMPLING_TIMETE);
		ETG_TRACE_USR1(("vWritePddData: DAMPING SAMPLING TIME  set with result %d ", _s32ReturnValue ));
		}

		//Damping_ConstantK_u8
		{
		dp_tclDimmingDpPDDDAMPING_CONSTANTK _oDpValue;
		_s32ReturnValue = _oDpValue.s32SetData(m_oKDSConfigData.DAMPING_CONSTANTK.DAMPING_CONSTANTK);
		ETG_TRACE_USR1(("vWritePddData: DAMPING CONSTANT K  set with result %d ", _s32ReturnValue ));
		}

		// Damping_ConstantJ_u8
		{
		dp_tclDimmingDpPDDDAMPING_CONSTANTJ _oDpValue;
		_s32ReturnValue = _oDpValue.s32SetData(m_oKDSConfigData.DAMPING_CONSTANTJ.DAMPING_CONSTANTJ);
		ETG_TRACE_USR1(("vWritePddData: DAMPING CONSTANT J  set with result %d ", _s32ReturnValue ));

		}

		//BrightnessBacklightDayButtonsDay_u16
		{
		dp_tclDimmingDpPDDBRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY _oDpValue;
		_s32ReturnValue = _oDpValue.s32SetData(m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY);
		ETG_TRACE_USR1(("vWritePddData: Brightness Backlight Day Buttons Day  set with result %d ", _s32ReturnValue ));

		}

		//BrightnessBacklightDayButtonsNight_u16
		{
		dp_tclDimmingDpPDDBRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT _oDpValue;
		_s32ReturnValue = _oDpValue.s32SetData(m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT);
		ETG_TRACE_USR1(("vWritePddData: Brightness Backlight Day Buttons Night  set with result %d ", _s32ReturnValue ));

		}

		//BrightnessBacklightNightButtonsNight_u16
		{
		dp_tclDimmingDpPDDBRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT _oDpValue;
		_s32ReturnValue = _oDpValue.s32SetData(m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT.BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT);
		ETG_TRACE_USR1(("vWritePddData: Brightness Backlight Night Buttons Night  set with result %d ", _s32ReturnValue ));

		}

		// BrightnessBacklightDayButtonsON_u16
		{
		dp_tclDimmingDpPDDBRIGTHNESSBACKLIGHT_DAYBUTTONS_ON _oDpValue;
		_s32ReturnValue = _oDpValue.s32SetData(m_oKDSConfigData.BRIGTHNESSBACKLIGHT_DAYBUTTONS_ON.BRIGTHNESSBACKLIGHT_DAYBUTTONS_ON);
		ETG_TRACE_USR1(("vWritePddData: Brightness Backlight Day Buttons ON  set with result %d ", _s32ReturnValue ));

		}

		//BrightnessBacklightDayButtonsOFF_u16
		{
		dp_tclDimmingDpPDDBRIGTHNESS_BACKLIGHT_DAYBUTTONS_OFF _oDpValue;
		_s32ReturnValue = _oDpValue.s32SetData(m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_DAYBUTTONS_OFF.BRIGTHNESS_BACKLIGHT_DAYBUTTONS_OFF);
		ETG_TRACE_USR1(("vWritePddData: Brightness Backlight Day Buttons OFF  set with result %d ", _s32ReturnValue ));

		}
		
		// BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS
		{
			dp_tclDimmingDpPDDBRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON _oDpValue;
			_s32ReturnValue = _oDpValue.s32SetData(m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON,
					(sizeof(m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON)/sizeof(m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON[0]) ));
			ETG_TRACE_USR1(("vWritePddData: BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS set with result %d ", _s32ReturnValue ));
		}


		// DAY_NIGHT_DELAY
		{
			dp_tclDimmingDpPDDDAY_NIGHT_DELAY _oDpValue;
			_s32ReturnValue = _oDpValue.s32SetData(m_oKDSConfigData.DAY_NIGHT_DELAY.DAY_NIGHT_DELAY);
			ETG_TRACE_USR1(("vWritePddData: DAY_NIGHT_DELAY  set with result %d ", _s32ReturnValue ));
		}

		// BRANDNAME
		{
			tU8 V850BrandType = 0x01;	//Set Nissan as default
			/*Set to 0x01 -> Nissan for BrandType--->	0x01	Nissan
														0x02	Infiniti
														0x03	EV(Nissan)

			Set to 0x00   -> Renault for BrandType->	0x00	Renault
														0x04	Dacia
														0x05	Daimler
														0x06	Avtovaz
														0x07	RSM
			*/
			
			dp_tclDimmingDpPDDBRANDNAME _oDpValue;

			if ( NISSAN_TYPE == CHECK_VARIANT_NISSAN(m_oKDSConfigData.BRANDNAME.BRANDNAME) )
			{
				ETG_TRACE_USR1(("vWritePddData: BRANDTYPE(%d): Nissan(0x01) will be send to V850",m_oKDSConfigData.BRANDNAME.BRANDNAME));
				V850BrandType = NISSAN_TYPE;
			}
			else if ( RENAULT_TYPE == CHECK_VARIANT_RENAULT(m_oKDSConfigData.BRANDNAME.BRANDNAME) )
			{
				ETG_TRACE_USR1(("vWritePddData: BRANDTYPE(%d): Renault(0x00) will be send to V850",m_oKDSConfigData.BRANDNAME.BRANDNAME));
				V850BrandType = RENAULT_TYPE;
			}
			else
			{
				ETG_TRACE_USR1(("vWritePddData: BRANDTYPE(%d): Invalid BrandType",m_oKDSConfigData.BRANDNAME.BRANDNAME));
			}

			_s32ReturnValue = _oDpValue.s32SetData(V850BrandType);
			ETG_TRACE_ERRMEM(( "vWritePddData: BRANDNAME set with result %d, (BRANDNAME:%d)",_s32ReturnValue,V850BrandType ));
		}

		//DIMMING_ARCHITECTURE
		{
		dp_tclDimmingDpPDDDIMMING_ARCHITECTURE _oDpValue;
		//Read the new PD Config item when available then do as below:TODO
		_oDpValue.s32SetData(m_oKDSConfigData.DIMMING_ARCHITECTURE.DIMMING_ARCHITECTURE);
							//0 - > Old Implementation
							//6->   P42M device with 9 key illumination values
		ETG_TRACE_USR1(("vWritePddData: DIMMING_ARCHITECTURE  set with result %d ", _s32ReturnValue ));
		}
		// trigger storing of PDD dimming datapool
		dp_tclSrvIf _oDpSrvIf;
		tU32 _u32DpID = DP_U32_POOL_ID_DIMMINGDPPDD;
		tS32 _s32ErrorCode = _oDpSrvIf.s32StorePoolNow(_u32DpID);
		if (DP_S32_NO_ERR != _s32ErrorCode)
		{
		   ETG_TRACE_FATAL(("vWritePddData: storing of PDD 0x%08x failed with error code 0x%08x!", _u32DpID, _s32ErrorCode ));
		}
	}
	else
	{
		ETG_TRACE_USR1(("vWritePddData: unchanged"));
	}
	return TRUE;
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfoEarlyConfig() const
*
* DESCRIPTION: Trace information EarlyConfig
*
* PARAMETER:   const char* strIdentifier
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid dispvidctrl_tclConfigPdd::vTraceInfoConfigPDD(const char* strIdentifier) const
{
    ETG_TRACE_USR4(("vTraceInfoConfigPDD() entered."));
	char _strAttrName[60] = {0};
	ETG_TRACE_FATAL(("%26s: <<< PDD DIMMING CONFIG >>> ", strIdentifier ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "      KDS_DIM_LOADED",
			ETG_ENUM(DISPVIDCTRL_Bool, m_bKDSLoaded) ));
	ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "       KDS_DIM_VALID",
			ETG_ENUM(DISPVIDCTRL_Bool, bIsValidKDSData()) ));

	ETG_TRACE_FATAL(("%26s: <<< Version >>> ", strIdentifier ));
	{
		// KDS value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "   KDS_VERSION_MAJOR",
				m_u8MajorVersion ));
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "   KDS_VERSION_MINOR",
				m_u8MinorVersion ));
	}
	ETG_TRACE_FATAL(("%26s: <<< BRIGTHNESS_CURVE_HMI_NIGHT >>> ", strIdentifier ));
	{
		dp_tclDimmingDpPDDBRIGTHNESS_CURVE_HMI_NIGHT _oDpValue;
		tU16 _au16BRIGTHNESS_CURVE_HMI_NIGHT[DATAPOOL_SCC_BRIGTHNESS_CURVE_HMI_NIGHT_NUMBER] = {0};
		(tVoid)_oDpValue.s32GetData(_au16BRIGTHNESS_CURVE_HMI_NIGHT,
				(sizeof(_au16BRIGTHNESS_CURVE_HMI_NIGHT)/sizeof(_au16BRIGTHNESS_CURVE_HMI_NIGHT[0])));
		tBool _bChanged = (OSAL_NULL != OSAL_s32MemoryCompare(_au16BRIGTHNESS_CURVE_HMI_NIGHT,
				m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_NIGHT.BRIGTHNESS_CURVE_HMI_NIGHT,
				sizeof(m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_NIGHT.BRIGTHNESS_CURVE_HMI_NIGHT)));
		ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "BCURVE_NIGHT_CHANGED",
				ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
		for (tU16 _u16Idx = 0; DATAPOOL_SCC_BRIGTHNESS_CURVE_HMI_NIGHT_NUMBER > _u16Idx; ++_u16Idx)
		{
			// KDS value
			(tVoid)memset(_strAttrName, OSAL_NULL, sizeof(_strAttrName));
			sprintf(_strAttrName, " KDS_BCURVE_%02u_NIGHT", _u16Idx);
			ETG_TRACE_FATAL(("%26s: %30s=%d [percent] ", strIdentifier, _strAttrName,
					m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_NIGHT.BRIGTHNESS_CURVE_HMI_NIGHT[_u16Idx] ));
			// PDD value
			(tVoid)memset(_strAttrName, OSAL_NULL, sizeof(_strAttrName));
			sprintf(_strAttrName, " PDD_BCURVE_%02u_NIGHT", _u16Idx);
			ETG_TRACE_FATAL(("%26s: %30s=%d [percent] ", strIdentifier, _strAttrName,
					_au16BRIGTHNESS_CURVE_HMI_NIGHT[_u16Idx] ));
		}
	}
	ETG_TRACE_FATAL(("%26s: <<< BRIGTHNESS_CURVE_HMI_DAY >>> ", strIdentifier ));
	{
		dp_tclDimmingDpPDDBRIGTHNESS_CURVE_HMI_DAY _oDpValue;
		tU16 _au16BRIGTHNESS_CURVE_HMI_DAY[DATAPOOL_SCC_BRIGTHNESS_CURVE_HMI_DAY_NUMBER] = {0};
		(tVoid)_oDpValue.s32GetData(_au16BRIGTHNESS_CURVE_HMI_DAY,
				(sizeof(_au16BRIGTHNESS_CURVE_HMI_DAY)/sizeof(_au16BRIGTHNESS_CURVE_HMI_DAY[0])));
		tBool _bChanged = (OSAL_NULL != OSAL_s32MemoryCompare(_au16BRIGTHNESS_CURVE_HMI_DAY,
				m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_DAY.BRIGTHNESS_CURVE_HMI_DAY,
				sizeof(m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_DAY.BRIGTHNESS_CURVE_HMI_DAY)));
		ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "  BCURVE_DAY_CHANGED",
				ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
		for (tU16 _u16Idx = 0; DATAPOOL_SCC_BRIGTHNESS_CURVE_HMI_DAY_NUMBER > _u16Idx; ++_u16Idx)
		{
			// KDS value
			(tVoid)memset(_strAttrName, OSAL_NULL, sizeof(_strAttrName));
			sprintf(_strAttrName, "   KDS_BCURVE_%02u_DAY", _u16Idx);
			ETG_TRACE_FATAL(("%26s: %30s=%d [percent] ", strIdentifier, _strAttrName,
					m_oKDSConfigData.BRIGTHNESS_CURVE_HMI_DAY.BRIGTHNESS_CURVE_HMI_DAY[_u16Idx] ));
			// PDD value
			(tVoid)memset(_strAttrName, OSAL_NULL, sizeof(_strAttrName));
			sprintf(_strAttrName, "   PDD_BCURVE_%02u_DAY", _u16Idx);
			ETG_TRACE_FATAL(("%26s: %30s=%d [percent] ", strIdentifier, _strAttrName,
					_au16BRIGTHNESS_CURVE_HMI_DAY[_u16Idx] ));
		}
	}
	ETG_TRACE_FATAL(("%26s: <<< DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY >>> ", strIdentifier ));
	{
		dp_tclDimmingDpPDDDISPLAY_ILLUMINATION_OUTPUT_FREQUENCY _oDpValue;
		tBool _bChanged = (_oDpValue.tGetData() !=
				m_oKDSConfigData.DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY.DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY );
		ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "DISP_PWMOUT_FREQ_CHA",
				ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
		// KDS value
		ETG_TRACE_FATAL(("%26s: %30s=%d [Hz] ", strIdentifier, "KDS_DISP_PWMOUT_FREQ",
				m_oKDSConfigData.DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY.DISPLAY_ILLUMINATION_OUTPUT_FREQUENCY ));
		// PDD value
		ETG_TRACE_FATAL(("%26s: %30s=%d [Hz] ", strIdentifier, "PDD_DISP_PWMOUT_FREQ",
				_oDpValue.tGetData() ));
	}
	ETG_TRACE_FATAL(("%26s: <<< KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY >>> ", strIdentifier ));
	{
		dp_tclDimmingDpPDDKEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY _oDpValue;
		tBool _bChanged = (_oDpValue.tGetData() !=
				m_oKDSConfigData.KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY.KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY );
		ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "KEYB_PWMOUT_FREQ_CHA",
				ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
		// KDS value
		ETG_TRACE_FATAL(("%26s: %30s=%d [Hz] ", strIdentifier, "KDS_KEYB_PWMOUT_FREQ",
				m_oKDSConfigData.KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY.KEYS_ENCODER_ILLUMINATION_OUTPUT_FREQUENCY ));
		// PDD value
		ETG_TRACE_FATAL(("%26s: %30s=%d [Hz] ", strIdentifier, "PDD_KEYB_PWMOUT_FREQ",
				_oDpValue.tGetData() ));
	}
	ETG_TRACE_FATAL(("%26s: <<< BRIGTHNESS_NO_OF_HMI_OFFSET >>> ", strIdentifier ));
	{
		dp_tclDimmingDpPDDBRIGTHNESS_NO_OF_HMI_OFFSET _oDpValue;
		tBool _bChanged = (_oDpValue.tGetData() !=
				m_oKDSConfigData.BRIGTHNESS_NO_OF_HMI_OFFSET.BRIGTHNESS_NO_OF_HMI_OFFSET );
		ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "NO_OF_HMI_OFFSET_CHA",
				ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
		// KDS value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "KDS_NO_OF_HMI_OFFSET",
				m_oKDSConfigData.BRIGTHNESS_NO_OF_HMI_OFFSET.BRIGTHNESS_NO_OF_HMI_OFFSET ));
		// PDD value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "PDD_NO_OF_HMI_OFFSET",
				_oDpValue.tGetData() ));
	}
	ETG_TRACE_FATAL(("%26s: <<< NUMBER_OF_KEYS >>> ", strIdentifier ));
	{
		dp_tclDimmingDpPDDNUMBER_OF_KEYS _oDpValue;
		tBool _bChanged = (_oDpValue.tGetData() !=
				m_oKDSConfigData.NUMBER_OF_KEYS.NUMBER_OF_KEYS );
		ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "  NO_OF_KEYS_CHANGED",
				ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
		// KDS value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      KDS_NO_OF_KEYS",
				m_oKDSConfigData.NUMBER_OF_KEYS.NUMBER_OF_KEYS ));
		// PDD value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      PDD_NO_OF_KEYS",
				_oDpValue.tGetData() ));
	}
	ETG_TRACE_FATAL(("%26s: <<< HIGH_TEMPERATURE >>> ", strIdentifier ));
	{
		dp_tclDimmingDpPDDHIGH_TEMPERATURE _oDpValue;
		tBool _bChanged = (_oDpValue.tGetData() !=
				m_oKDSConfigData.HIGH_TEMPERATURE.HIGH_TEMPERATURE );
		ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "   HIGH_TEMP_CHANGED",
				ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
		// KDS value
		ETG_TRACE_FATAL(("%26s: %30s=%d [C] ", strIdentifier, "       KDS_HIGH_TEMP",
				m_oKDSConfigData.HIGH_TEMPERATURE.HIGH_TEMPERATURE ));
		// PDD value
		ETG_TRACE_FATAL(("%26s: %30s=%d [C] ", strIdentifier, "       PDD_HIGH_TEMP",
				_oDpValue.tGetData() ));
	}
	ETG_TRACE_FATAL(("%26s: <<< CRITICAL_TEMPERATURE >>> ", strIdentifier ));
	{
		dp_tclDimmingDpPDDCRITICAL_TEMPERATURE _oDpValue;
		tBool _bChanged = (_oDpValue.tGetData() !=
				m_oKDSConfigData.CRITICAL_TEMPERATURE.CRITICAL_TEMPERATURE );
		ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "CRITHIGH_TEMP_CHANGE",
				ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
		// KDS value
		ETG_TRACE_FATAL(("%26s: %30s=%d [C] ", strIdentifier, "   KDS_CRITHIGH_TEMP",
				m_oKDSConfigData.CRITICAL_TEMPERATURE.CRITICAL_TEMPERATURE ));
		// PDD value
		ETG_TRACE_FATAL(("%26s: %30s=%d [C] ", strIdentifier, "   PDD_CRITHIGH_TEMP",
				_oDpValue.tGetData() ));
	}
	ETG_TRACE_FATAL(("%26s: <<< LOW_TEMPERATURE >>> ", strIdentifier ));
	{
		dp_tclDimmingDpPDDLOW_TEMPERATURE _oDpValue;
		tBool _bChanged = (_oDpValue.tGetData() !=
				m_oKDSConfigData.LOW_TEMPERATURE.LOW_TEMPERATURE );
		ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "     LOW_TEMP_CHANGE",
				ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
		// KDS value
		ETG_TRACE_FATAL(("%26s: %30s=%d [C] ", strIdentifier, "        KDS_LOW_TEMP",
				m_oKDSConfigData.LOW_TEMPERATURE.LOW_TEMPERATURE ));
		// PDD value
		ETG_TRACE_FATAL(("%26s: %30s=%d [C] ", strIdentifier, "        PDD_LOW_TEMP",
				_oDpValue.tGetData() ));
	}
	ETG_TRACE_FATAL(("%26s: <<< TEMP_HYSTERESIS >>> ", strIdentifier ));
	{
		dp_tclDimmingDpPDDTEMP_HYSTERESIS _oDpValue;
		tBool _bChanged = (_oDpValue.tGetData() !=
				m_oKDSConfigData.TEMP_HYSTERESIS.TEMP_HYSTERESIS );
		ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "    TEMP_HYST_CHANGE",
				ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
		// KDS value
		ETG_TRACE_FATAL(("%26s: %30s=%d [C] ", strIdentifier, "       KDS_TEMP_HYST",
				m_oKDSConfigData.TEMP_HYSTERESIS.TEMP_HYSTERESIS ));
		// PDD value
		ETG_TRACE_FATAL(("%26s: %30s=%d [C] ", strIdentifier, "       PDD_TEMP_HYST",
				_oDpValue.tGetData() ));
	}
	ETG_TRACE_FATAL(("%26s: <<< TEMP_COMPENSATION_PWM_MIN >>> ", strIdentifier ));
	{
		dp_tclDimmingDpPDDTEMP_COMPENSATION_PWM_MIN _oDpValue;
		tBool _bChanged = (_oDpValue.tGetData() !=
				m_oKDSConfigData.TEMP_COMPENSATION_PWM_MIN.TEMP_COMPENSATION_PWM_MIN );
		ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "TEMP_COMP_PWMMIN_CHA",
				ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
		// KDS value
		ETG_TRACE_FATAL(("%26s: %30s=%d [percent] ", strIdentifier, "KDS_TEMP_COMP_PWMMIN",
				m_oKDSConfigData.TEMP_COMPENSATION_PWM_MIN.TEMP_COMPENSATION_PWM_MIN ));
		// PDD value
		ETG_TRACE_FATAL(("%26s: %30s=%d [percent] ", strIdentifier, "PDD_TEMP_COMP_PWMMIN",
				_oDpValue.tGetData() ));
	}
	ETG_TRACE_FATAL(("%26s: <<< KEYB_LED_CALIBRATION_VALUE >>> ", strIdentifier ));
	{
		dp_tclDimmingDpPDDKEYB_LED_CALIBRATION_VALUE _oDpValue;
		tU16 _au16KEYB_LED_CALIBRATION_VALUE[DATAPOOL_SCC_KEYB_LED_CALIBRATION_VALUE_NUMBER] = {0};
		(tVoid)_oDpValue.s32GetData(_au16KEYB_LED_CALIBRATION_VALUE,
				(sizeof(_au16KEYB_LED_CALIBRATION_VALUE)/sizeof(_au16KEYB_LED_CALIBRATION_VALUE[0])));
		tBool _bChanged = (OSAL_NULL != OSAL_s32MemoryCompare(_au16KEYB_LED_CALIBRATION_VALUE,
				m_oKDSConfigData.KEYB_LED_CALIBRATION_VALUE.KEYB_LED_CALIBRATION_VALUE,
				sizeof(m_oKDSConfigData.KEYB_LED_CALIBRATION_VALUE.KEYB_LED_CALIBRATION_VALUE)));
		ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "KEYLED_CALVAL_CHANGE",
				ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
		ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "KEYLED_CALVAL_USEDEF",
				ETG_ENUM(DISPVIDCTRL_Bool, m_bUseDefaultCalibrationValues) ));
		for (tU16 _u16Idx = 0; DATAPOOL_SCC_KEYB_LED_CALIBRATION_VALUE_NUMBER > _u16Idx; ++_u16Idx)
		{
			// KDS value
			(tVoid)memset(_strAttrName, OSAL_NULL, sizeof(_strAttrName));
			sprintf(_strAttrName, "KDS_KEYLED_CALVAL_%02u", _u16Idx);
			ETG_TRACE_FATAL(("%26s: %30s=%d [percent] ", strIdentifier, _strAttrName,
					m_oKDSConfigData.KEYB_LED_CALIBRATION_VALUE.KEYB_LED_CALIBRATION_VALUE[_u16Idx] ));
			// PDD value
			(tVoid)memset(_strAttrName, OSAL_NULL, sizeof(_strAttrName));
			sprintf(_strAttrName, "PDD_KEYLED_CALVAL_%02u", _u16Idx);
			ETG_TRACE_FATAL(("%26s: %30s=%d [percent] ", strIdentifier, _strAttrName,
					_au16KEYB_LED_CALIBRATION_VALUE[_u16Idx] ));
		}
	}

	ETG_TRACE_FATAL(("%26s: <<< DISPLAY_TYPE >>> ", strIdentifier ));
	{
		dp_tclDimmingDpPDDDISPLAY_TYPE _oDpValue;
		tBool _bChanged = (_oDpValue.tGetData() !=
				m_oKDSConfigData.DISPLAY_TYPE.DISPLAY_TYPE );
		ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "  DISPLAY_TYPE",
				ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
		// KDS value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      KDS_DISPLAY_TYPE",
				m_oKDSConfigData.DISPLAY_TYPE.DISPLAY_TYPE ));
		// PDD value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      PDD_DISPLAY_TYPE",
				_oDpValue.tGetData() ));
	}

	ETG_TRACE_FATAL(("%26s: <<< DIMMING_COCKPIT_STRATEGY >>> ", strIdentifier ));
	{
		dp_tclDimmingDpPDDDIMMING_COCKPIT_STRATEGY _oDpValue;
		tBool _bChanged = (_oDpValue.tGetData() !=
				m_oKDSConfigData.DIMMING_COCKPIT_STRATEGY.DIMMING_COCKPIT_STRATEGY );
		ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "  DISPLAY_TYPE",
				ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
		// KDS value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      KDS_DISPLAY_TYPE",
				m_oKDSConfigData.DIMMING_COCKPIT_STRATEGY.DIMMING_COCKPIT_STRATEGY ));
		// PDD value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      PDD_DISPLAY_TYPE",
				_oDpValue.tGetData() ));
	}

	ETG_TRACE_FATAL(("%26s: <<< KEY_BRIGHTNESS_NUMOF_OFFSETS >>> ", strIdentifier ));
	{
		dp_tclDimmingDpPDDKEY_BRIGHTNESS_NUMOF_OFFSETS _oDpValue;
		tBool _bChanged = (_oDpValue.tGetData() !=
				m_oKDSConfigData.KEY_BRIGHTNESS_NUMOF_OFFSETS.KEY_BRIGHTNESS_NUMOF_OFFSETS );
		ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "  DISPLAY_TYPE",
				ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
		// KDS value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      KDS_KEY_BRIGHTNESS_NUMOF_OFFSETSE",
				m_oKDSConfigData.KEY_BRIGHTNESS_NUMOF_OFFSETS.KEY_BRIGHTNESS_NUMOF_OFFSETS ));
		// PDD value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      PDD_KEY_BRIGHTNESS_NUMOF_OFFSETS",
				_oDpValue.tGetData() ));
	}

	ETG_TRACE_FATAL(("%26s: <<< KEY_BRIGHTNESS_NUMOF_OFFSETS >>> ", strIdentifier ));
	{
		dp_tclDimmingDpPDDKEY_BRIGHTNESS_NUMOF_OFFSETS _oDpValue;
		tBool _bChanged = (_oDpValue.tGetData() !=
				m_oKDSConfigData.KEY_BRIGHTNESS_NUMOF_OFFSETS.KEY_BRIGHTNESS_NUMOF_OFFSETS );
		ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "  DISPLAY_TYPE",
				ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
		// KDS value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      KDS_KEY_BRIGHTNESS_NUMOF_OFFSETSE",
				m_oKDSConfigData.KEY_BRIGHTNESS_NUMOF_OFFSETS.KEY_BRIGHTNESS_NUMOF_OFFSETS ));
		// PDD value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      PDD_KEY_BRIGHTNESS_NUMOF_OFFSETS",
				_oDpValue.tGetData() ));
	}

	ETG_TRACE_FATAL(("%26s: <<< DAMPING_STRATEGY >>> ", strIdentifier ));
	{
		dp_tclDimmingDpPDDDAMPING_STRATEGY _oDpValue;
		tBool _bChanged = (_oDpValue.tGetData() !=
				m_oKDSConfigData.DAMPING_STRATEGY.DAMPING_STRATEGY );
		ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "  DISPLAY_TYPE",
				ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
		// KDS value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      KDS_DAMPING_STRATEGY",
				m_oKDSConfigData.DAMPING_STRATEGY.DAMPING_STRATEGY ));
		// PDD value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      PDD_DAMPING_STRATEGY",
				_oDpValue.tGetData() ));
	}

	ETG_TRACE_FATAL(("%26s: <<< DAMPING_SAMPLING_TIMETE >>> ", strIdentifier ));
	{
		dp_tclDimmingDpPDDDAMPING_SAMPLING_TIMETE _oDpValue;
		tBool _bChanged = (_oDpValue.tGetData() !=
				m_oKDSConfigData.DAMPING_SAMPLING_TIMETE.DAMPING_SAMPLING_TIMETE );
		ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "  DISPLAY_TYPE",
				ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
		// KDS value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      KDS_DAMPING_SAMPLING_TIMETE",
				m_oKDSConfigData.DAMPING_SAMPLING_TIMETE.DAMPING_SAMPLING_TIMETE ));
		// PDD value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      PDD_DAMPING_SAMPLING_TIMETE",
				_oDpValue.tGetData() ));
	}

	ETG_TRACE_FATAL(("%26s: <<< DAMPING_CONSTANTK >>> ", strIdentifier ));
	{
		dp_tclDimmingDpPDDDAMPING_CONSTANTK _oDpValue;
		tBool _bChanged = (_oDpValue.tGetData() !=
				m_oKDSConfigData.DAMPING_CONSTANTK.DAMPING_CONSTANTK );
		ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "  DISPLAY_TYPE",
				ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
		// KDS value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      KDS_DAMPING_CONSTANTK",
				m_oKDSConfigData.DAMPING_CONSTANTK.DAMPING_CONSTANTK ));
		// PDD value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      PDD_DAMPING_CONSTANTK",
				_oDpValue.tGetData() ));
	}

	ETG_TRACE_FATAL(("%26s: <<< DAMPING_CONSTANTJ >>> ", strIdentifier ));
	{
		dp_tclDimmingDpPDDDAMPING_CONSTANTJ _oDpValue;
		tBool _bChanged = (_oDpValue.tGetData() !=
				m_oKDSConfigData.DAMPING_CONSTANTJ.DAMPING_CONSTANTJ );
		ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "  DAMPING_CONSTANTJ",
				ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
		// KDS value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      KDS_DAMPING_CONSTANTJ",
				m_oKDSConfigData.DAMPING_CONSTANTJ.DAMPING_CONSTANTJ ));
		// PDD value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      PDD_DAMPING_CONSTANTJ",
				_oDpValue.tGetData() ));
	}

	ETG_TRACE_FATAL(("%26s: <<< BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY >>> ", strIdentifier ));
	{
		dp_tclDimmingDpPDDBRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY _oDpValue;
		tBool _bChanged = (_oDpValue.tGetData() !=
				m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY );
		ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "  BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY",
				ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
		// KDS value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      KDS_BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY",
				m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY ));
		// PDD value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      PDD_BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_DAY",
				_oDpValue.tGetData() ));
	}

	ETG_TRACE_FATAL(("%26s: <<< BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT >>> ", strIdentifier ));
	{
		dp_tclDimmingDpPDDBRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT _oDpValue;
		tBool _bChanged = (_oDpValue.tGetData() !=
				m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT );
		ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "  BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT",
				ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
		// KDS value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      KDS_BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT",
				m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT.BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT ));
		// PDD value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      PDD_BRIGTHNESS_BACKLIGHT_DAY_BUTTONS_NIGHT",
				_oDpValue.tGetData() ));
	}

	ETG_TRACE_FATAL(("%26s: <<< BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT >>> ", strIdentifier ));
	{
		dp_tclDimmingDpPDDBRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT _oDpValue;
		tBool _bChanged = (_oDpValue.tGetData() !=
				m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT.BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT );
		ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "  BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT",
				ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
		// KDS value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT",
				m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT.BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT ));
		// PDD value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      PDD_BRIGTHNESS_BACKLIGHT_NIGHT_BUTTONS_NIGHT",
				_oDpValue.tGetData() ));
	}

	ETG_TRACE_FATAL(("%26s: <<< BRIGTHNESSBACKLIGHT_DAYBUTTONS_ON >>> ", strIdentifier ));
	{
		dp_tclDimmingDpPDDBRIGTHNESSBACKLIGHT_DAYBUTTONS_ON _oDpValue;
		tBool _bChanged = (_oDpValue.tGetData() !=
				m_oKDSConfigData.BRIGTHNESSBACKLIGHT_DAYBUTTONS_ON.BRIGTHNESSBACKLIGHT_DAYBUTTONS_ON );
		ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "  BRIGTHNESSBACKLIGHT_DAYBUTTONS_ON",
				ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
		// KDS value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      BRIGTHNESSBACKLIGHT_DAYBUTTONS_ON",
				m_oKDSConfigData.BRIGTHNESSBACKLIGHT_DAYBUTTONS_ON.BRIGTHNESSBACKLIGHT_DAYBUTTONS_ON ));
		// PDD value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      PDD_BRIGTHNESSBACKLIGHT_DAYBUTTONS_ON",
				_oDpValue.tGetData() ));
	}


	ETG_TRACE_FATAL(("%26s: <<< BRIGTHNESS_BACKLIGHT_DAYBUTTONS_OFF >>> ", strIdentifier ));
	{
		dp_tclDimmingDpPDDBRIGTHNESS_BACKLIGHT_DAYBUTTONS_OFF _oDpValue;
		tBool _bChanged = (_oDpValue.tGetData() !=
				m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_DAYBUTTONS_OFF.BRIGTHNESS_BACKLIGHT_DAYBUTTONS_OFF );
		ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "  BRIGTHNESS_BACKLIGHT_DAYBUTTONS_OFF",
				ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
		// KDS value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      KDS_BRIGTHNESS_BACKLIGHT_DAYBUTTONS_OFF",
				m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_DAYBUTTONS_OFF.BRIGTHNESS_BACKLIGHT_DAYBUTTONS_OFF ));
		// PDD value
		ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      PDD_BRIGTHNESS_BACKLIGHT_DAYBUTTONS_OFF",
				_oDpValue.tGetData() ));

	}

	// brightness Btutton night On
	ETG_TRACE_FATAL(("%26s: <<< BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON >>> ", strIdentifier ));
		{
		dp_tclDimmingDpPDDBRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON _oDpValue;
			tU16 _au16BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON[DATAPOOL_SCC_BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON_NUMBER] = {0};
			(tVoid)_oDpValue.s32GetData(_au16BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON,
					(sizeof(_au16BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON)/sizeof(_au16BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON[0])));
			tBool _bChanged = (OSAL_NULL != OSAL_s32MemoryCompare(_au16BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON,
					m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON,
					sizeof(m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON)));
			ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "B_NIGHT_BUTTONS",
					ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
			ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "B_NIGHT_BUTTONS",
					ETG_ENUM(DISPVIDCTRL_Bool, m_bUseDefaultCalibrationValues) ));
			for (tU16 _u16Idx = 0; DATAPOOL_SCC_BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON_NUMBER > _u16Idx; ++_u16Idx)
			{
				// KDS value
				(tVoid)memset(_strAttrName, OSAL_NULL, sizeof(_strAttrName));
				sprintf(_strAttrName, "B_NIGHT_BUTTONS_%02u", _u16Idx);
				ETG_TRACE_FATAL(("%26s: %30s=%d [percent] ", strIdentifier, _strAttrName,
						m_oKDSConfigData.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON.BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON[_u16Idx] ));
				// PDD value
				(tVoid)memset(_strAttrName, OSAL_NULL, sizeof(_strAttrName));
				sprintf(_strAttrName, "B_NIGHT_BUTTONS_%02u", _u16Idx);
				ETG_TRACE_FATAL(("%26s: %30s=%d [percent] ", strIdentifier, _strAttrName,
						_au16BRIGTHNESS_BACKLIGHT_NIGHTBUTTONS_ON[_u16Idx] ));
			}
		}

		ETG_TRACE_FATAL(("%26s: <<< DAY_NIGHT_DELAY >>> ", strIdentifier ));
		{
			dp_tclDimmingDpPDDDAY_NIGHT_DELAY _oDpValue;
			tBool _bChanged = (_oDpValue.tGetData() !=
					m_oKDSConfigData.DAY_NIGHT_DELAY.DAY_NIGHT_DELAY );
			ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "  DAY_NIGHT_DELAY",
					ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
			// KDS value
			ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      KDS_DAY_NIGHT_DELAY",
					m_oKDSConfigData.DAY_NIGHT_DELAY.DAY_NIGHT_DELAY ));
			// PDD value
			ETG_TRACE_FATAL(("%26s: %30s=%d ", strIdentifier, "      PDD_DAY_NIGHT_DELAY",
					_oDpValue.tGetData() ));
		}


		ETG_TRACE_FATAL(("%26s: <<< BRANDNAME >>> ", strIdentifier ));
		{
			//dp_tclDimmingDpPDDRESERVED _oDpValue;
			dp_tclDimmingDpPDDBRANDNAME _oDpValue;
			tBool _bChanged = (_oDpValue.tGetData() !=
					m_oKDSConfigData.BRANDNAME.BRANDNAME );
			ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "    BRANDNAME_CHANGED",
					ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));
			// KDS value
			tU8 u8BrandType = 0x01; // 0x01 means Nissan, default initialize to Nissan
			//Now Reading Brand Type from KDS
		    /*AllianceBrandType 3 Bit
		    * [0x00] Renault
		    * [0x01] Nissan
		    * [0x02] Infiniti*/
			if ( DP_S32_NO_ERR == DP_s32GetConfigItem("SystemConfiguration1", "Brand", &u8BrandType, 1) )
			{
				ETG_TRACE_USR1((" Brand Type is ..  0x%02x.", u8BrandType));
			}
			else
			{
				ETG_TRACE_ERR(("u8BrandType(): Read has failed !!! "));
			}
		    //Reading Brand Time Finished..
			ETG_TRACE_FATAL(("m_oKDSConfigData.BRANDNAME.BRANDNAME= %d",m_oKDSConfigData.BRANDNAME.BRANDNAME));

			ETG_TRACE_FATAL(("%26s: %30s=%d  ", strIdentifier, "        KDS_BRANDNAME",
					u8BrandType));
			// PDD value
			ETG_TRACE_FATAL(("%26s: %30s=%d  ", strIdentifier, "        PDD_BRANDNAME",
					_oDpValue.tGetData() ));
		}

		ETG_TRACE_FATAL(("%26s: <<< DIMMING_ARCHITECTURE >>> ", strIdentifier ));
				{
					//dp_tclDimmingDpPDDRESERVED _oDpValue;
					dp_tclDimmingDpPDDDIMMING_ARCHITECTURE _oDpValue;
					tBool _bChanged = (_oDpValue.tGetData() !=
							m_oKDSConfigData.DIMMING_ARCHITECTURE.DIMMING_ARCHITECTURE );
					ETG_TRACE_FATAL(("%26s: %30s=%u ", strIdentifier, "    DIMMING_ARCHITECTURE_CHANGED",
							ETG_ENUM(DISPVIDCTRL_Bool, _bChanged) ));


					ETG_TRACE_FATAL(("%26s: %30s=%d  ", strIdentifier, "        KDS_DIMMING_ARCHITECTURE",
							m_oKDSConfigData.DIMMING_ARCHITECTURE.DIMMING_ARCHITECTURE));
					// PDD value
					ETG_TRACE_FATAL(("%26s: %30s=%d  ", strIdentifier, "        PDD_DIMMING_ARCHITECTURE",
							_oDpValue.tGetData() ));
				}
}
