/************************************************************************
* FILE:        dispvidctrl_tclBacklightError.cpp
* PROJECT:
* SW-COMPONENT:vd_dimming
*----------------------------------------------------------------------
*
* DESCRIPTION: implementation of back light error detection class
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 17.10.2016| CM-AI/EPB2 Bernard | init
*
*************************************************************************/

/*******************************************************************************
| includes: system- and project- includes
|-----------------------------------------------------------------------------*/
// Basic OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>

/*******************************************************************************
| includes: needed interfaces from external components
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| includes: internal and external interfaces from this component
|-----------------------------------------------------------------------------*/

#include "dispvidctrl_tclBacklightError.h"
#include "Idispvidctrl_ClientDimming.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_APPLICATION
#include "trcGenProj/Header/dispvidctrl_tclBacklightError.cpp.trc.h"
#endif

/*******************************************************************************
| defines and macros (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| typedefs (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function prototype (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function implementation (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
*
* FUNCTION:    dispvidctrl_tclBacklightError()
*
* DESCRIPTION: constructor
*
* PARAMETER:   const dispvidctrl_tclAppMain*
*
* RETURNVALUE: none
*
*******************************************************************************/
dispvidctrl_tclBacklightError::dispvidctrl_tclBacklightError(const dispvidctrl_tclAppMain* poMainAppl)
: I_dispvidctrl_tclBacklightError(poMainAppl)
{
   ETG_TRACE_USR4(("dispvidctrl_tclBacklightError() entered."));

   m_hTimer = OSAL_NULL;
   (tVoid)memset((tVoid*)m_tErrorInfo,OSAL_NULL,sizeof(m_tErrorInfo));
   m_bDisplayActive = FALSE;
   m_bStateNotHandled = FALSE;
}

/*******************************************************************************
*
* FUNCTION:    ~dispvidctrl_tclBacklightError()
*
* DESCRIPTION: destructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*******************************************************************************/
dispvidctrl_tclBacklightError::~dispvidctrl_tclBacklightError()
{
   ETG_TRACE_USR4(("~dispvidctrl_tclBacklightError() entered."));

   if (OSAL_NULL != m_hTimer)
   {
	   (tVoid)OSAL_s32TimerSetTime(m_hTimer, 0, 0);
	   (tVoid)OSAL_s32TimerDelete(m_hTimer);
   }
}

/*******************************************************************************
*
* FUNCTION:    vHandleMessage()
*
* DESCRIPTION: handle internal message
*
* PARAMETER:   dispvidctrl_tclBaseIf::TMsg* pMsg
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid dispvidctrl_tclBacklightError::vHandleMessage(dispvidctrl_tclBaseIf::TMsg* pMsg)
{
      DISPVIDCTRL_NULL_POINTER_CHECK(pMsg);
      ETG_TRACE_USR4(("vHandleMessage() entered %u -> data: %d.", ETG_ENUM(HC_INT_MSG , (tU32)pMsg->eCmd), pMsg->u.u32Data));

      switch(pMsg->eCmd)
      {
      case dispvidctrl_tclBaseIf::eTriggerBerrUpdate:
		  {
			  if (OSAL_NULL != m_hTimer)
			  {
				   (tVoid)OSAL_s32TimerSetTime(m_hTimer, 0, 0);
				   vMonitor();
				   (tVoid)OSAL_s32TimerSetTime(m_hTimer, DIM_C_U32_BACKLIGHTERROR_CYCLETIME_MS, 0);
			  }
		  }
		  break;
      default:
		  {

		  }
		  break;
      }
}

/*******************************************************************************
*
* FUNCTION:    vHandleTraceMessage()
*
* DESCRIPTION:    handler for trace command
*
* PARAMETER:      const tUChar* puchData: reference of received message
*
* RETURNVALUE:    None.
*
*******************************************************************************/
tVoid dispvidctrl_tclBacklightError::vHandleTraceMessage(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceMessage() entered (data: 0x%08x).", puchData ));
   DISPVIDCTRL_NULL_POINTER_CHECK(puchData);

    tU32 u32MsgCode = (puchData[1]<<8) | puchData[2];
   switch (u32MsgCode)
   {
   case DISPVIDCTRL_DISPVIDCTRL_TRACEINFO:
      {
         // Prints the display setting information.
         ETG_TRACE_USR4(("%26s: DISPVIDCTRL_CmdTraceinfo ",
               "DIM.APPL.BERR.CMD__.START" ));
         vTraceInfo();
         ETG_TRACE_USR4(("%26s: finished with errors ",
               "DIM.APPL.BERR.CMD__.RES__" ));
      }
      break;
   case DISPVIDCTRL_DIMMING_GETBERR_STATE:
       {
           // Get the backlight error detetction status information.


           // DISPVIDCTRL_DIMMING_GETBERR_STATE: user specific code start
    	   ETG_TRACE_FATAL(("%26s: DIMMING_GetBErr_State ",
                 "DIM.APPL.BERR.CMD__.START" ));
    	   vTraceInfo();
    	   ETG_TRACE_FATAL(("%26s: finished with errors ",
                 "DIM.APPL.BERR.CMD__.RES__" ));
           // DISPVIDCTRL_DIMMING_GETBERR_STATE: user specific code end

       }
       break;
   case DISPVIDCTRL_DIMMING_SETBERR:
       {
           // Set the state of specified backlight error.
           tU8 param1 = puchData[3];
           tU8 param2 = puchData[4];


           // DISPVIDCTRL_DIMMING_SETBERR: user specific code start
           ETG_TRACE_USR4(("%26s: DIMMING_SetBErr %u %u ",
                 "DIM.APPL.BERR.CMD__.START",
                 ETG_ENUM(BERR_NO, param1),
                 ETG_ENUM(BERR_STATE, param2) ));
           if (0 > s32SetSimulateState(param1, param2))
           {
               ETG_TRACE_FATAL(("%26s: finished with errors ",
                     "DIM.APPL.BERR.CMD__.RES__" ));
           }
           else
           {
               ETG_TRACE_FATAL(("%26s: finished without errors ",
                     "DIM.APPL.BERR.CMD__.RES__" ));
           }
           // DISPVIDCTRL_DIMMING_SETBERR: user specific code end

       }
       break;

   default:
      {
	  }
      break;
   }
}

/*******************************************************************************
*
* FUNCTION:    vGetReferences(tVoid)
*
* DESCRIPTION:    Function to get all reference needed by this class.
*             A reference should always be the Interface class of the object
*
* PARAMETER:    None.
*
* RETURNVALUE:    None.
*
*******************************************************************************/
tVoid dispvidctrl_tclBacklightError::vGetReferences(tVoid)
{
   ETG_TRACE_USR4(("vGetReferences() entered."));

   // create timer to poll back light error GPIN
   tS32 _s32Return = OSAL_s32TimerCreate((OSAL_tpfCallback)cb_vTimer, (tVoid*)this, &m_hTimer);
   if (OSAL_OK != _s32Return)
   {
	   ETG_TRACE_FATAL(("vGetReferences: timer creation failed with %d!", _s32Return ));
	   m_hTimer = OSAL_NULL;
   }
   else
   {
	   (tVoid)OSAL_s32TimerSetTime(m_hTimer, 0, 0);
   }
}

/*******************************************************************************
*
* FUNCTION:    tVoid vStartCommunication()
*
* DESCRIPTION:    Function to start all dynamic objects e.g. threads, ...
*
* PARAMETER:    None.
*
* RETURNVALUE:    None.
*
*******************************************************************************/
tVoid dispvidctrl_tclBacklightError::vStartCommunication(tVoid)
{
   ETG_TRACE_USR4(("vStartCommunication() entered."));

   if (OSAL_NULL != m_hTimer)
   {
	   (tVoid)OSAL_s32TimerSetTime(m_hTimer, 0, 0);
	   vMonitor();
	   (tVoid)OSAL_s32TimerSetTime(m_hTimer, DIM_C_U32_BACKLIGHTERROR_CYCLETIME_MS, 0);
   }
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfo()
*
* DESCRIPTION:    Trace information
*
* PARAMETER:    None.
*
* RETURNVALUE:    None.
*
*******************************************************************************/
tVoid dispvidctrl_tclBacklightError::vTraceInfo()
{
   ETG_TRACE_USR4(("vTraceInfo() entered."));
   tChar _strIdentStart[30] = "DIM.APPL.BERR.DETEC.START";
   tChar _strIdentAttr[30]  = "DIM.APPL.BERR.DETEC.ATTR_";
   tChar _strIdentEnd[30]   = "DIM.APPL.BERR.DETEC.END__";
   ETG_TRACE_USR2(("%026s: ", _strIdentStart ));
   vTraceBEerr(_strIdentAttr);
   ETG_TRACE_USR2(("%026s: ", _strIdentEnd ));
}

/*******************************************************************************
*
* FUNCTION:    cb_vTimer()
*
* DESCRIPTION: timer call back
*
* PARAMETER:   tVoid* pArg
*
* RETURNVALUE: OSAL_tpfCallback
*
*******************************************************************************/
OSAL_tpfCallback dispvidctrl_tclBacklightError::cb_vTimer(tVoid* pArg)
{
	ETG_TRACE_USR4(("vTraceInfo() entered."));

	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);
	dispvidctrl_tclAppMain* _poApp = dispvidctrl_tclAppMain::theServer();
	if (OSAL_NULL != _poApp)
	{
		dispvidctrl_tclBaseIf::TMsg _oMsg;
		(tVoid)memset((tVoid*)&_oMsg,OSAL_NULL,sizeof(_oMsg));
		_oMsg.eCmd = dispvidctrl_tclBaseIf::eTriggerBerrUpdate;
		(tVoid)strncpy(_oMsg.strClassName,"I_dispvidctrl_tclDisplaySettings",sizeof(_oMsg.strClassName)-1);
		_poApp->vPostInternalMessage(&_oMsg);
	}

	return OSAL_NULL;
}

/*******************************************************************************
*
* FUNCTION:    vMonitor()
*
* DESCRIPTION: polling of back light error GPIN
*
* PARAMETER:   None.
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid dispvidctrl_tclBacklightError::vMonitor(tVoid)
{
   ETG_TRACE_USR4(("vMonitor() entered."));

   tBool _bDoUpdate = FALSE;
   dispvidctrl_tclAppMain* _poApp = dispvidctrl_tclAppMain::theServer();
   tChar strIfName[] = "I_dispvidctrl_tclClientDimming";
   I_dispvidctrl_tclClientDimming* _poIncDimClient = (OSAL_NULL != _poApp)?(dynamic_cast<I_dispvidctrl_tclClientDimming*>(_poApp->getHandler(strIfName))):OSAL_NULL;
   // get current display activation state
   if ((OSAL_NULL != _poIncDimClient) && (m_bDisplayActive != _poIncDimClient->bGetDimState()))
   {
	   m_bDisplayActive = _poIncDimClient->bGetDimState();
	   _bDoUpdate = TRUE;
   }
   for (tU16 _u16Idx = 0; DISPVIDCTRL_C_U16_ARRAYELEMENTS(m_tErrorInfo) > _u16Idx; ++_u16Idx)
   {
	   if (0 == strlen(m_tErrorInfo[_u16Idx].strPath))
	   {
		   // path not set ... find the path and read the value
		   vFindGPIN(_u16Idx);
	   }
	   else
	   {
		   // read value of back light error device
		   tenBERR_State _enState = m_tErrorInfo[_u16Idx].enState;
		   m_tErrorInfo[_u16Idx].enState = enReadGPIN(m_tErrorInfo[_u16Idx].strPath);
		   if ((enBERR_zero != m_tErrorInfo[_u16Idx].enState) && (enBERR_one != m_tErrorInfo[_u16Idx].enState))
		   {
			   ETG_TRACE_USR1(("vMonitor: invalid value %u (currently %u) read from %s!",
					   ETG_CENUM(tenBERR_State, m_tErrorInfo[_u16Idx].enState), ETG_CENUM(tenBERR_State, _enState), m_tErrorInfo[_u16Idx].strPath));
			   (tVoid)memset((tVoid*)m_tErrorInfo[_u16Idx].strPath,OSAL_NULL,sizeof(m_tErrorInfo[_u16Idx].strPath));
		   }
		   else
		   {
			   ETG_TRACE_USR4(("vMonitor: new value %u (previously %u) read from %s.",
					   ETG_CENUM(tenBERR_State, m_tErrorInfo[_u16Idx].enState), ETG_CENUM(tenBERR_State, _enState), m_tErrorInfo[_u16Idx].strPath));
			   if (_enState != m_tErrorInfo[_u16Idx].enState)
			   {
				   _bDoUpdate = TRUE;
				   m_bStateNotHandled = TRUE;
			   }
		   }
	   }
   }
   if (TRUE == _bDoUpdate || TRUE == m_bStateNotHandled)
   {
	   vUpdateDetetctionState();
   }
}

/*******************************************************************************
*
* FUNCTION:    vFindGPIN()
*
* DESCRIPTION: find back light error GPIN pat and read the value
*
* PARAMETER:   tU16 u16Idx	... index of back light error GPIN (0 ...)
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid dispvidctrl_tclBacklightError::vUpdateDetetctionState(tVoid)
{
   ETG_TRACE_USR4(("vUpdateDetetctionState() entered."));

   tChar strIfName[] = "I_dispvidctrl_tclClientDimming";
   tU32 _u32Data = (tU32) eBackLight_BERR_Auto;
   dispvidctrl_tclAppMain* _poApp = dispvidctrl_tclAppMain::theServer();
   if (OSAL_NULL != _poApp)
   {
	   if (TRUE == m_bDisplayActive)
	   {
		   for (tU16 _u16Idx = 0; (DISPVIDCTRL_C_U16_ARRAYELEMENTS(m_tErrorInfo) > _u16Idx) && ((tU32) eBackLight_BERR_Off != _u32Data); ++_u16Idx)
		   {
			   if (TRUE == bIsSimuActive((tU8)_u16Idx))
			   {
				   if (DIM_C_U8_BACKLIGHTERROR_ACTIVE == m_tErrorInfo[_u16Idx].enSimulateState)
				   {
					   _u32Data = (tU32) eBackLight_BERR_Off;
				   }
			   }
			   else if (DIM_C_U8_BACKLIGHTERROR_ACTIVE == m_tErrorInfo[_u16Idx].enState)
			   {
				   _u32Data = (tU32) eBackLight_BERR_Off;
			   }
		   }
	   }
	   else
	   {
		   ETG_TRACE_USR3(("vUpdateDetetctionState: display inactive"));
		   // back light error state handled
		   m_bStateNotHandled = FALSE;
	   }
	   vTraceInfo();
	   _poApp->vPostMsgDimStateChanged(strIfName, _u32Data);
   }
}

/*******************************************************************************
*
* FUNCTION:    vFindGPIN()
*
* DESCRIPTION: find back light error GPIN pat and read the value
*
* PARAMETER:   tU16 u16Idx	... index of back light error GPIN (0 ...)
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid dispvidctrl_tclBacklightError::vFindGPIN(tU16 u16Idx)
{
   ETG_TRACE_USR4(("vFindGPIN(index: %d) entered.", u16Idx));

   if ((DISPVIDCTRL_C_U16_ARRAYELEMENTS(m_tErrorInfo) > u16Idx)
	   && (0 == strlen(m_tErrorInfo[u16Idx].strPath)))
   {
	   // path not set ... find the path and read the value
	   (tVoid)memset((tVoid*)m_tErrorInfo[u16Idx].strPath,OSAL_NULL,sizeof(m_tErrorInfo[u16Idx].strPath));
	   for (tU16 _u16Idy = 0; DIM_C_U8_BACKLIGHTERROR_MAXINPUTS > _u16Idy; ++_u16Idy)
	   {
		   char _strPath[sizeof(m_tErrorInfo[u16Idx].strPath)] = {0};
		   (tVoid)memset((tVoid*)_strPath,OSAL_NULL,sizeof(_strPath));
		   snprintf(_strPath, sizeof(_strPath)-1, "%s%s%s%d%s%s%d",
				   DIM_C_STR_BACKLIGHTERROR_INPUTPATH,
				   DIM_C_STR_BACKLIGHTERROR_DELIMITER,
				   DIM_C_STR_BACKLIGHTERROR_INPUTDIRNAME, _u16Idy,  // e.g.: input0
				   DIM_C_STR_BACKLIGHTERROR_DELIMITER,
				   DIM_C_STR_BACKLIGHTERROR_BERRORNAME, u16Idx+1 ); // e.g.: backlight_error1
		   tenBERR_State _enState = enReadGPIN(_strPath);
		   ETG_TRACE_USR3(("vFindGPIN(index: %d) state=%u, path=%100s", u16Idx, ETG_CENUM(tenBERR_State, _enState), _strPath ));
		   if (enBERR_zero == _enState || enBERR_one == _enState)
		   {
			   (tVoid)memcpy((tVoid*)m_tErrorInfo[u16Idx].strPath, _strPath, sizeof(m_tErrorInfo[u16Idx].strPath));
			   m_tErrorInfo[u16Idx].enState = _enState;
			   return;
		   }
	   }
   }
}

/*******************************************************************************
*
* FUNCTION:    enReadGPIN()
*
* DESCRIPTION: reading the specified back light error GPIN device
*
* PARAMETER:   const char *			full qualified path
*
* RETURNVALUE: tenBERR_State		back light error GPIN state
*
*******************************************************************************/
tenBERR_State dispvidctrl_tclBacklightError::enReadGPIN(const char* cstrPath) const
{
   ETG_TRACE_USR4(("enReadGPIN() entered."));
   tenBERR_State _enReturn = enBERR_undef;
   if (OSAL_NULL != cstrPath)
   {
	   int _iFd = open(cstrPath, O_RDONLY | O_NONBLOCK);
	   if (-1 != _iFd)
	   {
		   char _szBuffer[DIM_C_U8_BACKLIGHTERROR_MAXBUFLEN];
		   (tVoid)memset((tVoid*)_szBuffer,OSAL_NULL,sizeof(_szBuffer));
		   ssize_t _tReturn = read(_iFd, _szBuffer, sizeof(_szBuffer)-1);
		   ETG_TRACE_USR4(("enReadGPIN(%100s) buffer: %10s (length: %d), return: %d ",
				   cstrPath, _szBuffer, strlen(_szBuffer), _tReturn ));
		   close(_iFd);
		   if (0 < _tReturn)
		   {
			   switch((int)atoi(_szBuffer))
			   {
			   case 0:
				   _enReturn = enBERR_zero;
				   break;
			   case 1:
				   _enReturn = enBERR_one;
				   break;
			   default:
				   _enReturn = enBERR_max;
				   break;
			   }
		   }
		   else
		   {
			   _enReturn = enBERR_size;
		   }
	   }
	   else
	   {
		   _enReturn = enBERR_unexist;
	   }
   }
   return _enReturn;
}

/*******************************************************************************
*
* FUNCTION:    vTraceBEerr()
*
* DESCRIPTION: trace back light error status information
*
* PARAMETER:   const char *			prefix of every trace line
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid dispvidctrl_tclBacklightError::vTraceBEerr(const char* strIdentifier) const
{
   ETG_TRACE_USR4(("vTraceBEerr() entered."));

   char _strAttrName[60] = {0};
   ETG_TRACE_USR2(("%26s: %30s=%u ", strIdentifier, "         DISP_ACTIVE",
		   ETG_ENUM(DISPVIDCTRL_Bool, m_bDisplayActive) ));
   ETG_TRACE_USR2(("%26s: %30s=%u ", strIdentifier, "      ERR_NOTHANDLED",
		   ETG_ENUM(DISPVIDCTRL_Bool, m_bStateNotHandled) ));
   for (tU16 _u16Idx = 0; DISPVIDCTRL_C_U16_ARRAYELEMENTS(m_tErrorInfo) > _u16Idx; ++_u16Idx)
   {
       (tVoid)memset(_strAttrName, OSAL_NULL, sizeof(_strAttrName));
       sprintf(_strAttrName, "    FAIL%d_DevicePath", _u16Idx);
       ETG_TRACE_USR2(("%26s: %30s=%100s ", strIdentifier, _strAttrName,
    		   m_tErrorInfo[_u16Idx].strPath ));
       (tVoid)memset(_strAttrName, OSAL_NULL, sizeof(_strAttrName));
       sprintf(_strAttrName, "     FAIL%d_GPinState", _u16Idx);
       ETG_TRACE_USR2(("%26s: %30s=%u ", strIdentifier, _strAttrName,
    		   ETG_CENUM(tenBERR_State, m_tErrorInfo[_u16Idx].enState) ));
       (tVoid)memset(_strAttrName, OSAL_NULL, sizeof(_strAttrName));
       sprintf(_strAttrName, "  FAIL%d_GPinStateSim", _u16Idx);
       ETG_TRACE_USR2(("%26s: %30s=%u ", strIdentifier, _strAttrName,
    		   ETG_CENUM(tenBERR_State, m_tErrorInfo[_u16Idx].enSimulateState) ));
   }
}

/*******************************************************************************
*
* FUNCTION:    s32SetSimulateState()
*
* DESCRIPTION: set simulated back light error state of specified GPin
*
* PARAMETER:   tU8 u8GPin, tU8 u8State
*
* RETURNVALUE: none
*
*******************************************************************************/
tS32 dispvidctrl_tclBacklightError::s32SetSimulateState(tU8 u8GPin, tU8 u8State)
{
   ETG_TRACE_USR4(("s32SetSimulateState(no: %u, state: %u) entered.",
		   ETG_ENUM(BERR_NO, u8GPin), ETG_ENUM(BERR_STATE, u8State) ));

   tS32 _s32Return = -1;
   if (DISPVIDCTRL_C_U16_ARRAYELEMENTS(m_tErrorInfo) > (tU16)u8GPin)
   {
	   if (0 == u8State)
	   {
		   m_tErrorInfo[u8GPin].enSimulateState = DIM_C_U8_BACKLIGHTERROR_INACTIVE;
	   }
	   else if (1 == u8State)
	   {
		   m_tErrorInfo[u8GPin].enSimulateState = DIM_C_U8_BACKLIGHTERROR_ACTIVE;
	   }
	   else
	   {
		   m_tErrorInfo[u8GPin].enSimulateState = enBERR_max;
	   }
	   _s32Return = 0;
   }

   return _s32Return;
}
