/************************************************************************
* FILE:        dispvidctrl_tclGstIpa.cpp
* PROJECT:
* SW-COMPONENT: DisplayVideoController
*----------------------------------------------------------------------
*
* DESCRIPTION: Audio player module
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 19.11.2017| RBEI/ECV4 deo2kor  | init
*
*************************************************************************/

/*******************************************************************************
                        Includes
*******************************************************************************/
#include "dispvidctrl_tclGstIpa.h"
#include <string.h>

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DISPVIDCTRL_CLIENT_AUDIOPLAYER
#include "trcGenProj/Header/dispvidctrl_tclGstIpa.cpp.trc.h"
#endif

/*******************************************************************************
                        Defines
*******************************************************************************/
dispvidctrl_tclGstIpa* dispvidctrl_tclGstIpa::_poMyStaticRef = NULL;

#define IPA_AUDIO_BASE_PATH (tCString)"/opt/bosch/early/audio/"

#define ALSA_DEVICE_NAME_MAX_LENGTH  30


/******************************************************************************/
/* FUNCTION     dispvidctrl_tclGstIpa()                                       */
/******************************************************************************/
/**
*  \brief       constructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclGstIpa::dispvidctrl_tclGstIpa() : dispvidctrl_tclGstAudioPipeline() 
{
    ETG_TRACE_FATAL(("dispvidctrl_tclGstIpa::dispvidctrl_tclGstIpa() entered."));

    _poMyStaticRef = this;
    _u8IpaActiveRequestId = 0;    
    _u32HmiLanguage = 0;    
    _bIpaActive = FALSE;    
    _szDevName = NULL;
}

/******************************************************************************/
/* FUNCTION     dispvidctrl_tclGstIpa()                                       */
/******************************************************************************/
/**
*  \brief       constructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclGstIpa::dispvidctrl_tclGstIpa(const TEarlyConfiguration& tEarlyConfig) 
: dispvidctrl_tclGstAudioPipeline()
{
    ETG_TRACE_FATAL(("dispvidctrl_tclGstIpa::dispvidctrl_tclGstIpa() entered."));

    _poMyStaticRef = this;
    _u8IpaActiveRequestId = 0;
    _bIpaActive = FALSE;

    _u32HmiLanguage = tEarlyConfig.tEarlyNor.u32GuidelineWarningLanguage;
    
    _szDevName = new char[ALSA_DEVICE_NAME_MAX_LENGTH];
}

/******************************************************************************/
/* FUNCTION     ~dispvidctrl_tclGstIpa                                        */
/******************************************************************************/
/**
*  \brief       destructor
*
*  \param       none
*  \return      none
*/
/******************************************************************************/
dispvidctrl_tclGstIpa::~dispvidctrl_tclGstIpa(tVoid)
{
   ETG_TRACE_FATAL(("dispvidctrl_tclGstIpa::~dispvidctrl_tclGstIpa() entered."));

    _poMyStaticRef = NULL;
    _u8IpaActiveRequestId = 0;
    _u32HmiLanguage = 0;
    
    if (_szDevName)
    {
       delete _szDevName;
       _szDevName = NULL;
    }
    
}

tVoid dispvidctrl_tclGstIpa::vGetConfiguration(const TConfiguration* pStConfigurationValues)
{
    ETG_TRACE_FATAL(("[%d ms] dispvidctrl_tclGstIpa::vGetConfiguration() entered.", OSAL_ClockGetElapsedTime()));
}

tBool dispvidctrl_tclGstIpa::bInit(GMainLoop *pGstMainLoop)
{
    if (_pGstSource != NULL && _pGstAudioSink != NULL) {
       // we are already initialized
       return true;
    }

    _pGstMainLoop = pGstMainLoop;
    DISPVIDCTRL_NULL_POINTER_CHECK_VAL(_pGstMainLoop);

    _pGstPipeline = gst_pipeline_new ("IPAVOICE");
    DISPVIDCTRL_NULL_POINTER_CHECK_VAL(_pGstPipeline);

    /* we add a message handler */
    _pGstBus = gst_pipeline_get_bus (GST_PIPELINE (_pGstPipeline)); //lint !e826 PQM_authorized_multi_415
    DISPVIDCTRL_NULL_POINTER_CHECK_VAL(_pGstBus);

    gst_bus_add_watch (_pGstBus, dispvidctrl_tclGstIpa::bBusCall, _pGstMainLoop);
    
    tU32 u32StartTime = OSAL_ClockGetElapsedTime();
    
    _pGstSource = gst_element_factory_make("filesrc","source");
    if ( NULL == _pGstSource ) {
       dispvidctrl_tclAppMain::vWriteStartupLog("[RVC-EARLY]: IPA Audio pipeline create filesrc done", OSAL_ClockGetElapsedTime() - u32StartTime);
       ETG_TRACE_FATAL (("dispvidctrl_tclGstIpa::bInit source Element could not be created.\n"));
    }
    
    _pGstAudioParse = gst_element_factory_make("mpegaudioparse","audparse");
    if ( NULL == _pGstAudioParse ) {
       ETG_TRACE_FATAL (("dispvidctrl_tclGstIpa::bInit source Element could not be created.\n"));
    }

    _pGstAudioDecoder = gst_element_factory_make("beepdec", "play");
    if ( NULL == _pGstAudioDecoder ) {
       ETG_TRACE_FATAL (("dispvidctrl_tclGstIpa::bInit AudioDecoder Element could not be created.\n"));
    }
    
    _pGstAudioConverter = gst_element_factory_make("audioconvert","convert");
    if ( NULL == _pGstAudioConverter ) {
       ETG_TRACE_FATAL (("dispvidctrl_tclGstIpa::bInit AudioConverter Element could not be created.\n"));
    }
    
    _pGstAudioSink = gst_element_factory_make("alsasink", "sink");
    if ( NULL == _pGstAudioSink ) {
       ETG_TRACE_FATAL (("dispvidctrl_tclGstIpa::bInit sink Element could not be created.\n"));
    }

    gst_bin_add_many((GstBin*)_pGstPipeline, _pGstSource, _pGstAudioParse, _pGstAudioDecoder, _pGstAudioConverter, _pGstAudioSink, NULL);
    
    // Linking source and decode bin
    if (!gst_element_link (_pGstSource, _pGstAudioParse))
    {
       ETG_TRACE_FATAL (("source and mp3 parser Elements could not be linked.\n"));
       gst_object_unref (_pGstPipeline);
       return false;
    }
    
    // Linking source and decode bin
    if (!gst_element_link (_pGstAudioParse, _pGstAudioDecoder))
    {
       ETG_TRACE_FATAL (("mp3 parser and decoder Elements could not be linked.\n"));
       gst_object_unref (_pGstPipeline);
       return false;
    }
    
    // Linking source and decode bin
    if (!gst_element_link (_pGstAudioDecoder, _pGstAudioConverter))
    {
       ETG_TRACE_FATAL (("decoder and Audio converter Elements could not be linked.\n"));
       gst_object_unref (_pGstPipeline);
       return false;
    }
    
    // Linking audioconverter and sink
    if (!gst_element_link (_pGstAudioConverter, _pGstAudioSink))
    {
       ETG_TRACE_FATAL (("converter and sink Elements could not be linked.\n"));
       gst_object_unref (_pGstPipeline);
       return false;
    }
    
    ETG_TRACE_FATAL (("dispvidctrl_tclGstIpa::IPA Audio pipeline created"));
    dispvidctrl_tclAppMain::vWriteStartupLog("[RVC-EARLY]: IPA Audio pipeline created", OSAL_ClockGetElapsedTime() - u32StartTime);
	return true; // to check
}

tVoid dispvidctrl_tclGstIpa::vUpdateIpaRequest(const tU8 Request)
{
   ETG_TRACE_FATAL (("dispvidctrl_tclGstIpa::vUpdateIpaRequest Request %u _u8IpaActiveRequestId %u", Request, _u8IpaActiveRequestId));

   if ( _u8IpaActiveRequestId != Request )
   {
      _u8IpaActiveRequestId = Request;
      
      if ( 0x00 == _u8IpaActiveRequestId || 0x0f == _u8IpaActiveRequestId ) {
         vStop();
      } else {
         (void) bSetIpaAudioFilePath();
      }
   }
}

tVoid dispvidctrl_tclGstIpa::vStart()
{
   ETG_TRACE_FATAL(("dispvidctrl_tclGstIpa::vStart() entered."));
   DISPVIDCTRL_NULL_POINTER_CHECK(_pGstPipeline);
   
   if ( !_bIpaActive && ( 0x00 != _u8IpaActiveRequestId && 0x0f != _u8IpaActiveRequestId ) )
   {
      _bIpaActive = TRUE;
      gst_element_set_state(GST_ELEMENT(_pGstPipeline), GST_STATE_PLAYING); //lint !e826 PQM_authorized_multi_415
   }
   else
   {
      ETG_TRACE_FATAL(("dispvidctrl_tclGstIpa::vStart() cannot Play because no valid Request selected."));
   }
}

tVoid dispvidctrl_tclGstIpa::vPause()
{
   ETG_TRACE_FATAL(("dispvidctrl_tclGstIpa::vPause() entered."));
   DISPVIDCTRL_NULL_POINTER_CHECK(_pGstPipeline);
   
   if ( _bIpaActive )
   {
      _bIpaActive = FALSE;
      gst_element_set_state(GST_ELEMENT(_pGstPipeline), GST_STATE_PAUSED); //lint !e826 PQM_authorized_multi_415
   }
}

tVoid dispvidctrl_tclGstIpa::vStop()
{
   ETG_TRACE_FATAL(("dispvidctrl_tclGstIpa::vStop() entered."));
   DISPVIDCTRL_NULL_POINTER_CHECK(_pGstPipeline);
   
   if ( _bIpaActive )
   {
      _bIpaActive = FALSE;
      gst_element_set_state(GST_ELEMENT(_pGstPipeline), GST_STATE_NULL); //lint !e826 PQM_authorized_multi_415
   }
}

tVoid dispvidctrl_tclGstIpa::vTraceDebugInformation(tVoid)
{
   // unused
}

tVoid dispvidctrl_tclGstIpa::vSetAlsaDeviceName(const tChar* szDevName)
{
   ETG_TRACE_FATAL(("dispvidctrl_tclGstIpa::vSetAlsaDeviceName() AlsaDeviceName %s.", szDevName));
   char u8Length = strlen(szDevName);
   
   if ( u8Length < ALSA_DEVICE_NAME_MAX_LENGTH )
   {
      strcpy(_szDevName, szDevName);
   }
   else 
   {
      ETG_TRACE_FATAL(("dispvidctrl_tclGstIpa::vSetAlsaDeviceName() AlsaDeviceName >30 chars, length is %u", u8Length));
   }
}

gboolean dispvidctrl_tclGstIpa::bBusCall(GstBus* /*bus*/, GstMessage* msg, gpointer /*data*/)
{
   if(msg == NULL)
      return FALSE;

   GstState oldstate;
   GstState newstate;
   GstState pending;
   
   switch (GST_MESSAGE_TYPE(msg))
   {
   case GST_MESSAGE_EOS:
      {
         // media file finished
         ETG_TRACE_USR4(("dispvidctrl_tclGstIpa::EOS STREAM RECEIVED STOPPING THE PLAYER"));
         //Set the pipeline state to NULL
         if (_poMyStaticRef) {
            _poMyStaticRef->vStop();
         }
         //playStatus = PLAY_STATE_NULL;
         //Invoke message informing that the playback has stopped
      }
      break;

   case GST_MESSAGE_STATE_CHANGED:
      gst_message_parse_state_changed (msg, &oldstate, &newstate, &pending);
      ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstIpa::bBusCall() old: %d new: %d pending: %d state-changed: %s", OSAL_ClockGetElapsedTime(),oldstate,newstate,pending, GST_MESSAGE_SRC_NAME(msg)));
      break;

   case GST_MESSAGE_WARNING:
      break;

   case GST_MESSAGE_ERROR:
      {
          gchar* debug;
          GError* error;

          gst_message_parse_error(msg, &error, &debug);
          g_free(debug);

          ETG_TRACE_FATAL( ( "[%d ms] dispvidctrl_tclGstIpa::bBusCall() Error: %s", OSAL_ClockGetElapsedTime(), error->message));

          g_error_free(error);

          // TBD: error handling can be optimized for msg->src. In case of v4lsink
          // being the source switch to gst_apx_sink may solve the issue
          //g_main_loop_quit(loop);
      }
      break;

   default:
      //Unhandle messages, ignore
      break;
   }
   
   return TRUE;
}

tVoid dispvidctrl_tclGstIpa::vAudioPlayerInit()
{
   // unused
}

// inputs - language, requestId
tBool dispvidctrl_tclGstIpa::bSetIpaAudioFilePath() const
{
   if ( 0x00 == _u8IpaActiveRequestId || 0x0f == _u8IpaActiveRequestId )
   {
      return false;
   }
   
   char  as8Filename_AudioFile[13];
   char  as8path_AudioFile[50];
   
   //char  as8AlsaDeviceName[] = "AdevAcousticoutSpeech";
   
   //get file name extensions from configuration value
   /* put each nibble of configuration value as char in the string
   *  (0x30 to 0x39 if nibble value is going from  0 to 9
   *   and 0x61 to 0x66 if nibble value is going from 0xa to 0xf (10 to 15))    add 39 for hex a..f (use small letters in file name)
   */
   as8Filename_AudioFile[0]  = (char) 'i';
   as8Filename_AudioFile[1]  = (char) 'p';
   as8Filename_AudioFile[2]  = (char) 'a';

   // Hmi Language
   as8Filename_AudioFile[3]  = (char)(0x30 + (0x0f & (_u32HmiLanguage >> 4)) + (((0x0f & (_u32HmiLanguage >> 4)) < 10) ? 0:39));
   as8Filename_AudioFile[4]  = (char)(0x30 + (0x0f & _u32HmiLanguage) + (((0x0f & _u32HmiLanguage) <10) ? 0:39));
                               
   as8Filename_AudioFile[5]  = (char) '_';

   // Request Id
   as8Filename_AudioFile[6]  = (char) '0';
   as8Filename_AudioFile[7]  = (char)(0x30 + (0x0f & _u8IpaActiveRequestId) + (((0x0f & _u8IpaActiveRequestId) <10) ? 0:39));
                               
   as8Filename_AudioFile[8]  = (char) '.';
   as8Filename_AudioFile[9]  = (char) 'm';
   as8Filename_AudioFile[10] = (char) 'p';
   as8Filename_AudioFile[11] = (char) '3';
   as8Filename_AudioFile[12] = 0; // zero termination
   
   
   strcpy(as8path_AudioFile, IPA_AUDIO_BASE_PATH);
   strcpy(as8path_AudioFile + strlen(IPA_AUDIO_BASE_PATH), as8Filename_AudioFile);
   
   ETG_TRACE_USR4(("dispvidctrl_tclGstIpa::bSetIpaAudioFilePath() Audio mp3 file path %s", as8path_AudioFile));
   
   if (_pGstSource)
      g_object_set((GObject*) _pGstSource, "location", as8path_AudioFile, NULL);
   
   if (_pGstAudioSink)
      g_object_set((GObject*) _pGstAudioSink, "device", _szDevName, NULL);

   return true;
}


