/* Description and history of csm_stack_m.h at the end of the file   */
#ifndef CSM_STACK_M_HEADER
/** File name for preprozessor commands */
#define CSM_STACK_M_HEADER

/* csm_stack_mp.h has to be included afterwards because some defines
 * and typedefs have to be made before they are used in csm_stack_mp.h.
 *
 * No other includes allowed.
 */

//******************************************************************************
//* defines and macros (scope: csm stack internal)
//******************************************************************************
//******************************************************************************
//  CCCC  SSSSS  MM   MM
// C      S      M M M M
// C       SSS   M  M  M        Defines
// C          S  M     M
//  CCCC  SSSSS  M     M
//******************************************************************************
/** Macro for error code generation.        \n
 *  global error code:                      \n
 *  description          position 12345678  \n
 *  EG: Error Grade (g)           gg        \n
 *  EL: Error Location (l)          ll      \n
 *  EC: error code (e)                eeee
 */
#if (CSM_S_REDUCED_ERROR_INTERFACE == CSM_C_F_ON)
  #define CSM_M_MAKE_ERROR(EG, EL, EC) (LONG)(-(((EG) & 0x7F) << 24))
#elif (CSM_S_EXCEPTION_HANDLER_ACTIVATED == CSM_C_F_ON)
  #define CSM_M_MAKE_ERROR(EG, EL, EC) (LONG)CSM_lExceptionHandler(EG,EL,EC)
#else
  #define CSM_M_MAKE_ERROR(EG, EL, EC) (LONG)( - ( (((EG) & 0x7F) << 24) + (((EL) & 0xFF) << 16) + ((EC) & 0xFFFF) ) )
#endif

// #ToDo: check if this could be removed
#ifndef PFN_CALLBACK
  #define PFN_CALLBACK void*
#endif

// compatibility text replacement due to text correction
#define CSM_C_TRACE_FKT_CSM_TIMER_EXPIERED    CSM_C_TRACE_FKT_CSM_TIMER_EXPIRED

// Interface for vector IRQ initializing function
#define CSM_C_CAN_IRQSETUP                   0
#define CSM_C_CAN_EI                         1
#define CSM_C_CAN_DI                         2
#define CSM_C_CAN_IRQEXIT                    3

/** CSM telegram types: */
#define  CSM_C_TEL_TYPE_NO_TYPE              0x00
#define  CSM_C_TEL_TYPE_MCNET_ASDT           0x01
#define  CSM_C_TEL_TYPE_MCNET_USDT_RX        0x02
#define  CSM_C_TEL_TYPE_MCNET_USDT_TX        0x03
#define  CSM_C_TEL_TYPE_MCNET_WD_RX          0x04
#define  CSM_C_TEL_TYPE_MCNET_WD_TX          0x05

#define  CSM_C_TEL_TYPE_VAG_TP20             0x10
#define  CSM_C_TEL_TYPE_VAG_BAP_RAW          0x11
#define  CSM_C_TEL_TYPE_VAG_BAP              0x14
#define  CSM_C_TEL_TYPE_VAG_EID              0x15

#define  CSM_C_TEL_TYPE_ISO_TP               0x20
#define  CSM_C_TEL_TYPE_ISO_TP_AREA          0x21
#define  CSM_C_TEL_TYPE_MPDT_TP              0x22

#define  CSM_C_TEL_TYPE_LIN_USDT_RX          0x23

#define  CSM_C_TEL_TYPE_OSEK_NM              0x40
#define  CSM_C_TEL_TYPE_OSEKI_NM             0x41
#define  CSM_C_TEL_TYPE_GMLAN_NM             0x42
#define  CSM_C_TEL_TYPE_IVLAN_NM             0x43
#define  CSM_C_TEL_TYPE_J1939_NM             0x44
#define  CSM_C_TEL_TYPE_VWHIGH_NM            0x45

#define  CSM_C_TEL_TYPE_FIAT_NM_MASTER_RX    0x51
#define  CSM_C_TEL_TYPE_FIAT_NM_MASTER_TX    0x52
#define  CSM_C_TEL_TYPE_FIAT_NM_SLAVE_RX     0x53
#define  CSM_C_TEL_TYPE_FIAT_NM_SLAVE_TX     0x54

#define  CSM_C_TEL_TYPE_GATEWAY              0x60  // standard case: CDA gateway type
#define  CSM_C_TEL_TYPE_GATEWAY_PREDEF       0x61  // special case: different GW receive function, different GW configuration
#define  CSM_C_TEL_TYPE_GATEWAY_TX_NON_FIFO  0x62  // special case: same as the standard case, but TX without FIFO
#define  CSM_C_TEL_TYPE_GATEWAY_TX_FIFO      CSM_C_TEL_TYPE_GATEWAY

#define  CSM_C_TEL_TYPE_UUDT_RX              0x70
#define  CSM_C_TEL_TYPE_UUDT_TX              0x71

#define  CSM_C_TEL_TYPE_CBR_RX               0x81
#define  CSM_C_TEL_TYPE_CBR_TX               0x82
#define  CSM_C_TEL_TYPE_CBR_TX_2QUEUE        0x83

#define  CSM_C_TEL_TYPE_CNP_APPL_TEL_RX      0x91
#define  CSM_C_TEL_TYPE_CNP_APPL_TEL_TX      0x92
#define  CSM_C_TEL_TYPE_XCP_RX               0x93
#define  CSM_C_TEL_TYPE_XCP_TX               0x94

#define  CSM_C_TEL_TYPE_J1939_CNP_RX         0xA0
#define  CSM_C_TEL_TYPE_J1939_CNP_TX         0xA1

/** macro for determination of the number of elements in a array                                                 */
#define CSM_M_ELEMENTS(array)  (sizeof(array)/sizeof(array[0]))

/* suppress lint warning 572 "Excessive shift value", if BitPos <= 3:                                            */
/*lint -emacro(572,CSM_M_READBITINBYTEFIELD,CSM_M_SETBITINBYTEFIELD,CSM_M_CLEARBITINBYTEFIELD) PQM_authorized_78 */
#define CSM_M_READBITINBYTEFIELD(Adr,BitPos) \
          (*(((BYTE*)Adr) + ((BitPos)>>3)) & (1<<((BitPos) & 0x07)))

#define CSM_M_SETBITINBYTEFIELD(Adr,BitPos) \
          (*(((BYTE*)Adr) + ((BitPos)>>3)) |= (1<<((BitPos) & 0x07)))

#define CSM_M_CLEARBITINBYTEFIELD(Adr,BitPos) \
           (*(((BYTE*)Adr) + (BitPos>>3)) &= ~((DWORD)(1<<(BitPos & 0x07))))

// remap makro for error generation
#define CSM_STACK_M_MAKE_ERROR(SecurityLevel,ErrorCode)  CSM_M_MAKE_ERROR(SecurityLevel,CSM_C_ERR_LOC_CSM_STACK,ErrorCode)

// suppress LINT warning for unused parameters of functions
#define CSM_M_PARAMETER_INTENTIONALLY_UNUSED(x)  ((void)(x))

// marker to identify extended CAN IDs (bit 31 set)
#define CSM_C_CDA_EXTENDED_ID   ((DWORD) 0x80000000)

// VW specific CSM extensions
#if ( CSM_S_CSM_VW == CSM_C_F_ON )
  /* Makro to calculate Event-Listern event constant value.
   * Inputs:
   *   b: Is the event which constant has to be calculated.
   *   a: Is the event which has to be removed if event b is stored.
   */
  #define CSM_VW_CALC_EVNT_NR(a,b)            (((((DWORD)(~a))<<16) & 0xFFFF0000) | b)

  // Internal Events
  #define CSM_VW_C_E_VOID_L                   0x0001UL
  #define CSM_VW_C_E_TIMER_TS_ELAPSED_L       0x0002UL
  #define CSM_VW_C_E_TIMER_TW_ELAPSED_L       0x0004UL
  #define CSM_VW_C_E_TIMER_RKA_ELAPSED_L      0x0008UL
  #define CSM_VW_C_E_BUS_WAKE_TEMP_L          0x0010UL
  #define CSM_VW_C_E_FNL_CHANGED_L            0x0020UL
  #define CSM_VW_C_E_FORCE_SHUTDOWN_L         0x0040UL
  #define CSM_VW_C_E_RADIO_ON_L               0x0080UL
  #define CSM_VW_C_E_RADIO_OFF_L              0x0100UL
  #define CSM_VW_C_E_REMOTE_CAN_SLEEP_L       0x0200UL
  #define CSM_VW_C_E_REMOTE_CAN_AWAKE_L       0x0400UL
  #define CSM_VW_C_E_CHK_BAP_PHONE_MUTE_L     0x0800UL
  #define CSM_VW_C_E_VOID                     CSM_VW_CALC_EVNT_NR(0, CSM_VW_C_E_VOID_L)
  #define CSM_VW_C_E_TIMER_TS_ELAPSED         CSM_VW_CALC_EVNT_NR(0, CSM_VW_C_E_TIMER_TS_ELAPSED_L)
  #define CSM_VW_C_E_TIMER_TW_ELAPSED         CSM_VW_CALC_EVNT_NR(0, CSM_VW_C_E_TIMER_TW_ELAPSED_L)
  #define CSM_VW_C_E_TIMER_RKA_ELAPSED        CSM_VW_CALC_EVNT_NR(0, CSM_VW_C_E_TIMER_RKA_ELAPSED_L)
  #define CSM_VW_C_E_BUS_WAKE_TEMP            CSM_VW_CALC_EVNT_NR(CSM_VW_C_E_FORCE_SHUTDOWN_L , CSM_VW_C_E_BUS_WAKE_TEMP_L)
  #define CSM_VW_C_E_FNL_CHANGED              CSM_VW_CALC_EVNT_NR(0, CSM_VW_C_E_FNL_CHANGED_L)
  #define CSM_VW_C_E_FORCE_SHUTDOWN           CSM_VW_CALC_EVNT_NR(CSM_VW_C_E_BUS_WAKE_TEMP_L, CSM_VW_C_E_FORCE_SHUTDOWN_L)
  #define CSM_VW_C_E_RADIO_ON                 CSM_VW_CALC_EVNT_NR(CSM_VW_C_E_RADIO_OFF_L, CSM_VW_C_E_RADIO_ON_L)
  #define CSM_VW_C_E_RADIO_OFF                CSM_VW_CALC_EVNT_NR(CSM_VW_C_E_RADIO_ON_L, CSM_VW_C_E_RADIO_OFF_L)
  #define CSM_VW_C_E_REMOTE_CAN_SLEEP         CSM_VW_CALC_EVNT_NR(CSM_VW_C_E_REMOTE_CAN_AWAKE_L, CSM_VW_C_E_REMOTE_CAN_SLEEP_L)
  #define CSM_VW_C_E_REMOTE_CAN_AWAKE         CSM_VW_CALC_EVNT_NR(CSM_VW_C_E_REMOTE_CAN_SLEEP_L, CSM_VW_C_E_REMOTE_CAN_AWAKE_L)
  #define CSM_VW_C_E_CHK_BAP_PHONE_MUTE       CSM_VW_CALC_EVNT_NR(0, CSM_VW_C_E_CHK_BAP_PHONE_MUTE_L)

  // External Events
  #define CSM_VW_C_E_WAKE_UP_L                0x1000UL
  #define CSM_VW_C_E_KL15_ON_L                0x2000UL
  #define CSM_VW_C_E_KL15_OFF_L               0x4000UL
  #define CSM_VW_C_E_WAKE_UP                  CSM_VW_CALC_EVNT_NR(0,CSM_VW_C_E_WAKE_UP_L)
  #define CSM_VW_C_E_KL15_ON                  CSM_VW_CALC_EVNT_NR(CSM_VW_C_E_KL15_OFF_L, CSM_VW_C_E_KL15_ON_L)
  #define CSM_VW_C_E_KL15_OFF                 CSM_VW_CALC_EVNT_NR(CSM_VW_C_E_KL15_ON_L, CSM_VW_C_E_KL15_OFF_L)
#endif

//******************************************************************************
//  CCCC  DDDD   AAAAA
// C      D   D  A   A
// C      D   D  AAAAA          Defines
// C      D   D  A   A
//  CCCC  DDDD   A   A
//******************************************************************************
// the following error codes are used in CDA_hecc.c in case of initialisation
// problems, just to be able to track them later on with the help of the
// em_trace command in TTFis. Values are written to wCSM_EmTraceError (csm.c)
// Value of wCSM_EmTraceError can be set and get by access functions
#define CSM_EMTRACE_C_NO_ERROR                           0x00000000
#define CSM_EMTRACE_C_CDA_HECC_INIT_MODE_FAILED          0x00000001
#define CSM_EMTRACE_C_CDA_HECC_HW_SETUP_FAILED           0x00000002
#define CSM_EMTRACE_C_CDA_HECC_C_GET_HW_SETUP_FAILED     0x00000004
#define CSM_EMTRACE_C_CDA_HECC_C_ERROR_HW_CONFIG         0x00000008
#define CSM_EMTRACE_C_CDA_HECC_C_NORM_MODE_FAILED        0x00000010

// no mask used for the CAN message buffer
#define CSM_C_HW_LIST_NOMASK      0xFF

/** Defines for CSM_HW_LIST */
/** \defgroup CSM_C_HW_LIST_MASK_ */
/*@{*/
#define CSM_C_HW_LIST_MASK_0      0                      /**< number of CAN MASK */
#define CSM_C_HW_LIST_MASK_1      1                      /**< number of CAN MASK */
#define CSM_C_HW_LIST_MASK_2      2                      /**< number of CAN MASK */
#define CSM_C_HW_LIST_MASK_3      3                      /**< number of CAN MASK */
/*@}*/

/** \defgroup CSM_C_HW_LIST_OBJECT_TYPE_ */
/*@{*/
#define CSM_C_HW_LIST_OBJECT_TYPE_EMPTY            0     /**< CAN object is not used  */
#define CSM_C_HW_LIST_OBJECT_TYPE_RX_FULL_CAN_1    1     /**< CAN object is used as full CAN */
#define CSM_C_HW_LIST_OBJECT_TYPE_RX_FULL_CAN_2    2     /**< CAN object is used as full CAN on the second bus (NEC V850 only) */
#define CSM_C_HW_LIST_OBJECT_TYPE_RX_BASIC_CAN_1   3     /**< CAN object is used as basic CAN */
#define CSM_C_HW_LIST_OBJECT_TYPE_RX_BASIC_CAN_2   4     /**< CAN object is used as basic CAN on the second bus (NEC V850 only) */
#define CSM_C_HW_LIST_OBJECT_TYPE_TX_CAN_1        11     /**< CAN object is used for transmit */
#define CSM_C_HW_LIST_OBJECT_TYPE_TX_CAN_2        12     /**< CAN object is used for transmit on the second bus (NEC V850 only) */
/*@}*/

#if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
  // these values are not allowed to be changed
  // number of CAN hardware objects of the FCAN/FCN -> the defines are necessary only if FCAN/FCN is used !!!
  #define CSM_C_CAN_HW_OBJECTS_V850               32     // old V850 for Car Radio projects
  #define CSM_C_CAN_HW_OBJECTS_V850E2             64     // Autosar V850 for Gen3 projects
#endif // CSM_S_REMOTE_STACK

// Defines for CAN controller states
/** CAN controller Tx/Rx error counter 0 .. 95 */
#define CSM_C_CAN_CONTROLLER_STATE_NO_ERROR              0x00
/** CAN controller Tx/Rx error counter 96 .. 127 */
#define CSM_C_CAN_CONTROLLER_STATE_ERROR_WARNING         0x01
/** CAN controller Tx/Rx error counter 128 .. 255 */
#define CSM_C_CAN_CONTROLLER_STATE_ERROR_PASSIVE         0x02
/** CAN controller Tx error counter > 255 */
#define CSM_C_CAN_CONTROLLER_STATE_BUS_OFF               0x03

//******************************************************************************
//  CCCC  BBBB   RRRR
// C      B   B  R   R
// C      BBBB   RRRR             Defines
// C      B   B  R   R
//  CCCC  BBBB   R   R
//******************************************************************************
#if (CSM_S_CBR_AVAILABLE == CSM_C_F_ON)
  /** message types of CAN telegrams */
  #define CBR_TMSG_CYCLIC                   1            /**< cyclic           */
  #define CBR_TMSG_CYCLIC_IF_ACTIVE         2            /**< CyclicIfActive   */
  #define CBR_TMSG_SPONTAN                  3            /**< spontan          */
  #define CBR_TMSG_CYCLIC_AND_SPONTAN       4            /**< CyclicAndSpontan */
  //#define CBR_TMSG_CYC_AND_SPONT_WITH_DEL 5            // CyclicAndSpontanWithDelay hasn't been coded yet,
                                                         // because we don't know what is exactly meant
  //#define CBR_TMSG_SPONTAN_WITH_REP       6            // SpontanWithRepetition hasn't been coded yet,
                                                         // because we don't know what is exactly meant
  #define CBR_TMSG_CYC_IF_ACT_AND_SPONT     7            /**< CyclicIfActiveAndSpontan */
  //#define CBR_TMSG_SPONTAN_WITH_DELAY     8            // SpontanWithDelay hasn't been coded yet,
                                                         // because we don't know what is exactly meant
  #define CBR_TMSG_NO_TRANSMISSION          9            /**< message will never be sent */

  #ifdef CBR_SELF
    #define CBR_CONVERT_VECTOR_DBC_BIG_SIGNAL( bStartBit, bLength)  ( (8 * (7 - (( ( (8 * (7 - (bStartBit/8) ) ) + (bStartBit%8) ) - (bLength-1) )/8) ) ) + (( ( (8 * (7 - (bStartBit/8) ) ) + (bStartBit%8) ) - (bLength-1) )%8) ) , bLength
    /**< CBR_CONVERT_VECTOR_DBC_BIG_SIGNAL
    -Vector zeigt auf das MSBit und benutzt die Bit Nummerirung wie bei Littel Endian
     also muss man das DBC-Big-Startbit in das ESA3-Startbit umrechnen.
     temp1 = ( (8 * (7 - (DBC_Big_Startbit/8) ) ) + (DBC_Big_Startbit%8) )
     temp2 = temp1 - (bLaenge - 1)
     ESA_Startbit = ( (8 * (7 - (temp2/8) ) ) + (temp2%8) )
    */
    #define CBR_CONVERT_VECTOR_DBC_LITTLE_SIGNAL( bStartBit, bLength) bStartBit, bLength
    #define CBR_CONVERT_VW_SIGNAL( bStartBit, bLength) bStartBit, bLength
    #define CBR_CONVERT_FORD_SIGNAL( bStartBit, bLength) (7-((bStartBit+bLength-1)%8))+(8*((bStartBit+bLength-1)/8)), bLength
    //#define CBR_CONVERT_FIAT_MSB_SIGNAL_LSBIT_LENTGH( bStartBit, bLength) ((63-bStartBit) -7 +2*( bStartBit % 8)) , bLength
    #define CBR_CONVERT_FIAT_MSB_SIGNAL_LSBIT_LENTGH( bStartBit, bLength, bDlc) ((((bDlc*8)-1)-bStartBit) -7 +2*( bStartBit % 8)) , bLength
    #define CBR_CONVERT_FIAT_MSB_SIGNAL_MSBIT_LENTGH( bStartBit, bLength, bDlc) ((((bDlc*8)-1)-(bStartBit-(bLength-1))) -7 +2*( (bStartBit-(bLength-1)) %8)) , bLength          /*warning! noch nicht getestet*/
  #endif // CBR_SELF
#endif // CSM_S_CBR_AVAILABLE

//******************************************************************************
//  CCCC  N   N  PPPP
// C      NN  N  P   P
// C      N N N  PPPP           Defines
// C      N  NN  P
//  CCCC  N   N  P
//******************************************************************************
#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
  // ISO TP ---------------------------------------------------------------
  #if (CSM_S_ISO_TP == CSM_C_F_ON)
    // Addressing Modes
    // Used in cnp_itp.c and csm_stack_xxx.c
    #define  CSM_C_ITP_AMODE_NORMAL          1           /**< Normal addressing (single channel)       */
    #if (CSM_S_ITP_EXTENDED_ADDRESSING == CSM_C_F_ON)
      #define  CSM_C_ITP_AMODE_EXTENDED      2           /**< Ext. addressing (single channel)         */
      #define  CSM_C_ITP_AMODE_EXTENDED_MUX  3           /**< Ext. addressing, multipl. channel        */
    #endif
    #define  CSM_C_ITP_AMODE_NORMAL_FIXED    4           /**< Normal fixed addressing (single channel) */

    // Iso-TP types
    #define  CSM_C_ITP_TYPE_USDT             1           /**< "Standard" protocol version (USDT) */
    #define  CSM_C_ITP_TYPE_ASDT_BP          2           /**< BP ASDT protocol version           */
    #define  CSM_C_ITP_TYPE_ASDT_LIGHT       3           /**< "ASDT Light" protocol version      */
    #define  CSM_C_ITP_TYPE_ASDT_RN          4           /**< Renault's ASDT protocol version    */

    #if (CSM_S_ICCC == CSM_C_F_ON)
      // Iso-TP bConnectType: connection setup handling is different
      // channel is connected if...
      #define CSM_C_ITP_CON_TYPE_REGULAR     1            /**< ... CAN is active (rx and tx)        */
      #define CSM_C_ITP_CON_TYPE_ICCC        2            /**< .. own and peer node take part in NM */
      #define CSM_C_ITP_CON_TYPE_OWN_NODE    3            /**< .. own node takes part in NM         */
    #endif // CSM_S_ICCC
  #endif // CSM_S_ISO_TP

  // MCNet -----------------------------------------------------------------
  #if (CSM_S_MCNET == CSM_C_F_ON)
    // NM types: Master/Slave
    #define CSM_C_MCNET_SLAVE                 0x00
    #define CSM_C_MCNET_MASTER                0x01

    // Commands for NM services
    #define CSM_C_MCNET_WD_SEND_ENABLE        0x01
    #define CSM_C_MCNET_WD_SEND_DISABLE       0x02

    #define CSM_C_MCNET_WD_MON_ENABLE         0x01
    #define CSM_C_MCNET_WD_MON_DISABLE        0x02

    // CAN identifier for ASDT
    #define CSM_C_MCNET_ASDT_CAN_ID_MAS_AUT   0x418
    #define CSM_C_MCNET_ASDT_CAN_ID_AUT_MAS   0x218

    #define CSM_C_MCNET_ASDT_CAN_ID_MAS_TMC   0x41C
    #define CSM_C_MCNET_ASDT_CAN_ID_TMC_MAS   0x21C

    #define CSM_C_MCNET_ASDT_CAN_ID_MAS_NAV   0x448
    #define CSM_C_MCNET_ASDT_CAN_ID_NAV_MAS   0x248

    #define CSM_C_MCNET_ASDT_CAN_ID_MAS_DNL   0x44C
    #define CSM_C_MCNET_ASDT_CAN_ID_DNL_MAS   0x24C

    #define CSM_C_MCNET_ASDT_CAN_ID_NAV_TMC   0x508
    #define CSM_C_MCNET_ASDT_CAN_ID_TMC_NAV   0x509

    #define CSM_C_MCNET_ASDT_CAN_ID_TES_MAS   0x601
    #define CSM_C_MCNET_ASDT_CAN_ID_MAS_TES   0x701

    #define CSM_C_MCNET_ASDT_CAN_ID_TES_AUT   0x618
    #define CSM_C_MCNET_ASDT_CAN_ID_AUT_TES   0x718

    #define CSM_C_MCNET_ASDT_CAN_ID_TES_NAV   0x648
    #define CSM_C_MCNET_ASDT_CAN_ID_NAV_TES   0x748

    #define CSM_C_MCNET_ASDT_CAN_ID_MAS_DPL   0x4D0
    #define CSM_C_MCNET_ASDT_CAN_ID_DPL_MAS   0x2D0

    // Other general purpose defines
    #define CSM_MCNET_C_UNDEFINED             0xFF
  #endif // CSM_S_MCNET

  // J1939 -----------------------------------------------------------------
  #if (CSM_S_J1939 == CSM_C_F_ON)
    #define CSM_J1939_ADDR_NULL               0xFE
    #define CSM_J1939_ADDR_BROADCAST          0xFF
  #endif // CSM_S_J1939
#endif // CSM_S_CNP_AVAILABLE

//******************************************************************************
// TTTTT  RRRR   AAAAA   CCCC  EEEEE
//   T    R   R  A   A  C      E
//   T    RRRR   AAAAA  C      EEEE           Defines
//   T    R   R  A   A  C      E
//   T    R   R  A   A   CCCC  EEEEE
//******************************************************************************
#if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
    //*******************************************************************************************
    //* ETG trace active
    //*******************************************************************************************
    // trace functions (Func) (better described as trace events, in ETG used for TTFis input handling)
    enum
    {
      // CDA
      CSM_C_TRACE_FKT_CDA_SEND                     = 0x01,
      CSM_C_TRACE_FKT_CXX_CDA_RECEIVE_IND,
      CSM_C_TRACE_FKT_CXX_CDA_DATA_CON,
      CSM_C_TRACE_FKT_CDA_INIT,
      CSM_C_TRACE_FKT_CDA_ACTIVE,
      CSM_C_TRACE_FKT_CDA_DEACTIVE,
      CSM_C_TRACE_FKT_CDA_EXIT,
      CSM_C_TRACE_FKT_CDA_CAN_INTERRUPTS_DISABLE,
      CSM_C_TRACE_FKT_CDA_CAN_INTERRUPTS_RESTORE,
      CSM_C_TRACE_FKT_CDA_BUSOFF_ENTRY,
      CSM_C_TRACE_FKT_CDA_BUSOFF_RECOVERY,
      CSM_C_TRACE_FKT_CDA_GET_ERROR_STATE,
      CSM_C_TRACE_FKT_CSM_CDA_ERROR_IND,
      CSM_C_TRACE_FKT_CSM_CDA_WAKEUP,
      // CSM
      CSM_C_TRACE_FKT_CSM_COMMUNICATION_CON        = 0x20,
      CSM_C_TRACE_FKT_CSM_COMMUNICATION_IND,
      CSM_C_TRACE_FKT_CSM_COMMUNICATION_REQ,
      #if (CSM_S_REMOTE_STACK == CSM_C_F_ON)
        CSM_C_TRACE_FKT_CSM_COMMUNICATION_CON_REJ,
        CSM_C_TRACE_FKT_CSM_COMMUNICATION_CON_TIME,
      #endif
      // communication related
      CSM_C_TRACE_FKT_CSM_GET_COMMUNICATION_STATE = 0x25,
      CSM_C_TRACE_FKT_CSM_COMMUNICATION_REQ_RETVAL,
      CSM_C_TRACE_FKT_CSM_APPL_CALLBACK_INIT,
      CSM_C_TRACE_FKT_CSM_ERROR_IND,
      #if (CSM_S_DEACTIVATE_COMMUNICATION_REQUESTS == CSM_C_F_ON)
        CSM_C_TRACE_FKT_CSM_DEACTIVATE_COMMUNICATION_REQ,
        CSM_C_TRACE_FKT_CSM_ACTIVATE_COMMUNICATION_REQ,
      #endif
      CSM_C_TRACE_FKT_CSM_GET_ALL_SIGNALS_AGAIN   = 0x2B,
      // CBR signals
      CSM_C_TRACE_FKT_CSM_SIGNAL_READ             = 0x30,
      CSM_C_TRACE_FKT_CSM_SIGNAL_WRITE,
      CSM_C_TRACE_FKT_CSM_SIGNAL_IND,
      CSM_C_TRACE_FKT_CSM_SIGNAL_CALLBACK_INIT,
      CSM_C_TRACE_FKT_CSM_MULTIPLE_SIGNAL_READ,
      CSM_C_TRACE_FKT_CSM_MULTIPLE_SIGNAL_WRITE,
      CSM_C_TRACE_FKT_CSM_SIGNAL_CALLBACK_DEINIT_ALL,     // 0x36

      // Data related
      CSM_C_TRACE_FKT_CSM_DATA_REQ                = 0x40,
      CSM_C_TRACE_FKT_CSM_DATA_CON,
      CSM_C_TRACE_FKT_CSM_DATA_IND,
      CSM_C_TRACE_FKT_CSM_DATA_IND_PROCESSED,
      CSM_C_TRACE_FKT_CSM_DATA_READ,
      CSM_C_TRACE_FKT_CSM_DATA_REQ_RETVAL,
      CSM_C_TRACE_FKT_CSM_DATA_IND_FF,
      CSM_C_TRACE_FKT_CSM_DATA_ERROR_IND,

      // Timer related
      CSM_C_TRACE_FKT_CSM_TIMER_START             = 0x50,
      CSM_C_TRACE_FKT_CSM_TIMER_STOP,
      CSM_C_TRACE_FKT_CSM_TIMER_EXPIRED,
      CSM_C_TRACE_FKT_CSM_TIMER_CYCLIC_START,
      CSM_C_TRACE_FKT_CSM_TIMER_CYCLIC_STOP,

      // Init, Exit
      CSM_C_TRACE_FKT_CSM_INIT                    = 0x60,
      CSM_C_TRACE_FKT_CSM_EXIT,                           //0x61
      CSM_C_TRACE_FKT_CSM_APL_CALLBACK_PREINIT,           //0x62

      // system checks
      CSM_C_TRACE_FKT_CSM_SET_PROCESSORLOAD       = 0x66, /**< Command for setting processor load within kernel upper tester */
      CSM_C_TRACE_FKT_CSM_GET_SYSTEMRESOURCE      = 0x67, /**< Command for getting system resources within kernel upper tester */
      CSM_C_TRACE_FKT_CSM_SET_TEST_NUMBER,                /**< Command to path test number from CANoe server to the CSM stack so that it will be traced out. */
      CSM_C_TRACE_FKT_CSM_TRACE_TEST_NUMBER,              /**< Test number will be traced out. */
      CSM_C_TRACE_FKT_TSTRETVAL                   = 0x6F,

      // Bus state
      CSM_C_TRACE_FKT_CSM_BUSSTATE_CHANGED        = 0x80,
      CSM_C_TRACE_FKT_CSM_CALC_BUSSTATE,
      CSM_C_TRACE_FKT_CSM_CHANGE,

      // NVM related
      CSM_C_TRACE_FKT_CSM_LOCAL_NVM_SET           = 0x95,
      CSM_C_TRACE_FKT_CSM_LOCAL_NVM_GET           = 0x96,
      CSM_C_TRACE_FKT_CSM_NVN_WRITE_FAIL          = 0x97,
      CSM_C_TRACE_FKT_CSM_NVM_WRITE_PENDING       = 0x98,
      CSM_C_TRACE_FKT_CSM_NVN_RETRIGGER           = 0x99,
      CSM_C_TRACE_FKT_CSM_KDS_MIRROR_GET          = 0x9A,

      // CBR start, stop
      CSM_C_TRACE_FKT_CBR_STOP_RX                 = 0xA0,
      CSM_C_TRACE_FKT_CBR_STOP_TX,
      CSM_C_TRACE_FKT_CBR_START_RX,
      CSM_C_TRACE_FKT_CBR_START_TX,

      CSM_C_TRACE_FKT_CSM_SIGNAL_SUPPRESS         = 0xB0,

      // special cases translated in csm_p.trc
      #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
        #if (CSM_S_OSEK_NM == CSM_C_F_ON)
          CSM_C_TRACE_FKT_CNM_PROCESS_NM_STATE       = 0xC0,
          CSM_C_TRACE_FKT_CNM_UNDERVOLTAGE           = 0xC1,
        #endif

        #if (CSM_S_OSEKI_NM == CSM_C_F_ON)
          CSM_C_TRACE_FKT_CNM_OSEKI_RX_NODE_MONITOR  = 0xC0,
          CSM_C_TRACE_FKT_CNM_OSEKI_TX_MUTE_MONITOR  = 0xC1,
          CSM_C_TRACE_FKT_CNM_OSEKI_BUS_OFF_MONITOR  = 0xC2,
          CSM_C_TRACE_FKT_CNM_OSEKI_EVENT            = 0xC3,
        #endif

        #if (CSM_S_FIATSLAVE_NM == CSM_C_F_ON)
          CSM_C_TRACE_FKT_CNM_PROCESS_NM_STATE       = 0xC0,
        #endif
      #endif // CSM_S_CNM_AVAILABLE

      // The area 0xE0 .. 0xEF is reserved for project specific traces.
      CSM_C_TRACE_FKT_PROJECT_E0                     = 0xE0,

      CSM_C_TRACE_FKT_CSM_DUMMY                      = 0xF0,
      CSM_C_TRACE_FKT_CSM_STRING                     = 0xF1,
      CSM_C_TRACE_FKT_CSM_UPPERTST                   = 0xF2, // This trace input command starts/stops the upper tester in kernel or user mode.
      CSM_C_TRACE_FKT_CSM_SOURCEID                   = 0xF3, // For GMLAN NM - reads out the current source id list and traces it out
      CSM_C_TRACE_FKT_DOWNSTREAM_TRACE               = 0xF4, // CSM_C_DOWN_REASON_TRACE

      CSM_C_TRACE_FKT_MPDT_JOIN_CHECK_HELPER         = 0xF5, // Trigger a certain MPDT join check helper functionality
      CSM_C_TRACE_FKT_IMX_CONFIG_CHECK_HELPER        = 0xF6  // Trigger a certain config check helper functionality
    };

  #else // CSM_S_TRACE_ETG
    //*******************************************************************************************
    //* no ETG trace active -> we do standard tracing for high sophisticated systems like AUTOSAR
    //*******************************************************************************************
    // Trace transmit buffer size, to be determined according to the needs.
    // So the trace buffer size is defined here as 40, to fit most of the cases with a big reserve.
    // It can happen, in case of big amounts of trace data that they are been truncated in the trace output itself at runtime.
    // If you need more output for error investigation, just increase this define temporary
    #define CSM_MAX_TRACE_LENGTH             40
    // Length of trace header
    #define CSM_TRACE_HEADER                  6
    // trace makro to map trace output to a function (abbrevations used here are defined as enums below )
    #define CSM_M_TRACEOUT(Lvl, IF, Func, pbData, DataLen)       CSM_vTraceOut(Lvl, IF, Func, pbData, DataLen)

    // trace level (Lvl)
    enum
    {
      CSM_C_TRACEOUT_LEVEL_INTERFACE  = 1,    /**< only interface according to CAN SWA, information only */
      CSM_C_TRACEOUT_LEVEL_INFO,              /**< only internal information */
      CSM_C_TRACEOUT_LEVEL_WARNING,           /**< warnings */
      CSM_C_TRACEOUT_LEVEL_ERROR,             /**< error */
      CSM_C_TRACEOUT_LEVEL_SEVERE_ERROR       /**< error */
    };

    // trace interface (IF)
    enum
    {
      CSM_C_TRACEOUT_INTERFACE_CSM = 1,             /**< Interface 1# only */
      CSM_C_TRACEOUT_INTERFACE_CNM,                 /**< Interface 2# and CNM internal */
      CSM_C_TRACEOUT_INTERFACE_CNP,                 /**< Interface 3# and CNP internal */
      CSM_C_TRACEOUT_INTERFACE_CBR,                 /**< Interface 4# and CBR internal */
      CSM_C_TRACEOUT_INTERFACE_CDA,                 /**< Interface 5# and CDA internal */
      CSM_C_TRACEOUT_INTERFACE_CDA_SYSTEM,          /**< Interface 6# */
      CSM_C_TRACEOUT_INTERFACE_CSM_SYSTEM,          /**< Interface 7# */
      CSM_C_TRACEOUT_INTERFACE_CDA_CONTROLLER,      /**< Interface 8# and CDA_CCC internal */
      CSM_C_TRACEOUT_INTERFACE_CDA_TRANSCEIVER,     /**< Interface 9# and CDA_TTT internal */
      CSM_C_TRACEOUT_INTERFACE_OS,                  /**< CSM OS abstraction. timer, task trigger, ... */
      CSM_C_TRACEOUT_CSM_INTERNAL                   /**< CSM internal */
    };

    // trace functions (Func) (better described as trace events)
    enum
    {
      // CDA
      CSM_C_TRACE_FKT_CDA_SEND                     = 0x01,
      CSM_C_TRACE_FKT_CXX_CDA_RECEIVE_IND,
      CSM_C_TRACE_FKT_CXX_CDA_DATA_CON,
      CSM_C_TRACE_FKT_CDA_INIT,
      CSM_C_TRACE_FKT_CDA_ACTIVE,
      CSM_C_TRACE_FKT_CDA_DEACTIVE,
      CSM_C_TRACE_FKT_CDA_EXIT,
      CSM_C_TRACE_FKT_CDA_CAN_INTERRUPTS_DISABLE,
      CSM_C_TRACE_FKT_CDA_CAN_INTERRUPTS_RESTORE,
      CSM_C_TRACE_FKT_CDA_BUSOFF_ENTRY,
      CSM_C_TRACE_FKT_CDA_BUSOFF_RECOVERY,
      CSM_C_TRACE_FKT_CDA_GET_ERROR_STATE,
      CSM_C_TRACE_FKT_CSM_CDA_ERROR_IND,
      CSM_C_TRACE_FKT_CSM_CDA_WAKEUP,
      // CSM
      CSM_C_TRACE_FKT_CSM_COMMUNICATION_CON        = 0x20,
      CSM_C_TRACE_FKT_CSM_COMMUNICATION_IND,
      CSM_C_TRACE_FKT_CSM_COMMUNICATION_REQ,
      #if (CSM_S_REMOTE_STACK == CSM_C_F_ON)
        CSM_C_TRACE_FKT_CSM_COMMUNICATION_CON_REJ,
        CSM_C_TRACE_FKT_CSM_COMMUNICATION_CON_TIME,
      #endif
      // communication related
      CSM_C_TRACE_FKT_CSM_GET_COMMUNICATION_STATE = 0x25,
      CSM_C_TRACE_FKT_CSM_COMMUNICATION_REQ_RETVAL,
      CSM_C_TRACE_FKT_CSM_APPL_CALLBACK_INIT,
      CSM_C_TRACE_FKT_CSM_ERROR_IND,
      #if (CSM_S_DEACTIVATE_COMMUNICATION_REQUESTS == CSM_C_F_ON)
        CSM_C_TRACE_FKT_CSM_DEACTIVATE_COMMUNICATION_REQ,
        CSM_C_TRACE_FKT_CSM_ACTIVATE_COMMUNICATION_REQ,
      #endif
      CSM_C_TRACE_FKT_CSM_GET_ALL_SIGNALS_AGAIN   = 0x2B,
      // CBR signals
      CSM_C_TRACE_FKT_CSM_SIGNAL_READ             = 0x30,
      CSM_C_TRACE_FKT_CSM_SIGNAL_WRITE,
      CSM_C_TRACE_FKT_CSM_SIGNAL_IND,
      CSM_C_TRACE_FKT_CSM_SIGNAL_CALLBACK_INIT,
      CSM_C_TRACE_FKT_CSM_MULTIPLE_SIGNAL_READ,
      CSM_C_TRACE_FKT_CSM_MULTIPLE_SIGNAL_WRITE,
      CSM_C_TRACE_FKT_CSM_SIGNAL_CALLBACK_DEINIT_ALL,     // 0x36

      // Data related
      CSM_C_TRACE_FKT_CSM_DATA_REQ                = 0x40,
      CSM_C_TRACE_FKT_CSM_DATA_CON,
      CSM_C_TRACE_FKT_CSM_DATA_IND,
      CSM_C_TRACE_FKT_CSM_DATA_IND_PROCESSED,
      CSM_C_TRACE_FKT_CSM_DATA_READ,
      CSM_C_TRACE_FKT_CSM_DATA_REQ_RETVAL,
      CSM_C_TRACE_FKT_CSM_DATA_IND_FF,
      CSM_C_TRACE_FKT_CSM_DATA_ERROR_IND,

      // Timer related
      CSM_C_TRACE_FKT_CSM_TIMER_START             = 0x50,
      CSM_C_TRACE_FKT_CSM_TIMER_STOP,
      CSM_C_TRACE_FKT_CSM_TIMER_EXPIRED,
      CSM_C_TRACE_FKT_CSM_TIMER_CYCLIC_START,
      CSM_C_TRACE_FKT_CSM_TIMER_CYCLIC_STOP,

      // Init, Exit
      CSM_C_TRACE_FKT_CSM_INIT                    = 0x60,
      CSM_C_TRACE_FKT_CSM_EXIT,                           //0x61
      CSM_C_TRACE_FKT_CSM_APL_CALLBACK_PREINIT,           //0x62

      // system checks
      CSM_C_TRACE_FKT_CSM_SET_PROCESSORLOAD       = 0x66, /**< Command for setting processor load within kernel upper tester */
      CSM_C_TRACE_FKT_CSM_GET_SYSTEMRESOURCE      = 0x67, /**< Command for getting system resources within kernel upper tester */
      CSM_C_TRACE_FKT_CSM_SET_TEST_NUMBER,                /**< Command to path test number from CANoe server to the CSM stack so that it will be traced out. */
      CSM_C_TRACE_FKT_CSM_TRACE_TEST_NUMBER,              /**< Test number will be traced out. */
      CSM_C_TRACE_FKT_TSTRETVAL                   = 0x6F,

      // Bus state
      CSM_C_TRACE_FKT_CSM_BUSSTATE_CHANGED        = 0x80,
      CSM_C_TRACE_FKT_CSM_CALC_BUSSTATE,
      CSM_C_TRACE_FKT_CSM_CHANGE,

      // NVM related
      CSM_C_TRACE_FKT_CSM_LOCAL_NVM_SET           = 0x95,
      CSM_C_TRACE_FKT_CSM_LOCAL_NVM_GET           = 0x96,
      CSM_C_TRACE_FKT_CSM_NVN_WRITE_FAIL          = 0x97,
      CSM_C_TRACE_FKT_CSM_NVM_WRITE_PENDING       = 0x98,
      CSM_C_TRACE_FKT_CSM_NVN_RETRIGGER           = 0x99,
      CSM_C_TRACE_FKT_CSM_KDS_MIRROR_GET          = 0x9A,

      // CBR start, stop
      CSM_C_TRACE_FKT_CBR_STOP_RX                 = 0xA0,
      CSM_C_TRACE_FKT_CBR_STOP_TX,
      CSM_C_TRACE_FKT_CBR_START_RX,
      CSM_C_TRACE_FKT_CBR_START_TX,

      CSM_C_TRACE_FKT_CSM_SIGNAL_SUPPRESS         = 0xB0,

      // special cases translated in csm_p.trc
      #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
        #if (CSM_S_OSEK_NM == CSM_C_F_ON)
          CSM_C_TRACE_FKT_CNM_PROCESS_NM_STATE       = 0xC0,
          CSM_C_TRACE_FKT_CNM_UNDERVOLTAGE           = 0xC1,
        #endif

        #if (CSM_S_OSEKI_NM == CSM_C_F_ON)
          CSM_C_TRACE_FKT_CNM_OSEKI_RX_NODE_MONITOR  = 0xC0,
          CSM_C_TRACE_FKT_CNM_OSEKI_TX_MUTE_MONITOR  = 0xC1,
          CSM_C_TRACE_FKT_CNM_OSEKI_BUS_OFF_MONITOR  = 0xC2,
          CSM_C_TRACE_FKT_CNM_OSEKI_EVENT            = 0xC3,
        #endif

        #if (CSM_S_FIATSLAVE_NM == CSM_C_F_ON)
          CSM_C_TRACE_FKT_CNM_PROCESS_NM_STATE       = 0xC0,
        #endif
      #endif // CSM_S_CNM_AVAILABLE

      // The area 0xE0 .. 0xEF is reserved for project specific traces.
      CSM_C_TRACE_FKT_PROJECT_E0                     = 0xE0,

      CSM_C_TRACE_FKT_CSM_DUMMY                      = 0xF0,
      CSM_C_TRACE_FKT_CSM_STRING                     = 0xF1,
      CSM_C_TRACE_FKT_CSM_UPPERTST                   = 0xF2, // This trace input command starts/stops the upper tester in kernel or user mode.
      CSM_C_TRACE_FKT_CSM_SOURCEID                   = 0xF3, // For GMLAN NM - reads out the current source id list and traces it out
      CSM_C_TRACE_FKT_DOWNSTREAM_TRACE               = 0xF4, // CSM_C_DOWN_REASON_TRACE

      CSM_C_TRACE_FKT_MPDT_JOIN_CHECK_HELPER         = 0xF5, // Trigger a certain MPDT join check helper functionality
      CSM_C_TRACE_FKT_IMX_CONFIG_CHECK_HELPER        = 0xF6  // Trigger a certain config check helper functionality
    };

    // required for CSM_C_DOWN_REASON_TRACE:
    enum
    {
      CSM_C_DOWNSTREAM_TRACE_OVERRUN = 1,
      CSM_C_DOWNSTREAM_TRACE_UUDT_RX_QUEUE_OVERFLOW
    };
  #endif  // CSM_S_TRACE_ETG
#else // CSM_S_TRACE
  // no trace active -> empty makro
  #define CSM_M_TRACEOUT(wTraceOutLevel, wTraceOutInterface, wTraceOutFunktion, pbTraceOutData, wTraceOutDataLength)
#endif // CSM_S_TRACE

//*******************************************************************************
//* typedefs (scope: csm stack local)
//*******************************************************************************
//*******************************************************************************
//  CCCC  SSSSS  MM   MM
// C      S      M M M M
// C       SSS   M  M  M        Types
// C          S  M     M
//  CCCC  SSSSS  M     M
//*******************************************************************************
// Queue
typedef struct
{
  BYTE  bMaxElements;
  BYTE  bElementSize;
  BYTE  bWritePointer;
  BYTE  bReadPointer;
  BYTE *pbData;
}tCSM_QUEUE;

// communication indication functions
typedef struct
{
  void (*pfvComInd) (BYTE, DWORD, const void*, BYTE); /* Communication Indication */
} CSM_COMMUNICATION_IND;

// csm state machine state change indication
typedef struct
{
  void (*pvFunctionCall)(BYTE ubBus, BYTE dwCsmActState);
} tCSM_APL_STATE_CHANGE_IND;

// CAN Rx overrun indication
typedef struct
{
  void (*pvFunctionCall)(BYTE ubBus);
} tCSM_APL_OVERRUN_IND;

// #ToDo: ???
typedef void (*tCSM_PFN_GET_TX_ID) (DWORD *,BYTE *, BYTE, BYTE);

//******************************************************************************
//  CCCC  DDDD   AAAAA
// C      D   D  A   A
// C      D   D  AAAAA          Types
// C      D   D  A   A
//  CCCC  DDDD   A   A
//******************************************************************************
#if (CSM_S_CDA_GATEWAY == CSM_C_F_ON)
typedef struct
{
  BYTE bBus;                                   /**< CSM Bus                           */
  DWORD dwIdentifier;                          /**< Standard oder Enhanced Identifier */
  BYTE  bTelegramType;                         /**< Telegrammtyp                      */
  BYTE  bTelegramIndex;                        /**< Listenindex fr den Telegrammtyp  */
  BYTE  bDivider ;                             /**< Teiler fuer Zyklus-Reduzierung    */
} tCDA_GATEWAY_CFG;
#endif // CSM_S_CDA_GATEWAY

#if (CSM_S_CDA_GATEWAY_PREDEF == CSM_C_F_ON)
typedef struct
{
  BYTE bBus;                                   /**< CSM Bus                           */
  DWORD dwIdentifier;                          /**< Standard oder Enhanced Identifier */
  BYTE  bTelegramType;                         /**< Telegrammtyp                      */
  BYTE  bTelegramIndex;                        /**< Listenindex fr den Telegrammtyp  */
}tCDA_GATEWAY_PREDEF_CFG;
#endif // CSM_S_CDA_GATEWAY_PREDEF

// Struct is used in XXX_vCDAReceiveInd() indication function from CDA
typedef struct
{
  DWORD dwIdentifier;   /**< Standard or Enhanced Identifier */
  BYTE  abData[8];      /**< CAN Datenbytes (array) */
  BYTE  bTelegramType;  /**< Telegrammtyp */
  BYTE  bTelegramIndex; /**< Listenindex fr jew. Telegrammtyp,
                         *   Bedeutung abhngig vom jew. Protokoll */
  CSM_BITFIELD_TYPE_8 bBus:2;         /**< CAN Bus Nummer (0-3) */
  CSM_BITFIELD_TYPE_8 bLengthCode:4;  /**< Data Length Code (0-8) */
  CSM_BITFIELD_TYPE_8 fNError:1;      /**< (optional) Zustand des NError Pin am Transceiver */
} CSM_CAN_RX_DATA;

// Struct is used in CDA_lSend() function to CDA
typedef struct
{
  DWORD dwIdentifier;                 /**< Standard or Enhanced Identifier */
  const BYTE * pbData;                /**< CAN Datenbytes (pointer) */
  BYTE  bTelegramType;                /**< Telegrammtyp */
  BYTE  bTelegramIndex;               /**< Listenindex fr jew. Telegrammtyp, Bedeutung abhngig vom jew. Protokoll */
  CSM_BITFIELD_TYPE_8 bBus:2;         /**< CAN Bus Nummer (0-3) */
  CSM_BITFIELD_TYPE_8 bLengthCode:4;  /**< Data Length Code (0-8) */

  #if (CSM_S_CDA_TRANSMIT_TIMEOUT == CSM_C_F_ON)
  DWORD dwTimeout;                    /**< Timeout for CAN Controller TX repetition */
  #endif

  #if (CSM_S_CDA_TEL_SUB_TYPE_SUPPORT == CSM_C_F_ON)
  BYTE bTelegramSubType;              /**< Telegrammuntertyp bei gleichem Haupttyp  */
  #endif
} CSM_CAN_TX_DATA;

// Struct is used in XXX_vCDADataCon() confirmation function from CDA
typedef struct
{
  DWORD dwIdentifier;                       /**< Standard or Enhanced Identifier */
  BYTE  bStatus;                            /**< CAN confirmation status         */
  BYTE  bTelegramType;                      /**< Telegrammtyp                    */
  BYTE  bTelegramIndex;                     /**< Listenindex fr jew. Telegrammtyp, Bedeutung abhngig vom jew. Protokoll */
  CSM_BITFIELD_TYPE_8 bBus:2;               /**< CAN Bus Nummer (0-3)            */

  #if (CSM_S_CDA_TEL_SUB_TYPE_SUPPORT == CSM_C_F_ON)
  BYTE  bTelegramSubType;
  #endif
} CSM_CAN_CON_DATA;

// Struct to store data in CDA
typedef struct
{
  DWORD dwIdentifier;
  BYTE  bBus;                         /**< Nummer des Verwendeten CAN Bus            */
  BYTE  bLengthCode;
  BYTE  bTelegramType;                /**< Zuordnung zu einem TP oder NM Protokoll   */
  BYTE  bTelegramIndex;
  BYTE  abDaten[8];

  #if (CSM_S_CDA_TRANSMIT_TIMEOUT == CSM_C_F_ON)
  DWORD dwTimeout;                    /**< Timeout for CAN Controller TX repetition  */
  #endif

  #ifdef ONLY_OPTION
  BYTE  bMsgObjNum;
  BYTE  bExtraInfo;
  CSM_BITFIELD_TYPE_8 fNERR:1;        /**< RX: Zustand des NERR Pin am Transceiver   */
  #endif

  #if (CSM_S_CDA_TEL_SUB_TYPE_SUPPORT == CSM_C_F_ON)
  BYTE  bTelegramSubType;
  #endif
} tCDA_CAN_DATA;

// data structure for hardware list (CDA configuration)
typedef CSM_M_ROM_TABLE struct _CSM_HW_LIST
{
  BYTE   bObjectType;         /**< possible value #CSM_C_HW_LIST_OBJECT_TYPE_ #CSM_C_HW_LIST_OBJECT_TYPE_EMPTY, #CSM_C_HW_LIST_OBJECT_TYPE_RX_FULL_CAN_1 and ...*/
  union
  {
    BYTE   bTelegramType;     /**< possible value #CSM_C_TEL_TYPE_MCNET_WD_RX, #CSM_C_TEL_TYPE_CBR_RX, and ... */
    BYTE   bBasicTabIndex;    /**< index for csm_aprRxIdBasicList[] */
  }Id1;                       // V.K.: compiler do not accept annonymous union
  union
  {
    BYTE   bTelegramIndex;    /**< TelegramIndex */
    BYTE   bBasicTabLength;   /**< length of csm_aprRxIdBasicList[ xx.Id1.bBasicTabIndex]*/
  }Id2;                       // V.K.: compiler do not accept annonymous union

  BYTE   bMaskId;             /**< Index to #csm_adwHWCanMask[]. Possible value #CSM_C_HW_LIST_MASK_0, #CSM_C_HW_LIST_MASK_1, ...*/
  DWORD  dwCanId;             /**< Standard/Enhanced Identifier */

  #if (CSM_S_CDA_DISCARD_MSG_SHORTER_DLC == CSM_C_F_ON) || (CSM_S_CDA_DISCARD_MSG_LONGER_DLC == CSM_C_F_ON)
  BYTE   bExpectedDLC;        /**< DLC expected to be received */
  #endif
} CSM_HW_LIST;

// central data structure table for Basic Can Rx telegrams
typedef CSM_M_ROM_TABLE struct
{
  DWORD  dwCanId;                            /**< Standard/Enhanced Identifier */
  BYTE   bTelegramType;                      /**< telegram-type, s. Tabelle 2  */
  BYTE   bTelegramIndex;                     /**< Index number in table        */

  #if (CSM_S_CDA_MASK_CAN_ID == CSM_C_F_ON)
  DWORD  dwMaskCanId;                        /**< mask out not care bit of Standard/Enhanced Identifier  1 -> care bit, 0 -> not care bit */
  #endif

  #if (CSM_S_CDA_DISCARD_MSG_SHORTER_DLC == CSM_C_F_ON) || (CSM_S_CDA_DISCARD_MSG_LONGER_DLC == CSM_C_F_ON)
  BYTE   bExpectedDLC;                       /**< DLC expected to be received  */
  #endif
} CSM_RX_ID_LIST;


//******************************************************************************
//  CCCC  BBBB   RRRR
// C      B   B  R   R
// C      BBBB   RRRR        Types
// C      B   B  R   R
//  CCCC  BBBB   R   R
//******************************************************************************
// typedef for a potential conversion function, CSM internally used only
#if (CSM_S_CBR_AVAILABLE == CSM_C_F_ON)
  // TX telegrams
  typedef struct
  {
    BYTE      bBus;                  /**< Bus on which this msg shall be transmitted */
    DWORD     dwCanId;               /**< identifier of telegram */
    BYTE      bDlc;                  /**< dlc of telegram */
    void*     pvBuffer;              /**< pointer to telegram data buffer */
    BYTE      bMessageType;          /**< message type:
                                          a) CBR_TMSG_CYCLIC: cyclic
                                          b) CBR_TMSG_CYCLIC_IF_ACTIVE: CyclicIfActive
                                          c) CBR_TMSG_SPONTAN: spontan
                                          d) CBR_TMSG_SPONTAN_WITH_DELAY: SpontanWithDelay
                                          e) CBR_TMSG_CYCLIC_AND_SPONTAN: CyclicAndSpontan
                                          f) CBR_TMSG_CYC_AND_SPONT_WITH_DEL: CyclicAndSpontanWithDelay
                                          g) CBR_TMSG_SPONTAN_WITH_REP: SpontanWithRepetition
                                          h) CBR_TMSG_CYC_IF_ACT_AND_SPONT: CyclicIfActiveAndSpontan */
    WORD      wCycleTime;            /**< cycle time of cyclic telegrams in ms, for a,b,e,f,h */
    BYTE      bNumberOfRepetitions;  /**< number of repetitions, for g */
    WORD      wMinDist;              /**< minimum distance for event messages in cyclic periods in ms, for c,d,f,g,h */
    WORD      wDelay;                /**< for delay types in ms, for d,f*/

    #if (CSM_S_VECTOR_TXOBJ_USED == CSM_C_F_ON)
    BYTE      bMsgObjNum;            /**< Number of message object.    */
    #endif
  } CSM_CBR_TX;

  // RX telegramms
  typedef struct
  {
    BYTE      bBus;               /**< Bus on which this msg shall be received */
    DWORD     dwCanId;            /**< identifier of telegram */
    BYTE      bDlc;               /**< dlc of telegram */
    void*     pvBuffer;           /**< pointer to telegram data */
    BYTE      bMessageType;       /**< message type, see below */
    WORD      wTimeoutTime;       /**< timeout time for cyclic messages in ms, for message types a,b,e,f,h */

    #if (CSM_S_CBR_RX_TIMEOUT_GROUPS == CSM_C_F_ON)
    BYTE      bRxGroup;           /**< Partial disabled timeout supervision support. */
    #endif
  } CSM_CBR_RX;

  // communication confirmation
  typedef struct
  {
     void (CSM_CALLBACK *pfvCommConf) (BYTE, DWORD, const void*, BYTE, WORD);    /**< Appl. callback fct. Connect.con */
     DWORD     dwPTYpe;            /**< Protocoll Type */
     DWORD     AddrField;          /**< addressField */
  } CSM_CBR_COMM_CONF;

  // CBR configuration
  typedef struct
  {
    BYTE bBus;                                 /**< CSM Bus */
    DWORD dwProtocolType;                      /**< CSM Protokoll Type */
    BYTE au8AddressField[ 2];                  /**< CSM Address Feld   */
    WORD wApplId;                              /**< CSM Appl Id   */

    #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
      #if (CSM_S_VIRTUAL_NETWORK == CSM_C_F_ON)
      BYTE bVirtualNetwork;                    /**< CSM coressponding Virtual Network. Byte representation as used in the VNMF frame. */
      #endif
      #if (CSM_S_OSEK_NM == CSM_C_F_ON)
        #if (CNM_OSEK_S_PAYLOAD == CSM_C_F_ON)
        DWORD dwNmPayload;                     /**< payload for network management */
        #endif
      #endif /* CSM_S_OSEK_NM == CSM_C_F_ON */
      #if (CSM_S_VWHIGH_NM == CSM_C_F_ON)
      DWORD dwNmPayload;                       /**< payload for network management */
      #endif
    #endif // CSM_S_CNM_AVAILABLE
  } tCSM_CBR_CFG;

  // rolling counter configuration within a TX message, usually instantiated as an array
  #if (CSM_S_CBR_TX_MSG_ROLL_COUNTER == CSM_C_F_ON)
  typedef struct
  {
    DWORD dwSignalId;                           /**< Signal Id within the telegramm  */
    BYTE  bUpperThreshold;                      /**< max. Counter value =! 0         */
    BYTE  bLowerThreshold;                      /**< min. Counter value              */
    BYTE  bGroupAndLen;                         /**< group nr. and Len of group -> evaluated if CSM_S_CBR_TX_MSG_ROLL_COUNTER_GROUPS == CSM_C_F_ON     */
    BYTE  bLinkIndex;                           /**< next/first group index (ring) -> evaluated if CSM_S_CBR_TX_MSG_ROLL_COUNTER_GROUPS == CSM_C_F_ON  */
  } CSM_TX_ROLLCOUNTER_CFG;

  typedef struct
  {
    BYTE  bRollCount;                           /**< current Roll Counter            */
    #if (CSM_S_CBR_TX_MSG_ROLL_COUNTER_GROUPS == CSM_C_F_ON)
    BYTE  bGroupHandled;                        /**< Marker, that correcponding group has been handled for 1 TX cycle */
    BYTE  bGroupImDelay;                        /**< Marker, that corresponding group is entirely delayed */
    #endif
  } CSM_TX_ROLLCOUNTER_ADMIN;
  #endif

  // TX CRC configuration, usually instantiated as an array
  #if (CSM_S_CBR_TX_MSG_CRC == CSM_C_F_ON)
  typedef struct
  {
    DWORD dwSignalId;                           /**< Signal Id within the telegramm  */
    BOOL  fCrcActive;                           /**< CRC generation is active        */
    #if (CSM_S_CBR_TX_MSG_CRC_TYPE_RN == CSM_C_F_ON)
    BYTE  bCrcIndex;                               /**< byte index where to put the CRC */
    #endif
  } CSM_TX_CRC_CFG;
  #endif

  // rolling counter configuration within a RX message, usually instantiated as an array
  #if (CSM_S_CBR_RX_MSG_ROLL_COUNTER == CSM_C_F_ON)
  typedef struct
  {
    DWORD dwSignalId;                           /**< Signal Id within the telegramm  */
    BYTE  bUpperThreshold;                      /**< max. Counter value =! 0         */
    BYTE  bLowerThreshold;                      /**< min. Counter value              */
  } CSM_RX_ROLLCOUNTER_CFG;
  #endif

  // RX CRC configuration, usually instantiated as an array
  #if (CSM_S_CBR_RX_MSG_CRC == CSM_C_F_ON)
  typedef struct
  {
    DWORD dwSignalId;                           /**< Signal Id within the telegramm  */
    BOOL  fCrcActive;                           /**< CRC check is active             */
    BYTE  bDataLength;                          /**< len                             */
    #if (CSM_S_CBR_RX_MSG_CRC_TYPE_RN == CSM_C_F_ON)
    BYTE  bCrcIndex;                               /**< byte index where to get the CRC from */
    #endif
  } CSM_RX_CRC_CFG;
  #endif

  // signal based gateway rule
  #if (CSM_S_CBR_GATEWAY == CSM_C_F_ON)
  typedef struct tyCBR_GATEWAY_RULE
  {
    DWORD dwSrcSignalId;
    DWORD dwDestSignalId;
    BYTE  bRuleType;
    WORD  wTransformerIndex;
  } tyCBR_GATEWAY_RULE;

  // signal based gateway configuration, usually instantiated as an array
  typedef struct tyCBR_GATEWAY_CFG
  {
    DWORD dwFrameSignalId;
    const tyCBR_GATEWAY_RULE * parRules;
    BYTE  bLenOfRules;
  } tyCBR_GATEWAY_CFG;
  #endif // CSM_S_CBR_GATEWAY
#endif // CSM_S_CBR_AVAILABLE

//******************************************************************************
// L       CCCC  TTTTT  RRRR   L
// L      C        T    R   R  L
// L      C        T    RRRR   L        Types
// L      C        T    R   R  L
// LLLLL   CCCC    T    R   R  LLLLL
//******************************************************************************
#if (CSM_S_CSM_LOCAL_CTRL == CSM_C_F_ON)
typedef struct
{
  BYTE bBus;                                              /**< CSM Bus            */
  DWORD dwProtocolType;                                   /**< CSM Protokoll Type */
  WORD wApplId;                                           /**< CSM Appl Id        */
  #if (CSM_S_CSM_LOCAL_CTRL_USER_FUNCTION == CSM_C_F_ON)
    BOOL (* pfCheckAwakeFunction) (BYTE);
    void (* pvProcessNewAction) (BYTE);
  #endif
}tCSM_CSM_CFG;
#endif // CSM_S_CSM_LOCAL_CTRL

//******************************************************************************
//  CCCC  N   N  PPPP
// C      NN  N  P   P
// C      N N N  PPPP           Types
// C      N  NN  P
//  CCCC  N   N  P
//******************************************************************************
#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
  // J1939 ---------------------------------------------------------------------
  #if (CSM_S_J1939 == CSM_C_F_ON)
  typedef struct
  {
    BYTE bBus;              /**< CSM Bus */
    BYTE bNodeID;           /**< J1939 node */
    DWORD dwProtocolType;   /**< CSM Protokoll Type */
    WORD wApplId;           /**< CSM Appl Id */
  }tCSM_J1939_APPL_CFG;

  typedef struct
  {
    BYTE bAddr;             /**< address of the network node */
  }tCSM_J1939_ADDR;

  typedef struct            /**< NAME of a network node */
  {
    DWORD bIdentNum:21;
    DWORD bManuCode:11;
    DWORD bECUInst:3;
    DWORD bFuncInst:5;
    DWORD bFunc:8;
    DWORD bReserved:1;
    DWORD bVehSys:7;
    DWORD bVehSysInst:4;
    DWORD bIndGrp:3;
    DWORD bAAC:1;
  }tCSM_J1939_NAME;

  typedef struct
  {
    union tID
    {
      tCSM_J1939_NAME rName;  /**< NAME to ID of a node */
      BYTE abData[8];
    }ID;
  }tCSM_J1939_NODE;

  typedef struct
  {
    BYTE  bBus:2;           /**< bus of the PGN */
    BYTE  bPrio:3;          /**< priority of the PGN */
    #if (CSM_S_J1939_STARTSTOP_BROADCAST == CSM_C_F_ON)
    BYTE  fMutable:1;       /**< PGN is affected by DM13 */
    #endif
    DWORD dwPGN:24;         /**< PGN */
    WORD  wInterframe;      /**< interframe space used for this PGN (if multipacketed) */
    WORD  wMaxTxLen;        /**< maximum supported length for this PGN */
    WORD  wTxTimerID;
    WORD  wMaxRxLen;        /**< maximum supported length for this PGN */
    WORD  wRxTimerID;
    BYTE  bCCSNPChannel;    /**< defines the NP channel for CCS module */
  }tCSM_J1939_PGN_CFG;

  typedef struct
  {
    enum
    {
      CSM_C_J1939_TXSTATE_IDLE = 0,
      CSM_C_J1939_TXSTATE_BAM,
      CSM_C_J1939_TXSTATE_RTS,
      CSM_C_J1939_TXSTATE_PDU,
      CSM_C_J1939_TXSTATE_PREPSEND,
    } enState;

    BYTE  bTgtAddr;         /**< address of the target of this PDU */
    BYTE  bSrcAddr;         /**< address of the source of this PDU */
    BYTE* pbBuffer;         /**< pointer to assembly buffer for incoming PDUs */
    WORD  wBufferLen;       /**< length of the received PDU */
    BYTE  bConsFrames;      /**< number of frames until CM_CTS is required */
    WORD  bNextFrame;       /**< number of next expected/send frame */
    BYTE  bAllFrames;
  }tCSM_J1939_PGN_TXDATA;

  typedef struct
  {
    enum
    {
      CSM_C_J1939_RXSTATE_IDLE = 0,
      CSM_C_J1939_RXSTATE_BAM,
      CSM_C_J1939_RXSTATE_RTS,
      CSM_C_J1939_RXSTATE_CTS,
      CSM_C_J1939_RXSTATE_PDU,
      CSM_C_J1939_RXSTATE_EOM,
    } enState;

    BYTE  bTgtAddr;         /**< address of the target of this PDU */
    BYTE  bSrcAddr;         /**< address of the source of this PDU */
    BYTE* pbBuffer;         /**< pointer to assembly buffer for incoming PDUs */
    WORD  wBufferLen;       /**< length of the received PDU */
    BYTE  bConsFrames;      /**< number of frames until CM_CTS is required */
    WORD  bNextFrame;       /**< number of next expected/send frame */
    BYTE  bAllFrames;
  }tCSM_J1939_PGN_RXDATA;

  typedef struct
  {
    tCSM_J1939_PGN_TXDATA rTxData;
    tCSM_J1939_PGN_RXDATA rRxData;
  }tCSM_J1939_PGN_DATA;

  typedef struct
  {
    BYTE  fSendReq:1;
    BYTE  fRecvInd:1;
    BYTE  fRecvIndFF:1;
    BYTE  fTxTimeoutInd:1;
    BYTE  fRxTimeoutInd:1;
    BYTE  fErrorInd:1;
    BYTE  fDataCon:4;
  }tCSM_J1939_PGN_IND_FLAGS;
  #endif // CSM_S_J1939

  // MCNet -----------------------------------------------------------------
  #if (CSM_S_MCNET == CSM_C_F_ON)
  // MCNet logical components
  typedef struct tagCSM_MCNet_LOGICAL_COMPONENT
  {
    #ifdef TSI
    TCHAR szLc[9];       /**< Name of logical component                   */
    #endif
    BYTE  bLc;           /**< Number of   -- "" --                        */
    WORD  wCanId;        /**< CAN ident. for WD and "standard Broadcast"  */
  }CSM_MCNet_LOGICAL_COMPONENT;

  // MCNet local component list
  typedef struct tagCSM_MCNet_LOC_COMPLIST
  {
    #ifdef TSI
    TCHAR szLc[9];
    #endif
    BYTE    bLc;
    BOOLEAN fWdSendFlag;
    BOOLEAN fWdMonFlag;
  }CSM_MCNet_LOC_COMPLIST;

  // MCNet local and remote master
  typedef struct tagCSM_MCNet_MASTER
  {
    #ifdef TSI
    TCHAR szLc[9];        /**< Name of logical component                   */
    #endif
    BYTE     bBus;           /**< Bus number                                  */
    BYTE     bMasterLC;      /**< Number of  -- "" --                         */
    DWORD    dwCanId;        /**< CAN ident. for WD and "standard-Broadcast"  */
    BOOLEAN  fWdEnableFlag;  /**< Flag for WD sending or monitoring           */
  }CSM_MCNet_MASTER;

  // MCNet broadcast channels
  typedef struct tagCSM_MCNet_USDT_TX_CHANNEL
  {
    #ifdef TSI
    TCHAR szName[9];          /**< Symbolic channel name                  */
    TCHAR szLc[9];            /**< Name of logical component (sender)     */
    #endif
    BYTE  bSenderLC;          /**< Number of   -- ""---                   */
    BYTE  bBus;               /**< Bus number                             */
    DWORD dwCanId;            /**< CAN identifier                         */
    BYTE  bTDB;               /**< Transmit Delay Timer Broadcast [ms]    */
    WORD  wMaxTxLen;          /**< Max. length tx application message     */
    void (CSM_CALLBACK *pfvUSDTDataCon)(DWORD, const void*, BYTE );  /**< Appl. callback fct. Data.con */
  }CSM_MCNetUSDT_TX_CHANNEL;

  typedef struct tagCSM_MCNet_USDT_RX_CHANNEL
  {
    #ifdef TSI
    TCHAR szName[9];          /**< Symbolic channel name                  */
    TCHAR szLc[9];            /**< Name of logical component (sender)     */
    #endif
    BYTE  bSenderLC;          /**< Number of   -- ""---                   */
    BYTE  bBus;               /**< Bus number                             */
    DWORD dwCanId;            /**< CAN identifier                         */
    WORD  wMaxRxLen;          /**< Max. length rx application message     */
    void (CSM_CALLBACK *pfvUSDTDataInd) (DWORD, const void*, BYTE *, WORD); /**< Appl. callback fct. Data.ind  */
    #ifdef TSI /* Not used till now: */
    void (CSM_CALLBACK *pfvUSDTErrorInd)(BYTE, BYTE, BYTE); /**< Appl. callback fct. Error.ind  */
    #endif
  }CSM_MCNetUSDT_RX_CHANNEL;

  // MCNet ASDT connection table (ROM part)
  typedef struct tagCSM_MCNet_CONT
  {
    #ifdef TSI
    TCHAR szLc[2][9];             /**< Name of log. component 1 & 2                 */
    #endif
    BYTE    bBus;                 /**< Number of CAN bus                            */
    BYTE    bLocalLC;             /**< Local MCNet component number                 */
    BYTE    bRemoteLC;            /**< Remote MCNet component number                */
    DWORD   dwLocalId;            /**< TX identifier local component                */
    DWORD   dwRemoteId;           /**< TX identifier remote component               */
    WORD    wMaxTxLen;            /**< Max. TX application message length           */
    WORD    wMaxRxLen;            /**< Max. RX application message length           */
    BYTE    bSMConfigured;        /**< Configured Service Mode (STANDARD/ENHANCED)  */
    BYTE    bBSConfigured;        /**< Configured Block Size (1-15)                 */
    BYTE    bTD;                  /**< Transmit Delay Timer Value [ms]              */
    BYTE    bNMType;              /**< Master NM or Slave NM                        */

    #if (CSM_S_MCNET_APPL_TX_BUFFER_USED == CSM_C_F_ON)
    /** CSM uses the Tx buffer of the application. The application is not allowed to
     * write to the buffer until YYY_vCSMDataCon().
     * Configurable for each channel (fApplTxBufferUsed = TRUE). Value for wMaxTxLen
     * must be defined by application.
     */
    BOOLEAN fApplTxBufferUsed;
    #endif

    #if (CSM_S_MCNET_APPL_SETS_RX_BUFFER_FREE == CSM_C_F_ON)
    /** If fApplSetsRxBufferFree is TRUE (configurable for each channel) then the
     * application is responsible to free the Rx buffer after YYY_vCSMDataInd().
     * See CSM_lDataIndProcessed() */
    BOOLEAN fApplSetsRxBufferFree;
    #endif

    #if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_OFF)
    void (CSM_CALLBACK *pfvASDTDataCon)(DWORD, const void*, BYTE );        /**< Appl. callback fct. Data.con    */
    void (CSM_CALLBACK *pfvASDTDataInd)(DWORD, const void*, BYTE *, WORD); /**< Appl. callback fct. Data.ind    */
    void (CSM_CALLBACK *pfvASDTCommCon) (BYTE, DWORD, const void*, BYTE, WORD);  /**< Appl. callback fct. Communication.con */
    void (CSM_CALLBACK *pfvASDTCommInd) (BYTE, DWORD, const void*, BYTE, WORD);  /**< Appl. callback fct. Communication.ind */
    #endif

    #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
      #if (CSM_S_OSEK_NM == CSM_C_F_ON)
        #if (CNM_OSEK_S_PAYLOAD == CSM_C_F_ON)
        DWORD dwNmPayload;                     /**< payload for network management */
        #endif
      #endif // CSM_S_OSEK_NM
    #endif // CSM_S_CNM_AVAILABLE
  }CSM_MCNet_CONT;

  // MCNet ASDT connection table (RAM part)
  typedef volatile struct tagCSM_MCNet_CONT_ACTUAL
  {
    BYTE    bSMActual;             /**< Actual (= negotiated) Service Mode           */
    BYTE    bBSActual;             /**< Actual (= negotiated Block Size              */
    BYTE    bNMIdx;                /**< Index of local NM instance                   */
    BOOLEAN fConTestFlag;          /**< Connection Test flag                         */

    #if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
    BOOLEAN fApplicationInit;
    #endif
    BYTE    bReqAction;            /**< last requested action from CSM_MCNET_lCommunicationReq()
                                   possible values CSM_C_CONNECT and CSM_C_DISCONNECT.
                                   CSM_C_CONNECT keep the bus alive. */
  }CSM_MCNet_CONT_ACTUAL;
  #endif // CSM_S_MCNET

  // ISO TP ---------------------------------------------------------------------------
  #if (CSM_S_ISO_TP == CSM_C_F_ON)
  typedef struct
  {
    DWORD   dwProtocolType;  /**< CSM Protokoll Type */
    BYTE    bBus;            /**< Number of CAN bus       */
    BYTE    bAddrMode;       /**< Addressing mode (Normal/Extended/Ext.Mux) */

    #if (CSM_S_ITP_ASDT_EXTENSIONS == CSM_C_F_ON)
    BYTE    bItpType;        /**< USDT or ASDT versions   */   /* U.Z.(3) Neu, fuer Test */
    #endif

    #if (CSM_S_ICCC == CSM_C_F_ON)
    BYTE    bConnectType;    /**< CSM_C_ITP_CON_TYPE_REGULAR, ICCC or OWN_NODE channel:
                              * connection setup handling is different. */
    #endif

    #if (CSM_S_ITP_APPL_TX_BUFFER_USED == CSM_C_F_ON)
    /** CSM uses the Tx buffer of the application. The application is not allowed to
    * write to the buffer until YYY_vCSMDataCon().
    * Configurable for each channel (fApplTxBufferUsed = TRUE). Value for wMaxTxLen
    * must be defined by application.
    */
    BOOLEAN fApplTxBufferUsed;
    #endif

    #if (CSM_S_ITP_APPL_SETS_RX_BUFFER_FREE == CSM_C_F_ON)
    /** If fApplSetsRxBufferFree is TRUE (configurable for each channel) then the
    * application is responsible to free the Rx buffer after YYY_vCSMDataInd().
    * See CSM_lDataIndProcessed() */
    BOOLEAN fApplSetsRxBufferFree;
    #endif
    BYTE     bLocalComp;     /**< Local component number  */
    BYTE     bRemoteComp;    /**< Remote component number (not used for Ext./Ext.Mux,
                              * Value in Tx-/Assemblybuffer is used there) */
    union
    {
      DWORD  dwLocalId;      /**< TX Id for local component (normal and ext. addressing) */
      DWORD  dwIdBase;       /**< Base TX/RX Id (for multiplexed extended addressing)
                              * TX CAN Id = dwIdBase + bLocalComp
                              * TX Addr. (2. PCI Byte) = NetId + bRemoteComp */  /* V.K.: Net Id statt dwBase oder sogar nur NET ID */
    }Id1;                    /* V.K.: compiler do not accept annonymous union */

    union
    {
      DWORD  dwRemoteId;     /**< TX Id for remote component (Normal and ext. addressing)
                            * = RX Id for local component */
      DWORD  dwIdRange;      /**< TX/RX Id range (for multiplexed extended addressing)
                              * local RX CAN Id = dwIdBase masked with dwIdRange
                              * RemoteId = RxCanId - dwIdBase
                              * local RX Addr. (2. PCI Byte) = NetId + bLocalComp */
    }Id2;                    /* V.K.: compiler do not accept annonymous union */
    BYTE bBSRXConfigured;    /**< Konfig. RX-Blocksize (0-255) */
    BYTE bSTMinRXConfigured; /**< Konfig. STMin (ms) fr Empfang */
    WORD wMaxTxLen;          /**< Max. number of bytes for TX message */
    WORD wMaxRxLen;          /**< Max. number of bytes for RX message */

    #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
      #if (CSM_S_OSEK_NM == CSM_C_F_ON)
        #if (CNM_OSEK_S_PAYLOAD == CSM_C_F_ON)
        DWORD dwNmPayload;   /**< payload for network management */
        #endif
      #endif // CSM_S_OSEK_NM
      #if (CSM_S_VWHIGH_NM == CSM_C_F_ON)
      DWORD dwNmPayload;    /**< payload for network management */
      #endif // CSM_S_VWHIGH_NM
    #endif // CSM_S_CNM_AVAILABLE */
  }CSM_ISO_TP_CONT;

  typedef struct
  {
    BYTE bBSTXActual;                           /**< Aktuelle (d.h. ausgehandelte) TX-Blocksize */
    BYTE bSTMinTXActual;                        /**< Aktuelle (d.h. ausgehandelte) STMin fr Senden */
    CSM_BITFIELD_TYPE_8  bfConnectionState : 4; /**< Actual state of the connection (derived from NM etc.)*/
    CSM_BITFIELD_TYPE_8  fTxChannelFree    : 1; /**< FALSE after DataReq with no error return value,
                                                TRUE directly before DataCon (and after init) */
    CSM_BITFIELD_TYPE_8  bfReserved        : 3; /**< Fill to 8 bit */

  }CSM_ISO_TP_CONT_ACTUAL;

  typedef struct
  {
    tCSM_ISO_TP_USDT_APPL_CALLBACK rCallback;  /**< Structure to store the callbacks passed from application to CSM via CSM_lApplCallbackInit() */
    #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
    void * pvHandle;                           /**< Handle is used for AccessLayer. */
    #endif
    #if (CSM_S_USE_TKSE == CSM_C_F_ON)
    tCSM_TASK_INFO rTaskInfo;
    #endif
  }tCSM_ISO_TP_USDT_APPL_CALLBACK_ADMIN;       /**< ISO-TP channel administration data for application
                                                * callback functions. Access Layer configuration during callback init. */
  #endif // ISO TP

  // MPDT ---------------------------------------------------------------------------
  #define CSM_MPDT_C_MAX_IL                        2  /**< Number of possible Interrupt Level (IL)*/

  // CAN telegram data storage
  typedef struct
  {
    DWORD dwIdentifier;         /**< CAN indentifyer */
    BYTE  bBus;                 /**< Nummer des Verwendeten CAN Bus */
    BYTE  bLengthCode;          /**< Data length */
    BYTE  bTelegramType;        /**< Zuordnung zu einem TP oder NM Protokoll */
    BYTE  bTelegramIndex;       /**< Index number */
    BYTE  abDaten[8];           /**< data */
    #if (CSM_S_CDA_TEL_SUB_TYPE_SUPPORT == CSM_C_F_ON)
    BYTE  bTelegramSubType;     /**< Untertyp von Telegramm bei gleichem Haupttyp */
    #endif
  }CNP_MPDT_tCAN_DATA;

  typedef struct
  {
    BYTE  bBus;           /**< Bus number */
    DWORD dwProtocolType; /**< Protocoll type */
    BYTE  bLocalComp;     /**< Local component id. */
    BYTE  bRemoteComp;    /**< Remote component id. */
    DWORD dwLocalId;      /**< TX identifier local component  */
    DWORD dwRemoteId;     /**< TX identifier remote component */
    WORD  wRPS;           /**< maximum Reception Packet Size (6<= RPS <= 4094) */
    WORD  wTPS;           /**< maximum Transmission Packet Size */
    BYTE  bTTmin;         /**< Transmission separation Time Minimum (time between frames) (0<=TTmin<=10)*/

    BYTE  bBSRXConfigured;    /**< Konfig. RX-Blocksize (0-255) */
    BYTE  bSTMinRXConfigured; /**< Konfig. STMin (ms) fr Empfang */

    WORD  wP1Timer;       /**< Timer number for P1 timer */
    WORD  wP2Timer;       /**< Timer number for P2 timer */
    WORD  wP3Timer;       /**< Timer number for P3 timer */

    BYTE  *apbTxMessageBuffer[ CSM_MPDT_C_MAX_IL]; /**< Pointer to transmit message buffer */
    WORD  awTxMessageMaxLength[ CSM_MPDT_C_MAX_IL];/**< Maximum length of transmit message buffer */
    BYTE  *apbTxMessagePacketBuffer[ CSM_MPDT_C_MAX_IL];/**< Pointer to transmit message packet buffer */
    BYTE  *pbTxPacketBuffer;/**< Pointer to transmit packet buffer */

    BYTE  *apbRxMessageBuffer[ CSM_MPDT_C_MAX_IL]; /**< Pointer to receive message buffer */
    WORD  awRxMessageMaxLength[ CSM_MPDT_C_MAX_IL];/**< Maximum length of receive message buffer */

    BYTE  *pbRxCMessageBuffer; /**< Pointer to receive control message buffer. Used by CSM_MPDT to store the data before indicate the data to application */
    WORD  wRxCMessageMaxLength;/**< Maximum length of receive control message buffer */
    BYTE  *pbRxDMessageBuffer; /**< Pointer to receive data message buffer. Used by CSM_MPDT to store the data before indicate the data to application */
    WORD  wRxDMessageMaxLength;/**< Maximum length of receive data message buffer */

    BYTE  *pbRxPacketBuffer;/**< Pointer to receive packet buffer */
    WORD  wRxPacketMaxLength;/**< Maximum length of receive data packet buffer */

    WORD  wTxTimer;       /**< Timer number for transmit timer */
    WORD  wRxTimer;       /**< Timer number for receive timer */
    WORD  wTTminTimer;    /**< Timer number for TTmin timer */

    WORD  wTxTelegramQueueMaxLength;
    CNP_MPDT_tCAN_DATA *prTxTelegramQueue;
  }tCSM_MPDT_CONFIG;      /**< CSM MPDT configuration */

  // UUDT ---------------------------------------------------------------------------
  #if (CSM_S_UUDT_TX == CSM_C_F_ON)
  // GM Gen1 project needs old incompatible struct, gen2 the new one. CSM_C_PTYPE_GM_UUDT_TX used in Gen1 and Gen2.
  #if 0
  typedef struct
  {
    BYTE bBus;                                 /**< CSM bus */
    DWORD dwProtocolType;                      /**< CSM protocol Type */
    WORD awAddressField[1];                    /**< CSM UUDT Tx address */
  }tCSM_UUDT_TX_CFG;                           /* for UUDT TX configuration */
  #endif

  typedef struct
  {
    BYTE bBus;                                 /**< CSM bus */
    DWORD dwProtocolType;                      /**< CSM protocol Type */
    WORD wAddress;                             /**< CSM UUDT Tx address */
  }tCSM_UUDT_TX_CFG;                           /* for UUDT TX configuration */

  typedef struct
  {
    tCSM_UUDT_TX_APPL_CALLBACK rCallback;  /**< Structure to store the callbacks passed from application
                                            *   to CSM via CSM_lApplCallbackInit() */
    /* handle is used for AccessLayer */
    #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
    void * pvHandle;                       /**< Handle is used for AccessLayer. */
    #endif

    #if (CSM_S_USE_TKSE == CSM_C_F_ON)
    tCSM_TASK_INFO rTaskInfo;
    #endif
  }tCSM_UUDT_TX_APPL_CALLBACK_ADMIN;   /**< UUDT TX channel administration data for application
                                        *   callback functions. Access Layer configuration during
                                        *   callback init. */
  #endif // CSM_S_UUDT_TX

  #if (CSM_S_UUDT_RX == CSM_C_F_ON)
  // GM Gen1 project needs old incompatible struct, gen2 the new one. CSM_C_PTYPE_GM_UUDT_RX used in Gen1 and Gen2.
  #if 0
  typedef struct
  {
    BYTE bBus;                                 /**< CSM bus */
    DWORD dwProtocolType;                      /**< CSM protocol type */
    WORD awAddressField[1];                    /**< CSM UUDT Rx address field */
  } tCSM_UUDT_RX_CFG;                          /* for UUDT RX configuration */
  #endif

  typedef struct
  {
    BYTE bBus;                                 /**< CSM bus */
    DWORD dwProtocolType;                      /**< CSM protocol type */
    WORD wAddress;                             /**< CSM UUDT Rx address */
  }tCSM_UUDT_RX_CFG;                           /* for UUDT RX configuration */

  typedef struct
  {
    tCSM_UUDT_RX_APPL_CALLBACK rCallback;  /**< Structure to store the callbacks passed from application
                                            *   to CSM via CSM_lApplCallbackInit() */
    #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
    void * pvHandle;                       /**< Handle is used for AccessLayer. */
    #endif

    #if (CSM_S_USE_TKSE == CSM_C_F_ON)
    tCSM_TASK_INFO rTaskInfo;
    #endif
  }tCSM_UUDT_RX_APPL_CALLBACK_ADMIN;       /**< UUDT RX channel administration data for application
                                            *   callback functions. Access Layer configuration during
                                            *   callback init. */
  #endif // CSM_S_UUDT_RX

  // VW TP 1.6 -------------------------------------------------------------------------
  #if ( CSM_S_VW_TP_1_6 == CSM_C_F_ON )
  typedef struct
  {
    BYTE bLocalComp;          /**< Local component id.                          */
    BYTE bRemoteComp;         /**< Remote component id.                         */
    BYTE bBus;                /**< Bus number.                                  */
    BYTE bTpHandle;           /**< Vector connection TP API-handle.             */
    WORD wTxToInitValue;      /**< Tx timeout init value (0 = no timeout).      */

    BYTE *pabTxBuffer;        /**< Transmit buffer.                             */
    WORD wTxBufferSize;       /**< Transmit buffer size.                        */

    BYTE *pabRxBuffer;        /**< Receive buffer.                              */
    WORD wRxBufferSize;       /**< Receive buffer size.                         */

    // call back functions
    void (CSM_CALLBACK *pfvDataCon)(DWORD, const void*, BYTE );               /**< Appl. callback fct. Data.con    */
    void (CSM_CALLBACK *pfvDataInd)(DWORD, const void*, BYTE *, WORD);        /**< Appl. callback fct. Data.ind    */
    void (CSM_CALLBACK *pfvCommCon) (BYTE, DWORD, const void*, BYTE, WORD);   /**< Appl. callback fct. Connect.con */
    void (CSM_CALLBACK *pfvCommInd) (BYTE, DWORD, const void*, BYTE, WORD);   /**< Appl. callback fct. Connect.ind */
  }CSM_VW_TP16_CONT;

  typedef struct
  {
    struct
    {
      CSM_BITFIELD_TYPE_8 Connect    :1; /**< Connect request.                   */
      CSM_BITFIELD_TYPE_8 Disconnect :1; /**< Disconnect request.                */
      CSM_BITFIELD_TYPE_8 TxData     :1; /**< Transfer Data request.             */

      // Internal aux. requests                                                  */
      CSM_BITFIELD_TYPE_8 CallConf   :1; /**< Generate YYY_lCommunicationCon().  */
    }stReq;

    BYTE bCurrentState;         /**< Current state.                              */
    WORD wTxTOTimer;            /**< Transmit timeout timer.                     */

    // TP watchdog (workarounds)                                                 */
    WORD wRNRWDCnt;             /**< Receiver not ready (RNR) watchdog counter.  */
    WORD wTxWDCnt;              /**< Tx watchdog timer.                          */

    WORD wTxDataLength;         /**< Data length of current tx data.             */
    const BYTE *pabTxBuffer;    /**< Pointer to current tx buffer.               */

    BYTE bTpDDC;                /**< Data Direction Change (fast/slow).          */
    WORD wTpRxOffset;           /**< Receive buffer current write position.      */
    WORD wLength;               /**< Length of received data.                    */
  }CSM_VW_TP16_STAT;
  #endif // CSM_S_VW_TP_1_6

  // VW TP 2.0 -------------------------------------------------------------------
  #if ( CSM_S_VW_TP_2_0 == CSM_C_F_ON )
  typedef struct
  {
    BYTE bLocalComp;          /**< Local component id.                          */
    BYTE bRemoteComp;         /**< Remote component id.                         */
    BYTE bBus;                /**< Bus number.                                  */
    BYTE bAPI;                /**< VW TP 2.0 application type.                  */
    BYTE bConnType;           /**< Static or dynamic channel.                   */
    BYTE bPartnerID;          /**< PartnerID for vector driver.                 */
    #if (CSM_S_VW_TP20_MULTI_ECU == CSM_C_F_ON)
    BYTE bTpAddr;             /**< ECU-Addr for Multi ECU support               */
    #endif

    BYTE *pabTxBuffer;        /**< Transmit buffer.                             */
    WORD wTxBufferSize;       /**< Transmit buffer size.                        */

    BYTE *pabRxBuffer;        /**< Receive buffer.                              */
    WORD uwRxBufferSize;      /**< Receive buffer size.                         */

    // callback functions
    WORD (CSM_CALLBACK *pfvDirectConnInd)(BYTE, BYTE**);
    #if (CSM_S_VW_TP20_PARA_EXCHG == CSM_C_F_ON)
    void (CSM_CALLBACK *pfvPara)(DWORD, const void*, BYTE );                 /**< Appl. callback fct. ParamSet    */
    #endif
    void (CSM_CALLBACK *pfvDataCon)(DWORD, const void*, BYTE );              /**< Appl. callback fct. Data.con    */
    void (CSM_CALLBACK *pfvDataInd)(DWORD, const void*, BYTE *, WORD);       /**< Appl. callback fct. Data.ind    */
    void (CSM_CALLBACK *pfvCommCon) (BYTE, DWORD, const void*, BYTE, WORD);  /**< Appl. callback fct. Connect.con */
    void (CSM_CALLBACK *pfvCommInd) (BYTE, DWORD, const void*, BYTE, WORD);  /**< Appl. callback fct. Connect.ind */
  }CSM_VW_TP20_CONT;

  typedef struct
  {
    struct
    {
      CSM_BITFIELD_TYPE_8 ConState : 4;   /**< Current state of connection.      */
      CSM_BITFIELD_TYPE_8 RxState  : 2;   /**< Current rx substate.              */
      CSM_BITFIELD_TYPE_8 TxState  : 2;   /**< Current tx substate.              */
    }fCurrState;

    struct
    {
      CSM_BITFIELD_TYPE_8 ConConReq     : 1;  /**< Send connect confirmation
                                               * if con. state change into
                                               * CONNECT or DISCONNECT.          */
      CSM_BITFIELD_TYPE_8 SendConWithDC : 1;  /**< Send confirmation with
                                               * CSM_C_DISCONNECTED.             */
      CSM_BITFIELD_TYPE_8 SendConWithCO : 1;  /**< Send confirmation with
                                               * CSM_C_CONNECTED.                */
      CSM_BITFIELD_TYPE_8 ConReady      : 1;  /**< Connect ready
                                               *         / not ready.  (d7)      */
    }fAuxFlags;

    BYTE bHandle;                             /**< Handle for vector API.        */
    WORD wLength;                             /**< Length of received data.      */
  }CSM_VW_TP20_STAT;

  // BR service type definitions
  typedef struct
  {
    BYTE ubLastKey;                           /**< Last received key.            */
    BYTE TimeOutInvKey;                       /**< Max. time to get inv. key.    */

    // Flags
    BYTE ApplInformed  :1;                    /**< Set if application was
                                               * informed in current sequence.   */
  }CSM_VW_TP20_BRSERV_STAT;

  typedef struct
  {
    BYTE ubSID;                               /**< Service identifier.           */
                                              /** Pointer to CB-function.        */
    void (CSM_CALLBACK *pfvBrDataInd) (DWORD, const void*, BYTE*, WORD);
    BYTE aubAddress[2];                       /**< AddressField (CSM). */
    BOOL fRetriggerRequired;                  /** Service needs to be retriggered */
  }CSM_VW_TP20_BRSERV_CONT;
  #endif // CSM_S_VW_TP_2_0 enabled
#endif // CSM_S_CNP_AVAILABLE

// VW BAP (Transportprotocol) ---------------------------------------------------
#if ( CSM_S_VW_BAP_USE_ACKNOWLEDGE == CSM_C_F_ON )
typedef struct
{
  BYTE                    bLsgId;       /* LSG-ID */
  BYTE                    bFctId;       /* FCT-ID */
} tCSM_BAP_AckSearchTable;
#endif

//*******************************************************************************
//  CCCC  N   N  MM   MM
// C      NN  N  M M M M
// C      N N N  M  M  M        Types
// C      N  NN  M     M
//  CCCC  N   N  M     M
//*******************************************************************************
#if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
  #if (CSM_S_GMLAN_NM == CSM_C_F_ON)
  typedef struct
  {
    BYTE bBus;                                 /**< CSM Bus */
    DWORD dwProtocolType;                      /**< CSM Protokoll Type */
    BYTE au8AddressField[ 2];                  /**< CSM Address Feld   */
    WORD wApplId;                              /**< CSM Appl Id   */
  } tCSM_GMLAN_CFG;
  #endif // CSM_S_GMLAN_NM
#endif // CSM_S_CNM_AVAILABLE

// NWM configuration
typedef struct
{
  BYTE bBus;                                   /**< CSM Bus */
  DWORD dwProtocolType;                        /**< CSM Protokoll Type */
  BYTE au8AddressField[ 2];                    /**< CSM Address Feld   */
  WORD wApplId;                                /**< CSM Appl Id   */

  #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
    #if (CSM_S_GMLAN_NM == CSM_C_F_ON)
    BYTE bVirtualNetwork;                      /**< CSM coressponding Virtual Network  */
    #endif /* CSM_S_GMLAN_NM */

    #if (CSM_S_OSEK_NM == CSM_C_F_ON)
      #if (CNM_OSEK_S_PAYLOAD == CSM_C_F_ON)
      DWORD dwNmPayload;                       /**< payload for network management     */
      #endif
    #endif

    #if (CSM_S_OSEKI_NM == CSM_C_F_ON)
    BYTE bMonitorIndex;                        /**< index to node/bus specific monitor */
    #endif
  #endif // CSM_S_CNM_AVAILABLE
}tCSM_NWM_CFG;

// data structure for FIAT NM
typedef struct
{
  DWORD dwCanId;                               /**< CAN identifier */
  BYTE  bDlc;                                  /**< dlc of telegram */
  void* pvBuffer;                              /**< pointer to telegram data buffer */
}CSM_FIAT_NM_TX;

//******************************************************************************
// TTTTT  I  MM   MM  EEEEE  RRRR
//   T    I  M M M M  E      R   R
//   T    I  M  M  M  EEEE   RRRR       Types
//   T    I  M     M  E      R   R
//   T    I  M     M  EEEEE  R   R
//******************************************************************************
typedef struct
{
  void (*pvFunctionCall)(void);
} tCSM_TIMER_CALLBACK;

typedef struct
{
  void (*pvFunctionCall)(DWORD);
} tCSM_TIMER_CALLBACK_NEU;

typedef void (*tCSM_TIMER_CALLBACK_NEW) ( DWORD);

typedef struct
{
  tCSM_TIMER_CALLBACK_NEW   vpfnCallback;
  DWORD dwValue;
}tCSM_TIMER;

//******************************************************************************
// MM   MM  I  SSSSS   CCCC
// M M M M  I  S      C
// M  M  M  I   SSS   C            Types
// M     M  I      S  C
// M     M  I  SSSSS   CCCC
//******************************************************************************
// BKA2HI: the following code was disabled, after having replaced the define by a newer one.
// -> since it did not compile properly: to be removed if not needed
#if 0
#ifdef CSM_STACK_ENV_SELF
/* Definitions for CAN HW Test for V850 projects */
typedef struct
{
  void (*pvFunctionCall)(GLB_TY_InterfaceStruct * ptyInterfaceStruct);
} tCSM_CanTestHandler;

void CSM_vCanTestLowSpeed(GLB_TY_InterfaceStruct * ptyInterfaceStruct);
void CSM_vCanTestHighSpeed(GLB_TY_InterfaceStruct * ptyInterfaceStruct);
#endif
#endif

//------------------------------------------------------------------------------
// we need the following typedef struct in csm_apl.c and cda.c and cnm_oseki.c
#if (defined CSM_APL_SELF) || (defined CDA_SELF) || (defined CNM_OSEKI_SELF) || (defined CSM_PROXY_SELF)
#if (CSM_S_VARIANTS_MANAGEMENT == CSM_C_F_ON)
// data structure for configuration SLOTS *********************************
#if (CSM_S_SLOT_BASED_CONFIG == CSM_C_F_ON)
struct _CNM_OSEKI_tBusConfig;                               // forward declaration needed for cnm_oseki.c build
                                                            // because there cnm_oseki_m.h is not included before but after this include file
typedef const struct
{
  const CSM_HW_LIST * pCanHwConfigTable;                    // pointer to a CAN HW configuration table

  const CSM_RX_ID_LIST * const * pCanRxIdList;              // pointer to an array of pointers to "RX Multiple Data Basic Objects"
  const CSM_CBR_RX * pRxTelegramDescriptorTable;            // pointer to an array of RX Telegram Descriptors
  const CSM_TELEGRAM * const * pRxSignalDescriptorTable;    // pointer to an array of pointers to RX Signal Descriptors

  const CSM_CBR_TX * pTxTelegramDescriptorTable;            // pointer to an array of TX Telegram Descriptors
  const CSM_TELEGRAM * const * pTxSignalDescriptorTable;    // pointer to an array of pointers to TX Signal Descriptors

  const BYTE bCbrRxMax;                                     // number of max. RX messages (CBR)
  const BYTE bCbrTxMax;                                     // number of max. TX messages (CBR)

  #if (CSM_S_OSEKI_NM == CSM_C_F_ON)
  const struct _CNM_OSEKI_tBusConfig  * pOsekiBusConfig;    // pointer to a OSEKI Bus Config, here not using the type but the tagname of the struct
  #endif

  #if (CSM_S_CBR_TX_MSG_ROLL_COUNTER == CSM_C_F_ON)
  const CSM_TX_ROLLCOUNTER_CFG * pTxRollingCounterCfg;      // pointer to configuration for the TX rolling counter
  #endif

  #if (CSM_S_CBR_RX_MSG_ROLL_COUNTER == CSM_C_F_ON)
  const CSM_RX_ROLLCOUNTER_CFG * pRxRollingCounterCfg;      // pointer to configuration for the RX rolling counter
  #endif

  #if (CSM_S_CBR_TX_MSG_CRC == CSM_C_F_ON)
  const CSM_TX_CRC_CFG * cbr_pTxCrcCfg;                     // pointer to configuration for the TX CRC
  #endif

  #if (CSM_S_CBR_RX_MSG_CRC == CSM_C_F_ON)
  const CSM_RX_CRC_CFG * cbr_pRxCrcCfg;                     // pointer to configuration for the RX CRC
  #endif

  #if (CSM_S_CDA_GATEWAY == CSM_C_F_ON)
  const tCDA_GATEWAY_CFG * pCdaGatewayConfig;               // pointer to a CDA gateway configuration
  #endif

  #if (CSM_S_CBR_GATEWAY == CSM_C_F_ON)
  const tyCBR_GATEWAY_CFG * pCbrGatewayConfig;              // pointer to a CBR gateway configuration
  #endif

  // end of struct definition
}tCSM_SLOT_CONFIG;
#endif // CSM_S_SLOT_BASED_CONFIG
#endif // CSM_S_VARIANTS_MANAGEMENT
#endif // CSM_APL_SELF

//**************************************************************************************
//* project specific include after typedef of CSM_RX_ID_LIST and definitions of tables
//**************************************************************************************
#include "csm_stack_mp.h"

// check amount of CAN busses
#if (CSM_C_MAX_CAN_BUS > 4)
  #error "The struct CSM_CAN_RX_DATA supports only 4 busses !"
#endif

#ifndef CSM_M_GET_TASK_CONTEXT
  #define CSM_M_GET_TASK_CONTEXT(prTaskContext )                          // no process model is used
#endif

#ifndef CSM_M_SET_TASK_CONTEXT
  #define CSM_M_SET_TASK_CONTEXT(prTaskContext, pvCallback) pvCallback    // no process model is used
#endif

//**************************************************************************************
//* variable declaration (scope: csm stack local)
//**************************************************************************************
#if (CSM_S_BLOCKABLE_UPSTREAM_TASK == CSM_C_F_ON)
// suppress event handling in upstream task
extern BOOLEAN csm_fBlockUpstreamTask;
#endif

#if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
  #ifdef CSM_SELF
         BOOLEAN csm_fActivationFlag;
  #else
  extern BOOLEAN csm_fActivationFlag;
  #endif
#endif

/*------------------------------------------------------------------------------
 * upstream
 *
 * Trigger reasons and macros for CSM indication/upstream task (define single bits !)
 * If more then 16 reasons are necessary in one project, change variable
 * csm_dwIndicationReasonBitfield, parameter in csm_vTriggerCsmIndicationTask()
 * and makros below.
 * Project dependent macros are defined in csm_stack_mp.h and used as feature
 * switches.
 *
 *
 * not all defines are used in a project: renumber your used defines in csm_stack_mp.h
 * define CSM_C_IND_REASON x,y,z         0x0020, 0x0040 ...
 *
 *------------------------------------------------------------------------------
 * downstream
 *
 * Trigger reasons and macros for CSM downstream task (define single bits !)
 * If more then 32 reasons are necessary in one project, change variable
 * csm_dwIndicationReasonBitfield, parameter in csm_vTriggerCsmIndicationTask()
 * and makros below
 * Project dependent macros are defined in csm_stack_mp.h and used as feature
 * switches.
 * not all defines are used in a project: renumber your used defines in
 * csm_stack_mp.h if necessary.
 * Further comments see CSM_vDownstreamTask()
 *
 *
 */

#ifdef CSM_STACK_SELF
  // indication bitfield for upstream task
  DWORD csm_dwIndicationReasonBitfield;
  // indication bitfield for downstream task
  DWORD csm_dwDownReasonBitfield;
#else
  extern DWORD csm_dwIndicationReasonBitfield;
  extern DWORD csm_dwDownReasonBitfield;
#endif

/*#define CSM_C_IND_REASON_CBR_TX_MANAGER       0x0001  Broadcast server Tx manager */
/* define CSM_C_IND_REASON x,y,z         0x10, 0x20 ...*/

//**************************************************************************************
//*FUNCTION/MACRO: CSM_M_SET_IND_REASON, CSM_M_CLEAR_IND_REASON, CSM_M_CLEAR_DOWN_REASON
//*DESCRIPTION:    It may save ROM (depends on project), if these makros are
//*                replaced by functions.
//*
//*PARAMETER:      WORD IndReason or ReqReason
//*
//*RETURNVALUE:    void
//*
//*access level   (Init/interrupt/CAN task/main loop): all
//*reentrant:      yes
//*
//*HISTORY:
//*
//* 11.02.03 CM-DI/ESU3-Koehler
//* Initial Revision.
//**************************************************************************************
// Makros or functions for Upstream Task -----------------------------------------------
#if (0)
static void CSM_M_SET_IND_REASON(WORD IndReason)   // function instead of makro costs 16 Byte ROM (Opel)
{
  INT iIsrStat;

  iIsrStat = CSM_M_UPSTREAM_TASK_LOCK();           // lock semaphore or disable interrupts
  (csm_dwIndicationReasonBitfield |= IndReason);
  CSM_M_UPSTREAM_TASK_UNLOCK(iIsrStat);            // unlock semaphore or enable interrupts again
}
#endif

// MACRO to set an indication:
// - lock semaphore or disable interrupts
// - set indication bit
// - unlock semaphore or enable interrupts
#define CSM_M_SET_IND_REASON(IndReason) { INT iIsrStat = CSM_M_UPSTREAM_TASK_LOCK();    \
                                          (csm_dwIndicationReasonBitfield |= IndReason);\
                                          CSM_M_UPSTREAM_TASK_UNLOCK(iIsrStat);         \
                                        }

#if (CSM_S_USE_CLEAR_IND_REASON_FUNCTION == CSM_C_F_ON)
  static void CSM_M_CLEAR_IND_REASON(WORD IndReason) // function instead of makro saves 64 Byte ROM (Opel)
  {
    INT iIsrStat;

    iIsrStat = CSM_M_UPSTREAM_TASK_LOCK();           // lock semaphore or disable interrupts
    (csm_dwIndicationReasonBitfield &= ~IndReason);
    CSM_M_UPSTREAM_TASK_UNLOCK(iIsrStat);}           // unlock semaphore or enable interrupts again
  }
#else
  // MACRO to clear an indication:
  // - lock semaphore or disable interrupts
  // - clear indication bit
  // - unlock semaphore or enable interrupts
  #define CSM_M_CLEAR_IND_REASON(IndReason) { INT iIsrStat = CSM_M_UPSTREAM_TASK_LOCK();        \
                                              (csm_dwIndicationReasonBitfield &= ~(IndReason)); \
                                              CSM_M_UPSTREAM_TASK_UNLOCK(iIsrStat);             \
                                            }
#endif // CSM_S_USE_CLEAR_IND_REASON_FUNCTION

#define CSM_M_IS_IND_REASON_SET(IndReason) (((csm_dwIndicationReasonBitfield & IndReason) == IndReason) ? TRUE : FALSE)

// Makros or functions for Downstream Task --------------------------------------------------------
#define CSM_M_SET_DOWN_REASON(ReqReason)    { INT iIsrStat = CSM_M_DOWNSTREAM_TASK_LOCK(); \
                                              (csm_dwDownReasonBitfield |= (ReqReason));   \
                                              CSM_M_DOWNSTREAM_TASK_UNLOCK(iIsrStat);      \
                                            }

#if (0)
static void CSM_M_SET_DOWN_REASON(WORD ReqReason) // function instead of makro costs 16 Byte ROM (Opel)
{
  INT iIsrStat;

  iIsrStat = CSM_M_DOWNSTREAM_TASK_LOCK();        // lock semaphore or disable interrupts
  (csm_dwDownReasonBitfield |= ReqReason);
  CSM_M_DOWNSTREAM_TASK_UNLOCK(iIsrStat);         // unlock semaphore or enable interrupts again
}
#endif

#define CSM_M_CLEAR_DOWN_REASON(ReqReason)  { INT iIsrStat = CSM_M_DOWNSTREAM_TASK_LOCK();                       \
                                              (csm_dwDownReasonBitfield &= (DWORD)(0xFFFFFFFF & (~(ReqReason))));\
                                              CSM_M_DOWNSTREAM_TASK_UNLOCK(iIsrStat);                            \
                                            }

#if (0)
static void CSM_M_CLEAR_DOWN_REASON(WORD ReqReason) // function instead of makro costs 4 Byte ROM (Opel)
{
  INT iIsrStat;

  iIsrStat = CSM_M_DOWNSTREAM_TASK_LOCK();          // lock semaphore or disable interrupts
  (csm_dwDownReasonBitfield &= ~ReqReason);
  CSM_M_DOWNSTREAM_TASK_UNLOCK(iIsrStat);           // unlock semaphore or enable interrupts again
}
#endif
#define CSM_M_IS_DOWN_REASON_SET(ReqReason) (((csm_dwDownReasonBitfield & ReqReason) == ReqReason) ? TRUE : FALSE)


//******************************************************************************
//  CCCC  N   N  PPPP
// C      NN  N  P   P
// C      N N N  PPPP           Variables and defines
// C      N  NN  P
//  CCCC  N   N  P
//******************************************************************************
#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
  // VW-TP -----------------------------------------------------------------------
  // TP 2.0
  #if (CSM_S_VW_TP_2_0 == CSM_C_F_ON )
    #if (CSM_S_VWTP20_IND_ON_ML_LEVEL == CSM_C_F_ON)
      #ifdef CSM_SELF
        DWORD csm_wConnInd;
        DWORD csm_wConnCon;
        DWORD csm_wConnStat;
        DWORD csm_wDataInd;
        DWORD csm_wDataCon;
        DWORD csm_wDataResult;
      #else
        extern DWORD csm_wConnInd;
        extern DWORD csm_wConnCon;
        extern DWORD csm_wConnStat;
        extern DWORD csm_wDataInd;
        extern DWORD csm_wDataCon;
        extern DWORD csm_wDataResult;
      #endif

      #define CSM_M_IS_VWTP20_CONNECT_IND_FLAG(ConNr)     (csm_wConnInd & (0x01U << ConNr))

      #define CSM_M_SET_VWTP20_CONNECT_IND_FLAG(ConNr)    { INT iIsrStat = CSM_iInterruptsDisable(); \
                                                            csm_wConnInd |= (1 << ConNr);            \
                                                            CSM_vInterruptsRestore(iIsrStat);        \
                                                          }

      #define CSM_M_CLR_VWTP20_CONNECT_IND_FLAG(ConNr)    { INT iIsrStat = CSM_iInterruptsDisable(); \
                                                            csm_wConnInd &= ~(1 << ConNr);           \
                                                            CSM_vInterruptsRestore(iIsrStat);        \
                                                          }

      #define CSM_M_IS_VWTP20_CONNECT_CON_FLAG_SET(ConNr) (csm_wConnCon & (0x01U << ConNr))

      #define CSM_M_SET_VWTP20_CONNECT_CON_FLAG(ConNr)    { INT iIsrStat = CSM_iInterruptsDisable(); \
                                                            csm_wConnCon |= (1 << ConNr);            \
                                                            CSM_vInterruptsRestore(iIsrStat);        \
                                                          }

      #define CSM_M_CLR_VWTP20_CONNECT_CON_FLAG(ConNr)    { INT iIsrStat = CSM_iInterruptsDisable(); \
                                                            csm_wConnCon &= ~(1 << ConNr);           \
                                                            CSM_vInterruptsRestore(iIsrStat);        \
                                                          }

      #define CSM_M_IS_VWTP20_CONNECT_STATE_FLAG_SET(ConNr) (csm_wConnStat & (0x01U << ConNr))

      #define CSM_M_SET_VWTP20_CONNECT_STATE_FLAG(ConNr)  { INT iIsrStat = CSM_iInterruptsDisable(); \
                                                            csm_wConnStat |= (1 << ConNr);           \
                                                            CSM_vInterruptsRestore(iIsrStat);        \
                                                          }

      #define CSM_M_CLR_VWTP20_CONNECT_STATE_FLAG(ConNr)  { INT iIsrStat = CSM_iInterruptsDisable(); \
                                                            csm_wConnStat &= ~(1 << ConNr);          \
                                                            CSM_vInterruptsRestore(iIsrStat);        \
                                                          }

      #define CSM_M_IS_VWTP20_DATA_IND_FLAG_SET(ConNr)    (csm_wDataInd & (0x01U << ConNr))

      #define CSM_M_SET_VWTP20_DATA_IND_FLAG(ConNr)       { INT iIsrStat = CSM_iInterruptsDisable(); \
                                                            csm_wDataInd |= (1 << ConNr);            \
                                                            CSM_vInterruptsRestore(iIsrStat);        \
                                                          }

      #define CSM_M_CLR_VWTP20_DATA_IND_FLAG(ConNr)       { INT iIsrStat = CSM_iInterruptsDisable(); \
                                                            csm_wDataInd &= ~(1 << ConNr);           \
                                                            CSM_vInterruptsRestore(iIsrStat);        \
                                                          }

      #define CSM_M_IS_VWTP20_DATA_CON_FLAG_SET(ConNr)    (csm_wDataCon & (0x01U << ConNr))

      #define CSM_M_SET_VWTP20_DATA_CON_FLAG(ConNr)       { INT iIsrStat = CSM_iInterruptsDisable(); \
                                                            csm_wDataCon |= (1 << ConNr);            \
                                                            CSM_vInterruptsRestore(iIsrStat);        \
                                                          }

      #define CSM_M_CLR_VWTP20_DATA_CON_FLAG(ConNr)       { INT iIsrStat = CSM_iInterruptsDisable(); \
                                                            csm_wDataCon &= ~(1 << ConNr);           \
                                                            CSM_vInterruptsRestore(iIsrStat);        \
                                                          }

      #define CSM_M_IS_VWTP20_DATA_RESULT_FLAG_SET(ConNr) (csm_wDataResult & (0x01U << ConNr))

      #define CSM_M_SET_VWTP20_DATA_RESULT_FLAG(ConNr)    { INT iIsrStat = CSM_iInterruptsDisable(); \
                                                            csm_wDataResult |= (1 << ConNr);         \
                                                            CSM_vInterruptsRestore(iIsrStat);        \
                                                          }

      #define CSM_M_CLR_VWTP20_DATA_RESULT_FLAG(ConNr)    { INT iIsrStat = CSM_iInterruptsDisable(); \
                                                            csm_wDataResult &= ~(1 << ConNr);        \
                                                            CSM_vInterruptsRestore(iIsrStat);        \
                                                          }
    #endif // CSM_S_VWTP20_IND_ON_ML_LEVEL

    extern const CSM_VW_TP20_CONT CSM_VwTp20Connection [CSM_C_VW_TP20_CONNECTIONS];
    extern volatile CSM_VW_TP20_STAT CSM_VwTp20Status[CSM_C_VW_TP20_CONNECTIONS];

    #if ( CSM_S_VW_TP20_BROADCAST_SERVICE == CSM_C_F_ON )
      extern const BYTE CNP_VW20_abSupportedTpTargets[CSM_C_VW_TP20_MAX_BR_TARGET_ADDR];
      extern const CSM_VW_TP20_BRSERV_CONT CSM_VwTp20BrServConnection[CSM_C_VW_TP20_BR_NR_OF_SERVICES];
      extern volatile CSM_VW_TP20_BRSERV_STAT CSM_VwTp20BrServStatus[CSM_C_VW_TP20_BR_NR_OF_SERVICES];
    #endif // CSM_S_VW_TP20_BROADCAST_SERVICE
  #endif // CSM_S_VW_TP_2_0

  // TP 1.6 -----------------------------------------------------------------------
  #if ( CSM_S_VW_TP_1_6 == CSM_C_F_ON )
    extern const CSM_VW_TP16_CONT CSM_VwTp16Connection [CSM_C_VW_TP16_CONNECTIONS];
    extern volatile CSM_VW_TP16_STAT CSM_VwTp16Status[CSM_C_VW_TP16_CONNECTIONS];
  #endif

  // MPDT -------------------------------------------------------------------------
  #if (CSM_S_MPDT == CSM_C_F_ON)
    // this is required to compile cnp_mpdt.c
    extern const tCSM_MPDT_CONFIG  CSM_MPDT_arConfig[CSM_MPDT_C_MAX_CONNECTIONS];
  #endif

  // ISO-TP -----------------------------------------------------------------------
  #if (CSM_S_ISO_TP == CSM_C_F_ON)
    // this is required to compile cnp_itp.c
    extern CSM_M_ROM_TABLE   CSM_ISO_TP_CONT           csm_rIsoTpConnection[];
    extern                   CSM_ISO_TP_CONT_ACTUAL    csm_rIsoTpConnectionAct[];

    #if (defined CNP_ITP_SELF) || (defined CNP_LIN_USDT_SELF)
      // Declare in csm_stack_mp.h const or not const
      #if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
      extern                 tCSM_ISO_TP_USDT_APPL_CALLBACK_ADMIN   csm_itp_arCallbackFkt[];
      #else
      extern CSM_M_ROM_TABLE tCSM_ISO_TP_USDT_APPL_CALLBACK_ADMIN   csm_itp_arCallbackFkt[];
      #endif
    #endif
  #endif

  // MCNet -----------------------------------------------------------------------
  #if (CSM_S_MCNET == CSM_C_F_ON)
    #ifdef CSM_APL_SELF
      #define STORAGE_CL
      // List of local connection masters
      STORAGE_CL  CSM_MCNet_MASTER  csm_rMCNetLocalMaster[CSM_C_MCNET_MAX_CONNECTIONS];
      // List of remote connection masters
      STORAGE_CL  CSM_MCNet_MASTER  csm_rMCNetRemoteMaster[CSM_C_MCNET_MAX_CONNECTIONS];
      // (CSM-) Connection Test States
      STORAGE_CL  BYTE              csm_abMCNetConTestState[CSM_C_MCNET_MAX_CONNECTIONS];

      // csm_rMCNetUsdtTxChannel[CSM_C_MCNET_MAX_CONNECTIONS] and csm_wMCNetUsdtTxChannels
      // need to be defined/initialized in csm_stack_mp.h/csm_apl.c
    #else // CSM_APL_SELF
      #define STORAGE_CL  extern

      STORAGE_CL CSM_MCNet_MASTER csm_rMCNetLocalMaster[];
      STORAGE_CL CSM_MCNet_MASTER csm_rMCNetRemoteMaster[];
      STORAGE_CL BYTE             csm_abMCNetConTestState[];

      #ifdef TSI
      STORAGE_CL                 CSM_MCNet_LOC_COMPLIST      csm_rMCNetLocCompList[];
      STORAGE_CL CSM_M_ROM_TABLE CSM_MCNet_LOGICAL_COMPONENT csm_rMCNet_LCT[];
      #endif // TSI

      // the following variables are defined and initialized in csm_stack_mp.h ifdef CSM_STACK_APL
      /*
      #if (CSM_S_MCNET == CSM_C_F_ON)  not used any longer
      STORAGE_CL                 tCSM_PWR_CALLBACK csm_rMCNetPwrCallback;
      #endif
      */
      #if (CSM_S_MCNET_ASDT == CSM_C_F_ON)
        STORAGE_CL                 WORD                     csm_wMCNetLogComponents;
        STORAGE_CL CSM_M_ROM_TABLE CSM_MCNet_CONT           csm_rMCNetConnection[];
        STORAGE_CL                 CSM_MCNet_CONT_ACTUAL    csm_rMCNetConnectionAct[];
      #endif

      #if (CSM_S_MCNET_USDT_TX == CSM_C_F_ON)
        STORAGE_CL CSM_M_ROM_TABLE CSM_MCNetUSDT_TX_CHANNEL csm_rMCNetUsdtTxChannel[];
        STORAGE_CL CSM_M_ROM_TABLE WORD                     csm_wMCNetUsdtTxChannels;
      #endif

      #if (CSM_S_MCNET_USDT_RX == CSM_C_F_ON)
      STORAGE_CL CSM_M_ROM_TABLE   CSM_MCNetUSDT_RX_CHANNEL csm_rMCNetUsdtRxChannel[];
      STORAGE_CL                   WORD                     csm_wMCNetUsdtRxChannels;
      #endif
    #endif // CSM_APL_SELF

    // definition and declaration
    STORAGE_CL WORD csm_wMCNetLocalMasters;   /**< Number of local master components  */
    STORAGE_CL WORD csm_wMCNetRemoteMasters;  /**< Number of remote master components */
    #undef STORAGE_CL
  #endif // CSM_S_MCNET

  // J1939 -----------------------------------------------------------------------
  #if (CSM_S_J1939 == CSM_C_F_ON)
    // declaration here, definition in csm_stack_mp.h
    // declare in csm_stack_mp.h const or not const
    #if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
      //#error "CSM_S_CALLBACK_INIT_BY_APPLICATION is not yet supported by CSM."
    #else
      extern CSM_M_ROM_TABLE tCSM_J1939_APPL_CALLBACK    csm_j1939_arApplCallbackFkt[];
      extern CSM_M_ROM_TABLE tCSM_J1939_PGN_CALLBACK     csm_j1939_arPGNCallbackFkt[];
    #endif // CSM_S_CALLBACK_INIT_BY_APPLICATION
  #endif // CSM_S_J1939
#endif // CSM_S_CNP_AVAILABLE

// ISO-TP -----------------------------------------------------------------------
#ifdef CSM_C_DOWN_REASON_ITP_DATA_REQ
  #ifdef CSM_SELF
    #if (CSM_C_ISO_TP_MAX_CONNECTIONS > 32)
      #error "CSM_dwItpDataRequestFlags only capable of 32 channels"
    #endif
    DWORD CSM_dwItpDataRequestFlags; /**< One bit for each ISO-TP channel where
                                      * the data request function has
                                      * to be called (max 32 channels) */
  #else
    extern DWORD CSM_dwItpDataRequestFlags;
  #endif

  #define CSM_M_SET_ITP_DATAREQUEST_FLAG(bChannel)    ( CSM_dwItpDataRequestFlags |= (DWORD)(0x00000001U << bChannel) )  /**< set data request bit for this channel */
  #define CSM_M_CLEAR_ITP_DATAREQUEST_FLAG(bChannel)  ( CSM_dwItpDataRequestFlags &= ~(0x00000001U << bChannel) )        /**< clear */
  #define CSM_M_IS_ITP_DATAREQUEST_FLAG_SET(bChannel) ( CSM_dwItpDataRequestFlags & (0x00000001U << bChannel) )          /**< check if flag is set */
#endif // CSM_C_DOWN_REASON_ITP_DATA_REQ

// ------------------------------------------------------------------------------
#ifdef CSM_C_IND_REASON_ITP_DATA_IND
  #ifdef CSM_SELF
    #if (CSM_C_ISO_TP_MAX_CONNECTIONS > 32)
      #error "CSM_dwItpIndicationFlags only capable of 32 channels"
    #endif

    DWORD CSM_dwItpIndicationFlags; /**< One bit for each ISO-TP channel where
                                     * the application indication function has
                                     * to be called (max 32 channels) */
    #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
    DWORD CSM_dwItpIndicationFfFlags; /**< One bit for each ISO-TP channel where
                                       * the application indication function has
                                       * to be called (max 32 channels) */
    #endif // CSM_S_ITP_FF_INDICATION_AVAILABLE

    #if (CSM_S_ITP_ERROR_INDICATION_AVAILABLE == CSM_C_F_ON)
    DWORD CSM_dwItpErrorIndicationFlags; /**< One bit for each ISO-TP channel where
                                          * the application indication function has
                                          * to be called (max 32 channels) */
    #endif // CSM_S_ITP_ERROR_INDICATION_AVAILABLE
  #else
    extern DWORD CSM_dwItpIndicationFlags;
    #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
      extern DWORD CSM_dwItpIndicationFfFlags;
    #endif // CSM_S_ITP_FF_INDICATION_AVAILABLE

    #if (CSM_S_ITP_ERROR_INDICATION_AVAILABLE == CSM_C_F_ON)
      extern DWORD CSM_dwItpErrorIndicationFlags;
    #endif // CSM_S_ITP_ERROR_INDICATION_AVAILABLE
  #endif // CSM_SELF

  #define CSM_M_SET_ITP_INDICATION_FLAG(bChannel)   { INT iIsrStat = CSM_iInterruptsDisable();                      \
                                                      CSM_dwItpIndicationFlags |= (DWORD)(0x00000001UL << bChannel);\
                                                      CSM_vInterruptsRestore(iIsrStat);                             \
                                                    }

  #define CSM_M_CLEAR_ITP_INDICATION_FLAG(bChannel) { INT iIsrStat = CSM_iInterruptsDisable();                      \
                                                      CSM_dwItpIndicationFlags &= ~(0x00000001UL << bChannel);      \
                                                      CSM_vInterruptsRestore(iIsrStat);                             \
                                                    }

  #define CSM_M_IS_ITP_INDICATION_FLAG_SET(bChannel)( CSM_dwItpIndicationFlags & (0x00000001UL << bChannel) )

  #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
    #define CSM_M_SET_ITP_INDICATION_FF_FLAG(bChannel)   { INT iIsrStat = CSM_iInterruptsDisable();                          \
                                                           CSM_dwItpIndicationFfFlags |= (DWORD)(0x00000001UL << bChannel);  \
                                                           CSM_vInterruptsRestore(iIsrStat);                                 \
                                                         }

    #define CSM_M_CLEAR_ITP_INDICATION_FF_FLAG(bChannel) { INT iIsrStat = CSM_iInterruptsDisable();                          \
                                                           CSM_dwItpIndicationFfFlags &= ~(0x00000001UL << bChannel);        \
                                                           CSM_vInterruptsRestore(iIsrStat);                                 \
                                                         }

    #define CSM_M_IS_ITP_INDICATION_FF_FLAG_SET(bChannel) ( CSM_dwItpIndicationFfFlags & (0x00000001UL << bChannel) )
  #endif // CSM_S_ITP_FF_INDICATION_AVAILABLE

  #if (CSM_S_ITP_ERROR_INDICATION_AVAILABLE == CSM_C_F_ON)
    #define CSM_M_SET_ITP_ERROR_INDICATION_FLAG(bChannel)   { INT iIsrStat = CSM_iInterruptsDisable();                            \
                                                              CSM_dwItpErrorIndicationFlags |= (DWORD)(0x00000001UL << bChannel); \
                                                              CSM_vInterruptsRestore(iIsrStat);                                   \
                                                            }

    #define CSM_M_CLEAR_ITP_ERROR_INDICATION_FLAG(bChannel) { INT iIsrStat = CSM_iInterruptsDisable();                            \
                                                              CSM_dwItpErrorIndicationFlags &= ~(0x00000001UL << bChannel);       \
                                                              CSM_vInterruptsRestore(iIsrStat);                                   \
                                                            }

    #define CSM_M_IS_ITP_ERROR_INDICATION_FLAG_SET(bChannel) ( CSM_dwItpErrorIndicationFlags & (0x00000001UL << bChannel) )
  #endif // CSM_S_ITP_ERROR_INDICATION_AVAILABLE
#endif // CSM_C_IND_REASON_ITP_DATA_IND

// ------------------------------------------------------------------------------
#ifdef CSM_C_IND_REASON_ITP_DATA_CON
  #ifdef CSM_SELF
    #if (CSM_C_ISO_TP_MAX_CONNECTIONS > 32)
      #error "CSM_dwItpConfirmationFlags only capable of 32 channels"
    #endif

    DWORD CSM_dwItpConfirmationFlags; /**< One bit for each ISO-TP channel where
                                       * the application confirmation function has
                                       * to be called (max 32 channels) */
    BYTE CSM_abItpConfirmationResults[CSM_C_ISO_TP_MAX_CONNECTIONS];
  #else
    extern DWORD CSM_dwItpConfirmationFlags;
    extern BYTE CSM_abItpConfirmationResults[];
  #endif // CSM_SELF

  #define CSM_M_SET_ITP_CONFIRMATION_FLAG(bChannel)    ( CSM_dwItpConfirmationFlags |= (DWORD)(0x00000001UL << bChannel) )

  #define CSM_M_CLEAR_ITP_CONFIRMATION_FLAG(bChannel)  { INT iIsrStat = CSM_iInterruptsDisable();                   \
                                                         CSM_dwItpConfirmationFlags &= ~(0x00000001UL << bChannel); \
                                                         CSM_vInterruptsRestore(iIsrStat);                          \
                                                       }

  #define CSM_M_IS_ITP_CONFIRMATION_FLAG_SET(bChannel) ( CSM_dwItpConfirmationFlags & (0x00000001UL << bChannel) )
#endif // CSM_C_IND_REASON_ITP_DATA_CON

// ------------------------------------------------------------------------------
#ifdef CSM_C_IND_REASON_ITP_COMM_IND
  #ifdef CSM_SELF
    #if (CSM_C_ISO_TP_MAX_CONNECTIONS > 32)
      #error "CSM_dwItpCommIndFlags only capable of 32 channels"
    #endif

    DWORD CSM_dwItpCommIndFlags; /**< One bit for each ISO-TP channel where
                                  * the application communication indication function has
                                  * to be called (max 32 channels) */
  #else
    extern DWORD CSM_dwItpCommIndFlags;
  #endif // CSM_SELF

  #define CSM_M_SET_ITP_COMM_IND_FLAG(bChannel)    ( CSM_dwItpCommIndFlags |= (DWORD)(0x00000001UL << bChannel) )

  #define CSM_M_CLEAR_ITP_COMM_IND_FLAG(bChannel)  { INT iIsrStat = CSM_iInterruptsDisable();              \
                                                     CSM_dwItpCommIndFlags &= ~(0x00000001UL << bChannel); \
                                                     CSM_vInterruptsRestore(iIsrStat);                     \
                                                   }

  #define CSM_M_IS_ITP_COMM_IND_FLAG_SET(bChannel) ( CSM_dwItpCommIndFlags & (0x00000001UL << bChannel) )  /**< check if flag is set */
#endif // CSM_C_IND_REASON_ITP_COMM_IND

// MCNet -----------------------------------------------------------------------
// watchdog
#ifdef CSM_C_DOWN_REASON_MCNET_WD_MON_IND
  #if (CSM_C_MCNET_MAX_CONNECTIONS > 16)             /* Number of Slave NM instances <= number of connections */
    #error "change CSM_wMCNetWdPresent/MissingFlags"
  #endif

  #ifdef CSM_SELF
    WORD CSM_wMCNetWdPresentFlags;       /**< One bit for each MCNet Slave NM instance (max. 16 instances) */
    WORD CSM_wMCNetWdPresentFirstFlags;  /**< One bit for each MCNet Slave NM instance (max. 16 instances) */
    WORD CSM_wMCNetWdMissingFlags;       /**< One bit for each MCNet Slave NM instance (max. 16 instances) */
  #else
    extern WORD CSM_wMCNetWdPresentFlags;
    extern WORD CSM_wMCNetWdPresentFirstFlags;
    extern WORD CSM_wMCNetWdMissingFlags;
  #endif // CSM_SELF

  #define CSM_M_SET_MCNET_WD_PRESENT_FLAG(bSlaveNmIdx)          (CSM_wMCNetWdPresentFlags |= (0x01U << bSlaveNmIdx))
  #define CSM_M_SET_MCNET_WD_PRESENT_FIRST_FLAG(bSlaveNmIdx)    (CSM_wMCNetWdPresentFirstFlags |= (0x01U << bSlaveNmIdx))
  #define CSM_M_SET_MCNET_WD_MISSING_FLAG(bSlaveNmIdx)          (CSM_wMCNetWdMissingFlags |= (0x01U << bSlaveNmIdx))

  #define CSM_M_CLEAR_MCNET_WD_PRESENT_FLAG(bSlaveNmIdx)        (CSM_wMCNetWdPresentFlags &= ~(0x01U << bSlaveNmIdx))
  #define CSM_M_CLEAR_MCNET_WD_PRESENT_FIRST_FLAG(bSlaveNmIdx)  (CSM_wMCNetWdPresentFirstFlags &= ~(0x01U << bSlaveNmIdx))
  #define CSM_M_CLEAR_MCNET_WD_MISSING_FLAG(bSlaveNmIdx)        (CSM_wMCNetWdMissingFlags &= ~(0x01U << bSlaveNmIdx))

  #define CSM_M_IS_MCNET_WD_PRESENT_FLAG_SET(bSlaveNmIdx)       (CSM_wMCNetWdPresentFlags & (0x01U << bSlaveNmIdx))
  #define CSM_M_IS_MCNET_WD_PRESENT_FIRST_FLAG_SET(bSlaveNmIdx) (CSM_wMCNetWdPresentFirstFlags & (0x01U << bSlaveNmIdx))
  #define CSM_M_IS_MCNET_WD_MISSING_FLAG_SET(bSlaveNmIdx)       (CSM_wMCNetWdMissingFlags & (0x01U << bSlaveNmIdx))
#endif // CSM_C_DOWN_REASON_MCNET_WD_MON_IND

// Connection Request ---------------------------------------------------------------------
#ifdef CSM_C_DOWN_REASON_MCNET_CONNECT_REQ
  #if (CSM_C_MCNET_MAX_CONNECTIONS > 16)
    #error "change CSM_wMCNetConnectRequestFlags"
  #endif
  #ifdef CSM_SELF
    WORD CSM_wMCNetConnectRequestFlags;       /**< One bit for each MCNet ASDT TP channel where
                                               * the data request function has
                                               * to be called (max 16 channels) */
    WORD CSM_wMCNetConReqTriggeredByAppFlags; /**< Auxiliary flags: Connect.Req by application (vs. NM) */
  #else
    extern WORD CSM_wMCNetConnectRequestFlags;
    extern WORD CSM_wMCNetConReqTriggeredByAppFlags;
  #endif // CSM_SELF

  #define CSM_M_SET_MCNET_CONNECT_REQUEST_FLAG(bChannel)    ( CSM_wMCNetConnectRequestFlags |= (WORD)(0x01U << bChannel) )
  #define CSM_M_CLEAR_MCNET_CONNECT_REQUEST_FLAG(bChannel)  ( CSM_wMCNetConnectRequestFlags &= ~(0x01U << bChannel) )
  #define CSM_M_IS_MCNET_CONNECT_REQUEST_FLAG_SET(bChannel) ( CSM_wMCNetConnectRequestFlags & (0x01U << bChannel) )

  #define CSM_M_SET_MCNET_CON_REQ_TRIGGERED_BY_APPL_FLAG(bChannel)    ( CSM_wMCNetConReqTriggeredByAppFlags |= (WORD)(0x01U << bChannel) )
  #define CSM_M_CLEAR_MCNET_CON_REQ_TRIGGERED_BY_APPL_FLAG(bChannel)  ( CSM_wMCNetConReqTriggeredByAppFlags &= ~(0x01U << bChannel) )
  #define CSM_M_IS_MCNET_CON_REQ_TRIGGERED_BY_APPL_FLAG_SET(bChannel) ( CSM_wMCNetConReqTriggeredByAppFlags & (0x01U << bChannel) )
#endif // CSM_C_DOWN_REASON_MCNET_CONNECT_REQ


// ASDT Data Request ---------------------------------------------------------------------
#ifdef CSM_C_DOWN_REASON_MCNET_ASDT_DATA_REQ
  #if (CSM_C_MCNET_MAX_CONNECTIONS > 16)
    #error "change CSM_wMCNetLd/EdRequestFlags"
  #endif
  #ifdef CSM_SELF
    WORD CSM_wMCNetAsdtLdRequestFlags; /**< One bit for each MCNet ASDT TP channel where
                                        * the long data request function has
                                        * to be called (max 16 channels) */
    WORD CSM_wMCNetAsdtEdRequestFlags; /**< Same for expedited data */
  #else
    extern WORD CSM_wMCNetAsdtLdRequestFlags;
    extern WORD CSM_wMCNetAsdtEdRequestFlags;
  #endif // CSM_SELF

  #define CSM_M_SET_MCNET_ASDT_LDATA_REQUEST_FLAG(bChannel)    ( CSM_wMCNetAsdtLdRequestFlags |= (WORD)(0x01U << bChannel) )
  #define CSM_M_SET_MCNET_ASDT_EDATA_REQUEST_FLAG(bChannel)    ( CSM_wMCNetAsdtEdRequestFlags |= (WORD)(0x01U << bChannel) )
  #define CSM_M_CLEAR_MCNET_ASDT_LDATA_REQUEST_FLAG(bChannel)  ( CSM_wMCNetAsdtLdRequestFlags &= ~(0x01U << bChannel) )
  #define CSM_M_CLEAR_MCNET_ASDT_EDATA_REQUEST_FLAG(bChannel)  ( CSM_wMCNetAsdtEdRequestFlags &= ~(0x01U << bChannel) )
  #define CSM_M_IS_MCNET_ASDT_LDATA_REQUEST_FLAG_SET(bChannel) ( CSM_wMCNetAsdtLdRequestFlags & (0x01U << bChannel) )
  #define CSM_M_IS_MCNET_ASDT_EDATA_REQUEST_FLAG_SET(bChannel) ( CSM_wMCNetAsdtEdRequestFlags & (0x01U << bChannel) )
#endif // CSM_C_DOWN_REASON_MCNET_ASDT_DATA_REQ

// USDT Data Request ---------------------------------------------------------------------
#ifdef CSM_C_DOWN_REASON_MCNET_USDT_DATA_REQ
  #if (CSM_C_MCNET_MAX_USDT_TX_CHANNELS > 16)
    #error "change CSM_wMCNetUsdtDataRequestFlags"
  #endif
  #ifdef CSM_SELF
    WORD CSM_wMCNetUsdtDataRequestFlags; /**< One bit for each MCNet USDT channel where
                                          * the data request function has
                                          * to be called (max 16 channels) */
  #else
    extern WORD CSM_wMCNetUsdtDataRequestFlags;
  #endif // CSM_SELF

  #define CSM_M_SET_MCNET_USDT_DATA_REQUEST_FLAG(bChannel)    ( CSM_wMCNetUsdtDataRequestFlags |= (0x01U << bChannel) )
  #define CSM_M_CLEAR_MCNET_USDT_DATA_REQUEST_FLAG(bChannel)  ( CSM_wMCNetUsdtDataRequestFlags &= ~(0x01U << bChannel) )
  #define CSM_M_IS_MCNET_USDT_DATA_REQUEST_FLAG_SET(bChannel) ( CSM_wMCNetUsdtDataRequestFlags & (0x01U << bChannel) )
#endif // CSM_C_DOWN_REASON_MCNET_USDT_DATA_REQ

// Connection Confirmation ---------------------------------------------------------------------
#ifdef CSM_C_IND_REASON_MCNET_COMM_CON
  #if (CSM_C_MCNET_MAX_CONNECTIONS > 16)
    #error "change CSM_wMCNetConnectConFlags"
  #endif
  #ifdef CSM_SELF
    WORD CSM_wMCNetConnectConFlags; /**< One bit for each MCNet-TP channel where
                                     * the application connection confirmation function has
                                     * to be called (max 16 channels) */
  #else
    extern WORD CSM_wMCNetConnectConFlags;
  #endif // CSM_SELF

  #define CSM_M_SET_MCNET_CONNECT_CON_FLAG(bChannel)    ( CSM_wMCNetConnectConFlags |= (WORD)(0x01U << bChannel) )

  #define CSM_M_CLEAR_MCNET_CONNECT_CON_FLAG(bChannel)  { INT iIsrStat = CSM_iInterruptsDisable();           \
                                                          CSM_wMCNetConnectConFlags &= ~(0x01U << bChannel); \
                                                          CSM_vInterruptsRestore(iIsrStat);                  \
                                                        }

  #define CSM_M_IS_MCNET_CONNECT_CON_FLAG_SET(bChannel) ( CSM_wMCNetConnectConFlags & (0x01U << bChannel) )
#endif // CSM_C_IND_REASON_MCNET_COMM_CON

// Connection Indication ---------------------------------------------------------------------
#ifdef CSM_C_IND_REASON_MCNET_COMM_IND
  #if (CSM_C_MCNET_MAX_CONNECTIONS > 16)
    #error "change CSM_wMCNetConnectIndFlags"
  #endif
  #ifdef CSM_SELF
    WORD CSM_wMCNetConnectIndFlags; /**< One bit for each MCNet-TP channel where
                                     * the application connection indication function has
                                     * to be called (max 16 channels) */
  #else
    extern WORD CSM_wMCNetConnectIndFlags;
  #endif // CSM_SELF

  #define CSM_M_SET_MCNET_CONNECT_IND_FLAG(bChannel)    ( CSM_wMCNetConnectIndFlags |= (WORD)(0x01U << bChannel) )

  #define CSM_M_CLEAR_MCNET_CONNECT_IND_FLAG(bChannel)  { INT iIsrStat = CSM_iInterruptsDisable();           \
                                                          CSM_wMCNetConnectIndFlags &= ~(0x01U << bChannel); \
                                                          CSM_vInterruptsRestore(iIsrStat);                  \
                                                        }

  #define CSM_M_IS_MCNET_CONNECT_IND_FLAG_SET(bChannel) ( CSM_wMCNetConnectIndFlags & (0x01U << bChannel) )
#endif // CSM_C_IND_REASON_MCNET_COMM_IND

// ASDT Data Confirmation ---------------------------------------------------------------------
#ifdef CSM_C_IND_REASON_MCNET_ASDT_DATA_CON
  #if (CSM_C_MCNET_MAX_CONNECTIONS > 16)
    #error "change CSM_wMCNetAsdtDataConFlags"
  #endif
  #ifdef CSM_SELF
    WORD CSM_wMCNetAsdtLdDataConFlags; /**< One bit for each MCNet-TP channel where
                                         * the application data confirmation function has
                                         * to be called (max 16 channels) */
    WORD CSM_wMCNetAsdtEdDataConFlags; /**< Same as above, expedited data service */

    BYTE CSM_abMCNetAsdtLdDataConResults[CSM_C_MCNET_MAX_CONNECTIONS];  /**< Results (standard service)  */
    BYTE CSM_abMCNetAsdtEdDataConResults[CSM_C_MCNET_MAX_CONNECTIONS];  /**< Results (expedited service) */
  #else
    extern WORD CSM_wMCNetAsdtLdDataConFlags;
    extern WORD CSM_wMCNetAsdtEdDataConFlags;
    extern BYTE CSM_abMCNetAsdtLdDataConResults[CSM_C_MCNET_MAX_CONNECTIONS];
    extern BYTE CSM_abMCNetAsdtEdDataConResults[CSM_C_MCNET_MAX_CONNECTIONS];
  #endif // CSM_SELF

  #define CSM_M_SET_MCNET_ASDT_LD_DATA_CON_FLAG(bChannel)    ( CSM_wMCNetAsdtLdDataConFlags |= (WORD)(0x01U << bChannel) )
  #define CSM_M_SET_MCNET_ASDT_ED_DATA_CON_FLAG(bChannel)    ( CSM_wMCNetAsdtEdDataConFlags |= (WORD)(0x01U << bChannel) )                                                                                                 * bit for this channel (on CAN task level) */

  #define CSM_M_CLEAR_MCNET_ASDT_LD_DATA_CON_FLAG(bChannel)  { INT iIsrStat = CSM_iInterruptsDisable();              \
                                                               CSM_wMCNetAsdtLdDataConFlags &= ~(0x01U << bChannel); \
                                                               CSM_vInterruptsRestore(iIsrStat);                     \
                                                             }

  #define CSM_M_CLEAR_MCNET_ASDT_ED_DATA_CON_FLAG(bChannel)  { INT iIsrStat = CSM_iInterruptsDisable();              \
                                                               CSM_wMCNetAsdtEdDataConFlags &= ~(0x01U << bChannel); \
                                                               CSM_vInterruptsRestore(iIsrStat);                     \
                                                             }

  #define CSM_M_IS_MCNET_ASDT_LD_DATA_CON_FLAG_SET(bChannel) ( CSM_wMCNetAsdtLdDataConFlags & (0x01U << bChannel) )
  #define CSM_M_IS_MCNET_ASDT_ED_DATA_CON_FLAG_SET(bChannel) ( CSM_wMCNetAsdtEdDataConFlags & (0x01U << bChannel) )
#endif // CSM_C_IND_REASON_MCNET_ASDT_DATA_CON

// ASDT Data Indication ---------------------------------------------------------------------
#ifdef CSM_C_IND_REASON_MCNET_ASDT_DATA_IND
  #if (CSM_C_MCNET_MAX_CONNECTIONS > 16)
    #error "change CSM_wMCNetAsdtDataIndFlags"
  #endif
  #ifdef CSM_SELF
    WORD CSM_wMCNetAsdtDataIndFlags; /**< One bit for each MCNet-TP channel where
                                      * the application data indication function has
                                      * to be called (max 16 channels) */
  #else
    extern WORD CSM_wMCNetAsdtDataIndFlags;
  #endif // CSM_SELF
  #define CSM_M_SET_MCNET_ASDT_DATA_IND_FLAG(bChannel)    ( CSM_wMCNetAsdtDataIndFlags |= (WORD)(0x01U << bChannel) )

  #define CSM_M_CLEAR_MCNET_ASDT_DATA_IND_FLAG(bChannel)  { INT iIsrStat = CSM_iInterruptsDisable();            \
                                                            CSM_wMCNetAsdtDataIndFlags &= ~(0x01U << bChannel); \
                                                            CSM_vInterruptsRestore(iIsrStat);                   \
                                                          }

  #define CSM_M_IS_MCNET_ASDT_DATA_IND_FLAG_SET(bChannel) ( CSM_wMCNetAsdtDataIndFlags & (0x01U << bChannel) )
#endif // CSM_C_IND_REASON_MCNET_ASDT_DATA_IND

// USDT Data Indication ---------------------------------------------------------------------
#ifdef CSM_C_IND_REASON_MCNET_USDT_DATA_IND
  #if (CSM_C_MCNET_MAX_USDT_RX_CHANNELS > 16)
    #error "change CSM_wMCNetUsdtDataIndFlags"
  #endif
  #ifdef CSM_SELF
    WORD CSM_wMCNetUsdtDataIndFlags; /**< One bit for each MCNet USDT RX channel where
                                      * the application data indication function has
                                      * to be called (max 16 channels) */
  #else
    extern WORD CSM_wMCNetUsdtDataIndFlags;
  #endif // CSM_SELF

  #define CSM_M_SET_MCNET_USDT_DATA_IND_FLAG(bChannel)    ( CSM_wMCNetUsdtDataIndFlags |= (0x01U << bChannel) )

  #define CSM_M_CLEAR_MCNET_USDT_DATA_IND_FLAG(bChannel)  { INT iIsrStat = CSM_iInterruptsDisable();             \
                                                            CSM_wMCNetUsdtDataIndFlags &= ~(0x01U << bChannel);  \
                                                            CSM_vInterruptsRestore(iIsrStat);                    \
                                                          }
  #define CSM_M_IS_MCNET_USDT_DATA_IND_FLAG_SET(bChannel) ( CSM_wMCNetUsdtDataIndFlags & (0x01U << bChannel) )
#endif // CSM_C_IND_REASON_MCNET_USDT_DATA_IND

// USDT Data Confirmation ---------------------------------------------------------------------
#ifdef CSM_C_IND_REASON_MCNET_USDT_DATA_CON
  #if (CSM_C_MCNET_MAX_USDT_TX_CHANNELS > 16)
    #error "change CSM_wMCNetUsdtDataConFlags"
  #endif
  #ifdef CSM_SELF
    WORD CSM_wMCNetUsdtDataConFlags; /**< One bit for each MCNet USDT TX channel where
                                      * the application data indication function has
                                      * to be called (max 16 channels) */
  #else
    extern WORD CSM_wMCNetUsdtDataConFlags;
  #endif // CSM_SELF
#define CSM_M_SET_MCNET_USDT_DATA_CON_FLAG(bChannel)    ( CSM_wMCNetUsdtDataConFlags |= (0x01U << bChannel) )

#define CSM_M_CLEAR_MCNET_USDT_DATA_CON_FLAG(bChannel)  { INT iIsrStat = CSM_iInterruptsDisable();            \
                                                          CSM_wMCNetUsdtDataConFlags &= ~(0x01U << bChannel); \
                                                          CSM_vInterruptsRestore(iIsrStat);                   \
                                                        }

#define CSM_M_IS_MCNET_USDT_DATA_CON_FLAG_SET(bChannel) ( CSM_wMCNetUsdtDataConFlags & (0x01U << bChannel) )
#endif // CSM_C_IND_REASON_MCNET_USDT_DATA_CON

//******************************************************************************
//  CCCC  SSSSS  MM   MM
// C      S      M M M M
// C       SSS   M  M  M        Variables and macros
// C          S  M     M
//  CCCC  SSSSS  M     M
//******************************************************************************
// Indication Bus State changed ------------------------------------------------
#if (defined CSM_C_IND_REASON_BUS_STATE_CHANGE) || (defined CSM_C_DOWN_REASON_BUS_STATE_CHANGE)
  // Target state for next CSM_lBusStateChange call. Access Macro available.
  #ifdef CSM_SELF
    BYTE CSM_abBusStateChangeTargetState[CSM_C_MAX_CAN_BUS];
  #else
    extern BYTE CSM_abBusStateChangeTargetState[CSM_C_MAX_CAN_BUS];
  #endif // CSM_SELF

  // access macro
  #define CSM_M_SET_BUS_STATE_CHANGE_TARGET_STATE(bBus, bTargetState)  { INT iIsrStat = CSM_iInterruptsDisable();              \
                                                                         CSM_abBusStateChangeTargetState[bBus] = bTargetState; \
                                                                         CSM_vInterruptsRestore(iIsrStat);                     \
                                                                       }
#endif // CSM_C_IND_REASON_BUS_STATE_CHANGE or CSM_C_DOWN_REASON_BUS_STATE_CHANGE


// error grade evaluation and comparison at function call ----------------------
#define CSM_M_SET_HIGHEST_ERROR_GRADE(ActErrorValue, Function)              \
  {                                                                         \
    LONG s32ErrorValueTmp;                                                  \
    s32ErrorValueTmp = Function;                                            \
    if (ActErrorValue < 0)                                                  \
    {                                                                       \
      if (s32ErrorValueTmp < 0)                                             \
      {                                                                     \
        if(CSM_M_GET_EG(s32ErrorValueTmp) > CSM_M_GET_EG(ActErrorValue))    \
        {                                                                   \
          ActErrorValue = s32ErrorValueTmp;                                 \
        }                                                                   \
      }                                                                     \
    }                                                                       \
    else                                                                    \
    {                                                                       \
      ActErrorValue = s32ErrorValueTmp;                                     \
    }                                                                       \
  } // end of macro

// Critical Voltage handling
#if ( CSM_S_CRITICAL_LOW_VOLTAGE == CSM_C_F_ON )
  extern BOOL CSM_fCriticalLowVoltageActive;
#endif // CSM_S_CRITICAL_LOW_VOLTAGE

/*******************************************************************************
 * function prototypes (scope: csm stack local)
 ******************************************************************************/
void CSM_VW_vNMEventListener(DWORD lEvent);
void CSM_VW_vNMEventListenerPCAN(void);
void CSM_vHandleCodePlus(void);

void csm_vTriggerCsmUpstreamTask(DWORD dwIndicationReason);
void csm_vActivateUpstreamTask(void);

#if (CSM_S_TRIGGER_UPSTREAM_TASK_FROM_IRQ_AVAILABLE == CSM_C_F_ON)
  void csm_vTriggerCsmUpstreamTaskFromIrq(DWORD dwIndicationReason);
  void csm_vActivateUpstreamTaskFromIrq(void);
#endif // CSM_S_TRIGGER_UPSTREAM_TASK_FROM_IRQ_AVAILABLE

void csm_vTriggerCsmDownstreamTask(DWORD dwRequestReason);
void csm_vActivateDownstreamTask(void);
void csm_vTriggerCsmDownstreamTaskFromIrq(DWORD wRequestReason);
void csm_vActivateDownstreamTaskFromIrq(void);

void CSM_vDownstreamTraceOut(const WORD wTraceOutLevel,const WORD wTraceOutInterface,
const WORD wTraceOutFunktion,const BYTE *pbTraceOutData,const WORD wTraceOutDataLength);

void csm_vOSspecificUpstream(void);
void csm_vOSspecificDownstream(void);

/*******************************************************************************
 * function prototypes (scope: global)
 ******************************************************************************/
// Timer related
LONG  CSM_lTimerStart(WORD wTimerId, DWORD dwTime);
LONG  CSM_lTimerStop(WORD wTimerId);
LONG  CSM_lCyclicTimerStart(WORD wTimerId, DWORD dwStartDelay, DWORD dwCycleTime);
LONG  CSM_lCyclicTimerStop(WORD wTimerId);

#if (CSM_S_TRACE == CSM_C_F_ON)
void CSM_vTracePrintf (WORD wTraceLevel, const char* format_string, ...);
#endif

#if (CSM_S_CYCLIC_TASK == CSM_C_F_ON)
void CSM_vCyclicTask(void);
#endif

#if (CSM_S_REQUESTABLE_TIMER == CSM_C_F_ON)
LONG CSM_lRequestTimer(void);
LONG CSM_lReleaseTimer (WORD wTimerId);
BOOL CSM_fTimerExpired (WORD wTimerId);
#endif // CSM_S_REQUESTABLE_TIMER

#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
  #if (CSM_S_ISO_TP == CSM_C_F_ON)
  void csm_vGetItpAdrParameter( DWORD *pdwProtocollType, tCSM_ADDR_ISO_TP * prAddrField, BYTE bIsoTpChannel);
  #endif // CSM_S_ISO_TP
#endif // CSM_S_CNP_AVAILABLE

#if (CSM_S_DEBUG == CSM_C_F_ON)
  void CSM_vDebPrintf(BYTE bDebugLevel, const char *arg);
#else
  #define  CSM_vDebPrintf(a,b)   // no function, no code
#endif

void CSM_vTestPrintf(const char *arg);

#if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_LIMITED_SIGNAL == CSM_C_F_ON)
    BOOL CSM_fTraceCheckSignal( DWORD dwSignalId);
    BOOL CSM_fTraceCheckTxSignal( DWORD dwSignalId);
  #endif
#endif

void CSM_vTraceOut(const WORD wTraceOutLevel, const WORD wTraceOutInterface, const WORD wTraceOutFunktion,
                   const BYTE *pbTraceOutData,const WORD wTraceOutDataLength);

void CSM_vMicroSecDelay(WORD wDelay);

void CSM_vMemCopy(void* pvDestination, const void* pvSource, WORD wLength);

void CSM_vMemSet(void* pvDestination, const BYTE cubValue, WORD NumObj);

BYTE CSM_bMemCompare(const void* pvSource1, const void* pvSource2, WORD wLength);

void CSM_vMemCopyDWordInMotorolaOrder(BYTE* pbDestination, DWORD dwInput);

//*****************************************************************
//* prototypes for CM-CR projects with ercosek OS
void CSM_vSYS_PrepareSleepCon(void);
void CSM_vSYS_PrepareNormalInd(void);
void CSM_vSYS_ReqBusStateMsgTx2PWR(void);

#if (CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON)
void CSM_vIndicateBusError(BYTE bBus, BYTE bError, BYTE bState);
BYTE CSM_bGetBusError(BYTE bBus);
#endif

#if (CSM_S_ENABLE_INDICATE_STATIC_ERROR == CSM_C_F_ON)
void CSM_vIndicateStaticError(BYTE bBus, LONG lError);
#endif

void  CSM_vSetEmTraceError(DWORD dwErrorId);
DWORD CSM_dwGetEmTraceError(void);

#if (CSM_S_CSM_VW == CSM_C_F_ON)
  CSM_INLINE void CSM_vCDABusOff(BYTE bBus, BYTE bState)
  {
    CSM_vIndicateBusError(bBus, CSM_C_BUS_OFF, bState);
  }

  CSM_INLINE void CSM_vCNMNoNetWork(BYTE bBus, BYTE bState)
  {
    CSM_vIndicateBusError(bBus, CSM_C_NO_NETWORK, bState);
  }
#endif

#if (CSM_S_EXCEPTION_HANDLER_ACTIVATED == CSM_C_F_ON)
LONG CSM_lExceptionHandler(BYTE u8ErrorGrade, BYTE u8ErrorLocation, WORD u8ErrorCode);
#endif

INT CSM_iInterruptsDisable(void);
void CSM_vInterruptsRestore(INT iIsrStatus);

LONG CSM_lSetBusHold(BYTE bBus, DWORD dwProtocolType, WORD wApplID);
LONG CSM_lClearBusHold(BYTE bBus, DWORD dwProtocolType, WORD wApplID);
BOOL CSM_bGetBusHold(BYTE bBus);

LONG CSM_lBusStateChange(BYTE bBus, BYTE bCsmTargetState);

#if (CSM_S_RX_OVERRUN_INDICATION == CSM_C_F_ON)
  // Function CSM_vRxOverrunOccured() available to indicate an overrun in a Rx register or a queue
  void CSM_vRxOverrunOccured(BYTE bBus);
#endif

// internal assert function, to be mapped OS specific assert function or macro if needed
void CSM_vStackAssert(DWORD dwExpression);

LONG CSM_lInitQueue( tCSM_QUEUE *prQueue, BYTE bMaxElements, BYTE bElementSize, BYTE *pbData);
BOOL CSM_fWriteQueue( tCSM_QUEUE *prQueue, const BYTE* const pbData, WORD wLength);
BOOL CSM_fReadQueue( tCSM_QUEUE *prQueue, BYTE *pbData);

#ifdef CSM_EV_TRIGGER_LINE_CHANGED
// Coded up to now in csm_stack_tengine.c. CAN is active when com_enable HW line is active (interface to WUP driver).
BYTE CSM_STACK_bDetermineBusStateFromLine(BYTE bBus);
#endif

#else
  #error CSM_STACK_M.H included several times
#endif
/*******************************************************************************
 * \doxydocu
 * \file          csm_stack_m.h
 * \brief         Header file for CAN stack internal used data and defines.
 *
 *                Header file for CAN stack internal used data and defines.
 *
 * \sw_component  CAN stack
 * \project       Pool module
 * \path          /di_can/modules_swa/csm_stack/src/
 *
 * \ingroup       CSM
 * \ingroup       CSM_STACK
 *
 *
 * \authors       V-Team
 *
 * COPYRIGHT:      (c) 2001 Blaupunkt GmbH
 *
 * \history_begin
 *
 *27.02.04  Rev. \main\3  CM-DI/ESU3-K"ohler
 *          - Prototype CSM_vTriggerBusStateChange() added.
 *          - Up- and downstream reason bus state change added.
 *
 *24.05.04  Rev. \main\5  CM-DI/ESU3-Prhl
 *          csm_vActivateDownstreamTaskFromIrq() added
 *
 *25.05.04  Rev. \main\6  CM-DI/ESU3-Prhl
 *          warnings removed
 *
 *28.06.04  Rev. \main\7  CM-DI/ESU3-Prhl
 *          CSM_CAN_TX_DATA and CSM_CAN_RX_DATA
 *
 * 28.06.04  Rev. \main\9  CM-DI/ESU3-K"ohler
 * - First commands for doxygen added.
 * - Defines for CSM_HW_LIST renamed.
 * - CAN controller state defines added.
 * - Some #if "featureswitch" added.
 *
 * 14.07.04  Rev. \main\10  CM-DI/ESU3-Prhl
 * - bugfix for \main\9 changes
 *
 * 16.07.04  Rev. \main\11  CM-DI/ESU3-Prhl
 * - old typedefs removed
 * - typedefs moved from cda_m.h
 *
 * 19.07.04  Rev. \main\12  CM-DI/ESU3-Prhl
 * - bugfix for \main\11
 *
 * 19.07.04  Rev. \main\13  CM-DI/ESU3-Prhl
 * - bugfix for \main\12
 *
 * 20.07.04  \main\14  CM-DI/ESU3-Battis
 * - ifdef movements
 *
 * 20.07.04  \main\15  CM-DI/ESU3-Prhl
 * - some doxygen comments added
 * - typedefs moved
 *
 * 15.09.04  \main\16  CM-DI/ESA3-Battis
 * - CSM_bDisableAllInterruptsAndSave() renamed to CSM_iInterruptsDisable()
 * - CSM_vRestoreAllInterrupts() renamed to CSM_vInterruptsRestore()
 *
 * 23.09.04  \main\17  CM-DI/ESA3-K"ohler
 * - C++ comments replaced by C comments.
 *
 * 29.09.04  \main\18  CM-DI/ESA3-Battis
 * - new down reasons
 *
 * 29.09.04  \main\19  CM-DI/ESA3-Battis
 * - down reasons renamed
 *
 * 01.11.04  \main\20  CM-DI/ESA3-Battis
 * - CBR defines added
 *
 * 22.11.04  \main\20  CM-DI/ESA3-Battis
 * - old CBR defines removed
 *
 * 02.12.04  \main\21 CM-DI/ESA3-Battis
 * - "blockable upstream task" feature added
 *
 * 02.12.04  \main\22 CM-DI/ESA3-Battis
 * - "blockable upstream task" feature added
 *
 * 03.02.05  \main\23 CM-DI/ESA3-Prhl
 * - compiler warnings removed
 *
 * 07.02.05  \main\24 CM-DI/ESA3-Prhl
 * - typedef tCSM_TIMER added
 *
 * 16.02.05  \main\25 CM-DI/ESA3-Prhl
 * - enum for trace modified (CSM_TRACE_C_FKT_...).
 *
 * 22.02.05  \main\26 CM-DI/ESA3-Prhl
 * - remove callbacks from CSM_ISO_TP_CONT
 *
 * 07.03.05  \main\27 CM-DI/ESA3-Battis
 * - message type CBR_TMSG_NO_TRANSMISSION added
 * - extern definition for Fiat NWM added
 *
 * 07.03.05  \main\28 CM-DI/ESA3-Zurmhl
 * - csm_rMCNetLocCompList transferred to csm_stack_tsi_m.h
 *   (only relevant for PC test / TSI)
 *
 * 09.03.05  \main\29 CM-DI/ESA3-Battis
 * - RX groups support for timeout supervision implemented
 *
 * 10.03.05  \main\30 CM-DI/ESA3-Prhl
 * - compiler warning removed
 *
 * 10.03.05  \main\31 CM-DI/ESA3-Prhl
 * - typedef moved from csm_stack_mp.h
 *
 * 05.04.05  \main\32  CM-DI/ESA3-K"ohler
 * - Function prototypes CSM_vNetworkManagement(Goes)Up/Down() and
 *   CSM_vIgnitionIndication() removed.
 *
 * 07.04.05  \main\33  CM-DI/ESA3-K"ohler
 * - Macro ELEMENTS renamed CSM_M_ELEMENTS.
 * - Feature switches CSM_S_APPLICATION_TELEGRAMS and CSM_S_VW_TP_2_0 used to
 *   encapsulate struct definitions.
 * - A lot of old defines/declarations removed.
 *
 * 08.04.05  \main\34  CM-DI/ESA3-K"ohler
 * - C++ comments removed.
 *
 * 08.04.05  \main\35  CM-DI/ESA3-K"ohler
 * - MCNet callback function csm_rMCNetPwrCallback removed.
 *
 * 20.04.05  \main\36 CM-DI/ESA3-Zurmhl
 * - "const" replaced by "CSM_M_ROM_TABLE" in CDA typedefs
 *   (only affects PC test / TSI)
 *
 * 25.04.05  \main\37  CM-DI/ESA3-K"ohler
 * - Connection indication function in Iso-Tp ROM structure reactivated for Opel
 *   Ratio projects.
 *
 * 28.04.05  \main\38  CM-DI/ESA3-Battis
 * CSM_FIAT_NM_TX re-inserted
 *
 * 02.05.05  \main\39  CM-CR/ESD4-Tuschik
 * Add VW-Features.
 *
 * 10.05.05  \main\41  CM-DI/ESA3-Battis
 * changes to make MISRA happy
 *
 * 13.05.05  \main\42  CM-DI/ESA3-Zurmhl
 * structure CSM_ISO_TP_CONT_ACTUAL: now CSM_BITFIELD_TYPE_8 is used
 *
 * 19.05.05  \main\43  CM-DI/ESA3-Prhl
 * change CSM_lTimerStart() and CSM_lTimerStop(). Now are more than 256 timer possible
 *
 * 19.05.05  \main\44  CM-DI/ESA3-Zurmhl
 * change CSM_lTimerStart() and CSM_lTimerStop(). DWORD->WORD for wTimerId
 *
 * 02.06.05  \main\45  CM-DI/ESA3-K"ohler
 * - Macros CSM_M_SET/CLEAR_DOWN_REASON changed.
 *
 * 08.06.05  \main\46  CM-DI/ESA3-Battis
 * - define CSM_C_HW_LIST_NOMASK added
 *
 * 08.06.05  \main\47  CM-DI/ESA3-K"ohler
 * - Prototype CSM_vRxOverrunOccured() added, featured with switch
 *   CSM_S_RX_OVERRUN_INDICATION.
 *
 * 09.06.05  \main\48  CM-DI/ESA3-Battis
 * - Protype CSM_vEnableSendRestriction() added
 *
 * 09.06.05  \main\49  CM-DI/ESA3-Battis
 * - MISRA happyness
 *
 * ... comments ??
 *
 * 05.07.05  \main\53  CM-DI/ESA3-Zurmuehl
 * - Error codes moved to csm_stack_i.h
 *
 * 09.08.05  \main\54  CM-DI/ESA3-Zurmuehl
 * - MCNet CAN IDs added for connections MAS-AUT and TES-AUT
 * - New flag for CSM_C_DOWN_REASON_MCNET_WD_MON_IND (WdPresentFirst)
 *
 * ... comments ??
 *
 * 19.05.05  \main\56  CM-DI/ESA3-Prhl
 * - warning removed
 *
 * 01.09.05  \main\57  CM-DI/ESA3-Battis
 * - prototype for CSM_dwTimerValue() removed
 *
 * 06.09.05  \main\58  CM-DI/ESA3-Battis
 * - MISRA
 *
 * 15.09.05  \main\59  CM-DI/ESA3-Zurmhl
 * - CSM_M_SET_HIGHEST_ERROR_GRADE() corrected (now handles functions with positive return values)
 *
 * 19.09.05  \main\60 CM-DI/ESA3-Prhl
 * - CSM_vTraceOut() replaced by CSM_M_TRACEOUT()
 *
 * 19.09.05  \main\61 CM-DI/ESA3-Prhl
 * - change defines for trace
 *
 * ... comments ??
 *
 * 23.09.05  \main\64  CM-DI/ESA3-K"ohler
 * - CSM_vCDABusOff, CSM_vCNMNoNetWork commented out.
 *
 * 27.09.05  \main\65  CM-DI/ESA3-Battis
 * - corrected VW Event listener defines
 * - fixed PCAN level event listener prototype
 * - CSM_vCDABusOff, CSM_vCNMNoNetWork with VW feature switch.
 *   Must be kept in mind for further busoff handling design
 *
 * 28.09.05  \main\66  CM-DI/ESA3-Battis
 * - writing error removed
 *
 * 07.10.05  \main\67  CM-DI/ESA3-Battis
 * - changes for multichannel functionality
 * - prototype CSM_vSYS_SleepConfirmation() removed
 * - prototype CSM_vSYS_ActiveIndication() removed
 * - prototype CSM_vSYS_NormalIndication() removed
 *
 * 21.10.05  \main\68  CM-DI/ESA3-Battis
 * - define CSM_C_MCNET_ASDT_CAN_ID_MAS_DPL added
 * - define CSM_C_MCNET_ASDT_CAN_ID_DPL_MAS added
 *
 * 21.10.05  \main\69  CM-DI/ESA3-Battis
 * - expected DLC handling integrated in CSM_HW_LIST
 *
 * 04.11.05  \main\70  CM-DI/ESA3-Battis
 * - "connect" related callbacks removed from CSM_MCNet_CONT
 * - CSM_C_IND_REASON_MCNET_CONNECT_CON renamed to CSM_C_IND_REASON_MCNET_COMM_CON
 * - CSM_C_IND_REASON_MCNET_CONNECT_IND renamed to CSM_C_IND_REASON_MCNET_COMM_IND
 *
 * 07.11.05  \main\71  CM-DI/ESA3-Battis
 * - prototype CSM_lCyclicTimerStart() added
 * - prototype CSM_lCyclicTimerStop() added
 *
 * 17.11.05  \main\72  CM-DI/ESA3-Battis
 * - CSM_C_DOWN_REASON_CSM_POST_INIT_START renamed to CSM_C_DOWN_REASON_CSM_STACK_ACTIVATED
 *
 * 22.11.05  \main\73  CM-DI/ESA3-Battis
 * - down reason suggestions renamed:
 *   CSM_C_DOWN_REASON_CDA_BUSOFF_ENTRY_CAN0 renamed to
 *     CSM_C_DOWN_REASON_CDA_BUSOFF_ENTRY
 *   CSM_C_DOWN_REASON_CDA_BUSOFF_RECOVERY_CAN0 renamed to
 *     CSM_C_DOWN_REASON_CDA_BUSOFF_RECOVERY
 *
 * 07.12.05  \main\74  CM-DI/ESA3-Prhl
 * - change CBR_CONVERT_VECTOR_DBC_BIG_SIGNAL marco
 *
 * 07.12.05  \main\75  CM-DI/ESA3-Prhl
 * - change CBR_CONVERT_VECTOR_DBC_BIG_SIGNAL marco
 *
 * 24.02.06  \main\76  CM-DI/ESA3-Franke
 * - suppress lint warning 572 "Excessive shift value", if BitPos <= 3 for
 *   macros CSM_M_READBITINBYTEFIELD,CSM_M_SETBITINBYTEFIELD,CSM_M_CLEARBITINBYTEFIELD)
 *
 * 28.02.06  \main\77  CM-DI/ESA3-K"ohler
 * - Macro CSM_M_PARAMETER_INTENTIONALLY_UNUSED() addd.
 *
 * 03.04.06  \main\78  CM-DI/ESA3-Battis
 * - prototype definitions for V850 CAN HW test added
 *
 * 12.04.06  \main\79  CM-DI/ESA3-Battis
 * - new telegram type CSM_C_TEL_TYPE_VAG_BAP_RAW introduced.
 *
 * 28.04.06  \main\80  CM-DI/ESA3-Battis
 * - usage of switch CSM_S_REMOTE_STACK corrected.
 * - bBus added to struct CSM_CBR_RX
 *
 * 30.05.06  \main\81  CM-DI/ESA3-Prhl
 * - add CSM_C_TEL_TYPE_GMLAN_NM
 * - add CSM_TRACE_C_FKT_CSM_DATA_READ
 * - add define CSM_C_CDA_EXTENDED_ID
 *
 * 01.06.06  \main\82  CM-DI/ESA3-Prhl
 * - add dwProtocolType to CSM_ISO_TP_CONT
 * - tCSM_ITP_CFG deleted
 * - wApplId and bBus added to tCSM_CBR_CFG
 * - tCSM_CSM_CFG move from csm.c
 *
 * 06.06.06  \main\83  CM-DI/ESA3-Prhl
 * - switch CSM_S_CDA_MASK_CAN_ID added
 *
 * 14.06.06  \main\84  CM-DI/ESA3-Prhl
 * - virtual network support added
 *
 * 14.06.06  \main\85  CM-DI/ESA3-Prhl
 * - CSM_C_TEL_TYPE_GATEWAY and CSM_C_TEL_TYPE_GATEWAY_PREDEF added
 *
 * 03.07.06  \main\86  CM-DI/ESA3-Prhl
 * - configuration struct tCDA_GATEWAY_CFG added
 * - configuration struct tCDA_GATEWAY_PREDEF_CFG added
 *
 * 19.07.06  \main\87  CM-DI/ESA3-Battis
 * - struct CSM_VW_TP20_CONT extended for direct TP control by applications.
 *
 * 25.07.06  \main\88  CM-DI/ESA3-K"ohler
 * - Callback function pointer in table CSM_ISO_TP_CONT removed and element
 *   order changed to optimise alignment.
 * - Doxygen comments added, some function prototype descriptions removed.
 * - Declaration of csm_itp_arCallbackFkt[] added.
 *
 * 26.07.06  \main\89  CM-DI/ESA3-Battis
 * - pbData made const in CSM_CAN_TX_DATA
 *
 * 01.08.06  \main\90  CM-DI/ESA3-Zurmhl
 * - CSM_ISO_TP_CONT_ACTUAL.bfConnectionState now 4 bits long
 * - CSM_C_ITP_TYPE_ASDT_RN added
 *
 * 18.08.06  \main\91  CM-DI/ESA3-Prhl
 * - add CSM_C_TRACEOUT_CSM_INTERNAL to the enum
 *
 * 18.08.06  \main\92  CM-DI/ESA3-Khler
 * - CSM_TRACE_C_FKT renamed CSM_C_TRACE_FKT, hex instead of decimal values.
 * - Doxygen comments added.
 *
 * 22.08.06  \main\93  CM-DI/ESA3-Prhl
 * - change in CSM_ISO_TP_CONT. now we have the old sturct order
 *
 * 30.08.06  \main\94  CM-DI/ESA3-Prhl
 * - undo \main\93
 *
 * 30.08.06  \main\95  CM-DI/ESA3-Khler
 * - Trace define CSM_C_TRACE_FKT_CSM_DUMMY addd.
 *
 * 27.09.06  \main\96  CM-DI/ESA3-Khler
 * - Lint warnings removed which are caused by undefined feature switches.
 *
 * 06.10.06  \main\97  CM-DI/ESA3-Khler
 * - Prototype csm_vTriggerCsmUpstreamTaskFromIrq() added.
 *
 * 09.10.06  \main\98  CM-DI/ESA3-Khler
 * - Lint warnings removed, doxygen comments added.
 *
 * 09.10.06  \main\99  CM-DI/ESA3-Khler
 * - Prototype CSM_vSystemWakeup() removed and
 *   csm_vActivateUpstreamTaskFromIrq() added.
 *
 * 10.10.06  \main\100  CM-DI/ESA3-Zurmhl
 * - PVOID replaced by void* (because of const PVOID confusion!)
 *
 * 23.10.06  \main\101  CM-DI/ESA3-Khler
 * - Define CSM_C_TRACE_FKT_CSM_SIGNAL_IND added.
 *
 * 02.11.06  \main\102  CM-DI/ESA3-Prhl
 * - move csm_vGetItpAdrParameter() from csm_m.h
 *
 * 03.11.06  \main\103  CM-DI/ESA3-Haffner
 * - added CSM_C_TRACE_FKT_CSM_STRING
 *
 * 07.11.06  \main\104  CM-DI/ESA3-Khler
 * - CSM_ADDR_OPEL_USDT_ISO replaced by tCSM_ADDR_ISO_TP.
 *
 * 01.12.06  \main\105  CM-DI/ESA3-Prhl
 * - add macros CSM_M_GET_TASK_CONTEXT and CSM_M_SET_TASK_CONTEXT if not defined in csm_stack_mp.h
 *
 * 07.12.06  \main\106  CM-DI/ESA3-Khler
 * - Lint authorization PQM_authrized_78 for macro addd.
 *
 * 08.12.06  \main\107  CM-DI/ESA3-Prhl
 * - first try for generic queues
 *
 * 23.01.07  \main\108  CM-DI/ESA3-Prhl
 * - Typedefs tCSM_UUDT_TX_CFG and tCSM_UUDT_RX_CFG added
 *
 * 14.02.07  \main\109  VTeam-Prhl
 * - some Trace defines added
 *
 * 16.02.07  \main\110  VTeam-Prhl
 * - some Trace defines added
 *
 * 23.02.07  \main\111  VTeam-Prhl
 * - typedef CSM_GMLAN_CFG added
 *
 * 07.03.07  \main\112  VTeam-Battis
 * - tCSM_BAP_AckSearchTable for VW BAP TP added
 *
 * 07.03.07  \main\113  VTeam-Battis
 * - tCSM_BAP_AckSearchTable moved out of an ifdef group.
 *
 * 21.03.07  \main\114  VTeam-Prhl
 * - some Trace defines added
 *
 * 21.03.07  \main\115  VTeam-Battis
 * - one feature usage switch corrected.
 *
 * 21.03.07  \main\116  VTeam-Battis
 * - feature usage switch corrected.
 *
 * 22.03.07  \main\117 VTeam-Prhl
 * - old OPEL feature switch removed
 *
 * 28.03.07  \main\119 VTeam-Prhl
 * - CSM_C_TEL_TYPE_VAG_BAP added.
 *
 * 14.04.07  \main\120 VTeam-Brunotte
 * CSM_C_TRACE_FKT_ constants for API test added
 *
 * 24.04.07  \main\121 VTeam-Prhl
 * CSM_C_TRACE_FKT_ constants added
 *
 * 15.05.07  \main\122 VTeam-Prhl
 * MCNet "RAM callbacks" added
 *
 * 13.05.07  \main\bugfix_bai2hi_di_can_vw_rns300nf_v01.10\1 VTeam-Battis
 * - retrigger feature added to VW TP20 broadcast
 *
 * 15.05.07  \main\123 VTeam-Prhl
 * merge from \main\bugfix_bai2hi_di_can_vw_rns300nf_v01.10\1
 *
 * 15.05.07  \main\124 VTeam-Prhl
 * :-)
 *
 * 15.05.07  \main\125 VTeam-Prhl
 * CSM_C_TRACE_FKT_CSM_SIGNAL_CALLBACK_INIT added to enum.
 *
 * 15.05.07  \main\126 CM-DI/VTeam-Prhl
 * - spaces after # deleted
 *
 * 14.06.07  \main\127 CM-DI/VTeam-Prhl
 * - some lint warnings removed
 *
 * 19.06.07  \main\128 CM-DI/VTeam-Prhl
 * - payload for network management added to tCSM_CBR_CFG and CSM_ISO_TP_CONT.
 *
 * 19.06.07  \main\129 CM-DI/VTeam-Prhl
 * - payload for network management added to tCSM_CBR_CFG and CSM_ISO_TP_CONT.
 *
 * 22.06.07  \main\130 CM-DI/VTeam-Battis
 * - VW event listener event added: CHK_BAP_PHONE_MUTE
 *
 * 25.06.07  \main\131 CM-DI/VTeam-Prhl
 * - now we have 32-bit up/down-resons
 *
 * 26.06.07  \main\132 CM-DI/VTeam-Prhl
 * - trace define for csm error indication added
 *
 * 02.07.07  \main\133 CM-DI/VTeam-Prhl
 * - feature CSM_S_CSM_LOCAL_CTRL_USER_FUNCTION added.
 *
 * 05.07.07  \main\134 CM-DI/VTeam-Battis
 * - VW events corrected.
 * - tCSM_APL_BUSERROR_IND removed.
 *
 * 11.07.07  \main\135 CM-DI/VTeam-Brunotte
 * -CSM_C_TRACE_FKT_CSM_UPPERTST constant added.
 *
 * 17.07.07  \main\136 CM-DI/VTeam-Brunotte
 * -CSM_C_CAN_IRQSETUP, CSM_C_CAN_EI, CSM_C_CAN_DI constant added.
 *
 * 24.07.07  \main\137 VTeam-Prhl
 * - bReqAction added to tydef CSM_MCNet_CONT_ACTUAL. MCNet can now keep the bus alive.
 * - dwNmPayload added to tydef CSM_MCNet_CONT. MCNet can now set OSEK payload.
 *
 * 31.07.07  \main\138 VTeam-Prhl
 * - lint warnings removed or commented out.
 *
 * 10.08.07  \main\139 VTeam-Prhl
 * - dis/enable signal monitoring added.
 *
 * 22.10.07  \main\140 VTeam-Brunotte
 * Interface for vector node learning function (Appl_vFFSAccess)
 * macro APPL_FFS_WRITE and APPL_FFS_READ declared
 *
 * 05.11.07  \main\141  VTeam-Battis
 * - function header CSM_vStackAssert() added
 * - VOID -> void
 *
 * 06.11.07  \main\142 VTeam-Brunotte
 * - macro APPL_FFS_WRITE and APPL_FFS_READ moved to csm_stack_mp.h
 *
 * 06.11.07  \main\143  VTeam-Brunotte
 * - Macro CSM_C_TRACE_FKT_CSM_SOURCEID for trace added
 *
 * 15.11.07  \main\144  VTeam-Prhl
 * - dis/enable signal monitoring added for tx signals.
 *
 * 22.11.07  \main\145  VTeam-Prhl
 * - CNM_OSEK_S_VW_PAYLOAD replaced by CNM_OSEK_S_PAYLOAD
 *
 * 06.12.07  \main\146  VTeam-Khler
 * - Lint warnings for CSM_S_CDA_GATEWAY and CSM_S_CDA_GATEWAY_PREDEF removed.
 *
 * 12.12.07  \main\147 VTeam-Prhl
 * - low voltage handling added.
 *
 * 24.01.08  \main\148  VTeam-Zurmhl
 * - module history comment block revised (doxygen \ingroup tag etc.)
 *
 * 25.02.08  \main\149  VTeam-Prhl
 * - CSM_S_CRITICAL_LOW_VOLTATE renamed to CSM_S_CRITICAL_LOW_VOLTAGE
 *
 * 06.02.08  \main\150  VTeam-Prhl
 * - feature CSM_S_ITP_FF_INDICATION_AVAILABLE and CSM_S_ITP_ERROR_INDICATION_AVAILABLE added.
 *
 * 07.02.08  \main\151  VTeam-Battis
 * - TEST
 *
 * 25.02.08  \main\152  VTeam-Funke
 * - added function CSM_vIndicateStaticError for CSM_S_ENABLE_INDICATE_STATIC_ERROR
 *
 * 27.02.08  \main\153  VTeam-Brunotte
 * - Trace commands CSM_C_TRACE_FKT_CNM_GMLAN_GET_ACTIVE_VN, CSM_C_TRACE_FKT_CNM_GMLAN_GET_API_VN_REQ added
 *
 * 27.02.08  \main\154  VTeam-Lansley / Funke
 * - added OSEK NM state callback indication
 *
 * 06.03.08  \main\155  VTeam-Lansley / Funke
 * - macro CSM_M_SET_HIGHEST_ERROR_GRADE changed
 * - changes were controlled by VTeam-Zurmhl and -Pistoor
 *
 * 06.03.08  \main\156  VTeam-Lansley / Funke
 * - error in last implementation of macro CSM_M_SET_HIGHEST_ERROR_GRADE corrected
 *
 * 07.04.08  \main\157  VTeam-Funke
 * - added trace function CSM_C_TRACE_FKT_CSM_CHANGE
 *
 * 10.04.08  \main\158  VTeam-Prhl
 * - added CSM_C_TRACE_FKT_CSM_MULTIPLE_SIGNAL_READ and CSM_C_TRACE_FKT_CSM_MULTIPLE_SIGNAL_WRITE to enum.
 *
 * 29.04.08  \main\159  VTeam-Funke
 * - CSM_bGetBusError() added.
 *
 * 19.05.08  \main\160  VTeam-Funke
 * - commented out CSM_C_TRACE_FKT_CSM_CHANGE_.. IDs
 * - added trace number CSM_C_TRACE_FKT_CNM_PROCESS_NM_STATE
 * - covered GMLAN trace defines in #if (CSM_S_GMLAN_NM == CSM_C_F_ON)
 *
 * 29.05.08  \main\161  VTeam-Zurmhl
 * - TEL_TYPES MPDT_TP, UUDT_RX and UUDT_TX added
 *
 * 18.06.08  \main\162  VTeam-Brunotte
 * - macro CSM_C_CAN_IRQEXIT added.
 *
 * 15.07.08  \main\163  VTeam-Funke / Basavaraj
 * - changes for reducing Lint and Compiler warnings
 *
 * 18.07.08  \main\164  VTeam-Funke
 * - added CSM_C_TRACE_FKT_CNM_UNDERVOLTAGE
 *
 * 01.09.08  \main\165  VTeam-Khler
 * - Prototypes for T-Kernel access layer and subsystem added.
 *
 * 12.09.08  \main\166 VTeam-Pistoor
 * -  Added function declarations CSM_vCyclicTask() and void CSM_vTracePrintf()
 *
 * 22.09.08  \main\167 Prhl
 * - tCSM_MPDT_CONFIG added.
 *
 * 24.09.08  \main\168 Prhl
 * - tCSM_MPDT_CONFIG modified.
 *
 * 25.09.08  \main\169 Prhl
 * - tCSM_MPDT_CONFIG modified.
 *
 * 29.09.08  \main\170 Prhl
 * - lint warnings removed.2
 *
 * 01.10.08  \main\171 Prhl
 * - MPDT buffer handling changed.
 *
 * 07.10.08  \main\172 Prhl
 * - define CSM_MPDT_C_MAX_IL added and set to 2.
 *
 * 08.10.08  \main\173 Prhl
 * - tCSM_MPDT_CONFIG modified.
 *
 * 21.10.08  \main\174 Prhl
 * - tCSM_MPDT_CONFIG modified.
 *
 * 17.11.08  \main\175 Prhl
 * - typedef CSM_CAN_CON_DATA added.
 *
 * 19.11.08  \main\176 Prhl
 * - typedef CNP_MPDT_tCAN_DATA added.
 *
 * 19.11.08  \main\177 Prhl
 * - tCSM_MPDT_CONFIG modified.
 *
 * 24.02.09  \main\178  Khler
 * - Trace enum CSM_C_TRACE_FKT_PROJECT_E0 added.
 *
 * 09.03.09  \main\179  Khler
 * - Trace enums CSM_C_TRACE_FKT_CSM_SET_TEST_NUMBER and
 *   CSM_C_TRACE_FKT_CSM_TRACE_TEST_NUMBER added.
 *
 * 09.03.09  \main\180  Khler
 * - History added.
 *
 * 10.03.09  \main\181 Prhl
 * - tCSM_MPDT_CONFIG modified.
 *
 * 10.03.09  \main\182 Prhl
 * - make lint happy
 *
 * 08.06.09  \main\183 Khler
 * - Prototype CSMAL_lGetAccessHandle() changed, parameter pvHandle added.
 *
 * 25.08.09  \main\184  VTeam-Khler
 * - Defines for down and indication reasons rearranged.
 *
 * 04.09.09  \main\185 Prhl
 * - CSM_C_TEL_TYPE_OSEKI_NM added
 *
 * 08.09.09  \main\186 VTeam-Franke
 * - added some CSM_C_IND_REASON_...
 *
 * 22.09.09  \main\187 VTeam-Franke
 * - new feature CSM_S_DEACTIVATE_COMMUNICATION_REQUESTS
 *
 * 28.10.09  \main\188  VTeam-Khler
 * - Comment for struct tCSM_CBR_CFG changed.
 *
 * 16.11.09  \main\189  VTeam-Khler
 * - Prototype CSM_vMemCopyDWordInMotorolaOrder() added.
 * - Prototypes CSM_vEnableSendRestriction() and CSM_vTriggerBusStateChange()
 *   removed.
 *
 * 07.12.09  \main\190  VTeam-Franke
 * - tCSM_UUDT_RX/TX_CFG: address field changed to wAddress
 * - new: CSM_C_DOWN_REASON_TRACE
 *
 * 12.12.09  \main\191  VTeam-Franke
 * - feature CSM_S_ENABLE_UPSTREAM_TRACE removed
 *
 * 14.12.09  \main\192 Riecker
 * - CSM_C_TRACE_FKT_CNM_PROCESS_NM_STATE         = 0xC0 for Fiat
 *
 * 19.01.10  \main\193  VTeam-Franke
 * - trace length defines moved here
 * - lint warnings removed
 *
 * 25.01.10  \main\194  VTeam-Khler
 * - Old typedefs tCSM_UUDT_RX/TX_CFG added and encapsulated with defines
 *   CSM_C_PTYPE_GM_UUDT_RX/TX (because the new type is incompatible with the
 *   old one and gmripper, ... have to be changed before new type can be used).
 *
 * 25.01.10  \main\195  Prhl
 * - CSM_C_TEL_TYPE_IVLAN_NM added.
 *
 * 26.01.10  \main\196  VTeam-Franke
 * - upstream define values added (only comment)
 *
 * 02.03.10  \main\197 Riecker
 * - add CSM_C_TRACE_FKT_CNM_PROCESS_NM_STATE for IVLAN
 *
 * 28.04.10  \main\198 VTeam-Pistoor
 * - Changed in structures tCSM_UUDT_RX_CFG and tCSM_UUDT_TX_CFG
 *   WORD awAddressField[1]; to WORD wAddress;
 *
 * 06.05.10  \main\199 Borck
 * - spelling error corrected and compatibility ensured by an additional define
 * - project dependend trace function IDs added (for OSEKI NM)
 *   CSM_C_TRACE_FKT_CNM_OSEKI_RX_NODE_MONITOR  = 0xC0
 *   CSM_C_TRACE_FKT_CNM_OSEKI_TX_MUTE_MONITOR  = 0xC1
 *   CSM_C_TRACE_FKT_CNM_OSEKI_BUS_OFF_MONITOR  = 0xC2
 *
 * 16.05.10  \main\200 Borck
 * - struct CSM_CAN_CON_DATA extended by element bBus for OSEKI node monitoring
 *
 * 15.06.10  \main\201 Borck
 * - another new trace ID for NM (0xC3) added.
 *
 * 07.09.10  \main\202  Feldhaus
 * - CSM extension for J1939 protocol
 *
 * 08.09.10  \main\203 VTeam-Pistoor
 * - Added comment define CSM_C_DOWN_REASON_CDA_LLD_LIN_TX_CON
 *
 * 17.09.10  \main\204 VTeam-Pistoor
 * - Added comment define CSM_C_TEL_TYPE_LIN_USDT_RX
 *
 * 27.09.10  \main\205  Feldhaus
 * - Change in tCSM_J1939_NODE: removed bus number, bugfix for NAME member
 *
 * 06.10.10  \main\206  VTeam-Khler
 * - Prototype CSM_STACK_bDetermineBusStateFromLine() added.
 *
 * 07.10.10  \main\207  VTeam-Franke
 * - ISO TP flags datatype WORD -> DWORD (enhance maximum channels to 32)
 *   use along csm.c \main\270
 *             cnp_itp.c  \main\70
 *                        \main\vector_adaptation\29
 *             cnp_itp_i.h \main\26
 *
 * 15.10.10  \main\208  Feldhaus
 * - J1939: extension for separate Rx and Tx buffers to enable parallel
 *   transfer
 *
 * 15.12.10  \main\209  Feldhaus
 * - added feature StartStop broadcasts (DM13)
 * - added PGN for language setting
 *
 * 21.12.10  \main\210  VTeam-Khler
 * - Structs tCSM_ISO_TP_USDT_APPL_CALLBACK_ADMIN,
 *   tCSM_UUDT_TX_APPL_CALLBACK_ADMIN and tCSM_UUDT_RX_APPL_CALLBACK_ADMIN added.
 *
 * 25.1.11  \main\211/212  VTeam-Khler
 * - Type for csm_itp_arCallbackFkt[] changed from
 *   tCSM_ISO_TP_USDT_APPL_CALLBACK to tCSM_ISO_TP_USDT_APPL_CALLBACK_ADMIN.
 *   csm_stack_i.h main\134, csm_itp.c main\64, cnp_itp.c update necessary.
 *
 * 28.1.11  \main\213  VTeam-Feldhaus
 * - Type for tCSM_J1939_PGN_TXDATA.bNextFrame and
 *   tCSM_J1939_PGN_RXDATA.bNextFrame changed to WORD. No further dependencies.
 *
 * 02.2.11  \main\214  VTeam-Feldhaus
 * - Layout for type tCSM_J1939_NAME changed; added variables for NAME fields.
 *
 * 31.03.11  \main\215  VTeam-Pistoor
 * - Moved macro CSM_M_MAKE_ERROR from csm_stack_i.h version 135, update also csm_stack_i.h!
 *
 * 06.04.11  \main\216  Borck
 * - Nissan LCN2 branch merged back to trunc:
 * - struct CSM_CAN_TX_DATA extended by dwTimeout under feature switch
 * - struct tCDA_CAN_DATA extended by dwTimeout under feature switch
 *
 * 12.05.11  \main\217  Prhl
 * - bExpectedDLC added to CSM_RX_ID_LIST
 *
 * 20.05.11  \main\218  Borck
 * - struct type tCSM_NWM_CFG has been extended by a monitor index for OSEKI NM
 *
 * 23.05.11  \main\219  Prhl
 * - #ifdef/#endif to remove lint warning.
 *
 * 08.06.11  \main\220  Borck
 * - some defines added which are used in csm.c for em_trace output
 * - 2 prototypes added, so that 2 new functions in csm.c can be used to set an
 *   em_trace related var.
 *
 * 08.06.11  \main\221  Borck
 * - em trace errors extended to DWORD so that or-ed values can be used
 *
 * 18.08.11  \main\222  Borck
 * - TelegramSubType introduced in several data structs, needed to be able
 *   to filter within data confirmations of the same TelegramType.
 *
 * 30.08.11  \main\223  Feldhaus
 * - renamed TgtNodeID and SrcNodeID variables for J1939 protocol to TgtAddr
 *   and SrcAddr.
 *
 * 09.09.11  \main\224  CM-AI/PJ-GM58-Pistoor
 * - Removed GMLAN project specific defines CSM_C_TRACE_FKT_CNM_GMLAN_SET_BUSSTATE,
 *   CSM_C_TRACE_FKT_CNM_GMLAN_GET_ACTIVE_VN and
 *   CSM_C_TRACE_FKT_CNM_GMLAN_GET_API_VN_REQ
 *
 * 29.09.11 \main\225 CM/PJ-H- Basavaraj patil
 * - CDA adaptations for UUDT configuration tables.
 *
 * 17.11.11 \main\226 Kollai / Borck
 * - prototypes for T-Kernel access layer and subsystem removed
 *
 * 21.06.12  \main\227  Khler
 * - UUDT types corrected for GM Gen2.
 * - Macro CSM_M_PARAMETER_INTENTIONALLY_UNUSED changed to void cast.
 *
 * 04.01.13  \main\228  Borck
 * - warnings fixed (merged from Dirk Janssens branch)
 *
 * 10.01.13  \main\229  Feldhaus
 * - added macro CSM_C_TEL_TYPE_CBR_TX_2QUEUE for backward compatibility with
 *   old Fiat SB08 CAN stack
 * - added macro CSM_C_ITP_AMODE_NORMAL_FIXED for Normal Fixed addressing mode
 *
 * 07.05.13  \main\230  Prhl
 * - define CSM_C_TEL_TYPE_VAG_EID added.
 *
 * 10.05.13  \main\231  Prhl
 * - define CSM_C_TEL_TYPE_VWHIGH_NM added.
 *
 * 17.06.13  \main\232  Pistoor
 * - Corrected macro CBR_CONVERT_FIAT_MSB_SIGNAL_LSBIT_LENTGH
 *
 * 26.06.13  \main\233  Pistoor
 * - Corrected and used macro CBR_CONVERT_FIAT_MSB_SIGNAL_LSBIT_LENTGH
 *
 * 05.08.13  \main\234  Prhl
 * - dwNmPayload added to tCSM_CBR_CFG if VWHIGH NM is active.
 *
 * 15.08.13  \main\235  Prhl
 * - define CSM_C_CAN_HW_OBJECTS_V850E2 added.
 *
 * 02.09.13  \main\236  Prhl
 * - corrected CBR_CONVERT_FIAT_MSB_SIGNAL_LSBIT and CBR_CONVERT_FIAT_MSB_SIGNAL_MSBIT macros
 *
 * 26.11.13  \main\237  Pistoor
 * - added CSM_C_TRACE_FKT_CSM_GET_ALL_SIGNALS_AGAIN
 *
 * 07.01.14  \main\238  Pistoor
 * - introduced CSM_M_UPSTREAM_TASK_LOCK, CSM_M_DOWNSTREAM_TASK_LOCK,
 *   CSM_M_UPSTREAM_TASK_UNLOCK and CSM_M_DOWNSTREAM_TASK_UNLOCK, to be defined in csm_stack_mp.h
 *   Typically interrupt disable/restore(but also can be something else like semaphore lock/unlock):
 *   #define CSM_M_UPSTREAM_TASK_LOCK() CSM_iInterruptsDisable()
 *   #define CSM_M_UPSTREAM_TASK_UNLOCK(iIsrStat) CSM_vInterruptsRestore(iIsrStat)
 *   #define CSM_M_DOWNSTREAM_TASK_LOCK() CSM_iInterruptsDisable()
 *   #define CSM_M_DOWNSTREAM_TASK_UNLOCK(iIsrStat) CSM_vInterruptsRestore(iIsrStat)
 *
 * 05.02.14  \main\239  Prhl
 * - dwNmPayload added to CSM_ISO_TP_CONT if VWHIGH NM is active.
 *
 * 14.02.14  \main\240  Prhl
 * - telegram type define for XCP added (CSM_C_TEL_TYPE_XCP_RX, CSM_C_TEL_TYPE_XCP_TX)
 *
 * 25.06.14  \main\241  Feldhaus
 * - added new field bCCSNPChannel in tCSM_J1939_PGN_CFG.
 *
 * 09.12.14  \main\242  Prhl
 * - LINT warning removed.
 *
 * 12.12.14  \main\243  Borck
 * - Feature CSM_S_CBR_MSG_ROLL_COUNTER introduced and required data struct added here.
 *
 * 04.03.15  \main\244  Pistoor
 * - removed IVLAN
 *
 * 10.03.15  \main\245  Borck
 * - small adaptions: MPDT added in CNP section, ISO-TP comments changed
 *
 * 10.03.15  \main\246  Pistoor
 * - made extension for second LIN bus
 *
 * 24.03.15  \main\247  Pistoor
 * - made correction for GM MY18
 *
 * 27.04.15  \main\248  Borck
 * - CSM internal function typedef for abstract signal feature moved here from project interface file
 * - some comments added.
 *
 * 27.04.15  \main\249  Borck
 * - comment from version 248 was incomplete:
 * - CSM internal typedef for abstract signal translation table entry moved here as well !!
 * - finally the last 2 changes require the definition of a new feature switch introduced for the latest cbr.c implementation (version 154)
 *   -- please add CSM_S_CBR_ABSTRACT_SIGNALS (set it to OFF) to your csm_stack_p.h (refer to the ITG5 or RN_AIVI branches of this file)
 *
 * 30.04.15  \main\250  Borck
 * - disabled some code that was featured with outdated CSM_STACK_ESD_SELF (replaced by CSM_STACK_ENV_SELF)
 * - introduced a data type tCSM_SLOT_CONFIG, featured with CSM_S_SLOT_BASED_CONFIG
 *   -- this is a comprehensive data struct with pointers to many configuration structs and arrays.
 *   -- one single parameter (the SLOT) now allows to switch the entire CSM configuration (currently msg-boxes, Rx, TX, oseki monitoring, abstract signals, rolling counter)
 *   -- this can/will be extended later on for more config stuff
 *
 * 05.05.15  \main\251  Borck
 * - no functional changes
 * - file structure reorganized a little bit by grouping content that belongs together
 * - fit in some BIG section comments to be able to find sections faster (cda, cnp, csm, cnm,)
 * - indention and comment style changed slightly
 *
 * 15.05.15  \main\252  Borck
 * - feature switch for variant management added if using SLOT based config
 *
 * 23.07.15  \main\253  Borck
 * - provide SLOT config struct for proxy component as well
 *
 * 17.09.15  \main\rn_aivi_generated\1  A.Borck
 * - SLOT configuration extended by 2 CBR related variables and 1 Gateway related one
 *
 * 08.10.15  \main\rn_aivi_generated\2  A.Borck
 * - feature mini abstraction (CSM_S_CBR_ABSTRACT_SIGNALS) removed completely
 *
 * 19.10.15  \main\rn_aivi_generated\3  A.Borck
 * - small LINT issue fixed.
 *
 * 28.10.15  \main\rn_aivi_generated\4  A.Borck
 * - MPDT config made const
 *
 * 29.02.16  \main\rn_aivi_generated\5  A.Borck
 * - new data type for CBR gateway introduced
 * - new pointer to CBR gateway configuration defined
 *
 * 27.05.16  \main\rn_aivi_generated\6  M.Prhl
 * - define CSM_S_CBR_MSG_ROLL_COUNTER changed to CSM_S_CBR_TX_MSG_ROLL_COUNTER
 *
 * 16.06.16  \main\rn_aivi_generated\7  A.Borck
 * - structure of tyCBR_GATEWAY_RULE changed
 *
 * 30.06.16  \main\rn_aivi_generated\8  A.Borck
 * - new telegram type CSM_C_TEL_TYPE_GATEWAY_TX_NON_FIFO introduced
 * - new telegram type CSM_C_TEL_TYPE_GATEWAY_TX_FIFO defined as CSM_C_TEL_TYPE_GATEWAY
 *
 * 09.08.16  \main\rn_aivi_generated\9  M.Prhl
 * - CSM_C_TRACE_FKT_CSM_TIMER_CYCLIC_START and CSM_C_TRACE_FKT_CSM_TIMER_CYCLIC_STOP added.
 *
 * 20.09.16  \main\rn_aivi_generated\10  A.Borck
 * - tCDA_GATEWAY_CFG extended by a time divider value for cycle reduction
 *
 * 29.09.16  \main\rn_aivi_generated\11  A.Borck
 * - added 2 new trace function IDs for local NVM access
 *
 * 07.11.16  \main\rn_aivi_generated\12  A.Borck
 * - added 2 new trace function IDs for MPDT join check support and config info support
 *
 * 11.11.16  \main\rn_aivi_generated\13  A.Borck
 * - define for trace buffer size decreased to 40
 *
 * 14.11.16  \main\rn_aivi_generated\14  A.Borck
 * - some trace related code made dependend on ETG feature since it is not needed for the corresponding subsystem (iMX or v850)
 *
 * 08.02.17  \main\rn_aivi_generated\15  A.Borck
 * - some trace function IDs added for NVM access
 *
 * 28.02.17  \main\rn_aivi_generated\16  A.Borck
 * - some trace function IDs removed/added
 *
 * 19.06.17  \main\rn_aivi_generated\17  A.Borck
 * - prototype array declarations do NOT need a size dimension
 * - check for amount of ITP connections > 32 only needed, if compiling CSM_SELF
 *
 * 07.07.17  \main\rn_aivi_generated\18  A.Borck
 * - new typedefs and pointers added for RxRollCount, TxCrc, RxCrc
 *
 * 16.08.17  \main\rn_aivi_generated\19  A.Borck
 * - TX Rolling Counter related structs extended for group support
 *
 * \history_end
 ****** END OF FILE ***********************************************************/
