/* Description and history of csm_stack_i.h at the end of the file   */
#ifndef CSM_STACK_I_HEADER
#define CSM_STACK_I_HEADER

/*******************************************************************************
 * includes:
 * Only csm_stack_datatypes.h allowed here.
 * csm_stack_ip.h included later in file after type definitions.
 ******************************************************************************/
#include "csm_stack_datatypes.h"

/*******************************************************************************
 * defines and macros (scope: CAN stack and application)
 ******************************************************************************/

/** logic name for bus number one*/
#define  CSM_C_CAN_1                          0
/** logic name for bus number two*/
#define  CSM_C_CAN_2                          1
/** logic name for bus number three*/
#define  CSM_C_CAN_3                          2
/** logic name for bus number four*/
#define  CSM_C_CAN_4                          3

#define  CSM_C_GENERAL                        0xFF

#define  CSM_C_NO_ERROR                       0L

#define  CSM_C_APPL_ID_DEFAULT                0xFFFF

#if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
  // Default handle value if no handler is used (kernel mode application) in systems with CSM access layer.
  #define CSM_C_NO_HANDLE_PTR      NULL                /**< Handle pointer value for kernel mode applications. */
  #define CSM_C_INVALID_HANDLE    ((void*)0x01L)       /**< Handle value if error in CSM_lApplCallbackPreInit() occures (e.g no resources). */
#endif

/* defines for xxx_lInit() */
#define  CSM_C_INIT_COLD          0
#define  CSM_C_INIT_WARM          1
#define  CSM_C_INIT_START         2
#define  CSM_C_INIT_ACTIVATE      3

/* defines for xxx_lExit() */
#define CSM_C_EXIT_STOP           0
#define CSM_C_EXIT_OFF            1

/* defines for CSM setup (init/exit )in t-engine platform */
#define CSM_C_STR_STARTCSM          "C" /**< This string forces an CSM initialisation */
#define CSM_C_STR_STARTSUBSYSTEM    "S" /**< This string forces the CSM subsystem initialization */
#define CSM_C_STR_SLEEP             "X" /**< This string forces the stack to go into the sleep mode with disabled interrupts */
#define CSM_C_STR_RESUME            "R" /**< This string forces the stack to go into the sleep mode with disabled interrupts */

/* defines for xxx_vDataCon() */
#define CSM_C_CONF_OK             0x00
#define CSM_C_CONF_FAULT          0x01

/* defines for xxx_lConTest() */
#define CSM_C_ENABLE              0x00
#define CSM_C_DISABLE             0x01


/* Defines for bus states. Should be used only in the CAN stack modules CSM and
 * CNM and not in CNP, CBR and CDA.
 * parameter of function yyy_vCSMBusStateInd.
 * Also state of the CAN stack manager state machine
 * attention: the last byte is reserved for substates!!!
 * See CAN-SWA Table CAN Stack Manager States. */

typedef enum
{
  _UNDEFINED            = 0x0000,
  _DOWN_____            = 0x0001,
  _LISTEN___            = 0x0002,
  _STARTUP__            = 0x0003,
  _UP_______            = 0x0004,
  _SHUTDOWN_            = 0x0005
} tCSM_BUS_STATE;

#define CSM_C_STATE_BUS_UNDEFINED   CSM_M_GET_LOBYTE(_UNDEFINED)  /**< State of this CAN bus is undefined/not initialized. */
#define CSM_C_STATE_BUS_DOWN        CSM_M_GET_LOBYTE(_DOWN_____)  /**< State of this CAN bus is down. The Transceiver is switched off. */
#define CSM_C_STATE_BUS_LISTEN      CSM_M_GET_LOBYTE(_LISTEN___)  /**< State of this CAN bus is in listen mode. Project specific restricted
                                                                    *  sending of frames (e.g. ISO-TP Flow Control) is possible. */
#define CSM_C_STATE_BUS_STARTUP     CSM_M_GET_LOBYTE(_STARTUP__)  /**< State of this CAN bus is in (project specific) startup mode. */
#define CSM_C_STATE_BUS_UP          CSM_M_GET_LOBYTE(_UP_______)  /**< State of this CAN bus is up. SPM|application|CSM ... keeps bus alive. */
#define CSM_C_STATE_BUS_SHUTDOWN    CSM_M_GET_LOBYTE(_SHUTDOWN_)  /**< State of this CAN bus is in shutdown mode. The node do not need the bus
                                                                    *   but external CAN nodes keeps it still alive. */

/* defines for CBR_vSetRxGrpMask() / CBR_vClrRxGrpMask */
#define  CSM_C_RX_GROUP_NONE      0x00
#define  CSM_C_RX_GROUP_ALL       0xff
#define  CSM_C_RX_GROUP_1         0x01
#define  CSM_C_RX_GROUP_2         0x02
#define  CSM_C_RX_GROUP_3         0x04
#define  CSM_C_RX_GROUP_4         0x08
#define  CSM_C_RX_GROUP_5         0x10
#define  CSM_C_RX_GROUP_6         0x20
#define  CSM_C_RX_GROUP_7         0x40
#define  CSM_C_RX_GROUP_8         0x80

#define CSM_M_NUMBER_OF_CBR_SIGNALS(structure) \
            (sizeof(structure) / sizeof(CSM_SIGNAL))

#define CSM_M_NUMBER_OF_CBR_MULTIPLE_SIGNALS(structure) \
            (sizeof(structure) / sizeof(CSM_MULTIPLE_SIGNAL))

/******************* Signal ID ***********************************
   Byte 3    |  Byte 2    |  Byte 1   |  Byte 0
   1111 1111 | 11 11 1111 | 1111 1111 | 1111 1111
  |rrrr rrrr   rr|ss ssss | rrrr rtqv | iiii iiii
  |              |        |       | | |
  |<------------>|<------>|<----->| | |<-------->
  |              |        |       | |      |
  |              |        |       | |      ----> i: Telegram Index
  |              |        |       | |----------> v: Virtual Signal (Not used)
  |              |        |       |------------> q: Queued Signal (Not used)
  |              |        |--------------------> r: Reserved bits
  |              |-----------------------------> s: Signal index
  |--------------------------------------------> r: Reserved bits,
  |                                                 bits 30 and 31 possibly for bus number

******************************************************************/

#define CSM_M_MAKE_VIRTUAL_SIGNAL_ID(SignalId) \
            (DWORD)((SignalId & 0x3F) + 0x100)

#define CSM_M_IS_VIRTUAL_SIGNAL_ID(SignalId) \
            ((SignalId & 0x100) != 0)

#define CSM_M_GET_VIRTUAL_SIGNAL_ID(SignalId) \
            (SignalId & 0xFF)

#define CSM_M_SET_QUEUED_SIGNAL_ID(SignalId) \
            (SignalId + 0x200)

#define CSM_M_CLR_QUEUED_SIGNAL_ID(SignalId) \
            (SignalId - 0x200)

#define CSM_M_IS_QUEUED_SIGNAL_ID(SignalId) \
            ((SignalId & 0x200) != 0)

#define CSM_M_MAKE_SIGNAL_ID(TelIndex, SingleSignalId) \
            (DWORD)(((SingleSignalId & 0x3F) << 16) | (TelIndex & 0xFF)  )

#define CSM_M_GET_BUS_NO_FROM_SIGNAL_ID(SignalId) \
            ((SignalId & 0xC0000000) >> 30)

#define CSM_M_GET_TELINDEX_FROM_SIGNAL_ID(SignalId) \
            (SignalId & 0x000000FF)

#define CSM_M_GET_SIGINDEX_FROM_SIGNAL_ID(SignalId) \
            ((SignalId & 0x003F0000)>>16)

#define CSM_M_GET_BUSNUMBER_FROM_SIGNAL_ID(SignalId) \
            ((SignalId & 0xC0000000)>>30)


/* this index is to be used on an early development state if signals used by
 * the applications shall be removed but the code shall be compilable.
 * Usage of such defined signals will raise an Assert in cbr.c
 */
#define CBR_C_DEPRECATED_IDX    0xFF

// Signal State Information (for ETG tracing)
typedef enum
{
  _ARRIVED                             = 0x00000001UL,  /**< Signal received      */
  _ARRIVED_FIRST                       = 0x00000002UL,  /**< First time reception */
  _ARRIVED_FIRST__ARRIVED              = 0x00000003UL,  // assembled value
  _DATA_INVALID                        = 0x00000004UL,  /**< data invalid         */
  _TIMEOUT                             = 0x00000008UL,  /**< receive Timeout      */
  _ARRIVED__TIMEOUT                    = 0x00000009UL,  // assembled value
  _ARRIVED_FIRST__ARRIVED__TIMEOUT     = 0x0000000BUL,  // assembled value
  _NOT_RECEIVED                        = 0x00000010UL,  /**< no signal received   */
  _NOT_RECEIVED__DATA_INVALID__TIMEOUT = 0x0000001CUL,  // assembled value
  _DATA_OUTDATED                       = 0x00000020UL,  /**< bus sleep, no actual value */
  _RETRIGGERED                         = 0x00000040UL,  /**< forced signal indication   */
  _SIGNAL_HAS_NO_DATA_SOURCE           = 0x00000080UL,  // used by CBR_AL_lSignalRead() for abstract Rx signals which don't have an associated "physical" CSM source signal
  _DATA_QUEUED                         = 0x00000100UL,  /**< signal/message is read from queue */
  _DATA_QUEUE_OVERRUN                  = 0x00000200UL,  /**< signal/message is read from queue */
  _DATA_CRC_MISSMATCH                  = 0x00000400UL,  /**< data CRC check failed */
  _ROLL_COUNTER_FAIL                   = 0x00000800UL   /**< rolling counter check failes */

} tCSM_SIGNAL_STATUS_INFO;

// defines for signal states (only the real bits are derived from ETG enum)
#define CSM_C_SIGNAL_ARRIVED            ((DWORD) _ARRIVED )
#define CSM_C_SIGNAL_ARRIVED_FIRST      ((DWORD) _ARRIVED_FIRST )
#define CSM_C_SIGNAL_DATA_INVALID       ((DWORD) _DATA_INVALID )
#define CSM_C_SIGNAL_TIMEOUT            ((DWORD) _TIMEOUT )
#define CSM_C_SIGNAL_NOT_RECEIVED       ((DWORD) _NOT_RECEIVED )
#define CSM_C_SIGNAL_DATA_OUTDATED      ((DWORD) _DATA_OUTDATED )
#define CSM_C_SIGNAL_RETRIGGERED        ((DWORD) _RETRIGGERED )
#define CSM_C_SIGNAL_HAS_NO_DATA_SOURCE ((DWORD) _SIGNAL_HAS_NO_DATA_SOURCE )
#define CSM_C_SIGNAL_DATA_QUEUED        ((DWORD) _DATA_QUEUED )
#define CSM_C_SIGNAL_DATA_QUEUE_OVERRUN ((DWORD) _DATA_QUEUE_OVERRUN )
#define CSM_C_SIGNAL_DATA_CRC_MISSMATCH ((DWORD) _DATA_CRC_MISSMATCH )
#define CSM_C_SIGNAL_ROLL_COUNTER_FAIL  ((DWORD) _ROLL_COUNTER_FAIL )

/* ------------------------------------------------------------------------- */
/* defines for structure CSM_SIGNAL */
// abstract indicator introduced in level info (bit 7) !!!
#define CSM_C_ALWAYS_IND                 0x01  /**< always indicate after reception */
#define CSM_C_IND_AFTER_CHANGE           0x02  /**< indicate only, if value has changed */
#define CSM_C_IND_AFTER_CHANGE_ABSTRACT  0x82  /**< indicate only, if value has changed, signal is abstract !!!! */
#define CSM_C_NEVER_IND                  0x03  /**< indicate never */
#define CSM_C_CALLBACK_DIRECT            0x10  /**< indicate direct from CBR_vCDAReceiveInd (on PCAN level) */
#define CSM_C_CALLBACK_LEVEL_MASK        0x0F  /**< mask for indication modes without level info or abstraction indicator*/
#define CSM_C_ABSTRACTION_MASK           0x80  /**< mask for abstraction indication of a signal */

/** endians for signals */
#define CSM_C_SIGNAL_ENDIANS_LITTLE         0  /**> signal has little endian */
#define CSM_C_SIGNAL_ENDIANS_BIG            1  /**> signal has big endian */

/* defines for CBR_lSignalWrite() */
#define CSM_C_TX_CYCLIC                  0x01  /**< transmit cyclic */
#define CSM_C_TX_IMMEDIATELY             0x02  /**< transmit immediately */
#define CSM_C_TX_BUS_WAKE_TEMP           0x04  /**< transmit with Bus wake temporary */
#define CSM_C_TX_FCT_ACTIVE              0x08  /**< transmit and set function active */
#define CSM_C_TX_AS_CONFIGURED           0x10  /**< transmit, like configured */
#define CSM_C_TX_STORE_ONLY              0x20  /**< no transmit trigger, store value */
#define CSM_C_TX_NO_TRANSMISSION         0x40  /**< no transmission, for spontan messages */
#define CSM_C_TX_TYPE_VALID_MASK         0x7F  /**< used to check the TX type */

/* -------------------------------------------------------------------------- */
/* Defines for CNP connections (attention: ITP uses bitfield to store the     */
/* value, see csm_stack_m.h)                                                  */
/* Defines used for CSM_lCommunicationReq/Con/Ind() and/or main state         */
/* variables (depending on the protocol type).                                */

// Connection state (for ETG tracing)
typedef enum
{
  _INACTIVE_____  = 0x00,                 /**< "INACTIVE" (not initialized) state  */
  _CONNECTED____  = 0x01,                 /**< Generic "CONNNECTED" state          */
  _DISCONNECTED_  = 0x02                  /**< Generic "DISCONNECTED" state        */
} tCSM_CONNECTION_STATE;

#define CSM_C_INACTIVE        _INACTIVE_____
#define CSM_C_CONNECTED       _CONNECTED____
#define CSM_C_DISCONNECTED    _DISCONNECTED_

#define CSM_C_CONNECT         CSM_C_CONNECTED          /**< Request for connection. */
#define CSM_C_DISCONNECT      CSM_C_DISCONNECTED       /**< Request for disconnect. */

#if (CSM_S_DEACTIVATE_COMMUNICATION_REQUESTS == CSM_C_F_ON)
  #define CSM_C_DEACTIVATE_COMMUNICATION_REQUESTS 0x03 /**< Request for ignoring all communication requests. */
  #define CSM_C_ACTIVATE_COMMUNICATION_REQUESTS   0x04 /**< Request for process all communication requests. */
#endif // CSM_S_DEACTIVATE_COMMUNICATION_REQUESTS

#define CSM_C_STACK_UP        CSM_C_CONNECTED
#define CSM_C_STACK_DOWN      CSM_C_DISCONNECTED

#define CSM_C_STARTUP         0x06                     /**< "STARTUP" state                     */
#define CSM_C_SHUTDOWN        0x07                     /**< "SHUTDOWN state                     */

#define CSM_C_NORMAL_COMMUNICATION_ON   0x08           /**< Normal CAN communication allowed. */
#define CSM_C_NORMAL_COMMUNICATION_OFF  0x09           /**< Only specific diagnotic communication allowed (e.g. to keep bandwidth free for download). */

/* --- VW BR extensions. ---------------------------------------------------- */
#define CSM_C_FNL_ON                      0x05
#define CSM_C_CONNECT_FKT_NACHLAUF        CSM_C_FNL_ON
#define CSM_C_FNL_OFF                     CSM_C_DISCONNECT
#define CSM_C_CONNECTED_FKT_NACHLAUF      CSM_C_CONNECT_FKT_NACHLAUF

#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)

  /* for future use (for example VAG TP connect/disconnect) */
  #define CSM_C_CONNECT_READY              0x11   /**< ?? */
  #define CSM_C_CONNECT_NOT_READY          0x12   /**< ?? */
  #define CSM_C_CONNECT_VWTP16_STD         0x13   /**< ?? */
  #define CSM_C_CONNECT_VWTP16_DIRTY       0x14   /**< ?? */

  #if ( CSM_S_VW_TP_2_0 == CSM_C_F_ON )
    /* --- TP 2.0 Connection type of TP channel --------------------- */
    #define CSM_C_CONNECT_STATIC     0x0             /**< VW TP 20 static type of channel. */
    #define CSM_C_CONNECT_DYNAMIC    0x1             /**< VW TP 20 dynamic type of channel */
  #endif

  #if (CSM_S_MCNET == CSM_C_F_ON)
    /* MCNet specific connection states and actions */
    #define CSM_C_CONNECTED_MCNET_STD    CSM_C_CONNECTED     /**< "Standard" MCNet connection */
    #define CSM_C_DISCONNECTED_MCNET     CSM_C_DISCONNECTED
    #define CSM_C_CONNECTED_MCNET_ENH    0x03                /**< "Enhanced" MCNet connection */

    #define CSM_C_CONNECT_MCNET_STD    CSM_C_CONNECTED_MCNET_STD  /**< "Standard" MCNet connection */
    #define CSM_C_DISCONNECT_MCNET     CSM_C_DISCONNECTED_MCNET
    #define CSM_C_CONNECT_MCNET_ENH    CSM_C_CONNECTED_MCNET_ENH  /**< "Enhanced" MCNet connection */

    /*------ MCNet ASDT Service Modes ----------------------------------*/
    #define CSM_MCNET_C_SM_STANDARD    0x00  /**< Standard Service Mode   */
    #define CSM_MCNET_C_SM_ENHANCED    0x01  /**< Enhanced Service Mode   */

    /*--- MCNet Logical Components (according to MCNet specification) --*/
    #define CSM_C_MCNET_LC_MAS   0x01  /**< (HMI) Master                  */
    #define CSM_C_MCNET_LC_MAS1  0x01  /**< Master 1, synonymous with MAS */
    #define CSM_C_MCNET_LC_MAS2  0x02  /**< Second (HMI) Master           */
    #define CSM_C_MCNET_LC_MAS3  0x03  /**< Third (HMI) Master            */
    #define CSM_C_MCNET_LC_TES   0x07  /**< Test Node                     */
    #define CSM_C_MCNET_LC_CDC   0x08  /**< Compact Disc Changer          */
    #define CSM_C_MCNET_LC_CAS   0x10  /**< Compact Cass. (obsolete, name "DRV" to be used instead!) */
    #define CSM_C_MCNET_LC_DRV   0x10  /**< (Single) Drive Unit (incl. Cassette, CD)  */
    #define CSM_C_MCNET_LC_AUT   0x18  /**< Audio Tuner                   */
    #define CSM_C_MCNET_LC_TMC   0x1C  /**< Traffic Message Channel Unit  */
    #define CSM_C_MCNET_LC_PRA   0x20  /**< Audio Pre-Amplifier           */
    #define CSM_C_MCNET_LC_POW   0x24  /**< Audio Power Amplifier         */
    #define CSM_C_MCNET_LC_DAB   0x28  /**< Digital Audio Broadcasting    */
    #define CSM_C_MCNET_LC_DMB   0x2C  /**< Digital Multimedia Broadcast. */
    #define CSM_C_MCNET_LC_CON   0x30  /**< (Remote Control)              */
    #define CSM_C_MCNET_LC_CON2  0x32  /**< (Remote Control 2)            */
    #define CSM_C_MCNET_LC_AVS   0x38  /**< Audio Video Switch Unit       */
    #define CSM_C_MCNET_LC_ICL   0x40  /**< Instrument Cluster            */
    #define CSM_C_MCNET_LC_NAV   0x48  /**< Navigation Unit               */
    #define CSM_C_MCNET_LC_DNL   0x4C  /**< Download Unit                 */
    #define CSM_C_MCNET_LC_TEL   0x50  /**< Telephone Unit                */
    #define CSM_C_MCNET_LC_CRD   0x54  /**< Card Reader                   */
    #define CSM_C_MCNET_LC_VID   0x60  /**< Video Decoder                 */
    #define CSM_C_MCNET_LC_VIT   0x64  /**< Video Tuner                   */
    #define CSM_C_MCNET_LC_VIB   0x68  /**< Video Box                     */
    #define CSM_C_MCNET_LC_SPR   0x70  /**< Speech Recognition            */
    #define CSM_C_MCNET_LC_IPS   0x78  /**< Intelligent Prediction System */
    #define CSM_C_MCNET_LC_DSP   0x90  /**< Digital Signal Processor      */
    #define CSM_C_MCNET_LC_DTE   0xB0  /**< Data Teminal Equipment        */
    #define CSM_C_MCNET_LC_DCE   0xB8  /**< Data Circuit Equipm. (Modem)  */
    #define CSM_C_MCNET_LC_CHP   0xC8  /**< Chipcard Unit                 */
    #define CSM_C_MCNET_LC_DPL   0xD0  /**< (Secondary) Diplay            */
    #define CSM_C_MCNET_LC_GAT   0xE8  /**< Gateway                       */
  #endif /* CSM_S_MCNET */

/* defines for yyy_vCSM_DataErrorInd: */
#define CSM_C_MULTI_FRAME_RECEPTION_ABORTED         0x0001
#define CSM_C_TELEGRAM_TIMEOUT                      0x0002

  #if (CSM_S_J1939 == CSM_C_F_ON)

    /* extension for xxx_vDataCon() that is required by J1939 */
    #define CSM_C_CONF_REQ                          0x02
    #define CSM_C_CONF_ACK                          0x03
    #define CSM_C_CONF_NACK                         0x04
    #define CSM_C_CONF_ACCDENIED                    0x05
    #define CSM_C_CONF_BUSY                         0x06
    #define CSM_C_CONF_NONE                         0x07

    #define CSM_C_POSACK_LEN                        0xFFFE
    #define CSM_C_NEGACK_LEN                        0xFFFD
    #define CSM_C_ACCDENIED_LEN                     0xFFFC
    #define CSM_C_CANNOTRESP_LEN                    0xFFFB

  #endif

#endif // CSM_S_CNP_AVAILABLE

/* defines for yyy_vCSMBusErrorInd: */
#define CSM_C_BUS_ON             0x0
#define CSM_C_BUS_OFF            0x1
#define CSM_C_TRANSC_ERR_STABLE  0x2
#define CSM_C_NO_NETWORK         0x4

/** Macro for error code generation.        \n
 *  global error code:                      \n
 *  description          position 12345678  \n
 *  EG: Error Grade (g)           gg        \n
 *  EL: Error Location (l)          ll      \n
 *  EC: error code (e)                eeee
 */
/** macro for regeneration of error grade (EG) */
#define CSM_M_GET_EG(Error) ( (BYTE) ((((DWORD)(-Error)) & 0xFF000000) >> 24) )

/** macro for regeneration of error location (EL) */
#define CSM_M_GET_EL(Error) ( (BYTE) ((((DWORD)(-Error)) & 0x00FF0000) >> 16) )

/** macro for regeneration of error code (EC) */
#define CSM_M_GET_EC(Error) ( (WORD)  (((DWORD)(-Error)) & 0x0000FFFF) )

// Errors
typedef enum
{
  _NO_ERROR_____ =  0x00,
  _WARNING______ =  0x01,
  _ERROR________ =  0x02,
  _SEVERE_ERROR_ =  0x03
} tCSM_ERROR_GRADE;

// defines for error grade
#define CSM_C_ERR_WARNING        ((BYTE) _WARNING______)
#define CSM_C_ERR_ERROR          ((BYTE) _ERROR________)
#define CSM_C_ERR_SEVERE_ERROR   ((BYTE) _SEVERE_ERROR_)

// Error locations
typedef enum
{
  _GLB_________ = 0x00, /**< global or unspecified error location */
  _CSM_________ = 0x10, /**< Error location CAN Stack Manager */
  _CNM_________ = 0x20, /**< Error location CAN network Management */
  _CNM_MCNET___ = 0x21, /**< Error location CNM: MCNet NM    */
  _CNM_OSEK____ = 0x22, /**< Error location CNM: OSEK NM    */
  _CNM_GMLAN___ = 0x23, /**< Error location CNM: GMLAN NM    */
  _CNM_IVLAN___ = 0x24, /**< Error location CNM: IVLAN NM    */
  _CNM_J1939___ = 0x25, /**< Error location CNM: J1939 NM    */
  _CNM_VWHIGH__ = 0x26, /**< Error location CNM: VWHIGH NM    */
  _CNM_OSEKI___ = 0x27, /**< Error location CNM: OSEK NM    */
  _CNP_________ = 0x30, /**< Error location CNP (interface) */
  _CNP_MCNET___ = 0x31, /**< Error location CNP: MCNet TP       */
  _CNP_ISO_TP__ = 0x32, /**< Error location CNP: ISO TP         */
  _CNP_VW16_TP_ = 0x33, /**< Error location CNP: VW/VAG TP 1.6  */
  _CNP_VW20_TP_ = 0x34, /**< Error location CNP: VAG/VAG TP 2.0 */
  _CNP_BAP_____ = 0x35, /**< Error location CNP: VAG/BAP  */
  _CNP_UUDT____ = 0x36, /**< Error location CNP: UUDT  */
  _CNP_MPDT____ = 0x37, /**< Error location CNP: MPDT  */
  _CNP_LIN_DIAG = 0x38, /**< Error location CNP: LIN DIAG  */
  _CNP_LCTR____ = 0x39, /**< Error location CNP: LCTR  */
  _CNP_J1939_TP = 0x3A, /**< Error location CNP: J1939  */
  _CNP_LIN_USDT = 0x3B, /**< Error location CNP: LIN USDT  */
  _CBR_________ = 0x40, /**< Error location CAN BRoadcast server */
  _CDA_________ = 0x50, /**< Error location CAN Driver Access (manager) */
  _CDA_FCAN____ = 0x51, /**< Error location CAN controller FCAN */
  _CDA_TUAREG__ = 0x52, /**< Error location CAN controller TUAREG */
  _CDA_ARION___ = 0x53, /**< Error location CAN controller ARION/HECC */
  _CDA_AFCAN___ = 0x54, /**< Error location CAN controller AFCAN */
  _CDA_FCN_____ = 0x55, /**< Error location CAN controller FCN */
  _CSM_STACK___ = 0x60, /**< Error location CAN Stack */
  _CSM_APL_____ = 0x70, /**< Error location csm_apl.c */
  _CSM_AL______ = 0x80, /**< Error location csm access layer (general) */
  _CSM_AL_KRNL_ = 0x81, /**< Error location csm_access_krnl.c, csmal_krnl_os.c, csmal_krnl_trace.c */
  _CSM_AL_USR__ = 0x82, /**< Error location csm_access_usr.c */
  _CSM_TRACE___ = 0x90  /**< Error location csm trace */
} tCSM_ERROR_LOCATION;

// defines for error location
#define CSM_C_ERR_LOC_GLB          ((BYTE) _GLB_________)
#define CSM_C_ERR_LOC_CSM          ((BYTE) _CSM_________)
#define CSM_C_ERR_LOC_CNM          ((BYTE) _CNM_________)
#define CSM_C_ERR_LOC_CNM_MCNET    ((BYTE) _CNM_MCNET___)
#define CSM_C_ERR_LOC_CNM_OSEK     ((BYTE) _CNM_OSEK____)
#define CSM_C_ERR_LOC_CNM_GMLAN    ((BYTE) _CNM_GMLAN___)
#define CSM_C_ERR_LOC_CNM_IVLAN    ((BYTE) _CNM_IVLAN___)
#define CSM_C_ERR_LOC_CNM_J1939    ((BYTE) _CNM_J1939___)
#define CSM_C_ERR_LOC_CNM_VWHIGH   ((BYTE) _CNM_VWHIGH__)
#define CSM_C_ERR_LOC_CNM_OSEKI    ((BYTE) _CNM_OSEKI___)
#define CSM_C_ERR_LOC_CNP          ((BYTE) _CNP_________)
#define CSM_C_ERR_LOC_CNP_MCNET    ((BYTE) _CNP_MCNET___)
#define CSM_C_ERR_LOC_CNP_ISO_TP   ((BYTE) _CNP_ISO_TP__)
#define CSM_C_ERR_LOC_CNP_VW16_TP  ((BYTE) _CNP_VW16_TP_)
#define CSM_C_ERR_LOC_CNP_VW20_TP  ((BYTE) _CNP_VW20_TP_)
#define CSM_C_ERR_LOC_CNP_BAP      ((BYTE) _CNP_BAP_____)
#define CSM_C_ERR_LOC_CNP_UUDT     ((BYTE) _CNP_UUDT____)
#define CSM_C_ERR_LOC_CNP_MPDT     ((BYTE) _CNP_MPDT____)
#define CSM_C_ERR_LOC_CNP_LIN_DIAG ((BYTE) _CNP_LIN_DIAG)
#define CSM_C_ERR_LOC_CNP_LCTR     ((BYTE) _CNP_LCTR____)
#define CSM_C_ERR_LOC_CNP_J1939_TP ((BYTE) _CNP_J1939_TP)
#define CSM_C_ERR_LOC_CNP_LIN_USDT ((BYTE) _CNP_LIN_USDT)
#define CSM_C_ERR_LOC_CBR          ((BYTE) _CBR_________)
#define CSM_C_ERR_LOC_CDA          ((BYTE) _CDA_________)
#define CSM_C_ERR_LOC_CDA_FCAN     ((BYTE) _CDA_FCAN____)
#define CSM_C_ERR_LOC_CDA_TUAREG   ((BYTE) _CDA_TUAREG__)
#define CSM_C_ERR_LOC_CDA_ARION    ((BYTE) _CDA_ARION___)
#define CSM_C_ERR_LOC_CDA_AFCAN    ((BYTE) _CDA_AFCAN___)
#define CSM_C_ERR_LOC_CDA_FCN      ((BYTE) _CDA_FCN_____)
#define CSM_C_ERR_LOC_CSM_STACK    ((BYTE) _CSM_STACK___)
#define CSM_C_ERR_LOC_CSM_APL      ((BYTE) _CSM_APL_____)
#define CSM_C_ERR_LOC_CSM_AL       ((BYTE) _CSM_AL______)
#define CSM_C_ERR_LOC_CSM_AL_KRNL  ((BYTE) _CSM_AL_KRNL_)
#define CSM_C_ERR_LOC_CSM_AL_USR   ((BYTE) _CSM_AL_USR__)
#define CSM_C_ERR_LOC_CSM_TRACE    ((BYTE) _CSM_TRACE___)

/* Macro for errors with global error location:
 * if necessary use CSM_M_.... and not GLB_
   #define GLB_M_MAKE_ERROR(SecurityLevel,ErrorCode)            \
  CSM_M_MAKE_ERROR(SecurityLevel,CSM_C_ERR_LOC_GLB,ErrorCode)*/

/** ----------------------------------------------------------------
   list of global error codes (includes IPNET I/O conformant codes)
    -> these codes may be used together with any error location!
    used region: 0x0001 - 0x0FFF
---------------------------------------------------------------------*/

typedef enum
{
  _NO_ERROR______________  = 0x0000,
  _GENERAL_ERROR_________  = 0x0001,  /**< General (unspecified) error          */
  _OS_ERROR______________  = 0x0002,  /**< Operating system error               */
  _I_O_DEVICE____________  = 0x0003,  /**< General I/O device error             */
  _RESOURCE______________  = 0x0004,  /**< Resource problem (memory, timer,..)  */
  _BUFFER_ERROR__________  = 0x0010,  /**< General buffer error (e.g. not available) */
  _BUFFER_FULL___________  = 0x0011,  /**< Buffer full error                    */
  _RX_BUFFER_OVERRUN_____  = 0x0012,  /**< Rx Buffer overwritten                */
  _TX_BUFFER_OVERRUN_____  = 0x0013,  /**< Tx Buffer overwritten                */
  _TIMER_ERROR___________  = 0x0014,  /**< General timer error (e.g. not available) */
  _TIMER_ID_NOT_VALID____  = 0x0015,  /**< Timer ID not valid                   */
  _TIMER_VALUE_NOT_VALID_  = 0x0016,  /**< Timer value not valid                */
  _CONFIGURATION_________  = 0x0020,  /**< General data configuration error     */
  _CONFIG_FILE___________  = 0x0021,  /**< Error in configuration file          */
  _NO_CONFIG_FILE________  = 0x0022,  /**< No configuration file found          */
  _INIT_WARNING__________  = 0x00FF,  /**< Standard initialization of lReturn   */
  _UNKNOWN_SRVC__________  = 0x0101,  /**< Service unknown (e.g. for requests)  */
  _COMP_INACTIVE_________  = 0x0102,  /**< Component inactive (e.g. "power off" */
  _INVALID_PARA__________  = 0x0103,  /**< Invalid parameter                    */
  _SRVC_UNAVAILABLE______  = 0x0104,  /**< Service not available                */
  _NULL_POINTER__________  = 0x0105,  /**< (Illegal) null pointer               */
  _INVALID_PROTOCOL_TYPE_  = 0x0106,  /**< Invalid protocol type                */
  _DATA_LENGTH_EXCEEDED__  = 0x0107,  /**< Too much data (e.g. for tx/rx)       */
  _ITEM_NOT_FOUND________  = 0x0108,  /**< Item not found in search list        */
  _INVALID_BUS_NUMBER____  = 0x0110,  /**< Bus number not valid                 */
  _INVALID_CHANNEL_______  = 0x0111,  /**< Invalid channel / connection number  */
  _INVALID_ADDRESS_______  = 0x0112,  /**< Invalid address                      */
  _INVALID_TARGET_ADDR___  = 0x0113,  /**< Invalid target address               */
  _INVALID_SOURCE_ADDR___  = 0x0114,  /**< Invalid source address               */
  _CSM_STATE_FORBIDS_CALL  = 0x0115,  /**< (CSM-API) call not allowed in this state */
  _CALLBACK_MISSING______  = 0x0116,  /**< Invalid request call. Mandantory callback functions are not registered */
  _NO_FREE_CALLBACK______  = 0x0117,  /**< callback registration not possible, maximum number of callbacks already used */
  _WRONG_BUS_STATE_______  = 0x0118,  /**< (CSM-API) wrong bus state e.g. DataReq during sleep */
  _INVALID_CAN_DLC_______  = 0x0120,  /**< Invalid CAN data length code         */
  _INVALID_PCI___________  = 0x0121,  /**< Invalid protocol control information */
  _TX_QUEUE_OVERRUN______  = 0x0128,  /**< Overrun of TX queue                  */
  _RX_QUEUE_OVERRUN______  = 0x0129,  /**< Overrun of RX queue                  */
  _UNKNOWN_EVENT_________  = 0x0130,  /**< Unknown event (e.g. in state machine)*/
  _UNKNOWN_STATE_________  = 0x0131,  /**< Unknown state (e.g. in state machine)*/
  _INVALID_EVENT_________  = 0x0132,  /**< Invalid event (e.g. in state machine)*/
  _INVALID_STATE_________  = 0x0133,  /**< Invalid state (e.g. in state machine)*/
  _INVALID_SIG_INDEX_____  = 0x0141,  /**< Invalid signal index                 */
  _INVALID_TEL_INDEX_____  = 0x0142,  /**< Invalid telegram index               */
  _STACK_INACTIVE________  = 0x0200,  /**< Stack not active (e.g. not initial.) */
  _STACK_BUSY____________  = 0x0201,  /**< Stack busy (e.g. no Data.Con issued) */
  _NO_CONNECTION_________  = 0x0202,  /**< Connection not established/broken    */
  _BUS_OFF_______________  = 0x0204   /**< CAN controller in state "bus off"    */
} tCSM_ERROR_CODE;

#define CSM_E_GENERAL_ERROR            ((WORD) _GENERAL_ERROR_________)
#define CSM_E_OS_ERROR                 ((WORD) _OS_ERROR______________)
#define CSM_E_I_O_DEVICE               ((WORD) _I_O_DEVICE____________)
#define CSM_E_RESOURCE                 ((WORD) _RESOURCE______________)
#define CSM_E_BUFFER_ERROR             ((WORD) _BUFFER_ERROR__________)
#define CSM_E_BUFFER_FULL              ((WORD) _BUFFER_FULL___________)
#define CSM_E_RX_BUFFER_OVERRUN        ((WORD) _RX_BUFFER_OVERRUN_____)
#define CSM_E_TX_BUFFER_OVERRUN        ((WORD) _TX_BUFFER_OVERRUN_____)
#define CSM_E_TIMER_ERROR              ((WORD) _TIMER_ERROR___________)
#define CSM_E_TIMER_ID_NOT_VALID       ((WORD) _TIMER_ID_NOT_VALID____)
#define CSM_E_TIMER_VALUE_NOT_VALID    ((WORD) _TIMER_VALUE_NOT_VALID_)
#define CSM_E_CONFIGURATION            ((WORD) _CONFIGURATION_________)
#define CSM_E_CONFIG_FILE              ((WORD) _CONFIG_FILE___________)
#define CSM_E_NO_CONFIG_FILE           ((WORD) _NO_CONFIG_FILE________)
#define CSM_E_INIT_WARNING             ((WORD) _INIT_WARNING__________)
#define CSM_E_UNKNOWN_SRVC             ((WORD) _UNKNOWN_SRVC__________)
#define CSM_E_COMP_INACTIVE            ((WORD) _COMP_INACTIVE_________)
#define CSM_E_INVALID_PARA             ((WORD) _INVALID_PARA__________)
#define CSM_E_SRVC_UNAVAILABLE         ((WORD) _SRVC_UNAVAILABLE______)
#define CSM_E_NULL_POINTER             ((WORD) _NULL_POINTER__________)
#define CSM_E_INVALID_PROTOCOL_TYPE    ((WORD) _INVALID_PROTOCOL_TYPE_)
#define CSM_E_DATA_LENGTH_EXCEEDED     ((WORD) _DATA_LENGTH_EXCEEDED__)
#define CSM_E_ITEM_NOT_FOUND           ((WORD) _ITEM_NOT_FOUND________)
#define CSM_E_INVALID_BUS_NUMBER       ((WORD) _INVALID_BUS_NUMBER____)
#define CSM_E_INVALID_CHANNEL          ((WORD) _INVALID_CHANNEL_______)
#define CSM_E_INVALID_ADDRESS          ((WORD) _INVALID_ADDRESS_______)
#define CSM_E_INVALID_TARGET_ADDRESS   ((WORD) _INVALID_TARGET_ADDR___)
#define CSM_E_INVALID_SOURCE_ADDRESS   ((WORD) _INVALID_SOURCE_ADDR___)
#define CSM_E_CSM_STATE_FORBIDS_CALL   ((WORD) _CSM_STATE_FORBIDS_CALL)
#define CSM_E_CALLBACK_MISSING         ((WORD) _CALLBACK_MISSING______)
#define CSM_E_NO_FREE_CALLBACK         ((WORD) _NO_FREE_CALLBACK______)
#define CSM_E_WRONG_BUS_STATE          ((WORD) _WRONG_BUS_STATE_______)
#define CSM_E_INVALID_CAN_DLC          ((WORD) _INVALID_CAN_DLC_______)
#define CSM_E_INVALID_PCI              ((WORD) _INVALID_PCI___________)
#define CSM_E_TX_QUEUE_OVERRUN         ((WORD) _TX_QUEUE_OVERRUN______)
#define CSM_E_RX_QUEUE_OVERRUN         ((WORD) _RX_QUEUE_OVERRUN______)
#define CSM_E_UNKNOWN_EVENT            ((WORD) _UNKNOWN_EVENT_________)
#define CSM_E_UNKNOWN_STATE            ((WORD) _UNKNOWN_STATE_________)
#define CSM_E_INVALID_EVENT            ((WORD) _INVALID_EVENT_________)
#define CSM_E_INVALID_STATE            ((WORD) _INVALID_STATE_________)
#define CSM_E_INVALID_SIG_INDEX        ((WORD) _INVALID_SIG_INDEX_____)
#define CSM_E_INVALID_TEL_INDEX        ((WORD) _INVALID_TEL_INDEX_____)
#define CSM_E_STACK_INACTIVE           ((WORD) _STACK_INACTIVE________)
#define CSM_E_STACK_BUSY               ((WORD) _STACK_BUSY____________)
#define CSM_E_NO_CONNECTION            ((WORD) _NO_CONNECTION_________)
#define CSM_E_BUS_OFF                  ((WORD) _BUS_OFF_______________)

/* The following obsolete codes are provided for compatibility with former versions,  */
/* not to be used any longer, to be deleted after replacement in the respective code! */
#define CSM_C_GENERAL_ERROR           CSM_E_GENERAL_ERROR
#define CSM_C_OS_ERROR                CSM_E_OS_ERROR

#define CSM_C_UNKNOWN_SRVC            CSM_E_UNKNOWN_SRVC
#define CSM_C_COMP_INACTIVE           CSM_E_COMP_INACTIVE
#define CSM_C_INVALID_PARA            CSM_E_INVALID_PARA
#define CSM_C_SRVC_UNAVAILABLE        CSM_E_SRVC_UNAVAILABLE
#define CSM_C_DATA_LENGTH_EXCEEDED    CSM_E_DATA_LENGTH_EXCEEDED
#define CSM_C_NULL_POINTER            CSM_E_NULL_POINTER
#define CSM_C_INVALID_PROTOCOL_TYPE   CSM_E_INVALID_PROTOCOL_TYPE
#define CSM_C_INVALID_ADDRESS         CSM_E_INVALID_ADDRESS
#define CSM_C_RESOURCE                CSM_E_RESOURCE
#define CSM_C_I_O_DEVICE              CSM_E_I_O_DEVICE
#define CSM_C_BUFFER_FULL             CSM_E_BUFFER_FULL
#define CSM_C_CONFIG_FILE             CSM_E_CONFIG_FILE
#define CSM_C_NO_CONFIG_FILE          CSM_E_NO_CONFIG_FILE

#define CSM_C_STACK_INACTIVE          CSM_E_STACK_INACTIVE
#define CSM_C_STACK_BUSY              CSM_E_STACK_BUSY
#define CSM_C_NO_CONNECTION           CSM_E_NO_CONNECTION

#define CSM_E_BR_TEL_TX_QUEUE_OVERRUN CSM_E_TX_QUEUE_OVERRUN
#define CSM_E_BUFFER_OVERRUN          CSM_E_RX_BUFFER_OVERRUN

/* Unused (obsolete?) error codes, to be deleted after final check: */
/* #define CSM_E_PARAMETER_POINTER_NULL               0x0007  */
/* #define CSM_E_DATA_BUFFER_TO_SHORT                 0x0001  */

/********************************************************************/
/* defines for specific error codes (depending on error location)   */
/* to be found within module header files                           */
/*    used region: 0x1000 - 0xFFFF                                  */
/********************************************************************/

/*******************************************************************************
 * typedefs (scope: CAN stack and application)
 ******************************************************************************/
#if (CSM_S_VIRTUAL_NETWORK == CSM_C_F_ON)
  typedef void (*tCSM_PFN_BUS_STATE_IND) ( BYTE, BYTE, BYTE);   // only internal CB function!
#else
  typedef void (*tCSM_PFN_BUS_STATE_IND) ( BYTE, BYTE);         // only internal CB function!
#endif // CSM_S_VIRTUAL_NETWORK

#if (CSM_S_USE_TKSE == CSM_C_F_ON)
typedef struct
{
  LONG idTask;
}tCSM_TASK_INFO;
#endif // CSM_S_USE_TKSE

//*********************************************************************************************
//  CCCC  AAAAAA  L     L     BBBB   AAAAAA   CCCC  K   K    TTTTT  Y    Y  PPPPP  EEEEE  SSSSS
// C      A    A  L     L     B   B  A    A  C      K  K       T     Y  Y   P   P  E      S
// C      AAAAAA  L     L     BBBB   AAAAAA  C      KKK        T      Y     PPPP   EEEE    SSS
// C      A    A  L     L     B   B  A    A  C      K  K       T      Y     P      E          S
//  CCCC  A    A  LLLLL LLLLL BBBB   A    A   CCCC  K   K      T      Y     P      EEEEE  SSSSS
//*********************************************************************************************
// typedefs for a application callback function pointers
typedef void (CSM_CALLBACK *tCSM_PFN_BUS_ERROR_IND)        (
                                                            #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
                                                              void *pvHandle,
                                                            #endif
                                                            BYTE bBus,
                                                            WORD wErrorType);

typedef void (CSM_CALLBACK *tCSM_PFN_TP_COMMUNICATION_CON) (
                                                            #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
                                                              void *pvHandle,
                                                            #endif
                                                            BYTE bBus,
                                                            DWORD dwProtocolType,
                                                            const void *pvAddressField,  /**< wie sieht es mit doxygen Kommentaren aus? */
                                                            BYTE bConnectState,
                                                            WORD wApplID);
typedef void (CSM_CALLBACK *tCSM_PFN_TP_COMMUNICATION_IND) (
                                                            #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
                                                              void *pvHandle,
                                                            #endif
                                                            BYTE bBus,
                                                            DWORD dwProtocolType,
                                                            const void *pvAddressField,
                                                            BYTE bConnectState,
                                                            WORD wApplID);

// network protocols and CNP
typedef void (CSM_CALLBACK *tCSM_PFN_TP_DATA_CON) (
                                                   #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
                                                     void *pvHandle,
                                                   #endif
                                                   DWORD dwProtocolType,
                                                   const VOID *pvAddressField,
                                                   BYTE bTransferResult);
typedef void (CSM_CALLBACK *tCSM_PFN_TP_DATA_IND) (
                                                   #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
                                                     void *pvHandle,
                                                   #endif
                                                   DWORD dwProtocolType,
                                                   const VOID *pvAddressField,
                                                   BYTE *pbData,
                                                   WORD wDataLength);
typedef void (CSM_CALLBACK *tCSM_PFN_TP_DATA_IND_FF) (
                                                   #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
                                                     void *pvHandle,
                                                   #endif
                                                   DWORD dwProtocolType,
                                                   const VOID *pvAdressField,
                                                   WORD wDataLength);

#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
  #if (CSM_S_ISO_TP == CSM_C_F_ON)
    #if (CSM_S_ITP_ERROR_INDICATION_AVAILABLE == CSM_C_F_ON)
      #define CSM_LOCAL_DUMMY_DATA_ERROR_IND_AVAILABLE   // local help switch, used below
    #endif // CSM_S_ITP_ERROR_INDICATION_AVAILABLE
  #endif // CSM_S_ISO_TP enabled

  #if (CSM_S_UUDT_RX == CSM_C_F_ON) && (!defined CSM_LOCAL_DUMMY_DATA_ERROR_IND_AVAILABLE)
    #if  (CSM_S_UUDT_RX_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
      #define CSM_LOCAL_DUMMY_DATA_ERROR_IND_AVAILABLE   // local help switch, used below
    #endif // (CSM_S_UUDT_RX_DATA_ERROR_IND_AVAILABLE
  #endif // (CSM_S_UUDT_RX

  #if (CSM_S_J1939 == CSM_C_F_ON) && (!defined CSM_LOCAL_DUMMY_DATA_ERROR_IND_AVAILABLE)
  #if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
    #define CSM_LOCAL_DUMMY_DATA_ERROR_IND_AVAILABLE     // local help switch, used below
  #endif
  #endif

  #ifdef CSM_LOCAL_DUMMY_DATA_ERROR_IND_AVAILABLE
    /** Any data error ind available.  CSM_S_DATA_ERROR_IND_AVAILABLE is defined here in
     *  the pool file csm_stack_i.h instead of the project specific file csm_stack_p.h. */
    typedef void (CSM_CALLBACK *tCSM_PFN_TP_DATA_ERROR_IND) (
                                                             #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
                                                              void *pvHandle,
                                                             #endif
                                                             DWORD dwProtocolType,
                                                             const VOID *pvAddrssField,
                                                             DWORD dwDataErrorCode);
  #endif
#endif // CSM_S_CNP_AVAILABLE

// CBR
typedef void (CSM_CALLBACK *tCSM_PFN_CBR_SIGNAL_IND) (
                                                      #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
                                                        void *pvHandle,
                                                      #endif
                                                      DWORD dwSignalId,
                                                      DWORD dwSignalStatus);

//*********************************************************************************************
// DDDD   AAAAAA  TTTTT  AAAAAA    TTTTT  Y    Y  PPPPP  EEEEE  SSSSS
// D   D  A    A    T    A    A      T     Y  Y   P   P  E      S
// D   D  AAAAAA    T    AAAAAA      T      Y     PPPP   EEEE    SSS
// D   D  A    A    T    A    A      T      Y     P      E          S
// DDDD   A    A    T    A    A      T      Y     P      EEEEE  SSSSS
//*********************************************************************************************
// for broadcast signals **********************************
typedef struct
{
  tCSM_PFN_CBR_SIGNAL_IND pfvSignalInd1;       /**< Zeiger auf ein Feld von Callbackfunktionen */
  #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
    void * pvHandle;
  #endif
  #if (CSM_S_USE_TKSE == CSM_C_F_ON)
    tCSM_TASK_INFO rTaskInfo;
    //DWORD idTask
  #endif // CSM_S_USE_TKSE
}tCSM_BR_SIGNAL_CALLBACK;

// Typedefs for CNP application callback structures
#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
  // MPDT **********************************
  typedef struct
  {
    tCSM_PFN_TP_COMMUNICATION_CON pfvCommunicationCon;  /**< Pointer to communication confirmation callback function. */
    tCSM_PFN_TP_COMMUNICATION_IND pfvCommunicationInd;  /**< Pointer to communication indication callback function. */
    tCSM_PFN_TP_DATA_CON pfvCDataCon;                   /**< Pointer to data confirmation callback function for IL=Control. */
    tCSM_PFN_TP_DATA_IND pfvCDataInd;                   /**< Pointer to data indication callback function for IL=Control. */
    tCSM_PFN_TP_DATA_CON pfvDDataCon;                   /**< Pointer to data confirmation callback function for IL=Data. */
    tCSM_PFN_TP_DATA_IND pfvDDataInd;                   /**< Pointer to data indication callback function for IL=Data. */
    #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
      void * pvHandle;
    #endif // CSM_S_USE_CALLBACK_WITH_HANDLE
    #if (CSM_S_USE_TKSE == CSM_C_F_ON)
      tCSM_TASK_INFO rTaskInfo;
    #endif // CSM_S_USE_TKSE
  }tCSM_MPDT_APPL_CALLBACK;

  // VW TP2.0 **********************************
  typedef struct
  {
    tCSM_PFN_TP_COMMUNICATION_CON pfvCommunicationCon;  /**< Pointer to communication confirmation callback function. */
    tCSM_PFN_TP_COMMUNICATION_IND pfvCommunicationInd;  /**< Pointer to communication indication callback function. */
    tCSM_PFN_TP_DATA_CON pfvDataCon;                    /**< Pointer to data confirmation callback function. */
    tCSM_PFN_TP_DATA_IND pfvDataInd;                    /**< Pointer to data indication callback function. */
    #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
      void * pvHandle;
    #endif // CSM_S_USE_CALLBACK_WITH_HANDLE
    #if (CSM_S_USE_TKSE == CSM_C_F_ON)
      tCSM_TASK_INFO rTaskInfo;
    #endif // CSM_S_USE_TKSE
  }tCSM_TP20_APPL_CALLBACK;

  // ASDT **********************************
  typedef struct
  {
    tCSM_PFN_TP_COMMUNICATION_CON pfvCommunicationCon;
    tCSM_PFN_TP_COMMUNICATION_IND pfvCommunicationInd;
    tCSM_PFN_TP_DATA_CON pfvDataCon;
    tCSM_PFN_TP_DATA_IND pfvDataInd;
    #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
      void * pvHandle;
    #endif // CSM_S_USE_CALLBACK_WITH_HANDLE
    #if (CSM_S_USE_TKSE == CSM_C_F_ON)
      tCSM_TASK_INFO rTaskInfo;
    #endif // CSM_S_USE_TKSE
  }tCSM_MCNET_ASDT_APPL_CALLBACK;

  // USDT **********************************
  typedef struct
  {
    tCSM_PFN_TP_DATA_CON pfvDataCon;
    tCSM_PFN_TP_DATA_IND pfvDataInd;
    #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
      void * pvHandle;
    #endif // CSM_S_USE_CALLBACK_WITH_HANDLE
    #if (CSM_S_USE_TKSE == CSM_C_F_ON)
      tCSM_TASK_INFO rTaskInfo;
    #endif // CSM_S_USE_TKSE
  }tCSM_MCNET_USDT_APPL_CALLBACK;

  // ISO TP **********************************
  #if (CSM_S_ISO_TP == CSM_C_F_ON)
  typedef struct
  {
    tCSM_PFN_TP_COMMUNICATION_CON pfvCommunicationCon;
    tCSM_PFN_TP_COMMUNICATION_IND pfvCommunicationInd;
    tCSM_PFN_TP_DATA_CON pfvDataCon;
    tCSM_PFN_TP_DATA_IND pfvDataInd;

    #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
      tCSM_PFN_TP_DATA_IND_FF pfvDataIndFF;
    #endif // CSM_S_ITP_FF_INDICATION_AVAILABLE
    #if (CSM_S_ITP_ERROR_INDICATION_AVAILABLE == CSM_C_F_ON)
      tCSM_PFN_TP_DATA_ERROR_IND pfvDataErrorInd;
    #endif // CSM_S_ITP_ERROR_INDICATION_AVAILABLE

    /* If any application uses these elements for initialization it can be necessary to
     * add them again (e.g. via new switch) temporarily untill the applications are changed.
    #if (CSM_S_ISO_TP_USDT_APPL_CALLBACK_STRUCT_WITH_DUMMY_HANDLE_AND_TASKINFO == CSM_C_F_ON)
      #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
        void * pvHandle;
      #endif
      #if (CSM_S_USE_TKSE == CSM_C_F_ON)
        tCSM_TASK_INFO rTaskInfo;
      #endif
    #endif
    */
  }tCSM_ISO_TP_USDT_APPL_CALLBACK;
  #endif // CSM_S_ISO_TP

  // UUDT TX **********************************
  #if (CSM_S_UUDT_TX == CSM_C_F_ON)
  typedef struct
  {
    tCSM_PFN_TP_DATA_CON pfvDataCon;
    #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
      void * pvHandle;
    #endif // CSM_S_USE_CALLBACK_WITH_HANDLE
    #if (CSM_S_USE_TKSE == CSM_C_F_ON)
      tCSM_TASK_INFO rTaskInfo;
    #endif // CSM_S_USE_TKSE
  }tCSM_UUDT_TX_APPL_CALLBACK;
  #endif // CSM_S_UUDT_TX

  // UUDT RX ********************************
  #if (CSM_S_UUDT_RX == CSM_C_F_ON)
  typedef struct
  {
    tCSM_PFN_TP_DATA_IND pfvDataInd;                         /**< mandatory: Indication function on UUDT data reception */
    #if (CSM_S_UUDT_RX_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
      tCSM_PFN_TP_DATA_ERROR_IND pfvDataErrorInd;            /**< optional: Indication function on UUDT data reception */
    #endif // CSM_S_UUDT_RX_DATA_ERROR_IND_AVAILABLE
    #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
      void * pvHandle;
    #endif // CSM_S_USE_CALLBACK_WITH_HANDLE
    #if (CSM_S_USE_TKSE == CSM_C_F_ON)
      tCSM_TASK_INFO rTaskInfo;
    #endif // CSM_S_USE_TKSE
  }tCSM_UUDT_RX_APPL_CALLBACK;
  #endif // CSM_S_UUDT_RX

  // J1939 PGN **********************************
  #if (CSM_S_J1939 == CSM_C_F_ON)
  typedef struct
  {
    tCSM_PFN_TP_DATA_CON pfvDataCon;
    tCSM_PFN_TP_DATA_IND pfvDataInd;
    tCSM_PFN_TP_DATA_IND_FF pfvDataIndFF;
    #if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
      tCSM_PFN_TP_DATA_ERROR_IND pfvDataErrorInd;
    #else
      //#error "CSM_S_DATA_ERROR_IND_AVAILABLE must be defined above in this file."
    #endif
  #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
     void * pvHandle;
    #endif /* CSM_S_USE_CALLBACK_WITH_HANDLE */
  }tCSM_J1939_PGN_CALLBACK;
  #endif // CSM_S_J1939
#endif // CSM_S_CNP_AVAILABLE

// broadcast signals
typedef struct
{
  WORD u16ApplID;
  tCSM_PFN_TP_COMMUNICATION_CON pfvCommunicationCon;
  tCSM_PFN_TP_COMMUNICATION_IND pfvCommunicationInd;
  #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
    void * pvHandle;
  #endif // CSM_S_USE_CALLBACK_WITH_HANDLE
  #if (CSM_S_USE_TKSE == CSM_C_F_ON)
    tCSM_TASK_INFO rTaskInfo;
  #endif // CSM_S_USE_TKSE
}tCSM_BR_SIGNAL_APPL_CALLBACK;

// LCTRL
typedef struct
{
  WORD u16ApplID;
  tCSM_PFN_TP_COMMUNICATION_CON pfvCommunicationCon;
  tCSM_PFN_TP_COMMUNICATION_IND pfvCommunicationInd;
  /* If any application uses these elements for initialization it can be necessary to
   * add them again (e.g. via new switch) untill the applications are changed.
    #if (CSM_S_LOCAL_CTRL_CALLBACK_STRUCT_WITH_DUMMY_HANDLE_AND_TASKINFO == CSM_C_F_ON)
  #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
    void * pvHandle;
  #endif
  #if (CSM_S_USE_TKSE == CSM_C_F_ON)
    tCSM_TASK_INFO rTaskInfo;
  #endif
  #endif*/
}tCSM_LOCAL_CTRL_APPL_CALLBACK;

// J1939
typedef struct
{
  tCSM_PFN_TP_COMMUNICATION_CON pfvCommunicationCon;
  tCSM_PFN_TP_COMMUNICATION_IND pfvCommunicationInd;

#if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
  tCSM_PFN_TP_DATA_CON pfvDataCon;
  tCSM_PFN_TP_DATA_IND pfvDataInd;

  tCSM_PFN_TP_DATA_IND_FF pfvDataIndFF;

  #if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
  tCSM_PFN_TP_DATA_ERROR_IND pfvDataErrorInd;
  #else
  //#error "CSM_S_DATA_ERROR_IND_AVAILABLE must be defined above in this file."
  #endif

#endif

  #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
    void * pvHandle;
  #endif // CSM_S_USE_CALLBACK_WITH_HANDLE
  #if (CSM_S_USE_TKSE == CSM_C_F_ON)
    tCSM_TASK_INFO rTaskInfo;
  #endif // CSM_S_USE_TKSE
}tCSM_J1939_APPL_CALLBACK;

// VW BAP
typedef struct
{
  tCSM_PFN_TP_COMMUNICATION_CON  pfvCommunicationCon;   /**< optional: Asynchronous confirmation after CSM communication request*/
  tCSM_PFN_TP_DATA_CON           pfvDataCon;            /**< optional: Asynchronous confirmation after CSM data request */
  tCSM_PFN_TP_DATA_IND           pfvDataInd;            /**< mandatory: Indication function on BAP data or error reception */
  #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
    void * pvHandle;
  #endif // CSM_S_USE_CALLBACK_WITH_HANDLE
  #if (CSM_S_USE_TKSE == CSM_C_F_ON)
    tCSM_TASK_INFO rTaskInfo;
  #endif // CSM_S_USE_TKSE
}tCSM_VW_BAP_APPL_CALLBACK;

// GM LAN
typedef struct
{
  WORD u16ApplID;   /**< Application ID for protocol type GM NWM. */
  tCSM_PFN_TP_COMMUNICATION_IND pfvCommunicationInd;  /**< Pointer to communication indication callback function. */
  #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
    void * pvHandle;           /**< Buffer to store the handle given in CSM_lApplCallbackInit().
                                *   Should not be in this struct, only for CSM/CSMAL internal use. */
  #endif // CSM_S_USE_CALLBACK_WITH_HANDLE
  #if (CSM_S_USE_TKSE == CSM_C_F_ON)
    tCSM_TASK_INFO rTaskInfo;  /**< Task info. Should not be in this struct, only for CSM/CSMAL internal use. */
  #endif // CSM_S_USE_TKSE
}tCSM_GMLAN_NWM_APPL_CALLBACK;

// OSEK NWM
typedef struct
{
  WORD u16ApplID;
  tCSM_PFN_TP_COMMUNICATION_IND pfvCommunicationInd;
  #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
    void * pvHandle;
  #endif // CSM_S_USE_CALLBACK_WITH_HANDLE
  #if (CSM_S_USE_TKSE == CSM_C_F_ON)
    tCSM_TASK_INFO rTaskInfo;
  #endif // CSM_S_USE_TKSE
}tCSM_OSEK_NWM_APPL_CALLBACK;

// OSEKI NWM
typedef struct
{
  // this struct defines the types of callback functions for indirect OSEK NWM
  // currently, a Communication Indication is used to report errors to the Vehicle Data Server
  WORD u16ApplID;
  tCSM_PFN_TP_COMMUNICATION_IND pfvCommunicationInd; /**< Pointer to comm. indication callback function */
  #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
    void * pvHandle;
  #endif
  #if (CSM_S_USE_TKSE == CSM_C_F_ON)
    tCSM_TASK_INFO rTaskInfo;
  #endif
}tCSM_OSEKI_NWM_APPL_CALLBACK;

//*******************************************************************************************
// AAAAAA   DDDD   RRRR   EEEEE  SSSSS  SSSSS   FFFFF  I  EEEEE  L     DDDD   SSSSS
// A    A   D   D  R   R  E      S      S       F      I  E      L     D   D  S
// AAAAAA   D   D  RRRR   EEEE    SSS    SSS    FFFF   I  EEEE   L     D   D   SSS
// A    A   D   D  R   R  E          S      S   F      I  E      L     D   D      S
// A    A   DDDD   R   R  EEEEE  SSSSS  SSSSS   F      I  EEEEE  LLLLL DDDD   SSSSS
//*******************************************************************************************
/* CAN-SWA AddressFields */
typedef struct tagCSM_ADDR_ISO_TP
{
  BYTE bLocalComp;
  BYTE bRemoteComp;
  WORD wReserved;
} tCSM_ADDR_ISO_TP;

typedef struct
{
  BYTE bLocalComp;    /**< local component number */
  BYTE bRemoteComp;   /**< remote component number */
}tCSM_ADDR_MPDT;      /**< typedef for MPDT address field und in interface 1# functions */

typedef struct tagCSM_ADDR_MCNET_ASDT
{
  BYTE  bLocalComp;
  BYTE  bRemoteComp;
} CSM_ADDR_MCNET_ASDT;

typedef struct tagCSM_ADDR_MCNET_USDT
{
  BYTE  bSenderComp;
} CSM_ADDR_MCNET_USDT;

typedef struct
{
  BYTE  bSrcNodeID;
  BYTE  bTgtNodeID;
  WORD  wPGN;
} CSM_ADDR_J1939;

#if (CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON)
typedef struct
{
  tCSM_PFN_BUS_ERROR_IND pfvBusErrorInd;
  #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
    void * pvHandle;
  #endif // CSM_S_USE_CALLBACK_WITH_HANDLE
  #if (CSM_S_USE_TKSE == CSM_C_F_ON)
    tCSM_TASK_INFO rTaskInfo;
  #endif // CSM_S_USE_TKSE
}tCSM_BUS_ERROR_IND_CALLBACK;
#endif // CSM_S_ENABLE_INDICATE_BUSERROR

//*******************************************************************************************
//  CCCC  SSSSS  MM   MM        /   AAAAAA  PPPP   L
// C      S      M M M M       /    A    A  P   P  L
// C       SSS   M  M  M      /     AAAAAA  PPPP   L
// C          S  M     M     /      A    A  P      L
//  CCCC  SSSSS  M     M    /       A    A  P      LLLLL
//*******************************************************************************************
#if (defined CSM_CAN_STACK || defined CSM_APL_SELF)
typedef struct
{
  BYTE                   bStartBit;             /**< Startbit of signal in telegram */
  BYTE                   bLength;               /**< Length of signal in telegram */
  union
  {
    BYTE                 bIndicationMode;       /**< different kinds of indication
                                                  possible values:
                                                   CSM_C_ALWAYS_IND
                                                   CSM_C_IND_AFTER_CHANGE
                                                   CSM_NEVER_IND */
    BYTE                 bTxMode;               /**< different kinds of transmit mode
                                                  possible values:
                                                   CBR_TMSG_CYCLIC
                                                   CBR_TMSG_CYCLIC_IF_ACTIVE
                                                   CBR_TMSG_SPONTAN
                                                   CBR_TMSG_CYCLIC_AND_SPONTAN
                                                   CBR_TMSG_CYC_IF_ACT_AND_SPONT */
  } rMode;
  BYTE                   bMaxCallbackFunctions; /**< Anzahl der Callbackfunktionen */
  #if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
    #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
      tCSM_BR_SIGNAL_CALLBACK  *pfnSignalCallback;
    #else
      tCSM_PFN_CBR_SIGNAL_IND *pfvSignalInd1;           /**< Zeiger auf ein Feld von Callbackfunktionen */
    #endif // CSM_S_USE_CALLBACK_WITH_HANDLE
  #else // CSM_S_CALLBACK_INIT_BY_APPLICATION
    const tCSM_PFN_CBR_SIGNAL_IND *pfvSignalInd1;       /**< Zeiger auf ein Feld von Callbackfunktionen */
  #endif // CSM_S_CALLBACK_INIT_BY_APPLICATION
  #if (CSM_S_CBR_AVAILABLE == CSM_C_F_ON)
  #if (CSM_S_CBR_BOTH_ENDIANS_SIGNALS == CSM_C_F_ON)
    BYTE bEndians;        /**< Endians of signal: Possible values: CSM_C_SIGNAL_ENDIANS_LITTLE, CSM_C_SIGNAL_ENDIANS_BIG*/
  #endif // CSM_S_CBR_BOTH_ENDIANS_SIGNALS
  #endif // CSM_S_CBR_AVAILABLE
} CSM_SIGNAL;

typedef struct
{
  BYTE                         bNumberOfSignals;            /**< number of signals in telegram */
  const CSM_SIGNAL* const      prSignals;                   /**< pointer to signal definitions */

  #if (CSM_S_CBR_AVAILABLE == CSM_C_F_ON)
    #if CSM_S_DEFAULT_SIGNAL_VALUES == CSM_C_F_ON
      const BYTE* const        pvTelegramDefaultData;       /**< default data for each telegram */
      const BYTE* const        pvTelegramDefaultMask;       /**< default data for each telegram */
    #endif
    #if CSM_S_TIMEOUT_SIGNAL_VALUES == CSM_C_F_ON
      const BYTE* const        pvTelegramTimeoutData;       /**< default data for each telegram */
      const BYTE* const        pvTelegramTimeoutMask;       /**< default data for each telegram */
    #endif
    #if CSM_S_CBR_RX_MASK == CSM_C_F_ON
      const BYTE* const        pvTelegramRxMask;            /**< receive mask for each telegram */
    #endif
    #if CSM_S_CBR_PROXY_MASK == CSM_C_F_ON
      const BYTE* const        pvTelegramProxyMask;         /**< proxy mask for each telegram  */
    #endif
  #endif // CSM_S_CBR_AVAILABLE
} CSM_TELEGRAM;
#endif // CSM_CAN_STACK || CSM_APL_SELF

typedef struct
{
  DWORD dwSignalId;             /**< Signal Id */
  void *pvActSignalData;        /**< pointer to output data buffer */
  BYTE  bDataBufferLength;      /**< Length of output data buffer in bytes */
  DWORD *pdwSignalStatus;       /**< pointer to global signal state (bitfield) */
} tCSM_MultipleSignalRead;       /**< help structure for CSM_lMultipleSignalRead() */

typedef struct
{
  DWORD dwSignalId;             /**< Signal Id */
  void *pvNewSignalData;        /**< pointer to input data buffer */
  BYTE  bDataBufferLength;      /**< Length of input data buffer in bytes */
  BYTE  bTxType;                /**< transmit type */
} tCSM_MultipleSignalWrite;      /**< help structure for CSM_lMultipleSignalWrite() */

//*******************************************************************************************
// BBBB    AAAAA   PPPP
// B   B   A   A   P   P
// BBBB    AAAAA   PPPP
// B   B   A   A   P
// BBBB    A   A   P
//*******************************************************************************************
/* Please add this feature switch to your csm_stack_p.h, regardless of CSM_S_CNP_AVAILABLE */
#if (CSM_S_VW_BAP_TP == CSM_C_F_ON)
  #ifndef CSM_CAN_STACK
    /* include bap defines for applications */
    /* wenn, dann in der csm_ip.h
    #include "bap_types.h"*/
  #endif
typedef struct
{
  /* Important note:
   * If you change something in this structure please take also care of
   * adjusting CSM_C_BAP_BAPINOUT_MAXSIZE in csm_bap_m.h !!
   * It's purpose is commented there.
   */
  BYTE ubQualifier;    /* Type qualifier to specify the data type as void, int8, etc.  */
                       /* Defines like CSM_C_BAP_INT8_REQ are used here                */
  BYTE ubFctId;        /* BAP function ID which identifies BAP data or method          */
  union
  {
    BYTE ubRequest;    /* Request type e.g. BapReq_DataSet                             */
    BYTE ubIndication; /* Indication type e.g. BapInd_Data                             */
    BYTE ubAcknowledge;/* Acknowledge type e.g. BapAck_Property_DataSet                */
  } reason;
  union
  {
    void* pvIn;         /* Pointer to appl. buffer the data shall be stored to         */
    const void* pvOut;  /* Pointer to appl. buffer of data which shall be sent         */
  } buffer;             /* Buffer can be of type BYTE, uword, DWORD or BYTE[]          */
                        /* Must be casted to (void *)                                  */
  DWORD ulwLength;      /* Length of data in pvData                                    */
                        /* Redundant for atomic types since ubQualifier still holds    */
                        /* this info but mandatory for BYTE arrays.                    */
} tCSM_BAPInOut;
#endif // CSM_S_VW_BAP_TP
//*******************************************************************************************

#include "csm_stack_ip.h"

/*******************************************************************************
 * defines and macros which have to be set after include of csm_stack_ip.h
 * (scope: CAN stack and application)
 ******************************************************************************/
/* --none-- */

/*******************************************************************************
 * variable definition (scope: CAN stack and application)
 ******************************************************************************/
/* --none-- */

#if (CSM_S_DOXYGEN_HIDE_NON_API_FCT == CSM_C_F_OFF)
/*******************************************************************************
 * function prototypes (scope: CAN stack and  operating system)
 ******************************************************************************/
CSM_STATIC BOOL CSM_fCheckCommunicationReq(BYTE bBus);

#endif /* CSM_S_DOXYGEN_HIDE_NON_API_FCT */
/*******************************************************************************
 * function prototypes (scope: CAN stack #1 API)
 ******************************************************************************/
void CSM_vUpstreamTask(void);
void CSM_vDownstreamTask(void);
void CSM_vCBRTxRxTimeoutManager(void);
void CNM_VW_vNMTask(void);
void CSM_VW_vCyclicTimer(void);
void CNP_VW20_vTPManager(void);
void CNP_VW16_vTPManager(void);


#ifndef CSM_CAN_STACK
#ifdef GLB_CAR_RADIO
/*******************************************************************************
 *FUNCTION:    CSM_vCanTest
 *DESCRIPTION: function for testing of CAN hardware (department ESD)
 *
 *PARAMETER:   GLB_TY_InterfaceStruct* tyInterfaceStruct (->I ->O)
 *
 *RETURNVALUE: void
 *
 *HISTORY:
 *13.06.02 CM-DI/EHS3-Pistoor
 *Initial Revision.
 ******************************************************************************/
#ifndef _GLOBAL_H
  #include "global.h"
#endif
void CSM_vCanTest(GLB_TY_InterfaceStruct* ptyInterfaceStruct);
#endif /* GLB_CAR_RADIO */
#endif /* CSM_CAN_STACK */

#else
#error CSM_STACK_I.H included several times
#endif
/*******************************************************************************
 * \doxydocu
 * \file          csm_stack_i.h
 * \brief         Project independent part of module external and internal used
 *                data/definitions
 *
 * \see           CAN-SW Architecture, written by CM-DI/ESA3
 *
 * \sw_component  CAN stack
 * \project       Pool module
 * \path          /di_can/modules_swa/csm_stack/src/
 *
 * \ingroup       CSM
 * \ingroup       CSM_STACK
 *
 * \authors       V-Team
 *
 * COPYRIGHT      (c) 2004 Blaupunkt GmbH
 *
 * \history_begin
 *
 *30.11.01        Rev. 0.1 CM-DI/ECC13-Pistoor
 *                Initial Version.
 *
 *15.02.02        Rev. 0.2 CM-DI/EHS3-Pistoor
 *                All functions are new
 *
 *15.02.02        Rev. 0.3 CM-DI/EHS3-Pistoor
 *                Added definition of CSM_C_NO_ERROR
 *
 *28.02.02        Rev. 0.4 CM-DI/EHS3-Pistoor
 *                Changed error macros
 *18.03.02        Rev. 1.4 CM-DI/EHS3-Pistoor
 *                - Merged revisions 0.4 and 1.3
 *                - Renamed can_stack_i.h to csm_stack_i.h
 *                - Moved dwProtocolTypes to csm_ip.h
 *
 *18.03.02        Rev. 1.0 CM-DI/EHS3-Pistoor
 *                - Initial revision of csm_stack_i.h
 *
 *16.04.02        Rev. 1.1 CM-DI/EHS3-K"ohler, Pistoor
 *                - Comments for dwProtocolTypes added.
 *                - Makros for bus access changed.
 *
 *25.04.02  Rev. 1.2 CM-DI/EHS3-K"ohler
 *          - CSM_C_PTYPE_ defines moved to csm_stack_i.h
 *
 *20.06.02  Rev. 1.4 CM-DI/EHS3-K"ohler
 *          - Telegram types moved to csm_stack_m.h.
 *
 *09.08.02  Rev. 1.5 CM-DI/EHS3-Pistoor
 *          - Added error macros from csm_stack_m.h
 *          - Rename CSM_M_GET_MULTSIGID_FROM_SIGNAL_ID to
 *            CSM_M_GET_MULTSIGINDEX_FROM_SIGNAL_ID
 *            and CSM_M_GET_SIGID_FROM_SIGNAL_ID to
 *            CSM_M_GET_SIGINDEX_FROM_SIGNAL_ID
 *
 *30.08.02  Rev. 1.6 CM-DI/EHS3-K"ohler
 *          - CSM_vIndicationTask renamed CSM_vUpstreamTask
 *          - Prototypes CSM_vSYS_SleepConfirmation and
 *            CSM_vSYS_ActiveIndication moved to csm_stack_m.h
 *
 *13.09.02  Rev. 1.7 CM-DI/EHS3-Zurm"uhl, -K"ohler
 *          - Defines for MCNet, ISO-TP and VAG-TP error locations added.
 *
 *18.09.02  Rev. 1.8 CM-DI/EHS3-Pistoor
 *          - Changed CSM_NEVER_IND to CSM_C_NEVER_IND
 *
 *01.10.02  Rev. 1.9 CM-DI/EHS3-Pistoor
 *          - Added definition of CSM_C_GENERAL
 *
 *23.10.02  Rev. 1.10 CM-DI/EHS3-Pistoor
 *          - Changed definition of CSM_C_TX_TELEGRAM from 0 to 1
 *
 *04.11.02  Rev. 1.11 CM-DI/EHS3-Pistoor
 *          - Added defines CSM_C_INIT_START,
 *            CSM_C_EXIT_STOP and CSM_C_EXIT_OFF
 *          - Made corrections for broadcast server callback
 *            function configurations
 *          - Deleted Rx/Tx type of CSM_TELEGRAM structure
 *          - Added function declaration of CSM_vDownstreamTask()
 *
 *08.11.02  Rev. 1.12 CM-DI/EHS3-Pistoor
 *          - Added error location CSM_APL
 *          - Corrected macros for multiple signal generation
 *
 *12.11.02  Rev. 1.13 CM-DI/EHS3-Pistoor
 *          - Added defines of CSM_C_CONNECT and CSM_C_DISCONNECT
 *
 *14.11.02  Rev. 1.14 CM-DI/EHS3-Pistoor
 *          - Added pvSignalDefaultData and pvSignalTimeoutData
 *            to struct CSM_SIGNAL
 *          - Moved error code definitions from csm_stack_m.h
 *06.12.02  Rev. 1.15 CM-DI/EHS3-K"ohler
 *          - Four global/IPNET I/O error defines added.
 *
 *11.12.02  Rev. 1.17 CM-DI/EHS3-Pistoor
 *          - Added pvTelegramTimeoutMask in structure CSM_TELEGRAM
 *
 *21.02.03  Rev. 1.18 CM-DI/ESU3-Pistoor
 *          - Added preinclude section
 *
 *21.02.03  Rev. 1.19 CM-DI/ESU3-Pistoor
 *          - Made correction because of compiler error
 *
 *07.03.03  Rev. 1.20 CM-DI/ESU3-FEM5HI
 *          - defines for CSM_vCBRSignalInd extented to 16bit
 *          - type CSM_SIGNAL for CSM_S_FORD_IL moved to csm_stack_ip.h
 *
 *25.03.03  Rev. 1.21 CM-DI/ESU3-Pistoor
 *          - deleted CSM_C_ERR_INFO and incremented values for
 *            CSM_C_ERR_WARNING, CSM_C_ERR_ERROR and CSM_C_ERR_SEVERE_ERROR
 *
 *24.04.03  Rev. 1.22 CM-DI/ESU3-Pistoor
 *          - Added pvTelegramDefaultMask to structure CSM_TELEGRAM
 *
 *28.04.03  Rev. 1.23 CM-DI/ESU3-Pistoor
 *          - Added switch CSM_S_REMOTE_STACK
 *          - Added defines CSM_C_SIGNAL_IND_STATUS_MASK and
 *
 *15.05.03  Rev. 1.24 CM-DI/ESU3-Pistoor
 *          - defined CSM_C_BUS_OFF, CSM_C_TRANSC_ERR_STABLE
 *            and CSM_C_NO_NETWORK
 *            CSM_C_SIGNAL_READ_STATUS_MASK
 *
 *20.05.03  Rev. 1.25 CM-DI/ESU3-Pistoor
 *          - deleted pvTelegramData in structure CSM_TELEGRAM
 *
 *21.05.03  Rev. 1.26 CM-DI/ESU3-Pistoor
 *          - Added declaration CSM_vCanTest()
 *
 *18.06.03  Rev. 1.27 CM-DI/ESU3-Pistoor
 *          - Changed defines for CSM_vCBRSignalInd(), CSM_vCBRMultipleSignalInd()
 *            and xxx_lSignalRead()
 *
 *24.06.03  Rev. 1.28 CM-DI/ESU3-Pr"ohl
 *          - CSM_vCanTest() disabled for remote stack.
 *          - ESD types replaced by CSM SW-architecture types.
 *
 *28.07.03  Rev. 1.29 CM-DI/ESU3-K"ohler
 *          - All struct typedefs  moved from csm_stack_i.h to csm_stack_m.h.
 *            Preinclude not any longer necessary for csm_tack_i(p).h.
 *          - Typedef CSM_ADDR_OPEL_USDT_ISO moved from _ip.h to _i.h.
 *
 *06.08.03  Rev. 1.30 CM-DI/ESU3-Pistoor
 *          - Moved typedefs CSM_SIGNAL, CSM_MULTIPLE_SIGNAL and CSM_TELEGRAM back from csm_stack_m.h
 *          - Moved second include of csm_stack_ip.h after these typedefs again
 *
 *20.08.03  Rev. 1.31 CM-DI/ESU3-Franke
 *          - Ford IL project specific defines CSM_SIGNAL and CSM_TELEGRAM to save ROM
 *
 *22.09.03  Rev. 1.32 CM-DI/ESU3-Franke
 *          - added project specific define CSM_C_INIT_ACTIVATE
 *            (prepared for pool use)
 *
 *10.10.03  Rev. 1.33 CM-DI/ESU3-Zurm"uhl, -K"ohler
 *          - Include of csm_datatypes added.
 *          - MCNET defines added.
 *
 *01.12.03  Rev. 2.0 CM-DI/ESU3-Zurm"uhl, -K"ohler
 *          - New Include and feature switch structure.
 *          - Preinclude part removed.
 *          - Defines for xxx_vDataCon() and
 *          - MCNet addressfields added.
 *          - Define PFN_CALLBACK removed.
 *          - CSM_C_CONF_OK/FAULT changed to 0/1
 *
 *16.12.03  Rev. 2.1 CM-DI/ESU3-K"ohler
 *          - Error locations for VW TP 1.6 and 2.0 added.
 *
 *17.02.04  Rev. 2.2 CM-DI/ESU3-K"ohler
 *          - Error locations for CAN driver CDA_FCAN and CDA_TUAREG added.
 *
 *24.02.04  Rev. 2.3 CM-DI/ESU3-Prhl
 *          - typdefs added
 *
 *01.04.04  Rev. 2.4.X CM-DI/ESU3-Prhl
 *
 **********************  Switch from V-Team PVCS to ClearCase
 *
 * 30.04.04  Rev. \main\4 CM-DI/ESU3-K"ohler
 * - Define for error location CSM_C_ERR_LOC_CDA_ARION added.
 *
 * 18.06.04  Rev. \main\5 CM-DI/ESU3-Prhl
 * - change CSM_SIGNAL and CSM_TELEGRAM
 * - tCSM_PFN_BUS_STATE_IND added
 *
 * 23.06.04  Rev. \main\6 CM-DI/ESU3-Prhl
 *
 * 23.06.04  Rev. \main\7 CM-DI/ESU3-Prhl
 * - change CSM_SIGNAL
 *
 * 25.06.04  Rev. \main\8 CM-DI/ESU3-Prhl
 * - add tCSM_ISO_TP_USDT_APPL_CALLBACK and tCSM_BR_SIGNAL_APPL_CALLBACK
 *
 * 28.06.04  Rev. \main\9 CM-DI/ESU3-Prhl
 * - new CSM_C_SIGNAL_... defines added
 *
 * 30.08.04  Rev. \main\10 CM-DI/ESA3-Battis
 * - new error location CSM_C_ERR_LOC_CDA_AFCAN inserted
 *
 * 02.09.04  Rev. \main\11 CM-DI/ESA3-Prhl
 * - CSM_C_STACK_UP and CSM_C_STACK_DOWN defines added
 *
 * 30.09.04  Rev. \main\12 CM-DI/ESA3-K"ohler
 * - Bus state defines moved from csm_i.h to csm_stack_i.h because they are used
 *   in CNM too.
 *
 * 01.11.04  Rev. \main\13 CM-DI/ESA3-Battis
 * - feature switch CSM_S_CNP_AVAILABLE moved
 * - pointer changed to const in struct CSM_SIGNAL
 *
 * 22.11.04  Rev. \main\14 CM-DI/ESA3-Battis
 * - define CSM_C_SIGNAL_IND_STATUS_MASK removed
 *
 * 18.01.05  Rev. \main\15 CM-DI/ESA3-Battis
 * - SM_C_TX_AS_CONFIGURED renamed to SM_C_TX_CYCLIC
 *
 * 02.02.05  Rev. \main\16 CM-DI/ESA3-Prhl
 * - tCSM_MCNET_USDT_APPL_CALLBACK added,
 *   tCSM_MCNET_USDT_TX_APPL_CALLBACK and tCSM_MCNET_USDT_RX_APPL_CALLBACK removed
 *
 * 07.02.05  Rev. \main\17 CM-DI/ESA3-Prhl
 * - CSM_SIGNAL changed. tCSM_PFN_CBR_SIGNAL_IND *pfvSignalInd1 is not const
 *
 * 09.02.05  Rev. \main\18 CM-DI/ESA3-Prhl
 * - change tVoid to void
 *
 * 16.02.05  Rev. \main\19 CM-DI/ESA3-Prhl
 * - typedef tCSM_LOCAL_CTRL_SIGNAL_APPL_CALLBACK added
 *
 * 24.02.05  Rev. \main\20 CM-DI/ESA3-Prhl
 * - CSM_SIGNAL modified
 *
 * 24.02.05  Rev. \main\21 CM-DI/ESA3-Prhl
 * - CSM_SIGNAL modified
 *
 * 25.02.05  Rev. \main\22 CM-DI/ESA3-Battis
 * - CSM_C_TX_TYPE_VALID_MASK added
 *
 * 07.03.05  Rev. \main\23 CM-DI/ESA3-Zurmhl
 * - "Logical Component Table" definitions for MCNet added
 * - file history changed to "doxygen-style"
 *
 * 21.03.05  Rev. \main\24 CM-DI/ESA3-Zurmhl
 * - CSM_CALLBACK modifier added to application callback functions
 *
 * 04.04.05  \main\25 CM-DI/ESA3-Prhl
 * - old define CSM_C_TX_AS_CONFIGURED added again. Please not use for new projects.
 *
 * 04.04.05  \main\26 CM-DI/ESA3-Prhl
 * - new CSM_SIGNAL and CSM_TELEGRAM
 *
 * 08.04.05  \main\27  CM-DI/ESA3-K"ohler
 * - Callback function types renamed from vPFNTp to pfv.
 *
 * 28.04.05  \main\28  CM-DI/ESA3-Battis
 * - VW merge
 *
 * 02.05.05  \main\29  CM-CR/ESD4-Tuschik
 * - Make MISRA happy
 *
 * 10.05.05  \main\30  CM-DI/ESA3-Battis
 * - Make MISRA happy (no unnamed unions allowed)
 *
 * 10.05.05  \main\31  CM-DI/ESA3-Battis
 * - define CSM_C_SIGNAL_READ_STATUS_MASK removed
 *
 * 12.05.05  \main\33 CM-DI/ESA3-Zurmhl
 * - further general error codes added
 *
 * 12.05.05  \main\34 CM-DI/ESA3-Zurmhl
 * - comments added/changed
 *
 * 05.07.05  \main\36 CM-DI/ESA3-Zurmhl
 * - further general error codes added (some removed from other header files)
 *
 * 06.07.05  \main\37 CM-DI/ESA3-Zurmhl
 * - some error codes adapted to new conventions
 *
 * 06.07.05  \main\38 CM-DI/ESA3-Zurmhl
 * - further general error codes added (some removed from other header files)
 *
 * 07.07.05  \main\39 CM-DI/ESA3-Zurmhl
 * - suffix "UL" added to signal defines (CSM_C_SIGNAL_*)
 * - further general error codes added
 *
 * 01.08.05  \main\40 CM-DI/ESA3-Battis
 * - defines for CALLBACK DIRECT handling added
 *   CBR: indication callbacks can be called directly on downstream task level
 *
 * 15.08.05  \main\41 CM-DI/ESA3-Zurmhl
 * - new error code: CSM_E_CSM_STATE_FORBIDS_CALL
 *
 * 23.08.05  \main\42 CM-DI/ESA3-Zurmhl
 * - new defines: CSM_C_ENABLE/CSM_C_DISABLE
 *
 * 24.10.05  \main\43 CM-DI/ESA3-Battis
 * - MCNet timer callback functions added.
 *
 * 01.11.05  \main\44 CM-DI/ESA3-Battis
 * - Comment modified.
 *
 * 04.11.05  \main\45 CM-DI/ESA3-Battis
 * - define CSM_C_APPL_DEFAULT added
 *
 * 16.11.05  \main\46 CM-DI/ESA3-Battis
 * - telegram index for deprecated telegrams added (CBR_C_DEPRECATED_IDX)
 *
 * 17.11.05  \main\47 CM-DI/ESA3-Battis
 * - CSM_C_APPL_DEFAULT renamed to CSM_C_APPL_ID_DEFAULT
 * - MCNet function headers inserted.
 *
 * 17.11.05  \main\48 CM-DI/ESA3-Prhl
 * - tCSM_LOCAL_CTRL_APPL_CALLBACK typing changed
 *
 * 17.11.05  \main\49 CM-DI/ESA3-Khler
 * - Defines CSM_C_NORMAL_COMMUNICATION_ON and _OFF added.
 *
 * 13.12.05  \main\50 CM-DI/ESA3-Khler
 * - Define CSM_E_INVALID_SIG_INDEX added.
 *
 * 28.02.06  \main\51 CM-DI/ESA3-Khler
 * - Macro CSM_M_MAKE_ERROR changed.
 * - Macro GLB_M_MAKE_ERROR removed.
 * - Doxygen comments added.
 *
 * 03.04.06  \main\52 CM-DI/ESA3-Battis
 * - error location CSM_C_ERR_LOC_CNP_BAP added
 * - typedefs for network protocol callbacks adjusted to CSM coding guidelines
 * - struct tCSM_VW_BAP_APPL_CALLBACK added fro BAP callback handlers
 *
 * 21.04.06  \main\53 CM-DI/ESA3-Battis
 * - tCSM_BAPInOut inserted for BAP application usage.
 *
 * 11.05.06  \main\54 CM-DI/ESA3-Battis
 * - global.h included on demand to resolve GLB_TY_InterfaceStruct on V850 systems.
 *
 * 11.05.06  \main\55 CM-DI/ESA3-Battis
 * - uwLength member of rBAPInOut changed to ulwLength
 *
 * 07.06.06  \main\56 CM-DI/ESA3-Prhl
 * - add error location CSM_C_ERR_LOC_CNM_OSEK CSM_C_ERR_LOC_CNM_GMLAN
 *
 * 14.06.06  \main\57 CM-DI/ESA3-Prhl
 * - virtual network support added
 *
 * 26.06.06  \main\58 CM-DI/ESA3-Zurmhl
 * - support for third CAN added
 *
 * 29.06.06  \main\59 CM-DI/ESA3-Haffner
 * - changes to make new proxy compilable
 *
 * 29.06.06  \main\60 CM-DI/ESA3-Prhl
 * - change tU16 to WORD
 *
 * 14.07.06  \main\61 CM-DI/ESA3-Prhl
 * - add feature CSM_S_USE_CALLBACK_WITH_HANDLE.
 *   Add the pvHandle from CSM_lApplCallbackInit() to callback routines.
 *   This is used to handle the process border jump (e.g. TENGINE subsystem)
 *
 * 19.07.06  \main\62 CM-DI/ESA3-Prhl
 * - modify feature CSM_S_USE_CALLBACK_WITH_HANDLE for signal indication
 *
 * 01.08.06  \main\63 CM-DI/ESA3-Zurmhl
 * - Added: CSM_C_INACTIVE, CSM_C_STARTUP, CSM_C_SHUTDOWN
 * - Values changed for CSM_C_NORMAL_COMMUNICATION_ON/OFF
 *
 * 21.08.06  \main\64  CM-DI/ESA3-Khler
 * - Protocoltype placeholder CSM_C_PTYPE_USED_STD_USDT_ISO added.
 * - Doxygen comments added.
 *
 * 22.08.06  \main\65  CM-DI/ESA3-K"ohler
 * - Define CSM_C_PTYPE_OPEL_USDT_ISO moved from csm_stack_i.h to csm_i.h
 *   because protocol types should be used only in the CAN stack manager and
 *   application and in no other CAN stack module.
 *
 * 27.09.06  \main\66 CM-DI/ESA3-Zurmhl
 * - Bugfix in CSM_M_MAKE_ERROR(): Now EC >0xFF handled correctly
 *
 * 28.09.06  \main\67 CM-DI/ESA3-Zurmhl
 * - CSM_E_INIT_WARNING added (can be used as standard initialization for error return)
 *
 * 28.09.06  \main\68  CM-DI/ESA3-K"ohler
 * - CSM_S_VW_BAP_TP featured with switch CSM_S_CNP_AVAILABLE to avoid warning.
 *
 * 18.10.06  \main\69  CM-DI/ESA3-Prhl
 * - PVOID replaced
 *
 * 26.10.06  \main\70  CM-DI/ESA3-Haffner
 * - Removed dependency of CSM_S_VW_BAP_TP on CSM_S_CNP_AVAILABLE.
 *
 * 02.11.06  \main\71  CM-DI/ESA3-Looby
 * - Added Renault ISO typedef,CSM_ADDR_RENAULT_ISO.
 *
 * 02.11.06  \main\72  CM-DI/ESA3-Looby
 * - Removed OPEL and Renault types and added generic CSM_ADDR_ISO_TP.
 *
 * 07.11.06  \main\73  CM-DI/ESA3-Khler
 * - CSM_ADDR_ISO_TP renamed tCSM_ADDR_ISO_TP.
 *
 * 07.11.06  \main\74  CM-DI/ESA3-Battis
 * - CSM_M_MAKE_CAN_1_SIGNAL_ID renamed to CSM_M_MAKE_SIGNAL_ID
 *   -> No bus dependent signals anymore.
 * - CSM_M_MAKE_CAN_2_SIGNAL_ID deleted.
 * - CSM_M_GET_BUS_NO_FROM_SIGNAL_ID deleted.
 *
 * 01.12.06  \main\75  CM-DI/ESA3-Prhl
 * - add switch CSM_S_USE_TKSE
 *
 * 12.01.07  \main\76  CM-DI/ESA3-Prhl
 * - add tCSM_UUDT_TX_APPL_CALLBACK and tCSM_UUDT_RX_APPL_CALLBACK
 *
 * 12.01.07  \main\77  CM-DI/ESA3-Prhl
 * - add define  CSM_C_ERR_LOC_CNP_UUDT
 *
 * 23.02.07  \main\78  VTeam-Prhl
 * - add typedef tCSM_GMLAN_NWM_APPL_CALLBACK
 *
 * 07.03.07  \main\79  VTeam-Battis
 * - ubAcknowledge added to union of tCSM_BAPInOut
 *
 * 28.03.07  \main\80  VTeam-Prhl
 * - make ready to compile with BAP
 *
 * 03.04.07  \main\81  VTeam-Prhl
 * - include bap_types.h removed
 *
 * 15.05.07  \main\82  VTeam-Prhl
 * - tCSM_TP20_APPL_CALLBACK added
 *
 * 14.06.07  \main\83  VTeam-Prhl
 * - old code removed to avoid lint warning
 *
 * 25.06.07  \main\84  VTeam-Prhl
 * - handle added to tCSM_PFN_BUS_ERROR_IND
 *
 * 03.08.07  \main\85  VTeam-Battis
 * - comment added.
 *
 * 19.09.07  \main\86  VTeam-Khler
 * - Signal Tx Type CSM_C_TX_STORE_ONLY added.
 *
 * 25.09.07  \main\87  VTeam-Khler
 * - Signal Id comment from PAG included.
 *
 * 15.10.07  \main\88  VTeam-Khler
 * - Define CSM_C_NO_HANDLE_PTR added.
 *
 * 29.10.07  \main\89  VTeam-Khler
 * - CSM_C_FNL_ON corrected to CSM_C_F_ON.
 *
 * 07.12.07  \main\90  VTeam-Battis
 * - error indication interface adjusted.
 *
 * 24.01.08  \main\91  VTeam-Zurmhl
 * - module history comment block enhanced (doxygen \ingroup tag etc.)
 *
 * 06.02.08  \main\92  VTeam-Prhl
 * - feature CSM_S_ITP_FF_INDICATION_AVAILABLE and CSM_S_ITP_ERROR_INDICATION_AVAILABLE added.
 *
 * 14.02.08  \main\93  VTeam-Prhl
 * - error indication interface corrected.
 *
 * 22.02.08  \main\94  VTeam-Prhl
 * - make doxygen happy.
 *
 * 27.02.08  \main\95  VTeam-Lansley / Funke
 * - added tCSM_OSEK_NWM_APPL_CALLBACK
 *
 * 22.02.08  \main\96  VTeam-Brunotte
 * - Error macro CSM_E_CALLBACK_MISSING added
 *
 * 10.04.08  \main\97  VTeam-Prhl
 * - typedef tCSM_MultipleSignalRead added.
 *
 * 16.04.08  \main\98  VTeam-Prhl
 * - make lint happy.
 *
 * 25.04.08  \main\99  VTeam-Prhl
 * - typedef tCSM_MultipleSignalWrite added.
 *
 * 02.05.08  \main\100  VTeam-Khler
 * - Comments in struct tCSM_GMLAN_NWM_APPL_CALLBACK added.
 *
 * 02.05.08  \main\101  VTeam-Prhl
 * - typedef tCSM_MPDT_APPL_CALLBACK added.
 *
 * 13.06.08  \main\102  VTeam-Khler
 * - Defines for CSM setup (init/exit )in t-engine platform added.
 *
 * 27.06.08  \main\103  VTeam-Pistoor
 * - Added macro CSM_M_GET_BUSNUMBER_FROM_SIGNAL_ID
 *
 * 04.07.08  \main\104  VTeam-Pistoor
 * - Added parameter variable names to callback functions to make it easier for developers
 *
 * 08.07.08  \main\105  VTeam-Khler
 * - Error location defines for access layer added.
 *
 * 17.07.08  \main\106  VTeam-Zurmhl
 * - Error macros CSM_M_GET_Ex enhanced
 *
 * 11.09.08  \main\107  VTeam-Funke/Basavaraj
 * - added switches for CSM_S_CNP_AVAILABLE
 *
 * 15.09.08  \main\108  Prhl
 * - added switches for CSM_S_CBR_AVAILABLE
 *
 * 15.09.08  \main\109  Prhl
 * - define CSM_C_ERR_LOC_CNP_MPDT added
 *
 * 22.09.08  \main\110  Prhl
 * - typedef tCSM_ADDR_MPDT added, typedef tCSM_MPDT_APPL_CALLBACK modified
 *
 * 11.11.08  \main\111  Pistoor
 * - Added element pfvDataErrorInd in typedef tCSM_UUDT_RX_APPL_CALLBACK
 *
 * 12.11.08  \main\112  Pistoor
 * - Renamed precompiler switch CSM_S_UUDT_IND_STATUS to CSM_S_UUDT_DATA_ERROR_IND_AVAILABLE
 *
 * 19.11.08  \main\113  VTeam-Khler
 * - Prototype for YYY_cCSMDataErrorInd() callback available for UUDT, too.
 * - Doxygen comments added, indentation and preprocessor statements changed.
 *
 * 29.11.08  \main\114  VTeam-Pistoor
 * - Added structure tCSM_BUS_ERROR_IND_CALLBACK
 *
 * 07.01.09  \main\115  VTeam-Franke
 * - error code CSM_E_NO_FREE_CALLBACK, if number of possible callbacks exceeded
 *
 * 04.02.09  \main\116  VTeam-Franke
 * - added CSM_C_ERR_LOC_CNP_LIN_DIAG
 *
 * 28.02.09  \main\117  VTeam-Franke
 * - CSM_S_UUDT_DATA_ERROR_IND_AVAILABLE -> CSM_S_UUDT_RX_DATA_ERROR_IND_AVAILABLE
 *
 * 18.03.09  \main\118  VTeam-Franke
 * - defines for yyy_vCSM_DataErrorInd added
 *
 * 25.03.09  \main\119  VTeam-Franke
 * - tCSM_PFN_BUS_ERROR_IND: parameters named
 *
 * 26.03.09  \main\120  Prhl
 * - make it ready to compile.
 *
 * 08.06.09  \main\121  Khler
 * - Define CSM_C_INVALID_HANDLE added.
 *
 * 25.03.09  \main\122  VTeam-Franke
 * - CSM_E_WRONG_BUS_STATE added
 *
 * 17.08.09  \main\123  VTeam-Franke
 * - CSM_C_DISCONNECT_ALL added
 *
 * 28.08.09  \main\124  VTeam-Khler
 * - Define CSM_C_MCNET_LC_CON2 added.
 *
 * 17.09.09  \main\125  VTeam-Franke
 * - CSM_C_DISCONNECT_ALL exchanged by CSM_C_ACTIVATE/DEACTIVATE_COMMUNICATION_REQUESTS.
 *
 * 26.10.09  \main\126  Prhl
 * - CSM_C_ERR_LOC_CNP_LCTR added
 * - missing feature switch added.
 *
 * 12.12.09  \main\127  VTeam-Franke
 * - added some error defines and location CSM_TRACE
 *
 * 25.01.10  \main\128  Prhl
 * - CSM_C_ERR_LOC_CNM_IVLAN added
 *
 * 12.08.10  \main\129  Pistoor
 * - Corrected CSM_M_GET_BUSNUMBER_FROM_SIGNAL_ID
 *
 * 07.09.10  \main\130  Feldhaus
 * - Defines and structs for J1939 TP added.
 *
 * 17.09.10  \main\131  Pistoor
 * - Added CSM_C_ERR_LOC_CNP_LIN_USDT
 *
 * 25.10.10  \main\132  Borck
 * - Added tCSM_OSEKI_NWM_APPL_CALLBACK for indirect NWM (used for Nissan))
 *
 * 21.12.10  \main\133  VTeam-Khler
 * - pvHandle and rTaskInfo in UUDT and J1939 application callback structs
 *   removed. => Incompatible API #1 structs.
 *   cnp_uudt.c main\21, cnp_uudt_m.h main\7, csm_stack_m.h main\210 necessary,
 *   perhaps some applications has to be changed, too.
 * - Feature switches CSM_S_J1939, ISO-TP and UUDT used for TP callback structs.
 * - Feature switches for data error indication changed.
 *
 * 26.01.11  \main\134  VTeam-Khler
 * - pvHandle and rTaskInfo in tCSM_LOCAL_CTRL_APPL_CALLBACK and
 *   tCSM_ISO_TP_USDT_APPL_CALLBACK removed.
.*   => Incompatible API #1 structs.
 *   csm_lctr.c main\17
 *   csm_itp.c  main\64
 *   cnp_itp.c  vector_adaptation\30, main\?
 *   csm_stack_m.h main\211, csm_stack_mp.h
 *   update necessary, perhaps some applications have to be changed, too.
 *
 * 31.03.11  \main\135  VTeam-Pistoor
 * - Moved macro CSM_M_MAKE_ERROR to csm_stack_m.h version 215, update also csm_stack_m.h!
 *
 * 21.06.11  \main\136  Borck
 * - bus state #defines derived now from ETG trace enums.
 *
 * 29.09.11 \main\137 CM/PJ-H- Basavaraj patil
 * - Modified UUDT calback structures based on general needs.
 *
 * 28.11.11 \main\141 CM/PJ-H- Basavaraj patil
 * - Support for both Big and Small endian signals within the project and within a message.
 * - i.e. add additional endian's parameter within the CBR signal configuration table.
 * - Used feature switch CSM_S_CBR_BOTH_ENDIANS_SIGNALS to implement this feature
 *
 * 10.01.13 \main\142 Feldhaus
 * - added CSM_C_TX_NO_TRANSMISSION for backwards compatibility with old Fiat SB08 CAN stack
 *
 * 21.02.13 \main\143 Pistoor
 * - added macro CSM_M_GET_BUS_NO_FROM_SIGNAL_ID again
 *
 * 10.05.13 \main\144 Prhl
 * - define CSM_C_ERR_LOC_CNM_VWHIGH added
 *
 * 25.05.13 \main\145 Prhl
 * - define SM_C_ERR_LOC_CDA_FCN added.
 *
 * 19.07.13 \main\146 Feldhaus
 * - added 4 defines for J1939 which are used to trigger a J1939 Acknowledge frame
 *   (the definitions specify the send ACK reason)
 *
 * 01.08.13 \main\147 Feldhaus
 * - added declaration of CSM_fCheckCommunicationReq to header which is needed by VW HIGH NM
 *
 * 09.12.14 \main\148 Prhl
 * - LINT warning removed.
 *
 * 04.03.15 \main\149 Pistoor
 * - Added bus number 4
 *
 * 27.04.15 \main\150 Borck
 * merged from ITG5/RNAIVI
 * - some #defines for signal abstraction added.
 * - struct tCSM_BR_SIGNAL_CALLBACK extended by marker for abstract signal registration
 *   This allows to pass the abstract signalID back to the registered caller in case of a signal change.
 *
 * 27.04.15 \main\151 Borck
 * - no functional changes. code clean-up for better readability to identify data types and functional sections.
 *
 * 08.06.15 \main\152 Borck
 * - ETG trace enums for generic output added (e.g. used in csm.c)
 *
 * 09.06.15 \main\153 Borck
 * - define for connect states reverted back to absolute values due to problems in MCAN early up (preprocessor cannot handle it)
 *
 * 05.08.15 \main\154 Borck
 * - ETG trace enum for signal states extended
 *
 * 31.08.15 \main\155 Borck
 * - ETG trace enum for signal states extended
 *
 * 08.10.15 \main\156 Borck
 * - feature mini abstraction (CSM_S_CBR_ABSTRACT_SIGNALS) removed completely
 *
 * 21.10.15 \main\157 Borck
 * - CSM_C_SIGNAL_HAS_NO_DATA_SOURCE introduced as additional info for abstract signal status
 *
 * 22.01.16 \main\158 M.Prhl
 * - element pvTelegramRxMask added to typedef CSM_TELEGRAM
 *
 * 02.06.16 \main\159 M.Prhl
 * - define CSM_C_SIGNAL_DATA_CRC_MISSMATCH and CSM_C_SIGNAL_ROLL_COUNTER_FAIL added.
 *
 * 23.06.16 \main\160 A.Borck
 * - element pvTelegramProxyMask added to typedef CSM_TELEGRAM
 *
 * \history_end
 *//**** END OF FILE **********************************************************/
