// Description and history of csm_lctr.c at the end of the file.
// File name for preprozessor commands
#define CSM_LCTR_SELF
// csm_lctr.c is a file according V-team CAN-SW architecture implementation rules
#define CSM_CAN_STACK

//*******************************************************************************
//* includes
//*******************************************************************************
#include "csm_stack_i.h"
//#include "csm_i.h"         // include currently not necessary
#include "csm_stack_m.h"
//#include "csm_m.h"         // include currently not necessary

#include "csm_lctr_i.h"    // local control interface
#include "csm_lctr_m.h"    // local control specific functions/defines of the can stack manager

#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
#if (CSM_S_CNP_LOCAL_CTRL == CSM_C_F_ON)
  #include "cnp_lctr_i.h"
#endif // CSM_S_CNP_LOCAL_CTRL == CSM_C_F_ON
#endif // CSM_S_CNP_AVAILABLE == CSM_C_F_ON

//#include "csm_c.h"         // include currently not necessary

#include "csm_m.h"

//************** ETG Tracing ******
#if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #define ETRACE_S_IMPORT_INTERFACE_GENERIC
  #include "etrace_if.h"

  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
    #include "csm_etg_trace.h"
    #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FD_CSM
    /* #define ETG_INLINE */
    #include "trcGenProj/Header/csm_lctr.c.trc.h"
  #endif
  #endif
#endif // CSM_S_TRACE

//*******************************************************************************
//* defines and macros (scope: csm module local)
//*******************************************************************************
// none or in csm_lctr_m(p).h

//*******************************************************************************
//* typedefs (scope: csm module local)
//*******************************************************************************
// data structure for CSM local control state
typedef struct
{
  BYTE  abReqAction[ CSM_LCTR_C_MAX_USER];           /**< Array to store the bAction parameter from
                                                      *   csm_lLocalCtrlCommunicationReq() from every user.
                                                      *   Possible values: CSM_C_STACK_DOWN, CSM_C_STACK_UP */
  BYTE  abActAction[ CSM_LCTR_C_MAX_USER];           /**< Array to store the actual action value from
                                                      *   every user. Possible values: CSM_C_STACK_DOWN, CSM_C_STACK_UP */
  /*BYTE  bBusState;*/                               /**< State of the bus. Possible values: CSM_C_STATE_BUS_DOWN,
                                                      *   CSM_C_STATE_BUS_LISTEN, CSM_C_STATE_BUS_STARTUP,
                                                      *   CSM_C_STATE_BUS_SHUTDOWN, CSM_C_STATE_BUS_UP */
  BOOLEAN afCommunicationReq[ CSM_LCTR_C_MAX_USER];  /**< Trigger array Flag for Communication request,
                                                      *   if TRUE communication request is pennding.
                                                      *   Possible value TRUE/FALSE */
  BOOLEAN afCommunicationCon[ CSM_LCTR_C_MAX_USER];  /**< Trigger array Flag for Communication confirmation,
                                                      *   if TRUE communication confirmation is pennding.
                                                      *   Possible value TRUE/FALSE */
  BOOLEAN afCommunicationInd[ CSM_LCTR_C_MAX_USER];  /**< Trigger array Flag for Communication indication,
                                                      *   if TRUE communication indication is pennding.
                                                      *   Possible value TRUE/FALSE */
}tCSM_LCTR_CONTROL_STATE;

typedef struct
{
  tCSM_LOCAL_CTRL_APPL_CALLBACK rCallback;            /**< Structure to store the callbacks passed from application to CSM via CSM_lApplCallbackInit() */

  #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
  void * pvHandle;                                    /**< Handle is used for AccessLayer. */
  #endif

  #if (CSM_S_USE_TKSE == CSM_C_F_ON)
  tCSM_TASK_INFO rTaskInfo;                           /**< Info which is necessary to call the callback in the uptream task on the
                                                       *   level of the task which made the PreInit before callback registration. */
  #endif
}tCSM_LOCAL_CTRL_APPL_CALLBACK_ADMIN;

//*******************************************************************************
//* variable definition (scope: csm module local)
//*******************************************************************************
// States for local control handling
CSM_STATIC tCSM_LCTR_CONTROL_STATE  csm_lctr_rLocalControlState;
tCSM_LOCAL_CTRL_APPL_CALLBACK_ADMIN csm_lctr_arCallbackFkt[CSM_LCTR_C_MAX_USER];  /**< This area stores the callback function for each user */


//*******************************************************************************
//* variable definition (scope: CAN stack)
//*******************************************************************************
#if (CSM_S_DEACTIVATE_COMMUNICATION_REQUESTS == CSM_C_F_ON)
BOOL csm_fIgnoreCommReq;
#endif // CSM_S_DEACTIVATE_COMMUNICATION_REQUESTS

// additional variables may be defined in csm_lctr_i(p).h

//*******************************************************************************
//* constants (scope: file local)
//*******************************************************************************
// none

//*******************************************************************************
//* function prototypes (scope: csm module local)
//*******************************************************************************
// none

// Feature switch is used only to control doxygen documentation
#if (CSM_S_DOXYGEN_HIDE_NON_API_FCT == CSM_C_F_OFF)
//*******************************************************************************
//* function implementation (scope: file local = static)
//*******************************************************************************

 /*******************************************************************************
 * function    CSM_LCTR_bGetUserNumber
 * \doxydocu
 * \brief      get user number.
 *
 *             get user number for local ctrl user
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESA3
 *
 * \param[in]  bBus (I)
 *             bus number
 *
 * \param[in]  dwProtocolType (I)
 *             protocol type, see table 2 of CAn SW architecture
 *
 * \param[in]  *pvAddressField (->I)
 *             addressing, dependent on protocol
 *
 * \param[in]  wApplId (I)
 *               Appl Id
 *
 * \return     BYTE
 *               user number = from 0 to CSM_LCTR_C_MAX_USER-1. CSM_LCTR_C_MAX_USER is the error value.
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \lastreview 25.01.08  CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 08.11.06 CM-DI/ESA3-Prhl
 * Initial Revision.
 * \history_end
 ******************************************************************************/
BYTE CSM_LCTR_bGetUserNumber(BYTE bBus, DWORD dwProtocolType, const void * pvAddressField, WORD wApplId)
{
  // parameter address field is not used
  BYTE bUserNumber;

  for (bUserNumber = 0; bUserNumber < CSM_LCTR_C_MAX_USER; bUserNumber++)
  {
    if( (bBus == csm_arCsmCfg[ bUserNumber].bBus) &&
        (dwProtocolType == csm_arCsmCfg[bUserNumber].dwProtocolType) &&
        (wApplId == csm_arCsmCfg[ bUserNumber].wApplId)
      )
    {
      return(bUserNumber);
    }
  }
  return(bUserNumber);
}/*lint !e715 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally */
#endif // CSM_S_DOXYGEN_HIDE_NON_API_FCT

//*******************************************************************************
//* function implementation (scope: CAN stack global)
//*******************************************************************************

/*******************************************************************************
 * function    CSM_LCTR_lInit
 * \doxydocu
 * \brief      Initializes Local Corntrol Server.
 *
 *             Initializes Local Control Server .
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESA3
 *
 * \param      bInitMode
 *             Allowed values:
 *                CSM_C_INIT_COLD, CSM_C_INIT_WARM, CSM_C_INIT_START (see csm_stack_i.h)
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR, CBR_E_ERROR_STATE_WHILE_INIT
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \lastreview 11.12.09 Prhl
 *             13.10.08 VTeam-Funke   .. returnvalue without type
 *             24.07.08 VTeam-Zurmhl .. Finding: Function to be moved to (new) csm_lctr.c
 *
 * \history_begin
 *
 * 23.07.09 VTeam-Jablonski
 * - Init of csm_arBusErrorIndCallbackFkt moved to CSM_lnit
 *
 * 13.10.08 VTeam-Funke
 * - reduce to one return point
 *
 * 24.07.08 VTeam-Zurmhl
 * - FS CSM_S_USE_TKSE added.
 *
 * 12.06.08 VTeam-Brunotte
 * - Initial Revision.
 * \history_end
 ******************************************************************************/
LONG CSM_LCTR_lInit(BYTE bInitMode)
{
  LONG lRet = CSM_C_NO_ERROR;
  BYTE bUserNumber;

  if (bInitMode == CSM_C_INIT_COLD)
  {
    for( bUserNumber = 0; bUserNumber < CSM_LCTR_C_MAX_USER; bUserNumber++)
    {
      csm_lctr_rLocalControlState.abReqAction[bUserNumber] = CSM_C_STACK_DOWN;
      csm_lctr_rLocalControlState.abActAction[bUserNumber] = CSM_C_STACK_DOWN;
      csm_lctr_rLocalControlState.afCommunicationReq[bUserNumber] = FALSE;
      csm_lctr_rLocalControlState.afCommunicationCon[bUserNumber] = FALSE;
      csm_lctr_rLocalControlState.afCommunicationInd[bUserNumber] = FALSE;
    }

    #if (CSM_S_DEACTIVATE_COMMUNICATION_REQUESTS == CSM_C_F_ON)
    csm_fIgnoreCommReq = FALSE;
    #endif // CSM_S_DEACTIVATE_COMMUNICATION_REQUESTS
  }
  else if (bInitMode == CSM_C_INIT_WARM)
  {
    // for all users:
    for (bUserNumber=0; bUserNumber < CSM_LCTR_C_MAX_USER; bUserNumber++)
    {
      #if (CSM_S_USE_TKSE == CSM_C_F_ON)
      {
        // init task info
        csm_lctr_arCallbackFkt[bUserNumber].rTaskInfo.idTask = 0;
      }
      #endif // CSM_S_USE_TKSE

      #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
      {
        // init handle
        csm_lctr_arCallbackFkt[bUserNumber].pvHandle = NULL;
      }
      #endif // CSM_S_USE_CALLBACK_WITH_HANDLE

      // init callback pointers
      csm_lctr_arCallbackFkt[bUserNumber].rCallback.pfvCommunicationCon = NULL;
      csm_lctr_arCallbackFkt[bUserNumber].rCallback.pfvCommunicationInd = NULL;
    } // loop end for all users

    #if (CSM_S_DEACTIVATE_COMMUNICATION_REQUESTS == CSM_C_F_ON)
    csm_fIgnoreCommReq = FALSE;
    #endif // CSM_S_DEACTIVATE_COMMUNICATION_REQUESTS
  }
  else if (bInitMode == CSM_C_INIT_START)
  {
    // empty
  }
  else
  {
    // invalid init mode -> error
    lRet = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_UNKNOWN_STATE);
  }

  return(lRet);
}

/*******************************************************************************
 * function    CSM_LCTR_lExit
 * \doxydocu
 * \brief      Stops CSM local control server.
 *
 *             Stops CAN local control server.
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESA3
 *             When the stop state is called, the CSM state is set to DOWN. This state
 *             switch is not indicated to registered applications.
 *
 * \param[in]  bExitMode
 *             Allowed values (see csm_stack_i.h):
 *                - CSM_C_EXIT_STOP
 *                - CSM_C_EXIT_OFF
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \lastreview 13.10.08 VTeam-Funke   .. returnvalue without type
 *             24.07.08 VTeam-Zurmhl .. Finding: Function to be moved to (new) csm_lctr.c
 *
 * \history_begin
 * 13.10.08 VTeam-Funke
 * - align return value to same name
 *
 * 24.07.08 VTeam-Zurmhl
 * - FS CSM_S_USE_TKSE added, definition of bUserNumber moved
 *
 * 12.06.08 VTeam-Brunotte
 * - csm_cbr_vApplCallbackDelAll is called in off state
 * \history_end
 ******************************************************************************/
LONG CSM_LCTR_lExit(BYTE bExitMode)
{
  LONG lRet = CSM_C_NO_ERROR;
  switch (bExitMode)
  {
    case CSM_C_EXIT_STOP:
    {
      // empty
    }
    break;

    case CSM_C_EXIT_OFF:
    {
      BYTE bUserNumber;
      for (bUserNumber=0; bUserNumber < CSM_LCTR_C_MAX_USER; bUserNumber++)
      {
        #if (CSM_S_USE_TKSE == CSM_C_F_ON)
        {
          csm_lctr_arCallbackFkt[bUserNumber].rTaskInfo.idTask = 0;
        }
        #endif // CSM_S_USE_TKSE

        #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
        {
          csm_lctr_arCallbackFkt[bUserNumber].pvHandle = NULL;
        }
        #endif // CSM_S_USE_CALLBACK_WITH_HANDLE
        csm_lctr_arCallbackFkt[bUserNumber].rCallback.pfvCommunicationCon = NULL;
        csm_lctr_arCallbackFkt[bUserNumber].rCallback.pfvCommunicationInd = NULL;
      }
    }
    break;

    default:
    {
      // invalid parameter -> error
      lRet = CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_UNKNOWN_STATE);
    }
    break;
  }

  return(lRet);
}

/*******************************************************************************
 * function    CSM_LCTR_lApplCallbackInit
 * \doxydocu
 * \brief      Initial callback function
 *
 *             Initial callback function. Please call this function only once for each channel
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/EHS3
 *
 * \param[in]  pvHandle
 *              pointer to handle. For future use! May be a deinit() function
 *
 * \param[in]  bBus (I)
 *             CAN bus number
 *
 * \param[in]  dwProtocolType (I)
 *             protocol type, see table 2 of CAn SW architecture
 *
 * \param[in]  *pvAddressField (->I)
 *             addressing, dependent on protocol
 *
 * \param[in]  *prCallBackFkt (->I)
 *             pointer to callback function
 *
 * \return     Error value
 *              CSM_C_NO_ERROR, CSM_E_INVALID_PARA
 *
 * \access     application
 * \reentrant  yes
 *
 *
 * \lastreview 13.10.08 VTeam-Funke
 *             05.12.06 CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 13.10.08 VTeam-Funke
 * - added static error handling
 *
 * 17.10.06 CM-DI/ESA3-Khler
 * - PVOID replaced by void *.
 *
 * 14.07.06 CM-DI/ESA3-Prhl
 * add bBus
 *
 * 27.02.06 CM-DI/ESA3-K"ohler
 * Parameter pvAddressField is now a pointer to const.
 *
 * 11.02.05 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
LONG CSM_LCTR_lApplCallbackInit(
                                #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
                                void* pvHandle,
                                #endif
                                BYTE  bBus,
                                DWORD dwProtocolType,
                                const void* pvAddressField,
                                tCSM_LOCAL_CTRL_APPL_CALLBACK * prCallBackFkt)
{
  LONG lRet = CSM_C_NO_ERROR;

  BYTE bUserNumber;
  WORD wApplId = prCallBackFkt->u16ApplID;

  bUserNumber = CSM_LCTR_bGetUserNumber(bBus, dwProtocolType, pvAddressField, wApplId);
  if (bUserNumber < CSM_LCTR_C_MAX_USER)
  {
    #if (CSM_S_USE_TKSE == CSM_C_F_ON)
    {
      // store task context
      CSM_M_GET_TASK_CONTEXT( &(csm_lctr_arCallbackFkt[ bUserNumber].rTaskInfo));
    }
    #endif // CSM_S_USE_TKSE

    #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
    {
      // store handle
      csm_lctr_arCallbackFkt[ bUserNumber].pvHandle = pvHandle;
    }
    #endif // CSM_S_USE_CALLBACK_WITH_HANDLE

    // store confirmation and indication callback
    csm_lctr_arCallbackFkt[ bUserNumber].rCallback.pfvCommunicationCon = prCallBackFkt->pfvCommunicationCon;
    csm_lctr_arCallbackFkt[ bUserNumber].rCallback.pfvCommunicationInd = prCallBackFkt->pfvCommunicationInd;
  }
  else
  {
    // invalid parameter -> eroor
    lRet = CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_INVALID_PARA);
  }
  return( lRet);
}/*lint !e818 PQM_authorized_multi_123. Reason: Interface not defined as const*/

/*******************************************************************************
 * function    CSM_LCTR_lCommunicationReq
 * \doxydocu
 * \brief      Requests the CAN bus communication form system control.
 *
 *             Requests the CAN bus communication form system control.
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/EHS3
 *
 * \param[in]  bBus (I)
 *             bus number
 *
 * \param[in]  dwProtocolType (I)
 *             protocol type, see table 2 of CAn SW architecture
 *
 * \param[in]  *pvAddressField (->I)
 *             addressing, dependent on protocol
 *
 * \param[in]  bAction (I)
 *             Allowed values: CSM_C_STACK_UP, CSM_C_STACK_DOWN
 *
 * \param[in]  wApplID
 *             Allowed values: CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL...
 *
 * \return     Error value
 *               CSM_C_NO_ERROR, CSM_E_INVALID_PARA
 *
 * \access     application
 * \reentrant  no, but it is only alowed to call from a single application
 *
 * \lastreview 13.10.08  VTeam-Funke
 *             09.11.06  CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 13.10.08  VTeam-Funke
 * - align return value to same name
 *
 * 08.11.06  CM-DI/ESA3-Prhl
 * multi bus support added
 *
 * 11.02.05 CM-DI/ESA3 Prhl
 * Initial Revision.
 *
 * \history_end
 * \todo bBus is not used. CSM only works with one bus
******************************************************************************/
LONG CSM_LCTR_lCommunicationReq(BYTE bBus, DWORD dwProtocolType, const void * pvAddressField, BYTE bAction, WORD wApplID)
{
  LONG  lRet = CSM_C_NO_ERROR;
  BYTE bUserNumber;

  // get user number from LCTRL configuration
  bUserNumber = CSM_LCTR_bGetUserNumber(bBus, dwProtocolType, pvAddressField, wApplID);

  if (bUserNumber < CSM_LCTR_C_MAX_USER)
  {
    // user number is in range
    #if (CSM_S_DEACTIVATE_COMMUNICATION_REQUESTS == CSM_C_F_ON)
    // if we have the option to deactivate com requests, check the action
    if (bAction == CSM_C_DEACTIVATE_COMMUNICATION_REQUESTS)
    {
      // DE-ACTIVATE
      csm_fIgnoreCommReq = TRUE;
      csm_vTriggerCsmDownstreamTask(CSM_C_DOWN_REASON_CSM_BUS_STATE_CHANGED);

      #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      {
        #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
        ETG_TRACE_USR2(("CSM_LCTR_lCommunicationReq - DeActivateComRequests - Bus:%X  Prot:%X  ActAction:0x%X",
                         ETG_CENUM(tCSM_BUS_NUMBER,               (tU8)  bBus),
                         ETG_CENUM(tCSM_PROTOCOL_TYPE,            (tU32) dwProtocolType),
                         (tU8) csm_lctr_rLocalControlState.abActAction[bUserNumber]
                      ));
        #endif
      }
      #else // CSM_S_TRACE_ETG
      {
        // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
        BYTE abData[10];
        abData[0] = csm_arCsmCfg[bUserNumber].bBus;
        abData[1] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(csm_arCsmCfg[bUserNumber].dwProtocolType));
        abData[2] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(csm_arCsmCfg[bUserNumber].dwProtocolType));
        abData[3] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(csm_arCsmCfg[bUserNumber].dwProtocolType));
        abData[4] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(csm_arCsmCfg[bUserNumber].dwProtocolType));

        abData[5] = 0x00;
        abData[6] = 0x00;

        abData[7] = csm_lctr_rLocalControlState.abActAction[bUserNumber];
        abData[8] = CSM_M_GET_HIBYTE(csm_arCsmCfg[bUserNumber].wApplId);
        abData[9] = CSM_M_GET_LOBYTE(csm_arCsmCfg[bUserNumber].wApplId);

        CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_DEACTIVATE_COMMUNICATION_REQ, abData, 10);
      }
      #endif // CSM_S_TRACE_ETG
      #endif // CSM_S_TRACE
    } // CSM_C_DEACTIVATE_COMMUNICATION_REQUESTS
    else
      // not DEACTIVATE, so check for ACTIVATE
      if( bAction == CSM_C_ACTIVATE_COMMUNICATION_REQUESTS)
      {
        // ACTIVATE
        csm_fIgnoreCommReq = FALSE;
        csm_vTriggerCsmDownstreamTask(CSM_C_DOWN_REASON_CSM_BUS_STATE_CHANGED);

        #if (CSM_S_TRACE == CSM_C_F_ON)
        #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
        {
          #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
          ETG_TRACE_USR2(("CSM_LCTR_lCommunicationReq - ActivateComRequests - Bus:%X  Prot:%X  ActAction:0x%X",
                          ETG_CENUM(tCSM_BUS_NUMBER,               (tU8)  bBus),
                          ETG_CENUM(tCSM_PROTOCOL_TYPE,            (tU32) dwProtocolType),
                          (tU8) csm_lctr_rLocalControlState.abActAction[bUserNumber]
                        ));
          #endif
        }
        #else // CSM_S_TRACE_ETG
        {
          // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
          BYTE abData[10];
          abData[0] = csm_arCsmCfg[bUserNumber].bBus;
          abData[1] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(csm_arCsmCfg[bUserNumber].dwProtocolType));
          abData[2] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(csm_arCsmCfg[bUserNumber].dwProtocolType));
          abData[3] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(csm_arCsmCfg[bUserNumber].dwProtocolType));
          abData[4] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(csm_arCsmCfg[bUserNumber].dwProtocolType));

          abData[5] = 0x00;
          abData[6] = 0x00;

          abData[7] = csm_lctr_rLocalControlState.abActAction[ bUserNumber];
          abData[8] = CSM_M_GET_HIBYTE(csm_arCsmCfg[bUserNumber].wApplId);
          abData[9] = CSM_M_GET_LOBYTE(csm_arCsmCfg[bUserNumber].wApplId);

          CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_ACTIVATE_COMMUNICATION_REQ, abData, 10);
        }
        #endif // CSM_S_TRACE_ETG
        #endif // CSM_S_TRACE
      }
    else
    #endif // CSM_S_DEACTIVATE_COMMUNICATION_REQUESTS
    {
      if (csm_lctr_rLocalControlState.abReqAction[bUserNumber] != bAction)
      {
        // nderung
        csm_lctr_rLocalControlState.abReqAction[bUserNumber] = bAction;
        csm_lctr_rLocalControlState.afCommunicationReq[bUserNumber] = TRUE;
        csm_vTriggerCsmDownstreamTask(CSM_C_DOWN_REASON_CSM_COMMUNICATION_REQ);
      }
    }
    // communication confirmation direkt anstoen
    csm_lctr_rLocalControlState.afCommunicationCon[bUserNumber] = TRUE;
    csm_vTriggerCsmUpstreamTask(CSM_C_IND_REASON_CSM_COMMUNICATION_CON);
  }
  else
  {
    // user number out of range -> error
    lRet = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_INVALID_PARA);
  }
  return lRet;
}

 /*******************************************************************************
 * function    CSM_LCTR_lDataReq
 * \doxydocu
 * \brief      Request to send a message to the CAN stack
 *
 * \see        CAN-SW Architektur, written by CM-DI/VTeam
 *
 * \param[in]  dwProtocolType
 *              Protocol type, see table 2 of CAn SW architecture.
 *
 * \param[in]  pvAddressField
 *              Protocol dependent address field.
 *
 * \param[in]  pbData
 *              Pointer to data which shall be sent
 *
 * \param[in]  wDataLength
 *              Number of data bytes which shall be sent
 *
 * \return     Error/no error value
 *
 * \access     application
 * \reentrant  only for different channels
 *
 * \lastreview 11.12.09 Prhl
 *
 * \history_begin
 * 03.04.09 Prhl
 * - Initial Revision.
 * \history_end
 ******************************************************************************/
LONG  CSM_LCTR_lDataReq(DWORD dwProtocolType, const void* pvAddressField, const BYTE* pbData, WORD wDataLength)
{
  LONG  lRet = CSM_C_NO_ERROR;

  // if you got linkage problems because of missing CNP LCTRL functions,
  // then the solution is NOT to introduce a compiler switch which is never used in an example configuration.
  // Take the required CNP LCTR module into your project instead and leave the called
  // function empty, as other projects do it as well.
  #if ((CSM_S_CNP_AVAILABLE == CSM_C_F_ON))
  #if (CSM_S_CNP_LOCAL_CTRL == CSM_C_F_ON)
  lRet = CNP_LCTR_lDataReq(dwProtocolType,
                           pvAddressField,
                           pbData,
                           wDataLength);
  #endif /* CSM_S_CNP_AVAILABLE */
  #else  /* CSM_S_CNP_AVAILABLE */
  CSM_M_PARAMETER_INTENTIONALLY_UNUSED(dwProtocolType);
  CSM_M_PARAMETER_INTENTIONALLY_UNUSED(pvAddressField);
  CSM_M_PARAMETER_INTENTIONALLY_UNUSED(pbData);
  CSM_M_PARAMETER_INTENTIONALLY_UNUSED(wDataLength);
  lRet = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_GENERAL_ERROR);
  #endif /* CSM_S_CNP_AVAILABLE */

  return lRet;
}/*lint !e715 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally*/

/*******************************************************************************
 * function    CSM_LCTR_lDataRead
 * \doxydocu
 * \brief      Synchronous read request on the CSM
 *
 * \param[in]  dwProtocolType
 *              Protocol type, see table 2 of CAN SW architecture.
 *
 * \param[in]  pvAddressField
 *              Protocol dependend address field.
 *
 * \param[in]  pbData
 *              Pointer to buffer in which data shall be read.
 *
 * \param[in]  wDataLength
 *              Number of data bytes which shall be send.
 *
 *
 * \return     Error value
 *
 * \access     CSM module
 * \reentrant  no
 *
 * \lastreview 11.12.09 Prhl
 *
 * \history_begin
 * 03.04.09 Prhl
 * - Initial Revision.
 * \history_end
 ******************************************************************************/
LONG  CSM_LCTR_lDataRead(DWORD dwProtocolType, const void* pvAddressField, BYTE* pbData, WORD  wDataLength)
{
  LONG  lRet = CSM_C_NO_ERROR;

  CSM_M_PARAMETER_INTENTIONALLY_UNUSED(dwProtocolType);
  CSM_M_PARAMETER_INTENTIONALLY_UNUSED(pvAddressField);
  CSM_M_PARAMETER_INTENTIONALLY_UNUSED(pbData);
  CSM_M_PARAMETER_INTENTIONALLY_UNUSED(wDataLength);

  return lRet;
}

/*******************************************************************************
 * function    CSM_LCTR_lGetCommunicationState
 * \doxydocu
 * \brief      The actual state of the "connection" can be
 *             checked with this function.
 *
 * \see        CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \param[in]  bBus
 *              Bus number, only needed for "broadcast" protocol types
 *
 * \param[in]  dwProtocolType
 *              Protocol type, see corresponding table of CAN SW architecture
 *
 * \param[in]  pvAddressField
 *              Protocol dependend address field
 *
 * \param[out] pbConnectState
 *              CSM_C_CONNECTED     /  CSM_C_STACK_UP
 *              CSM_C_DISCONNECTED  /  CSM_C_STACK_DOWN
 *
 * \param[in]  wApplId
 *              Application Id, only needed for "broadcast" protocol types
 *
 * \return     Error value (wrong Parameter) if < CSM_C_NO_ERROR
 *             else no error
 *
 * \access     all
 * \reentrant  yes
 *
 * \lastreview 09.11.06  CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 08.11.06  CM-DI/ESA3-Prhl
 * multi bus support added
 * 21.08.06 CM-DI/ESA3-Prhl
 * First version
 *
 * \history_end
 * \todo bBus is not used. CSM only works with one bus
 ******************************************************************************/
CSM_API LONG CSM_LCTR_lGetCommunicationState(BYTE  bBus, DWORD dwProtocolType, const void* pvAddressField, BYTE* pbConnectState, WORD  wApplId)
{
  LONG  lRet = CSM_C_NO_ERROR;
  BYTE bUserNumber;

  bUserNumber = CSM_LCTR_bGetUserNumber(bBus, dwProtocolType, pvAddressField, wApplId);

  if( bUserNumber < CSM_LCTR_C_MAX_USER)
  {
    *pbConnectState = csm_lctr_rLocalControlState.abActAction[ bUserNumber];
  }
  else
  {
    lRet = CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_INVALID_PARA);
  }

  return (lRet);
}/* CSM_LCTR_lGetCommunicationState() */

/*******************************************************************************
 * function    CSM_LCTR_vCSMBusStateInd
 * \doxydocu
 * \brief      Bus State indication callback.
 *
 *             This funktion is call after the bus state is changed.
 *
 * \param[in]  bBus
 *             Bus
 *
 * \param[in]  bCsmActState
 *             New bus state.
 *             possible values:
 *                CSM_C_STATE_BUS_DOWN, CSM_C_STATE_BUS_LISTEN, CSM_C_STATE_BUS_STARTUP,
 *                CSM_C_STATE_BUS_UP, CSM_C_STATE_BUS_SHUTDOWN
 *
 * \param[in]  bVN
 *             Bit field for virtual networks
 *
 * \return     void
 *
 * \access     interrupt
 * \reentrant  yes
 *
 * \lastreview  09.11.06 CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 08.11.06  CM-DI/ESA3-Prhl
 * multi bus support added
 *
 * 14.06.06  CM-DI/ESA3-Prhl
 * virtual network support added
 *
 * 11.02.05 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void CSM_LCTR_vCSMBusStateInd(
                              BYTE bBus,
                              BYTE bCsmActState
                              #if (CSM_S_VIRTUAL_NETWORK == CSM_C_F_ON)
                              , BYTE bVN
                              #endif /* CSM_S_VIRTUAL_NETWORK */
                              )
{
  WORD wCount;
  #if (CSM_S_VIRTUAL_NETWORK == CSM_C_F_ON)
    CSM_M_PARAMETER_INTENTIONALLY_UNUSED(bVN);
  #endif /* CSM_S_VIRTUAL_NETWORK */

    switch( bCsmActState)
    {
      case CSM_C_STATE_BUS_DOWN:
      {
        for( wCount = 0; wCount < CSM_LCTR_C_MAX_USER; wCount++)
        {
          if( csm_arCsmCfg[ wCount].bBus == bBus)
          {
            if( csm_lctr_rLocalControlState.abActAction[ wCount] != CSM_C_STACK_DOWN)
            {
              csm_lctr_rLocalControlState.abActAction[ wCount] = CSM_C_STACK_DOWN;
              csm_lctr_rLocalControlState.afCommunicationInd[ wCount] = TRUE;
              csm_vTriggerCsmUpstreamTask( CSM_C_IND_REASON_CSM_COMMUNICATION_IND);
            }
          }
        }
      }
      break;
      case CSM_C_STATE_BUS_LISTEN:
      case CSM_C_STATE_BUS_STARTUP:
      case CSM_C_STATE_BUS_UP:
      case CSM_C_STATE_BUS_SHUTDOWN:
      {
        for( wCount = 0; wCount < CSM_LCTR_C_MAX_USER; wCount++)
        {
          if( csm_arCsmCfg[ wCount].bBus == bBus)
          {
            if( csm_lctr_rLocalControlState.abActAction[ wCount] != CSM_C_STACK_UP)
            {
              csm_lctr_rLocalControlState.abActAction[ wCount] = CSM_C_STACK_UP;
              csm_lctr_rLocalControlState.afCommunicationInd[ wCount] = TRUE;
              csm_vTriggerCsmUpstreamTask( CSM_C_IND_REASON_CSM_COMMUNICATION_IND);
            }
          }
        }
      }
      break;
      default:
      {
        CSM_vAssert( FALSE);   /*lint !e506 PQM_authorized_multi_226 */ /* Invalid bus state shall cause an assert - 21.01.2011 krv2hi */
      }
      break;
    }
}

/*******************************************************************************
 * function    CSM_LCTR_bGetUserAssignedBus
 * \doxydocu
 * \brief      return the configured bus for configured user
 *
 * \param[in]  bUserNumber
 *             requested User
 *
 * \return     Bus number
 *
 * \access     from CSM modules
 * \reentrant  yes
 *
 * \lastreview  21.04.15 Borck
 *
 * \history_begin
 *
 * 21.04.15 Borck
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
BYTE CSM_LCTR_bGetUserAssignedBus(BYTE bUserNumber)
{
  return csm_arCsmCfg[bUserNumber].bBus;
}

/*******************************************************************************
 * function    CSM_LCTR_fCheckCommunicationReq
 * \doxydocu
 * \brief      Calculate Communication request.
 *
 *             Calculate Communication request.
 *             This function is called to calculate the CommunicationReq state.
 *
 * \param[in]  bBus
 *
 * \return     BOOL
 *               FALSE: the bus is not needed by CSM
 *               TRUE: the bus is neened by CSM
 *
 * \access     CAN-Task
 * \reentrant  yes, but only call by CAN-Task
 *
 * \lastreview 09.11.06  CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 08.11.06  CM-DI/ESA3-Prhl
 * multi bus support added
 *
 * 22.09.04 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
BOOL CSM_LCTR_fCheckCommunicationReq(BYTE bBus)
{
  WORD wCount;

  for( wCount = 0; wCount < CSM_LCTR_C_MAX_USER; wCount++)
  {
    if( csm_arCsmCfg[ wCount].bBus == bBus)
    {
      #if (CSM_S_CSM_LOCAL_CTRL_USER_FUNCTION == CSM_C_F_ON)
      if( csm_arCsmCfg[ wCount].pfCheckAwakeFunction != NULL)
      {
        BOOL fTemp;
        fTemp = csm_arCsmCfg[ wCount].pfCheckAwakeFunction( csm_lctr_rLocalControlState.abReqAction[ wCount]);
        if( fTemp == TRUE)
        {
          return TRUE;
        }
      }
      else
      #endif // CSM_S_CSM_LOCAL_CTRL_USER_FUNCTION
      {
        if( csm_lctr_rLocalControlState.abReqAction[ wCount] != CSM_C_STACK_DOWN)
        {
          return TRUE;
        }
      }
    }
  }
  return FALSE;
}

/*******************************************************************************
 * function    CSM_LCTR_vProcessCommunicationReq
 * \doxydocu
 * \brief      Handle communication request.
 *
 *             Handle communication request.
 *             This function is call in downstream-task after trigger with CSM_C_DOWN_REASON_CSM_COMMUNICATION_REQ.
 *
 * \return     void
 *
 * \access     CAN-Task
 * \reentrant  no, but only call by CAN-Task
 *
 * \lastreview 09.11.06  CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 08.11.06  CM-DI/ESA3-Prhl
 * multi bus support added
 * 22.09.04 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void CSM_LCTR_vProcessCommunicationReq(void)
{
  BYTE bUserNumber;

  for( bUserNumber = 0; bUserNumber < CSM_LCTR_C_MAX_USER; bUserNumber++)
  {
    if( csm_lctr_rLocalControlState.afCommunicationReq[ bUserNumber] == TRUE)
    {
      csm_lctr_rLocalControlState.afCommunicationReq[ bUserNumber] = FALSE;

      // hier knnten Funktionen stehen die auf CAN-Task level nach einem CommunicationReq gemacht werden sollen
      #if (CSM_S_CSM_LOCAL_CTRL_USER_FUNCTION == CSM_C_F_ON)
      if( csm_arCsmCfg[ bUserNumber].pvProcessNewAction != NULL)
      {
        csm_arCsmCfg[ bUserNumber].pvProcessNewAction( csm_lctr_rLocalControlState.abReqAction[ bUserNumber]);
      }
      #endif // CSM_S_CSM_LOCAL_CTRL_USER_FUNCTION

      csm_vTriggerCsmDownstreamTask( CSM_C_DOWN_REASON_CSM_BUS_STATE_CHANGED);
    }
  }
}

/*******************************************************************************
 * function    CSM_LCTR_vProcessCommunicationCon
 * \doxydocu
 * \brief      Handle communication confimation.
 *
 *             Handle communication confimation.
 *             This function is call in upstream-task after trigger with CSM_C_IND_REASON_CSM_COMMUNICATION_CON.
 *
 * \return     void
 *
 * \access     Upstream-Task
 * \reentrant  no, but only call by Upstream-Task
 *
 * \lastreview 25.01.08 CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 08.11.06  CM-DI/ESA3-Prhl
 * multi bus support added
 * 01.06.05 CM-DI/ESA3-Prhl
 * all call back parameter now used from configuration struct
 *
 * 22.09.04 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void CSM_LCTR_vProcessCommunicationCon(void)
{
  BYTE bUserNumber;

  for (bUserNumber = 0; bUserNumber < CSM_LCTR_C_MAX_USER; bUserNumber++)
  {
    if (csm_lctr_rLocalControlState.afCommunicationCon[bUserNumber] == TRUE)
    {
      csm_lctr_rLocalControlState.afCommunicationCon[bUserNumber] = FALSE;

      if (csm_lctr_arCallbackFkt[bUserNumber].rCallback.pfvCommunicationCon != NULL)
      {
        #if (CSM_S_TRACE == CSM_C_F_ON)
          #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
            #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
            #if (CSM_S_ETG_SHORT_OUTPUT == CSM_C_F_OFF)
            ETG_TRACE_USR1(("CSM_LCTR_vProcessCommunicationCon - Bus:%X  Protocol:0x%X  Addr:0x%02X%02X  Action:0x%02X  ApplID:0x%04X",
            ETG_CENUM(tCSM_BUS_NUMBER,      (tU8)  csm_arCsmCfg[bUserNumber].bBus),
            ETG_CENUM(tCSM_PROTOCOL_TYPE,   (tU32) csm_arCsmCfg[bUserNumber].dwProtocolType),
            0x00,
            0x00,
            csm_lctr_rLocalControlState.abActAction[ bUserNumber],
            csm_arCsmCfg[bUserNumber].wApplId ));
            #else
            ETG_TRACE_ERR(("%04x%02x%08x%02x%02x%02x%04x", CSM_C_ETG_FKT_CSM_LCTR_VPROCESSCOMMUNICATIONCON,
            ETG_CENUM(tCSM_BUS_NUMBER,      (tU8)  csm_arCsmCfg[bUserNumber].bBus),
            ETG_CENUM(tCSM_PROTOCOL_TYPE,   (tU32) csm_arCsmCfg[bUserNumber].dwProtocolType),
            0x00,
            0x00,
            csm_lctr_rLocalControlState.abActAction[bUserNumber],
            csm_arCsmCfg[bUserNumber].wApplId));
            #endif
            #endif
          #else // CSM_S_TRACE_ETG
          {
            // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
            BYTE abData[10];
            abData[0] = csm_arCsmCfg[bUserNumber].bBus;
            abData[1] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(csm_arCsmCfg[bUserNumber].dwProtocolType));
            abData[2] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(csm_arCsmCfg[bUserNumber].dwProtocolType));
            abData[3] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(csm_arCsmCfg[bUserNumber].dwProtocolType));
            abData[4] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(csm_arCsmCfg[bUserNumber].dwProtocolType));

            abData[5] = 0x00;
            abData[6] = 0x00;

            abData[7] = csm_lctr_rLocalControlState.abActAction[bUserNumber];
            abData[8] = CSM_M_GET_HIBYTE(csm_arCsmCfg[bUserNumber].wApplId);
            abData[9] = CSM_M_GET_LOBYTE(csm_arCsmCfg[bUserNumber].wApplId);

            CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_COMMUNICATION_CON, abData, 10);
          }
          #endif // CSM_S_TRACE_ETG
        #endif // CSM_S_TRACE

        #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
          CSM_M_SET_TASK_CONTEXT( &(csm_lctr_arCallbackFkt[ bUserNumber].rTaskInfo),
                                  csm_lctr_arCallbackFkt[ bUserNumber].rCallback.pfvCommunicationCon(
                                                          csm_lctr_arCallbackFkt[ bUserNumber].pvHandle,
                                                          csm_arCsmCfg[ bUserNumber].bBus,
                                                          csm_arCsmCfg[ bUserNumber].dwProtocolType,
                                                          NULL,
                                                          csm_lctr_rLocalControlState.abActAction[ bUserNumber],
                                                          csm_arCsmCfg[ bUserNumber].wApplId) );
        #else // CSM_S_USE_CALLBACK_WITH_HANDLE
          CSM_M_SET_TASK_CONTEXT( &(csm_lctr_arCallbackFkt[ bUserNumber].rTaskInfo),
                                 csm_lctr_arCallbackFkt[ bUserNumber].rCallback.pfvCommunicationCon(
                                                          csm_arCsmCfg[ bUserNumber].bBus,
                                                          csm_arCsmCfg[ bUserNumber].dwProtocolType,
                                                          NULL,
                                                          csm_lctr_rLocalControlState.abActAction[ bUserNumber],
                                                          csm_arCsmCfg[ bUserNumber].wApplId) );
        #endif // CSM_S_USE_CALLBACK_WITH_HANDLE
      }
    }
  }
}

/*******************************************************************************
 * function    CSM_ITP_vPROXYTriggerCommunicationCon
 * \doxydocu
 * \brief      Callback function from PROXY to trigger a communication confirmation
 *
 * \param[in]  bUserNumber
 *                 Data
 *
 * \return     void
 *
 * \access     CSM task level
 * \reentrant  no, but only call from interrupt
 *
 * \lastreview
 *
 * \history_begin
 *
 * 23.08.13 Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
#if (CSM_S_REMOTE_STACK == CSM_C_F_ON)
void CSM_LCTR_vPROXYCommunicationCon( BYTE bBus, DWORD dwProtokollType, BYTE bState, WORD wApplID)
{
  BYTE bUserNumber;

  bUserNumber = CSM_LCTR_bGetUserNumber(bBus, dwProtokollType, NULL, wApplID);

  if( bUserNumber < CSM_LCTR_C_MAX_USER)
  {
    csm_lctr_rLocalControlState.abActAction[ bUserNumber] = bState;
    csm_lctr_rLocalControlState.afCommunicationCon[ bUserNumber] = TRUE;
    csm_vTriggerCsmUpstreamTask( CSM_C_IND_REASON_CSM_COMMUNICATION_CON);
  }
}

/*******************************************************************************
 * function    CSM_ITP_vPROXYTriggerCommunicationInd
 * \doxydocu
 * \brief      Callback function from PROXY to trigger a communication indication
 *
 * \param[in]  bUserNumber
 *                 Data
 *
 * \return     void
 *
 * \access     CSM task level
 * \reentrant  no, but only call from interrupt
 *
 * \lastreview
 *
 * \history_begin
 *
 * 23.08.13 Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void CSM_LCTR_vPROXYCommunicationInd( BYTE bBus, DWORD u32ProtokollType, BYTE bAction, WORD wApplID)
{
  BYTE bUserNumber;

  bUserNumber = CSM_LCTR_bGetUserNumber(bBus, u32ProtokollType, NULL, wApplID);

  if (bUserNumber < CSM_LCTR_C_MAX_USER)
  {
    csm_lctr_rLocalControlState.abActAction[bUserNumber] = bAction;
    csm_lctr_rLocalControlState.afCommunicationInd[bUserNumber] = TRUE;
    csm_vTriggerCsmUpstreamTask( CSM_C_IND_REASON_CSM_COMMUNICATION_IND);
  }
  else
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
        #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
          #if (CSM_S_ETG_SHORT_OUTPUT == CSM_C_F_OFF)
            ETG_TRACE_FATAL(("CSM_LCTR_vPROXYCommunicationInd - !(bUserNumber < CSM_LCTR_C_MAX_USER) bUserNumber = %02x!", bUserNumber ));
          #else
            ETG_TRACE_FATAL(("%04x%02x", CSM_C_ETG_FKT_CSM_LCTR_VPROXYCOMMUNICATIONIND, bUserNumber));
          #endif
        #endif
      #endif
    #endif
  }
}
#endif // CSM_S_REMOTE_STACK

/*******************************************************************************
 * function    CSM_LCTR_vProcessCommunicationInd
 * \doxydocu
 * \brief      Handle communication indication.
 *
 *             Handle communication indication.
 *             This function is call in upstream-task after trigger with CSM_C_IND_REASON_CSM_COMMUNICATION_IND.
 *
 * \return     void
 *
 * \access     Upstream-Task
 * \reentrant  no, but only call by Upstream-Task
 *
 * \lastreview 25.01.08 CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 28.11.06  CM-DI/ESA3-Zurmhl
 * bugfix for branch "(CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_OFF)"
 *
 * 08.11.06  CM-DI/ESA3-Prhl
 * multi bus support added
 * 01.06.05 CM-DI/ESA3-Prhl
 * all call back parameter now used from configuration struct
 *
 * 22.09.04 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void CSM_LCTR_vProcessCommunicationInd(void)
{
  BYTE bUserNumber;

  for (bUserNumber = 0; bUserNumber < CSM_LCTR_C_MAX_USER; bUserNumber++)
  {
    if (csm_lctr_rLocalControlState.afCommunicationInd[bUserNumber] == TRUE)
    {
      csm_lctr_rLocalControlState.afCommunicationInd[bUserNumber] = FALSE;

      if (csm_lctr_arCallbackFkt[bUserNumber].rCallback.pfvCommunicationInd != NULL)
      {
        #if (CSM_S_TRACE == CSM_C_F_ON)
          #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
            #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
            #if (CSM_S_ETG_SHORT_OUTPUT == CSM_C_F_OFF)
            ETG_TRACE_USR1(("CSM_LCTR_vProcessCommunicationInd - Bus:%X  Protocol:0x%X  Addr:0x%02X%02X  Action:0x%02X  ApplID:0x%04X",
            ETG_CENUM(tCSM_BUS_NUMBER,      (tU8)  csm_arCsmCfg[bUserNumber].bBus),
            ETG_CENUM(tCSM_PROTOCOL_TYPE,   (tU32) csm_arCsmCfg[bUserNumber].dwProtocolType),
            0x00,
            0x00,
            csm_lctr_rLocalControlState.abActAction[bUserNumber],
            csm_arCsmCfg[bUserNumber].wApplId ));
            #else
            ETG_TRACE_ERR(("%04x%02x%08x%02x%02x%02x%04x", CSM_C_ETG_FKT_CSM_LCTR_VPROCESSCOMMUNICATIONIND,
            ETG_CENUM(tCSM_BUS_NUMBER,      (tU8)  csm_arCsmCfg[bUserNumber].bBus),
            ETG_CENUM(tCSM_PROTOCOL_TYPE,   (tU32) csm_arCsmCfg[bUserNumber].dwProtocolType),
            0x00,
            0x00,
            csm_lctr_rLocalControlState.abActAction[bUserNumber],
            csm_arCsmCfg[bUserNumber].wApplId));
            #endif
            #endif
          #else // CSM_S_TRACE_ETG
          {
            // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
            BYTE abData[10];
            abData[0] = csm_arCsmCfg[bUserNumber].bBus;
            abData[1] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(csm_arCsmCfg[bUserNumber].dwProtocolType));
            abData[2] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(csm_arCsmCfg[bUserNumber].dwProtocolType));
            abData[3] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(csm_arCsmCfg[bUserNumber].dwProtocolType));
            abData[4] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(csm_arCsmCfg[bUserNumber].dwProtocolType));

            abData[5] = 0x00;
            abData[6] = 0x00;

            abData[7] = csm_lctr_rLocalControlState.abActAction[bUserNumber];
            abData[8] = CSM_M_GET_HIBYTE(csm_arCsmCfg[bUserNumber].wApplId);
            abData[9] = CSM_M_GET_LOBYTE(csm_arCsmCfg[bUserNumber].wApplId);

            CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_COMMUNICATION_IND, abData, 10);
          }
          #endif // CSM_S_TRACE_ETG
        #endif // CSM_S_TRACE

        #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
          CSM_M_SET_TASK_CONTEXT( &(csm_lctr_arCallbackFkt[ bUserNumber].rTaskInfo),
                                  csm_lctr_arCallbackFkt[ bUserNumber].rCallback.pfvCommunicationInd( csm_lctr_arCallbackFkt[ bUserNumber].pvHandle,
                                                                                                csm_arCsmCfg[ bUserNumber].bBus,
                                                                                                csm_arCsmCfg[ bUserNumber].dwProtocolType,
                                                                                                NULL,
                                                                                                csm_lctr_rLocalControlState.abActAction[ bUserNumber],
                                                                                                csm_arCsmCfg[ bUserNumber].wApplId) );
        #else // CSM_S_USE_CALLBACK_WITH_HANDLE
          csm_lctr_arCallbackFkt[ bUserNumber].rCallback.pfvCommunicationInd( csm_arCsmCfg[ bUserNumber].bBus,
                                                                        csm_arCsmCfg[ bUserNumber].dwProtocolType,
                                                                        NULL, csm_lctr_rLocalControlState.abActAction[ bUserNumber],
                                                                        csm_arCsmCfg[ bUserNumber].wApplId);
        #endif // CSM_S_USE_CALLBACK_WITH_HANDLE
      }
    }
  }
}

/*******************************************************************************
 * function    CSM_LCTR_vSendAllIndicationsAgain
 * \doxydocu
 * \brief      Deliver all LCTR communication indications again to the applications.
 *
 *             Set a flag to force indication for communication indications to the
 *             application via the upsteam task.
 *
 * \return     void
 *
 * \access     CAN-Task
 * \reentrant  yes
 *
 * \lastreview
 *
 * \history_begin
 * 22.10.13  Prhl
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#if (CSM_S_CCS_USED == CSM_C_F_ON)
void CSM_LCTR_vSendAllIndicationsAgain(void)
{
  WORD wCount;
  for( wCount = 0; wCount < CSM_LCTR_C_MAX_USER; wCount++)
  {
    csm_lctr_rLocalControlState.afCommunicationInd[ wCount] = TRUE;
  }
  csm_vTriggerCsmUpstreamTask( CSM_C_IND_REASON_CSM_COMMUNICATION_IND);
}
#endif // CSM_S_CCS_USED

#undef CSM_LCTR_SELF
#undef CSM_CAN_STACK
/*******************************************************************************
 * \doxydocu
 * \file          csm_lctr.c
 * \brief         Local control part of the CAN Stack Manager
 *
 * \see           CAN-SW Architektur, written by CM-AI/PJ-VW32
 *
 * \sw_component  CAN stack
 * \project       Pool module
 * \path          /di_can/modules_swa/csm/src/
 *
 * \ingroup       CSM
 * \ingroup       CSM_LCTR
 *
 * \authors       Prhl
 *
 * COPYRIGHT      (c) 2009 Blaupunkt GmbH
 *
 * \history_begin
 *
 * 03.04.09  main\1   Prhl
 * Initial revision.
 *
 * 17.07.09  main\2 Prhl
 * - CSM_vLocalCtrlProcessCommunicationCon(), CSM_vLocalCtrlProcessCommunicationInd() and CSM_vLocalCtrlProcessCommunicationReq() added.
 *
 * 23.07.09  main\3 VTeam-Jablonski
 * - Init of csm_arBusErrorIndCallbackFkt moved to CSM_lnit
 *
 * 23.07.09  main\4 VTeam-Jablonski
 * - (csm_lActStaticError[ bBus] != CSM_C_NO_ERROR)) removed
 *
 * 17.08.09  main\5 VTeam-Franke
 * - new: CSM_LCTR_lCommunicationReq(CSM_C_DISCONNECT_ALL)
 * - CSM_S_CSM_LCTR_CTRL_USER_FUNCTION -> CSM_S_CSM_LOCAL_CTRL_USER_FUNCTION
 *
 * 09.09.09  main\6 Prhl
 * - CSM_C_DISCONNECT_ALL removed from application task context. Please add this function to a "pvProcessNewAction" function.
 *   This functionality has to be called on downstream task level.
 *
 * 10.09.09  main\7 VTeam-Franke
 * - CSM_LCTR_lCommunicationReq(CSM_C_DISCONNECT_ALL)
 *
 * 22.09.09  main\8 VTeam-Franke
 * - new feature CSM_S_DEACTIVATE_COMMUNICATION_REQUESTS
 *
 * 23.09.09  main\9 VTeam-Franke
 * - CSM_LCTR_vEndCommunicationRequests removed
 *
 * 01.10.09  main\10 VTeam-Jablonski
 * - call of CNP_LCTR_lDataReq added
 * - please update also /cnp/src/cnp_lctr.c/main/1
 * - please update also /cnp/src/cnp_lctr_i.h/main/1
 * - please update also /cnp/src/main/21
 * - please update the src.sml on your project specific branch
 *
 * 14.10.09  main\12 Prhl
 * - missing feature switch added
 *
 * 20.10.09  main\13 Prhl
 * - missing include added
 *
 * 11.12.09  main\14 Prhl
 * - only review done
 *
 * 21.05.10  main\15 VTeam-Franke
 * - csm_fIgnoreCommReq definition moved here
 *
 * 21.01.11  main\16 VTeam-Khler
 * - Lint warning commented out.
 *
 * 25.01.11  main\17 VTeam-Khler
 * - Struct tCSM_LOCAL_CTRL_APPL_CALLBACK_ADMIN added to separate callback
 *   pointer from application and internal administration data.
 *   csm_stack_i.h main/ with reduced tCSM_LOCAL_CTRL_APPL_CALLBACK should be
 *   used, too
 * - Not used includes commented out.
 *
 * 18.02.11  \main\18  VTeam-Khler
 * - Lint deactivation authorization strings added.
 *
 * 13.07.11  \main\19  Borck
 * - ETG tracing introduced.
 *
 * 29.09.11 \main\20 CM/PJ-H- Basavaraj patil
 * - Compiler switch CSM_S_CNP_LOCAL_CTRL used instead of CSM_S_CSM_LOCAL_CTRL at relevant places
 *
 * 08.01.13 \main\21 Borck
 * - Previous changes reverted back (non-referenced compiler switch removed)
 *   (refer to comment at line 625 ff.)
 *
 * 09.07.13 \main\22 Prhl
 * - missing feature switch added.
 *
 * 12.09.13 \main\23 Feldhaus
 * - integrated CSM_ITP_vPROXYTriggerCommunicationCon
 *
 * 26.09.13 \main\24 Prhl
 * - CSM_LCTR_vPROXYCommunicationCon() added.
 * - CSM_LCTR_vPROXYCommunicationInd() modified.
 *
 * 15.10.13 \main\25 Prhl
 * - CSM_LCTR_vPROXYCommunicationCon() modified.
 * - CSM_LCTR_vPROXYCommunicationInd() modified.
 *
 * 22.10.13 \main\26 Prhl
 * - CSM_LCTR_vSendAllIncicationsAgain() added.
 *
 * 24.10.13 \main\27 Pistoor
 * - Renamed CSM_LCTR_vSendAllIncicationsAgain() to CSM_LCTR_vSendAllIndicationsAgain().
 *
 * 04.12.13  main\28 Prhl
 * - only small lint fixes.
 *
 * 25.02.14  \main\29  Pistoor
 * - Added short output for DLT
 *
 * 27.02.14  \main\30  Pistoor
 * - Optimized short output for DLT
 *
 * 22.02.14  \main\31  Pistoor
 * - usage of pvHandle only in case (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
 *
 * 06.03.15  \main\32  Borck
 * - enabled usage of csm_lctr_m.h with sub-including csm_lctr_mp.h !!!
 *   -> definition of array csm_arCsmCfg[] has been moved from csm_stack_mp.h to csm_lctr_mp.h
 *   -> in order to reduce the size and complexity of csm_stack_mp.h (more auto-generated content intended for the future)
 *
 * 13.03.15  \main\33  Borck
 * - trace without ETG for Autosar environment.
 *
 * 21.04.15  \main\34  Borck
 * - added function CSM_LCTR_bGetUserAssignedBus() to fix an access issue in csm_proxy.c
 * - some function signatures put into 1 single line for readability reasons
 *
 * 29.05.15  \main\35 Borck
 * - made this module compileable for the case that tracing is set to OFF !!!
 *
 * 23.07.15  \main\36 Borck
 * - local function made public, prototye moved to header, function renamed
 *
 * 01.11.16  \main\37 Borck
 * - added ETG tracing for sections that are used by the project (so that old trace output is disabled)
 *
 * 09.11.16  \main\38 Borck
 * - trace buffer sizes have been checked against CSM_MAX_TRACE_LENGTH due to reduction of the same in csm_stack_m.h
 *
 * \history_end
 *//**** END OF FILE **********************************************************/
