/* Description and history of csm_cbr.c at the end of the file.               */
/** File name for preprozessor commands */
#define CSM_CBR_SELF
/** csm_cbr.c is a file according V-team CAN-SW architecture implementation rules */
#define CSM_CAN_STACK

/*******************************************************************************
 * includes:
 *  1)system- and project- includes
 *  2)needed interfaces from external components
 *...3)internal and external interfaces from this component
 ******************************************************************************/
#include "csm_stack_i.h"   /* CSM_C_INIT_..., */
#include "csm_i.h"
#include "csm_stack_m.h"   /* for CSM_HW_LIST, CSM_vMemCopy(), CSM_vMicroSecDelay() etc */
#include "csm_m.h"

#include "csm_cbr_m.h"
#include "csm_cbr_c.h"

#if ( CSM_S_CSM_VW == CSM_C_F_ON )
  #include "csm_vw_i.h"
#endif
#if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
  #if (CSM_S_GMLAN_NM == CSM_C_F_ON)
    #include "cnm_gmlan_i.h"
    #include "csm_vecgmlan_i.h"
   #endif /* CSM_S_GMLAN_NM */
  #if (CSM_S_OSEK_NM == CSM_C_F_ON)
    #include "cnm_osek_i.h"
  #endif /* CSM_S_OSEK_NM */
  #if (CSM_S_VWHIGH_NM == CSM_C_F_ON)
    #include "cnm_vwhigh_i.h"
  #endif /* CSM_S_VWHIGH_NM */
#endif /* CSM_S_CNM_AVAILABLE */

/*************** ETG Tracing ******/
#if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #define ETRACE_S_IMPORT_INTERFACE_GENERIC
  #include "etrace_if.h"

  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
    #include "csm_etg_trace.h"
    #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FD_CSM
    /* #define ETG_INLINE */
    #include "trcGenProj/Header/csm_cbr.c.trc.h"
  #endif
  #endif
#endif /* (CSM_S_TRACE == CSM_C_F_ON) */


#if (CSM_CBR_MAX_USER > 32)
  #error "maximum CSM_CBR_MAX_USER = 32 permitted ! "
#endif

/*******************************************************************************
 * defines and macros (scope: file local)
 ******************************************************************************/
/* see csm_cbr_m(p).h, too */
#if (CSM_S_FNL == CSM_C_F_ON)
  #define CSM_CBR_C_FNL_WAIT_TIME   (1000*60*10)
  #define CSM_CBR_C_FNL_ALIVE_TIME  (6500) /*6,5s . Mit Ingo besprochen */
  /*TESTONLY #define CSM_CBR_C_FNL_WAIT_TIME   (1000*60)*/
  /*TESTONLY #define CSM_CBR_C_FNL_ALIVE_TIME  (6500)*/
#endif

/*******************************************************************************
 * typedefs (scope: file local)
 ******************************************************************************/
/** data structure for CBR state */
typedef struct
{
  #if (CSM_S_FNL == CSM_C_F_ON)
  BYTE abCsmActState[ CSM_C_MAX_CAN_BUS];         /**< needed for VW Funktionsnachlauf */
  #endif

  BYTE abBusState[ CSM_C_MAX_CAN_BUS];            /**< Last by CBR requested bus state. */

  BYTE abReqAction[ CSM_CBR_MAX_USER];            /**< Array to store the bAction parameter form CBR_lCommunicationReq() from every user.
                                                   *   Possible values: CSM_C_STACK_DOWN, CSM_C_STACK_UP */
  BYTE abActAction[ CSM_CBR_MAX_USER];            /**< Array to store the bAction parameter form CBR_lCommunicationReq() from every user.
                                                   *   Possible values: CSM_C_STACK_DOWN, CSM_C_STACK_UP
                                                   *   Nur bei einem GMLAN CAN Bus kann der Wert ActAction (aktiviertes VN)
                                                   *   unterschiedlich zu abReqAction (angefordertes VN) sein!*/
  BOOLEAN afCommunicationReq[ CSM_CBR_MAX_USER];  /**< Trigger array Flag for Communication request, if TRUE communication request is pending.
                                                   *   Possible value TRUE/FALSE */
  BOOLEAN afCommunicationCon[ CSM_CBR_MAX_USER];  /**< Trigger array Flag for Communication confirmation, if TRUE communication confirmation is pending.
                                                   * Possible value TRUE/FALSE */
  BOOLEAN afCommunicationInd[ CSM_CBR_MAX_USER];  /**< Trigger array Flag for Communication indication, if TRUE communication indication is pending.
                                                   * Possible value TRUE/FALSE */
}tCBR_STATE ;

/*******************************************************************************
 * variable definition (scope: file local)
 ******************************************************************************/
/* CSM_STATIC */ tCBR_STATE CBR_rState;                                            /**< This variable stores the CBR state */

#if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
CSM_STATIC tCSM_BR_SIGNAL_APPL_CALLBACK cbr_arCallbackFkt[ CSM_CBR_MAX_USER];      /**< This area stores the callbach function for each user */
#endif

/*******************************************************************************
 * variable definition (scope: CAN stack)
 ******************************************************************************/
#if (CSM_S_FNL == CSM_C_F_ON)
static BOOL csm_cbr_afFnlWaitTimerExpired[ CSM_C_MAX_CAN_BUS] = {FALSE};
static BOOL csm_cbr_afFnlAliveTimerExpired[ CSM_C_MAX_CAN_BUS] = {FALSE};
BOOL CSM_CBR_afFnlWake[ CSM_C_MAX_CAN_BUS] = {FALSE};
#endif /* (CSM_S_BUS_WAKE_TEMP == CSM_C_F_ON) */
/* --none-- or in csm_cbr_m(p).h */

/*******************************************************************************
 * function prototypes (scope: file local)
 ******************************************************************************/
/* --none-- or in csm_cbr_m(p).h */


/* Feature switch is used only to control doxygen documentation */
#if (CSM_S_DOXYGEN_HIDE_NON_API_FCT == CSM_C_F_OFF)
/*******************************************************************************
 * function implementation (scope: file local = static)
 ******************************************************************************/
/*******************************************************************************
 * function    csm_cbr_vApplCallbackDelAll
 * \doxydocu
 * \brief      Removes all registered rx callback functions
 *
 * \param[in]  void
 * \return     void
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \lastreview 17.02.09 VTeam-Zurmhl
 *
 * \history_begin
 * 11.06.08 VTeam-Brunotte
 * Initial Revision.
 * \history_end
 ******************************************************************************/
CSM_STATIC void csm_cbr_vApplCallbackDelAll(void)
{
  WORD wCountUser;
  #if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
    for( wCountUser = 0; wCountUser < CSM_CBR_MAX_USER; wCountUser++)
    {/* CBR Status zurck setzen */
      cbr_arCallbackFkt[ wCountUser].u16ApplID = csm_arCbrCfg[ wCountUser].wApplId;
      #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
        cbr_arCallbackFkt[ wCountUser].pvHandle = NULL;
      #endif /* CSM_S_USE_CALLBACK_WITH_HANDLE */
      cbr_arCallbackFkt[ wCountUser].pfvCommunicationCon = NULL;
      cbr_arCallbackFkt[ wCountUser].pfvCommunicationInd = NULL;
    }
  #endif
}

/*******************************************************************************
 * function    csm_cbr_bGetVNBit
 * \doxydocu
 * \brief      Get virtual network in bitfield representation of given VN number.
 *
 * \param[in]  bRemoteComp
 *             component number/ number of virtual network
 *
 * \return     Bitfield representation for given VN number.
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \lastreview 12.02.08 VTeam-Brunotte
 *
 * \history_begin
 * 22.10.07 VTeam D. Brunotte
 *  - Remote component "Intermediate" and "Remote Diagnostic" added.
 * 14.06.06 CM-DI/ESU3-Prhl
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#if (CSM_S_VIRTUAL_NETWORK == CSM_C_F_ON)
#if (CSM_S_GMLAN_NM == CSM_C_F_ON)
BYTE csm_cbr_bGetVNBit(BYTE bRemoteComp)
{
  BYTE bRemotedVN = 0;

  /* VK 7.10.2009: evtl auf CNM_GMLAN_arVnConfigMs[] umstellen ?              */
  switch( bRemoteComp)
  {
    case CSM_C_ADDRESS_FIELD_BR_SIGNAL_DIAGNOSTIC:
    {
      bRemotedVN = CNM_GMLAN_C_VN_DIAGNOSTIC;
    }
    break;
    case CSM_C_ADDRESS_FIELD_BR_SIGNAL_ALLNODES:
    {
      bRemotedVN = CNM_GMLAN_C_VN_ALLNODES;
    }
    break;
    case CSM_C_ADDRESS_FIELD_BR_SIGNAL_INFOTAINMENT:
    {
      bRemotedVN = CNM_GMLAN_C_VN_INFOTAINMENT;
    }
    break;
    case CSM_C_ADDRESS_FIELD_BR_SIGNAL_PARKEDVEHICLE:
    {
      bRemotedVN = CNM_GMLAN_C_VN_PARKEDVEHICLE;
    }
    break;
    case CSM_C_ADDRESS_FIELD_BR_SIGNAL_PRIMARYTRIGGERING:
    {
      bRemotedVN = CNM_GMLAN_C_VN_PRIMARYTRIGGERING;
    }
    break;
    case CSM_C_ADDRESS_FIELD_BR_SIGNAL_INTERMEDIATE:
    {
      bRemotedVN = CNM_GMLAN_C_VN_INTERMEDIATE;
    }
    break;
    case CSM_C_ADDRESS_FIELD_BR_SIGNAL_REMOTEDIACNOSTICS:
    {
      bRemotedVN = CNM_GMLAN_C_VN_REMOTEDIAGNOSTICS;
    }
    break;
    default:
    {
      /* empty */
    }
    break;
  }
  return bRemotedVN;
}
#endif /* CSM_S_GMLAN_NM */
#endif /* CSM_S_VIRTUAL_NETWORK */
/*******************************************************************************
 * function    CSM_CBR_bQueryRequestedVNs
 * \doxydocu
 * \brief      Calculate and return the bitfield representation of all requested
 *             virtual networks for this bus.
 *
 * \param[in]  bBus
 *             bus number
 *
 * \return     BYTE
 *              Bit field with all requested virtual networks.
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \lastreview 12.02.08 VTeam-Brunotte
 *
 * \history_begin
 * - csm_cbr_bCalculateActiveVN() renamed CSM_CBR_bQueryRequestedVNs()
 * 14.06.06 CM-DI/ESU3-Prhl
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#if (CSM_S_VIRTUAL_NETWORK == CSM_C_F_ON)
//#if (CSM_S_GMLAN_NM == CSM_C_F_ON)
BYTE CSM_CBR_bQueryRequestedVNs(BYTE bBus)
{
  BYTE bRequestedVN = 0;
  WORD wCount;

  for( wCount = 0; wCount < CSM_CBR_MAX_USER; wCount++)
  {
    if( bBus == csm_arCbrCfg[ wCount].bBus)
    {/* nur alle im gleichen Bus berprfen */
      if( CBR_rState.abReqAction[ wCount] != CSM_C_STACK_DOWN)
      { // ist Aktiv, jetzt VN Bit raussuchen
        bRequestedVN |= csm_cbr_bGetVNBit( csm_arCbrCfg[ wCount].au8AddressField[ 1]);
      }
    }
  }

  return( bRequestedVN);
} /* end CSM_CBR_bQueryRequestedVNs() */
//#endif /* CSM_S_GMLAN_NM */
#endif /* CSM_S_VIRTUAL_NETWORK */

/*******************************************************************************
 * function    csm_cbr_bCalculateBusState
 * \doxydocu
 * \brief      get the bus state form CBR.
 *
 *             get the bus state form CBR.
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESA3
 *
 * \param[in]  bBus (I)
 *             bus number
 *
 * \return     BYTE
 *              CBR bus state
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \lastreview 18.04.08 VTeam-Prhl
 *
 * \history_begin
 * 21.02.07 VTeam-Prhl
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#if (CSM_S_FNL == CSM_C_F_ON)
BYTE csm_cbr_bCalculateBusState(BYTE bBus)
{
  BYTE bCbrBusState = CSM_C_STACK_DOWN;
  WORD wCount;

  /* loop over all configured CBR user                                        */
  for (wCount = 0; wCount < CSM_CBR_MAX_USER; wCount++)
  {
    /* check if given bus is used by any client                               */
    if (bBus == csm_arCbrCfg[ wCount].bBus)
    {
      /* registered user is on the relevant bus                               */
      if (CBR_rState.abReqAction[ wCount] == CSM_C_STACK_UP)
      {
        /* an UP request is pending so set new state                          */
        bCbrBusState = CSM_C_STACK_UP;
        /* leave loop since more than STACK_UP is not possible                */
        break;
      }

      /* if reason was different from UP, it might be FNL                     */
      if (CBR_rState.abReqAction[ wCount] == CSM_C_FNL_ON)
      {
        /* if so, set the new state                                           */
        bCbrBusState = CSM_C_FNL_ON;
      }
    }
  }

  return bCbrBusState;
}
#endif /* CSM_S_FNL */

/*******************************************************************************
 * function    CSM_CBR_bGetUserNumber
 * \doxydocu
 * \brief      get user number.
 *
 *             get user number.
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESA3
 *
 * \param[in]  bBus (I)
 *             bus number
 *
 * \param[in]  dwProtocolType (I)
 *             protocol type, see table 2 of CAn SW architecture
 *
 * \param[in]  *pvAddressField (->I)
 *             addressing, dependent on protocol
 *
 * \param[in]  wApplId (I)
 *               Appl Id
 *
 * \return     BYTE
 *               user number = from 0 to CSM_CBR_MAX_USER-1. CSM_CBR_MAX_USER is the error value.
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \lastreview 22.03.11 Borck
 *
 * \history_begin
 * 10.10.06 CM-DI/ESA3-Zurmhl
 * pvAddressField changed: PVOID-> void*.
 *
 * 14.06.06 CM-DI/ESU3-Prhl
 * Initial Revision.
 * \history_end
 ******************************************************************************/
BYTE CSM_CBR_bGetUserNumber(BYTE bBus, DWORD dwProtocolType, const void * pvAddressField, WORD wApplId)
{
  BYTE bUserNumber;

  for (bUserNumber = 0; bUserNumber < CSM_CBR_MAX_USER; bUserNumber++)
  {
    if ( ( bBus == csm_arCbrCfg[ bUserNumber].bBus) &&
         ( dwProtocolType == csm_arCbrCfg[ bUserNumber].dwProtocolType) &&
         ( *((((BYTE *)pvAddressField)+0)) == csm_arCbrCfg[ bUserNumber].au8AddressField[ 0]) &&
         ( *((((BYTE *)pvAddressField)+1)) == csm_arCbrCfg[ bUserNumber].au8AddressField[ 1]) &&
         ( wApplId == csm_arCbrCfg[ bUserNumber].wApplId) )
    {
      return bUserNumber;
    }
  }
  return bUserNumber;
}

/*******************************************************************************
 * function implementation (scope: CAN stack and operating system)
 *******************************************************************************/
/*******************************************************************************
 * function    CSM_CBR_lInit
 * \doxydocu
 * \brief      Initializes CSM Broadcast Server module.
 *
 *             Initializes CSM Broadcast Server module.
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESA3
 *
 * \param      bInitMode
 *             Allowed values:
 *                CSM_C_INIT_COLD, CSM_C_INIT_WARM, CSM_C_INIT_START (see csm_stack_i.h)
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR, CBR_E_ERROR_STATE_WHILE_INIT
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \lastreview 22.03.11 Borck
 *             17.02.09 VTeam-Zurmhl .. Finding: Unused "#if (defined CSM_C_IND_REASON_CBR_COMMUNICATION_CON .."
 *             05.12.06 CM-DI/ESA3-Prhl
 *
 * \history_begin
 * 12.06.08 VTeam-Brunotte
 * - function csm_cbr_vApplCallbackDelAll is called in init warm state
 * - CSM_VECGMLAN_lInit is called to synchronize GMLAN CBR.
 * 24.05.06 CM-DI/ESA3-Prhl
 * old variables removed
 *
 * 18.06.04 CM-DI/ESU3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
LONG CSM_CBR_lInit(BYTE bInitMode)
{
  LONG lReturn = CSM_C_NO_ERROR;
  WORD wCountUser;

  #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
    #if (CSM_S_GMLAN_NM == CSM_C_F_ON)
    lReturn = CSM_VECGMLAN_lInit(bInitMode);
    #endif /* (CSM_S_GMLAN_NM == CSM_C_F_ON) */
  #endif /* (CSM_S_CNM_AVAILABLE == CSM_C_F_ON) */

  if(bInitMode == CSM_C_INIT_COLD)
  {
    /* intentionally empty */
  }
  else if (bInitMode == CSM_C_INIT_WARM)
  {
    /*#if (defined CSM_C_IND_REASON_CBR_COMMUNICATION_CON || defined CSM_C_IND_REASON_CBR_NACHLAUF_CON)
      // Set flags for communication requests to defined states.
      VK: alredy done? See
          CBR_rState.afCommunicationCon[ wCountUser] = FALSE;
          few lines below
    #endif*/

    for (wCountUser = 0; wCountUser < CSM_C_MAX_CAN_BUS; wCountUser++)
    {
      /* reset CBR Status                                                     */
      CBR_rState.abBusState[ wCountUser] = CSM_C_STACK_DOWN;
    }
    for (wCountUser = 0; wCountUser < CSM_CBR_MAX_USER; wCountUser++)
    {
      /* reset CBR Status                                                     */
      CBR_rState.abReqAction[ wCountUser] = CSM_C_STACK_DOWN;
      CBR_rState.abActAction[ wCountUser] = CSM_C_STACK_DOWN;
      CBR_rState.afCommunicationCon[ wCountUser] = FALSE;
      CBR_rState.afCommunicationInd[ wCountUser] = FALSE;
    }
    csm_cbr_vApplCallbackDelAll();
  }
  else if (bInitMode == CSM_C_INIT_START)
  {
  }
  else
  {
    return CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_UNKNOWN_STATE);
  }

  return lReturn;
}

/*******************************************************************************
 * function    CSM_CBR_lExit
 * \doxydocu
 * \brief      Stops CSM CAN Broadcast server.
 *
 *             Stops CAN Broadcast server.
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESA3
 *             When the stop state is called, the CBR state is set to DOWN. This state
 *             switch is not indicated to registered applications.
 * \param[in]  bExitMode
 *             Allowed values (see csm_stack_i.h):
 *                - CSM_C_EXIT_STOP
 *                - CSM_C_EXIT_OFF
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \lastreview 22.03.11 Borck
 *             17.02.09 VTeam-Zurmhl
 *             24.07.07 VTeam-Prhl
 *
 * \history_begin
 * 12.06.08 VTeam-Brunotte
 * - csm_cbr_vApplCallbackDelAll is called in off state
 * - CSM_VECGMLAN_lExit is called to synchronize GMLAN CBR
 * 19.07.07 Brunotte
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
LONG CSM_CBR_lExit(const BYTE bExitMode )
{
  LONG lReturn = CSM_C_NO_ERROR;
  WORD wCountUser;

  switch (bExitMode)
  {
    case CSM_C_EXIT_STOP:
    {
      for( wCountUser = 0; wCountUser < CSM_C_MAX_CAN_BUS; wCountUser++)
      {
        /* Reset CBR state */
        CBR_rState.abBusState[ wCountUser] = CSM_C_STACK_DOWN;
      }
      for( wCountUser = 0; wCountUser < CSM_CBR_MAX_USER; wCountUser++)
      {
        /* Reset CBR state */
        CBR_rState.abReqAction[ wCountUser] = CSM_C_STACK_DOWN;
        CBR_rState.abActAction[ wCountUser] = CSM_C_STACK_DOWN;
        CBR_rState.afCommunicationCon[ wCountUser] = FALSE;
        CBR_rState.afCommunicationInd[ wCountUser] = FALSE;
      }
    }
    break;
    case CSM_C_EXIT_OFF:
    {
      csm_cbr_vApplCallbackDelAll();
    }
    break;
    default:
    {
      lReturn = CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_UNKNOWN_STATE);
    }
    break;
  }

  #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
    #if (CSM_S_GMLAN_NM == CSM_C_F_ON)
      lReturn = CSM_VECGMLAN_lExit(bExitMode);
    #endif /* (CSM_S_GMLAN_NM == CSM_C_F_ON) */
  #endif /* (CSM_S_CNM_AVAILABLE == CSM_C_F_ON) */

  return lReturn;
}

/*******************************************************************************
 * function    CSM_CBR_bGetUserAssignedBus
 * \doxydocu
 * \brief      return the configured bus for configured user
 *
 * \param[in]  bUserNumber
 *             requested User
 *
 * \return     Bus number
 *
 * \access     from CSM modules
 * \reentrant  yes
 *
 * \lastreview  21.04.15 Borck
 *
 * \history_begin
 *
 * 21.04.15 Borck
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
BYTE CSM_CBR_bGetUserAssignedBus(BYTE bUserNumber)
{
  return csm_arCbrCfg[bUserNumber].bBus;
}

/*******************************************************************************
 * function    CSM_CBR_lApplCallbackInit
 * \doxydocu
 * \brief      Initial callback function
 *
 *             Initial callback function. Please call this function only once for each channel
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/EHS3
 *
 * \param[in]  pvHandle
 *              pointer to handle. For future use! May be a deinit() function
 *
 * \param[in]  bBus
 *              Bus number
 *
 * \param[in]  dwProtocolType (I)
 *             protocol type, see table 2 of CAn SW architecture
 *
 * \param[in]  *pvAddressField (->I)
 *             addressing, dependent on protocol
 *
 * \param[in]  *prCallBackFkt (->I)
 *             pointer to callback function
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR
 *
 * \access     application
 * \reentrant  yes
 *
 *
 * \lastreview 22.03.11 Borck
 *
 * \history_begin
 * 29.08.07 VTeam-Brunotte
 * A error number is returned, if no valid usernumber is found.
 * 10.10.06 CM-DI/ESA3-Zurmhl
 * pvAddressField changed: PVOID-> void*.
 *
 * 29.08.08 VTeam-Brunotte
 * - macro CSM_S_USE_TKSE check added.
 * 14.07.06 CM-DI/ESA3-Prhl
 * add bBus
 * 14.06.06 CM-DI/ESU3-Prhl
 * Paramter check added (csm_cbr_bGetUserNumber)
 * 01.10.03 CM-DI/ESU3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
#if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
LONG CSM_CBR_lApplCallbackInit(void * pvHandle, BYTE bBus, DWORD dwProtocolType,
                               const void * pvAddressField, tCSM_BR_SIGNAL_APPL_CALLBACK * prCallBackFkt)
{
  LONG lRetVal = CSM_C_NO_ERROR;
  BYTE bUserNumber;
  WORD wApplID = prCallBackFkt->u16ApplID;

  bUserNumber = CSM_CBR_bGetUserNumber( bBus, dwProtocolType, pvAddressField, wApplID);

  if (bUserNumber < CSM_CBR_MAX_USER)
  {
    #if (CSM_S_USE_TKSE == CSM_C_F_ON)
      CSM_M_GET_TASK_CONTEXT( &(cbr_arCallbackFkt[ bUserNumber].rTaskInfo));
    #endif /* (CSM_S_USE_TKSE == CSM_C_F_ON) */
    #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
      cbr_arCallbackFkt[ bUserNumber].pvHandle = pvHandle;
    #endif /* CSM_S_USE_CALLBACK_WITH_HANDLE */
    CSM_vAssert( cbr_arCallbackFkt[ bUserNumber].pfvCommunicationCon == NULL);
    CSM_vAssert( cbr_arCallbackFkt[ bUserNumber].pfvCommunicationInd == NULL);
    cbr_arCallbackFkt[ bUserNumber].pfvCommunicationCon = prCallBackFkt->pfvCommunicationCon;
    cbr_arCallbackFkt[ bUserNumber].pfvCommunicationInd = prCallBackFkt->pfvCommunicationInd;
  }
  else
  {
    lRetVal = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_INVALID_PARA);
    CSM_vAssert( FALSE);
  }
  return( lRetVal);
}/*lint !e818 PQM_authorized_multi_123. Reason: Interface not defined as const */
#endif /* CSM_S_CALLBACK_INIT_BY_APPLICATION */

/*******************************************************************************
 * function    CSM_CBR_lCommunicationReq
 * \doxydocu
 * \brief      Requests the CAN bus for broadcast communication.
 *
 *             Requests the CAN bus for broadcast communication.
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/EHS3
 *
 * \param[in]  bBus (I)
 *             bus number
 *
 * \param[in]  dwProtocolType (I)
 *             protocol type, see table 2 of CAn SW architecture
 *
 * \param[in]  *pvAddressField (->I)
 *             addressing, dependent on protocol
 *
 * \param[in]  bAction (I)
 *             Allowed values: CSM_C_STACK_UP, CSM_C_STACK_DOWN
 *
 * \param[in]  wApplID (I)
 *               Appl Id
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR
 *
 * \access     application
 * \reentrant  no, but the you can call it with diffrend wApplID without a problem
 *
 * \lastreview 22.03.11 Borck
 *
 * \history_begin
 * 28.03.08 VTeam-Brunotte
 * - Check if a callback function is registered before the request
 *
 * 10.10.06 CM-DI/ESA3-Zurmhl
 * pvAddressField changed: PVOID-> void*.
 *
 * 14.06.06 CM-DI/ESU3-Prhl
 * Paramter check added (csm_cbr_bGetUserNumber)
 * 24.05.06 CM-DI/ESA3-Prhl
 * trigger CSM_C_DOWN_REASON_CSM_BUS_STATE_CHANGED direct
 *
 * 26.07.04 CM-DI/ESU-3 Prhl
 * Initial Revision.
 *
 * \history_end
 * \todo bBus is not used. CBR only works with one bus
 ******************************************************************************/
LONG CSM_CBR_lCommunicationReq(BYTE bBus, DWORD dwProtocolType, const void * pvAddressField,
                               BYTE bAction, WORD wApplID)
{
  LONG  lReturnValue = CSM_C_NO_ERROR;
  BYTE bUserNumber;

  bUserNumber = CSM_CBR_bGetUserNumber(bBus, dwProtocolType, pvAddressField, wApplID);

  if (bUserNumber < CSM_CBR_MAX_USER)
  {
    /* Check if a callback function is registered */
    if ( (cbr_arCallbackFkt[ bUserNumber].pfvCommunicationCon != NULL) ||
         (cbr_arCallbackFkt[ bUserNumber].pfvCommunicationInd != NULL) )
    {
      if (CBR_rState.abReqAction[ bUserNumber] != bAction)
      {
        /* change detected                                                    */
        CBR_rState.abReqAction[ bUserNumber] = bAction;
        CBR_rState.afCommunicationReq[ bUserNumber] = TRUE;
        csm_vTriggerCsmDownstreamTask( CSM_C_DOWN_REASON_CBR_COMMUNICATION_REQ);

        csm_vTriggerCsmDownstreamTask( CSM_C_DOWN_REASON_CSM_BUS_STATE_CHANGED);
      }
      else
      {
        /* trigger communication confirmation directly                        */
        CBR_rState.afCommunicationCon[ bUserNumber] = TRUE;
        csm_vTriggerCsmUpstreamTask( CSM_C_IND_REASON_CBR_COMMUNICATION_CON);
      }
    }
    else
    {
      /* error: no callbacks available                                        */
      lReturnValue = CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_CALLBACK_MISSING);
    }
  }
  else
  {
    /* error: invalid user number                                             */
    lReturnValue = CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_E_INVALID_PARA);
  }

  return lReturnValue;
} /* end CSM_lCommunicationReq() */

/*******************************************************************************
 * function    CSM_CBR_fCheckCommunicationReq
 * \doxydocu
 * \brief      Calculate Communication request.
 *
 *             Calculate Communication request.
 *             This function is called to calculate the CommunicationReq state.
 *
 * \param[in]  bBus
 *
 * \return     BOOL
 *               FALSE: the bus is not needed by CBR
 *               TRUE: the bus is neened by CBR
 *
 * \access     CAN-Task
 * \reentrant  yes, but only called by CAN-Task
 *
 * \lastreview 22.03.11 Borck
 *
 * \history_begin
 * 14.06.06 CM-DI/ESU3-Prhl
 * bBus is used now
 * 22.09.04 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
BOOL CSM_CBR_fCheckCommunicationReq(BYTE bBus)
{
  WORD wCount;

  /* loop over all possible clients                                           */
  for (wCount = 0; wCount < CSM_CBR_MAX_USER; wCount++)
  {
    /* if given bus matches a registered client for this bus                  */
    if (bBus == csm_arCbrCfg[ wCount].bBus)
    {
      /* client found for this bus                                            */
      #if (CSM_S_FNL == CSM_C_F_ON)
      if (CBR_rState.abReqAction[ wCount] == CSM_C_STACK_UP)
      {
        /* CBR needs the bus for at least one of its FNL clients              */
        return TRUE;
      }
      #else /* CSM_S_FNL */
      if (CBR_rState.abReqAction[ wCount] != CSM_C_STACK_DOWN)
      {
        /* CBR needs the bus for at least one of its clients                  */
        return TRUE;
      }
      #endif /* CSM_S_FNL */
    }
  }

  /* CBR does not need the bus                                                */
  return FALSE;
}

/*******************************************************************************
 * function    CSM_CBR_vProcessCommunicationReq
 * \doxydocu
 * \brief      Handle communication request.
 *
 *             Handle communication request.
 *             This function is called in downstream-task after trigger with CSM_C_DOWN_REASON_CBR_COMMUNICATION_REQ.
 *
 * \return     VOID
 *
 * \access     CAN-Task
 * \reentrant  no, but only called by CAN-Task
 *
 * \lastreview 22.03.11 Borck
 *
 * \history_begin
 * 14.06.06 CM-DI/ESA3-Prhl
 * GMLAN needs function call on CAN task level
 *
 * 24.05.06 CM-DI/ESA3-Prhl
 * function is now empty
 *
 * 22.09.04 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
VOID CSM_CBR_vProcessCommunicationReq(VOID)
{
  WORD wCount;

  /**************************************************/
  /* dedicated part if Funktionsnachlauf is ACTIVE  */
  /**************************************************/
  #if (CSM_S_FNL == CSM_C_F_ON)
  {
    BYTE bBus;
    for (bBus = 0; bBus < CSM_C_MAX_CAN_BUS; bBus++)
    {
      if (csm_cbr_afFnlWaitTimerExpired[ bBus] == TRUE)
      {
        csm_cbr_afFnlWaitTimerExpired[ bBus] = FALSE;
        /* Bus einschalten, weil die FNL-WAIT Zeit abgelaufen ist */
        #if ( CSM_S_OSEK_NM == CSM_C_F_ON )
          #if ( CNM_OSEK_S_VW_PAYLOAD == CSM_C_F_ON )
            CNM_OSEK_lSetPayload( bBus, CNM_OSEK_C_NMINFO_WAKEUP_FNL);
            {
              BYTE bData = 1;
              CSM_lSignalWrite( CSM_C_SIG_TX_RA4_Radio_Wake_Up_Komfort_mRadio_4,
                                &bData, sizeof(bData),
                                CSM_C_TX_IMMEDIATELY);
            }
          #endif /* CNM_OSEK_S_VW_PAYLOAD == CSM_C_F_ON */
        #endif /* CSM_S_OSEK_NM == CSM_C_F_ON */
        CSM_CBR_afFnlWake[ bBus] = TRUE;
        CSM_lTimerStart( (WORD ) (CSM_TIMER_CBR_FNL_ALIVE_TIMER_0 + bBus), CSM_CBR_C_FNL_ALIVE_TIME);
        csm_vTriggerCsmDownstreamTask( CSM_C_DOWN_REASON_CSM_BUS_STATE_CHANGED);
      }
      if (csm_cbr_afFnlAliveTimerExpired[ bBus] == TRUE)
      {
        csm_cbr_afFnlAliveTimerExpired[ bBus] = FALSE;

        /* jetzt Bus freigeben, da die FNL active Zeit abgelaufen ist. Der FNL-WAIT Timer wird erst beim Busstate change auf DOWN aufgezogen */
        #if ( CSM_S_OSEK_NM == CSM_C_F_ON )
          #if ( CNM_OSEK_S_VW_PAYLOAD == CSM_C_F_ON )
          {
            BYTE bData = 0;
            CSM_lSignalWrite( CSM_C_SIG_TX_RA4_Radio_Wake_Up_Komfort_mRadio_4,
                              &bData, sizeof(bData),
                              CSM_C_TX_IMMEDIATELY);
          }
          #endif /* CNM_OSEK_S_VW_PAYLOAD == CSM_C_F_ON */
        #endif /* CSM_S_OSEK_NM == CSM_C_F_ON */
        CSM_CBR_afFnlWake[ bBus] = FALSE;
        csm_vTriggerCsmDownstreamTask( CSM_C_DOWN_REASON_CSM_BUS_STATE_CHANGED);
      }
    } /* end for bBus*/
  }
  #endif /* CSM_S_FNL */

  /**************************************************/
  /* generic part continues here                    */
  /**************************************************/
  for (wCount = 0; wCount < CSM_CBR_MAX_USER; wCount++)
  {
    if (CBR_rState.afCommunicationReq[ wCount] == TRUE)
    {
      CBR_rState.afCommunicationReq[ wCount] = FALSE;

      #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
      #if ( CSM_S_OSEK_NM == CSM_C_F_ON )
        #if ( CNM_OSEK_S_PAYLOAD == CSM_C_F_ON )
          if( CBR_rState.abReqAction[ wCount] == CSM_C_STACK_DOWN)
          {
            CNM_OSEK_lClrPayload( csm_arCbrCfg[ wCount].bBus, csm_arCbrCfg[ wCount].dwNmPayload);
          }
          else
          {
            CNM_OSEK_lSetPayload( csm_arCbrCfg[ wCount].bBus, csm_arCbrCfg[ wCount].dwNmPayload);
          }
        #endif /* CNM_OSEK_S_PAYLOAD == CSM_C_F_ON */
      #endif /* CSM_S_OSEK_NM == CSM_C_F_ON */
      #if ( CSM_S_VWHIGH_NM == CSM_C_F_ON )
        if( CBR_rState.abReqAction[ wCount] == CSM_C_STACK_DOWN)
        {
          CNM_VWHIGH_vClrPayload( csm_arCbrCfg[ wCount].bBus, csm_arCbrCfg[ wCount].dwNmPayload);
        }
        else
        {
          CNM_VWHIGH_vSetPayload( csm_arCbrCfg[ wCount].bBus, csm_arCbrCfg[ wCount].dwNmPayload);
        }
      #endif /* CSM_S_VWHIGH_NM == CSM_C_F_ON */
      #endif /* CSM_S_CNM_AVAILABLE == CSM_C_F_ON */

      /* hier knnten Funktionen stehen die auf CAN-Task level nach einem CommunicationReq gemacht werden sollen */
      #if (CSM_S_CNM_AVAILABLE == CSM_C_F_ON)
      #if (CSM_S_GMLAN_NM == CSM_C_F_ON)
        #if (CSM_S_VIRTUAL_NETWORK == CSM_C_F_ON)
        {
          BYTE bRequestedVNs;

          /* Query all requested VNs. => again a for (CSM_CBR_MAX_USER) loop */
          bRequestedVNs = CSM_CBR_bQueryRequestedVNs( csm_arCbrCfg[ wCount].bBus);
          CNM_GMLAN_lSetVN( csm_arCbrCfg[ wCount].bBus, bRequestedVNs);
        }
      #else  /* CSM_S_VIRTUAL_NETWORK */
        {
          if( CBR_rState.abReqAction[ wCount] == CSM_C_STACK_DOWN)
          {
            CNM_GMLAN_lSetVN(csm_arCbrCfg[ wCount].bBus, 0);
          }
          else
          {
            CNM_GMLAN_lSetVN(csm_arCbrCfg[ wCount].bBus, 0xff);
          }
        }
      #endif /* CSM_S_VIRTUAL_NETWORK */
      #endif /* CSM_S_GMLAN_NM */
      #endif /* CSM_S_CNM_AVAILABLE == CSM_C_F_ON */

      #if (CSM_S_FNL == CSM_C_F_ON)
      {
        BYTE bNewState;
        bNewState = csm_cbr_bCalculateBusState( csm_arCbrCfg[ wCount].bBus);
        if( CBR_rState.abBusState[ csm_arCbrCfg[ wCount].bBus] != bNewState)
        {
          if( (bNewState == CSM_C_FNL_ON) &&
              (CBR_rState.abBusState[ csm_arCbrCfg[ wCount].bBus] == CSM_C_STACK_DOWN))
          {/* Wir waren aus und jetzt ist FNL aktiv */
            CSM_CBR_afFnlWake[ csm_arCbrCfg[ wCount].bBus] = TRUE;
            CSM_lTimerStart( (WORD ) (CSM_TIMER_CBR_FNL_ALIVE_TIMER_0 + csm_arCbrCfg[ wCount].bBus), CSM_CBR_C_FNL_ALIVE_TIME);
            CSM_lTimerStop( (WORD ) (CSM_TIMER_CBR_FNL_WAIT_TIMER_0 + csm_arCbrCfg[ wCount].bBus));
            csm_vTriggerCsmDownstreamTask( CSM_C_DOWN_REASON_CSM_BUS_STATE_CHANGED);
          }
          if( CBR_rState.abBusState[ csm_arCbrCfg[ wCount].bBus] == CSM_C_FNL_ON)
          {/* der alte Zustand war FNL, der ist jetzt aber aus */
            #if ( CSM_S_OSEK_NM == CSM_C_F_ON )
            #if ( CNM_OSEK_S_VW_PAYLOAD == CSM_C_F_ON )
            {
              BYTE bData = 0;
              CSM_lSignalWrite( CSM_C_SIG_TX_RA4_Radio_Wake_Up_Komfort_mRadio_4,
                                &bData, sizeof(bData),
                                CSM_C_TX_IMMEDIATELY);
            }
            #endif /* CNM_OSEK_S_VW_PAYLOAD == CSM_C_F_ON */
            #endif /* CSM_S_OSEK_NM == CSM_C_F_ON */
            CSM_CBR_afFnlWake[ csm_arCbrCfg[ wCount].bBus] = FALSE;
            CSM_lTimerStop( (WORD ) (CSM_TIMER_CBR_FNL_ALIVE_TIMER_0 + csm_arCbrCfg[ wCount].bBus));
            CSM_lTimerStop( (WORD ) (CSM_TIMER_CBR_FNL_WAIT_TIMER_0 + csm_arCbrCfg[ wCount].bBus));
          }
          CBR_rState.abBusState[ csm_arCbrCfg[ wCount].bBus] = bNewState;
        }
      }
      #endif /* CSM_S_FNL*/

      /* trigger communication confirmation                                   */
      CBR_rState.afCommunicationCon[ wCount] = TRUE;
      csm_vTriggerCsmUpstreamTask( CSM_C_IND_REASON_CBR_COMMUNICATION_CON);
    }
  }
} /* end CSM_CBR_vProcessCommunicationReq() */

#if (CSM_S_REMOTE_STACK == CSM_C_F_ON)
/*******************************************************************************
 * function    CSM_CBR_vPROXYCommunicationCon
 * \doxydocu
 * \brief      Callback function from PROXY to trigger a communication confirmation
 *
 * \param[in]  wApplId
 *             bAction
 *
 * \return     void
 *
 * \access     CSM task level
 * \reentrant  no, but only call from interrupt
 *
 * \lastreview
 *
 * \history_begin
 *
 * 23.08.13 Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void CSM_CBR_vPROXYCommunicationCon(BYTE bUserNumber, BYTE bAction)
{
  CBR_rState.abActAction[ bUserNumber] = bAction;
  CBR_rState.afCommunicationCon[ bUserNumber] = TRUE;
  csm_vTriggerCsmUpstreamTask( CSM_C_IND_REASON_CBR_COMMUNICATION_CON);
}

/*******************************************************************************
 * function    CSM_CBR_vPROXYCommunicationInd
 * \doxydocu
 * \brief      Callback function from PROXY to trigger a communication confirmation
 *
 * \param[in]  wApplId
 *             bAction
 *
 * \return     void
 *
 * \access     CSM task level
 * \reentrant  no, but only call from interrupt
 *
 * \lastreview
 *
 * \history_begin
 *
 * 18.10.13 Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void CSM_CBR_vPROXYCommunicationInd(BYTE bUserNumber, BYTE bAction)
{
  CBR_rState.abActAction[ bUserNumber] = bAction;
  CBR_rState.afCommunicationInd[ bUserNumber] = TRUE;
  csm_vTriggerCsmUpstreamTask( CSM_C_IND_REASON_CBR_COMMUNICATION_IND);
}
#endif /* CSM_S_REMOTE_STACK == CSM_C_F_ON */

/*******************************************************************************
 * function    CSM_CBR_vProcessCommunicationCon
 * \doxydocu
 * \brief      Handle communication confimation.
 *
 *             Handle communication confimation.
 *             This function is called in upstream-task after trigger with CSM_C_IND_REASON_CBR_COMMUNICATION_CON.
 *
 * \return     VOID
 *
 * \access     Upstream-Task
 * \reentrant  no, but only called by Upstream-Task
 *
 * \lastreview 22.03.11 Borck
 *
 * \history_begin
 * 01.06.06 CM-DI/ESA3-Prhl
 * call back parameter get from csm_arCbrCfg configuration structure
 *
 * 24.05.06 CM-DI/ESA3-Prhl
 * old code removed
 *
 * 22.09.04 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
VOID CSM_CBR_vProcessCommunicationCon(VOID)
{
  WORD wCount;

  /* loop over all possible clients                                           */
  for (wCount = 0; wCount < CSM_CBR_MAX_USER; wCount++)
  {
    /* check confirmation is pending                                          */
    if (CBR_rState.afCommunicationCon[ wCount] == TRUE)
    {
      /* reset pending flag                                                   */
      CBR_rState.afCommunicationCon[ wCount] = FALSE;

      /* check if valid callback is registered                                */
      if (cbr_arCallbackFkt[ wCount].pfvCommunicationCon != NULL)
      {
        #if (CSM_S_TRACE == CSM_C_F_ON)
          #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
            #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
            ETG_TRACE_USR1(("CommunicationCon       --> Bus: %x Protocol: %x Addr: %02x%02x Action: %02x ApplID: %04x",
            ETG_CENUM(tCSM_BUS_NUMBER,      (tU8)  csm_arCbrCfg[ wCount].bBus),
            ETG_CENUM(tCSM_PROTOCOL_TYPE,   (tU32) csm_arCbrCfg[ wCount].dwProtocolType),
            csm_arCbrCfg[ wCount].au8AddressField[0],
            csm_arCbrCfg[ wCount].au8AddressField[1],
            CBR_rState.abActAction[ wCount],
            csm_arCbrCfg[ wCount].wApplId ));
            #endif
          #else
          {
            // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
            BYTE abData[10];
            abData[0] = csm_arCbrCfg[wCount].bBus;
            abData[1] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(csm_arCbrCfg[wCount].dwProtocolType));
            abData[2] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(csm_arCbrCfg[wCount].dwProtocolType));
            abData[3] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(csm_arCbrCfg[wCount].dwProtocolType));
            abData[4] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(csm_arCbrCfg[wCount].dwProtocolType));

            abData[5] = csm_arCbrCfg[wCount].au8AddressField[0];
            abData[6] = csm_arCbrCfg[wCount].au8AddressField[1];

            abData[7] = CBR_rState.abActAction[wCount];
            abData[8] = CSM_M_GET_HIBYTE(csm_arCbrCfg[wCount].wApplId);
            abData[9] = CSM_M_GET_LOBYTE(csm_arCbrCfg[wCount].wApplId);

            CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_COMMUNICATION_CON, abData, 10);
          }
          #endif
        #endif // CSM_S_TRACE

        #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
          CSM_M_SET_TASK_CONTEXT( &(cbr_arCallbackFkt[wCount].rTaskInfo),
                                  cbr_arCallbackFkt[wCount].pfvCommunicationCon
                                  (
                                    cbr_arCallbackFkt[wCount].pvHandle,
                                    csm_arCbrCfg[wCount].bBus,
                                    csm_arCbrCfg[wCount].dwProtocolType,
                                    csm_arCbrCfg[wCount].au8AddressField,
                                    CBR_rState.abActAction[wCount],
                                    csm_arCbrCfg[wCount].wApplId
                                  )
                                );
        #else // CSM_S_USE_CALLBACK_WITH_HANDLE
          cbr_arCallbackFkt[wCount].pfvCommunicationCon( csm_arCbrCfg[wCount].bBus,
                                                         csm_arCbrCfg[wCount].dwProtocolType,
                                                         csm_arCbrCfg[wCount].au8AddressField,
                                                         CBR_rState.abActAction[wCount],
                                                         csm_arCbrCfg[wCount].wApplId
                                                       );
        #endif // CSM_S_USE_CALLBACK_WITH_HANDLE
      }
    }
  } // loop
}

/*******************************************************************************
 * function    CSM_CBR_vProcessCommunicationInd
 * \doxydocu
 * \brief      Handle communication indication.
 *
 *             Handle communication indication.
 *             This function is called in upstream-task after trigger with CSM_C_IND_REASON_CBR_COMMUNICATION_IND.
 *
 * \return     VOID
 *
 * \access     Upstream-Task
 * \reentrant  no, but only called by Upstream-Task
 *
 * \lastreview 18.04.08 VTeam-Prhl
 *
 * \history_begin
 * 01.06.06 CM-DI/ESA3-Prhl
 * call back parameter get from csm_arCbrCfg configuration structure
 *
 * 24.05.06 CM-DI/ESA3-Prhl
 * old code removed
 *
 * 22.09.04 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
VOID CSM_CBR_vProcessCommunicationInd(VOID)
{
  WORD wCount;

  for( wCount = 0; wCount < CSM_CBR_MAX_USER; wCount++)
  {
    if( CBR_rState.afCommunicationInd[ wCount] == TRUE)
    {
      CBR_rState.afCommunicationInd[ wCount] = FALSE;

      if( cbr_arCallbackFkt[ wCount].pfvCommunicationInd != NULL)
      {
        #if (CSM_S_TRACE == CSM_C_F_ON)
          #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
            #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
            ETG_TRACE_USR1(("CommunicationInd       --> Bus: %x Protocol: %x Addr: %02x%02x Action: %02x ApplID: %04x",
            ETG_CENUM(tCSM_BUS_NUMBER,      (tU8)  csm_arCbrCfg[ wCount].bBus),
            ETG_CENUM(tCSM_PROTOCOL_TYPE,   (tU32) csm_arCbrCfg[ wCount].dwProtocolType),
            csm_arCbrCfg[ wCount].au8AddressField[0],
            csm_arCbrCfg[ wCount].au8AddressField[1],
            CBR_rState.abActAction[ wCount],
            csm_arCbrCfg[ wCount].wApplId ));
            #endif
          #else
          {
            // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
            BYTE abData[10];
            abData[0] = csm_arCbrCfg[ wCount].bBus;
            abData[1] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(csm_arCbrCfg[wCount].dwProtocolType));
            abData[2] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(csm_arCbrCfg[wCount].dwProtocolType));
            abData[3] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(csm_arCbrCfg[wCount].dwProtocolType));
            abData[4] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(csm_arCbrCfg[wCount].dwProtocolType));

            abData[5] = csm_arCbrCfg[wCount].au8AddressField[0];
            abData[6] = csm_arCbrCfg[wCount].au8AddressField[1];

            abData[7] = CBR_rState.abActAction[wCount];
            abData[8] = CSM_M_GET_HIBYTE(csm_arCbrCfg[wCount].wApplId);
            abData[9] = CSM_M_GET_LOBYTE(csm_arCbrCfg[wCount].wApplId);

            CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_COMMUNICATION_IND, abData, 10);
          }
          #endif
        #endif // CSM_S_TRACE

        #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
          CSM_M_SET_TASK_CONTEXT( &(cbr_arCallbackFkt[wCount].rTaskInfo),
                                  cbr_arCallbackFkt[wCount].pfvCommunicationInd
                                  (
                                    cbr_arCallbackFkt[wCount].pvHandle,
                                    csm_arCbrCfg[wCount].bBus,
                                    csm_arCbrCfg[wCount].dwProtocolType,
                                    csm_arCbrCfg[wCount].au8AddressField,
                                    CBR_rState.abActAction[wCount],
                                    csm_arCbrCfg[wCount].wApplId
                                  )
                                );
        #else // CSM_S_USE_CALLBACK_WITH_HANDLE
          cbr_arCallbackFkt[ wCount].pfvCommunicationInd( csm_arCbrCfg[wCount].bBus,
                                                          csm_arCbrCfg[wCount].dwProtocolType,
                                                          csm_arCbrCfg[wCount].au8AddressField,
                                                          CBR_rState.abActAction[wCount],
                                                          csm_arCbrCfg[wCount].wApplId
                                                        );
        #endif // CSM_S_USE_CALLBACK_WITH_HANDLE
      }
    }
  } // loop
}

/*******************************************************************************
 * function    CBR_vProcessNachlaufCon
 * \doxydocu
 * \brief      Handle Nachlauf confimation.
 *
 *             Handle Nachlauf confimation.
 *             This function is call in upstream-task after trigger with
 *             CSM_C_IND_REASON_CBR_NACHLAUF_CON.
 *
 * \return     VOID
 *
 * \access     Upstream-Task
 * \reentrant  no, but only called by Upstream-Task
 *
 * \lastreview 22.03.11 Borck
 *
 * \history_begin
 * 14.04.05 CM-CR/ESD  Peter Tuschik
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
#ifdef CSM_C_IND_REASON_CBR_NACHLAUF_CON
VOID CBR_vProcessNachlaufCon(VOID)
{
  DWORD dwLoop;

  /* Loop over all configured elements                                        */
  for (dwLoop=0; dwLoop < CSM_M_ELEMENTS(csm_rCbrCommConf); dwLoop++)
  {
     const CSM_CBR_COMM_CONF * CommConf = & csm_rCbrCommConf[dwLoop];
     if (CSM_M_IS_NACHLAUF_CONF_REQ_1_SET(dwLoop) && (CommConf->pfvCommConf != NULL))
     {
       ubyte ubConnectState;             /* review: for what is this needed ? */
       CSM_M_CLR_NACHLAUF_CONF_REQ_1(dwLoop);
       /* call registered function for confirmation                           */
       (CommConf->pfvCommConf)(CSM_C_CAN_1,CommConf->dwPTYpe, (const void *) & CommConf->AddrField, CSM_C_CONNECTED_FKT_NACHLAUF, dwLoop);
     }
  }
}
#endif /* #ifdef CSM_C_IND_REASON_CBR_NACHLAUF_CON */

/*******************************************************************************
 * function    CSM_CBR_vCSMBusStateInd
 * \doxydocu
 * \brief      Bus State indication callback.
 *
 *             This funktion is called after the bus state is changed.
 *
 * \param[in]  bBus
 *             Bus
 *
 * \param[in]  bCsmActState
 *             New bus state.
 *             possible values:
 *                CSM_C_STATE_BUS_DOWN, CSM_C_STATE_BUS_LISTEN, CSM_C_STATE_BUS_STARTUP,
 *                CSM_C_STATE_BUS_UP, CSM_C_STATE_BUS_SHUTDOWN
 *
 * \param[in]  bVN
 *             Available if feaure switch CSM_S_VIRTUAL_NETWORK is CSM_C_F_ON.\n
 *             Bit field with (foreign or self) activated virtual networks
 *
 * \return     VOID
 *
 * \access     CAN-Task
 * \reentrant  yes
 *
 * \lastreview 22.03.11 Borck
 *
 * \history_begin
 * 20.10.09 VTeam-Khler
 * - else { if( exchanged by else if(
 * - indentation changed.
 *
 * 16.02.07 VTeam-Prhl
 * bugfix for multi bus systems
 *
 * 14.06.06 CM-DI/ESA3-Prhl
 * virtual network support added
 *
 * 24.05.06 CM-DI/ESA3-Prhl
 * old code removed
 *
 * 11.02.05 CM-DI/ESA3-Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
/* different method of pre-processor usage for function signature eliminates  */
/* problems with some code parsers that identify functions.                   */
#if (CSM_S_VIRTUAL_NETWORK == CSM_C_F_ON)
VOID CSM_CBR_vCSMBusStateInd(BYTE bBus, BYTE bCsmActState, BYTE bVN)
#else
VOID CSM_CBR_vCSMBusStateInd(BYTE bBus, BYTE bCsmActState)
#endif
{
  WORD wCount;

  /**************************************************/
  /* dedicated part if Funktionsnachlauf is ACTIVE  */
  /**************************************************/
  #if (CSM_S_FNL == CSM_C_F_ON)
    if( CBR_rState.abCsmActState[ bBus] != bCsmActState)
    {/* Es hat sich was gendert */
      switch( CBR_rState.abCsmActState[ bBus])
      {
        case CSM_C_STATE_BUS_DOWN:
        case CSM_C_STATE_BUS_LISTEN:
        {
          switch( bCsmActState)
          {
            case CSM_C_STATE_BUS_STARTUP:
            case CSM_C_STATE_BUS_UP:
            case CSM_C_STATE_BUS_SHUTDOWN:
            {/* stop FNL-WAIT Timer, because bus is on */
              CSM_lTimerStop( (WORD ) (CSM_TIMER_CBR_FNL_WAIT_TIMER_0 + bBus));
            }
            break;
            default:
            {/*notthing to do*/
            }
            break;
          }
        }
        break;
        case CSM_C_STATE_BUS_UNDEFINED:
        case CSM_C_STATE_BUS_STARTUP:
        case CSM_C_STATE_BUS_UP:
        case CSM_C_STATE_BUS_SHUTDOWN:
        {
          switch( bCsmActState)
          {
            case CSM_C_STATE_BUS_DOWN:
            case CSM_C_STATE_BUS_LISTEN:
            {
              if( CBR_rState.abBusState[ bBus] == CSM_C_FNL_ON)
              {/* Bus ist ausgegangen und deshalb Restart timer if FNL is on */
                CSM_lTimerStart( (WORD ) (CSM_TIMER_CBR_FNL_WAIT_TIMER_0 + bBus), CSM_CBR_C_FNL_WAIT_TIME);
                CSM_lTimerStop( (WORD ) (CSM_TIMER_CBR_FNL_ALIVE_TIMER_0 + bBus));
              }
            }
            break;
            default:
            {/*notthing to do*/
            }
            break;
          }
        }
        break;
        default:
        {
          CSM_vAssert( FALSE);
        }
        break;
      }
      CBR_rState.abCsmActState[ bBus] = bCsmActState; /* set new value */
    }
  #endif /* CSM_S_FNL */

  /**************************************************/
  /* generic part continues here                    */
  /**************************************************/
  switch( bCsmActState)
  {
    case CSM_C_STATE_BUS_DOWN:
    case CSM_C_STATE_BUS_LISTEN:
    {
      for( wCount = 0; wCount < CSM_CBR_MAX_USER; wCount++)
      {
        if( bBus == csm_arCbrCfg[ wCount].bBus)
        {
        if( CBR_rState.abActAction[ wCount] != CSM_C_STACK_DOWN)
        {
          CBR_rState.abActAction[ wCount] = CSM_C_STACK_DOWN;
          CBR_rState.afCommunicationInd[ wCount] = TRUE;
          csm_vTriggerCsmUpstreamTask( CSM_C_IND_REASON_CBR_COMMUNICATION_IND);
        }
        }
      }
    }
    break;
    case CSM_C_STATE_BUS_STARTUP:
    case CSM_C_STATE_BUS_UP:
    case CSM_C_STATE_BUS_SHUTDOWN:
    {
      for( wCount = 0; wCount < CSM_CBR_MAX_USER; wCount++)
      {
        if( bBus == csm_arCbrCfg[ wCount].bBus)
        {
          if( (CBR_rState.abActAction[ wCount] != CSM_C_STACK_UP)
            #if (CSM_S_VIRTUAL_NETWORK == CSM_C_F_ON)
              && (csm_arCbrCfg[ wCount].bVirtualNetwork & bVN) // Das notwendige VN muss auch aktiv gesetzt sein
            #endif /* CSM_S_VIRTUAL_NETWORK */
              )
          {
            CBR_rState.abActAction[ wCount] = CSM_C_STACK_UP;
            CBR_rState.afCommunicationInd[ wCount] = TRUE;
            csm_vTriggerCsmUpstreamTask( CSM_C_IND_REASON_CBR_COMMUNICATION_IND);
          }
          #if (CSM_S_VIRTUAL_NETWORK == CSM_C_F_ON)
                                                                          /* The bus is up and .. */
          else if( (CBR_rState.abActAction[ wCount] == CSM_C_STACK_UP) && /* the last indication was up but .. */
                   ((csm_arCbrCfg[ wCount].bVirtualNetwork & bVN) !=      /* the necessary VN is not any longer active */
                    csm_arCbrCfg[ wCount].bVirtualNetwork) )
            { /* trigger an indikation with DOWN */
              CBR_rState.abActAction[ wCount] = CSM_C_STACK_DOWN;
              CBR_rState.afCommunicationInd[ wCount] = TRUE;
              csm_vTriggerCsmUpstreamTask( CSM_C_IND_REASON_CBR_COMMUNICATION_IND);
            }
          #endif /* CSM_S_VIRTUAL_NETWORK */
        }
      }
    }
    break;
    default:
    {
      CSM_vAssert( FALSE);
    }
    break;
  }
} /* end CSM_CBR_vCSMBusStateInd() */

/*******************************************************************************
 * function    CSM_CBR_vCSMBusStateReInitInd
 * \doxydocu
 * \brief      Bus State Reinit indication callback.
 *
 *             This funktion is called after a reinit request.
 *
 * \param[in]  bBus
 *             Bus
 * \param[in]  bVN
 *         current VN. CNM_GMLAN_C_VN_ defines are used
 *
 * \return     VOID
 *
 * \access     CAN-Task
 * \reentrant  yes
 *
 * \lastreview 22.03.11 Borck
 *
 * \history_begin
 * 10.07.07 CM-DI/ESA3-Brunotte
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
#if (CSM_S_VIRTUAL_NETWORK == CSM_C_F_ON)
VOID CSM_CBR_vCSMBusStateReInitInd(BYTE bBus, BYTE bVN)
{
  #if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
  WORD wCount;

  for( wCount = 0; wCount < CSM_CBR_MAX_USER; wCount++)
  {
    if ( (bBus == csm_arCbrCfg[ wCount].bBus) &&
         (csm_arCbrCfg[ wCount].bVirtualNetwork & bVN) )
    {
      if( CBR_rState.abActAction[ wCount] == CSM_C_STACK_UP)
      {
        CBR_rState.afCommunicationInd[ wCount] = TRUE;
        csm_vTriggerCsmUpstreamTask( CSM_C_IND_REASON_CBR_COMMUNICATION_IND);
      }
    }
  }
  #endif /* CSM_S_CALLBACK_INIT_BY_APPLICATION */
}
#endif /* CSM_S_VIRTUAL_NETWORK */

/*******************************************************************************
 * function    CSM_CBR_lGetCommunicationState
 * \doxydocu
 * \brief      The actual state of the "connection" can be
 *             checked with this function.
 *
 * \see        CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \param[in]  bBus
 *              Bus number, only needed for "broadcast" protocol types
 *
 * \param[in]  dwProtocolType
 *              Protocol type, see corresponding table of CAN SW architecture
 *
 * \param[in]  pvAddressField
 *              Protocol dependend address field
 *
 * \param[out] pbConnectState
 *              CSM_C_CONNECTED     /  CSM_C_STACK_UP
 *              CSM_C_DISCONNECTED  /  CSM_C_STACK_DOWN
 *
 * \param[in]  wApplId
 *              Application Id, only needed for "broadcast" protocol types
 *
 * \return     Error value (wrong Parameter) if < CSM_C_NO_ERROR
 *             else no error
 *
 * \access     all
 * \reentrant  yes
 *
 * \lastreview 18.04.08 VTeam-Prhl
 *
 * \history_begin
 * 22.08.06 CM-DI/ESA3-Prhl
 * First version
 *
 * \history_end
 ******************************************************************************/
LONG CSM_CBR_lGetCommunicationState(BYTE  bBus, DWORD dwProtocolType,
                                    const void * pvAddressField, BYTE * pbConnectState,
                                    WORD  wApplId)
{
  LONG lRet = CSM_C_NO_ERROR;
  BYTE bUserNumber;

  bUserNumber = CSM_CBR_bGetUserNumber(bBus, dwProtocolType, pvAddressField, wApplId);

  if (bUserNumber < CSM_CBR_MAX_USER)
  {
    *pbConnectState = CBR_rState.abActAction[ bUserNumber];
  }
  else
  {
    lRet = CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_C_INVALID_PARA);
  }

  return lRet;
}/* CSM_lGetCommunicationState() */

/*******************************************************************************
 * function    CSM_CBR_vFnlWaitTimeout
 * \doxydocu
 * \brief      The FNL wait timeout timer callback.
 *
 * \param[in]  bBus
 *              Bus number
 *
 * \return     void
 *
 * \access     Timer interrupt
 * \reentrant  yes
 *
 * \lastreview 22.03.11 Borck
 *
 * \history_begin
 * 01.01.06 CM-DI/ESA3-Prhl
 * First version
 *
 * \history_end
 ******************************************************************************/
#if (CSM_S_FNL == CSM_C_F_ON)
void CSM_CBR_vFnlWaitTimeout(DWORD bBus)
{
  csm_cbr_afFnlWaitTimerExpired[ bBus] = TRUE;
  /* trigger down stream task                                                 */
  /* different trigger parameter is used here, since there is no one for FNL  */
  csm_vTriggerCsmDownstreamTask( CSM_C_DOWN_REASON_CBR_COMMUNICATION_REQ);
}
#endif /* CSM_S_FNL */

/*******************************************************************************
 * function    CSM_CBR_vFnlAliveTimeout
 * \doxydocu
 * \brief      The FNL alive timeout timer callback.
 *
 * \param[in]  bBus
 *              Bus number
 *
 * \return     void
 *
 * \access     Timer interrupt
 * \reentrant  yes
 *
 * \lastreview 18.04.08 VTeam-Prhl
 *
 * \history_begin
 * 01.01.06 CM-DI/ESA3-Prhl
 * First version
 *
 * \history_end
 ******************************************************************************/
#if (CSM_S_FNL == CSM_C_F_ON)
void CSM_CBR_vFnlAliveTimeout(DWORD bBus)
{
  csm_cbr_afFnlAliveTimerExpired[ bBus] = TRUE;
  /* trigger down stream task                                                 */
  /* different trigger parameter is used here, since there is no one for FNL  */
  csm_vTriggerCsmDownstreamTask( CSM_C_DOWN_REASON_CBR_COMMUNICATION_REQ);
}
#endif /* CSM_S_FNL */

#if (CSM_S_CCS_USED == CSM_C_F_ON)
/*******************************************************************************
 * function    CSM_CBR_vSendAllIndicationsAgain
 * \doxydocu
 * \brief      Deliver all CBR communication indications again to the applications.
 *
 *             Set a flag to force indication for communication indications to the
 *             application via the upsteam task.
 *
 * \return     void
 *
 * \access     CAN-Task
 * \reentrant  yes
 *
 * \lastreview
 *
 * \history_begin
 * 22.10.13  Prhl
 * Initial Revision.
 * \history_end
 ******************************************************************************/
void CSM_CBR_vSendAllIndicationsAgain(void)
{
  WORD wCount;
  for( wCount = 0; wCount < CSM_CBR_MAX_USER; wCount++)
  {
    CBR_rState.afCommunicationInd[ wCount] = TRUE;
  }
  csm_vTriggerCsmUpstreamTask( CSM_C_IND_REASON_CBR_COMMUNICATION_IND);
}
#endif /* CSM_S_CCS_USED */

#endif /* CSM_S_DOXYGEN_HIDE_NON_API_FCT */
/*******************************************************************************
 * function implementation (scope: CAN stack #1 API)
 ******************************************************************************/

#undef CSM_CBR_SELF
#undef CSM_CAN_STACK

/*******************************************************************************
 *                MODULE GROUPING INFORMATION FOR DOXYGEN
 * \doxydocu
 *
 * \defgroup      CSM_CBR   CSM_CBR
 *                This group contains all files belonging to broadcast server
 *                functionality of the CAN stack manager.
 * \ingroup       CSM
 *******************************************************************************/

 /*******************************************************************************
 * \doxydocu
 * \file          csm_cbr.c
 * \brief         Broadcast server functionality of the can stack manager.
 *
 *                Detailed description\n
 *                (if available).
 *
 * \see           CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \sw_component  CAN stack
 * \project       Pool module
 * \path          /di_can/modules_swa/csm/src/
 *
 * \ingroup       CSM
 * \ingroup       CSM_CBR
 *
 * \authors       CM-DI/ESA3-Name
 *
 * COPYRIGHT      (c) 2004 Blaupunkt GmbH
 *
 * \history_begin
 *
 * 18.01.04  main\1 CM-DI/ESA3-Name
 * Initial revision.
 *
 * 09.03.05  \main\2 CM-DI/ESA3-Prhl
 * - move broadcast communication request to csm_cbr.c
 *
 * 01.04.05  \main\3 CM-DI/ESA3-Zurmhl
 * - "static" keyword changed to "CSM_STATIC"
 *
 * 08.04.05  main\3 CM-DI/ESA3-Zurmhl
 * - Callback function types renamed from vPFNTp to pfv
 *
 * 07.07.05  main\5 CM-DI/ESA3-Zurmhl
 * - CSM_C_GENERAL_ERROR replaced by CSM_E_UNKNOWN_STATE
 *
 * 19.09.05  \main\6 CM-DI/ESA3-Prhl
 * - add handle to all ...CallbackInit() funktions
 *
 * 28.09.05  \main\7 CM-DI/ESA3-Battis
 * - First version after VW merge
 *   USE WITH CAUTION !
 *
 * 07.11.05  \main\8 CM-DI/ESA3-Prhl
 * - make ready to compile
 *   USE WITH CAUTION !
 *
 * 16.11.05  \main\9 CM-DI/ESA3-Prhl
 * - tCSM_LOCAL_CTRL_SIGNAL_APPL_CALLBACK renamed to tCSM_LOCAL_CTRL_APPL_CALLBACK
 *
 * 20.04.06  \main\10 CM-DI/ESA3-Prhl
 * - temporary integration of old code
 *
 * 29.05.06  \main\11 CM-DI/ESA3-Prhl
 * - temporary integration of old code
 *
 * 01.06.06  \main\12 CM-DI/ESA3-Prhl
 * - CSM_CBR_vProcessCommunicationInd() and CSM_CBR_vProcessCommunicationCon() modified
 * - CBR_MAX_APPL_ID renamed to CSM_CBR_MAX_USER. CSM_CBR_MAX_USER is defined im csm_stack_mp.h
 *
 * 02.06.06  \main\13 CM-DI/ESA3-Prhl
 * - comments corrected
 *
 * 14.06.06  \main\14 CM-DI/ESA3-Prhl
 * - virtual network support added
 * - CBR_vCSMBusStateInd() renamed to CSM_CBR_vCSMBusStateInd()
 * - Please handle careful if you are using this file with a different project as GM GE 09. many things are changed!!!!
 *
 * 14.06.06  \main\15 CM-DI/ESA3-Prhl
 * - move #endif to make it ready to compile with out CSM_S_VIRTUAL_NETWORK
 *
 * 14.07.06  \main\16 CM-DI/ESA3-Prhl
 * - modify CSM_CBR_lApplCallbackInit(). add bBus.
 * - add feature CSM_S_USE_CALLBACK_WITH_HANDLE.
 *   Add the pvHandle from CSM_lApplCallbackInit() to callback routines.
 *   This is used to handle the process border jump (e.g. TENGINE subsystem)
 *
 * 19.07.06 \main\17 CM-DI/ESA3-Prhl
 * - modify feature CSM_S_USE_CALLBACK_WITH_HANDLE.
 *   add PVOID pvHandle to tCSM_ISO_TP_USDT_APPL_CALLBACK if feature CSM_S_USE_CALLBACK_WITH_HANDLE is active
 *
 * 22.08.06 \main\18 CM-DI/ESA3-Prhl
 * - add CSM_CBR_lGetCommunicationState().
 *
 * 20.09.06 \main\19 CM-DI/ESA3-Zurmhl
 * - CSM_API modifier deleted for CSM_CBR_lGetCommunicationState()
 *
 * 10.10.06 \main\20  CM-DI/ESA3-Zurmhl
 * - PVOID replaced by void* (because of const PVOID confusion!)
 * - "static" keyword changed to "CSM_STATIC"
 *
 * 18.10.06  \main\21  CM-DI/ESA3-Prhl
 * - missing const added
 *
 * 01.12.06  \main\22  CM-DI/ESA3-Prhl
 * - add macros CSM_M_GET_TASK_CONTEXT and CSM_M_SET_TASK_CONTEXT for all callback functions
 *
 * 05.12.06  \main\23  CM-DI/ESA3-Prhl
 * - remove bug from CSM_CBR_lApplCallbackInit()
 *
 * 12.01.07  \main\24  CM-DI/ESA3-Prhl
 * - make ready to compile for WIN32. some #if moved
 *
 * 14.02.07  \main\25  VTeam-Prhl
 * - add some CSM_M_TRACEOUT()
 *
 * 16.02.07  \main\26  VTeam-Prhl
 * - Bugfix in CSM_CBR_vCSMBusStateInd() for multi bus systems
 *
 * 15.03.07  \main\27  VTeam-Prhl
 * - FNL code added
 *
 * 22.03.07  \main\28  VTeam-Prhl
 * - bugfix for FNL
 *
 * 22.03.07  \main\29  VTeam-Prhl
 * - compiler warning removed (type changed)
 *
 * 22.03.07  \main\30  VTeam-Prhl
 * - OSEK payload handling added
 *
 * 23.03.07  \main\31  VTeam-Prhl
 * - enable K-CAN added
 *
 * 04.04.07  \main\32  VTeam-Prhl
 * - warning removed
 *
 * 04.04.07  \main\33  VTeam-Prhl
 * - buxfix for FNL. Wrong bBus was used.
 *
 * 10.04.07  \main\34  VTeam-Prhl
 * - only Trace output changed.
 *
 * 17.04.07  \main\35  VTeam-Prhl
 * - missing feature switch CSM_S_FNL added.
 *
 * 22.05.07  \main\36  VTeam-Khler
 * - Wrong csm include order corrected.
 *
 * 15.06.07  \main\37  VTeam-Prhl
 * - stupid asserts removed
 *
 * 19.06.07  \main\38  VTeam-Prhl
 * - some asserts aded to CSM_CBR_lApplCallbackInit()
 * - handling for network management payload added
 *
 * 10.07.07  \main\39  VTeam-Brunotte
 * - CSM_CBR_vCSMBusStateReInitInd function added
 *
 * 19.07.07  \main\40  VTeam-Brunotte
 * - CSM_CBR_lExit added.
 *
 * 24.07.07  \main\41  VTeam-Prhl
 * - some review done
 *
 * 31.07.07  \main\42  VTeam-Prhl
 * - lint warnings removed or commented out.
 *
 * 31.07.07  \main\43  VTeam-Prhl
 * - CSM_C_STATE_BUS_UNDEFINED was not handled in CSM_CBR_vCSMBusStateInd()
 *
 * 29.08.07 \main\44  VTeam-Brunotte
 * - In function CSM_CBR_lApplCallbackInit: A error number is returned, if no valid user number
 *   is found.
 *
 * 14.09.07  \main\45  VTeam-Prhl
 * - CSM_CBR_C_FNL_ALIVE_TIME set to 6500ms
 *
 * 17.09.07  \main\46  VTeam-Prhl
 * - Bugfix for MMS 156965. Wake_Up_Komfort was not cleared if FNL was switched off.
 *
 * 22.10.07  \main\47 VTeam D. Brunotte
 *  - Remote component "Intermediate" and "Remote Diagnostic" in function csm_cbr_bGetVNBit added.
 *
 * 15.11.07  \main\48  VTeam-Prhl
 * - CNM_OSEK_S_VW_PAYLOAD replaced by CNM_OSEK_S_PAYLOAD
 *
 * 27.11.07  \main\49  VTeam-Prhl
 * - wrong CNM_OSEK_S_PAYLOAD replaced by CNM_OSEK_S_VW_PAYLOAD
 *
 * 18.01.08  \main\50 VTeam-Zurmhl
 * - Doxygen tags (defgroup/ingroup) re-arranged
 *
 * 05.02.08  \main\51  VTeam-Khler
 * - Defines CSM_CBR_C_FNL_WAIT_TIME and CSM_CBR_C_FNL_ALIVE_TIME and struct
 *   element abCsmActState[] encapsulated with switch CSM_S_FNL.
 *
 * 07.02.08  \main\52  VTeam-Khler
 * - CBR_rState not CSM_STATIC (GE09 Workaround).
 *
 * 07.02.08  \main\53  VTeam-Khler
 * - CBR_rState CSM_STATIC again.
 *
 * 11.02.08  \main\54  VTeam-Prhl
 * - lint authorisation added.
 *
 * 13.02.08  \main\55  VTeam-Brunotte
 * - Review for csm_cbr_bCalculateActiveVN, csm_cbr_bGetVNBit.
 *
 * 21.02.08  \main\56  VTeam-Prhl
 * - lint authorisation added.
 *
 * 22.02.08  \main\57  VTeam-Prhl
 * - make doxygen happy.
 *
 * 28.03.08 \main\58 VTeam-Brunotte
 * - Before the request is called at least one callback funtion has to be registered. Therefore a
 *   check if this is made is added in CSM_CBR_lCommunicationReq
 * - CBR_rState not CSM_STATIC (GE09 Workaround).
 *
 * 18.04.08  \main\59  VTeam-Prhl
 * - some reviews added.
 *
 * 12.06.08 \main\60 VTeam-Brunotte
 * - Exit functionality added. As a result the function csm_cbr_vApplCallbackDelAll is added.
 *   This function is called in INIT_WARM and EXIT_OFF state. To setup signal callback
 *   functions CSM_VECGMLAN_lInit is called in CSM_CBR_lInit. CSM_VECGMLAN_lExit is called
 *   in CSM_CBR_lExit.
 *
 * 12.06.08 \main\61 VTeam-Brunotte
 * - Macro CSM_S_USE_TKSE check in several functions added.
 *
 * 17.02.09 \main\62 VTeam-Zurmhl
 * - Reviews done (no change in code)
 *
 * 17.08.09 \main\63 VTeam-Franke
 * - new: CSM_CBR_vEndCommunicationRequests
 *
 * 23.09.09 \main\64 VTeam-Franke
 * - CSM_CBR_vEndCommunicationRequests removed
 *
 * 28.10.09  \main\65  VTeam-Khler
 * - csm_cbr_bCalculateActiveVN() renamed CSM_CBR_bQueryRequestedVNs().
 *
 * 06.08.10  \main\66  VTeam-Pistoor
 * - Added precompiler switch CSM_S_VIRTUAL_NETWORK
 *
 * 22.03.11  \main\67  Borck
 * - reviewed
 *
 * 13.07.11  \main\68  Borck
 * - ETG tracing introduced.
 *
 * 05.08.13  \main\69  Prhl
 * - Payload handling for VW NM HIGH added.
 *
 * 12.09.13  \main\70  Feldhaus
 * - integrated CSM_CBR_vPROXYCommunicationCon for CSM Proxy support
 *
 * 18.10.13  \main\71  Prhl
 * - cleanup of ComminicationReq/Con/Ind handling.
 *
 * 22.10.13  \main\72  Prhl
 * - CSM_CBR_vSendAllIncicationsAgain() added.
 *
 * 26.10.13  \main\73  Prhl
 * - Renamed CSM_CBR_vSendAllIncicationsAgain() to CSM_CBR_vSendAllIndicationsAgain().
 *
 * 07.11.13  \main\74  Prhl
 * - missing #ifdef added to avoid lint wornings
 *
 * 18.12.13  \main\75  Prhl
 * - wrong indication reason corrected (CSM_C_IND_REASON_CBR_COMMUNICATION_CON/IND)
 *
 * 13.03.15  \main\76  Borck
 * - trace without ETG for Autosar environment.
 *
 * 21.04.15  \main\77  Borck
 * - added function CSM_CBR_bGetUserAssignedBus() to fix access to config issue in csm_proxy.c
 *
 * 23.07.15  \main\78 Borck
 * - local function made public (name changed)
 *
 * 10.11.16  \main\79 Borck
 * - trace buffer sizes have been checked against CSM_MAX_TRACE_LENGTH due to reduction of the same in csm_stack_m.h
 * - no functional changes
 *
 * \history_end
 *//**** END OF FILE **********************************************************/
