
/* Description and history of cnm_osek.c at the end of the file.              */
/** File name for preprozessor commands */
#define CNM_OSEKI_SELF
/** cnm_osek.c is a file according V-team CAN-SW architecture implementation rules */
#define CSM_CAN_STACK

/*************** can stack and CSM callbacks **************/
#include "csm_stack_i.h"
#include "csm_stack_m.h"

#include "csm_c.h"
#include "csm_m.h"

#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
#if (CSM_S_MPDT == CSM_C_F_ON)
//#include "cnp_mpdt_m.h"
#endif
#endif

/*************** CNM **************/
#include "cnm_m.h"

#include "cnm_oseki_i.h"
#include "cnm_oseki_m.h"

/*************** CDA **************/
#include "cda_i.h"

/*************** APL **************/
#include "csm_apl.h"

/*************** ETG Tracing ******/
#define TR_CLASS_CSMOSEKI (TR_COMP_CSM + 0x14)

#if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #define ETRACE_S_IMPORT_INTERFACE_GENERIC
  #include "etrace_if.h"

  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
    #include "csm_etg_trace.h"
    #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CSMOSEKI
    /* #define ETG_INLINE */
    #include "trcGenProj/Header/cnm_oseki.c.trc.h"
  #endif
  #endif
#endif // CSM_S_TRACE

/*******************************************************************************
 * check of all feature switches which affects this file :
 * The following #defines should be declared in csm_stack_p.h:
 ******************************************************************************/

/*******************************************************************************
 * defines and macros (scope: file local)
 ******************************************************************************/


/*******************************************************************************
 * typedefs (scope: file local)
 ******************************************************************************/
/* Attention: do not change these enums without cross-changing OSEKI state    */
/* values in csm_stack_ip.h, which are located there for exports              */
typedef enum
{
  _RX_OK___________     = 0x0000,
  _ABSENT__________     = 0x0001,
  _CONFIRMED_ABSENT     = 0x0002
} tCNM_OSEKI_RX_ERROR_STATE;

typedef enum
{
  _TX_OK___________      = 0x0000,
  _MUTED___________      = 0x0001,
  _CONFIRMED_MUTED_      = 0x0002
} tCNM_OSEKI_TX_ERROR_STATE;

typedef enum
{
  _BUS_OK__________     = 0x0000,
  _BUS_OFF_________     = 0x0001,
  _CONFIRMED_BUSOFF     = 0x0002
} tCNM_OSEKI_BUS_ERROR_STATE;

/* internal struct for handling communication requests                        */
typedef struct
{
  BYTE bReqAction;                     // requested command by CommunicationReq
  BOOLEAN bCommunicationReq;           // trigger flag for asynch. evaluation
} tCNM_OSEKI_COMMUNICATION_STATE;

/*******************************************************************************
 * variable definition (scope: file local)
 ******************************************************************************/
#if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
/* Array to store the callback functions for each client */
CSM_STATIC tCSM_OSEKI_NWM_APPL_CALLBACK   csm_nmCallbackFkt[CSM_NWM_C_MAX_USER];
#endif

/* Initialisation of these vars is done with COLD_INIT via a function         */
CNM_OSEKI_tState            CNM_OSEKI_arState[CNM_OSEKI_ANZAHL_BUSSE];

/* Counters and states for RX/TX monitoring Bus 0 -> MCAN                     */
CNM_OSEKI_RxMonitorState     CNM_OSEKI_arRxStatesBus_MCAN[CNM_OSEKI_NR_OF_RX_NODE_MONITORS_MCAN];
CNM_OSEKI_TxMonitorState     CNM_OSEKI_arTxStatesBus_MCAN[CNM_OSEKI_NR_OF_TX_MUTE_MONITORS_MCAN];
CNM_OSEKI_BusOffMonitorState CNM_OSEKI_BusOffStates_MCAN;

/* Counters and states for RX/TX monitoring Bus 1 -> VCAN                     */
CNM_OSEKI_RxMonitorState     CNM_OSEKI_arRxStatesBus_VCAN[CNM_OSEKI_NR_OF_RX_NODE_MONITORS_VCAN];
CNM_OSEKI_TxMonitorState     CNM_OSEKI_arTxStatesBus_VCAN[CNM_OSEKI_NR_OF_TX_MUTE_MONITORS_VCAN];
CNM_OSEKI_BusOffMonitorState CNM_OSEKI_BusOffStates_VCAN;

static BOOL cnm_oseki_fTraceListenOnce[CNM_OSEKI_ANZAHL_BUSSE];
static tCNM_OSEKI_COMMUNICATION_STATE  cnm_oseki_arCommunicationState[CSM_NWM_C_MAX_COMMAND_USER];

/*******************************************************************************
 * variable definition (scope: CAN stack)
 ******************************************************************************/
/*-- none --*/

/*******************************************************************************
 * function prototypes (scope: file local = static)
 ******************************************************************************/
/* Init and Set Functions */
static void cnm_oseki_vInitVariables(void);
static void cnm_oseki_vSetupMonitoringStates(BYTE bCnmBus);
static void cnm_oseki_vSetBusState(BYTE bBus, BYTE bBusState);
static void cnm_oseki_vSetNmMode(BYTE bBus, BYTE bMode);
static VOID cnm_oseki_vSetMonitorState(BYTE bBus, BYTE bIndicator, BYTE bIndex, BYTE bState);

static void cnm_oseki_vStartStopRxNodeMonitoring(BYTE bBus, BYTE bMode);
static void cnm_oseki_vStartStopTxMuteMonitoring(BYTE bBus, BYTE bMode);
static void cnm_oseki_vStartStopBusOffMonitoring(BYTE bBus, BYTE bMode);

/* FSM Functions */
static tU32 cnm_oseki_fsm_u32CalcNewFsmState(CNM_OSEKI_FSM_tFsmInstance * pFsmInstance, tU32 u32Trigger, BOOL bState);
static void cnm_oseki_fsm_vActStateExit(CNM_OSEKI_FSM_tFsmInstance * pFsmInstance, tU32 u32FsmState);
static tU32 cnm_oseki_fsm_u32CallStateFunc(CNM_OSEKI_FSM_tFsmInstance * pFsmInstance, tU32 u32FsmState, tU8 u8SearchIndex);
#if 0
static void cnm_oseki_fsm_vTraceStatusInfo(CNM_OSEKI_FSM_tFsmInstance * pFsmInstance, tU32 u32RemainingTime);
#endif
static tU32 cnm_oseki_fsm_u32ActStateEntry(CNM_OSEKI_FSM_tFsmInstance * pFsmInstance, tU32 u32FsmState);
static void cnm_oseki_fsm_vStartFsmStateTimer(CNM_OSEKI_FSM_tFsmInstance * pFsmInstance, tU32 u32Time);
static void cnm_oseki_fsm_vNewTrigger(CNM_OSEKI_FSM_tFsmInstance * pFsmInstance, tenCNMFsmTrigger_ETG eTrigger, BOOL bTriggerState);
static void cnm_oseki_fsm_vStateNoTransition(CNM_OSEKI_FSM_tFsmInstance * pFsmInstance);
static void cnm_oseki_fsm_vStateTransition(CNM_OSEKI_FSM_tFsmInstance * pFsmInstance);

/* Node/Mute and BusOff Monitor Handler */
static void cnm_oseki_vRxNodeMonitoring(const CSM_CAN_RX_DATA * prData);
static void cnm_oseki_vTxMuteMonitoring(const CSM_CAN_CON_DATA * prData);
static void cnm_oseki_vBusOffMonitoring(BYTE bBus, BYTE bState);

static BOOLEAN cnm_oseki_fCheckForMaxTxMuteCounter(BYTE bBus, BYTE bMuteIndex);
static BOOLEAN cnm_oseki_fCheckForMaxRxNodeCounter(BYTE bBus, BYTE bNodeIndex);
static BOOLEAN cnm_oseki_fCheckForMaxBusOffCounter(BYTE bBus);

static void cnm_oseki_vSetErrorConfirmedTxMute(BYTE bBus, BYTE bMuteIndex);
static void cnm_oseki_vSetErrorConfirmedRxAbsent(BYTE bBus, BYTE bRxIndex);
static void cnm_oseki_vSetErrorConfirmedBusOff(BYTE bBus);

/* Bus State Handler */
static void cnm_oseki_vRxBusStateManagement(const CSM_CAN_RX_DATA * prData);

/* Timeout Handler */
static void cnm_oseki_vProcessRxNodeMonitorTimeouts(void);
static void cnm_oseki_vProcessTxMuteMonitorTimeouts(void);
static void cnm_oseki_vProcessBusOffMonitorTimeouts(void);
static void cnm_oseki_vProcessBusStateTimeouts(void);
static void cnm_oseki_vProcessFsmTimeouts(void);

/* Timer Starter */
static void cnm_oseki_vStartStopRxMonitorTimer(BYTE bBus, WORD wTimerbase, BYTE bNodeIndex, BYTE bMode);
static void cnm_oseki_vStartStopTxMonitorTimer(BYTE bBus, WORD wTimerbase, BYTE bMuteIndex, BYTE bMode);
static void cnm_oseki_vStartStopBusOffMonitorTimer(BYTE bBus, WORD wTimerbase, BYTE bMode);

static BYTE cnm_oseki_bGetRxNodeIndex(DWORD dwId, BYTE bBus);
static BYTE cnm_oseki_bGetTxNodeIndex(DWORD dwId, BYTE bBus);
//static BYTE cnm_oseki_bGetMonitorState(BYTE bBus, BYTE bIndicator, BYTE bIndex);
//static BYTE cnm_oseki_bGetResetMonitorStateChange(BYTE bBus, BYTE bIndicator, BYTE bIndex);

static WORD cnm_oseki_wGetTimerbase(BYTE bBus, BYTE bType);

/* Callback related */
static BYTE cnm_oseki_bGetUserNumber(BYTE bBus, DWORD dwProtocolType, const void * pvAddressField, WORD wApplId);
static BYTE cnm_oseki_bGetMonitorNodeIndex(BYTE bUserNumber);

static void cnm_oseki_vCheckForNormal(BYTE bBus);

/*******************************************************************************
 * constants and vars (scope: file local)
 ******************************************************************************/
static tCNM_OSEKI_RX_ERROR_STATE tenDummy1 =  _RX_OK___________;
static tCNM_OSEKI_TX_ERROR_STATE tenDummy2 =  _TX_OK___________;
static tCNM_OSEKI_BUS_ERROR_STATE tenDummy3 = _BUS_OK__________;

/* Feature switch is used only to control doxygen documentation */
#if (CSM_S_DOXYGEN_HIDE_NON_API_FCT == CSM_C_F_OFF)
/*******************************************************************************
 * function implementation (scope: file local = static)
 ******************************************************************************/
 /*******************************************************************************
 * function    csm_NM_bGetUserNumber
 * \doxydocu
 * \brief      get user number.
 *
 *             returns the user number (index) to a given parameter set
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESA3
 *
 * \param[in]  bBus (I)
 *             bus number
 *
 * \param[in]  dwProtocolType (I)
 *             protocol type, see table 2 of CAn SW architecture
 *
 * \param[in]  *pvAddressField (->I)
 *             addressing, dependent on protocol
 *
 * \param[in]  BYTE  wApplId (I)
 *             Appl Id
 *
 * \return     BYTE
 *             user number = from 0 to CSM_NWM_C_MAX_USER-1. CSM_NWM_C_MAX_USER is an error value.
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \lastreview 09.05.11 Prhl
 *
 * \history_begin
 * 22.10.10 Borck
 * Initial Revision.
 * \history_end
 ******************************************************************************/
static BYTE cnm_oseki_bGetUserNumber(BYTE bBus, DWORD dwProtocolType, const void * pvAddressField, WORD wApplId)
{
  BYTE bUserNumber;

  // NULL pointer check of pvAddressfield required to pass Coverity
  if (pvAddressField == NULL)
  {
    // bail out with invalid user number (too large)
    return (CSM_NWM_C_MAX_USER + 1);
  }

  // loop over the configuration array defined in cnm_oseki_mp.h or csm_stack_mp.h
  for (bUserNumber = 0; bUserNumber < CSM_NWM_C_MAX_USER; bUserNumber++)
  {
    if ( (bBus == csm_arNWMCfg[bUserNumber].bBus) &&
         (dwProtocolType == csm_arNWMCfg[ bUserNumber].dwProtocolType) &&
         ( *((((BYTE *)pvAddressField)+0)) == csm_arNWMCfg[bUserNumber].au8AddressField[0]) &&
         ( *((((BYTE *)pvAddressField)+1)) == csm_arNWMCfg[bUserNumber].au8AddressField[1]) &&
         (wApplId == csm_arNWMCfg[bUserNumber].wApplId) )
    {
      // all passed parameters are matching the current entry, so return the index of this entry
      return bUserNumber;
    }
  }
  return bUserNumber;
}

 /*******************************************************************************
 * function    cnm_oseki_bGetMonitorNodeIndex
 * \doxydocu
 * \brief      determine monitor index from given user number from configuration
 *
 * \param[in]  bUserNumber (I)
 *               index to configuration entry
 *
 * \return     BYTE
 *               index to the corresponding monitor array containing the requested state
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 * 18.05.11 Borck
 * Initial Revision.
 * \history_end
 ******************************************************************************/
static BYTE cnm_oseki_bGetMonitorNodeIndex(BYTE bUserNumber)
{
  return csm_arNWMCfg[bUserNumber].bMonitorIndex;
}

/*******************************************************************************
 * function    cnm_oseki_vInitVariables
 * \doxydocu
 * \brief      set administration struct to default
 *
 * \param      none
 *
 * \return     void
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \lastreview 12.05.11 Prhl
 *
 * \history_begin
 * 30.04.10 Borck
 * Initial Revision.
 * \history_end
 ******************************************************************************/
static void cnm_oseki_vInitVariables(void)
{
  BYTE bBus;
  BYTE bUnusedConfig;
  BYTE bIsaConfig;

  #if (CSM_S_VARIANTS_MANAGEMENT == CSM_C_F_ON)
  #if (CSM_S_SLOT_BASED_CONFIG == CSM_C_F_ON)
  BYTE bSlot = CSM_APL_bGetConfigurationSlotNr();
  extern tCSM_SLOT_CONFIG CSM_APL_SLOT_CONFIG[CSM_APL_MAX_CONFIG_SLOTS][CSM_C_MAX_CAN_BUS];
  #endif
  #endif

  /* dummy statements to avoid annoying LINT warnings as a result of ETG      */

  tenDummy1 = _CONFIRMED_ABSENT;
  tenDummy2 = _CONFIRMED_MUTED_;
  tenDummy3 = _CONFIRMED_BUSOFF;

  if ((tenDummy1 ==_RX_OK___________) || (tenDummy2 == _TX_OK___________) || (tenDummy3 == _BUS_OK__________))
  {
    tenDummy1 = _ABSENT__________;
    tenDummy2 = _MUTED___________;
    tenDummy3 = _BUS_OFF_________;
  }

  // now start with sensible code !!!
  #if (CSM_S_VARIANTS_MANAGEMENT == CSM_C_F_ON)
    #if (CSM_S_SLOT_BASED_CONFIG == CSM_C_F_ON)
      // in case of a SLOT based configuration, we choose our configuration array according to the active slot
      // since the bus configuration for OSEKI contains both busses, we here always set the first bus as a base, the pointer later dereferences
      // the correct bus
      CNM_OSEKI_arBusConfig = CSM_APL_SLOT_CONFIG[bSlot][0].pOsekiBusConfig;  // the pointers in [0] and [1] are identical within the same slot, but only the first one is used
    #else
      // use a valid and functional standard configuration (here SLOT 1, since SLOT 0 has no RX and no TX monitoring)
      CNM_OSEKI_arBusConfig = CNM_OSEKI_arBusConfig_Slot1;
    #endif
  #else
  // without variant management we use the SLOT 1 configuration (which at least should always be available)
  // even if no SLOT configuration is supported (the name can be changed in our ..._mp.h file)
  CNM_OSEKI_arBusConfig = CNM_OSEKI_arBusConfig_Slot1;
  #endif

  for (bBus = 0; bBus < CNM_OSEKI_ANZAHL_BUSSE; bBus ++)
  {
    CNM_OSEKI_arState[bBus].bBusStateTimeoutReason = CNM_OSEKI_C_TIMER_REASON_UNDEF;
    CNM_OSEKI_arState[bBus].bBusState = CSM_C_STATE_BUS_DOWN;
    /* der erste Initialisierungsschritt setzt erstmal einen NMOff Mode       */
    CNM_OSEKI_arState[bBus].bNMMode = CNM_OSEKI_C_NMOff;
    CNM_OSEKI_arState[bBus].fBusStateTimeout = FALSE;
    /* monitoring is disabled upon start                                      */
    CNM_OSEKI_arState[bBus].bRxMonitoringState     = CNM_OSEKI_MONITOR_OFF;
    CNM_OSEKI_arState[bBus].bTxMonitoringState     = CNM_OSEKI_MONITOR_OFF;
    CNM_OSEKI_arState[bBus].bBusOffMonitoringState = CNM_OSEKI_MONITOR_OFF;
    CNM_OSEKI_arState[bBus].fStartup = TRUE;
  }

  /* for generic usage, the state struct is initialised from data defined in  */
  /* the project specific file cnm_oseki_mp.h                                 */
  for (bBus = 0; bBus < CNM_OSEKI_ANZAHL_BUSSE; bBus ++)
  {
    CNM_OSEKI_arState[bBus].pRxStates     = CNM_OSEKI_arBusConfig[bBus].parRxStates;
    CNM_OSEKI_arState[bBus].pTxStates     = CNM_OSEKI_arBusConfig[bBus].parTxStates;
    CNM_OSEKI_arState[bBus].pBusOffStates = CNM_OSEKI_arBusConfig[bBus].parBusOffStates;

    CNM_OSEKI_arState[bBus].pRxNodeData   = CNM_OSEKI_arBusConfig[bBus].parRxNodeData;
    CNM_OSEKI_arState[bBus].pTxMuteData   = CNM_OSEKI_arBusConfig[bBus].parTxMuteData;
    CNM_OSEKI_arState[bBus].pBusOffData   = CNM_OSEKI_arBusConfig[bBus].parBusOffData;

    /* init the node specific sub-structs  (RX/TX States, Flags, Counters)    */
    cnm_oseki_vSetupMonitoringStates(bBus);

    /* reset "state trace control" to initial value                           */
    cnm_oseki_fTraceListenOnce[bBus] = FALSE;
  }

  /* init the communication request states                                    */
  for (bBus = 0; bBus < CSM_NWM_C_MAX_COMMAND_USER; bBus ++)
  {
    /* bBus wird hier als index missbraucht, der Name ist hier irrefhrend !  */
    cnm_oseki_arCommunicationState[bBus].bReqAction = CNM_OSEKI_CMD_NO_CMD;
    cnm_oseki_arCommunicationState[bBus].bCommunicationReq = FALSE;
  }
}

/*******************************************************************************
 * function    cnm_oseki_vSetupMonitoringStates
 * \doxydocu
 * \brief      internal function to set the monitoring states from Cold Init
 *
 * \param[in]  bBus
 *
 * \return     void
 *
 * \access     task
 * \reentrant  no
 *
 * \lastreview 09.05.11 Prhl
 *
 * \history_begin
 * 28.04.10 Borck
 * Initial Revision.
 * \history_end
 ******************************************************************************/
static void cnm_oseki_vSetupMonitoringStates(BYTE bBus)
{
  BYTE bTemp;
  BYTE bNrRxNodes;
  BYTE bNrTxNodes;

  /* get the max values for the loops below from bus config                   */
  bNrRxNodes = CNM_OSEKI_arBusConfig[bBus].bAmountRxNodes;
  bNrTxNodes = CNM_OSEKI_arBusConfig[bBus].bAmountTxNodes;

  /* reset RX counters and states for all nodes                               */
  for (bTemp = 0; bTemp < bNrRxNodes; bTemp ++)
  {
    /* since there might be unused busses, check the pointer first            */
    /* BKA2HI: default value changed according to NDS25953NDS06-2 for LCN2kai */
    if (CNM_OSEKI_arState[bBus].pRxStates != NULL)
    {
      CNM_OSEKI_arState[bBus].pRxStates[bTemp].fRxNodeTimeout = FALSE;
      CNM_OSEKI_arState[bBus].pRxStates[bTemp].wRxNodeCounter = 0;
      CNM_OSEKI_arState[bBus].pRxStates[bTemp].bRxNodeAbsentState = CNM_OSEKI_C_STATE_NODE_PRESENT;
      CNM_OSEKI_arState[bBus].pRxStates[bTemp].bStateChanged = 0x00;
    }
  }

  /* reset TX counters and states for all nodes                               */
  for (bTemp = 0; bTemp < bNrTxNodes; bTemp ++)
  {
    /* since there might be unused busses, check the pointer first            */
    /* BKA2HI: default value changed according to NDS25953NDS06-2 for LCN2kai */
    if (CNM_OSEKI_arState[bBus].pTxStates != NULL)
    {
      CNM_OSEKI_arState[bBus].pTxStates[bTemp].fTxMuteTimeout = FALSE;
      CNM_OSEKI_arState[bBus].pTxStates[bTemp].wTxMuteCounter = 0;
      CNM_OSEKI_arState[bBus].pTxStates[bTemp].bTxNodeMuteState = CNM_OSEKI_C_STATE_NODE_NOT_MUTED;
      CNM_OSEKI_arState[bBus].pTxStates[bTemp].bStateChanged = 0x00;
    }
  }

  /* reset BusOff counters and states                                         */
  if (CNM_OSEKI_arState[bBus].pBusOffStates != NULL)
  {
    CNM_OSEKI_arState[bBus].pBusOffStates->fBusOffTimeout = FALSE;
    CNM_OSEKI_arState[bBus].pBusOffStates->wBusOffCounter = 0 ;
    CNM_OSEKI_arState[bBus].pBusOffStates->bBusOffState = CNM_OSEKI_C_STATE_BUS_ON;
    CNM_OSEKI_arState[bBus].pBusOffStates->bStateChanged = 0x00;
  }
}

static BYTE cnm_oseki_bGetMonitorState(BYTE bBus, BYTE bIndicator, BYTE bIndex)
{
  BYTE bReturn = CNM_OSEKI_C_STATE_ERROR; /* assume ERROR as return value     */
                                          /* ensure that 0xFF is not used as a valid state value */
                                          /* check this in cnm_oseki_mp.h                        */

  switch(bIndicator)
  {
    /* --- RX absent state requested ---                                      */
    case CSM_C_ADR_FLD_DC_NM_RX_INDICATOR:
      /* check if requested node is configured for this bus                   */
      if ((CNM_OSEKI_arState[bBus].pRxStates != NULL) &&
          (CNM_OSEKI_arState[bBus].pRxNodeData[bIndex].dwId != 0x00000000))
      {
        bReturn = CNM_OSEKI_arState[bBus].pRxStates[bIndex].bRxNodeAbsentState;
      }
      break;

    /* --- TX mute state requested ---                                        */
    case CSM_C_ADR_FLD_DC_NM_TX_INDICATOR:
      /* check if requested node is configured for this bus                   */
      if ((CNM_OSEKI_arState[bBus].pTxStates != NULL) &&
          (CNM_OSEKI_arState[bBus].pTxMuteData[bIndex].dwId != 0x00000000))
      {
        bReturn = CNM_OSEKI_arState[bBus].pTxStates[bIndex].bTxNodeMuteState;
      }
      break;

    /* --- BusOff state requested ---                                         */
    case CSM_C_ADR_FLD_DC_NM_BUSOFF_INDICATOR:
      /* check if requested bus is configured for this request                */
      if (CNM_OSEKI_arState[bBus].pBusOffStates != NULL)
      {
        bReturn = CNM_OSEKI_arState[bBus].pBusOffStates->bBusOffState;
      }
      break;

    default:
      break;
  }
  return bReturn;
}

#if (CSM_S_REMOTE_STACK == CSM_C_F_ON)
static VOID cnm_oseki_vSetMonitorState(BYTE bBus, BYTE bIndicator, BYTE bIndex, BYTE bState)
{
  switch(bIndicator)
  {
    /* --- RX absent state requested ---                                      */
    case CSM_C_ADR_FLD_DC_NM_RX_INDICATOR:
      /* check if requested node is configured for this bus                   */
      if ((CNM_OSEKI_arState[bBus].pRxStates != NULL) &&
          (CNM_OSEKI_arState[bBus].pRxNodeData[bIndex].dwId != 0x00000000))
      {
        CNM_OSEKI_arState[bBus].pRxStates[bIndex].bRxNodeAbsentState = bState;
        CNM_OSEKI_arState[bBus].pRxStates[bIndex].bStateChanged = 0x01;
      }
      break;

    /* --- TX mute state requested ---                                        */
    case CSM_C_ADR_FLD_DC_NM_TX_INDICATOR:
      /* check if requested node is configured for this bus                   */
      if ((CNM_OSEKI_arState[bBus].pTxStates != NULL) &&
          (CNM_OSEKI_arState[bBus].pTxMuteData[bIndex].dwId != 0x00000000))
      {
        CNM_OSEKI_arState[bBus].pTxStates[bIndex].bTxNodeMuteState = bState;
        CNM_OSEKI_arState[bBus].pTxStates[bIndex].bStateChanged = 0x01;
      }
      break;

    /* --- BusOff state requested ---                                         */
    case CSM_C_ADR_FLD_DC_NM_BUSOFF_INDICATOR:
      /* check if requested bus is configured for this request                */
      if (CNM_OSEKI_arState[bBus].pBusOffStates != NULL)
      {
        CNM_OSEKI_arState[bBus].pBusOffStates->bBusOffState = bState;
        CNM_OSEKI_arState[bBus].pBusOffStates->bStateChanged = 0x01;
      }
      break;

    default:
      break;
  }
}
#endif //CSM_S_REMOTE_STACK == CSM_C_F_ON

/*******************************************************************************
 * function    cnm_oseki_bGetResetMonitorStateChange
 * \doxydocu
 * \brief      delivers if the state of a specified monitor (RX, TX, BusOff) has changed
 *             resets this flag, since the state change will be forwarded right now
 *
 * \param[in]  bBus, bIndicator, bIndex
 *
 * \return     0x00 or 0x01
 *
 * \access     task
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 * 26.10.10 Borck
 * Initial Revision.
 * \history_end
 ******************************************************************************/
static BYTE cnm_oseki_bGetResetMonitorStateChange(BYTE bBus, BYTE bIndicator, BYTE bIndex)
{
  BYTE bReturn = 0x00;   /* assume state has not changed as return value     */

  switch(bIndicator)
  {
    /* --- RX absent state requested ---                                      */
    case CSM_C_ADR_FLD_DC_NM_RX_INDICATOR:
      /* check if requested node is configured for this bus                   */
      if ((CNM_OSEKI_arState[bBus].pRxStates != NULL) &&
          (CNM_OSEKI_arState[bBus].pRxNodeData[bIndex].dwId != 0x00000000))
      {
        bReturn = CNM_OSEKI_arState[bBus].pRxStates[bIndex].bStateChanged;
        /* reset the changed flag, since changed status will be forwarded to another instance */
        CNM_OSEKI_arState[bBus].pRxStates[bIndex].bStateChanged = 0x00;
      }
      break;

    /* --- TX mute state requested ---                                        */
    case CSM_C_ADR_FLD_DC_NM_TX_INDICATOR:
    /* check if requested node is configured for this bus                   */
      if ((CNM_OSEKI_arState[bBus].pTxStates != NULL) &&
          (CNM_OSEKI_arState[bBus].pTxMuteData[bIndex].dwId != 0x00000000))
      {
        bReturn = CNM_OSEKI_arState[bBus].pTxStates[bIndex].bStateChanged;
        /* reset the changed flag, since changed status will be forwarded to another instance */
        CNM_OSEKI_arState[bBus].pTxStates[bIndex].bStateChanged = 0x00;
      }
      break;

    /* --- BusOff state requested ---                                       */
    case CSM_C_ADR_FLD_DC_NM_BUSOFF_INDICATOR:
    /* check if requested bus is configured for this request                */
      if (CNM_OSEKI_arState[bBus].pBusOffStates != NULL)
      {
        bReturn = CNM_OSEKI_arState[bBus].pBusOffStates->bStateChanged;
        /* reset the changed flag, since changed status will be forwarded to another instance */
        CNM_OSEKI_arState[bBus].pBusOffStates->bStateChanged = 0x00;
      }
      break;

    default:
      break;
  }
  return bReturn;
}

/*******************************************************************************
 * function implementation (scope: CAN stack and operating system)
 ******************************************************************************/

/*******************************************************************************
 * function    CNM_OSEKI_lInit
 * \doxydocu
 * \brief      Initializes OSEKI NM module.
 *
 *             Initializes OSEKI NM module.
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESA3
 *
 * \param      bInitMode
 *             Allowed values:
 *                CSM_C_INIT_COLD, CSM_C_INIT_WARM, CSM_C_INIT_START (see csm_stack_i.h)
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR, TODO: CNM_E_ERROR_STATE_
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \lastreview 09.05.11 Prhl
 *
 * \history_begin
 * 04.09.09 Prhl
 * Initial Revision.
 * \history_end
 ******************************************************************************/
LONG CNM_OSEKI_lInit(BYTE bInitMode)
{
  LONG lRet = CSM_C_NO_ERROR;

  #ifdef WIN32
  printf("CNM_OSEKI__lInit(%02X)\n", bInitMode);
  #endif

  switch(bInitMode)
  {
    case CSM_C_INIT_COLD:
      /* initialize our administration structs and the FSM                    */
      /* Bus states start in DOWN, NM in OFF, all monitors are OFF            */
      cnm_oseki_vInitVariables();
      break;

    case CSM_C_INIT_WARM:
      /* bus states are still DOWN for all configured busses                  */
      break;

    case CSM_C_INIT_START:
      /* the NM FSM cannot be used at this point, because triggering a state  */
      /* change from here requires the availibilty of semaphors for the timer */
      /* system. Since semaphores are NOT available at this point, the first  */
      /* NM state transition has to be triggered indirectly after the CSM     */
      /* has been fully activated and will calculate its bus states for the   */
      /* first time. This happens automatically triggered by an upper layer   */
      break;

    default:
      lRet = (CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CNM_OSEK, CSM_E_INVALID_PARA));
      break;
  }
  return lRet;
}

/*******************************************************************************
 * function    CNM_OSEKI_lGotoMode
 * \doxydocu
 * \brief      Switches the NM state maschine into bMode mode.
 *
 *             Switches the NM state maschine into bMode mode.
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/EHS3
 *
 * \param[in]  bBus
 *                CAN bus
 *
 * \param[in]  bMode
 *             Allowed values:
 *                CNM_OSEKI_C_NMNormal (AWAKE), CNM_OSEKI_C_NMBusSleep (SLEEP)
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \lastreview 09.05.11 Prhl
 *
 * \history_begin
 * 23.11.09 CM-AI/Borck
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
LONG CNM_OSEKI_lGotoMode(BYTE bBus, BYTE bMode)
{
  LONG lRet = CSM_C_NO_ERROR;

  return lRet;
}

/*******************************************************************************
 * function    CNM_OSEKI_lExit
 * \doxydocu
 * \brief      Stops OSEKI NM module.
 *
 *             Stops OSEKI NM module
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/EHS3
 *
 * \param[in]  bExitMode
 *             Allowed values:
 *                CSM_C_EXIT_STOP, CSM_C_EXIT_OFF (see csm_stack_i.h)
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \lastreview 09.05.11 Prhl
 *
 * \history_begin
 * 04.09.09 Prhl
 * Initial Revision.
 * \history_end
 ******************************************************************************/
LONG CNM_OSEKI_lExit(BYTE bExitMode)
{
  LONG lRet = CSM_C_NO_ERROR;
  BYTE bBus;

  #ifdef WIN32
  printf("CNM_OSEKI__lExit(%02X)\n", bExitMode);
  #endif

  if (bExitMode == CSM_C_EXIT_STOP)
  {
    /* the device will shutdown now, so the NM can be switched OFF here already */
    /* the NM bus states are depending on the NM state and will be set to DOWN  */
  }
  else if( bExitMode == CSM_C_EXIT_OFF)
  {
    /* device is OFF now, nothing left to do                                  */
  }
  else
  {
    lRet = (CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CNM_OSEK, CSM_E_INVALID_PARA));
  }

  return lRet;
}


/************************************************************************
 * function    CNM_OSEKI_bGetStatus
 * \doxydocu
 * \brief      Get OSEKI NM status
 *
 *             Get OSEK NM status (function currently left void,
 *             just implemented to satisfy generic interface)
 *
 * \param[in]  bBus
 *             bus number
 *
 * \return     bStatus
 *             NM status
 *
 * \access     application (?)
 * \reentrant  no
 *
 * \lastreview 12.05.11 Prhl
 *
 * \history_begin
 * 23.11.09 CM-AI/Borck
 * Initial Revision.
 *
 * \history_end
 ************************************************************************/
BYTE CNM_OSEKI_bGetStatus(BYTE bBus)
{
  BYTE bReturn;

  /* currently without any sense !!!                                          */
  switch(bBus)
  {
    case CNM_C_CAN_BUS_0:
    case CNM_C_CAN_BUS_1:
    default:
      bReturn = 0;
      break;
  }
  return bReturn;
}

#endif /* CSM_S_DOXYGEN_HIDE_NON_API_FCT */

/*******************************************************************************
 * function implementation (scope: CAN stack #1 API)
 ******************************************************************************/

/*******************************************************************************
 * function    CSM_OSEKI_lApplCallbackInit
 * \doxydocu
 * \brief      Initial callback function
 *
 *             Initial callback function to register an Application here at NWM.
 *
 * \param[in]  pvHandle
 *             pointer to handle. For future use! May be a deinit() function
 *
 * \param[in]  dwProtocolType (I)
 *             protocol type, see table 2 of CAn SW architecture
 *
 * \param[in]  *pvAddressField (->I)
 *             addressing, dependent on protocol
 *
 * \param[in]  *prCallBackFkt (->I)
 *             pointer to callback function
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR
 *
 * \access     application
 * \reentrant  yes
 *
 * \lastreview 09.05.11 Prhl
 *
 * \history_begin
 * 22.10.10 Borck
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
#if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
LONG CSM_OSEKI_lApplCallbackInit(void * pvHandle, BYTE bBus, DWORD dwProtocolType,
                                 const void * pvAddressField, tCSM_OSEKI_NWM_APPL_CALLBACK * prCallBackFkt)
{
  LONG lRetVal = CSM_C_NO_ERROR;                   // assume NO ERROR
  BYTE bUserNumber;                                // client number from configuration
  WORD wApplID;                                    // client ID passed indirectly by struct

  /* get the client application which wants to register the callback          */
  wApplID = prCallBackFkt->u16ApplID;
  /* calculate the corresponding user number configured in csm_stack_mp.h     */
  bUserNumber = cnm_oseki_bGetUserNumber(bBus, dwProtocolType, pvAddressField, wApplID);

  if (bUserNumber < CSM_NWM_C_MAX_USER)
  {
    /* user number is valid, so store task info in management struct          */
    CSM_M_GET_TASK_CONTEXT( &(csm_nmCallbackFkt[bUserNumber].rTaskInfo));
    /* also store handle if used                                              */
    #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
    csm_nmCallbackFkt[bUserNumber].pvHandle = pvHandle;
    #endif
    /* ensure that a callback ptr field is only initialized once              */
    CSM_vAssert(csm_nmCallbackFkt[bUserNumber].pfvCommunicationInd == NULL);
    csm_nmCallbackFkt[bUserNumber].pfvCommunicationInd = prCallBackFkt->pfvCommunicationInd;
  }
  else
  {
    /* user number out of range */
    lRetVal = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_INVALID_PARA);
    CSM_vAssert(FALSE);
  }
  return lRetVal;
}/*lint !e818 */
#endif /* CSM_S_CALLBACK_INIT_BY_APPLICATION */

/*******************************************************************************
 * function    CSM_OSEKI_lGetCommunicationState
 * \doxydocu
 * \brief      The actual state of the "connection" can be
 *             checked with this function.
 *
 * \see        CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \param[in]  bBus
 *             Bus number, only needed for "broadcast" protocol types
 *
 * \param[in]  dwProtocolType
 *             Protocol type, see corresponding table of CAN SW architecture
 *
 * \param[in]  pvAddressField
 *             Protocol dependend address field
 *
 * \param[out] pbConnectState
 *             CSM_C_CONNECTED     /  CSM_C_STACK_UP
 *             CSM_C_DISCONNECTED  /  CSM_C_STACK_DOWN
 *
 * \param[in]  wApplId
 *             Application Id, only needed for "broadcast" protocol types
 *
 * \return     Error value (wrong Parameter) if < CSM_C_NO_ERROR
 *             else no error
 *
 * \access     all
 * \reentrant  yes
 *
 * \lastreview 09.05.11 Prhl
 *
 * \history_begin
 * 25.10.10 Borck
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
LONG CSM_OSEKI_lGetCommunicationState(BYTE bBus, DWORD dwProtocolType,
                                      const void * pvAddressField,
                                      BYTE * pbState,
                                      WORD wApplId)
{
  LONG lRet = CSM_C_NO_ERROR;         // assume NO ERROR
  BYTE bUserNumber;                   // client number from configuration
  BYTE bArrayIndex;                   // index to target state array
  BYTE bTargetIndicator;              // part of the Addressfield to indicate the target array
  BYTE bState;

  /* calculate the corresponding user number configured in csm_stack_mp.h     */
  /* the user number is directly the index to the config array                */
  bUserNumber = cnm_oseki_bGetUserNumber(bBus, dwProtocolType, pvAddressField, wApplId);

  if (bUserNumber < CSM_NWM_C_MAX_USER)
  {
    /* user number is valid, so continue with getting the requested data      */
    /* by determining the target indicator first (what data to get)           */
    bTargetIndicator = *((((BYTE *)pvAddressField)+0));

    /* by determining the index for the corresponding monitor array           */
    bArrayIndex = cnm_oseki_bGetMonitorNodeIndex(bUserNumber);

    /* now we can get the requested data by accessing the correct monitor     */
    bState = cnm_oseki_bGetMonitorState(bBus, bTargetIndicator, bArrayIndex);

    /* evaluate the monitor state                                             */
    if (bState != CNM_OSEKI_C_STATE_ERROR)
    {
      * pbState = bState;
    }
    else
    {
      lRet = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_C_INVALID_PARA);
    }
  }
  else
  {
    lRet = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM, CSM_C_INVALID_PARA);
  }
  return lRet;
}

#if (CSM_S_REMOTE_STACK == CSM_C_F_ON)
/*******************************************************************************
 * function    CSM_OSEKI_vPROXYCommunicationInd
 * \doxydocu
 * \brief      Callback function from PROXY to trigger a communication indication
 *
 * \param[in]  bUserNumber
 *                 Data
 *
 * \return     void
 *
 * \access     CSM task level
 * \reentrant  no, but only call from interrupt
 *
 * \lastreview
 *
 * \history_begin
 *
 * 02.06.15 Prhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void CSM_OSEKI_vPROXYCommunicationInd(BYTE bBus, DWORD dwProtocolType, void *pvAddressField, BYTE bAction, WORD wApplId)
{
  BYTE bUserNumber;                   // client number from configuration
  BYTE bArrayIndex;                   // index to target state array
  BYTE bTargetIndicator;              // part of the Addressfield to indicate the target array
  BYTE bMonitorComponent;             // the component to which the CAN msg is belonging

  // user number is the index within the configuration table
  bUserNumber = cnm_oseki_bGetUserNumber(bBus, dwProtocolType, pvAddressField, wApplId);

  if (bUserNumber < CSM_NWM_C_MAX_USER)
  {
    // for a valid vonfig entry we now need the index within the TX/RX/BUSOff monitor array (monitor index)
    bArrayIndex = cnm_oseki_bGetMonitorNodeIndex(bUserNumber);
    // the target indicator specifies if RX, TX or BusOff monitor is meant
    bTargetIndicator  = *((((BYTE *)pvAddressField)+0));
    bMonitorComponent = *((((BYTE *)pvAddressField)+1));

    // now trace the specific interpretation of the entire Communication Indication
    #if (CSM_S_TRACE == CSM_C_F_ON)
      #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
        #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
        switch(bTargetIndicator)
        {
          case CSM_C_ADR_FLD_DC_NM_RX_INDICATOR:
            ETG_TRACE_USR3(("CSM_OSEKI_vPROXYCommunicationInd: Bus:%x  Protocol:%x  Type:%x  Component:%x  State:%x",
                            ETG_CENUM(tCSM_BUS_NUMBER,            (tU8)  bBus),
                            ETG_CENUM(tCSM_PROTOCOL_TYPE,         (tU32) dwProtocolType),
                            ETG_CENUM(tCSM_ADDRESS_FIELD_0_OSEKI, (tU8)  bTargetIndicator),
                            ETG_CENUM(tCSM_ADDRESS_FIELD_1_OSEKI, (tU8)  bMonitorComponent),
                            ETG_CENUM(tCNM_OSEKI_RX_ERROR_STATE,  (tU8)  bAction)
                          ));
            break;
          case CSM_C_ADR_FLD_DC_NM_TX_INDICATOR:
            ETG_TRACE_USR3(("CSM_OSEKI_vPROXYCommunicationInd: Bus:%x  Protocol:%x  Type:%x  Component:%x  State:%x",
                            ETG_CENUM(tCSM_BUS_NUMBER,            (tU8)  bBus),
                            ETG_CENUM(tCSM_PROTOCOL_TYPE,         (tU32) dwProtocolType),
                            ETG_CENUM(tCSM_ADDRESS_FIELD_0_OSEKI, (tU8)  bTargetIndicator),
                            ETG_CENUM(tCSM_ADDRESS_FIELD_1_OSEKI, (tU8)  bMonitorComponent),
                            ETG_CENUM(tCNM_OSEKI_TX_ERROR_STATE,  (tU8)  bAction)
                          ));
            break;
          case CSM_C_ADR_FLD_DC_NM_BUSOFF_INDICATOR:
            ETG_TRACE_USR3(("CSM_OSEKI_vPROXYCommunicationInd: Bus:%x  Protocol:%x  Type:%x  Component:%x  State:%x",
                            ETG_CENUM(tCSM_BUS_NUMBER,            (tU8)  bBus),
                            ETG_CENUM(tCSM_PROTOCOL_TYPE,         (tU32) dwProtocolType),
                            ETG_CENUM(tCSM_ADDRESS_FIELD_0_OSEKI, (tU8)  bTargetIndicator),
                            ETG_CENUM(tCSM_ADDRESS_FIELD_1_OSEKI, (tU8)  bMonitorComponent),
                            ETG_CENUM(tCNM_OSEKI_BUS_ERROR_STATE, (tU8)  bAction)
                          ));
            break;
          default:
            ETG_TRACE_FATAL(("CSM_OSEKI_vPROXYCommunicationInd: INVALID target Indicator"));
            break;
        }
        #endif
      #endif
    #endif

    cnm_oseki_vSetMonitorState(bBus, bTargetIndicator, bArrayIndex, bAction);
    csm_vTriggerCsmUpstreamTask(CSM_C_IND_REASON_CSM_OSEKI_NM_STATE_CHANGED);
  }
  else
  {
    #if (CSM_S_ETG_SHORT_OUTPUT == CSM_C_F_OFF)
    ETG_TRACE_FATAL(("!(bUserNumber < CSM_NWM_C_MAX_USER) bUserNumber = %02x!", bUserNumber ));
    #else
    ETG_TRACE_FATAL(("%04x%02x", CSM_C_ETG_FKT_CSM_LCTR_VPROXYCOMMUNICATIONIND, bUserNumber));
    #endif
  }
}
#endif //CSM_S_REMOTE_STACK == CSM_C_F_ON


/*******************************************************************************
 * function    CSM_OSEKI_vProcessCommunicationInd
 * \doxydocu
 * \brief      Handle communication indication.
 *
 *             Handle communication indication.
 *             This function is called in upstream-task after trigger with CSM_C_UP_REASON_CSM_OSEK_NM_STATE_CHANGED.
 *
 * \param[in]  VOID
 *
 * \return     VOID
 *
 * \access     Upstream-Task
 * \reentrant  no, but only called by Upstream-Task
 *
 * \lastreview 09.05.11 Prhl
 *
 * \history_begin
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
VOID CSM_OSEKI_vProcessCommunicationInd(VOID)
{
  #if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
  WORD wCount;
  BYTE bTargetIndex;
  BYTE bTargetIndicator;
  BYTE bState;
  BYTE bBus;
  BYTE bStateChanged;

  /* loop over all possible client callback functions                         */
  for (wCount = 0; wCount < CSM_NWM_C_MAX_USER; wCount++)
  {
    if (csm_nmCallbackFkt[wCount].pfvCommunicationInd != NULL)
    {
      /* callback is available (registered by client)                         */
      /* get the bus of the configured array entry first                      */
      bBus             = csm_arNWMCfg[wCount].bBus;
      /* by determining the target indicator first (what data to get)         */
      bTargetIndicator = csm_arNWMCfg[wCount].au8AddressField[0];

      /* by determining the index for the corresponding monitor array         */
      bTargetIndex     = csm_arNWMCfg[wCount].bMonitorIndex;

      /* now we can get the requested data by accessing the correct monitor     */
      bState           = cnm_oseki_bGetMonitorState(bBus, bTargetIndicator, bTargetIndex);
      bStateChanged    = cnm_oseki_bGetResetMonitorStateChange(bBus, bTargetIndicator, bTargetIndex);

      /* check if the state is valid                                          */
      if (bState != CNM_OSEKI_C_STATE_ERROR)
      {
        if (bStateChanged != 0x00)
        {
          /*************** Tracing ******/
          #if (CSM_S_TRACE == CSM_C_F_ON)
            #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
              #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
              {
                ETG_TRACE_USR3(("ERROR Indication       --> Bus: %x Protocol: %x Indicator: %x Node: [%x] State: %x ApplID: %x",
                ETG_CENUM(tCSM_BUS_NUMBER              , (tU8)  csm_arNWMCfg[wCount].bBus),
                ETG_CENUM(tCSM_PROTOCOL_TYPE           , (tU32) csm_arNWMCfg[wCount].dwProtocolType),
                ETG_CENUM(tCSM_ADDRESS_FIELD_0_OSEKI   , (tU8)  csm_arNWMCfg[wCount].au8AddressField[0]),
                ETG_CENUM(tCSM_ADDRESS_FIELD_1_OSEKI   , (tU8)  csm_arNWMCfg[wCount].au8AddressField[1]),
                ETG_CENUM(tCSM_OSEKI_ERROR_STATE       , (tU8)  bState),
                ETG_CENUM(tCSM_APPL_ID                 , (tU16) csm_arNWMCfg[wCount].wApplId) ));
              }
              #endif
            #else
            {
              /* trace out the registered data related to the callback        */
              BYTE abData[10];

              abData[ 0] = csm_arNWMCfg[wCount].bBus;
              abData[ 1] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(csm_arNWMCfg[wCount].dwProtocolType));
              abData[ 2] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(csm_arNWMCfg[wCount].dwProtocolType));
              abData[ 3] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(csm_arNWMCfg[wCount].dwProtocolType));
              abData[ 4] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(csm_arNWMCfg[wCount].dwProtocolType));
              abData[ 5] = csm_arNWMCfg[wCount].au8AddressField[0];
              abData[ 6] = csm_arNWMCfg[wCount].au8AddressField[1];
              abData[ 7] = bState;
              abData[ 8] = CSM_M_GET_HIBYTE(csm_arNWMCfg[wCount].wApplId);
              abData[ 9] = CSM_M_GET_LOBYTE(csm_arNWMCfg[wCount].wApplId);

              CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_COMMUNICATION_IND, abData , 10);
            }
            #endif
          #endif /* CSM_S_TRACE */

          #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
          /* set the appropriate task context                                 */
          CSM_M_SET_TASK_CONTEXT( &(csm_nmCallbackFkt[wCount].rTaskInfo),
                                  csm_nmCallbackFkt[wCount].pfvCommunicationInd(csm_nmCallbackFkt[wCount].pvHandle,
                                  bBus,
                                  csm_arNWMCfg[wCount].dwProtocolType,
                                  csm_arNWMCfg[wCount].au8AddressField,
                                  bState,
                                  csm_arNWMCfg[ wCount].wApplId) );
          #else
          /* directly use the registered callback to inform client            */
          csm_nmCallbackFkt[wCount].pfvCommunicationInd( bBus,
                                                         csm_arNWMCfg[ wCount].dwProtocolType,
                                                         csm_arNWMCfg[ wCount].au8AddressField,
                                                         bState,
                                                         csm_arNWMCfg[ wCount].wApplId);
          #endif
        }
      }
    }
    else
    {
      #if 0 // BKA2HI: enable this section if you want to trace the registered ITC indications
      #if (CSM_S_TRACE == CSM_C_F_ON)
        #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
          #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
          {
            ETG_TRACE_USR4(("ERROR Indication - No Callback registered for Bus:%x  Prot:%x  Indicator:%x  Node:[%x]",
                            ETG_CENUM(tCSM_BUS_NUMBER              , (tU8)  csm_arNWMCfg[wCount].bBus),
                            ETG_CENUM(tCSM_PROTOCOL_TYPE           , (tU32) csm_arNWMCfg[wCount].dwProtocolType),
                            ETG_CENUM(tCSM_ADDRESS_FIELD_0_OSEKI   , (tU8)  csm_arNWMCfg[wCount].au8AddressField[0]),
                            ETG_CENUM(tCSM_ADDRESS_FIELD_1_OSEKI   , (tU8)  csm_arNWMCfg[wCount].au8AddressField[1])
                          ));
          }
          #endif
        #endif
      #endif
      #endif

      /* reset the changed flag even if no client has registered              */
      /* determine, what to get where and then get it                         */
      bBus             = csm_arNWMCfg[wCount].bBus;
      bTargetIndicator = csm_arNWMCfg[wCount].au8AddressField[0];
      bTargetIndex     = csm_arNWMCfg[wCount].bMonitorIndex;
      /* reset monitor state                                                  */
      bStateChanged = cnm_oseki_bGetResetMonitorStateChange(bBus, bTargetIndicator, bTargetIndex);
    }
  }
  #endif /* CSM_S_CALLBACK_INIT_BY_APPLICATION */
}

/*******************************************************************************
 * function    CNM_OSEKI_lCommunicationReq
 * \doxydocu
 * \brief      Used to pass commands from Middleware components to CSM CNM module.
 *
 *
 * \param      none
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR
 *
 * \access     application
 * \reentrant  yes
 *
 * \lastreview 09.05.11 Prhl
 *
 * \history_begin
 * 22.10.10 Borck
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
VOID CNM_OSEKI_vProcessCommunicationReq(VOID)
{
  BYTE bCount;
  BYTE bBus;

  /* loop over all configured command clients                                 */
  for (bCount = 0; bCount < CSM_NWM_C_MAX_COMMAND_USER; bCount++)
  {
    if (cnm_oseki_arCommunicationState[bCount].bCommunicationReq == TRUE)
    {
      /* pending request detected, clear the marker                           */
      cnm_oseki_arCommunicationState[bCount].bCommunicationReq = FALSE;

      /* evaluate given command                                               */
      switch (cnm_oseki_arCommunicationState[bCount].bReqAction)
      {
        case CNM_OSEKI_CMD_REINIT_MONITOR:
          /* reInit ALL monitors                                              */
          for (bBus= 0; bBus < CNM_OSEKI_ANZAHL_BUSSE; bBus++)
          {
            CNM_OSEKI_vReInitMonitoringStates(bBus);
          }
          /* clear the command here, so that invalid commands remain in the var */
          cnm_oseki_arCommunicationState[bCount].bReqAction = CNM_OSEKI_CMD_NO_CMD;
          break;

        default:
          break;
      }
    }
  }
}

/*******************************************************************************
 * function    CNM_OSEKI_lCommunicationReq
 * \doxydocu
 * \brief      Used to pass commands from Middleware components to CSM CNM module.
 *
 *
 * \param[in]  bBus (I)
 *             Bus index for which the function is called
 *
 * \param[in]  dwProtocolType (I)
 *             protocol type, see table 2 of CAn SW architecture
 *
 * \param[in]  *pvAddressField (->I)
 *             addressing, dependent on protocol
 *
 * \param[in]  bAction
 *             pointer to callback function
 *
 * \param[in]  wApplID
 *             Application ID
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR
 *
 * \access     application
 * \reentrant  yes
 *
 * \lastreview 09.05.11 Prhl
 *
 * \history_begin
 * 22.10.10 Borck
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
LONG CNM_OSEKI_lCommunicationReq(BYTE bBus, DWORD dwProtocolType, const void * pvAddressField,
                                 BYTE bAction, WORD wApplID)
{
  LONG lReturnValue = CSM_C_NO_ERROR;
  BYTE bUserNumber;
  BYTE bUser;

  /* calculate the corresponding user number configured in csm_stack_mp.h     */
  bUserNumber = cnm_oseki_bGetUserNumber(bBus, dwProtocolType, pvAddressField, wApplID);

  if (bUserNumber < CSM_NWM_C_MAX_USER)
  {
    /* a valid entry with matching params has been found in table             */
    /* get the index to the communication state array                         */
    bUser = csm_arNWMCfg[bUserNumber].bMonitorIndex;
  }
  else
  {
    /* user number out of range */
    lReturnValue = CSM_M_MAKE_CSM_ERROR(CSM_C_ERR_ERROR, CSM_E_INVALID_PARA);
    CSM_vAssert(FALSE);
    return lReturnValue;
  }

  if (cnm_oseki_arCommunicationState[bUser].bReqAction != bAction)
  {
    /* store command in struct and trigger downstream task for later handling */
    /* all other given params are now lost forever, since the current         */
    /* requirements only are: clear and reset the entire MONITORING           */
    cnm_oseki_arCommunicationState[bUser].bReqAction = bAction;
    cnm_oseki_arCommunicationState[bUser].bCommunicationReq = TRUE;
    csm_vTriggerCsmDownstreamTask(CSM_C_DOWN_REASON_CNM_OSEKI_COMMUNICATION_REQ);
  }
  else
  {
    /* command already in progress, so do nothing                             */
  }

  return lReturnValue;
}

/*******************************************************************************
 * function    CNM_OSEKI_vReInitMonitoringStates
 * \doxydocu
 * \brief      internal function to Re-Init the monitoring states on demand
 *             for example triggered by the station management
 *
 * \param[in]  none
 *
 * \return     void
 *
 * \access     task
 * \reentrant  no
 *
 * \lastreview 12.05.11 Prhl
 *
 * \history_begin
 * 28.04.10 Borck
 * Initial Revision.
 * \history_end
 ******************************************************************************/
void CNM_OSEKI_vReInitMonitoringStates(BYTE bBus)
{
  BYTE bTemp;
  BYTE bChanged = 0;
  BYTE bNrRxNodes;
  BYTE bNrTxNodes;

  BYTE bRxMonitoringState;
  BYTE bTxMonitoringState;
  BYTE bBoMonitoringState;

  #if (CSM_S_TRACE == CSM_C_F_ON)
    #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR4(("CNM_OSEKI_vReInitMonitoringStates: called"));
      #endif
    #endif
  #endif

  /* avoid any action with invalid bus param                                  */
  if (bBus >= CNM_OSEKI_ANZAHL_BUSSE)
  {
    return;
  }

  /* get the max values for the loops below from bus config                   */
  bNrRxNodes = CNM_OSEKI_arBusConfig[bBus].bAmountRxNodes;
  bNrTxNodes = CNM_OSEKI_arBusConfig[bBus].bAmountTxNodes;

  /* reset RX counters and states for all nodes                               */
  for (bTemp = 0; bTemp < bNrRxNodes; bTemp ++)
  {
    /* since there might be unused busses, check the pointer first            */
    if (CNM_OSEKI_arState[bBus].pRxStates != NULL)
    {
      // simply reset the state and wait, until the V850 states will come over INC
      CNM_OSEKI_arState[bBus].pRxStates[bTemp].fRxNodeTimeout = FALSE;
      CNM_OSEKI_arState[bBus].pRxStates[bTemp].wRxNodeCounter = 0;
      CNM_OSEKI_arState[bBus].pRxStates[bTemp].bRxNodeAbsentState = CNM_OSEKI_C_STATE_NODE_PRESENT;
      CNM_OSEKI_arState[bBus].pRxStates[bTemp].bStateChanged = 0;
    }
  }

  /* reset TX counters and states for all nodes                               */
  for (bTemp = 0; bTemp < bNrTxNodes; bTemp ++)
  {
    /* since there might be unused busses, check the pointer first            */
    if (CNM_OSEKI_arState[bBus].pTxStates != NULL)
    {
      // simply reset the state and wait, until the V850 states will come over INC
      CNM_OSEKI_arState[bBus].pTxStates[bTemp].fTxMuteTimeout = FALSE;
      CNM_OSEKI_arState[bBus].pTxStates[bTemp].wTxMuteCounter = 0;
      CNM_OSEKI_arState[bBus].pTxStates[bTemp].bTxNodeMuteState = CNM_OSEKI_C_STATE_NODE_NOT_MUTED;
      CNM_OSEKI_arState[bBus].pTxStates[bTemp].bStateChanged = 0;
    }
  }

  /* reset BusOff counters and states                                         */
  if (CNM_OSEKI_arState[bBus].pBusOffStates != NULL)
  {
    // simply reset the state and wait, until the V850 states will come over INC
    CNM_OSEKI_arState[bBus].pBusOffStates->fBusOffTimeout = FALSE;
    CNM_OSEKI_arState[bBus].pBusOffStates->wBusOffCounter = 0;
    CNM_OSEKI_arState[bBus].pBusOffStates->bBusOffState = CNM_OSEKI_C_STATE_BUS_ON;
    CNM_OSEKI_arState[bBus].pBusOffStates->bStateChanged = 0;
  }
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/
/******************************************************************************/
/******************************************************************************/
/* Entry Function for OFF state on MCAN */
#undef CNM_OSEKI_SELF
#undef CSM_CAN_STACK

/*******************************************************************************
 * \doxydocu
 * \file          cnm_oseki.c
 * \brief         OSEK NetworkManagement indirect
 *                OSEK NetworkManagement indirect based on OSEK NM 2.53
 *
 * \see           CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \sw_component  CAN stack
 * \project       Pool module
 * \path          /di_can/modules_swa/cnm/src/
 *
 * \ingroup       CNM
 * \ingroup       CNM_OSEK
 *
 * \authors       V-Team
 *
 * COPYRIGHT      (c) 2015 Bosch Car Multimedia GmbH
 *
 * \history_begin
 *
 * 15.05.15  main\rn_aivi\5  Borck
 * - function to get SLOT from NVM introduced.
 *
 * 20.05.15  main\rn_aivi_proxy\1  M. Prhl
 * - merge from main\rn_aivi\5
 *
 * 01.06.15  main\rn_aivi_proxy\2  M. Prhl
 * - unused stuff removed.
 *
 * 02.06.15  main\rn_aivi_proxy\3  M. Prhl
 * - CSM_OSEKI_vPROXYCommunicationInd() and cnm_oseki_vSetMonitorState() added
 *
 * 19.06.15  main\rn_aivi_proxy\4  Borck
 * - enum for NWM address field renamed
 * - ETG trace output for PROXY extendend for the CommunicationInd (Monitor) path
 *
 * 09.07.15  main\rn_aivi_proxy\5  Borck
 * - appl related stuff moved to csm_apl.c
 *
 * 16.07.15  main\rn_aivi_proxy\6  Borck
 * - unneeded stuff (state machine) removed
 * - required modifications for SLOT configuration added and re-activated
 * - local ReInitMonitoringStates added to avoid temporary inconsistence while waiting for V850 monitor states
 *
 * 06.09.16  main\rn_aivi_proxy\7  Borck
 * - dedicated trace class for this module introduced
 *
 * 30.11.16  main\rn_aivi_proxy\8  Borck
 * - added but disabled a trace output to observe the monitoring registrations of Vehicle Data (can be enabled on demand)
 *
 * 06.09.17  main\rn_aivi_proxy\9  Borck
 * - added NULL pointer check for AddressField to fix coverity finding
 *
 * \history_end
 *//**** END OF FILE **********************************************************/
