// Description and history of cbr.c at the end of the file.
// File name for preprozessor commands
#define CBR_SELF
// cbr.c is a file according V-team CAN-SW architecture implementation rules.
#define CSM_CAN_STACK

/**
 * \defgroup      CBR   CBR (Interface #4)
 *                This group contains all files which upper interface
 *                belongs to CSM interface #4.
 */

//*******************************************************************************
//* includes:
//*   1)system- and project- includes
//*   2)needed interfaces from external components
//*...3)internal and external interfaces from this component
//*******************************************************************************

//*************** CSM and stack **************
#include "csm_stack_i.h"
#include "csm_stack_m.h"

#if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
//*************** CDA *************
 #include "cda_i.h"
#else
 #include "csm_proxy_i.h"
#endif

//*************** CBR **************
#include "cbr_i.h"
#include "cbr_m.h"
#include "cbr_c.h"

//*************** CSM APL **********
#include "csm_apl.h"

//*************** ETG Tracing ******
#if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #define ETRACE_S_IMPORT_INTERFACE_GENERIC
  #include "etrace_if.h"

  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
    #include "csm_etg_trace.h"
    #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FD_CSM
    // #define ETG_INLINE
    #include "trcGenProj/Header/cbr.c.trc.h"
  #endif
  #endif
#endif // CSM_S_TRACE == CSM_C_F_ON

//*****************************************************************
//* check of all feature switches which affect the CBR :
//* The following #defines should be declared in csm_stack_p.h:
//*****************************************************************
#ifndef CSM_S_TRACE
  #error "CSM_S_TRACE must be set to CSM_C_F_ON or -OFF !"
#endif

#if (CSM_S_TRACE == CSM_C_F_ON)
  #ifndef CSM_S_TRACE_LIMITED_SIGNAL
    #error "CSM_S_TRACE_LIMITED_SIGNAL must be set to CSM_C_F_ON or -OFF !"
  #endif
#endif

#ifndef CSM_S_REMOTE_STACK
  #error "CSM_S_REMOTE_STACK must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_CBR_LITTLE_ENDIAN_SIGNALS
  #error "CSM_S_CBR_LITTLE_ENDIAN_SIGNALS must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_CBR_BIG_ENDIAN_SIGNALS
  #error "CSM_S_CBR_BIG_ENDIAN_SIGNALS must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_CBR_BOTH_ENDIANS_SIGNALS
  #error "CSM_S_CBR_BOTH_ENDIANS_SIGNALS must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_TIMEOUT_SIGNAL_VALUES
  #error "CSM_S_TIMEOUT_SIGNAL_VALUES must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_CBR_QUEUED_SIGNALS_USED
  #error "CSM_S_CBR_QUEUED_SIGNALS_USED must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_CBR_RX_TIMEOUT_GROUPS
  #error "CSM_S_CBR_RX_TIMEOUT_GROUPS must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_CBR_MIN_DELAY
  #error "CSM_S_CBR_MIN_DELAY must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_LITTLE_ENDIAN_CPU
  #error "CSM_S_LITTLE_ENDIAN_CPU must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_BIG_ENDIAN_CPU
  #error "CSM_S_BIG_ENDIAN_CPU must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_CBR_TX_IF_APPL_ACTIVE
  #error "CSM_S_CBR_TX_IF_APPL_ACTIVE must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_DEFAULT_SIGNAL_VALUES
  #error "CSM_S_DEFAULT_SIGNAL_VALUES must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_CBR_RESYNCHRONIZED_CYCLE
  #error "CSM_S_CBR_RESYNCHRONIZED_CYCLE must be set to CSM_C_F_ON or -OFF !"
#endif
#ifndef CSM_S_CBR_USES_DIRECT_CALLS
  #error "CSM_S_CBR_USES_DIRECT_CALLS must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_CBR_TIMEOUT_DEFAULT_LAST_VALUE
  #error "CSM_S_CBR_TIMEOUT_DEFAULT_LAST_VALUE must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_CCS_USED
  #error "CSM_S_CCS_USED must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_CBR_USES_COMMUNICATION_REQ
  #error "CSM_S_CBR_USES_COMMUNICATION_REQ must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_REQUESTABLE_TIMER
  #error "CSM_S_REQUESTABLE_TIMER must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_CALLBACK_INIT_BY_APPLICATION
  #error "CSM_S_CALLBACK_INIT_BY_APPLICATION must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_CSM_VW
  #error "CSM_S_CSM_VW must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_CBR_DISCARD_MSG_SHORTER_DLC
  #error "CSM_S_CBR_DISCARD_MSG_SHORTER_DLC must be set to CSM_C_F_OFF or -ON !"
#endif

#ifndef CSM_S_CBR_TX_MSG_ROLL_COUNTER
  #error "CSM_S_CBR_TX_MSG_ROLL_COUNTER must be set to CSM_C_F_ON or -OFF !"
#endif

#if (CSM_S_CBR_TX_MSG_ROLL_COUNTER == CSM_C_F_ON)
#ifndef CSM_S_CBR_TX_MSG_ROLL_COUNTER_GROUPS
  #error "CSM_S_CBR_TX_MSG_ROLL_COUNTER_GROUPS must be set to CSM_C_F_ON or -OFF !"
#endif
#endif

#ifndef CSM_S_CBR_TX_MSG_CRC
  #error "CSM_S_CBR_TX_MSG_CRC must be set to CSM_C_F_ON or -OFF !"
#endif

#if (CSM_S_CBR_TX_MSG_CRC == CSM_C_F_ON)
#ifndef CSM_S_CBR_TX_MSG_CRC_TYPE_RN
  #error "CSM_S_CBR_TX_MSG_CRC_TYPE_RN must be set to CSM_C_F_ON or -OFF !"
#endif
#endif

#ifndef CSM_S_CBR_RX_MSG_ROLL_COUNTER
  #error "CSM_S_CBR_RX_MSG_ROLL_COUNTER must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_CBR_RX_MSG_CRC
  #error "CSM_S_CBR_RX_MSG_CRC must be set to CSM_C_F_ON or -OFF !"
#endif

#ifndef CSM_S_CBR_GATEWAY
  #error "CSM_S_CBR_GATEWAY must be set to CSM_C_F_ON or -OFF !"
#endif

//*****************************************************************
//* defines, macros (scope: modul-local)
//*****************************************************************
#define CBR_STATUS_TX_RUNNING   0x01U
#define CBR_STATUS_RX_RUNNING   0x02U
#if (CSM_S_CCS_USED == CSM_C_F_ON)
  #define CBR_STATUS_RX_RETRIGGER 0x04U
#endif

#if (CSM_C_MAX_CAN_BUS > 4)
  #error "The following code supports only 4 busses !"
#endif
#define CBR_SET_STATUS(bBus, bStatus) cbr_dwStatus |= ((bStatus & 0x0F) << (bBus*4));
#define CBR_CLR_STATUS(bBus, bStatus) cbr_dwStatus &= (DWORD)~((bStatus & 0x0F) << (bBus*4));
#define CBR_IS_STATUS(bBus, bStatus) ((cbr_dwStatus & ((bStatus & 0x0F) << (bBus*4))) > 0)

//****************************************************************
//* typedefs (scope: modul-local)
//****************************************************************
// none

//****************************************************************
//* variable defintion (scope: global)
//****************************************************************
// none

//****************************************************************
//* variable defintion (scope: modul-local)
//****************************************************************
// BKA2HI: Nissan Renault AIVI projects require a variable amount of RX/TX messages dependent on a configuration item called "SLOT".
// The entire CSM configuration is tool based generated for these projects.
// This all results in the requirement always to define the max. number of buffers, state arrays etc. that can occur.
// To unambiguously handle this, the following measures have been applied:
// For Renault/Nissan with variable amount of RX/TX messages:
// - the old defines used here as "array dimension" have been replaced by new defines which cover the maximum of a variable size.
// - for other occurences, the original defines have been mapped to 2 variables which are handled under the feature "variant management"
// For all other projects with a fixed amount RX/TX messages:
// - the only thing you have to do is to map CBR_TX_ABSOLUT_MAX_NUMBER to CBR_TX_MAX_NUMBER
//                                       and CBR_RX_ABSOLUT_MAX_NUMBER to CBR_RX_MAX_NUMBER
CSM_STATIC tCBR_TX_TELEGRAM_STATE CBR_arTxTelegramState[CBR_TX_ABSOLUT_MAX_NUMBER]; /**< This area stores all state values for each transmit telegram */
CSM_STATIC tCBR_RX_TELEGRAM_STATE CBR_arRxTelegramState[CBR_RX_ABSOLUT_MAX_NUMBER]; /**< This area stores all state values for each receive telegram*/

// each byte of this long word stores the states for one bus !!!
CSM_STATIC DWORD cbr_dwStatus;

// New CBR mechanism to handle a variable amount of RX/TX messages at runtime, based on the SLOT
#if ((CSM_S_VARIANTS_MANAGEMENT == CSM_C_F_ON) && (CSM_S_SLOT_BASED_CONFIG == CSM_C_F_ON))
// the new variables to be used are defined in cbr_mp.h and initialized in csm_apl.c
// BYTE  bCbrRxMaxNumber; // do NOT uncomment this line - its only to show the used names
// BYTE  bCbrTxMaxNumber; // do NOT uncomment this line - its only to show the used names
#endif

// Rolling counters for TX - 1 for each telegram
#if (CSM_S_CBR_TX_MSG_ROLL_COUNTER == CSM_C_F_ON)
CSM_STATIC CSM_TX_ROLLCOUNTER_ADMIN cbr_aTxRollingCounterAdmin[CBR_TX_ABSOLUT_MAX_NUMBER];
#endif

// Rolling counters for RX - 1 for each telegram
#if (CSM_S_CBR_RX_MSG_ROLL_COUNTER == CSM_C_F_ON)
CSM_STATIC BYTE cbr_abRxRollingCounter[CBR_RX_ABSOLUT_MAX_NUMBER];
#endif

//*****************************************************************************
//* variable Cbr_u8RxGrpMask
//*
//* comment: flag mask to disable time out checking for 8 different
//*          groups of rx-messages
//*
//* range: TRUE, FALSE for every bit
//*
//* access level (loop / interrupt): Can task
//*****************************************************************************
#if (CSM_S_CBR_RX_TIMEOUT_GROUPS == CSM_C_F_ON)
static BYTE cbr_bRxGrpMask;
#endif // CSM_S_CBR_RX_TIMEOUT_GROUPS

//*****************************************************************************
//* variable Cbr_wRxTimeOutDelay
//*
//* comment: During this period of time, the timeout supervision is extended.
//*****************************************************************************
#ifdef CBR_RXTIMEOUT_DELAY
static WORD cbr_wRxTimeOutDelay;
#define CBR_EXTEND_TO_DELAY(a) ((a)+(cbr_wRxTimeOutDelay))
#else
#define CBR_EXTEND_TO_DELAY(a) (a)
#endif // CBR_RXTIMEOUT_DELAY

//*****************************************************************
//* function prototype (scope: modul-local)
//*****************************************************************
#if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
  static void CBR_vStartTransmitRepeatTimer(BYTE bTelegramIndex);

  #if (CSM_S_CBR_MIN_DELAY == CSM_C_F_ON)
  static void CBR_vStartTransmitMinDelayTimer(BYTE bTelegramIndex);
  #endif // CSM_S_CBR_MIN_DELAY

  static void CBR_vStartReceiveTimer(BYTE bTelegramIndex);
  static void CBR_vStopTransmitRepeatTimer(BYTE bTelegramIndex);

  #if (CSM_S_CBR_MIN_DELAY == CSM_C_F_ON)
  static void CBR_vStopTransmitMinDelayTimer(BYTE bTelegramIndex);
  #endif // CSM_S_CBR_MIN_DELAY

  static void CBR_vStopReceiveTimer(BYTE bTelegramIndex);
#endif  // CSM_S_REMOTE_STACK

#if (CSM_S_CBR_TX_MSG_ROLL_COUNTER == CSM_C_F_ON)
static void cbr_vHandleRollingCounter(BYTE bIndex);
#endif

#if (CSM_S_CBR_TX_MSG_CRC == CSM_C_F_ON)
static void cbr_vHandleTxCRC(BYTE bTelegramIndex);
#endif

#if (CSM_S_CBR_RX_MSG_ROLL_COUNTER == CSM_C_F_ON)
static BOOL cbr_fCheckRxRollCounter(BYTE bTelegramIndex, const BYTE *pbData);
#endif

#if (CSM_S_CBR_RX_MSG_CRC == CSM_C_F_ON)
BOOL cbr_fCheckRxCRC(BYTE bTelegramIndex, const BYTE *pbData, BYTE bDataLength);
#endif

static void cbr_vReceiveInd(const CSM_CAN_RX_DATA * const prData, const DWORD dwSignalStatus);
static BOOL cbr_fIsTxRequiredAfterSignalWrite(const BYTE * pabTempData, const CSM_CBR_TX * prCbrTxCfg, tCBR_TX_TELEGRAM_STATE * pCbrTxTelegramState, const CSM_SIGNAL * prCsmSignal, BYTE bTxType, BYTE bTelIndex, BYTE bSignalIndex);

//*****************************************************************
//* constants (scope: modul-local)
//*****************************************************************
static const BYTE cbr_abSMask[8] = { 0xFF, 0x01, 0x03, 0x07, 0x0F, 0x1F, 0x3F, 0x7F }; /**< This area is use to mask signals from telegram */
static const BYTE cbr_abRMask[8] = { 0x00, 0x01, 0x03, 0x07, 0x0F, 0x1F, 0x3F, 0x7F }; /**< This area is use to mask signals from telegram */
static const BYTE cbr_abLMask[8] = { 0xFE, 0xFC, 0xF8, 0xF0, 0xE0, 0xC0, 0x80, 0x00 }; /**< This area is use to mask signals from telegram */

//*****************************************************************
//* function implementation (scope: modul-local)
//*****************************************************************
#define  cbr_vSetIndicationFlag(Tel,Sig)  CBR_arRxTelegramState[Tel].abIndicate[(Sig)>>3] |= (1U << ((Sig)%8));
#define  cbr_vClrIndicationFlag(Tel,Sig)  CBR_arRxTelegramState[Tel].abIndicate[(Sig)>>3] &= ~(1U << ((Sig)%8));
#define  cbr_fGetIndicationFlag(Tel,Sig)  ((CBR_arRxTelegramState[Tel].abIndicate[(Sig)>>3] & (1U << ((Sig)%8))) != 0)

#if  (CSM_S_CBR_TX_IF_APPL_ACTIVE == CSM_C_F_ON)
  #define  cbr_vSetActiveFlag(f, Sig)     f |= (1 << Sig);
  #define  cbr_vClrActiveFlag(f,Sig)      f &= ~(1 << Sig);
  #define  cbr_fIsFunctionActive(f)       (f != 0)
#endif // CSM_S_CBR_TX_IF_APPL_ACTIVE

/************************************************************************
 * function    cbr_vCheckConfig
 * \doxydocu
 * \brief      Performs some basic integrity checks of the CBR message and
 *             signal config. This is used for development.
 *
 * \lastreview  16.04.08 VTeam-Pröhl
 *
 * \history_begin
 * 31.07.07 CM-DI/ESA3-Looby
 * - Added #ifdef for CSM_S_CBR_BIG_ENDIAN_SIGNALS, DLC consistency check, for Tx.
 * 08.11.06 CM-DI/ESA3-Battis
 * Initial Revision.
 *
 * \history_end
 ************************************************************************/
static void cbr_vCheckConfig(void)
{
  BYTE bTelCount, bSigCount;
  const CSM_TELEGRAM * prCsmTelegram;      // pointer to table of pointers
  const CSM_SIGNAL * pCsmSignal;           // pointer to table of pointers

  // ensure that the variable mapped by the makro has been initialized already !!!
  CSM_vAssert(CBR_RX_MAX_NUMBER != 0);
  CSM_vAssert(CBR_TX_MAX_NUMBER != 0);

  // ******* RX ************************************************************************************
  for (bTelCount = 0; bTelCount < CBR_RX_MAX_NUMBER; bTelCount++)
  {
    CSM_vAssert(csm_arCbrRxTelegram[bTelCount].bDlc > 0);
    CSM_vAssert(csm_arCbrRxTelegram[bTelCount].dwCanId > 0);
    CSM_vAssert(csm_arCbrRxTelegram[bTelCount].pvBuffer != NULL);

    prCsmTelegram = cbr_arRxTelegramIndex[bTelCount];

    // loop over all signals of a telegram
    for (bSigCount = 0; bSigCount < prCsmTelegram->bNumberOfSignals; bSigCount++)
    {
      // get pointer for each signal
      pCsmSignal = ((prCsmTelegram->prSignals) + bSigCount);
      // ensure that the pointer is not NULL
      CSM_vAssert(pCsmSignal != NULL);

      // check if signal definition exceeds the DLC
      #if (CSM_S_CBR_BIG_ENDIAN_SIGNALS == CSM_C_F_OFF)
      CSM_vAssert(csm_arCbrRxTelegram[bTelCount].bDlc >= ((pCsmSignal->bStartBit + pCsmSignal->bLength) / 8));
      #endif

      // check if number of callbacks and pointer to callback array is consistent
      #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_OFF)
      CSM_vAssert(!((pCsmSignal->bMaxCallbackFunctions > 0) ^ (pCsmSignal->pfvSignalInd1 != NULL)));
      #endif // CSM_S_USE_CALLBACK_WITH_HANDLE
    }
  }

  // ******* TX ************************************************************************************
  for (bTelCount = 0; bTelCount < CBR_TX_MAX_NUMBER; bTelCount++)
  {
    CSM_vAssert(csm_arCbrTxTelegram[bTelCount].bDlc > 0);
    CSM_vAssert(csm_arCbrTxTelegram[bTelCount].dwCanId > 0);
    CSM_vAssert(csm_arCbrTxTelegram[bTelCount].pvBuffer != NULL);

    prCsmTelegram = cbr_arTxTelegramIndex[bTelCount];

    // loop over all signals of a telegram
    for (bSigCount = 0; bSigCount < prCsmTelegram->bNumberOfSignals; bSigCount++)
    {
      // get pointer for each signal
      pCsmSignal = ((prCsmTelegram->prSignals) + bSigCount);
      // ensure that the pointer is not NULL
      CSM_vAssert(pCsmSignal != NULL);

      // check if signal definition exceeds the DLC
      #if (CSM_S_CBR_BIG_ENDIAN_SIGNALS == CSM_C_F_OFF)
      CSM_vAssert(csm_arCbrTxTelegram[bTelCount].bDlc >= ((pCsmSignal->bStartBit + pCsmSignal->bLength) / 8));
      #endif

      // No application callback for TX signals !
      CSM_vAssert(pCsmSignal->bMaxCallbackFunctions == 0);

      #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
      CSM_vAssert(pCsmSignal->pfnSignalCallback == NULL);
      #else
      CSM_vAssert(pCsmSignal->pfvSignalInd1 == NULL);
      #endif // CSM_S_USE_CALLBACK_WITH_HANDLE
    }
  }
}/*lint !e438 !e550 yes, pCsmSignal is not used and not accessed, since we only check the pointer here*/

/************************************************************************
 * function    CBR_lSignalWriteRaw
 * \doxydocu
 * \brief      Writes raw signal data to telegram buffer.
 *
 *             Writes raw signal data to telegram buffer.
 *
 * \param[out]  pbTelegrammData (->O)
 *             Base address of write signal buffer
 *
 * \param[in]  bTelegrammBufferLength (I)
 *             Length of data buffer *pbTelegrammData
 *
 * \param[in]  bStartBit (I)
 *             Start bit of signal. bStart is the lsb
 *             Range: 0..62
 *
 * \param[in]  bLength (I)
 *             Bit Length of signal
 *             Range 1..64
 *
 * \param[in]  *pbSignalData (->I)
 *             pointer to data of the signal
 *
 * \param[in]  bSignalBufferLength (I)
 *             Length of data buffer *pbSignalData
 *
 * \param[in]  bSignalEndian (I)
 *             Endianness of the signal. i.e. big or small endian.
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR, CSM_E_DATA_BUFFER_TO_SHORT
 *
 * \access     application
 * \reentrant  no, protection with CBR_LOCK_TX_DATA/CBR_UNLOCK_TX_DATA;
 *
 * \lastreview 21.11.2008 Pröhl
 *
 * \history_begin
 * 18.06.04 CM-DI/ESU3-Pröhl
 * Initial Revision.
 *
 * \history_end
 ************************************************************************/
#if (CSM_S_CBR_BOTH_ENDIANS_SIGNALS == CSM_C_F_ON)
static LONG CBR_lSignalWriteRaw(BYTE * const pbTelegrammData, const BYTE bTelegrammBufferLength,
                                const BYTE bStartBit, const BYTE bLength, const BYTE * const pbSignalData,
                                const BYTE bSignalBufferLength, const BYTE bSignalEndian
                               )
#else
static LONG CBR_lSignalWriteRaw(BYTE * const pbTelegrammData, const BYTE bTelegrammBufferLength,
                                const BYTE bStartBit, const BYTE bLength, const BYTE * const pbSignalData,
                                const BYTE bSignalBufferLength
                               )
#endif
{
  BYTE  bShift;
  DWORD dwNumberOfBytes;
  DWORD dwStartByte;
  DWORD dwTemp;
  BYTE  bTemp;
  DWORD dwCount;
  DWORD dwRightMaskIndex; // = shift !!!!
  DWORD dwLeftMaskIndex;

  // bStartBit zeigt auf LSB
  if (bLength == 0)
  {
    return CSM_C_NO_ERROR;
  }
  if (((bLength+7)>>3) > bSignalBufferLength)
  {
    // The signal width is too long for the given application buffer.
    // You are trying to write only a part of a single signal.
    // This is not permitted. A signal must be written as a whole.
    CSM_vAssert(0);
    return CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR,CSM_C_ERR_LOC_CBR,CSM_E_DATA_LENGTH_EXCEEDED);
  }

  #if (CSM_S_CBR_BOTH_ENDIANS_SIGNALS == CSM_C_F_ON)
  if (bSignalEndian == CSM_C_SIGNAL_ENDIANS_LITTLE)
  {
  #endif
    #if (CSM_S_CBR_LITTLE_ENDIAN_SIGNALS == CSM_C_F_ON)
    if (((bStartBit+bLength+7)>>3) > bTelegrammBufferLength)
    {
      CSM_vAssert(0);
      return CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR,CSM_C_ERR_LOC_CBR,CSM_E_DATA_LENGTH_EXCEEDED);
    }
    #endif
  #if (CSM_S_CBR_BOTH_ENDIANS_SIGNALS == CSM_C_F_ON)
  }
  #endif

  // calculate shift
  bShift = bStartBit % 8;

  // calculate start byte
  dwStartByte = bStartBit >> 3;

  // calc num of bytes
  dwNumberOfBytes = ((bShift+bLength-1U)>>3)+1;

  // calculate mask index
  dwRightMaskIndex = bStartBit % 8; // = shift !!!!
  dwLeftMaskIndex = (bLength+bStartBit-1) % 8;

  for (dwCount = 0; dwCount < dwNumberOfBytes; dwCount++)
  {
    BYTE bMask = 0x00; // wenn ein Bit 1 ist dann bleiben dort die alten bits unverändert

    #if (CSM_S_LITTLE_ENDIAN_CPU == CSM_C_F_ON)
    dwTemp = pbSignalData[dwCount - 1] + ((WORD)pbSignalData[dwCount] << 8);
    #endif

    #if (CSM_S_BIG_ENDIAN_CPU == CSM_C_F_ON)
    #error Herzlichen Glückwunsch Du bist der erste der eine big endian CPU benutzt. Bitte alles gründlich testen :-)
    dwTemp = pbSignalData[dwNumberOfBytes - 1 - dwCount + 1] + (pbSignalData[dwNumberOfBytes - 1 - dwCount] << 8);
    #endif
    bTemp = (BYTE) ((dwTemp << bShift) >> 8) & 0xff;

    if (dwCount == (DWORD)0)
    {
      bMask = cbr_abRMask[dwRightMaskIndex];
    }
    if (dwCount == (dwNumberOfBytes-1))
    {
      bMask |= cbr_abLMask[dwLeftMaskIndex];
    }

    #if (CSM_S_CBR_BOTH_ENDIANS_SIGNALS == CSM_C_F_ON)
    if (bSignalEndian == CSM_C_SIGNAL_ENDIANS_LITTLE)
    {
    #endif
      #if (CSM_S_CBR_LITTLE_ENDIAN_SIGNALS == CSM_C_F_ON)
      pbTelegrammData[dwStartByte + dwCount] = (pbTelegrammData[dwStartByte + dwCount] & bMask) | (bTemp & ~bMask);
      #endif
    #if (CSM_S_CBR_BOTH_ENDIANS_SIGNALS == CSM_C_F_ON)
    }
    else
    {
    #endif
    #if (CSM_S_CBR_BIG_ENDIAN_SIGNALS == CSM_C_F_ON)
      pbTelegrammData[dwStartByte - dwCount] = (pbTelegrammData[dwStartByte - dwCount] & bMask) | (bTemp & ~bMask);
    #endif
    #if (CSM_S_CBR_BOTH_ENDIANS_SIGNALS == CSM_C_F_ON)
    }
    #endif
  }

  return CSM_C_NO_ERROR;
}/*lint !e715 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally */

/************************************************************************
 * function    CBR_lSignalReadRaw
 * \doxydocu
 * \brief      Reads raw signal data from telegram buffer.
 *
 *             Reads raw signal data from telegram buffer.
 *             Attention: After address pvActSignalData there has to be
 *             enough place for the data (1 + Length/8 bytes) !!!
 *
 * \param[in]  pbTelegrammData (->I)
 *             Base address of read signal buffer
 *
 * \param[in]  bTelegrammBufferLength (I)
 *             Length of data buffer *pbTelegrammData
 *
 * \param[in]  bStartBit (I)
 *             Start bit of signal. bStart is the lsb
 *             Range: 0..62
 *
 * \param[in]  bLength (I)
 *             Bit Length of signal
 *             Range 1..64
 *
 * \param[out] *pbSignalData (->O)
 *             pointer to data of the signal
 *
 * \param[in]  bSignalBufferLength (I)
 *             Length of data buffer *pbSignalData
 *
 * \param[in]  bSignalEndian (I)
 *             Endianness of the signal. i.e. big or small endian.
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR, CSM_E_DATA_BUFFER_TO_SHORT
 *
 * \access     application
 * \reentrant  no, protection with CBR_LOCK_RX_DATA/CBR_UNLOCK_RX_DATA;
 *
 * \lastreview 21.11.2008 Pröhl
 *
 * \history_begin
 * 18.06.04 CM-DI/ESU3-Pröhl
 * Initial Revision.
 *
 * \history_end
 ************************************************************************/
#if (CSM_S_CBR_BOTH_ENDIANS_SIGNALS == CSM_C_F_ON)
static LONG CBR_lSignalReadRaw(const BYTE * const pbTelegrammData, const BYTE bTelegrammBufferLength,
                               const BYTE   bStartBit, const BYTE bLength,
                               BYTE * const  pbSignalData, const BYTE bSignalBufferLength,
                               const BYTE bSignalEndian
                              )
#else
static LONG CBR_lSignalReadRaw(const BYTE * const pbTelegrammData, const BYTE bTelegrammBufferLength,
                               const BYTE   bStartBit, const BYTE bLength,
                               BYTE * const  pbSignalData, const BYTE bSignalBufferLength
                              )
#endif
{
  BYTE  bShift;
  DWORD dwNumberOfBytes;
  DWORD dwStartByte;
  DWORD dwTemp;
  DWORD dwCount;
  DWORD dwMaskIndex;

  /* bStartBit zeigt auf LSB */

  if (bLength == 0)
  {
    return CSM_C_NO_ERROR;
  }

  if (((bLength+7)>>3) > bSignalBufferLength)
  {
    return CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR,CSM_C_ERR_LOC_CBR,CSM_E_DATA_LENGTH_EXCEEDED);
  }

  #if (CSM_S_CBR_BOTH_ENDIANS_SIGNALS == CSM_C_F_ON)
  if (bSignalEndian == CSM_C_SIGNAL_ENDIANS_LITTLE)
  {
  #endif
    #if (CSM_S_CBR_LITTLE_ENDIAN_SIGNALS == CSM_C_F_ON)
    if (((bStartBit+bLength+7)>>3) > bTelegrammBufferLength)
    {
      return CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR,CSM_C_ERR_LOC_CBR,CSM_E_DATA_LENGTH_EXCEEDED);
    }
    #endif
  #if (CSM_S_CBR_BOTH_ENDIANS_SIGNALS == CSM_C_F_ON)
  }
  #endif

  // Clear application buffer now.
  // Reason: if application gives a buffer bigger than needed, the code below
  // would only fill bytes of the buffer which are needed for the signal value.
  // Remaining bytes would be untouched which can result in an unexpected
  // inconsistent buffer content if the buffer is not initialized to 0 prior
  // to SignalRead.
  CSM_vMemSet((void*)pbSignalData, 0, bSignalBufferLength);

  // calculate shift
  bShift = bStartBit % 8;

  // calculate start byte
  dwStartByte = bStartBit >> 3;

  // calc num of bytes
  dwNumberOfBytes = (bLength + 7)>>3;

  // calculate mask index
  dwMaskIndex = bLength % 8;

  for (dwCount = 0; dwCount < dwNumberOfBytes; dwCount++)
  {
    #if (CSM_S_CBR_BOTH_ENDIANS_SIGNALS == CSM_C_F_ON)
    if (bSignalEndian == CSM_C_SIGNAL_ENDIANS_LITTLE)
    {
    #endif
      #if (CSM_S_CBR_LITTLE_ENDIAN_SIGNALS == CSM_C_F_ON)
      dwTemp = pbTelegrammData[dwStartByte + dwCount] + (pbTelegrammData[dwStartByte + dwCount + 1] << 8);
      #endif
    #if (CSM_S_CBR_BOTH_ENDIANS_SIGNALS == CSM_C_F_ON)
    }
    else
    {
    #endif
      #if (CSM_S_CBR_BIG_ENDIAN_SIGNALS == CSM_C_F_ON)
      dwTemp = pbTelegrammData[dwStartByte - dwCount] + ((WORD)pbTelegrammData[dwStartByte - (dwCount + 1)] << 8);
      #endif
    #if (CSM_S_CBR_BOTH_ENDIANS_SIGNALS == CSM_C_F_ON)
    }
    #endif

    dwTemp = dwTemp >> bShift;

    #if (CSM_S_LITTLE_ENDIAN_CPU == CSM_C_F_ON)
    pbSignalData[dwCount] = (BYTE)(dwTemp & 0xff);
    #endif

    #if (CSM_S_BIG_ENDIAN_CPU == CSM_C_F_ON)
    #error Herzlichen Glückwunsch Du bist der erste der eine big endian CPU benutzt. Bitte alles gründlich testen :-)
    pbSignalData[dwNumberOfBytes - 1 - dwCount] = (BYTE)(dwTemp & 0xff);
    #endif
  } // loop

  // mask out the upper part of the most significant byte which is not part of the signal
  #if (CSM_S_LITTLE_ENDIAN_CPU == CSM_C_F_ON)
  pbSignalData[dwNumberOfBytes-1] &= cbr_abSMask[dwMaskIndex];
  #endif

  #if (CSM_S_BIG_ENDIAN_CPU == CSM_C_F_ON)
  #error Herzlichen Glückwunsch Du bist der erste der eine big endian CPU benutzt. Bitte alles gründlich testen :-)
  pbSignalData[0] &= cbr_abSMask[dwMaskIndex];
  #endif

  return CSM_C_NO_ERROR;
}/*lint !e715 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally */

/*******************************************************************************
 * function    CBR_vStartTransmitRepeatTimer
 * \doxydocu
 * \brief      Starts the transmit repeat timer.
 *
 * \param[in]  bTelegramIndex
 *             Index of telegram
 *
 * \return     void
 *
 * \access     CAN-Task
 * \reentrant  yes
 *
 * \lastreview 21.11.2008 Pröhl
 *
 * \history_begin
 * 22.09.04 CM-DI/ESA3-Pröhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
#if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
static void CBR_vStartTransmitRepeatTimer(BYTE bTelegramIndex)
{
  #if (CSM_S_REQUESTABLE_TIMER == CSM_C_F_ON)
  // dynamic mapping of timer handles
  CSM_lTimerStart(CBR_arTxTelegramState[bTelegramIndex].hRepeatTimer, csm_arCbrTxTelegram[bTelegramIndex].wCycleTime);
  #else
  CSM_lTimerStart(CSM_TIMER_CBR_TRANSMIT_TIMEOUT_0 + bTelegramIndex, csm_arCbrTxTelegram[bTelegramIndex].wCycleTime);
  #endif
}

/*******************************************************************************
 * function    CBR_vStartTransmitMinDelayTimer
 * \doxydocu
 * \brief      Starts the transmit minimum delay timer.
 *
 * \param[in]  bTelegramIndex
 *             Index of telegram
 *
 * \return     void
 *
 * \access     CAN-Task
 * \reentrant  yes
 *
 * \lastreview  27.09.2007 VTeam-Pröhl
 *
 * \history_begin
 * 22.09.04 CM-DI/ESA3-Pröhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
#if (CSM_S_CBR_MIN_DELAY == CSM_C_F_ON)
static void CBR_vStartTransmitMinDelayTimer(BYTE bTelegramIndex)
{
  #if (CSM_S_REQUESTABLE_TIMER == CSM_C_F_ON)
  // dynamic mapping of timer handles
  #error "Hier muß man sich noch etwas ausdenken !"
  #else
  if (csm_arCbrTxTelegram[bTelegramIndex].wMinDist != 0)
  {
    CSM_lTimerStart(CSM_TIMER_CBR_DELAY_TIMEOUT_0 + bTelegramIndex, csm_arCbrTxTelegram[bTelegramIndex].wMinDist);
  }
  #endif
}
#endif // CSM_S_CBR_MIN_DELAY

/*******************************************************************************
 * function    CBR_vStartReceiveTimer
 * \doxydocu
 * \brief      Start the receive timer.
 *
 *             The receive timer calls the receive timeout function
 *
 * \param[in]  bTelegramIndex
 *             Index of telegram
 *
 * \return     void
 *
 * \access     CAN-Task
 * \reentrant  yes
 *
 * \lastreview 21.11.2008 Pröhl
 *
 * \history_begin
 * 22.09.04 CM-DI/ESA3-Pröhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
static void CBR_vStartReceiveTimer(BYTE bTelegramIndex)
{
  if (csm_arCbrRxTelegram[bTelegramIndex].wTimeoutTime > 0)
  {
    #if (CSM_S_REQUESTABLE_TIMER == CSM_C_F_ON)
    // dynamic mapping of timer handles
    CSM_lTimerStart(CBR_arRxTelegramState[bTelegramIndex].hReceiveTimer, CBR_EXTEND_TO_DELAY(csm_arCbrRxTelegram[bTelegramIndex].wTimeoutTime));
    #else
    CSM_lTimerStart(CSM_TIMER_CBR_RECEIVE_TIMEOUT_0 + bTelegramIndex, CBR_EXTEND_TO_DELAY(csm_arCbrRxTelegram[bTelegramIndex].wTimeoutTime));
    #endif
  }
}

/*******************************************************************************
 * function    CBR_vStopTransmitRepeatTimer
 * \doxydocu
 * \brief      Stop the receive timer.
 *
 * \param[in]  bTelegramIndex
 *             Index of telegram
 *
 * \return     void
 *
 * \access     CAN-Task
 * \reentrant  yes
 *
 * \lastreview 21.11.2008 Pröhl
 *
 * \history_begin
 * 22.09.04 CM-DI/ESA3-Pröhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
static void CBR_vStopTransmitRepeatTimer(BYTE bTelegramIndex)
{
  #if (CSM_S_REQUESTABLE_TIMER == CSM_C_F_ON)
  // dynamic mapping of timer handles
  CSM_lTimerStop(CBR_arTxTelegramState[bTelegramIndex].hRepeatTimer );
  #else
  CSM_lTimerStop(CSM_TIMER_CBR_TRANSMIT_TIMEOUT_0 + bTelegramIndex );
  #endif

  CBR_arTxTelegramState[bTelegramIndex].fRepeatTimeout = FALSE;
}

/*******************************************************************************
 * function    CBR_vStopTransmitMinDelayTimer
 * \doxydocu
 * \brief      Stop the transmit minimum delay timer.
 *
 * \param[in]  bTelegramIndex
 *             Index of telegram
 *
 * \return     void
 *
 * \access     CAN-Task
 * \reentrant  yes
 *
 * \lastreview 21.11.2008 Pröhl
 *
 * \history_begin
 * 22.09.04 CM-DI/ESA3-Pröhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
#if (CSM_S_CBR_MIN_DELAY == CSM_C_F_ON)
static void CBR_vStopTransmitMinDelayTimer(BYTE bTelegramIndex)
{
  #if (CSM_S_REQUESTABLE_TIMER == CSM_C_F_ON)
  // dynamic mapping of timer handles
  #error "Hier muß man sich noch etwas ausdenken !"
  #else
  CSM_lTimerStop(CSM_TIMER_CBR_DELAY_TIMEOUT_0 + bTelegramIndex);
  #endif
}
#endif // CSM_S_CBR_MIN_DELAY

/*******************************************************************************
 * function    CBR_vStopReceiveTimer
 * \doxydocu
 * \brief      Stop the receive timer.
 *
 * \param[in]  bTelegramIndex
 *             Index of telegram
 *
 * \return     void
 *
 * \access     CAN-Task
 * \reentrant  yes
 *
 * \lastreview 21.11.2008 Pröhl
 *
 * \history_begin
 * 22.09.04 CM-DI/ESA3-Pröhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
static void CBR_vStopReceiveTimer(BYTE bTelegramIndex)
{
  #if (CSM_S_REQUESTABLE_TIMER == CSM_C_F_ON)
  // dynamic mapping of timer handles
  CSM_lTimerStop(CBR_arRxTelegramState[bTelegramIndex].hReceiveTimer );
  #else
  CSM_lTimerStop(CSM_TIMER_CBR_RECEIVE_TIMEOUT_0 + bTelegramIndex );
  #endif
  CBR_arRxTelegramState[bTelegramIndex].fReceiveTimeout = FALSE;
}
#endif // CSM_S_REMOTE_STACK

//*****************************************************************
//* function implementation (scope: global)
//*****************************************************************

/*******************************************************************************
 * function    CBR_lInit
 * \doxydocu
 * \brief      Initializes CAN Broadcast Server module.
 *
 *             Initializes CAN Broadcast Server module.
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESA3
 *
 * \param      bInitMode
 *             Allowed values (see csm_stack_i.h):
 *                - CSM_C_INIT_COLD
 *                - CSM_C_INIT_WARM
 *                - CSM_C_INIT_START
 *
 * \return     LONG
 *             Error value:
 *              - CSM_C_NO_ERROR
 *              - CBR_E_ERROR_STATE_WHILE_INIT
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \lastreview 21.11.2008 Pröhl
 *
 * \history_begin
 * 18.06.04 CM-DI/ESU3-Pröhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
LONG CBR_lInit(const BYTE bInitMode)
{
  LONG return_value = CSM_C_NO_ERROR;
  BYTE bCountTelegramIndex;

  // ensure that the variable mapped by the makro has been initialized already !!!
  CSM_vAssert(CBR_RX_MAX_NUMBER != 0);
  CSM_vAssert(CBR_TX_MAX_NUMBER != 0);

  if (bInitMode == CSM_C_INIT_COLD)
  {
    #if (CSM_S_CBR_RX_TIMEOUT_GROUPS == CSM_C_F_ON)
    cbr_bRxGrpMask = 0;               // default: Always timeout check !
    #endif

    for (bCountTelegramIndex = 0; bCountTelegramIndex < CBR_RX_MAX_NUMBER; bCountTelegramIndex++)
    {
      // Beware of variants management: variants pointers are assigned not until WARM INIT !!
      // Only basic init is permitted here !!
      CBR_arRxTelegramState[bCountTelegramIndex].dwState = (CSM_C_SIGNAL_NOT_RECEIVED | CSM_C_SIGNAL_DATA_INVALID);
    }

    // perform CBR config integrity tests now
    cbr_vCheckConfig();
  }
  else if (bInitMode == CSM_C_INIT_WARM)
  {
    CBR_M_INIT_VARS(); // set macro in csm_stack_mp.h

    cbr_dwStatus = 0;

    for (bCountTelegramIndex = 0; bCountTelegramIndex < CBR_TX_MAX_NUMBER; bCountTelegramIndex++)
    {
      #if (CSM_S_CBR_TX_MSG_ROLL_COUNTER == CSM_C_F_ON)
      // reset all rolling counters
      cbr_aTxRollingCounterAdmin[bCountTelegramIndex].bRollCount = 0;
      #if (CSM_S_CBR_TX_MSG_ROLL_COUNTER_GROUPS == CSM_C_F_ON)
      cbr_aTxRollingCounterAdmin[bCountTelegramIndex].bGroupHandled = 0;
      cbr_aTxRollingCounterAdmin[bCountTelegramIndex].bGroupImDelay = 0;
      #endif
      #endif

      /* transmit telegramm Status zurück setzen */
      CBR_arTxTelegramState[bCountTelegramIndex].fTransmit = FALSE;
      CBR_arTxTelegramState[bCountTelegramIndex].fRepeatTimeout = FALSE;

      #if (CSM_S_CBR_MIN_DELAY == CSM_C_F_ON)
      CBR_arTxTelegramState[bCountTelegramIndex].fDelayTransmit = FALSE;
      #endif

      CBR_arTxTelegramState[bCountTelegramIndex].fValid = FALSE;

      #if (CSM_S_CBR_TX_IF_APPL_ACTIVE == CSM_C_F_ON)
      CBR_arTxTelegramState[bCountTelegramIndex].bRepeatCount = 0;
      #endif

      #if (CSM_S_REQUESTABLE_TIMER == CSM_C_F_ON)
      // dynamic mapping of timer handles
      CBR_arTxTelegramState[bCountTelegramIndex].hRepeatTimer = CSM_lRequestTimer();

      if (CBR_arTxTelegramState[bCountTelegramIndex].hRepeatTimer == 0)
      {
        // no more timer available!
        return_value = CBR_M_MAKE_ERROR(CSM_C_ERR_ERROR,CSM_E_GENERAL_ERROR);
        CSM_vAssert(FALSE);
      }
      #endif // CSM_S_REQUESTABLE_TIMER

      // check for correct pointer setting
      CSM_vAssert(cbr_arTxTelegramIndex[bCountTelegramIndex] != NULL);

      #if (CSM_S_DEFAULT_SIGNAL_VALUES == CSM_C_F_ON)
      // set default data if available
      // TODO: consider MASK
      if ((cbr_arTxTelegramIndex[bCountTelegramIndex] != NULL) &&
          (cbr_arTxTelegramIndex[bCountTelegramIndex]->pvTelegramDefaultData != NULL) )
      {
        CBR_arTxTelegramState[bCountTelegramIndex].fValid = TRUE;
        CSM_vMemCopy(csm_arCbrTxTelegram[bCountTelegramIndex].pvBuffer,
                     cbr_arTxTelegramIndex[bCountTelegramIndex]->pvTelegramDefaultData,
                     (WORD)csm_arCbrTxTelegram[bCountTelegramIndex].bDlc);
      }
      #endif // CSM_S_DEFAULT_SIGNAL_VALUES
    }

    for (bCountTelegramIndex = 0; bCountTelegramIndex < CBR_RX_MAX_NUMBER; bCountTelegramIndex++)
    {
      // receive telegramm Status zurück setzen
      CBR_arRxTelegramState[bCountTelegramIndex].fReceiveTimeout = FALSE;

      #if (CSM_S_REQUESTABLE_TIMER == CSM_C_F_ON)
      // dynamic mapping of timer handles
      CBR_arRxTelegramState[bCountTelegramIndex].hReceiveTimer = CSM_lRequestTimer();

      if (CBR_arRxTelegramState[bCountTelegramIndex].hReceiveTimer == 0)
      {
        // no more timer available!
        return_value = CBR_M_MAKE_ERROR(CSM_C_ERR_ERROR,CSM_E_GENERAL_ERROR); /* ToDo: assign specific error code!*/
        CSM_vAssert(FALSE);
      }
      #endif // CSM_S_REQUESTABLE_TIMER

      // check for correct pointer setting
      CSM_vAssert(cbr_arRxTelegramIndex[bCountTelegramIndex] != NULL);

      #if (CSM_S_DEFAULT_SIGNAL_VALUES == CSM_C_F_ON)
      // set default data if available
      // TODO: consider MASK
      if (cbr_arRxTelegramIndex[bCountTelegramIndex]->pvTelegramDefaultData != NULL)
      {
        CBR_arRxTelegramState[bCountTelegramIndex].dwState &= ~CSM_C_SIGNAL_DATA_INVALID;
        CSM_vMemCopy(csm_arCbrRxTelegram[bCountTelegramIndex].pvBuffer,
                      cbr_arRxTelegramIndex[bCountTelegramIndex]->pvTelegramDefaultData,
                      (WORD)csm_arCbrRxTelegram[bCountTelegramIndex].bDlc);
      }
      #endif // CSM_S_DEFAULT_SIGNAL_VALUES
    }
  }
  else if (bInitMode == CSM_C_INIT_START)
  {
    // nothing
  }
  else
  {
    return CBR_M_MAKE_ERROR(CSM_C_ERR_ERROR,CSM_E_UNKNOWN_STATE);
  }

  return return_value;
}

/*******************************************************************************
 * function    CBR_lExit
 * \doxydocu
 * \brief      Stops CAN Broadcast server.
 *
 *             Stops CAN Broadcast server.
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESA3
 *
 * \param[in]  bExitMode
 *             Allowed values (see csm_stack_i.h):
 *                - CSM_C_EXIT_STOP
 *                - CSM_C_EXIT_OFF
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR
 *
 * \access     CAN-task
 * \reentrant  no
 *
 * \lastreview 21.11.2008 Pröhl
 *
 * \history_begin
 * 18.06.04 CM-DI/ESU3-Pröhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
LONG CBR_lExit(const BYTE bExitMode)
{
  LONG return_value = CSM_C_NO_ERROR;

  #if (CSM_S_REQUESTABLE_TIMER == CSM_C_F_ON)
  if (CSM_C_EXIT_STOP == bExitMode)
  {
    BYTE bCountTelegramIndex;

    // release dynamically allocated TX and RX timers
    for (bCountTelegramIndex = 0; bCountTelegramIndex < CBR_TX_MAX_NUMBER; bCountTelegramIndex++)
    {
      // first step: no return errors used/necessary
      CSM_lReleaseTimer((BYTE) CBR_arTxTelegramState[bCountTelegramIndex].hRepeatTimer);
    }

    for (bCountTelegramIndex = 0; bCountTelegramIndex < CBR_RX_MAX_NUMBER; bCountTelegramIndex++)
    {
      // first step: no return errors used/necessary
      CSM_lReleaseTimer((BYTE) CBR_arRxTelegramState[bCountTelegramIndex].hReceiveTimer);
    }
  } // CSM_C_EXIT_STOP
  #endif // CSM_S_REQUESTABLE_TIMER

  return return_value;
}/*lint !e715 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally */

/*******************************************************************************
 * function    CBR_lSignalCallBackInit
 * \doxydocu
 * \brief      Stores the callback function.
 *
 *             Stores the callback function.
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESA3
 *
 * \param[in]  pvHandle (I)
 *             pointer to handle. May be helpful for a deinit() function.
 *
 * \param[in]  dwSignalId (I)
 *             signal identifier (signal name)
 *
 * \param[in]  *vPFNCBRSignalInd (->I)
 *             pointer to callback function
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR
 *
 * \access     application
 * \reentrant  yes, in the near future
 *              TODO, Odometer is not working with DI/EI
 *
 * \lastreview 19.04.07 VTeam-Pröhl
 *
 * \history_begin
 * 01.01.03 CM-DI/ESU3-Pröhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
#if (CSM_S_CALLBACK_INIT_BY_APPLICATION == CSM_C_F_ON)
LONG CBR_lSignalCallBackInit(void * pvHandle, const DWORD dwSIGNALID, const tCSM_PFN_CBR_SIGNAL_IND vPFNCBRSignalInd)
{
  BYTE bCount;
  BYTE bTelegramIndex;
  BYTE bSignalIndex;
  LONG lRetVal;
  DWORD dwSignalId = dwSIGNALID;

  // ensure that the variable mapped by the makro has been initialized already !!!
  CSM_vAssert(CBR_RX_MAX_NUMBER != 0);

  bTelegramIndex = (BYTE) CSM_M_GET_TELINDEX_FROM_SIGNAL_ID(dwSignalId);
  bSignalIndex   = (BYTE) CSM_M_GET_SIGINDEX_FROM_SIGNAL_ID(dwSignalId);
  lRetVal = (CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CBR, CSM_E_NO_FREE_CALLBACK));

  if (bTelegramIndex < CBR_RX_MAX_NUMBER)
  {
    if (bSignalIndex < (cbr_arRxTelegramIndex[bTelegramIndex]->bNumberOfSignals))
    {
      // Signal ist gültig
      // INT iInterruptOldFlag;
      // CBR_LOCK_RX_DATA(iInterruptOldFlag);
      for (bCount=0; bCount < ((cbr_arRxTelegramIndex[bTelegramIndex]->prSignals) + bSignalIndex)->bMaxCallbackFunctions; bCount++)
      {
        #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
        if (((cbr_arRxTelegramIndex[bTelegramIndex]->prSignals) + bSignalIndex)->pfnSignalCallback[bCount].pfvSignalInd1 == NULL)
        {
          ((cbr_arRxTelegramIndex[bTelegramIndex]->prSignals) + bSignalIndex)->pfnSignalCallback[bCount].pvHandle = pvHandle;
          ((cbr_arRxTelegramIndex[bTelegramIndex]->prSignals) + bSignalIndex)->pfnSignalCallback[bCount].pfvSignalInd1 = vPFNCBRSignalInd;
          lRetVal = CSM_C_NO_ERROR;
          break; // for-Scheife  verlassen
        }
        #else
        if (((cbr_arRxTelegramIndex[bTelegramIndex]->prSignals) + bSignalIndex)->pfvSignalInd1[bCount] == NULL)
        {
          ((cbr_arRxTelegramIndex[bTelegramIndex]->prSignals) + bSignalIndex)->pfvSignalInd1[bCount] = vPFNCBRSignalInd;
          lRetVal = CSM_C_NO_ERROR;
          break; // for-Scheife  verlassen
        }
        #endif
      }
      // CBR_UNLOCK_RX_DATA(iInterruptOldFlag);
    }
    else
    {
      lRetVal = (CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CBR, CSM_E_INVALID_PARA));
    }
  }
  else
  {
    lRetVal = (CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CBR, CSM_E_INVALID_PARA));
  }

  return lRetVal;
}
#endif // CSM_S_CALLBACK_INIT_BY_APPLICATION

/*******************************************************************************
 * function    CBR_lStartTx
 * \doxydocu
 * \brief      Starts CAN Broadcast Server transmission telegrams.
 *
 *             Starts CAN Broadcast Server transmission telegrams.
 *             From now on Tx is runnig.
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESA3
 *
 * \param[in]  bBus (I)
 *             bus number
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR
 *
 * \access     CAN-Task
 * \reentrant  yes, but only call by CAN-Task
 *
 * \lastreview 26.09.06 CM-DI/ESA3-Battis
 *
 * \history_begin
 * 25.09.06 CM-DI/ESA3-Haffner
 * Changes for CSM proxy usage.
 *
 * 26.07.04 CM-DI/ESU-3 Pröhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
LONG CBR_lStartTx(const BYTE bBus)
{
  #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
  BYTE bCountTelegramIndex;
  BOOL fXmit = FALSE;
  #endif // CSM_S_REMOTE_STACK

  #if (CSM_S_TRACE == CSM_C_F_ON)
    #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR3(("CBR_lStartTx - Bus:%x",
      ETG_CENUM(tCSM_BUS_NUMBER, (tU8)  bBus) ));
      #endif
    #else
    {
      // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
      BYTE abData[1];
      abData[0] = bBus;
      CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_INTERFACE_CBR, CSM_C_TRACE_FKT_CBR_START_TX, abData, 1);
    }
    #endif
  #endif // CSM_S_TRACE

  CBR_SET_STATUS(bBus, CBR_STATUS_TX_RUNNING);

  #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
  // starte Tx timer
  for (bCountTelegramIndex = 0; bCountTelegramIndex < CBR_TX_MAX_NUMBER; bCountTelegramIndex++)
  {
    if (csm_arCbrTxTelegram[bCountTelegramIndex].bBus == bBus)
    {
      switch(csm_arCbrTxTelegram[bCountTelegramIndex].bMessageType)
      {
        case CBR_TMSG_CYCLIC:
        case CBR_TMSG_CYCLIC_AND_SPONTAN:
        {
          if (CBR_arTxTelegramState[bCountTelegramIndex].fValid == TRUE)
          {
            CBR_arTxTelegramState[bCountTelegramIndex].fTransmit = TRUE;   // Telegramm senden
            fXmit = TRUE;
            #if (CSM_S_CBR_RESYNCHRONIZED_CYCLE == CSM_C_F_ON)
              CBR_vStartTransmitRepeatTimer(bCountTelegramIndex);
            #endif
          }
        }
        break;
        #if  (CSM_S_CBR_TX_IF_APPL_ACTIVE == CSM_C_F_ON)
          case CBR_TMSG_CYCLIC_IF_ACTIVE:
          case CBR_TMSG_CYC_IF_ACT_AND_SPONT:
          {
            if (CBR_arTxTelegramState[bCountTelegramIndex].fValid == TRUE)
            {
              if (cbr_fIsFunctionActive(CBR_arTxTelegramState[bCountTelegramIndex].dwActive))
              {
                CBR_arTxTelegramState[bCountTelegramIndex].fTransmit = TRUE;   // Telegramm senden
                fXmit = TRUE;
                #if (CSM_S_CBR_RESYNCHRONIZED_CYCLE == CSM_C_F_ON)
                  CBR_vStartTransmitRepeatTimer(bCountTelegramIndex);
                #endif
              }
            }
          }
          break;
        #endif // CSM_S_CBR_TX_IF_APPL_ACTIVE
        case CBR_TMSG_SPONTAN:
        case CBR_TMSG_NO_TRANSMISSION:
        default:
        {
          // NOTHING TO DO
        }
        break;
      }
    }
  }

  if (fXmit == TRUE)
  {
    csm_vTriggerCsmDownstreamTask(CSM_C_DOWN_REASON_CBR_SEND);
  }
  #endif // CSM_S_REMOTE_STACK

  return CSM_C_NO_ERROR;
}

/*******************************************************************************
 * function    CBR_lStartRx
 * \doxydocu
 * \brief      Starts CAN Broadcast Server reception telegrams.
 *
 *             Starts CAN Broadcast Server reception telegrams.
 *             From now on Rx timers are running.
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESA3
 *
 * \param[in]  bBus (I)
 *             bus number
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR
 *
 * \access     CAN-Task
 * \reentrant  yes, but only call by CAN-Task
 *
 * \lastreview 26.09.06 CM-DI/ESA3-Battis
 *
 * \history_begin
 * 25.09.06 CM-DI/ESA3-Haffner
 * Changes for CSM proxy usage.
 *
 * 26.07.04 CM-DI/ESU-3 Pröhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
LONG CBR_lStartRx(const BYTE bBus)
{
  #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
  BYTE bCountTelegramIndex;
  #endif  // CSM_S_REMOTE_STACK

  // ensure that the variable mapped by the makro has been initialized already !!!
  CSM_vAssert(CBR_RX_MAX_NUMBER != 0);

  #if (CSM_S_TRACE == CSM_C_F_ON)
    #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR3(("CBR_lStartRx - Bus:%x",
      ETG_CENUM(tCSM_BUS_NUMBER, (tU8)  bBus) ));
      #endif
    #else
    {
      // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
      BYTE abData[1];
      abData[0] = bBus;
      CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_INTERFACE_CBR, CSM_C_TRACE_FKT_CBR_START_RX, abData, 1);
    }
    #endif
  #endif // CSM_S_TRACE

  CBR_SET_STATUS(bBus, CBR_STATUS_RX_RUNNING);

  #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
  #ifdef CBR_RXTIMEOUT_DELAY
  cbr_wRxTimeOutDelay = CBR_RXTIMEOUT_DELAY;   // Delay time out supervision.
  #endif

  for (bCountTelegramIndex = 0; bCountTelegramIndex < CBR_RX_MAX_NUMBER; bCountTelegramIndex++)
  {
    if (csm_arCbrRxTelegram[bCountTelegramIndex].bBus == bBus)
    {
      CBR_vStartReceiveTimer(bCountTelegramIndex);
    }
  }

  #ifdef CBR_RXTIMEOUT_DELAY
  cbr_wRxTimeOutDelay = 0;
  #endif
  #endif  // CSM_S_REMOTE_STACK

  return CSM_C_NO_ERROR;
}

/*******************************************************************************
 * function    CBR_lStopTx
 * \doxydocu
 * \brief      Stops CAN Broadcast Server module transmission telegrams.
 *
 *             Stops CAN Broadcast Server module transmission telegrams.
 *             Tx Timers will be stopped.
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESU3
 *
 * \param[in]  bBus (I)
 *             bus number
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR
 *
 * \access     CAN-Task
 * \reentrant  yes, but only call by CAN-Task
 *
 * \lastreview 26.09.06 CM-DI/ESA3-Battis
 *
 * \history_begin
 * 25.09.06 CM-DI/ESA3-Haffner
 * Changes for CSM proxy usage.
 *
 * 26.07.04 CM-DI/ESU-3 Pröhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
LONG CBR_lStopTx(const BYTE bBus)
{
  #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
  BYTE bCountTelegramIndex;
  #endif  // CSM_S_REMOTE_STACK

  #if (CSM_S_TRACE == CSM_C_F_ON)
    #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR3(("CBR_lStopTx - Bus:%x",
      ETG_CENUM(tCSM_BUS_NUMBER, (tU8)  bBus) ));
      #endif
    #else
    {
      // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
      BYTE abData[1];

      abData[0] = bBus;
      CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_INTERFACE_CBR, CSM_C_TRACE_FKT_CBR_STOP_TX, abData, 1);
    }
    #endif
  #endif // CSM_S_TRACE

  CBR_CLR_STATUS(bBus, CBR_STATUS_TX_RUNNING);

  #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
  for (bCountTelegramIndex = 0; bCountTelegramIndex < CBR_TX_MAX_NUMBER; bCountTelegramIndex++)
  {
    if (csm_arCbrTxTelegram[bCountTelegramIndex].bBus == bBus)
    {
      #if (CSM_S_CBR_TX_IF_APPL_ACTIVE == CSM_C_F_ON)
      CBR_arTxTelegramState[bCountTelegramIndex].bRepeatCount = 0;
      #endif

      #if (CSM_S_CBR_MIN_DELAY == CSM_C_F_ON)
      CBR_vStopTransmitMinDelayTimer(bCountTelegramIndex);
      CBR_arTxTelegramState[bCountTelegramIndex].fDelayTransmit = FALSE;
      #endif // CSM_S_CBR_MIN_DELAY

      CBR_vStopTransmitRepeatTimer(bCountTelegramIndex);
    }
  }
  #endif // CSM_S_REMOTE_STACK

  return CSM_C_NO_ERROR;
}

/*******************************************************************************
 * function    CBR_lStopRx
 * \doxydocu
 * \brief      Stops CAN Broadcast Server module reception telegrams.
 *
 *             Stops CAN Broadcast Server module reception telegrams.
 *             Rx Timers will be stopped.
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESU3
 *
 * \param[in]  bBus (I)
 *             bus number
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR
 *
 * \access     CAN-Task
 * \reentrant  yes, but only call by CAN-Task
 *
 * \lastreview 26.09.06 CM-DI/ESA3-Battis
 *
 * \history_begin
 * 25.09.06 CM-DI/ESA3-Haffner
 * Changes for CSM proxy usage.
 *
 * 26.07.04 CM-DI/ESU-3 Pröhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
LONG CBR_lStopRx(const BYTE bBus)
{
  #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
  BYTE bCountTelegramIndex;
  #endif  // CSM_S_REMOTE_STACK

  // ensure that the variable mapped by the makro has been initialized already !!!
  CSM_vAssert(CBR_RX_MAX_NUMBER != 0);

  #if (CSM_S_TRACE == CSM_C_F_ON)
    #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
      #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      ETG_TRACE_USR3(("CBR_lStopRx - Bus:%x",
      ETG_CENUM(tCSM_BUS_NUMBER, (tU8)  bBus) ));
      #endif
    #else
    {
      // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
      BYTE abData[1];

      abData[0] = bBus;
      CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_INTERFACE_CBR, CSM_C_TRACE_FKT_CBR_STOP_RX, abData, 1);
    }
    #endif
  #endif // CSM_S_TRACE

  CBR_CLR_STATUS(bBus, CBR_STATUS_RX_RUNNING);

  #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
  for (bCountTelegramIndex = 0; bCountTelegramIndex < CBR_RX_MAX_NUMBER; bCountTelegramIndex++)
  {
    if (csm_arCbrRxTelegram[bCountTelegramIndex].bBus == bBus)
    {
      CBR_vStopReceiveTimer(bCountTelegramIndex);
      CBR_arRxTelegramState[bCountTelegramIndex].dwState |= CSM_C_SIGNAL_DATA_OUTDATED;
    }
  }
  #endif // CSM_S_REMOTE_STACK

  return CSM_C_NO_ERROR;
}

/************************************************************************
 * function    CBR_lSignalWrite
 * \doxydocu
 * \brief      Stores signal data.
 *
 *             Stores signal data in telegram buffer.
 *             If transmission type is CSM_C_TX_IMMEDIATELY, telegram has to be
 *             transmitted immediately, otherwise (CSM_C_TX_CYCLIC) as
 *             configured (for cyclic/event mixed modes cyclic).
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESA3
 *
 * \param[in]  dwSignalId        (I)
 *             signal identifier (signal name)
 *
 * \param[in]  *pvNewSignalData  (->I)
 *             pointer to data buffer
 *
 * \param[in]  bDataBufferLength (I)
 *             length of data buffer
 *
 * \param[in]  bTxType           (I)
 *             Allowed values:
 *                 CSM_C_TX_CYCLIC -   like in config.-Tabelle
 *                 CSM_C_TX_IMMEDIATELY -     immediately transmission
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR
 *
 * \access     application
 * \reentrant  yes
 *
 * \lastreview 26.09.06 CM-DI/ESA3-Battis
 *
 * \history_begin
 * 25.09.06 CM-DI/ESA3-Haffner
 * Changes for CSM proxy usage.
 *
 * 28.07.06  CM-DI/ESA3-Battis
 * Bugfix: Transmit Repeat Timer has never been started if telegram is
 * not valid when StartTx() is called.
 *
 * 26.07.04 CM-DI/ESU-3 Pröhl
 * Initial Revision.
 *
 * \history_end
 ************************************************************************/
LONG CBR_lSignalWrite(const DWORD dwSignalId, const void *pvNewSignalData, const BYTE bDataBufferLength, const BYTE bTxType)
{
  INT iInterruptOldFlag;
  LONG return_value = CSM_C_NO_ERROR;
  DWORD dwSigId = CSM_M_GET_SIGNAL_ID_FROM_VIRTUAL_SIGNAL_ID(dwSignalId);
  BYTE  bTelIndex;

  bTelIndex = (BYTE) CSM_M_GET_TELINDEX_FROM_SIGNAL_ID(dwSigId);

  if (pvNewSignalData == NULL)
  {
    // got null pointer with pvNewSignalData
    CSM_vAssert(FALSE);
    return (CSM_M_MAKE_ERROR(CSM_C_ERR_SEVERE_ERROR, CSM_C_ERR_LOC_CBR, CSM_E_NULL_POINTER));
  }

  if (bTxType > CSM_C_TX_TYPE_VALID_MASK)
  {
    // TX type is not valid
    CSM_vAssert(FALSE);
    return (CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CBR, CSM_E_INVALID_PARA));
  }

  if (bTelIndex < CBR_TX_MAX_NUMBER)
  {
    const CSM_TELEGRAM * prCsmTelegram = cbr_arTxTelegramIndex[bTelIndex];
    BYTE  bSignalIndex = (BYTE) CSM_M_GET_SIGINDEX_FROM_SIGNAL_ID(dwSigId);

    if (bSignalIndex < (prCsmTelegram->bNumberOfSignals))
    {
      // Signal ist gültig
      const CSM_SIGNAL * prCsmSignal               = prCsmTelegram->prSignals + bSignalIndex;
      const CSM_CBR_TX * prCbrTxCfg                = &csm_arCbrTxTelegram[bTelIndex];
      tCBR_TX_TELEGRAM_STATE * pCbrTxTelegramState = &CBR_arTxTelegramState[bTelIndex];
      BOOL fXmit = FALSE;
      BYTE abTempData[8];

      // lock TX buffer
      // jetzt können die Daten manipuliert werden
      CBR_LOCK_TX_DATA(iInterruptOldFlag);
      // create working copy of current telegram content
      CSM_vMemCopy(abTempData, prCbrTxCfg->pvBuffer, (WORD)prCbrTxCfg->bDlc);
      // write the signal data to the correct position in the working buffer
      return_value = CBR_lSignalWriteRaw(abTempData, prCbrTxCfg->bDlc, prCsmSignal->bStartBit,
                                         prCsmSignal->bLength, (BYTE *) pvNewSignalData,
                                         bDataBufferLength
                                         #if (CSM_S_CBR_BOTH_ENDIANS_SIGNALS == CSM_C_F_ON)
                                         ,prCsmSignal->bEndians
                                         #endif
                                        );

      if (return_value == CSM_C_NO_ERROR)
      {
        // first let a function check, if TX is required due to several criteria
        fXmit = cbr_fIsTxRequiredAfterSignalWrite(&abTempData[0], prCbrTxCfg, pCbrTxTelegramState, prCsmSignal, bTxType, bTelIndex, bSignalIndex);

        // then copy new data to TX buffer
        CSM_vMemCopy(prCbrTxCfg->pvBuffer, abTempData, (WORD) prCbrTxCfg->bDlc);
      }

      // unlock TX buffer
      CBR_UNLOCK_TX_DATA(iInterruptOldFlag);
      // check the result of required TX
      if (fXmit == TRUE)
      {
        pCbrTxTelegramState->fTransmit = TRUE;
        csm_vTriggerCsmDownstreamTask(CSM_C_DOWN_REASON_CBR_SEND);
      }
    }
    else
    {
      // ERROR
      return_value = (CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CBR, CSM_E_INVALID_PARA));
      // signal ID not valid
      CSM_vAssert(FALSE); 
    }
  }
  else
  {
    // ERROR
    return_value = (CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CBR, CSM_E_INVALID_PARA));

    // Only throw an Assert if the tel index is not 'deprecated'
    // Assume that param is invalid.
    // On deprecated tel indices only return with error value. Appl shall
    // check !
    if (bTelIndex != CBR_C_DEPRECATED_IDX)
    {
      // got invalid message ID
      CSM_vAssert(FALSE);
    }
  }

  return return_value;
}

/************************************************************************
 * function    CBR_lFullFrameSignalWrite
 * \doxydocu
 * \brief      Proxy (CCC) has requested the storage of a full frame due to changed signal data.
 *             Since this can collide with locally written gateway signals for the same frame,
 *             a special handling is required. This function should only be called (directly or indirectly) from the CCS.
 *
 *             Dedicated signal data have to be masked and then the standard signal write can be executed.
 *
 * \history_begin
 * 22.06.16 Andreas Borck
 * - initial version.
 *
 * \history_end
 ************************************************************************/
#if (CSM_S_CBR_GATEWAY == CSM_C_F_ON)
LONG CBR_lFullFrameSignalWrite(const DWORD dwSignalId, const void * pvNewSignalData, const BYTE bDataBufferLength, const BYTE bTxType)
{
  INT iInterruptOldFlag;
  LONG return_value = CSM_C_NO_ERROR;
  DWORD dwSigId = CSM_M_GET_SIGNAL_ID_FROM_VIRTUAL_SIGNAL_ID(dwSignalId);
  BYTE bTelIndex;
  BYTE bSignalIndex;
  BYTE * pabProxyMask;
  BYTE fXmit = FALSE;

  // ROM constant (default mask)
  static const BYTE abOpenMask[8] = {0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF};

  bTelIndex    = (BYTE) CSM_M_GET_TELINDEX_FROM_SIGNAL_ID(dwSigId);
  bSignalIndex = (BYTE) CSM_M_GET_SIGINDEX_FROM_SIGNAL_ID(dwSigId);

  // check used parameters
  if ((pvNewSignalData == NULL) || (bSignalIndex != 0))
  {
    // got null pointer with pvNewSignalData or the signal index is NOT the full frame signal
    // assumption: the full frame signal is always the first one within the signal descriptions !!!
    CSM_vAssert(FALSE);
    return (CSM_M_MAKE_ERROR(CSM_C_ERR_SEVERE_ERROR, CSM_C_ERR_LOC_CBR, CSM_E_NULL_POINTER));
  }

  if (bTelIndex < CBR_TX_MAX_NUMBER)
  {
    // the telegram index must be in range
    // get the correct TX telegram descriptor and configuration
    const CSM_TELEGRAM * prCsmTelegram           = cbr_arTxTelegramIndex[bTelIndex];
    const CSM_CBR_TX   * prCbrTxCfg              = &csm_arCbrTxTelegram[bTelIndex];
    tCBR_TX_TELEGRAM_STATE * pCbrTxTelegramState = &CBR_arTxTelegramState[bTelIndex];
    const CSM_SIGNAL   * prCsmSignal             = prCsmTelegram->prSignals + bSignalIndex;

    // create work buffers
    BYTE abOldData[8] = {0,0,0,0,0,0,0,0};
    BYTE abNewData[8] = {0,0,0,0,0,0,0,0};
    BYTE i;
    BYTE DLC = prCbrTxCfg->bDlc;

    if ((DLC > bDataBufferLength) || (DLC >8))
    {
      CSM_vAssert(FALSE);
      return (CSM_M_MAKE_ERROR(CSM_C_ERR_SEVERE_ERROR, CSM_C_ERR_LOC_CBR, CSM_E_INVALID_PARA));
    }

    // the signal index should be 0 but we check it anyway
    if (bSignalIndex < (prCsmTelegram->bNumberOfSignals))
    {
      // Signal index ist gültig
      if (prCsmTelegram->pvTelegramProxyMask != NULL)
      {
        pabProxyMask = (BYTE *) prCsmTelegram->pvTelegramProxyMask;
      }
      else
      {
        pabProxyMask = (BYTE *) &abOpenMask[0];
      }

      // Lock TX buffer
      CBR_LOCK_TX_DATA(iInterruptOldFlag);

      // only keep the new data signal bits that are allowed to be written (defined via the mask)
      for (i=0; i < DLC; i++)
      {
        #if (CSM_S_CBR_BIG_ENDIAN_SIGNALS == CSM_C_F_ON)
        abNewData[i]  =  (*((BYTE *)pvNewSignalData + ((DLC-1)-i))) & (* (pabProxyMask + i));
        abOldData[i]  =  (*((BYTE *)prCbrTxCfg->pvBuffer + i)) & ~(* (pabProxyMask + i));
        abNewData[i] |=  abOldData[i];
        #endif

        #if (CSM_S_CBR_LITTLE_ENDIAN_SIGNALS == CSM_C_F_ON)
        abNewData[i]  =  (*((BYTE *)pvNewSignalData + i))      &  (* (pabProxyMask + i));
        abOldData[i]  =  (*((BYTE *)prCbrTxCfg->pvBuffer + i)) & ~(* (pabProxyMask + i));
        abNewData[i] |=  abOldData[i];
        #endif

        #if (CSM_S_CBR_BOTH_ENDIANS_SIGNALS == CSM_C_F_ON)
        #error TODO
        #endif
      }

      // first let a function check, if TX is required due to several criteria
      fXmit = cbr_fIsTxRequiredAfterSignalWrite(&abNewData[0], prCbrTxCfg, pCbrTxTelegramState, prCsmSignal, bTxType, bTelIndex, bSignalIndex);
      // then copy new data back to TX buffer
      CSM_vMemCopy(prCbrTxCfg->pvBuffer, abNewData, (WORD) prCbrTxCfg->bDlc);
      // unlock TX buffer
      CBR_UNLOCK_TX_DATA(iInterruptOldFlag);
      // check the result of required TX
      if (fXmit == TRUE)
      {
        pCbrTxTelegramState->fTransmit = TRUE;
        csm_vTriggerCsmDownstreamTask(CSM_C_DOWN_REASON_CBR_SEND);
      }
    }
  }

  return return_value;
}
#endif

/************************************************************************
 * function    CBR_lMultipleSignalWrite
 * \doxydocu
 * \brief      Stores signal data.
 *
 *             Stores signal data in telegram buffer.
 *             If transmission type is CSM_C_TX_IMMEDIATELY, telegram has to be
 *             transmitted immediately, otherwise (CSM_C_TX_CYCLIC) as
 *             configured (for cyclic/event mixed modes cyclic).
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESA3
 *
 * \param[in]  dwSignalId        (I)
 *             signal identifier (signal name)
 *
 * \param[in]  *pvNewSignalData  (->I)
 *             pointer to data buffer
 *
 * \param[in]  bDataBufferLength (I)
 *             length of data buffer
 *
 * \param[in]  bTxType           (I)
 *             Allowed values:
 *                 CSM_C_TX_CYCLIC -   like in config.-Tabelle
 *                 CSM_C_TX_IMMEDIATELY -     immediately transmission
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR
 *
 * \access     application
 * \reentrant  yes
 *
 * \lastreview 21.11.2008 Pröhl
 *
 * \history_begin
 * 29.04.08 VTeam-Pröhl
 * Initial Revision.
 * \history_end
 ************************************************************************/
LONG CBR_lMultipleSignalWrite(const tCSM_MultipleSignalWrite *prSignalList, const BYTE  bNumberOfSignals)
{
  BYTE bCount;
  INT iInterruptOldFlag;
  LONG return_value = CSM_C_NO_ERROR;
  DWORD dwFirstSignalId;
  BYTE  bTelIndex;
  DWORD dwSigId;
  BYTE bSignalIndex;

  // get the signal ID from first entry of array
  dwFirstSignalId = CSM_M_GET_SIGNAL_ID_FROM_VIRTUAL_SIGNAL_ID(prSignalList[0].dwSignalId );

  // get telegramm index from signal ID
  bTelIndex = (BYTE) CSM_M_GET_TELINDEX_FROM_SIGNAL_ID(dwFirstSignalId);

  // loop over the signal list and perform some checks
  for (bCount = 0; bCount < bNumberOfSignals; bCount++)
  {
    dwSigId = CSM_M_GET_SIGNAL_ID_FROM_VIRTUAL_SIGNAL_ID(prSignalList[bCount].dwSignalId);

    if (bTelIndex != (BYTE) CSM_M_GET_TELINDEX_FROM_SIGNAL_ID(dwSigId))
    {
      // all signals must be in the same telegramm !!!
      return_value = CSM_M_MAKE_ERROR(CSM_C_ERR_SEVERE_ERROR, CSM_C_ERR_LOC_CBR, CSM_E_INVALID_PARA);
    }

    if (prSignalList[bCount].pvNewSignalData == NULL)
    {
      // got null pointer with pvNewSignalData
      CSM_vAssert(FALSE);
      return_value = CSM_M_MAKE_ERROR(CSM_C_ERR_SEVERE_ERROR, CSM_C_ERR_LOC_CBR, CSM_E_NULL_POINTER);
    }

    if (prSignalList[bCount].bTxType > CSM_C_TX_TYPE_VALID_MASK)
    {
      // TX type is not valid
      CSM_vAssert(FALSE);
      return_value = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CBR, CSM_E_INVALID_PARA);
    }
  }

  if (return_value == CSM_C_NO_ERROR)
  {
    if (bTelIndex < CBR_TX_MAX_NUMBER)
    {
      const CSM_TELEGRAM     * prCsmTelegram       = cbr_arTxTelegramIndex[bTelIndex];   // name is confusing, should be arpTxTelegramIndex, but would require changing ALL referencesd in code !!
      const CSM_CBR_TX       * prCbrTxCfg          = &csm_arCbrTxTelegram[bTelIndex];
      tCBR_TX_TELEGRAM_STATE * pCbrTxTelegramState = &CBR_arTxTelegramState[bTelIndex];

      BOOL fXmit = FALSE;                                                                // presume NO transmission yet

      // lock the TX buffer
      // jetzt können die Daten manipuliert werden
      CBR_LOCK_TX_DATA(iInterruptOldFlag);

      for (bCount = 0; bCount < bNumberOfSignals; bCount++)
      {
        BYTE bTxType = prSignalList[bCount].bTxType;                                       // extract the type already here
        dwSigId = CSM_M_GET_SIGNAL_ID_FROM_VIRTUAL_SIGNAL_ID(prSignalList[bCount].dwSignalId);
        bSignalIndex = (BYTE) CSM_M_GET_SIGINDEX_FROM_SIGNAL_ID(dwSigId);

        if (bSignalIndex < (prCsmTelegram->bNumberOfSignals))
        {
          // Signal ist gültig
          LONG temp_return_value;
          BYTE abTempData[8];
          const CSM_SIGNAL * prCsmSignal = prCsmTelegram->prSignals + bSignalIndex;
          // create working copy of current telegram content
          CSM_vMemCopy(abTempData, prCbrTxCfg->pvBuffer, (WORD)prCbrTxCfg->bDlc);
          // write the signal data to the correct position in the working buffer
          temp_return_value = CBR_lSignalWriteRaw(abTempData, prCbrTxCfg->bDlc, prCsmSignal->bStartBit,
                                                  prCsmSignal->bLength, (BYTE *) prSignalList[bCount].pvNewSignalData,
                                                  prSignalList[bCount].bDataBufferLength
                                                  #if (CSM_S_CBR_BOTH_ENDIANS_SIGNALS == CSM_C_F_ON)
                                                  ,prCsmSignal->bEndians
                                                  #endif
                                                 );

          if (temp_return_value == CSM_C_NO_ERROR)
          {
            // first let a function check, if TX is required due to several criteria
            fXmit = cbr_fIsTxRequiredAfterSignalWrite(&abTempData[0], prCbrTxCfg, pCbrTxTelegramState, prCsmSignal, bTxType, bTelIndex, bSignalIndex);

            // then copy new data back to TX buffer
            CSM_vMemCopy(prCbrTxCfg->pvBuffer, abTempData, (WORD) prCbrTxCfg->bDlc);
          }
          else
          {
            // store error code
            return_value = temp_return_value;
          }
        }
        else
        {
          // ERROR
          return_value = (CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CBR, CSM_E_INVALID_PARA));
          // signal ID not valid
          CSM_vAssert(FALSE);
        }
      }

      // unlock the TX buffer
      CBR_UNLOCK_TX_DATA(iInterruptOldFlag);

      // check the result of required TX
      if (fXmit == TRUE)
      {
        // set TX state of telegram and trigger the TX
        pCbrTxTelegramState->fTransmit = TRUE;
        csm_vTriggerCsmDownstreamTask(CSM_C_DOWN_REASON_CBR_SEND);
      }
    }
    else
    {
      // ERROR
      return_value = (CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CBR, CSM_E_INVALID_PARA));

      /* Only throw an Assert if the tel index is not 'deprecated'
       * Assume that param is invalid.
       * On deprecated tel indices only return with error value. Appl shall
       * check !
      */
      if (bTelIndex != CBR_C_DEPRECATED_IDX)
      {
        // got invalid message ID
        CSM_vAssert(FALSE);
      }
    }
  }
  return return_value;
}/*lint !e818 PQM_authorized_multi_123. Reason: Interface not defined as const */

/************************************************************************
 * function    cbr_fIsTxRequiredAfterSignalWrite
 * \doxydocu
 * \brief      Checks, if after SignalWriteRaw a tranmission of the frame must be triggered
 *
 * \param[in]  dwSignalId        (I)
 *             signal identifier (signal name)
 *
 * \param[out] *pvActSignalData (->O)
 *             pointer to data buffer
 *
 * \param[in]  bDataBufferLength (I)
 *             Length of output data buffer in bytes.
 *             best way: sizeof(databuffer)
 *
 * \param[out] *pdwSignalStatus (->O)
 *             pointer to global signal state (bitfield)
 *             possible values:
 *                CSM_C_SIGNAL_DATA_INVALID, CSM_C_SIGNAL_TIMEOUT, CSM_C_SIGNAL_DATA_OUTDATED
 *                CSM_C_SIGNAL_NOT_RECEIVED
 *
 * \return     BOOL
 *             True: TX required False: no TX required
 *
 * \access     CBR internal
 * \reentrant  no
 *
 * \lastreview 30.06.16 A.Borck
 *
 * \history_begin
 * 30.06.2016 A.Borck
 * Initial Revision.
 *
 * \history_end
 ************************************************************************/
static BOOL cbr_fIsTxRequiredAfterSignalWrite(const BYTE * pabTempData, const CSM_CBR_TX * prCbrTxCfg, tCBR_TX_TELEGRAM_STATE * pCbrTxTelegramState, const CSM_SIGNAL * prCsmSignal, BYTE bTxType, BYTE bTelIndex, BYTE bSignalIndex)
{
  // ATTENTION: it has been dicovered, that combined bit-values of bTxType are not properly handled by this function, since
  // it has been assumed, that only non-combined values are passed for bTxType.
  // Hence a modification has been done, so that the switch-case statements are evaluating real single-bit values only.
  // The bits CSM_C_TX_FCT_ACTIVE and CSM_C_TX_BUS_WAKE_TEMP are supressed, if evaluating the TX type within switch-case.
  BOOL fReturn = FALSE;

  #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
  // on local stack directly compare old and new buffer content
  BOOL fDataChanged = CSM_bMemCompare(pabTempData, prCbrTxCfg->pvBuffer, (WORD)prCbrTxCfg->bDlc);
  #endif

  #if (CSM_S_REMOTE_STACK == CSM_C_F_ON)
    // SignalWrite auf dem Proxy wird immer an RU weitergeleitet -> fReturn = TRUE
    pCbrTxTelegramState->fValid = TRUE;

    #if (CSM_S_CBR_TX_IF_APPL_ACTIVE == CSM_C_F_ON)
    if (bTxType & CSM_C_TX_FCT_ACTIVE)
    {
      cbr_vSetActiveFlag(pCbrTxTelegramState->dwActive, bSignalIndex);
    }
    else
    {
      cbr_vClrActiveFlag(pCbrTxTelegramState->dwActive, bSignalIndex);
    }
    #endif // CSM_S_CBR_TX_IF_APPL_ACTIVE

    if (bTxType & CSM_C_TX_BUS_WAKE_TEMP)
    {
      //this flag is needed to send the bus wake temp information to the V850
      pCbrTxTelegramState->fBusWakeTemp = TRUE;
    }

    fReturn = TRUE;
  #else
  {
    // on local stack now check, if TX is required
    switch(prCsmSignal->rMode.bTxMode)
    {
      //case CBR_TMSG_CYC_AND_SPONT_WITH_DEL:
      //case CBR_TMSG_SPONTAN_WITH_REP:
      //case CBR_TMSG_SPONTAN_WITH_DELAY:
      case CBR_TMSG_CYCLIC:
      {
        switch(bTxType & ~(CSM_C_TX_FCT_ACTIVE | CSM_C_TX_BUS_WAKE_TEMP))
        {
          case CSM_C_TX_AS_CONFIGURED:
          case CSM_C_TX_CYCLIC:
          {
            if (pCbrTxTelegramState->fValid == FALSE)
            {
              #if (CSM_S_CBR_RESYNCHRONIZED_CYCLE == CSM_C_F_ON)
              CBR_vStartTransmitRepeatTimer(bTelIndex);
              #endif
              pCbrTxTelegramState->fValid = TRUE;
              fReturn = TRUE;
            }
          }
          break;

          case CSM_C_TX_IMMEDIATELY:
          {
            // tx type immediately is not permitted here !
            CSM_vAssert(FALSE);
          }
          break;

          default:
          break;
        }
      }
      break; // CBR_TMSG_CYCLIC

      #if  (CSM_S_CBR_TX_IF_APPL_ACTIVE == CSM_C_F_ON)
      case CBR_TMSG_CYCLIC_IF_ACTIVE:
      {
        switch(bTxType & ~(CSM_C_TX_FCT_ACTIVE | CSM_C_TX_BUS_WAKE_TEMP))
        {
          case CSM_C_TX_AS_CONFIGURED:
          case CSM_C_TX_CYCLIC:
          {
            pCbrTxTelegramState->fValid = TRUE; // das Telegramm ist nun gültig
            if (bTxType & CSM_C_TX_FCT_ACTIVE)
            {
              if (cbr_fIsFunctionActive(pCbrTxTelegramState->dwActive) == FALSE)
              {
                fReturn = TRUE;
                pCbrTxTelegramState->bRepeatCount = prCbrTxCfg->bNumberOfRepetitions;
                #if  (CSM_S_CBR_RESYNCHRONIZED_CYCLE == CSM_C_F_ON)
                CBR_vStartTransmitRepeatTimer(bTelIndex);
                #endif
              }
              cbr_vSetActiveFlag(pCbrTxTelegramState->dwActive, bSignalIndex);
            }
            else
            {
              cbr_vClrActiveFlag(pCbrTxTelegramState->dwActive, bSignalIndex);
            }
          }
          break;

          case CSM_C_TX_IMMEDIATELY:
          {
            // tx type immediately is not permitted here !
            CSM_vAssert(FALSE);
          }
          break;
        }
      }
      break; //CBR_TMSG_CYCLIC_IF_ACTIVE
      #endif // CSM_S_CBR_TX_IF_APPL_ACTIVE

      case CBR_TMSG_SPONTAN:
      {
        switch(bTxType & ~(CSM_C_TX_FCT_ACTIVE | CSM_C_TX_BUS_WAKE_TEMP))
        {
          case CSM_C_TX_CYCLIC:
          {
            // tx type cyclic is not permitted here !
            CSM_vAssert(FALSE);
          }
          break;

          case CSM_C_TX_AS_CONFIGURED:
          case CSM_C_TX_IMMEDIATELY:
          {
            if (fDataChanged != 0)
            {
              //??
            }
            fReturn = TRUE;
          }
          break;

          default:
          break;
        }
      }
      break; // CBR_TMSG_SPONTAN

      case CBR_TMSG_CYCLIC_AND_SPONTAN:
      {
        switch(bTxType & ~(CSM_C_TX_FCT_ACTIVE | CSM_C_TX_BUS_WAKE_TEMP))
        {
          case CSM_C_TX_CYCLIC:
          {
            if (pCbrTxTelegramState->fValid == FALSE)
            {
              #if (CSM_S_CBR_RESYNCHRONIZED_CYCLE == CSM_C_F_ON)
              CBR_vStartTransmitRepeatTimer(bTelIndex);
              #endif
              pCbrTxTelegramState->fValid = TRUE; // telegramm is valid now
              fReturn = TRUE;
            }
          }
          break;

          case CSM_C_TX_AS_CONFIGURED:
          case CSM_C_TX_IMMEDIATELY:
          {
            if (pCbrTxTelegramState->fValid == FALSE)
            {
              pCbrTxTelegramState->fValid = TRUE; // telegramm is valid now
              #if (CSM_S_CBR_RESYNCHRONIZED_CYCLE == CSM_C_F_ON)
              CBR_vStartTransmitRepeatTimer(bTelIndex);
              #endif
              fReturn = TRUE;
            }

            if (fDataChanged == TRUE)
            {
              fReturn = TRUE;
            }
          }
          break;

          default:
          break;
        }
      }
      break; // CBR_TMSG_CYCLIC_AND_SPONTAN

      #if (CSM_S_CBR_TX_IF_APPL_ACTIVE == CSM_C_F_ON)
      case CBR_TMSG_CYC_IF_ACT_AND_SPONT:
      {
        switch(bTxType & ~(CSM_C_TX_FCT_ACTIVE | CSM_C_TX_BUS_WAKE_TEMP))
        {
          case CSM_C_TX_CYCLIC:
          {
            pCbrTxTelegramState->fValid = TRUE; // das Telegramm ist nun gültig
            if (bTxType & CSM_C_TX_FCT_ACTIVE)
            {
              if (cbr_fIsFunctionActive(pCbrTxTelegramState->dwActive) == FALSE)
              {
                fReturn = TRUE;
                pCbrTxTelegramState->bRepeatCount = prCbrTxCfg->bNumberOfRepetitions;
                #if (CSM_S_CBR_RESYNCHRONIZED_CYCLE == CSM_C_F_ON)
                CBR_vStartTransmitRepeatTimer(bTelIndex);
                #endif
              }
              cbr_vSetActiveFlag(pCbrTxTelegramState->dwActive, bSignalIndex);
            }
            else
            {
              cbr_vClrActiveFlag(pCbrTxTelegramState->dwActive, bSignalIndex);
            }
          }
          break;

          case CSM_C_TX_AS_CONFIGURED:
          case CSM_C_TX_IMMEDIATELY:
          {
            // Karnaugh variables
            BOOL fA = pCbrTxTelegramState->fValid;
            BOOL fB = ((bTxType & CSM_C_TX_FCT_ACTIVE) > 0);
            BOOL fC = fDataChanged;
            BOOL fD = (pCbrTxTelegramState->bRepeatCount > 0);

            if (bTxType & CSM_C_TX_FCT_ACTIVE)
            {
              #if (CSM_S_CBR_RESYNCHRONIZED_CYCLE == CSM_C_F_ON)
              if (cbr_fIsFunctionActive(pCbrTxTelegramState->dwActive) == FALSE)
              {
                CBR_vStartTransmitRepeatTimer(bTelIndex);
                cbr_vSetActiveFlag(pCbrTxTelegramState->dwActive, bSignalIndex);
              }
              #else
              cbr_vSetActiveFlag(pCbrTxTelegramState->dwActive, bSignalIndex);
              #endif
            }
            else
            {
              cbr_vClrActiveFlag(pCbrTxTelegramState->dwActive, bSignalIndex);
            }
            pCbrTxTelegramState->fValid = TRUE; // das Telegramm ist nun gültig

            /*    Karnaugh-Table
                  AB
                 |00 |01 |11 |10
                 -----------------
               CD 00| 0 | 1 | 1 | 0   A: valid before
                  01| d | d | 0 | 0   B: request active
                  11| d | d | 1 | 1   C: data value changed
                  10| 0 | 1 | 1 | 0   D: active at present (RepeatCount > 0)
                 -----------------
            */
            // karnaugh-minimized :
            if ((fB && (!fD)) || (fA && fC && fD))
            {
              if (!(fA && (!fB) && fC && fD))
              {
                pCbrTxTelegramState->bRepeatCount = prCbrTxCfg->bNumberOfRepetitions;
              }
              fReturn = TRUE;
            }
          }
          break;
        }
      }
      break; // CBR_TMSG_CYC_IF_ACT_AND_SPONT
      #endif // CSM_S_CBR_TX_IF_APPL_ACTIVE

      default:
      {
        // wrong message type !
        CSM_vAssert(FALSE);
      }
      break;
    } // switch prCsmSignal->rMode.bTxMode
  }
  #endif // CSM_S_REMOTE_STACK is OFF

  // LINT happy
  #if (CSM_S_CBR_TX_IF_APPL_ACTIVE == CSM_C_F_OFF)
  CSM_M_PARAMETER_INTENTIONALLY_UNUSED(bSignalIndex);
  #endif

  #if (CSM_S_CBR_RESYNCHRONIZED_CYCLE == CSM_C_F_OFF)
  CSM_M_PARAMETER_INTENTIONALLY_UNUSED(bTelIndex);
  #endif

  return fReturn;
}

/************************************************************************
 * function    CBR_lSignalRead
 * \doxydocu
 * \brief      Read signal data.
 *
 *             Read signal data from telegram buffer.
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESA3
 *
 * \param[in]  dwSignalId        (I)
 *             signal identifier (signal name)
 *
 * \param[out] *pvActSignalData (->O)
 *             pointer to data buffer
 *
 * \param[in]  bDataBufferLength (I)
 *             Length of output data buffer in bytes.
 *             best way: sizeof(databuffer)
 *
 * \param[out] *pdwSignalStatus (->O)
 *             pointer to global signal state (bitfield)
 *             possible values:
 *                CSM_C_SIGNAL_DATA_INVALID, CSM_C_SIGNAL_TIMEOUT, CSM_C_SIGNAL_DATA_OUTDATED
 *                CSM_C_SIGNAL_NOT_RECEIVED
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR
 *
 * \access     application
 * \reentrant  yes
 *
 * \lastreview  16.04.08 VTeam-Pröhl
 *
 * \history_begin
 * 18.10.06 CM-DI/ESA3-Haffner
 * Const for parameter pvActSignalData in CBR_lSignalRead() removed.
 * (see cbr_i.h \main\22)
 *
 * 26.07.04 CM-DI/ESU-3 Pröhl
 * Initial Revision.
 *
 * \history_end
 ************************************************************************/
LONG CBR_lSignalRead(const DWORD dwSignalId, void *pvActSignalData, const BYTE  bDataBufferLength, DWORD *pdwSignalStatus)
{
  INT iInterruptOldFlag;
  LONG return_value = CSM_C_NO_ERROR;
  DWORD dwSigId = CSM_M_GET_SIGNAL_ID_FROM_VIRTUAL_SIGNAL_ID(dwSignalId);

    // ensure that the variable mapped by the makro has been initialized already !!!
  CSM_vAssert(CBR_RX_MAX_NUMBER != 0);

  /* first check params */
  if (pdwSignalStatus == NULL)
  {
    return_value = CSM_M_MAKE_ERROR(CSM_C_ERR_SEVERE_ERROR, CSM_C_ERR_LOC_CBR, CSM_E_NULL_POINTER);
    /* got null pointer with pdwSignalStatus */
    CSM_vAssert(FALSE);
  }
  else
  {
    *pdwSignalStatus = 0;                              /* Init default-value. */
  }

  if (pvActSignalData == NULL)
  {
    if (pdwSignalStatus != NULL)
    {
      *pdwSignalStatus |= CSM_C_SIGNAL_DATA_INVALID;
    }
    return_value = CSM_M_MAKE_ERROR(CSM_C_ERR_SEVERE_ERROR, CSM_C_ERR_LOC_CBR, CSM_E_NULL_POINTER);
    // got null pointer with pbActSignalData
    CSM_vAssert(FALSE);
  }

  if (return_value == CSM_C_NO_ERROR)
  {
    BYTE  bTelIndex = (BYTE) CSM_M_GET_TELINDEX_FROM_SIGNAL_ID(dwSigId);

    if (bTelIndex < CBR_RX_MAX_NUMBER)
    {
      BYTE  bSignalIndex = (BYTE) CSM_M_GET_SIGINDEX_FROM_SIGNAL_ID(dwSigId);

      if (bSignalIndex < (cbr_arRxTelegramIndex[bTelIndex]->bNumberOfSignals))
      {
        // Signal ist gültig
        CBR_LOCK_RX_DATA(iInterruptOldFlag);
        // Jetzt können die Daten manipuliert werden
        return_value = CBR_lSignalReadRaw((BYTE *)csm_arCbrRxTelegram[bTelIndex].pvBuffer,
                                           csm_arCbrRxTelegram[bTelIndex].bDlc,
                                           ((cbr_arRxTelegramIndex[bTelIndex]->prSignals) + bSignalIndex)->bStartBit,
                                           ((cbr_arRxTelegramIndex[bTelIndex]->prSignals) + bSignalIndex)->bLength,
                                           (BYTE *) pvActSignalData,
                                           bDataBufferLength
                                           #if (CSM_S_CBR_BOTH_ENDIANS_SIGNALS == CSM_C_F_ON)
                                           , ((cbr_arRxTelegramIndex[bTelIndex]->prSignals) + bSignalIndex)->bEndians
                                           #endif
                                         );
        CBR_UNLOCK_RX_DATA(iInterruptOldFlag);

        if (return_value != CSM_C_NO_ERROR)
        {
          *pdwSignalStatus |= CSM_C_SIGNAL_DATA_INVALID; /*lint !e613 lint is wrong. is protected with pdwSignalStatus return_value */
          // buffer size too short !
          CSM_vAssert(FALSE);
        }
        else
        {
          // determine signal status:
          *pdwSignalStatus = CBR_arRxTelegramState[bTelIndex].dwState; /*lint !e613 lint is wrong. is protected with pdwSignalStatus return_value */
        }
      }
      else
      {
        return_value = (CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CBR, CSM_E_INVALID_PARA));
        // got invalid signal ID
        CSM_vAssert(FALSE); 
      }
    }
    else
    {
      return_value = (CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CBR, CSM_E_INVALID_PARA));
      *pdwSignalStatus |= CSM_C_SIGNAL_DATA_INVALID;  /*lint !e613 lint is wrong. is protected with pdwSignalStatus return_value */

      // Only throw an Assert if the tel index is not 'deprecated'
      // Assume that param is invalid.
      // On deprecated tel indices only return with error value. Appl shall check !
      if (bTelIndex != CBR_C_DEPRECATED_IDX)
      {
        // got invalid message ID
        CSM_vAssert(FALSE);
      }
    }
  }
  return return_value;
}

/************************************************************************
 * function    CBR_lMultipleSignalRead
 * \doxydocu
 * \brief      Read multiple signal data.
 *
 *             Read multiple signal data from telegram buffer.
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESA3
 *
 * \param[in]  *prSignals (I)
 *             pointer to signal data structure
 *
 * \param[in]  bNumberOfSignals (I)
 *             number of signals
 *
 * \return     LONG
 *             Error value: CSM_C_NO_ERROR
 *
 * \access     application
 * \reentrant  yes
 *
 * \lastreview   21.11.2008 Pröhl
 *
 * \history_begin
 * 29.04.08 VTeam-Pröhl
 * Initial Revision.
 * \history_end
 ************************************************************************/
LONG CBR_lMultipleSignalRead(tCSM_MultipleSignalRead *prSignalList, const BYTE  bNumberOfSignals)
{
  INT iInterruptOldFlag;
  LONG return_value = CSM_C_NO_ERROR;
  BYTE  bTelIndex;
  BYTE bCount;
  DWORD dwFirstSignalId;
  BYTE bSignalIndex;
  DWORD dwSigId;

  // ensure that the variable mapped by the makro has been initialized already !!!
  CSM_vAssert(CBR_RX_MAX_NUMBER != 0);

  // get the signal ID from first entry of array
  dwFirstSignalId = CSM_M_GET_SIGNAL_ID_FROM_VIRTUAL_SIGNAL_ID(prSignalList[0].dwSignalId );

  // get telegramm index from signal ID
  bTelIndex = (BYTE) CSM_M_GET_TELINDEX_FROM_SIGNAL_ID(dwFirstSignalId);

  // loop over the signal list and perform some checks
  for (bCount = 0; bCount < bNumberOfSignals; bCount++)
  {
    // get the signal ID
    dwSigId = CSM_M_GET_SIGNAL_ID_FROM_VIRTUAL_SIGNAL_ID(prSignalList[bCount].dwSignalId);

    if (bTelIndex != (BYTE) CSM_M_GET_TELINDEX_FROM_SIGNAL_ID(dwSigId))
    {
      // all signals must be within the same telegramm !!!
      return_value = CSM_M_MAKE_ERROR(CSM_C_ERR_SEVERE_ERROR, CSM_C_ERR_LOC_CBR, CSM_E_INVALID_PARA);
    }

    if (prSignalList[bCount].pdwSignalStatus == NULL)
    {
      return_value = CSM_M_MAKE_ERROR(CSM_C_ERR_SEVERE_ERROR, CSM_C_ERR_LOC_CBR, CSM_E_NULL_POINTER);
      // got null pointer with pdwSignalStatus
      CSM_vAssert(FALSE);
    }
    else
    {
      *prSignalList[bCount].pdwSignalStatus = 0; // Init default-value
    }

    if (prSignalList[bCount].pvActSignalData == NULL)
    {
      if (prSignalList[bCount].pdwSignalStatus != NULL)
      {
        *prSignalList[bCount].pdwSignalStatus |= CSM_C_SIGNAL_DATA_INVALID;
      }
      return_value = CSM_M_MAKE_ERROR(CSM_C_ERR_SEVERE_ERROR, CSM_C_ERR_LOC_CBR, CSM_E_NULL_POINTER);
      // got null pointer with pbActSignalData
      CSM_vAssert(FALSE);
    }
  }

  if (return_value == CSM_C_NO_ERROR)
  {
    // now go and process the reading
    if (bTelIndex < CBR_RX_MAX_NUMBER)
    {
      CBR_LOCK_RX_DATA(iInterruptOldFlag);
      for (bCount = 0; bCount < bNumberOfSignals; bCount++)
      {
        dwSigId = CSM_M_GET_SIGNAL_ID_FROM_VIRTUAL_SIGNAL_ID(prSignalList[bCount].dwSignalId);
        bSignalIndex = (BYTE) CSM_M_GET_SIGINDEX_FROM_SIGNAL_ID(dwSigId);

        if (bSignalIndex < (cbr_arRxTelegramIndex[bTelIndex]->bNumberOfSignals))
        {
          LONG temp_return_value;
          // Signal ist gültig
          // Jetzt können die Daten manipuliert werden.
          temp_return_value = CBR_lSignalReadRaw((BYTE *)csm_arCbrRxTelegram[bTelIndex].pvBuffer,
                                                 csm_arCbrRxTelegram[bTelIndex].bDlc,
                                                 ((cbr_arRxTelegramIndex[bTelIndex]->prSignals) + bSignalIndex)->bStartBit,
                                                 ((cbr_arRxTelegramIndex[bTelIndex]->prSignals) + bSignalIndex)->bLength,
                                                 (BYTE *) prSignalList[bCount].pvActSignalData,
                                                  prSignalList[bCount].bDataBufferLength
                                                 #if (CSM_S_CBR_BOTH_ENDIANS_SIGNALS == CSM_C_F_ON)
                                                 , ((cbr_arRxTelegramIndex[bTelIndex]->prSignals) + bSignalIndex)->bEndians
                                                 #endif
                                                );

          if (temp_return_value != CSM_C_NO_ERROR)
          {
            *prSignalList[bCount].pdwSignalStatus |= CSM_C_SIGNAL_DATA_INVALID; /*lint !e613 lint is wrong. is protected with pdwSignalStatus return_value */
            // buffer size too short !
            return_value = temp_return_value; // store error code
            CSM_vAssert(FALSE);
          }
          else
          {
            // determine signal status
            *prSignalList[bCount].pdwSignalStatus = CBR_arRxTelegramState[bTelIndex].dwState; /*lint !e613 lint is wrong. is protected with pdwSignalStatus return_value */
          }
        }
        else
        {
          return_value = (CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CBR, CSM_E_INVALID_PARA));
          // got invalid signal ID
          CSM_vAssert(FALSE);
        }
      }
      CBR_UNLOCK_RX_DATA(iInterruptOldFlag);
    }
    else
    {
      return_value = (CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CBR, CSM_E_INVALID_PARA));
      for (bCount = 0; bCount < bNumberOfSignals; bCount++)
      {
        // set all signals to invalid
        *prSignalList[bCount].pdwSignalStatus |= CSM_C_SIGNAL_DATA_INVALID;  /*lint !e613 lint is wrong. is protected with pdwSignalStatus return_value */
      }
      // Only throw an Assert if the tel index is not 'deprecated'
      // Assume that param is invalid.
      // On deprecated tel indices only return with error value. Appl shall
      // check !
      if (bTelIndex != CBR_C_DEPRECATED_IDX)
      {
        // got invalid message ID
        CSM_vAssert(FALSE);
      }
    }
  }
  return return_value;
}/*lint !e818 PQM_authorized_multi_123. Reason: Interface not defined as const */

/************************************************************************
 * function    CBR_vCDAReceiveInd
 * \doxydocu
 * \brief      Callback function from CDA to indicate new data.
 *
 *             Callback function from CDA to indicate new data
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESA3
 *
 * \param[in]  *prData
 *             CAN telegram
 *
 * \return     void
 *
 * \access     CAN-Task
 * \reentrant  yes, but only call from CAN-task
 *
 * \lastreview 21.11.2008 Pröhl
 *
 * \history_begin
 * 26.07.04 CM-DI/ESU-3 Pröhl
 * Initial Revision.
 *
 * \history_end
 ************************************************************************/
void CBR_vCDAReceiveInd(const CSM_CAN_RX_DATA* const prData)
{
  #if CSM_S_CBR_RX_MASK == CSM_C_F_ON
  // we have a LOCAL stack and RX is active
  CSM_CAN_RX_DATA rData;

  CSM_vMemCopy(&rData , prData, sizeof(rData));
  if (prData->bTelegramType == CSM_C_TEL_TYPE_CBR_RX)
  {
    // the telegram is CBR related
    if (prData->bTelegramIndex < CBR_RX_MAX_NUMBER)
    {
      // the telegramm index is valid according to the configured amount of RX telegramms
      const CSM_CBR_RX *prCbrRxCfg = &csm_arCbrRxTelegram[prData->bTelegramIndex];

      if ((cbr_arRxTelegramIndex[prData->bTelegramIndex] != NULL) &&
          (cbr_arRxTelegramIndex[prData->bTelegramIndex]->pvTelegramRxMask != NULL) )
      {
        DWORD dwCount;
        const BYTE *pMaskData = cbr_arRxTelegramIndex[prData->bTelegramIndex]->pvTelegramRxMask;
        for (dwCount=0; dwCount < prCbrRxCfg->bDlc; dwCount++)
        {
          rData.abData[dwCount] &= pMaskData[dwCount];
        }
      }
    }
  }
  cbr_vReceiveInd(&rData, 0x0000);
  #else
  cbr_vReceiveInd(prData, 0x0000);
  #endif // CSM_S_CBR_RX_MASK
}

/************************************************************************
 * function    CBR_vPROXYReceiveInd
 * \doxydocu
 * \brief      Callback function from PROXY to indicate new data.
 *
 *             Callback function from PROXY to indicate new data
 *             Design like written in CAN SW architecture
 *             document from group CM-DI/ESA3
 *
 * \param[in]  *prData
 *             CAN telegram
 *
 * \param[in]  dwSignalStatus
 *             Signal status
 *
 * \return     void
 *
 * \access     CAN-Task
 * \reentrant  yes, but only call from CAN-task
 *
 * \lastreview 21.11.2008 Pröhl
 *
 * \history_begin
 * 26.07.04 CM-DI/ESU-3 Pröhl
 * Initial Revision.
 *
 * \history_end
 ************************************************************************/
void CBR_vPROXYReceiveInd(const CSM_CAN_RX_DATA* const prData, const DWORD dwSignalStatus)
{
  cbr_vReceiveInd(prData, dwSignalStatus);
}

/************************************************************************
 * function    CBR_vCDAReceiveInd / CBR_vPROXYReceiveInd
 * \doxydocu
 * \brief      Generic function for data delivery from CDA/PROXY.
 *
 * \param[in]  *prData
 *             CAN telegram
 *
 * \param[in]  dwSignalStatus
 *             Signal status
 *
 * \return     void
 *
 * \access     CAN-Task
 * \reentrant  yes, but only call from CAN-task
 *
 * \lastreview 30.07.07 CM-DI/ERM13-Battis
 *
 * \history_begin
 * 29.07.07 VTeam Pröhl
 * handel to short telegramms.
 * 26.07.04 CM-DI/ESU-3 Pröhl
 * Initial Revision.
 *
 * \history_end
 ************************************************************************/
static void cbr_vReceiveInd(const CSM_CAN_RX_DATA* const prData, const DWORD dwSignalStatus)
{
  // Diese Funktion läuft auf CAN Task level
  INT iInterruptOldFlag;
  BYTE abXorData[8];
  BOOL fDataChanged;
  BYTE bSignalIndex;
  DWORD dwOldSignalStatus;
  DWORD dwTempSignalStatus = 0;

  // ensure that the variable mapped by the makro has been initialized already !!!
  CSM_vAssert(CBR_RX_MAX_NUMBER != 0);

  #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
  if (CBR_IS_STATUS(prData->bBus, CBR_STATUS_RX_RUNNING))
  #endif
  {
    // we have a LOCAL stack and RX is active
    if (prData->bTelegramType == CSM_C_TEL_TYPE_CBR_RX)
    {
      // the telegram is CBR related
      if (prData->bTelegramIndex < CBR_RX_MAX_NUMBER)
      {
        // the telegramm index is valid according to the configured amount of RX telegramms
        const CSM_CBR_RX *prCbrRxCfg = &csm_arCbrRxTelegram[prData->bTelegramIndex];
        const CSM_TELEGRAM *prCsmTelegram = cbr_arRxTelegramIndex[prData->bTelegramIndex];

        #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
        // in a LOCAL Stack: prevent from CBR timeout by restarting the timer
        CBR_vStopReceiveTimer(prData->bTelegramIndex);
        CBR_vStartReceiveTimer(prData->bTelegramIndex);
        #endif

        #if (CSM_S_CBR_DISCARD_MSG_SHORTER_DLC == CSM_C_F_ON)
        if (prData->bLengthCode < prCbrRxCfg->bDlc)
        {
          // the message is shorter than expected and the feature says: discard it then
          // so bail out of the entire function here
          return;
        }
        #endif

        CBR_LOCK_RX_DATA(iInterruptOldFlag);
        if (prData->bLengthCode < prCbrRxCfg->bDlc)
        {
          // if the received telegramm is to short than add the rest from the actual buffer to avoid ghost indications
          CSM_vMemCopy((void*) (prData->abData + prData->bLengthCode), (((BYTE *)prCbrRxCfg->pvBuffer) + prData->bLengthCode), (prCbrRxCfg->bDlc - prData->bLengthCode));
        }
        fDataChanged = CSM_bMemCompare(prCbrRxCfg->pvBuffer,prData->abData,(WORD)prCbrRxCfg->bDlc);

        dwOldSignalStatus = CBR_arRxTelegramState[prData->bTelegramIndex].dwState;

        #if (CSM_S_REMOTE_STACK == CSM_C_F_ON)
        CBR_arRxTelegramState[prData->bTelegramIndex].dwState = dwSignalStatus;
        #else
        {
          #if (CSM_S_CBR_RX_MSG_CRC == CSM_C_F_ON)
          // Now we can check the CRC of a RX telegram
          if (TRUE == cbr_fCheckRxCRC(prData->bTelegramIndex, prData->abData, prData->bLengthCode) )
          {
            // no action if ok
          }
          else
          {
            // otherwise we mark data as invalid and the reason as well (temp storage first)
            dwTempSignalStatus = CSM_C_SIGNAL_DATA_CRC_MISSMATCH | CSM_C_SIGNAL_DATA_INVALID;
          }
          #endif

          #if (CSM_S_CBR_RX_MSG_ROLL_COUNTER == CSM_C_F_ON)
          // Now we can check the rolling counter of a RX telegram
          if (TRUE == cbr_fCheckRxRollCounter(prData->bTelegramIndex, prData->abData) )
          {
            // no action if ok
          }
          else
          {
            // data might be corrct but counter is definitely wrong (temp storage)
            dwTempSignalStatus |= CSM_C_SIGNAL_ROLL_COUNTER_FAIL;
          }
          #endif

          if ((dwOldSignalStatus & CSM_C_SIGNAL_NOT_RECEIVED) == CSM_C_SIGNAL_NOT_RECEIVED)
          {
            // Wenn CSM_C_SIGNAL_NOT_RECEIVED noch gesetzt ist, dann ist es der erste Empfang des Signals
            CBR_arRxTelegramState[prData->bTelegramIndex].dwState = dwTempSignalStatus |
                                                                  CSM_C_SIGNAL_ARRIVED |
                                                                  CSM_C_SIGNAL_ARRIVED_FIRST;
          }
          else
          {
            CBR_arRxTelegramState[prData->bTelegramIndex].dwState = dwTempSignalStatus |
                                                                   CSM_C_SIGNAL_ARRIVED;
          }
        }
        #endif

        // now check which signals of the telegramm have changed
        for (bSignalIndex = 0; bSignalIndex < cbr_arRxTelegramIndex[prData->bTelegramIndex]->bNumberOfSignals; bSignalIndex++)
        {
          if (((((prCsmTelegram->prSignals) + bSignalIndex)->rMode.bIndicationMode) & CSM_C_CALLBACK_LEVEL_MASK) == CSM_C_ALWAYS_IND)
          {
            fDataChanged = 1;
          }
        }

        #if (CSM_S_REMOTE_STACK == CSM_C_F_ON)
        // Route timeouts through proxy
        if ((dwSignalStatus & CSM_C_SIGNAL_TIMEOUT) != 0)
        {
          CBR_vReceiveTimeout(prData->bTelegramIndex);
        }
        else
        #endif
        {
          if ((fDataChanged != 0)
           || ((dwOldSignalStatus & (CSM_C_SIGNAL_TIMEOUT |
                                     CSM_C_SIGNAL_DATA_INVALID |
                                     CSM_C_SIGNAL_DATA_OUTDATED |
                                     CSM_C_SIGNAL_NOT_RECEIVED)) != 0))
          {
            // ok, something has changed.
            // It is not needed to check the DLC for abXorData here because later
            // on only the valid bytes are evaluated by CBR_lSignalReadRaw()
            abXorData[0] = *(((BYTE*)prCbrRxCfg->pvBuffer) + 0) ^ prData->abData[0];
            abXorData[1] = *(((BYTE*)prCbrRxCfg->pvBuffer) + 1) ^ prData->abData[1];
            abXorData[2] = *(((BYTE*)prCbrRxCfg->pvBuffer) + 2) ^ prData->abData[2];
            abXorData[3] = *(((BYTE*)prCbrRxCfg->pvBuffer) + 3) ^ prData->abData[3];
            abXorData[4] = *(((BYTE*)prCbrRxCfg->pvBuffer) + 4) ^ prData->abData[4];
            abXorData[5] = *(((BYTE*)prCbrRxCfg->pvBuffer) + 5) ^ prData->abData[5];
            abXorData[6] = *(((BYTE*)prCbrRxCfg->pvBuffer) + 6) ^ prData->abData[6];
            abXorData[7] = *(((BYTE*)prCbrRxCfg->pvBuffer) + 7) ^ prData->abData[7];

            if (prData->bLengthCode > prCbrRxCfg->bDlc)
            {
              // use the length from the configuration
              CSM_vMemCopy(prCbrRxCfg->pvBuffer,prData->abData,(WORD)prCbrRxCfg->bDlc);
            }
            else
            {
              // use the length of the received number of bytes
              CSM_vMemCopy(prCbrRxCfg->pvBuffer,prData->abData,(WORD)prData->bLengthCode);
            }

            for (bSignalIndex = 0; bSignalIndex < prCsmTelegram->bNumberOfSignals; bSignalIndex++)
            {
              BYTE bLen = ((prCsmTelegram->prSignals) + bSignalIndex)->bLength;
              BYTE bRxIndicationMode = ((((prCsmTelegram->prSignals) + bSignalIndex)->rMode.bIndicationMode) & CSM_C_CALLBACK_LEVEL_MASK);

              #if (CSM_S_CBR_USES_DIRECT_CALLS == CSM_C_F_ON)
              BOOL fCallbackDirect = (((((prCsmTelegram->prSignals) + bSignalIndex)->rMode.bIndicationMode) & CSM_C_CALLBACK_DIRECT) > 0);
              #endif

              BYTE abTempData[8] = {0};
              BYTE bTempByte;

              if (bRxIndicationMode != CSM_C_NEVER_IND)
              {
                if (bRxIndicationMode == CSM_C_IND_AFTER_CHANGE)
                {
                  CBR_lSignalReadRaw(abXorData,
                                     prCbrRxCfg->bDlc,
                                     ((prCsmTelegram->prSignals) + bSignalIndex)->bStartBit,
                                     bLen,
                                     abTempData,
                                     (BYTE)((bLen+7)>>3)
                                     #if (CSM_S_CBR_BOTH_ENDIANS_SIGNALS == CSM_C_F_ON)
                                     , ((prCsmTelegram->prSignals) + bSignalIndex)->bEndians
                                     #endif
                                    );
                  bTempByte = abTempData[0] | abTempData[1] | abTempData[2] | abTempData[3] | abTempData[4] | abTempData[5] | abTempData[6] | abTempData[7];
                }
                else
                {
                  // jetzt bleibt nur noch CSM_C_ALWAYS_IND über
                  bTempByte = 0xff;
                }
                if ((bTempByte != 0x00)  ||
                    ((dwOldSignalStatus & (CSM_C_SIGNAL_TIMEOUT |
                                           CSM_C_SIGNAL_DATA_INVALID |
                                           CSM_C_SIGNAL_DATA_OUTDATED |
                                           CSM_C_SIGNAL_NOT_RECEIVED)) != 0))
                {
                  // Das Signal hat sich geändert bitte Callback Funktionen aufrufen
                  #if (CSM_S_CBR_USES_DIRECT_CALLS == CSM_C_F_ON)
                  if (fCallbackDirect == TRUE)
                  {
                    // delivery on downstream task level
                    const CSM_TELEGRAM *prCsmTelegram = cbr_arRxTelegramIndex[prData->bTelegramIndex];
                    const CSM_SIGNAL *pCsmSignal = ((prCsmTelegram->prSignals) + bSignalIndex);
                    DWORD dwSignalId = CSM_M_MAKE_SIGNAL_ID(prData->bTelegramIndex, bSignalIndex);
                    BYTE bCount;

                    if (CSM_M_CHECK_FOR_VALID_SIGNAL_ID(&dwSignalId))
                    {
                      for (bCount = 0; bCount < pCsmSignal->bMaxCallbackFunctions; bCount++)
                      {
                        if (pCsmSignal->pfvSignalInd1[bCount] != NULL)
                        {
                          // invoke Callback function
                          pCsmSignal->pfvSignalInd1[bCount](dwSignalId, CBR_arRxTelegramState[prData->bTelegramIndex].dwState);
                        }
                      } // loop
                    }
                  }
                  else
                  #endif
                  {
                    // delivery on upstream task level
                    cbr_vSetIndicationFlag(prData->bTelegramIndex, bSignalIndex);
                    csm_vTriggerCsmUpstreamTask(CSM_C_IND_REASON_CBR_SIGNAL_IND);
                  }
                }
              }
            }
          }
        }
        CBR_UNLOCK_RX_DATA(iInterruptOldFlag);
      }
      else
      {
        // Index zu groß
      }
    }
    else
    {
      // Telegramm Type falsch
    }
  } // RX_RUNNING
}/*lint !e715 PQM_authorized_multi_122. Reason: Variable is a preset by interface, but not used intentionally */

/*******************************************************************************
 * function    CBR_vProcessTransmit
 * \doxydocu
 * \brief      Transmit data.
 *
 *             Transmit data to CAN bus and calculate the timer handling.
 *             This function is called from downstream-task after trigger with CSM_C_DOWN_REASON_CBR_SEND.
 *
 * \return     void
 *
 * \access     CAN-Task
 * \reentrant  no, but only call by CAN-Task
 *
 * \lastreview 26.09.06 CM-DI/ESA3-Battis
 *
 * \history_begin
 * 25.09.06 CM-DI/ESA3-Haffner
 * Changes for CSM proxy usage.
 *
 * 22.09.04 CM-DI/ESA3-Pröhl
 * Initial Revision.
 *
 * 27.06.06 CM-DI/ESA3-Haffner
 * Remote stack calls CSM_PROXY_lCbrSendTelegram() instead of CDA_lSend().
 *
 * \history_end
 ******************************************************************************/
void CBR_vProcessTransmit(void)
{
  #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
  INT iInterruptOldFlag;
  CSM_CAN_TX_DATA rCanData;
  #endif

  BYTE bCount;
  #if (CSM_S_CBR_TX_MSG_ROLL_COUNTER == CSM_C_F_ON)
  #if (CSM_S_CBR_TX_MSG_ROLL_COUNTER_GROUPS ==  CSM_C_F_ON)
  BYTE bGroupCount;
  BYTE bGroupCount2;
  BYTE bGroup;
  BYTE bLink;
  BYTE bLink2;
  #endif
  #endif

  #if (CSM_S_VARIANTS_MANAGEMENT == CSM_C_F_ON)
  #if (CSM_S_SLOT_BASED_CONFIG == CSM_C_F_ON)
  #if (CSM_S_RUNTIME_FEATURE_BASED_CONFIG == CSM_C_F_ON)
  for (bCount = 0; bCount < CBR_TX_MAX_NUMBER; bCount++)
  {
    // make sure that restricted TX frames can't neither trigger the rolling counter group mechanism nor be sent further down
    if (0x01 == CSM_APL_bGetTxRestriction(bCount))
      CBR_arTxTelegramState[bCount].fTransmit = FALSE;
  }
  #endif
  #endif
  #endif

  #if (CSM_S_CBR_TX_MSG_ROLL_COUNTER == CSM_C_F_ON)
  #if (CSM_S_CBR_TX_MSG_ROLL_COUNTER_GROUPS ==  CSM_C_F_ON)
  // handle rolling counter groups before transmitting anything
  // first we clear all group-handled markers
  for (bCount = 0; bCount < CBR_TX_MAX_NUMBER; bCount++)
  {
    cbr_aTxRollingCounterAdmin[bCount].bGroupHandled = 0;
  }

  // now loop over all TX frames
  for (bCount = 0; bCount < CBR_TX_MAX_NUMBER; bCount++)
  {
    if (CBR_arTxTelegramState[bCount].fTransmit == TRUE)
    {
      // we start at the position of the first configured frame of a group (a group is a closed loop)
      if (((pTxRollingCounterCfg + bCount)->bGroupAndLen != 0) && (cbr_aTxRollingCounterAdmin[bCount].bGroupHandled != 1 ))
      {
        BYTE bGroupIsInDelay;   // this is the marker, that at least 1 frame of the group is in TX delay -> entire group is delayed

        // a TX ready telegramm has been identified to belong to a group AND the group has not been handled yet
        bGroupCount  = ((pTxRollingCounterCfg + bCount)->bGroupAndLen & 0x0F);   // get number of group members (including the current one)
        bGroupCount2 = bGroupCount;
        bGroup       = (pTxRollingCounterCfg + bCount)->bGroupAndLen & 0xF0;     // get current group nr
        bLink        = (pTxRollingCounterCfg + bCount)->bLinkIndex;              // get the telegramm index of the next group member
        bLink2       = bLink;

        bGroupIsInDelay = FALSE;                                                 // assumption so far
        while(bGroupCount)                                                       // as long as group members are to be handled (the current telegramm will be handled as last in the following loop)
        {
          bGroupCount -= 1;                                                      // decrease loop counter
          if (((pTxRollingCounterCfg + bLink)->bGroupAndLen & 0xF0) == bGroup)   // ensure that the next linked entry belongs to current group
          {
            // linked entry is of same group
            #if (CSM_S_VARIANTS_MANAGEMENT == CSM_C_F_ON)
            #if (CSM_S_SLOT_BASED_CONFIG == CSM_C_F_ON)
            #if (CSM_S_RUNTIME_FEATURE_BASED_CONFIG == CSM_C_F_ON)
            if (0x00 == CSM_APL_bGetTxRestriction(bLink))                        // prevent, that un-restricted TX frame triggers a restricted one
            #endif
            #endif
            #endif
            {
              CBR_arTxTelegramState[bLink].fTransmit = TRUE;                     // mark linked next TX telegram for transmission
              cbr_aTxRollingCounterAdmin[bLink].bGroupImDelay = FALSE;           // assumption for now
              if (CBR_arTxTelegramState[bLink].fDelayTransmit)
              {
                bGroupIsInDelay = TRUE;                                          // mark the entire group is in delay
              }

              cbr_aTxRollingCounterAdmin[bLink].bGroupHandled = 1;               // mark the linked next telegram as group-handled
            }
          }
          bLink = (pTxRollingCounterCfg + bLink)->bLinkIndex;                // get next linked TX telegram of group
        }
        if (bGroupIsInDelay) // if the assumption above (bGroupIsInDelay = FALSE) was wrong:
        {
          // this loop again starts from the current TX entry
          while(bGroupCount2)
          {
            // any group entry "delay flag" has to be duplicated to ALL the other entries
            bGroupCount2 -= 1;
            if (((pTxRollingCounterCfg + bLink2)->bGroupAndLen & 0xF0) == bGroup)   // ensure that the next linked entry belongs to current group
            {

              if (CBR_arTxTelegramState[bLink2].fTransmit)
              {
                cbr_aTxRollingCounterAdmin[bLink2].bGroupImDelay = TRUE;        // correct the assumption from above if needed
              }
            }
            bLink2 = (pTxRollingCounterCfg + bLink2)->bLinkIndex;             // get next linked TX telegram of group
          }
        }
        else
        {
          // hier könnte man jetzt die Rolling counter sofort neu berechnen, um sicherzustellen, dass sie wirklich alle den gleichen Wert haben,
          // denn die grosse Schleife weiter unten könnte z.B. durch ein BUS OFF unterbrochen werden, mit zunächst mal ungeklärten Konsequenzen
          // für die Synchronität der Rolling Counter innerhalb einer Gruppe.
        }

        // the entire group of the origin TX telegram has been finished
        // we can continue to check the next TX ready telegram (of another group)
      }
    }
  }
  #endif
  #endif

  // loop over all TX telegram states
  for (bCount = 0; bCount < CBR_TX_MAX_NUMBER; bCount++)
  {
    if ((CBR_arTxTelegramState[bCount].fTransmit == TRUE)
        #if (CSM_S_CBR_MIN_DELAY == CSM_C_F_ON)
        && (CBR_arTxTelegramState[bCount].fDelayTransmit == FALSE)
        #endif
       )
    {
      #if (CSM_S_CBR_TX_MSG_ROLL_COUNTER == CSM_C_F_ON)
      #if (CSM_S_CBR_TX_MSG_ROLL_COUNTER_GROUPS ==  CSM_C_F_ON)
      if (cbr_aTxRollingCounterAdmin[bCount].bGroupImDelay)
        continue;
      #endif
      #endif

      if (CBR_IS_STATUS(csm_arCbrTxTelegram[bCount].bBus, CBR_STATUS_TX_RUNNING))
      {
        if (csm_arCbrTxTelegram[bCount].bMessageType != CBR_TMSG_NO_TRANSMISSION)
        {
          #if (CSM_S_REMOTE_STACK == CSM_C_F_ON)
          {
            BYTE  bTelegramTxType = CSM_C_TX_AS_CONFIGURED;

            #if (CSM_S_CBR_TX_IF_APPL_ACTIVE == CSM_C_F_ON)
            if (cbr_fIsFunctionActive(CBR_arTxTelegramState[bCount].dwActive) == TRUE)
            {
              bTelegramTxType = CSM_C_TX_AS_CONFIGURED | CSM_C_TX_FCT_ACTIVE;
            }
            #endif // CSM_S_CBR_TX_IF_APPL_ACTIVE

            if (CBR_arTxTelegramState[bCount].fBusWakeTemp == TRUE)
            {
              CBR_arTxTelegramState[bCount].fBusWakeTemp = FALSE;
              bTelegramTxType |= CSM_C_TX_BUS_WAKE_TEMP;
            }

            CBR_arTxTelegramState[bCount].fTransmit = FALSE;

            CSM_PROXY_lCbrSendTelegram (csm_arCbrTxTelegram[bCount].bBus,
                                        CSM_C_TEL_TYPE_CBR_TX,
                                        bCount,
                                        bTelegramTxType);
          }
          #else // no CSM_S_REMOTE_STACK configured
          {
            // csm_arCbrTxTelegram[bCount].pvBuffer muss geschützt werden
            CBR_LOCK_TX_DATA(iInterruptOldFlag);

            #if (CSM_S_CBR_TX_MSG_ROLL_COUNTER == CSM_C_F_ON)
            // This Feature allows to have a rolling counter in EVERY transmission of the message
            // we delegate the handling to a dedicated function
            // we modify the telegramm data (the counter signal) within that function
            cbr_vHandleRollingCounter(bCount);
            #endif

            #if (CSM_S_CBR_TX_MSG_CRC == CSM_C_F_ON)
            // This feature allows to have a CRC in EVERY transmission of the message
            // we delegate the handling to a dedicated function
            // we modifiy the telegram data (the CRC signal) within that function
            cbr_vHandleTxCRC(bCount);
            #endif

            rCanData.dwIdentifier   = csm_arCbrTxTelegram[bCount].dwCanId;
            rCanData.bBus           = csm_arCbrTxTelegram[bCount].bBus;
            rCanData.bLengthCode    = csm_arCbrTxTelegram[bCount].bDlc;
            rCanData.bTelegramType  = CSM_C_TEL_TYPE_CBR_TX;
            rCanData.bTelegramIndex = bCount;
            rCanData.pbData         = csm_arCbrTxTelegram[bCount].pvBuffer;

            #if (CSM_S_CDA_TRANSMIT_TIMEOUT == CSM_C_F_ON)
            // Nissan LCN2 requires a timeout in case of disturbed transmission
            rCanData.dwTimeout   = CBR_TX_TIMEOUT; // Controller Repeat timeout
            #endif

            if (CDA_lSend(&rCanData) == CSM_C_NO_ERROR)     // transmit via CDA
            {
              CBR_arTxTelegramState[bCount].fTransmit = FALSE;

              #if (CSM_S_CBR_MIN_DELAY == CSM_C_F_ON)
              CBR_vStartTransmitMinDelayTimer(bCount);
              CBR_arTxTelegramState[bCount].fDelayTransmit = (csm_arCbrTxTelegram[bCount].wMinDist != 0);
              #endif
            }
            #if 0 // TO BE CHECKED !! This else is reached in BusOff condition
            else CSM_vAssert(FALSE);
            #endif

            CBR_UNLOCK_TX_DATA(iInterruptOldFlag);
          }
          #endif // CSM_S_REMOTE_STACK
        }
        else
        {
          CBR_arTxTelegramState[bCount].fTransmit = FALSE;
        }
      }

      #if ((CSM_S_CBR_RESYNCHRONIZED_CYCLE == CSM_C_F_OFF) && (CSM_S_REMOTE_STACK == CSM_C_F_OFF))
      switch(csm_arCbrTxTelegram[bCount].bMessageType)
      {
        case CBR_TMSG_CYCLIC:
        case CBR_TMSG_CYCLIC_AND_SPONTAN:
        {
          CBR_vStartTransmitRepeatTimer(bCount);
        }
        break;

        #if (CSM_S_CBR_TX_IF_APPL_ACTIVE == CSM_C_F_ON)
        case CBR_TMSG_CYCLIC_IF_ACTIVE:
        case CBR_TMSG_CYC_IF_ACT_AND_SPONT:
        {
          if (cbr_fIsFunctionActive(CBR_arTxTelegramState[bCount].dwActive) == TRUE)
          {
            CBR_vStartTransmitRepeatTimer(bCount);
          }
          else
          {
            if (CBR_arTxTelegramState[bCount].bRepeatCount > 0)
            {
              CBR_arTxTelegramState[bCount].bRepeatCount--;
              CBR_vStartTransmitRepeatTimer(bCount);
            }
          }
        }
        break;
        #endif // CSM_S_CBR_TX_IF_APPL_ACTIVE

        case CBR_TMSG_SPONTAN:
        break;

        case CBR_TMSG_NO_TRANSMISSION:
        break;

        default:
        {
          CSM_vAssert(FALSE);
        }
        break;
      }
      #endif // CSM_S_CBR_RESYNCHRONIZED_CYCLE && CSM_S_REMOTE_STACK
    }
  }
}

/*******************************************************************************
 * function    CBR_vProcessTimeout
 * \doxydocu
 * \brief      Handle timer event.
 *
 *             Handle timer event.
 *             This function is called from downstream-task after trigger with CSM_C_DOWN_REASON_CBR_TIMEOUT.
 *
 * \return     void
 *
 * \access     CAN-Task
 * \reentrant  no, but only call by CAN-Task
 *
 * \lastreview 21.11.2008 Pröhl
 *
 * \history_begin
 * 22.09.04 CM-DI/ESA3-Pröhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void CBR_vProcessTimeout(void)
{
  BYTE bCountTelegramIndex;
  BYTE bCountSignalIndex;
  #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
  BOOL fXmit = FALSE;
  #endif

  // ensure that the variable mapped by the makro has been initialized already !!!
  CSM_vAssert(CBR_RX_MAX_NUMBER != 0);
  CSM_vAssert(CBR_TX_MAX_NUMBER != 0);

  // if we have a local stack, then we handle TX here
  #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
  // Handle TX telegrams first
  for (bCountTelegramIndex = 0; bCountTelegramIndex < CBR_TX_MAX_NUMBER; bCountTelegramIndex++)
  {
    // check if TX is required due to timeout
    if (CBR_arTxTelegramState[bCountTelegramIndex].fRepeatTimeout == TRUE)
    {
      // reset TX marker
      CBR_arTxTelegramState[bCountTelegramIndex].fRepeatTimeout = FALSE;
      // force TX
      CBR_arTxTelegramState[bCountTelegramIndex].fTransmit = TRUE;

      // handle special feature for resyncronisation
      #if (CSM_S_CBR_RESYNCHRONIZED_CYCLE == CSM_C_F_ON)
      switch(csm_arCbrTxTelegram[bCountTelegramIndex].bMessageType)
      {
        case CBR_TMSG_CYCLIC:
        case CBR_TMSG_CYCLIC_AND_SPONTAN:
        {
          CBR_vStartTransmitRepeatTimer(bCountTelegramIndex);
        }
        break;

        #if (CSM_S_CBR_TX_IF_APPL_ACTIVE == CSM_C_F_ON)
        case CBR_TMSG_CYCLIC_IF_ACTIVE:
        case CBR_TMSG_CYC_IF_ACT_AND_SPONT:
        {
          if (cbr_fIsFunctionActive(CBR_arTxTelegramState[bCountTelegramIndex].dwActive) == TRUE)
          {
            CBR_vStartTransmitRepeatTimer(bCountTelegramIndex);
          }
          else
          {
            if (CBR_arTxTelegramState[bCountTelegramIndex].bRepeatCount > 0)
            {
              CBR_arTxTelegramState[bCountTelegramIndex].bRepeatCount--;
              CBR_vStartTransmitRepeatTimer(bCountTelegramIndex);
            }
          }
        }
        break;
        #endif // CSM_S_CBR_TX_IF_APPL_ACTIVE

        default:
        {
          CSM_vAssert(FALSE);
        }
        break;
      }
      #endif // CSM_S_CBR_RESYNCHRONIZED_CYCLE

    #if (CSM_S_CBR_MIN_DELAY == CSM_C_F_ON)
    }
    if (CBR_arTxTelegramState[bCountTelegramIndex].fTransmit == TRUE)
    {
    #endif
      fXmit = TRUE;
    }
  }

  // all TX telegrams have been handled, so if TX required, trigger task now
  if (fXmit == TRUE)
  {
    csm_vTriggerCsmDownstreamTask(CSM_C_DOWN_REASON_CBR_SEND);
  }
  #endif // CSM_S_REMOTE_STACK

  // Handle RX Telegrams now
  for (bCountTelegramIndex = 0; bCountTelegramIndex < CBR_RX_MAX_NUMBER; bCountTelegramIndex++)
  {
    // check the timeout flag for timeout occurence
    if (CBR_arRxTelegramState[bCountTelegramIndex].fReceiveTimeout == TRUE)
    {
      // by setting the cbr_bRxGrpMask to 0x00, the next if-statement will
      // always be TRUE (not relevant) and the feature has no effect
      #if (CSM_S_CBR_RX_TIMEOUT_GROUPS == CSM_C_F_ON)
      if ((csm_arCbrRxTelegram[bCountTelegramIndex].bRxGroup & cbr_bRxGrpMask) == 0)
      #endif
      {
        INT iInterruptOldFlag;

        // reset the timeout flag
        CBR_arRxTelegramState[bCountTelegramIndex].fReceiveTimeout = FALSE;

        #if (CSM_S_TRACE == CSM_C_F_ON)
          #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
            #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
            ETG_TRACE_USR2(("CBR_vProcessTimeout - TelIndex:%02x",
            (tU8)  bCountTelegramIndex) );
            #endif
          #endif
        #endif

        // first reset flag OUTDATED
        CBR_arRxTelegramState[bCountTelegramIndex].dwState &= ~(CSM_C_SIGNAL_DATA_OUTDATED);
        // then mark the timeout
        CBR_arRxTelegramState[bCountTelegramIndex].dwState |= CSM_C_SIGNAL_TIMEOUT;

        #if (CSM_S_TIMEOUT_SIGNAL_VALUES == CSM_C_F_ON)
        // set timeout default data
        // in case of no timeout default data INVALID flag is set in SignalRead() function
        // TODO: consider MASK
        if (cbr_arRxTelegramIndex[bCountTelegramIndex]->pvTelegramTimeoutData != NULL)
        {
          CSM_vMemCopy(csm_arCbrRxTelegram[bCountTelegramIndex].pvBuffer,
                       cbr_arRxTelegramIndex[bCountTelegramIndex]->pvTelegramTimeoutData,
                       (WORD)csm_arCbrRxTelegram[bCountTelegramIndex].bDlc);
          CBR_arRxTelegramState[bCountTelegramIndex].dwState &= ~CSM_C_SIGNAL_DATA_INVALID;
        }
        else
        #endif // CSM_S_TIMEOUT_SIGNAL_VALUES
        {
          #if (CSM_S_CBR_TIMEOUT_DEFAULT_LAST_VALUE == CSM_C_F_ON)
            // If no timeout default is given, use the last arrived value as
            // timeout default. Only the status INVALD is not set in this case
            // because the last value is already stored in the msg buffer
            // If the signal has never been arrived, status is already INVALID
          #else
            CBR_arRxTelegramState[bCountTelegramIndex].dwState |= CSM_C_SIGNAL_DATA_INVALID;
          #endif
        }

        CBR_LOCK_RX_DATA(iInterruptOldFlag);
        for (bCountSignalIndex = 0;
             bCountSignalIndex < cbr_arRxTelegramIndex[bCountTelegramIndex]->bNumberOfSignals;
             bCountSignalIndex++)
        {
          if (((cbr_arRxTelegramIndex[bCountTelegramIndex]->prSignals) + bCountSignalIndex)->rMode.bIndicationMode != CSM_C_NEVER_IND)
          {
            // Nur Flag setzen. Die Callback Funktion muss von der Upstream Task aufgerufen werden
            cbr_vSetIndicationFlag(bCountTelegramIndex, bCountSignalIndex);
          }
        }
        CBR_UNLOCK_RX_DATA(iInterruptOldFlag);
        csm_vTriggerCsmUpstreamTask(CSM_C_IND_REASON_CBR_SIGNAL_IND);
      }
    }
  }
}

/*******************************************************************************
 * function    CBR_vProcessSignalInd
 * \doxydocu
 * \brief      Handle signal indications.
 *
 *             Handle signal indications.
 *             This function is called from upstream-task after trigger with CSM_C_IND_REASON_CBR_SIGNAL_IND.
 *
 * \return     void
 *
 * \access     Upstream-Task
 * \reentrant  no, but only call by Upstream-Task
 *
 * \lastreview 15.07.06 CM-DI/ESA3-Pröhl
 *
 * \history_begin
 * 22.09.04 CM-DI/ESA3-Pröhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void CBR_vProcessSignalInd(void)
{
  BYTE bTelIdx;
  BYTE bSigIdx;
  BYTE bCount;
  BYTE bBus;
  BOOLEAN fRxRunning;
  INT iIsrStat;

  // ensure that the variable mapped by the makro has been initialized already !!!
  CSM_vAssert(CBR_RX_MAX_NUMBER != 0);

  #if (CSM_S_CCS_USED == CSM_C_F_ON)
  // If retrigger is set for CAN1 then retrigger all busses. ATTENTION:
  // Bus separation is currently not supported for SendAllSignalsAgain
  if (CBR_IS_STATUS(CSM_C_CAN_1, CBR_STATUS_RX_RETRIGGER))
  {
    CBR_CLR_STATUS(CSM_C_CAN_1, CBR_STATUS_RX_RETRIGGER);
    for (bTelIdx = 0; bTelIdx < CBR_RX_MAX_NUMBER; bTelIdx++)
    {
      const CSM_TELEGRAM *prCsmTelegram = cbr_arRxTelegramIndex[bTelIdx];

      for (bSigIdx = 0; bSigIdx < prCsmTelegram->bNumberOfSignals; bSigIdx++)
      {
        const CSM_SIGNAL *pCsmSignal = ((prCsmTelegram->prSignals) + bSigIdx);
        DWORD dwSignalId = CSM_M_MAKE_SIGNAL_ID(bTelIdx, bSigIdx);

        if (CSM_M_CHECK_FOR_VALID_SIGNAL_ID(&dwSignalId)) //lint !e774 PQM_authorized_127
        {
          for (bCount = 0; bCount < pCsmSignal->bMaxCallbackFunctions; bCount++)
          {
            if (pCsmSignal->pfnSignalCallback[bCount].pfvSignalInd1 != NULL)
            {
              pCsmSignal->pfnSignalCallback[bCount].pfvSignalInd1(pCsmSignal->pfnSignalCallback[bCount].pvHandle,
                                                                    dwSignalId,
                                                                    CSM_C_SIGNAL_RETRIGGERED);
            }
          } // loop over callback functions
        }
      } // loop over signals */
    } // loop over telegram
  }
  #endif

  // Process all RX telegrams configured for CBR
  for (bTelIdx = 0; bTelIdx < CBR_RX_MAX_NUMBER; bTelIdx++)
  {
    // create local pointer to array of pointers with telegramm descriptors
    // and consider directly the offset to the correct descriptor by index
    const CSM_TELEGRAM * prCsmTelegram = cbr_arRxTelegramIndex[bTelIdx];

    // also get the relevant bus for the currently handled telegram
    bBus = csm_arCbrRxTelegram[bTelIdx].bBus;
    // check if that bus is already running RX
    iIsrStat = CSM_iInterruptsDisable();
    fRxRunning = CBR_IS_STATUS(bBus, CBR_STATUS_RX_RUNNING);
    CSM_vInterruptsRestore(iIsrStat);

    // if that bus is running, then process all configured signals
    if (fRxRunning)
    {
      // loop over all signals of each telegramm defined in the descriptor
      for (bSigIdx = 0; bSigIdx < prCsmTelegram->bNumberOfSignals; bSigIdx++)
      {
        // create local pointer to the correct signal descripton table entry
        const CSM_SIGNAL * pCsmSignal = ((prCsmTelegram->prSignals) + bSigIdx);
        // use MACRO to check indication flag of the telegramm
        if (cbr_fGetIndicationFlag(bTelIdx, bSigIdx) == TRUE )
        {
          // RX indication is set for that telegramm
          DWORD dwSignalId = CSM_M_MAKE_SIGNAL_ID(bTelIdx, bSigIdx);
          DWORD dwBackupSigId = dwSignalId;
          INT iInterruptOldFlag;

          CBR_LOCK_RX_DATA(iInterruptOldFlag);   // Take care for interrupts.
          cbr_vClrIndicationFlag(bTelIdx, bSigIdx);
          CBR_UNLOCK_RX_DATA(iInterruptOldFlag);

          if (CSM_M_CHECK_FOR_VALID_SIGNAL_ID(&dwSignalId))  //lint !e774 PQM_authorized_127
          {
            for (bCount = 0; bCount < pCsmSignal->bMaxCallbackFunctions; bCount++)
            {
              // reset original internal signal ID for each handled callback in this loop
              dwSignalId = dwBackupSigId;

              #if (CSM_S_USE_CALLBACK_WITH_HANDLE == CSM_C_F_ON)
              if (pCsmSignal->pfnSignalCallback[bCount].pfvSignalInd1 != NULL)
              {
                #if (CSM_S_TRACE == CSM_C_F_ON)
                  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
                    #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
                    ETG_TRACE_USR4(("CBR_vProcessSignalInd - Bus:%x  Signal:%08x  State:%08x  Handle:%08x  Callback:%08x",
                    ETG_CENUM(tCSM_BUS_NUMBER, (tU8)  bBus),
                    (tU32)  dwSignalId,
                    (tU32)  CBR_arRxTelegramState[bTelIdx].dwState,
                    (tU32)  pCsmSignal->pfnSignalCallback[bCount].pvHandle,
                    (tU32)  pCsmSignal->pfnSignalCallback[bCount].pfvSignalInd1 ));
                    #endif
                  #else
                    #if (CSM_S_TRACE_LIMITED_SIGNAL == CSM_C_F_ON)
                    if (CSM_fTraceCheckSignal(dwSignalId) )
                    #endif // CSM_S_TRACE_LIMITED_SIGNAL
                    {
                      // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
                      BYTE abData[16];

                      abData[0] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(pCsmSignal->pfnSignalCallback[bCount].pfvSignalInd1));
                      abData[1] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(pCsmSignal->pfnSignalCallback[bCount].pfvSignalInd1));
                      abData[2] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(pCsmSignal->pfnSignalCallback[bCount].pfvSignalInd1));
                      abData[3] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(pCsmSignal->pfnSignalCallback[bCount].pfvSignalInd1));
                      abData[4] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(pCsmSignal->pfnSignalCallback[bCount].pvHandle));
                      abData[5] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(pCsmSignal->pfnSignalCallback[bCount].pvHandle));
                      abData[6] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(pCsmSignal->pfnSignalCallback[bCount].pvHandle));
                      abData[7] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(pCsmSignal->pfnSignalCallback[bCount].pvHandle));
                      abData[8] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(dwSignalId));
                      abData[9] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(dwSignalId));
                      abData[10] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(dwSignalId));
                      abData[11] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(dwSignalId));
                      abData[12] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(CBR_arRxTelegramState[bTelIdx].dwState));
                      abData[13] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(CBR_arRxTelegramState[bTelIdx].dwState));
                      abData[14] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(CBR_arRxTelegramState[bTelIdx].dwState));
                      abData[15] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(CBR_arRxTelegramState[bTelIdx].dwState));

                      CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_SIGNAL_IND, abData, 16);
                    }
                  #endif
                #endif

                // call the registered callback for the signal
                pCsmSignal->pfnSignalCallback[bCount].pfvSignalInd1(pCsmSignal->pfnSignalCallback[bCount].pvHandle,
                                                                    dwSignalId,
                                                                    CBR_arRxTelegramState[bTelIdx].dwState);
              }
              #else // CSM_S_USE_CALLBACK_WITH_HANDLE
              if (pCsmSignal->pfvSignalInd1[bCount] != NULL)
              {
                #if (CSM_S_TRACE == CSM_C_F_ON)
                #if (CSM_S_TRACE_LIMITED_SIGNAL == CSM_C_F_ON)
                if (CSM_fTraceCheckSignal(dwSignalId) )
                #endif // CSM_S_TRACE_LIMITED_SIGNAL
                {
                  // checked buffer size against CSM_MAX_TRACE_LENGTH !! BKA2HI: 11/2016
                  BYTE abData[16];

                  abData[0] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(pCsmSignal->pfvSignalInd1[bCount]));
                  abData[1] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(pCsmSignal->pfvSignalInd1[bCount]));
                  abData[2] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(pCsmSignal->pfvSignalInd1[bCount]));
                  abData[3] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(pCsmSignal->pfvSignalInd1[bCount]));
                  abData[4] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(0));
                  abData[5] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(0));
                  abData[6] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(0));
                  abData[7] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(0));
                  abData[8] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(dwSignalId));
                  abData[9] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(dwSignalId));
                  abData[10] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(dwSignalId));
                  abData[11] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(dwSignalId));
                  abData[12] = CSM_M_GET_HIBYTE(CSM_M_GET_HIWORD(CBR_arRxTelegramState[bTelIdx].dwState));
                  abData[13] = CSM_M_GET_LOBYTE(CSM_M_GET_HIWORD(CBR_arRxTelegramState[bTelIdx].dwState));
                  abData[14] = CSM_M_GET_HIBYTE(CSM_M_GET_LOWORD(CBR_arRxTelegramState[bTelIdx].dwState));
                  abData[15] = CSM_M_GET_LOBYTE(CSM_M_GET_LOWORD(CBR_arRxTelegramState[bTelIdx].dwState));

                  CSM_M_TRACEOUT(CSM_C_TRACEOUT_LEVEL_INTERFACE, CSM_C_TRACEOUT_INTERFACE_CSM, CSM_C_TRACE_FKT_CSM_SIGNAL_IND, abData, 16);
                }
                #endif // CSM_S_TRACE
                pCsmSignal->pfvSignalInd1[bCount](dwSignalId, CBR_arRxTelegramState[bTelIdx].dwState);
              }
              #endif // CSM_S_USE_CALLBACK_WITH_HANDLE
            } // Loop over all callback functions
          } // signal is valid
        }
      } // loop over all signals
    } // bus is in RX
  } // loop over all RX telegrams
}

/*******************************************************************************
 * function    CBR_vTransmitTimeout
 * \doxydocu
 * \brief      Transmit timeout callback.
 *
 *             This funktion is called after a transmit timeout is expired.
 *             Only the downstream-task is triggered in this function.
 *
 * \param[in]  dwTelegramIndex
 *             Index of telegram
 *
 * \return     void
 *
 * \access     CAN-Task
 * \reentrant  yes
 *
 * \lastreview 21.11.2008 Pröhl
 *
 * \history_begin
 * 22.09.04 CM-DI/ESA3-Pröhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
#if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
void CBR_vTransmitTimeout(const DWORD dwTelegramIndex)
{
  CBR_arTxTelegramState[dwTelegramIndex].fRepeatTimeout = TRUE;
  // wird in CBR_vProcessTimeout() gesetzt. CBR_arTxTelegramState[dwTelegramIndex].fTransmit = TRUE;
  csm_vTriggerCsmDownstreamTaskFromIrq(CSM_C_DOWN_REASON_CBR_TIMEOUT);
}

/*******************************************************************************
 * function    CBR_vTransmitMinDelayTimeout
 * \doxydocu
 * \brief      Transmit minimum delay timeout callback.
 *
 *             This funktion is called after the transmit minimum delay
 *             timeout is expired.
 *
 * \param[in]  dwTelegramIndex
 *             Index of telegram
 *
 * \return     void
 *
 * \access     CAN-Task
 * \reentrant  yes
 *
 * \lastreview 21.11.2008 Pröhl
 *
 * \history_begin
 * 22.09.04 CM-DI/ESA3-Pröhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
#if (CSM_S_CBR_MIN_DELAY == CSM_C_F_ON)
void CBR_vTransmitMinDelayTimeout(const DWORD dwTelegramIndex)
{
  CBR_arTxTelegramState[dwTelegramIndex].fDelayTransmit = FALSE;
  if (CBR_arTxTelegramState[dwTelegramIndex].fTransmit == TRUE)
  {
    csm_vTriggerCsmDownstreamTaskFromIrq(CSM_C_DOWN_REASON_CBR_TIMEOUT);
  }
}
#endif // CSM_S_CBR_MIN_DELAY
#endif // CSM_S_REMOTE_STACK

/*******************************************************************************
 * function    CBR_vReceiveTimeout
 * \doxydocu
 * \brief      Receive timeout callback.
 *
 *             This funktion is called after the receive timeout is expired.
 *             Only the downstream-task is triggered in this function.
 *
 * \param[in]  dwTelegramIndex
 *             Index of telegram
 *
 * \return     void
 *
 * \access     CAN-Task
 * \reentrant  yes
 *
 * \lastreview 21.11.2008 Pröhl
 *
 * \history_begin
 * 22.09.04 CM-DI/ESA3-Pröhl
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void CBR_vReceiveTimeout(const DWORD dwTelegramIndex)
{
  #if (CSM_S_CBR_RX_TIMEOUT_GROUPS == CSM_C_F_ON)
  // Only if Rx-Group is enabled, set Receive Timeout
  if ((csm_arCbrRxTelegram[dwTelegramIndex].bRxGroup & cbr_bRxGrpMask) == 0)
  #endif
  {
    // set corresponding timeout flag and trigger Downstream Task
    CBR_arRxTelegramState[dwTelegramIndex].fReceiveTimeout = TRUE;
    csm_vTriggerCsmDownstreamTaskFromIrq(CSM_C_DOWN_REASON_CBR_TIMEOUT);
  }
}

/*******************************************************************************
 * function    CBR_vSetRxGrpMask
 * \doxydocu
 * \brief      Set flags for disable the timeout check for Rx-Messages.
 *
 *             This function disables the timeout check for Rx-Messages
 *             of selected Group, passed as parameter in mask.
 *
 * \param[in]  bMask
 *
 * \return     void
 *
 * \access     CAN-Task
 * \reentrant  yes
 *
 * \lastreview 26.09.06 CM-DI/ESA3-Battis
 *
 * \history_begin
 * 25.09.06 CM-DI/ESA3-Haffner
 * Changes for CSM proxy usage.
 *
 * 11.03.04 CM-CR/ESD-Peter Tuschik
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
#if (CSM_S_CBR_RX_TIMEOUT_GROUPS == CSM_C_F_ON)
void CBR_vSetRxGrpMask(const BYTE bMask)
{
  #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
    cbr_bRxGrpMask |= bMask;     // set RX-Group to disable time out checking
  #endif // CSM_S_REMOTE_STACK
}

/*******************************************************************************
 * function    CBR_vClrRxGrpMask
 * \doxydocu
 * \brief      Clear flags to enable the timeout check for Rx-messages.
 *
 *             This function clears flags to enable the timeout check for
 *             Rx-messages of selected group, passed as parameter in mask.
 *             For the messages, that belongs to the selected group, we must
 *             restart the receive timer.
 *
 * \param[in]  bMask
 *
 * \return     void
 *
 * \access     CAN-Task
 * \reentrant  yes
 *
 * \lastreview 26.09.06 CM-DI/ESA3-Battis
 *
 * \history_begin
 * 25.09.06 CM-DI/ESA3-Haffner
 * Changes for CSM proxy usage.
 *
 * 11.03.04 CM-CR/ESD-Peter Tuschik
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
void CBR_vClrRxGrpMask(const BYTE bMask)
{
  #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
  BYTE bCountTelegramIndex;
  #endif

  // ensure that the variable mapped by the makro has been initialized already !!!
  CSM_vAssert(CBR_RX_MAX_NUMBER != 0);

  #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
  for (bCountTelegramIndex = 0; bCountTelegramIndex < CBR_RX_MAX_NUMBER; bCountTelegramIndex++)
  {
    if (csm_arCbrRxTelegram[bCountTelegramIndex].bRxGroup & bMask)
    {                                               // Restart Timeout for
      CBR_vStartReceiveTimer(bCountTelegramIndex);  // enabled Rx-Group.
    }
  }

  cbr_bRxGrpMask &= ~bMask;  // Clear RX-Group to enable time out checking.
  #endif // CSM_S_REMOTE_STACK
}

/*******************************************************************************
 * function    CBR_bGetRxGrpMask
 * \doxydocu
 * \brief      Returns present group mask.
 *
 * \return     bMask
 *
 * \access     CAN-Task
 * \reentrant  yes
 *
 * \lastreview  16.04.08 VTeam-Pröhl
 *
 * \history_begin
 *
 * 13.02.07 CM-DI/ERM13-Battis
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
BYTE CBR_bGetRxGrpMask(void)
{
  #if (CSM_S_REMOTE_STACK == CSM_C_F_OFF)
  return cbr_bRxGrpMask;
  #else
  return 0;
  #endif  // CSM_S_REMOTE_STACK
}

/*******************************************************************************
 * function    CBR_vSetRxGroup2Outdated
 * \doxydocu
 * \brief      Sets signal status of bRxGroup signals to outdated.
 *
 *             This function sets signal status of bRxGroup signals to outdated.
 *             (e.g.: This is used for VW projects were it were a part
 *                    of the Rx-messages comes from a remote CAN. If the
 *                    remoute CAN is sleeping we have to change the signal
 *                    state to outdated.)
 *
 * \param[in]  bRxGroup
 *             Rx group of signal which has to be set to outdated.
 *
 * \return     void
 *
 * \access     CAN-Task
 * \reentrant  yes
 *
 * \lastreview 21.11.2008 Pröhl
 *
 * \history_begin
 * 08.03.2005 (CM-DI/ESA3) Ingo Battis
 * adapted to new CBR
 *
 * 04.08.2004 (CM-CR/ESD3) Thomas Mehring
 * Initial Revision.
 * \history_end
 ******************************************************************************/
void CBR_vSetRxGroup2Outdated(const BYTE bRxGroup)
{
  DWORD dwTelIdx;

  // ensure that the variable mapped by the makro has been initialized already !!!
  CSM_vAssert(CBR_RX_MAX_NUMBER != 0);

  for (dwTelIdx = 0; dwTelIdx < CBR_RX_MAX_NUMBER; dwTelIdx++)
  {
    // Check if rx message belongs to bRxGroup.
    if ((csm_arCbrRxTelegram[dwTelIdx].bRxGroup & bRxGroup) != 0)
    {
      // Message belongs to bRxGroup ==> set outdated flag.
      CBR_arRxTelegramState[dwTelIdx].dwState |= CSM_C_SIGNAL_DATA_OUTDATED;
    }
  }
}
#endif // CSM_S_CBR_RX_TIMEOUT_GROUPS

/*******************************************************************************
 * function    CBR_vSendAllSignalsAgain
 * \doxydocu
 * \brief      Deliver all CBR signals again to the applications.
 *
 *             Set a flag to force indication for all signals to the
 *             application via the upsteam task.
 *
 * \return     void
 *
 * \access     CAN-Task
 * \reentrant  yes
 *
 * \lastreview 21.11.2008 Pröhl
 *
 * \history_begin
 * 03.03.04 CM-CR/ESD-Peter Tuschik
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#if (CSM_S_CCS_USED == CSM_C_F_ON)
void CBR_vSendAllSignalsAgain(void)
{
  CBR_SET_STATUS(CSM_C_CAN_1, CBR_STATUS_RX_RETRIGGER);
  csm_vTriggerCsmUpstreamTask(CSM_C_IND_REASON_CBR_SIGNAL_IND);
}
#endif // CSM_S_CCS_USED

/*******************************************************************************
 * function    cbr_vHandleRollingCounter
 * \doxydocu
 * \brief      Implements a rolling counter for all TX messages, that have
 *             been configured accordingly and if the feature is enabled
 *
 * \return     void
 *
 * \access     called from CBR_vProcessTransmit, directly before TX
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 * 11.11.2014  Andreas Borck
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#if (CSM_S_CBR_TX_MSG_ROLL_COUNTER == CSM_C_F_ON)
static void cbr_vHandleRollingCounter(BYTE bTelegramIndex)
{
  DWORD dwSignalId;         // signal Id of counter signal within telegram
  BYTE  bCounter;

  if (pTxRollingCounterCfg == NULL)
  {
    CSM_vAssert(pTxRollingCounterCfg != NULL);
    return; // bail out if config is invalid
  }

  // get the relevant signal ID of the Counter Signal within the telegramm
  dwSignalId = (pTxRollingCounterCfg + bTelegramIndex)->dwSignalId;

  // check if the upper configured counter value != 0, otherwise it indicates, that no
  // counter is used for that telegram
  // also check if the lower configured value is really lower than the upper one
  if (((pTxRollingCounterCfg + bTelegramIndex)->bUpperThreshold == 0) || ((pTxRollingCounterCfg + bTelegramIndex)->bLowerThreshold >= (pTxRollingCounterCfg + bTelegramIndex)->bUpperThreshold))
  {
    return; // bail out if no counter handling required
  }

  // get the current, locally hosted counter value
  bCounter = cbr_aTxRollingCounterAdmin[bTelegramIndex].bRollCount;

  // write counter to signal
  CBR_lSignalWrite(dwSignalId, (tVoid*) &bCounter, 1, CSM_C_TX_STORE_ONLY);

  // increment counter value and check for max threshold and turn around in case of overflow
  if (bCounter < (pTxRollingCounterCfg + bTelegramIndex)->bUpperThreshold)
  {
    bCounter += 1;
  }
  else
  {
    bCounter = (pTxRollingCounterCfg + bTelegramIndex)->bLowerThreshold;
  }
  // write back counter
  cbr_aTxRollingCounterAdmin[bTelegramIndex].bRollCount = bCounter;
}
#endif // CSM_S_CBR_TX_MSG_ROLL_COUNTER

/*******************************************************************************
 * function    cbr_vHandleTxCRC
 * \doxydocu
 * \brief      Implements a CRC for all TX messages, that have
 *             been configured accordingly and if the feature is enabled
 *
 * \return     void
 *
 * \access     called from CBR_vProcessTransmit, directly before TX
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 * 27.05.2016  Markus Pröhl
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#if (CSM_S_CBR_TX_MSG_CRC == CSM_C_F_ON)
static void cbr_vHandleTxCRC(BYTE bTelegramIndex)
{
  INT iInterruptOldFlag;
  LONG return_value;
  DWORD dwSignalId;         // signal Id of CRC signal within telegram
  const CSM_TELEGRAM * prCsmTelegram = cbr_arTxTelegramIndex[bTelegramIndex];
  BYTE bSignalIndex;
  BYTE bCount;
  BYTE bCrcData = 0;        // we start with a clean Crc
  #if (CSM_S_CBR_TX_MSG_CRC_TYPE_RN == CSM_C_F_ON)
  BYTE bCrcIndex;           // Byte nr. which is containing the CRC
  #endif

  if (cbr_pTxCrcCfg == NULL)
  {
    CSM_vAssert(cbr_pTxCrcCfg != NULL);
    return; // bail out if config is invalid
  }

  // get the relevant signal ID of the Counter Signal within the telegramm
  dwSignalId = (cbr_pTxCrcCfg + bTelegramIndex)->dwSignalId;

  // check if the CRC is activated for this TX message
  if ((cbr_pTxCrcCfg + bTelegramIndex)->fCrcActive == FALSE)
  {
    return; // bail out if no CRC handling required
  }

  // get the signal index for the signal ID (CRC signal)
  bSignalIndex = (BYTE) CSM_M_GET_SIGINDEX_FROM_SIGNAL_ID(dwSignalId);
  // check if access index is in range
  if (bSignalIndex < (prCsmTelegram->bNumberOfSignals))
  {
    // index is valid, so get some helpers
    const CSM_SIGNAL * prCsmSignal = prCsmTelegram->prSignals + bSignalIndex;
    const CSM_CBR_TX * prCbrTxCfg = &csm_arCbrTxTelegram[bTelegramIndex];

    // lock data buffer - jetzt können die Daten manipuliert werden
    CBR_LOCK_TX_DATA(iInterruptOldFlag);

    // Set CRC signal to 0 in Telegramm
    return_value = CBR_lSignalWriteRaw(prCbrTxCfg->pvBuffer, prCbrTxCfg->bDlc, prCsmSignal->bStartBit,
                                       prCsmSignal->bLength, (BYTE *) &bCrcData,
                                       1
                                       #if (CSM_S_CBR_BOTH_ENDIANS_SIGNALS == CSM_C_F_ON)
                                       , prCsmSignal->bEndians
                                       #endif
                                      );
    if (return_value != CSM_C_NO_ERROR)
    {
      CSM_vAssert(return_value == CSM_C_NO_ERROR);
    }

    // Calculate CRC
    #if (CSM_S_CBR_TX_MSG_CRC_TYPE_RN == CSM_C_F_ON)
    {
      // this is Renault Nissan CRC implementation
      BYTE bTemp_Result = 0;  // temp CRC result according to Renault Nissan Spec
      bCrcIndex = (cbr_pTxCrcCfg + bTelegramIndex)->bCrcIndex;

      for (bCount=0; bCount < prCbrTxCfg->bDlc; bCount++)
      {
        // CRC all bytes except the one containing the CRC itself
        if (bCount != bCrcIndex)
        {
          // XOR data byte
          bTemp_Result ^= ((BYTE*)(prCbrTxCfg->pvBuffer))[bCount];
        }
      }

      // now reduce the CRC to 4 Bits by keeping the upper MSBs of the byte containing the CRC
      bCrcData = ((bTemp_Result & 0xF0) >> 4) ^ (bTemp_Result & 0x0F);
      bCrcData ^= ((((BYTE*)(prCbrTxCfg->pvBuffer))[bCrcIndex] & 0xF0) >> 4);
    }
    #else
    // this is AUTOSAR CRC standard
    for (bCount=0; bCount < prCbrTxCfg->bDlc; bCount++)
    {
      bCrcData ^= ((BYTE*)(prCbrTxCfg->pvBuffer))[bCount];
    }
    #endif

    // Store CRC in Telegramm
    return_value = CBR_lSignalWriteRaw(prCbrTxCfg->pvBuffer, prCbrTxCfg->bDlc, prCsmSignal->bStartBit,
                                       prCsmSignal->bLength, (BYTE *) &bCrcData,
                                       1
                                       #if (CSM_S_CBR_BOTH_ENDIANS_SIGNALS == CSM_C_F_ON)
                                       , prCsmSignal->bEndians
                                       #endif
                                      );
    if (return_value != CSM_C_NO_ERROR)
    {
      CSM_vAssert(return_value == CSM_C_NO_ERROR);
    }

    // unlock data buffer
    CBR_UNLOCK_TX_DATA(iInterruptOldFlag);
  }
}
#endif // CSM_S_CBR_TX_MSG_CRC

/*******************************************************************************
 * function    cbr_fCheckRxRollCounter
 * \doxydocu
 * \brief      Implements a rolling counter check for all RX messages, that have
 *             been configured accordingly and if the feature is enabled
 *
 * \return     BOOL
 *
 * \access     called from cbr_vReceiveInd()
 * \reentrant  no
 *
 * \lastreview
 *
 * \history_begin
 * 11.11.2014  Andreas Borck
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#if (CSM_S_CBR_RX_MSG_ROLL_COUNTER == CSM_C_F_ON)
static BOOL cbr_fCheckRxRollCounter(BYTE bTelegramIndex, const BYTE *pbData)
{
  DWORD dwSignalId;                                  // signal Id of counter signal within telegram
  BYTE  bSignalIndex;
  BYTE  bCounter;
  BYTE  bNewCounter;
  LONG  return_value;

  if (pRxRollingCounterCfg == NULL)
  {
    CSM_vAssert(pRxRollingCounterCfg != NULL);
    return; // bail out if config is invalid
  }

  // get the relevant signal ID of the Counter Signal within the telegramm
  dwSignalId = (pRxRollingCounterCfg + bTelegramIndex)->dwSignalId;

  // check if the upper configured counter value != 0, otherwise it indicates, that no
  // counter is used for that telegram
  // also check if the lower configured value is really lower than the upper one
  if (((pRxRollingCounterCfg + bTelegramIndex)->bUpperThreshold == 0) || ((pRxRollingCounterCfg + bTelegramIndex)->bLowerThreshold >= (pRxRollingCounterCfg + bTelegramIndex)->bUpperThreshold))
  {
    return TRUE; // bail out if no counter handling required
  }

  // get the current, locally hosted counter value (the start value is 0xFF to be able to start with 0 here)
  bCounter = cbr_abRxRollingCounter[bTelegramIndex];

  // increment counter value and check for max threshold and turn around in case of overflow
  if (bCounter < (pRxRollingCounterCfg + bTelegramIndex)->bUpperThreshold)
  {
    bCounter += 1;
  }
  else
  {
    bCounter = (pRxRollingCounterCfg + bTelegramIndex)->bLowerThreshold;
  }

  // read signal to counter
  bSignalIndex = (BYTE) CSM_M_GET_SIGINDEX_FROM_SIGNAL_ID(dwSignalId);
  if (bSignalIndex < (cbr_arRxTelegramIndex[bTelegramIndex]->bNumberOfSignals))
  {
    return_value = CBR_lSignalReadRaw((BYTE *)pbData,
                                       csm_arCbrRxTelegram[bTelegramIndex].bDlc,
                                       ((cbr_arRxTelegramIndex[bTelegramIndex]->prSignals) + bSignalIndex)->bStartBit,
                                       ((cbr_arRxTelegramIndex[bTelegramIndex]->prSignals) + bSignalIndex)->bLength,
                                       (BYTE *) &bNewCounter,
                                       1
                                       #if (CSM_S_CBR_BOTH_ENDIANS_SIGNALS == CSM_C_F_ON)
                                       , ((cbr_arRxTelegramIndex[bTelegramIndex]->prSignals) + bSignalIndex)->bEndians
                                       #endif
                                     );
    if (return_value != CSM_C_NO_ERROR)
    {
    }
  }

  // write back counter
  cbr_abRxRollingCounter[bTelegramIndex] = bNewCounter;

  if (bCounter == bNewCounter)
  {
    return TRUE;
  }
  else
  {
    return FALSE;
  }
}
#endif // CSM_S_CBR_RX_MSG_ROLL_COUNTER

#if (CSM_S_CBR_RX_MSG_CRC == CSM_C_F_ON)
BOOL cbr_fCheckRxCRC(BYTE bTelegramIndex, const BYTE *pbData, BYTE bDataLength)
{
  if (cbr_pRxCrcCfg == NULL)
  {
    CSM_vAssert(cbr_pRxCrcCfg != NULL);
    return; // bail out if config is invalid
  }

  if ((cbr_pRxCrcCfg + bTelegramIndex)->fCrcActive == TRUE)
  {
    if ((cbr_pRxCrcCfg + bTelegramIndex)->bDataLength == bDataLength)
    {
      BYTE bCount;
      BYTE bData = 0;

      for (bCount=0; bCount < bDataLength; bCount++)
      {
        bData ^= pbData[bCount];
      }
      if (bData == 0)
      {
        return TRUE;
      }
      else
      {
        return FALSE;
      }
    }
    else
    {
      return FALSE;
    }
  }
  else
  {
    return TRUE;
  }
}
#endif // CSM_S_CBR_RX_MSG_CRC

#undef CBR_SELF
#undef CSM_CAN_STACK

/*******************************************************************************
 * \doxydocu
 * \file          cbr.c
 * \brief         CAN broadcast server module
 *
 * \see           CAN-SW Architektur, written by CM-DI/ESA3
 *
 * \sw_component  CAN stack
 * \project       Pool module
 * \path          /di_can/modules_swa/cbr/src/
 *
 * \ingroup       CBR
 *
 * \authors       V-Team
 *
 * COPYRIGHT      (c) 2004 Blaupunkt GmbH
 *
 * \history_begin
 *
 *
 *04.12.03  Rev. 1.34 CM-DI/ESU3-K"ohler
 *          - Include csm_datatypes.h removed.
 *
 *18.06.04  Rev. \main\4  CM-DI/ESU3-Pröhl
 * - empty CBR for new CDA
 *
 *23.06.04  Rev. \main\5  CM-DI/ESU3-Pröhl
 * - first try
 *
 *25.06.04  Rev. \main\6  CM-DI/ESU3-Pröhl
 * - CBR_lApplCallbackInit added
 *
 *25.06.04  Rev. \main\7  CM-DI/ESU3-Pröhl
 *
 *28.06.04  Rev. \main\8  CM-DI/ESU3-Pröhl
 * - CBR_lApplCallbackInit() modified
 *
 *28.06.04  Rev. \main\9  CM-DI/ESU3-Pröhl
 * - CDA_lSend() and CBR_vCDAReceiveInd modified
 *
 *08.07.04  Rev. \main\10  CM-DI/ESU3-Pröhl
 * - CBR_lSignalWrite() modified
 *
 *20.07.04  \main\11  CM-DI/ESU3-Pröhl
 * - bugfix
 * - CBR_lSignalWrite() geht
 * - Signal Indication geht
 * - Timeout geht
 * - alles mit FTPEX getestet
 *
 *23.07.04  \main\12  CM-DI/ESU3-Pröhl
 * - CBR_lSignalWrite() modified
 * - timeout modified
 *
 *24.07.04  \main\13  CM-DI/ESU3-Pröhl
 * - CBR_lSignalWriteRawNeu() and CBR_lSignalReadRawNeu() added
 *
 *30.07.04  \main\14  CM-DI/ESU3-Pröhl
 * - new CBR_CONVERT_.... macros added
 *
 *03.09.04  \main\15  CM-DI/ESA3-Pröhl
 * - CBR_fCheckCommunicationReq(), CBR_vProcessCommunicationReg(), CBR_vProcessCommunicationCon(), CBR_vProcessCommunicationInd() added
 *
 *07.09.04  \main\16  CM-DI/ESA3-Pröhl
 * - code reading, doxygen tag added
 *
 *22.09.04  \main\17  CM-DI/ESA3-Pröhl
 * - code reading, doxygen tags added
 *
 *29.10.04  \main\18  CM-DI/ESA3-Battis
 * - defines moved to csm_stack_m.h
 * - CBR_vStartReceiveTimer does not start timer if timout value is 0 now
 * - bugfix: don't change indication signal is pointer is NULL
 * - CBR_vProcessCommunicationCon and CBR_vProcessCommunicationInd
 *   feature switches inserted
 * - compiler warnings removed
 *
 *22.11.04  \main\19  CM-DI/ESA3-Battis
 * - tCBR_TX_TELEGRAM_STATE and tCBR_RX_TELEGRAM_STATE moved to cbr_m.h
 * - CBR_vStartReceiveTimer() renamed to CBR_vStartReceiveTimer()
 * - define CBR_M_INIT_VARS() called in CBR_lInit()
 * - some bugfixes
 * - CBR_vReceiveTimeout() renamed to CBR_vReceiveTimeout()
 *
 *02.12.04  \main\20  CM-DI/ESA3-Battis
 * - CBR Status flag line added
 * - now all TX messages are suppressed, if CBR is not started
 *
 *22.12.04  \main\21  CM-DI/ESA3-Battis
 * - local variables added for interrupt enable/disable defines
 *
 *18.01.05  \main\22  CM-DI/ESA3-Battis
 * - CSM_C_TX_AS_CONFIGURED renamed to CSM_C_TX_CYCLIC
 * - 2 bugfixes in CBR_lSignalWrite()
 *
 *20.01.05  \main\23  CM-DI/ESA3-Battis
 * - CBR_vStartReceiveTimer() corrected
 *
 *21.01.05  \main\24  CM-DI/ESA3-Battis
 * - in CBR_lSignalWrite() for CBR_TMSG_CYCLIC_AND_SPONTAN:
 *   transmission only if data has changed
 *
 *03.02.05  \main\25  CM-DI/ESA3-Battis
 * - bugfix: CBR does only support one CAN bus at present
 *
 *08.02.05  \main\26  CM-DI/ESA3-Battis
 * - set default data on WARM init
 * - set default data on timeout
 * - signal status determined when signal is read
 *
 *09.02.05  \main\27  CM-DI/ESA3-Battis
 * - set TX default data on WARM init
 * - return value of CBR_SignalRead and CBR_SignalWrite is now determined
 *
 *09.02.05  \main\28  CM-DI/ESA3-Battis
 * - bugfix
 *
 *10.02.05  \main\29  CM-DI/ESA3-Battis
 * - CSM_C_ALWAYS_IND is now processed correctly
 *
 *11.02.05  \main\30  CM-DI/ESA3-Pröhl
 * - CBR_vCSMBusStateInd() added
 * - bugfix for communication request.
 * - bugfix CBR_lStartTx() for activ signals
 * - CSM_C_TX_BUS_WAKE_TEMP function added
 * - CBR_STATUS_TX_RUNNING handling moved to CBR_vProcessTransmit()
 *
 *18.02.05  \main\31  CM-DI/ESA3-Battis
 * - bugfix
 *
 *24.02.05  \main\32 CM-DI/ESA3-Pröhl
 * - CSM_SIGNAL modified
 *
 *24.02.05  \main\33 CM-DI/ESA3-Pröhl
 * - CSM_SIGNAL modified
 *
 *25.02.05  \main\34 CM-DI/ESA3-Battis
 * - valid flag now officially used
 * - signal indication behaviour aligned to present CBR spec.
 * - signal indication bug fixes
 * - several asserts inserted for our dear Pappenheimers
 * - new implementation of FCT_IF_ACTIVE
 * - CBR_vStartTransmitRepeatTimer shall only be started during ProcessTransmit
 * - feature CSM_S_CBR_TIMEOUT_DEFAULT_LAST_VALUE implemented
 * - comments added
 *
 *25.02.05  \main\35 CM-DI/ESA3-Battis
 * - bugfix
 *
 *08.03.05  \main\36 CM-DI/ESA3-Battis
 * - recieve renamed to receive
 * - evaluation of type CBR_TMSG_NO_TRANSMISSION in CBR_vProcessTransmit()
 *
 *09.03.05  \main\37 CM-DI/ESA3-Battis
 * - RX groups support for timeout supervision implemented
 * - old functions removed
 *
 *09.03.05  \main\38 CM-DI/ESA3-Pröhl
 * - move broadcast communication request to csm_cbr.c
 *
 *23.03.05  \main\39 CM-DI/ESA3-Pröhl
 * - NULL pointer check added to CBR_lInit()
 *
 *23.03.05  \main\40 CM-DI/ESA3-Pröhl
 * - bugfix for /39
 *
 *23.03.05  \main\41 CM-DI/ESA3-Zurmühl
 * - enhancements for "CSM_S_REQUESTABLE_TIMER" in CBR_lInit, CBR_lExit
 *
 *01.04.05  \main\42 CM-DI/ESA3-Pröhl
 * - die ersten 20 Byte sind eingespart
 *
 *04.04.05  \main\43 CM-DI/ESA3-Pröhl
 * - die 202 Byte ROM sind eingespart
 *
 *04.04.05  \main\44 CM-DI/ESA3-Pröhl
 * - change indication and activ flags
 *
 *05.04.05  \main\45 CM-DI/ESA3-Battis
 * - bugfix in CBR_vCDAReceiveInd()
 * - fix in CBR_vProcessSignalInd()
 * - more code featured with CSM_S_CBR_TX_IF_APPL_ACTIVE
 * - feature switch CSM_S_CBR_RETRIGGERD corrected
 * - dummy function inserted if CSM_S_CBR_USES_COMMUNICATION_REQ is not used
 * - CPP comments removed
 * - writing errors removed
 *
 *07.04.05  \main\46 CM-DI/ESA3-Battis
 * - static function cbr_fIsFunctionActive() added
 * - some code for usage of CSM_S_REQUESTABLE_TIMER added
 * - signal init CSM_C_SIGNAL_NOT_RECEIVED and CSM_C_SIGNAL_DATA_INVALID moved
 *   from WARM to COLD init
 * - some code optimizations to save RAM/ROM
 * - CSM_C_TX_AS_CONFIGURED reintroduced
 * - bugfix in CBR_vSendAllSignalsAgain()
 *
 *07.04.05  \main\47 CM-DI/ESA3-Pröhl
 * - CSM_S_CBR_CALL_CALLBACK_DIRECT added
 *
 *07.04.05  \main\48 CM-DI/ESA3-Battis
 * - function fGetActiveFlag() removed
 * - bugfix for CSM_S_REQUESTABLE_TIMER (hRepeatTimer -> hReceiveTimer)
 * - fDelayTimeout optimized away
 * - usage of CSM_S_CBR_RETRIGGERD moved/renamed to CSM_S_CCS_USED
 *
 *26.04.05  \main\49 CM-DI/ESA3-Pröhl
 * - remove bug in CBR_lInit()
 *
 *27.04.05  \main\50 CM-DI/ESA3-Pröhl
 * - remove bug in CBR_lInit() next try
 *
 *28.04.05  \main\51 CM-DI/ESA3-Battis
 * - VW merge
 *
 *02.05.05  \main\52 CM-DI/ESA3-Zurmühl
 * - "static" for 3 variable definitions replaced by "CSM_STATIC"
 *   (needed for PC test environment)
 *
 *09.05.05  \main\55 CM-DI/ESA3-Battis
 * - changes to make MISRA happy
 *
 *10.05.05  \main\56 CM-DI/ESA3-Battis
 * - usage of CSM_C_SIGNAL_READ_STATUS_MASK removed
 *
 *11.05.05  \main\57 CM-CR/ESD4-Tuschk
 * - Add support for virtual signals if variants used.
 *
 *27.05.05  \main\59 CM-DI/ESA3-Battis
 * - CBR_vCDAReceiveInd :
 *   reject messages with DLC shorter than expected
 *   accept messages with DLC longer than expected, use the expected bytes and
 *   reject the rest
 *
 *06.07.05  \main\60 CM-DI/ESA3-Zurmühl
 * - Some error codes adapted to new conventions
 *
 *06.07.05  \main\61 CM-DI/ESA3-Zurmühl
 * - Error code CBR_E_ERROR_STATE_WHILE_INIT changed to global code CSM_E_UNKNOWN_STATE
 *
 *01.08.05  \main\63 CM-DI/ESA3-Battis
 * - optional "callback direct" handling (on downstream task) implemented
 *
 *18.08.05  \main\66 CM-DI/ESA3-Zurmühl
 * - bugfix in CBR_vStartTransmitRepeatTimer for (CSM_S_REQUESTABLE_TIMER == CSM_C_F_ON)
 *
 *31.08.05  \main\67 CM-DI/ESA3-Battis
 * - feature CSM_S_CBR_RESYNCHRONIZED_CYLCE implemented
 *
 *31.08.05  \main\68 CM-DI/ESA3-Battis
 * - writing error in feature switch CSM_S_CBR_RESYNCHRONIZED_CYLCE removed
 *
 *31.08.05  \main\69 CM-DI/ESA3-Battis
 * - dependancy of CSM_S_CBR_RESYNCHRONIZED_CYLCE to MIN_DELAY removed
 *
 *08.09.05  \main\70 CM-DI/ESA3-Battis
 * - bugfix in CBR_vProcessTimeout()
 *
 *19.09.05  \main\71 CM-DI/ESA3-Pröhl
 * - add handle to all ...CallbackInit() funktions
 *
 *22.09.05  \main\72 CM-DI/ESA3-Battis
 * - doxygen
 *
 *28.09.05  \main\73 CM-DI/ESA3-Battis
 * - checks now all affecting feature switches
 *
 *07.10.05  \main\74 CM-DI/ESA3-Battis
 * - changes to finalize multi channel functionality
 *
 *10.10.05  \main\75 CM-CR/ESD4-Tuschik
 * - Bug fix in start of repeat timer for "cyclic_if_active" messages.
 *
 *10.10.05  \main\76 CM-DI/ESA3-Battis
 * - Bugfix in RX Handling. Reception was started also without StartRx().
 *
 *21.10.05  \main\77 CM-DI/ESA3-Battis
 * - Bugfix in CSM_S_CBR_USES_DIRECT_CALLS handling
 *
 *01.11.05  \main\78 CM-DI/ESA3-Battis
 * - Bugfix: ALWAYS_IND was not possible with CALLBACK_DIRECT at the same time.
 *
 *08.11.05  \main\79 CM-CR/ESD4-Tuschik
 * - Add Delay for Time out supervision at StartRx.
 *
 *08.11.05  \main\80 CM-CR/ESD4-Tuschik
 * - use macro to store time out delay to save code.
 *
 *09.11.05  \main\81 CM-CR/ESD4-Tuschik
 * -
 *
 *14.11.05  \main\82 CM-DI/ESA3-Pröhl
 * - remove some compiler warnings
 *
 *16.11.05  \main\83 CM-DI/ESA3-Battis
 * - Check if signal is linked to CBR_C_DEPRECATED_IDX and raise error then.
 *
 *19.12.05  \main\84 CM-DI/ESA3-Battis
 * - error in timeout supervision handling solved.
 *
 *25.01.06  \main\85 CM-DI/ESA3-Battis
 * - CBR_vCSMBusStateInd() dummy removed
 *   Decision if this function is to be called is now done in csm.c directly.
 *
 *26.04.06  \main\86 CM-DI/ESA3-Battis
 * - Bugfix: Start and Stop functions did not take care of two busses correctly.
 *
 *28.04.06  \main\87 CM-DI/ESA3-Battis
 * - First changes for Proxy usage.
 *
 *02.05.06  \main\88 CM-DI/ESA3-
 * -
 *
 *15.05.06  \main\89 CM-DI/ESA3-Pröhl
 * - change date in comment
 * - add some \lastreview tags
 *
 *02.06.06  \main\90 CM-DI/ESA3-Pröhl
 * - change name in comment
 *
 *27.06.06  \main\91 CM-DI/ESA3-Haffner
 * - Prepared sending telegrams by remote stack (proxy).
 *
 *14.07.06  \main\92 CM-DI/ESA3-Pröhl
 * - add feature CSM_S_USE_CALLBACK_WITH_HANDLE.
 *   Add the pvHandle from CSM_lApplCallbackInit() to callback routines.
 *   This is used to handle the process border jump (e.g. TENGINE subsystem)
 *
 *19.07.06  \main\93 CM-DI/ESA3-Pröhl
 * - modify feature CSM_S_USE_CALLBACK_WITH_HANDLE.
 *   add PVOID pvHandle to tCSM_ISO_TP_USDT_APPL_CALLBACK if feature CSM_S_USE_CALLBACK_WITH_HANDLE is active
 *
 *28.07.06  \main\94 CM-DI/ESA3-Battis
 * - Bugfix: Transmit Repeat Timer has never been started if telegram is
 *   not valid when StartTx() is called.
 *
 *21.08.06  \main\95 CM-DI/ESA3-Pröhl
 * - add parameter check to CBR_lSignalCallBackInit()
 *
 *07.09.06  \main\96 CM-DI/ESA3-Battis
 * - Bugfix: call of CBR_vStartTransmitRepeatTimer() only permitted in
 *   CBR_lSignalWrite() if CSM_S_CBR_RESYNCHRONIZED_CYCLE is switched on
 *
 *07.09.06  \main\97 CM-DI/ESA3-Haffner
 * - Add a lot of preprocessor conditions for remote stack.
 *   Add CBR_vProxyReceiveInd() for remote stack only.
 *
 *26.09.06  \main\98 CM-DI/ESA3-Battis
 * - proxy changes reviewed.
 *
 *27.09.06  \main\99 CM-DI/ESA3-Haffner
 * - activated CBR_vProcessSignalInd() again, which disappeared in review.
 *
 *29.09.06  \main\100 CM-DI/ESA3-Haffner
 * - switched PROXY-CBR interface to CDA style
 *
 *13.10.06  \main\101 CM-DI/ESA3-Haffner
 * - removed dead code
 *
 *18.10.06  \main\102  CM-DI/ESA3-Pröhl
 * - PVOID  and BYTE replaced
 *
 *02.11.06  \main\103  CM-DI/ESA3-Haffner
 * - Const for parameter pvActSignalData in CBR_lSignalRead() removed.
 *
 *03.11.06  \main\104  CM-DI/ESA3-Battis
 * - asserts added to SignalWriteRaw
 *
 *07.11.06  \main\105  CM-DI/ESA3-Battis
 * - usage of CSM_M_MAKE_CAN_1_SIGNAL_ID changed to CSM_M_MAKE_SIGNAL_ID
 *
 *08.11.06  \main\106  CM-DI/ESA3-Battis
 * - cbr_vCheckConfig() added
 *
 *08.11.06  \main\107  CM-DI/ESA3-Battis
 * - compilation problem solved.
 *
 *09.11.06  \main\108  CM-DI/ESA3-Battis
 * - cbr_vCheckConfig() improved
 *
 *05.02.07  \main\109  CM-DI/ERM13-Battis
 * - CBR_vCDAReceiveInd and CBR_vPROXYReceiveInd implemented as single functions
 *   mapping to cbr_vReceiveInd
 * - bugfix on timeout handling for proxy CBR
 *
 *05.02.07  \main\110  CM-DI/ERM13-Battis
 * - syntax fixes
 *
 *05.02.07  \main\111  CM-DI/ERM13-Battis
 * - proxy timeout handling still not sufficient - fixed now.
 * - code cosmetics
 *
 * 13.02.07  \main\112  CM-DI/ERM13-Battis
 * - VOID replaced by void
 * - bugfix: clearing group mask worked only by chance with KCAN config.
 * - function CBR_bGetRxGrpMask() added.
 *   Needed for decision with tout timer to be restated on busoff recovery.
 *
 * 21.02.07  \main\113  CM-DI/ERM14-Battis
 * - Clear application buffer in SignalReadRaw before writing the signal value
 *   to it. Avoid partly filled buffers.
 *
 * 21.03.07  \main\114  VTeam-Pröhl
 * - some trace code added
 *
 * 16.04.07  \main\115  VTeam-Battis
 * - compiler warings removed.
 *
 * 19.04.07  \main\116  VTeam-Pröhl
 * - bugfix in CBR_lSignalCallBackInit(). Callback was set more than ones.
 *
 * 15.06.07  \main\117  VTeam-Pröhl
 * - feature CSM_S_USE_CALLBACK_WITH_HANDLE adapted to cbr_vCheckConfig()
 *
 * 05.07.07  \main\118  VTeam-Pröhl
 * - lint warnings removed/disabled
 *
 * 29.07.07  \main\119  VTeam-Pröhl
 * - handle to short telegramms in function cbr_vReceiveInd(). Renault projects
 *   cannot use CSM_S_CDA_DISCARD_MSG_SHORTER_DLC.
 *
 * 30.07.07  \main\120  VTeam-Battis
 * - dependency to big endian signals added in consistency check of cbr_vCheckConfig()
 * - code change reviewed.
 *
 * 31.07.07 \main\121  CM-DI/ESA3-Looby
 * - Dependency to big endian signals added in consistency check of cbr_vCheckConfig()
 * - Added #ifdef for CSM_S_CBR_BIG_ENDIAN_SIGNALS, DLC consistency check, for Tx.
 *
 * 10.08.07  \main\122  VTeam-Pröhl
 * - dis/enable signal monitoring added.
 *
 * 13.08.07  \main\123  VTeam-Pröhl
 * - check for CSM_S_TRACE_LIMITED_SIGNAL added.
 *
 * 27.09.07  \main\124  VTeam-Pröhl
 * - fix in CBR_vStartTransmitMinDelayTimer(). Timer is only set if timeout value is != 0.
 *
 * 10.10.07  \main\125  VTeam-Battis
 * - in case located on a proxy: accept receive indications also if official RX path
 *   is not switched on. Needed for SendAllSignalsagain after undervoltage in case
 *   the bus is sleeping.
 *
 * 19.10.07  \main\126  VTeam-Pröhl
 * - bugfix for timer handling to avoid race conditions between stop-timer call and timer expired call.
 *
 * 19.10.07  \main\127  VTeam-Pröhl
 * - bugfix for timer handling to avoid race conditions between stop-timer call and timer expired call. secound try.
 *
 * 11.02.08  \main\128  VTeam-Pröhl
 * - lint authorisation added.
 *
 * 21.02.08  \main\129  VTeam-Pröhl
 * - lint authorisation added.
 *
 * 22.02.08  \main\130  VTeam-Pröhl
 * - make doxygen happy.
 *
 * 10.04.08  \main\131  VTeam-Pröhl
 * - Function CBR_lMultipleSignalRead() added.
 *
 * 16.04.08  \main\132  VTeam-Pröhl
 * - make lint happy.
 *
 * 29.04.08  \main\133  VTeam-Pröhl
 * - CBR_lMultipleSignalWrite() added.
 * - CBR_lMultipleSignalRead() modified.
 *
 * 28.05.08  \main\134  VTeam-Pröhl
 * - bugfix for CBR_lMultipleSignalWrite().
 *
 * 29.05.08  \main\135  VTeam-Pröhl
 * - lint warning removed.
 *
 * 21.11.08  \main\136  VTeam-Pröhl
 * - some reviews done.
 *
 * 24.11.08  \main\137  VTeam-Pröhl
 * - CBR_lSignalCallBackInit rebuild to \main\135 Version.
 *
 * 15.04.09  \main\138  Pröhl
 * - const added to  CBR_lMultipleSignalWrite().
 *
 * 21.01.10  \main\139  Pröhl
 * - lint warning removed.
 *
 * 25.01.11  \main\140  Borck
 * - CBR_vProcessTransmit() modified for usage with global extended data struct
 *   CSM_CAN_TX_DATA. The element dwTimeout is preset with the value CBR_TX_TIMEOUT
 *   to be passed to lowest CDA layer.
 *
 * 26.01.11  \main\141  Borck
 * - previously introduced changes put under feature switch CSM_S_CDA_TRANSMIT_TIMEOUT
 *
 * 12.07.11  \main\142  Borck
 * - ETG tracing introduced
 * - some comments added while searching for Nissan bug
 * - CBR_ProcessSignalInd() changed due to bug. -> fixed missing multi bus support
 *
 * 28.11.11 \main\143 CM/PJ-H- Basavaraj patil
 * - Support for both Big and Small endian signals within the project and within a message.
 * - i.e. add additional endian's parameter for function CBR_lSignalWriteRaw and CBR_lSignalReadRaw.
 * - Used feature switch CSM_S_CBR_BOTH_ENDIANS_SIGNALS to implement this feature
 *
 * 14.09.12  \main\144  Borck
 * - feature CSM_S_CBR_DISCARD_MSG_SHORTER_DLC added and used:
 * - CBR RX messages will NOT be updated if DLC is too short
 * - code readability improved within some lines to properly support notepads "function list" feature
 *
 * 05.03.13  \main\145 Borck
 * - current implementation allows only up to 32 signals per message.
 *   ConfigCheck() has been extended by a check (Assertion)
 *
 * 03.04.13  \main\146 Borck
 * - possible amount of signals per telegramm now extended to 64 (by change in cbr_m.h)
 * - assertion from previous change removed again
 * - macros cbr_vSetIndicationFlag, cbr_vClrIndicationFlag and cbr_fGetIndicationFlag adapted against mentioned header file
 *
 * 26.07.13  \main\147 Pröhl
 * - in CBR_vProcessSignalInd(void) CCS part adapted to latest CSM.
 *
 * 07.11.13  \main\148 Pröhl
 * - lint warning removed.
 *
 * 19.11.13  \main\149 Pröhl
 * - missing error code in return value added in CBR_lSignalCallBackInit() if no free palce for call back is availiable
 *
 * 04.12.13  \main\150 Pröhl
 * - only small lint fixes.
 *
 * 06.03.14  \main\151 Pröhl
 * - CSM_S_CBR_TX_IF_APPL_ACTIVE works now if CSM_S_REMOTE_STACK is ON
 *
 * 06.03.14  \main\152 Pröhl
 * - :-) CSM_S_CBR_TX_IF_APPL_ACTIVE works now if CSM_S_REMOTE_STACK is ON
 *
 * 11.12.14  \main\153 Borck
 * - feature CSM_S_CBR_MSG_ROLL_COUNTER introduced
 *   This allows to have a rolling counter for TX messages, if configured (do not ask if this is usefull for broadcasts)
 *   A static counter-array with an init-sequence and the function cbr_vHandleRollingCounter()
 *   called from within CBR_vProcessTransmit() have been added. The mentioned feature switch has to be added in csm_stack_p.h !!!
 *   csm_stack_mp.h has to be extended with a configuration array. To have an example, pls. use the branch "nissan_itg5_sd" and have a look !!!
 *
 * 27.04.15  \main\154 Borck
 * merged from ITG5-branch to have new features available here.
 * for the additionally required switches (csm_stack_mp.h, csm_stack_p.h please refer to the branch "nissan_itg5_sd"
 * - feature CSM_S_CBR_ABSTRACT_SIGNALS introduced (first approach for Non-Proxy systems ONLY !!!)
 *   This allows to keep signal access (callback-init, read/write, multiple read/write) independent from the internal signal definition.
 *   That internal "signal access definitions" contains a telegram and a signal index. If telegrams and signal locations
 *   have to be moved within the RX/TX configuration due to a complete customer re-configuration (via diag tester),
 *   the signal access would not work correctly anymore.
 *   The feature supports a translation for an abstract signal to the internal signal, based on tables.
 *   The mentioned feature switch has to be added in the projects' csm_stack_p.h !!!
 *   csm_stack_mp.h has to be extended with a configuration array. To have an example, pls. use the branch "nissan_itg5_sd" and have a look !!!
 * - some code indentions and line feeds optimized for better readability.
 * - rolling counter mechanism reworked slightly -> global pointer to config array introduced to be able to switch between several configurations
 * - ETG trace output featured with the usual switches
 * - definition of some local vars moved to the top of their function due to errors with v850 compiler
 * - trace without ETG for Autosar environment.
 * - a lot of non-functional changes (code cleanup) for better readability
 *   -- function signatures written into 1 single line
 *   -- indentions corrected
 *   -- assignments using CSM_M_MAKE_ERROR written into 1 single line
 *   -- for(...) statement written to 1 single line
 *
 * 19.05.15  \main\155 Pröhl
 * - CBR_vProcessTransmit() works now if CSM_S_REMOTE_STACK is ON
 *
 * 10.09.15  \main\156 Pröhl
 * - bus wake temp handling added for proxy (REMOTE_STACK) software.
 *   Bus wake temp are now send to the V850. The V850 does the wakeup.
 *
 * 17.09.15  \main\157    A.Borck
 * - 2 variables introduced to handle the nr. of RX/TX messages (instead of #defines)
 *   -- refer to line 163ff. to find a short explanition
 *
 * 22.09.15  \main\158    A.Borck
 * - some assertions added to check, that the new code (variables) is working as intended
 *
 * 22.09.15  \main\159    A.Borck
 * - the v850 C-compiler does not allow declarations after statements
 *   -> asserts put in for iMX site had to be moved in some cases
 *
 * 08.10.15  \main\160    A.Borck
 * - feature CSM_S_CBR_ABSTRACT_SIGNALS (Mini Abstraction) removed due to introduction of CBR_AL
 *
 * 13.10.15  \main\161    A.Borck
 * - take usage of feature switch CSM_S_RUNTIME_FEATURE_BASED_CONFIG for slot-based variants to switch of TX of certain messages
 *   -- evaluation which message of an entire configuration is inhibited for TX is done in csm_apl.c
 *
 * 16.10.15  \main\162    A.Borck
 * - some LINT fixes related to version 160, some comments added
 *
 * 22.01.16  \main\163    M.Pröhl
 * - feature CSM_S_CBR_RX_MASK added.
 *
 * 17.05.16  \main\164    M.Pröhl
 * - fix for return value handling in CBR_lMultipleSignalWrite() and CBR_lMultipleSignalRead() added.
 *
 * 27.05.16  \main\165    M.Pröhl
 * - define CSM_S_CBR_MSG_ROLL_COUNTER changed to CSM_S_CBR_TX_MSG_ROLL_COUNTER
 *
 * 02.06.16  \main\166    M.Pröhl
 * - feature CSM_S_CBR_TX_MSG_CRC, CSM_S_CBR_RX_MSG_ROLL_COUNTER and CSM_S_CBR_RX_MSG_CRC added.
 *
 * 23.06.16  \main\167    A.Borck
 * - Function CBR_lFullFrameSignalWrite() added (required for CBR gateway feature along with CCS)
 *
 * 01.07.16  \main\168    A.Borck
 * - update for the previous full frame write feature to improve, simplify and share code
 *
 * 13.07.16  \main\169    M.Pröhl
 * - support for CSM_S_CBR_LITTLE_ENDIAN_SIGNALS and CSM_S_CBR_BIG_ENDIAN_SIGNALS added in function  CBR_lFullFrameSignalWrite()
 *
 * 14.07.16  \main\170    M.Pröhl
 * - fix in function  CBR_lFullFrameSignalWrite()
 *
 * 31.08.16  \main\171    A.Borck
 * - fix for situations, where checking if a TX telegramm buffer has changed after signal writings
 *
 * 22.09.16  \main\172    A.Borck
 * - fix in cbr_fIsTxRequiredAfterSignalWrite() reagrding the TX-type bits for CSM_C_TX_FCT_ACTIVE and CSM_C_TX_BUS_WAKE_TEMP
 *
 * 11.11.16  \main\173    A.Borck
 * - trace buffer sizes have been checked against CSM_MAX_TRACE_LENGTH due to reduction of the same in csm_stack_m.h
 * - some code alignment, no functional changes
 * - some LIN issues
 *
 * 07.07.17  \main\174    A.Borck
 * - robustness improved by checking config pointers before using (TxCrc,TxRollCount, RxCrc, RxRollCount)
 * - TX CRC handling allows AUTOSAR or Renault/Nissan algorithm by using feature switch CSM_S_CBR_TX_MSG_CRC_TYPE_RN == Off/On
 *
 * 10.08.17  \main\175    A.Borck
 * - final optimisations regarding ETG TRACE output (iMX site) -> USER3 should be used as standard, USER4 is with "Process Signal Indication"
 *
 * 16.08.17  \main\176    A.Borck
 * - Transmit processing function modified:
 *   - to support TX Rolling Counter groups
 *   - TX restriction handling moved upwards to take use of it in Rolling Counter group handling as well
 * - struct to maintain a TX rolling counter introduced (simple byte array replaced)
 * - version history updated due to incorrect numbers
 *
 * \history_end */
/* *** END OF FILE ************************************************************ */
