/*****************************************************************************
| FILE:         ProcCsm.cpp
| PROJECT:      NISSAN LCN2kai
| SW-COMPONENT: Process
|-----------------------------------------------------------------------------
| DESCRIPTION:  Process Base Entry
|
|-----------------------------------------------------------------------------
| COPYRIGHT:    (c) 2016 Bosch Car Multimegia GmbH
| HISTORY:
| Date      | Modification                                    | Author
| 10.02.16  |                                                 | Kollai
| 22.11.16  | HandleInit is now storing the queue item
|             element size -> CSMAL_lStoreHandleData extended | Borck
|
| --.--.--  | ----------------           | -------, -----
|
|*****************************************************************************/
#include <algorithm>
#include <vector>

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#define CSM_S_IMPORT_INTERFACE_GENERIC_USER
#include "csm_if.h"
#include "csm_stack_m.h"
#include "csm_access_krnl.h"

//#include "ProcCsmMain.h"


// Trace class definition for ETG (extented trace generator).

#define TR_CLASS_CSMPROC (TR_COMP_CSM + 0x10)

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CSMPROC
#include "trcGenProj/Header/ProcCsmAccessKrnl.cpp.trc.h"
#endif


#define CSM_PROC_WORKER_THREAD_NAME       "PROC_CSM_WORK"
#define CSM_PROC_WORKER_THREAD_STACKSIZE  4096
#define CSM_PROC_WORKER_THREAD_PRIO       100

#define CSM_C_PROC_MESSAGE_QUEUE_MAX_ENTRIES  20

typedef struct
{
   OSAL_tProcessID     hProcessID;
   tU32              u32MsgQueueCnt;
   OSAL_tMQueueHandle  hMessageQueueID;
} trCSMParentMessageQueueAdmin;

std::vector<trCSMParentMessageQueueAdmin, std::allocator<trCSMParentMessageQueueAdmin> > vecParentMessageQueueAdmin;

OSAL_tMQueueHandle hMessageQueueProc = OSAL_C_INVALID_HANDLE;

tVoid vStartWorkerThread( tVoid );
tVoid vWorkerThreadFunction( tPVoid pvArg );

tVoid vHandleInit( trCSMProcMessageQueueItemInit rInit );
tVoid vHandleRegisterAppl( trCSMProcMessageQueueItemRegisterAppl rRegisterAppl );
tVoid vHandleGetCommunicationState( trCSMProcMessageQueueItemGetCommunicationState rGetCommunicationState );
tVoid vHandleSignalRead( trCSMProcMessageQueueItemSignalRead rSignalRead );
tVoid vHandleMultipleSignalRead( trCSMProcMessageQueueItemMultipleSignalRead rMultipleSignalRead );
tVoid vHandleDataRead( trCSMProcMessageQueueItemDataRead rDataRead );

tVoid vSendGeneralResult( tCString coszDebugString, tenCSMParentMessageQueueItemID enItemID, OSAL_tProcessID hProcessID, tU32 u32MsgQueueCnt, tS32 s32Result );

OSAL_tMQueueHandle hGetParentMessageQueueID( OSAL_tProcessID hProcessID, tU32 u32MsgQueueCnt );
tVoid vCloseAllParentMessageQueues( tVoid );


//**************************************************************************************************
// start our worker thread
//**************************************************************************************************
tVoid vStartWorkerThread( tVoid )
{

   CSMAL_lInit();

   OSAL_trThreadAttribute  rAttr;
   OSAL_tThreadID          hThreadId = 0;
   tC8                     szThreadName[OSAL_C_U32_MAX_NAMELENGTH] = CSM_PROC_WORKER_THREAD_NAME;

   rAttr.szName       = szThreadName;
   rAttr.s32StackSize = CSM_PROC_WORKER_THREAD_STACKSIZE;
   rAttr.u32Priority  = CSM_PROC_WORKER_THREAD_PRIO;
   rAttr.pfEntry      = (OSAL_tpfThreadEntry)vWorkerThreadFunction;
   rAttr.pvArg        = NULL;

   hThreadId = OSAL_ThreadSpawn(&rAttr);
}

//**************************************************************************************************
// our worker thread function
//**************************************************************************************************
tVoid vWorkerThreadFunction( tPVoid pvArg )
{
  (tVoid)pvArg;
   #if (CSM_S_TRACE == CSM_C_F_ON)
     ETG_TRACE_USR1(("ProcCsm: start worker thread of CSM process"));
   #endif

   if ( OSAL_s32MessageQueueCreate( CSM_C_PROC_MESSAGE_QUEUE_NAME,
                                    CSM_C_PROC_MESSAGE_QUEUE_MAX_ENTRIES,
                                    sizeof(trCSMProcMessageQueueItem),
                                    OSAL_EN_READWRITE,
                                    &hMessageQueueProc )
        == OSAL_OK )
   {
      tS32 s32ReadBytes;
      trCSMProcMessageQueueItem  rItem;

      while ( TRUE )
      {
         s32ReadBytes = OSAL_s32MessageQueueWait( hMessageQueueProc,
                                                  (tU8*)&rItem,
                                                  sizeof(rItem),
                                                  OSAL_NULL,  // equal priorities
                                                  OSAL_C_U32_INFINITE );

         if ( s32ReadBytes > 0 )
         {
            switch ( rItem.enID )
            {
               case CSM_PROC_MESSAGE_QUEUE_ID_INIT:
               {
                  vHandleInit( rItem.uData.rInit );
               }
               break;
               case CSM_PROC_MESSAGE_QUEUE_ID_REGISTER_APPL:
               {
                  vHandleRegisterAppl( rItem.uData.rRegisterAppl );
               }
               break;
               case CSM_PROC_MESSAGE_QUEUE_ID_UNREGISTER_APPL:
               {
                  tS32 s32RetVal;

                  #if (CSM_S_TRACE == CSM_C_F_ON)
                     ETG_TRACE_USR1(("ProcCsm: message UNREGISTER APPL received; handle = 0x%08x",
                                      rItem.uData.rUnregisterAppl.pvHandle));
                  #endif

                  s32RetVal = CSMAL_lUnregisterAppl( rItem.uData.rUnregisterAppl.pvHandle );

                  vSendGeneralResult( (tCString)"UNREGISTER APPL",
                                      CSM_PARENT_MESSAGE_QUEUE_ID_UNREGISTER_APPL,
                                      rItem.uData.rUnregisterAppl.hProcessID,
                                      rItem.uData.rUnregisterAppl.u32MsgQueueCnt,
                                      s32RetVal );
               }
               break;
               case CSM_PROC_MESSAGE_QUEUE_ID_REGISTER_SIGNAL:
               {
                  tS32 s32RetVal;

                  #if (CSM_S_TRACE == CSM_C_F_ON)
                     ETG_TRACE_USR1(("ProcCsm: message REGISTER SIGNAL received; handle = 0x%08x, signal = 0x%08x, callback = 0x%08x",
                                      rItem.uData.rRegisterSignal.pvHandle,
                                      rItem.uData.rRegisterSignal.u32SignalId,
                                      rItem.uData.rRegisterSignal.vPFNCBRSignalInd));
                  #endif

                  s32RetVal = CSMAL_lRegisterSignal( rItem.uData.rRegisterSignal.pvHandle,
                                                     rItem.uData.rRegisterSignal.u32SignalId,
                                                     rItem.uData.rRegisterSignal.vPFNCBRSignalInd );

                  vSendGeneralResult( (tCString)"REGISTER SIGNAL",
                                      CSM_PARENT_MESSAGE_QUEUE_ID_REGISTER_SIGNAL,
                                      rItem.uData.rRegisterSignal.hProcessID,
                                      rItem.uData.rRegisterSignal.u32MsgQueueCnt,
                                      s32RetVal );
               }
               break;
               case CSM_PROC_MESSAGE_QUEUE_ID_UNREGISTER_SIGNAL:
               {
                  tS32 s32RetVal;

                  #if (CSM_S_TRACE == CSM_C_F_ON)
                     ETG_TRACE_USR1(("ProcCsm: message UNREGISTER SIGNAL received; handle = 0x%08x, signal = 0x%08x",
                                      rItem.uData.rUnregisterSignal.pvHandle,
                                      rItem.uData.rUnregisterSignal.u32SignalId));
                  #endif

                  s32RetVal = CSMAL_lUnregisterSignal( rItem.uData.rUnregisterSignal.pvHandle,
                                                       rItem.uData.rUnregisterSignal.u32SignalId );

                  vSendGeneralResult( (tCString)"UNREGISTER SIGNAL",
                                      CSM_PARENT_MESSAGE_QUEUE_ID_UNREGISTER_SIGNAL,
                                      rItem.uData.rUnregisterSignal.hProcessID,
                                      rItem.uData.rUnregisterSignal.u32MsgQueueCnt,
                                      s32RetVal );
               }
               break;
               case CSM_PROC_MESSAGE_QUEUE_ID_UNREGISTER_ALL_SIGNALS:
               {
                  tS32 s32RetVal;

                  #if (CSM_S_TRACE == CSM_C_F_ON)
                     ETG_TRACE_USR1(("ProcCsm: message UNREGISTER ALL SIGNALS received; handle = 0x%08x",
                                      rItem.uData.rUnregisterAllSignals.pvHandle));
                  #endif

                  s32RetVal = CSMAL_lUnregisterAllSignals( rItem.uData.rUnregisterAllSignals.pvHandle );

                  vSendGeneralResult( (tCString)"UNREGISTER ALL SIGNALS",
                                      CSM_PARENT_MESSAGE_QUEUE_ID_UNREGISTER_ALL_SIGNALS,
                                      rItem.uData.rUnregisterAllSignals.hProcessID,
                                      rItem.uData.rUnregisterAllSignals.u32MsgQueueCnt,
                                      s32RetVal );
               }
               break;
               case CSM_PROC_MESSAGE_QUEUE_ID_REGISTER_BUS_ERROR_IND:
               {
                  tS32 s32RetVal;

                  #if (CSM_S_TRACE == CSM_C_F_ON)
                     ETG_TRACE_USR1(("ProcCsm: message REGISTER BUS ERROR IND received; handle = 0x%08x, callback = 0x%08x",
                                      rItem.uData.rRegisterBusErrorInd.pvHandle,
                                      rItem.uData.rRegisterBusErrorInd.vPFNBusErrorInd));
                  #endif

#if (CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON)

                  s32RetVal = CSMAL_lRegisterBusErrorInd( rItem.uData.rRegisterBusErrorInd.pvHandle,
                                                          rItem.uData.rRegisterBusErrorInd.vPFNBusErrorInd );
#else
                  s32RetVal = CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_SRVC_UNAVAILABLE );
#endif
                  vSendGeneralResult( (tCString)"REGISTER BUS ERROR IND",
                                      CSM_PARENT_MESSAGE_QUEUE_ID_REGISTER_BUS_ERROR_IND,
                                      rItem.uData.rRegisterBusErrorInd.hProcessID,
                                      rItem.uData.rRegisterBusErrorInd.u32MsgQueueCnt,
                                      s32RetVal );
               }
               break;
               case CSM_PROC_MESSAGE_QUEUE_ID_COMMUNICATION_REQ:
               {
                  tS32 s32RetVal;

                  #if (CSM_S_TRACE == CSM_C_F_ON)
                     ETG_TRACE_USR1(("ProcCsm: message COMMUNICATION REQ received; bus = %d, protocol = 0x%08x, AddrField = %d %d, Action = %d, AppID = 0x%04x",
                                      rItem.uData.rCommunicationReq.u8Bus,
                                      rItem.uData.rCommunicationReq.u32ProtocolType,
                                      rItem.uData.rCommunicationReq.au8AddressField[0],
                                      rItem.uData.rCommunicationReq.au8AddressField[1],
                                      rItem.uData.rCommunicationReq.u8Action,
                                      rItem.uData.rCommunicationReq.u16ApplID));
                  #endif

                  s32RetVal = CSM_lCommunicationReq( rItem.uData.rCommunicationReq.u8Bus,
                                                     rItem.uData.rCommunicationReq.u32ProtocolType,
                                                     (tPCVoid)&(rItem.uData.rCommunicationReq.au8AddressField[0]),
                                                     rItem.uData.rCommunicationReq.u8Action,
                                                     rItem.uData.rCommunicationReq.u16ApplID );

                  vSendGeneralResult( (tCString)"COMMUNICATION REQ",
                                      CSM_PARENT_MESSAGE_QUEUE_ID_COMMUNICATION_REQ,
                                      rItem.uData.rCommunicationReq.hProcessID,
                                      rItem.uData.rCommunicationReq.u32MsgQueueCnt,
                                      s32RetVal );
               }
               break;
               case CSM_PROC_MESSAGE_QUEUE_ID_DATA_REQUEST:
               {
                  tS32 s32RetVal;

                  #if (CSM_S_TRACE == CSM_C_F_ON)
                     ETG_TRACE_USR1(("ProcCsm: message DATA REQ received; protocol = 0x%08x, AddrField = %d %d, length = %d, data = 0x%02x 0x%02x",
                                      rItem.uData.rDataRequest.u32ProtocolType,
                                      rItem.uData.rDataRequest.au8AddressField[0],
                                      rItem.uData.rDataRequest.au8AddressField[1],
                                      rItem.uData.rDataRequest.u16DataLength,
                                      rItem.uData.rDataRequest.au8Data[0],
                                      rItem.uData.rDataRequest.au8Data[1]));
                  #endif

                  s32RetVal = CSM_lDataReq( rItem.uData.rDataRequest.u32ProtocolType,
                                            (tPCVoid)&(rItem.uData.rDataRequest.au8AddressField[0]),
                                            rItem.uData.rDataRequest.au8Data,
                                            rItem.uData.rDataRequest.u16DataLength );

                  vSendGeneralResult( (tCString)"DATA REQ",
                                      CSM_PARENT_MESSAGE_QUEUE_ID_DATA_REQUEST,
                                      rItem.uData.rDataRequest.hProcessID,
                                      rItem.uData.rDataRequest.u32MsgQueueCnt,
                                      s32RetVal );
               }
               break;
               case CSM_PROC_MESSAGE_QUEUE_ID_DATA_IND_PROCESSED:
               {
                  tS32 s32RetVal;

                  #if (CSM_S_TRACE == CSM_C_F_ON)
                     ETG_TRACE_USR1(("ProcCsm: message DATA IND PROCESSED received; protocol = 0x%08x, AddrField = %d %d",
                                      rItem.uData.rDataIndProcessed.u32ProtocolType,
                                      rItem.uData.rDataIndProcessed.au8AddressField[0],
                                      rItem.uData.rDataIndProcessed.au8AddressField[1]));
                  #endif

                  s32RetVal = CSM_lDataIndProcessed( rItem.uData.rDataIndProcessed.u32ProtocolType,
                                                     (tPCVoid)&(rItem.uData.rDataIndProcessed.au8AddressField[0]) );

                  vSendGeneralResult( (tCString)"IND PROCESSED",
                                      CSM_PARENT_MESSAGE_QUEUE_ID_DATA_IND_PROCESSED,
                                      rItem.uData.rDataIndProcessed.hProcessID,
                                      rItem.uData.rDataIndProcessed.u32MsgQueueCnt,
                                      s32RetVal );
               }
               break;
               case CSM_PROC_MESSAGE_QUEUE_ID_GET_COMMUNICATION_STATE:
               {
                  vHandleGetCommunicationState( rItem.uData.rGetCommunicationState );
               }
               break;
               case CSM_PROC_MESSAGE_QUEUE_ID_SIGNAL_WRITE:
               {
                  tS32 s32RetVal;

                  #if (CSM_S_TRACE == CSM_C_F_ON)
                     ETG_TRACE_USR1(("ProcCsm: message SIGNAL WRITE received; signal = 0x%08x, length = %d, Type 0x%02x",
                                      rItem.uData.rSignalWrite.u32SignalId,
                                      rItem.uData.rSignalWrite.u8Length,
                                      rItem.uData.rSignalWrite.u8TxType));
                  #endif

                  s32RetVal = CSM_lSignalWrite( rItem.uData.rSignalWrite.u32SignalId,
                                                (tPCVoid)&(rItem.uData.rSignalWrite.au8Data[0]),
                                                rItem.uData.rSignalWrite.u8Length,
                                                rItem.uData.rSignalWrite.u8TxType);

                  vSendGeneralResult( (tCString)"SIGNAL WRITE",
                                      CSM_PARENT_MESSAGE_QUEUE_ID_SIGNAL_WRITE,
                                      rItem.uData.rSignalWrite.hProcessID,
                                      rItem.uData.rSignalWrite.u32MsgQueueCnt,
                                      s32RetVal );
               }
               break;
               case CSM_PROC_MESSAGE_QUEUE_ID_MULTIPLE_SIGNAL_WRITE:
               {
                  tS32 s32RetVal;
                  tU8   u8NumberOfSignals;
                  tCSM_MultipleSignalWrite *prSignalList;

                  u8NumberOfSignals = rItem.uData.rMultipleSignalWrite.u8NumberOfSignals;

                  #if (CSM_S_TRACE == CSM_C_F_ON)
                     ETG_TRACE_USR1(("ProcCsm: message MULTIPLE SIGNAL WRITE received; number of signals %d",
                                      u8NumberOfSignals));
                  #endif

                  prSignalList = OSAL_NEW tCSM_MultipleSignalWrite[u8NumberOfSignals];

                  if ( NULL != prSignalList )
                  {
                     for ( tU8 u8SignalCnt = 0; u8SignalCnt < u8NumberOfSignals; u8SignalCnt++ )
                     {
                        #if (CSM_S_TRACE == CSM_C_F_ON)
                           ETG_TRACE_USR1(("ProcCsm: signal = 0x%08x, length = %d, Type 0x%02x",
                                            rItem.uData.rMultipleSignalWrite.arSignalData[u8SignalCnt].u32SignalId,
                                            rItem.uData.rMultipleSignalWrite.arSignalData[u8SignalCnt].u8Length,
                                            rItem.uData.rMultipleSignalWrite.arSignalData[u8SignalCnt].u8TxType));
                        #endif

                        (prSignalList + u8SignalCnt)->dwSignalId        = rItem.uData.rMultipleSignalWrite.arSignalData[u8SignalCnt].u32SignalId;
                        (prSignalList + u8SignalCnt)->pvNewSignalData   = (tPVoid)&(rItem.uData.rMultipleSignalWrite.arSignalData[u8SignalCnt].au8Data[0]);
                        (prSignalList + u8SignalCnt)->bDataBufferLength = rItem.uData.rMultipleSignalWrite.arSignalData[u8SignalCnt].u8Length;
                        (prSignalList + u8SignalCnt)->bTxType           = rItem.uData.rMultipleSignalWrite.arSignalData[u8SignalCnt].u8TxType;
                     }

                     #if ((CSM_S_iMX_SIGNAL_ABSTRACTION == CSM_C_F_ON) || (CSM_S_v850_SIGNAL_ABSTRACTION == CSM_C_F_ON))
                     s32RetVal = CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_SRVC_UNAVAILABLE );
                     #else
                     s32RetVal = CSM_lMultipleSignalWrite( prSignalList, u8NumberOfSignals );
                     #endif

                     OSAL_DELETE[] prSignalList;
                  }
                  else
                  {
                     s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_RESOURCE);
                  }

                  vSendGeneralResult( (tCString)"MULTIPLE SIGNAL WRITE",
                                      CSM_PARENT_MESSAGE_QUEUE_ID_MULTIPLE_SIGNAL_WRITE,
                                      rItem.uData.rMultipleSignalWrite.hProcessID,
                                      rItem.uData.rMultipleSignalWrite.u32MsgQueueCnt,
                                      s32RetVal );
               }
               break;
               case CSM_PROC_MESSAGE_QUEUE_ID_SIGNAL_READ:
               {
                  vHandleSignalRead( rItem.uData.rSignalRead );
               }
               break;
               case CSM_PROC_MESSAGE_QUEUE_ID_MULTIPLE_SIGNAL_READ:
               {
                  vHandleMultipleSignalRead( rItem.uData.rMultipleSignalRead );
               }
               break;
               case CSM_PROC_MESSAGE_QUEUE_ID_GET_ALL_STATE_IND_AGAIN:
               {
                  tS32 s32RetVal;

                  #if (CSM_S_TRACE == CSM_C_F_ON)
                     ETG_TRACE_USR1(("ProcCsm: message GET ALL STATE IND AGAIN received"));
                  #endif

#if ( (CSM_S_CBR_AVAILABLE == CSM_C_F_ON) && (CSM_S_CCS_USED == CSM_C_F_ON) )
                  s32RetVal = CSM_lGetAllStateIndicationsAgain();
#else
                  s32RetVal = CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_SRVC_UNAVAILABLE );
#endif

                  vSendGeneralResult( (tCString)"GET ALL STATE IND AGAIN",
                                      CSM_PARENT_MESSAGE_QUEUE_ID_GET_ALL_STATE_IND_AGAIN,
                                      rItem.uData.rGetAllStateIndAgain.hProcessID,
                                      rItem.uData.rGetAllStateIndAgain.u32MsgQueueCnt,
                                      s32RetVal );
               }
               break;
               case CSM_PROC_MESSAGE_QUEUE_ID_CON_TEST:
               {
                  tS32 s32RetVal;

                  #if (CSM_S_TRACE == CSM_C_F_ON)
                     ETG_TRACE_USR1(("ProcCsm: message CON TEST received; protocol = 0x%08x, AddrField = %d %d, Action = %d",
                                      rItem.uData.rConTest.u32ProtocolType,
                                      rItem.uData.rConTest.au8AddressField[0],
                                      rItem.uData.rConTest.au8AddressField[1],
                                      rItem.uData.rConTest.u8Action));
                  #endif

                  s32RetVal = CSM_lConTest( rItem.uData.rConTest.u32ProtocolType,
                                            (tPVoid)&(rItem.uData.rConTest.au8AddressField[0]),
                                            rItem.uData.rConTest.u8Action );

                  vSendGeneralResult( (tCString)"COMMUNICATION REQ",
                                      CSM_PARENT_MESSAGE_QUEUE_ID_CON_TEST,
                                      rItem.uData.rConTest.hProcessID,
                                      rItem.uData.rConTest.u32MsgQueueCnt,
                                      s32RetVal );
               }
               break;
               case CSM_PROC_MESSAGE_QUEUE_ID_DATA_READ:
               {
                  vHandleDataRead( rItem.uData.rDataRead );
               }
               break;
               case CSM_PROC_MESSAGE_QUEUE_ID_MAX:
               default:
               {
               }
               break;
            }
         }
      }
   }

   if ( hMessageQueueProc != OSAL_C_INVALID_HANDLE )
   {
      if ( OSAL_OK == OSAL_s32MessageQueueClose( hMessageQueueProc ) )
      {
         OSAL_s32MessageQueueDelete( CSM_C_PROC_MESSAGE_QUEUE_NAME );
      }
   }
}


tVoid vHandleInit( trCSMProcMessageQueueItemInit rInit )
{
   #if (CSM_S_TRACE == CSM_C_F_ON)
   ETG_TRACE_USR1(("ProcCsm->vHandleInit - ProcessID:0x%08X  MsgQueueCnt:%04d  QueueElemSize:%04d",
                   rInit.hProcessID,
                   rInit.u32MsgQueueCnt,
                   rInit.u32MsgQueueElementSize));
   #endif
   OSAL_tMQueueHandle hMessageQueueID = hGetParentMessageQueueID( rInit.hProcessID, rInit.u32MsgQueueCnt );

   if ( OSAL_C_INVALID_HANDLE != hMessageQueueID )
   {
      pCSMHandle  pAccessUsrHandle;
      trCSMParentMessageQueueItem  rMessage;
      tS32  s32Res;

      // get a CSM access handle, fill it with data, and send it back to user application
      if ( CSMAL_lGetAccessHandle( &pAccessUsrHandle ) >= CSM_C_NO_ERROR )
      {
        CSMAL_lStoreHandleData((void *)pAccessUsrHandle, (LONG)rInit.u32MsgQueueCnt, (LONG)rInit.hProcessID, (LONG)rInit.u32MsgQueueElementSize);
      }

      rMessage.enID = CSM_PARENT_MESSAGE_QUEUE_ID_INIT;
      rMessage.uData.rInit.pvHandle = pAccessUsrHandle;

      s32Res = OSAL_s32MessageQueuePost( hMessageQueueID, (tPCU8)&rMessage, sizeof(rMessage), OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST );
   }
   else
   {
      #if (CSM_S_TRACE == CSM_C_F_ON)
         ETG_TRACE_ERR(("ProcCsm->vHandleInit: could not open message queue"));
      #endif
   }
}


tVoid vHandleRegisterAppl( trCSMProcMessageQueueItemRegisterAppl rRegisterAppl )
{
   tS32  s32RetVal;
   tBool bRegistered = FALSE;

   #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_USR1(("ProcCsm: message REGISTER APPL received; handle = 0x%08x, bus = %d, protocol = 0x%08x, AddrField = %02x %02x u16ApplId = 0x%04x",
                       rRegisterAppl.pvHandle,
                       rRegisterAppl.u8Bus,
                       rRegisterAppl.u32ProtocolType,
                       rRegisterAppl.au8AddressField[0],
                       rRegisterAppl.au8AddressField[1],
                       rRegisterAppl.u16ApplId));

   #endif

   switch ( rRegisterAppl.u32ProtocolType )
   {
#ifdef CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL
      case CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL:
      {
         tCSM_LOCAL_CTRL_APPL_CALLBACK rCallBackFkt;

         rCallBackFkt.pfvCommunicationCon = (tCSM_PFN_TP_COMMUNICATION_CON)rRegisterAppl.pcvCallBackCommunicationCon;
         rCallBackFkt.pfvCommunicationInd = (tCSM_PFN_TP_COMMUNICATION_IND)rRegisterAppl.pcvCallBackCommunicationInd;
         rCallBackFkt.u16ApplID =           rRegisterAppl.u16ApplId;

         s32RetVal = CSMAL_lRegisterAppl( rRegisterAppl.pvHandle,
                                          rRegisterAppl.u8Bus,
                                          rRegisterAppl.u32ProtocolType,
                                          (tPCVoid)&(rRegisterAppl.au8AddressField[0]),
                                          (tPCVoid)&(rCallBackFkt) );
         bRegistered = TRUE;
      }
      break;
#endif /* CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL */

#ifdef CSM_C_PTYPE_GM_NWM
      case CSM_C_PTYPE_GM_NWM:
      {
         tCSM_GMLAN_NWM_APPL_CALLBACK rCallBackFkt;

         rCallBackFkt.pfvCommunicationInd = (tCSM_PFN_TP_COMMUNICATION_IND)rRegisterAppl.pcvCallBackCommunicationInd;

         s32RetVal = CSMAL_lRegisterAppl( rRegisterAppl.pvHandle,
                                          rRegisterAppl.u8Bus,
                                          rRegisterAppl.u32ProtocolType,
                                          (tPCVoid)&(rRegisterAppl.au8AddressField[0]),
                                          (tPCVoid)&(rCallBackFkt) );
         bRegistered = TRUE;
      }
      break;
#endif /* CSM_C_PTYPE_GM_NWM */

#ifdef CSM_C_PTYPE_USED_STD_USDT_ISO
      case CSM_C_PTYPE_USED_STD_USDT_ISO:
      {
  #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
    #if (CSM_S_ISO_TP == CSM_C_F_ON)
         tCSM_ISO_TP_USDT_APPL_CALLBACK rCallBackFkt;

         rCallBackFkt.pfvCommunicationCon = (tCSM_PFN_TP_COMMUNICATION_CON)rRegisterAppl.pcvCallBackCommunicationCon;
         rCallBackFkt.pfvCommunicationInd = (tCSM_PFN_TP_COMMUNICATION_IND)rRegisterAppl.pcvCallBackCommunicationInd;
         rCallBackFkt.pfvDataCon          = (tCSM_PFN_TP_DATA_CON)rRegisterAppl.pcvCallBackDataCon;
         rCallBackFkt.pfvDataInd          = (tCSM_PFN_TP_DATA_IND)rRegisterAppl.pcvCallBackDataInd;
      #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
         rCallBackFkt.pfvDataIndFF        = (tCSM_PFN_TP_DATA_IND_FF)rRegisterAppl.pcvCallBackDataIndFF;
      #endif
      #if (CSM_S_ITP_ERROR_INDICATION_AVAILABLE == CSM_C_F_ON)
         rCallBackFkt.pfvDataErrorInd     = (tCSM_PFN_TP_DATA_ERROR_IND)rRegisterAppl.pcvCallBackDataErrorInd;
      #endif

         s32RetVal = CSMAL_lRegisterAppl( rRegisterAppl.pvHandle,
                                          rRegisterAppl.u8Bus,
                                          rRegisterAppl.u32ProtocolType,
                                          (tPCVoid)&(rRegisterAppl.au8AddressField[0]),
                                          (tPCVoid)&(rCallBackFkt) );
         bRegistered = TRUE;
    #endif
  #endif
      }
      break;
#endif /* CSM_C_PTYPE_USED_STD_USDT_ISO */

#ifdef CSM_C_PTYPE_USED_STD_LIN_DIAG
      case CSM_C_PTYPE_USED_STD_LIN_DIAG:
      {
  #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
    #if (CSM_S_ISO_TP == CSM_C_F_ON)
         tCSM_ISO_TP_USDT_APPL_CALLBACK rCallBackFkt;

         rCallBackFkt.pfvCommunicationCon = (tCSM_PFN_TP_COMMUNICATION_CON)rRegisterAppl.pcvCallBackCommunicationCon;
         rCallBackFkt.pfvCommunicationInd = (tCSM_PFN_TP_COMMUNICATION_IND)rRegisterAppl.pcvCallBackCommunicationInd;
         rCallBackFkt.pfvDataCon          = (tCSM_PFN_TP_DATA_CON)rRegisterAppl.pcvCallBackDataCon;
         rCallBackFkt.pfvDataInd          = (tCSM_PFN_TP_DATA_IND)rRegisterAppl.pcvCallBackDataInd;
      #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
         rCallBackFkt.pfvDataIndFF        = (tCSM_PFN_TP_DATA_IND_FF)rRegisterAppl.pcvCallBackDataIndFF;
      #endif
      #if (CSM_S_ITP_ERROR_INDICATION_AVAILABLE == CSM_C_F_ON)
         rCallBackFkt.pfvDataErrorInd     = (tCSM_PFN_TP_DATA_ERROR_IND)rRegisterAppl.pcvCallBackDataErrorInd;
      #endif

         s32RetVal = CSMAL_lRegisterAppl( rRegisterAppl.pvHandle,
                                          rRegisterAppl.u8Bus,
                                          rRegisterAppl.u32ProtocolType,
                                          (tPCVoid)&(rRegisterAppl.au8AddressField[0]),
                                          (tPCVoid)&(rCallBackFkt) );
         bRegistered = TRUE;
    #endif
  #endif
      }
      break;
#endif /* CSM_C_PTYPE_USED_STD_LIN_DIAG */

#ifdef CSM_C_PTYPE_USED_STD_LIN_USDT
      case CSM_C_PTYPE_USED_STD_LIN_USDT:
      {
  #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
    #if (CSM_S_ISO_TP == CSM_C_F_ON)
         tCSM_ISO_TP_USDT_APPL_CALLBACK rCallBackFkt;

         rCallBackFkt.pfvCommunicationCon = (tCSM_PFN_TP_COMMUNICATION_CON)rRegisterAppl.pcvCallBackCommunicationCon;
         rCallBackFkt.pfvCommunicationInd = (tCSM_PFN_TP_COMMUNICATION_IND)rRegisterAppl.pcvCallBackCommunicationInd;
         rCallBackFkt.pfvDataCon          = (tCSM_PFN_TP_DATA_CON)rRegisterAppl.pcvCallBackDataCon;
         rCallBackFkt.pfvDataInd          = (tCSM_PFN_TP_DATA_IND)rRegisterAppl.pcvCallBackDataInd;
      #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
         rCallBackFkt.pfvDataIndFF        = (tCSM_PFN_TP_DATA_IND_FF)rRegisterAppl.pcvCallBackDataIndFF;
      #endif
      #if (CSM_S_ITP_ERROR_INDICATION_AVAILABLE == CSM_C_F_ON)
         rCallBackFkt.pfvDataErrorInd     = (tCSM_PFN_TP_DATA_ERROR_IND)rRegisterAppl.pcvCallBackDataErrorInd;
      #endif

         s32RetVal = CSMAL_lRegisterAppl( rRegisterAppl.pvHandle,
                                          rRegisterAppl.u8Bus,
                                          rRegisterAppl.u32ProtocolType,
                                          (tPCVoid)&(rRegisterAppl.au8AddressField[0]),
                                          (tPCVoid)&(rCallBackFkt) );
         bRegistered = TRUE;
    #endif
  #endif
      }
      break;
#endif /* CSM_C_PTYPE_USED_STD_LIN_USDT */

#ifdef CSM_C_PTYPE_USED_STD_UUDT_TX
      case CSM_C_PTYPE_USED_STD_UUDT_TX:
      {
  #if (CSM_S_UUDT_TX == CSM_C_F_ON)
         tCSM_UUDT_TX_APPL_CALLBACK rCallBackFkt;

         rCallBackFkt.pfvDataCon = (tCSM_PFN_TP_DATA_CON)rRegisterAppl.pcvCallBackDataCon;

         s32RetVal = CSMAL_lRegisterAppl( rRegisterAppl.pvHandle,
                                          rRegisterAppl.u8Bus,
                                          rRegisterAppl.u32ProtocolType,
                                          (tPCVoid)&(rRegisterAppl.au8AddressField[0]),
                                          (tPCVoid)&(rCallBackFkt) );
         bRegistered = TRUE;
  #endif
      }
      break;
#endif /* CSM_C_PTYPE_USED_STD_UUDT_TX */

#ifdef CSM_C_PTYPE_USED_STD_UUDT_RX
      case CSM_C_PTYPE_USED_STD_UUDT_RX:
      {
  #if (CSM_S_UUDT_RX == CSM_C_F_ON)
         tCSM_UUDT_RX_APPL_CALLBACK rCallBackFkt;

         rCallBackFkt.pfvDataInd = (tCSM_PFN_TP_DATA_IND)rRegisterAppl.pcvCallBackDataInd;
    #if (CSM_S_UUDT_RX_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
         rCallBackFkt.pfvDataErrorInd = (tCSM_PFN_TP_DATA_ERROR_IND)rRegisterAppl.pcvCallBackDataErrorInd;
    #endif

         s32RetVal = CSMAL_lRegisterAppl( rRegisterAppl.pvHandle,
                                          rRegisterAppl.u8Bus,
                                          rRegisterAppl.u32ProtocolType,
                                          (tPCVoid)&(rRegisterAppl.au8AddressField[0]),
                                          (tPCVoid)&(rCallBackFkt) );
         bRegistered = TRUE;
  #endif
      }
      break;
#endif /* CSM_C_PTYPE_USED_STD_UUDT_RX */

#ifdef CSM_C_PTYPE_USED_STD_BR_SIGNAL
      case CSM_C_PTYPE_USED_STD_BR_SIGNAL:
      {
         tCSM_BR_SIGNAL_APPL_CALLBACK rCallBackFkt;

         rCallBackFkt.pfvCommunicationCon = (tCSM_PFN_TP_COMMUNICATION_CON)rRegisterAppl.pcvCallBackCommunicationCon;
         rCallBackFkt.pfvCommunicationInd = (tCSM_PFN_TP_COMMUNICATION_IND)rRegisterAppl.pcvCallBackCommunicationInd;
         rCallBackFkt.u16ApplID =           rRegisterAppl.u16ApplId;

         s32RetVal = CSMAL_lRegisterAppl( rRegisterAppl.pvHandle,
                                          rRegisterAppl.u8Bus,
                                          rRegisterAppl.u32ProtocolType,
                                          (tPCVoid)&(rRegisterAppl.au8AddressField[0]),
                                          (tPCVoid)&(rCallBackFkt) );
         bRegistered = TRUE;
      }
      break;
#endif /* CSM_C_PTYPE_USED_STD_BR_SIGNAL */

#ifdef CSM_C_PTYPE_MCNET_ASDT_STD
      case CSM_C_PTYPE_MCNET_ASDT_STD:
      {
        /* not supported yet */
      }
#endif

#ifdef CSM_C_PTYPE_MCNET_USDT
      case CSM_C_PTYPE_MCNET_USDT:
      {
        /* not supported yet */
      }
#endif

#ifdef CSM_C_PTYPE_VW_BAP
      case CSM_C_PTYPE_VW_BAP:
      {
  #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
         tCSM_VW_BAP_APPL_CALLBACK rCallBackFkt;

         rCallBackFkt.pfvCommunicationCon = (tCSM_PFN_TP_COMMUNICATION_CON)rRegisterAppl.pcvCallBackCommunicationCon;
         rCallBackFkt.pfvDataCon          = (tCSM_PFN_TP_DATA_CON)rRegisterAppl.pcvCallBackDataCon;
         rCallBackFkt.pfvDataInd          = (tCSM_PFN_TP_DATA_IND)rRegisterAppl.pcvCallBackDataInd;

         s32RetVal = CSMAL_lRegisterAppl( rRegisterAppl.pvHandle,
                                          rRegisterAppl.u8Bus,
                                          rRegisterAppl.u32ProtocolType,
                                          (tPCVoid)&(rRegisterAppl.au8AddressField[0]),
                                          (tPCVoid)&(rCallBackFkt) );
         bRegistered = TRUE;
  #endif
      }
      break;
#endif /* CSM_C_PTYPE_VW_BAP */

#ifdef CSM_C_PTYPE_PAG_OSEK_NWM
      case CSM_C_PTYPE_PAG_OSEK_NWM:
      {
         tCSM_OSEK_NWM_APPL_CALLBACK rCallBackFkt;

         rCallBackFkt.pfvCommunicationInd = (tCSM_PFN_TP_COMMUNICATION_IND)rRegisterAppl.pcvCallBackCommunicationInd;

         s32RetVal = CSMAL_lRegisterAppl( rRegisterAppl.pvHandle,
                                          rRegisterAppl.u8Bus,
                                          rRegisterAppl.u32ProtocolType,
                                          (tPCVoid)&(rRegisterAppl.au8AddressField[0]),
                                          (tPCVoid)&(rCallBackFkt) );
         bRegistered = TRUE;
      }
      break;
#endif /* CSM_C_PTYPE_PAG_OSEK_NWM */

      #ifdef CSM_C_PTYPE_RN_MPDT_C
      case CSM_C_PTYPE_RN_MPDT_C:
      case CSM_C_PTYPE_RN_MPDT_D:
      {
         #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
         tCSM_MPDT_APPL_CALLBACK rCallBackFkt;

         rCallBackFkt.pfvCommunicationCon = (tCSM_PFN_TP_COMMUNICATION_CON)rRegisterAppl.pcvCallBackCommunicationCon;
         rCallBackFkt.pfvCommunicationInd = (tCSM_PFN_TP_COMMUNICATION_IND)rRegisterAppl.pcvCallBackCommunicationInd;
         rCallBackFkt.pfvCDataCon         = (tCSM_PFN_TP_DATA_CON)rRegisterAppl.pcvCallBackDataCon;
         rCallBackFkt.pfvCDataInd         = (tCSM_PFN_TP_DATA_IND)rRegisterAppl.pcvCallBackDataInd;
         rCallBackFkt.pfvDDataCon         = (tCSM_PFN_TP_DATA_CON)rRegisterAppl.pcvCallBackDDataCon;
         rCallBackFkt.pfvDDataInd         = (tCSM_PFN_TP_DATA_IND)rRegisterAppl.pcvCallBackDDataInd;

         s32RetVal = CSMAL_lRegisterAppl( rRegisterAppl.pvHandle,
                                          rRegisterAppl.u8Bus,
                                          rRegisterAppl.u32ProtocolType,
                                          (tPCVoid)&(rRegisterAppl.au8AddressField[0]),
                                          (tPCVoid)&(rCallBackFkt) );
         bRegistered = TRUE;
         #endif
      }
      break;
      #endif /* CSM_C_PTYPE_RN_MPDT_C */

#ifdef CSM_C_PTYPE_RN_OSEKI_NWM
      case CSM_C_PTYPE_RN_OSEKI_NWM:
      {
         tCSM_OSEKI_NWM_APPL_CALLBACK rCallBackFkt;

         rCallBackFkt.u16ApplID =           rRegisterAppl.u16ApplId;
         rCallBackFkt.pfvCommunicationInd = (tCSM_PFN_TP_COMMUNICATION_IND)rRegisterAppl.pcvCallBackCommunicationInd;

         s32RetVal = CSMAL_lRegisterAppl( rRegisterAppl.pvHandle,
                                          rRegisterAppl.u8Bus,
                                          rRegisterAppl.u32ProtocolType,
                                          (tPCVoid)&(rRegisterAppl.au8AddressField[0]),
                                          (tPCVoid)&(rCallBackFkt) );
         bRegistered = TRUE;
      }
      break;
#endif /* CSM_C_PTYPE_RN_OSEKI_NWM */
#ifdef CSM_C_PTYPE_J1939
      case CSM_C_PTYPE_J1939:
      {
  #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
    #if (CSM_S_J1939 == CSM_C_F_ON)
         tCSM_J1939_APPL_CALLBACK rCallBackFkt;

         rCallBackFkt.pfvCommunicationCon = (tCSM_PFN_TP_COMMUNICATION_CON)rRegisterAppl.pcvCallBackCommunicationCon;
         rCallBackFkt.pfvCommunicationInd = (tCSM_PFN_TP_COMMUNICATION_IND)rRegisterAppl.pcvCallBackCommunicationInd;
         rCallBackFkt.pfvDataCon          = (tCSM_PFN_TP_DATA_CON)rRegisterAppl.pcvCallBackDataCon;
         rCallBackFkt.pfvDataInd          = (tCSM_PFN_TP_DATA_IND)rRegisterAppl.pcvCallBackDataInd;
      #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
         rCallBackFkt.pfvDataIndFF        = (tCSM_PFN_TP_DATA_IND_FF)rRegisterAppl.pcvCallBackDataIndFF;
      #endif
      #if (CSM_S_ITP_ERROR_INDICATION_AVAILABLE == CSM_C_F_ON)
         rCallBackFkt.pfvDataErrorInd     = (tCSM_PFN_TP_DATA_ERROR_IND)rRegisterAppl.pcvCallBackDataErrorInd;
      #endif

         s32RetVal = CSMAL_lRegisterAppl( rRegisterAppl.pvHandle,
                                          rRegisterAppl.u8Bus,
                                          rRegisterAppl.u32ProtocolType,
                                          (tPCVoid)&(rRegisterAppl.au8AddressField[0]),
                                          (tPCVoid)&(rCallBackFkt) );
         bRegistered = TRUE;
    #endif
  #endif
      }
      break;
#endif /* CSM_C_PTYPE_J1939 */

      default:
      break;
   }

   if ( FALSE == bRegistered )
   {
      s32RetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_INVALID_PROTOCOL_TYPE);
   }

   vSendGeneralResult( (tCString)"REGISTER APPL",
                       CSM_PARENT_MESSAGE_QUEUE_ID_REGISTER_APPL,
                       rRegisterAppl.hProcessID,
                       rRegisterAppl.u32MsgQueueCnt,
                       s32RetVal );
}


tVoid vHandleGetCommunicationState( trCSMProcMessageQueueItemGetCommunicationState rGetCommunicationState )
{
   #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_USR1(("ProcCsm: message GET COMMUNICATION STATE received; ProcessID = %08x, MsgQueueCnt = %04d, bus = %d, protocol = 0x%08x, AddrField = %d %d, AppID = 0x%04x",
                       rGetCommunicationState.hProcessID,
                       rGetCommunicationState.u32MsgQueueCnt,
                       rGetCommunicationState.u8Bus,
                       rGetCommunicationState.u32ProtocolType,
                       rGetCommunicationState.au8AddressField[0],
                       rGetCommunicationState.au8AddressField[1],
                       rGetCommunicationState.u16ApplID));
   #endif

   OSAL_tMQueueHandle hMessageQueueID = hGetParentMessageQueueID( rGetCommunicationState.hProcessID, rGetCommunicationState.u32MsgQueueCnt );

   if ( OSAL_C_INVALID_HANDLE != hMessageQueueID )
   {
      tS32  s32Result;
      tU8    u8State = CSM_C_INACTIVE;
      trCSMParentMessageQueueItem  rMessage;

      s32Result = CSM_lGetCommunicationState( rGetCommunicationState.u8Bus,
                                              rGetCommunicationState.u32ProtocolType,
                                              (tPCVoid)&(rGetCommunicationState.au8AddressField[0]),
                                              &u8State,
                                              rGetCommunicationState.u16ApplID );

      rMessage.enID = CSM_PARENT_MESSAGE_QUEUE_ID_GET_COMMUNICATION_STATE;
      rMessage.uData.rGetCommunicationState.s32Result = s32Result;
      rMessage.uData.rGetCommunicationState.u8State = u8State;

      s32Result = OSAL_s32MessageQueuePost( hMessageQueueID, (tPCU8)&rMessage, sizeof(rMessage), OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST );
   }
   else
   {
      #if (CSM_S_TRACE == CSM_C_F_ON)
         ETG_TRACE_ERR(("ProcCsm->vHandleGetCommunicationState: could not open message queue"));
      #endif
   }
}


tVoid vHandleSignalRead( trCSMProcMessageQueueItemSignalRead rSignalRead )
{
   #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_USR1(("ProcCsm: message SIGNAL READ received; ProcessID = %08x, MsgQueueCnt = %04d, signal = 0x%08x, length = %d",
                       rSignalRead.hProcessID,
                       rSignalRead.u32MsgQueueCnt,
                       rSignalRead.u32SignalId,
                       rSignalRead.u8Length));
   #endif

   OSAL_tMQueueHandle hMessageQueueID = hGetParentMessageQueueID( rSignalRead.hProcessID, rSignalRead.u32MsgQueueCnt );

   if ( OSAL_C_INVALID_HANDLE != hMessageQueueID )
   {
      tS32  s32Result;
      trCSMParentMessageQueueItem  rMessage;

      if ( rSignalRead.u8Length <= CSM_C_ACCESS_LAYER_SIGNAL_DATA_MAXLENGTH )
      {
         s32Result = CSM_lSignalRead( rSignalRead.u32SignalId, (tPVoid)&(rMessage.uData.rSignalRead.au8Data[0]), rSignalRead.u8Length, (DWORD*)(&(rMessage.uData.rSignalRead.u32SignalStatus) ));
      }
      else
      {
         s32Result = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_DATA_LENGTH_EXCEEDED);
      }

      rMessage.enID = CSM_PARENT_MESSAGE_QUEUE_ID_SIGNAL_READ;
      rMessage.uData.rSignalRead.s32Result = s32Result;

      s32Result = OSAL_s32MessageQueuePost( hMessageQueueID, (tPCU8)&rMessage, sizeof(rMessage), OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST );
   }
   else
   {
      #if (CSM_S_TRACE == CSM_C_F_ON)
         ETG_TRACE_ERR(("ProcCsm->vHandleSignalRead: could not open message queue"));
      #endif
   }
}


tVoid vHandleMultipleSignalRead( trCSMProcMessageQueueItemMultipleSignalRead rMultipleSignalRead )
{
   tU8  u8NumberOfSignals = rMultipleSignalRead.u8NumberOfSignals;

   #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_USR1(("ProcCsm: message MULTIPLE SIGNAL READ received; ProcessID = %08x, MsgQueueCnt = %04d, number of signals = %d",
                       rMultipleSignalRead.hProcessID,
                       rMultipleSignalRead.u32MsgQueueCnt,
                       u8NumberOfSignals));
   #endif

   OSAL_tMQueueHandle hMessageQueueID = hGetParentMessageQueueID( rMultipleSignalRead.hProcessID, rMultipleSignalRead.u32MsgQueueCnt );

   if ( OSAL_C_INVALID_HANDLE != hMessageQueueID )
   {
      tS32  s32Result = CSM_C_NO_ERROR;
      trCSMParentMessageQueueItem  rMessage;

      if ( u8NumberOfSignals <= CSM_C_ACCESS_LAYER_MULTI_SIGNAL_MAXNUMBER )
      {
         tCSM_MultipleSignalRead *prSignalList = OSAL_NEW tCSM_MultipleSignalRead[u8NumberOfSignals];

         if ( NULL != prSignalList )
         {
            for ( tU8 u8SignalCnt = 0; u8SignalCnt < u8NumberOfSignals; u8SignalCnt++ )
            {
               #if (CSM_S_TRACE == CSM_C_F_ON)
                  ETG_TRACE_USR1(("ProcCsm: signal = 0x%08x, length = %d",
                                   rMultipleSignalRead.arSignalData[u8SignalCnt].u32SignalId,
                                   rMultipleSignalRead.arSignalData[u8SignalCnt].u8Length));
               #endif

               if ( rMultipleSignalRead.arSignalData[u8SignalCnt].u8Length <= CSM_C_ACCESS_LAYER_SIGNAL_DATA_MAXLENGTH )
               {
                  // fill the requested values to the output message
                  rMessage.uData.rMultipleSignalRead.arSignalData[u8SignalCnt].u32SignalId =
                     rMultipleSignalRead.arSignalData[u8SignalCnt].u32SignalId;
                  rMessage.uData.rMultipleSignalRead.arSignalData[u8SignalCnt].u8Length =
                     rMultipleSignalRead.arSignalData[u8SignalCnt].u8Length;

                  // handle over the requested values and the addresses of the output message parameters to the signal list
                  (prSignalList + u8SignalCnt)->dwSignalId        = rMultipleSignalRead.arSignalData[u8SignalCnt].u32SignalId;
                  (prSignalList + u8SignalCnt)->bDataBufferLength = rMultipleSignalRead.arSignalData[u8SignalCnt].u8Length;
                  (prSignalList + u8SignalCnt)->pvActSignalData   = (tPVoid)&(rMessage.uData.rMultipleSignalRead.arSignalData[u8SignalCnt].au8Data[0]);
                  (prSignalList + u8SignalCnt)->pdwSignalStatus   = (DWORD*)(&(rMessage.uData.rMultipleSignalRead.arSignalData[u8SignalCnt].u32SignalStatus));
               }
               else
               {
                  s32Result = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_DATA_LENGTH_EXCEEDED);
                  break;  // leave for-loop
               }
            }

            if ( CSM_C_NO_ERROR == s32Result )
            {
               // now read the data to the list, i.e. to the addresses of the output message parameters
               #if ((CSM_S_iMX_SIGNAL_ABSTRACTION == CSM_C_F_ON) || (CSM_S_v850_SIGNAL_ABSTRACTION == CSM_C_F_ON))
               s32Result = CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_SRVC_UNAVAILABLE );
               #else
               s32Result = CSM_lMultipleSignalRead( prSignalList, u8NumberOfSignals );
               #endif
            }

            OSAL_DELETE[] prSignalList;
         }
         else
         {
            s32Result = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_RESOURCE);
         }
      }
      else
      {
         s32Result = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_INVALID_PARA);
      }

      rMessage.enID = CSM_PARENT_MESSAGE_QUEUE_ID_MULTIPLE_SIGNAL_READ;
      rMessage.uData.rMultipleSignalRead.s32Result = s32Result;
      if ( CSM_C_NO_ERROR == s32Result )
      {
         rMessage.uData.rMultipleSignalRead.u8NumberOfSignals = u8NumberOfSignals;
      }
      else
      {
         rMessage.uData.rMultipleSignalRead.u8NumberOfSignals = 0;
      }

      s32Result = OSAL_s32MessageQueuePost( hMessageQueueID, (tPCU8)&rMessage, sizeof(rMessage), OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST );
   }
   else
   {
      #if (CSM_S_TRACE == CSM_C_F_ON)
         ETG_TRACE_ERR(("ProcCsm->vHandleMultipleSignalRead: could not open message queue"));
      #endif
   }
}


tVoid vHandleDataRead( trCSMProcMessageQueueItemDataRead rDataRead )
{
   #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_USR1(("ProcCsm: message DATA READ received; ProcessID = %08x, MsgQueueCnt = %04d, protocol = 0x%08x, AddrField = %d %d, length = %d",
                       rDataRead.hProcessID,
                       rDataRead.u32MsgQueueCnt,
                       rDataRead.u32ProtocolType,
                       rDataRead.au8AddressField[0],
                       rDataRead.au8AddressField[1],
                       rDataRead.u16DataLength));
   #endif

   OSAL_tMQueueHandle hMessageQueueID = hGetParentMessageQueueID( rDataRead.hProcessID, rDataRead.u32MsgQueueCnt );

   if ( OSAL_C_INVALID_HANDLE != hMessageQueueID )
   {
      tS32  s32Result;
      trCSMParentMessageQueueItem  rMessage;

      if ( rDataRead.u16DataLength <= CSM_C_ACCESS_LAYER_DATA_READ_MAXLENGTH )
      {
         s32Result = CSM_lDataRead( rDataRead.u32ProtocolType, (tPCVoid)&(rDataRead.au8AddressField[0]), &(rMessage.uData.rDataRead.au8Data[0]), rDataRead.u16DataLength );
      }
      else
      {
         s32Result = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_DATA_LENGTH_EXCEEDED);
      }

      rMessage.enID = CSM_PARENT_MESSAGE_QUEUE_ID_DATA_READ;
      rMessage.uData.rDataRead.s32Result = s32Result;

      s32Result = OSAL_s32MessageQueuePost( hMessageQueueID, (tPCU8)&rMessage, sizeof(rMessage), OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST );
   }
   else
   {
      #if (CSM_S_TRACE == CSM_C_F_ON)
         ETG_TRACE_ERR(("ProcCsm->vHandleDataRead: could not open message queue"));
      #endif
   }
}


tVoid vSendGeneralResult( tCString coszDebugString, tenCSMParentMessageQueueItemID enItemID, OSAL_tProcessID hProcessID, tU32 u32MsgQueueCnt, tS32 s32Result )
{
   #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_USR1(("vSendGeneralResult for %s", coszDebugString));
      ETG_TRACE_USR1(("ProcessID = %08x, MsgQueueCnt = %04d, result = 0x%08x",
                       hProcessID,
                       u32MsgQueueCnt,
                       s32Result));
   #endif

   OSAL_tMQueueHandle hMessageQueueID = hGetParentMessageQueueID( hProcessID, u32MsgQueueCnt );

   if ( OSAL_C_INVALID_HANDLE != hMessageQueueID )
   {
      trCSMParentMessageQueueItem  rMessage;

      rMessage.enID = enItemID;
      rMessage.uData.rGeneralResult.s32Result = s32Result;

      s32Result = OSAL_s32MessageQueuePost( hMessageQueueID, (tPCU8)&rMessage, sizeof(rMessage), OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST );
   }
   else
   {
      #if (CSM_S_TRACE == CSM_C_F_ON)
         ETG_TRACE_ERR(("vSendGeneralResult: could not open message queue"));
      #endif
   }
}


OSAL_tMQueueHandle hGetParentMessageQueueID( OSAL_tProcessID hProcessID, tU32 u32MsgQueueCnt )
{
   OSAL_tMQueueHandle hResult = OSAL_C_INVALID_HANDLE;
   tU32 u32Idx = 0;

   // search whether queue is already opened
   while ( ( u32Idx < vecParentMessageQueueAdmin.size() ) && ( OSAL_C_INVALID_HANDLE == hResult ) )
   {
      if (
            ( vecParentMessageQueueAdmin[u32Idx].hProcessID == hProcessID )
            &&
            ( vecParentMessageQueueAdmin[u32Idx].u32MsgQueueCnt == u32MsgQueueCnt )
         )
      {
         hResult = vecParentMessageQueueAdmin[u32Idx].hMessageQueueID;
      }
      u32Idx++;
   }

   // no matching open message queue found -> open it
   if ( OSAL_C_INVALID_HANDLE == hResult )
   {
      tChar acMessageQueueName[OSAL_C_U32_MAX_NAMELENGTH];
      tChar acNamePrefix[] = CSMAL_C_PARENT_MESSAGE_QUEUE_NAME_PREFIX;

      sprintf( acMessageQueueName, "%s%08x_%04d", acNamePrefix, hProcessID, u32MsgQueueCnt );

      if ( OSAL_OK == OSAL_s32MessageQueueOpen( acMessageQueueName, OSAL_EN_READWRITE, &hResult ) )
      {
         trCSMParentMessageQueueAdmin rNewMessageQueueAdmin;

         rNewMessageQueueAdmin.hProcessID      = hProcessID;
         rNewMessageQueueAdmin.u32MsgQueueCnt  = u32MsgQueueCnt;
         rNewMessageQueueAdmin.hMessageQueueID = hResult;

         vecParentMessageQueueAdmin.push_back( rNewMessageQueueAdmin );

         #if (CSM_S_TRACE == CSM_C_F_ON)
         ETG_TRACE_USR1(("ProcCsm new parent message queue opened: ID = 0x%08x, Name = %s",
                          hResult, acMessageQueueName));
         #endif
      }
   }

   return hResult;
}


tVoid vCloseAllParentMessageQueues( tVoid )
{
   for ( tU32 u32Idx = 0; u32Idx < vecParentMessageQueueAdmin.size(); ++u32Idx )
   {
      if ( OSAL_C_INVALID_HANDLE != vecParentMessageQueueAdmin[u32Idx].hMessageQueueID )
      {
         OSAL_s32MessageQueueClose( vecParentMessageQueueAdmin[u32Idx].hMessageQueueID );
      }
   }

   vecParentMessageQueueAdmin.clear();
}

//EOF

