/* Description and history of csmal_krnl_os.c at the end of the file.         */
#define CSMAL_KRNL_OS_SELF /**< File name for preprozessor commands */
/** csmal_krnl_os.c is a file according V-team CAN-SW architecture implementation rules */
#define CSM_CAN_STACK


#ifndef VARIANT_S_FTR_ENABLE_OSAL_CSM_SOCKET_SIMU
/* TKSE */
#include <extension/outer.h>
#endif

// #define CSM_NO_TYPE_REDEFINITION    // not used anywhere !!
#include "csm_stack_i.h"
#include "csm_stack_m.h"

#include "csmal_krnl_trace.h"
#include "csmal_krnl_os.h"
#include "csm_access_m.h"

#if (CSM_AL_S_USE_OSAL_MESSAGEBUFFER == CSM_C_F_ON)
  #define OSAL_S_IMPORT_INTERFACE_GENERIC
  #include "osal_if.h"
#endif  /* (CSM_AL_S_USE_OSAL_MESSAGEBUFFER == CSM_C_F_ON) */


/*******************************************************************************
 * function    CSMAL_lDeleteUserResource
 * \doxydocu
 * \brief      Releases the wait state of the specified user task and its corres-
 *             ponding message buffer.
 *
 * \param[in]  lTaskId
 *              ID of the task, which has to be deleted.
 *
 * \param[in]  lMbfId
 *              ID of the message buffer, on which the above mentioned task is
 *              waiting for.
 *
 * \return     CSM error code CSM_E_OS_ERROR or CSM_C_NO_ERROR
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview 09.09.08 CM-AI/VTeam-Brunotte
 * \history_begin
 * 19.12.2008  VTeam-Khler
 * - (tPCU8) cast in OSAL_s32MessageQueuePost call added.
 * 20.11.2008  VTeam-Brunotte
 * - osal message buffer is used
 * 20.11.2008  VTeam-Khler
 * - Task will be deleted by sending CSM_ACCESS_TerminateTask message.
 * 18.06.2008  VTeam-Brunotte
 * - Initial Revision.
 * \history_end
 ******************************************************************************/
LONG CSMAL_lDeleteUserResource(LONG lTaskId, LONG lMbfId)
{
  LONG lRet;
  LONG ercd;
  trMsgHeader rMsgHeader;
  rMsgHeader.enMsgType = CSM_ACCESS_TerminateTask;
  CSM_M_PARAMETER_INTENTIONALLY_UNUSED(lTaskId);
  #if (CSM_AL_S_USE_OSAL_MESSAGEBUFFER == CSM_C_F_ON)
    ercd = OSAL_s32MessageQueuePost((OSAL_tMQueueHandle) lMbfId,
                                         (tPCU8)&rMsgHeader,
                                         (tU32) sizeof(rMsgHeader),
                                         OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST);

  #else /* (CSM_AL_S_USE_OSAL_MESSAGEBUFFER == CSM_C_F_ON) */
    ercd = tkse_snd_mbf((ID)lMbfId, &rMsgHeader, (INT)sizeof(rMsgHeader), (INT)0);
  #endif /* (CSM_AL_S_USE_OSAL_MESSAGEBUFFER == CSM_C_F_ON) */
  if (ercd < E_OK)
  {
    lRet = CSM_M_MAKE_ERROR( CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_OS_ERROR);
  }
  else
  {
    lRet = CSM_C_NO_ERROR;
  }

  return(lRet);
}


/*******************************************************************************
 * function    CSMAL_lDeleteLock
 * \doxydocu
 * \brief      Deletes the lock designated in lLockID
 *
 * \param[in]  lLockID
 *              ID of the lock, which has to be deleted.
 *
 * \return     t-Kernel error code
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview 09.09.08 CM-AI/VTeam-Brunotte
 * \history_begin
 * 18.06.2008  VTeam-Brunotte
 * - Initial Revision.
 * \history_end
 ******************************************************************************/
LONG CSMAL_lDeleteLock(LONG lLockID)
{
  (tVoid) lLockID; // LINT happy
  #ifndef VARIANT_S_FTR_ENABLE_OSAL_CSM_SOCKET_SIMU
  return (LONG)tk_del_sem((ID) lLockID);
  #else
  return 0L;
  #endif
}

/*******************************************************************************
 * function    CSMAL_lCreateLock
 * \doxydocu
 * \brief      Creates a lock with the designated name. As lock a semaphore is
 *             used.
 *
 * \param[in]  pbSemName
 *              Pointer to the lock name
 *
 * \return     t-Kernel error code
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview 09.09.08 CM-AI/VTeam-Brunotte
 * \history_begin
 * 27.08.2008  VTeam-Khler
 * - type changed form B* to const CHAR*.
 * 18.06.2008  VTeam-Brunotte
 * - Initial Revision.
 * \history_end
 ******************************************************************************/
LONG CSMAL_lCreateLock(const CHAR* pbSemName)
{
  #ifndef VARIANT_S_FTR_ENABLE_OSAL_CSM_SOCKET_SIMU
  {
    T_CSEM  rSem;

    rSem.exinf    = 0;
    rSem.isemcnt  = 1;
    rSem.maxsem   = 1;
    rSem.sematr   = TA_TFIFO | TA_DSNAME;
    if (pbSemName != NULL)
    {
      CSM_vMemCopy(&rSem.dsname[0],pbSemName,7);
    }
    else
    {
      CSM_vMemCopy(&rSem.dsname[0],"CSMALSE",7);
    }
    rSem.dsname[7] = '\0';
    return tk_cre_sem(&rSem);
  }
  #else // VARIANT_S_FTR_ENABLE_OSAL_CSM_SOCKET_SIMU
  {
    OSAL_tSemHandle hSem = OSAL_C_INVALID_HANDLE;            // semaphore handle to synchronize data access
    tChar sTempSemName[] = "CSMALSE";
    if (pbSemName != NULL)
    {
      CSM_vMemCopy(&sTempSemName[0],pbSemName,7);
    }
    if (OSAL_ERROR == OSAL_s32SemaphoreCreate(sTempSemName, &hSem, 1))
    {
      hSem = OSAL_C_INVALID_HANDLE;
    }
    return (LONG) hSem;
  }
  #endif
}

/*******************************************************************************
 * function    CSMAL_lLock
 * \doxydocu
 * \brief      Waits for unlock of the designated lLockID. This function blocks
 *             the calling task without timeout.
 *
 * \param[in]  lLockID
 *              ID of the lock, on which a calling task has to be waiting for.
 *
 * \return     t-Kernel error code
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview 09.09.08 CM-AI/VTeam-Brunotte
 * \history_begin
 * 18.06.2008  VTeam-Brunotte
 * - Initial Revision.
 * \history_end
 ******************************************************************************/
LONG CSMAL_lLock(LONG lLockID)
{
  #ifndef VARIANT_S_FTR_ENABLE_OSAL_CSM_SOCKET_SIMU
  return tk_wai_sem((ID)lLockID, 1, TMO_FEVR);
  #else
  return OSAL_s32SemaphoreWait((OSAL_tSemHandle) lLockID, OSAL_C_TIMEOUT_FOREVER);
  #endif
}

/*******************************************************************************
 * function    CSMAL_lUnLock
 * \doxydocu
 * \brief      Sends a signal to a lock designated with lLockID.
 *
 * \param[in]  lLockID
 *              ID of the lock
 *
 * \return     t-Kernel error code
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview 09.09.08 CM-AI/VTeam-Brunotte
 * \history_begin
 * 18.06.2008  VTeam-Brunotte
 * - Initial Revision.
 * \history_end
 ******************************************************************************/
LONG CSMAL_lUnLock(LONG lLockID)
{
  #ifndef VARIANT_S_FTR_ENABLE_OSAL_CSM_SOCKET_SIMU
  return tk_sig_sem((ID)lLockID, 1);
  #else
  OSAL_s32SemaphorePost((OSAL_tSemHandle)lLockID);
  return 0L;
  #endif
}

#undef CSMAL_KRNL_OS_SELF
#undef CSM_CAN_STACK
/*******************************************************************************
 * \doxydocu
 * \file          csmal_krnl_os.c
 * \brief         Operating system dependant functions of CSM Access Layer
 *                (Kernel part)
 *
 * \project       Nissan LCN2 kai
 * \path          /di_can/adapt/osal/csm_access_krnl/include/
 *
 * \ingroup       CSM_ACCESS_KRNL
 *
 * \authors       VTeam
 *
 * COPYRIGHT      (c) 2011 Bosch GmbH
 *
 * \history_begin
 *
 * 17.08.11  /main/1 Kollai
 * - Initial revision.
 *
 * 16.05.12  /main/2 Kollai Kempen
 * - Function CSMAL_vSendMessage removed
 *
 * 02.01.13  /main/3 Kempen Borck
 * - Semaphore access fixed
 *
 * 01.03.17  /main/4 Borck
 * - LINT issues fixed
 *
 * \history_end
 *//***********************************************************************/
