// Description and history of csm_access_krnl.c at the end of the file.
// File name for preprozessor commands
#define CSM_ACCESS_KRNL_SELF
// csm_access_krnl.c is a file according V-team CAN-SW architecture implementation rules
#define CSM_CAN_STACK

// #define CSM_NO_TYPE_REDEFINITION   // not used anywhere !!
#include "csm_stack_i.h"
#include "csm_i.h"
#include "csm_stack_m.h"
#include "csm_access_krnl.h"
#include "csm_access_m.h"

#if (CSM_AL_S_USE_OSAL_MESSAGEBUFFER == CSM_C_F_ON)
  #define OSAL_S_IMPORT_INTERFACE_GENERIC
  #include "osal_if.h"
#endif

#include "csmal_krnl_os.h"

// Trace class definition for ETG (extented trace generator).
#define TR_CLASS_CSMKRNL (TR_COMP_CSM + 0x11)

#if (CSM_S_TRACE == CSM_C_F_ON)
  #if (CSM_S_TRACE_ETG == CSM_C_F_ON)
  #define ETRACE_S_IMPORT_INTERFACE_GENERIC
  #include "etrace_if.h"

  #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
    #include "csm_etg_trace.h"
    #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CSMKRNL
    //#define ETG_INLINE
    #include "trcGenProj/Header/csm_access_krnl.c.trc.h"
  #endif
  #endif
#endif // CSM_S_TRACE

//******************************************************************************
//* defines and macros (scope: file local)
//******************************************************************************
// none

//******************************************************************************
//* typedefs (scope: file local)
//******************************************************************************
/** type for the signal notification table */
typedef struct
{
  LONG             lIndex;   /**< index of next free position in table */
  trSignalIndMsg   arSignalIndNotifications[CSMAL_MAX_SIGNALIND_NOTIFICATIONS];  /**< Access layer kernel table with fix and actual data to
                                                                                  * call signal indication callbacks */
} trSignalIndNotificationTable;

/** type for the communication confirmation notification table */
typedef struct
{
  LONG                  lIndex;  /**< index of next free position in table */
  trCommunicationConMsg arCommunicationConNotifications[CSMAL_MAX_COMMUNICATIONCON_NOTIFICATIONS];
} trCommunicationConNotificationTable;

/** type for the communication indication notification table */
typedef struct
{
  LONG                  lIndex;  /**< index of next free position in table */
  trCommunicationIndMsg arCommunicationIndNotifications[CSMAL_MAX_COMMUNICATIONIND_NOTIFICATIONS];
} trCommunicationIndNotificationTable;


#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
  /** type for the data confirmation notification table */
  typedef struct
  {
    LONG         lIndex;  /**< index of next free position in table */
    trDataConMsg arDataConNotifications[CSMAL_MAX_DATACON_NOTIFICATIONS];
  } trDataConNotificationTable;

  /** type for the data indication notification table */
  typedef struct
  {
    LONG         lIndex;  /**< index of next free position in table */
    trDataIndMsg arDataIndNotifications[CSMAL_MAX_DATAIND_NOTIFICATIONS];
  } trDataIndNotificationTable;

  #if (CSM_S_ISO_TP == CSM_C_F_ON)
    #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
      /** type for the first frame data indication notification table */
      typedef struct
      {
        LONG         lIndex;  /**< index of next free position in table */
        trDataIndFFMsg arDataIndFFNotifications[CSMAL_MAX_DATAINDFF_NOTIFICATIONS];
      } trDataIndFFNotificationTable;
    #endif // CSM_S_ITP_FF_INDICATION_AVAILABLE
  #endif // CSM_S_ISO_TP

  #if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
    /** type for the data error indication notification table */
    typedef struct
    {
      LONG         lIndex;  /**< index of next free position in table */
      trDataErrorIndMsg arDataErrorIndNotifications[CSMAL_MAX_DATAERRORIND_NOTIFICATIONS];
    } trDataErrorIndNotificationTable;
  #endif // CSM_S_DATA_ERROR_IND_AVAILABLE
#endif // CSM_S_CNP_AVAILABLE

typedef struct
{
  void * pvHandle;
  LONG  lIndexCommunicationCon;
  LONG  lIndexCommunicationInd;
  #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
  LONG  lIndexDataCon;
  LONG  lIndexDataInd;
  #if (CSM_S_MPDT == CSM_C_F_ON)
  LONG  lIndexDDataCon;
  LONG  lIndexDDataInd;
  #endif // CSM_S_MPDT
  #if (CSM_S_ISO_TP == CSM_C_F_ON)
  #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
  LONG  lIndexDataIndFF;
  #endif
  #endif // CSM_S_ISO_TP
  #if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
  LONG  lIndexDataErrorInd;
  #endif
  #endif // CSM_S_CNP_AVAILABLE
} trApplNotificationLookupEntry;

typedef trApplNotificationLookupEntry * pApplNotificationLookupEntry;

typedef struct
{
  LONG lIndex;
  trApplNotificationLookupEntry arEntries[CSMAL_MAX_APPL_NOTIFICATIONS];
} trApplNotificationLookupTable;

/** type to create a CSM access handle table */
typedef struct
{
  LONG        lIndex;                                  /**< the next free table entry */
  trCSMHandle arCSMAccessHandles[CSMAL_NUM_HANDLES];   /**< the table array itself */
} trCSMAccessHandleTable;

#if (CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON)
/** type for the bus error indication notification table */
typedef struct
{
  LONG             lIndex;                                                           /**< index of next free position in table */
  trBusErrorIndMsg arBusErrorIndNotifications[CSMAL_MAX_BUSERRORIND_NOTIFICATIONS];  /**< the table array itself */
} trBusErrorIndNotificationTable;
#endif // CSM_S_ENABLE_INDICATE_BUSERROR

//******************************************************************************
//* variable declaration (scope: file local)
//******************************************************************************
// none

//******************************************************************************
//* variable definition (scope: file local)
//******************************************************************************
/** the signal indication notification table */
static trSignalIndNotificationTable        rSignalIndNotificationTable;

/** the communication confirmation notification table */
static trCommunicationConNotificationTable rCommunicationConNotificationTable;

/** the communication indication notification table */
static trCommunicationIndNotificationTable rCommunicationIndNotificationTable;

#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
  /** the data confirmation notification table */
  static trDataConNotificationTable          rDataConNotificationTable;

  /** the data indication notification table */
  static trDataIndNotificationTable          rDataIndNotificationTable;

  #if (CSM_S_ISO_TP == CSM_C_F_ON)
    #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
      /** the first frame data indication notification table */
      static trDataIndFFNotificationTable      rDataIndFFNotificationTable;
    #endif
  #endif // CSM_S_ISO_TP

  #if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
    /** the data error indication notification table */
    static trDataErrorIndNotificationTable          rDataErrorIndNotificationTable;
  #endif
#endif // CSM_S_CNP_AVAILABLE

static trApplNotificationLookupTable       rApplNotificationLookupTable;

// this is a table to store the csm access layer handles
static trCSMAccessHandleTable              rCSMAccessHandleTable;

static LONG ghSignalNotifyTableLock; /**< semaphore protecting the signal notification table. */
static LONG ghApplNotifyTableLock;   /**< semaphore protecting the communication/data con/ind
                                      *   notification table. */
static LONG ghAccessHandleTableLock; /**< semaphore protecting the access handle table. */

#if (CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON)
  static trBusErrorIndNotificationTable    rBusErrorIndNotificationTable;
#endif

//******************************************************************************
//* function prototypes (scope: file local)
//******************************************************************************
static pSignalIndMsg CSMAL_lAddSignalIndNotification(void * pvAccessUsrHandle, DWORD dwSignalId, tCSM_PFN_CBR_SIGNAL_IND vPFNCBRSignalInd);
static LONG CSMAL_lAddCommunicationIndNotification(void * pvHandle,
                                                   BYTE bBus,
                                                   DWORD dwProtocolType,
                                                   const void * pvAddressField,
                                                   tCSM_PFN_TP_COMMUNICATION_IND vPFNTPCommunicationInd);
static LONG CSMAL_lAddCommunicationConNotification(void * pvHandle,
                                                   BYTE bBus,
                                                   DWORD dwProtocolType,
                                                   const void * pvAddressField,
                                                   tCSM_PFN_TP_COMMUNICATION_CON vPFNTPCommunicationCon);
#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
  static LONG CSMAL_lAddDataConNotification(void * pvHandle,
                                            BYTE bBus,
                                            DWORD dwProtocolType,
                                            const void * pvAddressField,
                                            tCSM_PFN_TP_DATA_CON pPFNTPDataCon);
  static LONG CSMAL_lAddDataIndNotification(void * pvHandle,
                                            BYTE bBus,
                                            DWORD dwProtocolType,
                                            const void * pvAddressField,
                                            tCSM_PFN_TP_DATA_IND pPFNTPDataInd);
  #if (CSM_S_ISO_TP == CSM_C_F_ON)
    #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
      static LONG CSMAL_lAddDataIndFFNotification(void * pvHandle,
                                                  BYTE bBus,
                                                  DWORD dwProtocolType,
                                                  const void * pvAddressField,
                                                  tCSM_PFN_TP_DATA_IND_FF pPFNTPDataIndFF);
    #endif
  #endif // CSM_S_ISO_TP

  #if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
    static LONG CSMAL_lAddDataErrorIndNotification(void * pvHandle,
                                                   BYTE bBus,
                                                   DWORD dwProtocolType,
                                                   const void * pvAddressField,
                                                   tCSM_PFN_TP_DATA_ERROR_IND pPFNTPDataErrorInd);
  #endif
#endif // CSM_S_CNP_AVAILABLE

#if (CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON)
  static pBusErrorIndMsg CSMAL_lAddBusErrorIndNotification(void * pvHandle,
                                                           tCSM_PFN_BUS_ERROR_IND pPFNBusErrorInd);
#endif

static pApplNotificationLookupEntry CSMAL_lAddApplNotification(void * pvHandle, BYTE bBus, DWORD dwProtocolType, const void * pvAddressField,
                                                               tCSM_PFN_TP_COMMUNICATION_CON pPFNTPCommunicationCon,
                                                               tCSM_PFN_TP_COMMUNICATION_IND pPFNTPCommunicationInd
                                                               #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
                                                               ,tCSM_PFN_TP_DATA_CON pPFNTPDataCon
                                                               ,tCSM_PFN_TP_DATA_IND pPFNTPDataInd
                                                               #if (CSM_S_MPDT == CSM_C_F_ON )
                                                               ,tCSM_PFN_TP_DATA_CON pPFNTPDDataCon
                                                               ,tCSM_PFN_TP_DATA_IND pPFNTPDDataInd
                                                               #endif
                                                               #if (CSM_S_ISO_TP == CSM_C_F_ON)
                                                               #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
                                                               ,tCSM_PFN_TP_DATA_IND_FF pDataIndFF
                                                               #endif
                                                               #endif // CSM_S_ISO_TP
                                                               #if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
                                                               ,tCSM_PFN_TP_DATA_ERROR_IND pPFNTPDataErrorInd
                                                               #endif
                                                               #endif // CSM_S_CNP_AVAILABLE
                                                               );

// Prototypes for local CSMAL Registration Functions
#ifdef CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL
static LONG lRegisterCSM_LocalCtrl_ApplCallback(void * pvHandle, BYTE bBus, DWORD dwProtocolType,
                                                const void * pvAddressField, const void * pvCallBackFkt);
#endif

#ifdef CSM_C_PTYPE_GM_NWM
static LONG lRegisterCSM_GMLAN_NWM_ApplCallback(void * pvHandle, BYTE bBus, DWORD dwProtocolType,
                                                const void * pvAddressField, const void * pvCallBackFkt);
#endif

#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
#if (CSM_S_ISO_TP == CSM_C_F_ON)
//#ifdef CSM_C_PTYPE_USED_STD_USDT_ISO
static LONG lRegisterCSM_ITP_ApplCallback(void * pvHandle, BYTE bBus, DWORD dwProtocolType,
                                          const void * pvAddressField, const void * pvCallBackFkt);
//#endif
#endif // CSM_S_ISO_TP

#if (CSM_S_UUDT_TX == CSM_C_F_ON)
//#ifdef CSM_C_PTYPE_USED_STD_UUDT_TX
static LONG lRegisterCSM_UUDT_TX_ApplCallback(void * pvHandle, BYTE bBus, DWORD dwProtocolType,
                                              const void * pvAddressField, const void * pvCallBackFkt);
//#endif
#endif // CSM_S_UUDT_TX

#if (CSM_S_UUDT_RX == CSM_C_F_ON)
//#ifdef CSM_C_PTYPE_USED_STD_UUDT_RX
static LONG lRegisterCSM_UUDT_RX_ApplCallback(void * pvHandle, BYTE bBus, DWORD dwProtocolType,
                                              const void * pvAddressField, const void * pvCallBackFkt);
//#endif
#endif // CSM_S_UUDT_RX
#endif // CSM_S_CNP_AVAILABLE

#ifdef CSM_C_PTYPE_VW_BAP
static LONG lRegisterCSM_BAP_lApplCallbackInit(void * pvHandle, BYTE bBus, DWORD dwProtocolType,
                                               const void * pvAddressField, const void * pvCallBackFkt);
#endif // CSM_C_PTYPE_VW_BAP

#ifdef CSM_C_PTYPE_PAG_OSEK_NWM
static LONG lRegisterCSM_OSEK_lApplCallbackInit(void * pvHandle, BYTE bBus, DWORD dwProtocolType,
                                                const void * pvAddressField, const void * pvCallBackFkt);
#endif // CSM_C_PTYPE_PAG_OSEK_NWM

#ifdef CSM_C_PTYPE_USED_STD_BR_SIGNAL
static LONG lRegisterCSM_CBR_ApplCallback(void * pvHandle, BYTE bBus, DWORD dwProtocolType,
                                          const void * pvAddressField, const void * pvCallBackFkt);
#endif // CSM_C_PTYPE_USED_STD_BR_SIGNAL

#ifdef CSM_C_PTYPE_RN_MPDT_C
static LONG lRegisterCSM_MPDT_C_ApplCallback(void * pvHandle, BYTE bBus, DWORD dwProtocolType,
                                             const void * pvAddressField, const void * pvCallBackFkt);
#endif // CSM_C_PTYPE_RN_MPDT_C

#ifdef CSM_C_PTYPE_J1939
static LONG lRegisterCSM_J1939_ApplCallback(void * pvHandle,
                                              BYTE bBus,
                                              DWORD dwProtocolType,
                                              const void * pvAddressField,
                                              const void * pvCallBackFkt);
#endif


#ifdef CSM_C_PTYPE_RN_OSEKI_NWM
static LONG lRegisterCSM_OSEKI_lApplCallbackInit(void * pvHandle, BYTE bBus, DWORD dwProtocolType,
                                                 const void * pvAddressField, const void * pvCallBackFkt);
#endif // CSM_C_PTYPE_RN_OSEKI_NWM

// other local helper prototypes
VOID CSMAL_vSendMessage(void *pvHandle, void const *pvMsg, LONG lMessageLength);

//******************************************************************************
//* function prototypes (scope: external access by functionpointer)
//******************************************************************************
VOID CSMAL_vSignalInd_Handler(void * pvMsg, DWORD dwSignalId, DWORD dwSignalStatus);
#if (CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON)
  VOID CSMAL_vBusErrorInd_Handler(void *pvEntry, BYTE bBus, WORD wBusError);
#endif

VOID CSMAL_vCommunicationCon_Handler(void * pvEntry, BYTE bBus, DWORD dwProtocolType, const VOID* pvAddressField, BYTE bConnectState, WORD wAppId);
VOID CSMAL_vCommunicationInd_Handler(void * pvEntry, BYTE bBus, DWORD dwProtocolType, const VOID* pvAddressField, BYTE bConnectState, WORD wAppId);

#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
  VOID CSMAL_vDataCon_Handler(void * pvEntry, DWORD dwProtocolType, const VOID* pvAddressField, BYTE bTransferResult);
  VOID CSMAL_vDataInd_Handler(void * pvEntry, DWORD dwProtocolType, const VOID* pvAddressField, BYTE * pbData, WORD wDataLength);
  #if (CSM_S_ISO_TP == CSM_C_F_ON)
    #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
      VOID CSMAL_vDataIndFF_Handler(void * pvEntry, DWORD dwProtocolType, const VOID* pvAddressField, WORD wDataLength);
    #endif
  #endif
  #if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
    VOID CSMAL_vDataErrorInd_Handler(void * pvEntry, DWORD dwProtocolType, const VOID* pvAddressField, DWORD dwDataErrorCode);
  #endif
#endif // CSM_S_CNP_AVAILABLE

// Feature switch is used only to control doxygen documentation
#if (CSM_S_DOXYGEN_HIDE_NON_API_FCT == CSM_C_F_OFF)
//******************************************************************************
//* function implementation (scope: file local = static)
//******************************************************************************

/*******************************************************************************
 * function    CSMAL_lAddSignalIndNotification
 * \doxydocu
 * \brief      Adds the data for a signal indication notification callback
 *             to the corresponding notification table.
 *
 * \param[in]  pvAccessUsrHandle
 *             CSM Access Layer Handle
 *
 * \param[in]  dwSignalId
 *             ID of the signal that shall be notified
 *
 * \param[in]  vPFNCBRSignalInd
 *             address of the callback function that shall be called for notify
 *
 * \return     Pointer to the entry in the notification table
 *
 * \access     user mode application
 * \reentrant  yes, saved with semaphore
 *
 * \lastreview 04.09.08 VTeam-Khler
 * \history_begin
 * 04.09.08 VTeam-Khler
 * - Unlock before error trace.
 * 10.10.06 3SOFT Schubart
 * Initial Revision.
 * \history_end
 ******************************************************************************/
static pSignalIndMsg CSMAL_lAddSignalIndNotification(void * pvAccessUsrHandle, DWORD dwSignalId, tCSM_PFN_CBR_SIGNAL_IND vPFNCBRSignalInd)
{
  pSignalIndMsg phCSM = NULL;  // pointer to element in signal indication notification table

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR2(("CSMAL_lAddSignalIndNotification - AccessUsrHandle:0x%X  SignalId:0x%X  PFNCBRSignalInd:0x%X",
                  pvAccessUsrHandle,
                  dwSignalId,
                  vPFNCBRSignalInd));
  #endif

  // lock table for inserting
  CSMAL_lLock(ghSignalNotifyTableLock);

  // get message placeholder if current index is in a valid range
  if ((rSignalIndNotificationTable.lIndex >= 0) && (rSignalIndNotificationTable.lIndex < CSMAL_MAX_SIGNALIND_NOTIFICATIONS))
  {
    phCSM = &rSignalIndNotificationTable.arSignalIndNotifications[rSignalIndNotificationTable.lIndex];

    phCSM->rMsgHeader.enMsgType          = CSM_ACCESS_SignalInd;
    phCSM->rMsgHeader.pvAccessUsrHandle  = pvAccessUsrHandle;
    phCSM->dwSignalId                    = dwSignalId;
    phCSM->vPFNCBRSignalInd              = vPFNCBRSignalInd;

    // increase current access index
    rSignalIndNotificationTable.lIndex++;
  }

  // unlock table (after insertion)
  CSMAL_lUnLock(ghSignalNotifyTableLock);

  // Error trace if insertion failed
  if (NULL == phCSM)
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_ERR(("CSMAL_lAddSignalIndNotification - ERROR - Table Index out of bounds:0x%X",
                  rSignalIndNotificationTable.lIndex));
    #endif
  }

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR4(("CSMAL_lAddSignalIndNotification - phCSM:0x%X", phCSM));
  #endif

  return phCSM;
}

/*******************************************************************************
 * function    CSMAL_lAddCommunicationConNotification
 * \doxydocu
 * \brief      Adds the data for a communication confirmation notification
 *             callback to the corresponding notification table.
 *
 * \param[in]  pvHandle
 *             CSM Access Layer Handle
 *
 * \param[in]  bBus
 *             CAN bus number
 *
 * \param[in]  dwProtocolType
 *             protocol type
 *
 * \param[in]  pvAddressField
 *             addressing, depends on protocol
 *
 * \param[in]  vPFNTPCommunicationCon
 *             Callback function
 *
 * \return     index in the corresponding notification table
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview 09.09.08 CM-AI/VTeam-Brunotte
 * \history_begin
 * 17.07.07  VTeam-Khler
 * - Storage of pvAddressField commented out.
 * 11.10.06 3SOFT Schubart
 * Initial Revision.
 * \history_end
 ******************************************************************************/
static LONG CSMAL_lAddCommunicationConNotification(void * pvHandle, BYTE bBus, DWORD dwProtocolType, const void * pvAddressField,
                                                   tCSM_PFN_TP_COMMUNICATION_CON vPFNTPCommunicationCon)
{
  pCommunicationConMsg phCSM = NULL;
  LONG lRetVal = -1 ;
  BYTE bAddr0 = *((((BYTE *)pvAddressField)+0));
  BYTE bAddr1 = *((((BYTE *)pvAddressField)+1));

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR2(("CSMAL_lAddCommunicationConNotification - Handle:0x%X  Bus:%x  Prot:%x  AddrField:0x%02X%02X  PFNTPCommunicationCon:0x%X",
                  pvHandle,
                  ETG_CENUM(tCSM_BUS_NUMBER,               (tU8)  bBus),
                  ETG_CENUM(tCSM_PROTOCOL_TYPE,            (tU32) dwProtocolType),
                  (tU8)  bAddr0,
                  (tU8)  bAddr1,
                  vPFNTPCommunicationCon));
  #endif

  // get message placeholder if current access index is in a valid range
  if ((rCommunicationConNotificationTable.lIndex >= 0) && (rCommunicationConNotificationTable.lIndex < CSMAL_MAX_COMMUNICATIONCON_NOTIFICATIONS))
  {
    INT iAddrFieldIndex;
    phCSM = &rCommunicationConNotificationTable.arCommunicationConNotifications[rCommunicationConNotificationTable.lIndex];

    phCSM->rMsgHeader.enMsgType   = CSM_ACCESS_CommunicationCon;
    phCSM->rMsgHeader.pvAccessUsrHandle    = pvHandle;
    phCSM->bBus                   = bBus;
    phCSM->dwProtocolType         = dwProtocolType;
    for (iAddrFieldIndex = 0; iAddrFieldIndex < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; iAddrFieldIndex++)
    {
      phCSM->abAddressField[iAddrFieldIndex] = *((((BYTE *)pvAddressField)+iAddrFieldIndex));
    }
    phCSM->vPFNTPCommunicationCon = vPFNTPCommunicationCon;

    lRetVal = rCommunicationConNotificationTable.lIndex++;
  }
  else
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_ERR(("CSMAL_lAddCommunicationConNotification - ERROR - Table Index out of bounds: 0x%X",
                   rCommunicationConNotificationTable.lIndex));
    #endif
  }

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR4(("CSMAL_lAddCommunicationConNotification - RetVal:0x%X", lRetVal));
  #endif

  return lRetVal;
}

/*******************************************************************************
 * function    CSMAL_lAddCommunicationIndNotification
 * \doxydocu
 * \brief      Adds the data for a communication indication notification
 *             callback to the corresponding notification table.
 *
 * \param[in]  pvHandle
 *             CSM Access Layer Handle
 *
 * \param[in]  bBus
 *             CAN bus number
 *
 * \param[in]  dwProtocolType
 *             protocol type
 *
 * \param[in]  pvAddressField
 *             addressing, depends on protocol
 *
 * \param[in]  vPFNTPCommunicationInd
 *             Callback function
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview  09.09.08 CM-AI/VTeam-Brunotte
 * \return     index in the corresponding notification table
 *
 * \history_begin
 * 17.07.07  VTeam-Khler
 * - Storage of pvAddressField commented out.
 * 11.10.06 3SOFT Schubart
 * Initial Revision.
 * \history_end
 ******************************************************************************/
static LONG CSMAL_lAddCommunicationIndNotification(void * pvHandle, BYTE bBus, DWORD dwProtocolType, const void * pvAddressField,
                                                   tCSM_PFN_TP_COMMUNICATION_IND vPFNTPCommunicationInd)
{
  pCommunicationIndMsg phCSM = NULL;
  LONG lRetVal = -1;
  BYTE bAddr0 = *((((BYTE *)pvAddressField)+0));
  BYTE bAddr1 = *((((BYTE *)pvAddressField)+1));

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR2(("CSMAL_lAddCommunicationIndNotification - Handle:0x%X  Bus:%x  Prot:%x  AddrField:0x%02X%02X  PFNTPCommunicationInd:0x%X",
                  pvHandle,
                  ETG_CENUM(tCSM_BUS_NUMBER,               (tU8)  bBus),
                  ETG_CENUM(tCSM_PROTOCOL_TYPE,            (tU32) dwProtocolType),
                  (tU8)  bAddr0,
                  (tU8)  bAddr1,
                  vPFNTPCommunicationInd));
  #endif

  // get message placeholder if current access index is in a valid range
  if ((rCommunicationIndNotificationTable.lIndex >= 0) && (rCommunicationIndNotificationTable.lIndex < CSMAL_MAX_COMMUNICATIONIND_NOTIFICATIONS))
  {
    INT iAddrFieldIndex;
    phCSM = &rCommunicationIndNotificationTable.arCommunicationIndNotifications[rCommunicationIndNotificationTable.lIndex];

    phCSM->rMsgHeader.enMsgType   = CSM_ACCESS_CommunicationInd;
    phCSM->rMsgHeader.pvAccessUsrHandle    = pvHandle;
    phCSM->bBus                   = bBus;
    phCSM->dwProtocolType         = dwProtocolType;
    for (iAddrFieldIndex = 0; iAddrFieldIndex < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; iAddrFieldIndex++)
    {
      phCSM->abAddressField[iAddrFieldIndex] = *((((BYTE *)pvAddressField)+iAddrFieldIndex));
    }
    phCSM->vPFNTPCommunicationInd = vPFNTPCommunicationInd;

    lRetVal = rCommunicationIndNotificationTable.lIndex++;
  }
  else
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_ERR(("CSMAL_lAddCommunicationIndNotification - ERROR - Table Index out of bounds:0x%X",
                   rCommunicationIndNotificationTable.lIndex));
    #endif
  }

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR4(("CSMAL_lAddCommunicationIndNotification - RetVal:0x%X", lRetVal));
  #endif

  return lRetVal;
}

/*******************************************************************************
 * function    CSMAL_lAddDataConNotification
 * \doxydocu
 * \brief      Adds the data for a data confirmation notification
 *             callback to the corresponding notification table.
 *
 * \param[in]  pvHandle
 *             CSM Access Layer Handle
 *
 * \param[in]  bBus
 *             CAN bus number
 *
 * \param[in]  dwProtocolType
 *             protocol type
 *
 * \param[in]  pvAddressField
 *             addressing, depends on protocol
 *
 * \param[in]  pPFNTPDataCon
 *             Callback function
 *
 * \return     index in the corresponding notification table
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview 09.09.08 CM-AI/VTeam-Brunotte
 * \history_begin
 * 17.07.07  VTeam-Khler
 * - Storage of pvAddressField commented out.
 * 11.10.06 3SOFT Schubart
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
static LONG CSMAL_lAddDataConNotification(void * pvHandle, BYTE bBus, DWORD dwProtocolType, const void * pvAddressField,
                                          tCSM_PFN_TP_DATA_CON pPFNTPDataCon)
{
  pDataConMsg phCSM = NULL;
  LONG lRetVal = -1;
  BYTE bAddr0 = *((((BYTE *)pvAddressField)+0));
  BYTE bAddr1 = *((((BYTE *)pvAddressField)+1));

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR2(("CSMAL_lAddDataConNotification - Handle:0x%X  Bus:%x  Prot:%x  AddrField:0x%02X%02X  PFNTPDataCon:0x%X",
                  pvHandle,
                  ETG_CENUM(tCSM_BUS_NUMBER,               (tU8)  bBus),
                  ETG_CENUM(tCSM_PROTOCOL_TYPE,            (tU32) dwProtocolType),
                  (tU8)  bAddr0,
                  (tU8)  bAddr1,
                  pPFNTPDataCon));
  #endif

  // get message placeholder if current access index is in a valid range
  if ((rDataConNotificationTable.lIndex >= 0) && (rDataConNotificationTable.lIndex < CSMAL_MAX_DATACON_NOTIFICATIONS))
  {
    INT iAddrFieldIndex;

    phCSM = &rDataConNotificationTable.arDataConNotifications[rDataConNotificationTable.lIndex];

    phCSM->rMsgHeader.enMsgType = CSM_ACCESS_DataCon;
    phCSM->rMsgHeader.pvAccessUsrHandle  = pvHandle;
    phCSM->bBus                 = bBus;
    phCSM->dwProtocolType       = dwProtocolType;
    for (iAddrFieldIndex = 0; iAddrFieldIndex < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; iAddrFieldIndex++)
    {
      phCSM->abAddressField[iAddrFieldIndex] = *((((BYTE *)pvAddressField)+iAddrFieldIndex));
    }
    phCSM->vPFNTPDataCon        = pPFNTPDataCon;

    lRetVal = rDataConNotificationTable.lIndex++;
  }
  else
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_ERR(("CSMAL_lAddDataConNotification - ERROR - Table Index out of bounds:0x%X",
                  rDataConNotificationTable.lIndex));
    #endif
  }

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR4(("CSMAL_lAddDataConNotification - RetVal:0x%X", lRetVal));
  #endif

  return lRetVal;
}

/*******************************************************************************
 * function    CSMAL_lAddDataIndNotification
 * \doxydocu
 * \brief      Adds the data for a data indication notification
 *             callback to the corresponding notification table.
 *
 * \param[in]  pvHandle
 *             CSM Access Layer Handle
 *
 * \param[in]  bBus
 *             CAN bus number
 *
 * \param[in]  dwProtocolType
 *             protocol type
 *
 * \param[in]  pvAddressField
 *             addressing, depends on protocol
 *
 * \param[in]  pPFNTPDataInd
 *             Callback function
 *
 * \return     index in the corresponding notification table
 *             or error code
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview 09.09.08 CM-AI/VTeam-Brunotte
 * \history_begin
 * 03.09.08  VTeam-Khler
 * - Return value initialized with error macro and init value.
 * 07.07.08  VTeam-Khler
 * - CSM_M_MAKE_ERROR macro used for negative return value.
 * 17.07.07  VTeam-Khler
 * - Storage of pvAddressField commented out.
 * 11.10.06 3SOFT Schubart
 * Initial Revision.
 * \history_end
 ******************************************************************************/
static LONG CSMAL_lAddDataIndNotification(void * pvHandle, BYTE bBus, DWORD dwProtocolType, const void * pvAddressField,
                                          tCSM_PFN_TP_DATA_IND pPFNTPDataInd)
{
  pDataIndMsg phCSM = NULL;
  LONG lRetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_WARNING, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_INIT_WARNING);
  BYTE bAddr0 = *((((BYTE *)pvAddressField)+0));
  BYTE bAddr1 = *((((BYTE *)pvAddressField)+1));

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR2(("CSMAL_lAddDataIndNotification - Handle:0x%X  Bus:%x  Prot:%x  AddrField:0x%02X%02X  PFNTPDataInd:0x%X",
                  pvHandle,
                  ETG_CENUM(tCSM_BUS_NUMBER,               (tU8)  bBus),
                  ETG_CENUM(tCSM_PROTOCOL_TYPE,            (tU32) dwProtocolType),
                  (tU8)  bAddr0,
                  (tU8)  bAddr1,
                  pPFNTPDataInd));
  #endif

  // get message placeholder if current access index is in a valid range
  if ((rDataIndNotificationTable.lIndex >= 0) && (rDataIndNotificationTable.lIndex < CSMAL_MAX_DATAIND_NOTIFICATIONS))
  {
    INT iAddrFieldIndex;

    phCSM = &rDataIndNotificationTable.arDataIndNotifications[rDataIndNotificationTable.lIndex];

    phCSM->rMsgHeader.enMsgType = CSM_ACCESS_DataInd;
    phCSM->rMsgHeader.pvAccessUsrHandle  = pvHandle;
    phCSM->bBus                 = bBus;
    phCSM->dwProtocolType       = dwProtocolType;
    for (iAddrFieldIndex = 0; iAddrFieldIndex < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; iAddrFieldIndex++)
    {
      phCSM->abAddressField[iAddrFieldIndex] = *((((BYTE *)pvAddressField)+iAddrFieldIndex));
    }
    phCSM->vPFNTPDataInd        = pPFNTPDataInd;

    lRetVal = rDataIndNotificationTable.lIndex++;
  }
  else
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_ERR(("CSMAL_lAddDataIndNotification - ERROR - Table Index out of bounds:0x%X",
                   rDataIndNotificationTable.lIndex));
    #endif
    lRetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_BUFFER_FULL);
  }

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR4(("CSMAL_lAddDataIndNotification - RetVal:0x%X", lRetVal));
  #endif

  return lRetVal;
}

/*******************************************************************************
 * function    CSMAL_lAddDataIndFFNotification
 * \doxydocu
 * \brief      Adds the data for a first frame data indication notification
 *             callback to the corresponding notification table.
 *
 * \param[in]  pvHandle
 *             CSM Access Layer Handle
 *
 * \param[in]  bBus
 *             CAN bus number
 *
 * \param[in]  dwProtocolType
 *             protocol type
 *
 * \param[in]  pvAddressField
 *             addressing, depends on protocol
 *
 * \param[in]  pPFNTPDataIndFF
 *             Callback function
 *
 * \return     index in the corresponding notification table
 *             or error code
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview
 * \history_begin
 * 17.11.08  VTeam-Khler
 * - Initial Revision.
 * \history_end
 ******************************************************************************/
#if (CSM_S_ISO_TP == CSM_C_F_ON)
#if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
static LONG CSMAL_lAddDataIndFFNotification(void * pvHandle, BYTE bBus, DWORD dwProtocolType, const void * pvAddressField,
                                            tCSM_PFN_TP_DATA_IND_FF pPFNTPDataIndFF)
{
  pDataIndFFMsg phCSM = NULL;
  LONG lRetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_WARNING, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_INIT_WARNING);
  BYTE bAddr0 = *((((BYTE *)pvAddressField)+0));
  BYTE bAddr1 = *((((BYTE *)pvAddressField)+1));

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR2(("CSMAL_lAddDataIndFFNotification - Handle:0x%X  Bus:%x  Prot:%x  AddrField:0x%02X%02X  PFNTPDataIndFF:0x%X",
                  pvHandle,
                  ETG_CENUM(tCSM_BUS_NUMBER,               (tU8)  bBus),
                  ETG_CENUM(tCSM_PROTOCOL_TYPE,            (tU32) dwProtocolType),
                  (tU8)  bAddr0,
                  (tU8)  bAddr1,
                  pPFNTPDataIndFF));
  #endif

  // get message placeholder if current access index is in a valid range
  if ((rDataIndFFNotificationTable.lIndex >= 0) && (rDataIndFFNotificationTable.lIndex < CSMAL_MAX_DATAINDFF_NOTIFICATIONS))
  {
    INT iAddrFieldIndex;

    phCSM = &rDataIndFFNotificationTable.arDataIndFFNotifications[rDataIndFFNotificationTable.lIndex];

    phCSM->rMsgHeader.enMsgType = CSM_ACCESS_DataIndFF;
    phCSM->rMsgHeader.pvAccessUsrHandle  = pvHandle;
    phCSM->bBus                 = bBus;
    phCSM->dwProtocolType       = dwProtocolType;
    for (iAddrFieldIndex = 0; iAddrFieldIndex < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; iAddrFieldIndex++)
    {
      phCSM->abAddressField[iAddrFieldIndex] = *((((BYTE *)pvAddressField)+iAddrFieldIndex));
    }
    phCSM->vPFNTPDataIndFF      = pPFNTPDataIndFF;

    lRetVal = rDataIndFFNotificationTable.lIndex++;
  }
  else
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_ERR(("CSMAL_lAddDataIndFFNotification - ERROR - Table Index out of bounds:0x%X",
                   rDataIndFFNotificationTable.lIndex));
    #endif
    lRetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_BUFFER_FULL);
  }

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR4(("CSMAL_lAddDataIndFFNotification - RetVal:0x%X", lRetVal));
  #endif

  return lRetVal;
}
#endif // CSM_S_ITP_FF_INDICATION_AVAILABLE
#endif // CSM_S_ISO_TP

/*******************************************************************************
 * function    CSMAL_lAddDataErrorIndNotification
 * \doxydocu
 * \brief      Adds the data for a data error indication notification
 *             callback function to the corresponding notification table.
 *
 * \param[in]  pvHandle
 *             CSM Access Layer Handle
 *
 * \param[in]  bBus
 *             CAN bus number
 *
 * \param[in]  dwProtocolType
 *             protocol type
 *
 * \param[in]  pvAddressField
 *             addressing, depends on protocol
 *
 * \param[in]  pPFNTPDataErrorInd
 *             Callback function pointer
 *
 * \return     >=0: index in the corresponding notification table or
 *             < 0: error code
 *
 * \access     Called by CSMAL_lAddApplNotification() within CSMAL_lLock/Unlock
 * \reentrant  no
 *
 * \lastreview
 * \history_begin
 * 13.11.08  VTeam-Khler
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
static LONG CSMAL_lAddDataErrorIndNotification(void * pvHandle, BYTE bBus, DWORD dwProtocolType, const void * pvAddressField,
                                               tCSM_PFN_TP_DATA_ERROR_IND pPFNTPDataErrorInd)
{
  LONG lRetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_WARNING, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_INIT_WARNING);
  BYTE bAddr0 = *((((BYTE *)pvAddressField)+0));
  BYTE bAddr1 = *((((BYTE *)pvAddressField)+1));

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR2(("CSMAL_lAddDataErrorIndNotification - Handle:0x%X  Bus:%x  Prot:%x  AddrField:0x%02X%02X  PFNTPDataErrorInd0x%X",
                  pvHandle,
                  ETG_CENUM(tCSM_BUS_NUMBER,               (tU8)  bBus),
                  ETG_CENUM(tCSM_PROTOCOL_TYPE,            (tU32) dwProtocolType),
                  (tU8)  bAddr0,
                  (tU8)  bAddr1,
                  pPFNTPDataErrorInd));
  #endif

  // get message placeholder if current access index is in a valid range
  if ((rDataErrorIndNotificationTable.lIndex >= 0) && (rDataErrorIndNotificationTable.lIndex < CSMAL_MAX_DATAERRORIND_NOTIFICATIONS))
  {
    pDataErrorIndMsg phCSM = NULL;
    INT  iAddrFieldIndex;

    phCSM = &rDataErrorIndNotificationTable.arDataErrorIndNotifications[rDataErrorIndNotificationTable.lIndex];

    phCSM->rMsgHeader.enMsgType = CSM_ACCESS_DataErrorInd;
    phCSM->rMsgHeader.pvAccessUsrHandle  = pvHandle;
    phCSM->bBus                 = bBus;
    phCSM->dwProtocolType       = dwProtocolType;
    for (iAddrFieldIndex = 0; iAddrFieldIndex < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; iAddrFieldIndex++)
    {
      phCSM->abAddressField[iAddrFieldIndex] = *((((BYTE *)pvAddressField)+iAddrFieldIndex));
    }
    phCSM->vPFNTPDataErrorInd   = pPFNTPDataErrorInd;

    lRetVal = rDataErrorIndNotificationTable.lIndex++;
  }
  else
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_ERR(("CSMAL_lAddDataErrorIndNotification - ERROR - Table Index out of bounds:0x%X",
                  rDataErrorIndNotificationTable.lIndex));
    #endif
    lRetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_BUFFER_FULL);
  }

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR4(("CSMAL_lAddDataErrorIndNotification - RetVal:0x%X", lRetVal));
  #endif

  return lRetVal;
}
#endif // CSM_S_DATA_ERROR_IND_AVAILABLE
#endif // CSM_S_CNP_AVAILABLE

/*******************************************************************************
 * function    CSMAL_lAddBusErrorIndNotification
 * \doxydocu
 * \brief      Adds the data for a bus error indication notification
 *             callback function to the corresponding notification table.
 *
 * \param[in]  pvHandle
 *             CSM Access Layer Handle
 *
 * \param[in]  pPFNTPBusErrorInd
 *             Callback function pointer
 *
 * \return     >=0: index in the corresponding notification table or
 *             < 0: error code
 *
 * \access     Called by CSMAL_lAddApplNotification() within CSMAL_lLock/Unlock
 * \reentrant  no
 *
 * \lastreview 24.03.09 VTeam-Franke
 * \history_begin
 *
 * 01.02.09  VTeam-Pistoor
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#if (CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON)
static pBusErrorIndMsg CSMAL_lAddBusErrorIndNotification(void * pvHandle, tCSM_PFN_BUS_ERROR_IND pPFNBusErrorInd)
{
  pBusErrorIndMsg phCSM = NULL;

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR2(("CSMAL_lAddBusErrorIndNotification - Handle:0x%X  PFNBusErrorInd:0x%X",
                  pvHandle,
                  pPFNBusErrorInd));
  #endif

  // get message placeholder if current access index is in a valid range
  if ((rBusErrorIndNotificationTable.lIndex >= 0) && (rBusErrorIndNotificationTable.lIndex < CSMAL_MAX_BUSERRORIND_NOTIFICATIONS))
  {
    phCSM = &rBusErrorIndNotificationTable.arBusErrorIndNotifications[rBusErrorIndNotificationTable.lIndex];

    phCSM->rMsgHeader.enMsgType = CSM_ACCESS_BusErrorInd;
    phCSM->rMsgHeader.pvAccessUsrHandle  = pvHandle;
    phCSM->vPFNBusErrorInd   = pPFNBusErrorInd;

    rBusErrorIndNotificationTable.lIndex++;
  }
  else
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_ERR(("CSMAL_lAddBusErrorIndNotification - ERROR - Table Index out of bounds:0x%X",
                   rBusErrorIndNotificationTable.lIndex));
    #endif
  }

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR4(("CSMAL_lAddBusErrorIndNotification - RetVal:0x%X", phCSM));
  #endif

  return phCSM;
}
#endif // CSM_S_ENABLE_INDICATE_BUSERROR

/*******************************************************************************
 * function    CSMAL_lAddApplNotification
 * \doxydocu
 * \brief      TODO
 *
 * \param[in]  pvHandle
 *             CSM Access Layer Handle
 *
 * \param[in]  Bus
 *             CAN bus number
 *
 * \param[in]  dwProtocolType
 *             protocol type
 *
 * \param[in]  vAddressField
 *             addressing, depends on protocol
 *
 * \param[in]  pPFNTPCommunicationCon
 *             Callback function for communication confirmation
 *
 * \param[in]  pPFNTPCommunicationInd
 *             Callback function for communication indication
 *
 * \param[in]  pPFNTPDataCon
 *             Callback function for data confirmation
 *
 * \param[in]  pPFNTPDataInd
 *             Callback function for data indication
 *
 * \param[in]  pPFNTPDataIndFF
 *             featured with #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
 *             Callback function for First Frame data indication
 *
 * \param[in]  pPFNTPDataErrorInd
 *             featured with #if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
 *             Callback function for data error indication
 *
 * \return     LONG
 *             index in the corresponding notification table
 *
 * \access     ? called by lRegister..._ApplCallback functions
 * \reentrant  ?
 *
 * \lastreview 09.09.08 CM-AI/VTeam-Brunotte
 * \history_begin
 * 14.11.08  VTeam-Khler
 * DataIndFF and DataErrorInd added.
 * 11.10.06 3SOFT Schubart
 * Initial Revision.
 * \history_end
 ******************************************************************************/
static pApplNotificationLookupEntry CSMAL_lAddApplNotification(void * pvHandle, BYTE bBus, DWORD dwProtocolType, const void * pvAddressField,
                                                               tCSM_PFN_TP_COMMUNICATION_CON pPFNTPCommunicationCon,
                                                               tCSM_PFN_TP_COMMUNICATION_IND pPFNTPCommunicationInd
                                                               #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
                                                               ,tCSM_PFN_TP_DATA_CON pPFNTPDataCon
                                                               ,tCSM_PFN_TP_DATA_IND pPFNTPDataInd
                                                               #if (CSM_S_MPDT == CSM_C_F_ON )
                                                               ,tCSM_PFN_TP_DATA_CON pPFNTPDDataCon
                                                               ,tCSM_PFN_TP_DATA_IND pPFNTPDDataInd
                                                               #endif
                                                               #if (CSM_S_ISO_TP == CSM_C_F_ON)
                                                               #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
                                                               ,tCSM_PFN_TP_DATA_IND_FF pPFNTPDataIndFF
                                                               #endif // CSM_S_ITP_FF_INDICATION_AVAILABLE
                                                               #endif // CSM_S_ISO_TP
                                                               #if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
                                                               ,tCSM_PFN_TP_DATA_ERROR_IND pPFNTPDataErrorInd
                                                               #endif
                                                               #endif // CSM_S_CNP_AVAILABLE
                                                              )
{
  LONG lIndexCommunicationCon = -1;
  LONG lIndexCommunicationInd = -1;
  #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
  LONG lIndexDataCon = -1;
  LONG lIndexDataInd = -1;
  //#if (CSM_S_MPDT == CSM_C_F_ON)  // always allocate these vars, since used for tracing
  LONG lIndexDDataCon = -1;
  LONG lIndexDDataInd = -1;
  //#endif  // CSM_S_MPDT
  #if (CSM_S_ISO_TP == CSM_C_F_ON)
  #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
  LONG lIndexDataIndFF = -1;
  #endif
  #endif // CSM_S_ISO_TP
  #if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
  LONG lIndexDataErrorInd = -1;
  #endif
  #endif // CSM_S_CNP_AVAILABLE

  BYTE bAddr0 = *((((BYTE *)pvAddressField)+0));
  BYTE bAddr1 = *((((BYTE *)pvAddressField)+1));

  pApplNotificationLookupEntry phCSM = NULL;

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR2(("CSMAL_lAddApplNotification - Handle:0x%X  Bus:%x  Prot:%x  AddrField:0x%02X%02X  PFNTPCommunicationCon:0x%X  PFNTPCommunicationInd:0x%X  PFNTPDataCon:0x%X  PFNTPDataInd:0x%X",
                  pvHandle,
                  ETG_CENUM(tCSM_BUS_NUMBER,               (tU8)  bBus),
                  ETG_CENUM(tCSM_PROTOCOL_TYPE,            (tU32) dwProtocolType),
                  (tU8)  bAddr0,
                  (tU8)  bAddr1,
                  pPFNTPCommunicationCon,
                  pPFNTPCommunicationInd,
                  pPFNTPDataCon,
                  pPFNTPDataInd));
  #endif

  // add the given callback functions to their corresponding notification tables;
  // first check if they are not NULL, so no registering is needed for the function

  // lock table for inserting
  CSMAL_lLock(ghApplNotifyTableLock);

  // check CommunicationCON pointer
  if (NULL != pPFNTPCommunicationCon)
  {
    lIndexCommunicationCon = CSMAL_lAddCommunicationConNotification(pvHandle, bBus, dwProtocolType, pvAddressField, pPFNTPCommunicationCon);
  }

  // check CommunicationIND pointer
  if (NULL != pPFNTPCommunicationInd)
  {
    lIndexCommunicationInd = CSMAL_lAddCommunicationIndNotification(pvHandle, bBus, dwProtocolType, pvAddressField, pPFNTPCommunicationInd);
  }

  #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
  {
    // check DataCON pointer
  if (NULL != pPFNTPDataCon)
    {
      lIndexDataCon = CSMAL_lAddDataConNotification(pvHandle, bBus, dwProtocolType, pvAddressField, pPFNTPDataCon);
    }

  // check DataIND pointer
    if (NULL != pPFNTPDataInd)
    {
      lIndexDataInd = CSMAL_lAddDataIndNotification(pvHandle, bBus, dwProtocolType, pvAddressField, pPFNTPDataInd);
    }

  #if (CSM_S_MPDT == CSM_C_F_ON)
    // check DataCon2 pointer
  if (NULL != pPFNTPDDataCon)
    {
      // this is the special case for the MPDT, where D-Data also require a notification
    lIndexDDataCon = CSMAL_lAddDataConNotification(pvHandle, bBus, CSM_C_PTYPE_RN_MPDT_D, pvAddressField, pPFNTPDDataCon);
    }

  // check DataIND2 pointer
    if (NULL != pPFNTPDDataInd)
    {
      // this is the special case for the MPDT, where D-Data also require a notification
      lIndexDDataInd = CSMAL_lAddDataIndNotification(pvHandle, bBus, CSM_C_PTYPE_RN_MPDT_D, pvAddressField, pPFNTPDDataInd);
    }
  #endif

  #if (CSM_S_ISO_TP == CSM_C_F_ON)
    {
      #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
      // check FF DataIND pointer
      if (NULL != pPFNTPDataIndFF)
      {
        lIndexDataIndFF = CSMAL_lAddDataIndFFNotification(pvHandle, bBus, dwProtocolType, pvAddressField, pPFNTPDataIndFF);
      }
      #endif // CSM_S_ITP_FF_INDICATION_AVAILABLE
    }
    #endif // CSM_S_ISO_TP

    #if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
    if (NULL != pPFNTPDataErrorInd)
    {
      lIndexDataErrorInd = CSMAL_lAddDataErrorIndNotification(pvHandle, bBus, dwProtocolType, pvAddressField, pPFNTPDataErrorInd);
    }
    #endif // CSM_S_DATA_ERROR_IND_AVAILABLE
  }
  #endif // CSM_S_CNP_AVAILABLE

  // insert data into lookup table, if the insertion index is in a valid range
  if( (rApplNotificationLookupTable.lIndex >= 0) && (rApplNotificationLookupTable.lIndex < CSMAL_MAX_APPL_NOTIFICATIONS))
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_USR2(("CSMAL_lAddApplNotification - IndexCommCon:0x%X  IndexCommInd:0x%X  IndexDataCon:0x%X  IndexDataInd:0x%X  IndexDataCon2:0x%X  IndexDataInd2:0x%X",
                    lIndexCommunicationCon,
                    lIndexCommunicationInd,
                    lIndexDataCon,
                    lIndexDataInd,
                    lIndexDDataCon,
                    lIndexDDataInd
                  ));
    #endif

    // within the lookup table select the struct element "arEntries" and use the the current access index to
    // reference it with the help of a pointer now
    phCSM = &rApplNotificationLookupTable.arEntries[rApplNotificationLookupTable.lIndex];

    // add data to table entry
    phCSM->lIndexCommunicationCon = lIndexCommunicationCon;
    phCSM->lIndexCommunicationInd = lIndexCommunicationInd;

    #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
    phCSM->lIndexDataCon          = lIndexDataCon;
    phCSM->lIndexDataInd          = lIndexDataInd;

    #if (CSM_S_MPDT == CSM_C_F_ON)
    phCSM->lIndexDDataCon         = lIndexDDataCon;
    phCSM->lIndexDDataInd         = lIndexDDataInd;
    #endif  // CSM_S_MPDT

    #if (CSM_S_ISO_TP == CSM_C_F_ON)
    #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
    phCSM->lIndexDataIndFF        = lIndexDataIndFF;
    #endif
    #endif

    #if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
    phCSM->lIndexDataErrorInd     = lIndexDataErrorInd;
    #endif
    #endif // CSM_S_CNP_AVAILABLE

    phCSM->pvHandle               = pvHandle;

    // increase the current lookup index
    rApplNotificationLookupTable.lIndex++;
  }

  // unlock table (after insertion)
  CSMAL_lUnLock(ghApplNotifyTableLock);

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR4(("CSMAL_lAddApplNotification - phCSM:0x%X", phCSM));
  #endif

  // return the lookup entry index, so that upon callback the entry can be accessed directly
  return phCSM;
}

/*******************************************************************************
 *  function implementation (scope: module-global)
 ******************************************************************************/

/*******************************************************************************
 * function    CSMAL_vInitAllNotificationTables
 * \doxydocu
 * \brief      initializes internal data of kernel mode part of CSM Access Layer
 *
 * \return     VOID
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview 09.09.08 CM-AI/VTeam-Brunotte
 * \history_begin
 * 19.11.08 VTeam-Khler, -Pistoor
 * DataIndFF and DataErrorInd added.
 * 16.06.08 VTeam-Brunotte
 * Initial Revision.
 * \history_end
 ******************************************************************************/
static void CSMAL_vInitAllNotificationTables(void)
{
  LONG lCount;
  INT  iAddrFieldIndex;

  // start with the signal indication table
  rSignalIndNotificationTable.lIndex = 0;
  for(lCount = 0; lCount < CSMAL_MAX_SIGNALIND_NOTIFICATIONS; lCount++)
  {
    rSignalIndNotificationTable.arSignalIndNotifications[lCount].rMsgHeader.enMsgType  = CSM_ACCESS_NoEvent;
    rSignalIndNotificationTable.arSignalIndNotifications[lCount].rMsgHeader.pvAccessUsrHandle   = NULL;
    rSignalIndNotificationTable.arSignalIndNotifications[lCount].vPFNCBRSignalInd      = NULL;
    rSignalIndNotificationTable.arSignalIndNotifications[lCount].dwSignalId            = 0;
    rSignalIndNotificationTable.arSignalIndNotifications[lCount].dwState               = 0;
  }

  // next is the communication confirmation table
  rCommunicationConNotificationTable.lIndex   = 0;
  for(lCount = 0; lCount < CSMAL_MAX_COMMUNICATIONCON_NOTIFICATIONS; lCount++)
  {
    rCommunicationConNotificationTable.arCommunicationConNotifications[lCount].rMsgHeader.enMsgType   = CSM_ACCESS_NoEvent;
    rCommunicationConNotificationTable.arCommunicationConNotifications[lCount].rMsgHeader.pvAccessUsrHandle    = NULL;
    rCommunicationConNotificationTable.arCommunicationConNotifications[lCount].vPFNTPCommunicationCon = NULL;
    rCommunicationConNotificationTable.arCommunicationConNotifications[lCount].bBus                   = 0;
    rCommunicationConNotificationTable.arCommunicationConNotifications[lCount].dwProtocolType         = 0;
    for (iAddrFieldIndex = 0; iAddrFieldIndex < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; iAddrFieldIndex++)
    {
      rCommunicationConNotificationTable.arCommunicationConNotifications[lCount].abAddressField[iAddrFieldIndex] = 0;
    }
    rCommunicationConNotificationTable.arCommunicationConNotifications[lCount].bConnectState          = 0;
    rCommunicationConNotificationTable.arCommunicationConNotifications[lCount].wAppId                 = 0;
  }

  // next is the communication indication table
  rCommunicationIndNotificationTable.lIndex   = 0;
  for(lCount = 0; lCount < CSMAL_MAX_COMMUNICATIONIND_NOTIFICATIONS; lCount++)
  {
    rCommunicationIndNotificationTable.arCommunicationIndNotifications[lCount].rMsgHeader.enMsgType   = CSM_ACCESS_NoEvent;
    rCommunicationIndNotificationTable.arCommunicationIndNotifications[lCount].rMsgHeader.pvAccessUsrHandle    = NULL;
    rCommunicationIndNotificationTable.arCommunicationIndNotifications[lCount].vPFNTPCommunicationInd = NULL;
    rCommunicationIndNotificationTable.arCommunicationIndNotifications[lCount].bBus                   = 0;
    rCommunicationIndNotificationTable.arCommunicationIndNotifications[lCount].dwProtocolType         = 0;
    for (iAddrFieldIndex = 0; iAddrFieldIndex < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; iAddrFieldIndex++)
    {
      rCommunicationIndNotificationTable.arCommunicationIndNotifications[lCount].abAddressField[iAddrFieldIndex] = 0;
    }
    rCommunicationIndNotificationTable.arCommunicationIndNotifications[lCount].bConnectState          = 0;
    rCommunicationIndNotificationTable.arCommunicationIndNotifications[lCount].wAppId                 = 0;
  }

  #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
  {
    // next is the data confirmation table
    rDataConNotificationTable.lIndex   = 0;
    for(lCount = 0; lCount < CSMAL_MAX_DATACON_NOTIFICATIONS; lCount++)
    {
      rDataConNotificationTable.arDataConNotifications[lCount].rMsgHeader.enMsgType = CSM_ACCESS_NoEvent;
      rDataConNotificationTable.arDataConNotifications[lCount].rMsgHeader.pvAccessUsrHandle  = NULL;
      rDataConNotificationTable.arDataConNotifications[lCount].vPFNTPDataCon        = NULL;
      rDataConNotificationTable.arDataConNotifications[lCount].bBus                 = 0;
      rDataConNotificationTable.arDataConNotifications[lCount].dwProtocolType       = 0;
      for (iAddrFieldIndex = 0; iAddrFieldIndex < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; iAddrFieldIndex++)
      {
        rDataConNotificationTable.arDataConNotifications[lCount].abAddressField[iAddrFieldIndex] = 0;
      }
      rDataConNotificationTable.arDataConNotifications[lCount].bTransferResult      = 0;
    }

    // next is the data indication table
    rDataIndNotificationTable.lIndex   = 0;
    for(lCount = 0; lCount < CSMAL_MAX_DATAIND_NOTIFICATIONS; lCount++)
    {
      rDataIndNotificationTable.arDataIndNotifications[lCount].rMsgHeader.enMsgType = CSM_ACCESS_NoEvent;
      rDataIndNotificationTable.arDataIndNotifications[lCount].rMsgHeader.pvAccessUsrHandle  = NULL;
      rDataIndNotificationTable.arDataIndNotifications[lCount].vPFNTPDataInd        = NULL;
      rDataIndNotificationTable.arDataIndNotifications[lCount].bBus                 = 0;
      rDataIndNotificationTable.arDataIndNotifications[lCount].dwProtocolType       = 0;
      //  rDataIndNotificationTable.arDataIndNotifications[lCount].pbData               = NULL;
      for (iAddrFieldIndex = 0; iAddrFieldIndex < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; iAddrFieldIndex++)
      {
        rDataIndNotificationTable.arDataIndNotifications[lCount].abAddressField[iAddrFieldIndex] = 0;
      }
      rDataIndNotificationTable.arDataIndNotifications[lCount].wDataLength          = 0;
    }

    #if (CSM_S_ISO_TP == CSM_C_F_ON)
    {
      #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
      {
        // next is the first frame data indication table
        rDataIndFFNotificationTable.lIndex   = 0;
        for(lCount = 0; lCount < CSMAL_MAX_DATAIND_NOTIFICATIONS; lCount++)
        {
          rDataIndFFNotificationTable.arDataIndFFNotifications[lCount].rMsgHeader.enMsgType          = CSM_ACCESS_NoEvent;
          rDataIndFFNotificationTable.arDataIndFFNotifications[lCount].rMsgHeader.pvAccessUsrHandle  = NULL;
          rDataIndFFNotificationTable.arDataIndFFNotifications[lCount].vPFNTPDataIndFF            = NULL;
          rDataIndFFNotificationTable.arDataIndFFNotifications[lCount].bBus                 = 0;
          rDataIndFFNotificationTable.arDataIndFFNotifications[lCount].dwProtocolType       = 0;
          for (iAddrFieldIndex = 0; iAddrFieldIndex < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; iAddrFieldIndex++)
          {
            rDataIndFFNotificationTable.arDataIndFFNotifications[lCount].abAddressField[iAddrFieldIndex] = 0;
          }
          rDataIndFFNotificationTable.arDataIndFFNotifications[lCount].wDataLength      = 0;
        }
      }
      #endif // CSM_S_ITP_FF_INDICATION_AVAILABLE
    }
    #endif // CSM_S_ISO_TP

    #if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
    {
      // next is the data error indication table
      rDataErrorIndNotificationTable.lIndex   = 0;
      for(lCount = 0; lCount < CSMAL_MAX_DATAIND_NOTIFICATIONS; lCount++)
      {
        rDataErrorIndNotificationTable.arDataErrorIndNotifications[lCount].rMsgHeader.enMsgType          = CSM_ACCESS_NoEvent;
        rDataErrorIndNotificationTable.arDataErrorIndNotifications[lCount].rMsgHeader.pvAccessUsrHandle  = NULL;
        rDataErrorIndNotificationTable.arDataErrorIndNotifications[lCount].vPFNTPDataErrorInd            = NULL;
        rDataErrorIndNotificationTable.arDataErrorIndNotifications[lCount].bBus                 = 0;
        rDataErrorIndNotificationTable.arDataErrorIndNotifications[lCount].dwProtocolType       = 0;
        for (iAddrFieldIndex = 0; iAddrFieldIndex < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; iAddrFieldIndex++)
        {
          rDataErrorIndNotificationTable.arDataErrorIndNotifications[lCount].abAddressField[iAddrFieldIndex] = 0;
        }
        rDataErrorIndNotificationTable.arDataErrorIndNotifications[lCount].dwDataErrorCode      = 0;
      }
    }
    #endif // CSM_S_DATA_ERROR_IND_AVAILABLE
  }
  #endif // CSM_S_CNP_AVAILABLE
}

/*******************************************************************************
 * function    CSMAL_vInit
 * \doxydocu
 * \brief      initializes internal data of kernel mode part of CSM Access Layer
 *
 * \param[in]  VOID
 *
 * \return     Fix CSM_C_NO_ERROR
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview 09.09.08 CM-AI/VTeam-Brunotte
 * \history_begin
 * 17.07.07  VTeam-Khler
 * - Address field byte array instead of pointer.
 * 04.10.06 3SOFT Schubart
 * Initial Revision.
 * \history_end
 ******************************************************************************/
LONG CSMAL_lInit(VOID)
{
  LONG lCount;

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR2(("CSMAL_vInit"));
  ETG_TRACE_USR2(("##########################################"));
  ETG_TRACE_USR2(("# Configuration"));
  ETG_TRACE_USR2(("##########################################"));
  ETG_TRACE_USR2(("# CSMAL_MAX_SIGNALIND_NOTIFICATIONS = 0x%X", CSMAL_MAX_SIGNALIND_NOTIFICATIONS));
  ETG_TRACE_USR2(("# CSMAL_MAX_APPL_NOTIFICATIONS      = 0x%X", CSMAL_MAX_APPL_NOTIFICATIONS));
  ETG_TRACE_USR2(("# CSMAL_NUM_HANDLES                 = 0x%X", CSMAL_NUM_HANDLES));
  ETG_TRACE_USR2(("##########################################"));
  #endif

  // create semaphore for signal notifications
  ghSignalNotifyTableLock = CSMAL_lCreateLock("CSMALSi");

  #ifndef VARIANT_S_FTR_ENABLE_OSAL_CSM_SOCKET_SIMU
  if(ghSignalNotifyTableLock < 0)
  #else
  if(ghSignalNotifyTableLock == (LONG)OSAL_C_INVALID_HANDLE)
  #endif
  {
    // error creating semaphore
    return CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_RESOURCE);
  }

  // create semaphore for application notifications
  ghApplNotifyTableLock = CSMAL_lCreateLock("CSMALAp");

  #ifndef VARIANT_S_FTR_ENABLE_OSAL_CSM_SOCKET_SIMU
  if(ghApplNotifyTableLock < 0)
  #else
  if(ghApplNotifyTableLock == (LONG)OSAL_C_INVALID_HANDLE)
  #endif
  {
    // error creating semaphore
    return CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_RESOURCE);
  }

  // create semaphore for
  ghAccessHandleTableLock = CSMAL_lCreateLock("CSMALHa");

  #ifndef VARIANT_S_FTR_ENABLE_OSAL_CSM_SOCKET_SIMU
  if(ghAccessHandleTableLock < 0)
  #else
  if(ghAccessHandleTableLock == (LONG)OSAL_C_INVALID_HANDLE)
  #endif
  {
    // error creating semaphore
    return CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_RESOURCE);
  }

  // initialize the access handle table
  rCSMAccessHandleTable.lIndex = 0;
  for (lCount = 0; lCount < CSMAL_NUM_HANDLES; lCount++)
  {
    // clear all elements of table entry
    rCSMAccessHandleTable.arCSMAccessHandles[lCount].lMQueueID = 0;
    rCSMAccessHandleTable.arCSMAccessHandles[lCount].lMQueueElemSize = 0;
    rCSMAccessHandleTable.arCSMAccessHandles[lCount].lTaskID   = 0;
  }

  // initialize the notification tables
  CSMAL_vInitAllNotificationTables();

  return CSM_C_NO_ERROR;
}

/*******************************************************************************
 * function    CSMAL_lExit()
 * \doxydocu
 * \brief      initializes internal data of kernel mode part of CSM Access Layer
 *
 * \param[in]  VOID
 *
 * \return     error value
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview
 * \history_begin
 * 16.06.08  VTeam-Brunotte
 * Initial Revision.
 * \history_end
 ******************************************************************************/
LONG CSMAL_lExit(VOID)
{
  LONG lCount;
  LONG lRet = CSM_C_NO_ERROR;
  LONG ercd;

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR2(("CSMAL_lExit"));
  ETG_TRACE_USR2(("##########################################"));
  ETG_TRACE_USR2(("# Configuration"));
  ETG_TRACE_USR2(("##########################################"));
  ETG_TRACE_USR2(("# CSMAL_MAX_SIGNALIND_NOTIFICATIONS = 0x%X", CSMAL_MAX_SIGNALIND_NOTIFICATIONS));
  ETG_TRACE_USR2(("# CSMAL_MAX_APPL_NOTIFICATIONS      = 0x%X", CSMAL_MAX_APPL_NOTIFICATIONS));
  ETG_TRACE_USR2(("# CSMAL_NUM_HANDLES                 = 0x%X", CSMAL_NUM_HANDLES));
  ETG_TRACE_USR2(("##########################################"));
  #endif

  // Delete all message buffer and task objects, which are created in usermode
  CSMAL_lLock(ghAccessHandleTableLock);
  for (lCount = 0; lCount < CSMAL_NUM_HANDLES; lCount++)
  {
    if (rCSMAccessHandleTable.arCSMAccessHandles[lCount].lTaskID>E_OK)
    {
      ercd = CSMAL_lDeleteUserResource(rCSMAccessHandleTable.arCSMAccessHandles[lCount].lTaskID,
                                       rCSMAccessHandleTable.arCSMAccessHandles[lCount].lMQueueID);
      if (ercd < E_OK)
      {
        // ToDo Error Handling . Ignoring errors in hope that the process deletes all resources
        lRet = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_STACK, CSM_E_OS_ERROR);
      }
      rCSMAccessHandleTable.arCSMAccessHandles[lCount].lTaskID = 0;
      rCSMAccessHandleTable.arCSMAccessHandles[lCount].lMQueueID = 0;
      rCSMAccessHandleTable.arCSMAccessHandles[lCount].lMQueueElemSize = 0;
    }
  }

  // deinitialize the notification tables
  CSMAL_lLock(ghSignalNotifyTableLock);
  CSMAL_lLock(ghApplNotifyTableLock);
  CSMAL_vInitAllNotificationTables();

  CSMAL_lUnLock(ghSignalNotifyTableLock);
  CSMAL_lUnLock(ghApplNotifyTableLock);
  CSMAL_lUnLock(ghAccessHandleTableLock);

  // delete semaphores
  CSMAL_lDeleteLock(ghAccessHandleTableLock);
  CSMAL_lDeleteLock(ghApplNotifyTableLock);
  CSMAL_lDeleteLock(ghSignalNotifyTableLock);

  return lRet;
}

/*******************************************************************************
 * function    CSMAL_lRegisterAppl
 * \doxydocu
 * \brief      adds an application notification by putting the given data into the
 *             application notification table
 *
 * \param[in]  pvHandle
 *             CSM Access Layer Handle
 *
 * \param[in]  bBus
 *             CAN bus number
 *
 * \param[in]  dwProtocolType
 *             protocol type
 *
 * \param[in]  pvAddressField
 *             addressing, depends on protocol
 *
 * \param[in]  pvCallBackFkt
 *             Callback function(s)
 *
 * \return     Error code
 *              >= 0 successfull, <0 error
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview 09.09.08 CM-AI/VTeam-Brunotte
 * \history_begin
 * 28.09.06 3SOFT Schubart
 * Initial Revision.
 *
 * 15.01.09 CM-AI/PJ-FO45-Pistoor
 * Added CSM_C_PTYPE_USED_STD_LIN_DIAG
 *
 * 29.09.10 CM-AI/PJ-GM58-Pistoor
 * Added CSM_C_PTYPE_USED_STD_LIN_USDT
 *
 * \history_end
 ******************************************************************************/
LONG CSMAL_lRegisterAppl(void * pvHandle, BYTE bBus, DWORD dwProtocolType, const void * pvAddressField, const void * pvCallBackFkt)
{
  LONG lRetVal = CSM_C_NO_ERROR;
  BYTE bAddr0 = *((((BYTE *)pvAddressField)+0));
  BYTE bAddr1 = *((((BYTE *)pvAddressField)+1));

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR1(("CSMAL_lRegisterAppl - Handle:0x%X  Bus:%x  Prot:%x  AddrField:0x%02X%02X  CallBackFktStruct:0x%X",
                  pvHandle,
                  ETG_CENUM(tCSM_BUS_NUMBER,               (tU8)  bBus),
                  ETG_CENUM(tCSM_PROTOCOL_TYPE,            (tU32) dwProtocolType),
                  (tU8)  bAddr0,
                  (tU8)  bAddr1,
                  pvCallBackFkt));
  #endif

  // the pointer to the callback functions is of type void, so the correct amount of callbacks
  // have already been filled by the upper handler function dependend on the protocoltype (in ProcCsm)
  // here we can simply forward the request to our corresponding local handler function
  switch(dwProtocolType)
  {
    #ifdef CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL
    case CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL:
    {
      lRetVal = lRegisterCSM_LocalCtrl_ApplCallback(pvHandle, bBus, dwProtocolType, pvAddressField, pvCallBackFkt);
    }
    break;
    #endif // CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL

    #ifdef CSM_C_PTYPE_GM_NWM
    case CSM_C_PTYPE_GM_NWM:
    {
      lRetVal = lRegisterCSM_GMLAN_NWM_ApplCallback(pvHandle, bBus, dwProtocolType, pvAddressField, pvCallBackFkt);
    }
    break;
    #endif // CSM_C_PTYPE_GM_NWM

    #ifdef CSM_C_PTYPE_USED_STD_USDT_ISO
    case CSM_C_PTYPE_USED_STD_USDT_ISO:
    {
      lRetVal = lRegisterCSM_ITP_ApplCallback(pvHandle, bBus, dwProtocolType, pvAddressField, pvCallBackFkt);
    }
    break;
    #endif // CSM_C_PTYPE_USED_STD_USDT_ISO

    #ifdef CSM_C_PTYPE_USED_STD_LIN_DIAG
    case CSM_C_PTYPE_USED_STD_LIN_DIAG:
    {
      lRetVal = lRegisterCSM_ITP_ApplCallback(pvHandle, bBus, dwProtocolType, pvAddressField, pvCallBackFkt);
    }
    break;
    #endif // CSM_C_PTYPE_USED_STD_LIN_DIAG

    #ifdef CSM_C_PTYPE_USED_STD_LIN_USDT
    case CSM_C_PTYPE_USED_STD_LIN_USDT:
    {
      lRetVal = lRegisterCSM_ITP_ApplCallback(pvHandle, bBus, dwProtocolType, pvAddressField, pvCallBackFkt);
    }
    break;
    #endif // CSM_C_PTYPE_USED_STD_LIN_USDT

    #ifdef CSM_C_PTYPE_USED_STD_UUDT_TX
    case CSM_C_PTYPE_USED_STD_UUDT_TX:
    {
      lRetVal = lRegisterCSM_UUDT_TX_ApplCallback(pvHandle, bBus, dwProtocolType, pvAddressField, pvCallBackFkt);
    }
    break;
    #endif // CSM_C_PTYPE_USED_STD_UUDT_TX

    #ifdef CSM_C_PTYPE_USED_STD_UUDT_RX
    case CSM_C_PTYPE_USED_STD_UUDT_RX:
    {
      lRetVal = lRegisterCSM_UUDT_RX_ApplCallback(pvHandle, bBus, dwProtocolType, pvAddressField, pvCallBackFkt);
    }
    break;
    #endif // CSM_C_PTYPE_USED_STD_UUDT_RX

    #ifdef CSM_C_PTYPE_USED_STD_BR_SIGNAL
    case CSM_C_PTYPE_USED_STD_BR_SIGNAL:
    {
      lRetVal = lRegisterCSM_CBR_ApplCallback(pvHandle, bBus, dwProtocolType, pvAddressField, pvCallBackFkt);
    }
    break;
    #endif // CSM_C_PTYPE_USED_STD_BR_SIGNAL

    #ifdef CSM_C_PTYPE_MCNET_ASDT_STD
    case CSM_C_PTYPE_MCNET_ASDT_STD:
    {
      // not supported yet
      lRetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_INVALID_PROTOCOL_TYPE);
    }
    #endif

    #ifdef CSM_C_PTYPE_MCNET_USDT
    case CSM_C_PTYPE_MCNET_USDT:
    {
      // not supported yet
      lRetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_INVALID_PROTOCOL_TYPE);
    }
    #endif

    #ifdef CSM_C_PTYPE_VW_BAP
    case CSM_C_PTYPE_VW_BAP:
    {
      lRetVal = lRegisterCSM_BAP_lApplCallbackInit(pvHandle, bBus, dwProtocolType, pvAddressField, pvCallBackFkt);
    }
    break;
    #endif // CSM_C_PTYPE_VW_BAP

    #ifdef CSM_C_PTYPE_PAG_OSEK_NWM
    case CSM_C_PTYPE_PAG_OSEK_NWM:
    {
      lRetVal = lRegisterCSM_OSEK_lApplCallbackInit(pvHandle, bBus, dwProtocolType, pvAddressField, pvCallBackFkt);
    }
    break;
    #endif // CSM_C_PTYPE_PAG_OSEK_NWM

    #ifdef CSM_C_PTYPE_RN_MPDT_C
    case CSM_C_PTYPE_RN_MPDT_C:
    case CSM_C_PTYPE_RN_MPDT_D:
    {
      lRetVal = lRegisterCSM_MPDT_C_ApplCallback(pvHandle, bBus, dwProtocolType, pvAddressField, pvCallBackFkt);
    }
    break;
    #endif // CSM_C_PTYPE_RN_MPDT_C

  #ifdef CSM_C_PTYPE_J1939
      case CSM_C_PTYPE_J1939:
      {
        lRetVal = lRegisterCSM_J1939_ApplCallback(pvHandle, bBus, dwProtocolType, pvAddressField, pvCallBackFkt);
      }
      break;
    #endif /* CSM_C_PTYPE_J1939 */

    #ifdef CSM_C_PTYPE_RN_OSEKI_NWM
    case CSM_C_PTYPE_RN_OSEKI_NWM:
    {
      lRetVal = lRegisterCSM_OSEKI_lApplCallbackInit(pvHandle, bBus, dwProtocolType, pvAddressField, pvCallBackFkt);
    }
    break;
    #endif // CSM_C_PTYPE_RN_OSEKI_NWM

    default:
    {
      lRetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_INVALID_PROTOCOL_TYPE);
    }
    break;
  }

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR1(("CSMAL_lRegisterAppl - RetVal:0x%X", lRetVal));
  #endif

  return lRetVal;
}

/*******************************************************************************
 * function    CSMAL_lUnregisterAppl
 * \doxydocu
 * \brief      Depend on pvHandle all application callback message types entries
 *             are switched to CSM_ACCESS_NoEvent and initialized.
 *
 * \param[in]  pvHandle
 *
 * \return     Always CSM_C_NO_ERROR
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview
 * \history_begin
 * \history_end
 ******************************************************************************/
LONG CSMAL_lUnregisterAppl(void * pvHandle)
{
  pCSMHandle pHandle = (pCSMHandle) pvHandle;

  LONG lCount;
  INT  iAddrFieldIndex;

  // Search in communication confirmation table for handle relevant entries
  for(lCount = 0; lCount < CSMAL_MAX_COMMUNICATIONCON_NOTIFICATIONS; lCount++)
  {
    if (rCommunicationConNotificationTable.arCommunicationConNotifications[lCount].rMsgHeader.pvAccessUsrHandle == pHandle)
    {
      CSMAL_lLock(ghApplNotifyTableLock);
      rCommunicationConNotificationTable.arCommunicationConNotifications[lCount].rMsgHeader.enMsgType   = CSM_ACCESS_NoEvent;
      rCommunicationConNotificationTable.arCommunicationConNotifications[lCount].rMsgHeader.pvAccessUsrHandle    = NULL;
      rCommunicationConNotificationTable.arCommunicationConNotifications[lCount].vPFNTPCommunicationCon = NULL;
      rCommunicationConNotificationTable.arCommunicationConNotifications[lCount].bBus                   = 0;
      rCommunicationConNotificationTable.arCommunicationConNotifications[lCount].dwProtocolType         = 0;
      for (iAddrFieldIndex = 0; iAddrFieldIndex < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; iAddrFieldIndex++)
      {
        rCommunicationConNotificationTable.arCommunicationConNotifications[lCount].abAddressField[iAddrFieldIndex] = 0;
      }
      rCommunicationConNotificationTable.arCommunicationConNotifications[lCount].bConnectState          = 0;
      rCommunicationConNotificationTable.arCommunicationConNotifications[lCount].wAppId                 = 0;
      CSMAL_lUnLock(ghApplNotifyTableLock);
    }
  }

  // Search in communication indication table for handle relevant entries
  for(lCount = 0; lCount < CSMAL_MAX_COMMUNICATIONIND_NOTIFICATIONS; lCount++)
  {
    if (rCommunicationIndNotificationTable.arCommunicationIndNotifications[lCount].rMsgHeader.pvAccessUsrHandle == pHandle)
    {
      CSMAL_lLock(ghApplNotifyTableLock);
      rCommunicationIndNotificationTable.arCommunicationIndNotifications[lCount].rMsgHeader.enMsgType   = CSM_ACCESS_NoEvent;
      rCommunicationIndNotificationTable.arCommunicationIndNotifications[lCount].rMsgHeader.pvAccessUsrHandle    = NULL;
      rCommunicationIndNotificationTable.arCommunicationIndNotifications[lCount].vPFNTPCommunicationInd = NULL;
      rCommunicationIndNotificationTable.arCommunicationIndNotifications[lCount].bBus                   = 0;
      rCommunicationIndNotificationTable.arCommunicationIndNotifications[lCount].dwProtocolType         = 0;
      for (iAddrFieldIndex = 0; iAddrFieldIndex < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; iAddrFieldIndex++)
      {
        rCommunicationIndNotificationTable.arCommunicationIndNotifications[lCount].abAddressField[iAddrFieldIndex] = 0;
      }
      rCommunicationIndNotificationTable.arCommunicationIndNotifications[lCount].bConnectState          = 0;
      rCommunicationIndNotificationTable.arCommunicationIndNotifications[lCount].wAppId                 = 0;
      CSMAL_lUnLock(ghApplNotifyTableLock);
    }
  }

  #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
  {
    // Search in data confirmation table for handle relevant entries
    for(lCount = 0; lCount < CSMAL_MAX_DATACON_NOTIFICATIONS; lCount++)
    {
      if (rDataConNotificationTable.arDataConNotifications[lCount].rMsgHeader.pvAccessUsrHandle == pHandle)
      {
        CSMAL_lLock(ghApplNotifyTableLock);
        rDataConNotificationTable.arDataConNotifications[lCount].rMsgHeader.enMsgType = CSM_ACCESS_NoEvent;
        rDataConNotificationTable.arDataConNotifications[lCount].rMsgHeader.pvAccessUsrHandle  = NULL;
        rDataConNotificationTable.arDataConNotifications[lCount].vPFNTPDataCon        = NULL;
        rDataConNotificationTable.arDataConNotifications[lCount].bBus                 = 0;
        rDataConNotificationTable.arDataConNotifications[lCount].dwProtocolType       = 0;
        for (iAddrFieldIndex = 0; iAddrFieldIndex < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; iAddrFieldIndex++)
        {
          rDataConNotificationTable.arDataConNotifications[lCount].abAddressField[iAddrFieldIndex] = 0;
        }
        rDataConNotificationTable.arDataConNotifications[lCount].bTransferResult      = 0;
        CSMAL_lUnLock(ghApplNotifyTableLock);
      }
    }

    // data indication table
    for(lCount = 0; lCount < CSMAL_MAX_DATAIND_NOTIFICATIONS; lCount++)
    {
      if (rDataIndNotificationTable.arDataIndNotifications[lCount].rMsgHeader.pvAccessUsrHandle == pHandle)
      {
        CSMAL_lLock(ghApplNotifyTableLock);
        rDataIndNotificationTable.arDataIndNotifications[lCount].rMsgHeader.enMsgType = CSM_ACCESS_NoEvent;
        rDataIndNotificationTable.arDataIndNotifications[lCount].rMsgHeader.pvAccessUsrHandle  = NULL;
        rDataIndNotificationTable.arDataIndNotifications[lCount].vPFNTPDataInd        = NULL;
        rDataIndNotificationTable.arDataIndNotifications[lCount].bBus                 = 0;
        rDataIndNotificationTable.arDataIndNotifications[lCount].dwProtocolType       = 0;
        //rDataIndNotificationTable.arDataIndNotifications[lCount].pbData               = NULL;
        for (iAddrFieldIndex = 0; iAddrFieldIndex < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; iAddrFieldIndex++)
        {
          rDataIndNotificationTable.arDataIndNotifications[lCount].abAddressField[iAddrFieldIndex] = 0;
        }
        rDataIndNotificationTable.arDataIndNotifications[lCount].wDataLength          = 0;
        CSMAL_lUnLock(ghApplNotifyTableLock);
      }
    }

    #if (CSM_S_ISO_TP == CSM_C_F_ON)
    {
      #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
      // First frame data indication table
      for(lCount = 0; lCount < CSMAL_MAX_DATAERRORIND_NOTIFICATIONS; lCount++)
      {
        if (rDataIndFFNotificationTable.arDataIndFFNotifications[lCount].rMsgHeader.pvAccessUsrHandle == pHandle)
        {
          CSMAL_lLock(ghApplNotifyTableLock);
          rDataIndFFNotificationTable.arDataIndFFNotifications[lCount].rMsgHeader.enMsgType = CSM_ACCESS_NoEvent;
          rDataIndFFNotificationTable.arDataIndFFNotifications[lCount].rMsgHeader.pvAccessUsrHandle  = NULL;
          rDataIndFFNotificationTable.arDataIndFFNotifications[lCount].vPFNTPDataIndFF        = NULL;
          rDataIndFFNotificationTable.arDataIndFFNotifications[lCount].bBus                 = 0;
          rDataIndFFNotificationTable.arDataIndFFNotifications[lCount].dwProtocolType       = 0;
          for (iAddrFieldIndex = 0; iAddrFieldIndex < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; iAddrFieldIndex++)
          {
            rDataIndFFNotificationTable.arDataIndFFNotifications[lCount].abAddressField[iAddrFieldIndex] = 0;
          }
          rDataIndFFNotificationTable.arDataIndFFNotifications[lCount].wDataLength          = 0;
          CSMAL_lUnLock(ghApplNotifyTableLock);
        }
      }
      #endif // CSM_S_ITP_FF_INDICATION_AVAILABLE
    }
    #endif // CSM_S_ISO_TP

    #if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
    {
      // data error indication table
      for(lCount = 0; lCount < CSMAL_MAX_DATAERRORIND_NOTIFICATIONS; lCount++)
      {
        if (rDataErrorIndNotificationTable.arDataErrorIndNotifications[lCount].rMsgHeader.pvAccessUsrHandle == pHandle)
        {
          CSMAL_lLock(ghApplNotifyTableLock);
          rDataErrorIndNotificationTable.arDataErrorIndNotifications[lCount].rMsgHeader.enMsgType = CSM_ACCESS_NoEvent;
          rDataErrorIndNotificationTable.arDataErrorIndNotifications[lCount].rMsgHeader.pvAccessUsrHandle  = NULL;
          rDataErrorIndNotificationTable.arDataErrorIndNotifications[lCount].vPFNTPDataErrorInd        = NULL;
          rDataErrorIndNotificationTable.arDataErrorIndNotifications[lCount].bBus                 = 0;
          rDataErrorIndNotificationTable.arDataErrorIndNotifications[lCount].dwProtocolType       = 0;
          for (iAddrFieldIndex = 0; iAddrFieldIndex < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; iAddrFieldIndex++)
          {
            rDataErrorIndNotificationTable.arDataErrorIndNotifications[lCount].abAddressField[iAddrFieldIndex] = 0;
          }
          rDataErrorIndNotificationTable.arDataErrorIndNotifications[lCount].dwDataErrorCode          = 0;
          CSMAL_lUnLock(ghApplNotifyTableLock);
        }
      }
    }
    #endif // CSM_S_DATA_ERROR_IND_AVAILABLE
  }
  #endif // CSM_S_CNP_AVAILABLE

  return CSM_C_NO_ERROR;
}

/*******************************************************************************
 * function    CSMAL_lRegisterSignal
 * \doxydocu
 * \brief      Adds a signal notification by
 *             1. putting the given data into the signal notification table and
 *             2. initialize the signal indication handler as CSM sig ind
 *                callback. The table element pointer is used as handle for the
 *                CSM callback.
 *
 * \param[in]  pvAccessUsrHandle
 *             NULL for kernel mode applications or
 *             CSM Access Layer Handle (generated by CSM_lApplCallbackPreInit())
 *
 * \param[in]  dwSignalId
 *             ID of the signal that shall be notified
 *
 * \param[in]  vPFNCBRSignalInd
 *             address of the callback function that shall be called for notify
 *
 * \return     Error code
 *              >= 0 successfull, <0 error
 *
 * \access     user mode application
 * \reentrant  ? (only if CSM_lSignalCallbackInit() is reentrant)
 *
 * \lastreview 03.09.08 VTeam-Khler
 * \history_begin
 * 03.09.08  VTeam-Khler
 * - Return value initialization changed.
 * 20.09.06 3SOFT Schubart
 * Initial Revision.
 * \history_end
 ******************************************************************************/
LONG CSMAL_lRegisterSignal(void * pvAccessUsrHandle, DWORD dwSignalId, tCSM_PFN_CBR_SIGNAL_IND vPFNCBRSignalInd)
{
  LONG lRetVal = CSM_C_NO_ERROR;
  pSignalIndMsg phCSM = NULL;

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR1(("CSMAL_lRegisterSignal - AccessUsrHandle:0x%X  SignalId:0x%X  PFNCBRSignalInd:0x%X",
                  pvAccessUsrHandle,
                  dwSignalId,
                  vPFNCBRSignalInd));
  #endif

  // add data to notification table
  if (NULL != pvAccessUsrHandle)
  {
    phCSM = CSMAL_lAddSignalIndNotification(pvAccessUsrHandle, dwSignalId, vPFNCBRSignalInd);

    if (NULL != phCSM)
    {
      // now register signal at CSM
      lRetVal = CSM_lSignalCallbackInit((void *)phCSM, dwSignalId, CSMAL_vSignalInd_Handler);
    }
    else
    {
      lRetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_NULL_POINTER);
      #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_ERR(("CSMAL_lRegisterSignal - ERROR - phCSM is NULL !!!"));
      #endif
    }
  }
  else
  {
    lRetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_NULL_POINTER);
    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_ERR(("CSMAL_lRegisterSignal - ERROR - pvAccessUsrHandle is NULL !!!"));
    #endif
  }

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR1(("CSMAL_lRegisterSignal - lRetVal:0x%X  phCSM:0x%X", lRetVal, phCSM));
  #endif

  return lRetVal;
}

/*******************************************************************************
 * function    CSMAL_lUnregisterSignal
 * \doxydocu
 * \brief      Depend on pvHandle the signal callback message type entry of the
 *             given signal ID is switched to CSM_ACCESS_NoEvent and initialized.
 *
 * \param[in]  pvHandle
 *
 * \param[in]  dwSignalId
 *             ID of the signal that shall be unregisterd
 *
 * \return     Always CSM_C_NO_ERROR
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview
 * \history_begin
 * \history_end
 ******************************************************************************/
LONG CSMAL_lUnregisterSignal(void * pvHandle, DWORD dwSignalId)
{
  LONG lRetVal = CSM_C_NO_ERROR;
  pCSMHandle pHandle = (pCSMHandle) pvHandle;

  LONG lCount;

  // Search in signal indication table for handle relevant entries
  for(lCount = 0; lCount < CSMAL_MAX_SIGNALIND_NOTIFICATIONS; lCount++)
  {
    if ( (rSignalIndNotificationTable.arSignalIndNotifications[lCount].rMsgHeader.pvAccessUsrHandle == pHandle)
         &&
         (rSignalIndNotificationTable.arSignalIndNotifications[lCount].dwSignalId == dwSignalId)
       )
    {
      CSMAL_lLock(ghSignalNotifyTableLock);
      rSignalIndNotificationTable.arSignalIndNotifications[lCount].rMsgHeader.enMsgType  = CSM_ACCESS_NoEvent;
      rSignalIndNotificationTable.arSignalIndNotifications[lCount].rMsgHeader.pvAccessUsrHandle   = NULL;
      rSignalIndNotificationTable.arSignalIndNotifications[lCount].vPFNCBRSignalInd      = NULL;
      rSignalIndNotificationTable.arSignalIndNotifications[lCount].dwSignalId            = 0;
      rSignalIndNotificationTable.arSignalIndNotifications[lCount].dwState               = 0;
      CSMAL_lUnLock(ghSignalNotifyTableLock);
    }
  }
  return lRetVal;
}

/*******************************************************************************
 * function    CSMAL_lUnregisterAllSignals
 * \doxydocu
 * \brief      Depend on pvHandle all signal callback message types entries
 *             are switched to CSM_ACCESS_NoEvent and initialized.
 *
 * \param[in]  pvHandle
 *
 * \return     Always CSM_C_NO_ERROR
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview
 * \history_begin
 * \history_end
 ******************************************************************************/
LONG CSMAL_lUnregisterAllSignals(void * pvHandle)
{
  LONG lRetVal = CSM_C_NO_ERROR;
  pCSMHandle pHandle = (pCSMHandle) pvHandle;

  LONG lCount;

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR2(("CSMAL_lUnregisterAllSignals - AccessUsrHandle:0x%X", pvHandle));
  #endif

  CSMAL_lLock(ghSignalNotifyTableLock);

  // Search in signal indication table for handle relevant entries
  for(lCount = 0; lCount < CSMAL_MAX_SIGNALIND_NOTIFICATIONS; lCount++)
  {
    if (rSignalIndNotificationTable.arSignalIndNotifications[lCount].rMsgHeader.pvAccessUsrHandle == pHandle)
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_USR4(("CSMAL_lUnregisterAllSignals - TableElement:0x%X", &rSignalIndNotificationTable.arSignalIndNotifications[lCount]));
      ETG_TRACE_USR4(("CSMAL_lUnregisterAllSignals - TableEntry:0x%X - Signal Removed", lCount));
      #endif

      rSignalIndNotificationTable.arSignalIndNotifications[lCount].rMsgHeader.enMsgType           = CSM_ACCESS_NoEvent;
      rSignalIndNotificationTable.arSignalIndNotifications[lCount].rMsgHeader.pvAccessUsrHandle   = NULL;
      rSignalIndNotificationTable.arSignalIndNotifications[lCount].vPFNCBRSignalInd               = NULL;
      rSignalIndNotificationTable.arSignalIndNotifications[lCount].dwSignalId                     = 0;
      rSignalIndNotificationTable.arSignalIndNotifications[lCount].dwState                        = 0;
    }
  }

  CSMAL_lUnLock(ghSignalNotifyTableLock);
  return lRetVal;
}

/*******************************************************************************
 * function    CSMAL_lRegisterBusErrorInd
 * \doxydocu
 * \brief      Adds a bus error indication by
 *             1. putting the given data into the bus error indication table and
 *             2. initialize the bus error indication handler as CSM
 *                callback. The table element pointer is used as handle for the
 *                CSM callback.
 *
 * \param[in]  pvAccessUsrHandle
 *             NULL for kernel mode applications or
 *             CSM Access Layer Handle (generated by CSM_lApplCallbackPreInit())
 *
 * \param[in]  vPFNBusErrorInd
 *             callback function that shall be called for notify
 *
 * \return     Error code
 *              >= 0 successfull, <0 error
 *
 * \access     user mode application
 * \reentrant  ? (only if CSM_lSignalCallbackInit() is reentrant)
 *
 * \lastreview
 * \history_begin
 *
 * 29.11.08  VTeam-Pistoor
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
#if (CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON)
LONG CSMAL_lRegisterBusErrorInd(void * pvAccessUsrHandle, tCSM_PFN_BUS_ERROR_IND vPFNBusErrorInd)
{
  LONG lRetVal = CSM_C_NO_ERROR;
  pBusErrorIndMsg phCSM = NULL;

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR1(("CSMAL_lRegisterBusErrorInd - AccessUsrHandle:0x%X  PFNBusErrorInd:0x%X",
                  pvAccessUsrHandle,
                  vPFNBusErrorInd));
  #endif

  // add data to notification table
  if(NULL != pvAccessUsrHandle)
  {
    phCSM = CSMAL_lAddBusErrorIndNotification(pvAccessUsrHandle,vPFNBusErrorInd);

    if (NULL != phCSM)
    {
      // now register callback at CSM
      lRetVal = CSM_lBusErrorIndCallbackInit((void *)phCSM, CSMAL_vBusErrorInd_Handler);
    }
    else
    {
      lRetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_NULL_POINTER);
      #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_ERR(("CSMAL_lRegisterBusErrorInd - ERROR - phCSM is NULL !!!"));
     #endif
    }
  }
  else
  {
    lRetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_NULL_POINTER);
    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_ERR(("CSMAL_lRegisterBusErrorInd - ERROR - pvAccessUsrHandle is NULL !!!"));
    #endif
  }

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR1(("CSMAL_lRegisterBusErrorInd - RetVal:0x%X  phCSM:0x%X", lRetVal, phCSM));
  #endif

  return lRetVal;
}
#endif // CSM_S_ENABLE_INDICATE_BUSERROR

/*******************************************************************************
 * function    CSMAL_vSignalInd_Handler
 * \doxydocu
 * \brief      Handler for Signal indication. Retrieves the registered
 *             callback data and sends it to the service task.
 *             This function is registered with the CSM.
 *
 * \param[in]  pvTableElement
               Pointer to signal indication table element in kernel space.
 *             The content is send as t-kernel message to the related user task.
 *
 * \param[in]  dwSignalId
 *             ID of signal which is to be notified
 *
 * \param[in]  dwSignalStatus
 *             status of the signal which is to be notified
 *
 * \return     VOID
 *
 * \access     upstream task
 * \reentrant  ?
 *
 * \lastreview 04.09.08  VTeam-Khler
 * \history_begin
 * 02.12.07  VTeam-Khler
 * - Cast (pCSMHandle) for pMsg->rMsgHeader.pvAccessUsrHandle added.
 * - Trace call with pHandle moved into if(pHandle != NULL) tree.
 * 11.10.06 3SOFT Schubart
 * adapted to new handle mechanism
 * 09.10.06 3SOFT Schubart
 * renamed from CSM_vSignalCallbackHandler to CSMAL_vSignalInd_Handler
 * 20.09.06 3SOFT Schubart
 * Initial Revision.
 * \history_end
 ******************************************************************************/
VOID CSMAL_vSignalInd_Handler(void * pvTableElement, DWORD dwSignalId, DWORD dwSignalStatus)
{
  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR4(("CSMAL_vSignalInd_Handler - TableElement:0x%X  dwSignalId:0x%X  dwSignalStatus:0x%X",
                  pvTableElement, dwSignalId, dwSignalStatus
                ));
  #endif

  // before working with a pointer, check if it is valid !!!
  if (pvTableElement != NULL)
  {
    // use the table element for access to the notification table
    pSignalIndMsg pMsg = (pSignalIndMsg) pvTableElement;
    // get the access handle from that table
    pCSMHandle pHandle = (pCSMHandle) pMsg->rMsgHeader.pvAccessUsrHandle;
    // add status for the signal, received from CSM
    pMsg->dwState = dwSignalStatus;

    if (pHandle != NULL)
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_USR4(("CSMAL_vSignalInd_Handler - Message will be sent to queue of handle 0x%08X", pHandle));
      #endif
      // send the callback data to message queue
      CSMAL_vSendMessage(pHandle, (void *)pMsg, sizeof(trSignalIndMsg));
    }
    else
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_ERR(("CSMAL_vSignalInd_Handler - ERROR - pHandle is NULL, no message sent"));
      #endif
    }
  }
  else
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_ERR(("CSMAL_vSignalInd_Handler - ERROR - pvTableElement is NULL!"));
    #endif
  }
}

/*******************************************************************************
 * function    CSM_vBusErrorInd_Handler
 * \doxydocu
 * \brief      Handler for bus error indication. Retrieves the registered
 *             callback bus error value and sends it to the service task.
 *             This function is registered with the CSM.
 *
 * \param[in]  pvEntry
 *             pointer to lookup table, to get message data
 *
 * \param[in]  bBus
 *             bus number which is to be notified
 *
 * \param[in]  wBusError
 *             Bus error value which is to be notified
 *
 * \return     VOID
 *
 * \access     upstream task
 * \reentrant  ?
 *
 * \lastreview
 * \history_begin
 *
 * 29.11.08  VTeam-Pistoot
 * Initial Revision.
 *
 * \history_end
 ******************************************************************************/
#if (CSM_S_ENABLE_INDICATE_BUSERROR == CSM_C_F_ON)
VOID CSMAL_vBusErrorInd_Handler(void *pvEntry, BYTE bBus, WORD wBusError)
{
  pCSMHandle pHandle;

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR2(("CSMAL_vBusErrorInd_Handler - pvEntry:0x%X  Bus:%x  BusError0x%X",
                  pvEntry,
                  ETG_CENUM(tCSM_BUS_NUMBER,            (tU8)  bBus),
                  wBusError));
  #endif

  if (pvEntry != NULL)
  {
    pBusErrorIndMsg pMsg = pvEntry;
    //&rBusErrorIndNotificationTable.arBusErrorIndNotifications[lIndexBusErrorInd];

    // fill in new data
    pMsg->bBus           = bBus;
    pMsg->wDataErrorCode = wBusError;
    pHandle              = (pCSMHandle)pMsg->rMsgHeader.pvAccessUsrHandle;

    if(pHandle != NULL)
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_USR3(("CSMAL_vBusErrorInd_Handler - Message will be sent to queue of handle 0x%08X",
                      pHandle));
      #endif

      // send the callback data to message queue
      CSMAL_vSendMessage(pHandle, (void *)pMsg, sizeof(trBusErrorIndMsg));
    }
    else
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_ERR(("CSMAL_vBusErrorInd_Handler - ERROR - pHandle is NULL, no message sent"));
      #endif
    }
  }
  else
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_ERR(("CSMAL_vBusErrorInd_Handler - ERROR - pvHandle is NULL, no message sent"));
    #endif
  }
}
#endif // CSM_S_ENABLE_INDICATE_BUSERROR

/*******************************************************************************
 * function    CSMAL_vCommunicationCon_Handler
 * \doxydocu
 * \brief      Handler for communication confirmation. Retrieves the registered
 *             callback data and sends it to the service task.
 *             This function is registered with the CSM.
 *
 * \param[in]  pvEntry
 *             pointer to lookup table, to get message data
 *
 * \param[in]  bBus
 *             CAN bus
 *
 * \param[in]  dwProtocolType
 *             protocol type
 *
 * \param[in]  pvAddressField
 *             addressing, depends on protocol
 *
 * \param[in]  bConnectState
 *             The CAN bus/conection state
 *
 * \param[in]  wAppId
 *             Application ID
 *
 * \return     VOID
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview  09.09.08 CM-AI/VTeam-Brunotte
 * \history_begin
 * 26.03.09  VTeam-Khler
 * - Elements of pMsg are copied only if pHandle is not NULL.
 * 20.09.07  VTream Brunotte
 * - Check adressfield pointer for NULL.
 * 17.07.07  VTeam-Khler
 * - Copy content of address field pointer instead of pointer itself.
 * 09.10.06 3SOFT Schubart
 * Initial Revision.
 * \history_end
 ******************************************************************************/
VOID CSMAL_vCommunicationCon_Handler(void * pvEntry,
                                     BYTE bBus,
                                     DWORD dwProtocolType,
                                     const VOID* pvAddressField,
                                     BYTE bConnectState,
                                     WORD wAppId)
{
  LONG lIndexCommunicationCon = -1;
  BYTE bAddr0 = *((((BYTE *)pvAddressField)+0));
  BYTE bAddr1 = *((((BYTE *)pvAddressField)+1));

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR2(("CSMAL_vCommunicationCon_Handler - pvEntry:0x%X  Bus%x  Prot:%x  AddrField:0x%02X%02X",
                  pvEntry,
                  ETG_CENUM(tCSM_BUS_NUMBER,               (tU8)  bBus),
                  ETG_CENUM(tCSM_PROTOCOL_TYPE,            (tU32) dwProtocolType),
                  (tU8)  bAddr0,
                  (tU8)  bAddr1
                ));
  #endif

  // handle is pointer into lookup table
  pApplNotificationLookupEntry pEntry = (pApplNotificationLookupEntry) pvEntry;

  if (pEntry != NULL)
  {
    lIndexCommunicationCon = pEntry->lIndexCommunicationCon;
  }

  if ((lIndexCommunicationCon >= 0) && (lIndexCommunicationCon < CSMAL_MAX_COMMUNICATIONCON_NOTIFICATIONS))
  {
    // lookup message in corresponding notification table
    pCSMHandle pHandle;
    pCommunicationConMsg pMsg = &rCommunicationConNotificationTable.arCommunicationConNotifications[lIndexCommunicationCon];

    pHandle = (pCSMHandle)pMsg->rMsgHeader.pvAccessUsrHandle;

    if (pHandle != NULL)
    {
      INT  iAddrFieldIndex;
      BYTE *pbAddressField = (BYTE *) pvAddressField;

      // fill in new data
      pMsg->bBus           = bBus;
      pMsg->dwProtocolType = dwProtocolType;
      for (iAddrFieldIndex = 0; iAddrFieldIndex < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; iAddrFieldIndex++)
      {
        if (pbAddressField != NULL)
        {
          pMsg->abAddressField[iAddrFieldIndex] = *pbAddressField++;
        }
        else
        {
          pMsg->abAddressField[iAddrFieldIndex] = 0;
        }
      }
      pMsg->bConnectState  = bConnectState;
      pMsg->wAppId         = wAppId;

      #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_USR3(("CSMAL_vCommunicationCon_Handler - Message will be sent to queue of handle 0x%08X",
                      pHandle));
      #endif

      // send the callback data to message queue
      CSMAL_vSendMessage(pHandle, (void *)pMsg, sizeof(trCommunicationConMsg));
    }
    else
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_ERR(("CSMAL_vCommunicationCon_Handler - ERROR - pHandle is NULL, no message sent"));
      #endif
    }
  }
  else
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_ERR(("CSMAL_vCommunicationCon_Handler - ERROR - Table Index out of bounds:0x%X", lIndexCommunicationCon));
    #endif
  }
}

/*******************************************************************************
 * function    CSMAL_vCommunicationInd_Handler
 * \doxydocu
 * \brief      Handler for communication indication. Retrieves the registered
 *             callback data and sends it to the service task.
 *             This function is registered with the CSM.
 *
 * \param[in]  pvEntry
 *             pointer to lookup table, to get message data
 *
 * \param[in]  bBus
 *             CAN bus number
 *
 * \param[in]  dwProtocolType
 *             protocol type
 *
 * \param[in]  pvAddressField
 *             addressing, depends on protocol
 *
 * \param[in]  bConnectState
 *             The CAN bus state
 *
 * \param[in]  wAppId
 *             Application ID
 *
 * \return     VOID
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview 09.09.08 CM-AI/VTeam-Brunotte
 * \history_begin
 * 20.09.07 VTeam-Brunotte
 * - Checks adressfiled pointer for NULL. Avoid TLB exceptions
 * 17.07.07  VTeam-Khler
 * - Copy content of address field pointer instead of pointer itself.
 * 09.10.06 3SOFT Schubart
 * Initial Revision.
 * \history_end
 ******************************************************************************/
VOID CSMAL_vCommunicationInd_Handler(void * pvEntry, BYTE bBus, DWORD dwProtocolType, const VOID* pvAddressField, BYTE bConnectState, WORD wAppId)
{
  LONG lIndexCommunicationInd = -1;
  pCSMHandle pHandle;
  BYTE bAddr0 = *((((BYTE *)pvAddressField)+0));
  BYTE bAddr1 = *((((BYTE *)pvAddressField)+1));

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR2(("CSMAL_vCommunicationInd_Handler - pvEntry:0x%X  Bus:%x  Prot:%x  AddrField:0x%02X%02X",
                   pvEntry,
                   ETG_CENUM(tCSM_BUS_NUMBER,               (tU8)  bBus),
                   ETG_CENUM(tCSM_PROTOCOL_TYPE,            (tU32) dwProtocolType),
                   (tU8)  bAddr0,
                   (tU8)  bAddr1
                ));
  #endif

  // handle is pointer into lookup table
  pApplNotificationLookupEntry pEntry = (pApplNotificationLookupEntry) pvEntry;

  if (pEntry != NULL)
  {
    lIndexCommunicationInd = pEntry->lIndexCommunicationInd;
  }

  if ((lIndexCommunicationInd >= 0) && (lIndexCommunicationInd < CSMAL_MAX_COMMUNICATIONIND_NOTIFICATIONS))
  {
    // lookup message in corresponding notification table
    pCommunicationIndMsg pMsg = &rCommunicationIndNotificationTable.arCommunicationIndNotifications[lIndexCommunicationInd];
    INT  iAddrFieldIndex;
    BYTE *pbAddressField = (BYTE *) pvAddressField;

    // fill in new data
    pMsg->bBus           = bBus;
    pMsg->dwProtocolType = dwProtocolType;
    for (iAddrFieldIndex = 0; iAddrFieldIndex < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; iAddrFieldIndex++)
    {
      if (pbAddressField != NULL)
      {
        pMsg->abAddressField[iAddrFieldIndex] = *pbAddressField++;
      }
      else
      {
        pMsg->abAddressField[iAddrFieldIndex] = 0;
      }
    }
    pMsg->bConnectState  = bConnectState;
    pMsg->wAppId         = wAppId;

    pHandle = (pCSMHandle)pMsg->rMsgHeader.pvAccessUsrHandle;

    if (pHandle != NULL)
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_USR3(("CSMAL_vCommunicationInd_Handler - Message will be sent to queue of handle 0x%08X",
                      pHandle));
      #endif

      // send the callback data to message queue
      CSMAL_vSendMessage(pHandle, (void *)pMsg, sizeof(trCommunicationIndMsg));
    }
    else
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_ERR(("CSMAL_vCommunicationInd_Handler - ERROR - pHandle is NULL, no message sent"));
      #endif
    }
  }
  else
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_ERR(("CSMAL_vCommunicationInd_Handler - ERROR - Table Index out of bounds:0x%X", lIndexCommunicationInd));
    #endif
  }
}

/*******************************************************************************
 * function    CSMAL_vDataCon_Handler
 * \doxydocu
 * \brief      Handler for data confirmation. Retrieves the registered
 *             callback data and sends it to the service task.
 *             This function is registered with the CSM.
 *
 * \param[in]  pvEntry
 *             pointer to lookup table, to get message data
 *
 * \param[in]  dwProtocolType
 *             protocol type
 *
 * \param[in]  pvAddressField
 *             addressing, depends on protocol
 *
 * \param[in]  bTransferResult
 *             TODO
 *
 * \return     VOID
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview  09.09.08 CM-AI/VTeam-Brunotte
 * \history_begin
  * 20.09.07 VTeam-Brunotte
 * - Checks adressfiled pointer for NULL. Avoid TLB exceptions
 * 17.07.07  VTeam-Khler
 * - Copy content of address field pointer instead of pointer itself.
 * 09.10.06 3SOFT Schubart
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
VOID CSMAL_vDataCon_Handler(void * pvEntry, DWORD dwProtocolType, const VOID* pvAddressField, BYTE bTransferResult)
{
  LONG lIndexDataCon = -1;
  LONG lIndexDDataCon = -1;
  pCSMHandle pHandle;
  BYTE bAddr0 = *((((BYTE *)pvAddressField)+0));
  BYTE bAddr1 = *((((BYTE *)pvAddressField)+1));

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR2(("CSMAL_vDataCon_Handler - LookupEntry:0x%X  Prot:%x  AddrField:0x%02X%02X  TXResult:0x%X",
                  pvEntry,
                  ETG_CENUM(tCSM_PROTOCOL_TYPE,         (tU32) dwProtocolType),
                  (tU8)  bAddr0,
                  (tU8)  bAddr1,
                  bTransferResult));
  #endif

  // cast the entry pointer to the appropriate struct for accessing our lookup table
  pApplNotificationLookupEntry pEntry = (pApplNotificationLookupEntry) pvEntry;

  if (pEntry != NULL)
  {
    // get the correct entry index to the confirmation lookup table
    lIndexDataCon = pEntry->lIndexDataCon;
    #if (CSM_S_MPDT == CSM_C_F_ON)
    lIndexDDataCon = pEntry->lIndexDDataCon;
    #endif
  }

  // check if standard DataCon index is valid
  if ((lIndexDataCon >= 0) && (lIndexDataCon < CSMAL_MAX_DATACON_NOTIFICATIONS))
  {
    INT  iAddrFieldIndex;
    BYTE *pbAddressField = (BYTE *) pvAddressField;

    // lookup data in corresponding notification table
    pDataConMsg pMsg1 = &rDataConNotificationTable.arDataConNotifications[lIndexDataCon];
    #if (CSM_S_MPDT == CSM_C_F_ON)
    pDataConMsg pMsg2 = pMsg1;
    #endif
    pDataConMsg pMsg = pMsg1;

    // for MPDT also check if second notification index is valid
    #if (CSM_S_MPDT == CSM_C_F_ON)
    if ((lIndexDDataCon >= 0) && (lIndexDataCon < CSMAL_MAX_DATACON_NOTIFICATIONS))
    {
      // lookup data in corresponding notification table and overwrite pointer declared above
      pMsg2 = &rDataConNotificationTable.arDataConNotifications[lIndexDDataCon];
    }
    #endif

    #if (CSM_S_MPDT == CSM_C_F_ON)
    // in case of MPDT support we now must check the protocol type given by CSM
    if (dwProtocolType == pMsg2->dwProtocolType)
    {
      // D-Data confirmation by CSM, so we use the corresponding second notification entry
      pMsg = pMsg2;
      #if (CSM_S_TRACE == CSM_C_F_ON)
      if (pMsg1 == pMsg2)
      {
        ETG_TRACE_USR1(("CSMAL_vDataCon_Handler - Primary Callback for Protocol:%x chosen",
                        ETG_CENUM(tCSM_PROTOCOL_TYPE,         (tU32) dwProtocolType)
                      ));
      }
      else
      {
        ETG_TRACE_USR1(("CSMAL_vDataCon_Handler - Secondary Callback for Protocol:%x chosen",
                        ETG_CENUM(tCSM_PROTOCOL_TYPE,         (tU32) dwProtocolType)
                      ));
      }
      #endif
    }
    else
    {
      // it could be any other protocol except MPDT or MPDT C-DATA
      // in this case the standard notification entry is used
      pMsg = pMsg1;
      #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_USR1(("CSMAL_vDataCon_Handler - Primary Callback for Protocol:%x chosen",
                      ETG_CENUM(tCSM_PROTOCOL_TYPE,         (tU32) dwProtocolType)
                    ));
      #endif
    }
    #else
    // No MPDT support -> the standard notification can be chosen directly to go on with
    pMsg = pMsg1;
    #endif

    // warum muss der Protocol Typ nochmal in die Tabelle geschrieben werden ?? steht eigentlich schon drin !!!
    pMsg->dwProtocolType = dwProtocolType;
    // fill address field (dies wurde beim ApplCallbackInit nicht gespeichert !!!)
    for (iAddrFieldIndex = 0; iAddrFieldIndex < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; iAddrFieldIndex++)
    {
      if (pbAddressField != NULL)
      {
        pMsg->abAddressField[iAddrFieldIndex] = *pbAddressField++;
      }
      else
      {
        pMsg->abAddressField[iAddrFieldIndex] = 0;
      }
    }
    // fill transfer result
    pMsg->bTransferResult = bTransferResult;
    // get handle from table (this identifies the process/queue)
    pHandle = (pCSMHandle) pMsg->rMsgHeader.pvAccessUsrHandle;

    if (pHandle != NULL)
    {
      LONG lMsgQElemSize = pHandle->lMQueueElemSize;
      LONG lMsgSize = sizeof(trDataConMsg);

      #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_USR3(("CSMAL_vDataCon_Handler - Message will be sent to queue of handle 0x%08x", pHandle));
      #endif

      #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_USR4(("CSMAL_vDataCon_Handler - MsgSize:%d  MsgQueueElemSize:%d", lMsgSize, lMsgQElemSize));
      #endif

      // send the callback data to message queue
      CSMAL_vSendMessage(pHandle, (void *)pMsg, lMsgSize);
    }
    else
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_ERR(("CSMAL_vDataCon_Handler - ERROR - pHandle is NULL, no message sent"));
      #endif
    }
  }
  else
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_ERR(("CSMAL_vDataCon_Handler - ERROR - Index out of bounds:%d", lIndexDataCon));
    #endif
  }
}

/*******************************************************************************
 * function    CSMAL_vDataInd_Handler
 * \doxydocu
 * \brief      Handler for data indication. Retrieves the registered
 *             callback data and sends it to the service task.
 *             This function is registered with the CSM.
 *
 * \param[in]  pvEntry
 *             pointer to lookup table, to get message data
 *
 * \param[in]  dwProtocolType
 *             protocol type
 *
 * \param[in]  pvAddressField
 *             addressing, depends on protocol
 *
 * \param[in]  pbData
 *             pointer to data
 *
 * \param[in]  wDataLength
 *             length of data
 *
 * \return     VOID
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview 09.09.08 CM-AI/VTeam-Brunotte
 * \history_begin
 * 20.09.07 VTeam-Brunotte
 * - Checks adressfiled pointer for NULL. Avoid TLB exceptions
 * 17.07.07  VTeam-Khler
 * - Copy content of address field pointer instead of pointer itself.
 * 08.06.07 VTeam-Khler
 * memcpy replaced by CSM_vMemCopy.
 * 09.10.06 3SOFT Schubart
 * Initial Revision.
 * \history_end
 ******************************************************************************/
VOID CSMAL_vDataInd_Handler(void * pvEntry, DWORD dwProtocolType, const VOID* pvAddressField, BYTE* pbData, WORD wDataLength)
{
  LONG lIndexDataInd = -1;
  LONG lIndexDDataInd = -1;
  pCSMHandle pHandle;
  BYTE bAddr0 = *((((BYTE *)pvAddressField)+0));
  BYTE bAddr1 = *((((BYTE *)pvAddressField)+1));

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR2(("CSMAL_vDataInd_Handler - pvEntry:0x%X  Prot:%x  AddrField:0x%02X%02X  pData:0x%X  DataLen:0x%X",
                  pvEntry,
                  ETG_CENUM(tCSM_PROTOCOL_TYPE,         (tU32) dwProtocolType),
                  (tU8)  bAddr0,
                  (tU8)  bAddr1,
                  pbData,
                  wDataLength));
  #endif

  // data length must not be bigger than maximum payload
  // BKA2HI: but keep in mind that the updated access_usr interface allows now, to give the payload size as a parameter,
  // so that the real service message queue size might be much smaller than CSMAL_MAX_DATA_PAYLOAD.
  // This following check only helps if default values are used (parameterless interface call in vApplCallbackPreInit).
  // the real message queue size is to be checked further down again, if the handle is available.
  if (wDataLength <= CSMAL_MAX_DATA_PAYLOAD)
  {
    // cast the entry pointer to the appropriate struct for accessing our lookup table
    pApplNotificationLookupEntry pEntry = (pApplNotificationLookupEntry) pvEntry;

    if (pEntry != NULL)
    {
      lIndexDataInd = pEntry->lIndexDataInd;
      #if (CSM_S_MPDT == CSM_C_F_ON)
      lIndexDDataInd = pEntry->lIndexDDataInd;
      #endif
    }

    // check if standard DataInd index is valid
    if ((lIndexDataInd >= 0) && (lIndexDataInd < CSMAL_MAX_DATAIND_NOTIFICATIONS))
    {
      INT  iAddrFieldIndex;
      BYTE *pbAddressField = (BYTE *) pvAddressField;
      // lookup data in corresponding notification table
      pDataIndMsg pMsg1 = &rDataIndNotificationTable.arDataIndNotifications[lIndexDataInd];
      #if (CSM_S_MPDT == CSM_C_F_ON)
      pDataIndMsg pMsg2 = pMsg1;
      #endif
      pDataIndMsg pMsg = pMsg1;

      // for MPDT also check if second notification index is valid
      #if (CSM_S_MPDT == CSM_C_F_ON)
      if ((lIndexDDataInd >= 0) && (lIndexDataInd < CSMAL_MAX_DATAIND_NOTIFICATIONS))
      {
        // lookup data in corresponding notification table and overwrite pointer declared above
        pMsg2 = &rDataIndNotificationTable.arDataIndNotifications[lIndexDDataInd];
      }
      #endif

      #if (CSM_S_MPDT == CSM_C_F_ON)
      // in case of MPDT support we now must check the protocol type given by CSM
      if (dwProtocolType == pMsg2->dwProtocolType)
      {
        // D-Data indication by CSM, so we use the corresponding second notification entry
        pMsg = pMsg2;
        #if (CSM_S_TRACE == CSM_C_F_ON)
        if (pMsg1 == pMsg2)
        {
          ETG_TRACE_USR1(("CSMAL_vDataInd_Handler - Primary Callback for Protocol:%x chosen",
                          ETG_CENUM(tCSM_PROTOCOL_TYPE,         (tU32) dwProtocolType)
                        ));
        }
        else
        {
          ETG_TRACE_USR1(("CSMAL_vDataInd_Handler - Secondary Callback for Protocol:%x chosen",
                          ETG_CENUM(tCSM_PROTOCOL_TYPE,         (tU32) dwProtocolType)
                        ));
        }
        #endif
      }
      else
      {
        // it could be any other protocol except MPDT or MPDT C-DATA
        // in this case the standard notification entry is used
        pMsg = pMsg1;
        #if (CSM_S_TRACE == CSM_C_F_ON)
        ETG_TRACE_USR1(("CSMAL_vDataInd_Handler - Primary Callback for Protocol:%x chosen",
                        ETG_CENUM(tCSM_PROTOCOL_TYPE,         (tU32) dwProtocolType)
                      ));
        #endif
      }
      #else
      // No MPDT support -> the standard notification can be chosen directly to go on with
      pMsg = pMsg1;
      #endif

      // warum muss der Protocol Typ nochmal in die Tabelle geschrieben werden ?? steht eigentlich schon drin !!!
      pMsg->dwProtocolType = dwProtocolType;
      // fill address field (dies wurde beim ApplCallbackInit nicht gespeichert !!!)
      for (iAddrFieldIndex = 0; iAddrFieldIndex < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; iAddrFieldIndex++)
      {
        if (pbAddressField != NULL)
        {
          pMsg->abAddressField[iAddrFieldIndex] = *pbAddressField++;
        }
        else
        {
          pMsg->abAddressField[iAddrFieldIndex] = 0;
        }
      }
      // fill the data length
      pMsg->wDataLength = wDataLength;

      #if (CSMAL_S_DATA_IND_STRUCT_WITH_DATA_ARRAY == CSM_C_F_ON)
      // if we use the integrated struct buffer -> copy the data to the msg
      CSM_vMemCopy(pMsg->abData, pbData, wDataLength);
      #endif

      // now get the access handle (this identifies the process/queue/queuesize)
      pHandle = (pCSMHandle)pMsg->rMsgHeader.pvAccessUsrHandle;

      if (pHandle != NULL)
      {
        #if (CSM_S_TRACE == CSM_C_F_ON)
        ETG_TRACE_USR3(("CSMAL_vDataInd_Handler - Message will be sent to queue of handle 0x%08x", pHandle));
        #endif

        // send the callback data to message queue
        #if (CSMAL_S_DATA_IND_STRUCT_WITH_DATA_ARRAY == CSM_C_F_ON)
        {
          // Old handling with fix 4k+ messages
          // CSMAL_vSendMessage(pHandle, (void *)pMsg, (sizeof(trDataIndMsg) /*+ wDataLength*/));
          // Newer handling with flexible messages but still global buffers, one for each possible application
          CSMAL_vSendMessage(pHandle, (void *)pMsg, (sizeof(trDataIndMsg) - CSMAL_MAX_DATA_PAYLOAD + wDataLength));
        }
        #else
        {
          // New handling without fix data array
          // we have to ensure, that the data size matches the service message queue element size !!!
          LONG lMsgQElemSize = pHandle->lMQueueElemSize;
          LONG lMsgSize = sizeof(trDataIndMsg) + wDataLength;
          // create buffer of max. size that is possible
          BYTE abMsgAndData[sizeof(trDataIndMsg) + CSMAL_MAX_DATA_PAYLOAD];  // buffer for message info and received data

          if (lMsgSize <= lMsgQElemSize)
          {
            #if (CSM_S_TRACE == CSM_C_F_ON)
            ETG_TRACE_USR4(("CSMAL_vDataInd_Handler - MsgSize:%d  MsgQueueElemSize:%d", lMsgSize, lMsgQElemSize));
            #endif
            // copy the message info to the buffer
            CSM_vMemCopy(abMsgAndData, pMsg, sizeof(trDataIndMsg));
            // copy the data to the correct position in buffer
            CSM_vMemCopy(abMsgAndData + sizeof(trDataIndMsg), pbData, wDataLength);
            // send the callback data to message queue
            CSMAL_vSendMessage(pHandle, (void *)abMsgAndData, lMsgSize);
          }
          else
          {
            #if (CSM_S_TRACE == CSM_C_F_ON)
            ETG_TRACE_ERR(("CSMAL_vDataInd_Handler - ERROR - MsgSize:%d > MsgQueueElemSize:%d", lMsgSize, lMsgQElemSize));
            #endif
          }
        }
        #endif
      }
      else
      {
        #if (CSM_S_TRACE == CSM_C_F_ON)
        ETG_TRACE_ERR(("CSMAL_vDataInd_Handler - ERROR - pHandle is NULL, no message sent"));
        #endif
      }
    }
    else
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
        ETG_TRACE_ERR(("CSMAL_vDataInd_Handler - ERROR - Table Index out of bounds:0x%X", lIndexDataInd));
      #endif
    }
  } // payload length pre-check
  else
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_ERR(("CSMAL_vDataInd_Handler - ERROR - Payload DataLen exceeded:0x%X", wDataLength));
    #endif
  }
}/*lint !e818*/ /* pbData could be declared as pointing to const but specified interface is not const - 27.08.2008 krv2hi */

/*******************************************************************************
 * function    CSMAL_vDataIndFF_Handler
 * \doxydocu
 * \brief      Handler for first frame data indication. Retrieves the registered
 *             callback data and sends it to the service task.
 *             This function is registered with the CSM.
 *
 * \param[in]  pvEntry
 *             pointer to lookup table, to get message data
 *
 * \param[in]  dwProtocolType
 *             protocol type
 *
 * \param[in]  pvAddressField
 *             addressing, depends on protocol
 *
 * \param[in]  wDataLength
 *             Message length in byte.
 *
 * \return     VOID
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview
 * \history_begin
 * 13.11.08  VTeam-Khler
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#if (CSM_S_ISO_TP == CSM_C_F_ON)
#if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
VOID CSMAL_vDataIndFF_Handler(void * pvEntry, DWORD dwProtocolType, const VOID* pvAddressField, WORD wDataLength)
{
  LONG lIndexDataIndFF = -1;
  pCSMHandle pHandle;
  BYTE bAddr0 = *((((BYTE *)pvAddressField)+0));
  BYTE bAddr1 = *((((BYTE *)pvAddressField)+1));

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR2(("CSMAL_vDataIndFF_Handler - pvEntry:0x%X  Prot:0x%X  AddrField:0x%02X%02X  DataLen:0x%X",
                  pvEntry,
                  dwProtocolType,
                  (tU8)  bAddr0,
                  (tU8)  bAddr1,
                  wDataLength
                ));
  #endif

  // handle is pointer into lookup table
  pApplNotificationLookupEntry pEntry = (pApplNotificationLookupEntry)pvEntry;

  if (pEntry != NULL)
  {
    lIndexDataIndFF = pEntry->lIndexDataIndFF;
  }

  if ((lIndexDataIndFF >= 0) && (lIndexDataIndFF < CSMAL_MAX_DATAINDFF_NOTIFICATIONS))
  {
    // lookup message in corresponding notification table
    pDataIndFFMsg pMsg = &rDataIndFFNotificationTable.arDataIndFFNotifications[lIndexDataIndFF];
    INT  iAddrFieldIndex;
    BYTE *pbAddressField = (BYTE *) pvAddressField;

    // fill in new data
    pMsg->dwProtocolType  = dwProtocolType;
    for (iAddrFieldIndex = 0; iAddrFieldIndex < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; iAddrFieldIndex++)
    {
      if (pbAddressField != NULL)
      {
        pMsg->abAddressField[iAddrFieldIndex] = *pbAddressField++;
      }
      else
      {
        pMsg->abAddressField[iAddrFieldIndex] = 0;
      }
    }
    pMsg->wDataLength = wDataLength;

    pHandle = (pCSMHandle)pMsg->rMsgHeader.pvAccessUsrHandle;

    if (pHandle != NULL)
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_USR3(("CSMAL_vDataIndFF_Handler - Message will be sent to queue of handle 0x%08x",
                      pHandle));
      #endif
      // send the callback data to message buffer
      CSMAL_vSendMessage(pHandle, (void *)pMsg, sizeof(trDataIndFFMsg));
    }
    else
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_ERR(("CSMAL_vDataIndFF_Handler - ERROR - pHandle is NULL, no message sent"));
      #endif
    }
  }
  else
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_ERR(("CSMAL_vDataIndFF_Handler - ERROR - Index out of bounds:%d", lIndexDataIndFF));
    #endif
  }
}
#endif // CSM_S_ITP_FF_INDICATION_AVAILABLE
#endif // CSM_S_ISO_TP

/*******************************************************************************
 * function    CSMAL_vDataErrorInd_Handler
 * \doxydocu
 * \brief      Handler for data error indication. Retrieves the registered
 *             callback data and sends it to the service task.
 *             This function is registered with the CSM.
 *
 * \param[in]  pvEntry
 *             pointer to lookup table, to get message data
 *
 * \param[in]  dwProtocolType
 *             protocol type
 *
 * \param[in]  pvAddressField
 *             addressing, depends on protocol
 *
 * \param[in]  dwDataErrorCode
 *             Indicated error code
 *
 * \return     VOID
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview
 * \history_begin
 * 13.11.08  VTeam-Khler
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
VOID CSMAL_vDataErrorInd_Handler(void * pvEntry, DWORD dwProtocolType, const VOID* pvAddressField, DWORD dwDataErrorCode)
{
  LONG lIndexDataErrorInd = -1;
  pCSMHandle pHandle;
  BYTE bAddr0 = *((((BYTE *)pvAddressField)+0));
  BYTE bAddr1 = *((((BYTE *)pvAddressField)+1));

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR2(("CSMAL_vDataErrorInd_Handler - pvEntry:0x%X  Prot:0x%X  AddrField:0x%02X%02X  dwDataErrorCode:0x%X",
                  pvEntry,
                  dwProtocolType,
                  (tU8)  bAddr0,
                  (tU8)  bAddr1,
                  dwDataErrorCode
                ));
  #endif

  // handle is pointer into lookup table
  pApplNotificationLookupEntry pEntry = (pApplNotificationLookupEntry)pvEntry;

  if (pEntry != NULL)
  {
    lIndexDataErrorInd = pEntry->lIndexDataErrorInd;
  }

  if ((lIndexDataErrorInd >= 0) && (lIndexDataErrorInd < CSMAL_MAX_DATAERRORIND_NOTIFICATIONS))
  {
    // lookup message in corresponding notification table
    pDataErrorIndMsg pMsg = &rDataErrorIndNotificationTable.arDataErrorIndNotifications[lIndexDataErrorInd];
    INT  iAddrFieldIndex;
    BYTE *pbAddressField = (BYTE *) pvAddressField;

    // fill in new data
    pMsg->dwProtocolType  = dwProtocolType;
    for (iAddrFieldIndex = 0; iAddrFieldIndex < CSM_C_ACCESS_LAYER_ADDRESS_FIELD_SIZE; iAddrFieldIndex++)
    {
      if (pbAddressField != NULL)
      {
        pMsg->abAddressField[iAddrFieldIndex] = *pbAddressField++;
      }
      else
      {
        pMsg->abAddressField[iAddrFieldIndex] = 0;
      }
    }
    pMsg->dwDataErrorCode = dwDataErrorCode;

    pHandle = (pCSMHandle)pMsg->rMsgHeader.pvAccessUsrHandle;

    if (pHandle != NULL)
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_USR3(("CSMAL_vDataErrorInd_Handler - Message will be sent to queue of handle 0x%08x",
                      pHandle));
      #endif
      // send the callback data to message buffer
      CSMAL_vSendMessage(pHandle, (void *)pMsg, sizeof(trDataErrorIndMsg));
    }
    else
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_ERR(("CSMAL_vDataErrorInd_Handler - ERROR - pHandle is NULL, no message sent"));
      #endif
    }
  }
  else
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_ERR(("CSMAL_vDataErrorInd_Handler - ERROR - Index out of bounds:%d", lIndexDataErrorInd));
    #endif
  }
}
#endif // CSM_S_DATA_ERROR_IND_AVAILABLE

#if (CSM_S_ISO_TP == CSM_C_F_ON)
#if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
// empty
#endif
#endif

#endif // CSM_S_CNP_AVAILABLE

/*******************************************************************************
 * function    CSMAL_lGetAccessHandle
 * \doxydocu
 * \brief      retrieves a handle for the CSM Access Layer for use with
 *             callback functionality
 *
 * \param[in]  pvHandle
 *               Pointer to access layer handle for use with callback
 *               functionality. The handle or CSM_C_INVALID_HANDLE is copied
 *               to this address.
 *
 * \return     error value
 *
 *
 * \access     Application
 * \reentrant  yes
 *
 * \lastreview 05.09.08 VTeam-Khler
 * \history_begin
 * 04.06.09 VTeam-Khler
 * - Parameter 'pointer to handle' added, return value now error value.
 * 19.09.06 3SOFT Schubart
 * Initial Revision.
 * \history_end
 ******************************************************************************/
LONG CSMAL_lGetAccessHandle(void * pvHandle)
{
  pCSMHandle pAccessUsrHandle;  // Pointer to a handle struct, in fact a pointer to a table entry
  LONG lRetVal;

  // cast the given pointer to the correct type (array table of structs)
  pCSMHandle *prHandle = (pCSMHandle *) pvHandle;

  // lock the handle table
  CSMAL_lLock(ghAccessHandleTableLock);

  if (rCSMAccessHandleTable.lIndex < CSMAL_NUM_HANDLES)
  {
    // assign next free handle & increase index
    pAccessUsrHandle = &rCSMAccessHandleTable.arCSMAccessHandles[rCSMAccessHandleTable.lIndex++];
    lRetVal = CSM_C_NO_ERROR;
  }
  else
  {
    // all handles used, table is full
    pAccessUsrHandle = CSM_C_INVALID_HANDLE;
    lRetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_RESOURCE);
  }

  // unlock the handle table
  CSMAL_lUnLock(ghAccessHandleTableLock);

  *prHandle = pAccessUsrHandle;  // Copy handle to caller's reference
  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR2(("CSMAL_lGetAccessHandle - pAccessUsrHandle:0x%X", pAccessUsrHandle));
  #endif

  return lRetVal;
}

/*******************************************************************************
 * function    CSMAL_lStoreHandleData
 * \doxydocu
 * \brief      stores message buffer id and task id in handle. This function
 *             is necessary because the storing must be done in kernel space,
 *             so the user mode app cannot do this itself.
 *
 * \param[in]  pvHandle
 *             pointer to access handle
 *
 * \param[in]  lMessageBufferID
 *             message buffer id to store
 *
 * \param[in]  lTaskID
 *             task id to store
 *
 * \return     error code
 *
 * \access     ?
 * \reentrant  yes if handles are different
 *
 * \lastreview 04.09.08  VTeam-Khler
 * \history_begin
 * 04.09.08  VTeam-Khler
 * - Handle will be checked.
 * 17.10.2006 3SOFT Schubart
 * Initial Revision.
 * \history_end
 ******************************************************************************/
LONG CSMAL_lStoreHandleData(void * pvHandle, LONG lMessageBufferID, LONG lTaskID, LONG lMessageBufferSize)
{
  LONG lRetVal;

  #if (CSM_S_TRACE == CSM_C_F_ON)
  ETG_TRACE_USR2(("CSMAL_lStoreHandleData - Handle:0x%X  MsgBufferID:%d  MsgBufferSize:%d  TaskID:0x%X",
                  pvHandle,
                  lMessageBufferID,
                  lMessageBufferSize,
                  lTaskID));
  #endif

  if (NULL != pvHandle)
  {
    // the handle is nothing else than a pointer to a table entry in the handle table
    // cast to the correct struct type
    pCSMHandle pHandle = (pCSMHandle) pvHandle;

    // store MessageQueueID, MessageQueueElementSize and ProcessID in table
    pHandle->lMQueueID = lMessageBufferID;
    pHandle->lMQueueElemSize = lMessageBufferSize;
    pHandle->lTaskID   = lTaskID;
    lRetVal = CSM_C_NO_ERROR;
  }
  else
  {
    // invalid handle -> error
    lRetVal = CSM_M_MAKE_ERROR(CSM_C_ERR_ERROR, CSM_C_ERR_LOC_CSM_AL_KRNL, CSM_E_NULL_POINTER);
  }

  return lRetVal;
}

/*******************************************************************************
 * function    lRegisterCSM_LocalCtrl_ApplCallback
 * \doxydocu
 * \brief      Registers callbacks for local ctrl
 *
 * \param[in]  pvHandle
 *             CSM Access Layer Handle
 *
 * \param[in]  bBus
 *             CAN bus number
 *
 * \param[in]  dwProtocolType
 *             protocol type
 *
 * \param[in]  pvAddressField
 *             addressing, depends on protocol
 *
 * \param[in]  pvCallBackFkt
 *             Callback function(s)
 *
 * \return     error code
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview 09.09.08 CM-AI/VTeam-Brunotte
 * \history_begin
 * 25.10.2006 3SOFT Schubart
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#ifdef CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL
static LONG lRegisterCSM_LocalCtrl_ApplCallback(void * pvHandle, BYTE bBus, DWORD dwProtocolType, const void * pvAddressField, const void * pvCallBackFkt)
{
  LONG lRetVal;
  pApplNotificationLookupEntry phCSM;

  tCSM_LOCAL_CTRL_APPL_CALLBACK tCallbackData = { 0 }; /* default = NULL */
  tCSM_LOCAL_CTRL_APPL_CALLBACK* pCallBackFkt;

  // cast pointer to appropriate struct type (here LCTRL)
  pCallBackFkt = (tCSM_LOCAL_CTRL_APPL_CALLBACK*) pvCallBackFkt;

  phCSM = CSMAL_lAddApplNotification( pvHandle, bBus, dwProtocolType, pvAddressField,                      // mandatory
                                      pCallBackFkt->pfvCommunicationCon,                                   // mandatory
                                      pCallBackFkt->pfvCommunicationInd                                    // mandatory
                                      #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
                                      ,NULL                                                                // DataCon not used
                                      ,NULL                                                                // DataInd not used

                                      #if (CSM_S_MPDT == CSM_C_F_ON )
                                      ,NULL                                                                // DataCon2 not used
                                      ,NULL                                                                // DataInd2 not used
                                      #endif // CSM_S_MPDT

                                      #if (CSM_S_ISO_TP == CSM_C_F_ON)
                                      #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
                                      ,NULL                                                                // DataIndFF not used
                                      #endif
                                      #endif // CSM_S_ISO_TP

                                      #if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
                                      ,NULL                                                                // DataErrorInd not used
                                      #endif

                                      #endif // CSM_S_CNP_AVAILABLE
                                     );

  tCallbackData.u16ApplID           = pCallBackFkt->u16ApplID;
  tCallbackData.pfvCommunicationCon = CSMAL_vCommunicationCon_Handler;
  tCallbackData.pfvCommunicationInd = CSMAL_vCommunicationInd_Handler;

  // now register protocoltype at CSM
  lRetVal = CSM_lApplCallbackInit((void *)phCSM, bBus, dwProtocolType, pvAddressField, &tCallbackData);

  return lRetVal;
}
#endif // CSM_C_PTYPE_USED_STD_CAN_LOCAL_CTRL

/*******************************************************************************
 * function    lRegisterCSM_GMLAN_NWM_ApplCallback
 * \doxydocu
 * \brief      Registers callbacks for GM NWM
 *
 * \param[in]  pvHandle
 *             CSM Access Layer Handle
 *
 * \param[in]  bBus
 *             CAN bus number
 *
 * \param[in]  dwProtocolType
 *             protocol type
 *
 * \param[in]  pvAddressField
 *             addressing, depends on protocol
 *
 * \param[in]  pvCallBackFkt
 *             Callback function(s)
 *
 * \return     error code
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview 09.09.08 CM-AI/VTeam-Brunotte
 * \history_begin
 * 23.02.2007 VTeam-Prhl
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#ifdef CSM_C_PTYPE_GM_NWM
static LONG lRegisterCSM_GMLAN_NWM_ApplCallback(void * pvHandle, BYTE bBus, DWORD dwProtocolType, const void * pvAddressField,
                                                const void * pvCallBackFkt)
{
  LONG lRetVal;
  pApplNotificationLookupEntry phCSM;

  tCSM_GMLAN_NWM_APPL_CALLBACK tCallbackData = { 0 }; /* default = NULL */
  tCSM_GMLAN_NWM_APPL_CALLBACK* pCallBackFkt;

  pCallBackFkt = (tCSM_GMLAN_NWM_APPL_CALLBACK*)pvCallBackFkt;

  phCSM = CSMAL_lAddApplNotification(pvHandle, bBus, dwProtocolType, pvAddressField,
                                     NULL,
                                     pCallBackFkt->pfvCommunicationInd
                                     #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
                                     ,NULL                                                                // DataCon not used
                                     ,NULL                                                                // DataInd not used
                                     #if (CSM_S_MPDT == CSM_C_F_ON )
                                     ,NULL                                                                // DataCon2 not used
                                     ,NULL                                                                // DataInd2 not used
                                     #endif  // CSM_S_MPDT
                                     #if (CSM_S_ISO_TP == CSM_C_F_ON)
                                     #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
                                     ,NULL
                                     #endif
                                     #endif
                                     #if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
                                     ,NULL
                                     #endif
                                     #endif // CSM_S_CNP_AVAILABLE
                                     );

  // copy relevant data to temporary NMW_APPL_CALLBACK structure. rTaskInfo and
  // pvHandle are calculated and stored in CSM_lApplCallbackInit
  tCallbackData.u16ApplID = pCallBackFkt->u16ApplID;
  tCallbackData.pfvCommunicationInd = CSMAL_vCommunicationInd_Handler;

  // now register protocoltype at CSM
  lRetVal = CSM_lApplCallbackInit((void *)phCSM, bBus, dwProtocolType, pvAddressField, &tCallbackData);

  return lRetVal;
}
#endif // CSM_C_PTYPE_GM_NWM

/*******************************************************************************
 * function    lRegisterCSM_ITP_ApplCallback
 * \doxydocu
 * \brief      Registers callbacks for ISO TP
 *
 * \param[in]  pvHandle
 *             CSM Access Layer Handle
 *
 * \param[in]  bBus
 *             CAN bus number
 *
 * \param[in]  dwProtocolType
 *             protocol type
 *
 * \param[in]  pvAddressField
 *             addressing, depends on protocol
 *
 * \param[in]  pvCallBackFkt
 *             Callback function(s)
 *
 * \return     error code
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview  09.09.08 CM-AI/VTeam-Brunotte
 * \history_begin
 * 14.11.08  VTeam-Khler
 * DataIndFF and DataErrorInd added.
 * 25.10.2006 3SOFT Schubart
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
#if (CSM_S_ISO_TP == CSM_C_F_ON)
//#ifdef CSM_C_PTYPE_USED_STD_USDT_ISO
static LONG lRegisterCSM_ITP_ApplCallback(void * pvHandle, BYTE bBus, DWORD dwProtocolType, const void * pvAddressField,
                                          const void * pvCallBackFkt)
{
  LONG lRetVal;
  pApplNotificationLookupEntry    phCSM;
  tCSM_ISO_TP_USDT_APPL_CALLBACK  tCallbackData = { 0 }; /* default = NULL */
  tCSM_ISO_TP_USDT_APPL_CALLBACK* pCallBackFkt;

  pCallBackFkt = (tCSM_ISO_TP_USDT_APPL_CALLBACK*)pvCallBackFkt;

  phCSM = CSMAL_lAddApplNotification(pvHandle, bBus, dwProtocolType, pvAddressField,
                                     pCallBackFkt->pfvCommunicationCon,
                                     pCallBackFkt->pfvCommunicationInd,
                                     pCallBackFkt->pfvDataCon,
                                     pCallBackFkt->pfvDataInd

                                     #if (CSM_S_MPDT == CSM_C_F_ON )
                                     ,NULL                                                                // DataCon2 not used
                                     ,NULL                                                                // DataInd2 not used
                                     #endif  // CSM_S_MPDT

                                     #if (CSM_S_ISO_TP == CSM_C_F_ON)
                                     #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
                                     ,pCallBackFkt->pfvDataIndFF
                                     #endif // CSM_S_ITP_FF_INDICATION_AVAILABLE
                                     #endif // CSM_S_ISO_TP

                                     #if (CSM_S_ITP_ERROR_INDICATION_AVAILABLE == CSM_C_F_ON)
                                     ,pCallBackFkt->pfvDataErrorInd
                                     #elif (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
                                     ,NULL  /* pPFNTPDataErrorInd */
                                     #endif
                                     );

  tCallbackData.pfvCommunicationCon = CSMAL_vCommunicationCon_Handler;
  tCallbackData.pfvCommunicationInd = CSMAL_vCommunicationInd_Handler;
  tCallbackData.pfvDataCon          = CSMAL_vDataCon_Handler;
  tCallbackData.pfvDataInd          = CSMAL_vDataInd_Handler;
  #if (CSM_S_ISO_TP == CSM_C_F_ON)
    #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
      tCallbackData.pfvDataIndFF = CSMAL_vDataIndFF_Handler;
    #endif
  #endif // CSM_S_ISO_TP

  #if (CSM_S_ITP_ERROR_INDICATION_AVAILABLE == CSM_C_F_ON)
    tCallbackData.pfvDataErrorInd = CSMAL_vDataErrorInd_Handler;
  #elif (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
    //tCallbackData.pfvDataErrorInd = NULL;
  #endif

  // now register protocol at CSM
  lRetVal = CSM_lApplCallbackInit((void *)phCSM, bBus, dwProtocolType, pvAddressField, &tCallbackData);

  return lRetVal;
}
//#endif //CSM_C_PTYPE_USED_STD_USDT_ISO
#endif // CSM_S_ISO_TP
#endif // CSM_S_CNP_AVAILABLE

/*******************************************************************************
 * function    lRegisterCSM_UUDT_TX_ApplCallback
 * \doxydocu
 * \brief      Registers callbacks for UUDT TX
 *
 * \param[in]  pvHandle
 *             CSM Access Layer Handle
 *
 * \param[in]  bBus
 *             CAN bus number
 *
 * \param[in]  dwProtocolType
 *             protocol type
 *
 * \param[in]  pvAddressField
 *             addressing, depends on protocol
 *
 * \param[in]  pvCallBackFkt
 *             Callback function(s)
 *
 * \return     error code
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview  09.09.08 CM-AI/VTeam-Brunotte
 * \history_begin
 * 24.01.2007 CM-DI/PJ-VW48 Prhl
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
#if (CSM_S_UUDT_TX == CSM_C_F_ON)
//#ifdef CSM_C_PTYPE_USED_STD_UUDT_TX
static LONG lRegisterCSM_UUDT_TX_ApplCallback(void * pvHandle, BYTE bBus, DWORD dwProtocolType, const void * pvAddressField,
                                              const void * pvCallBackFkt)
{
  LONG lRetVal;
  pApplNotificationLookupEntry phCSM;

  tCSM_UUDT_TX_APPL_CALLBACK tCallbackData = { 0 }; /* default = NULL */
  tCSM_UUDT_TX_APPL_CALLBACK* pCallBackFkt;

  pCallBackFkt = (tCSM_UUDT_TX_APPL_CALLBACK*)pvCallBackFkt;

  phCSM = CSMAL_lAddApplNotification(pvHandle, bBus, dwProtocolType, pvAddressField,
                                     NULL,
                                     NULL,
                                     #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
                                     pCallBackFkt->pfvDataCon,
                                     NULL

                                     #if (CSM_S_MPDT == CSM_C_F_ON )
                                     ,NULL                                                                // DataCon2 not used
                                     ,NULL                                                                // DataInd2 not used
                                     #endif  // CSM_S_MPDT

                                     #if (CSM_S_ISO_TP == CSM_C_F_ON)
                                     #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
                                     ,NULL
                                     #endif
                                     #endif // CSM_S_ISO_TP
                                     #if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
                                     ,NULL
                                     #endif
                                     #endif // CSM_S_CNP_AVAILABLE
                                     );

  tCallbackData.pfvDataCon = CSMAL_vDataCon_Handler;

  // now register protocol at CSM
  lRetVal = CSM_lApplCallbackInit((void *)phCSM, bBus, dwProtocolType, pvAddressField, &tCallbackData);

  return lRetVal;
}
//#endif // CSM_C_PTYPE_USED_STD_UUDT_TX
#endif // CSM_S_UUDT_TX
#endif // CSM_S_CNP_AVAILABLE

/*******************************************************************************
 * function    lRegisterCSM_UUDT_RX_ApplCallback
 * \doxydocu
 * \brief      Registers callbacks for UUDT RX
 *
 * \param[in]  pvHandle
 *             CSM Access Layer Handle
 *
 * \param[in]  bBus
 *             CAN bus number
 *
 * \param[in]  dwProtocolType
 *             protocol type
 *
 * \param[in]  pvAddressField
 *             addressing, depends on protocol
 *
 * \param[in]  pvCallBackFkt
 *             Callback function(s)
 *
 * \return     error code
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview 09.09.08 CM-AI/VTeam-Brunotte
 * \history_begin
 * 13.11.2008  VTeam-Khler
 * - CSMAL_vDataErrorInd_Handler added.
 * 24.01.2007 CM-DI/PJ-VW48 Prhl
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
#if (CSM_S_UUDT_RX == CSM_C_F_ON)
//#ifdef CSM_C_PTYPE_USED_STD_UUDT_RX
static LONG lRegisterCSM_UUDT_RX_ApplCallback(void * pvHandle, BYTE bBus, DWORD dwProtocolType, const void * pvAddressField,
                                              const void * pvCallBackFkt)
{
  LONG lRetVal;
  pApplNotificationLookupEntry phCSM;

  tCSM_UUDT_RX_APPL_CALLBACK tCallbackData = { 0 }; /* default = NULL */
  tCSM_UUDT_RX_APPL_CALLBACK* pCallBackFkt;

  pCallBackFkt = (tCSM_UUDT_RX_APPL_CALLBACK*)pvCallBackFkt;

  phCSM = CSMAL_lAddApplNotification(pvHandle, bBus, dwProtocolType, pvAddressField,
                                     NULL,
                                     NULL,
                                     #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
                                     NULL,
                                     pCallBackFkt->pfvDataInd

                                     #if (CSM_S_MPDT == CSM_C_F_ON )
                                     ,NULL                                                                // DataCon2 not used
                                     ,NULL                                                                // DataInd2 not used
                                     #endif  // CSM_S_MPDT

                                     #if (CSM_S_ISO_TP == CSM_C_F_ON)
                                     #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
                                     ,NULL
                                     #endif
                                     #endif // CSM_S_ISO_TP

                                     #if (CSM_S_UUDT_RX_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
                                     ,pCallBackFkt->pfvDataErrorInd
                                     #elif (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
                                     ,NULL
                                     #endif
                                     #endif // CSM_S_CNP_AVAILABLE
                                     );

  tCallbackData.pfvDataInd = CSMAL_vDataInd_Handler;
  #if (CSM_S_UUDT_RX_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
  tCallbackData.pfvDataErrorInd   = CSMAL_vDataErrorInd_Handler;
  #elif (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
  //tCallbackData.pfvDataErrorInd =  NULL;
  #endif

  // now register protocol at CSM
  lRetVal = CSM_lApplCallbackInit((void *)phCSM, bBus, dwProtocolType, pvAddressField, &tCallbackData);

  return lRetVal;
}
//#endif // CSM_C_PTYPE_USED_STD_UUDT_RX
#endif // CSM_S_UUDT_RX
#endif // CSM_S_CNP_AVAILABLE

/*******************************************************************************
 * function    lRegisterCSM_CBR_ApplCallback
 * \doxydocu
 * \brief      Registers callbacks for broadcast
 *
 * \param[in]  pvHandle
 *             CSM Access Layer Handle
 *
 * \param[in]  bBus
 *             CAN bus number
 *
 * \param[in]  dwProtocolType
 *             protocol type
 *
 * \param[in]  pvAddressField
 *             addressing, depends on protocol
 *
 * \param[in]  pvCallBackFkt
 *             Callback function(s)
 *
 * \return     error code
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview 09.09.08 CM-AI/VTeam-Brunotte
 * \history_begin
 * 25.10.2006 3SOFT Schubart
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#ifdef CSM_C_PTYPE_USED_STD_BR_SIGNAL
static LONG lRegisterCSM_CBR_ApplCallback(void * pvHandle, BYTE bBus, DWORD dwProtocolType, const void * pvAddressField,
                                          const void * pvCallBackFkt)
{
  LONG lRetVal;
  pApplNotificationLookupEntry phCSM;

  tCSM_BR_SIGNAL_APPL_CALLBACK tCallbackData = { 0 }; /* default = NULL */
  tCSM_BR_SIGNAL_APPL_CALLBACK* pCallBackFkt;

  // cast pointer to appropriate struct type (here a CBR Signal)
  pCallBackFkt = (tCSM_BR_SIGNAL_APPL_CALLBACK*) pvCallBackFkt;

  phCSM = CSMAL_lAddApplNotification( pvHandle, bBus, dwProtocolType, pvAddressField,              // mandatory
                                      pCallBackFkt->pfvCommunicationCon,                           // mandatory
                                      pCallBackFkt->pfvCommunicationInd                            // mandatory

                                      #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
                                      ,NULL                                                        // DataCon not used
                                      ,NULL                                                        // DataInd not used

                                      #if (CSM_S_MPDT == CSM_C_F_ON )
                                      ,NULL                                                                // DataCon2 not used
                                      ,NULL                                                                // DataInd2 not used
                                      #endif  // CSM_S_MPDT

                                      #if (CSM_S_ISO_TP == CSM_C_F_ON)
                                      #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
                                      ,NULL                                                        // DataIndFF not used
                                      #endif
                                      #endif // CSM_S_ISO_TP

                                      #if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
                                      ,NULL                                                        // DataErrorInd not used
                                      #endif

                                      #endif // CSM_S_CNP_AVAILABLE
                                     );

  tCallbackData.u16ApplID           = pCallBackFkt->u16ApplID;
  tCallbackData.pfvCommunicationCon = CSMAL_vCommunicationCon_Handler;
  tCallbackData.pfvCommunicationInd = CSMAL_vCommunicationInd_Handler;

  // now register signal at CSM
  lRetVal = CSM_lApplCallbackInit((void *)phCSM, bBus, dwProtocolType, pvAddressField, &tCallbackData);

  return lRetVal;
}
#endif // CSM_C_PTYPE_USED_STD_BR_SIGNAL

/*******************************************************************************
 * function    lRegisterCSM_MPDT_C_ApplCallback
 * \doxydocu
 * \brief      Registers callbacks for MCAN
 *
 * \param[in]  pvHandle
 *             CSM Access Layer Handle
 *
 * \param[in]  bBus
 *             CAN bus number
 *
 * \param[in]  dwProtocolType
 *             protocol type
 *
 * \param[in]  pvAddressField
 *             addressing, depends on protocol
 *
 * \param[in]  pvCallBackFkt
 *             Callback function(s)
 *
 * \return     error code
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview
 * \history_begin
 * 17.04.2012 K&K
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#ifdef CSM_C_PTYPE_RN_MPDT_C
static LONG lRegisterCSM_MPDT_C_ApplCallback(void * pvHandle, BYTE bBus, DWORD dwProtocolType, const void * pvAddressField,
                                             const void * pvCallBackFkt)
{
  LONG lRetVal;
  pApplNotificationLookupEntry phCSM_C;

  // local instance of callbacks, will be filled later on
  tCSM_MPDT_APPL_CALLBACK   tCallbackData = { 0 };
  tCSM_MPDT_APPL_CALLBACK * pCallBackFkt;

  // cast pointer to appropriate struct type (here MPDT)
  pCallBackFkt = (tCSM_MPDT_APPL_CALLBACK*) pvCallBackFkt;

  // create a notification entry for MPDT C-Data protocol
  // One problem regarding MPDT here now is, that the CSM in general requires seperate callbacks for C-Data and D-Data,
  // but on the other hand only allows one ApplCallbackInit for MPDT_C (passing 4 callbacks for C and D directly).
  // The one and only notification reference is passed to CSM (phCSM_C) so on the way back later upon callback,
  // the IND and CON Handlers have to differentiate between MPDT C-Data and D-Data.

  phCSM_C = CSMAL_lAddApplNotification( pvHandle, bBus, CSM_C_PTYPE_RN_MPDT_C, pvAddressField,   // mandatory
                                        pCallBackFkt->pfvCommunicationCon,                       // mandatory
                                        pCallBackFkt->pfvCommunicationInd                        // mandatory

                                        #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
                                        ,pCallBackFkt->pfvCDataCon                               // mandatory for CNP
                                        ,pCallBackFkt->pfvCDataInd                               // mandatory for CNP

                                        #if (CSM_S_MPDT == CSM_C_F_ON )
                                        ,pCallBackFkt->pfvDDataCon                               // mandatory for CNP MPDT
                                        ,pCallBackFkt->pfvDDataInd                               // mandatory for CNP MPDT
                                        #endif // CSM_S_MPDT

                                        #if (CSM_S_ISO_TP == CSM_C_F_ON)
                                        #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
                                        ,NULL                                                    // mandatory for ITP FF Indication
                                        #endif
                                        #endif // CSM_S_ISO_TP

                                        #if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
                                        ,NULL                                                    // mandatory for data error Indication
                                        #endif

                                        #endif // CSM_S_CNP_AVAILABLE
                                       );

  // now setup a complete set of CSMAL callbacks for the CSM
  tCallbackData.pfvCommunicationCon  = CSMAL_vCommunicationCon_Handler;
  tCallbackData.pfvCommunicationInd  = CSMAL_vCommunicationInd_Handler;
  tCallbackData.pfvCDataCon          = CSMAL_vDataCon_Handler;
  tCallbackData.pfvCDataInd          = CSMAL_vDataInd_Handler;
  tCallbackData.pfvDDataCon          = CSMAL_vDataCon_Handler;
  tCallbackData.pfvDDataInd          = CSMAL_vDataInd_Handler;

  // and register the protocol at CSM (upon callback, MPDT C and D data will lead to the same callback here in CSMAL)
  lRetVal = CSM_lApplCallbackInit((void *)phCSM_C, bBus, dwProtocolType, pvAddressField, &tCallbackData);

  return lRetVal;
}
#endif // CSM_C_PTYPE_RN_MPDT_C

/*******************************************************************************
 * function    lRegisterCSM_BAP_lApplCallbackInit
 * \doxydocu
 * \brief      Registers callbacks for VW BAP
 *
 * \param[in]  pvHandle
 *             CSM Access Layer Handle
 *
 * \param[in]  bBus
 *             CAN bus number
 *
 * \param[in]  dwProtocolType
 *             protocol type
 *
 * \param[in]  pvAddressField
 *             addressing, depends on protocol
 *
 * \param[in]  pvCallBackFkt
 *             Callback function(s)
 *
 * \return     error code
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview
 * \history_begin
 * 15.07.2013 K&K
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#ifdef CSM_C_PTYPE_VW_BAP
static LONG lRegisterCSM_BAP_lApplCallbackInit(void * pvHandle, BYTE bBus, DWORD dwProtocolType, const void * pvAddressField,
                                               const void * pvCallBackFkt)
{
  LONG lRetVal;
  pApplNotificationLookupEntry phCSM;

  tCSM_VW_BAP_APPL_CALLBACK tCallbackData = { 0 }; /* default = NULL */
  tCSM_VW_BAP_APPL_CALLBACK* pCallBackFkt;

  pCallBackFkt = (tCSM_VW_BAP_APPL_CALLBACK*)pvCallBackFkt;

  phCSM = CSMAL_lAddApplNotification(pvHandle, bBus, dwProtocolType, pvAddressField,
                                     pCallBackFkt->pfvCommunicationCon,
                                     NULL
                                     #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
                                     ,pCallBackFkt->pfvDataCon
                                     ,pCallBackFkt->pfvDataInd

                                     #if (CSM_S_MPDT == CSM_C_F_ON )
                                     ,NULL                                                                // DataCon2 not used
                                     ,NULL                                                                // DataInd2 not used
                                     #endif // CSM_S_MPDT

                                     #if (CSM_S_ISO_TP == CSM_C_F_ON)
                                     #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
                                     ,NULL
                                     #endif
                                     #endif // CSM_S_ISO_TP

                                     #if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
                                     ,NULL
                                     #endif
                                     #endif // CSM_S_CNP_AVAILABLE
                                     );

  tCallbackData.pfvCommunicationCon = CSMAL_vCommunicationCon_Handler;
  tCallbackData.pfvDataCon          = CSMAL_vDataCon_Handler;
  tCallbackData.pfvDataInd          = CSMAL_vDataInd_Handler;

  // now register protocol at CSM
  lRetVal = CSM_lApplCallbackInit((void *)phCSM, bBus, dwProtocolType, pvAddressField, &tCallbackData);

  return lRetVal;
}
#endif // CSM_C_PTYPE_VW_BAP

/*******************************************************************************
 * function    lRegisterCSM_OSEK_lApplCallbackInit
 * \doxydocu
 * \brief      Registers callbacks for PAG Osek NM
 *
 * \param[in]  pvHandle
 *             CSM Access Layer Handle
 *
 * \param[in]  bBus
 *             CAN bus number
 *
 * \param[in]  dwProtocolType
 *             protocol type
 *
 * \param[in]  pvAddressField
 *             addressing, depends on protocol
 *
 * \param[in]  pvCallBackFkt
 *             Callback function(s)
 *
 * \return     error code
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview 09.09.08 CM-AI/VTeam-Brunotte
 * \history_begin
 * 15.05.2008 VTeam-Funke
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#ifdef CSM_C_PTYPE_PAG_OSEK_NWM
static LONG lRegisterCSM_OSEK_lApplCallbackInit(void * pvHandle, BYTE bBus, DWORD dwProtocolType, const void * pvAddressField,
                                                const void * pvCallBackFkt)
{
  LONG lRetVal;
  pApplNotificationLookupEntry phCSM;

  tCSM_OSEK_NWM_APPL_CALLBACK tCallbackData = { 0 }; /* default = NULL */
  tCSM_OSEK_NWM_APPL_CALLBACK* pCallBackFkt;

  pCallBackFkt = (tCSM_OSEK_NWM_APPL_CALLBACK*)pvCallBackFkt;

  phCSM = CSMAL_lAddApplNotification(pvHandle, bBus, dwProtocolType, pvAddressField,
                                     NULL,
                                     pCallBackFkt->pfvCommunicationInd
                                     #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
                                     ,NULL
                                     ,NULL
                                     #if (CSM_S_MPDT == CSM_C_F_ON )
                                     ,NULL                                                                // DataCon2 not used
                                     ,NULL                                                                // DataInd2 not used
                                     #endif // CSM_S_MPDT

                                     #if (CSM_S_ISO_TP == CSM_C_F_ON)
                                     #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
                                     ,NULL
                                     #endif
                                     #endif // CSM_S_ISO_TP

                                     #if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
                                     ,NULL
                                     #endif
                                     #endif // CSM_S_CNP_AVAILABLE
                                     );

  // copy relevant data to temporary NMW_APPL_CALLBACK structure. rTaskInfo and
  // pvHandle are calculated and stored in CSM_lApplCallbackInit
  tCallbackData.u16ApplID = pCallBackFkt->u16ApplID;
  tCallbackData.pfvCommunicationInd = CSMAL_vCommunicationInd_Handler;

  // now register protocoltype at CSM
  lRetVal = CSM_lApplCallbackInit((void *)phCSM, bBus, dwProtocolType, pvAddressField, &tCallbackData);

  return lRetVal;
}
#endif // CSM_C_PTYPE_PAG_OSEK_NWM

/*******************************************************************************
 * function    lRegisterCSM_J1939_ApplCallback
 * \doxydocu
 * \brief      Registers callbacks for J1939
 *
 * \param[in]  pvHandle
 *             CSM Access Layer Handle
 *
 * \param[in]  bBus
 *             CAN bus number
 *
 * \param[in]  dwProtocolType
 *             protocol type
 *
 * \param[in]  pvAddressField
 *             addressing, depends on protocol
 *
 * \param[in]  pvCallBackFkt
 *             Callback function(s)
 *
 * \return     error code
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview  09.09.08 CM-AI/VTeam-Brunotte
 * \history_begin
 * 17.10.14  NW Team-RBEI
 * Initial Version.
 * \history_end
 ******************************************************************************/
#ifdef CSM_C_PTYPE_J1939
LONG lRegisterCSM_J1939_ApplCallback(void * pvHandle,
                                   BYTE bBus,
                                   DWORD dwProtocolType,
                                   const void * pvAddressField,
                                   const void * pvCallBackFkt)
{
  LONG lRetVal;
  pApplNotificationLookupEntry    phCSM;
  tCSM_J1939_APPL_CALLBACK  tCallbackData = { 0 }; /* default = NULL */
  tCSM_J1939_APPL_CALLBACK* pCallBackFkt;

  pCallBackFkt = (tCSM_J1939_APPL_CALLBACK*)pvCallBackFkt;

  phCSM = CSMAL_lAddApplNotification(pvHandle,
                                     bBus,
                                     dwProtocolType,
                                     pvAddressField,
                                     pCallBackFkt->pfvCommunicationCon,
                                     pCallBackFkt->pfvCommunicationInd,
                                     pCallBackFkt->pfvDataCon,
                                     pCallBackFkt->pfvDataInd

                   #if (CSM_S_MPDT == CSM_C_F_ON )
                                     ,NULL,                                                                // DataCon2 not used
                                     ,NULL,                                                                // DataInd2 not used
                                     #endif  // CSM_S_MPDT

                   #if (CSM_S_ISO_TP == CSM_C_F_ON)
                                     #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
                                     ,pCallBackFkt->pfvDataIndFF
                                     #endif // CSM_S_ITP_FF_INDICATION_AVAILABLE
                                     #endif // CSM_S_ISO_TP

                   #if   (CSM_S_ITP_ERROR_INDICATION_AVAILABLE == CSM_C_F_ON)
                                     ,pCallBackFkt->pfvDataErrorInd
                                     #elif (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
                                     ,NULL  /* pPFNTPDataErrorInd */
                                     #endif
                                     );

  tCallbackData.pfvCommunicationCon = CSMAL_vCommunicationCon_Handler;
  tCallbackData.pfvCommunicationInd = CSMAL_vCommunicationInd_Handler;
  tCallbackData.pfvDataCon          = CSMAL_vDataCon_Handler;
  tCallbackData.pfvDataInd          = CSMAL_vDataInd_Handler;
  #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
    tCallbackData.pfvDataIndFF = CSMAL_vDataIndFF_Handler;
  #endif
  #if   (CSM_S_ITP_ERROR_INDICATION_AVAILABLE == CSM_C_F_ON)
    tCallbackData.pfvDataErrorInd = CSMAL_vDataErrorInd_Handler;
  #elif (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
    tCallbackData.pfvDataErrorInd = NULL;
  #endif

  /* now register signal at CSM */
  lRetVal = CSM_lApplCallbackInit((void *)phCSM,
                                  bBus,
                                  dwProtocolType,
                                  pvAddressField,
                                  &tCallbackData);

  return lRetVal;
}
#endif /*CSM_C_PTYPE_J1939 */


/*******************************************************************************
 * function    lRegisterCSM_OSEKI_lApplCallbackInit
 * \doxydocu
 * \brief      Registers callbacks for RN Osek indirect NM
 *
 * \param[in]  pvHandle
 *             CSM Access Layer Handle
 *
 * \param[in]  bBus
 *             CAN bus number
 *
 * \param[in]  dwProtocolType
 *             protocol type
 *
 * \param[in]  pvAddressField
 *             addressing, depends on protocol
 *
 * \param[in]  pvCallBackFkt
 *             Callback function(s)
 *
 * \return     error code
 *
 * \access     ?
 * \reentrant  ?
 *
 * \lastreview
 * \history_begin
 * 02.06.15 M.Prhl
 * Initial Revision.
 * \history_end
 ******************************************************************************/
#ifdef CSM_C_PTYPE_RN_OSEKI_NWM
static LONG lRegisterCSM_OSEKI_lApplCallbackInit(void * pvHandle, BYTE bBus, DWORD dwProtocolType, const void * pvAddressField,
                                                 const void * pvCallBackFkt)
{
  LONG lRetVal;
  pApplNotificationLookupEntry phCSM;

  tCSM_OSEKI_NWM_APPL_CALLBACK tCallbackData = { 0 }; /* default = NULL */
  tCSM_OSEKI_NWM_APPL_CALLBACK* pCallBackFkt;

  pCallBackFkt = (tCSM_OSEKI_NWM_APPL_CALLBACK*)pvCallBackFkt;

  phCSM = CSMAL_lAddApplNotification(pvHandle, bBus, dwProtocolType, pvAddressField,
                                     NULL,
                                     pCallBackFkt->pfvCommunicationInd
                                     #if (CSM_S_CNP_AVAILABLE == CSM_C_F_ON)
                                     ,NULL
                                     ,NULL

                                     #if (CSM_S_MPDT == CSM_C_F_ON )
                                     ,NULL                                                                // DataCon2 not used
                                     ,NULL                                                                // DataInd2 not used
                                     #endif // CSM_S_MPDT

                                     #if (CSM_S_ISO_TP == CSM_C_F_ON)
                                     #if (CSM_S_ITP_FF_INDICATION_AVAILABLE == CSM_C_F_ON)
                                     ,NULL
                                     #endif
                                     #endif // CSM_S_ISO_TP

                                     #if (CSM_S_DATA_ERROR_IND_AVAILABLE == CSM_C_F_ON)
                                     ,NULL
                                     #endif
                                     #endif // CSM_S_CNP_AVAILABLE
                                     );

  // copy relevant data to temporary NMW_APPL_CALLBACK structure. rTaskInfo and
  // pvHandle are calculated and stored in CSM_lApplCallbackInit
  tCallbackData.u16ApplID = pCallBackFkt->u16ApplID;
  tCallbackData.pfvCommunicationInd = CSMAL_vCommunicationInd_Handler;

  // now register protocoltype at CSM
  lRetVal = CSM_lApplCallbackInit((void *)phCSM, bBus, dwProtocolType, pvAddressField, &tCallbackData);

  return lRetVal;
}
#endif // CSM_C_PTYPE_RN_OSEKI_NWM

//***********************************************************************************************************************
// local helper function
//***********************************************************************************************************************
VOID CSMAL_vSendMessage(void *pvHandle, void const *pvMsg, LONG lMessageLength)
{
  // This function is sending data to the service queue of an access usr instance

  pCSMHandle pAccessUsrHandle;
  tChar acMessageQueueName[OSAL_C_U32_MAX_NAMELENGTH];
  tChar acNamePrefix[] = CSMAL_C_SERVICE_MESSAGE_QUEUE_NAME_PREFIX;
  OSAL_tMQueueHandle hMessageQueueID = OSAL_C_INVALID_HANDLE;
  LONG lMsgQElemSize;

  // validate handle
  if (( NULL != pvHandle) && (CSM_C_INVALID_HANDLE != pvHandle))
  {
    // cast handle to correct type
    pAccessUsrHandle = (pCSMHandle) pvHandle;
    // get the queue element size of the instance
    lMsgQElemSize = pAccessUsrHandle->lMQueueElemSize;
    // create queue name from data given by handle
    sprintf(acMessageQueueName, "%s%08x_%04d", acNamePrefix, (unsigned int) pAccessUsrHandle->lTaskID, (int) pAccessUsrHandle->lMQueueID);

    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_USR4(("CSMAL_vSendMessage - MsgLen:%d  MQElemSize:%d  MQName:%s",
                    lMessageLength,
                    lMsgQElemSize,
                    acMessageQueueName
                  ));
    #endif

    // this check is new, since the MQ element size can now be parametrized by the application.
    // The default value is still CSM_C_SERVICE_MESSAGE_QUEUE_MAX_MESSAGE_SIZE but can be decreased by
    // the corresponding parameter in vApplCallbackPreInit in access_usr module.
    if (lMessageLength <= lMsgQElemSize)
    {
      // open message queue
      if (OSAL_OK == OSAL_s32MessageQueueOpen(acMessageQueueName, OSAL_EN_READWRITE, &hMessageQueueID))
      {
        // post message and close queue
        OSAL_s32MessageQueuePost(hMessageQueueID, (tPCU8)pvMsg, (DWORD) lMessageLength, OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST);
        OSAL_s32MessageQueueClose(hMessageQueueID);
      }
      else
      {
        #if (CSM_S_TRACE == CSM_C_F_ON)
        ETG_TRACE_ERR(("CSMAL_vSendMessage - ERROR - Could not open message queue"));
        #endif
      }
    }
    else
    {
      #if (CSM_S_TRACE == CSM_C_F_ON)
      ETG_TRACE_FATAL(("CSMAL_vSendMessage - ERROR - Message length exceeds message queue element size"));
      #endif
      BOOL Error = FALSE;
      CSM_vAssert(Error);
    }
  }
  else
  {
    #if (CSM_S_TRACE == CSM_C_F_ON)
    ETG_TRACE_ERR(("CSMAL_vSendMessage - ERROR - Invalid handle"));
    #endif
  }
}

#endif /* CSM_S_DOXYGEN_HIDE_NON_API_FCT */
/*******************************************************************************
 * function implementation (scope: CAN stack #1 API)
 ******************************************************************************/

#undef CSM_ACCESS_KRNL_SELF
#undef CSM_CAN_STACK
/*******************************************************************************
 *                MODULE GROUPING INFORMATION FOR DOXYGEN
 * \doxydocu
 *
 * \defgroup      CSM_ACCESS_LAYER   CSM (CAN Stack Manager) Access Layer
 *                This group contains all files belonging to CAN Stack Manager
 *                Access Layer for T-Kernel process model
 */
/**
 * \defgroup      CSM_ACCESS_KRNL   CSM Access Layer Kernel part
 *                This group contains all files belonging to CAN Stack Manager
 *                Access Layer Kernel part for T-Kernel process model
 *
 * \ingroup       CSM_ACCESS_LAYER
 *
 */
/*******************************************************************************
 * \doxydocu
 * \file          csm_access_krnl.c
 * \brief         Functionality for usermode access to CSM
 *                This library must be linked to the kernel (interacts with
 *                CSM_SUBSYS)
 *
 * \project       Nissan LCN2 kai
 * \path          /di_can/adapt/osal/csm_access_krnl/sources/
 *
 * \ingroup       CSM_ACCESS_KRNL
 *
 * \authors
 *
 * COPYRIGHT      (c) 2011 Bosch GmbH
 *
 * \history_begin
 *
 * 17.11.11  /main/1 Kollai
 * - Initial revision.
 *
 * 16.05.12  /main/2 Kollai Kempen
 * - new local functions: lRegisterCSM_MPDT_C_ApplCallback, CSMAL_vSendMessage
 * - new function prototypes: CSMAL_lUnregisterApp,
 *                            CSMAL_lUnregisterSignal,
 *                            CSMAL_lUnregisterAllSignals
 * - renamed function prototypes: CSMAL_lGetAccessHandle,
 *                                CSMAL_lStoreHandleData
 *
 * 02.05.13  /main/4 Kempen
 * - some Problems with Semaphores fixed.
 *
 * 08.07.13  /main/5
 * - update from KPK for gen3 enviroment.
 *
 * 15.07.13  /main/6
 * - update from Reinhold for BAP support.
 *
 * 20.11.13  /main/7
 * - trace functionality uses now ETG traces
 *
 * 02.06.15  /main/8
 * - support for CSM_C_PTYPE_RN_OSEKI_NWM added.
 *
 * 01.10.15  /main/9  A.Borck
 * - some adaptions to identify invalid parameters -> helped to find iMX resets due to new CBR_AL
 *
 * 14.09.16  /main/10 A.Borck
 * - CSM_C_PTYPE_RN_MPDT_D allowed as well where CSM_C_PTYPE_RN_MPDT_C is allowed
 *
 * 20.10.16  /main/11 A.Borck
 * - intermediate version regarding MPDT D-Data extension
 *
 * 27.10.16  /main/12 A.Borck
 * - Notification lookup table extended by 2 indizees for MPDT
 * - AddApplNotification extended for 2 further Callbacks (MPDT special)
 * - complete rework of trace output
 * - ApplNotification automaticall adds a second notification for MPDT D-Data Confirmation and Indication if valid pointers are given
 * - DataCon and DataInd handler modified for proper MPDT D-Data handling
 * - all calls of ApplCallbackInit extended by passing 2 additional pointer params, if MPDT is enabled.
 *
 * 27.10.16  /main/13 A.Borck
 * - Indention fixed for some feature switches, no functional changes
 *
 * 22.11.16  /main/14 A.Borck
 * - access handle table is now additionally storing the element size of a service queue item
 * - as an example, the DataInd Handler is checking this element size before calling SendMessage
 * - the SendMessage function is also checking, since it is the central point for all handler functions
 * - SendMessage is doing an ASSERT if MsgSize exceeds the QueueElementSize
 * - Trace outputs have been modified, unified or extended for the entire module
 *
 * 23.11.16  /main/15 A.Borck
 * - some trace outputs reworked: content of AddrField traced not the pointer to content
 *
 * 30.11.16  /main/16 A.Borck
 * - more trace outputs reworked
 * - for the notification add functions, the address field is now stored in the table as well
 *
 * 28.02.17  /main/17 A.Borck
 * - bug fixes for Signal and Appl DeRegistration !! (pointer cast was wrong)
 * - LINT issues fixed
 * - trace output added
 *
 * 19.05.17  /main/18 A.Borck
 * - small fix in trace output line
 *
 * 24.05.17  /main/19 A.Borck
 * - modified a confusing trace output, if handling ISO-TP besides MPDT -> for DataCon
 *
 * 30.05.17  /main/20 A.Borck
 * - modified a confusing trace output, if handling ISO-TP besides MPDT -> this time for DataInd
 *
 * 10.08.17  /main/21 A.Borck
 * - final optimisations regarding TRACE output -> USER3 should be used as standard, USER4 is with CBR output
 *
 * \history_end
 ***** END OF FILE ***********************************************************/
