/*******************************************************************************************
* FILE:			 dl_V850UpdateCtrlApp.cpp
*
* SW-COMPONENT: Software Download
*
* DESCRIPTION:  This file is a Test Application to v850UpdateCtrl.
*				    
*
* AUTHOR:		 Aditya Kumar Jha
*
* COPYRIGHT:	 (c) 2013  2014 Robert Bosch Engineering and Business Solutions Ltd, Bangalore.
*
*********************************************************************************************
* HISTORY:
*					10.09.2013	Aditya Kumar Jha
*					Initial Version.
*					18.02.2014	Aditya Kumar Jha : Change for command option.
*					03.04.2014	Aditya Kumar Jha : Added ERROR Code and history for all functions.
*					12.06.2014  Aditya Kumar Jha , Neera.K.Ponnappa : Implemented No CRC and No Reset 
*                   30.06.2014  Neera.K.Ponnappa : Implemented Security Bytes and Jtag ID
*                   25.07.2014  Neera.K.Ponnappa : Implemented ClearCase label entry to registry
*                   06.08.2014  Neera.K.Ponnappa : Bug Fix CMG3GB-reset issue when Security Bytes/Jtag Bytes executed.
*----------------------------------------------------------------------------------------------
* DATE			| VERSION		| AUTHOR		| MODIFICATION
* ---------------------------------------------------------------------------------------------
* 21-05-2015	| 2.0			| VRI7COB(RBEI)	| Changes are Made to Adapt For Adding the New Command line
* 				|				|				| Parameter "screenreinit"
*----------------------------------------------------------------------------------------------
* ---------------------------------------------------------------------------------------------
* 19-02-2016	| 2.0			| VRI7COB(RBEI)	| Changes Done For New CRQ-CMG3G9596
* 				|				|				| New Command Line Arguments are added.
*				|				|				| /opt/bosch/base/bin/swu_common_v850_app_out.out -s 0x01 -j 0x010203040506070809101112
*               |               |               | /opt/bosch/base/bin/swu_common_v850_app_out.out -s 0x01
*               |               |               | /opt/bosch/base/bin/swu_common_v850_app_out.out -j 0x010203040506070809101112
*				|				|				| /opt/bosch/base/bin/swu_common_v850_app_out.out -jverify 0xbb99ff8a
*				|				|				| /opt/bosch/base/bin/swu_common_v850_app_out.out -sverify 0x08
*				|				|				| /opt/bosch/base/bin/swu_common_v850_app_out.out -sverify 0x08  -jverify 0xbb99ff8a
*				|				|				| ./swu_common_v850_app_out.out -jverify 0xbb99ff8a -sread -token 12345 -o /tmp/text.txt
*				|				|				| ./swu_common_v850_app_out.out -s 0x1F -j 0x...
*----------------------------------------------------------------------------------------------
*25-04-2016	    | 2.1	        | VRI7COB (RBEI)| Fixing the Issue GMMY17-13603
*----------------------------------------------------------------------------------------------
* 18-05-2016	| 2.2			| VRI7COB (RBEI)| Fixing the Issue Little Endian and Big Endian Conversion For Setting JTAG.
*				|				|				|  Issue Number: CMG3GB-3323
*----------------------------------------------------------------------------------------------
* 18-07-2016	| 2.3			| VRI7COB		| Fixing the Issue CMG3GB-3514
*----------------------------------------------------------------------------------------------
* 14-09-2016	| 2.4			| VRI7COB		| Fixing the Issue CMG3GB-3637|
*----------------------------------------------------------------------------------------------
********************************************************************************************/

#include "dl_V850ClientInterface.h"
using namespace std;
#include <string>
#include <sstream>
#include <list>
#include <algorithm>
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
#include <byteswap.h>
#include <iomanip>
#define ETG_ENABLED
#include "trace_interface.h"

#define ONE_ARG		         1
#define TWO_ARG		         2
#define THREE_ARG		     3
#define FOUR_ARG		     4
#define FIVE_ARG		     5
#define SIX_ARG		         6
#define SEVEN_ARG			 7
#define EIGHT_ARG			 8


#define VER_FLAG		     7
#define CRC_FLAG		     8
#define SEC_FLAG		     9
#define JTAG_FLAG		    10
#define INITAPP_FLAG		11
#define UPDATE_FLAG		    12
#define RESET_FLAG		    13

const int JTAG_MAX_LENGTH = 24;
const int JTAG_VALIDATE_MAX_LENGTH = 8; //32 Bits


T_fCallback CALLBACK;

std::string sBinaryFilePath = "";
std::string sBinaryName = "";
std::string sVerCrcPath = "";
std::string sSecurityByte = "";
std::string sJtagByte = "";

//Start of CRQ-CMG3G9596
std::string sSecurityValidateByte = "";
bool bSecurityValidation = false;

std::string sJTAGValidateByte = "";
bool bJTAGValidation = false;
bool bSecurityByteRead = false;
string strToken("");
string strWriteFile("");
//End of CRQ-CMG3G9596

bool bVerDisplay = false;
bool bCrcDisplay = false;
bool bIsDownloadable = false;
bool bIsOnlyInit = false;
bool bWriteIntoFileOption = false;
bool bResetEnabled = true;
bool bCrcEnabled = true;
bool bResetReq = false;
static bool bSecurityAccess = false;
static bool bJtagAccess = false;


int    FlagVal = 0;
dl_tclV850ClientInterface* pclV850ClientIf = NULL;
std::list<EN_EXTRA_OPERATIONS> g_ListOfOpera;

void help( void );
void vGetCmdLineArgs(char *argv[]);
void vParseCmdLineArgs( int argc, char *argv[]);  
int iProcessExecution( void );
int iGetSwVersion( void );
int iGetCrcValue( void );
int iUpdateBinary( void );
int iUpdateValueToFile( const char * chTempStr );
void vParseOneAgr(char *argv[]);
void vParseTwoAgrs(char *argv[]);
void vParseThreeAgrs(char *argv[]);
void vParseFourAgrs(int argc, char *argv[]);
void vParseFiveAgrs(int argc, char *argv[]);
void vParseSixAgrs(int argc, char* argv[]);
void vParseSevenArgs(int argc, char* argv[]);
int iAssignBinaryPathAndName( char *argv[], int iIndex );
int iAssignBinaryName( char *argv[], int iIndex );
int iAssignPathToWriteCrc( char *argv[], int iIndex );
int iAssignSecurityByte( char *argv[], int iIndex );
int iAssignJtagByte( char *argv[], int iIndex );

//Start of CRQ-CMG3G9596
int iAssignSecurityValidateByte(char *argv[], int iIndex);
int iAssignJTAGValidateByte(char *argv[], int iIndex);
//End of  CRQ-CMG3G9596



int main( int argc, char *argv[] )
{
   vInitPlatformEtg();

	int iRetVal = INIT_SUCCESS;
	g_ListOfOpera.clear();
	if (argc == 1)  
	{
		bIsOnlyInit = true;
	
	}
	else if (argc > EIGHT_ARG)
	{ 
		//If not enough parameters have been passed, inform user and exit.
		INFO_MSG (INFO_LEVEL_1, "INFO: argc:%d\n", argc);
		help();
		OSAL_vSetProcessExitCode(ARGUMENT_ERROR);
        OSAL_vProcessExit();
	}
	//coverity fix for 51497
	try{
	pclV850ClientIf = new dl_tclV850ClientInterface();
	}
	catch (std::exception& e)
	{
		INFO_MSG(INFO_LEVEL_1,"bad_alloc caught in pclV850ClientIf for the new operator in v850UpdateCtrlApp file\n");
	}
	if ( pclV850ClientIf != NULL )
	{
		if( bIsOnlyInit == true )
		{
	        FlagVal = INITAPP_FLAG;
			//coverity fix for 51497
			try{
			if ( -1 == pclV850ClientIf->s32InitV850ClientHandler( CALLBACK,  bIsOnlyInit , FlagVal) )
				{
					iRetVal = INIT_ERROR;
				}
			}
			catch (std::exception& e)
			{
				INFO_MSG(INFO_LEVEL_1,"bad_alloc caught in s32InitV850ClientHandler function call in v850UpdateCtrlApp file\n");
				iRetVal = INIT_ERROR;
			}
		}
		else
		{
			//coverity fix for 51497
			try{
			vParseCmdLineArgs(argc, argv);   
			iRetVal = iProcessExecution();
			//pclV850ClientIf->KillManagerThread();
			}
			catch (std::exception& e)
			{
				INFO_MSG(INFO_LEVEL_1,"bad_alloc caught in vParseCmdLineArgs function call in v850UpdateCtrlApp file\n");
				iRetVal = INIT_ERROR;
			}
		}
		//If the socket failure happened after receiver thread creation then it needs to be killed.
		pclV850ClientIf->KillManagerThread();	
		delete pclV850ClientIf;
	}
	OSAL_vSetProcessExitCode(iRetVal);
    OSAL_vProcessExit();	
	return iRetVal;
}

/*******************************************************************************************
* FUNCTION:		help
* DESCRIPTION:	This function display the help for the application.
* PARAMETER:
*					tVoid
* RETURNVALUE:	tU8 
*
* HISTORY:
*					04.03.2014	Aditya Kumar Jha
*					Initial Version.
**----------------------------------------------------------------------------------------------
* DATE			| VERSION		| AUTHOR		| MODIFICATION
* ---------------------------------------------------------------------------------------------
* 29-05-2015	| 2.0			| VRI7COB(RBEI)	| Changes are Made to Adapt For Adding the New Command line
* 				|				|				| Parameter "screenreinit"
*----------------------------------------------------------------------------------------------
********************************************************************************************/
void help( void)
{
	printf("Usage: ./swu_common_v850_app_out.out : Check presence of binary data and read version\n"); 
	printf("Usage: ./swu_common_v850_app_out.out -reset: reset V850 and start in application mode\n");
	printf("\t ./swu_common_v850_app_out.out -v [-bm / -bl / -app]   : print version of binary file\n"); 
	printf("\t ./swu_common_v850_app_out.out -crc [-bm / -bl / -app]  : print checksum of binary file\n"); 
	printf("\t ./swu_common_v850_app_out.out -crc [-bm / -bl / -app] -f [Destination_File_Path] :\
			 print checksum of binary file and write into the file\n");
	printf("\t ./swu_common_v850_app_out.out [-bin / -dnl] [Binary File Path] [-bl / -app]	-screenreinit: Execute V850 download\n");
	printf("\t ./swu_common_v850_app_out.out [-bin / -dnl] [Binary File Path] [-bl / -app]	-nocrc -screenreinit:\
			 Execute V850 download without checking CRC\n");
	printf("\t ./swu_common_v850_app_out.out [-bin / -dnl] [Binary File Path] [-bl / -app]	-noreset -screenreinit:\
			 Execute V850 download and v850 remains in bootloader mode\n");
	printf("\t ./swu_common_v850_app_out.out [-bin / -dnl] [Binary File Path] [-bl / -app]	-nocrc -noreset -screenreinit:\
			 Execute V850 download without checking CRC and v850 remains in bootloader mode\n");
	printf("\t ./swu_common_v850_app_out.out -j [12bytes]; Set V850 JTAG Bytes\n");
	printf("\t ./swu_common_v850_app_out.out -s 0xBYTE[01 to 05]: Set V850 security Bytes\n \
            0x01 : Set write protection flag \n \
            0x02 : Set read protection flag \n \
            0x03 : Set Boot cluster  protection flag \n \
            0x04 : Set Block erase protection flag \n \
            0x05 : Set Chip erase protection flag \n");
	printf("\t ./swu_common_v850_app_out.out -s 0xBYTE -j 0xBYTE; Set the security Id and JTAG Id\n");
	printf("\t ./swu_common_v850_app_out.out -sverify 0xBYTE; Validate Security Bytes with 0xBYTE\n \
            0x01 : Boot block erase protection flag \n \
            0x02 : Block erase protection flag \n \
            0x04 : Chip erase protection flag  \n \
            0x08 : Write protection flag \n \
            0x10 : Read protection flag \n \
           RETURN: \n \
               0 : Security Verify Success \n \
              18 : Security Verify Failure \n");
	printf("\t ./swu_common_v850_app_out.out -jverify 0xBYTE; To Validate JTAG Fingerprint\n \
           RETURN: \n \
               0 : JTAG Verify Success \n \
              19 : JTAG Verify Failure \n \
              20 : JTAG Fingerprint is not Implemented - OLD V850 \n");
	printf("\t ./swu_common_v850_app_out.out  -sverify 0x08  -jverify 0xbb99ff8a\n \
           RETURN: \n \
               0 : Validation Sucess \n \
              18 : Secuirty Failure \n \
              19 : JTAG FingerPrint Failure \n \
              20 : JTAG Fingerprint is not Implemented - OLD V850\n \
       (20+18)38 : JTAG and Security is Failed\n \
       (19+18)37 : Security Failed and JTAG Fingerprint is not Implemented - OLD V850\n");
	printf("\t ./swu_common_v850_app_out.out  -jverify 0xbb99ff8a -sread -token <token> -o <outfile>\n \
	           RETURN: \n \
	               0 : Validation Sucess \n \
	              18 : Secuirty Failure \n \
	              19 : JTAG FingerPrint Failure \n \
	              20 : JTAG Fingerprint is not Implemented - OLD V850\n \
				  21 : Token is empty\n \
				  22 : File Name with path is empty\n \
				  You will get the sum of above integers depends on the failure.\n");

}


/*******************************************************************************************
* FUNCTION:		vGetCmdLineArgs
* DESCRIPTION:	This function validate ane read command line parameters.
* PARAMETER:
*					char *
* RETURNVALUE:	void 
*
* HISTORY:
*					04.03.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
void vGetCmdLineArgs(char *argv[])
{
	int iArgumentNumber = 1;
	
	//get the first argument
	if ( strcmp(argv[iArgumentNumber], "-v") == 0 )
	{
		bVerDisplay = true;
	}
	else if ( strcmp(argv[iArgumentNumber], "-crc") == 0 ) 
	{
		bCrcDisplay = true;				
	}
	else if ( (strcmp(argv[iArgumentNumber], "-bin") == 0)
		||
		(strcmp(argv[iArgumentNumber], "-dnl") == 0) )
	{
		iArgumentNumber++;
		if ( argv[iArgumentNumber] != NULL )
		{
			sBinaryFilePath.assign(argv[iArgumentNumber]);
			bIsDownloadable = true;	
		}
	}
	else
	{
		help();
	
	    OSAL_vSetProcessExitCode(ARGUMENT_ERROR);
        OSAL_vProcessExit();
	}

	//get the second argument: for which binary
	iArgumentNumber++;
	sBinaryName.assign(argv[iArgumentNumber]);

	if ( bWriteIntoFileOption == true )
	{
		//check if -f is present as a argument
		iArgumentNumber++;

		if ( strcmp(argv[iArgumentNumber], "-f") == 0 )
		{
			iArgumentNumber++;
			if ( argv[iArgumentNumber] != NULL )
			{
				sVerCrcPath.assign(argv[iArgumentNumber]);
			}
		}
	}
	
}
/*******************************************************************************************
* FUNCTION:		vParseCmdLineArgs
* DESCRIPTION:	This function validate ane read command line parameters.
* PARAMETER:
*					char *
* RETURNVALUE:	void 
*
* HISTORY:
*					04.03.2014	Aditya Kumar Jha
*					Initial Version.
**----------------------------------------------------------------------------------------------
* DATE			| VERSION		| AUTHOR		| MODIFICATION
* ---------------------------------------------------------------------------------------------
* 29-05-2015	| 2.0			| VRI7COB(RBEI)	| Changes are Made to Adapt For Adding the New Command line
* 				|				|				| Parameter "screenreinit"
*----------------------------------------------------------------------------------------------
********************************************************************************************/
void vParseCmdLineArgs(int argc, char *argv[])
{
	INFO_MSG (INFO_LEVEL_1, "INFO: argc:%d\n", argc);
	switch( argc )
	{
	default:
		{
		//help file for commandline args
		help();                                        
	    OSAL_vSetProcessExitCode(ARGUMENT_ERROR);
        OSAL_vProcessExit();
		break;
		}
	case ONE_ARG:
		break;
	case TWO_ARG:
		vParseOneAgr(argv);
		break;
	case THREE_ARG:
		vParseTwoAgrs(argv);
		break;
	case FOUR_ARG:
		vParseThreeAgrs(argv);
		break;
	case FIVE_ARG:
		vParseFourAgrs(argc, argv);
		break;
	case SIX_ARG:
		vParseFiveAgrs(argc, argv);
		break;
	case SEVEN_ARG:
		vParseSixAgrs(argc, argv);
		break;
	case EIGHT_ARG:
		vParseSevenArgs(argc, argv);
		break;
	}
}
/*******************************************************************************************
* FUNCTION:		vParseOneAgr
* DESCRIPTION:	This function validate read command line parameters and checks for the reset command.
* PARAMETER:
*					char *
* RETURNVALUE:	void 
*
* HISTORY:
*					04.03.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
void vParseOneAgr(char *argv[])
{
	int iArgumentNumber = 1;

	//check if reset is demanded
	if(strcmp(argv[iArgumentNumber], "-reset") == 0)   
	{
		bResetReq = true;
		bResetEnabled = true;
		FlagVal = RESET_FLAG;
		
	}
	else
	{
		//help file for commandline args
		help();                                        
	    OSAL_vSetProcessExitCode(ARGUMENT_ERROR);
        OSAL_vProcessExit();
	}


}
/*******************************************************************************************
* FUNCTION:		vParseTwoAgrs
* DESCRIPTION:	TThis function validate read command line parameters and checks the argument for Crc or version
* PARAMETER:
*					char *
* RETURNVALUE:	void 
*
* HISTORY:
*					04.03.2014	Aditya Kumar Jha
                    30.06.2014  Neera.K.Ponnappa
*					Initial Version.
********************************************************************************************/
void vParseTwoAgrs(char *argv[])
{
	int iArgumentNumber = 1;
			
	//check if crc is enabled
	if(strcmp(argv[iArgumentNumber], "-crc") == 0)            
	{
		bCrcDisplay = true;
		FlagVal = CRC_FLAG;
		iArgumentNumber = iAssignBinaryName( argv, iArgumentNumber );

	}
	//check if version check is enabled
	else if(strcmp(argv[iArgumentNumber], "-v") == 0)           
	{
		bVerDisplay = true;
		FlagVal = VER_FLAG;
		iArgumentNumber = iAssignBinaryName( argv, iArgumentNumber );
	}
	else if(strcmp(argv[iArgumentNumber], "-s") == 0)  
	{
		bSecurityAccess = true;
		FlagVal = SEC_FLAG;
		iArgumentNumber = iAssignSecurityByte( argv, iArgumentNumber );
	}
	else if(strcmp(argv[iArgumentNumber], "-j") == 0)  
	{
		bJtagAccess = true;
		FlagVal = JTAG_FLAG;
		iArgumentNumber = iAssignJtagByte( argv, iArgumentNumber );
	}	
	else if(strcmp(argv[iArgumentNumber], "-sverify") == 0)							//Start of CRQ-CMG3G9596
	{
		bSecurityValidation = true;
		iArgumentNumber = iAssignSecurityValidateByte( argv, iArgumentNumber );
	}
	else if(strcmp(argv[iArgumentNumber], "-jverify") == 0)
	{
		bJTAGValidation = true;
		iArgumentNumber = iAssignJTAGValidateByte( argv, iArgumentNumber );			//End of CRQ-CMG3G9596
	}
	else
	{
		//help file for commandline args
		help();                                        
	    OSAL_vSetProcessExitCode(ARGUMENT_ERROR);
        OSAL_vProcessExit();
	}


}
/*******************************************************************************************
* FUNCTION:		vParseThreeAgrs
* DESCRIPTION:	This function validate ane read command line parameters for bin or dnl file .
* PARAMETER:
*					char *
* RETURNVALUE:	void 
*
* HISTORY:
*					04.03.2014	Aditya Kumar Jha
                    30.06.2014  Neera.K.Ponnappa
*					Initial Version.
**----------------------------------------------------------------------------------------------
* DATE			| VERSION		| AUTHOR		| MODIFICATION
* ---------------------------------------------------------------------------------------------
* 29-05-2015	| 2.0			| VRI7COB(RBEI)	| Changes are Made to Adapt For Adding the New Command line
* 				|				|				| Parameter "screenreinit"
*----------------------------------------------------------------------------------------------
********************************************************************************************/
void vParseThreeAgrs(char *argv[])
{
	
	int iArgumentNumber = 1;

	//check if dnl file or bin file
	if( (strcmp(argv[iArgumentNumber], "-dnl") == 0)
		|| (strcmp(argv[iArgumentNumber], "-bin") == 0) )
	{
		iArgumentNumber = iAssignBinaryPathAndName( argv, iArgumentNumber );
        FlagVal = UPDATE_FLAG;

	}
	else if(strcmp(argv[iArgumentNumber],"-s") == 0)
	{
	
        bSecurityAccess = true;
		iArgumentNumber = iAssignSecurityByte( argv, iArgumentNumber );
		iArgumentNumber++;
		if(strcmp(argv[iArgumentNumber],"-noreset") == 0)
		{
          FlagVal = SEC_FLAG;
		  printf("WARN: No Reset-v850 remains in bootloader mode\n");
		  bResetEnabled = false;
		}
	}
	else if(strcmp(argv[iArgumentNumber],"-j") == 0)
	{
		
        bJtagAccess = true;
		iArgumentNumber = iAssignJtagByte( argv, iArgumentNumber );
		iArgumentNumber++;
		if(strcmp(argv[iArgumentNumber],"-noreset") == 0)
		{
          FlagVal = JTAG_FLAG;
		  printf("WARN: No Reset-v850 remains in bootloader mode\n");
		  bResetEnabled = false;
		}
	}
	else if(strcmp(argv[iArgumentNumber], "-screenreinit") == 0)
		g_ListOfOpera.push_back(EN_SCREENREINIT_OPERATION);
	else if(strcmp(argv[iArgumentNumber], "-sverify") == 0)							//Start of CRQ-CMG3G9596
	{
		bSecurityValidation = true;
		iArgumentNumber = iAssignSecurityValidateByte( argv, iArgumentNumber );
		iArgumentNumber++;
		if(strcmp(argv[iArgumentNumber], "-noreset") == 0)
		{
			INFO_MSG (INFO_LEVEL_1, "INFO: No Reset\n");
			bResetEnabled = false;
		}
	}
	else if(strcmp(argv[iArgumentNumber], "-jverify") == 0)
	{
		bJTAGValidation = true;
		iArgumentNumber = iAssignJTAGValidateByte( argv, iArgumentNumber );
		iArgumentNumber++;
		if(strcmp(argv[iArgumentNumber], "-noreset") == 0)
		{
			INFO_MSG (INFO_LEVEL_1, "INFO: No Reset\n");
			bResetEnabled = false;
		}

	}																			//End of CRQ-CMG3G9596
	else
	{
		//help file for commandline args
		help();                                        
		OSAL_vSetProcessExitCode(ARGUMENT_ERROR);
        OSAL_vProcessExit();
	}

}
/*******************************************************************************************
* FUNCTION:		vParseFourAgrs
* DESCRIPTION:	This function validate ane read command line parameters.
* PARAMETER:
*					int , char *
* RETURNVALUE:	void 
*
* HISTORY:
*					04.03.2014	Aditya Kumar Jha
*					Initial Version.
**----------------------------------------------------------------------------------------------
* DATE			| VERSION		| AUTHOR		| MODIFICATION
* ---------------------------------------------------------------------------------------------
* 29-05-2015	| 2.0			| VRI7COB(RBEI)	| Changes are Made to Adapt For Adding the New Command line
* 				|				|				| Parameter "screenreinit"
*----------------------------------------------------------------------------------------------
********************************************************************************************/
void vParseFourAgrs(int argc, char *argv[])
{
	for( int iArgumentNumber = 1; iArgumentNumber< argc; )
	{
		 //check if dnl file or bin file
		if( (strcmp(argv[iArgumentNumber], "-dnl") == 0)
			|| (strcmp(argv[iArgumentNumber], "-bin") == 0) )
		{
			iArgumentNumber = iAssignBinaryPathAndName( argv, iArgumentNumber );
			FlagVal = UPDATE_FLAG;
		}
		//check if crc check is disabled
		else if(strcmp(argv[iArgumentNumber], "-nocrc") == 0)              
		{
			printf("WARN: Execute V850 download without checking CRC\n");
			bCrcEnabled = false;
		}
		//check if no reset
		else if(strcmp(argv[iArgumentNumber], "-noreset") == 0)          
		{
			printf("WARN: No Reset-v850 remains in bootloader mode\n");
			bResetEnabled = false;
		}
		//check if crc is enabled
		else if(strcmp(argv[iArgumentNumber], "-crc") == 0)            
		{
			bCrcDisplay = true;
			iArgumentNumber = iAssignBinaryName( argv, iArgumentNumber );
		}
		//check if crc storage file path check is specified
		else if(strcmp(argv[iArgumentNumber], "-f") == 0)        
		{
			iArgumentNumber = iAssignPathToWriteCrc( argv, iArgumentNumber );
		}
		else if(strcmp(argv[iArgumentNumber], "-screenreinit") == 0)
			g_ListOfOpera.push_back(EN_SCREENREINIT_OPERATION);

		else if(strcmp(argv[iArgumentNumber], "-s") == 0)							//Start of CRQ-CMG3G9596
		{
			iArgumentNumber = iAssignSecurityByte(argv, iArgumentNumber);
			iArgumentNumber++;
			bSecurityAccess = true;
			FlagVal = SEC_FLAG;

			if(strcmp(argv[iArgumentNumber], "-j") == 0)
			{
				iArgumentNumber = iAssignJtagByte(argv, iArgumentNumber);
				bJtagAccess = true;
				FlagVal = JTAG_FLAG;
			}
		}																			//End of CRQ-CMG3G9596
		else if(strcmp(argv[iArgumentNumber], "-sverify") == 0)
		{
			bSecurityValidation = true;
			iArgumentNumber = iAssignSecurityValidateByte( argv, iArgumentNumber );
			iArgumentNumber++;
			if(strcmp(argv[iArgumentNumber], "-jverify") == 0)
			{
				bJTAGValidation = true;
				iArgumentNumber = iAssignJTAGValidateByte( argv, iArgumentNumber );
				iArgumentNumber++;
			}
		}
		else
		{
			//help file for commandline args
			help();                                        
		    OSAL_vSetProcessExitCode(ARGUMENT_ERROR);
            OSAL_vProcessExit();
		}

		iArgumentNumber++;
	}
}
/*******************************************************************************************
* FUNCTION:		vParseFiveAgrs
* DESCRIPTION:	This function validate ane read command line parameters.
* PARAMETER:
*					int , char *
* RETURNVALUE:	void 
*
* HISTORY:
*					04.03.2014	Aditya Kumar Jha
*					Initial Version.
**----------------------------------------------------------------------------------------------
* DATE			| VERSION		| AUTHOR		| MODIFICATION
* ---------------------------------------------------------------------------------------------
* 29-05-2015	| 2.0			| VRI7COB(RBEI)	| Changes are Made to Adapt For Adding the New Command line
* 				|				|				| Parameter "screenreinit"
*----------------------------------------------------------------------------------------------
********************************************************************************************/
void vParseFiveAgrs(int argc, char *argv[])
{
	for( int iArgumentNumber = 1; iArgumentNumber< argc; )
	{
		 //check if dnl file or bin file
		if( (strcmp(argv[iArgumentNumber], "-dnl") == 0)
			|| (strcmp(argv[iArgumentNumber], "-bin") == 0) )
		{
			iArgumentNumber = iAssignBinaryPathAndName( argv, iArgumentNumber );
			FlagVal = UPDATE_FLAG;
		}
		//check if crc check is disabled
		else if(strcmp(argv[iArgumentNumber], "-nocrc") == 0)              
		{
			printf("WARN: Execute V850 download without checking CRC\n");
			bCrcEnabled = false;
		}
		//check if no reset
		else if(strcmp(argv[iArgumentNumber], "-noreset") == 0)          
		{
			printf("WARN: No Reset-v850 remains in bootloader mode\n");
			bResetEnabled = false;
		}
		else if(strcmp(argv[iArgumentNumber], "-screenreinit") == 0)
			g_ListOfOpera.push_back(EN_SCREENREINIT_OPERATION);
		else
		{
			//help file for commandline args
			help();                                        
		    OSAL_vSetProcessExitCode(ARGUMENT_ERROR);
            OSAL_vProcessExit();
		}

		iArgumentNumber++;
	}
}

/*******************************************************************************************
* FUNCTION:		vParseSixAgrs
* DESCRIPTION:	This function validate ane read command line parameters.
* PARAMETER:
*					int , char *
* RETURNVALUE:	void
*
* HISTORY:
*					04.03.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
void vParseSixAgrs(int argc, char *argv[])
{
	for( int iArgumentNumber = 1; iArgumentNumber< argc; )
	{
		 //check if dnl file or bin file
		if( (strcmp(argv[iArgumentNumber], "-dnl") == 0)
			|| (strcmp(argv[iArgumentNumber], "-bin") == 0) )
		{
			iArgumentNumber = iAssignBinaryPathAndName( argv, iArgumentNumber );
			FlagVal = UPDATE_FLAG;
		}
		//check if crc check is disabled
		else if(strcmp(argv[iArgumentNumber], "-nocrc") == 0)
		{
			printf("WARN: Execute V850 download without checking CRC\n");
			bCrcEnabled = false;
		}
		//check if no reset
		else if(strcmp(argv[iArgumentNumber], "-noreset") == 0)
		{
			printf("WARN: No Reset-v850 remains in bootloader mode\n");
			bResetEnabled = false;
		}
		else if(strcmp(argv[iArgumentNumber], "-screenreinit") == 0)
			g_ListOfOpera.push_back(EN_SCREENREINIT_OPERATION);
		else
		{
			//help file for commandline args
			help();
		    OSAL_vSetProcessExitCode(ARGUMENT_ERROR);
            OSAL_vProcessExit();
		}

		iArgumentNumber++;
	}
}


void vParseSevenArgs(int argc, char *argv[])
{
	for( int iArgumentNumber = 1; iArgumentNumber< argc; )
	{
		if(strcmp(argv[iArgumentNumber], "-jverify") == 0)
		{
			bJTAGValidation = true;
			iArgumentNumber = iAssignJTAGValidateByte( argv, iArgumentNumber );

		}
		else if(strcmp(argv[iArgumentNumber], "-sread") == 0)
			bSecurityByteRead = true;

		else if(strcmp(argv[iArgumentNumber], "-token") == 0)
		{
			iArgumentNumber++;
			strToken.clear();
			strToken.assign(argv[iArgumentNumber]);
		}
		else if(strcmp(argv[iArgumentNumber], "-o") == 0)
		{
			iArgumentNumber++;
			strWriteFile.clear();
			strWriteFile.assign(argv[iArgumentNumber]);
		}
		else
		{			
			help();
		    OSAL_vSetProcessExitCode(ARGUMENT_ERROR);
            OSAL_vProcessExit();
		}
		iArgumentNumber++;
	}
}




	

/*******************************************************************************************
* FUNCTION:		iProcessExecution
* DESCRIPTION:	This function process the execution.
* PARAMETER:
*					void
* RETURNVALUE:	int 
*
* HISTORY:
*					04.03.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
int iProcessExecution( void )
{
	int iRetVal = INIT_ERROR;
	int iRetSecValidation = INIT_SUCCESS;
	if ( pclV850ClientIf != NULL )
	{		
		if ( -1 == pclV850ClientIf->s32InitV850ClientHandler( CALLBACK , false , FlagVal) )
		{
			return INIT_ERROR;
		}

		//display version info
		if(true == bVerDisplay)
		{
			iRetVal = iGetSwVersion();
		}

		//display Check sum Value
		if(true == bCrcDisplay)
		{
			iRetVal = iGetCrcValue();
		}

		//update V850 binary
		if(true == bIsDownloadable)
		{
			iRetVal = iUpdateBinary();
		}

		if( (true == bResetReq)
			&&
			(true == bResetEnabled) )
		{
			pclV850ClientIf->vOnResetReq( );
			iRetVal = INIT_SUCCESS;
		}
		if(true == bSecurityAccess)
		{
			INFO_MSG (INFO_LEVEL_1, "INFO: Security Access");

			if(bJtagAccess)
				pclV850ClientIf->vOnGetSecurityByteVal(sSecurityByte,false);
			else
				pclV850ClientIf->vOnGetSecurityByteVal(sSecurityByte,true);

			iRetVal = INIT_SUCCESS;
			
		}
		if(true == bJtagAccess)
		{
			INFO_MSG (INFO_LEVEL_1, "\nINFO: JTAG Access");
		    pclV850ClientIf->vOnGetJtagByteVal(sJtagByte,bResetEnabled);
			iRetVal = INIT_SUCCESS;
		}
		if(true == bSecurityValidation) //CRQ-CMG3G9596
		{
			if(bJTAGValidation)
				iRetSecValidation = pclV850ClientIf->u16DoSecurityValidation(sSecurityValidateByte, false);
			else
				iRetVal = pclV850ClientIf->u16DoSecurityValidation(sSecurityValidateByte, true);
		}
		if(true == bJTAGValidation)//CRQ-CMG3G9596
		{
			int iRetJTAGValidation = INIT_SUCCESS;
			if(bSecurityByteRead)
				iRetJTAGValidation = pclV850ClientIf->u16DoJTAGFingerprintValidation(sJTAGValidateByte, false);
			else
				iRetJTAGValidation = pclV850ClientIf->u16DoJTAGFingerprintValidation(sJTAGValidateByte, true);
			iRetVal = iRetSecValidation + iRetJTAGValidation;
		}
		if(bSecurityByteRead )
		{
			string  strSecurityVal = pclV850ClientIf->strReadSecuiryBitFromV850();
			if(strSecurityVal.empty())
			{
				iRetVal = iRetVal + SECURITY_ERROR;
				strSecurityVal.assign("Issue in Security bits");
			}

			if(strToken.empty())
			{
				iRetVal = iRetVal + TOKEN_NOT_PRESENT;
				strToken.assign("Empty Token");
			}

			if(!strWriteFile.empty())
			{
				std::ofstream outfile (strWriteFile.c_str());
				outfile << strToken << std::endl;
				outfile << strSecurityVal << std::endl;
				outfile.close();
			}
			else
				iRetVal = iRetVal + INVALID_FILE;
		}

	}

	return iRetVal;
}

/*******************************************************************************************
* FUNCTION:		iGetSwVersion
* DESCRIPTION:	This function call the read SW Version interface.
* PARAMETER:
*					void
* RETURNVALUE:	int 
*
* HISTORY:
*					04.03.2014	Aditya Kumar Jha
*					Initial Version.
                    25.07.2014  Neera.K.Ponnappa : Implemented ClearCase label entry to registry
********************************************************************************************/
int iGetSwVersion( void )
{
	int iRetVal = -1;
	std::string version = "";
	std::string label = "";
	printf("INFO: Display version info\n");
	iRetVal = pclV850ClientIf->s8OnGetSWVer(sBinaryName, version , label);
	if( version.compare("") )
	{
		printf("INFO: %s version : %s\n", sBinaryName.c_str(), version.c_str());

		if( iRetVal )
		{
			printf("ERROR: Failed to Store %s version in registry\n", sBinaryName.c_str());
		}
	}
	if( label.compare("") )
	{
		printf("INFO: %s label : %s\n", sBinaryName.c_str(), label.c_str());

		if( iRetVal )
		{
			printf("ERROR: Failed to Store %s label in registry\n", label.c_str());
		}
	}
	return iRetVal;

}

/*******************************************************************************************
* FUNCTION:		iGetCrcValue
* DESCRIPTION:	This function call the read CRC value interface.
* PARAMETER:
*					void
* RETURNVALUE:	int 
*
* HISTORY:
*					04.03.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
int iGetCrcValue( void )
{
	int iRetVal = -1;
	std::string checksum = "";

	printf("INFO: Display Check sum Value\n");
	iRetVal = pclV850ClientIf->s8OnGetCheckSumVal(sBinaryName, checksum);
	if( iRetVal == -1 )
	{		
		printf("ERROR: Failed to get %s CRC\n", sBinaryName.c_str());
	}
	else
	{		
		if ( sVerCrcPath != "" )
		{
			iRetVal = iUpdateValueToFile( checksum.c_str() );
		}		
	}
	if( checksum.compare("") )
	{
		printf("INFO: %s CRC Value: %s\n", sBinaryName.c_str(), checksum.c_str());
	}

	return iRetVal;

}

/*******************************************************************************************
* FUNCTION:		iUpdateBinary
* DESCRIPTION:	This function call the download start interface.
* PARAMETER:
*					void
* RETURNVALUE:	int 
*
* HISTORY:
*					04.03.2014	Aditya Kumar Jha
*					Initial Version.
**----------------------------------------------------------------------------------------------
* DATE			| VERSION		| AUTHOR		| MODIFICATION
* ---------------------------------------------------------------------------------------------
* 29-05-2015	| 2.0			| VRI7COB(RBEI)	| Changes are Made to Adapt For Adding the New Command line
* 				|				|				| Parameter "screenreinit"
*----------------------------------------------------------------------------------------------
********************************************************************************************/
int iUpdateBinary( void )
{
	int iRetVal = UPDATE_SUCCESS;

	if( false == pclV850ClientIf->bOnDLStart( sBinaryFilePath.c_str(), sBinaryName.c_str(), g_ListOfOpera, bCrcEnabled, bResetEnabled ) )
	{
		iRetVal = UPDATE_FAILED;
	}

	return iRetVal;

}

/*******************************************************************************************
* FUNCTION:		iUpdateValueToFile
* DESCRIPTION:	This function write the value into given file.
* PARAMETER:
*					void
* RETURNVALUE:	int 
*
* HISTORY:
*					04.03.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
int iUpdateValueToFile( const char * chTempStr )
{
	int fd = -1;

	if ( sVerCrcPath == "" )
	{
		return FILE_CREATE_ERROR;
	}

	/* Open the file */
	fd = open(sVerCrcPath.c_str(), O_RDWR, S_IRUSR | S_IRGRP | S_IROTH);
	if ( -1 == fd )
	{
		/* Create the file */
		fd = open(sVerCrcPath.c_str(), O_RDWR | O_CREAT, S_IRUSR | S_IRGRP | S_IROTH);
		printf("WARN: CRC File not found! Creating...\n");
		if (-1 == fd )
		{
			printf("ERROR: CRC File could not be created!\n");
			return FILE_CREATE_ERROR;
		}

	}

	//write the file 
	write(fd, chTempStr, strlen(chTempStr));

	if ( -1 != fd )
	{
		close(fd);		
	}

	return CRC_SUCCESS;

}

/*******************************************************************************************
* FUNCTION:		iAssignBinaryPathAndName
* DESCRIPTION:	This function assign the binary location and type of binary to be flash into buffer.
* PARAMETER:
*					char *, int
* RETURNVALUE:	int ( return the incremented value of argv index )
*
* HISTORY:
*					11.06.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
int iAssignBinaryPathAndName( char *argv[], int iIndex )
{
	iIndex++;
	if ( argv[iIndex] != NULL )
	{
		sBinaryFilePath.assign(argv[iIndex]);
		bIsDownloadable = true;	
	}
	iIndex++;
	if ( argv[iIndex] != NULL )
	{
		sBinaryName.assign(argv[iIndex]);
	}

	return iIndex;

}

/*******************************************************************************************
* FUNCTION:		iAssignBinaryName
* DESCRIPTION:	This function assign the type of binary to be flash into buffer.
* PARAMETER:
*					char *, int
* RETURNVALUE:	int ( return the incremented value of argv index )
*
* HISTORY:
*					11.06.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
int iAssignBinaryName( char *argv[], int iIndex )
{
	iIndex++;
	if ( argv[iIndex] != NULL )
	{
		sBinaryName.assign(argv[iIndex]);
	}

	return iIndex;
}

/*******************************************************************************************
* FUNCTION:		iAssignPathToWriteCrc
* DESCRIPTION:	This function assign the file path to write crc value.
* PARAMETER:
*					char *, int
* RETURNVALUE:	int ( return the incremented value of argv index )
*
* HISTORY:
*					11.06.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
int iAssignPathToWriteCrc( char *argv[], int iIndex )
{
	iIndex++;
	if ( argv[iIndex] != NULL )
	{
		sVerCrcPath.assign(argv[iIndex]);
	}

	return iIndex;

}

/*******************************************************************************************
* FUNCTION:		iAssignSecurityByte
* DESCRIPTION:	This function Validates and assigns the Security Byte to be set.
* PARAMETER:
*					char *, int
* RETURNVALUE:	int ( return the incremented value of argv index )
*
* HISTORY:
*					11.06.2014	Neera.K.Ponnappa
*					Initial Version.
********************************************************************************************/
int iAssignSecurityByte( char *argv[], int iIndex )
{	
	iIndex++;
	if ( argv[iIndex] != NULL )
	{
		sSecurityByte.assign(argv[iIndex]);
		INFO_MSG (INFO_LEVEL_1, "INFO: Security:%s", sSecurityByte.c_str());
	}


   return iIndex;
}

/*******************************************************************************************
* FUNCTION       :iAssignSecurityValidateByte
* DESCRIPTION    :This function Validates and assigns the Security Byte to be set.
* PARAMETER      :char *, int
* RETURNVALUE    :int ( return the incremented value of argv index )
* HISTORY:       :
* DATE           |   VERSION          | AUTHOR         | MODIFICATION
-----------------------------------------------------------------------------------
* 19-02-2016     |   0.1              | VRI7COB        | Initial Version
********************************************************************************************/
int iAssignSecurityValidateByte(char *argv[], int iIndex)
{
	iIndex++;
	if(argv[iIndex] != NULL)
	{
		if(strtol(argv[iIndex], NULL, 16) == BOOT_BLOCK_ERASE_FLAG ||
				strtol(argv[iIndex], NULL, 16) == BLOCK_ERASE_FLAG ||
					strtol(argv[iIndex], NULL, 16) == CHIP_ERASE_FLAG ||
						strtol(argv[iIndex], NULL, 16) == WRITE_FLAG ||
							strtol(argv[iIndex], NULL, 16) == READ_FLAG )
		{
			sSecurityValidateByte.assign(argv[iIndex]);
		}
		else
		{
			help();
			OSAL_vSetProcessExitCode(ARGUMENT_ERROR);
	        OSAL_vProcessExit();
		}
	}
	return iIndex;
}

/*******************************************************************************************
* FUNCTION       :iAssignJTAGValidateByte
* DESCRIPTION    :This function Validates and assigns the JTAG
* PARAMETER      :char *, int
* RETURNVALUE    :int ( return the incremented value of argv index )
* HISTORY:       :
* DATE           |   VERSION          | AUTHOR         | MODIFICATION
-----------------------------------------------------------------------------------
* 19-02-2016     |   0.1              | VRI7COB        | Initial Version
-----------------------------------------------------------------------------------
* 18-05-2016	 |   0.2			  | VRI7COB		   | Fixing the Issue of Big Endian to Little Endian Conversion.
-----------------------------------------------------------------------------------
* 18-07-2016	 |   0.3			  | VRI7COB		   | CMG3GB-3514
-----------------------------------------------------------------------------------
* 08-09-2016	 |   0.4			  | VRI7COB		   | CMG3GB-3637
*----------------------------------------------------------------------------------
********************************************************************************************/
int iAssignJTAGValidateByte(char *argv[], int iIndex)
{
	iIndex++;

	if(argv[iIndex] != NULL)
	{		
		string teststring(argv[iIndex]);		
		size_t index = 0;
		index = teststring.find("x");
		teststring = teststring.substr(++index, teststring.length());
		INFO_MSG(INFO_LEVEL_1, "INFO: TestString:%s", teststring.c_str());
		
		if (teststring.length() == JTAG_VALIDATE_MAX_LENGTH) {
			tU32 u32BigEndianVal;
			sscanf(argv[iIndex], "%x", (unsigned int*)&u32BigEndianVal);

			std::stringstream stream;
			std::string result;

			if (u32BigEndianVal != 0) {
				tU32 u32LittleEndianVal = __bswap_32(u32BigEndianVal);
				stream << std::setfill('0') << std::setw(sizeof(tU32)* 2) << std::hex << u32LittleEndianVal;
				result = stream.str();
			}
			else {
				string strText(argv[iIndex]);
				result = strText.substr(2, strText.length());
			}

			INFO_MSG(INFO_LEVEL_1, "INFO: After JTAG Byte is:%s", result.c_str());

			result.insert(2, "-");
			result.insert(5, "-");
			result.insert(8, "-");

			sJTAGValidateByte.clear();
			sJTAGValidateByte.assign("-");
			sJTAGValidateByte.append(result);
			INFO_MSG(INFO_LEVEL_1, "INFO: ValidateJTAG:%s", sJTAGValidateByte.c_str());
		}
		else {
			INFO_MSG(INFO_LEVEL_1, "INFO: JTAG CRC Length(CurrentLength-%zdbytes, MaxLength-8bytes) is not matched", teststring.length());
			OSAL_vSetProcessExitCode(ARGUMENT_ERROR);
			OSAL_vProcessExit();
		}
		
	}
	else
	{
		help();
		OSAL_vSetProcessExitCode(ARGUMENT_ERROR);
        OSAL_vProcessExit();
	}

	return iIndex;

}
/*******************************************************************************************
* FUNCTION:		iAssignJtagByte
* DESCRIPTION:	This function validates and assigns the JTAG Byte to be set.
* PARAMETER:
*					char *, int
* RETURNVALUE:	int ( return the incremented value of argv index )
*
* HISTORY:
*					11.06.2014	Neera.K.Ponnappa
*					Initial Version.
********************************************************************************************/
int iAssignJtagByte( char *argv[], int iIndex )
{	
	iIndex++;

	if(argv[iIndex] != NULL)
	{
		string strText(argv[iIndex]);
		strText = strText.substr(2, strText.length());
		INFO_MSG (INFO_LEVEL_1, "INFO: JTAG Length:%zd", strText.length());
		if(strText.length() == JTAG_MAX_LENGTH)
		{
			strText.insert(2, "-");
			strText.insert(5, "-");
			strText.insert(8, "-");
			strText.insert(11, "-");
			strText.insert(14, "-");
			strText.insert(17, "-");
			strText.insert(20, "-");
			strText.insert(23, "-");
			strText.insert(26, "-");
			strText.insert(29, "-");
			strText.insert(32, "-");


			sJtagByte.clear();
			sJtagByte.assign("-");
			sJtagByte.append(strText);
			INFO_MSG (INFO_LEVEL_1, "INFO: JTAG:%s", sJtagByte.c_str());
		}
		else
		{
			bJtagAccess = false;
			INFO_MSG (INFO_LEVEL_1, "INFO: JTAG Bytes should be 12 Bytes Long");
			help();
			OSAL_vSetProcessExitCode(ARGUMENT_ERROR);
			OSAL_vProcessExit();
		}

	}
	else
	{
		help();
		OSAL_vSetProcessExitCode(ARGUMENT_ERROR);
		OSAL_vProcessExit();
	}
   return iIndex;
}



