/*
 * ep_mainloop.cpp
 *  ePoll based mainloop
 *  Created on: Mar 15, 2013
 *      Author: rjk2hi
 */
#include "ep_mainloop.h"

#include <sys/eventfd.h>
#include <sys/epoll.h>
#include <assert.h>
#include <stdio.h>
#include <unistd.h>
#include <poll.h>
#include <fcntl.h>

#define EPOLL_INITIAL_FD_SIZE  10
#define EP_OPERATION_FAILED			((int)-1)

/**
 * Buffer for polled events
 */
static struct epoll_event m_events[MAX_EVENTS];
/**
 * Constructor of ep_mainloop
 */
ep_mainloop::ep_mainloop()
{
	//Create epoll fd
	m_FdePoll = epoll_create(EPOLL_INITIAL_FD_SIZE);//Since Linux 2.6.8, the size argument is unused, but must be greater than zero.
	if(m_FdePoll == EP_HANDLE_INVALID)
	{
		perror("Failed to Create m_FdePoll");
		assert(m_FdePoll != EP_HANDLE_INVALID);
	}

	//Creating event fd for mainloop terminator
	m_evFdTerminator = eventfd(0,EFD_NONBLOCK);//Counter style, nonblocking fd
	if(m_evFdTerminator == EP_HANDLE_INVALID)
	{
		perror("Failed to Create evFdTerminator");
		assert(m_evFdTerminator != EP_HANDLE_INVALID);
	}
	m_bRunMainLoop = true;
}

/**
 * Destructor of ep_mainloop.
 * \n Close all pollfds *
 */
ep_mainloop::~ep_mainloop()
{
	if(m_FdePoll != EP_HANDLE_INVALID)
	{
		close(m_FdePoll);
	}
	if(m_evFdTerminator != EP_HANDLE_INVALID)
	{
		close(m_evFdTerminator);
	}
}
/**
 * Function > bAddePollFd
 * Description > Adds a epollfd to mainloop
 */
bool ep_mainloop::bAddePollFd(int fd,uint32_t epollevents)
{
	if(bIsFdValid(m_FdePoll)&& bIsFdValid(fd))
	{
		epoll_event ev;
		ev.data.fd = fd;
		ev.events = epollevents;

		int ret = epoll_ctl(m_FdePoll,EPOLL_CTL_ADD,fd,&ev);
		if(ret != EP_OPERATION_FAILED)
		{
			return true;
		}
	}
	return false;
}
/**
 * Function > bDelePollFd
 * Description > Removes a epollfd from mainloop
 */
bool ep_mainloop::bDelePollFd(int fd)
{
	if(bIsFdValid(m_FdePoll)&& bIsFdValid(fd))
	{
		epoll_event ev;

		int ret = epoll_ctl(m_FdePoll,EPOLL_CTL_DEL,fd,&ev);
		if(ret != EP_OPERATION_FAILED)
		{
			return true;
		}
	}
	return false;
}

/**
 * Function > vIterateMainLoop
 * Description > Run a single interation of the main loop
 */
void ep_mainloop::vIterateMainLoop()
{
	int nrofEvents = 0;

	if(m_FdePoll != EP_HANDLE_INVALID)
	{
		nrofEvents = epoll_wait(m_FdePoll,m_events, MAX_EVENTS, -1);
		for(int i=0; (i < nrofEvents); i++)
		{
			//Now we have the list of events fired .... process one by one
			if(m_events[i].data.fd == m_evFdTerminator)
			{
				//Read from fd, or next terminate events will not be caught
				uint64_t data;
				//Coverity fix for 10273
				if (-1 == read(m_evFdTerminator,&data,sizeof(data)))
				{
					fprintf(stderr,"DNL: file read error\n");
				}

				//Set terminator to true
				m_bRunMainLoop = false;
				break;
			}
			else
			{
				std::map<int,ep_fptrPollBreak>::iterator it = m_trigtable.find(m_events[i].data.fd);
				if(it != m_trigtable.end())
				{
					if(it->second != NULL)
					{
						//Trigger callback
						it->second(m_events[i].data.fd,m_events[i].events);
					}
				}
			}
		}
	}
}
/**
 * Function > vRunMainLoop
 * Description > Run the mainloop continuosly till you encounter a stop event.
 */
void ep_mainloop::vRunMainLoop()
{
	m_bRunMainLoop = true;
	if(bAddePollFd(m_evFdTerminator,POLLIN))
	{
		while(m_bRunMainLoop)
		{
			vIterateMainLoop();
		}
		//Delete terminator from poll
		bDelePollFd(m_evFdTerminator);
	}
}
/**
 * Function > vStopMainLoop()
 * Description > Stops the main loop in continuous mode
 */
void ep_mainloop::vStopMainLoop()
{
	//Is a check for running in continuous mode needed ????
	if(m_evFdTerminator != EP_HANDLE_INVALID)
	{
		uint64_t data = 1;
		write(m_evFdTerminator,&data,sizeof(data));
	}
}
/**
 * Function > bAddfdtoPoll
 * Description > Add an external fd to the main poll loop
 */
bool ep_mainloop::bAddfdtoPoll(int& fdpoll, unsigned int pflags, ep_fptrPollBreak pBase)
{
	bool ret = false;

	if((pBase != NULL)&& bIsFdValid(fdpoll))
	{
		//Check if fd already exists in poll
		if(m_trigtable.end() == m_trigtable.find(fdpoll))
		{
			if(bAddePollFd(fdpoll,pflags))
			{
				//Add to lookup table
				m_trigtable[fdpoll] = pBase;
				ret = true;
			}
		}
	}
	return ret;
}
/**
 * Function > bRemovefdfromPoll
 * Description > Remove fd from poll loop
 */
bool ep_mainloop::bRemovefdfromPoll(int& fdpoll)
{
	if(!bIsFdValid(fdpoll))
		return false;

	//Remove fd from poll loop
	bDelePollFd(fdpoll);

	//erase pollfd from lookup table
	m_trigtable.erase(fdpoll);
	return true;
}

/**
 * Additional helper functions
 */
bool ep_mainloop::bIsFdValid(int& fdpoll) const
{
	return (fcntl(fdpoll, F_GETFL) != -1);
}
