/*******************************************************************************************
* FILE:			dl_V850MessageFactory.cpp
*
* SW-COMPONENT:Software Download
*
* DESCRIPTION: 
*				
*
* AUTHOR:		Aditya Kumar Jha
*
* COPYRIGHT:	(c) 2013 - 2014 Robert Bosch Engineering and Business Solutions Ltd, Bangalore.
*
*********************************************************************************************
* HISTORY:
* ------------------------------------------------------------------------------------------------
* Date 				| Rev. 				| Author 		   	| Modification
* ------------------------------------------------------------------------------------------------
*					29.08.2013	Aditya Kumar Jha
*					Initial Version.
*					18.02.2014	Aditya Kumar Jha : Change for new trace class TR_CLASS_SWUPDATE_V850.
*              31.07.2014  Neera.K.Ponnappa : Lint Warnings solved
*              01.08.2014	Aditya Kumar Jha : IS1515_CRQ 4166-2: Prevention of: Linux root user can bypass 
*															 signature check of V850 code during software update.
*              04.08.2014	Aditya Kumar Jha : Review points incorporated.
*---------------------------------------------------------------------------------------------------
* 12-03-2015		| 2.0	| VRI7COB (RBEI)	| Enabling TTFI's Traces;Ticket Number:CMG3GB-1324
*---------------------------------------------------------------------------------------------------
* 23-03-2015		| 2.2	| VRI7COB (RBEI)	| Review Issues are corrected. & CMG3GB-860 is fixed.
* ---------------------------------------------------------------------------------------------------
* 14-09-2016		| 2.3	| ALA5COB (RBEI)	| Review Issues are corrected. & CMG3GB-3613 is fixed.
* ---------------------------------------------------------------------------------------------------
* 28-11-2016		| 2.4	| STP5COB (RBEI)	| CMG3G-11488,CMG3G-12148 is fixed.Review Issues are corrected
* ---------------------------------------------------------------------------------------------------
********************************************************************************************/

#include "ai_sw_update/common/base/imp/swupd_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
	#define ETG_I_TRACE_CHANNEL 			TR_TTFIS_SWUPDATE_CTRL
	#define ETG_I_TTFIS_CMD_PREFIX 			"DNL_"
	#define ETG_I_FILE_PREFIX 				dl_tclV850MessageFactory::
    #define ETG_DEFAULT_TRACE_CLASS 		TR_CLASS_SWUPDATE_V850
  #include "trcGenProj/Header/dl_V850MessageFactory.cpp.trc.h"
#endif

#include "dl_V850MessageFactory.h"

#define SWU_U8_MAX				0xFF
#define INFO_BLOCKID		0x8000
#define END_OF_TOC		0xFFFF
#define RSA_TYPE			0x8
#define TOC_OFFSET		13

using namespace std;

/*******************************************************************************************
* FUNCTION:		dl_tclV850MessageFactory
* DESCRIPTION:	Private Constructor
* PARAMETER:
*					NULL
*				
*
* RETURNVALUE:	Void
*
* HISTORY:
*					29.08.2013	Aditya Kumar Jha
*					Initial Version.
*					12-03-2015 VRI7COB (RBEI)
********************************************************************************************/
dl_tclV850MessageFactory::dl_tclV850MessageFactory()
{
	m_bDebugTraceStatus = true;
	//myStream.open("V850ResponseLog.txt",std::fstream::in | std::fstream::out | std::fstream::app);
	
}

/*******************************************************************************************
* FUNCTION:		~dl_tclV850MessageFactory
* DESCRIPTION:	Destructor
* PARAMETER:
*					NULL
*				
*
* RETURNVALUE:	Void
*
* HISTORY:
*					29.08.2013	Aditya Kumar Jha
*					Initial Version.
*					12-03-2015 VRI7COB (RBEI)
*					1.1 Adding the ET_TRACE_CLOSE
********************************************************************************************/
dl_tclV850MessageFactory::~dl_tclV850MessageFactory()
{
   //myStream.close();
}


/*******************************************************************************************
* FUNCTION:		u32PackDownloadRequestPacket
* DESCRIPTION:	This function is packs the download request packet to send to V850. 
*              
* PARAMETER:
*					tCString commandStr,
*					tU8* pTempBuf,
*					tU32 tempBufSize,
*					tU32 u32BinaryFileSize,
*					tU32 u32Address
*
* RETURNVALUE:	tU32
*
* HISTORY:
*					05.09.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tU32 dl_tclV850MessageFactory::u32PackDownloadRequestPacket( tCString commandStr, tString pTempBuf, tU32 tempBufSize, tU32 u32BinaryFileSize, tU32 u32Address ) const
{
	tCString method = "u32PackDownloadRequestPacket";
	ETG_TRACE_USR4 (("%s was entered", method));

	tU32 u32Length = 0;
	//char buffer[tempBufSize];

	if ( (NULL != commandStr )
		&&
		( NULL != pTempBuf ) )
	{

		memset ( pTempBuf, 0, tempBufSize );

		u32Length = u32ArgToPayload( pTempBuf, tempBufSize, commandStr );  

		//Fill address filed
		vSet_at( pTempBuf, u32Length, u32Address, sizeof(uint32_t) );

		//Fill Size filed
		vSet_at( pTempBuf, u32Length, u32BinaryFileSize, sizeof(uint32_t) );

		ETG_TRACE_USR4 (("%s was left", method));
		vHexDump( "SEND", pTempBuf , u32Length );
	}

	return u32Length;


}

/*******************************************************************************************
* FUNCTION:		u32PackCmdPacket
* DESCRIPTION:	This function is packing the command packet based on V850 message format 
*              and ready for write over inc driver.
* PARAMETER:
*				tCString cmdString,
*				tU8* pTempBuf,
*				tU32 tempBufSize
*
* RETURNVALUE:	tU32 //length of packet
*
* HISTORY:
*					04.09.2013	Aditya Kumar Jha
*					Initial Version.
*                   31.07.2014  Neera.K.Ponnappa
                    Fixed Prio2 Lint warning
********************************************************************************************/
tU32 dl_tclV850MessageFactory::u32PackCmdPacket( tCString strCmdPac, tString pTempBuf, tU32 tempBufSize ) const 
{
	tCString method = "u32PackCmdPacket";
	ETG_TRACE_USR4 (("%s was entered", method));

	tU32 u32Length = 0;

	if ( ( NULL != strCmdPac )
		&&
		( NULL != pTempBuf ) )
	{

		u32Length = u32ArgToPayload( pTempBuf, tempBufSize, strCmdPac );  

		vHexDump( "SEND", pTempBuf , u32Length );
	}

	ETG_TRACE_USR4 (("%s was left", method));

	return u32Length;

}

/*******************************************************************************************
* FUNCTION:		vSet_at
* DESCRIPTION:	This function place the tU32 data to the buffer at given position. 
* PARAMETER:
*					tU8* pTempBuf,
*					tU32 &start_pos,
*					tU32 data_in,
*					tU32 data_size
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					05.09.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850MessageFactory::vSet_at( tString pTempBuf, tU32 &start_pos, tU32 data_in, tU32 data_size) const 
{  
	tCString method = "vSet_at";
	ETG_TRACE_USR4 (("%s was entered", method));

	tU32 msb = start_pos + data_size;   
	tU32 j = 1; 

	for ( ; start_pos < msb; start_pos++, j++ ) 
	{ 
		pTempBuf[start_pos] = static_cast <char>(data_in >> ((data_size - j) *8) ); 
	} 

	ETG_TRACE_USR4 (("%s was left", method));

}  

/*******************************************************************************************
* FUNCTION:		u32GetFileSize
* DESCRIPTION:	This function returns the file size. 
*				
* PARAMETER:
*					tCString pcTempFileName
*				
*
* RETURNVALUE:	tU32 : file size on success : 0 on failed
*
* HISTORY:
*					29.08.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tU32 dl_tclV850MessageFactory::u32GetFileSize( tCString pcTempFileName ) const 
{
	tCString method = "u32GetFileSize";
    ETG_TRACE_USR4 (("%s was entered", method));

	tU32 u32FileSize = 0;
	FILE * pFile = NULL;

	 // Open file 
	pFile = fopen(pcTempFileName, "r");

	if ( NULL != pFile )
	{
		//read file size
		fseek( pFile, 0, SEEK_END );   	   
		u32FileSize =static_cast<tU32> (ftell( pFile )); 
		fseek( pFile, 0, SEEK_SET );

		//Close the file pointer
		fclose (pFile);
	}
// cast-for gen3 and gen4 compatibility 
	INFO_MSG (INFO_LEVEL_1, "INFO: Read data count =0x%8.8X", (unsigned int)u32FileSize);

	ETG_TRACE_USR4 (("%s was left", method));

	return u32FileSize;	

}

/*******************************************************************************************
* FUNCTION:		u32ReadBinaryFile
* DESCRIPTION:	This function open the binary file and read the binary of provided size 
*					and copy into a buffer.
* PARAMETER:
*					FILE *fp,
*					char * pcTempBuff ,
*					tU32 u32BytesToRead
*
* RETURNVALUE:	tU32 //byte read
*
* HISTORY:
*					29.08.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tU32 dl_tclV850MessageFactory::u32ReadBinaryFile( FILE *fp, char * pcTempBuff , tU32 u32BytesToRead ) const 
{
	tCString method = "u32ReadBinaryFile";
    ETG_TRACE_USR4 (("%s was entered", method));

	tU32 u32BytesRead = 0;

	if ( NULL != fp )
	{
		 u32BytesRead =static_cast<tU32> (fread ( pcTempBuff, 1, u32BytesToRead, fp ));

		 if( u32BytesRead != u32BytesToRead )
		 {
			 ETG_TRACE_USR4(("DNL: file read error"));
			 INFO_MSG(INFO_LEVEL_1, "ERROR: Download file read error");
		 }

	}

	ETG_TRACE_USR4 (("%s was left", method));

	return u32BytesRead;

}

/*******************************************************************************************
* FUNCTION:		u32ArgToPayload
* DESCRIPTION:	This function remove extra character and copy the numeric value in a buffer. 
*					
* PARAMETER:
*					char *buffer,  
*					int buflen,
*					tCString strArgPayLoad
*
* RETURNVALUE:	tU32
*
* HISTORY:
*					29.08.2013	Aditya Kumar Jha
*					Initial Version.
*                   31.07.2014  Neera.K.Ponnappa
                    Fixed Prio2 Lint warning
********************************************************************************************/
tU32 dl_tclV850MessageFactory::u32ArgToPayload( tString buffer, tU32 u32BufLen, tCString strArgPayLoad)
{
	tCString method = "u32ArgToPayload";
    ETG_TRACE_USR4 (("%s was entered", method));
  
   char *pos, *end;
   long result;
   tU32 TempLen = 0;
   
   if(!strArgPayLoad)
	{
      return 0;
	}

	char acStringBuffer[strlen(strArgPayLoad)+1];

   //memset(acStringBuffer, 0, (DEFAULT_MSGSZ+1) );
   strcpy(acStringBuffer, strArgPayLoad);   
   pos = acStringBuffer;
   end = acStringBuffer;
   while(*end != '\0' && (TempLen < u32BufLen)) 
	{
      result = strtol(pos, &end ,16);
      if(result > 255)
		{
         return 0;
		}
      if(result < 0)
		{
         return 0;
		}
      buffer[TempLen++] = static_cast<char> (result);

      if(*end != '\0') 
		{
         if(*end == '-')
			{
            end++;
			}
         else
			{
            return 0;
			}
      }
      pos = end;
   }

	ETG_TRACE_USR4 (("%s was left", method));

   return TempLen;
}


/*******************************************************************************************
* FUNCTION:		vHexDump
* DESCRIPTION:	This function convert the numeric value into hex format 
*					
* PARAMETER:
*					tCString intro,  
*					char *buffer,	
*					tU32 u32Len
*
* RETURNVALUE:	Void
*
* HISTORY:
*					29.08.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850MessageFactory::vHexDump(tCString intro, tString buffer, tU32 u32Len) const
{
	if(m_bDebugTraceStatus)
	{
		tCString method = "vHexDump";
		ETG_TRACE_USR4 (("%s was entered", method));

		tU32 u32Index;
		vPrintTimeStamp();
		fprintf(stderr,"%s:",intro);
		for ( u32Index=0; u32Index < u32Len; u32Index++ )
		{
			fprintf(stderr," 0x%02x", buffer[u32Index]);
		}

		fprintf(stderr,"\n");
		ETG_TRACE_USR4 (("%s was left", method));
	}
	else
		fprintf(stderr, ".");
}

/*******************************************************************************************
* FUNCTION:		DFInfo_ulwInit
* DESCRIPTION:	This function reads the DNL-File headers and store the
*              Filepositions in the DFInfo-Array 
*					
* PARAMETER:
*					FILE * , 
*					Flash_strTYInfo *	
*					
*
* RETURNVALUE:	tULong
*
* HISTORY:
*					24.12.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tULong dl_tclV850MessageFactory::DFInfo_ulwInit(FILE *DF_file,Flash_strTYInfo *Info)
{ 


	tU8 u8Buf[255];
	tU32 u32Pos = 0;
	Info->Count = 0;                                     /* default: no entrys */
	size_t result_for_fread = 0;

	rewind(DF_file);
	fseek(DF_file,0,SEEK_END);
	tU32 FileLen =static_cast<tU32>(ftell(DF_file));
	fseek(DF_file,0,0);

	while (fseek(DF_file, u32Pos, SEEK_SET) == 0)               /* locate FilePos */
	{
		//Coverity fix for 10374
		result_for_fread = fread((void*)u8Buf,0x80,1,DF_file);
		if (0 == result_for_fread)                        /* read header */
		{
			INFO_MSG (INFO_LEVEL_1, "WARN: Read error in DNL-file\n");
                        return 0;
                }

		if (Info->Count == FLASH_MAX_DEV)     /* check for device count overrun */
		{  
			return 0/*FLASH_NO_SUCCESS*/;
		}

		tU32 FixLen = (u8Buf[14] << 8) + u8Buf[15];
		tU32 Len = (u8Buf[6]<<24) + (u8Buf[7]<<16) + (u8Buf[8]<<8) + u8Buf[9];
		tU32 Start = (u8Buf[2]<<24) + (u8Buf[3]<<16) + (u8Buf[4]<<8) + u8Buf[5];
		tU32 CS = (u8Buf[10]<<24) + (u8Buf[11]<<16) + (u8Buf[12]<<8) + u8Buf[13];
		//tU16 HCS = (u8Buf[0x1E]<<8) + u8Buf[0x1F];
		tU32 HeadLen = (FixLen) ? FixLen : 32;    /* calc len of Header */
		/* store type */
		Info->Header[Info->Count].Typ =static_cast<tU16> ((tU16)(u8Buf[0] << 8) + u8Buf[1]);
		Info->Header[Info->Count].Pos = u32Pos + HeadLen; /* and FilePos to data */
		Info->Header[Info->Count].Length = Len;
		Info->Header[Info->Count].Start = Start;
		Info->Header[Info->Count].CS = CS;
		/* calc new pos of next header */
		if (FixLen)
		{  
			tU32 Rest = Len % FixLen;
			if (Rest) Len += FixLen - Rest;      /* Fill up to fix sector length */
		}


		u32Pos += (Len + HeadLen);              /* calculate addr to next header */
		Info->Count =static_cast<tU8> (Info->Count + 1);                       /* inc # of entrys */

		if (u32Pos >= FileLen) break;
	}

	return 1/*FLASH_SUCCESS*/;
} 

/*******************************************************************************************
* FUNCTION:		iReadBinaryDataPtr
* DESCRIPTION:	This function sets the pointer to the first byte of binary data specified
*              by the Typ, passed as parameter.
*					
* PARAMETER:
*					FILE * , 
*					Flash_strTYInfo *	
*					tU16
*
* RETURNVALUE:	tInt
*
* HISTORY:
*					24.12.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tInt dl_tclV850MessageFactory::iReadBinaryDataPtr (FILE *DF_file, Flash_strTYInfo *Info, tU16 Typ) const 
{ 
	tU8 u8Index = 0;

	while (u8Index < Info->Count)                    /* loop to search entry for typ */
	{  
		if (Info->Header[u8Index].Typ == Typ)           /* and return block length */
		{  
			if (fseek(DF_file, Info->Header[u8Index].Pos, SEEK_SET) != 0)
			{
				return 0;                            /* seek error in DNL-File */
			}
			else
			{
				return 1;
			}

		}
		u8Index++;
	}

	return 0;                                    /* return false, if not found */

} 

/*******************************************************************************************
* FUNCTION:		u32GetFileLen
* DESCRIPTION:	This function reads the length of the block, specified
*              by the Typ, passed as parameter
*					
* PARAMETER:
*					Flash_strTYInfo *	,
*					tU16
*
* RETURNVALUE:	tULong
*
* HISTORY:
*					24.12.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tULong dl_tclV850MessageFactory::u32GetFileLen(Flash_strTYInfo *Info, tU16 Typ) const
{ 
	tU8 u8Index = 0;

	while (u8Index < Info->Count)                   /* loop to search entry for typ */
	{     
		if (Info->Header[u8Index].Typ == Typ)          /* and return block length */
			return (Info->Header[u8Index].Length);
		u8Index++;
	}

	return 0;                                       /* return 0, if not found */

} 

/*******************************************************************************************
* FUNCTION:		u32GetMemoryAddress
* DESCRIPTION:	This function reads the start address of Device, specified
*              by the Typ, passed as parameter
*					
* PARAMETER:
*					Flash_strTYInfo *	,
*					tU16
*
* RETURNVALUE:	tULong
*
* HISTORY:
*					24.12.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tULong dl_tclV850MessageFactory::u32GetMemoryAddress(Flash_strTYInfo *Info, tU16 Typ) const
{ 
	tU8 u8Index = 0;

	while (u8Index < Info->Count)                   /* loop to search entry for typ */
	{     
		if (Info->Header[u8Index].Typ == Typ)          /* and return start address of Device */
			return (Info->Header[u8Index].Start);
		u8Index++;
	}

	return 0;                                       /* return 0, if not found */

} 

/*******************************************************************************************
* FUNCTION:		u32PackSecuritySignReqPacket
* DESCRIPTION:	This function is packs the Security signature request packet to send to V850. 
*              
* PARAMETER:
*					tCString commandStr,
*					tU8* pTempBuf,
*					tU32 tempBufSize,
*					tU32 u32BinaryFileSize
*
* RETURNVALUE:	tU32
*
* HISTORY:
*					24.07.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tU32 dl_tclV850MessageFactory::u32PackSecuritySignReqPacket( tCString commandStr, tString pTempBuf, tString pTempDataBuf, tU32 tempBufSize, tU32 u32BinaryFileSize ) const
{
	tCString method = "u32PackSecuritySignReqPacket";
	ETG_TRACE_USR4 (("%s was entered", method));

	tU32 u32Length = 0;
	//char buffer[tempBufSize];

	if ( (NULL != commandStr )
		&&
		( NULL != pTempBuf ) )
	{

		memset ( pTempBuf, 0, tempBufSize );

		u32Length = u32ArgToPayload( pTempBuf, tempBufSize, commandStr );  

		//Fill Size filed
		vSet_at( pTempBuf, u32Length, u32BinaryFileSize, sizeof(uint16_t) );
	
		memcpy ( &pTempBuf[u32Length], pTempDataBuf, u32BinaryFileSize );

		u32Length += u32BinaryFileSize;

		vHexDump( "SEND", pTempBuf , u32Length );
	}

	ETG_TRACE_USR4 (("%s was left", method));

	return u32Length;


}

/*******************************************************************************************
* FUNCTION:		u32GetRSASignature
* DESCRIPTION:	This function is to read the RSA start address. 
*              
* PARAMETER:
*					FILE *DF_file,
*					Flash_strTYInfo *Info,
*					tString pTempBuf,
*					tU32 MemBlock
*
* RETURNVALUE:	tU32 // return the length of the RSA bytes
*
* HISTORY:
*					28.07.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tU32 dl_tclV850MessageFactory::u32GetRSASignature (FILE *DF_file, Flash_strTYInfo *Info, tString pTempBuf, tU32 MemBlock)  
{ 
	tU8 u8Index = 0;
	tU32 u32BlockLen = 0;
	tU32 u32RSAOffset = 0;
	tU32 u32SIBLen = 0;
	tU32 u32RSAMaxLen = 0;
	tU32 u32LenCount = 0;
	size_t result_for_fread = 0;

	//Get the block index for structure
	u8Index = u8GetBlockIndex ( Info, INFO_BLOCKID );

	/* Block ID not found */
	if( SWU_U8_MAX == u8Index )
	{
		INFO_MSG (INFO_LEVEL_1, "WARN: Block ID not found\n");
		return 0;  
	}

	//seek the file pointer
	//Coverity fix for 126768
	if (-1 == fseek(DF_file, Info->Header[u8Index].Pos, SEEK_SET))
	{
		ETG_TRACE_USR4(("DNL: file seek error"));
		INFO_MSG (INFO_LEVEL_1, "WARN: file seek error in DNL-file\n");
	}

	// get the length of SIB bytes	
	u32SIBLen =static_cast<tU32> (Info->Header[u8Index].Length);	

	/* get the RSA offset */
	u32RSAOffset = u32GetRSAOffset(DF_file, u32SIBLen);

	if (u32RSAOffset == 0)
	{ 
		INFO_MSG (INFO_LEVEL_1, "WARN: No RSA-Signature data in DNL-file found\n");
		return 0;
	}
	//seek the file pointer
	//Coverity fix for 126768
	if (-1 == fseek(DF_file, (Info->Header[u8Index].Pos + u32RSAOffset), SEEK_SET))
	{
		ETG_TRACE_USR4(("DNL: file seek error"));
		INFO_MSG (INFO_LEVEL_1, "WARN: file seek error in DNL-file\n");
	}

	u32RSAMaxLen = u32SIBLen - u32RSAOffset;	
	u32LenCount = u8GetRSAStartPos( DF_file, u32RSAMaxLen, u32BlockLen, MemBlock);

	// NULL pointer check
	if ( u32LenCount == 0 || NULL == pTempBuf )
	{
		return 0;  
	}

	//Seek the file pointer to the
	//Coverity fix for 126768
	if (-1 == fseek(DF_file, (Info->Header[u8Index].Pos + u32RSAOffset + u32LenCount), SEEK_SET))
	{
		ETG_TRACE_USR4(("DNL: file seek error"));
		INFO_MSG (INFO_LEVEL_1, "WARN: file seek error in DNL-file\n");
	}

	/* Read data of RSA byte and assign to pTempBuf */
	//Coverity fix for CID 126770
	result_for_fread = fread(pTempBuf, u32BlockLen, 1, DF_file);
	if (0 == result_for_fread)
	{
		INFO_MSG (INFO_LEVEL_1, "WARN: Read error in DNL-file\n");
		return 0;
	}

	return u32BlockLen;                                    /* return RSA byte length */

}

/*******************************************************************************************
* FUNCTION:		u8GetBlockIndex
* DESCRIPTION:	This function is to Block type Index. 
*              
* PARAMETER:
*					Flash_strTYInfo *Info,
*					tU32 Typ,
*
* RETURNVALUE:	tU32 // return the Index of the structure Flash_strTYInfo
*
* HISTORY:
*					28.07.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tU8 dl_tclV850MessageFactory::u8GetBlockIndex ( Flash_strTYInfo *Info, tU32 Typ)
{
	tU8 u8Index = 0;
	tU8 u8RetVal = SWU_U8_MAX;

	while (u8Index < Info->Count)                    /* loop to search entry for typ */
	{ 
		if (Info->Header[u8Index].Typ == Typ)           /* and return block length */
		{ 
			u8RetVal = u8Index;
			break;
		}
		u8Index++;
	}

	return u8RetVal;
}

/*******************************************************************************************
* FUNCTION:		u32GetRSAOffset
* DESCRIPTION:	This function gives the RSA start address. 
*              
* PARAMETER:
*					FILE *DF_file,
*					tU32 u32SIBLen
*
* RETURNVALUE:	tU32 // return the Offset of the RSA bytes
*
* HISTORY:
*					28.07.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tU32 dl_tclV850MessageFactory::u32GetRSAOffset ( FILE *DF_file, tU32 u32SIBLen)
{
	tU8 *u8Ptr = NULL;             
	tU32 u32Type = 0;
	tU32 u32RSAOffset = 0, u32Offset = 0;
	 size_t return_for_fread = 0;

	/* Databuf for Complete RSA field */
	tU8 *u8DataBuf = NULL; 
	u8DataBuf = new tU8[u32SIBLen]; 

	//Check data buffer for null pointer
	if ( u8DataBuf == NULL )
	{
		return 0;  
	}

	//Read the content of RSA complete byte along with header
	//Coverity fix for 126767
	return_for_fread = fread((void*)u8DataBuf, u32SIBLen, 1, DF_file);
	if (0 == return_for_fread)
	{  
		INFO_MSG (INFO_LEVEL_1, "WARN: Read error in DNL-file\n");
		delete[] u8DataBuf;
		u8DataBuf = NULL;
		return 0;
	}

	u8Ptr = (u8DataBuf + TOC_OFFSET); 

	// Read TOC Entry and get Offset 
	do                                  
	{ 
		u32Type = (tU32)(*u8Ptr << 8) + (tU32)*(u8Ptr+1);
		u8Ptr+=2;
		u32Offset = ((tU32)*u8Ptr << 24) + ((tU32)*(u8Ptr+1) << 16) +
			((tU32)*(u8Ptr+2) << 8) + (tU32)*(u8Ptr+3);
		u8Ptr +=4;
		if (u32Type == RSA_TYPE)
		{
			u32RSAOffset = u32Offset;
		}
	}
	while (u32Type != END_OF_TOC);

	//free the RSA data buffer
	delete[] u8DataBuf;
	u8DataBuf = NULL;


	return u32RSAOffset;
}

/*******************************************************************************************
* FUNCTION:		u8GetRSAStartPos
* DESCRIPTION:	This function is to read the RSA data byte start position. 
*              
* PARAMETER:
*					FILE *DF_file,
*					tU32 u32RSAMaxLen,
*					tU32 &u32BlockLen,
*					tU32 MemBlock
*
* RETURNVALUE:	tU32 // return the position of the first RSA bytes
*
* HISTORY:
*					31.07.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tU32 dl_tclV850MessageFactory::u8GetRSAStartPos( FILE *DF_file, tU32 u32RSAMaxLen, tU32 &u32BlockLen, tU32 MemBlock)
{
	tU32 u32BlockType = 0;
	tU32 u32LenCount = 0;
	tBool bBlockPresent = false;
	size_t result_for_fread = 0;

	/* Databuf for RSA */
	tU8 *u8DataBuf = NULL; 
	u8DataBuf = new tU8[u32RSAMaxLen]; 

	//Check data buffer for null pointer
	if ( u8DataBuf == NULL )
	{
		return 0;  
	}

	//Read the content of RSA byte
	//Coverity fix for 126769
	result_for_fread = fread((void*)u8DataBuf, (u32RSAMaxLen), 1, DF_file);
	if (0 == result_for_fread)
	{  
		INFO_MSG (INFO_LEVEL_1, "WARN: Read error in DNL-file\n");
		delete[] u8DataBuf;
		u8DataBuf = NULL;
		return false;
	}

	tU8 *pRSADataBuf = u8DataBuf + 3;
	u32LenCount += 2;

	//Read the total number of key counts
	tU32 u32KeyCount = (tU32)(*pRSADataBuf << 8) + (tU32)*(pRSADataBuf+1);

	//Increment the pointers by key counts bytes
	pRSADataBuf += u32LenCount;
	u32LenCount += 2;

	// loop to find block ID in RSA field 
	for(tU32 u32Index = 0; u32Index < u32KeyCount; u32Index++ )
	{
		//Read the block ID
		u32BlockType = (tU32)(*pRSADataBuf << 8) + (tU32)*(pRSADataBuf+1);
		//Increment the pointers by block type bytes
		pRSADataBuf += 2;
		u32LenCount += 2;

		//Read the length of singnature bytes to be send
		u32BlockLen = (tU32)(*pRSADataBuf << 8) + (tU32)*(pRSADataBuf+1);
		if( u32BlockType == MemBlock )
		{
			u32LenCount += 3;	
			bBlockPresent = true;
			break;
		}

		//Increment the pointers by length + length bytes (2 bytess) 
		pRSADataBuf = pRSADataBuf + u32BlockLen + 2;
		u32LenCount = u32LenCount + u32BlockLen + 2;
	} 

	if( false == bBlockPresent )
	{
		u32LenCount = 0;
	}
	
	//free the RSA data buffer
	delete[] u8DataBuf;
	u8DataBuf = NULL;


	return u32LenCount;
}

/*******************************************************************************************
* FUNCTION:		vPrintTimeStamp
* DESCRIPTION:	This function prints the system up time in the format [ minutes : seconds : milliseconds ]
*					
* PARAMETER:  Void
*
* RETURNVALUE:	Void
*
* HISTORY:
*					21.11.2016	Naveen Moses
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850MessageFactory::vPrintTimeStamp(tVoid) const
{

	struct timespec SystemUpTime;

	if( !clock_gettime( CLOCK_MONOTONIC, &SystemUpTime ) )
		fprintf( stderr, "[%02ld:%02ld:%03ld]", SystemUpTime.tv_sec/60, SystemUpTime.tv_sec%60, SystemUpTime.tv_nsec/1000000 );


}
