/*******************************************************************************************
* FILE:			dl_V850ClientInterface.cpp
*
* SW-COMPONENT:Software Download
*
* DESCRIPTION: 
*				
*
* AUTHOR:		Aditya Kumar Jha
*
* COPYRIGHT:	(c) 2013  2014 Robert Bosch Engineering and Business Solutions Ltd, Bangalore.
*
*********************************************************************************************
* HISTORY:
* ------------------------------------------------------------------------------------------------
* Date 				| Rev. 				| Author 		   	| Modification
* ------------------------------------------------------------------------------------------------
*					26.09.2013	Aditya Kumar Jha
*					Initial Version.
*					18.02.2014	Aditya Kumar Jha : Change for new trace class TR_CLASS_SWUPDATE_V850.
*					Change for command line option and added tBool parameter to s32InitV850ClientHandler().
*					04.03.2014	Aditya Kumar Jha : Change bOnDLStart() from vOnDLStart() to return 
*					the download state to the client.
*              14.03.2014	Aditya Kumar Jha : Change for ERROR CODE.
*              21.05.2014	Aditya Kumar Jha : Remove mutex lock and introduce semaphore.
*              12.06.2014  Aditya Kumar Jha , Neera.K.Ponnappa : Implemented No CRC and No Reset
*              30.06.2014  Neera.K.Ponnappa : Implemented Security Bytes and Jtag ID 
*              25.07.2014  Neera.K.Ponnappa : Implemented ClearCase label entry to registry
*              31.07.2014  Neera.K.Ponnappa : Lint Warnings solved
*              06.08.2014  Neera.K.Ponnappa : Bug Fix CMG3GB-reset issue when Security Bytes/Jtag Bytes executed.
*              16.09.2014	Aditya Kumar Jha : Fixes for Jira ticket CMG3GB-1128.
*---------------------------------------------------------------------------------------------------
* 12-03-2015		| 2.0			| VRI7COB (RBEI)	| Enabling TTFI's Traces;Ticket Number:CMG3GB-1324
*---------------------------------------------------------------------------------------------------
* 21-05-2015		| 2.1			| VRI7COB (RBEI)	| Changes are Made to Adapt For Adding the New Command line
* 					|				|					| Parameter "screenreinit"
*----------------------------------------------------------------------------------------------
* 19-02-2016		| 2.3			| VRI7COB (RBEI)	| Implementing the CRQ, CMG3G-9596
*----------------------------------------------------------------------------------------------
* 18-05-2016		| 2.4			| VRI7COB (RBEI)	| Taking the 1's complement of Security Value from V850.
*					|				|					|   Ticket Number: CMG3GB-3323
* --------------------------------------------------------------------------------------------------------------------
********************************************************************************************/

#include "ai_sw_update/common/base/imp/swupd_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
	#define ETG_I_TRACE_CHANNEL 			TR_TTFIS_SWUPDATE_CTRL
	#define ETG_I_TTFIS_CMD_PREFIX 			"DNL_"
	#define ETG_I_FILE_PREFIX 				dl_tclV850ClientInterface::
    #define ETG_DEFAULT_TRACE_CLASS 		TR_CLASS_SWUPDATE_V850
  #include "trcGenProj/Header/dl_V850ClientInterface.cpp.trc.h"
#endif

#include <pthread.h>
#include "dl_V850ClientInterface.h"


/*******************************************************************************************
* FUNCTION:		dl_tclV850ClientInterface
* DESCRIPTION:	Private Constructor
* PARAMETER:
*					NULL
*				
*
* RETURNVALUE:	Void
*
* HISTORY:
*					26.09.2013	Aditya Kumar Jha
*					Initial Version.
*					12-03-2015 VRI7COB (RBEI)
********************************************************************************************/
dl_tclV850ClientInterface::dl_tclV850ClientInterface()
:m_oV850DlCtrlMgr()
{
}

/*******************************************************************************************
* FUNCTION:		~dl_tclMessageCompositor
* DESCRIPTION:	Destructor
* PARAMETER:
*					NULL
*				
*
* RETURNVALUE:	Void
*
* HISTORY:
*					26.09.2013	Aditya Kumar Jha
*					Initial Version.
*					12-03-2015 VRI7COB (RBEI)
********************************************************************************************/
dl_tclV850ClientInterface::~dl_tclV850ClientInterface()
{
}

/*******************************************************************************************
* FUNCTION:		s32InitV850ClientHandler
* DESCRIPTION:	This function returns the file size. 
*				
* PARAMETER:
*					T_fCallback fCallback
*				
*
* RETURNVALUE:	tS32 : file size on success : 0 on failed
*
* HISTORY:
*					26.09.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tS32 dl_tclV850ClientInterface::s32InitV850ClientHandler( T_fCallback fCallback, tBool bIsAppStartOnly , tInt AppFlagVal)
{
	vAssignAppFlags(AppFlagVal);
	//Call bInitV850Message() to initailized the messages
	if( true == m_oV850DlCtrlMgr.bInitV850Message( fCallback, bIsAppStartOnly ) )
	{
		return 0;
	}
	return -1;
	
}

/*******************************************************************************************
* FUNCTION:		vOnDLStart
* DESCRIPTION:	Client interface function to start the download. 
*					
* PARAMETER:
*					tCString CStrFilePath,  
*					tU32 u32MemoryAddress
*
* RETURNVALUE:	tU32
*
* HISTORY:
*					26.09.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850ClientInterface::vOnDLStart( tCString CStrFilePath, tU32 u32MemoryAddress )
{
	tU8 acReqMsgBuffer[DEFAULT_MSGSZ];

	memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );

	if ( NULL != CStrFilePath )
	{
		//Assign the file name and path to a const character array 
		m_oV850DlCtrlMgr.vSetFilePath( CStrFilePath );

		m_oV850DlCtrlMgr.vSetMemeoryAddress( u32MemoryAddress );

		acReqMsgBuffer[0] = DL_REQUEST_START;

		//Add event to message queue
		vAddRequestToMsgQ( acReqMsgBuffer );
		m_oV850DlCtrlMgr.vWaitForSemRelease();
	}

}

/*******************************************************************************************
* FUNCTION:		bOnDLStart
* DESCRIPTION:	Client interface function to start the download. 
*					
* PARAMETER:
*					tCString CStrFilePath,  
*					tU32 u32MemoryAddress
*
* RETURNVALUE:	tBool
*
* HISTORY:
*					26.09.2013	Aditya Kumar Jha
*					Initial Version.
***----------------------------------------------------------------------------------------------
* DATE			| VERSION		| AUTHOR		| MODIFICATION
* ---------------------------------------------------------------------------------------------
* 29-05-2015	| 2.0			| VRI7COB(RBEI)	| Changes are Made to Adapt For Adding the New Command line
* 				|				|				| Parameter "screenreinit"
*----------------------------------------------------------------------------------------------
********************************************************************************************/
tBool dl_tclV850ClientInterface::bOnDLStart
    ( 
        tCString CStrFilePath, 
        tCString CStrFileName, 
        list<EN_EXTRA_OPERATIONS> listofOpera,
        tBool bCrcEnabled , tBool bResetEnabled
    )
{
	tBool bRet = false;
	tU8 acReqMsgBuffer[DEFAULT_MSGSZ];

	memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );

	if ( (NULL != CStrFilePath) && (NULL != CStrFileName) )
	{
		//Assign the file name and path to a const character array 
		m_oV850DlCtrlMgr.vSetFilePath( CStrFilePath );

		m_oV850DlCtrlMgr.vSetFileName( CStrFileName );

		m_oV850DlCtrlMgr.vSetCrcEnable( bCrcEnabled );

		m_oV850DlCtrlMgr.vSetResetEnable( bResetEnabled );

		m_oV850DlCtrlMgr.vSetListOfOpera(listofOpera);


		acReqMsgBuffer[0] = DL_REQUEST_START;

		//Add event to message queue
		vAddRequestToMsgQ( acReqMsgBuffer );

		m_oV850DlCtrlMgr.vWaitForSemRelease();

		if ( m_oV850DlCtrlMgr.u8GetUpdateState() == DL_SUCCESS_STATE )
		{
			bRet = true;
		}
	}

	return bRet;

}

/*******************************************************************************************
* FUNCTION:		vOnDLAbort
* DESCRIPTION:	Client interface function to abort the download. 
*					
* PARAMETER:
*					NULL  
*					
*
* RETURNVALUE:	Void
*
* HISTORY:
*					26.09.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850ClientInterface::vOnDLAbort() const
{
	tU8 acReqMsgBuffer[DEFAULT_MSGSZ];

	memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );

	acReqMsgBuffer[0] = DL_REQUEST_ABORT;

	//Add event to message queue
	vAddRequestToMsgQ( acReqMsgBuffer );	
	
}

/*******************************************************************************************
* FUNCTION:		vOnDLUpdateProgress
* DESCRIPTION:	Client interface function to get the current update of the download. 
*					
* PARAMETER:
*					NULL  
*					
*
* RETURNVALUE:	Void
*
* HISTORY:
*					26.09.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850ClientInterface::vOnDLUpdateProgress() const
{
	tU8 acReqMsgBuffer[DEFAULT_MSGSZ];

	memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );

	acReqMsgBuffer[0] = DL_REQUEST_PROGRESS;

	//Add event to message queue
	vAddRequestToMsgQ( acReqMsgBuffer );
	
}

/*******************************************************************************************
* FUNCTION:		vOnDLState
* DESCRIPTION:	Client interface function to get the current state of the download.
*					
* PARAMETER:	
*					NULL
*					
*
* RETURNVALUE:	Void
*
* HISTORY:
*					26.09.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850ClientInterface::vOnDLState() const
{
	tU8 acReqMsgBuffer[DEFAULT_MSGSZ];

	memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );

	acReqMsgBuffer[0] = DL_REQUEST_STATE;

	//Add event to message queue
	vAddRequestToMsgQ( acReqMsgBuffer );
	
}

/*******************************************************************************************
* FUNCTION:		s8OnGetSWVer
* DESCRIPTION:	Client interface function to get the Software Version of the binary SW.
*					
* PARAMETER:	
*					std::string& CStrFile, 
*					std::string& strVer
*
* RETURNVALUE:	tS8
*
* HISTORY:
*					13.12.2013	Aditya Kumar Jha
*					Initial Version.
*                   25.07.2014  Neera.K.Ponnappa : Implemented ClearCase label entry to registry
********************************************************************************************/
tS8 dl_tclV850ClientInterface::s8OnGetSWVer(std::string& CStrFile, std::string& strVer , std::string& strLabel )
{
	tS8 s8Result = -1;
	tU8 acReqMsgBuffer[DEFAULT_MSGSZ];

	memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );
	m_oV850DlCtrlMgr.m_chSWVersion[0] = '\n';
	s8Result = s8OnGetSWVerExtended();
	if ( ("" != CStrFile)
		&&
		( (CStrFile.compare("-bl") == 0 )
		||
		(CStrFile.compare("-app") == 0 )
		||
		(CStrFile.compare("-bm") == 0 ) ) )
	{
		if( CStrFile.compare("-bl") == 0 )
		{
			acReqMsgBuffer[0] = DL_REQUEST_BL_VERSION;
		}
		else if( CStrFile.compare("-app") == 0 )
		{
			acReqMsgBuffer[0] = DL_REQUEST_APP_VERSION;
		}
		else if( CStrFile.compare("-bm") == 0 )
		{
			acReqMsgBuffer[0] = DL_REQUEST_BM_VERSION;
		}		

		//Add event to message queue
		vAddRequestToMsgQ( acReqMsgBuffer );

		m_oV850DlCtrlMgr.vWaitForSemRelease();

		if(m_oV850DlCtrlMgr.m_chSWVersion[0] != '\n')
		{
			strVer.append(m_oV850DlCtrlMgr.m_chSWVersion);
		}
	    if(m_oV850DlCtrlMgr.m_chSWLabel[0] != '\n')
		{
			strLabel.append(m_oV850DlCtrlMgr.m_chSWLabel);
		}
		//Assign the CRC result
		s8Result = m_oV850DlCtrlMgr.s8GetSwVersionResult();
		s8Result = (tS8)(int)(s8Result & m_oV850DlCtrlMgr.s8GetSwLabelResult());
		//return error code
		if( s8Result == -1 )
		{
			s8Result = TIMEOUT_ERROR;
		}
		else if( s8Result )
		{
			s8Result = (tS8)(int)(s8Result + REG_OPEN_ERROR - 1);
		}

	}

	return s8Result;

}
/*******************************************************************************************
* FUNCTION:		s8OnGetSWVerExtended
* DESCRIPTION:	Client interface function to get the Software Version of the binary SW extended for maintainability issue.
*					
* PARAMETER:	Void
*					
*
* RETURNVALUE:	tS8
*
* HISTORY:
*					Initial Version.
*                   13.08.2014  Neera.K.Ponnappa : Implemented ClearCase label entry to registry
********************************************************************************************/
tS8 dl_tclV850ClientInterface::s8OnGetSWVerExtended()
{
	tS8 s8Result = -1;

	 if(true == m_oV850DlCtrlMgr.m_bIsBLProgMode)
	 {
       s8Result = V850_MODE_ERROR;
	   OSAL_vSetProcessExitCode(V850_MODE_ERROR);
       OSAL_vProcessExit();
	 }
	 return s8Result;
}
/*******************************************************************************************
* FUNCTION:		s8OnGetCheckSumVal
* DESCRIPTION:	Client interface function to get the check sum value of binary SW.
*					
* PARAMETER:	
*					std::string& CStrFile, 
*					std::string& strCheckSum
*
* RETURNVALUE:	tS8   (Result of CheckSum) 
*
* HISTORY:
*					13.12.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tS8 dl_tclV850ClientInterface::s8OnGetCheckSumVal(std::string& CStrFile, std::string& strCheckSum)
{
	tS8 s8Result = -1;
	tU8 acReqMsgBuffer[DEFAULT_MSGSZ];

	memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );
	m_oV850DlCtrlMgr.m_chCheckSumVal[0] = '\n';
    if(true == m_oV850DlCtrlMgr.m_bIsBLProgMode)
    {
       s8Result = V850_MODE_ERROR ;	
	   OSAL_vSetProcessExitCode(V850_MODE_ERROR);
       OSAL_vProcessExit();
    }
	if ( ("" != CStrFile)
		&&
		( (CStrFile.compare("-bl") == 0 )
		||
		(CStrFile.compare("-app") == 0 )
		||
		(CStrFile.compare("-bm") == 0 ) ) )
	{
		//compare the string with type of binary file
		if( CStrFile.compare("-bl") == 0 )  
		{
			acReqMsgBuffer[0] = DL_REQUEST_BL_CRC;
		}
		else if( CStrFile.compare("-app") == 0 ) 
		{
			acReqMsgBuffer[0] = DL_REQUEST_APP_CRC;
		}
		else if( CStrFile.compare("-bm") == 0 ) 
		{
			acReqMsgBuffer[0] = DL_REQUEST_BM_CRC;
		}
		//Add event to message queue
		vAddRequestToMsgQ( acReqMsgBuffer );

		m_oV850DlCtrlMgr.vWaitForSemRelease();

		if(m_oV850DlCtrlMgr.m_chCheckSumVal[0] != '\n')
		{
			strCheckSum.append(m_oV850DlCtrlMgr.m_chCheckSumVal);			
		}

		//Assign the CRC result
		s8Result = m_oV850DlCtrlMgr.s8GetCheckSumResult();	

		//return error code
		if( s8Result == -1 )
		{
			s8Result = TIMEOUT_ERROR;
		}
		else if( s8Result )
		{
			s8Result = (tS8)(int) (s8Result + CRC_UNINIT - 1);
		}

	}

	return s8Result;

}

/*******************************************************************************************
* FUNCTION:		vOnResetReq
* DESCRIPTION:	Client interface function to reset the ECU.
*					
* PARAMETER:	
*					None
*
* RETURNVALUE:	tVoid
*
* HISTORY:
*					04.06.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850ClientInterface::vOnResetReq() 
{
	tU8 acReqMsgBuffer[DEFAULT_MSGSZ];
	memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );

	acReqMsgBuffer[0] = DL_REQUEST_ECU_RESET;

	//Add event to message queue
	vAddRequestToMsgQ( acReqMsgBuffer );

	m_oV850DlCtrlMgr.vWaitForSemRelease();

}

/*******************************************************************************************
* FUNCTION:		vAddRequestToMsgQ
* DESCRIPTION:	This function add the request to the queue and notify the epoll.
*					
* PARAMETER:
*					NULL  
*					
*
* RETURNVALUE:	Void
*
* HISTORY:
*					01.10.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850ClientInterface::vAddRequestToMsgQ( tU8* message ) const
{
	
	//Add event to message queue
	MessageQueue * pMessageQueue = dl_tclV850UpdateCtrlManager::pGetMessageQueuePtr();

	//Check the message queue pointer and post the message over queue
	if( ( NULL != pMessageQueue )
		&&
		( NULL != message ) )
	{
		
		pMessageQueue->vSendtoReqMsgQ(message);
		
		//Notify event
		tS32 s32TempFD = m_oV850DlCtrlMgr.s32GetClientReqEventFD();

		if ( s32TempFD != EP_HANDLE_INVALID )
		{
			write( s32TempFD, message ,sizeof(uint64_t) );
		}

	}
       
}

/*******************************************************************************************
* FUNCTION:		vPostErrorToClient
* DESCRIPTION:	This function post an error code to the client. 
*					
* PARAMETER:
*					NULL  
*					
*
* RETURNVALUE:	Void
*
* HISTORY:
*					26.09.2013	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tVoid dl_tclV850ClientInterface::vPostErrorToClient() const
{

	
}


/*******************************************************************************************
* FUNCTION:		vOnGetSecurityByteVal
* DESCRIPTION:	This function Post the request for Secuirty Byte Access.
*					
* PARAMETER:	
*					std::string& CStrFile, 
*					std::string& strCheckSum
*
* RETURNVALUE:	 Void
*
* HISTORY:
*					30.06.2014	Neera.K.Ponnappa
*					Initial Version.
*                   31.07.2014  Neera.K.Ponnappa
                    Fixed Lint warning
********************************************************************************************/
tVoid dl_tclV850ClientInterface::vOnGetSecurityByteVal(std::string& CStrSecByteVal, tBool bResetEnabled)
{
	tU8 acReqMsgBuffer[DEFAULT_MSGSZ];

	memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );

	tU8 u8ManipulatedVal = 0;
	// cast- for gen3 and gen4 compatibality
	sscanf(CStrSecByteVal.c_str(), "%x",(unsigned int*)&u8ManipulatedVal);


	memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );
	acReqMsgBuffer[0] = DL_DO_STARTUP_SEQUENCE;
	vAddRequestToMsgQ( acReqMsgBuffer );
	m_oV850DlCtrlMgr.vWaitForSemRelease();
	
	tBool IsValueGiven = false;
	if((u8ManipulatedVal & BOOT_BLOCK_ERASE_FLAG) == BOOT_BLOCK_ERASE_FLAG)
	{
		IsValueGiven = true;
		m_oV850DlCtrlMgr.vSetSecurityByte(BOOT_BLOCK_ERASE_INPUT_FLAG);
		memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );
		acReqMsgBuffer[0] = DL_REQUEST_SECURITY_ACCESS;
		vAddRequestToMsgQ( acReqMsgBuffer );
		m_oV850DlCtrlMgr.vWaitForSemRelease();

	}
	if((u8ManipulatedVal & BLOCK_ERASE_FLAG) == BLOCK_ERASE_FLAG)
	{
		IsValueGiven = true;
		m_oV850DlCtrlMgr.vSetSecurityByte(BLOCK_ERASE_INPUT_FLAG);
		memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );
		acReqMsgBuffer[0] = DL_REQUEST_SECURITY_ACCESS;
		vAddRequestToMsgQ( acReqMsgBuffer );
		m_oV850DlCtrlMgr.vWaitForSemRelease();
	}

	if((u8ManipulatedVal & CHIP_ERASE_FLAG) == CHIP_ERASE_FLAG)
	{
		IsValueGiven = true;
		m_oV850DlCtrlMgr.vSetSecurityByte(CHIP_ERASE_INPUT_FLAG);
		memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );
		acReqMsgBuffer[0] = DL_REQUEST_SECURITY_ACCESS;
		vAddRequestToMsgQ( acReqMsgBuffer );
		m_oV850DlCtrlMgr.vWaitForSemRelease();
	}

	if((u8ManipulatedVal & WRITE_FLAG) == WRITE_FLAG)
	{
		IsValueGiven = true;
		m_oV850DlCtrlMgr.vSetSecurityByte(WRITE_ERASE_INPUT_FLAG);
		memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );
		acReqMsgBuffer[0] = DL_REQUEST_SECURITY_ACCESS;
		vAddRequestToMsgQ( acReqMsgBuffer );
		m_oV850DlCtrlMgr.vWaitForSemRelease();
	}

	if((u8ManipulatedVal & READ_FLAG) == READ_FLAG)
	{
		IsValueGiven = true;
		m_oV850DlCtrlMgr.vSetSecurityByte(READ_ERASE_INPUT_FLAG);
		memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );
		acReqMsgBuffer[0] = DL_REQUEST_SECURITY_ACCESS;
		vAddRequestToMsgQ( acReqMsgBuffer );
		m_oV850DlCtrlMgr.vWaitForSemRelease();
	}

	if(!IsValueGiven)
	{
		ETG_TRACE_USR4 (("No Input is given for security"));
	}

	if(bResetEnabled)
	{
		memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );
		acReqMsgBuffer[0] = DL_DO_SCC_RESET;
		vAddRequestToMsgQ( acReqMsgBuffer );
		m_oV850DlCtrlMgr.vWaitForSemRelease();
	}

}

/*******************************************************************************************
* FUNCTION:		vOnGetJtagByteVal
* DESCRIPTION:	This function Post the request for Jtag Byte Access.
*					
* PARAMETER:	
*					std::string& CStrFile, 
*					std::string& strCheckSum
*
* RETURNVALUE:	Void 
*
* HISTORY:
*					30.06.2014	Neera.K.Ponnappa
*					Initial Version.
*                   31.07.2014  Neera.K.Ponnappa
                    Fixed Lint warning
             
********************************************************************************************/
tVoid dl_tclV850ClientInterface::vOnGetJtagByteVal(std::string& CStrJtagByteVal, tBool bResetEnabled)
{
	
	tU8 acReqMsgBuffer[DEFAULT_MSGSZ];

	memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );
	m_oV850DlCtrlMgr.vSetJtagByte((const char*)CStrJtagByteVal.c_str());

	memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );
	acReqMsgBuffer[0] = DL_DO_STARTUP_SEQUENCE;
	vAddRequestToMsgQ( acReqMsgBuffer );
	m_oV850DlCtrlMgr.vWaitForSemRelease();

	memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );
	acReqMsgBuffer[0] = DL_REQUEST_JTAG_ACCESS;
	vAddRequestToMsgQ( acReqMsgBuffer );
	m_oV850DlCtrlMgr.vWaitForSemRelease();

	if(bResetEnabled)
	{
		memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );
		acReqMsgBuffer[0] = DL_DO_SCC_RESET;
		vAddRequestToMsgQ( acReqMsgBuffer );
		m_oV850DlCtrlMgr.vWaitForSemRelease();
	}

}

/*******************************************************************************************
* FUNCTION			: u16DoSecurityValidation
* DESCRIPTION		: Do Security Validation
* PARAMETER			: string (IN), string (IN)	
* RETURNVALUE		: tU16 (OUT)
* HISTORY			:
--------------------------------------------------------------------------------------------
*  DATE			| VERSION			| AUTHOR			| MODIFICATION
*-------------------------------------------------------------------------------------------
* 19-02-2016	| 0.1				| VRI7COB			| CRQ,CMG3G-9596
********************************************************************************************/
tU16 dl_tclV850ClientInterface::u16DoSecurityValidation(std::string strSecurityVal, tBool bResetEnabled)
{
	tU16 u16Return = INVALID_ERROR;
	ETG_TRACE_USR4 (("u16DoSecurityValidation- START"));
	vAssignAppFlags(SECCHECK_FLAG);

	tU8 acReqMsgBuffer[DEFAULT_MSGSZ];
	memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );

	acReqMsgBuffer[0] = DL_DO_STARTUP_SEQUENCE;
	vAddRequestToMsgQ( acReqMsgBuffer );
	m_oV850DlCtrlMgr.vWaitForSemRelease();

	memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );
	acReqMsgBuffer[0] = DL_REQUEST_READ_SECURITY;
	vAddRequestToMsgQ( acReqMsgBuffer );
	m_oV850DlCtrlMgr.vWaitForSemRelease();

	tU8 u8InputCmdVal;
	// cast- for gen3 and gen4 compatibality
	sscanf(strSecurityVal.c_str(), "%x",(unsigned int*) &u8InputCmdVal);

	tU8 u8V850SecurityVal = m_oV850DlCtrlMgr.u8getSecurityBitVal();
	ETG_TRACE_USR4(("Security Value From V850:0x%x", u8V850SecurityVal));
	ETG_TRACE_USR4(("Security Value From CmdLine:0x%x", u8InputCmdVal));

	//u8V850SecurityVal bytes has to do bitwise Not and check. V850 will give as like that.
	if(u8InputCmdVal == (~u8V850SecurityVal & u8InputCmdVal))
	{
		ETG_TRACE_USR4(("Security is Matched"));
		u16Return = INIT_SUCCESS;
	}
	else
	{
		ETG_TRACE_USR4(("Security is Not Matched"));
		u16Return = SECURITY_ERROR;
	}

	if(bResetEnabled)
	{
		memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );
		acReqMsgBuffer[0] = DL_DO_SCC_RESET;
		vAddRequestToMsgQ( acReqMsgBuffer );
		m_oV850DlCtrlMgr.vWaitForSemRelease();
	}

	ETG_TRACE_USR4 (("u16DoSecurityValidation- END"));
	return u16Return;
}

/*******************************************************************************************
* FUNCTION			: u16DoSecurityValidation
* DESCRIPTION		: Do Security Validation
* PARAMETER			: None	
* RETURNVALUE		: string (OUT)
* HISTORY			:
--------------------------------------------------------------------------------------------
*  DATE			| VERSION			| AUTHOR			| MODIFICATION
*-------------------------------------------------------------------------------------------
* 15-03-2016	| 0.1				| VRI7COB			| CRQ,CMG3G-9596
*-------------------------------------------------------------------------------------------
* 18-05-2016	| 0.2				| VRI7COB			| Taking the 1's complement of security value from V850.
********************************************************************************************/
string dl_tclV850ClientInterface::strReadSecuiryBitFromV850()
{
	ETG_TRACE_USR4 (("u16DoSecurityValidation- START"));
	vAssignAppFlags(SECCHECK_FLAG);

	tU8 acReqMsgBuffer[DEFAULT_MSGSZ];
	memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );

	acReqMsgBuffer[0] = DL_DO_STARTUP_SEQUENCE;
	vAddRequestToMsgQ( acReqMsgBuffer );
	m_oV850DlCtrlMgr.vWaitForSemRelease();

	memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );
	acReqMsgBuffer[0] = DL_REQUEST_READ_SECURITY;
	vAddRequestToMsgQ( acReqMsgBuffer );
	m_oV850DlCtrlMgr.vWaitForSemRelease();

	tU8 u8V850SecurityVal = m_oV850DlCtrlMgr.u8getSecurityBitVal();

	memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );
	acReqMsgBuffer[0] = DL_DO_SCC_RESET;
	vAddRequestToMsgQ( acReqMsgBuffer );
	m_oV850DlCtrlMgr.vWaitForSemRelease();

	string result;
	result.clear();

	if(u8V850SecurityVal != 0) {
		tU8 u8SecurityOneComplement =(tU8)(int)( ~u8V850SecurityVal);
		char str[32] = {0};
		sprintf(str, "%x", u8SecurityOneComplement);
		result.assign(str);
	}

	return result;

}

/*******************************************************************************************
* FUNCTION			: u16DoJTAGFingerprintValidation
* DESCRIPTION		: Do Fingerprint Validation
* PARAMETER			: string (IN), string (IN)	
* RETURNVALUE		: tU16 (OUT)
* HISTORY			:
--------------------------------------------------------------------------------------------
*  DATE			| VERSION			| AUTHOR			| MODIFICATION
*-------------------------------------------------------------------------------------------
* 19-02-2016	| 0.1				| VRI7COB			| CRQ,CMG3G-9596
********************************************************************************************/
tU16 dl_tclV850ClientInterface::u16DoJTAGFingerprintValidation(std::string strjtagVal, tBool bResetEnabled)
{
	tU16 u16Return = INVALID_ERROR;
	ETG_TRACE_USR4 (("u16DoJTAGFingerprintValidation- START"));
	vAssignAppFlags(JTAGCHECK_FLAG);

	tU8 acReqMsgBuffer[DEFAULT_MSGSZ];
	memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );

	acReqMsgBuffer[0] = DL_DO_STARTUP_SEQUENCE;
	vAddRequestToMsgQ( acReqMsgBuffer );
	m_oV850DlCtrlMgr.vWaitForSemRelease();

	m_oV850DlCtrlMgr.vsetJTAG_CRC_Byte((const char*)strjtagVal.c_str());

	memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );
	acReqMsgBuffer[0] = DL_DO_JTAGFINGERPRINT_VALIDATION;
	vAddRequestToMsgQ( acReqMsgBuffer );
	m_oV850DlCtrlMgr.vWaitForSemRelease();

	switch(m_oV850DlCtrlMgr.getJTAGValidationStatus())
	{
	case E_JTAG_VALIDATION_SUCCESS:
		ETG_TRACE_USR4(("JTAG Validation is Success"));
		u16Return = INIT_SUCCESS;
		break;
	case E_JTAG_VALIDATION_FAILURE:
		ETG_TRACE_USR4(("JTAG Validation is Failed"));
		u16Return = JTAG_VALIDATION_ERROR;
		break;
	default:
	case E_JTAG_VALIDATION_NOTIMPLEMENTED:
		ETG_TRACE_USR4(("JTAG Validation is Failed"));
		u16Return = JTAG_VALIDATION_NOTIMPLEMENTED;
		break;

	}

	if(bResetEnabled)
	{
		memset(acReqMsgBuffer, 0, DEFAULT_MSGSZ );
		acReqMsgBuffer[0] = DL_DO_SCC_RESET;
		vAddRequestToMsgQ( acReqMsgBuffer );
		m_oV850DlCtrlMgr.vWaitForSemRelease();
	}

	ETG_TRACE_USR4 (("u16DoJTAGFingerprintValidation- END"));

	return u16Return;
}

	/*******************************************************************************************
* FUNCTION:		vAssignAppFlags
* DESCRIPTION:	This function sets the  App Flags .
*					
* PARAMETER:	None
*
* RETURNVALUE:	Void 
*
* HISTORY:
*					08.12.2014	Neera.K.Ponnappa
*					Initial Version.
            
********************************************************************************************/
tVoid dl_tclV850ClientInterface::vAssignAppFlags(int AppFlagVal)
{
	switch ( AppFlagVal )
		{
			//response for routine control
		case VERCHECK_FLAG:
			{
				m_oV850DlCtrlMgr.m_bVerFlag = true;  
		        break;
			}
		case CRCCHECK_FLAG:
			{
				m_oV850DlCtrlMgr.m_bCrcFlag = true;
				break;
			}
		case SECCHECK_FLAG:
			{
				m_oV850DlCtrlMgr.m_bSecFlag = true;
				break;
			}
		case JTAGCHECK_FLAG:
			{
				m_oV850DlCtrlMgr.m_bJtagFlag = true;
				break;
			}
		case UPDATECHECK_FLAG:
			{
				m_oV850DlCtrlMgr.m_bUpdateReqFlag = true; 
				break;
			}
		case RESETCHECK_FLAG:
			{
				m_oV850DlCtrlMgr.m_bResetFlag = true;
				break;
			}
		default: 
			{
				break;
		    }

	}
}
