/**
 * @file       swu_targetKey.h
 * @author     Philipp Blanke (blp4hi) <philipp.blanke@de.bosch.com>
 * @date       Thu 22 Sep 2016, 06:51
 * @copyright  Robert Bosch Car Multimedia GmbH
 * @brief      Access to the cryptographic private key of the target.
 */

#ifndef SWU_TARGET_KEY_H
#define SWU_TARGET_KEY_H

#include <openssl/rsa.h>

#include "util/swu_crypto.hpp"
#include "util/swu_trace.h"
#include "swu_singleton.hpp"

namespace swu {

/** @brief Interface for different accesses to a cryptographic private key.
 *  This has the purpose to decrypt messages encrypted with the corresponding
 *  public key.
 *
 * The private key is stored on the target. There are different types of 
 * storage, which is reflected through sub-classes (swu::NORTargetKey, 
 * swu::SdcTargetKey and swu::FileTargetKey).
 *
 * The interface supports loading a private key and decrypting a message
 * with it.
 */
class CTargetKeyIf {
public:
   CTargetKeyIf() : _rsa(0), _loaded(false) {}
   virtual ~CTargetKeyIf();

   /** Abstract method to be implemented.
    * Should return false, if the key can't be loaded, otherwise true. */
   virtual bool load()=0;

   /** Returns, if there is currently a key loaded. */
   virtual bool isLoaded () { return _loaded; }

   /** @brief Directly decrypt a message with the loaded key.
    *
    * This uses direct decryption of a message with the target key. The
    * length of this message is limited by the length of the key. (Compare 
    * libcrypto RSA_size()).
    * 
    * @param encrypted  Input. The encrypted message.
    * @param decrypted  Output. The decrypted message.
    * @return False, if the key was not loaded or the decryption fails.
    */
   bool decrypt(const std::vector<SWU_BYTE>& encrypted, std::string &decrypted);

   /** @brief Decrypt CMS container with a hybrid algorithm.
    *
    * This uses a hybrid algorithm, where a symmetric key is decrypted with
    * the private key and then used to decrypt the message. The message can 
    * have arbitrary length (it maybe padded to fit key length).
    *
    * @param encrypted  Input. The encrypted message.
    * @param decrypted  Output. The decrypted message.
    * @return False, if the key was not loaded or the decryption fails.
    */
   bool decryptCMS(const std::string& encrypted, std::string &decrypted);

   /** @brief Create an RSA data structure from the given data in PEM or
    *         DER format and store it.
    *
    *  @param certData  Data in PEM or DER format.
    *  @return  True, if the RSA key was successfully stored, otherwise false.
    */
   virtual bool readPemOrDer (const std::vector< SWU_BYTE >& keyData);
protected:
   RSA* _rsa;
   bool _loaded;
};

/** @brief Source of the Target Key is NOR memory. */
class NORTargetKey : public CTargetKeyIf, public Singleton < NORTargetKey >
{
   friend class Singleton< NORTargetKey >;
   NORTargetKey();
public:
   virtual ~NORTargetKey() {}
   /** The key will be loaded from one of the NOR memory locations defined in 
    *  swu::Constants::Mmc::MTD_TARGET_KEY1_OFFSET and 
    *  swu::Constants::Mmc::MTD_TARGET_KEY2_OFFSET.
    */
   virtual bool load();

private:
   bool setTargetKey (const std::vector<SWU_BYTE>& data);
};


/** @brief Source of the Target Key is a Secure Data Container. */
class SdcTargetKey : public CTargetKeyIf, public Singleton < SdcTargetKey >
{
   friend class Singleton< SdcTargetKey >;
   SdcTargetKey();
public:
   virtual ~SdcTargetKey() {}
   /** The key will be loaded from the file defined in 
    * SWU::Constants::SDC::PATH_PRIVKEY and decrypted using 
    * SourceSdc::read(). */
   virtual bool load();
};

/** @brief Source of the Target Key is a plain file. */
class FileTargetKey : public CTargetKeyIf, public Singleton < FileTargetKey >
{
   friend class Singleton< FileTargetKey >;
   FileTargetKey();
public:
   virtual ~FileTargetKey() {}
   /** The key will be loaded from the file defined in 
    * SWU::Constants::SDC::PATH_PRIVKEY. The file is loaded
    * directly without any decryption. This exists for testing purposes.
    */
   virtual bool load();
};
}
#endif   // #define SWU_TARGET_KEY_H
