/*
 * swu_securityEngine.h
 *
 *  Created on: Jul 30, 2015
 *      Author: sof4hi
 */

#ifndef SWU_SECURITY_ENGINE_H_
#define SWU_SECURITY_ENGINE_H_

#include <string>
#include <vector>
#include <istream>
#include <ostream>
#include <stdint.h>
#include <openssl/evp.h>
#include "tinyxml/tinyxml.h"

#include "util/swu_types.h"
#include "util/swu_crypto.hpp"

namespace swu {


/**
 * This class processes digest and encryption given from FC Update over ASF or from file.
 */
class SecurityEngine {

public:

   SecurityEngine();

   SecurityEngine(const SecurityEngine &copyFrom);

   // Deletes the SecurityEngine objects
   ~SecurityEngine() {};

   bool readParameters(const std::string &file);
   bool writeParameters(const std::string &file);

   std::string _encryptParam;
   swu::tenDigestType _enDigestType;
   off_t _blockLength;
   off_t _fileLength;
   ::std::vector< ::std::vector< uint8_t > > _digests;
   
   /** @brief Verifies or creates the digest and decrypts input data.
    *
    * @param inFile A file name where the digest needs to be verified or calculated
    * @param outFile Output file name.
    *           If that is NULL - the processed data are not output to file
    * @param doVerify if true the method verifies the digest, if false - generates one.
    * @return false on error, true if the verification could be checked. A wrong
    *         digest also result in a return value true, as it could be checked.
    *         Only result tells then if the the verification was successful.
    */
   bool processAndCatenate(const std::string &inFile, const std::string &outFile,
      bool doVerify=true);
      
   static const ::std::string PARAMETER_FILENAME() {return "/tmp/securityParameters.xml";}

   static const ::std::string DIGEST_TYPE_NONE() {return "NONE";}
   static const ::std::string DIGEST_TYPE_SHA1() {return "SHA1";}
   static const ::std::string DIGEST_TYPE_SHA256() {return "SHA256";}
   static const ::std::string DIGEST_TYPE_MD5() {return "MD5";}

   static const ::std::string ENCRYPT_TYPE_NONE() {return "encrypt_none";}
   static const ::std::string ENCRYPT_TYPE_AES_128_CBC() {return "aes-128-cbc";}
   static const ::std::string ENCRYPT_TYPE_AES_192_CBC() {return "aes-192-cbc";}
   static const ::std::string ENCRYPT_TYPE_AES_256_CBC() {return "aes-256-cbc";}

private:
   bool readFromXml(TiXmlElement *);
   TiXmlElement toXml();

   SecurityEngine& operator = ( const SecurityEngine & );  //Coverity fix for 47283
   static const ::std::string SECURITY_PARAM() {return "SECURITY_PARAM";}
   static const ::std::string ENCRYPT_PARAM() {return "ENCRYPT_PARAM";}
   static const ::std::string SYM_ENCRYPT_PARAM() {return "ENC";}
   static const ::std::string ENCRYPT_TYPE() {return "METHOD";}
   static const ::std::string ENCRYPT_KEY() {return "KEY";}
   static const ::std::string ENCRYPT_IV() {return "IV";}
   static const ::std::string DIGEST_TYPE() {return "DIGEST_TYPE";}
   static const ::std::string BLOCK_SIZE() {return "BLOCK_SIZE";}
   static const ::std::string FILE_SIZE() {return "FILE_SIZE";}
   static const ::std::string DIGEST_SERIES() {return "DIGEST_SERIES";}
   static const ::std::string DIGEST_VALUE() {return "DIGEST_VAL";}

/** @brief Helper method - used by SecurityEngine::processAndCatenate().
 *
 * This method can create or verify digests.
 *
 * @param inFile  A file handle. Source of the data which shall be decrypted & verified or for which
 *                the digests shall be created.
 * @param outFile A file handle. If this is not null, all decrypted data read from inData is written
 *                to this.
 * @param verify  If true the function takes the digests from the vector
 *                and checks if they are valid. if false the digests are
 *                computed and added to the vector.
 * @return        false on error, true if the verification could be checked or the
 *                digests could be created. A wrong digest in verification
 *                also results in a return value false. Only
 *                result tells whether the the verification was successful.
 */
   bool runThroughStream(int inFile, int *outFile, bool verify,
      EVP_MD_CTX *, EVP_CIPHER_CTX *);


   /** @brief Helper method - 
    *
    * This will decrypt the _encryptParams using the target's private key.
    */
   EVP_CIPHER_CTX *initCipherContext();

   int readChunk(int handle, char *buffer, off_t bufferSize, off_t readLength);

   bool writeChunk(int *handle, SWU_BYTE *buffer, off_t writeLength);

};

}

#endif /* SWU_SECURITY_ENGINE_H_ */
