#ifndef SWU_ROBUST_FILE_H
#define SWU_ROBUST_FILE_H

#include "swu_types.h"
#include <string>

namespace swu {

/*
  Message-classes the can be broadcasted
*/
// the base-class for all messages
class RobustFile {
   /**
    * Robust File can be used to secure a stored file. Therefore you give
    * it the path and the name of the file. To secure the file it stores a
    * copy of the file.
    */
public:
   /**
    * create a RobustFile in the given path
    *
    * Location for secondary copy and CRC are chosen automatically
    */
   RobustFile(std::string path);

   /**
    * create a RobustFile in the given path
    *
    * Location for secondary copy and CRC are chosen automatically
    */
   RobustFile(std::string path, std::string secure_area_path);

   virtual ~RobustFile() {
   }

   /**
    * Initialisation. Has to be called after constructor and before any other methon call.
    *
    * @return true on success, false if failed (this Object my not be used then, as it is not initialized!)
    */
   bool init(bool lazy = false);

   /**
    * The file has been written. Sync it.
    *
    * @return true iff everything is o.k.
    */
   bool sync();

   /**
    * This Version restores the file from the synced version if the actual version is damaged.
    */
   bool restore();

   /**
    * Removes the secured file incl. copy and metadata file.
    */
   bool remove();

   /**
    * Returns the secured file
    */
   inline const std::string &file_path();

private:
   enum status_t {
      NOT_INITIALIZED, INITIALIZED
   };

   static const std::string k_DEFAULT_PATH_EXTENSION;
   static const int k_DEFAULT_DIRMODE = 0700;
   static const int k_DEFAULT_FILEMODE = 0600;

   /**
    * Added some output in case of error
    */
   void print_error_out(uint line, const char *msg, ...);

   status_t _status;
   std::string _file_path;
   std::string _secure_area_path;
   bool _explicit_secure_path;
   std::string _file_copy_path;
   std::string _file_crc_path;
   std::string _file_crc_copy_path;
};

inline const std::string &RobustFile::file_path() {
   return _file_path;
}

}

#endif

