#include "util/swu_util.hpp"
#include "util/swu_kvstore.hpp"
#include "tinyxml/tinyxml.h"

#include "util/swu_trace.h"
#include "util/swu_util.hpp"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_UTIL
#include "trcGenProj/Header/swu_kvstore.cpp.trc.h"
#endif

using namespace std;

namespace swu {

const std::string KVStore::k_EMPTY_STRING = "";

KVStore::KVStore(const std::string &filename) :
         _xmlFile(filename), _xmlRoot(0), _initialized(false) {
   ETG_TRACE_COMP(("KVStore(%s) CTOR", filename.c_str()));
}

KVStore::~KVStore() {
   _xmlRoot = 0;
}

void KVStore::trace() {
   ETG_TRACE_COMP(("KVStore(_initialized=%u): trace:", _initialized));
   SWU_ASSERT_RETURN(_xmlRoot);
   for (TiXmlElement *item = _xmlRoot->FirstChildElement("ITEM");
            item; item = item->NextSiblingElement("ITEM")) {
      char const *key = getTextFromChild(item, "KEY");
      char const *val = getTextFromChild(item, "VALUE",false,"");
      ETG_TRACE_COMP(("    key=%15s val=%s", key ? key : "invalid key", val ? val : "invalid val"));
   }

}

void KVStore::init() {
   ETG_TRACE_COMP(("KVStore::init(): _initialized=%u", _initialized));
   if (_initialized)
      return;
   bool create = true;
   if (_xmlFile.exists()) {
      ETG_TRACE_COMP(("KVStore::init(): file exist!"));
      if (!_xmlFile.restore()) {
         ETG_TRACE_ERR(("KVStore::init(): restore failed!"));
         _xmlFile.remove();
      } else {
         _xmlRoot = _xmlFile.getXmlDoc().FirstChildElement("STORE");
         if (!_xmlRoot) {
            ETG_TRACE_ERR(("KVStore::init(): _xmlRoot (STORE) missing!"));
            _xmlFile.remove();
         } else {
            create = false;
         }
      }
   }
   if (create) {
      ETG_TRACE_ERR(("KVStore::init(): create new xmlDoc!"));

      _xmlRoot = new TiXmlElement("STORE");
      _xmlFile.getXmlDoc().LinkEndChild(_xmlRoot);
      SWU_ASSERT_RETURN(_xmlFile.store());
   }
   _initialized = true;
}

TiXmlElement *KVStore::getElem(const std::string &key) const {
   ETG_TRACE_COMP(("KVStore: getKeyElem=%s", key.c_str()));
   SWU_ASSERT_RETURN_VAL(_initialized, 0);
   SWU_ASSERT_RETURN_VAL(_xmlRoot, 0);

   for (TiXmlElement *itemElement = _xmlRoot->FirstChildElement("ITEM");
            itemElement;
            itemElement = itemElement->NextSiblingElement("ITEM")) {
      const char *keyEntry = getTextFromChild(itemElement, "KEY");
      if (keyEntry && (strcmp(key.c_str(), keyEntry) == 0)) {
         ETG_TRACE_COMP(("KVStore: getKeyElem: found %s", key.c_str()));
         return itemElement;
      }
   }
   return 0;
}

bool KVStore::read(const std::string &key, std::string &val) const {
   ETG_TRACE_COMP(("KVStore: getValue(key=%s)", key.c_str()));
   TiXmlElement *itemElem = getElem(key);
   if (!itemElem) {
      val="";
      ETG_TRACE_COMP(("KVStore: key not found!"));
      return false;
   }
   val = getTextFromChild(itemElem, "VALUE",false,"");
   ETG_TRACE_COMP(("KVStore: getValue(val=%s)", val.c_str()));
   return true;
}

bool KVStore::read(const std::string &key, tU32 &val) const {
   string strVal;
   bool res= read(key, strVal);
   val=atoi(strVal.c_str());
   return res;
}

bool KVStore::read(const std::string &key, std::vector<tU8> &val) const {
   string strVal;
   if (!read(key, strVal)) {
      return false;
   }
   return hexToBin(strVal, val);

}


bool KVStore::write(const std::string &key, const std::string &val) {
   ETG_TRACE_COMP(
            ("KVStore: write(val=%30s, key=%s)", val.c_str(), key.c_str()));
   SWU_ASSERT_RETURN_FALSE(_xmlRoot);
   TiXmlElement *itemElem = getElem(key);
   if (!itemElem) {
      ETG_TRACE_COMP(("KVStore: NEW KEY!"));
      itemElem = new TiXmlElement("ITEM");
      addTextChild(itemElem, "KEY", key.c_str());
      addTextChild(itemElem, "VALUE", val.c_str());
      _xmlRoot->LinkEndChild(itemElem);
   } else {
      std::string curVal = getTextFromChild(itemElem, "VALUE",false,"");
      if(curVal.empty())
      {
         setTextChild(itemElem, "VALUE", val.c_str()); 
      }
      else
      {
         TiXmlElement *valElem = itemElem->FirstChildElement("VALUE");
         valElem->ReplaceChild(valElem->FirstChild(), TiXmlText(val.c_str()));
      }
   }
   itemElem = 0;
   return true;
}

bool KVStore::write(const string &key, tU32 const &val) {
   ETG_TRACE_COMP(("KVStore: write(val=%u, key=%s)", val, key.c_str()));
   char str[10];
   sprintf(str, "%u",(unsigned int) val);
   return write(key, str);
}

bool KVStore::write(const string &key, std::vector<tU8> const &val) {
   std::string str=binToHex(val);
   ETG_TRACE_COMP(("KVStore: write(val=%50s, key=%s)", str.c_str(), key.c_str()));
   return write(key, str);
}


void KVStore::removeKey(const std::string &key) {
   ETG_TRACE_COMP(("KVStore: removeKey(key=%s)", key.c_str()));
   SWU_ASSERT_RETURN(_initialized);
   SWU_ASSERT_RETURN(_xmlRoot);
   TiXmlElement *itemElem = getElem(key);
   if (itemElem) {
      _xmlRoot->RemoveChild(itemElem);
   } else {
      ETG_TRACE_COMP(
               ("KVStore: removeKey(key=%s) KEY DOES NOT EXIST", key.c_str()));
   }
}
}
