#include <sstream>
#include <fstream>

#define DIAGNOSTICS_KDS_ENTRY_IMPORT_INTERFACE
#include "fc_diagnosis_if.h"

#include "util/swu_types.h"
#include "util/swu_filesystem.h"
#include "util/swu_kds.h"


#include "util/swu_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_UTIL
#include "trcGenProj/Header/swu_kds.cpp.trc.h"
#endif

using namespace std;

//#define FCSWUPD_SIM_KDS_FILE FCSWUPD_PERSISTENT_ROOT "fcswupd_simKds.txt"
namespace swu {
struct TsKDSEntry {
   // workaround for missing struct-tag in declaration of tsKDSEntry
   tsKDSEntry e;   
};


bool KdsDriver::write(TsKDSEntry *kdsEntry) {
   bool res=true;
   ETG_TRACE_USR4(("KdsDriver::write: table=0x%04x, len=%u data=%02x",
                   kdsEntry->e.u16Entry,
                   kdsEntry->e.u16EntryLength,
                   ETG_LIST_LEN(kdsEntry->e.u16EntryLength),
                   ETG_LIST_PTR_T8(kdsEntry->e.au8EntryData)));


   if (!open()) {
      return false;
   }
   if (!setWriteEnable(true)) {
      close();
      return false;
   }
   OSAL_s32IOControl(_IOKdsHandle,OSAL_C_S32_IOCTRL_KDS_INVALIDATE_ENTRY,kdsEntry->e.u16Entry);
   tU16 reqLen=kdsEntry->e.u16EntryLength;
   tS32 resLen=OSAL_s32IOWrite(_IOKdsHandle,(tPCS8)&kdsEntry->e, sizeof(kdsEntry->e));
   if (reqLen > resLen) {
      ETG_TRACE_ERR(("KdsDriver::write failed, reqLen=%d, resLen=%d",
                     reqLen, resLen));
      res=false;
   }
   else if (OSAL_s32IOControl(_IOKdsHandle, OSAL_C_S32_IOCTRL_KDS_WRITE_BACK, 0) == OSAL_ERROR) {
      ETG_TRACE_ERR(("KdsDriver::ioCrtr failed"));
#ifndef LSIM      
      res=false;
#endif
   } 
   sync();
   OSAL_s32ThreadWait(100);
   sync();
   OSAL_s32ThreadWait(100);

   setWriteEnable(false);
   close();
   return res;
}

bool KdsDriver::read(TsKDSEntry *kdsEntry) {
   bool res=true;
   if (!open()) {
      return false;
   }
   tU16 entry= kdsEntry->e.u16Entry;
   ETG_TRACE_USR4(("KdsDriver::read(key=0x%04x, len=%u flags=0x%08x) START",
                     kdsEntry->e.u16Entry, kdsEntry->e.u16EntryLength, kdsEntry->e.u16EntryFlags));
   tS32 osalRet=OSAL_s32IORead(_IOKdsHandle,(tPS8)&kdsEntry->e, sizeof(kdsEntry->e));
   kdsEntry->e.u16Entry=entry;
   if (osalRet == OSAL_ERROR) {
      ETG_TRACE_ERR(("KdsDriver::read(key=0x%04x, len=%u flags=0x%08x) failed",
                     kdsEntry->e.u16Entry, kdsEntry->e.u16EntryLength, kdsEntry->e.u16EntryFlags));
      res=false;
   }
   ETG_TRACE_USR4(("DATA:%02x", 
                   ETG_LIST_LEN(kdsEntry->e.u16EntryLength), ETG_LIST_PTR_T8(kdsEntry->e.au8EntryData)));
   close();
   return res;
      
}

bool KdsDriver::open() {
   _IOKdsHandle = OSAL_IOOpen(OSAL_C_STRING_DEVICE_KDS , OSAL_EN_READWRITE);
   if( OSAL_ERROR == _IOKdsHandle )
   {
      ETG_TRACE_ERR(( " KdsHandle::open(): -> KDS open operation failed, OSAL_u32ErrorCode=0x%08X"
                      ,OSAL_u32ErrorCode()));
      return false;
   }
   return true;
      
};

void KdsDriver::close() {
   if (_bWriteEnabled) {
      setWriteEnable(false);
   }
   if (_IOKdsHandle != OSAL_ERROR) {
      OSAL_s32IOClose(_IOKdsHandle);
      _IOKdsHandle = OSAL_ERROR;
   }
};

bool KdsDriver::setWriteEnable(bool enable) {
   if (_IOKdsHandle == OSAL_ERROR) {
      ETG_TRACE_ERR(("setWriteEnable failed, invalid handle!"));
      return false;
   }
   tS32 osalEnable=enable ? (tS32)TRUE : (tS32)FALSE;
   tS32 osalRes=OSAL_s32IOControl(_IOKdsHandle, OSAL_C_S32_IOCTRL_KDS_WRITE_ENABLE, osalEnable);
   if ( OSAL_OK !=  osalRes) {
      ETG_TRACE_ERR(("Enabling KDS write mode failed! osalRes=0x%x", osalRes));
      return false;
   }
   _bWriteEnabled=enable;
   return true;
};

KdsTable::KdsTable(tU16 key, tU16 len) {
   _kdsEntry=new TsKDSEntry;
   _kdsEntry->e.u16Entry=key;
   _kdsEntry->e.u16EntryLength=len;
   _kdsEntry->e.u16EntryFlags = M_KDS_ENTRY_FLAG_NONE;
   ETG_TRACE_USR4(("KdsTable CTOR:key=0x%04x, len=%u",
                   key, len));
}

KdsTable::KdsTable(KdsItemDescriptionBase const *item) {
   _kdsEntry=new TsKDSEntry;
   _kdsEntry->e.u16Entry=item->_table;
   _kdsEntry->e.u16EntryLength=item->_tableLen;
   _kdsEntry->e.u16EntryFlags = M_KDS_ENTRY_FLAG_NONE;
   ETG_TRACE_USR4(("KdsTable CTOR:key=0x%04x, len=%u",
                   item->_table, item->_tableLen));
}

KdsTable::~KdsTable() {
   if (_kdsEntry) {
      delete _kdsEntry;
      _kdsEntry=0;
   }
}

bool KdsTable::write(tU8 const *data, tU16 offset, tU16 len, bool doSync) {
   _kdsEntry->e.u16EntryFlags = M_KDS_ENTRY_FLAG_NONE;
   ETG_TRACE_USR4(("KdsTable::write(): table=0x%04x offset=%u len=%u doSync=%u data=%02x", 
                   _kdsEntry->e.u16Entry, offset, len, doSync,
                   ETG_LIST_LEN(len), ETG_LIST_PTR_T8((data))));
   (void) ::memcpy(_kdsEntry->e.au8EntryData+offset, data, len);
   if (doSync) {
      KdsDriver drv;
      if (!drv.write(_kdsEntry)) {
         return false;
      }
   }
   return true;
}

bool KdsTable::read() {
   bool res = true;
   ETG_TRACE_USR4(("KdsTable::read(0x%04x) ", _kdsEntry->e.u16Entry));

   KdsDriver drv;
   if (!drv.read(_kdsEntry)) {
      memset(_kdsEntry->e.au8EntryData, 0, _kdsEntry->e.u16EntryLength);
      res = false;
   }   
   return res;
}

tU32 KdsTable::getLen() {
   return _kdsEntry->e.u16EntryLength;
}

bool KdsTable::getBytes(tU8 *data, tU16 offset, tU16 len) {
   if (getLen() >= len + offset) {
      memcpy(data, _kdsEntry->e.au8EntryData + offset, len);
      ETG_TRACE_USR4(("KdsTable::getBytes offset=%u len=%u data=%02x",
                      offset, len,
                      ETG_LIST_LEN(len),
                      ETG_LIST_PTR_T8(data)));
      return true;
   }
   ETG_TRACE_USR4(("KdsTable::getBytes failed"));
   return false;
}

string KdsTable::getString(tU16 offset, tU16 len) {
   ScopedBuffer<tU8> buf(len + 1, true);
   memcpy(buf.getBuf(), _kdsEntry->e.au8EntryData+offset, len);
   std::string res((char const *)buf.getBuf());
   ETG_TRACE_USR4(("KdsTable::getString, offset=%u itemlen=%u strlen=%u val=%s",
                   offset, len, res.size(), res.c_str()));
   return res;
}

bool Kds::read(KdsItemDescriptionBase *item, string &str) {
   string res;
   if (item->_type != enKsdEntryType_String) {
      return false;
   }
   KdsTable table(item);
   if (!table.read()) {
      return false;
   }
   str = table.getString(item->_offset, item->_len);
   return true;
}

bool Kds::read(KdsItemDescriptionBase *item, vector<tU8> &val) {
   string res;
   if (item->_type != enKsdEntryType_Data) {
      return false;
   }
   KdsTable table(item);
   if (!table.read()) {
      return false;
   }
   val.resize(item->_len, 0);
   
   return table.getBytes(&val[0], item->_offset, item->_len);

}

bool Kds::read(KdsItemDescriptionBase *item, tU32 &val) {

   KdsTable table(item);
   if (!table.read()) {
      ETG_TRACE_ERR(("Kds::read: could not read table 0x%04x",
                     item->_table));
      return false;
   }

   tU8 tmp[4];
   bool res=table.getBytes(tmp, item->_offset, item->_len);

   val=0;
   for (unsigned int j=0;j<item->_len; ++j) {
      val<<=8;
      val|=tmp[j];
   }

   ETG_TRACE_USR4(("Kds::getU16Val:res=%u, val=%u (0x%04x)",
                   res, val, val));
   return res;
}

bool Kds::write(KdsItemDescriptionBase *item, tU32 val) {
   tU8 data[4];
   if (item->_len > 4) {
      return false;
   }
   if(item->_type!=enKsdEntryType_UInt) {
      return false;
   }  
   for(tU32 i=0; i<4 ; ++i) {
      data[3 -i]=static_cast<tU8> (val>>(i*8));
   }  
   KdsTable table(item);
   table.read();
   table.write(data + 4 - item->_len, item->_offset, item->_len);
   return true;
}

bool Kds::write(KdsItemDescriptionBase *item, string const &val) {
   if(item->_type!=enKsdEntryType_String) {
      return false;
   }  
   KdsTable table(item);
   table.read();
   vector<tU8> buf(item->_len, 0);
   memcpy(&buf[0], val.c_str(), val.size());
   table.write((tU8 const *)val.c_str(), item->_offset, item->_len);
   return true;
}

bool Kds::write(KdsItemDescriptionBase *item, vector<tU8> const &val) {
   if(item->_type!=enKsdEntryType_Data) {
      return false;
   }  
   KdsTable table(item);
   tU32 writeLen=static_cast<tU32> ((val.size() < item->_len) ? val.size() : item->_len );
   table.read();
   table.write(&val[0], item->_offset,static_cast<tU16> (writeLen) );
   return true;
}

}
