
#include <unistd.h>
#include <fcntl.h>

#include <errno.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <netdb.h>


// todo (hpe2hi): find better way to include
// taken from ai_osal_linux/components
#include "system/system_types.h"
#include "inc/inc.h"
#include "inc/inc_ports.h"
#include "inc/dgram_service.h"

#include "swu_incIf.hpp"


#include "util/swu_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_UTIL
#include "trcGenProj/Header/swu_incIf.cpp.trc.h"
#endif

using namespace std;

namespace swu {

struct DgramHolder {
   DgramHolder():
      _dgram(0)
   {}

   sk_dgram *_dgram;
};



IncIf::IncIf(uint16_t s16LocalPort):
   _dgramHolder(new DgramHolder),
   _sockfd(-1),
   _bOpen(false),
   _s16LocalPort(s16LocalPort)
{
   ETG_TRACE_USR4(("IncIf CTOR: _s16LocalPort=%u", _s16LocalPort));
};

IncIf::~IncIf() {
   ETG_TRACE_USR4(("IncIf DTOR"));
   vClose();
   if (_dgramHolder) {
      delete _dgramHolder;
      _dgramHolder=0;
   }
}

bool IncIf::bOpen() {
   ETG_TRACE_USR4(("IncIf::bOpen(): START"));
   SWU_ASSERT_RETURN_FALSE(_dgramHolder);

   if (_bOpen) {
      ETG_TRACE_ERR(("IncIfx::bOpen(): already initialized"));
      return _bOpen;
   }
   _sockfd = socket( AF_BOSCH_INC_AUTOSAR, SOCK_STREAM, 0 ); // wait for a message to arrive
   if (_sockfd < 0)
   {
      _sockfd=-1;
      ETG_TRACE_ERR(("IncIf::bOpen(): ERROR opening socket"));
      vClose();
      return _bOpen;
   }  
   //Data gram initiation 
   _dgramHolder->_dgram = dgram_init(_sockfd, DGRAM_MAX, NULL);
   if ( !_dgramHolder->_dgram)
   {
      ETG_TRACE_ERR(("IncIf::bOpen(): dgram_init failed"));
      vClose();
      return _bOpen;
   }

   // get local adress
   //local host address, iMX
   struct hostent *hostentLocal = gethostbyname("scc-local");	  
   if (hostentLocal == NULL)
   {
      ETG_TRACE_USR4(("IncIf::bOpen(): ERROR no such local host"));
      vClose();
      return false;
   }
   struct sockaddr_in sockaddrLocal;
   sockaddrLocal.sin_family = AF_INET;
   memcpy( (char *)&sockaddrLocal.sin_addr.s_addr, (char *)hostentLocal->h_addr, hostentLocal->h_length );
   sockaddrLocal.sin_port = htons(_s16LocalPort);

   //bind to local host
   if ( bind( _sockfd, (struct sockaddr *) &sockaddrLocal, sizeof(sockaddrLocal) ) < 0 )
   {
      ETG_TRACE_USR4(("IncIf::bOpen(): ERROR on binding"));
      vClose();
      return false;

   }

   // get remote addres
   struct hostent *hostentRemote = gethostbyname("scc"); 
   if (hostentRemote == NULL)
   {
      ETG_TRACE_USR4(("IncIf::bOpen(): gethostbyname failed"));
      vClose();
      return false;
   }

   struct sockaddr_in sockaddrRemote;
   sockaddrRemote.sin_family = AF_INET;
   memcpy ( (char *) &sockaddrRemote.sin_addr.s_addr, (char *) hostentRemote->h_addr, hostentRemote->h_length );
   sockaddrRemote.sin_port = htons(_s16LocalPort);

   //Connect to remote host
   if ( connect( _sockfd,(struct sockaddr *) &sockaddrRemote, sizeof(sockaddrRemote) ) < 0 )
   {
      ETG_TRACE_USR4(("IncIf::bInit(): ERROR connecting"));
   }

   fcntl(_sockfd, F_SETFL, O_NONBLOCK);


   _bOpen=true;
   ETG_TRACE_USR4(("IncIf::bInit(): END"));

   return true;
}

void IncIf::vClose() {
   ETG_TRACE_USR4(("IncIf DTOR"));
   SWU_ASSERT_RETURN(_dgramHolder);

   if (_dgramHolder->_dgram) {
      ETG_TRACE_USR4(("IncIf DTOR: delete _dgramHolder"));

      if (dgram_exit(_dgramHolder->_dgram) < 0) {
         ETG_TRACE_USR4(("IncIf::bOpen(): dgram_exit failed"));
      }
      _dgramHolder->_dgram=0;
   }
      
   if( _sockfd != -1 )
   {
      ETG_TRACE_USR4(("IncIf DTOR: close fd"));
      close(_sockfd);
      _sockfd = -1;
   }
   _bOpen=false;

}


bool IncIf::bSend(vector<tU8> &msg) {
   if (!bOpen()) {
      return false;
   }
   ETG_TRACE_USR4(("IncIf::bSend: %02x",
                   ETG_LIST_LEN(static_cast<unsigned int> (msg.size())), ETG_LIST_PTR_T8(&msg[0])));
   int retValue=static_cast<int> (dgram_send(_dgramHolder->_dgram, &msg[0], msg.size()) );
   if (retValue < 0) {
      ETG_TRACE_ERR(("IncIf::bSend(): dgram_send failed (retValue=%d)", retValue));
      return false;
   }

   ETG_TRACE_USR4(("IncIf::bSend(numBytes=%u): allOk, END (ret value == %d)",msg.size(), retValue));

   return true;

      
}

bool IncIf::bRecv(vector<tU8> &msg, tU32 rcvLen) {
   msg.clear();
   if (!bOpen()) {
      return false;
   }
   bool bOk=false;
   msg.resize(rcvLen);
     
   for(tUInt retries=0; retries<10;++retries) {
      int retValue =static_cast<int> (dgram_recv(_dgramHolder->_dgram, &msg[0], rcvLen) );
      if (retValue>=0) {
         msg.resize(retValue);
         ETG_TRACE_USR4(("IncIf::bRcv(%u bytes): %02x",
                         retValue,
                         ETG_LIST_LEN(static_cast<unsigned int> (msg.size())), ETG_LIST_PTR_T8(&msg[0])));
         bOk=true;
         break;
      } else {
         ETG_TRACE_ERR(("IncIf::bRcv: res=%d retries=%u", retValue, retries));
         sleep(1);
      }
   }
      
   if (!bOk)  {
      msg.clear();
      ETG_TRACE_ERR(("IncIf::bRcv: failed"));
   }
   return bOk;
}
   
}
