#define _XOPEN_SOURCE
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <ctime>
#include <algorithm>

#include <openssl/bio.h>
#include <openssl/evp.h>
#include <openssl/x509v3.h>
#include <openssl/pem.h>

#include "util/swu_crypto.hpp"
#include "util/swu_trace.h"
#include "util/swu_constants.hpp"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_UTIL
#include "trcGenProj/Header/swu_crypto.cpp.trc.h"
#endif 

namespace swu {

   time_t ASN1_GetTimeT(const ASN1_TIME* asn1_time)
   {
      struct tm t;
      const char* str = (const char*) asn1_time->data;
      size_t len = ::strlen(str);
      size_t i = 0;
      ::memset(&t, 0, sizeof(t));
      if (asn1_time->type == V_ASN1_UTCTIME) {// two digit year
         if (len < 12) {
            ETG_TRACE_ERR(("Time string too short."));
            return ::time(0);
         }
         t.tm_year  = (str[i++] - '0') * 10;
         t.tm_year += (str[i++] - '0');
         if (t.tm_year < 70) {
            t.tm_year += 100;
         }
      }
      else if (asn1_time->type == V_ASN1_GENERALIZEDTIME) {// four digit year
         if (len < 14) {
            ETG_TRACE_ERR(("Time string too short."));
            return ::time(0);
         }
         t.tm_year  = (str[i++] - '0') * 1000;
         t.tm_year += (str[i++] - '0') * 100;
         t.tm_year += (str[i++] - '0') * 10;
         t.tm_year += (str[i++] - '0');
         t.tm_year -= 1900;
      }
      else {
         ETG_TRACE_ERR(("Unknown time type."));
         return ::time(0);
      }
      t.tm_mon   = (str[i++] - '0') * 10;
      t.tm_mon  += (str[i++] - '0') - 1; // -1 since January is 0 not 1.
      t.tm_mday  = (str[i++] - '0') * 10;
      t.tm_mday += (str[i++] - '0');
      t.tm_hour  = (str[i++] - '0') * 10;
      t.tm_hour += (str[i++] - '0');
      t.tm_min   = (str[i++] - '0') * 10;
      t.tm_min  += (str[i++] - '0');
      t.tm_sec   = (str[i++] - '0') * 10;
      t.tm_sec  += (str[i]   - '0');
      return ::mktime(&t);
   }

   time_t parseTimeString(const std::string& time_string, const std::string& time_format)
   {
      struct tm tm;
      memset(&tm, 0, sizeof(struct tm));
      if ( 0 == strptime(time_string.c_str(), time_format.c_str(), &tm)) {
         ETG_TRACE_FATAL(("Can't parse time string '%s'", time_string.c_str() ));
         ETG_TRACE_FATAL(("            Time format '%s'", time_format.c_str() ));
         return -1;
      }
      return std::max(mktime(&tm), (time_t)0);
   }

   std::string writeTimeString(const time_t& time_posix, const std::string& time_format)
   {
      std::string time_string = ""; 
      if (time_posix < 0) {
         ETG_TRACE_FATAL(("Invalid POSIX time '%d'.", time_posix));
         return time_string;
      }
      char time_cstr[256];
      if ( 0 == strftime(time_cstr, sizeof(time_cstr), time_format.c_str(), localtime(&time_posix)) ) {
         ETG_TRACE_ERR(("Writing an empty time string. POSIX time: '%d'", time_posix));
         ETG_TRACE_ERR(("                           Format string: '%s'", time_format.c_str() ));
         return time_string;
      }
      time_string.assign(time_cstr);
      return time_string;
   }

   std::vector<SWU_BYTE> decodeBase64(const std::string & encstr)
   {
      std::vector<SWU_BYTE> dec;
      int enclen = (int)encstr.size();
      int declen = enclen;
      int readlen = 0;
      SWU_BYTE decbuf[declen];
      BIO* b64 = 0;
      BIO* enc = 0;

      do { // emulate 'goto cleanup'
         b64 = BIO_new(BIO_f_base64());
         if (b64 == 0) break;
         enc = BIO_new_mem_buf((void*)encstr.c_str(), enclen);
         if (enc == 0) break;

         enc = BIO_push(b64, enc);

         if (::strchr(encstr.c_str(), '\n') == 0) {
            BIO_set_flags(enc, BIO_FLAGS_BASE64_NO_NL);
         }
         readlen = BIO_read(enc, decbuf, declen);
         if (readlen < 1) {
            ETG_TRACE_ERR(("Can't decode Base64 encoded string '%s'", encstr.c_str() ));
            break;
         }

         dec.assign(decbuf, decbuf + readlen);
      } while (false);

      //cleanup
      if (enc) BIO_free_all(enc);

      return dec;
   }


   X509* stringToX509Cert(const std::string &cert)
   {
      OpenSSL_add_all_algorithms();
      BIO* certBio = BIO_new(BIO_s_mem());
      if  ( ! certBio ) {
         ETG_TRACE_ERR(("Can't create BIO for certificate"));
         return 0;
      }
      BIO_puts(certBio, cert.c_str());
      X509* certX509 = PEM_read_bio_X509(certBio, NULL, NULL, NULL);
      if ( ! certX509 ) {
         ETG_TRACE_ERR(("Can't convert string to X509 certificate."));
      }
      BIO_free(certBio);
      return certX509;
   }

   ::std::vector< SWU_BYTE > decodeHex(const ::std::string& hex)
   {
      return decodeHex(hex.c_str());
   }

   ::std::vector< SWU_BYTE > decodeHex(const char *in)
   {
      ::std::vector< SWU_BYTE > result;
      if(0 == in) {
         ETG_TRACE_ERR(("reading Vector from NULL is not possible."));
         return result;
      }
      size_t len = strlen(in);

      bool first = true;
      SWU_BYTE act_val = 0;
      if (len % 2) {
         first = false;
      }

      while (*in != 0) {
         SWU_BYTE i = *in;
         if ((i >= '0') && (i <= '9')) {
            i =static_cast<swu::SWU_BYTE> (i - '0');
         } else if ((i >= 'a') && (i <= 'f')) {
            i =static_cast<swu::SWU_BYTE> (i - 'a');
            i =static_cast<swu::SWU_BYTE> (i + 10);
         } else if ((i >= 'A') && (i <= 'F')) {
            i =static_cast<swu::SWU_BYTE> (i - 'A');
            i =static_cast<swu::SWU_BYTE> (i + 10);
         } else {
            ETG_TRACE_ERR(("found wrong hex digit %c while trying to load vector", i));
            result.clear();
            return result;
         }

         if (first) {
            act_val =static_cast<swu::SWU_BYTE> (i << 4);
            first = false;
         } else {
            act_val =static_cast<swu::SWU_BYTE> (act_val + i);
            result.push_back(act_val);
            first = true;
         }
         ++in;
      }
      return result;
   }

   ::std::vector< SWU_BYTE > decodeDigest(const ::std::string& enc, const tenDigestType& digestType)
   {
      size_t digestLength = 0;

      switch(digestType) {
      case tenDigestTypeMD5:    digestLength = 16; break;
      case tenDigestTypeSHA1:   digestLength = 20; break;
      case tenDigestTypeSHA256: digestLength = 32; break;
      case tenDigestTypeInvalid: 
      case tenDigestTypeNone:   digestLength = 0; break;
      default:
         ETG_TRACE_FATAL(("Unknown digest type: %d", digestType ));
         return ::std::vector< SWU_BYTE >(); 
      }

      if (digestLength == 0 && enc.length() == 0) {
         ETG_TRACE_USR1(("Zero-length digest returned."));
         return ::std::vector< SWU_BYTE >(); 
      }

      if (enc.length() == 2*digestLength) {
         return decodeHex(enc);
      }
      
      // defaulting to Base64
      return decodeBase64(enc);
   }

   bool isPEM(const ::std::vector< SWU_BYTE >& data)
   {
      const std::string &prefix = swu::Constants::Cert::PEM_BEGIN_STR;
      return std::equal(prefix.begin(), prefix.end(), data.begin());
   }

   int X509_verify_cb(int ok, X509_STORE_CTX *ctx)
   {
      int err = X509_STORE_CTX_get_error(ctx);
      if      (err == X509_V_ERR_CERT_NOT_YET_VALID) { return 1; }
      else if (err == X509_V_ERR_CERT_HAS_EXPIRED  ) { return 1; }
      // TODO (blp4hi): add a configuration item for this.
      // else if (err == X509_V_ERR_INVALID_PURPOSE) { return 1; }
      return ok;
   }
}
