#ifndef SWU_CONFIG_ITEM_HPP
#define SWU_CONFIG_ITEM_HPP


#include "util/swu_trace.h"
#include "util/swu_util.hpp"
#include <sys/stat.h>
//#include "util/swu_configBase.h"


namespace swu {
class ConfigBase;

// base-class giving acces to driver
class ItemDescriptionBase {
public:
   virtual ~ItemDescriptionBase() {
   }
   virtual void sync() {};
   virtual bool doSimulate() { return false;};
   virtual bool getVolatile() { return false;};
};

// access for config-item of specifc type
template<class VALTYPE>
class ItemDescription:public ItemDescriptionBase {
public:
   virtual bool write(VALTYPE const &val)=0;
   virtual bool read(VALTYPE &val)=0;
};

#define SWU_CFG_ITEM_VOLATILE 0x01
#define SWU_CFG_ITEM_EXPORT   0x02
class ConfigItemBase {
public:
   ConfigItemBase(ConfigBase *cfg, std::string name, tU32 flags, std::string format):
      _cfg(cfg),
      _name(name),
      _flags(flags),
      _format(format), // only valid for U32-items
      _valid(false),
      _doSimulate(false)
   {}
   virtual ~ConfigItemBase(){};

   virtual void sync()=0;
   std::string const &getName() const {
      return _name;
   };

   bool read(bool force=false);
   bool write();
   virtual void setDefault()=0;


   virtual bool readAsBool()=0;
   virtual std::string readAsString()=0;

   void setSim(bool doSimulate) {
      _doSimulate=doSimulate;
   }

   void setVolatile(bool volatile_) {
      if (volatile_) {
         _flags |= SWU_CFG_ITEM_VOLATILE;
      } else {
         _flags^=SWU_CFG_ITEM_VOLATILE;
      }
   }

   bool getVolatile() {
      bool descriptionVolatile=false;
      ItemDescriptionBase *description=getItemDescription();
      if (description) {
         descriptionVolatile=description->getVolatile();
      }
      return ((_flags & SWU_CFG_ITEM_VOLATILE) || descriptionVolatile) ? true : false;
   }

   void setExport(bool export_) {
      if (export_) {
         _flags |= SWU_CFG_ITEM_EXPORT;
      } else {
         _flags^=SWU_CFG_ITEM_EXPORT;
      }
   }

   bool getExport() {
      return (_flags & SWU_CFG_ITEM_EXPORT) ? true : false;
   }

   std::string toString(tU32 const &val) {
      if (!_format.empty()) {
         char resStr[9];
         snprintf(resStr, 9, _format.c_str(), val);
         resStr[9]=0;
         return resStr;
      }
      return intToString(val);
   }
   std::string toString(std::string const &val) {
      return val;
   }

   std::string toString(std::vector<tU8> const &val) {
      char *tmp=new char[val.size()*2 +1];
      binToHex(&val[0], val.size(), tmp);
      tmp[val.size()*2]=0;
      std::string res=tmp;
      delete [] tmp;  //Coverity fix for 86564
      return res;
   }

   bool toBool(tU32 val) {
      return val;
   }

   bool toBool(std::string const &val) {
      return val.size();
   }

   bool toBool(std::vector <tU8> const &val) {
      return val.size();
   }
   
   void fromBool(bool bVal, tU32 &val) {
      val = bVal;
   }

   void fromBool(bool bVal, std::string &val) {
      val = bVal ? "1" : "";
   }

   void fromBool(bool bVal, std::vector <tU8> &val) {
      val.clear();
      if (bVal) {
         val.push_back(1);
      }
   }

   void traceAction(const char *action);
   void traceVal(std::string const &val);
   void traceVal(tU32 const &val);
   void traceVal(std::vector<tU8> const &val);
   void traceReadAsBool(bool res);
   void traceWrite(bool doSim);

   ConfigBase *_cfg;
   std::string _name;
   tU32 _flags;
   std::string _format;
   bool _valid;
   bool _doSimulate;

protected:
   virtual bool hasItemDescription()=0;
   virtual ItemDescriptionBase *getItemDescription()=0;
   virtual bool readVal()=0;
   virtual bool writeVal()=0;
};



template<class VALTYPE >
class ConfigItem: public ConfigItemBase {
public:
   ConfigItem(ConfigBase* cfg, std::string name, VALTYPE defaultVal, ItemDescription<VALTYPE> *itemDescription=0, tU32 flags=0, std::string format=""):
      ConfigItemBase(cfg, name, flags, format),
      _itemDescription(itemDescription),
      _defaultVal(defaultVal),
      _val(defaultVal)
   {
      ConfigItem_addToCfg(_cfg, this);
   }


   ~ConfigItem() {
      if (_itemDescription) {
         delete _itemDescription;
         _itemDescription=0;
      }
   }

   void setDefaultVal(VALTYPE defaultVal) {
      _defaultVal=defaultVal;
   }
   
   void traceVal() {
      ConfigItemBase::traceVal(_val);
   }

   SWU_MOCK_VIRTUAL VALTYPE &get(bool force=false) {
      ConfigItemBase::traceAction("get():1 START");
      read(force);
      traceVal();
      return _val;
   }
   SWU_MOCK_VIRTUAL bool get(VALTYPE &val, bool force=false) {
      ConfigItemBase::traceAction("get():2 START");
      read(force);
      val=_val;
      traceVal();
      return _valid;
   }


   virtual std::string readAsString() {
      ConfigItemBase::traceAction("readAsString()");
      read();
      ConfigItemBase::traceVal(ConfigItemBase::toString(_val));
      return ConfigItemBase::toString(_val);
   }


   virtual bool readAsBool() {
      ConfigItemBase::traceAction("readAsBool()");
      read();
      traceVal();
      bool res = _valid && toBool(_val);
      ConfigItemBase::traceReadAsBool(res);
      return res;
   }

   SWU_MOCK_VIRTUAL bool setFromBool(bool bVal, bool doSync=true) {
      ConfigItemBase::traceAction("setFromBool()");
      VALTYPE val;
      fromBool(bVal, val);
      return set(val, doSync);
   }

   SWU_MOCK_VIRTUAL bool set(VALTYPE const &val, bool doSync=true) {
      ConfigItemBase::traceAction("set(): from->to:");

      ConfigItemBase::traceVal(_val);
      ConfigItemBase::traceVal(val);
      VALTYPE oldVal=_val;
      _val=val;
      if (_valid && oldVal==_val) {
         return true;
      }
      write();
      if (!_valid) {
         ConfigItemBase::traceAction("set(): not valid");
         _val=oldVal;
      }
      else if (_val!=oldVal) {
         if (doSync) {
            sync();
         }
         ConfigItemBase::traceAction("set(): calling ConfigItem_onChanged ");
         ConfigItem_onChanged(_cfg, this);
      }
      return _valid;
   }

   virtual void setDefault() {
      _val=_defaultVal;
   }

   virtual void sync() {
      ConfigItemBase::traceAction("sync()");
      if (_itemDescription) {
         _itemDescription->sync();
      }
   }


   void removeDescription() {
      ConfigItemBase::traceAction("removeDescription()");
      ConfigItemBase::_valid=false;
      if (_itemDescription) {
         delete _itemDescription;
      }
   }


   void setDescription(ItemDescription<VALTYPE> *itemDescription) {
      ConfigItemBase::traceAction("setDescription()");
      ConfigItemBase::_valid=false;
      if (_itemDescription) {
         delete _itemDescription;
      }
      _itemDescription=itemDescription;
   }

protected:
   virtual bool hasItemDescription() {
      return _itemDescription ? true : false;
   }
   virtual ItemDescriptionBase *getItemDescription() {
      return _itemDescription;
   }
   virtual bool readVal() {
         return _itemDescription->read(_val);
   }

   virtual bool writeVal() {
      return _itemDescription->write(_val);
   }

   ItemDescription<VALTYPE> *_itemDescription;
   VALTYPE _defaultVal;
   VALTYPE _val;

protected:
   ConfigItem();

private:
   ConfigItem ( const ConfigItem & ); //Coverity fix for 48140
   ConfigItem & operator = ( const ConfigItem & ); //Coverity fix for 47796

};


void ConfigItem_addToCfg(ConfigBase *cfg, ConfigItem<tU32> *item);
void ConfigItem_addToCfg(ConfigBase *cfg, ConfigItem<std::string> *item);
void ConfigItem_addToCfg(ConfigBase *cfg, ConfigItem<std::vector<tU8> > *item);

void ConfigItem_onChanged(ConfigBase *cfg, ConfigItem<tU32> *item);
void ConfigItem_onChanged(ConfigBase *cfg, ConfigItem<std::string> *item);
void ConfigItem_onChanged(ConfigBase *cfg, ConfigItem<std::vector<tU8> > *item);


struct MarkerFileItemDescription: public ItemDescription<tU32> {
   std::string _markerFileName;
   uid_t _user;
   gid_t _group;
   mode_t _perm;   
   MarkerFileItemDescription(char const *markerFileName, uid_t user=0, gid_t group=0, mode_t perm=0660);
   //   virtual void sync() { sync();};   
   virtual bool write(tU32 const &activate);
   virtual bool read(tU32 &activate);
};


class FileItemDescriptionHelpers {
   template<class VARTYPE>
   friend class FileItemDescription;
private:
   static bool writeFileItem(std::string const &fileName, std::string const &content);
   static bool readFileItem(std::string const &fileName, std::string &content);
   static bool writeFileItem(std::string const &fileName, tU32 const &content);
   static bool readFileItem(std::string const &fileName, tU32 &content);
   static bool writeFileItem(std::string const &fileName, std::vector<tU8> const &content);
   static bool readFileItem(std::string const &fileName, std::vector<tU8> &content);
};

template<class VARTYPE>
struct FileItemDescription: public ItemDescription<VARTYPE> {
   std::string _markerFileName;
   
   FileItemDescription(char const *markerFileName):
      _markerFileName(markerFileName)
   {}
   //   virtual void sync() { sync();};   
   virtual bool write(VARTYPE const &val) {
      return FileItemDescriptionHelpers::writeFileItem(_markerFileName, val);
   }
   virtual bool read(VARTYPE &val) {
      return FileItemDescriptionHelpers::readFileItem(_markerFileName, val);
   }
private:

};




template<class VARTYPE>
struct DummyItemDescription: public ItemDescription<VARTYPE> {
   bool write(VARTYPE const &val) {
      return true;
   };

   bool read(VARTYPE &val) {
      return true;
   };
};


template<class VARTYPE>
struct ConfigVarItemDescription: public ItemDescription<VARTYPE> {
   ConfigVarItemDescription(VARTYPE &tiedVal):
      _tiedVal(&tiedVal)
   {}
   bool write(VARTYPE const &val) {
      *_tiedVal=val;
      return true;
   };

   bool read(VARTYPE &val) {
      val=*_tiedVal;
      return true;
   };

   VARTYPE *_tiedVal;
};

 struct ConfigItemSliceDescription: public ItemDescription<tU32> {
    ConfigItemSliceDescription(ConfigItem<std::vector<tU8> > *parent, tU32 offset, tU32 numBytes, tU32 mask=0);

    virtual bool getVolatile() { return true;};

    bool write(tU32 const &val);
    bool read(tU32 &val);

    ConfigItem<std::vector<tU8> > *_parent;
    tU32 _offset;
    tU32 _numBytes;
    tU32 _mask;

 };

struct ConfigItemVectorSliceDescription: public ItemDescription<std::vector<tU8> > {
   ConfigItemVectorSliceDescription(ConfigItem<std::vector<tU8> > *parent, tU32 offset, tU32 numBytes);
    virtual bool getVolatile() { return true;};

   bool write(std::vector<tU8> const &val);
   bool read(std::vector<tU8> &val);

    ConfigItem<std::vector<tU8> > *_parent;
    tU32 _offset;
    tU32 _numBytes;
 };
}
#endif
