#include <openssl/pem.h>
#include <openssl/cms.h>
#include <openssl/err.h>

#include <cstdlib>
#include <cstring>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <libgen.h>
#include <fstream>
#include <sstream>
#include <map>
#include <algorithm>

#include "util/swu_cms.h"
#include "util/swu_types.h"
#include "util/swu_certificate.h"
#include "util/swu_trace.h"
#include "util/swu_filesystem.h"
#include "util/swu_util.hpp"
#include "util/swu_targetKey.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_UTIL
#include "trcGenProj/Header/swu_cms.cpp.trc.h"
#endif

#define XML_SUFFIX          "xml"
#define XML_SUFFIX_LEN      3

#define XML_TAG_ROOT        "cms"
#define XML_TAG_META        "meta"
#define XML_TAG_LIC_FILE    "licensefile"
#define XML_TAG_LIC_PATH    "licensepath"
#define XML_TAG_FILE        "file"
#define XML_TAG_NAME        "name"
#define XML_TAG_PATH        "path"
#define XML_TAG_DIGEST_TYPE "digesttype"
#define XML_TAG_DIGEST      "digest"

#define TARGET_UIN_PROXY     "$(uin)"
#define TARGET_UIN_PROXY_LEN 6

#define MAX_INT_VALUE_32  2147483647
namespace swu {

using namespace ::std;

// TODO: check if certificate is valid?
CMS::CMS (CACertIf* caCert, ::time_t lastAuthenticTime)
   : _caCert(caCert), _lastAuthenticTime(lastAuthenticTime)
{}

bool CMS::parse(const string& file, bool allowEmptyCert)
{
   string msgStr;
   if( ! swu::loadFile(file, msgStr)) {
      ETG_TRACE_ERR(("Can't read %s", file.c_str()));
      return false;
   }

   if (! validate(_message, msgStr, allowEmptyCert) ) {
      ETG_TRACE_ERR(("Can't validate %s", file.c_str()));
      return false;
   }

   char* path = ::strdup(file.c_str());
   _path = ::dirname(path);
   free(path);
   // prepare the XML payload for further parsing
   _xmlDoc.Parse(_message.c_str());

   return true;
}

bool CMS::validate(string& outStr, const string &msgStr, bool allowEmptyCert)
{
   if (_caCert == 0 ) {
      ETG_TRACE_ERR(("No CA certificate initialized. Can't validate CMS."));
      return false;
   }

   bool isValidated = false;

   // openssl types 
   BIO             *msgBio          = 0;
   BIO             *payloadBio      = 0;
   CMS_ContentInfo *cms             = 0;
   STACK_OF(X509)  *untrustedCerts  = 0;
   X509            *signerCert      = 0;
   X509_STORE      *trustedStore    = 0;
   //X509_STORE_CTX  *storeCtx        = 0;  coverity fix for 65970. We never used this variable so commenting this variable as it is giving coverity warning.
   BIO             *diagnosisBio    = 0;
   X509_NAME       *certSubject     = 0;

   OpenSSL_add_all_algorithms();
   ERR_load_BIO_strings();
   ERR_load_crypto_strings();

   do { // this loop emulates "goto cleanup;"
      // Divide message into CMS information and payload.
      msgBio = BIO_new(BIO_s_mem());
      if ( 0 == msgBio ) {
         ETG_TRACE_ERR(("Can't initialize message BIO."));
         break;
      }
      BIO_puts(msgBio, msgStr.c_str());
      cms = SMIME_read_CMS(msgBio, &payloadBio);
      if ((0 == cms) || (0 == payloadBio)) {
         ETG_TRACE_ERR(("Can't read S/MIME CMS"));
         ETG_TRACE_ERR(("%s", ERR_error_string(ERR_get_error(), NULL) ));
         break;
      }

      if ( ! _caCert->isEmpty() ) { // we have a CA certificate
         // CMS_verify() tries to build and verify a certificate chain from 
         // certificates included in the CMS and the CA certificate.
         // During verification, among other thingsm it checks if the CAcert
         // and intermediate certs have a 'smimesign' purpose. It also checks 
         // the certificates' validity against the current local time. 
         //
         // TODO (blp4hi) any CRL to be used has to be added to the trusted store.
         trustedStore = X509_STORE_new();
         if ( 0 == trustedStore ) {
            ETG_TRACE_ERR(( "Can't initialize X509 trusted store." ));
            break;
         }
         X509_STORE_add_cert(trustedStore, _caCert->getX509());
         // We add a verification callback to optionally suppress 
         //   - validity time errors - we have no reliable system time (in
         //     recovery mode),
         //   - purpose errors - old certificates may come with a wrong purpose
         X509_STORE_set_verify_cb(trustedStore, swu::X509_verify_cb);

         // CMS_DETACHED  The payload is not included in the cms information
         int flags = CMS_DETACHED;

         if (1 != CMS_verify(cms, NULL, trustedStore, payloadBio, NULL, flags)) {
            // Debugging output ...
            ETG_TRACE_ERR(("CMS validation failed. Reason:"));
            ETG_TRACE_ERR(("%s", ERR_error_string(ERR_get_error(), NULL) ));

            // extract the signer's certificates from CMS
            untrustedCerts = CMS_get1_certs(cms);
            if ( 0 == untrustedCerts || 1 > sk_X509_num(untrustedCerts) ) {
               ETG_TRACE_ERR(("Can't extract any certificates from CMS."));
               break;
            }
            certSubject = X509_get_subject_name(sk_X509_value(untrustedCerts, 0));

            diagnosisBio = BIO_new(BIO_s_mem());
            if ( 0 == diagnosisBio) { 
               ETG_TRACE_ERR(("CMS validation failed. BIO_new is NULL"));
               break; 
            }

            X509_NAME_print_ex(diagnosisBio, certSubject, 0, XN_FLAG_MULTILINE);
            char *diagStr = 0;
            long diagLen = BIO_get_mem_data(diagnosisBio, &diagStr);
            diagStr[diagLen] = 0;
            BIO_get_mem_data(diagnosisBio, &diagStr);
            ETG_TRACE_ERR(("Offending certificate:"));
            ETG_TRACE_ERR(("%s", diagStr));
            break;
         }
      }
      else if (allowEmptyCert) { // We have no CA certificate, but allow that explicitly
         //  Just check if the CMS looks good.
         ETG_TRACE_USR3(( "Empty CA certificate. Just checking syntax of CMS." ));
         int flags = CMS_DETACHED | CMS_NO_SIGNER_CERT_VERIFY;
         if ( 1 != CMS_verify(cms, NULL, trustedStore, payloadBio, NULL, flags)) {
            ETG_TRACE_ERR(("CMS syntax check failed:"));
            ETG_TRACE_ERR(("%s", ERR_error_string(ERR_get_error(), NULL) ));
            break;
         }
      }
      else {
         ETG_TRACE_ERR(("Empty or corrupted CA certificate in target. Can't validate CMS."));
         break;
      }

      // get the validity begin from the certificate
      signerCert = sk_X509_value(CMS_get0_signers(cms), 0);
      time_t signerCertValidityBegin = swu::ASN1_GetTimeT(X509_get_notBefore(signerCert));
      time_t signerCertValidityEnd   = swu::ASN1_GetTimeT(X509_get_notAfter(signerCert));

      // Dirty hack to fix the 2038 problem on 32-bit machines
      // Max Time representation in 32-bit machines 03:14:07 UTC on 19 January 2038 
      #ifndef VARIANT_S_FTR_ENABLE_G4G
      if(signerCertValidityEnd == -1)
      {
		ETG_TRACE_USR3(("Signer's certificate is valid after 2038!" ));
        signerCertValidityEnd = MAX_INT_VALUE_32;
      }
      #endif
      // the certificate is rejected if the lastAuthenticTime is greater
      // than the validity end.
      std::string t = writeTimeString(_lastAuthenticTime, "%d-%m-%Y %H:%M:%S");
      if ( _lastAuthenticTime > signerCertValidityEnd) {
         ETG_TRACE_ERR(("Signer's certificate is expired!" ));
         ETG_TRACE_ERR(("  Last authentic time on target: %s", t.c_str() ));
         t = writeTimeString(signerCertValidityEnd, "%d-%m-%Y %H:%M:%S");
         ETG_TRACE_ERR(("Signer's certificate expired on: %s", t.c_str() ));
         break;
      }
      
      // we have not broken from the loop => validation OK!
      isValidated = true;

      // if the certificate was verified, the last authentic time can be
      // fast-forwarded (if the validity begin is later)
      if (_lastAuthenticTime < signerCertValidityBegin ) {
         ETG_TRACE_USR3(("Updating last authentic time. Old: %s", t.c_str() ));
         _lastAuthenticTime = signerCertValidityBegin;
         t = writeTimeString(_lastAuthenticTime, "%d-%m-%Y %H:%M:%S");
         ETG_TRACE_USR3(("                              New: %s", t.c_str() ));
      }

      // output is the verified CMS content
      if (payloadBio == 0) {
         ETG_TRACE_ERR(("Can't get content from CMS."));
      }
      else {
         char* payloadStr = 0;
         long len = BIO_get_mem_data(payloadBio, &payloadStr); // gen4rcar: conversion to 'int' from 'long int' may alter its value
         outStr.assign(payloadStr, len);
      }

   } while (false);

   // cleanup:
   if(certSubject)    X509_NAME_free(certSubject);
   if(diagnosisBio)   BIO_free(diagnosisBio);
   //if(storeCtx)       X509_STORE_CTX_free(storeCtx);   coverity fix for 65970
   if(trustedStore) X509_STORE_free(trustedStore);
   if(untrustedCerts) sk_X509_pop_free(untrustedCerts, X509_free);
   // if(cms)          CMS_ContentInfo_free(cms);
   if(payloadBio)     BIO_free(payloadBio);
   if(msgBio)         BIO_free(msgBio);

   return isValidated;
}

bool CMS::isMIME(const ::std::string& msgStr) const
{
   // find first non-blank
   size_t begin = msgStr.find_first_not_of(" \t\n\r");
   if (begin == string::npos) {
      return false;
   }
   return ( 0 == msgStr.compare(begin, Constants::Cert::MIME_STRING.length(),
            Constants::Cert::MIME_STRING));
}

::std::string CMS::message () const
{
   return _message;
}

bool CMS::extractXmlNameAndDigest(string &name, string &digest, swu::tenDigestType& type)
{
   // look for <cms> => <file> => <name> until a file with name *.xml is found
   TiXmlElement* elFile = _xmlDoc.RootElement()->FirstChildElement(XML_TAG_FILE);
   while (elFile) {
      string nameTmp;
      if (elFile->FirstChildElement(XML_TAG_NAME)) {
         nameTmp = elFile->FirstChildElement(XML_TAG_NAME)->GetText();
      }

      // if name is empty or doesn't end with "xml", get the next element
      if ( ! hasSuffix(nameTmp, XML_SUFFIX) ) {
         elFile = elFile->NextSiblingElement(XML_TAG_FILE);
         continue;
      }

      if ( elFile->FirstChildElement(XML_TAG_PATH) ) {
         name += elFile->FirstChildElement(XML_TAG_PATH)->GetText();
         name += '/';
      }
      name += nameTmp;

      if ( ! elFile->FirstChildElement(XML_TAG_DIGEST) ) {
         ETG_TRACE_ERR(("Found XML file without digest in CMS. Skipping this."));
         elFile = elFile->NextSiblingElement(XML_TAG_FILE);
         continue;
      }
      else {
         digest = elFile->FirstChildElement(XML_TAG_DIGEST)->GetText();
      }

      if ( ! elFile->FirstChildElement(XML_TAG_DIGEST_TYPE) ) {
         ETG_TRACE_ERR(("Found XML file without digest type in CMS. Assuming SHA256"));
         type = tenDigestTypeSHA256; 
      }
      else {
         std::string typeStr = elFile->FirstChildElement(XML_TAG_DIGEST_TYPE)->GetText();
         std::transform(typeStr.begin(), typeStr.end(), typeStr.begin(), ::tolower);
         if      (typeStr == "sha256") { type = tenDigestTypeSHA256; }
         else if (typeStr == "md5")    { type = tenDigestTypeMD5;    }
         else if (typeStr == "sha1")   { type = tenDigestTypeSHA1;   }
         else {
            ETG_TRACE_ERR(("Unknown digest type %s", typeStr.c_str() ));
            return false;
         }
      }

      return true;
   }
   ETG_TRACE_ERR(("Found no valid XML file in CMS file."));
   return false;
}

bool CMS::extractEncryptParam(string& encryptParam, const string& uin, bool allowEmptyCert)
{
   // root tag <cms>
   TiXmlElement* elCms = _xmlDoc.RootElement();

   // XML path for license-file and -path is 
   // <cms> -> <meta> -> <licensefile> and
   // <cms> -> <meta> -> <licensepath>
   TiXmlElement* elMeta = elCms->FirstChildElement(XML_TAG_META);
   if (! elMeta) {
      ETG_TRACE_ERR(("%s: No XML tag <meta> found", __func__));
      return false;
   }

   TiXmlElement* elLicFile = elMeta->FirstChildElement(XML_TAG_LIC_FILE);
   if (! elLicFile) {
      ETG_TRACE_ERR(("%s: No XML tag <licensefile> found", __func__));
      return false;
   }

   // read the filename and path
   string filePath;
   TiXmlElement* elLicPath = elMeta->FirstChildElement(XML_TAG_LIC_PATH);
   if (elLicPath) {
      string path(elLicPath->GetText());
      if ( ! isAbsolutePath(path) ) {
         filePath = _path + '/';
      }
      filePath += path + '/';
   }
   else {
      ETG_TRACE_USR3(("%s: No XML tag <licensepath> found", __func__));
   }

   filePath += elLicFile->GetText();

   // replace placeholder $(uin) in filePath by actual target uin
   size_t start_pos = filePath.find(TARGET_UIN_PROXY);
   if(start_pos == string::npos) {
      ETG_TRACE_ERR(("Can't insert target UIN in license path. %s", filePath.c_str()));
      return false;
   }
   filePath.replace(start_pos, TARGET_UIN_PROXY_LEN, uin);

   string tmpEncParam;
   ETG_TRACE_USR3(("Opening license file %s", filePath.c_str()));
   if ( ! loadFile(filePath, tmpEncParam) ) {
      ETG_TRACE_ERR(("Can't open license file."));
      return false;
   }

   if (!isMIME(tmpEncParam)) {
      ETG_TRACE_USR1(( "Encryption parameters are not signed." ));
      return false;
   }

   if (!validate(encryptParam, tmpEncParam, allowEmptyCert)) {
      ETG_TRACE_ERR(( "Can't validate encryption parameters!" ));
      return false;
   }

   return true;
}

bool CMS::isUpdateEncrypted ()
{
   // root tag <cms>
   TiXmlElement* elCms = _xmlDoc.RootElement();

   // XML path for license-file and -path is 
   // <cms> -> <meta> -> <licensefile> and
   // <cms> -> <meta> -> <licensepath>
   TiXmlElement* elMeta = elCms->FirstChildElement(XML_TAG_META);
   if (! elMeta) {
      ETG_TRACE_ERR(("%s: No XML tag <meta> found", __func__));
      return false;
   }

   TiXmlElement* elLicFile = elMeta->FirstChildElement(XML_TAG_LIC_FILE);
   if (! elLicFile) {
      ETG_TRACE_ERR(("%s: No XML tag <licensefile> found", __func__));
      return false;
   }
   return true;
}

void CMS::setLastAuthenticTime(const ::time_t& lastAuthenticTime)
{ _lastAuthenticTime = lastAuthenticTime; }

::time_t CMS::getLastAuthenticTime() const
{ return _lastAuthenticTime; }

}
