#ifndef SWU_CA_CERT_H
#define SWU_CA_CERT_H
#include "util/swu_types.h"
#include "util/swu_trace.h"
#include "util/swu_crypto.hpp"
#include "util/swu_singleton.hpp"

namespace swu {

/** @brief Interface for a CA (Certificate Authority) certificate.
 * This has the purpose to verify other certificates to build a chain of trust.
 *
 * The CA certificate is stored on the target. There are different types of 
 * storage, which is reflected through sub-classes. For each type there is
 * another sub-class, e.g. NOR memory, SDC encrypted file, plain file.
 *
 * A CA certificate can only be used after a successful call of 
 * CACertIf::load().
 */
class CACertIf
{

public:
   CACertIf () : _cert(0), _isEmpty(true), _loaded(false) {}
   virtual ~CACertIf ();

   /** @brief Return a string containing the certificate in PEM format. */
   virtual std::string getPem ();

   /** @brief Direct access to the underlying data structure. */
   virtual X509* getX509 () { return _cert; }
   
   /** @brief Create an X509 data structure from the given data in PEM or
    *         DER format and store it.
    *
    *  @param certData  Data in PEM or DER format.
    *  @return  True, if the certificate was successfully stored, otherwise
    *           false.
    */
   virtual bool readPemOrDer (const std::vector< SWU_BYTE >& certData);
   /** @brief Verify, if the certificate validates against its own public key.
    *         This will be true, if the certificate is self-signed.
    */
   virtual bool verifyCertificate ();

   virtual bool isEmpty () const { return _isEmpty; }
   virtual bool isLoaded () const { return _loaded; }

   virtual bool load ()=0;

protected:
   X509* _cert; // cert storage. Has to be filled by derived classes.
   bool _isEmpty;
   bool _loaded;
};

/** @class NORCaCert
 * @brief Source of the CA certificate is NOR memory.
 */
class NORCaCert : public CACertIf, public Singleton<NORCaCert>
{
   friend class Singleton< NORCaCert >;
   NORCaCert();

public:
   virtual ~NORCaCert() {}
   
   virtual bool load ();
private:
   /** @brief Helper method - Checks the provided DER certificate and stores
    *  it in the internal X509 data structure.
    *
    * If the data contains only zero-values, isEmpty() will return true afterwards.
    *
    * @param data       The certificate in binary DER format.
    * @return True, if the certificate is valid, non-zero and set.
    */
   bool setCertificate(const std::vector<SWU_BYTE>& data);
};

/** @class SdcCaCert
 * @brief Source of the CA certificate is a Secure Data Container.
 */
class SdcCaCert : public CACertIf, public Singleton<SdcCaCert>
{
   friend class Singleton< SdcCaCert >;
   SdcCaCert ();

public:
   virtual ~SdcCaCert () {}

   /** @brief Checks the provided PEM certificate and loads it to the internal 
    *         X509 data structure.
	
    * If the data is zero-length, isEmpty() will return true afterwards.
    *
    * @return True, if the certificate is valid, non-zero and set.
    */
   virtual bool load ();
};

/** @class FileCaCert
 * @brief Source of the CA certificate is a plain file.
 */
class FileCaCert : public CACertIf, public Singleton<FileCaCert>
{
   friend class Singleton< FileCaCert >;
   FileCaCert ();

public:
   virtual ~FileCaCert () {}

private:
   /** @brief Checks the provided PEM certificate and loads it to the internal 
    *         X509 data structure.
	*
    * The member _isEmpty will be set to true, if the data is zero-length. 
    * I.e., isEmpty() will return true then.
    *
    * @return True, if the certificate is valid, non-zero and set.
    */
   virtual bool load ();
};
}
#endif
