#include <cstdlib>
#include <cstring>
#include <fstream>

#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/mman.h>

#include <openssl/pem.h>

#include "util/swu_caCert.h"
#include "util/swu_certificate.h"
#include "util/swu_constants.hpp"
#include "util/swu_filesystem.h"
#include "util/swu_sourceNor.hpp"
#include "util/swu_sourceSdc.hpp"

#include "util/swu_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_UTIL
#include "trcGenProj/Header/swu_caCert.cpp.trc.h"
#endif 

namespace swu {

typedef swu::Constants::Mmc MMC;
typedef swu::Constants::SDC SDC;

// ------------------------------------------------------------- class CaCertIf

CACertIf:: ~CACertIf ()
{
   if (_cert != 0) { X509_free(_cert); _cert = 0;}
}

bool CACertIf::readPemOrDer (const std::vector< SWU_BYTE >& certData)
{
   if (isPEM(certData)) {
      // we explicitly cast const away here, since BIO_new_mem_buf accepts 
      // no const void* as parameter, although it treats it as read-only
      // memory. This will change with OpenSSL 1.0.2g
      SWU_BYTE *data = const_cast<SWU_BYTE*>( &(certData[0]) );
      BIO *certBio = BIO_new_mem_buf(data, -1);
      if (! certBio) {
         ETG_TRACE_ERR(( "Cannot create certificate BIO." ));
         return false;
      }
      _cert = PEM_read_bio_X509(certBio, 0, 0, 0);
      BIO_free(certBio);
      if (! _cert ) {
         ETG_TRACE_ERR(("Error when parsing PEM certificate."));
         return false;
      }
   }
   else {
      const SWU_BYTE *data = &(certData[0]);
      _cert = d2i_X509(0, &data, certData.size());
      if ( ! _cert ) {
         ETG_TRACE_ERR(("Error when parsing DER certificate."));
         return false;
      }
   }
   return true;
}

bool CACertIf::verifyCertificate ()
{
   OpenSSL_add_all_algorithms();

   EVP_PKEY *pkey = X509_get_pubkey(_cert);
   if (pkey == 0) {
      ETG_TRACE_ERR(( "Could not read public key of CA cert." ));
      return false;
   }
   if (X509_verify(_cert, pkey) != 1) { // 0 check failed, -1 could not check
      ETG_TRACE_ERR(( "Could not verify CA certificate." ));
      X509_free(_cert);
      EVP_PKEY_free(pkey);
      _cert = 0;
      return false;
   }

   EVP_PKEY_free(pkey);
   EVP_cleanup();
   return true;
}

std::string CACertIf::getPem () 
{
   std::string pem;
   if (! load()) { // directly return if certificate is empty
      ETG_TRACE_USR3(( "Returning empty PEM certificate."));
      return pem;
   }

   BIO*  pemBio = 0;

   do { // emulate goto cleanup;
      pemBio = BIO_new(BIO_s_mem());
      if (! pemBio) {
         ETG_TRACE_ERR(( "Can not allocate PEM BIO." ));
         break;
      }
      // write certificate structure to PEM bio. AUX is used, since it respects
      // trust settings on the certificate (important for CA certs)
      if (! PEM_write_bio_X509_AUX(pemBio, _cert)) {
         ETG_TRACE_ERR(( "Can not write certificate to PEM BIO." ));
         break;
      }

      char* pemBuf = 0;
      long length = BIO_get_mem_data(pemBio, &pemBuf);   // gen4rcar: conversion to 'int' from 'long int' may alter its value
      if (! length) {
         ETG_TRACE_ERR(( "Can not read PEM BIO." ));
         break;
      }
      pem.assign(pemBuf, (unsigned int)length);
   } while (false);

   // cleanup:
   if (pemBio) BIO_free(pemBio);

   return pem;
}

// ------------------------------------------------------------ class NORCaCert
NORCaCert::NORCaCert()
{ }

bool NORCaCert::load()
{
   if (_loaded) { return true; }

   SourceNOR nor;
   std::vector<SWU_BYTE> data;

   ETG_TRACE_USR3(( "Trying to read CA certificate from first memory location." ));
   bool readSuccessfully = nor.read( data, MMC::MTD_CERTIFICATE1_OFFSET, MMC::MTD_CERTIFICATE_BUFFER_SIZE);
   if (readSuccessfully && setCertificate(data)) {
      return true;
   }

   ETG_TRACE_USR3(( "Trying to read CA certificate from second memory location." ));
   readSuccessfully = nor.read( data, MMC::MTD_CERTIFICATE2_OFFSET, MMC::MTD_CERTIFICATE_BUFFER_SIZE);
   if (readSuccessfully && setCertificate(data)) {
      return true;
   }

   return false;
}

bool NORCaCert::setCertificate(const std::vector<SWU_BYTE>& data)
{
   _isEmpty = false;
   // check if data contains only erased bytes
   std::vector<SWU_BYTE>::const_iterator it = data.begin();
   for ( ; it != data.end() && *it == MMC::MTD_ERASED_MASK; ++it ) {}
   if(it == data.end()) {
      ETG_TRACE_USR3(("Certificate is erased (all bytes == 0xFF) set _isEmpty")); 
      ETG_TRACE_USR3(("Length of data: %i", data.size())); 
      _isEmpty = true;
      return false;
   }

#if 0
   unsigned int bytesLeft=data.size();
   while (bytesLeft) {
      unsigned int n=(bytesLeft <=16) ? bytesLeft : 16;
      ETG_TRACE_USR4(("NORCaCert::setCertificate(): data: %02x",
                      ETG_LIST_LEN(n), ETG_LIST_PTR_T8(&data[data.size()-bytesLeft]) ));
      ETG_TRACE_USR4(("NORCaCert::setCertificate(): data: %16s",
                      (char *)&data[data.size()-bytesLeft]));
      bytesLeft-=n;
   }
#endif

   // check if data contains only zero-value bytes
   it = data.begin();
   for ( ; it != data.end() && *it == MMC::MTD_ZEROED_MASK; ++it ) {}
   if(it == data.end()) {
      ETG_TRACE_USR3(("Certificate is zeroed (all bytes == 0x00)  set _isEmpty"));
      ETG_TRACE_USR3(("Length of data: %i", data.size())); 
      // TODO: check if a zeroed certificate should be valid.
      _isEmpty = true;
      return false;
   }

   _loaded = ( readPemOrDer(data) && verifyCertificate() );
   return _loaded;
}

// ------------------------------------------------------------------ SdcCaCert
SdcCaCert::SdcCaCert()
{ }

bool SdcCaCert::load()
{
   if (_loaded) {
      ETG_TRACE_USR3(("SDC CaCert already loaded."));
      return true;
   }

   SourceSdc sdc;
   std::vector<SWU_BYTE> data;
   char const* cacert = SDC::PATH_CACERT;

   _isEmpty = false;

   if ( ! swu::exists(cacert) ) {
      ETG_TRACE_USR3(("No CA certificate installed in path %s", cacert));
      cacert = SDC::PSA_PATH_CACERT;
      ETG_TRACE_USR3(("Trying alternative path %s", cacert));
      if ( ! swu::exists(cacert) ) {
         ETG_TRACE_USR3(("No CA certificate installed."));
         return false;
      }
   }

   if ( ! sdc.read(data, cacert) ) {
      ETG_TRACE_ERR(("Could not read CA certificate from SDC encrypted file %s", cacert));
      return false;
   }

   if ( data.size() == 0) {
      _isEmpty = true;
      return false;
   }
   
   _loaded = ( readPemOrDer(data) && verifyCertificate() );
   return _loaded;
}

// ----------------------------------------------------------------- FileCaCert
FileCaCert::FileCaCert()
{ }

bool FileCaCert::load()
{
   if (_loaded) {
      return true;
   }

   std::vector<SWU_BYTE> data;

   _isEmpty = false;

   if ( ! swu::exists(SDC::PATH_CACERT) ) {
      ETG_TRACE_USR3(("No CA certificate installed in path %s", SDC::PATH_CACERT));
      return false;
   }

   if ( ! swu::loadFile(SDC::PATH_CACERT, data) ) {
      ETG_TRACE_ERR(("Could not read CA certificate"));
      return false;
   }

   if ( data.size() == 0) {
      _isEmpty = true;
      return false;
   }

   _loaded = ( readPemOrDer(data) && verifyCertificate() );
   return _loaded;
}

}
