/*******************************************************************************************
 * FILE:			 TftpClient.cpp
 *
 * SW-COMPONENT: Software Download
 *
 * DESCRIPTION:  A TFTP client that will request a connections from the server and transefet files.
 *					 Here we are using the sendto and recvfrom functions so the server and client   
 *				    can exchange data.
 *
 * AUTHOR:		 Aditya Kumar Jha
 *
 * COPYRIGHT:	 (c) 2013 – 2014 Robert Bosch Engineering and Business Solutions Ltd, Bangalore.
 *
 *********************************************************************************************
 * HISTORY:
 *					29.08.2013	Aditya Kumar Jha
 *					Initial Version.
 *
 ********************************************************************************************/

#include <stdlib.h>
#include <unistd.h>
#include <errno.h> 		
#include <cstdio>
#include <string.h>		
#include "tftpClient.h"
#include "base/imp/swupd_trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_TFTP_CLIENT
#include "trcGenProj/Header/tftpClient.cpp.trc.h"
#endif

tU32 _bytesWritten=0;

//struct sockaddr_in_server;

//#define PRINT_DEBUG(...) if (_fpLog) {fprintf (_fpLog, __VA_ARGS__);}


TftpClient::TftpClient(string serverIpAddress, tU32 maxPktSize):
   _serverIpAddress(serverIpAddress),
   _maxPktSize(maxPktSize),
   _sock(-1),
   _u32BlockNumber(0),
   _packetBuf(0),
   _bOpen(false),
   _logFile(TFTP_LOG_FILE),
   _fpLog(0),
   _tid(0)
{
   _server = sockaddr_in();
   _txBuf[0] = 0;   
   ETG_TRACE_USR4(("TftpClient:CTOR"));
}

TftpClient::~TftpClient()
{
   ETG_TRACE_USR4(("TftpClient:DTOR START"));
   close();
   _serverIpAddress="";
   _maxPktSize=0;
   ETG_TRACE_USR4(("TftpClient:DTOR END"));
}


/*******************************************************************************************
 * FUNCTION:		requestFile
 * DESCRIPTION:	This function is to create a TFTP request packet.
 ********************************************************************************************/ 
bool TftpClient::requestFile()
{
   ETG_TRACE_USR4(("TftpClient:requestFile START _sock=%d", _sock));
   tInt iLen = 0;
   tU32 u32Size = 0;
   _txBuf[0]=0;
   iLen = sprintf(_txBuf, "%c%c%s%c%s%c%s%c%d%c%s%c%d%c", 0x00,
                  TFTP_RRQ, _rcvFileName.c_str(), 0x00,
                  "octet", 0x00, "blksize",
                  0x00, _maxPktSize - TFTP_HEADER_SIZE, 0x00,
                  "tsize", 0x00, u32Size, 0x00);

   if (!iLen) {
      ETG_TRACE_USR1(("TftpClient:requestFile:err iLen=0"));
      return false;
   }

   if(sendto(_sock, _txBuf, iLen, 0, (struct sockaddr *) &_server, sizeof(_server)) != iLen) {
      ETG_TRACE_USR1(("TftpClient:requestFile: err sendto"));
      return false;
   }
   ETG_TRACE_USR1(("TftpClient:requestFile: END iLen=%u", iLen));

   return true;
}

/*******************************************************************************************
 * FUNCTION:		iSendAck
 * DESCRIPTION:	This function is to create a TFTP acknowledgement packet.
 ********************************************************************************************/
tInt TftpClient::iSendAck()
{
   ETG_TRACE_USR4(("TftpClient:iSendAck: START"));
   tInt iLen = 0;
   _txBuf[0]=0;
   iLen = sprintf(_txBuf, "%c%c%c%c", 0x00, TFTP_ACK, 0x00, 0x00);
   _txBuf[2] =static_cast<char> ( (_u32BlockNumber & 0xFF00) >> 8 );
   _txBuf[3] = (_u32BlockNumber & 0x00FF);
	
   if(sendto(_sock, _txBuf, iLen, 0, (struct sockaddr *) &_server,
             sizeof(_server)) != iLen) {
      ETG_TRACE_USR1(("TftpClient:iSendAck: err: sendto"));
      return TFTP_ERROR;
   }	
   ETG_TRACE_USR1(("TftpClient:iSendAck: END iLen=%u _u32BlockNumber=%u", iLen, _u32BlockNumber));
   return iLen;
}

/*******************************************************************************************
 * FUNCTION:		iSendError
 * DESCRIPTION:	This function is to create a TFTP error packet.
 *					Initial Version.
 ********************************************************************************************/
tInt TftpClient::iSendError(tU32 u32ErrCode, tCString pchErrMsg)
{
   ETG_TRACE_USR4(("TftpClient:iSendError: START"));
   tInt iLen = 0;
   _txBuf[0]=0;
   iLen = sprintf(_txBuf, "%c%c%c%c%s%c", 0x00, TFTP_ERR, 0x00, u32ErrCode, pchErrMsg, 0x00);
	
   if(sendto(_sock, _txBuf, iLen, 0, (struct sockaddr *) &_server, sizeof(_server)) != iLen) {
      ETG_TRACE_USR1(("TftpClient:iSendError: err sendto"));
      return TFTP_ERROR;
   }
   ETG_TRACE_USR3(("TftpClient:iSendError: END iLen=%u", iLen));
   return iLen;
}


void TftpClient::close() {
   ETG_TRACE_USR3(("TftpClient:close: START _bOpen=%u", _bOpen));
   _tid=0;
   if (_sock >=0 ) {
      //close the scoket
      ::close(_sock);
      _sock=-1;
   }
   
   if (_fpLog) {
      ::fclose(_fpLog);
      _fpLog=0;
   }

   if (_packetBuf) {
      delete[] _packetBuf;
      _packetBuf=0;
   }

   fflush(stdout);
   ::sync();
   _bOpen=false;
   ETG_TRACE_USR3(("TftpClient:close: END"));
}


bool TftpClient::open() {
   ETG_TRACE_USR3(("TftpClient:open: START _bOpen=%u", _bOpen));
   bool allOk=true;
   if (_bOpen) {
      close();
   }

   _fpLog = fopen(_logFile.c_str(), "w");
   if(!_fpLog) {
      ETG_TRACE_USR1(("TftpClient:open: err  fopen"));
      allOk=false;
   }
   //coverity fix for 51441
   try{
       _packetBuf= new tU8[_maxPktSize + 1];
   }
   catch (std::exception& e)
   {
	ETG_TRACE_ERR(("bad_alloc caught in  _packetBuf for the new operator in tftpClient file\n"));
   }
   if (!allOk) {
      close();
   }
   _bOpen=true;
   return _bOpen;
}


bool TftpClient::getFile(string rcvFileName) {
   ETG_TRACE_USR1(("TftpClient:getFile: START rcvFileName=%s", rcvFileName.c_str()));
   tU32 u32TryCounter=0;
   close();
   _rcvFileName=rcvFileName;
   bool success=false;
   while (u32TryCounter < TFTP_MAX_FILE_RETRIES) {
      ETG_TRACE_USR3(("TftpClient:getFile: u32TryCounter=%u", u32TryCounter));
      u32TryCounter++;
      if (!open()) {
         continue;
      }
      if (!openConnection()) {
         ETG_TRACE_USR1(("TftpClient:getFile: err openConnection"));
         continue;
      }

      if (!requestFile()) {
         ETG_TRACE_USR1(("TftpClient:getFile: err requestFile"));
         continue;         
      }
      if (!receiveFile()) {
         ETG_TRACE_USR1(("TftpClient:getFile: err receiveFile"));
         continue;
      }
      success=true;         
      break;   
   }

   _rcvFileName="";
   ETG_TRACE_USR3(("TftpClient:getFile: END success=%u _bytesWritten=%u", success, _bytesWritten));

   return success;
}


bool TftpClient::sendErrorMessage(string Message)
{
   ETG_TRACE_USR1(("TftpClient:sendErrorMessage: START Message=%s", Message.c_str()));
   tU32 u32TryCounter=0;
   close();
   bool success=false;
   while (u32TryCounter < TFTP_MAX_FILE_RETRIES) {
      ETG_TRACE_USR3(("TftpClient:sendErrorMessage: u32TryCounter=%u", u32TryCounter));
      u32TryCounter++;
      if (!open()) {
         continue;
      }
      if (!openConnection()) {
         ETG_TRACE_USR1(("TftpClient:sendErrorMessage: err openConnection"));
         continue;
      }

      if(iSendError(TFTP_ERR, Message.c_str())==TFTP_ERROR)
      {
         ETG_TRACE_USR1(("TftpClient:sendErrorMessage: err iSendError"));
         continue;
      }
      success=true;
      break;
   }

   ETG_TRACE_USR3(("TftpClient:sendErrorMessage: END success=%u _bytesWritten=%u", success, _bytesWritten));

   return success;
}


bool TftpClient::openConnection() {
   ETG_TRACE_USR3(("TftpClient:openConnection: START"));
   //create a socket connection
   _sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
   if (_sock < 0)
   {
      ETG_TRACE_USR1(("TftpClient:openConnection: END failed (socket)!!!"));
      return false;
   }
   
   //set the address values for the server 
   memset (&_server, 0, sizeof(_server));
   
   //address family for TCP and UDP
   _server.sin_family = AF_INET;
   _server.sin_port = htons(_tftpPort);
   tChar m_cInetAddress[20];
   memset(m_cInetAddress, 0, sizeof(m_cInetAddress));
   strncpy(m_cInetAddress, _serverIpAddress.c_str(), sizeof(m_cInetAddress) - 1);    //coverity fix for 17360
   _server.sin_addr.s_addr = inet_addr(m_cInetAddress);
   ETG_TRACE_USR3(("TftpClient:openConnection: END success!!!"));

   return true;
}
   
   

/*******************************************************************************************
 * FUNCTION:		receiveFile
 * DESCRIPTION:	This function is to receives a file over TFTP connection.
 ********************************************************************************************/
bool TftpClient::receiveFile()
{
   tU8 u8Opcode=0;
   tU32 u32TryCounter=0;
   tInt rcvPktSize;
   tU32 u32RBlockNumber=0;

   ETG_TRACE_USR1(("TftpClient:receiveFile:Receiving %s file",_rcvFileName.c_str()));

   rcvPktSize = _maxPktSize;
   tU32 rcvDataSize=0;
   while (rcvPktSize == _maxPktSize)
   {
      ETG_TRACE_USR4(("TftpClient:receiveFile: new round"));

      /* loop for maximum retries */
      for(u32TryCounter = 0; u32TryCounter < TFTP_MAX_PKT_RETRIES; u32TryCounter++) 
      {
         ETG_TRACE_USR4(("TftpClient:receiveFile:u32TryCounter=%u", u32TryCounter));

         //receive a packet
         rcvPktSize=iRxPacket();
         
         ETG_TRACE_USR3(("TftpClient:receiveFile:rcvPktSize=%u", rcvPktSize));
         if(rcvPktSize == 0) {
            ETG_TRACE_USR1(("TftpClient:receiveFile:got zero bytes"));
            continue;
         }
         rcvDataSize=rcvPktSize-TFTP_HEADER_SIZE;
         

         // check to first byte
         // assign the opcode
         u8Opcode = _packetBuf[1];
         if(_packetBuf[0] != 0x00)  {
            ETG_TRACE_USR1(("TftpClient:receiveFile:err Bad first nullbyte"));
         }
         else if(u8Opcode == TFTP_OACK) 
         {
            _u32BlockNumber = 0; 
            iSendAck();

            ETG_TRACE_USR1(("TftpClient:receiveFile:got OACK"));
            continue;
         }
         else if(u8Opcode == TFTP_ACK) 
         {
            ETG_TRACE_USR1(("TftpClient:receiveFile:err got ACK"));
            continue;
         }
			
         else if(u8Opcode != TFTP_DATA) 
         {
            ETG_TRACE_USR1(("TftpClient:receiveFile: err invalid opcode=0x%x", u8Opcode));
            // sending error message 
            iSendError(0x04, "Illegal TFTP operation");
            return false;
         } 
         else 
         {
            // Data packet
            u32RBlockNumber = _packetBuf[2] << 8;
            u32RBlockNumber &= 0xff00;
            u32RBlockNumber += (_packetBuf[3] & 0x00ff);
            ETG_TRACE_USR3(("TftpClient:receiveFile:got TFTP_DATA rcvDataSize=%u u32RBlockNumber=%u", rcvDataSize, u32RBlockNumber));

            if (u32RBlockNumber==_u32BlockNumber+1) {
               //send the acknoladge packet over socket
               _u32BlockNumber++;
               _bytesWritten+=(rcvDataSize);
               write(STDOUT_FILENO, &_packetBuf[4], rcvDataSize);
            }
            else {
               ETG_TRACE_USR1(("TftpClient:receiveFile:err _u32BlockNumber=%u u32RBlockNumber=%u", _u32BlockNumber, u32RBlockNumber));
            }
            iSendAck();
            break;
         }
      }

      if (u8Opcode == TFTP_OACK) {
         continue;
      }
      if(u32TryCounter == TFTP_MAX_PKT_RETRIES) {
         ETG_TRACE_USR1(("TftpClient:receiveFile: err TFTP_MAX_PKT_RETRIES reached"));
         break;
      }
		
   };

   // Are we at last packet?
   if(rcvPktSize != _maxPktSize) {
      ETG_TRACE_USR3(("TftpClient:receiveFile:got last chunk"));
      return true;
   }
	
   if (u32TryCounter == TFTP_MAX_PKT_RETRIES) {
      ETG_TRACE_USR1(("TftpClient:receiveFile: END TIMEOUT"));
      return false;
   }
   ETG_TRACE_USR1(("TftpClient:receiveFile: END FAILED"));
   return false;	
}



/*******************************************************************************************
 * FUNCTION:		iRxPacket
 * DESCRIPTION:	This function is to receive a packet over socket.
 * PARAMETER:
 *				
 *
 * RETURNVALUE:	tInt: length of the received packet/TFTP_ERROR (-1)
 ********************************************************************************************/
tInt TftpClient::iRxPacket()
{
   ETG_TRACE_USR3(("TftpClient:iRxPacket START"));
   tInt ret = 0;
   tInt tid = 0;
   tInt datalen = 0;
   tInt iRecBytes = -1;
   /*extern int errno;*/
   struct sockaddr_in data = {0};

   datalen = sizeof(data);
   errno = EAGAIN;

   for(tUInt iIndex = 0; errno == EAGAIN && iIndex <= TFTP_TIME_OUT && iRecBytes < 0; iIndex++) {
      ETG_TRACE_USR1(("TftpClient:iRxPacket iIndex=%u _maxPktSize=%u", iIndex, _maxPktSize));

      iRecBytes =static_cast<int> (recvfrom(_sock, _packetBuf,
                           _maxPktSize +10,
                           MSG_DONTWAIT/*MSG_WAITALL*/,
                           (struct sockaddr *) &data,
                           (socklen_t *) &datalen) );
      if (iRecBytes < 0) {
         ETG_TRACE_USR4(("TftpClient:iRxPacket errx iRecBytes=%d", iRecBytes));
         usleep (1000);
      }
   }

   tid = ntohs(data.sin_port);
   ETG_TRACE_USR3(("TftpClient:iRxPacket tid=%u _tid=%u", tid, _tid));
   if (!_tid) {
      _tid=tid;
      _server.sin_port = htons(tid);
   }

   if(iRecBytes < 0 && errno != EAGAIN) 
   {
      ETG_TRACE_USR1(("TftpClient:iRxPacket err Server receive error"));
   } 
   else if(iRecBytes < 0 && errno == EAGAIN) 
   {
      ETG_TRACE_USR1(("TftpClient:iRxPacket err timeout"));
   } 
   else 
   {
      if(_server.sin_addr.s_addr != data.sin_addr.s_addr) 
      {
         ETG_TRACE_USR1(("TftpClient:iRxPacket err addr"));
         return TFTP_ERROR;			
      }
      if(_tid != ntohs(_server.sin_port)) 
      {
         ETG_TRACE_USR1(("TftpClient:iRxPacket err port"));
         iSendError(TFTP_ERR, "Bad/Unknown TID");
         return TFTP_ERROR;			
      }
      ret = iRecBytes;
   }

   if (_packetBuf)
   {
      ETG_TRACE_USR3(("TftpClient:iRxPacket: %02x %02x %02x %02x", 
                      _packetBuf[0], _packetBuf[1], _packetBuf[2], _packetBuf[3]));
   }

   return ret;
}


int main(int argc, char *argv[]) {
   // evaluate command line options
   vInitPlatformEtg(); // initialize the trace backend no cleanup required

   int c;
   extern char *optarg;
   //extern int optind, optopt; //2015-06-11: commented out - lint info 752 - not referenced
   string fileName, errMessage;
   string ipAddr;
   while ((c = getopt(argc, argv, "f:e:i:")) != -1) {
      switch(c) {
         case 'f': // file
            fileName=optarg;
            break;

         case 'e': // error message
            errMessage=optarg;
            break;

         case 'i': // id-address
            ipAddr=optarg;
            break;
         default:
            break;
      }
   }
   ETG_TRACE_USR1(("TftpClient:main START:fileName=%30s ipAddr=%s",
                   fileName.c_str(), ipAddr.c_str()));
   if (!fileName.length() && !errMessage.length()) {
      exit(-1);
   }
   if (!ipAddr.length()) {
      exit(-1);
   }
   TftpClient tftpClient(ipAddr);

   bool success=false;
   if(fileName.length())
      success=tftpClient.getFile(fileName);
   if(errMessage.length())
      success=tftpClient.sendErrorMessage(errMessage);
   ETG_TRACE_USR1(("TftpClient:main END:fileName=%30s ipAddr=%30s success=%u",
                   fileName.c_str(), ipAddr.c_str(), success));
   exit(success ? 0 : 1);
}
