/************************************************************************
 * File: teseo_SWUpdateMgr.cpp
 * SW-Component: 
 * Description: Handles events from teseo DL client and teseo response receiver
 * Author:
 *	Rohit.ChannappiahRaman@in.bosch.com
 * Copyright:
 *   Robert Bosch Engineering and Business Solutions Ltd, Bangalore.
 *
 * History:
 * 14.10.2013 - Initial version - Rohit.ChannappiahRaman@in.bosch.com
 * 27.08.2013 - Implemented CMG3G-5318 optimized logging using update.log - Apoorva K R (RBEI/ECA1)
 * 6/4/2015 - Implemented CMG3G-6728 IS2030_Teseo update only if CRC changed@Integration Tooling - Apoorva K.R (RBEI/ECA1)
 * 2/7/2015 - Fixed CMG3GB-2169 - Segmentation fault when calculating CRC for teseo firmware - Apoorva K.R (RBEI/ECA1)
 *					CMG3GB-2167 - Exit codes for CRC calculation are treated as error codes while writing to errmem
*---------------------------------------------------------------------------------------------------
* 1/2/2016          | AGL5KOR (RBEI)    |     CMG3G-9450 - teseo update ctrl app to be modified to write and read CRC
* 				    | Anusha Ghali	    | 
*---------------------------------------------------------------------------------------------------
 ***********************************************************************/

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include "ai_sw_update/common/base/imp/swupd_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  #define ETG_DEFAULT_TRACE_CLASS 		TR_CLASS_SWUPDATE_TESEO
  #include "trcGenProj/Header/teseo_SWUpdateMgr.cpp.trc.h"
#endif


#include "teseo_SWUpdateMgr.h"

OSAL_tEventHandle teseo_SWUpdateMgr::m_evHandleMgr=0;
OSAL_tEventHandle teseo_SWUpdateMgr::m_evHandleSWUpdater=0;
tU8 teseo_SWUpdateMgr::m_evMgrToSWUpdater = EV_INVALID;
tU8 teseo_SWUpdateMgr::m_evSWUpdaterToMgr = EV_TESEO_INVALID;
tU8 teseo_SWUpdateMgr::m_evSWUpdateClientToMgr = EV_INVALID;
tU8 teseo_SWUpdateMgr::m_SWUpdateState = TESEO_SWUPDATE_NOT_STARTED;
tU8 teseo_SWUpdateMgr::m_SWUpdateStatus = STATUS_SWUPDATE_NOT_STARTED;
	
/************************************************************************
* FUNCTION: teseo_SWUpdateMgr()
*
* DESCRIPTION: Constructor
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
teseo_SWUpdateMgr::teseo_SWUpdateMgr()
{
	pthread_mutex_init(&m_swUpdateMutex, NULL);
	m_fptrClientCb = NULL;
	m_strFlasherBinPath="";
	m_strTeseoFWBinPath="";
}


/************************************************************************
* FUNCTION: ~teseo_SWUpdateMgr()
*
* DESCRIPTION: Destructor
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
teseo_SWUpdateMgr::~teseo_SWUpdateMgr()
{
	pthread_mutex_destroy(&m_swUpdateMutex);
	//close the OSAL events
	if(teseo_SWUpdateMgr::m_evHandleMgr)
	{
		OSAL_s32EventClose(teseo_SWUpdateMgr::m_evHandleMgr);
		OSAL_s32EventDelete(TESEO_SWUPDATE_MGR);
	}
	if(teseo_SWUpdateMgr::m_evHandleSWUpdater)
	{
		OSAL_s32EventClose(teseo_SWUpdateMgr::m_evHandleSWUpdater);
		OSAL_s32EventDelete(TESEO_SW_UPDATER);
	}

}


/*******************************************************************************************
* FUNCTION:	bInitUpdateMgr	
* DESCRIPTION:	interface for initializing the update manager 
*					
* PARAMETER: std::string Flasher bin Path
*		  	 std::string TeseoFWBinPath
*			 rTeseoBinImgOptions& rImgOpts	
* RETURNVALUE:	bool
********************************************************************************************/
tBool teseo_SWUpdateMgr::bInitUpdateMgr(std::string& strFlasherPath,
									  std::string& strTeseoFWBinPath)
{
	m_SWUpdateState = TESEO_SWUPDATE_NOT_STARTED;
	UpdateLog::vUpdateLogTrace ("%s Initializing Update manager", DL_LOG_INFO);
	//create event handles for events from DL client and SW updater
	if( OSAL_OK != OSAL_s32EventCreate\
		( TESEO_SWUPDATE_MGR, &teseo_SWUpdateMgr::m_evHandleMgr))
    {
		UpdateLog::vUpdateLogTrace ("%s Failed to create OSAL Event in %s %d", DL_LOG_ERROR, __FILE__, __LINE__);
		DEBUG_TRACE(TRACE_LEVEL_1, "ERROR: failed to create OSAL Event in %s %d",__FILE__,__LINE__);
		ETG_TRACE_USR4 (("ERROR: failed to create OSAL Event"));
		return false;
	}
	
	if( OSAL_OK != OSAL_s32EventCreate\
		( TESEO_SW_UPDATER, &teseo_SWUpdateMgr::m_evHandleSWUpdater))
    {
		UpdateLog::vUpdateLogTrace ("%s Failed to create OSAL Event in %s %d", DL_LOG_ERROR, __FILE__, __LINE__);
		DEBUG_TRACE(TRACE_LEVEL_1, "ERROR: failed to create OSAL Event in %s %d",__FILE__,__LINE__);
		ETG_TRACE_USR4 (("ERROR: failed to create OSAL Event"));
		return false;
	}
	
	//create thread to receive teseo responses
	if(false == m_teseoSWUpdater.bInitSWUpdater\
		(strFlasherPath,strTeseoFWBinPath))
		return false;

	pthread_t thrdUpdateMgrThread;
	if(pthread_create(&thrdUpdateMgrThread, NULL, \
		&teseo_SWUpdateMgr::vUpdateMgrThreadCallback, this))
    {
		UpdateLog::vUpdateLogTrace ("%s failed to create manager thread", DL_LOG_ERROR);
		DEBUG_TRACE(TRACE_LEVEL_1, "ERROR: failed to create manager thread");
		ETG_TRACE_USR4 (("ERROR: failed to create manager thread"));
		return false;
	}
	
	DEBUG_TRACE(TRACE_LEVEL_2, "INFO: Initializing teseo_SWUpdateMgr OK");
	m_strFlasherBinPath=strFlasherPath;
	m_strTeseoFWBinPath=strTeseoFWBinPath;
	return true;
}

/************************************************************************
* FUNCTION: u8InitUpdateMgrForCRC
* DESCRIPTION: Based on the bool values this function:
*				1. Gets CRC of flashed firmware from GNSS
*				2. Gets CRC of the new image to be flashed
*				3. Compares CRC of flashed and new firmware images
* PARAMETER: std::string& strTeseoFWBinPath - Path to new firmware image 
*			 const std::string& strPathToFile - Path to file where CRC has
*												to be written
*			 bool bCRCFromGNSS - whether to get Flashed firmware CRC	
*	     	 bool bCRCFromUpdtCtl - whether to get CRc of new image
* RETURNVALUE: tU8
*************************************************************************/	
tU8 teseo_SWUpdateMgr::u8InitUpdateMgrForCRC (std::string& strTeseoFWBinPath, \
		const std::string& strPathToFile, bool bCRCFromGNSS, \
				bool bCRCFromUpdtCtl, bool bFlasherCrc)
{
	tU8 u8Res = 0;
	
	if (bCRCFromGNSS)
	{
		UpdateLog::vUpdateLogTrace ("%s INFO: Getting CRC of flashed firmware", DL_LOG_INFO);
		DEBUG_TRACE (TRACE_LEVEL_1, "INFO: Getting CRC of flashed firmware");
		DEBUG_TRACE (TRACE_LEVEL_1, "INFO: Path to file where CRC has to be written to: %s", \
				strPathToFile.c_str());
		m_teseoSWUpdater.u8GetCRCFromGNSS (u8Res, strPathToFile);
	}
	else if (bCRCFromUpdtCtl)
	{
		UpdateLog::vUpdateLogTrace ("%s INFO: Getting CRC of new firmware", DL_LOG_INFO);
		DEBUG_TRACE (TRACE_LEVEL_1, "INFO: Getting CRC of new firmware");
		DEBUG_TRACE (TRACE_LEVEL_1, "INFO: Path to file where CRC has to be written to: %s", \
				strPathToFile.c_str());		
		m_teseoSWUpdater.u8CalcCRCForFwUpdate (u8Res, bFlasherCrc, strTeseoFWBinPath, strPathToFile);
	}
	else
	{
		UpdateLog::vUpdateLogTrace ("%s INFO: Comparing CRCs of flashed and new firmware", DL_LOG_INFO);
		DEBUG_TRACE (TRACE_LEVEL_1, "INFO: Comparing CRCs of flashed and new firmware");
		u8Res = m_teseoSWUpdater.u8CompareCRC (strTeseoFWBinPath);
	}

	return u8Res;
}

/*******************************************************************************************
* FUNCTION:		vSWUpdateMgrThreadFunction
* DESCRIPTION:	Worker thread of manager that waits for and process events from teseo DL client
*				and teseo response receiver. 
*					
* PARAMETER: none
*
* RETURNVALUE: void	
********************************************************************************************/
 tVoid teseo_SWUpdateMgr::vSWUpdateMgrThreadFunction()
 {
	while(true)
	{
		OSAL_tEventMask u32ResultMask = 0;
		
		ETG_TRACE_USR4 (("INFO: Manager waiting for events"));
		DEBUG_TRACE(TRACE_LEVEL_2, "INFO: Manager waiting for events");
		UpdateLog::vUpdateLogTrace ("%s Creating worker thread of manager and waiting for events", DL_LOG_INFO);
				
		//wait for events from DL client and response receiver
		if ( OSAL_ERROR == OSAL_s32EventWait\
							( m_evHandleMgr,
							(EV_TESEO_SWUPDATE_MGR | EV_TESEO_SW_UPDATER),
							OSAL_EN_EVENTMASK_OR,
							OSAL_C_TIMEOUT_FOREVER,
							&u32ResultMask ))
		{
			ETG_TRACE_USR4(("teseo_SWUpdateMgr::vSWUpdateMgrThreadFunction OSAL_s32EventWait error"));
			DEBUG_TRACE(TRACE_LEVEL_1, "ERROR: OSAL_s32EventWait failed in  %s %d",__FILE__,__LINE__);  
			continue;
		}
		/* Event wait returned with a sucess, Clear the event */
		if(OSAL_ERROR == OSAL_s32EventPost( m_evHandleMgr,
										   (OSAL_tEventMask) ~u32ResultMask,
										   OSAL_EN_EVENTMASK_AND))
		{
		   ETG_TRACE_USR4 (("teseo_SWUpdateMgr::vSWUpdateMgrThreadFunction OSAL_s32EventPost error"));
		   DEBUG_TRACE(TRACE_LEVEL_1, "ERROR: OSAL_s32EventPost failed in  %s %d",__FILE__,__LINE__);
		}
		
		/* Check event and process */
		if (( u32ResultMask == EV_TESEO_SWUPDATE_MGR ))
		{
			vProcessClientEvent();
		}
		else
		{
			if (( u32ResultMask == EV_TESEO_SW_UPDATER ))
			{
				vProcessSWUpdaterEvent();
			}
		}
		
	}
	
 }
 
 /*******************************************************************************************
* FUNCTION:		vProcessClientEvent
* DESCRIPTION:	process events from client
* PARAMETER: 	none
* RETURNVALUE:	none
********************************************************************************************/
tVoid teseo_SWUpdateMgr::vProcessClientEvent() 
{
	ETG_TRACE_USR4 (("INFO: teseo_SWUpdateMgr::vProcessClientEvent RXed"));
	DEBUG_TRACE(TRACE_LEVEL_2,"teseo_SWUpdateMgr::vProcessClientEvent RXed");

	switch(m_evSWUpdateClientToMgr)
	{
		case EV_TESEO_SWUPDATE_START:
			m_evMgrToSWUpdater = EV_TESEO_SWUPDATE_START;
			UpdateLog::vUpdateLogTrace ("%s Received TESEO_SWUPDATE_START event", DL_LOG_INFO);
			ETG_TRACE_USR4 (("INFO: Received TESEO_SWUPDATE_START event"));
			if(false == bPostEventtoSWUpdater())
			{
				pthread_mutex_unlock\
					( const_cast<pthread_mutex_t *>(&m_swUpdateMutex) );
			}	
			break;
					
		case EV_TESEO_SWUPDATE_ABORT:
			m_evMgrToSWUpdater = EV_TESEO_SWUPDATE_ABORT;
			ETG_TRACE_USR4 (("INFO: Received TESEO_SWUPDATE_ABORT event"));
			UpdateLog::vUpdateLogTrace ("%s Received TESEO_SWUPDATE_ABORT event", DL_LOG_INFO);
			if(false == bPostEventtoSWUpdater())
				m_SWUpdateState = TESEO_SWUPDATE_ABORT_FAILED;
			else
			{	
				//create swupdater thread again to receive teseo responses
				if(false == m_teseoSWUpdater.bInitSWUpdater\
						(m_strFlasherBinPath,m_strTeseoFWBinPath))
					m_SWUpdateState = TESEO_SWUPDATE_ABORT_FAILED;
			}
			break;

		case EV_TESEO_SWUPDATE_EXIT:
			//post event to updater to exit from its thread
			m_evMgrToSWUpdater = EV_TESEO_SWUPDATE_EXIT;
			ETG_TRACE_USR4 (("INFO: Received TESEO_SWUPDATE_EXIT event"));
			UpdateLog::vUpdateLogTrace ("%s Received TESEO_SWUPDATE_EXIT event", DL_LOG_INFO);
			if(bPostEventtoSWUpdater())
			{
				m_SWUpdateState = TESEO_EXIT_SUCCESS;				
			}
			
			pthread_mutex_unlock\
				( const_cast<pthread_mutex_t *>(&m_swUpdateMutex) );
			//exit manager thread
			pthread_exit(0);
			break;
		default:
		break;
	}
}

 /*******************************************************************************************
* FUNCTION:		vProcessSWUpdaterEvent
* DESCRIPTION:	process events from SW updater
* PARAMETER: 	none
* RETURNVALUE:	none
********************************************************************************************/
tVoid teseo_SWUpdateMgr::vProcessSWUpdaterEvent()
{
	switch(teseo_SWUpdateMgr::m_evSWUpdaterToMgr)
	{
		case EV_TESEO_SWUPDATE_STATUS:
		pthread_mutex_unlock( &m_swUpdateMutex);
		break;

		default:
		break;
	}
}

/************************************************************************
* FUNCTION: bPostEvent
*
* DESCRIPTION: post event to update manager
*
* PARAMETER: None
* RETURNVALUE: bool
*************************************************************************/
tBool teseo_SWUpdateMgr::bPostEventtoSWUpdater() const
{
   if(OSAL_ERROR == OSAL_s32EventPost( teseo_SWUpdateMgr::m_evHandleSWUpdater,
									 (OSAL_tEventMask) EV_TESEO_SWUPDATE_MGR,
									 OSAL_EN_EVENTMASK_OR))
   {
	   m_SWUpdateState = TESEO_OSAL_EVENT_ERROR; 	
	   ETG_TRACE_USR4 (("teseo_SWUpdateMgr::bPostEventtoSWUpdater OSAL_s32EventPost error"));
	   DEBUG_TRACE(TRACE_LEVEL_1, "ERROR: OSAL_s32EventPost failed in %s %d",__FILE__,__LINE__);
	   return false;
   }
   return true;
}

