/*******************************************************************************************
* FILE:			 teseoUpdateCtrlApp.cpp
*
* SW-COMPONENT: Software Update
*
* DESCRIPTION:  Standalone application for Teseo update
*				    
*
* AUTHOR:		 Rohit.C.R
*
* COPYRIGHT:	 (c) 2013  2014 Robert Bosch Engineering and Business Solutions Ltd, Bangalore.
*
*********************************************************************************************
* HISTORY:
*					12/4/2013	Rohit.C.R - Initial Version.
*					6/4/2015	Apoorva K.R - Implemented CMG3G-6728
*								IS2030_Teseo update only if CRC changed@Integration Tooling.
* 1/1/2016 - Implemented CMG3G-9339 Teseo-3 support in SWUpdate - Anusha Ghali (RBEI/ECA1)
* 1/2/2016 - Implemented CMG3G-9450 - teseo update ctrl app to be modified to write and read CRC - Anusha Ghali (RBEI/ECA1) 
* 3/1/2016 - Implemented fix for CMG3GB-2996 [CMD] TESEO update / downgrade failed
********************************************************************************************/

#include "teseo_ClientInterface.h"
using namespace std;
#include <iostream>
#include <string>
#include <sstream>
#include <stdlib.h>
#include <unistd.h>
#include <getopt.h>
#include "ErrorHandler.h"
#include "teseo_Global.h"

#include "ai_sw_update/common/base/imp/swupd_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  #define ETG_DEFAULT_TRACE_CLASS 		TR_CLASS_SWUPDATE_TESEO_APP
  #include "trcGenProj/Header/teseoUpdateCtrlApp.cpp.trc.h"
#endif

#define OSAL_EXIT_WITH_CODE(n) 	{								\
									OSAL_vSetProcessExitCode(n);\
									OSAL_vProcessExit();		\
								}
								
T_fCallback CALLBACK;

bool bCRCFromGNSS = false;
bool bCRCFromUpdtCtl = false;
std::string strFlasherBin = "", strTeseoFirmware = "";
std::string strTeseoVersionToGNSS = "teseo_2";
std::string strPathToWriteCRC = "";
bool bWrongArgs = false;
bool bUpdate = false, bCmpCrc = false;
bool bReadCrc = false, bNewCrc = false;
bool bFlasherCrc = false; 
teseo_ClientInterface* pclteseoUpdater = NULL;
int iEXitCode = TESEO_EXIT_FAILED;
extern char *optarg;

void vUpdateTeseoFirmware (void);
void vInitClientIntForCRC (void);
void help (void);
void vCheckArgs (int& c);
void vParseCmdArgs (int& c);
void vCheckForEmpty (void);
void vWriteErrorLogToErrmem(void);
bool bParseCmdArgsExtended (int& c);

// Teseo Error Codes
const char *strenTeseoSWUpdateState[] = {
	"TESEO_SWUPDATE_SUCCESS",
	"TESEO_EXIT_FAILED",
	"TESEO_SWUPDATE_NOT_STARTED",
	"TESEO_CLIENT_INTF_CREATION_OK",
	"TESEO_SWUPDATE_STARTED",
	"TESEO_SWUPDATE_FLASHING_FLASHER_BIN",
	"TESEO_SWUPDATE_FLASHING_FIRMWARE",
	"TESEO_SWUPDATE_ABORTED",
	"TESEO_SWUPDATE_ABORT_FAILED",
	"TESEO_FILE_CLOSE_ERROR",
	"TESEO_OSAL_WRITE_ERROR",
	"TESEO_OSAL_EVENT_ERROR",
	"TESEO_SWUPDATE_IOCTL_ERROR",
	"TESEO_SWUPDATE_CRC_ERROR",
	"TESEO_SWUPDATE_GNSS_OPEN_ERROR",
	"TESEO_SWUPDATE_GNSS_CLOSE_ERROR",
	"TESEO_SWUPDATE_BL_OPEN_ERROR",
	"TESEO_SWUPDATE_FW_OPEN_ERROR",
	"TESEO_SWUPDATE_FW_CRCS_SAME",
	"TESEO_SWUPDATE_FW_CRCS_DIFFERENT",
	"TESEO_SWUPDATE_CRC_CALC_OK",
	"TESEO_SWUPDATE_CRC_CALC_ERROR",
	"TESEO_SWUPDATE_PARSE_CMDARGS_FAILED",
	"TESEO_GNSS_NOTIFY_TYPE_ERROR",
	"INVALID_STATE",
	"TESEO_UPDTCTRL_VERSION"
};

tU32 u32gTeseo_Fw_Chunk_Size_Bytes=TESEO_FW_CHUNK_SIZE_BYTES_16K; // 16KB chunks for teseo_2 - default

int main(int argc, char *argv[])
{
	//opening and registering for TTFis traces
	vInitPlatformEtg();
	ETG_I_REGISTER_CHN();
	
	// To initialize SWUpdate Error Memory
	iInitializeSwUpdateErrorArray();
		
	if (1 < argc) 
	{
		// evaluate command line options
		int indexes = 0;
		int  c;
		
		static struct option long_options[]=
		{
			{"update"         , no_argument,  	   0,  'u'},		//to update teseo fw
			{"cmpcrc"         , no_argument,  	   0,  'c'},		//compare CRC of flashed and new fw
			{"readcrc"        , no_argument,   	   0,  'r'},		//get CRC of flashed fw from GNSS
			{"calccrc"        , no_argument,  	   0,  'n'},		//Calc CRC of new fw image
			{"help"           , no_argument,       0,  'h'},		//display cmd line opts & usage
			{"crcfile"        , required_argument, 0,  'o'},		//file to write CRC into
			{"bin"            , required_argument, 0,  'b'},		//path to flasher.bin
			{"firmware"       , required_argument, 0,  'f'},		//path to teseo_app.bin
			{"chiptype"       , required_argument, 0,  't'},		//to notify teseo version to GNSS
			{"teseoappversion", no_argument,       0,  'v'},		//to get tesep app version
			{0,               0,            0,   0 }
		};	
		
		while ((c = getopt_long_only (argc, argv, "vucrnho:b:f:t:",\
					long_options, &indexes)) != -1) 
				vParseCmdArgs (c);
		
		DEBUG_TRACE (TRACE_LEVEL_1, "INFO: Parsing cmd line args for their correctness");		
		//coverity fix for 51453
		try{
		vCheckForEmpty();
	}
		catch (std::exception& e)
		{
			ETG_TRACE_ERR(("bad_alloc caught when calling vCheckForEmpty function in teseoUpdateCtrlApp file\n"));
		}
	}
	else
	{
		// no command line options supplied, continue with default behaviour
		DEBUG_TRACE (TRACE_LEVEL_1, "INFO: no command line option provided");
		help ();
	}
	
		//closing and unregistering for TTFis traces
	ETG_I_UNREGISTER_CHN ();
	
	vWriteErrorLogToErrmem();
	OSAL_EXIT_WITH_CODE (iEXitCode);	//change exit code
}

void vCheckForEmpty ()
{
	if (bUpdate)			//perform firmware update. Path to bootloader and fw required
	{
		if ((strTeseoFirmware == "") || (strFlasherBin == ""))
		{
			DEBUG_TRACE (TRACE_LEVEL_1, "ERROR: strTeseoFirmware or strFlasherBin path or strTeseoVersionToGNSS or all are missing");
			help();
		}
		
		if ("teseo_3" == strTeseoVersionToGNSS)
		{
			u32gTeseo_Fw_Chunk_Size_Bytes=TESEO_FW_CHUNK_SIZE_BYTES_5K; // 5KB chunks for teseo_3 
		}		
		//coverity fix for 51453
		try{
		vUpdateTeseoFirmware();
		}
		catch (std::exception& e)
		{
			ETG_TRACE_ERR(("bad_alloc caught when calling vUpdateTeseoFirmware function in teseoUpdateCtrlApp file\n"));
		}
	}
	
	else if (bCmpCrc)			//compare crcs of flashe and new fw. Path to new fw required
	{
		if (strTeseoFirmware == "")
		{
			DEBUG_TRACE (TRACE_LEVEL_1, "ERROR: strTeseoFirmware path is missing");
			help();
		}	
		vInitClientIntForCRC();
	}
	
	else if (bReadCrc)		//get CRC of flashed fw
	{
		if (strPathToWriteCRC == "")
		{
			DEBUG_TRACE (TRACE_LEVEL_1, "ERROR: path to write CRC to file is missing");
			help();
		}
		vInitClientIntForCRC();
	}
	
	else if (bNewCrc)				//calculate CRC of new image to be flashed
	{
		if (bFlasherCrc)
		{
			if (strFlasherBin == "")
			{
				DEBUG_TRACE (TRACE_LEVEL_1, "ERROR: strFlasherBin path is missing");
				help();
			}
		}
		else if ((strPathToWriteCRC == "") || (strTeseoFirmware == ""))
		{
			DEBUG_TRACE (TRACE_LEVEL_1, "ERROR: strTeseoFirmware path or path to write CRC to file is missing");
			help();
		}
		
		vInitClientIntForCRC ();
	}
	
	else				//cmd opt other than -u, -c, -n, -r, -h
	{
		DEBUG_TRACE (TRACE_LEVEL_1, "ERROR: Please provide the cmd line option according to required functionality ");
		help();	
	}

}


void vParseCmdArgs (int& c)
{
	bool res = false;
	res = bParseCmdArgsExtended(c); //check extended args first
	
	if ( false == res )
	{
		switch (c)
		{
			case 'u':					//trying to update teseo fw
				bUpdate = true;
				break;
		
		case 't':					// to notify teseo version to GNSS
		    strTeseoVersionToGNSS = optarg;
			vCheckArgs(c);
		break;
		
		case 'c':						//compare CRC of flashed and new fw
			bCmpCrc = true;
		break;
		
		case 'r':							//get CRC of flashed image from GNSS
			bReadCrc = true;
			bCRCFromGNSS = true;
		break;
		
		case 'n':							//get CRC of new image to be flashed
			bNewCrc = true;
			bCRCFromUpdtCtl = true;		
		break;
		
		case 'b':							//load bootloader path
		    bFlasherCrc = true;
			strFlasherBin = optarg;
			vCheckArgs(c);
			break;
			
		case 'f':							//load firmware path
			strTeseoFirmware = optarg;
			vCheckArgs(c);
			break;
		case 'o':							//load file path to write CRC to 
			strPathToWriteCRC = optarg;
			vCheckArgs(c);
			break;
			case ':':		//arguments missing
			case '?':		//unrecognized option
			default:
				iEXitCode = TESEO_SWUPDATE_PARSE_CMDARGS_FAILED;
				help ();
				break;
		}
	}
}

bool bParseCmdArgsExtended (int& c)
{
	switch (c)
	{
		case 'h':		//display help
			iEXitCode = TESEO_EXIT_SUCCESS;
			help();
			break;
		case 'v':       // return teseo app version
		   iEXitCode = TESEO_UPDTCTRL_VERSION;
	       OSAL_EXIT_WITH_CODE (iEXitCode);
		   break;
		default:  
		   break;
	}
	return false;
}

void vCheckArgs (int& c)
{
	if ((c != 'h') && (c != ':') && (c != '?'))				//optarg will not be loaded to for any other cmd opt
	{													
		if ((optarg[0] == '\0') || (optarg[0] == '-'))		//check for empty and order of cmd line opts & arguments
		{
			DEBUG_TRACE (TRACE_LEVEL_1, "INFO: cmd line opt -%c has wrong args", c);
			iEXitCode = TESEO_SWUPDATE_PARSE_CMDARGS_FAILED;		
			help();
		}
	} 
}

void vInitClientIntForCRC ()
{
	pclteseoUpdater = new teseo_ClientInterface ();		//create client interface
	
	if (NULL == pclteseoUpdater)
	{	//client interface creation failed
		DEBUG_TRACE (TRACE_LEVEL_1, "INFO: pclteseoUpdater is null");	
		ETG_TRACE_USR4 (("ERROR: pclteseoUpdater is null"));	
		iEXitCode = TESEO_EXIT_FAILED;		
		vWriteErrorLogToErrmem();
		OSAL_EXIT_WITH_CODE (iEXitCode);

	}
	
	DEBUG_TRACE (TRACE_LEVEL_1, "INFO: Initializing teseo_ClientInterface for CRC");
			
	ETG_TRACE_USR4 (("INFO: Initializing teseo_ClientInterface for CRC"));	
	
	//inititalize client interface
	if (bFlasherCrc)
	{
		iEXitCode = pclteseoUpdater->u8InitClientInterfaceForCRC\
			(strFlasherBin, strPathToWriteCRC, bCRCFromGNSS, \
			 bCRCFromUpdtCtl,bFlasherCrc);
	}
	else
	{
		iEXitCode = pclteseoUpdater->u8InitClientInterfaceForCRC\
			(strTeseoFirmware, strPathToWriteCRC, bCRCFromGNSS, \
			 bCRCFromUpdtCtl,bFlasherCrc);
	}
		
	delete pclteseoUpdater;	
	pclteseoUpdater = NULL;
}

void vUpdateTeseoFirmware()
{
	
	DEBUG_TRACE (TRACE_LEVEL_1, "INFO: teseo flasher.bin path - %s teseo firmware path is - %s",\
					strFlasherBin.c_str(), strTeseoFirmware.c_str());
	//start teseo update
	pclteseoUpdater = new teseo_ClientInterface();	//create client interface
	
	if (NULL == pclteseoUpdater)
	{	//pclteseoUpdater is null
		DEBUG_TRACE (TRACE_LEVEL_1,"INFO: pclteseoUpdater is null");
		ETG_TRACE_USR4 (("ERROR: pclteseoUpdater is null"));	
		
		iEXitCode = TESEO_EXIT_FAILED;	
		vWriteErrorLogToErrmem();
		OSAL_EXIT_WITH_CODE (iEXitCode);
		
	}
	
	//initialize client interface
	iEXitCode = pclteseoUpdater->bInitClientInterface\
									(strFlasherBin, strTeseoFirmware);
	
	if(TESEO_CLIENT_INTF_CREATION_OK != iEXitCode)
	{
		DEBUG_TRACE (TRACE_LEVEL_1, "INFO: problem in creating Client interface");
		ETG_TRACE_USR4 (("ERROR: problem in creating Client interface"));
		//problem in creating Client interface
		delete pclteseoUpdater;	
		pclteseoUpdater = NULL;
	}
	else
	{	
		DEBUG_TRACE (TRACE_LEVEL_1, "INFO: Client interface created! Starting update...");
		
		ETG_TRACE_USR4 (("INFO: Client interface created! Starting update..."));
		
		iEXitCode = pclteseoUpdater->bOnStartSWUpdate (CALLBACK);
		
		if(iEXitCode != TESEO_SWUPDATE_SUCCESS)
		{
			//teseo update failed
			DEBUG_TRACE (TRACE_LEVEL_1, "ERROR: teseo firmware update Failed");
			ETG_TRACE_USR4 (("ERROR: teseo firmware update Failed"));
		}
		else
		{
			//teseo update success
			iEXitCode = TESEO_EXIT_SUCCESS;
			DEBUG_TRACE (TRACE_LEVEL_1, "INFO: teseo firmware update Success!!");
			ETG_TRACE_USR4 (("INFO: teseo firmware update Success!!"));
		}

	
		if(false == pclteseoUpdater->bExitClientInterface())
		{
			//teseo firmware upadte failed.
			iEXitCode = TESEO_EXIT_FAILED;
			DEBUG_TRACE (TRACE_LEVEL_2, "ERROR: Exit Client Interface failed!!");
			ETG_TRACE_USR4 (("ERROR: Exit Client Interface failed!!"));
		}	

		delete pclteseoUpdater;	
		pclteseoUpdater = NULL;
	}		
}

void help ( void)
{
	//print the usage of teseo update control
	cout <<"\n \
		Options supported:\n \
			1. -update (-u)\n \
			2. -cmpcrc (-c)\n \
			3. -readcrc (-r)\n \
			4. -calccrc (-n)\n \
			5. -help (-h)\n \
			6. teseoappversion (-v) \n\
		\n \
		Usage:\n \
			1. For teseo Update:\n \
				./swu_common_teseo_app_out.out -u -b <Absolute path of flasher.bin> -f <Absolute path of teseo firmware binary> -t <teseo chip type present on the board (teseo_2/teseo_3)>\n \
			2. Compare CRC of flashed and new teseo firmware images:\n \
				./swu_common_teseo_app_out.out -c -f <Absolute path of teseo firmware binary>\n \
			3. Get CRC of flashed teseo firmware from GNSS:\n \
				./swu_common_teseo_app_out.out -r -o <path to file>\n \
			4. Get CRC of new firmware image to be flashed:\n \
				./swu_common_teseo_app_out.out -n -o <path to file> -f <Absolute path of teseo firmware binary>\n \
			5. Get CRC of new flasher/bootloader image to be flashed:\n \
				./swu_common_teseo_app_out.out -n -o <path to file> -b <Absolute path of teseo flasher/bootloader binary>\n \
			6. Display options & usage:\n \
				./swu_common_teseo_app_out.out -h\n \
			7. Returns teseo update ctrl app version:\n\
			    ./swu_common_teseo_app_out.out -v\n";

	vWriteErrorLogToErrmem();
	OSAL_EXIT_WITH_CODE (iEXitCode);

}

void vWriteErrorLogToErrmem(void)
{
	// to capture teseo error exit code in Errmem
	if ( (TESEO_SWUPDATE_SUCCESS != iEXitCode) && \
			(TESEO_SWUPDATE_FW_CRCS_SAME != iEXitCode) &&\
				(TESEO_SWUPDATE_FW_CRCS_DIFFERENT != iEXitCode) && \
				  (TESEO_UPDTCTRL_VERSION != iEXitCode) && \
					(TESEO_SWUPDATE_CRC_CALC_OK != iEXitCode))
	{
		iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_TESEO,
                        (int)SW_UPDATE_ERROR_TYPE_VERIFY,
                        "Teseo EXIT Code:",
                        strenTeseoSWUpdateState[iEXitCode]);
	}
}

