/************************************************************************
 * FILE: 			sxm_sirius_update.cpp
 *
 * SW-COMPONENT: 	swu_common_sxm_lib
 *
 * DESCRIPTION: 	A server class to take calls for updating SXM firmware
 *
 * AUTHOR: 			Logesh Gopalakrishnan (RBEI/ECA)
 *
 * COPYRIGHT: 		(c) Robert Bosch Engineering and Business Solutions Ltd.
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
 *  20.06.2016  Rev. 1.1 Logesh
 * 				Choose firmware from a dir based on HType.
************************************************************************/

#include "sxm_gpio.h"
#include "sxm_version_utility.h"
#include "sxm_sirius_update.h"
#include "sxm_module_handler.h"
#include "sxm_update_log.h"
#undef fgets

#include "ai_sw_update/common/base/imp/swupd_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
	#define ETG_I_TRACE_CHANNEL 			TR_TTFIS_SWUPDATE_CTRL
	#define ETG_I_TTFIS_CMD_PREFIX 			"DNL_"
	#define ETG_I_FILE_PREFIX 				dl_tclSXM::
    #define ETG_DEFAULT_TRACE_CLASS 		TR_CLASS_SWUPDATE_SXM
  #include "trcGenProj/Header/sxm_sirius_update.cpp.trc.h"
#endif


using namespace SiriusModuleUpdateControlLib;
using namespace std;
using namespace ::ai_sw_update::common;


/************************************************************************
 * FUNCTION: SiriusUpdate::SiriusUpdate
 *
 * DESCRIPTION: Constructor
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE:
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
SiriusUpdate::SiriusUpdate()
{
	enSXMState = SXM_STATE_INVALID;
	enSXMError = SXM_NO_ERROR;
	_hSiriusModule = NULL;

	UpdateLog::vSetLogFile("sxm_update.log");
	UpdateLog::vUpdateLogOpen();

	ETG_I_REGISTER_FILE();
	ETG_I_REGISTER_CHN(0);
}

/************************************************************************
 * FUNCTION: SiriusUpdate::~SiriusUpdate
 *
 * DESCRIPTION: Destructor
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE:
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
SiriusUpdate::~SiriusUpdate()
{
	_hSiriusModule = NULL;

    UpdateLog::vUpdateLogClose();
}

/************************************************************************
 * FUNCTION: SiriusUpdate::setSiriusModuleHandler
 *
 * DESCRIPTION: Set SiriusModuleHandler implementation pointer
 *
 * PARAMETER:
 * 				SiriusModuleHandler*
 *
 * RETURNVALUE: tVoid
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tVoid SiriusUpdate::setSiriusModuleHandler(SiriusModuleHandler* siriusModuleHandler)
{
	_hSiriusModule = siriusModuleHandler;
}


// To provide client specific implementation of SiriusModuleHandler
/************************************************************************
 * FUNCTION: SiriusUpdate::initializeModuleHandlerIfNotInitialized
 *
 * DESCRIPTION: If siriusModuleHandler is not initialized and passed on,
 * 					create all controllers and initialize them now.
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: void
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tVoid SiriusUpdate::initializeModuleHandlerIfNotInitialized()
{

	// Already some one has given us the handle
	if(_hSiriusModule != NULL) return;

	// If dependency not injected, use conventional controllers.
	// Create new object for handling Sirius Module
	_hSiriusModule = SiriusModuleHandler::instance();

	//TODO: Should delete newly created items
	SXM_GPIO* oGPIOController = new tclSXM_GPIO();
	SXM_VersionUtility* oVersionUtility = new tclSXM_VersionUtility();

	_hSiriusModule->setControllers(oGPIOController, oVersionUtility);

	ETG_TRACE_USR4(("Controllers not provided. So creating them."));
	UpdateLogInfo("Controllers not provided. So creating them.");

}

/************************************************************************
 * FUNCTION: SiriusUpdate::isSiriusModuleHandleValid
 *
 * DESCRIPTION: Return true if SiriusModuleHandle is valid else false
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: tBool
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tBool SiriusUpdate::isSiriusModuleHandleValid()
{
	if(_hSiriusModule == NULL)
	{
		ETG_TRACE_ERR(("Sirius module handle not initialized"));
		UpdateLogError("Sirius module handle not initialized");
		enSXMError = SXM_ERROR_INITIALIZATION_FAILURE;
		enSXMState = SXM_STATE_INVALID;
		return false;
	}
	else
		return true;
}



/************************************************************************
 * FUNCTION: SiriusUpdate::isInitialized
 *
 * DESCRIPTION: Check if Sirius Module is already initialized
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: tBool
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tBool SiriusUpdate::isInitialized()
{
	return ( enSXMState == SXM_STATE_INITIALIZED || enSXMState == SXM_STATE_UPDATE_SUCCESS || enSXMState == SXM_STATE_UPDATE_FAILURE );
}

/************************************************************************
 * FUNCTION: SiriusUpdate::initialize
 *
 * DESCRIPTION: Ask SiriusModuleHandle to power on and initialize SXM if
 * 					it is in proper state.
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: tBool
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tBool SiriusUpdate::initialize()
{

	// If already initialized just return true
	if( isInitialized() )
	{
		ETG_TRACE_USR4(("Already initialized"));
		UpdateLogInfo("Already initialized");
		return true;
	}

	initializeModuleHandlerIfNotInitialized();

	if(! isSiriusModuleHandleValid() ) return false;

	if(enSXMState == SXM_STATE_INVALID || enSXMState == SXM_STATE_DEINITIALIZED)
	{
		if (!_hSiriusModule->powerOn()) {
			enSXMError = SXM_ERROR_POWER_ON_FAILURE;
			return false;
		}

		enSXMError = _hSiriusModule->initialize();
		if (enSXMError != SXM_NO_ERROR) {
			return false;
		}

		enSXMState = SXM_STATE_INITIALIZED;
		return true;
	}
	else if(enSXMState == SXM_STATE_UPDATING)
	{
		enSXMError = SXM_ERROR_UPDATE_IN_PROGRESS;
		ETG_TRACE_ERR(("Update is in progress"));
		UpdateLogError("Update is in progress");
		return false;
	}
	ETG_TRACE_ERR(("Initialize control should not be here"));
	UpdateLogError("Initialize control should not be here");
	return false;
}

/************************************************************************
 * FUNCTION: SiriusUpdate::updateFirmware
 *
 * DESCRIPTION: Start a firmware update if SXM is already initialized.
 *
 * PARAMETER:
 * 				string, cbProgressInformation
 *
 * RETURNVALUE: tBool
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
 *  20.06.2016  Rev. 1.1 Logesh
 * 				Choose firmware from a dir based on HType.
************************************************************************/
tBool SiriusUpdate::updateFirmware(string l_firmwareFilePath, tBool bForceUpdate, cbProgressInformation cbProgressUpdate)
{
	if(! isSiriusModuleHandleValid() ) return false;

	if( isInitialized() )
	{
		enSXMState = SXM_STATE_UPDATING;
		string foundFirmwarePath = "";

		enSXMError = chooseCompatibleFirmware(getHardwareType(), l_firmwareFilePath, foundFirmwarePath);
		if (enSXMError != SXM_NO_ERROR) {
			enSXMState = SXM_STATE_UPDATE_FAILURE;
			ETG_TRACE_ERR(("Update failed while choosing firmware %d", enSXMError));
			UpdateLogError("Update failed while choosing firmware  %d", enSXMError);
			return false;
		}

		enSXMError = _hSiriusModule->firmwareUpdate(foundFirmwarePath, bForceUpdate, cbProgressUpdate);
		if (enSXMError != SXM_NO_ERROR) {
			enSXMState = SXM_STATE_UPDATE_FAILURE;
			ETG_TRACE_ERR(("Update failed %d", enSXMError));
			UpdateLogError("Update failed %d", enSXMError);
			return false;
		}

		enSXMState = SXM_STATE_UPDATE_SUCCESS;
		return true;
	}
	else if( enSXMState == SXM_STATE_UPDATING )
	{
		ETG_TRACE_ERR(("Update is in progress"));
		UpdateLogError("Update is in progress");
		enSXMError = SXM_ERROR_UPDATE_IN_PROGRESS;
		return false;
	}

	else if( enSXMState == SXM_STATE_DEINITIALIZED )
	{
		ETG_TRACE_ERR(("Not initialized"));
		UpdateLogError("Not initialized");
		enSXMError = SXM_ERROR_NOT_INITIALIZED;
		return false;
	}
	ETG_TRACE_ERR(("updateFirmware control should not be here"));
	UpdateLogError("updateFirmware control should not be here");
	enSXMState = SXM_STATE_UPDATE_FAILURE;
	return false;
}

/************************************************************************
 * FUNCTION: SiriusUpdate::deinitialize
 *
 * DESCRIPTION: Deinitialize and power off SXM
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: tBool
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tBool SiriusUpdate::deinitialize()
{

	if(! isSiriusModuleHandleValid() ) return false;

	// If not INITIALIZED don't allow
	if( enSXMState == SXM_STATE_INVALID )
	{
		ETG_TRACE_USR4(("Not initialized to deinitialize"));
		UpdateLogInfo("Not initialized to deinitialize");
		enSXMError = SXM_ERROR_NOT_INITIALIZED;
		return false;
	}
	if( enSXMState == SXM_STATE_UPDATING )
	{
		ETG_TRACE_ERR(("Cannot deinitialize. Update in progress"));
		UpdateLogError("Cannot deinitialize. Update in progress");
		enSXMError = SXM_ERROR_UPDATE_IN_PROGRESS;
		return false;
	}


	_hSiriusModule->deInitialize();
	_hSiriusModule->powerOff();

	enSXMState = SXM_STATE_INVALID;
	return TRUE;
}

/************************************************************************
 * FUNCTION: SiriusUpdate::getError
 *
 * DESCRIPTION: Return recent error
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: SXM_ERROR_CODE_ENUM
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
SXM_ERROR_CODE_ENUM SiriusUpdate::getError()
{
	return enSXMError;
}

/************************************************************************
 * FUNCTION: SiriusUpdate::chooseCompatibleFirmware
 *
 * DESCRIPTION: Return error
 *
 * PARAMETER:
 *				string, string, string&
 *
 * RETURNVALUE: SXM_ERROR_CODE_ENUM
 *
 * HISTORY:
 * 	20.06.2016 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
SXM_ERROR_CODE_ENUM SiriusUpdate::chooseCompatibleFirmware(string sHardwareType, string sBasePath, string& sChosenBinary)
{
    char buffer[MAX_SIZE_FIND_BUFFER];

	// find all files with .bin extension in the given directory or file
	string command = "find " + sBasePath + " -type f -name '*.bin'";

    FILE* pipe = popen(command.c_str(), "r");
    if (!pipe)
    {
    	ETG_TRACE_ERR(("Could not get files from %s", sBasePath.c_str()));
		UpdateLogError("Could not get files from %s", sBasePath.c_str());
		enSXMError = SXM_ERROR_INVALID_ARGUMENTS;
		return enSXMError;
    }

	// read result line by line
	UpdateLogInfo("Looking for HW Type compatible firmware file");
	while (fgets(buffer, MAX_SIZE_FIND_BUFFER, pipe) != NULL)
	{
		buffer[strcspn(buffer, "\r\n")] = 0; // remove newlines
		if(isFirmwareFitForHardware(buffer, sHardwareType))
		{
			UpdateLogInfo("HW type Compatible Firmware file found!");
			sChosenBinary = buffer;
			pclose(pipe);
			return SXM_NO_ERROR;
		}

	}
	pclose(pipe);

	ETG_TRACE_ERR(("No firmware is found for hardware"));
	UpdateLogError("No firmware is found for hardware");
	enSXMError = SXM_ERROR_INVALID_ARGUMENTS;
	return enSXMError;

}

/************************************************************************
 * FUNCTION: SiriusUpdate::isFirmwareFitForHardware
 *
 * DESCRIPTION: Checks if the firmware file is compatible with the module
 * 					hardware type
 *
 * PARAMETER:
 * 				string, string
 *
 * RETURNVALUE: tBool
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tBool SiriusUpdate::isFirmwareFitForHardware(string sFrimwarePath, string sHType)
{
	string sFirmwareHardwareType = _hSiriusModule->getHardwareTypeFromFirmware(sFrimwarePath);
	tBool bResult = (sFirmwareHardwareType.compare(sHType) == 0);

	ETG_TRACE_USR4(("isFirmwareFitForHardware Firmware = %s", sFrimwarePath.c_str()));
	ETG_TRACE_USR4(("isFirmwareFitForHardware FimwareHardwareType = %s", sFirmwareHardwareType.c_str()));
	ETG_TRACE_USR4(("ModuleHardwareType = %s", sHType.c_str()));
	UpdateLogInfo("Firmware = %s, FrimwareHardwareType = %s, ModuleHardwareType = %s",
			sFrimwarePath.c_str(), sFirmwareHardwareType.c_str(), sHType.c_str());
	
	return bResult;
}

/************************************************************************
 * FUNCTION: SiriusUpdate::isFirmwareVersionNewer
 *
 * DESCRIPTION: Checks if the firmware file is newer than the one that is
 * 					already present in the module
 *
 * PARAMETER:
 * 				string
 *
 * RETURNVALUE: tBool
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tBool SiriusUpdate::isFirmwareVersionNewer(string l_firmwareFilePath)
{

	if(! isSiriusModuleHandleValid() ) return false;


	//TODO: Implementation needed

	(string) l_firmwareFilePath;

	return true;
}

/************************************************************************
 * FUNCTION: SiriusUpdate::getSerialVersion
 *
 * DESCRIPTION: Gets the serial number of SXM
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: string
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
string SiriusUpdate::getSerialVersion()
{
	if(_hSiriusModule)
		return _hSiriusModule->getSerialVersionOfModule();
	return "";
}

/************************************************************************
 * FUNCTION: SiriusUpdate::getHardwareVersion
 *
 * DESCRIPTION: Gets the Hardware version of SXM
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: string
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
string SiriusUpdate::getHardwareVersion()
{
	if(_hSiriusModule)
		return _hSiriusModule->getHardwareVersionOfModule();
	return "";
}

/************************************************************************
 * FUNCTION: SiriusUpdate::getHardwareType
 *
 * DESCRIPTION: Gets the Hardware type of SXM
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: string
 *
 * HISTORY:
 * 	20.06.2016 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
string SiriusUpdate::getHardwareType()
{
	if(_hSiriusModule)
		return _hSiriusModule->getHardwareTypeOfModule();
	return "";
}

/************************************************************************
 * FUNCTION: SiriusUpdate::getSoftwareVersion
 *
 * DESCRIPTION: Gets the software version of SXM
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: string
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
string SiriusUpdate::getSoftwareVersion()
{
	if(_hSiriusModule)
		return _hSiriusModule->getSoftwareVersionOfModule();
	return "";
}

/************************************************************************
 * FUNCTION: SiriusUpdate::getFirmwareVersion
 *
 * DESCRIPTION: get the software version of firmware file
 *
 * PARAMETER:
 * 				string
 *
 * RETURNVALUE: string
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
string SiriusUpdate::getFirmwareVersion(string l_firmwareFilePath)
{
	if(_hSiriusModule)
		return _hSiriusModule->getSoftwareVersionOfFirmware(l_firmwareFilePath);
	return "";
}

/************************************************************************
 * FUNCTION: SiriusUpdate::getHardwareTypeFromFirmware
 *
 * DESCRIPTION: get the compatible hardware type from firmware file
 *
 * PARAMETER:
 * 				string
 *
 * RETURNVALUE: string
 *
 * HISTORY:
 * 	20.06.2016 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
string SiriusUpdate::getHardwareTypeFromFirmware(string l_firmwareFilePath)
{
	if(_hSiriusModule)
		return _hSiriusModule->getHardwareTypeFromFirmware(l_firmwareFilePath);
	return "";
}

