/************************************************************************
 * FILE: 			sxm_module_handler.cpp
 *
 * SW-COMPONENT: 	swu_common_sxm_lib
 *
 * DESCRIPTION:   	A wrapper class for sms-lib, to provide update interfaces
 *
 * AUTHOR: 			Logesh Gopalakrishnan (RBEI/ECA)
 *
 * COPYRIGHT: 		(c) Robert Bosch Engineering and Business Solutions Ltd.
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
 *  20.06.2016  Rev. 1.1 Logesh
 * 				Choose firmware from a dir based on HType.
************************************************************************/

#include "sxm_module_handler.h"
#include "sxm_update_log.h"
#include "sxm_version_utility.h"
#include "sxm_gpio.h"
#include <unistd.h> //usleep
#include <sys/time.h> // gettimeofday
#include <errno.h>

#include "ai_sw_update/common/base/imp/swupd_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
	#define ETG_I_TRACE_CHANNEL 			TR_TTFIS_SWUPDATE_CTRL
	#define ETG_I_TTFIS_CMD_PREFIX 			"DNL_"
	#define ETG_I_FILE_PREFIX 				dl_tclSXM::
	#define ETG_DEFAULT_TRACE_CLASS 		TR_CLASS_SWUPDATE_SXM
	#include "trcGenProj/Header/sxm_module_handler.cpp.trc.h"
#endif


/*
 * Currently taken as input from caller. May need these path in future.

#define SMS_WORKING_DIR 			"/var/opt/bosch/dynamic/ffs/siriusxm/sms_hb"
#define SMS_CONFIGURATION_DIR 		"/var/opt/bosch/static/siriusxm/sms_hb"
#define SMS_CONFIGURATION_FILE 		"sms.cfg"
*/

#define SMS_WORKING_DIR 			"/tmp"
#define FC_SXM_RADIO_NAME 			"radio:"
#define FC_SXM_SERIAL_DEVICE 		"/dev/ttymxc2"
#define FC_SXM_SMS_UART_PARAMS 		("115200,N,8,1")
#define SMS_CONFIGURATION_WAIT_TIME (30 * 1000) // Thirty Seconds
#define SMS_UPDATE_WAIT_TIME 		(5 * 60 * 1000) // Five Minutes


using namespace SiriusModuleUpdateControlLib;
using namespace std; 
using namespace ::ai_sw_update::common;

/************************************************************************
 * FUNCTION: SiriusModuleHandler::SiriusModuleHandler
 *
 * DESCRIPTION: Constructor
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE:
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
SiriusModuleHandler::SiriusModuleHandler() {

		_hSRM = 0;
		_hModule = 0;

		oGPIO = 0;
		oVersionUtility = 0;

		resetCounter = 0;
		
		_cbProgressUpdate = 0;
		enModuleErrorCode = MODULE_ERROR_CODE_NONE;
		enUpdateErrorCode = SXM_NO_ERROR;

		isDriverReady = FALSE;
		isSRMReady = FALSE;
		isModuleReady = FALSE;
		enUpdateState = SXM_STATE_INVALID;

		softwareVersion = "";
		firmwareFileVersion = "";
		firmwareFilePath = "";

		pthread_mutex_init(&DriverReadyMutex, NULL);
		pthread_mutex_init(&SRMReadyMutex, NULL);
		pthread_mutex_init(&ModuleReadyMutex, NULL);
		pthread_mutex_init(&UpdateCompleteMutex, NULL);

		pthread_cond_init(&DriverReadyCondition, NULL);
		pthread_cond_init(&SRMReadyCondition, NULL);
		pthread_cond_init(&ModuleReadyCondition, NULL);
		pthread_cond_init(&UpdateCompleteCondition, NULL);
	}

/************************************************************************
 * FUNCTION: SiriusModuleHandler::~SiriusModuleHandler
 *
 * DESCRIPTION: Destructor
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE:
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
SiriusModuleHandler::~SiriusModuleHandler() {

	_hSRM = 0;
	_hModule = 0;

	oGPIO = 0;
	oVersionUtility = 0;

	pthread_mutex_destroy(&DriverReadyMutex);
	pthread_mutex_destroy(&SRMReadyMutex);
	pthread_mutex_destroy(&ModuleReadyMutex);
	pthread_mutex_destroy(&UpdateCompleteMutex);

	pthread_cond_destroy(&DriverReadyCondition);
	pthread_cond_destroy(&SRMReadyCondition);
	pthread_cond_destroy(&ModuleReadyCondition);
	pthread_cond_destroy(&UpdateCompleteCondition);
}
/************************************************************************
 * FUNCTION: SiriusModuleHandler::bRegisterDrivers
 *
 * DESCRIPTION: Register serial port and register SRH driver
 *
 * PARAMETER:
 * 				void*
 *
 * RETURNVALUE: tBool
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tBool SiriusModuleHandler::bRegisterDrivers(tVoid *pvArg)
{
   (tVoid)pvArg;
   tS32 s32RetVal;

   /////////////////////////////
   // Register Serial Port(s)...
   /////////////////////////////

   // Register LSIM serial port
   s32RetVal = OSAL.n32RegNode(
	  FC_SXM_SERIAL_DEVICE,
	  &GsPosixSerialInterface,
	  FC_SXM_SERIAL_DEVICE,
	  strlen(FC_SXM_SERIAL_DEVICE) + 1);

   if(s32RetVal != DEV_OK)
   {
	   ETG_TRACE_USR4(("Error! Cannot Register %s", FC_SXM_SERIAL_DEVICE));
	   UpdateLogInfo("Error! Cannot Register %s", FC_SXM_SERIAL_DEVICE);
	  return FALSE;
   }

   ETG_TRACE_USR4(("Registered device %s", FC_SXM_SERIAL_DEVICE));
   UpdateLogInfo("Registered device %s", FC_SXM_SERIAL_DEVICE);

   // Register LSIM serial port via a shim i/o driver.
   // This driver allows us to use the serial port's modem
   // signals as independently controlled gpio.
   s32RetVal = OSAL.n32RegNode(
	  FC_SXM_RADIO_NAME,
	  &GsSioInterface,
	  FC_SXM_SERIAL_DEVICE,
	  strlen(FC_SXM_SERIAL_DEVICE) + 1);

   if(s32RetVal != DEV_OK)
   {
	   ETG_TRACE_USR4(("Error! Cannot Register %s", FC_SXM_RADIO_NAME));
	   UpdateLogInfo("Error! Cannot Register %s", FC_SXM_RADIO_NAME);
	  return FALSE;
   }

   ETG_TRACE_USR4(("Registered device %s", FC_SXM_RADIO_NAME));
   UpdateLogInfo("Registered device %s", FC_SXM_RADIO_NAME);

   /////////////////////////////
   // Register SRH Driver
   /////////////////////////////
   SRH_vRegisterDriver(const_cast<char *> FC_SXM_SMS_UART_PARAMS);

   ETG_TRACE_USR4(("Registered SRH driver"));
   UpdateLogInfo("Registered SRH driver");

   return TRUE;
}

/************************************************************************
 * FUNCTION: SiriusModuleHandler::bUnRegisterDrivers
 *
 * DESCRIPTION: Unregister SRH driver
 *
 * PARAMETER:
 * 				void*
 *
 * RETURNVALUE: void
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tVoid SiriusModuleHandler::bUnRegisterDrivers(tVoid *pvArg)
{

   (tVoid)pvArg;

   ETG_TRACE_USR4(("Unregistering SRH driver"));
   UpdateLogInfo("Unregistering SRH driver");

   // Unregister SRH Drivers
   SRH_vUnRegisterDriver();

   OSAL.bUnRegNode(FC_SXM_RADIO_NAME);
   OSAL.bUnRegNode(FC_SXM_SERIAL_DEVICE);

}

/************************************************************************
 * FUNCTION: SiriusModuleHandler::bStartHandler
 *
 * DESCRIPTION: Called after driver init
 *
 * PARAMETER:
 * 				const void*
 *
 * RETURNVALUE: tBool
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tBool SiriusModuleHandler::bStartHandler(const tVoid *pvArg)
{
	(tVoid)pvArg;
	tBool bTempDirSet;

	bTempDirSet = OSAL.bFileSystemSetTempPath(SMS_WORKING_DIR);

	if (bTempDirSet == FALSE)
	{
		return FALSE;
	}

	ETG_TRACE_USR4(("bStartHandler started"));
	UpdateLogInfo("bStartHandler started");

	pthread_mutex_lock(&instance()->DriverReadyMutex);
		instance()->isDriverReady = TRUE;
	pthread_cond_signal(&instance()->DriverReadyCondition);
	pthread_mutex_unlock(&instance()->DriverReadyMutex);

	return TRUE;
}

/************************************************************************
 * FUNCTION: SiriusModuleHandler::pfSmsInitThreadFn
 *
 * DESCRIPTION: Starting function of OSAL Thread.
 *
 * PARAMETER:
 * 				void*
 *
 * RETURNVALUE: void*
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
void* SiriusModuleHandler::pfSmsInitThreadFn(void *)
{
	// OSAL Start Handlers
	static const OSAL_START_HANDLER_STRUCT gsStartHandlers =
	{
		// OSAL Driver Registration Handlers
			SiriusModuleHandler::bRegisterDrivers,
			SiriusModuleHandler::bUnRegisterDrivers,
		NULL,

		// OSAL Start Handler
		SiriusModuleHandler::bStartHandler,
		NULL,

		// OSAL Monitor Handlers
		{
			NULL,
			NULL,

			NULL,
			NULL,

			NULL,
			NULL,
			NULL,
			NULL
		}
	};

	void *pvRes = reinterpret_cast<void*>(OSAL.bStart(&gsStartHandlers));

	return pvRes;
}

/************************************************************************
 * FUNCTION: SiriusModuleHandler::initializeSRM_Object
 *
 * DESCRIPTION: Blocking call to initialize SRM object
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: tBool
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tBool SiriusModuleHandler::initializeSRM_Object() {
	// 3. Create a SRM object and wait for it to go to ready

	_hSRM = SRM.hGet("srh:", "srm:", SRM_OBJECT_EVENT_ALL, cb_vSRMEventCallbackLocal, NULL);
	tBool bResult = _hSRM != SRM_INVALID_OBJECT;
	if(bResult == FALSE)
	{
		enUpdateErrorCode = SXM_ERROR_INVALID_SRM;

		ETG_TRACE_ERR(("SRM object could not be created. Initialization failed."));
		UpdateLogError("SRM object could not be created. Initialization failed.");

	}
	else
	{

		ETG_TRACE_USR4(("waiting for SRM to get ready"));
		UpdateLogInfo("waiting for SRM to get ready");

		if(! isSRMReady) {
	        pthread_mutex_lock(&SRMReadyMutex);
	        cond_timedwait(SRMReadyCondition, SRMReadyMutex, SMS_CONFIGURATION_WAIT_TIME);
	        pthread_mutex_unlock(&SRMReadyMutex);
		}
		if( ! isSRMReady )
		{
			bResult = FALSE;
			ETG_TRACE_ERR(("Timed out. SRM failed to get ready"));
			UpdateLogError("Timed out. SRM failed to get ready");
		}
		else
	    {
			bResult = TRUE;
		    ETG_TRACE_USR4(("SRM is ready"));
		    UpdateLogInfo("SRM is ready");
	    }

	}
	return bResult;

}

/************************************************************************
 * FUNCTION: SiriusModuleHandler::initializeModule_Object
 *
 * DESCRIPTION: Blocking call to initialize Module object
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: tBool
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tBool SiriusModuleHandler::initializeModule_Object() {

	// 4. Create a Module object and wait for it to go to ready

	tUInt _u32Mask = (MODULE_OBJECT_EVENT_ALL ^ MODULE_OBJECT_EVENT_UPDATE_PROGRESS);
	_hModule = MODULE.hGet(_hSRM, "Audio", (MODULE_EVENT_MASK) _u32Mask, cb_vModuleEventCallbackLocal, NULL);
	tBool bResult = _hModule != MODULE_INVALID_OBJECT;
	if(bResult == FALSE)
	{
		enUpdateErrorCode = SXM_ERROR_INVALID_MODULE;
		ETG_TRACE_ERR(("Module object could not be created. Initialization failed."));
		UpdateLogError("Module object could not be created. Initialization failed.");
	}
	else
	{

		ETG_TRACE_USR4(("waiting for Module to get ready"));
		UpdateLogInfo("waiting for Module to get ready");

		if(! isModuleReady) {
	        pthread_mutex_lock(&ModuleReadyMutex);
	        cond_timedwait(ModuleReadyCondition, ModuleReadyMutex, SMS_CONFIGURATION_WAIT_TIME);
	        pthread_mutex_unlock(&ModuleReadyMutex);
		}
		if( ! isModuleReady )
		{
			bResult = FALSE;
			ETG_TRACE_ERR(("Timed out. Module failed to get ready"));
			UpdateLogError("Timed out. Module failed to get ready");
		}
    	else
    	{
			bResult = TRUE;
    		ETG_TRACE_USR4(("Module is ready"));
    		UpdateLogInfo("Module is ready");
    	}
	}

	return bResult;
}

/************************************************************************
 * FUNCTION: SiriusModuleHandler::cb_vSRMEventCallbackLocal
 *
 * DESCRIPTION: Call back to indicate SRM events
 *
 * PARAMETER:
 * 				SRM_OBJECT, SRM_EVENT_MASK, void*
 *
 * RETURNVALUE: void
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tVoid SiriusModuleHandler::cb_vSRMEventCallbackLocal(SRM_OBJECT hSRM,
		SRM_EVENT_MASK tEventMask, tVoid *pvEventCallbackArg) {

	(tVoid) pvEventCallbackArg; /* TO REMOVE LINT WARNINGS */

	if (tEventMask & SRM_OBJECT_EVENT_STATE) {

		SRM_STATE_ENUM eState = SRM.eState(hSRM);

		if( eState == SRM_STATE_READY )
		{
			ETG_TRACE_USR4(("SRM_STATE_READY"));
			UpdateLogInfo("SRM_STATE_READY");
			pthread_mutex_lock(&instance()->SRMReadyMutex);
			instance()->isSRMReady = true;
			pthread_cond_signal(&instance()->SRMReadyCondition);
			pthread_mutex_unlock(&instance()->SRMReadyMutex);

		}
		if( eState == SRM_STATE_ERROR)
		{
			ETG_TRACE_ERR(("SRM_STATE_ERROR"));
			UpdateLogError("SRM_STATE_ERROR");
			instance()->enUpdateErrorCode = SXM_ERROR_INVALID_SRM;
			pthread_mutex_lock(&instance()->SRMReadyMutex);

			instance()->isSRMReady = false;
			pthread_cond_signal(&instance()->SRMReadyCondition);
			pthread_mutex_unlock(&instance()->SRMReadyMutex);
		}

	}
}

/************************************************************************
 * FUNCTION: SiriusModuleHandler::cb_vModuleEventCallbackLocal
 *
 * DESCRIPTION: Call back to indicate Module events
 *
 * PARAMETER:
 * 				MODULE_OBJECT, MODULE_EVENT_MASK, void*
 *
 * RETURNVALUE: void
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tVoid SiriusModuleHandler::cb_vModuleEventCallbackLocal(MODULE_OBJECT hModule,
		MODULE_EVENT_MASK tEventMask, tVoid *pvEventCallbackArg) {
	(tVoid) pvEventCallbackArg; /* TO REMOVE LINT WARNINGS */

    SiriusModuleHandler* handler = instance();
	if(!handler)
	{
		ETG_TRACE_USR4(("SiriusModuleHandler is null in cb_vModuleEventCallbackLocal"));
		UpdateLogInfo("SiriusModuleHandler is null in cb_vModuleEventCallbackLocal");
		return;
	}
	if(hModule != handler->_hModule)
	{
		ETG_TRACE_USR4(("Another instance of module is active"));
		UpdateLogInfo("Another instance of module is active");
		return;
	}

	if(tEventMask & MODULE_OBJECT_EVENT_STATE)
	{
		MODULE_STATE_ENUM eModuleState = MODULE.eState(hModule);

		switch(eModuleState)
		{
		case MODULE_STATE_READY:

			ETG_TRACE_USR4(("MODULE_STATE_READY"));
			UpdateLogInfo("MODULE_STATE_READY");

			if(handler->enUpdateState != SXM_STATE_UPDATING)
			{
				pthread_mutex_lock(&handler->ModuleReadyMutex);
				handler->isModuleReady = true;
				pthread_cond_signal(&handler->ModuleReadyCondition);
				pthread_mutex_unlock(&handler->ModuleReadyMutex);
			}

			/* If an update completes module will restart and go to ready state
			 * So if update was running and module is going to ready then
			 * 	update completed successfully.
			 */
			if(handler->enUpdateState == SXM_STATE_UPDATING)
			{

				if(handler->isFimwareAndModuleVersionSame())
				{
					handler->enUpdateState = SXM_STATE_UPDATE_SUCCESS;
					ETG_TRACE_USR4(("SXM_STATE_UPDATE_SUCCESS"));
					UpdateLogInfo("SXM_STATE_UPDATE_SUCCESS");
				}
				else
				{
					handler->enUpdateState = SXM_STATE_UPDATE_FAILURE;
					ETG_TRACE_ERR(("SXM_STATE_UPDATE_FAILURE"));
					UpdateLogError("SXM_STATE_UPDATE_FAILURE");
				}
				pthread_mutex_lock(&handler->UpdateCompleteMutex);
				pthread_cond_signal(&handler->UpdateCompleteCondition);
				pthread_mutex_unlock(&handler->UpdateCompleteMutex);
			}
		break;

		case MODULE_STATE_UPDATING:

			// update started. wait for it to reset. and go to ready again.
			handler->enUpdateState = SXM_STATE_UPDATING;
			ETG_TRACE_USR4(("MODULE_STATE_UPDATING"));
			UpdateLogInfo("MODULE_STATE_UPDATING");

		break;

		case MODULE_STATE_ERROR:

			ETG_TRACE_ERR(("MODULE_STATE_ERROR"));
			UpdateLogError("MODULE_STATE_ERROR");

			if(handler->enUpdateState != SXM_STATE_UPDATING)
			{
				handler->enUpdateErrorCode = SXM_ERROR_INVALID_MODULE;

				MODULE_ERROR_CODE_ENUM eErrorCode = MODULE.eErrorCode(hModule);
				handler->enModuleErrorCode = eErrorCode;
				ETG_TRACE_ERR(("Module not ready. Error: %d", eErrorCode));
				UpdateLogError("Module not ready. Error: %d", eErrorCode);
				
				pthread_mutex_lock(&handler->ModuleReadyMutex);
				handler->isModuleReady = false;
				pthread_cond_signal(&handler->ModuleReadyCondition);
				pthread_mutex_unlock(&handler->ModuleReadyMutex);
			}

			if(handler->enUpdateState == SXM_STATE_UPDATING)
			{
				handler->enUpdateState = SXM_STATE_UPDATE_FAILURE;

				MODULE_ERROR_CODE_ENUM eErrorCode = MODULE.eErrorCode(hModule);
				handler->enModuleErrorCode = eErrorCode;
				ETG_TRACE_ERR(("Module error while updating: %d", eErrorCode));
				UpdateLogError("Module error while updating: %d", eErrorCode);
				handler->enUpdateErrorCode = SXM_ERROR_FIRMWARE_UPDATE_FAILED;
				
				pthread_mutex_lock(&handler->UpdateCompleteMutex);
				pthread_cond_signal(&handler->UpdateCompleteCondition);
				pthread_mutex_unlock(&handler->UpdateCompleteMutex);
			}
		break;

		default:
			break;

		} // switch(eModuleState)

	} // MODULE_OBJECT_EVENT_STATE

	if (tEventMask & MODULE_OBJECT_EVENT_UPDATE_PROGRESS)
	{

		tU8 u8UpdateProgress=0;
		SMSAPI_RETURN_CODE_ENUM enSmsRes = MODULE.eUpdateProgress (hModule, &u8UpdateProgress);

		if(enSmsRes != SMSAPI_RETURN_CODE_SUCCESS)
		{
			ETG_TRACE_USR4(("Cannot get update progress (%d)", enSmsRes));
			UpdateLogInfo("Cannot get update progress (%d)", enSmsRes);
			return;
		}

		// On registering there will be a temporary "100%" notification
		if(handler->enUpdateState == SXM_STATE_UPDATING)
		{
			if(handler->_cbProgressUpdate)
				handler->_cbProgressUpdate(u8UpdateProgress);

			if(0 == (u8UpdateProgress % 10) ) // To reduce load, print only multiples of 10.
			{
				ETG_TRACE_USR4(("%d percent complete", u8UpdateProgress));
				UpdateLogInfo("Updating...%d%%", u8UpdateProgress);
			}
		}

	} // MODULE_OBJECT_EVENT_UPDATE_PROGRESS

}

/************************************************************************
 * FUNCTION: SiriusModuleHandler::cond_timedwait
 *
 * DESCRIPTION: Wrapper to pthread_cond_timedwait. Waits for signal or times out
 *
 * PARAMETER:
 * 				pthread_cond_t&, pthread_mutex_t&, tInt
 *
 * RETURNVALUE: tInt
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tInt SiriusModuleHandler::cond_timedwait(pthread_cond_t& condition, pthread_mutex_t& mutex, tInt timeInMs)
{

	 struct timeval tv;
	 struct timespec ts;

	 gettimeofday(&tv, NULL);
	 ts.tv_sec = time(NULL) + timeInMs / 1000;
	 ts.tv_nsec = tv.tv_usec * 1000 + 1000 * 1000 * (timeInMs % 1000);
	 ts.tv_sec += ts.tv_nsec / (1000 * 1000 * 1000);
	 ts.tv_nsec %= (1000 * 1000 * 1000);

	 return pthread_cond_timedwait(&condition, &mutex, &ts);
}

/************************************************************************
 * FUNCTION: SiriusModuleHandler::isFimwareAndModuleVersionSame
 *
 * DESCRIPTION: Checks if Module version and firmware file version is equal
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: tBool
 * 				true => If equal
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
 *  20.06.2016  Rev. 1.1 Logesh
 				Return type changed to bool. 
************************************************************************/
tBool SiriusModuleHandler::isFimwareAndModuleVersionSame()
{

	if( oVersionUtility == NULL)
		return false;

	string l_ModuleVersion, l_firmwareFileVersion;

	oVersionUtility->getSoftwareVersionOfModule(_hModule, l_ModuleVersion);
	oVersionUtility->getSoftwareVersionOfFirmware(firmwareFilePath, l_firmwareFileVersion);
	return (l_ModuleVersion.compare(l_firmwareFileVersion) == 0);
}

/************************************************************************
 * FUNCTION: SiriusModuleHandler::isFirmwareNewer
 *
 * DESCRIPTION: Dummy implementation to check if the firmware file
 * 				is newer than running module version
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: tBool
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tBool SiriusModuleHandler::isFirmwareNewer() {
	// TODO: Implement firmware compare
	//tInt iReturn = compareFimwareAndModuleVersion();

	return true;
}

/************************************************************************
 * FUNCTION: SiriusModuleHandler::setControllers
 *
 * DESCRIPTION: To give GPIO and Version Utility implementations
 *
 * PARAMETER:
 * 				SXM_GPIO*, SXM_VersionUtility*
 *
 * RETURNVALUE: tVoid
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tVoid SiriusModuleHandler::setControllers(SXM_GPIO* gpio, SXM_VersionUtility* versionUtil)
{
	oGPIO = gpio;
	oVersionUtility = versionUtil;
}



/************************************************************************
 * FUNCTION: SiriusModuleHandler::powerOn
 *
 * DESCRIPTION: Powers on SXM
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: tBool
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tBool SiriusModuleHandler::powerOn() {

	ETG_TRACE_USR4(("Powering On"));
	UpdateLogInfo("Powering On");

	tBool bResult = FALSE;
	if(oGPIO)
	{
		bResult = oGPIO->initializeGPIOs();

		bResult &= oGPIO->setPower(TRUE);
		usleep(250000);

		bResult &= oGPIO->setReset(TRUE);
		usleep(100000);
	}

	return bResult;

}

/************************************************************************
 * FUNCTION: SiriusModuleHandler::powerOff
 *
 * DESCRIPTION: Powers off SXM
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: tBool
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tBool SiriusModuleHandler::powerOff() {

	ETG_TRACE_USR4(("Powering Off"));
	UpdateLogInfo("Powering Off");

	tBool bResult = FALSE;
	if(oGPIO)
	{
		bResult = oGPIO->initializeGPIOs();

		bResult &= oGPIO->setReset(FALSE);
		usleep(50000);

		bResult &= oGPIO->setPower(FALSE);
		usleep(250000);
	}

	return bResult;
}

/************************************************************************
 * FUNCTION: SiriusModuleHandler::reset
 *
 * DESCRIPTION: Resets SXM
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: tBool
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tBool SiriusModuleHandler::reset() {

	ETG_TRACE_USR4(("Resetting SXM"));
	UpdateLogInfo("Resetting SXM");

	tBool bResult = FALSE;

	bResult = powerOff();
	bResult &= powerOn();


	return bResult;

}

/************************************************************************
 * FUNCTION: SiriusModuleHandler::resetSXM
 *
 * DESCRIPTION: Module object should not be reset the first time.
 * 				Should reset on the subsequent calls.
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: void
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tVoid SiriusModuleHandler::resetSXM()
{
	if(resetCounter)
	{
		reset();
	}
	resetCounter++;
}

/************************************************************************
 * FUNCTION: SiriusModuleHandler::initializeSMSThread
 *
 * DESCRIPTION: Create a thread for SMS to work independently
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: tBool
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tBool SiriusModuleHandler::initializeSMSThread()
{
	pthread_t hSmsInitThread;
	tBool bResult = FALSE;

	int rc = pthread_create(&hSmsInitThread, NULL, pfSmsInitThreadFn, NULL);

	if(rc != 0)
	{
		ETG_TRACE_USR4(("Failed to start thread for OSAL_SMS"));
		UpdateLogInfo("Failed to start thread for OSAL_SMS");
		return false;
	}


	ETG_TRACE_USR4(("Starting thread for OSAL_SMS"));
	UpdateLogInfo("Starting thread for OSAL_SMS");

	if(! isDriverReady) {
        pthread_mutex_lock(&DriverReadyMutex);
        cond_timedwait(DriverReadyCondition, DriverReadyMutex, SMS_CONFIGURATION_WAIT_TIME);
        pthread_mutex_unlock(&DriverReadyMutex);
	}
	if( ! isDriverReady )
	{
		bResult = FALSE;
		enUpdateErrorCode = SXM_ERROR_DRIVER_INITIALIZATION_FAILURE;
		ETG_TRACE_ERR(("Timed out. Driver initialization failed"));
		UpdateLogError("Timed out. Driver initialization failed");
	}
	else
	{
		bResult = TRUE;
		ETG_TRACE_USR4(("Driver is ready"));
		UpdateLogInfo("Driver is ready");
	}

	return bResult;
}

/************************************************************************
 * FUNCTION: SiriusModuleHandler::initialize
 *
 * DESCRIPTION: Initialize SMS lib, SRM and MODULE Objects
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: SXM_ERROR_CODE_ENUM
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
SXM_ERROR_CODE_ENUM SiriusModuleHandler::initialize()
{

	SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;


	if(! initializeSMSThread() )
	{
		enUpdateErrorCode = SXM_ERROR_DRIVER_INITIALIZATION_FAILURE;
		ETG_TRACE_ERR(("SXM_ERROR_INITIALIZATION_FAILURE"));
		UpdateLogError("SXM_ERROR_INITIALIZATION_FAILURE");
		return enUpdateErrorCode;
	}

	tBool lreplace;
	tBool bResult = FALSE;
	for (int iRetry=1; iRetry<=10; iRetry++)
	{
			eReturnCode = SMS.eInitialize(
								SMS_WORKING_DIR, 		// Path solely used by SMS
								"", 					// Config file location (Directory)
								"",						// "sms.cfg", Config file name
								&lreplace
							);

			if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
			{
				bResult = TRUE;
				break;
			}
			else
			{
				ETG_TRACE_ERR(("Initialization try %d failed. Reason %d\nTrying again...\n", iRetry, eReturnCode));
				UpdateLogError("Initialization try %d failed. Reason %d\nTrying again...\n", iRetry, eReturnCode);
			}
	}
	if(bResult == FALSE)
	{
		enUpdateErrorCode = SXM_ERROR_INITIALIZATION_FAILURE;
		ETG_TRACE_ERR(("SMS Library Initialization failed"));
		UpdateLogError("SMS Library Initialization failed");
		return enUpdateErrorCode;
	}
	if(! (initializeSRM_Object() && initializeModule_Object()) )
	{
		return enUpdateErrorCode;
	}

	MODULE.eModifyRegisteredEventMask(_hModule, (MODULE_EVENT_MASK)MODULE_OBJECT_EVENT_ALL, SMSAPI_MODIFY_EVENT_MASK_START);

	return enUpdateErrorCode;
}

/************************************************************************
 * FUNCTION: SiriusModuleHandler::deInitialize
 *
 * DESCRIPTION: Relaese Module and SRM object
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: void
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tVoid SiriusModuleHandler::deInitialize() {

   ETG_TRACE_USR4(("De-initialize started"));
   UpdateLogInfo("De-initialize started");

	if (_hModule != MODULE_INVALID_OBJECT) {
		MODULE.vRelease(_hModule);
		_hModule = MODULE_INVALID_OBJECT;
	}

	if (_hSRM != SRM_INVALID_OBJECT) {
		SRM.vRelease(_hSRM);
		_hSRM = SRM_INVALID_OBJECT;
	}

	tVoid *pvArg = 0;
	bUnRegisterDrivers(pvArg);

}

/************************************************************************
 * FUNCTION: SiriusModuleHandler::logFirmwareAndModuleDetails
 *
 * DESCRIPTION: log details about firmware and hardware module
 *
 * PARAMETER:
 * 				string
 *
 * RETURNVALUE: tVoid
 *
 * HISTORY:
 * 	20.6.2016 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tVoid SiriusModuleHandler::logFirmwareAndModuleDetails(string l_firmwareFilePath)
{
	if( !oVersionUtility )
	{
		ETG_TRACE_ERR(("No Version Utility Available"));
		UpdateLogError("No Version Utility Available");
		return ;
	}
	string sFirmwareHardwareType = "";
	string sEarliestVersion = "";
	string sLatestVersion = "";

	oVersionUtility->getSoftwareVersionOfModule(_hModule, softwareVersion);

	oVersionUtility->getDetailsOfFirmware(firmwareFilePath, firmwareFileVersion,
			sFirmwareHardwareType, sEarliestVersion, sLatestVersion);


	ETG_TRACE_USR4(("Updating from %s", softwareVersion.c_str()));
	ETG_TRACE_USR4(("Updating to %s", firmwareFileVersion.c_str()));
	UpdateLogInfo("Updating from %s to %s", softwareVersion.c_str(), firmwareFileVersion.c_str());

	ETG_TRACE_USR4(("SXM_HW_TYPE  = %s", sFirmwareHardwareType.c_str()));
	ETG_TRACE_USR4(("SERIAL_VER   = %s", getSerialVersionOfModule().c_str()));
	ETG_TRACE_USR4(("SXM_HW_VER   = %s", getHardwareVersionOfModule().c_str()));
	ETG_TRACE_USR4(("SXM_SW_VER   = %s", getSoftwareVersionOfModule().c_str()));
	ETG_TRACE_USR4(("EARLIEST_VER = %s", sEarliestVersion.c_str()));
	ETG_TRACE_USR4(("LATEST_VER   = %s", sLatestVersion.c_str()));

	UpdateLogInfo("SXM_HW_TYPE  = %s", sFirmwareHardwareType.c_str());
	UpdateLogInfo("SERIAL_VER   = %s", getSerialVersionOfModule().c_str());
	UpdateLogInfo("SXM_HW_VER   = %s", getHardwareVersionOfModule().c_str());
	UpdateLogInfo("SXM_SW_VER   = %s", getSoftwareVersionOfModule().c_str());
	UpdateLogInfo("EARLIEST_VER = %s", sEarliestVersion.c_str());
	UpdateLogInfo("LATEST_VER   = %s", sLatestVersion.c_str());

}

/************************************************************************
 * FUNCTION: SiriusModuleHandler::firmwareUpdate
 *
 * DESCRIPTION: Update the firmware and get progress information
 *
 * PARAMETER:
 * 				string, cbProgressInformation, tBool
 *
 * RETURNVALUE: SXM_ERROR_CODE_ENUM
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
SXM_ERROR_CODE_ENUM SiriusModuleHandler::firmwareUpdate(string l_firmwareFilePath, tBool bForceUpdate, cbProgressInformation cbProgressUpdate) {

	_cbProgressUpdate = cbProgressUpdate;
	firmwareFilePath = l_firmwareFilePath;


	logFirmwareAndModuleDetails(l_firmwareFilePath);

	/*
	 * No need to disallow downgrade. At least not yet.

		if (FALSE == bAllowDowngrade) {
			if (FALSE == isFirmwareNewer()) {
				ETG_TRACE_ERR(("Trying to down-grade. Down-grade not allowed."));
				UpdateLogError("Trying to down-grade. Down-grade not allowed.");
				return SXM_ERROR_DOWNGRADE_NOT_ALLOWED;
			}
		}
	*/

	// If versions are identical and nobody forces, skip the update...
	if(isFimwareAndModuleVersionSame()  && bForceUpdate == FALSE)
	{
		ETG_TRACE_USR4(("SKIPPING UPDATE. No need to update same version."));
		UpdateLogInfo("SKIPPING UPDATE. No need to update same version.");
		return SXM_NO_ERROR;
	}


	SMSAPI_RETURN_CODE_ENUM enSmsRes = MODULE.eFirmwareUpdate(_hModule,
			firmwareFilePath.c_str());

	if(enSmsRes == SMSAPI_RETURN_CODE_MODULE_FWUPDATE_FILE_VERSION_INVALID)
	{
		enUpdateErrorCode = SXM_ERROR_FILE_VERSION_INVALID;
		ETG_TRACE_ERR(("SXM_ERROR_FILE_VERSION_INVALID"));
		UpdateLogError("SXM_ERROR_FILE_VERSION_INVALID");
	}
	else if(enSmsRes == SMSAPI_RETURN_CODE_MODULE_FWUPDATE_ALREADY_IN_PROGRESS)
	{
		enUpdateErrorCode = SXM_ERROR_UPDATE_IN_PROGRESS;
		ETG_TRACE_ERR(("SMSAPI_RETURN_CODE_MODULE_FWUPDATE_ALREADY_IN_PROGRESS"));
		UpdateLogError("SMSAPI_RETURN_CODE_MODULE_FWUPDATE_ALREADY_IN_PROGRESS");
	}
	else if( enSmsRes != SMSAPI_RETURN_CODE_ERROR)
	{
        tBool isTimedOut = true;
		pthread_mutex_lock(&UpdateCompleteMutex);
	    isTimedOut = (cond_timedwait(UpdateCompleteCondition, UpdateCompleteMutex, SMS_UPDATE_WAIT_TIME) == ETIMEDOUT);
	    pthread_mutex_unlock(&UpdateCompleteMutex);
	    
		if(isTimedOut)
		{
			enUpdateErrorCode = SXM_ERROR_FIRMWARE_UPDATE_FAILED;
			ETG_TRACE_ERR(("Update timed out"));
			UpdateLogError("Update timed out");
		}

		if(enUpdateState == SXM_STATE_UPDATE_FAILURE)
		{
			enUpdateErrorCode = SXM_ERROR_FIRMWARE_UPDATE_FAILED;
			ETG_TRACE_ERR(("Update Failure"));
			UpdateLogError("Update Failure");
		}
		else if(enUpdateState == SXM_STATE_UPDATE_SUCCESS)
		{
			enUpdateErrorCode = SXM_NO_ERROR;
			ETG_TRACE_USR4(("Update Successful"));
			UpdateLogInfo("Update Successful");
		}

	}
	else
	{
		enUpdateErrorCode = SXM_ERROR_FIRMWARE_UPDATE_FAILED;
		ETG_TRACE_ERR(("Update API Failed %d", enSmsRes));
		UpdateLogError("Update API Failed %d", enSmsRes);
	}


	return enUpdateErrorCode;

}
/************************************************************************
 * FUNCTION: SiriusModuleHandler::getSerialVersionOfModule
 *
 * DESCRIPTION: Get Serial number of module
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: string
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
string SiriusModuleHandler::getSerialVersionOfModule()
{
	if(oVersionUtility)
		oVersionUtility->getSerialVersionOfModule(_hModule, serialVersion);
	else
		serialVersion = "";

	return serialVersion;
}
/************************************************************************
 * FUNCTION: SiriusModuleHandler::getHardwareVersionOfModule
 *
 * DESCRIPTION: Get Hardware version of module
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: string
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
string SiriusModuleHandler::getHardwareVersionOfModule()
{
	if(oVersionUtility)
		oVersionUtility->getHardwareVersionOfModule(_hModule, hardwareVersion);
	else
		hardwareVersion = "";

	return hardwareVersion;
}
/************************************************************************
 * FUNCTION: SiriusModuleHandler::getHardwareTypeOfModule
 *
 * DESCRIPTION: Get Hardware type of module
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: string
 *
 * HISTORY:
 * 	20.06.2016 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
string SiriusModuleHandler::getHardwareTypeOfModule()
{
	if(oVersionUtility)
		oVersionUtility->getHardwareTypeOfModule(_hModule, hardwareType);
	else
		hardwareType = "";

	return hardwareType;
}
/************************************************************************
 * FUNCTION: SiriusModuleHandler::getSoftwareVersionOfModule
 *
 * DESCRIPTION: Get Software version of module
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: string
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
string SiriusModuleHandler::getSoftwareVersionOfModule()
{
	if(oVersionUtility)
		oVersionUtility->getSoftwareVersionOfModule(_hModule, softwareVersion);
	else
		softwareVersion = "";

	return softwareVersion;
}

/************************************************************************
 * FUNCTION: SiriusModuleHandler::getSoftwareVersionOfFirmware
 *
 * DESCRIPTION: Get software version of firmware file
 *
 * PARAMETER:
 * 				string
 *
 * RETURNVALUE: string
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
string SiriusModuleHandler::getSoftwareVersionOfFirmware(string l_firmwareFilePath)
{
	if(oVersionUtility)
		oVersionUtility->getSoftwareVersionOfFirmware(l_firmwareFilePath, firmwareFileVersion);
	else
		firmwareFileVersion = "";

	return firmwareFileVersion;
}

/************************************************************************
 * FUNCTION: SiriusModuleHandler::getHardwareTypeFromFirmware
 *
 * DESCRIPTION: Get compatible hardware type from firmware file
 *
 * PARAMETER:
 * 				string
 *
 * RETURNVALUE: string
 *
 * HISTORY:
 * 	20.06.2016 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
string SiriusModuleHandler::getHardwareTypeFromFirmware(string l_firmwareFilePath)
{
	if(oVersionUtility)
		oVersionUtility->getHardwareTypeFromFirmware(l_firmwareFilePath, firmwareHardwareType);
	else
		firmwareHardwareType = "";

	return firmwareHardwareType;
}

extern "C"
{
/************************************************************************
 * FUNCTION: fc_sxm_vSetPatchedPrio
 *
 * DESCRIPTION: To patch thread prio. But not needed in UpdateCtrlLib.
 *
 * PARAMETER:
 * 				const char*, OSAL_OBJECT_HDL, OSAL_TASK_PRIORITY_ENUM*
 *
 * RETURNVALUE: void
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tVoid fc_sxm_vSetPatchedPrio(tCString szThreadName, OSAL_OBJECT_HDL tThreadId,
		OSAL_TASK_PRIORITY_ENUM *piPrioVal) {

	(tCString) szThreadName;
	(OSAL_OBJECT_HDL) tThreadId;
	(OSAL_TASK_PRIORITY_ENUM*) piPrioVal;

}

/************************************************************************
 * FUNCTION: fc_sxm_vResetX65
 *
 * DESCRIPTION: Called by SMS to reset SXM before and after update
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: void
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tVoid fc_sxm_vResetX65() {
	SiriusModuleHandler::instance()->resetSXM();
}

/************************************************************************
 * FUNCTION: fc_sxm_vSwitchOnX65
 *
 * DESCRIPTION: Called by SMS to power on SXM
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: void
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tVoid fc_sxm_vSwitchOnX65() {
	SiriusModuleHandler::instance()->powerOn();
}

/************************************************************************
 * FUNCTION: fc_sxm_vSwitchOffX65
 *
 * DESCRIPTION: Called by SMS to power off SXM
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: void
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tVoid fc_sxm_vSwitchOffX65() {
	SiriusModuleHandler::instance()->powerOff();
}

/************************************************************************
 * FUNCTION: fc_sxm_bIsSMSLIB_CvOn
 *
 * DESCRIPTION: To check if UpdateCtrlLib handles critical voltage
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: unsigned char
 *
 * HISTORY:
 * 	04.12.2015 	Rev. 1.0 Logesh
 * 				Initial Revision.
************************************************************************/
tU8 fc_sxm_bIsSMSLIB_CvOn() {
		return FALSE;
}

} // extern


