/************************************************************************
 * File: fbhmi_KmsHandler.cpp
 * SW-Component: ASCII HMI for recovery download
 *
 * Description:
 *   This is a singleton class used to initialize and use the Linux
 *   framebuffer for displaying characters.
 * Author:
 *   Gururaj.B@in.bosch.com
 * Copyright:
 *   Robert Bosch Engineering and Business Solutions Ltd, Bangalore.
 *
 * History:
 * 21.05.2013 - Initial version - Gururaj.B@in.bosch.com
 ************************************************************************/

//--------------------------------------------------------------------------
// includes
//--------------------------------------------------------------------------
#include <iostream>
using namespace std;
#include <string.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/mman.h>
#include "fbhmi_KmsHandler.h"
#include <linux/kd.h>
#include <stdlib.h>
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>


/************************************************************************
 * FUNCTION: fbhmi_KmsHandler::fbhmi_KmsHandler()
 *
 * DESCRIPTION: private Constructor
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
fbhmi_KmsHandler::fbhmi_KmsHandler()
{
  DEBUG_TRACE("Enter");
  // Initialize framebuffer device
  m_s32FdKms=-1;
  m_DevList=NULL;
  bInitKmsBuffer(FBHMI_KMS_DEVICE);
}

/************************************************************************
 * FUNCTION: fbhmi_KmsHandler::~fbhmi_KmsHandler()
 *
 * DESCRIPTION: Destructor
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
fbhmi_KmsHandler::~fbhmi_KmsHandler()
{
  DEBUG_TRACE("Enter");
  bCleanupKmsBuffer();
  m_pGfxHandle=NULL;
}
;

/************************************************************************
 * FUNCTION: fbhmi_KmsHandler::pCreateInstance()
 *
 * DESCRIPTION: Static function to return instance of the class.
 *
 * PARAMETER: None
 *
 * RETURNVALUE: Pointer to instance of the class
 *************************************************************************/
fbhmi_GfxHandler* fbhmi_KmsHandler::pCreateInstance(void)
{
  DEBUG_TRACE("Enter");
  if(NULL==fbhmi_GfxHandler::m_pGfxHandle)
  {
    fbhmi_GfxHandler::m_pGfxHandle=new fbhmi_KmsHandler();
  }
  DEBUG_TRACE("Leave");
  return fbhmi_GfxHandler::m_pGfxHandle;
}

/************************************************************************
 * FUNCTION: fbhmi_KmsHandler::bInitKmsBuffer(_tChar *fbName)
 *
 * DESCRIPTION: Initialize the framebuffer device and structures
 *
 * PARAMETER: Name of the framebuffer device
 *
 * RETURNVALUE: 1 = SUCCESS, 0 = FAILURE
 *************************************************************************/
_tBool fbhmi_KmsHandler::bInitKmsBuffer(const _tChar *pFbName)
{
  DEBUG_TRACE("Enter");
  uint64_t  has_dumb;

  // Disable cursor on framebuffer console
  vDisableCursor();

  // Open the frame buffer device
  if((m_s32FdKms=open(pFbName, O_RDWR|O_CLOEXEC))<0)
  {
    DEBUG_TRACE("failed to open drm device '%s'", pFbName);
    return false;
  }

  if(drmGetCap(m_s32FdKms, DRM_CAP_DUMB_BUFFER, &has_dumb)<0 || !has_dumb)
  {
    DEBUG_TRACE("drm device '%s' does not support dumb buffers", pFbName);
    close(m_s32FdKms);
    return false;
  }

  // Get Screen info from driver
  if(!bGetKmsInfo())
  {
    return false;
  }

  // perform actual modesetting on each found connector+CRTC
  m_DevList->saved_crtc=drmModeGetCrtc(m_s32FdKms, m_DevList->crtc);
  if(drmModeSetCrtc(m_s32FdKms, m_DevList->crtc, m_DevList->fb, 0, 0, &m_DevList->conn, 1, &m_DevList->mode))
    return false;

  // Check number of bits per pixel supported.
  if(m_stGfxVar.bits_per_pixel!=15 && m_stGfxVar.bits_per_pixel!=16 && m_stGfxVar.bits_per_pixel!=24 && m_stGfxVar.bits_per_pixel!=32)
  {
    return false;
  }

  if(m_stGfxVar.bits_per_pixel==15)
    m_u32GfxPixelBytes=2;
  else
    m_u32GfxPixelBytes=m_stGfxVar.bits_per_pixel/8;

  m_u64GfxMemOffset=0;
  m_pGfxMemReal=m_DevList->map;

  if(-1L==(long)m_pGfxMemReal)
    return false;

  m_pGfxMemory=m_pGfxMemReal+m_u64GfxMemOffset;

  //assign x-resolution and y-resolution
  m_u32YRes=m_stGfxVar.yres;
  m_u32XRes=m_stGfxVar.xres;

  //Clear the screen
  vClearScreen();
  return true;
}

/************************************************************************
 * FUNCTION: fbhmi_KmsHandler::bGetKmsInfo(void)
 *
 * DESCRIPTION: Get inormation about the display and wrap it up in
 *              structures used for framebuffers because the rest of the
 *              code already uses them.
 *
 * PARAMETER: void
 *
 * RETURNVALUE: _tBool
 *************************************************************************/
_tBool fbhmi_KmsHandler::bGetKmsInfo(void)
{
  DEBUG_TRACE("Enter");
  drmModeRes        *res;
  drmModeConnector  *conn;
  unsigned int      i;
  struct kms_dev    *dev;
  int ret;

  // retrieve resources
  res=drmModeGetResources(m_s32FdKms);
  if(!res)
  {
    DEBUG_TRACE("cannot retrieve DRM resources (%d): %m", errno);
    return false;
  }

  // iterate all connectors
  for(i=0; i<res->count_connectors; ++i)
  {
    // get information for each connector
    conn=drmModeGetConnector(m_s32FdKms, res->connectors[i]);
    if(!conn)
    {
      DEBUG_TRACE("cannot retrieve DRM connector %u:%u (%d): %m", i, res->connectors[i], errno);
      continue;
    }

    // create a device structure
    dev=(struct kms_dev *)malloc(sizeof(*dev));
    memset(dev, 0, sizeof(*dev));
    dev->conn=conn->connector_id;

    // call helper function to prepare this connector
    ret=bSetupKmsDev(res, conn, dev);
    drmModeFreeConnector(conn);
    if(!ret)
    {
      free(dev);
DEBUG_TRACE("bSetupKmsDev failed");
      continue;
    }

    // free connector data and link device into global list
    dev->next=m_DevList;
    m_DevList=dev;
  }

  // free resources again
  drmModeFreeResources(res);

  // Note that while we acquire a list of devices, for now it is sufficient to use the first one only
  bzero(&m_stGfxFix, sizeof(m_stGfxFix));
  strcpy(m_stGfxFix.id, "drm/kms");
  m_stGfxFix.smem_len=m_DevList->stride*m_DevList->height;
  m_stGfxFix.type=FB_TYPE_PACKED_PIXELS;
  m_stGfxFix.line_length=m_DevList->stride;

  bzero(&m_stGfxVar, sizeof(m_stGfxVar));
  m_stGfxVar.xres=m_DevList->width;
  m_stGfxVar.yres=m_DevList->height;
  m_stGfxVar.bits_per_pixel=32;
  m_stGfxVar.red.offset=16;
  m_stGfxVar.red.length=8;
  //m_stGfxVar.red.msb_right;
  m_stGfxVar.green.offset=8;
  m_stGfxVar.green.length=8;
  m_stGfxVar.blue.offset=0;
  m_stGfxVar.blue.length=8;
  m_stGfxVar.transp.offset=24;
  m_stGfxVar.transp.length=8;

  DEBUG_TRACE("Leave");
  return true;
}



/************************************************************************
 * FUNCTION: fbhmi_KmsHandler::bSetupKmsDev(drmModeRes *res, drmModeConnector *conn, struct kms_dev *dev)
 *
 * DESCRIPTION: Setup a kms device for rendering.
 *
 * PARAMETER: drmModeRes *res:        drm resource, see drm lib
 *            drmModeConnector *conn: drm connector, see drm lib
 *            struct kms_dev *dev:    device to initialize, discovered in bGetKmsInfo
 *
 * RETURNVALUE: _tBool
 *************************************************************************/
_tBool fbhmi_KmsHandler::bSetupKmsDev(drmModeRes *res, drmModeConnector *conn, struct kms_dev *dev)
{
  DEBUG_TRACE("Enter");

  // check if a monitor is connected
  if(conn->connection!=DRM_MODE_CONNECTED)
  {
    DEBUG_TRACE("ignoring unused connector %u", conn->connector_id);
    return false;
  }

  // check if there is at least one valid mode
  if(conn->count_modes==0)
  {
    DEBUG_TRACE("no valid mode for connector %u", conn->connector_id);
    return false;
  }

  // copy the mode information into our device structure
  memcpy(&dev->mode, &conn->modes[0], sizeof(dev->mode));
  dev->width=conn->modes[0].hdisplay;
  dev->height=conn->modes[0].vdisplay;
  DEBUG_TRACE("mode for connector %u is %ux%u", conn->connector_id, dev->width, dev->height);

  // find a crtc for this connector
  if(!bFindCrtc(res, conn, dev))
  {
    DEBUG_TRACE("no valid crtc for connector %u", conn->connector_id);
    return false;
  }

  // create a framebuffer for this CRTC
  if(!bCreateFb(dev))
  {
    DEBUG_TRACE("cannot create framebuffer for connector %u", conn->connector_id);
    return false;
  }

  DEBUG_TRACE("Leave");
  return true;
}



/************************************************************************
 * FUNCTION: fbhmi_KmsHandler::bFindCrtc(drmModeRes *res, drmModeConnector *conn, struct kms_dev *dev)
 *
 * DESCRIPTION: Try to find a CRTC for the given device.
 *
 * PARAMETER: drmModeRes *res:        drm resource, see drm lib
 *            drmModeConnector *conn: drm connector, see drm lib
 *            struct kms_dev *dev:    device to find a CRTC for, initialized in bSetupKmsDev
 *
 * RETURNVALUE: _tBool
 *************************************************************************/
_tBool fbhmi_KmsHandler::bFindCrtc(drmModeRes *res, drmModeConnector *conn, struct kms_dev *dev)
{
  DEBUG_TRACE("Enter");
  drmModeEncoder  *enc;
  unsigned int    i, j;
  int32_t         crtc;
  struct kms_dev  *iter;

  // first try the currently conected encoder+crtc
  if(conn->encoder_id)
    enc=drmModeGetEncoder(m_s32FdKms, conn->encoder_id);
  else
    enc=NULL;

  if(enc)
  {
    if(enc->crtc_id)
    {
      crtc=enc->crtc_id;
      for(iter=m_DevList; iter; iter=iter->next)
      {
        if(iter->crtc==crtc)
        {
          crtc=-1;
          break;
        }
      }

      if(crtc>=0)
      {
        drmModeFreeEncoder(enc);
        dev->crtc=crtc;
        DEBUG_TRACE("Leave");
        return true;
      }
    }

    drmModeFreeEncoder(enc);
  }

  // If the connector is not currently bound to an encoder or if the
  // encoder+crtc is already used by another connector (actually unlikely
  // but lets be safe), iterate all other available encoders to find a
  // matching CRTC.
  for(i=0; i<conn->count_encoders; ++i)
  {
    enc=drmModeGetEncoder(m_s32FdKms, conn->encoders[i]);
    if(!enc)
    {
      DEBUG_TRACE("cannot retrieve encoder %u:%u (%d): %m", i, conn->encoders[i], errno);
      continue;
    }

    // iterate all global CRTCs
    for(j=0; j<res->count_crtcs; ++j)
    {
      // check whether this CRTC works with the encoder
      if(!(enc->possible_crtcs&(1<<j)))
        continue;

      // check that no other device already uses this CRTC
      crtc=res->crtcs[j];
      for(iter=m_DevList; iter; iter=iter->next)
      {
        if(iter->crtc==crtc)
        {
          crtc=-1;
          break;
        }
      }

      // we have found a CRTC, so save it and return
      if(crtc>=0)
      {
        drmModeFreeEncoder(enc);
        dev->crtc=crtc;
        DEBUG_TRACE("Leave");
        return true;
      }
    }

    drmModeFreeEncoder(enc);
  }

  DEBUG_TRACE("cannot find suitable CRTC for connector %u", conn->connector_id);
  return false;
}


/************************************************************************
 * FUNCTION: fbhmi_KmsHandler::bCreateFb(struct kms_dev *dev)
 *
 * DESCRIPTION: Create a dumb framebuffer for the given device.
 *
 * PARAMETER: struct kms_dev *dev:    device to create a framebuffer for, after a CRTC was assigend to it in bFindCrtc
 *
 * RETURNVALUE: _tBool
 *************************************************************************/
_tBool fbhmi_KmsHandler::bCreateFb(struct kms_dev *dev)
{
  DEBUG_TRACE("Enter");
  struct drm_mode_create_dumb   creq;
  struct drm_mode_destroy_dumb  dreq;
  struct drm_mode_map_dumb      mreq;

  // create dumb buffer
  memset(&creq, 0, sizeof(creq));
  creq.width=dev->width;
  creq.height=dev->height;
  creq.bpp=32;
  if(drmIoctl(m_s32FdKms, DRM_IOCTL_MODE_CREATE_DUMB, &creq)<0)
  {
    DEBUG_TRACE("cannot create dumb buffer (%d): %m", errno);
    return false;
  }
  dev->stride=creq.pitch;
  dev->size=creq.size;
  dev->handle=creq.handle;

  // create framebuffer object for the dumb-buffer
  if(drmModeAddFB(m_s32FdKms, dev->width, dev->height, 24, 32, dev->stride, dev->handle, &dev->fb))
  {
    DEBUG_TRACE("cannot create framebuffer (%d): %m", errno);
    goto err_destroy;
  }

  // prepare buffer for memory mapping
  memset(&mreq, 0, sizeof(mreq));
  mreq.handle=dev->handle;
  if(drmIoctl(m_s32FdKms, DRM_IOCTL_MODE_MAP_DUMB, &mreq))
  {
    DEBUG_TRACE("cannot map dumb buffer (%d): %m", errno);
    goto err_fb;
  }

  // perform actual memory mapping
  if((dev->map=(uint8_t *)mmap(0, dev->size, PROT_READ|PROT_WRITE, MAP_SHARED, m_s32FdKms, mreq.offset))==MAP_FAILED)
  {
    DEBUG_TRACE("cannot mmap dumb buffer (%d): %m", errno);
    goto err_fb;
  }

  // clear the framebuffer to 0
  memset(dev->map, 0, dev->size);

  DEBUG_TRACE("Leave");
  return true;

err_fb:
  drmModeRmFB(m_s32FdKms, dev->fb);
err_destroy:
  memset(&dreq, 0, sizeof(dreq));
  dreq.handle=dev->handle;
  drmIoctl(m_s32FdKms, DRM_IOCTL_MODE_DESTROY_DUMB, &dreq);
  return false;
}



/************************************************************************
 * FUNCTION: fbhmi_KmsHandler::bCleanupFrameBuffer()
 *
 * DESCRIPTION: De-Initialize and clean up kms structures
 *
 * PARAMETER: None
 *
 * RETURNVALUE: 1 = SUCCESS, 0 = FAILURE
 *************************************************************************/
_tBool fbhmi_KmsHandler::bCleanupKmsBuffer(void)
{
  DEBUG_TRACE("Enter");
/*  munmap(m_pFbMemReal, m_stFbFix.smem_len+m_u64FbMemOffset);
  close (m_s32FdFrameBuffer);*/

  struct kms_dev                *iter;
  struct drm_mode_destroy_dumb  dreq;

  while(m_DevList)
  {
    // remove from global list
    iter=m_DevList;
    m_DevList=iter->next;

    // restore saved CRTC configuration
    drmModeSetCrtc(m_s32FdKms, iter->saved_crtc->crtc_id, iter->saved_crtc->buffer_id, iter->saved_crtc->x, iter->saved_crtc->y, &iter->conn, 1, &iter->saved_crtc->mode);
    drmModeFreeCrtc(iter->saved_crtc);

    // unmap buffer
    munmap(iter->map, iter->size);

    // delete framebuffer
    drmModeRmFB(m_s32FdKms, iter->fb);

    // delete dumb buffer
    memset(&dreq, 0, sizeof(dreq));
    dreq.handle=iter->handle;
    drmIoctl(m_s32FdKms, DRM_IOCTL_MODE_DESTROY_DUMB, &dreq);

    // free allocated memory
    free(iter);
  }

  close(m_s32FdKms);

  return true;
}
