/************************************************************************
* File: fbhmi_FbHmiManager.cpp
* SW-Component: Text HMI for Recovery Downloader
*
* Description:
*   This is a basic window manager implementation for
*   framebuffer based text hmi display.
* Author:
*   Gururaj.B@in.bosch.com
* Copyright:
*   Robert Bosch Engineering and Business Solutions Ltd, Bangalore.
*
* History:
* 30.05.2013 - Initial version - Gururaj.B@in.bosch.com
* 27.02.2014 - Rohit C R - Added vProcessFbHMIEvent() to process fb hmi event.
* 08.09.2014 - Apoorva K R - Added bUpdateTitle() and FBHMI_EVNT_UPDATE_TITLE
* 							 to process the screen title text change
***********************************************************************/
#include "fbhmi_FbHmiManager.h"
#include <signal.h>
#include <pthread.h>

fbhmi_QueueHandler* fbhmi_FbHmiManager::m_pQueueHandle = NULL;

/************************************************************************
* FUNCTION: fbhmi_FbHmiManager::fbhmi_FbHmiManager()
*
* DESCRIPTION: Constructor.creates HMI queue
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
fbhmi_FbHmiManager::fbhmi_FbHmiManager(_tU32 u32Width, _tU32 u32Height, _tU32 u32BdrWidth, std::string	strTitle,
													_tU32 u32FgColor, _tU32 u32BgColor, _tU32 u32BdrColor,_tU32 u32UpdateType,fbhmiDisplayType u32DisplayType)
{

	DEBUG_TRACE("%s", __func__);
	// Create a default screen with default parameters
	m_bManagerInit = false;
	m_u32xRes = 0;
	m_u32yRes = 0;
	u32Width = u32Width;
	u32Height = u32Height;
	m_pQueueHandle = new fbhmi_QueueHandler();
	if (NULL == m_pQueueHandle) {
		DEBUG_TRACE("Queue creation failed");
		return;
	}

	fbhmi_Surface::DisplayType=u32DisplayType;
	fbhmi_Surface* surface = fbhmi_Surface::pGetInstance();
	if (NULL == surface) {
		DEBUG_TRACE("Surface pointer is NULL");
		delete m_pQueueHandle;
		return;
	}

	m_u32xRes = surface->u32GetXRes();
	m_u32yRes = surface->u32GetYRes();

	if( ( m_u32xRes >= (2*FBHMI_SCREEN_DEF_WIDTH) )
		&&
		( m_u32yRes >= (2*FBHMI_SCREEN_DEF_HEIGHT) ) )
	{
		m_pFontHandle = fbhmi_Font::pclCreateFromFile(FBHMI_10x20_FONT_FILE);
	}
	else
	{
		m_pFontHandle = fbhmi_Font::pclCreateFromFile(FBHMI_6x10_FONT_FILE);
	}

	if(NULL == m_pFontHandle)
	{
		DEBUG_TRACE("%s", "Font creation failed");
		delete m_pQueueHandle;
		return;
	}


	surface->vSetFont(m_pFontHandle);

	if(NULL != m_pQueueHandle)
	{
		fbhmi_Screen* pScreen = new fbhmi_Screen(
			FBHMI_SCREEN_DEF_ID, FBHMI_SCREEN_DEF_X, FBHMI_SCREEN_DEF_Y,\
			surface->u32GetYRes(), surface->u32GetXRes(), u32BdrWidth, u32BgColor,\
			u32FgColor, u32BdrColor, FBHMI_WIDGET_DEF_FILCOLOR,\
			strTitle, FBHMI_WIDGET_DEF_ALIGN,\
			FBHMI_WIDGET_DEF_VISIBLE);
		fbhmi_Screen *p1Screen;

		p1Screen = pScreen;
		if(NULL != pScreen) {
			m_vecScreenList.push_back(pScreen);
			if(u32UpdateType == FBHMI_NON_OTA_UPDATE) {
			pScreen->vDrawWidget();
			}
		}
		// Initialize HMI thread
		m_bManagerInit = bInitHMIManager();
	}
	DEBUG_TRACE("%s", __func__);
}

/************************************************************************
* FUNCTION: fbhmi_FbHmiManager::bInitHMIManager()
* DESCRIPTION: Initializes manager
* PARAMETER:
* RETURNVALUE: bool
*************************************************************************/
_tBool fbhmi_FbHmiManager::bInitHMIManager()
{
	DEBUG_TRACE("%s", __func__);
	//Create the HMI thread
	if(pthread_create(&m_stFbHmiThread, NULL, &fbhmi_FbHmiManager::vFbHMIThreadCallback, this))
		return false;
	return true;
}

/************************************************************************
* FUNCTION: fbhmi_FbHmiManager::bIsManagerInit()
* DESCRIPTION: Returns status of HMI thread creation
* PARAMETER: None*
* RETURNVALUE: bool
*************************************************************************/
_tBool fbhmi_FbHmiManager::bIsManagerInit() const
{
	DEBUG_TRACE("%s", __func__);
	return m_bManagerInit;
}

/************************************************************************
* FUNCTION: fbhmi_FbHmiManager::~fbhmi_FbHmiManager()
*
* DESCRIPTION: Destructor
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
fbhmi_FbHmiManager::~fbhmi_FbHmiManager()
{
	DEBUG_TRACE("%s", __func__);
	pthread_kill(m_stFbHmiThread, SIGTERM);
	for(std::vector<fbhmi_Screen*>::iterator pIt = m_vecScreenList.begin(); pIt != m_vecScreenList.end(); pIt++)
	{
		delete *pIt;
	}
	m_vecScreenList.clear();

	if(NULL != m_pFontHandle){
		delete m_pFontHandle;
		m_pFontHandle = NULL;
	}

	m_pSurfaceHandle->vDeleteSurfaceHandler();
	m_pSurfaceHandle = NULL;

	if (NULL != m_pQueueHandle)
		delete m_pQueueHandle;
}


/************************************************************************
* FUNCTION: fbhmi_FbHmiManager::bCreateLabel
* DESCRIPTION: posts a request to HMI thread to create a label
* PARAMETER: *
* RETURNVALUE: bool
*************************************************************************/
_tBool fbhmi_FbHmiManager::bCreateLabel(_tU16 u16ScreenId, _tU16 u16WidgetId,
													 _tU32 u32x, _tU32 u32y, _tU32 u32Width,
													 _tU32 u32Height, _tU32 u32BdrWidth,
													 _tU32 u32FgColor, _tU32 u32BgColor,
													 _tU32 u32BdrColor, string strText,
													 _tU8 u8Align, _tBool bVisible) const
{
	DEBUG_TRACE("%s", __func__);
	fbhmi_WidgetReq* pReq =
		new fbhmi_WidgetReq(FBHMI_EVNT_CREATE_LABEL, u16ScreenId, u16WidgetId);
	if (NULL == pReq) {
		DEBUG_TRACE("WidgetRequet is NULL");
		return false;
	}

	pReq->m_u32x			= u32x;
	pReq->m_u32y			= u32y;
	pReq->m_u32height		= u32Height;
	pReq->m_u32width		= u32Width;
	pReq->m_u32borderWidth	= u32BdrWidth;
	pReq->m_u32bgColor		= u32BgColor;
	pReq->m_u32fgColor		= u32FgColor;
	pReq->m_u32borderColor	= u32BdrColor;
	pReq->m_u32fillColor	= FBHMI_WIDGET_DEF_FILCOLOR;
	pReq->m_strText			= strText;
	pReq->m_u32textAlign	= u8Align;
	pReq->m_u32value		= FBHMI_WIDGET_DEF_PBARVAL;
	pReq->m_u32maxValue		= FBHMI_WIDGET_DEF_PBARMAX;
	pReq->m_bvisibility		= bVisible;

	return (m_pQueueHandle->bPushQueue(pReq));
}

/************************************************************************
* FUNCTION: fbhmi_FbHmiManager::bCreateProgressBar
* DESCRIPTION: posts a request to HMI thread to create a label
* PARAMETER: *
* RETURNVALUE: bool
*************************************************************************/
_tBool fbhmi_FbHmiManager::bCreateProgressBar(_tU16 u16ScreenId, _tU16 u16WidgetId,
															 _tU32 u32x, _tU32 u32y, _tU32 u32Width,
															 _tU32 u32Height, _tU32 u32BdrWidth,
															 _tU32 u32FgColor, _tU32 u32BgColor,
															 _tU32 u32BdrColor, string strText,
															 _tU8 u8Align, _tBool bVisible,
															 _tU32 u32Value, _tU32 u32MaxValue,
															 _tU32 u32FillColor) const
{
	DEBUG_TRACE("%s", __func__);
	fbhmi_WidgetReq* pReq =
		new fbhmi_WidgetReq(FBHMI_EVNT_CREATE_PROGRESSBAR, u16ScreenId, u16WidgetId);
	if (NULL == pReq) {
		DEBUG_TRACE("WidgetRequet pointer is NULL");
		return false;
	}

	pReq->m_u32x				= u32x;
	pReq->m_u32y				= u32y;
	pReq->m_u32height			= u32Height;
	pReq->m_u32width			= u32Width;
	pReq->m_u32borderWidth		= u32BdrWidth;
	pReq->m_u32bgColor			= u32BgColor;
	pReq->m_u32fgColor			= u32FgColor;
	pReq->m_u32borderColor		= u32BdrColor;
	pReq->m_u32fillColor		= FBHMI_WIDGET_DEF_FILCOLOR;
	pReq->m_strText				= strText;
	pReq->m_u32textAlign		= u8Align;
	pReq->m_u32value			= FBHMI_WIDGET_DEF_PBARVAL;
	pReq->m_u32maxValue			= FBHMI_WIDGET_DEF_PBARMAX;
	pReq->m_bvisibility			= bVisible;
	pReq->m_u32fillColor		= u32FillColor;
	pReq->m_u32value			= u32Value;
	pReq->m_u32maxValue			= u32MaxValue;

	return (m_pQueueHandle->bPushQueue(pReq));
}

/************************************************************************
* FUNCTION: fbhmi_FbHmiManager::bCreateScreen
* DESCRIPTION: posts a request to HMI thread to create a label
* PARAMETER: *
* RETURNVALUE: bool
*************************************************************************/
_tBool fbhmi_FbHmiManager::bCreateScreen(_tU16 u16ScreenId, _tU32 u32x, _tU32 u32y,
													  _tU32 u32Width,_tU32 u32Height, _tU32 u32BdrWidth,_tU32 u32FgColor,
													  _tU32 u32BgColor, _tU32 u32BdrColor, string strText,
													  _tU8 u8Align, _tBool bVisible, _tU32 u32FillColor)
{

	DEBUG_TRACE("%s", __func__);
	fbhmi_Screen* pScreen = new fbhmi_Screen();
	if (NULL == pScreen) {
		DEBUG_TRACE("pScreen pointer is NULL");
		return false;
	}

	pScreen->m_u16Id = u16ScreenId;
	pScreen->m_u32x = u32x;
	pScreen->m_u32y = u32y;
	pScreen->m_u32Height = u32Height;
	pScreen->m_u32Width = u32Width;
	pScreen->m_u16BdrWidth = u32BdrWidth;
	pScreen->m_u32BgColor = u32BgColor;
	pScreen->m_u32FgColor = u32FgColor;
	pScreen->m_u32BdrColor = u32BdrColor;
	pScreen->m_u32FillColor = u32FillColor;
	pScreen->m_strText = strText;
	pScreen->m_u8Align = u8Align;
	pScreen->m_bVisible = bVisible;

	/*To remove Lint warning*/
	fbhmi_Screen *p1Screen;
	p1Screen = pScreen;

	m_vecScreenList.push_back(pScreen);

	return true;
}

/************************************************************************
* FUNCTION: fbhmiClearScreen
* DESCRIPTION: posts a request to HMI thread to clear the screen
* PARAMETER: screen_id
* RETURNVALUE: bool
*************************************************************************/
_tBool fbhmi_FbHmiManager::bClearScreen(_tU16 u16ScreenId) const
{
	DEBUG_TRACE("%s", __func__);
	fbhmi_Request* pReq = new fbhmi_Request(FBHMI_EVNT_CLEAR_SCREEN,u16ScreenId, 0);
	if (NULL == pReq) {
		DEBUG_TRACE("Requet pointer is NULL");
		return false;
	}
	return (m_pQueueHandle->bPushQueue(pReq));
}

/************************************************************************
* FUNCTION: fbhmiGetScreen
* DESCRIPTION: gets screen with id
* PARAMETER: screen_id
* RETURNVALUE: fbhmi_Screen *
*************************************************************************/
fbhmi_Screen * fbhmi_FbHmiManager::pclGetScreen(_tU16 u16ScreenId)
{
	DEBUG_TRACE("%s", __func__);
	fbhmi_Screen* pclScr = NULL;
	for(std::vector<fbhmi_Screen*>::iterator itScr = m_vecScreenList.begin(); itScr != m_vecScreenList.end(); itScr++)
	{
		if(u16ScreenId == (*itScr)->u16GetScreenId()) {
			pclScr = *itScr;
			break;
		}
	}
	return pclScr;
}

/************************************************************************
* FUNCTION: bUpdateColor
* DESCRIPTION: posts a request to HMI thread to update the fbhmi_Color properties
* PARAMETER:
* RETURNVALUE: bool
*************************************************************************/
_tBool fbhmi_FbHmiManager::bUpdateColor(_tU16 u16ScreenId, _tU16 u16WidgetId, _tU32 u32BgColor, _tU32 u32FgColor,
													 _tU32 u32BdrColor, _tU32 u32FillColor) const
{
	DEBUG_TRACE("%s", __func__);
	fbhmi_ColorReq* pReq = new fbhmi_ColorReq(FBHMI_EVNT_UPDATE_COLOR, u16ScreenId, u16WidgetId);
	if (NULL == pReq) {
		DEBUG_TRACE("Color pointer pReq  is NULL");
		return false;
	}

	pReq->m_u32BgColor = u32BgColor;
	pReq->m_u32FgColor = u32FgColor;
	pReq->m_u32BdrColor = u32BdrColor;
	pReq->m_u32FillColor = u32FillColor;

	return (m_pQueueHandle->bPushQueue(pReq));
}

/************************************************************************
* FUNCTION: fbhmiUpdateDimension
* DESCRIPTION:posts a request to HMI thread to update the dimension
* PARAMETER: *
* RETURNVALUE: bool
*************************************************************************/
_tBool fbhmi_FbHmiManager::bUpdateDimension(_tU16 u16ScreenId, _tU16 u16WidgetId,
														  _tU32 u32Width, _tU32 u32Height,
														  _tU32 u32BdrWidth) const
{
	DEBUG_TRACE("%s", __func__);
	fbhmi_DimensionReq* pReq = new fbhmi_DimensionReq(FBHMI_EVNT_UPDATE_DIMENSION, u16ScreenId, u16WidgetId);

	if(NULL != pReq)
	{
		pReq->m_u32Height = u32Height;
		pReq->m_u32Width = u32Width;
		pReq->m_u32BdrWidth = u32BdrWidth;
		return (m_pQueueHandle->bPushQueue(pReq));
	}
	else
	{
		return false;
	}
}

/************************************************************************
* FUNCTION:bUpdateProgress
* DESCRIPTION:posts a request to HMI thread to update the prog bar status
* PARAMETER: *
* RETURNVALUE: bool
*************************************************************************/
_tBool fbhmi_FbHmiManager::bUpdateProgress(_tU16 u16ScreenId, _tU16 u16WidgetId, _tU32 u32Value, _tU32 u32MaxValue) const
{
	DEBUG_TRACE("%s", __func__);
	fbhmi_ProgressReq* pReq = new fbhmi_ProgressReq(FBHMI_EVNT_UPDATE_PROGRESSBAR, u16ScreenId, u16WidgetId);

	if (NULL == pReq) {
		DEBUG_TRACE("ProgressBar pointer pReq  is NULL");
		return false;
	}
	pReq->m_u32PbarVal = u32Value;
	pReq->m_u32PbarMax = u32MaxValue;

	return (m_pQueueHandle->bPushQueue(pReq));
}

/************************************************************************
* FUNCTION: fbhmiUpdatePosition
* DESCRIPTION: posts a request to HMI thread to update the position
* PARAMETER: *
* RETURNVALUE: bool
*************************************************************************/
_tBool fbhmi_FbHmiManager::bUpdatePosition(_tU16 u16ScreenId, _tU16 u16WidgetId, _tU32 u32x, _tU32 u32y) const
{
	DEBUG_TRACE("%s", __func__);
	fbhmi_PositionReq* pReq = new fbhmi_PositionReq(FBHMI_EVNT_UPDATE_POSITION, u16ScreenId, u16WidgetId);

	if (NULL == pReq) {
		DEBUG_TRACE("bUpdatePosition pointer pReq  is NULL");
		return false;
	}

	pReq->m_u32x = u32x;
	pReq->m_u32y = u32y;

	return (m_pQueueHandle->bPushQueue(pReq));
}

/************************************************************************
* FUNCTION: fbhmiUpdateText
* DESCRIPTION: posts a request to HMI thread to update the text
* PARAMETER: *
* RETURNVALUE: bool
*************************************************************************/
_tBool fbhmi_FbHmiManager::bUpdateText(_tU16 u16ScreenId, _tU16 u16WidgetId, string strText, _tU8 u8Align, _tBool bVisible) const
{
	DEBUG_TRACE("%s", __func__);
	fbhmi_TextReq* pReq = new fbhmi_TextReq(FBHMI_EVNT_UPDATE_TEXT, u16ScreenId, u16WidgetId);
	if (NULL == pReq) {
		DEBUG_TRACE("Position pointer pReq  is NULL");
		return false;
	}

	pReq->m_strText = strText;
	pReq->m_u8Align = u8Align;
	pReq->m_bVisible = bVisible;

	return (m_pQueueHandle->bPushQueue(pReq));
}

/************************************************************************
* FUNCTION: bDrawScreen
* DESCRIPTION: posts a request to HMI thread to Draw the current Screen
* PARAMETER: *
* RETURNVALUE: bool
*************************************************************************/
_tBool fbhmi_FbHmiManager::bDrawScreen(_tU16 u16ScreenId) const
{
	DEBUG_TRACE("%s", __func__);
	fbhmi_Request* pReq = new fbhmi_Request(FBHMI_EVNT_DRAW_SCREEN, u16ScreenId, 0);
	if (NULL == pReq) {
		DEBUG_TRACE("Request pointer pReq  is NULL");
		return false;
	}
	return (m_pQueueHandle->bPushQueue(pReq));
}


/************************************************************************
* FUNCTION: bUpdateTitle
* DESCRIPTION: Update screen title
* PARAMETER: *
* RETURNVALUE: bool
*************************************************************************/
_tBool fbhmi_FbHmiManager::bUpdateTitle(_tU16 u16ScreenId, string strText, _tU8 u8Align, _tBool bVisible) const
{

	DEBUG_TRACE("%s", __func__);
	fbhmi_TextReq* pReq = new fbhmi_TextReq(FBHMI_EVNT_UPDATE_TITLE, u16ScreenId, 0);

	if (NULL == pReq) {
		DEBUG_TRACE("Request  pointer pReq  is NULL");
		return false;
	}

	pReq->m_strText = strText;
	pReq->m_u8Align = u8Align;
	pReq->m_bVisible = bVisible;

	return (m_pQueueHandle->bPushQueue(pReq));
}


/************************************************************************
* FUNCTION: fbhmiGetQueue
* DESCRIPTION: return address of the static queue handler
* PARAMETER: None*
* RETURNVALUE: fbhmi_QueueHandler*
*************************************************************************/
fbhmi_QueueHandler* fbhmi_FbHmiManager::pclGetQueueHandle()
{
	DEBUG_TRACE("%s", __func__);
	return (m_pQueueHandle);
}

/************************************************************************
* FUNCTION:
* DESCRIPTION:The main HMI thread responsible for framebuffer display
*			  based on events from the user application.
* PARAMETER: None*
* RETURNVALUE: None
*************************************************************************/
_tVoid fbhmi_FbHmiManager::vFbHMIThreadFunction()
{
	DEBUG_TRACE("%s", __func__);
	fbhmi_QueueHandler* pclQueueHandle = fbhmi_FbHmiManager::pclGetQueueHandle();
	m_pSurfaceHandle = fbhmi_Surface::pGetInstance();
	if (NULL == pclQueueHandle || NULL == m_pSurfaceHandle) {
		DEBUG_TRACE("%s ptr is NULL",(NULL == pclQueueHandle)?"pclQueueHandle": "m_pSurfaceHandle");
		DEBUG_TRACE("Aborting Thread creation")
			return;
	}
	while(true)
	{
		//consume all HMI events
		if(pclQueueHandle->bIsQueueEmpty())
		{
			pclQueueHandle->vLockMQ();
			pclQueueHandle->m_isQLocked = true;
			pclQueueHandle->vWaitMQ();
		}

		while(false == pclQueueHandle->bIsQueueEmpty())
		{
			fbhmi_Request* pReq = pclQueueHandle->pQueueFront();

			if (NULL == pReq)
			{
				DEBUG_TRACE("NULL Object in the queue Removing it");
				pclQueueHandle->vPopQueue();
				continue;
			}
			//process the HMI req
			vProcessFbHMIEvent(pReq);

			//delete the event once processed
			pclQueueHandle->vPopQueue();
		}

		if(pclQueueHandle->m_isQLocked)
		{
			pclQueueHandle->vUnlockMQ();
			pclQueueHandle->m_isQLocked = false;
		}
	}

}


/************************************************************************
* FUNCTION: vProcessFbHMIEvent
* DESCRIPTION:This function process the hmi event present in a request queue.
*
* PARAMETER: None
* RETURNVALUE: None
*************************************************************************/
_tVoid fbhmi_FbHmiManager::vProcessFbHMIEvent(fbhmi_Request* pReq)
{
	_tU16 u16RequestID = pReq->u16GetRequestID();
	_tU16 u16ScreenID = pReq->u16GetScreenID();

	//get the screen
	fbhmi_Screen * pclScreen = NULL;
	pclScreen = pclGetScreen(u16ScreenID);

	if(NULL != pclScreen)
	{
		DEBUG_TRACE("ScrID: %u, ReqID: %u", u16ScreenID, u16RequestID);
		//process the HMI event
		switch(u16RequestID)
		{
		case FBHMI_EVNT_CREATE_LABEL:
		case FBHMI_EVNT_CREATE_PROGRESSBAR:
			pclScreen->bAddChildWidget(pReq);
			break;

		case FBHMI_EVNT_DRAW_SCREEN:
			m_pSurfaceHandle->vDrawToFrameBuffer();
			break;

		case FBHMI_EVNT_CREATE_SCREEN:
			break;

		case FBHMI_EVNT_CLEAR_SCREEN:
			pclScreen->vClearScreen();
			m_pSurfaceHandle->vDrawToFrameBuffer();
			break;

		case FBHMI_EVNT_UPDATE_DIMENSION:
		case FBHMI_EVNT_UPDATE_COLOR:
		case FBHMI_EVNT_UPDATE_PROGRESSBAR:
		case FBHMI_EVNT_UPDATE_POSITION:
		case FBHMI_EVNT_UPDATE_TEXT:
		case FBHMI_EVNT_UPDATE_TEXT_ALIGN:
		case FBHMI_EVNT_UPDATE_VISIBILITY:
		case FBHMI_EVNT_UPDATE_TITLE:
			pclScreen->vUpdateWidget(pReq);
			m_pSurfaceHandle->vDrawToFrameBuffer();
			break;
		default:
			break;
		}
	}
}

/*******************************************************************************************
* FUNCTION:		u32GetXRes
* DESCRIPTION:	This function returns the X-resolution.
* PARAMETER:
*					tVoid
* RETURNVALUE:	_tU32
*
* HISTORY:
*					14.03.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
_tU32 fbhmi_FbHmiManager::u32GetXRes()
{
	return m_u32xRes;
}

/*******************************************************************************************
* FUNCTION:		u32GetYRes
* DESCRIPTION:	This function returns the Y-resolution.
* PARAMETER:
*					tVoid
* RETURNVALUE:	_tU32
*
* HISTORY:
*					14.03.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
_tU32 fbhmi_FbHmiManager::u32GetYRes()
{
	return m_u32yRes;
}

