/************************************************************************
* File: fbhmi_BDFFont.cpp
* SW-Component: ASCII HMI for recovery download
*
* Description:
*   This is a singleton class used to initialize and use the Linux
*   framebuffer for displaying characters. 
* Author:
*   Adityakumar.jha@in.bosch.com
* Copyright:
*   Robert Bosch Engineering and Business Solutions Ltd, Bangalore.
*
* History:
* 07.06.2013 - Initial version - Adityakumar.jha@in.bosch.com
* 13.06.2013 - Updated for first release - Gururaj.B@in.bosch.com
* 19.03.2014 - Added s32GetFontXPlus() and s32GetFontYPlus()- Aditya Kumar Jha
************************************************************************/
#include "fbhmi_BDFFont.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "../fbhmi_Surface.h"

#define BULK_SIZE 4096

/************************************************************************
* FUNCTION: fbhmi_BDFFont::fbhmi_BDFFont()
*
* DESCRIPTION: Constructor
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
fbhmi_BDFFont::fbhmi_BDFFont()
{
	DEBUG_TRACE("%s", __func__);
	m_pchMemory = NULL;
	m_s32BdfPixelSize = 0;
	m_pchMemory = NULL;
	m_u32FontHeight = 0;
	m_u32FontWidth = 0;
	m_s32FontsXPlus = 0;
	m_s32FontsYPlus = 0;
}

/************************************************************************
* FUNCTION: fbhmi_BDFFont::~fbhmi_BDFFont()
*
* DESCRIPTION: Destructor
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
fbhmi_BDFFont::~fbhmi_BDFFont()
{
	DEBUG_TRACE("%s", __func__);
	if (m_pchMemory != NULL)
		free(m_pchMemory);
}

/************************************************************************
* FUNCTION: fbhmi_BDFFont::pGetMemory()
*
* DESCRIPTION: 
*
* PARAMETER: char *, char *, int
*
* RETURNVALUE: char *
*************************************************************************/
_tChar *fbhmi_BDFFont::pGetMemory(_tChar *pchBuf, _tChar *pchFile, _tS32 s32Buff_size)
{
	_tS32 s32Size = 0;

	if(pchBuf == NULL || pchFile == NULL) {
		DEBUG_TRACE("%s is NULL",(pchBuf == NULL)? "pchBuf":"pchFile");
		return 0;
	}

	while (*pchFile != 0x0a && s32Size < (s32Buff_size-1)) {
		pchBuf[s32Size] = *pchFile;
		pchFile++;
		s32Size++;
	}
	pchBuf[s32Size] = 0;
	if (pchBuf[s32Size - 1] == 0x0d)
		pchBuf[s32Size-1] = 0;

	if (*pchFile != 0x0a)
		return pchFile;
	else
		return (pchFile+1);
}

/************************************************************************
* FUNCTION: fbhmi_BDFFont::pGetSizePart()
*
* DESCRIPTION: 
*
* PARAMETER: char *
*
* RETURNVALUE: char *
*************************************************************************/
_tChar *fbhmi_BDFFont::pGetSizePart(_tChar *pchBuf)
{
	DEBUG_TRACE("%s", __func__);
	if(pchBuf == NULL ) {
		DEBUG_TRACE("pchBuf is NULL");
		return 0;
	}
	_tS32 s32I;
	_tChar *pchPart;
	for (s32I=0; s32I<8; s32I++) {
		pchPart = pchBuf;
		pchBuf = strchr(pchBuf, '-');
		if (!pchBuf) return NULL;
		pchBuf++;
	}
	pchBuf[-1] = 0;
	return pchPart;
}


/************************************************************************
* FUNCTION: fbhmi_BDFFont::vGetSizeAndEncoding()
*
* DESCRIPTION: 
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
_tVoid fbhmi_BDFFont::vGetSizeAndEncoding()
{
	DEBUG_TRACE("%s", __func__);
	_tChar achBuf[BULK_SIZE];

	_tChar *pchFile = this->m_pchMemory;
	if(NULL != pchFile )
	{
		while (*pchFile != 0) {
			pchFile = pGetMemory(achBuf, pchFile, BULK_SIZE);
			if (strncmp(achBuf, "FONT ", strlen("FONT "))==0)
			{
				_tBool bEncoding = false;
				if (strstr(achBuf, "iso10646-1") != NULL || strstr(achBuf, "ISO10646-1") != NULL)
					bEncoding = true;

				if (bEncoding == false)
					throw fbhmi_FontException("Invalid encoding in BDF font. Only iso10646-1 (range 0x0000-0x10ffff)");

				_tChar *pchFontSizePart = pGetSizePart(achBuf);
				sscanf(pchFontSizePart, "%d", &this->m_s32BdfPixelSize);

				return;
			}
		}
	}
	throw fbhmi_FontException("FONT directive not found in file");
}

/************************************************************************
* FUNCTION: fbhmi_BDFFont::vLoadFontToMem()
*
* DESCRIPTION: loads font info from file to memory
*
* PARAMETER: const string
*
* RETURNVALUE: None
*************************************************************************/
_tVoid fbhmi_BDFFont::vLoadFontToMem(const string strFontFileName)
{
	DEBUG_TRACE("%s", __func__);
	FILE *file = fopen(strFontFileName.c_str(), "r");
	if (file == NULL){
		throw fbhmi_FontException("Font file open failed");
	}
	else
	{
		fseek(file, 0, SEEK_END);
		long size = ftell(file) + 1;
		fseek(file, 0, SEEK_SET);

		this->m_pchMemory = (char *)malloc(size);
		_tChar *pchFont = NULL;
		if (NULL != this->m_pchMemory) {
			pchFont = this->m_pchMemory;
			_tS64 rsize = 1;
			while (!feof(file) && (rsize!=0)) {
				rsize = fread(pchFont, 1, BULK_SIZE, file);
				pchFont += rsize;
			}
			*pchFont = 0;
		}
		fclose(file);
	}
}

/************************************************************************
* FUNCTION: fbhmi_BDFFont::vFillMapping()
*
* DESCRIPTION: 
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
_tVoid fbhmi_BDFFont::vFillMapping()
{
	DEBUG_TRACE("%s", __func__);
	_tChar *pcharStart_font_addr = this->m_pchMemory;
	_tChar *pchFont = NULL;
	_tChar achBuf[BULK_SIZE];

	if (NULL != this->m_pchMemory)
		pchFont = this->m_pchMemory;

	_tS64 s64Start_char_loc = 0;

	if(NULL != pchFont && NULL != pcharStart_font_addr) {
		while (*pchFont != 0) {

			_tS64 s64Loc = pchFont - pcharStart_font_addr;
			_tChar *pcharNext_font = pGetMemory(achBuf, pchFont, BULK_SIZE);

			if (strncmp(achBuf, "STARTCHAR ", strlen("STARTCHAR "))==0)	
			{
				s64Start_char_loc = s64Loc;
			}
			if (strncmp(achBuf, "ENCODING ", strlen("ENCODING "))==0)
			{
				_tS32 enc;
				sscanf(achBuf, "ENCODING %d\n", &enc);

				m_mapping[enc] = s64Start_char_loc;

			}

			pchFont = pcharNext_font;
		}
	}
}

/************************************************************************
* FUNCTION: fbhmi_BDFFont::vLoadBDF()
*
* DESCRIPTION: 
*
* PARAMETER: const string
*
* RETURNVALUE: None
*************************************************************************/
_tVoid fbhmi_BDFFont::vLoadBDF(const string sFontFileName)
{
	DEBUG_TRACE("%s", __func__);
	vLoadFontToMem(sFontFileName);

	vGetSizeAndEncoding();
	this->m_s32PixelSize = m_s32BdfPixelSize;

	vFillMapping();
}

/************************************************************************
* FUNCTION: fbhmi_BDFFont::bIsCharDefined()
*
* DESCRIPTION: 
*
* PARAMETER: uint32_t
*
* RETURNVALUE: bool
*************************************************************************/
_tBool fbhmi_BDFFont::bIsCharDefined(_tU32 u32Ch)
{
	DEBUG_TRACE("");
	if (m_mapping.find(u32Ch) == m_mapping.end())
		return false;
	else
		return true;
}

/************************************************************************
* FUNCTION: fbhmi_BDFFont::s32GetAdvanceX()
*
* DESCRIPTION: 
*
* PARAMETER: uint32_t
*
* RETURNVALUE: int
*************************************************************************/
_tS32 fbhmi_BDFFont::s32GetAdvanceX(_tU32 s32Ch)
{
	_tU32 u32RealCh = u32GetRealCharIndex(s32Ch);
	DEBUG_TRACE("%s", __func__);
	if (u32RealCh == 0xFFFFFFFF)
		return m_s32PixelSize/2;

	_tChar achBuf[BULK_SIZE];
	_tChar *pchFont = NULL;

	if(this->m_pchMemory != NULL)
		pchFont = this->m_pchMemory + m_mapping[u32RealCh];

	if(pchFont != NULL)
	{
		while (*pchFont != 0) {
			pchFont = pGetMemory(achBuf, pchFont, BULK_SIZE);

			if (strncmp(achBuf, "ENDCHAR", strlen("ENDCHAR"))==0)
				return m_s32PixelSize/2;
			if (strncmp(achBuf, "DWIDTH " , strlen("DWIDTH "))==0)
			{
				_tS32 s32Advance_x, s32Advance_y;
				sscanf(achBuf, "DWIDTH %d %d", &s32Advance_x, &s32Advance_y);

				if (m_s32PixelSize == m_s32BdfPixelSize)
					return s32Advance_x;

				// TODO: do rescale of advance_x
				return s32Advance_x;
			}
		}
	}
	return m_s32PixelSize/2;
}

/************************************************************************
* FUNCTION: fbhmi_BDFFont::iGetAdvanceY()
*
* DESCRIPTION: 
*
* PARAMETER: uint32_t
*
* RETURNVALUE: int
*************************************************************************/
_tS32 fbhmi_BDFFont::s32GetAdvanceY(_tU32 u32Ch)
{
	_tU32 u32RealCh = u32GetRealCharIndex(u32Ch);
	DEBUG_TRACE("%s", __func__);
	if (u32RealCh == 0xFFFFFFFF)
		return 0;

	_tChar achBuf[BULK_SIZE];

	_tChar *pchFont = NULL;
	if (NULL != this->m_pchMemory)
		pchFont = this->m_pchMemory + m_mapping[u32RealCh];
	if(NULL != pchFont){
		while (*pchFont != 0) {
			pchFont = pGetMemory(achBuf, pchFont, BULK_SIZE);

			if (strncmp(achBuf, "ENDCHAR", strlen("ENDCHAR"))==0)
				return 0;
			if (strncmp(achBuf, "DWIDTH " , strlen("DWIDTH "))==0)
			{
				_tS32 s32Advance_x, s32Advance_y;
				sscanf(achBuf, "DWIDTH %d %d", &s32Advance_x, &s32Advance_y);

				if (m_s32PixelSize == m_s32BdfPixelSize)
					return s32Advance_y;

				// TODO: do rescale of advance_y
				return s32Advance_y;
			}
		}
	}
	return 0;
}

/************************************************************************
* FUNCTION: fbhmi_BDFFont::vDrawNotScaled()
*
* DESCRIPTION: 
*
* PARAMETER: fbhmi_Surface *, char *, unsigned int, unsigned int, 
*			 int, int, unsigned int, unsigned int
*
* RETURNVALUE: None
*************************************************************************/
_tVoid fbhmi_BDFFont::vDrawNotScaled(fbhmi_Surface *graphics,
												 _tChar *pchFont, 
												 _tU32 u32PosX, 
												 _tU32 u32PosY, 
												 _tS32 s32X_plus, 
												 _tS32 s32Y_plus, 
												 _tU32 u32Size_x, 
												 _tU32 u32Size_y
												 )
{
	_tChar achBuf[BULK_SIZE];
	_tU32 u32Y, u32X;

	for (u32Y=0; u32Y<u32Size_y; u32Y++)
	{
		pchFont = pGetMemory(achBuf, pchFont, BULK_SIZE);

		if (strncmp(achBuf, "ENDCHAR", strlen("ENDCHAR"))==0)
			break;

		for (u32X=0; u32X<u32Size_x; u32X++)
		{
			_tS32 s32B_s = u32X >> 2;
			_tChar bb = achBuf[s32B_s];
			bb = toupper(bb);

			_tChar achhex[2];
			achhex[0] = bb;
			achhex[1] = 0;

			_tU32 hhex;
			sscanf(achhex, "%x", &hhex);

			_tS32 s32Mask = 1 << (3 - (u32X % 4));

			if (hhex & s32Mask) {
				graphics->vSetPixel(u32PosX + u32X + s32X_plus, u32PosY - u32Size_y + u32Y - s32Y_plus);
			}
		}
	}
}

/************************************************************************
* FUNCTION: fbhmi_BDFFont::vDrawChar()
*
* DESCRIPTION: 
*
* PARAMETER: fbhmi_Surface *, uint32_t, unsigned int, unsigned int
*
* RETURNVALUE: None
*************************************************************************/
_tVoid fbhmi_BDFFont::vDrawChar(fbhmi_Surface *graphics, _tU32 u32Ch, _tU32 u32PosX, _tU32 u32PosY)
{
	_tU32 u32RealCh = u32GetRealCharIndex(u32Ch);
	DEBUG_TRACE("RealCh: %u Ch: %u X: %u Y: %u", u32RealCh, u32Ch, u32PosX, u32PosY);
	if (graphics == NULL){
		DEBUG_TRACE("Surface pointer is NULL");
		return;
	}
	if (u32RealCh == 0xFFFFFFFF) {
		vDrawBlock(graphics,u32PosX, u32PosY);
	}

	_tChar achBuf[BULK_SIZE];
	_tS32 s32Size_x = 0, s32Size_y = 0, s32X_plus = 0, s32Y_plus = 0 ;

	_tChar *pchFont = NULL;

	if (NULL != this->m_pchMemory)
		pchFont = this->m_pchMemory + m_mapping[u32RealCh];

	if (NULL != pchFont) {
		while (*pchFont != 0) {
			pchFont = pGetMemory(achBuf, pchFont, BULK_SIZE);

			if (strncmp(achBuf, "ENDCHAR", strlen("ENDCHAR"))==0)
				break;
			if (strncmp(achBuf, "BBX ", strlen("BBX "))==0)
				sscanf(achBuf, "BBX %d %d %d %d", &s32Size_x, &s32Size_y, &s32X_plus, &s32Y_plus);
			if (strncmp(achBuf, "BITMAP", strlen("BITMAP"))==0)
			{
				if (m_s32BdfPixelSize == m_s32PixelSize)
					vDrawNotScaled(graphics, pchFont, u32PosX, u32PosY, s32X_plus, s32Y_plus, s32Size_x, s32Size_y);
				else {
					//TODO: rescale char;
				}
			}
		}
	}
}
/************************************************************************
* FUNCTION: fbhmi_Font::vDrawBlock()
*
* DESCRIPTION: 
*
* PARAMETER: fbhmi_Surface *, posX, posY
*
* RETURNVALUE: None
*************************************************************************/
_tVoid fbhmi_BDFFont::vDrawBlock(fbhmi_Surface *graphics, _tU32 u32PosX, _tU32 u32PosY)
{
	_tS32 s32X, s32Y;
	DEBUG_TRACE("%s", __func__);

	if (graphics == NULL){
		DEBUG_TRACE("Surface pointer is NULL");
		return;
	}

	for (s32Y= 0; s32Y < m_s32PixelSize; s32Y++)
		for (s32X=0; s32X<(m_s32PixelSize/2); s32X++)
			fbhmi_Surface::pGetInstance()->vSetPixel(u32PosX + s32X, u32PosY - s32Y);
}
/************************************************************************
* FUNCTION: fbhmi_BDFFont::u32GetFontWidth()
*
* DESCRIPTION: 
*
* PARAMETER: NONE
*
* RETURNVALUE: _tU32
*************************************************************************/
_tU32 fbhmi_BDFFont::u32GetFontHeight()
{
	DEBUG_TRACE("%s", __func__);
	return m_u32FontHeight;
}

/************************************************************************
* FUNCTION: fbhmi_BDFFont::u32GetFontWidth()
*
* DESCRIPTION: 
*
* PARAMETER: NONE
*
* RETURNVALUE: _tU32
*************************************************************************/
_tU32 fbhmi_BDFFont::u32GetFontWidth()
{
	DEBUG_TRACE("%s", __func__);
	return m_u32FontWidth;

}
/************************************************************************
* FUNCTION: fbhmi_BDFFont::u32GetFontWidth()
*
* DESCRIPTION: 
*
* PARAMETER: NONE
*
* RETURNVALUE: _tU32
*************************************************************************/
_tVoid fbhmi_BDFFont::vExtractFontSize()
{
	_tChar achBuf[BULK_SIZE];
	_tChar *pchFile = this->m_pchMemory;
	DEBUG_TRACE("%s", __func__);

	if(NULL != pchFile) {
		while (*pchFile != 0){
			pchFile = pGetMemory(achBuf, pchFile, BULK_SIZE);
			if (strncmp(achBuf, "FONTBOUNDINGBOX ", strlen("FONTBOUNDINGBOX "))==0)
			{
				sscanf(achBuf, "FONTBOUNDINGBOX %d %d %d %d"\
					,&m_u32FontWidth, &m_u32FontHeight, &m_s32FontsXPlus, &m_s32FontsYPlus);
				return;
			}
		}
	}
	throw fbhmi_FontException("FONT directive not found in file");
}

/************************************************************************
* FUNCTION: fbhmi_BDFFont::s32GetFontXPlus()
*
* DESCRIPTION: 
*
* PARAMETER: NONE
*
* RETURNVALUE: _tS32
*************************************************************************/
_tS32 fbhmi_BDFFont::s32GetFontXPlus()
{
	DEBUG_TRACE("%s", __func__);
	return m_s32FontsXPlus;
}

/************************************************************************
* FUNCTION: fbhmi_BDFFont::s32GetFontYPlus()
*
* DESCRIPTION: 
*
* PARAMETER: NONE
*
* RETURNVALUE: _tS32
*************************************************************************/
_tS32 fbhmi_BDFFont::s32GetFontYPlus()
{
	DEBUG_TRACE("%s", __func__);
	return m_s32FontsYPlus;

}
